#include <linux/config.h>
#include <linux/utsname.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include <pcp/pmapi.h>
#include <pcp/pmda.h>

#include "lockstat.h"
#include <linux/lockmeter.h>


#include "domain.h"
#include "pmda_lockstat.h"

int opt_debug = 1;

__pmHashCtl			pcp_lockstat_hash;
lstat_user_request_t		*lockstat_req = NULL;
lstat_directory_entry_t		*lockstat_dir = NULL;
lstat_read_lock_cpu_counts_t	*lockstat_rcounts = NULL;
lstat_cpu_counts_t		*lockstat_counts = NULL;
char				*lockstat_mapfile = NULL;

static char *
get_lock_name(void *adr, char *buf)
{
    char *p;

    XlateAddressToSymbol(adr, buf);
    while ((p = strchr(buf, ' ')) != NULL) {
	if (*(p+1) == '\0')
	    *p = '\0';
	else
	    strcpy(buf, p+1);
    }
    return buf;
}


static int		_isDSO = 1;	/* =0 I am a daemon */

/*
 * Metric Instance Domains
 */
static pmdaIndom indomtab[] = {
/*
 * The "all" instance domain "lock"
 * (aggregated for all callers and all cpus)
 */
#define INDOM_ALL 0 
    { INDOM_ALL, 0, NULL },
/*
 * The "CALLER" instance domain "lock:caller"
 * (aggregated for all cpus)
 */
#define INDOM_CALLER 1 
    { INDOM_CALLER, 0, NULL },
/*
 * The "detail" instance domain lock:caller:cpu
 * (no aggregation)
 */
#define INDOM_DETAIL 2 
    { INDOM_DETAIL, 0, NULL },
};

/*
 * fetch clusters
 */
#define CLUSTER_ALL		0
#define CLUSTER_CALLER		1
#define CLUSTER_DETAIL		2

#define NUM_CLUSTERS		3 /* one more than highest numbered cluster */


#define PCP_TIME_COUNTER(CLUSTER, ID, INDOM) \
    { NULL, \
      { PMDA_PMID((CLUSTER),(ID)), PM_TYPE_U32, (INDOM), PM_SEM_COUNTER, \
      PMDA_PMUNITS(0,1,0,0,PM_TIME_MSEC,0) }, }

#define PCP_TIME_INSTANT(CLUSTER, ID, INDOM) \
    { NULL, \
      { PMDA_PMID((CLUSTER),(ID)), PM_TYPE_U32, (INDOM), PM_SEM_INSTANT, \
      PMDA_PMUNITS(0,1,0,0,PM_TIME_MSEC,0) }, }

#define PCP_EVENT_COUNTER(CLUSTER, ID, INDOM) \
    { NULL, \
      { PMDA_PMID((CLUSTER),(ID)), PM_TYPE_U32, (INDOM), PM_SEM_COUNTER, \
      PMDA_PMUNITS(0,0,1,0,0,0) }, }

/*
 * all metrics supported in this PMDA - one table entry for each
 */
static pmdaMetric metrictab[] = {

/*
 * detail cluster
 */

/* lockstat.detail.time.wait */
    PCP_TIME_COUNTER(CLUSTER_DETAIL, 0, INDOM_DETAIL),

/* lockstat.detail.time.hold */
    PCP_TIME_COUNTER(CLUSTER_DETAIL, 1, INDOM_DETAIL),

/* lockstat.detail.time.max_wait */
    PCP_TIME_INSTANT(CLUSTER_DETAIL, 2, INDOM_DETAIL),

/* lockstat.detail.time.max_hold */
    PCP_TIME_INSTANT(CLUSTER_DETAIL, 3, INDOM_DETAIL),

/* lockstat.detail.count.spin */
    PCP_EVENT_COUNTER(CLUSTER_DETAIL, 4, INDOM_DETAIL),

/* lockstat.detail.count.nowait */
    PCP_EVENT_COUNTER(CLUSTER_DETAIL, 5, INDOM_DETAIL),
};

static void
lockstat_refresh(int header_only)
{
    int					i;
    int					id;
    int					cpu;
    __pmHashNode			*node;
    lockstat_inst_entry_t		*entry;
    lstat_directory_entry_t             *d;
    pmdaIndom				*ip;
    char				tmpbuf[1024];

    /*
     * fetch new lock data
     */
    getKernelData(&lockstat_req, &lockstat_counts, &lockstat_dir,
	 &lockstat_rcounts, header_only);

    if (header_only) {
    	return;
    }

    /*
     * refresh the instance domains
     */
    for (i=0; i < LSTAT_MAX_STAT_INDEX; i++) {
        d = &lockstat_dir[i]; /* dir entry */
        if (d->caller_ra && d->lock_ptr) {
	    for (cpu=0; cpu < lockstat_req->maxcpus; cpu++) {
		id = PCP_LOCKSTAT_HASHKEY(i, cpu);
		node = __pmHashSearch(id, &pcp_lockstat_hash);
		if (node == NULL) {
		    /*
		     * Add a new lock:caller:cpu instance
		     */
		    entry = (lockstat_inst_entry_t *)malloc(sizeof(lockstat_inst_entry_t));
		    entry->dir = i;
		    entry->cpu = cpu;
		    entry->lockname = strdup(get_lock_name(d->lock_ptr, tmpbuf));
		    entry->callername = strdup(get_lock_name(d->caller_ra, tmpbuf));
		    __pmHashAdd(id, (void *)entry, &pcp_lockstat_hash);
		    /*
		     * add this instance to the instance domain
		     */
		    ip = &indomtab[INDOM_DETAIL];
		    ip->it_numinst++;
		    ip->it_set = (pmdaInstid *)realloc(ip->it_set,
		    	ip->it_numinst * sizeof(pmdaInstid));

		    sprintf(tmpbuf, "%s:%s:cpu%d", entry->lockname, entry->callername, cpu);
		    ip->it_set[ip->it_numinst-1].i_name = strdup(tmpbuf);
		    ip->it_set[ip->it_numinst-1].i_inst = id;
		}
	    }
        }
    }
}

static int
lockstat_instance(pmInDom indom, int inst, char *name, __pmInResult **result, pmdaExt *pmda)
{
    __pmInDom_int	*indomp = (__pmInDom_int *)&indom;

    switch (indomp->serial) {
    default:
	/* pmdaInstance will pick up errors */
    }

    return pmdaInstance(indom, inst, name, result, pmda);
}


/*
 * callback provided to pmdaFetch
 */

static int
lockstat_fetchCallBack(pmdaMetric *mdesc, unsigned int inst, pmAtomValue *atom)
{
    __pmID_int		*idp = (__pmID_int *)&(mdesc->m_desc.pmid);
    int			cpu;
    int			dir;
    __pmHashNode	*node;
    lockstat_inst_entry_t *inst_entry;

    cpu = inst & 0xffff;
    dir = (inst & 0xffff0000) >> 16;

    if (dir < 0 || dir >= LSTAT_MAX_STAT_INDEX)
    	return PM_ERR_INST;
    
    node = __pmHashSearch(inst, &pcp_lockstat_hash);
    if (node == NULL)
    	return PM_ERR_INST;

    inst_entry = (lockstat_inst_entry_t *)node->data;
    
    if (mdesc->m_user != NULL) {
	/* 
	 * The metric value is extracted directly via the address specified
	 * in metrictab.  Note: not all metrics support this - those that
	 * don't have NULL for the m_user field in their respective
         * metrictab slot.
	 */
    }
    else
    switch (idp->cluster) {
    case CLUSTER_DETAIL:
	switch(idp->item) {

#define CYCLES_TO_MILLISECS(x) 1000 * (x) / lockstat_req->cycleval

	case 0: /* lockstat.detail.time.wait */
	    atom->ul = CYCLES_TO_MILLISECS(lockstat_counts[cpu][dir].cum_wait_ticks);
	    break;

	case 1: /* lockstat.detail.time.hold */
	    atom->ul = CYCLES_TO_MILLISECS(lockstat_counts[cpu][dir].cum_hold_ticks);
	    break;

	case 2: /* lockstat.detail.time.max_wait */
	    atom->ul = CYCLES_TO_MILLISECS(lockstat_counts[cpu][dir].max_wait_ticks);
	    break;

	case 3: /* lockstat.detail.time.max_hold */
	    atom->ul = CYCLES_TO_MILLISECS(lockstat_counts[cpu][dir].max_hold_ticks);
	    break;

	case 4: /* lockstat.detail.count.spin */
	    atom->ul = lockstat_counts[cpu][dir].count[LSTAT_ACT_SPIN];
	    break;

	case 5: /* lockstat.detail.count.nowait */
	    atom->ul = lockstat_counts[cpu][dir].count[LSTAT_ACT_NO_WAIT];
	    break;
		
	default:
	    return PM_ERR_PMID;
	}
	break;
    default:
    	return PM_ERR_PMID;
    }

    return 1;
}


static int
lockstat_fetch(int numpmid, pmID pmidlist[], pmResult **resp, pmdaExt *pmda)
{
    lockstat_refresh(0);
    return pmdaFetch(numpmid, pmidlist, resp, pmda);
}

/*
 * Initialise the agent (both daemon and DSO).
 */

void 
lockstat_init(pmdaInterface *dp)
{
    int					i;
    int					cpu;
    lstat_directory_entry_t		*d;
    lstat_lock_counts_t			*c;
    lstat_read_lock_counts_t		*r;

    if (geteuid() != 0) {
    	fprintf(stderr, "FATAL Error: root access is required for this PMDA\n");
	exit(1);
    }

#if 0
    setCollectionState(LSTAT_ON);
#endif

    /*
     * initialize lockstat API and indom hash table
     */
    memset(&pcp_lockstat_hash, 0, sizeof(pcp_lockstat_hash));

    openKernelData(defaultDataFilename);
    openMapFile(lockstat_mapfile);

    lockstat_refresh(1);
    lockstat_refresh(0);

    if (_isDSO) {
	char helppath[MAXPATHLEN];
	sprintf(helppath, "%s/pmdas/lockstat/help", pmGetConfig("PCP_VAR_DIR"));
    	pmdaDSO(dp, PMDA_INTERFACE_3, "lockstat DSO", helppath);
    }

    if (dp->status != 0)
	return;

    dp->version.two.instance = lockstat_instance;
    dp->version.two.fetch = lockstat_fetch;
    pmdaSetFetchCallBack(dp, lockstat_fetchCallBack);

    pmdaInit(dp, indomtab, sizeof(indomtab)/sizeof(indomtab[0]), metrictab,
             sizeof(metrictab)/sizeof(metrictab[0]));
}

static void
usage(void)
{
    fprintf(stderr, "Usage: %s -s System.map [options]\n\n", pmProgname);
    fputs("Options:\n"
	  "  -d domain  use domain (numeric) for metrics domain of PMDA\n"
	  "  -l logfile write log into logfile rather than using default log name\n",
	  stderr);		
    exit(1);
}

/*
 * Set up the agent if running as a daemon.
 */

int
main(int argc, char **argv)
{
    int			err = 0;
    int			c = 0;
    pmdaInterface	dispatch;
    char		helppath[MAXPATHLEN];
    char		*p;

    /* trim cmd name of leading directory components */
    pmProgname = argv[0];
    for (p = pmProgname; *p; p++) {
	if (*p == '/')
	    pmProgname = p+1;
    }

    _isDSO = 0;

    sprintf(helppath, "%s/pmdas/lockstat/help", pmGetConfig("PCP_VAR_DIR"));
    pmdaDaemon(&dispatch, PMDA_INTERFACE_3, pmProgname, LOCKSTAT, "lockstat.log", helppath);

    while ((c = pmdaGetOpt(argc, argv, "s:D:d:l:?", &dispatch, &err)) != EOF) {
	switch(c) {
	case 's':
	    lockstat_mapfile = optarg;
	    break;
	default:
	    err++;
	}
    }

    if (lockstat_mapfile == NULL) {
	/* "-s" is a mandatory option */
    	err++;
    }

    if (err)
    	usage();

    pmdaOpenLog(&dispatch);
    lockstat_init(&dispatch);
    pmdaConnect(&dispatch);
    pmdaMain(&dispatch);

    exit(0);
    /*NOTREACHED*/
}
