package WebService::OurWorldInData::Catalog;
# ABSTRACT: Scrapes the Our World in Data search results for datasets

use Moo;
extends 'WebService::OurWorldInData';

use Carp;
use JSON qw(decode_json);
use PerlX::Maybe qw( maybe provided );
use Types::Standard qw( Str Bool Enum ); # Int ArrayRef HashRef InstanceOf ConsumerOf

has query => (
    is  => 'rw',
    isa => Str,
);

# list of all the known charts
# should make it a constant
my @data = <DATA>;
chomp @data;

=head1 SYNOPSIS

    my $chart = WebService::OurWorldInData::Chart->new({
        chart => 'sea-surface-temperature-anomaly', # dataset name
    });

    my $result = $chart->data(); # get the csv data
    my $rows = $chart->parse_data($result);

=head1 DESCRIPTION

Queries the Our World in Data Chart api which provides data and metadata
in CSV format. The Chart object can be created with the following attributes:

short_names - a boolean flag to affect the results
csv_type - either full (default) or filtered
time/country - filter the results you request

as described by the OWiD API.

    my $chile = WebService::OurWorldInData::Chart->new(
                    chart => 'life-expectancy',
                    csv_type => 'filtered',
                    country => '~CHL',
                    time => '1998..2023' );

=head1 Methods

=head2 find

Our World in Data holds the data on 14,368 charts across 123 topics
as of July 2025. Because they haven't opened up their Catalog API yet,
I've had to scrape their web page for results. This means that the
C<find> method is not dynamic, up-to-date or anywhere near complete.

There are currently 1822 charts known by this module.

=head2 get_topics

This will return a list of topics that I've scraped so far.
It is not the complete 123 as of July 2025, but it should
give you a start.

=head1 SEE ALSO

=over 4

=item * L<HTTP::Tiny>

=back

=cut

sub get_path {
    my $self = shift;
    return join '/', $self->base_url, 'data';
}

sub find {
    my ($self, $query) = @_;

    $query = qr/$query/ unless ref $query eq 'Regexp';
    my @hits = grep { /$query/ } @data;
    return \@hits;

    # if we ever get an api to the data endpoint
    #return $self->get_response( $url, { q => $query } );
}

sub get_topics {
    return qw(
        Age+Structure
        Air+Pollution
        Antibiotics+%26+Antibiotic+Resistance
        Biodiversity
        Burden+of+Disease
        Cancer
        Cardiovascular+Diseases
        Causes+of+Death
        Child+%26+Infant+Mortality
        Clean+Water
        Climate+Change
        Energy
        Fertility+Rate
        Fish+%26+Overfishing
        Forests+%26+Deforestation
        Gender+Ratio
        Global+Health
        Healthcare+Spending
        Life+Expectancy
        Mental+Health
        Migration
        Oil+Spills
        Pandemics
        Population+Growth
        Sanitation
        Suicides
        Urbanization
        Vaccination
        Wildfires
    );
}

1;

__DATA__
5-year-survival-rate-of-cancers-among-female-patients-in-england
above-ground-biomass-in-forest-per-hectare
abs-change-energy-consumption
absolute-number-of-deaths-from-ambient-particulate-air-pollution
absolute-number-of-deaths-from-household-air-pollution
absolute-number-of-deaths-from-outdoor-air-pollution
access-drinking-water-stacked
accessibility-of-statins-in-the-public-health-sector
access-to-basic-services
access-to-clean-fuels-and-technologies-for-cooking
access-to-clean-fuels-and-technologies-for-cooking-vs-per-capita-energy-consumption
access-to-clean-fuels-for-cooking-vs-gdp-per-capita
access-to-electricity-urban-vs-rural
access-to-electricity-vs-gdp-per-capita
access-to-sexual-health-care-and-education
adequacy-of-social-insurance-programs
adolescent-fertility
adolescent-fertility-15-19
adoption-of-new-icd-guidelines-who-mortality-database
adult-population-covered-in-primary-data-on-the-prevalence-of-major-depression
adult-population-covered-in-primary-data-on-the-prevalence-of-mental-illnesses
african-elephant-carcass-ratio
african-elephants
age-dependency-breakdown
age-dependency-ratio-of-working-age-population
age-dependency-ratio-old
age-dependency-ratio-projected-to-2100
age-dependency-ratio-young-of-working-age-population
age-group-with-the-largest-population
age-of-mothers-at-childbirth-by-year
age-standardized-death-rate-by-cause
age-standardized-deaths-from-all-causes
age-when-first-had-anxiety-depression
air-conditioners-projections
air-pollutant-emissions
air-pollution-deaths-by-age
air-pollution-deaths-country
air-pollution-london-vs-delhi
air-pollution-vs-gdp-per-capita
ambient-pollution-death-rates-2017-1990
anemia-risk-groups
annual-afforestation
annual-area-burnt-by-wildfires
annual-area-burnt-by-wildfires-gwis
annual-area-burnt-per-wildfire
annual-area-burnt-per-wildfire-vs-number-of-fires
annual-burned-area-by-landcover
annual-carbon-dioxide-emissions
annual-change-coal
annual-change-forest-area
annual-change-fossil-fuels
annual-change-gas
annual-change-hydro
annual-change-in-solar-and-wind-energy-generation
annual-change-low-carbon-energy
annual-change-nuclear
annual-change-oil
annual-change-primary-energy-source
annual-change-renewables
annual-change-solar
annual-change-wind
annual-co2-coal
annual-co2-emissions-per-country
annual-co2-gas
annual-co2-oil
annual-death-rate-by-age-group
annual-death-rates-in-different-age-groups
annual-death-rates-in-different-age-groups-by-sex
annual-death-rates-in-different-age-groups-females
annual-death-rates-in-different-age-groups-males
annual-deaths-by-age
annual-deaths-by-age-england-and-wales
annual-deaths-from-outdoor-air-pollution-by-region
annual-deforestation
annual-fish-catch-taxa
annual-forest-expansion
annual-growth-rate-of-urban-population-share
annual-healthcare-expenditure-per-capita
annual-mortality-rate-from-seasonal-influenza-ages-65
annual-number-of-births-by-world-region
annual-number-of-deaths-by-cause
annual-number-of-deaths-by-world-region
annual-number-of-dhs-surveys-by-type
annual-number-of-fires
annual-percentage-change-coal
annual-percentage-change-fossil-fuels
annual-percentage-change-gas
annual-percentage-change-hydro
annual-percentage-change-in-solar-and-wind-energy-generation
annual-percentage-change-low-carbon
annual-percentage-change-nuclear
annual-percentage-change-oil
annual-percentage-change-renewables
annual-percentage-change-solar
annual-percentage-change-wind
annual-pm25-emissions-from-wildfires
annual-population-growth
annual-primary-energy-fossil-vs-low-carbon
annual-rate-of-change-in-urban-populations
annual-temperature-anomalies
anorexia-bulimia-nervosa-estimated-cases
antarctica-sea-ice-extent
anthromes
antibiotic-consumption-rate
antibiotic-usage-by-surveillance-category-stacked
antibiotic-usage-by-surveillance-category-time
antibiotic-usage-in-children
antibiotic-usage-in-livestock
antibiotic-use-in-livestock-in-europe
antibiotic-use-intensity-in-livestock-by-region
antibiotic-use-livestock-marimekko
antibiotic-use-livestock-per-kg
antibiotic-use-livestock-tonnes
antidepressants-per-capita
antidepressants-per-capita-vs-gdp
antiretroviral-therapy-coverage-among-people-living-with-hiv
anxiety-disorders-prevalence
anxiety-disorders-prevalence-by-age
anxiety-disorders-prevalence-males-vs-females
anxiety-disorders-prevalence-vs-gdp
anxiety-disorders-treatment-gap
aquaculture-farmed-fish-production
arctic-sea-ice
area-burned-wildfires-by-type
asthma-prevalence
automated-teller-machines-atms-per-100000-adults
autopsy-rate
autumn-temperature-anomalies
availability-of-computed-tomography-ct-imaging
availability-of-positron-emission-tomography-pet-imaging
average-age-of-mothers-birth-cohort
average-battery-cell-price
average-cost-of-tuberculosis-treatment-by-type
average-effective-retirement-men
average-exposure-pm25-pollution
average-monthly-surface-temperature
average-non-hdl-cholesterol-levels
avertable-deaths-from-rotavirus-with-full-vaccine-coverage
aviation-accidents-by-flight-phase
aviation-demand-efficiency
aviation-fatalities-by-flight-phase
bank-capital-to-total-assets-ratio
bathing-sites-with-excellent-water-quality
bcg-immunization-coverage-for-tb-among-1-year-olds
bev-share-new-ev
biofuel-production
biofuels-production-by-region
biomass-fish-stocks-region
biomass-fish-stocks-taxa
biomass-vs-abundance-taxa
bipolar-disorder-prevalence
bipolar-disorder-prevalence-by-age
bipolar-disorder-prevalence-males-vs-females
bird-populations-eu
birth-rate-colored-by-peak-birth-month
birth-rate-total
birth-rate-vs-death-rate
births-and-deaths
births-and-deaths-projected-to-2100
births-by-age-of-mother
births-per-month-annual
births-registered
black-rhinos
bladder-cancer-death-rate
bleaching-events-enso
bottom-trawling
brain-and-nervous-system-cancer-death-rate
breakdown-of-child-mortality-by-cause
breast-cancer-death-rate-in-women
breast-cancer-screening-coverage-in-europe
broad-causes-of-death
budget-to-manage-invasive-alien-species
burden-disease-from-each-mental-illness
burden-of-disease
burden-of-disease-by-cause
burden-of-disease-from-injuries-by-age
burden-of-disease-rates-from-communicable-neonatal-maternal-nutritional-diseases
burden-of-disease-rates-from-injuries
burden-of-disease-rates-from-ncds
burden-of-disease-who
cancer-crude-death-rate-by-type
cancer-death-rate-by-age-group
cancer-death-rate-crude-vs-age-standardized-who-mdb
cancer-death-rate-in-children-under-5-years-old-by-type
cancer-death-rates
cancer-death-rates-by-age
cancer-death-rates-by-type
cancer-death-rates-by-type-who-db
cancer-death-rates-in-children-under-10-years-old
cancer-death-rate-who-mdb
cancer-deaths-by-age
cancer-deaths-by-type-grouped
cancer-deaths-in-children-under-5-years-old-by-type
cancer-deaths-rate-and-age-standardized-rate-index
cancer-incidence
cancer-incidence-rate-by-age-group
cancer-screening-rates-in-europe
capital-city-population
capture-and-aquaculture-production
capture-fisheries-vs-aquaculture
capture-fishery-production
carbon-dioxide-emissions-factor
carbon-intensity-electricity
cardiac-magnetic-resonance-imaging-mri-availability
cardiac-procedures-per-million-people
cardiac-surgeons-per-million-people
cardiovascular-death-rate-vs-gdp-per-capita
cardiovascular-disease-death-crude-rate-who-mdb
cardiovascular-disease-death-rate-age-group-who-mdb
cardiovascular-disease-death-rate-crude-versus-age-standardized
cardiovascular-disease-death-rate-males-vs-females
cardiovascular-disease-death-rate-over-time-males-vs-females
cardiovascular-disease-death-rates
cardiovascular-disease-death-rates-by-age
cardiovascular-disease-death-rate-who-mdb
cardiovascular-disease-deaths-by-age
cardiovascular-diseases-deaths-by-risk-factor
caribbean-acropora
car-sales
cases-and-deaths-from-vaccine-preventable-diseases
cases-of-extensively-drug-resistant-tuberculosis
cases-of-multidrug-resistant-tuberculosis
catch-fishing-power-uk
cause-of-death-registration-population
causes-of-death-in-15-49-year-olds
causes-of-death-in-50-69-year-olds
causes-of-death-in-5-14-year-olds
causes-of-death-in-70-year-olds
causes-of-death-in-children-under-5
cervical-cancer-death-rate-in-women
cervical-cancer-screening-coverage-in-europe
change-air-pollutant-emissions
change-energy-consumption
change-energy-gdp-per-capita
change-forest-area-share-total
change-forest-vs-gdp
change-heat-death-rate
change-heat-deaths-gdp
change-in-total-mangrove-area
change-marine-mammals
changes-uk-butterfly
child-and-maternal-malnutrition-death-rate
child-death-change
child-death-rates-by-cause
child-deaths-by-source
child-deaths-by-world-region-un-data
child-deaths-from-rotavirus
child-deaths-from-streptococcus-by-disease
child-deaths-igme-data
child-deaths-life-stage
childhood-deaths-from-the-five-most-lethal-infectious-diseases-worldwide
child-maternal-malnutrition-gdp
child-mortality
child-mortality-around-the-world
child-mortality-by-income-level-of-country
child-mortality-by-sex
child-mortality-by-the-type-of-congenital-birth-defect
child-mortality-from-ari-vs-treatment
child-mortality-gdp-per-capita
child-mortality-igme
child-mortality-line-chart-by-sex
child-mortality-rate-ihme
child-mortality-rate-including-un-projections-through-2100
child-mortality-rate-marimekko
child-mortality-rate-vs-mean-male-height-cm
child-mortality-respiratory-infections-vs-clean-fuels
child-mortality-under-five-death-rate-by-wealth-group
child-mortality-vs-co2-emissions-per-capita
child-mortality-vs-electoral-democracy
child-mortality-vs-health-expenditure
child-mortality-vs-level-of-prosperity-regression
child-mortality-vs-population-growth
child-mortality-vs-share-of-children-immunized-against-diphtheria-pertussis-and-tetanus
children-born-per-woman
children-died-before-five-per-woman
children-per-woman-fertility-rate-vs-level-of-prosperity
children-per-woman-un
children-per-woman-vs-human-development-index
children-per-woman-vs-population-growth
children-treated-ari
children-under-age-5
children-woman-death-vs-survival
chlorophyll-a-deviation-from-the-global-average
chronic-respiratory-diseases-death-rate-who-mdb
city-populations-to-2035
co2-deforestation-for-food
co2-emissions-by-fuel-line
co2-gdp-pop-growth
co2-long-term-concentration
co2-per-capita-vs-renewable-electricity
coal-by-end-user-uk
coal-consumption-by-country-terawatt-hours-twh
coal-consumption-by-region
coal-consumption-per-capita
coal-electricity-per-capita
coal-energy-share
coal-output-per-worker-in-the-united-kingdom
coal-output-uk-tonnes
coal-phase-out-timeline
coal-prices
coal-prod-per-capita
coal-production-by-country
coal-production-by-region
coal-production-country
coal-production-per-capita-over-the-long-term
coal-proved-reserves
coal-reserves-by-type
coal-uk-opencast-deep-mine
coal-vs-gdp-per-capita
cobalt-production
co-emissions-per-capita
co-emissions-per-capita-vs-fossil-fuel-consumption-per-capita
cohort-fertility-rate
cohort-fertility-rate-by-age
colon-and-rectum-cancer-death-rate
colon-and-rectum-cancer-screening-coverage-in-europe
comfort-speaking-vs-knowing-friends-family-with-anxiety-depression
commodity-deforestation-by-region
commodity-driven-deforestation
community-health-workers
comparison-of-fertility-rates
comparison-of-world-population-projections
confidence-in-the-effectiveness-of-vaccines-vs-perception-of-importance
confirmed-tb-cases-vs-rifampicin-resistant-tb-cases
congenital-birth-defect-mortality-rates-in-children-under-5
consumption-based-energy-per-capita
consumption-energy-intensity
consumption-of-oral-antibiotics-by-type
consumption-subsidies-for-fossil-fuels
contraceptive-prevalence-any-methods-of-women-ages-15-49
contraceptive-prevalence-any-methods-vs-modern-methods
contributions-global-temp-change
coral-bleaching-events
correlation-between-child-mortality-and-mean-years-of-schooling-for-those-aged-15-and-older
costs-of-66-different-technologies-over-time
costs-of-polio-vaccines-in-low-income-countries
countries-data-antibiotic-livestock
countries-enrolled-in-glass-surveillance-of-antimicrobial-consumption-and-resistance
countries-reporting-progress-in-multi-stakeholder-development-effectiveness-monitoring-frameworks-in-support-of-the-sdgs
countries-submit-antimicrobial-data-woah
countries-that-adopt-guarantees-for-public-access-to-information
countries-that-are-parties-to-the-nagoya-protocol
countries-to-access-and-benefit-sharing-clearing-house
countries-to-the-international-treaty-on-plant-genetic-resources
countries-using-ecosystem-based-approaches-to-manage-marine-areas
countries-with-minimum-urban-threshold
countries-with-national-adaptation-plans-for-climate-change
countries-with-national-cervical-cancer-screening-program
country-level-monthly-temperature-anomalies
country-scp-plan
coverage-by-protected-areas-of-important-sites-for-mountain-biodiversity
coverage-of-the-human-papillomavirus-vaccine
covid-people-vaccinated-marimekko
covid-vaccinations-vs-covid-death-rate
covid-vaccine-share-boosters
covid-world-unvaccinated-people
cross-country-literacy-rates
crude-birth-rate
crude-death-rate
crude-death-rate-hmd
crude-death-rate-the-share-of-the-population-that-dies-each-year-gbd
crude-death-rate-the-share-of-the-population-that-dies-each-year-who
crude-death-rate-the-share-of-the-population-that-dies-per-year
crude-oil-prices
crude-oil-spot-prices
cumulative-area-burnt-by-wildfires-by-week
cumulative-area-burnt-by-wildfires-in-2024
cumulative-co-emissions-released-by-wildfires-by-week
cumulative-co-emissions-released-by-wildfires-in-2024
cumulative-cohort-fertility-rate-by-age
cumulative-covid-vaccinations-income-group
cumulative-installed-wind-energy-capacity-gigawatts
cumulative-lives-saved-by-measles-vaccination-since-1974
cumulative-lives-saved-vaccination
cumulative-number-of-dhs-surveys-conducted
daily-cases-covid-region
daily-mean-income
daily-new-confirmed-covid-19-deaths-in-sweden-oct-2020
daily-tests-and-daily-new-confirmed-covid-cases
daly-rates-from-all-causes-by-age
dalys-particulate-matter
dalys-rate-from-all-causes
dalys-to-ncds-vs-gdp
date-of-the-peak-cherry-tree-blossom-in-kyoto
dealing-with-anxiety-depression-approaches
dealing-with-anxiety-depression-comparison
dealt-with-anxiety-depression-friends-family
dealt-with-anxiety-depression-religious-spiritual
dealt-with-anxiety-depression-took-prescribed-medication
death-rate-ambient-air-pollution
death-rate-amphetamine
death-rate-amphetamine-who
death-rate-by-age-group-in-england-and-wales
death-rate-by-cause-who-mdb
death-rate-by-source-from-air-pollution
death-rate-by-source-from-indoor-air-pollution
death-rate-from-air-pollution-per-100000
death-rate-from-alzheimers-other-dementias-gbd
death-rate-from-alzheimers-other-dementias-ghe
death-rate-from-ambient-particulate-air-pollution
death-rate-from-cancer
death-rate-from-cancer-by-sex
death-rate-from-cancer-for-15-to-49-year-olds
death-rate-from-cancer-gbd
death-rate-from-cancers-vs-average-income
death-rate-from-cancers-vs-income-inequality
death-rate-from-cardiomyopathy-myocarditis-endocarditis-who-ghe-age-standardized
death-rate-from-cardiovascular-disease-age-standardized-ghe
death-rate-from-cardiovascular-disease-for-15--to-49-year-olds
death-rate-from-cardiovascular-disease-ghe
death-rate-from-cardiovascular-diseases
death-rate-from-cardiovascular-diseases-age-group-who-mdb
death-rate-from-cardiovascular-diseases-by-risk-factor-age-standardized
death-rate-from-communicable-vs-non-communicable-diseases
death-rate-from-diabetes
death-rate-from-diabetes-gbd
death-rate-from-diabetes-ghe
death-rate-from-diabetes-who-mdb
death-rate-from-digestive-diseases
death-rate-from-digestive-diseases-gbd
death-rate-from-digestive-diseases-who-mdb
death-rate-from-drowning
death-rate-from-drowning-ghe
death-rate-from-drug-use-disorders-for-15--to-49-year-olds
death-rate-from-falls
death-rate-from-falls-gbd
death-rate-from-falls-gbd-rate
death-rate-from-fires-and-burns-ihme
death-rate-from-fires-and-burns-who
death-rate-from-hypertensive-heart-disease-who-ghe-age-standardized
death-rate-from-indoor-air-pollution-vs-per-capita-energy-consumption
death-rate-from-ischaemic-heart-disease-who-ghe-age-standardized
death-rate-from-liver-disease
death-rate-from-liver-disease-gbd
death-rate-from-malnutrition
death-rate-from-malnutrition-ghe
death-rate-from-maternal-disorders
death-rate-from-maternal-disorders-ihme
death-rate-from-meningitis
death-rate-from-meningitis-gbd
death-rate-from-meningitis-who
death-rate-from-mental-health-and-substance-use-disorders-who
death-rate-from-obesity-gbd
death-rate-from-opioid-use
death-rate-from-opioid-use-who
death-rate-from-ozone-pollution
death-rate-from-ozone-pollution-gbd
death-rate-from-parkinsons-disease
death-rate-from-parkinsons-disease-gbd
death-rate-from-parkinsons-disease-ghe
death-rate-from-pm25-vs-pm25-concentration
death-rate-from-pneumonia-gbd
death-rate-from-pneumonia-ghe
death-rate-from-poisoning
death-rate-from-poisoning-gbd
death-rate-from-poisoning-ghe
death-rate-from-poisonings
death-rate-from-respiratory-infections
death-rate-from-respiratory-infections-gbd
death-rate-from-rheumatic-heart-disease-who-ghe-age-standardized
death-rate-from-road-accidents
death-rate-from-road-accidents-for-15--to-49-year-olds
death-rate-from-road-accidents-ghe
death-rate-from-secondhand-smoke
death-rate-from-secondhand-smoke-gbd
death-rate-from-smoking
death-rate-from-snakebite-envenoming
death-rate-from-strokes
death-rate-from-strokes-who
death-rate-from-suicides
death-rate-from-suicides-ghe
death-rate-from-suicides-gho
death-rate-from-unsafe-sanitation
death-rate-from-unsafe-sanitation-gbd
death-rate-from-venomous-animal
death-rate-household-air-pollution
death-rate-household-and-ambient-air-pollution
death-rate-indoor-pollution-1990-2017
death-rate-road-traffic-injuries
death-rates-cocaine
death-rates-cocaine-overdoses-by-age
death-rates-cocaine-who
death-rates-due-to-low-physical-activity
death-rates-due-to-low-physical-activity-gbd
death-rates-from-air-pollution
death-rates-from-drug-use
death-rates-from-drug-use-disorders
death-rates-from-drug-use-disorders-gbd
death-rates-from-drug-use-disorders-who
death-rates-from-drug-use-gbd
death-rates-from-energy-production-per-twh
death-rates-from-high-blood-pressure
death-rates-from-high-blood-pressure-gbd
death-rates-from-high-blood-sugar
death-rates-from-high-blood-sugar-gbd
death-rates-from-indoor-air-pollution-gbd
death-rates-from-indoor-air-pollution-vs-gdp-per-capita
death-rates-from-indoor-air-pollution-vs-share-of-population-in-absolute-poverty
death-rates-from-mental-and-substance-disorders-by-age-who
death-rates-from-natural-disasters
death-rates-from-no-access-to-handwashing-facilities
death-rates-from-pneumonia-and-other-lower-respiratory-infections-vs-gdp-per-capita
death-rates-from-unsafe-sex
death-rates-from-unsafe-sex-gbd
death-rates-from-unsafe-water-sources-gbd
death-rates-opioid-overdoses-by-age
death-rates-road-incidents
death-rates-smoking-age
death-rates-through-the-20th-century
death-rates-unsafe-water
death-rates-unsafe-water-vs-gdp
death-rate-upper-respiratory-age
deaths-age-group-stacked-bar
deaths-amphetamine-overdoses-by-sex
deaths-and-disappearances-during-migration
deaths-attributed-to-household-air-pollution-vs-deaths-attributed-to-ambient-air-pollution
deaths-averted-due-to-rotavirus-vaccine
deaths-by-age-group
deaths-by-age-group-stacked
deaths-by-risk-factor-over-70
deaths-by-risk-under5s
deaths-caused-by-vaccine-preventable-diseases-over-time
deaths-cocaine-overdoses-by-sex
deaths-drug-overdoses
deaths-drug-overdoses-who
deaths-drug-use-disorders-by-sex
deaths-due-to-drug-use
deaths-due-to-high-blood-pressure
deaths-due-to-high-blood-sugar
deaths-due-to-illicit-drug-use-by-age
deaths-due-to-low-physical-activity
deaths-due-to-measles-by-region
deaths-due-to-measles-gbd
deaths-due-to-obesity
deaths-due-to-unsafe-sanitation
deaths-due-to-unsafe-sex
deaths-due-to-unsafe-water-sources
deaths-from-acute-hepatitis-by-age
deaths-from-acute-hepatitis-by-cause
deaths-from-air-pollution
deaths-from-alzheimers
deaths-from-alzheimers-other-dementias
deaths-from-cancer-gbd
deaths-from-cancer-who
deaths-from-cardiovascular-disease
deaths-from-cardiovascular-disease-ghe
deaths-from-cardiovascular-diseases-by-region
deaths-from-cardiovascular-disease-type
deaths-from-cysticercosis
deaths-from-diabetes
deaths-from-diabetes-by-age
deaths-from-diabetes-by-type
deaths-from-diabetes-ghe
deaths-from-digestive-diseases
deaths-from-drowning
deaths-from-drowning-ghe
deaths-from-drug-use-disorders
deaths-from-drug-use-disorders-by-age
deaths-from-drug-use-disorders-by-age-who
deaths-from-drug-use-disorders-ghe
deaths-from-encephalitis-by-age
deaths-from-falls
deaths-from-falls-gbd
deaths-from-fires-and-burns
deaths-from-fires-and-burns-who
deaths-from-household-and-outdoor-air-pollution
deaths-from-indoor-air-pollution-by-age
deaths-from-infectious-diseases
deaths-from-liver-disease
deaths-from-lower-respiratory-infections-ghe
deaths-from-malnutrition
deaths-from-malnutrition-ghe
deaths-from-measles-by-age
deaths-from-meningitis
deaths-from-meningitis-by-age
deaths-from-meningitis-ghe
deaths-from-mental-health-and-substance-use-disorders-who
deaths-from-natural-disasters
deaths-from-obesity-by-age
deaths-from-ozone-pollution
deaths-from-parkinsons-disease
deaths-from-parkinsons-disease-ghe
deaths-from-pneumonia
deaths-from-poisoning
deaths-from-poisonings-ghe
deaths-from-preterm-birth-complications
deaths-from-rabies-by-age
deaths-from-respiratory-disease
deaths-from-respiratory-infections
deaths-from-road-injuries
deaths-from-smallpox-in-london
deaths-from-smallpox-per-1000-population
deaths-from-snakebite-envenoming
deaths-from-strokes
deaths-from-strokes-ghe
deaths-from-substance-use-disorders
deaths-from-substance-use-disorders-who
deaths-from-syphilis-by-age
deaths-from-tobacco-smoking-by-age
deaths-from-venomous-animal-contact
deaths-from-yellow-fever-by-age
deaths-illicit-drugs
deaths-opioid-overdoses-by-sex
deaths-registered-with-an-ill-defined-cause-of-death
deaths-risk-factor-15-49years
deaths-risk-factor-50-69years
deaths-risk-factor-5-14years
deaths-substance-disorders
deaths-temperature-gasparrini
decadal-average-surface-temperature
decadal-temperature-anomaly
decade-in-which-smallpox-ceased-to-be-endemic-by-country
deforestation-by-commodity
deforestation-co2-trade-by-product
deforestation-drivers-indonesia
deforestation-share-forest-area
dementia-death-rates
dementia-related-deaths-by-age
demographic-and-health-surveys-that-were-ongoing-before-termination
demographic-transition-england-wales
demographic-transition-sweden
dengue-fever-deaths
dengue-incidence
dentistry-personnel-per-1000
depressive-disorders-prevalence-by-age
depressive-disorders-prevalence-ihme
depressive-disorders-prevalence-ihme-vs-who
depressive-disorders-prevalence-males-vs-females
depressive-disorders-prevalence-vs-gdp-per-capita
depressive-symptoms-across-us-population
dhs-is-the-only-source-of-maternal-mortality-data
diabetes-prevalence
diabetes-prevalence-vs-gdp-per-capita
diabetes-prevalence-who-gho
diamond-production
diarrheal-disease-death-rates-in-children-under-5
diarrheal-diseases-vs-handwashing-facilities
difference-in-female-and-male-life-expectancy-at-age-45
difference-in-female-and-male-life-expectancy-at-birth
difference-in-life-expectancy-between-females-and-males-vs-child-mortality
difference-in-life-expectancy-between-females-and-males-vs-gdp-per-capita
difference-in-male-and-female-life-expectancy
diphtheria-cases-in-the-united-states
diphtheria-deaths
diphtheria-tetanus-pertussis-vaccine-vs-pneumococcal-vaccine-coverage
discomfort-speaking-anxiety-depression
discomfort-speaking-anxiety-depression-by-gender
disease-burden-by-age
disease-burden-by-region
disease-burden-by-risk-factor
disease-burden-from-communicable-diseases
disease-burden-from-communicable-diseases-by-age
disease-burden-from-injuries
disease-burden-from-ncds
disease-burden-from-ncds-by-age
disease-burden-rates-by-cancer-types
disease-burden-rates-from-cancers
disease-burden-to-communicable-diseases-vs-gdp
disease-burden-vs-health-expenditure-per-capita
domestic-forest-change-vs-imported-deforestation
drinking-water-service-coverage
drinking-water-services-coverage-rural
drivers-forest-loss-brazil-amazon
drivers-of-recovery-in-european-bird-populations
drowning-death-rates
drowning-death-rates-under-5
drowning-deaths-by-age-group
drug-disorder-dalys-rate
drug-overdose-death-rates
drug-resistant-confirmed-cases-of-tuberculosis
dtp3-children-unvaccinated
eating-disorders-prevalence
eating-disorders-prevalence-by-age
eating-disorders-prevalence-males-vs-females
effective-fertility-rate-children-per-woman-who-are-expected-to-survive-until-childbearing-age
effective-fertility-rates-malani-and-jacob
efficiency-of-lighting-in-the-united-kingdom-lumen-hours-per-kilowatt-hour
elec-fossil-nuclear-renewables
elec-mix-bar
electric-car-sales
electric-car-sales-share
electric-car-stocks
electricity-air-conditioning
electricity-as-a-share-of-primary-energy
electricity-coal
electricity-demand
electricity-fossil-fuels
electricity-fossil-renewables-nuclear-line
electricity-gas
electricity-generation
electricity-generation-from-solar-and-wind-compared-to-coal
electricity-imports-share-demand
electricity-mix-uk
electricity-oil
electricity-prod-source-stacked
electricity-production-by-source
electricity-renewables
electricity-source-wb-stacked
emissions-of-air-pollutants
emissions-of-air-pollutants-oecd
emissions-of-particulate-matter
employed-fisheries-aquaculture
employed-fisheries-aquaculture-time
employment-in-agri-vs-urban-pop
employment-in-the-coal-industry-in-the-united-kingdom
endemic-amphibian-species-by-country
endemic-bird-species-by-country
endemic-freshwater-crab-species
endemic-mammal-species-by-country
endemic-reef-forming-coral-species
endemic-shark-and-ray-species
energy-consumption-by-source-and-country
energy-embedded-traded-goods
energy-imports-and-exports-energy-use
energy-intensity
energy-intensity-by-sector
energy-intensity-of-economies
energy-intensity-vs-gdp
energy-mineral-prices
energy-use-gdp-decoupling
energy-use-per-capita-vs-co2-emissions-per-capita
energy-use-per-capita-vs-gdp-per-capita
energy-use-per-person-vs-gdp-per-capita
england-covid-19-mortality-rate-by-vaccination-status
englands-economy-over-the-long-run-gdp-per-capita-vs-population
englands-economy-over-the-long-run-gdp-vs-population
estimated-prevalence-vs-burden-mental-illnesses
excess-deaths-cumulative-economist-single-entity
excess-deaths-cumulative-economist-who
excess-deaths-cumulative-who
excess-deaths-daily-economist-single-entity
exclusive-breastfeeding-rate
expected-years-of-living-with-disability-or-disease-burden
expenditure-on-cultural-and-natural-heritage-per-capita
experienced-calmness-a-lot-of-the-day-yesterday-by-age
experienced-calmness-a-lot-of-the-day-yesterday-by-gender
exported-deforestation
extensively-drug-resistant-tuberculosis
extinct-species-since-1500
extremely-important-fund-research-anxiety-depression-vs-gdp-per-capita
extreme-poverty-headcount-ratio-vs-life-expectancy-at-birth
extreme-poverty-vs-child-mortality
facilitate-orderly-safe-migration
fatal-occupational-injuries-among-employees
female-and-male-life-expectancy-at-birth-in-years
female-labor-force-participation-rates-by-national-per-capita-income
female-minus-male-life-expectancy-vs-non-communicable-disease-death-rates
female-population-by-age-group
female-to-male-life-expectancy-ratio
fertility-and-female-labor-force-participation
fertility-and-wanted-fertility
fertility-rate-by-age-group
fertility-rate-vs-mean-years-of-schooling
fertility-rate-vs-share-of-women-between-25-and-29-years-old-with-no-education
fertility-rate-vs-the-share-living-in-extreme-poverty
fertility-rate-with-projections
fertility-vs-child-mortality
fertility-vs-contraception
fertility-vs-unmet-contraception
financial-contributions-toward-polio-eradication
fire-death-rates
fire-deaths-by-age
fish-catch-by-fishing-sector
fish-catch-gear-type
fish-catch-region
fish-catch-uk
fish-discard-rates-gear
fish-discards
fish-discards-gear
fishing-pressure-by-region
fishing-pressure-by-taxa
fish-landings-and-discards
fish-seafood-production
fish-species-threatened
fish-stocks-by-region
fish-stocks-taxa
fish-stocks-within-sustainable-levels
five-year-net-cancer-survival-rates-by-sex
five-year-survival-rate-of-cancers-among-men-in-england
five-year-survival-rates-by-cancer-type
flu-vaccines-older-people-recommendation
forest-area-as-share-of-land-area
forest-area-km
forest-area-net-change-rate
forest-area-primary-planted
forest-transition-phase
fossil-electricity-per-capita
fossil-fuel-cons-per-capita
fossil-fuel-consumption-by-fuel-type
fossil-fuel-consumption-by-type
fossil-fuel-consumption-per-capita
fossil-fuel-price-index
fossil-fuel-primary-energy
fossil-fuel-production
fossil-fuel-production-over-the-long-term
fossil-fuel-production-per-capita
fossil-fuels-per-capita
fossil-fuels-share-energy
fossil-fuel-subsidies
fossil-fuel-subsidies-gdp
fossil-fuel-subsidies-per-capita
fraction-of-countries-with-available-data-on-deaths-from-suicides-by-income-group
freshwater-use-seafood
friends-family-anxious-depressed
future-life-expectancy-projections
gamma-camera-or-nuclear-medicine-availability
gas-consumption-by-country
gas-electricity-per-capita
gas-prod-per-capita
gas-production-by-country
gas-share-energy
gavi-supported-countries
gdp-per-capita-maddison-project-database
gender-gap-in-life-expectancy-by-age-group
gender-ratio-in-life-expectancy-at-birth
ghg-emissions-seafood
gini-coefficient-of-lifespan-inequality-by-sex
gini-coefficient-of-lifespan-inequality-in-females
gini-coefficient-of-lifespan-inequality-in-males
global-alien-species-first-record-database
global-antimicrobial-resistance-and-use-surveillance-system-glass-by-pathogen
global-aquaculture-wild-fish-feed
global-case-numbers-of-infectious-diseases
global-change-over-the-last-50-years
global-child-deaths-by-cause
global-child-mortality-timeseries
global-co2-concentration
global-deaths-by-syndrome-and-antimicrobial-resistance
global-deaths-from-infections-attributable-to-antimicrobial-resistance
global-energy-consumption-source
global-energy-substitution
global-first-records-of-established-alien-species
global-forestry-area-1958-2014
global-fossil-fuel-consumption
global-hydro-consumption
global-living-planet-index
global-malaria-deaths-by-world-region
global-marine-protected-area
global-methane-concentrations
global-nitrous-oxide-concentration
global-number-of-missing-women
global-primary-energy
global-primary-energy-share-inc-biomass
global-smallpox-cases
global-temperature-anomalies-by-el-nino-la-nina
global-temperature-anomalies-by-el-nino-la-nina-and-month
global-temperature-anomalies-by-month
global-vaccination-coverage
global-warming-by-gas-and-source
global-warming-fossil
global-whale-biomass
gold-production
good-cholesterol-levels-age-standardized
govt-expenditure-share-budget
graphite-production
green-climate-gcf-fund-pledges
h5n1-flu-reported-cases
habitat-loss-25-species
haemophilus-influenzae-b-vaccine-immunization-schedule
haq-by-level-of-healthcare-spending-endpoints
healthcare-access-quality-un
healthcare-expenditure-vs-gdp
health-coverage-vs-gdp-per-capita-simple
health-expenditure-and-financing-per-capita
health-expenditure-government-expenditure
health-insurance-coverage-in-the-us
health-protection-coverage
health-provider-absence-rate
health-spending-per-person-by-source
healthy-life-expectancy-and-years-lived-with-disability
healthy-life-expectancy-at-birth
healthy-life-expectancy-vs-health-expenditure-per-capita
hearing-loss-by-age-group
heat-death-rate-vs-co2
heat-deaths-vs-projected-death-rates
hepatitis-b-birth-dose-vaccine-immunization-schedule
hepatitis-b-deaths-by-type-of-disease
hepatitis-b-incidence-sdgs
hepatitis-b-number-of-deaths
hepatitis-b-total-number-of-deaths-by-age-group
hepatitis-b-vaccine-immunization-schedule
hepatitis-c-deaths-by-type-of-disease
hepatitis-c-number-of-deaths
hepatitis-c-total-number-of-deaths-by-age-group
hepatitis-death-rate
hepb3-number-unvaccinated
hib3-number-unvaccinated
hib-vaccine
historical-death-rates-from-each-cause-category-in-france
historical-share-of-population-living-on-less-than-5-per-day
historical-share-of-population-living-on-less-than-per-day2
hivaids-deaths-and-averted-due-to-art
hiv-funding-and-future-requirements
hiv-funding-by-region
hospital-beds-per-1000-people
household-air-pollution-deaths-by-region
households-air-conditioning
household-vs-outdoor-air-pollution-death-rates
how-much-we-think-we-spend-on-healthcare-vs-how-much-we-actually-do
hubberts-peak-vs-actual-oil-production-in-the-united-states
human-papillomavirus-vaccine-immunization-schedule
human-trafficking-victims
hydro-consumption-by-region
hydro-electricity-per-capita
hydropower-consumption
hydro-share-energy
hypertension-adults-30-79
hypothetical-number-of-deaths-from-energy-production
icd-code-version-used-to-classify-causes-of-death
ice-sheet-mass-balance
ihr-core-capacity-index-sdgs
immunization-coverage-against-diphtheria-tetanus-and-pertussis-dtp3-vs-gdp-per-capita
immunization-hepb3-of-one-year-old-children
immunization-measles-vs-vaccine-safety
importance-government-funding-research-anxiety-depression
importance-science-treating-anxiety-depression
imported-deforestation
imports-of-ppe-related-products
imports-of-ppe-related-products-per-capita
improved-sanitation-facilities-vs-gdp-per-capita
improved-water-sources-vs-gdp-per-capita
inactivated-poliovirus-vaccine-immunization-schedule
incidence-of-meningitis
incidence-of-tuberculosis
incidence-of-venomous-animal-contact
incidence-rate-of-cardiovascular-disease
indian-rhinos
indoor-pollution-death-rates-clean-fuels
indoor-pollution-deaths-1990-2017
inequality-in-life-expectancy-vs-health-expenditure-per-capita
infant-death-rates-by-cause-by-sex
infant-deaths
infant-mortality
infant-mortality-by-sex
infant-mortality-rate
infant-mortality-rates
infant-mortality-rates-uk
infant-mortality-rate-wdi
infant-mortality-vaccines
infant-mortality-vs-prenatal-care
infants-pre-term-share
infectious-and-parasitic-diseases-death-rate-who-mdb
infectious-disease-death-rates
infectious-disease-death-rates-gbd
influenza-vaccination-rate-elderly-oecd
influenza-vaccination-rate-elderly-who-unicef
installed-geothermal-capacity
installed-global-renewable-energy-capacity-by-technology
installed-solar-pv-capacity
international-finance-clean-energy
interventions-for-substance-use-disorders
interventions-ntds-sdgs
investment-in-renewable-energy-by-technology
ipv1-number-unvaccinated
ipv-vaccine-coverage
is-antibiotic-use-in-livestock-above-global-guidelines
iwc-members
javan-rhinos
kaya-identity-co2
land-use-seafood
large-household-expenditures-health
large-oil-spills-decadal
laws-antimicrobials-aquatic
laws-antimicrobials-livestock
leading-broad-cause-of-death
leading-cancer-types-causing-death-in-men
leading-cancer-types-causing-death-in-women
leading-cause-of-child-mortality-among-children-under-five
leading-cause-of-death
lead-paint-over-10000ppm
lead-paint-over-600ppm
levelized-cost-of-energy
life-expectancy
life-expectancy-at-age-15
life-expectancy-at-birth-including-the-un-projections
life-expectancy-at-birth-oecd
life-expectancy-at-birth-source-comparison
life-expectancy-at-birth-who-gho
life-expectancy-at-different-ages
life-expectancy-hmd-unwpp
life-expectancy-marimekko
life-expectancy-of-women-vs-life-expectancy-of-men
life-expectancy-un-vs-gdp-per-capita-wb
life-expectancy-unwpp
life-expectancy-versus-gini-lifespan-inequality
life-expectancy-vs-expected-years-lived-with-disability
life-expectancy-vs-gdp-per-capita
life-expectancy-vs-healthcare-expenditure
life-expectancy-vs-health-expenditure
life-expectation-at-birth-by-sex
life-satisfaction-vs-child-mortality
lithium-production
liver-cancer-death-rate
liver-cancer-incidence-rate-by-risk-factor
lives-saved-vaccines
lives-saved-vaccines-region
living-planet-index-by-region
living-planet-index-comparison
long-run-air-pollution
long-run-birth-rate
long-run-methane-concentration
long-term-cod-catch
long-term-energy-transitions
long-term-urban-population-region
low-carbon-elec-per-capita
low-carbon-electricity
low-carbon-energy-consumption
low-carbon-share-energy
lung-cancer-death-rates-males
lung-cancer-deaths-men-women
lung-cancer-deaths-per-100000-by-sex-1950-2002
magnetic-resonance-imaging-mri-units-availability
major-causes-of-child-deaths
male-female-ratio-suicides-rates
male-high-blood-pressure
male-population-by-age-group
male-vs-female-suicide-who
mandatory-childhood-vaccination
marine-protected-areas
mass-us-glaciers
material-footprint-per-capita
material-footprint-per-unit-of-gdp
maternal-mortality
maternal-mortality-vs-fertility
maternal-mortality-vs-neonatal-mortality
maximum-life-expectancy-sex-female
mcv2-vaccine-immunization-schedule
measles-cases-and-death
measles-cases-and-death-rate
measles-coverage-vs-cases-worldwide
measles-number-unvaccinated
measles-prevalence-vaccine-attitudes
median-age
median-age-vs-births-per-woman
medical-doctors-per-1000-people-vs-gdp-per-capita
meningitis-death-rates-children-under-5
mens-life-expectancy-at-birth
men-survival-to-age-65
mental-and-substance-use-as-share-of-disease
mental-health-as-risk-for-drug-dependency
mental-health-units-in-general-hospitals
mental-illnesses-prevalence
mental-illness-estimated-cases
men-women-friends-family-anxious-depressed
migrant-recruitment-costs
migrant-stock-share
migrant-stock-total
mining-electricity-sources-including-gas
missing-female-births-and-excess-mortality
mobile-cellular-subscriptions-per-100-people
modal-age-at-death-in-adults
modern-contraceptive-prevalence-vs-gdp-per-capita
modern-renewable-energy-consumption
modern-renewable-prod
money-committed-to-public-private-partnerships-for-infrastructure
money-sent-or-brought-back-by-migrants-as-a-share-of-gdp
monthly-average-surface-temperatures-by-decade
monthly-average-surface-temperatures-by-year
monthly-birth-rate-per-day
monthly-ocean-heat-2000m
monthly-sea-ice-extent-in-the-antarctic
monthly-sea-ice-extent-in-the-antarctic-decadal-average
monthly-sea-ice-extent-in-the-arctic
monthly-sea-ice-extent-in-the-arctic-decadal-average
monthly-surface-temperature-anomalies-by-decade
monthly-surface-temperature-anomalies-by-year
monthly-temperature-anomalies
monthly-upper-ocean-heat
mortality-from-ncds-gender
mortality-from-ncds-sdgs
mortality-neonatal-asphyxia-and-trauma
mortality-rate-attributable-to-wash
mortality-rates-age-25-by-sex
mortality-rates-age-65-by-sex
mountain-green-cover-index
multidrug-resistant-tuberculosis-without-extensive-drug-resistance
mumps-cases-and-deaths-in-the-united-states
mumps-vaccine-immunization-schedule
national-biodiversity-strategy-align-with-aichi-target-9
nationally-determined-contributions
national-progress-towards-aichi-biodiversity-target-2
national-system-to-monitor-antimicrobials-usage-for-human-medicine
national-system-to-monitor-spread-of-antimicrobial-resistance-in-humans
natural-gas-consumption-by-region
natural-gas-prices
natural-gas-production-by-region-terawatt-hours-twh
natural-gas-proved-reserves
natural-population-growth
natural-resource-rents
neonatal-deaths-by-cause
neonatal-deaths-from-bloodstream-infections-attributed-to-antimicrobial-resistance
neonatal-deaths-from-infections-attributed-to-antimicrobial-resistance-by-syndrome
neonatal-infant-and-child-deaths
neonatal-mortality-ihme
neonatal-mortality-rate-gho
neonatal-mortality-vs-births-attended-by-staff
neonatal-mortality-wdi
neonatal-sepsis-and-infection-death-rate-in-infants
net-deforestation-in-trade
net-electricity-imports
net-enrollment-rate-pre-primary
net-oda-to-medical-research
nitrogen-emissions-seafood
nitrous-oxide-long
northern-white-rhinos
nuclear-electricity-per-capita
nuclear-energy-generation
nuclear-primary-energy
number-bev-phev-sold
number-countries-mandatory-vaccination
number-of-accessions-of-plant-genetic-resources-secured-in-conservation-facilities
number-of-asian-elephants
number-of-births-per-year
number-of-child-deaths
number-of-child-deaths-by-the-type-of-congenital-birth-defect
number-of-child-deaths-igme
number-of-child-deaths-unwpp
number-of-children-newly-infected-with-hiv
number-of-children-that-die-from-road-injuries
number-of-children-who-are-underweight
number-of-commercial-bank-branches-per-100000-adults
number-of-countries-administering-each-major-vaccine
number-of-countries-reporting-confirmed-bacterial-infections
number-of-countries-with-a-vaccine-stock-out
number-of-countries-with-primary-data-on-prevalence-of-mental-illnesses-in-the-global-burden-of-disease-study
number-of-deaths
number-of-deaths-by-age-group
number-of-deaths-by-risk-factor
number-of-deaths-from-cervical-cancer
number-of-deaths-from-diarrheal-diseases-by-risk-factor-for-under-5s
number-of-deaths-from-hivaids
number-of-deaths-from-hivaids-who
number-of-deaths-from-pneumonia-in-children-under-5
number-of-deaths-from-road-injuries
number-of-deaths-from-suicide-ghe
number-of-deaths-from-tuberculosis-ghe
number-of-deaths-gbd
number-of-deaths-per-year
number-of-deaths-who
number-of-described-species
number-of-estimated-paralytic-polio-cases-by-world-region
number-of-hospitals-with-cardiovascular-units-per-million-people
number-of-infant-deaths
number-of-infant-deaths-gho
number-of-infant-deaths-ihme
number-of-infant-deaths-unwpp
number-of-infections-tested-for-antibiotic-susceptibility
number-of-measles-cases
number-of-neonatal-deaths-by-region
number-of-neonatal-deaths-igme
number-of-neonatal-deaths-ihme
number-of-new-cases-and-deaths-from-cervical-cancer
number-of-new-hiv-child-infections-vs-number-of-infections-averted-due-to-pmtct
number-of-new-tetanus-cases-in-children-under-5
number-of-one-year-olds
number-of-parties-env-agreements
number-of-people-in-the-world-with-and-without-access-to-improved-water-sources
number-of-people-with-and-without-electricity-access
number-of-people-with-cancer
number-of-people-with-cancer-by-age
number-of-people-with-cancer-by-type
number-of-refugees-per-100000
number-of-reported-cholera-deaths
number-of-reported-road-incidents
number-of-reported-yaws-cases
number-of-rhinos-poached
number-of-tanker-spills-by-operation-at-time-of-incident
number-of-tanker-spills-by-primary-cause-of-spill
number-of-transboundary-animal-breeds-which-have-genetic-resources-secured-in-conservation-facilities
number-of-unique-settlements
number-of-youth-deaths
number-oil-spills
number-outdoor-pollution-deaths-by-age
number-species-evaluated-iucn
number-species-threatened
number-suffering-from-depression-by-region
number-suicide-deaths
number-suicide-deaths-source-comparison
number-using-at-least-basic-sanitation
number-with-autistic-spectrum-disorder
number-with-drug-disorders-by-substance
number-with-drug-use-disorders-country
number-with-mental-health-disorders-by-sex
number-without-access-to-improved-sanitation
number-without-clean-cooking-fuel
number-without-electricity-by-region
number-without-improved-water
number-without-safe-drinking-water
number-with-without-clean-cooking-fuels
nurses-and-midwives-per-1000-people
obesity-prevalence-adults-who-gho
ocean-heat-content-upper
ocean-heat-top-2000m
oil-consumption-by-country
oil-consumption-by-region-terawatt-hours-twh
oil-electricity-per-capita
oil-prices-inflation-adjusted
oil-prod-per-capita
oil-production-by-country
oil-production-by-region
oil-proved-reserves
oil-rents-as-a-share-of-gdp
oil-share-energy
one-person-households
open-defecation-in-rural-areas-vs-urban-areas
opinions-young-people-climate
outdoor-pollution-death-rate
outdoor-pollution-deaths-1990-2017
outdoor-pollution-rates-by-age
outdoor-pollution-rate-vs-gdp
out-of-pocket-expenditure-per-capita-on-healthcare
ozone-o3-concentration-in-ppb
pancreatic-cancer-death-rate
patents-ccs
patents-electric-vehicles
patents-energy-storage
patents-filed-for-renewables
patents-for-renewables-by-country
pcv3-number-unvaccinated
peak-birth-month
people-practicing-open-defecation-of-population
people-with-access-to-at-least-basic-drinking-water
people-without-clean-cooking-fuels-region
people-without-electricity-country
per-capita-co2-by-source
per-capita-co2-coal
per-capita-co2-food-deforestation
per-capita-co2-fuel
per-capita-co2-gas
per-capita-co2-oil
per-capita-consumption-based-co-emissions-vs-per-capita-energy-consumption
per-capita-electricity-demand
per-capita-electricity-fossil-nuclear-renewables
per-capita-electricity-generation
per-capita-electricity-generation-from-solar-and-wind
per-capita-electricity-generation-vs-gdp-per-capita
per-capita-electricity-source-stacked
per-capita-energy-consumption-from-solar-and-wind
per-capita-energy-source-stacked
per-capita-energy-stacked
per-capita-energy-use
per-capita-fossil-energy-vs-gdp
per-capita-gas
per-capita-hydro
per-capita-low-carbon-energy
per-capita-low-carbon-vs-gdp
per-capita-nuclear
per-capita-oil
per-capita-renewables
per-capita-solar
per-capita-wind
perceived-comfort-speaking-anxiety-depression
perceived-comfort-speaking-anxiety-depression-vs-talked-to-family-friends-when-had-anxiety-depression
percentage-change-energy-by-source
percentage-of-persons-without-health-insurance-coverage-us
perception-of-the-safety-of-vaccines-vs-vaccine-coverage
period-average-age-of-mothers
period-average-age-of-mothers-birth-order
period-total-fertility-rate
period-versus-cohort-life-expectancy
personal-remittances-oda
pertussis-cases-and-deaths-in-the-united-states
pharmaceutical-personnel-per-1000
phosphorous-emissions-seafood
physicians-per-1000-people
pm25-air-pollution
pm25-exposure-gdp
pm-exposure-1990-2017
pneumococcal-conjugate-vaccine-immunization-schedule
pneumococcal-vaccination-averted-deaths
pneumonia-and-lower-respiratory-diseases-deaths
pneumonia-careseeking
pneumonia-death-rates-age-standardized
pneumonia-death-rates-in-children-under-5
pneumonia-risk-factors
polio-number-unvaccinated
polio-vaccine-coverage-of-one-year-olds
polio-vaccine-schedule
pollution-deaths-from-fossil-fuels
population
population-by-age-group
population-by-age-group-with-projections
population-by-five-year-age-group
population-by-income-level
population-density
population-density-by-city
population-density-of-the-capital-city
population-density-vs-prosperity
population-growth-rate
population-growth-rate-by-level-of-development
population-growth-rates
population-growth-rate-vs-median-age
population-growth-rate-with-and-without-migration
population-in-the-largest-city
population-long-run-with-projections
population-marimekko
population-of-cities-towns-and-villages
population-of-england-millennium
population-regions-with-projections
population-threshold-for-urban-area
population-unwpp
population-using-at-least-basic-drinking-water
population-using-solid-fuels-for-cooking
population-with-un-projections
population-younger-than-15-with-projections
population-young-working-elderly
population-young-working-elderly-with-projections
population-young-working-old
postnatal-care
power-outages-in-firms-per-month
prevalence-by-mental-and-substance-use-disorder
prevalence-cysticercosis
prevalence-of-diarrheal-disease-vs-ors
prevalence-of-drug-use-disorders-by-age
prevalence-of-drug-use-disorders-males-vs-females
prevalence-of-hepatitis-b-surface-antigen
prevalence-of-lymphatic-filariasis
prevalence-of-mental-and-substance-disorders-by-age
prevalence-of-modern-contraception-methods-vs-womens-mean-years-of-schooling
prevalence-of-onchocerciasis
prevalence-of-tuberculosis-by-drug-susceptibility
prevalence-of-undernourishment
prevalence-rate-of-cardiovascular-disease
primary-energy-cons
primary-energy-consumption-by-region
primary-energy-consumption-by-source
primary-energy-consumption-from-solar
primary-energy-consumption-from-solar-and-wind
primary-energy-fossil-nuclear-renewables
primary-energy-hydro
primary-energy-mix-uk
primary-energy-nuclear
primary-energy-renewables
primary-energy-share-nuclear-renewables
primary-energy-source-bar
primary-energy-wind
primary-sub-energy-source
prison-population-rate
private-health-expenditure-per-person
probability-of-dying-by-age
probability-of-dying-that-year-among-females-of-a-given-age
probability-of-dying-that-year-among-males-of-a-given-age
probability-that-a-male-will-outlive-a-female
probability-that-a-newborn-will-die-in-the-first-year-of-life-by-sex
production-consumption-carbon-intensity-energy
production-consumption-energy-per-person
production-vs-consumption-energy
projected-change-dependency-ratios
projected-cropland-by-2050
projected-habitat-loss-extent-bau
projected-population-under-age-5
projections-of-the-number-of-children-per-woman-by-education-scenario
proportion-countries-migration-policies-criteria
proportion-of-animal-breeds-genetic-conservation
proportion-of-forest-area-certified-under-an-independently-verified-certification-scheme
proportion-of-forest-area-within-legally-established-protected-areas
proportion-of-forest-area-with-long-term-management-plan
proportion-of-important-sites-for-freshwater-biodiversity-covered-by-protected-areas
proportion-of-local-breeds-at-risk-of-extinction
proportion-using-safely-managed-drinking-water
proportion-with-basic-handwashing-facilities-urban-vs-rural
prostate-cancer-death-rate-in-men
protected-area-coverage-of-marine-key-biodiversity-areas
protected-terrestrial-biodiversity-sites
protection-of-the-rights-of-small-scale-fisheries
protein-energy-malnutrition-mortality-rates-in-children
psychiatrists-working-in-the-mental-health-sector
public-healthcare-spending-share-gdp
public-health-expenditure-share-gdp
public-health-insurance-coverage
public-opposition-to-nuclear-energy-production
quantity-oil-spills
quantity-oil-spills-decadal-average
rate-of-child-deaths-attributed-to-child-and-maternal-malnutrition
rate-of-disease-burden-from-drug-use-disorders-who
rate-of-new-cases-of-cancer-by-type
rate-of-new-cervical-cancer-cases
rate-of-new-cervical-cancer-cases-gco
rate-of-twin-deliveries
ratio-of-female-to-male-labor-force-participation-rates-slopes
real-commodity-price-index-metals
red-list-index
reduction-global-antibiotic-use
refugee-population-by-country-or-territory-of-asylum
refugee-population-by-country-or-territory-of-origin
region-share-tropical-deforestation
regulation-illegal-fishing
regulations-antibiotics-growth-promotion
remaining-life-expectancy-at-different-ages
remaining-life-expectancy-at-different-ages-females
remaining-life-expectancy-at-different-ages-males
remittance-costs-as-share-of-amount-remitted
renewable-electricity-generating-capacity-per-capita
renewable-electricity-per-capita
renewable-energy-consumption
renewable-energy-gen
renewable-energy-investment-of-gdp
renewable-nuclear-direct-substitution
renewable-share-energy
replacement-fertility-rate
reported-cases-of-acute-african-trypanosomiasis
reported-cases-of-human-african-trypanosomiasis-t-b-gambiense
reported-cases-of-leprosy
reported-cases-of-measles
reported-rabies-deaths
reported-rate-of-new-cervical-cancer-cases
reporting-of-mental-health-system-data-in-the-past-two-years
representation-of-women-in-the-judiciary
respiratory-disease-death-rate
respiratory-disease-death-rate-gbd
respiratory-disease-deaths-by-age
respiratory-infection-death-rate-who-mdb
respiratory-infections-death-rate-age-group
respiratory-pathogens-share-detected-seattle-flu-study
risk-of-impoverishing-expenditure-for-surgical-care
road-accident-deaths-per-passenger-kilometers
road-death-rate-vs-gdp-per-capita
road-deaths-by-type
road-deaths-in-great-britain-by-road-user
road-deaths-over-the-long-term
road-incident-deaths-by-age
road-miles-nox-emissions
road-mortality-rate-comparison
rotavirus-number-unvaccinated
rotavirus-vaccine-immunization-schedule
routes-to-diagnosis
rubella-cases-in-the-united-states
rubella-number-unvaccinated
rubella-vaccination-coverage
rubella-vaccine-immunization-schedule
rural-population-with-improved-water
rural-without-improved-sanitation
rural-without-improved-water
safe-sanitation-without
sales-antibiotics-livestock-europe
sales-of-cigarettes-per-adult-per-day
sanitation-facilities-coverage
sanitation-facilities-coverage-in-rural-areas
sanitation-facilities-coverage-in-urban-areas
schizophrenia-prevalence
schizophrenia-prevalence-by-age
schizophrenia-prevalence-males-vs-females
schools-access-to-electricity
science-helps-a-lot-treating-anxiety-depression-vs-gdp-per-capita
sea-ice-anomaly-in-the-antarctic-by-month
sea-ice-anomaly-in-the-arctic-by-month
sea-level
seasonal-temp-anomaly-us
sea-surface-temperature
sea-surface-temperature-anomaly
seawater-ph
seized-rhino-horns
self-harm-death-rates-by-age
self-reported-hopefulness
severe-bleaching-events-enso
sex-ratio-at-birth
sex-ratio-at-birth-by-birth-order
sex-ratio-at-birth-vs-five-years-old
sex-ratio-at-five-years-old
sex-ratio-at-sixty-years-old
sex-ratio-by-age
share-agrees-vaccines-are-important-wellcome
share-agree-vaccines-are-safe-wellcome
share-believe-climate
share-cancers-diagnosed-via-screening-in-england-by-stage-of-cancer
share-car-sales-battery-plugin
share-car-stocks-electric
share-co2-emissions-deforestation
share-covered-by-one-social-protection-benefit
share-deaths-air-pollution
share-deaths-cardiovascular-disease
share-deaths-indoor-pollution
share-deaths-outdoor-pollution
share-deaths-secondhand-smoke
share-deaths-suicide
share-deaths-unsafe-sanitation
share-deaths-unsafe-water
share-deforestation-agricultural-products
share-deforestation-domestic-consumption
share-deforestation-exported
share-degraded-land
share-disagrees-vaccines-are-effective
share-disagrees-vaccines-are-important
share-disagree-vaccines-are-safe-wellcome-trust
share-elec-by-source
share-electricity-coal
share-electricity-fossil-fuels
share-electricity-gas
share-electricity-hydro
share-electricity-low-carbon
share-electricity-nuclear
share-electricity-oil
share-electricity-renewables
share-electricity-solar
share-electricity-source-facet
share-electricity-source-wb
share-electricity-wind
share-energy-source-sub
share-global-deforestation
share-global-forest
share-global-forest-expansion
share-of-adults-borrowing-from-a-financial-institution-vs-family-or-friends
share-of-adults-who-smoke
share-of-all-cancer-deaths-attributable-to-alcohol-use
share-of-all-cancer-deaths-by-risk-factor
share-of-births-by-age-of-mother
share-of-cancer-deaths-attributed-to-alcohol-use-by-type
share-of-cancer-deaths-attributed-to-all-risk-factors
share-of-cancer-deaths-attributed-to-different-risk-factors
share-of-cancer-deaths-attributed-to-risk-factors
share-of-cancer-deaths-by-cancer-type
share-of-cancer-deaths-linked-to-dietary-risk-factors-by-cancer-type
share-of-cancer-deaths-linked-to-high-bmi-by-type
share-of-cancer-deaths-linked-to-tobacco-use-by-cancer-type
share-of-cancers-attributed-to-9-hpv-types
share-of-cervical-and-all-cancer-deaths-due-to-unsafe-sex
share-of-children-immunized-dtp3
share-of-children-under-fifteen-years-old-with-hiv
share-of-children-vaccinated-against-measles
share-of-children-vaccinated-against-measles-vs-share-vaccinated-against-dpt3
share-of-children-vaccinated-with-mcv2
share-of-children-who-are-stunted-in-urban-and-rural-areas
share-of-clinical-trials-that-report-results-in-time-by-country
share-of-clinical-trials-that-report-results-within-a-year-over-time
share-of-countries-region-reporting-confirmed-infections-glass
share-of-countries-testing-confirmed-bacterial-infections-for-antibiotic-susceptibility
share-of-deaths-by-cause
share-of-deaths-by-risk-factor
share-of-deaths-cause-is-registered
share-of-deaths-from-cardiovascular-diseases-men-women
share-of-deaths-from-major-causes
share-of-deaths-from-opioid-overdoses
share-of-deaths-from-underlying-causes
share-of-deaths-registered
share-of-deliveries-by-singleton-twin-or-multiple-status
share-of-disease-burden-from-communicable-diseases-vs-gdp
share-of-disease-burden-from-ncds-vs-gdp
share-of-e-coli-bloodstream-infections-due-to-antimicrobial-resistant-bacteria
share-of-electricity-production-from-renewable-sources
share-of-electricity-production-from-solar-and-wind
share-of-final-energy-consumption-from-renewable-sources
share-of-firms-for-which-access-to-finance-is-a-major-constraint
share-of-fish-stocks-overexploited
share-of-fish-stocks-that-are-sustainably-fished
share-of-global-consumption-based-co-emissions-and-population
share-of-health-facilities-with-essential-medicines
share-of-invasive-cervical-cancers-caused-by-each-hpv-type
share-of-low-birthweight-babies-born
share-of-low-carbon-energy-vs-gdp
share-of-lung-cancer-deaths-attributed-to-air-pollution
share-of-married-women-ages-15-49-years-whose-need-for-family-planning-is-satisfied
share-of-mothers-in-low--and-middle-income-countries-who-lost-a-child-under-5
share-of-mothers-receiving-at-least-four-antenatal-visits-during-pregnancy
share-of-mothers-receiving-at-least-one-antenatal-visit-during-pregnancy
share-of-mothers-who-have-lost-a-child-under-5
share-of-mothers-who-have-lost-an-infant
share-of-new-cancers-caused-by-all-known-infections-by-type
share-of-new-cancers-caused-by-all-known-infectious-agents
share-of-new-cancers-caused-by-helicobacter-pylori-bacteria
share-of-one-year-olds-who-received-the-final-dose-of-pneumococcal-vaccine
share-of-one-year-olds-who-received-the-rotavirus-vaccine
share-of-out-of-pocket-expenditure-on-healthcare
share-of-out-of-pocket-expenditure-vs-gdp-per-capita
share-of-people-who-think-their-life-will-improve-in-the-next-years
share-of-population-covered-by-social-protection
share-of-population-in-extreme-poverty
share-of-population-in-multidimensional-poverty-urban-vs-rural-cme-scatter
share-of-population-living-in-cities-towns-and-villages
share-of-populations-increasing-stable-or-decreasing-in-the-living-planet-index
share-of-population-urban
share-of-population-using-safely-managed-drinking-water-services-rural-vs-urban
share-of-population-with-cancer-age-standardized
share-of-population-with-cancer-by-age
share-of-population-with-cancer-crude
share-of-population-with-cancer-types
share-of-population-with-cancer-vs-median-age
share-of-population-with-improved-sanitation-faciltities
share-of-population-with-large-household-expenditures-on-health-10pc
share-of-primary-energy-consumption-by-source
share-of-primary-energy-from-solar-and-wind
share-of-public-expenditure-on-healthcare-by-country
share-of-rural-population-near-a-road
share-of-rural-population-with-electricity-access-vs-share-of-total-population-with-electricity-access
share-of-rural-population-with-improved-sanitation-faciltities
share-of-s-aureus-bloodstream-infections-that-are-resistant-to-antibiotics
share-of-schools-with-access-to-single-sex-basic-sanitation
share-of-species-evaluated-iucn
share-of-stage-1-cancers-by-diagnosis-route-in-england
share-of-stage-4-cancers-by-diagnosis-route-in-england
share-of-the-population-at-risk-of-catastrophic-expenditure-when-surgical-care-is-required
share-of-the-population-infected-with-hiv
share-of-the-population-using-improved-sanitation-facilities
share-of-the-population-with-access-to-electricity
share-of-the-population-with-access-to-sanitation-facilities
share-of-the-population-with-latent-tuberculosis-infection
share-of-the-population-without-access-to-clean-fuels-for-cooking
share-of-the-rural-and-urban-population-with-access-to-clean-fuels-for-cooking
share-of-the-total-land-area-burnt-by-wildfires-each-year
share-of-the-urban-and-rural-population-using-safely-managed-drinking-water-services
share-of-the-urban-and-rural-population-using-safely-managed-sanitation-services
share-of-the-workforce-employed-in-the-coal-industry-united-kingdom
share-of-total-disease-burden-by-cause
share-of-total-energy-used-in-agriculture-and-forestry
share-of-urban-population-by-size-of-the-agglomeration
share-of-urban-population-by-source
share-of-urban-population-living-in-inadequate-housing
share-of-urban-population-living-in-slums
share-of-urban-population-with-convenient-access-to-open-public-spaces
share-of-urban-population-with-improved-sanitation-facilities
share-of-women-having-births
share-of-women-using-modern-contraceptive-methods
share-of-women-who-have-had-no-births-across-their-childbearing-years
share-of-young-women-vs-men-who-are-not-in-education-employment-or-training
share-population-female
share-species-traded
share-threatened-species
share-urban-and-rural-population
share-using-safely-managed-sanitation
share-who-agree-vaccines-are-effective
share-who-report-lifetime-anxiety-or-depression
share-who-say-its-extremely-important-for-the-national-government-to-fund-research-on-anxietydepression
share-with-access-to-electricity-vs-per-capita-energy-consumption
share-with-convenient-access-to-an-open-public-space
share-with-convenient-access-to-public-transport
share-with-drug-use-disorders
share-with-mental-and-substance-disorders
share-with-mental-health-males-vs-females
share-with-mental-health-or-development-disorder
share-without-improved-sanitation
share-without-improved-water
smalsmall-scale-industries-loan
snow-cover-north-america
so-emissions-by-world-region-in-million-tonnes
solar-and-wind-power-generation
solar-electricity-per-capita
solar-energy-consumption
solar-energy-consumption-by-region
solar-pv-cumulative-capacity
solar-pv-energy-consumption-vs-solar-pv-capacity
solar-pv-prices
solar-pv-prices-vs-cumulative-capacity
solar-pv-system-costs
solar-share-energy
sources-of-air-pollution-in-the-uk
southern-white-rhinos
species-represented-in-the-living-planet-index
spring-temperature-anomalies
stand-alone-policy-or-plan-for-mental-health
statin-use-cardiovascular-disease-death-rate
statin-use-vs-share-of-population-with-high-blood-pressure
status-of-buruli-ulcer-endemicity
status-of-endemicity-of-cutaneous-leishmaniasis
status-of-endemicity-of-onchocerciasis
status-of-endemicity-of-visceral-leishmaniasis
status-of-trachoma-as-a-public-health-problem
status-of-yaws-endemicity
stillbirth-rate
stomach-cancer-death-rate
stroke-death-rates
stroke-death-rates-by-age
stroke-deaths-by-age
sub-energy-fossil-renewables-nuclear
suicide-death-rates
suicide-death-rates-by-sex-who
suicide-deaths-by-age
suicide-rate-1990-2019
suicide-rate-2000-2019-who
suicide-rate-for-15--to-49-year-olds
suicide-rate-male-female-who-mdb
suicide-rates-among-young-men-who-mdb
suicide-rates-among-young-people-who-mdb
suicide-rates-among-young-women-who-mdb
suicide-rates-by-age-detailed
suicide-rates-by-age-detailed-who
suicide-rates-by-age-who-mdb
suicide-rates-source-comparison
suicide-rates-source-comparison-crude
suicide-rate-vs-income-inequality
suicide-rate-who-mdb
suicide-vs-violent-deaths
sulphur-dioxide-and-coal
sumatran-rhinos
summer-temperature-anomalies
support-policies-climate
support-political-climate-action
support-public-action-climate
surgeons-per-100000
survival-ages-across-the-population
survival-to-age-65-of-cohort
sustainable-fisheries-as-a-proportion-of-gdp
sweden-official-covid-deaths
tax-revenue-per-capita-and-public-health-spending-per-capita-in-developing-countries-in
technology-adoption-by-households-in-the-united-states
temp-deaths-zhao
temperature-anomaly
temperature-anomaly-vs-latitude
terrestrial-protected-areas
testing-coverage-of-confirmed-bacterial-infections
tetanus-cases-vs-tetanus-vaccination-coverage
tetanus-mortality-in-children-under-5
the-burden-of-disease-vs-gdp-per-capita
the-long-term-energy-transition-in-europe
the-number-of-cases-of-infectious-diseases
thermal-efficiency-factor
the-worlds-number-of-vaccinated-one-year-olds
threatened-bird-species
threatened-endemic-bird-species
threatened-endemic-coral
threatened-endemic-mammal-species
threatened-mammal-species
total-cancer-deaths-by-type
total-cardiologists-per-million-people
total-deaths-comparison
total-disease-burden
total-disease-burden-by-cause
total-disease-burden-who
total-fertility-rate-vs-cohort-fertility-rate-vs-tempo-adjusted-tfr
total-healthcare-expenditure-gdp
total-live-births-hfd
total-number-of-births-by-birth-order
total-number-of-deaths-by-cause
total-oda-for-biodiversity-by-donor
total-oda-for-biodiversity-by-recipient
total-oda-for-water-supply-and-sanitation-by-recipient
trachoma-prevalence-age-standardized-ihme
traded-energy-share-domestic
trawling-footprint-region
tree-cover-loss
tree-cover-loss-by-dominant-driver
tree-cover-loss-from-wildfires
trust-healthcare-vs-vaccine-safety
tuberculosis-case-detection-rate
tuberculosis-cases-attributable-to-risk-factors
tuberculosis-death-rate
tuberculosis-deaths-who
tuberculosis-funding-versus-budget-requirement
tuberculosis-incidence-uncertainty-intervals
tuberculosis-treatment-success-rate-by-type
twentieth-century-of-deaths-us
uk-covid-hospital-patients
uk-covid-positivity
uk-cumulative-covid-deaths-rate
uk-daily-covid-admissions
uk-daily-covid-cases-7day-average
uk-daily-covid-deaths
uk-daily-new-covid-cases
under-5-mortality-rate-sdgs
unemployment-rate-of-males-vs-females
unintentional-injuries-death-rate-by-age-group-who-mdb
universal-health-coverage-index
unmet-need-for-contraception-share-of-married-women-of-reproductive-age
un-population-2024-vs-2022
un-projection-of-annual-infant-deaths
uranium-production
urban-agglomerations-1-million-percent
urban-agglomerations-more-than-1-million
urban-and-rural-population
urban-and-rural-population-2050
urban-and-rural-population-stacked
urban-area-long-term
urban-improved-water-access-vs-rural-water-access
urbanization-vs-gdp
urban-land-area
urban-policies-population-dynamics
urban-pop-in-out-of-slums
urban-population-share-2050
urban-population-vs-female-labor-force-participation-rate
urban-population-with-improved-water
urban-share-european-commission
urban-vs-rural-majority
urban-vs-rural-population-safely-managed-sanitation
urban-vs-rural-population-using-at-least-basic-sanitation
urban-vs-rural-safely-managed-drinking-water-source
urban-vs-rural-using-at-least-basic-drinking-water
us-covid-19-share-people-vaccinated
us-covid-19-total-people-vaccinated
us-covid-number-fully-vaccinated
us-covid-share-fully-vaccinated
us-daily-covid-vaccine-doses-administered
us-daily-covid-vaccine-doses-per-100
use-of-statins-for-primary-prevention-of-cardiovascular-disease
use-of-statins-for-secondary-prevention-amongst-eligibible-individuals
us-healthcare-expenditure
us-state-covid-vaccines-per-100
us-total-covid-19-vaccine-doses-administered
utilization-of-statins
vaccination-coverage-by-income-in
vaccination-coverage-dpt-by-extreme-poverty
wanted-fertility-rate
warming-fossil-fuels-land-use
warm-season-deaths-climate
wealth-per-capita-by-asset-group-with-natural-capital-disaggregated-in-2014-us-dollars
weekly-area-burnt-by-wildfires
weekly-cumulative-share-of-the-area-burnt-by-wildfires-each-year
weekly-share-of-the-total-land-area-burnt-by-wildfires
were-flu-vaccines-available-in-the-country
whale-catch
whale-populations
whales-killed-per-decade
which-countries-include-malaria-vaccines-in-their-vaccination-schedules
wild-fish-allocation
wild-fish-catch-gear-type
wildlife-exports
willingness-climate-action
wind-electricity-per-capita
wind-energy-consumption-by-region
wind-energy-consumption-vs-installed-wind-energy-capacity
wind-generation
wind-share-energy
winter-temperature-anomalies
women-high-blood-pressure
womens-educational-attainment-vs-fertility
womens-life-expectancy-at-birth
women-survival-to-age-65
women-who-have-next-birth-parity-progression-ratio
world-crude-oil-price-vs-oil-consumption
world-population-comparison-historical-sources
world-population-projections-by-un-prospects-revision
years-lived-with-disability-vs-gdp-per-capita
years-lived-with-disability-vs-health-expenditure-per-capita
years-of-fossil-fuel-reserves-left
youth-mortality-rate
