use strict;
use warnings;
use utf8;

use Test::More;
if ( not $ENV{AUTHOR_TESTING} ) {
    my $msg = 'Set $ENV{AUTHOR_TESTING} to run author tests.';
    plan( skip_all => $msg );
}
if ( !eval { require Benchmark; 1 } ) {
    plan skip_all => "Benchmark required for this test";
}
$ENV{AUTHOR_TESTING} && eval { require Test::NoWarnings };

use version;

plan tests => 2;

sub benchmark {
    use Benchmark;
    use HTML::Hyphenate;

    my $EMPTY = q{};

    my $html = $EMPTY;
    while ( my $line = <DATA> ) {
        $html .= $line;
    }

    my $h = HTML::Hyphenate->new();
    diag(
        q{Hyphenating the HTML standard as a single 8MB HTML document can } .
        q{take a couple of minutes, starting time is } . localtime );
    my $t = timeit( 1, sub { $h->hyphenated($html) } );
    diag( q{Hyphenating took: } . timestr($t) );
    return 1;
}
ok( benchmark(), q{Benchmark});

$ENV{AUTHOR_TESTING} && Test::NoWarnings::had_no_warnings();

__DATA__
<!DOCTYPE html><html lang=en-US><meta charset=utf-8><meta content="width=device-width, initial-scale=1, shrink-to-fit=no" name=viewport><title>HTML Standard</title><meta content=#3c790a name=theme-color><link rel=stylesheet href=https://resources.whatwg.org/spec.css crossorigin=""><link rel=stylesheet href=https://resources.whatwg.org/standard.css crossorigin=""><link rel=stylesheet href=https://resources.whatwg.org/standard-shared-with-dev.css crossorigin=""><link rel=icon href=https://resources.whatwg.org/logo.svg crossorigin=""><link rel=stylesheet href=/styles.css crossorigin=""><script>
   function toggleStatus(div) {
     div.parentNode.classList.toggle('wrapped');
   }
   function setLinkFragment(link) {
     link.hash = location.hash;
   }
  </script><body>
  
  <script async="" src=/html-dfn.js crossorigin=""></script>
  <script data-file-issue-url=https://github.com/whatwg/html/issues/new async="" crossorigin="" src=https://resources.whatwg.org/file-issue.js></script>
  <script async="" crossorigin="" src=https://resources.whatwg.org/commit-snapshot-shortcut-key.js></script>
  <header id=head class="head with-buttons">
   <a href=https://whatwg.org/ class=logo><img src=https://resources.whatwg.org/logo.svg width=100 alt=WHATWG height=100 crossorigin=""></a>
   <hgroup><h1 class=allcaps>HTML</h1><h2 id=living-standard class="no-num no-toc">Living Standard — Last Updated <span class=pubdate>20 November 2020</span></h2></hgroup>
   
   <nav>
    
    <div>
     
     <a href=/><span><strong>One-Page Version</strong> <code>html.spec.whatwg.org</code></span></a>
     <a onclick=setLinkFragment(this); href=/multipage/ id=multipage-link><span><strong>Multipage Version</strong> <code>/multipage</code></span></a>
     <a onclick=setLinkFragment(this); href=/dev/><span><strong>Developer Version</strong> <code>/dev</code></span></a>
     <a href=/print.pdf><span><strong>PDF Version</strong> <code>/print.pdf</code></span></a>
     <a href=https://github.com/whatwg/html/wiki/Translations><span><strong>Translations</strong> <code>日本語 • 简体中文</code></span></a>
    </div>
    <div>
     <a href=https://github.com/whatwg/html/blob/master/FAQ.md class=misc><span><strong>FAQ</strong> <code>on GitHub</code></span></a>
     <a href=https://wiki.whatwg.org/wiki/IRC class=comms><span><strong>Join us on IRC</strong> <code>#whatwg on Freenode</code></span></a>
    </div>
    <div>
     <a href=https://github.com/whatwg/html class=changes><span><strong>Contribute on GitHub</strong> <code>whatwg/html repository</code></span></a>
     <a href=https://github.com/whatwg/html/commits class=changes><span><strong>Commits</strong> <code>on GitHub</code></span></a>
     <a id=commit-snapshot-link href=/commit-snapshots/922872b2b9a7999e879188bf9b7c99cae25328c0/ class=changes><span><strong>Snapshot</strong> <code>as of this commit</code></span></a>
     <a href=https://twitter.com/htmlstandard class=changes><span><strong>Twitter Updates</strong> <code>@htmlstandard</code></span></a>
    </div>
    <div>
     <a href=https://github.com/whatwg/html/issues class=feedback><span><strong>Open Issues</strong> <code>filed on GitHub</code></span></a>
     <a href=https://whatwg.org/newbug class=feedback><span><strong>Open an Issue</strong> <code>whatwg.org/newbug</code></span></a>
     <a href=https://github.com/web-platform-tests/wpt/tree/master/html class=feedback><span><strong>Tests</strong> <code>web-platform-tests html/</code></span></a>
     <a href=https://github.com/web-platform-tests/wpt/labels/html class=feedback><span><strong>Issues for Tests</strong> <code>ongoing work</code></span></a>
    </div>
   </nav>

   
  </header>

  

  

  <hr>

  <h2 id=table-of-contents class="no-num no-toc">Table of contents</h2>
  <ol class="brief toc"><li><a href=#toc-introduction><span class=secno>1</span> Introduction</a><li><a href=#toc-infrastructure><span class=secno>2</span> Common infrastructure</a><li><a href=#toc-dom><span class=secno>3</span> Semantics, structure, and APIs of HTML documents</a><li><a href=#toc-semantics><span class=secno>4</span> The elements of HTML</a><li><a href=#toc-microdata><span class=secno>5</span> Microdata</a><li><a href=#toc-editing><span class=secno>6</span> User interaction</a><li><a href=#toc-browsers><span class=secno>7</span> Loading web pages</a><li><a href=#toc-webappapis><span class=secno>8</span> Web application APIs</a><li><a href=#toc-comms><span class=secno>9</span> Communication</a><li><a href=#toc-workers><span class=secno>10</span> Web workers</a><li><a href=#toc-worklets><span class=secno>11</span> Worklets</a><li><a href=#toc-webstorage><span class=secno>12</span> Web storage</a><li><a href=#toc-syntax><span class=secno>13</span> The HTML syntax</a><li><a href=#toc-the-xhtml-syntax><span class=secno>14</span> The XML syntax</a><li><a href=#toc-rendering><span class=secno>15</span> Rendering</a><li><a href=#toc-obsolete><span class=secno>16</span> Obsolete features</a><li><a href=#toc-iana><span class=secno>17</span> IANA considerations</a><li><a href=#toc-index>Index</a><li><a href=#toc-references>References</a><li><a href=#toc-acknowledgments>Acknowledgments</a><li><a href=#toc-ipr>Intellectual property rights</a></ol>

  <h2 id=contents class="no-num no-toc">Full table of contents</h2>
  <ol class=toc><li id=toc-introduction><a href=#introduction><span class=secno>1</span> Introduction</a><ol><li><a href=#abstract><span class=secno>1.1</span> Where does this specification fit?</a><li><a href=#is-this-html5?><span class=secno>1.2</span> Is this HTML5?</a><li><a href=#background><span class=secno>1.3</span> Background</a><li><a href=#audience><span class=secno>1.4</span> Audience</a><li><a href=#scope><span class=secno>1.5</span> Scope</a><li><a href=#history-2><span class=secno>1.6</span> History</a><li><a href=#design-notes><span class=secno>1.7</span> Design notes</a><ol><li><a href=#serialisability-of-script-execution><span class=secno>1.7.1</span> Serializability of script execution</a><li><a href=#compliance-with-other-specifications><span class=secno>1.7.2</span> Compliance with other specifications</a><li><a href=#extensibility><span class=secno>1.7.3</span> Extensibility</a></ol><li><a href=#html-vs-xhtml><span class=secno>1.8</span> HTML vs XML syntax</a><li><a href=#structure-of-this-specification><span class=secno>1.9</span> Structure of this specification</a><ol><li><a href=#how-to-read-this-specification><span class=secno>1.9.1</span> How to read this specification</a><li><a href=#typographic-conventions><span class=secno>1.9.2</span> Typographic conventions</a></ol><li><a href=#a-quick-introduction-to-html><span class=secno>1.10</span> A quick introduction to HTML</a><ol><li><a href=#writing-secure-applications-with-html><span class=secno>1.10.1</span> Writing secure applications with HTML</a><li><a href=#common-pitfalls-to-avoid-when-using-the-scripting-apis><span class=secno>1.10.2</span> Common pitfalls to avoid when using the scripting APIs</a><li><a href=#how-to-catch-mistakes-when-writing-html:-validators-and-conformance-checkers><span class=secno>1.10.3</span> How to catch mistakes when writing HTML: validators and conformance checkers</a></ol><li><a href=#conformance-requirements-for-authors><span class=secno>1.11</span> Conformance requirements for authors</a><ol><li><a href=#presentational-markup><span class=secno>1.11.1</span> Presentational markup</a><li><a href=#syntax-errors><span class=secno>1.11.2</span> Syntax errors</a><li><a href=#restrictions-on-content-models-and-on-attribute-values><span class=secno>1.11.3</span> Restrictions on content models and on attribute values</a></ol><li><a href=#suggested-reading><span class=secno>1.12</span> Suggested reading</a></ol><li id=toc-infrastructure><a href=#infrastructure><span class=secno>2</span> Common infrastructure</a><ol><li><a href=#terminology><span class=secno>2.1</span> Terminology</a><ol><li><a href=#parallelism><span class=secno>2.1.1</span> Parallelism</a><li><a href=#resources><span class=secno>2.1.2</span> Resources</a><li><a href=#xml><span class=secno>2.1.3</span> XML compatibility</a><li><a href=#dom-trees><span class=secno>2.1.4</span> DOM trees</a><li><a href=#scripting-2><span class=secno>2.1.5</span> Scripting</a><li><a href=#plugins><span class=secno>2.1.6</span> Plugins</a><li><a href=#encoding-terminology><span class=secno>2.1.7</span> Character encodings</a><li><a href=#conformance-classes><span class=secno>2.1.8</span> Conformance classes</a><li><a href=#dependencies><span class=secno>2.1.9</span> Dependencies</a><li><a href=#extensibility-2><span class=secno>2.1.10</span> Extensibility</a><li><a href=#interactions-with-xpath-and-xslt><span class=secno>2.1.11</span> Interactions with XPath and XSLT</a></ol><li><a href=#case-sensitivity-and-string-comparison><span class=secno>2.2</span> String comparison</a><li><a href=#policy-controlled-features><span class=secno>2.3</span> Policy-controlled features</a><li><a href=#common-microsyntaxes><span class=secno>2.4</span> Common microsyntaxes</a><ol><li><a href=#common-parser-idioms><span class=secno>2.4.1</span> Common parser idioms</a><li><a href=#boolean-attributes><span class=secno>2.4.2</span> Boolean attributes</a><li><a href=#keywords-and-enumerated-attributes><span class=secno>2.4.3</span> Keywords and enumerated attributes</a><li><a href=#numbers><span class=secno>2.4.4</span> Numbers</a><ol><li><a href=#signed-integers><span class=secno>2.4.4.1</span> Signed integers</a><li><a href=#non-negative-integers><span class=secno>2.4.4.2</span> Non-negative integers</a><li><a href=#floating-point-numbers><span class=secno>2.4.4.3</span> Floating-point numbers</a><li><a href=#percentages-and-dimensions><span class=secno>2.4.4.4</span> Percentages and lengths</a><li><a href=#non-zero-percentages-and-lengths><span class=secno>2.4.4.5</span> Non-zero percentages and lengths</a><li><a href=#lists-of-floating-point-numbers><span class=secno>2.4.4.6</span> Lists of floating-point numbers</a><li><a href=#lists-of-dimensions><span class=secno>2.4.4.7</span> Lists of dimensions</a></ol><li><a href=#dates-and-times><span class=secno>2.4.5</span> Dates and times</a><ol><li><a href=#months><span class=secno>2.4.5.1</span> Months</a><li><a href=#dates><span class=secno>2.4.5.2</span> Dates</a><li><a href=#yearless-dates><span class=secno>2.4.5.3</span> Yearless dates</a><li><a href=#times><span class=secno>2.4.5.4</span> Times</a><li><a href=#local-dates-and-times><span class=secno>2.4.5.5</span> Local dates and times</a><li><a href=#time-zones><span class=secno>2.4.5.6</span> Time zones</a><li><a href=#global-dates-and-times><span class=secno>2.4.5.7</span> Global dates and times</a><li><a href=#weeks><span class=secno>2.4.5.8</span> Weeks</a><li><a href=#durations><span class=secno>2.4.5.9</span> Durations</a><li><a href=#vaguer-moments-in-time><span class=secno>2.4.5.10</span> Vaguer moments in time</a></ol><li><a href=#colours><span class=secno>2.4.6</span> Colors</a><li><a href=#space-separated-tokens><span class=secno>2.4.7</span> Space-separated tokens</a><li><a href=#comma-separated-tokens><span class=secno>2.4.8</span> Comma-separated tokens</a><li><a href=#syntax-references><span class=secno>2.4.9</span> References</a><li><a href=#mq><span class=secno>2.4.10</span> Media queries</a></ol><li><a href=#urls><span class=secno>2.5</span> URLs</a><ol><li><a href=#terminology-2><span class=secno>2.5.1</span> Terminology</a><li><a href=#resolving-urls><span class=secno>2.5.2</span> Parsing URLs</a><li><a href=#dynamic-changes-to-base-urls><span class=secno>2.5.3</span> Dynamic changes to base URLs</a></ol><li><a href=#fetching-resources><span class=secno>2.6</span> Fetching resources</a><ol><li><a href=#terminology-3><span class=secno>2.6.1</span> Terminology</a><li><a href=#content-type-sniffing><span class=secno>2.6.2</span> Determining the type of a resource</a><li><a href=#extracting-character-encodings-from-meta-elements><span class=secno>2.6.3</span> Extracting character encodings from <code>meta</code> elements</a><li><a href=#cors-settings-attributes><span class=secno>2.6.4</span> CORS settings attributes</a><li><a href=#referrer-policy-attributes><span class=secno>2.6.5</span> Referrer policy attributes</a><li><a href=#nonce-attributes><span class=secno>2.6.6</span> Nonce attributes</a><li><a href=#lazy-loading-attributes><span class=secno>2.6.7</span> Lazy loading attributes</a></ol><li><a href=#common-dom-interfaces><span class=secno>2.7</span> Common DOM interfaces</a><ol><li><a href=#reflecting-content-attributes-in-idl-attributes><span class=secno>2.7.1</span> Reflecting content attributes in IDL attributes</a><li><a href=#collections><span class=secno>2.7.2</span> Collections</a><ol><li><a href=#the-htmlallcollection-interface><span class=secno>2.7.2.1</span> The <code>HTMLAllCollection</code> interface</a><ol><li><a href=#HTMLAllCollection-call><span class=secno>2.7.2.1.1</span> [[Call]] ( <var>thisArgument</var>,  <var>argumentsList</var> )</a></ol><li><a href=#the-htmlformcontrolscollection-interface><span class=secno>2.7.2.2</span> The <code>HTMLFormControlsCollection</code> interface</a><li><a href=#the-htmloptionscollection-interface><span class=secno>2.7.2.3</span> The <code>HTMLOptionsCollection</code> interface</a></ol><li><a href=#the-domstringlist-interface><span class=secno>2.7.3</span> The <code>DOMStringList</code> interface</a></ol><li><a href=#safe-passing-of-structured-data><span class=secno>2.8</span> Safe passing of structured data</a><ol><li><a href=#serializable-objects><span class=secno>2.8.1</span> Serializable objects</a><li><a href=#transferable-objects><span class=secno>2.8.2</span> Transferable objects</a><li><a href=#structuredserializeinternal><span class=secno>2.8.3</span> StructuredSerializeInternal ( <var>value</var>, <var>forStorage</var> [ ,
  <var>memory</var> ] )</a><li><a href=#structuredserialize><span class=secno>2.8.4</span> StructuredSerialize ( <var>value</var> )</a><li><a href=#structuredserializeforstorage><span class=secno>2.8.5</span> StructuredSerializeForStorage ( <var>value</var> )</a><li><a href=#structureddeserialize><span class=secno>2.8.6</span> StructuredDeserialize ( <var>serialized</var>, <var>targetRealm</var> [ ,
  <var>memory</var> ] )</a><li><a href=#structuredserializewithtransfer><span class=secno>2.8.7</span> StructuredSerializeWithTransfer ( <var>value</var>, <var>transferList</var>
  )</a><li><a href=#structureddeserializewithtransfer><span class=secno>2.8.8</span> StructuredDeserializeWithTransfer ( <var>serializeWithTransferResult</var>,
  <var>targetRealm</var> )</a><li><a href=#performing-structured-clones-from-other-specifications><span class=secno>2.8.9</span> Performing serialization and
  transferring from other specifications</a></ol></ol><li id=toc-dom><a href=#dom><span class=secno>3</span> Semantics, structure, and APIs of HTML documents</a><ol><li><a href=#documents><span class=secno>3.1</span> Documents</a><ol><li><a href=#the-document-object><span class=secno>3.1.1</span> The <code>Document</code> object</a><li><a href=#the-documentorshadowroot-interface><span class=secno>3.1.2</span> The <code>DocumentOrShadowRoot</code> interface</a><li><a href=#resource-metadata-management><span class=secno>3.1.3</span> Resource metadata management</a><li><a href=#dom-tree-accessors><span class=secno>3.1.4</span> DOM tree accessors</a></ol><li><a href=#elements><span class=secno>3.2</span> Elements</a><ol><li><a href=#semantics-2><span class=secno>3.2.1</span> Semantics</a><li><a href=#elements-in-the-dom><span class=secno>3.2.2</span> Elements in the DOM</a><li><a href=#html-element-constructors><span class=secno>3.2.3</span> HTML element constructors</a><li><a href=#element-definitions><span class=secno>3.2.4</span> Element definitions</a><ol><li><a href=#attributes><span class=secno>3.2.4.1</span> Attributes</a></ol><li><a href=#content-models><span class=secno>3.2.5</span> Content models</a><ol><li><a href=#the-nothing-content-model><span class=secno>3.2.5.1</span> The "nothing" content model</a><li><a href=#kinds-of-content><span class=secno>3.2.5.2</span> Kinds of content</a><ol><li><a href=#metadata-content><span class=secno>3.2.5.2.1</span> Metadata content</a><li><a href=#flow-content><span class=secno>3.2.5.2.2</span> Flow content</a><li><a href=#sectioning-content><span class=secno>3.2.5.2.3</span> Sectioning content</a><li><a href=#heading-content><span class=secno>3.2.5.2.4</span> Heading content</a><li><a href=#phrasing-content><span class=secno>3.2.5.2.5</span> Phrasing content</a><li><a href=#embedded-content-2><span class=secno>3.2.5.2.6</span> Embedded content</a><li><a href=#interactive-content><span class=secno>3.2.5.2.7</span> Interactive content</a><li><a href=#palpable-content><span class=secno>3.2.5.2.8</span> Palpable content</a><li><a href=#script-supporting-elements><span class=secno>3.2.5.2.9</span> Script-supporting elements</a></ol><li><a href=#transparent-content-models><span class=secno>3.2.5.3</span> Transparent content models</a><li><a href=#paragraphs><span class=secno>3.2.5.4</span> Paragraphs</a></ol><li><a href=#global-attributes><span class=secno>3.2.6</span> Global attributes</a><ol><li><a href=#the-title-attribute><span class=secno>3.2.6.1</span> The <code>title</code> attribute</a><li><a href=#the-lang-and-xml:lang-attributes><span class=secno>3.2.6.2</span> The <code>lang</code> and <code>xml:lang</code>
  attributes</a><li><a href=#the-translate-attribute><span class=secno>3.2.6.3</span> The <code>translate</code> attribute</a><li><a href=#the-dir-attribute><span class=secno>3.2.6.4</span> The <code>dir</code> attribute</a><li><a href=#the-style-attribute><span class=secno>3.2.6.5</span> The <code>style</code> attribute</a><li><a href=#embedding-custom-non-visible-data-with-the-data-*-attributes><span class=secno>3.2.6.6</span> Embedding custom non-visible data with the <code>data-*</code> attributes</a></ol><li><a href=#the-innertext-idl-attribute><span class=secno>3.2.7</span> The <code>innerText</code> getter and
  setter</a><li><a href=#requirements-relating-to-the-bidirectional-algorithm><span class=secno>3.2.8</span> Requirements relating to the bidirectional algorithm</a><ol><li><a href=#authoring-conformance-criteria-for-bidirectional-algorithm-formatting-characters><span class=secno>3.2.8.1</span> Authoring conformance criteria for bidirectional-algorithm formatting characters</a><li><a href=#user-agent-conformance-criteria><span class=secno>3.2.8.2</span> User agent conformance criteria</a></ol><li><a href=#wai-aria><span class=secno>3.2.9</span> Requirements related to ARIA and to platform accessibility APIs</a></ol></ol><li id=toc-semantics><a href=#semantics><span class=secno>4</span> The elements of HTML</a><ol><li><a href=#the-root-element><span class=secno>4.1</span> The document element</a><ol><li><a href=#the-html-element><span class=secno>4.1.1</span> The <code>html</code> element</a></ol><li><a href=#document-metadata><span class=secno>4.2</span> Document metadata</a><ol><li><a href=#the-head-element><span class=secno>4.2.1</span> The <code>head</code> element</a><li><a href=#the-title-element><span class=secno>4.2.2</span> The <code>title</code> element</a><li><a href=#the-base-element><span class=secno>4.2.3</span> The <code>base</code> element</a><li><a href=#the-link-element><span class=secno>4.2.4</span> The <code>link</code> element</a><ol><li><a href=#processing-the-media-attribute><span class=secno>4.2.4.1</span> Processing the <code>media</code> attribute</a><li><a href=#processing-the-type-attribute><span class=secno>4.2.4.2</span> Processing the <code>type</code> attribute</a><li><a href=#fetching-and-processing-a-resource-from-a-link-element><span class=secno>4.2.4.3</span> Fetching and processing a resource
  from a <code>link</code> element</a><li><a href=#processing-link-headers><span class=secno>4.2.4.4</span> Processing `<code>Link</code>` headers</a><li><a href=#providing-users-with-a-means-to-follow-hyperlinks-created-using-the-link-element><span class=secno>4.2.4.5</span> Providing users with a means to follow hyperlinks created using the <code>link</code>
  element</a></ol><li><a href=#the-meta-element><span class=secno>4.2.5</span> The <code>meta</code>
  element</a><ol><li><a href=#standard-metadata-names><span class=secno>4.2.5.1</span> Standard metadata names</a><li><a href=#other-metadata-names><span class=secno>4.2.5.2</span> Other metadata names</a><li><a href=#pragma-directives><span class=secno>4.2.5.3</span> Pragma directives</a><li><a href=#charset><span class=secno>4.2.5.4</span> Specifying the document's character encoding</a></ol><li><a href=#the-style-element><span class=secno>4.2.6</span> The <code>style</code> element</a><li><a href=#interactions-of-styling-and-scripting><span class=secno>4.2.7</span> Interactions of styling and scripting</a></ol><li><a href=#sections><span class=secno>4.3</span> Sections</a><ol><li><a href=#the-body-element><span class=secno>4.3.1</span> The <code>body</code> element</a><li><a href=#the-article-element><span class=secno>4.3.2</span> The <code>article</code> element</a><li><a href=#the-section-element><span class=secno>4.3.3</span> The <code>section</code> element</a><li><a href=#the-nav-element><span class=secno>4.3.4</span> The <code>nav</code> element</a><li><a href=#the-aside-element><span class=secno>4.3.5</span> The <code>aside</code> element</a><li><a href=#the-h1,-h2,-h3,-h4,-h5,-and-h6-elements><span class=secno>4.3.6</span> The <code>h1</code>, <code>h2</code>, <code>h3</code>, <code>h4</code>, <code>h5</code>, and
  <code>h6</code> elements</a><li><a href=#the-hgroup-element><span class=secno>4.3.7</span> The <code>hgroup</code> element</a><li><a href=#the-header-element><span class=secno>4.3.8</span> The <code>header</code> element</a><li><a href=#the-footer-element><span class=secno>4.3.9</span> The <code>footer</code> element</a><li><a href=#the-address-element><span class=secno>4.3.10</span> The <code>address</code> element</a><li><a href=#headings-and-sections><span class=secno>4.3.11</span> Headings and sections</a><ol><li><a href=#outlines><span class=secno>4.3.11.1</span> Creating an outline</a><li><a href=#sample-outlines><span class=secno>4.3.11.2</span> Sample outlines</a><li><a href=#exposing-outlines-to-users><span class=secno>4.3.11.3</span> Exposing outlines to users</a></ol><li><a href=#usage-summary-2><span class=secno>4.3.12</span> Usage summary</a><ol><li><a href=#article-or-section><span class=secno>4.3.12.1</span> Article or section?</a></ol></ol><li><a href=#grouping-content><span class=secno>4.4</span> Grouping content</a><ol><li><a href=#the-p-element><span class=secno>4.4.1</span> The <code>p</code> element</a><li><a href=#the-hr-element><span class=secno>4.4.2</span> The <code>hr</code> element</a><li><a href=#the-pre-element><span class=secno>4.4.3</span> The <code>pre</code> element</a><li><a href=#the-blockquote-element><span class=secno>4.4.4</span> The <code>blockquote</code> element</a><li><a href=#the-ol-element><span class=secno>4.4.5</span> The <code>ol</code> element</a><li><a href=#the-ul-element><span class=secno>4.4.6</span> The <code>ul</code> element</a><li><a href=#the-menu-element><span class=secno>4.4.7</span> The <code>menu</code> element</a><li><a href=#the-li-element><span class=secno>4.4.8</span> The <code>li</code> element</a><li><a href=#the-dl-element><span class=secno>4.4.9</span> The <code>dl</code> element</a><li><a href=#the-dt-element><span class=secno>4.4.10</span> The <code>dt</code> element</a><li><a href=#the-dd-element><span class=secno>4.4.11</span> The <code>dd</code> element</a><li><a href=#the-figure-element><span class=secno>4.4.12</span> The <code>figure</code> element</a><li><a href=#the-figcaption-element><span class=secno>4.4.13</span> The <code>figcaption</code> element</a><li><a href=#the-main-element><span class=secno>4.4.14</span> The <code>main</code> element</a><li><a href=#the-div-element><span class=secno>4.4.15</span> The <code>div</code> element</a></ol><li><a href=#text-level-semantics><span class=secno>4.5</span> Text-level semantics</a><ol><li><a href=#the-a-element><span class=secno>4.5.1</span> The <code>a</code> element</a><li><a href=#the-em-element><span class=secno>4.5.2</span> The <code>em</code> element</a><li><a href=#the-strong-element><span class=secno>4.5.3</span> The <code>strong</code> element</a><li><a href=#the-small-element><span class=secno>4.5.4</span> The <code>small</code> element</a><li><a href=#the-s-element><span class=secno>4.5.5</span> The <code>s</code> element</a><li><a href=#the-cite-element><span class=secno>4.5.6</span> The <code>cite</code> element</a><li><a href=#the-q-element><span class=secno>4.5.7</span> The <code>q</code> element</a><li><a href=#the-dfn-element><span class=secno>4.5.8</span> The <code>dfn</code> element</a><li><a href=#the-abbr-element><span class=secno>4.5.9</span> The <code>abbr</code> element</a><li><a href=#the-ruby-element><span class=secno>4.5.10</span> The <code>ruby</code> element</a><li><a href=#the-rt-element><span class=secno>4.5.11</span> The <code>rt</code> element</a><li><a href=#the-rp-element><span class=secno>4.5.12</span> The <code>rp</code> element</a><li><a href=#the-data-element><span class=secno>4.5.13</span> The <code>data</code> element</a><li><a href=#the-time-element><span class=secno>4.5.14</span> The <code>time</code> element</a><li><a href=#the-code-element><span class=secno>4.5.15</span> The <code>code</code> element</a><li><a href=#the-var-element><span class=secno>4.5.16</span> The <code>var</code> element</a><li><a href=#the-samp-element><span class=secno>4.5.17</span> The <code>samp</code> element</a><li><a href=#the-kbd-element><span class=secno>4.5.18</span> The <code>kbd</code> element</a><li><a href=#the-sub-and-sup-elements><span class=secno>4.5.19</span> The <code>sub</code> and <code>sup</code> elements</a><li><a href=#the-i-element><span class=secno>4.5.20</span> The <code>i</code> element</a><li><a href=#the-b-element><span class=secno>4.5.21</span> The <code>b</code> element</a><li><a href=#the-u-element><span class=secno>4.5.22</span> The <code>u</code> element</a><li><a href=#the-mark-element><span class=secno>4.5.23</span> The <code>mark</code> element</a><li><a href=#the-bdi-element><span class=secno>4.5.24</span> The <code>bdi</code> element</a><li><a href=#the-bdo-element><span class=secno>4.5.25</span> The <code>bdo</code> element</a><li><a href=#the-span-element><span class=secno>4.5.26</span> The <code>span</code> element</a><li><a href=#the-br-element><span class=secno>4.5.27</span> The <code>br</code> element</a><li><a href=#the-wbr-element><span class=secno>4.5.28</span> The <code>wbr</code> element</a><li><a href=#usage-summary><span class=secno>4.5.29</span> Usage summary</a></ol><li><a href=#links><span class=secno>4.6</span> Links</a><ol><li><a href=#introduction-2><span class=secno>4.6.1</span> Introduction</a><li><a href=#links-created-by-a-and-area-elements><span class=secno>4.6.2</span> Links created by <code>a</code> and <code>area</code> elements</a><li><a href=#api-for-a-and-area-elements><span class=secno>4.6.3</span> API for <code>a</code> and <code>area</code> elements</a><li><a href=#following-hyperlinks><span class=secno>4.6.4</span> Following hyperlinks</a><li><a href=#downloading-resources><span class=secno>4.6.5</span> Downloading resources</a><ol><li><a href=#hyperlink-auditing><span class=secno>4.6.5.1</span> Hyperlink auditing</a></ol><li><a href=#linkTypes><span class=secno>4.6.6</span> Link types</a><ol><li><a href=#rel-alternate><span class=secno>4.6.6.1</span> Link type "<code>alternate</code>"</a><li><a href=#link-type-author><span class=secno>4.6.6.2</span> Link type "<code>author</code>"</a><li><a href=#link-type-bookmark><span class=secno>4.6.6.3</span> Link type "<code>bookmark</code>"</a><li><a href=#link-type-canonical><span class=secno>4.6.6.4</span> Link type "<code>canonical</code>"</a><li><a href=#link-type-dns-prefetch><span class=secno>4.6.6.5</span> Link type "<code>dns-prefetch</code>"</a><li><a href=#link-type-external><span class=secno>4.6.6.6</span> Link type "<code>external</code>"</a><li><a href=#link-type-help><span class=secno>4.6.6.7</span> Link type "<code>help</code>"</a><li><a href=#rel-icon><span class=secno>4.6.6.8</span> Link type "<code>icon</code>"</a><li><a href=#link-type-license><span class=secno>4.6.6.9</span> Link type "<code>license</code>"</a><li><a href=#link-type-manifest><span class=secno>4.6.6.10</span> Link type "<code>manifest</code>"</a><li><a href=#link-type-modulepreload><span class=secno>4.6.6.11</span> Link type "<code>modulepreload</code>"</a><li><a href=#link-type-nofollow><span class=secno>4.6.6.12</span> Link type "<code>nofollow</code>"</a><li><a href=#link-type-noopener><span class=secno>4.6.6.13</span> Link type "<code>noopener</code>"</a><li><a href=#link-type-noreferrer><span class=secno>4.6.6.14</span> Link type "<code>noreferrer</code>"</a><li><a href=#link-type-opener><span class=secno>4.6.6.15</span> Link type "<code>opener</code>"</a><li><a href=#link-type-pingback><span class=secno>4.6.6.16</span> Link type "<code>pingback</code>"</a><li><a href=#link-type-preconnect><span class=secno>4.6.6.17</span> Link type "<code>preconnect</code>"</a><li><a href=#link-type-prefetch><span class=secno>4.6.6.18</span> Link type "<code>prefetch</code>"</a><li><a href=#link-type-preload><span class=secno>4.6.6.19</span> Link type "<code>preload</code>"</a><li><a href=#link-type-prerender><span class=secno>4.6.6.20</span> Link type "<code>prerender</code>"</a><li><a href=#link-type-search><span class=secno>4.6.6.21</span> Link type "<code>search</code>"</a><li><a href=#link-type-stylesheet><span class=secno>4.6.6.22</span> Link type "<code>stylesheet</code>"</a><li><a href=#link-type-tag><span class=secno>4.6.6.23</span> Link type "<code>tag</code>"</a><li><a href=#sequential-link-types><span class=secno>4.6.6.24</span> Sequential link types</a><ol><li><a href=#link-type-next><span class=secno>4.6.6.24.1</span> Link type "<code>next</code>"</a><li><a href=#link-type-prev><span class=secno>4.6.6.24.2</span> Link type "<code>prev</code>"</a></ol><li><a href=#other-link-types><span class=secno>4.6.6.25</span> Other link types</a></ol></ol><li><a href=#edits><span class=secno>4.7</span> Edits</a><ol><li><a href=#the-ins-element><span class=secno>4.7.1</span> The <code>ins</code> element</a><li><a href=#the-del-element><span class=secno>4.7.2</span> The <code>del</code> element</a><li><a href=#attributes-common-to-ins-and-del-elements><span class=secno>4.7.3</span> Attributes common to <code>ins</code> and <code>del</code> elements</a><li><a href=#edits-and-paragraphs><span class=secno>4.7.4</span> Edits and paragraphs</a><li><a href=#edits-and-lists><span class=secno>4.7.5</span> Edits and lists</a><li><a href=#edits-and-tables><span class=secno>4.7.6</span> Edits and tables</a></ol><li><a href=#embedded-content><span class=secno>4.8</span> Embedded content</a><ol><li><a href=#the-picture-element><span class=secno>4.8.1</span> The <code>picture</code> element</a><li><a href=#the-source-element><span class=secno>4.8.2</span> The <code>source</code> element</a><li><a href=#the-img-element><span class=secno>4.8.3</span> The <code>img</code> element</a><li><a href=#images><span class=secno>4.8.4</span> Images</a><ol><li><a href=#introduction-3><span class=secno>4.8.4.1</span> Introduction</a><ol><li><a href=#adaptive-images><span class=secno>4.8.4.1.1</span> Adaptive images</a></ol><li><a href=#attributes-common-to-source-and-img-elements><span class=secno>4.8.4.2</span> Attributes common to <code>source</code>,
  <code>img</code>, and <code>link</code> elements</a><ol><li><a href=#srcset-attributes><span class=secno>4.8.4.2.1</span> Srcset attributes</a><li><a href=#sizes-attributes><span class=secno>4.8.4.2.2</span> Sizes attributes</a></ol><li><a href=#images-processing-model><span class=secno>4.8.4.3</span> Processing model</a><ol><li><a href=#when-to-obtain-images><span class=secno>4.8.4.3.1</span> When to obtain images</a><li><a href=#reacting-to-dom-mutations><span class=secno>4.8.4.3.2</span> Reacting to DOM mutations</a><li><a href=#the-list-of-available-images><span class=secno>4.8.4.3.3</span> The list of available images</a><li><a href=#decoding-images><span class=secno>4.8.4.3.4</span> Decoding images</a><li><a href=#updating-the-image-data><span class=secno>4.8.4.3.5</span> Updating the image data</a><li><a href=#selecting-an-image-source><span class=secno>4.8.4.3.6</span> Selecting an image source</a><li><a href=#updating-the-source-set><span class=secno>4.8.4.3.7</span> Updating the source set</a><li><a href=#parsing-a-srcset-attribute><span class=secno>4.8.4.3.8</span> Parsing a srcset attribute</a><li><a href=#parsing-a-sizes-attribute><span class=secno>4.8.4.3.9</span> Parsing a sizes attribute</a><li><a href=#normalizing-the-source-densities><span class=secno>4.8.4.3.10</span> Normalizing the source densities</a><li><a href=#reacting-to-environment-changes><span class=secno>4.8.4.3.11</span> Reacting to environment changes</a></ol><li><a href=#alt><span class=secno>4.8.4.4</span> Requirements for providing text to act as an alternative for images</a><ol><li><a href=#general-guidelines><span class=secno>4.8.4.4.1</span> General guidelines</a><li><a href=#a-link-or-button-containing-nothing-but-the-image><span class=secno>4.8.4.4.2</span> A link or button containing nothing but the image</a><li><a href=#a-phrase-or-paragraph-with-an-alternative-graphical-representation:-charts,-diagrams,-graphs,-maps,-illustrations><span class=secno>4.8.4.4.3</span> A phrase or paragraph with an alternative graphical representation: charts, diagrams, graphs, maps, illustrations</a><li><a href=#a-short-phrase-or-label-with-an-alternative-graphical-representation:-icons,-logos><span class=secno>4.8.4.4.4</span> A short phrase or label with an alternative graphical representation: icons, logos</a><li><a href=#text-that-has-been-rendered-to-a-graphic-for-typographical-effect><span class=secno>4.8.4.4.5</span> Text that has been rendered to a graphic for typographical effect</a><li><a href=#a-graphical-representation-of-some-of-the-surrounding-text><span class=secno>4.8.4.4.6</span> A graphical representation of some of the surrounding text</a><li><a href=#ancillary-images><span class=secno>4.8.4.4.7</span> Ancillary images</a><li><a href="#a-purely-decorative-image-that-doesn't-add-any-information"><span class=secno>4.8.4.4.8</span> A purely decorative image that doesn't add any information</a><li><a href=#a-group-of-images-that-form-a-single-larger-picture-with-no-links><span class=secno>4.8.4.4.9</span> A group of images that form a single larger picture with no links</a><li><a href=#a-group-of-images-that-form-a-single-larger-picture-with-links><span class=secno>4.8.4.4.10</span> A group of images that form a single larger picture with links</a><li><a href=#a-key-part-of-the-content><span class=secno>4.8.4.4.11</span> A key part of the content</a><li><a href=#an-image-not-intended-for-the-user><span class=secno>4.8.4.4.12</span> An image not intended for the user</a><li><a href=#an-image-in-an-e-mail-or-private-document-intended-for-a-specific-person-who-is-known-to-be-able-to-view-images><span class=secno>4.8.4.4.13</span> An image in an email or private document intended for a specific person who is known to be able to view images</a><li><a href=#guidance-for-markup-generators><span class=secno>4.8.4.4.14</span> Guidance for markup generators</a><li><a href=#guidance-for-conformance-checkers><span class=secno>4.8.4.4.15</span> Guidance for conformance checkers</a></ol></ol><li><a href=#the-iframe-element><span class=secno>4.8.5</span> The
  <code>iframe</code> element</a><li><a href=#the-embed-element><span class=secno>4.8.6</span> The <code>embed</code> element</a><li><a href=#the-object-element><span class=secno>4.8.7</span> The <code>object</code> element</a><li><a href=#the-param-element><span class=secno>4.8.8</span> The <code>param</code> element</a><li><a href=#the-video-element><span class=secno>4.8.9</span> The <code>video</code> element</a><li><a href=#the-audio-element><span class=secno>4.8.10</span> The <code>audio</code> element</a><li><a href=#the-track-element><span class=secno>4.8.11</span> The <code>track</code> element</a><li><a href=#media-elements><span class=secno>4.8.12</span> Media elements</a><ol><li><a href=#error-codes><span class=secno>4.8.12.1</span> Error codes</a><li><a href=#location-of-the-media-resource><span class=secno>4.8.12.2</span> Location of the media resource</a><li><a href=#mime-types><span class=secno>4.8.12.3</span> MIME types</a><li><a href=#network-states><span class=secno>4.8.12.4</span> Network states</a><li><a href=#loading-the-media-resource><span class=secno>4.8.12.5</span> Loading the media resource</a><li><a href=#offsets-into-the-media-resource><span class=secno>4.8.12.6</span> Offsets into the media resource</a><li><a href=#ready-states><span class=secno>4.8.12.7</span> Ready states</a><li><a href=#playing-the-media-resource><span class=secno>4.8.12.8</span> Playing the media resource</a><li><a href=#seeking><span class=secno>4.8.12.9</span> Seeking</a><li><a href=#media-resources-with-multiple-media-tracks><span class=secno>4.8.12.10</span> Media resources with multiple media tracks</a><ol><li><a href=#audiotracklist-and-videotracklist-objects><span class=secno>4.8.12.10.1</span> <code>AudioTrackList</code> and <code>VideoTrackList</code> objects</a><li><a href=#selecting-specific-audio-and-video-tracks-declaratively><span class=secno>4.8.12.10.2</span> Selecting specific audio and video tracks declaratively</a></ol><li><a href=#timed-text-tracks><span class=secno>4.8.12.11</span> Timed text tracks</a><ol><li><a href=#text-track-model><span class=secno>4.8.12.11.1</span> Text track model</a><li><a href=#sourcing-in-band-text-tracks><span class=secno>4.8.12.11.2</span> Sourcing in-band text tracks</a><li><a href=#sourcing-out-of-band-text-tracks><span class=secno>4.8.12.11.3</span> Sourcing out-of-band text tracks</a><li><a href=#guidelines-for-exposing-cues-in-various-formats-as-text-track-cues><span class=secno>4.8.12.11.4</span> Guidelines for exposing cues in various formats as text track cues</a><li><a href=#text-track-api><span class=secno>4.8.12.11.5</span> Text track API</a><li><a href=#cue-events><span class=secno>4.8.12.11.6</span> Event handlers for objects of the text track APIs</a><li><a href=#best-practices-for-metadata-text-tracks><span class=secno>4.8.12.11.7</span> Best practices for metadata text tracks</a></ol><li><a href=#identifying-a-track-kind-through-a-url><span class=secno>4.8.12.12</span> Identifying a track kind through a URL</a><li><a href=#user-interface><span class=secno>4.8.12.13</span> User interface</a><li><a href=#time-ranges><span class=secno>4.8.12.14</span> Time ranges</a><li><a href=#the-trackevent-interface><span class=secno>4.8.12.15</span> The <code>TrackEvent</code> interface</a><li><a href=#mediaevents><span class=secno>4.8.12.16</span> Events summary</a><li><a href=#security-and-privacy-considerations><span class=secno>4.8.12.17</span> Security and privacy considerations</a><li><a href=#best-practices-for-authors-using-media-elements><span class=secno>4.8.12.18</span> Best practices for authors using media elements</a><li><a href=#best-practices-for-implementers-of-media-elements><span class=secno>4.8.12.19</span> Best practices for implementers of media elements</a></ol><li><a href=#the-map-element><span class=secno>4.8.13</span> The <code>map</code> element</a><li><a href=#the-area-element><span class=secno>4.8.14</span> The <code>area</code> element</a><li><a href=#image-maps><span class=secno>4.8.15</span> Image maps</a><ol><li><a href=#authoring><span class=secno>4.8.15.1</span> Authoring</a><li><a href=#image-map-processing-model><span class=secno>4.8.15.2</span> Processing model</a></ol><li><a href=#mathml><span class=secno>4.8.16</span> MathML</a><li><a href=#svg-0><span class=secno>4.8.17</span> SVG</a><li><a href=#dimension-attributes><span class=secno>4.8.18</span> Dimension attributes</a></ol><li><a href=#tables><span class=secno>4.9</span> Tabular data</a><ol><li><a href=#the-table-element><span class=secno>4.9.1</span> The <code>table</code> element</a><ol><li><a href=#table-descriptions-techniques><span class=secno>4.9.1.1</span> Techniques for describing tables</a><li><a href=#table-layout-techniques><span class=secno>4.9.1.2</span> Techniques for table design</a></ol><li><a href=#the-caption-element><span class=secno>4.9.2</span> The <code>caption</code> element</a><li><a href=#the-colgroup-element><span class=secno>4.9.3</span> The <code>colgroup</code> element</a><li><a href=#the-col-element><span class=secno>4.9.4</span> The <code>col</code> element</a><li><a href=#the-tbody-element><span class=secno>4.9.5</span> The <code>tbody</code> element</a><li><a href=#the-thead-element><span class=secno>4.9.6</span> The <code>thead</code> element</a><li><a href=#the-tfoot-element><span class=secno>4.9.7</span> The <code>tfoot</code> element</a><li><a href=#the-tr-element><span class=secno>4.9.8</span> The <code>tr</code> element</a><li><a href=#the-td-element><span class=secno>4.9.9</span> The <code>td</code> element</a><li><a href=#the-th-element><span class=secno>4.9.10</span> The <code>th</code> element</a><li><a href=#attributes-common-to-td-and-th-elements><span class=secno>4.9.11</span> Attributes common to <code>td</code> and <code>th</code> elements</a><li><a href=#table-processing-model><span class=secno>4.9.12</span> Processing model</a><ol><li><a href=#forming-a-table><span class=secno>4.9.12.1</span> Forming a table</a><li><a href=#header-and-data-cell-semantics><span class=secno>4.9.12.2</span> Forming relationships between data cells and header cells</a></ol><li><a href=#table-examples><span class=secno>4.9.13</span> Examples</a></ol><li><a href=#forms><span class=secno>4.10</span> Forms</a><ol><li><a href=#introduction-4><span class=secno>4.10.1</span> Introduction</a><ol><li><a href="#writing-a-form's-user-interface"><span class=secno>4.10.1.1</span> Writing a form's user interface</a><li><a href=#implementing-the-server-side-processing-for-a-form><span class=secno>4.10.1.2</span> Implementing the server-side processing for a form</a><li><a href=#configuring-a-form-to-communicate-with-a-server><span class=secno>4.10.1.3</span> Configuring a form to communicate with a server</a><li><a href=#client-side-form-validation><span class=secno>4.10.1.4</span> Client-side form validation</a><li><a href=#enabling-client-side-automatic-filling-of-form-controls><span class=secno>4.10.1.5</span> Enabling client-side automatic filling of form controls</a><li><a href=#improving-the-user-experience-on-mobile-devices><span class=secno>4.10.1.6</span> Improving the user experience on mobile devices</a><li><a href=#the-difference-between-the-field-type,-the-autofill-field-name,-and-the-input-modality><span class=secno>4.10.1.7</span> The difference between the field type, the autofill field name, and the input modality</a><li><a href=#input-author-notes><span class=secno>4.10.1.8</span> Date, time, and number formats</a></ol><li><a href=#categories><span class=secno>4.10.2</span> Categories</a><li><a href=#the-form-element><span class=secno>4.10.3</span> The <code>form</code> element</a><li><a href=#the-label-element><span class=secno>4.10.4</span> The <code>label</code> element</a><li><a href=#the-input-element><span class=secno>4.10.5</span> The <code>input</code> element</a><ol><li><a href=#states-of-the-type-attribute><span class=secno>4.10.5.1</span> States of the <code>type</code> attribute</a><ol><li><a href="#hidden-state-(type=hidden)"><span class=secno>4.10.5.1.1</span> Hidden state (<code>type=hidden</code>)</a><li><a href="#text-(type=text)-state-and-search-state-(type=search)"><span class=secno>4.10.5.1.2</span> Text (<code>type=text</code>) state and Search state (<code>type=search</code>)</a><li><a href="#telephone-state-(type=tel)"><span class=secno>4.10.5.1.3</span> Telephone state (<code>type=tel</code>)</a><li><a href="#url-state-(type=url)"><span class=secno>4.10.5.1.4</span> URL state (<code>type=url</code>)</a><li><a href="#email-state-(type=email)"><span class=secno>4.10.5.1.5</span> Email state (<code>type=email</code>)</a><li><a href="#password-state-(type=password)"><span class=secno>4.10.5.1.6</span> Password state (<code>type=password</code>)</a><li><a href="#date-state-(type=date)"><span class=secno>4.10.5.1.7</span> Date state (<code>type=date</code>)</a><li><a href="#month-state-(type=month)"><span class=secno>4.10.5.1.8</span> Month state (<code>type=month</code>)</a><li><a href="#week-state-(type=week)"><span class=secno>4.10.5.1.9</span> Week state (<code>type=week</code>)</a><li><a href="#time-state-(type=time)"><span class=secno>4.10.5.1.10</span> Time state (<code>type=time</code>)</a><li><a href="#local-date-and-time-state-(type=datetime-local)"><span class=secno>4.10.5.1.11</span> Local Date and Time state (<code>type=datetime-local</code>)</a><li><a href="#number-state-(type=number)"><span class=secno>4.10.5.1.12</span> Number state (<code>type=number</code>)</a><li><a href="#range-state-(type=range)"><span class=secno>4.10.5.1.13</span> Range state (<code>type=range</code>)</a><li><a href="#color-state-(type=color)"><span class=secno>4.10.5.1.14</span> Color state (<code>type=color</code>)</a><li><a href="#checkbox-state-(type=checkbox)"><span class=secno>4.10.5.1.15</span> Checkbox state (<code>type=checkbox</code>)</a><li><a href="#radio-button-state-(type=radio)"><span class=secno>4.10.5.1.16</span> Radio Button state (<code>type=radio</code>)</a><li><a href="#file-upload-state-(type=file)"><span class=secno>4.10.5.1.17</span> File Upload state (<code>type=file</code>)</a><li><a href="#submit-button-state-(type=submit)"><span class=secno>4.10.5.1.18</span> Submit Button state (<code>type=submit</code>)</a><li><a href="#image-button-state-(type=image)"><span class=secno>4.10.5.1.19</span> Image Button state (<code>type=image</code>)</a><li><a href="#reset-button-state-(type=reset)"><span class=secno>4.10.5.1.20</span> Reset Button state (<code>type=reset</code>)</a><li><a href="#button-state-(type=button)"><span class=secno>4.10.5.1.21</span> Button state (<code>type=button</code>)</a></ol><li><a href=#input-impl-notes><span class=secno>4.10.5.2</span> Implementation notes regarding localization of form controls</a><li><a href=#common-input-element-attributes><span class=secno>4.10.5.3</span> Common <code>input</code> element attributes</a><ol><li><a href=#the-maxlength-and-minlength-attributes><span class=secno>4.10.5.3.1</span> The <code>maxlength</code> and <code>minlength</code> attributes</a><li><a href=#the-size-attribute><span class=secno>4.10.5.3.2</span> The <code>size</code> attribute</a><li><a href=#the-readonly-attribute><span class=secno>4.10.5.3.3</span> The <code>readonly</code> attribute</a><li><a href=#the-required-attribute><span class=secno>4.10.5.3.4</span> The <code>required</code> attribute</a><li><a href=#the-multiple-attribute><span class=secno>4.10.5.3.5</span> The <code>multiple</code> attribute</a><li><a href=#the-pattern-attribute><span class=secno>4.10.5.3.6</span> The <code>pattern</code> attribute</a><li><a href=#the-min-and-max-attributes><span class=secno>4.10.5.3.7</span> The <code>min</code> and <code>max</code> attributes</a><li><a href=#the-step-attribute><span class=secno>4.10.5.3.8</span> The <code>step</code> attribute</a><li><a href=#the-list-attribute><span class=secno>4.10.5.3.9</span> The <code>list</code> attribute</a><li><a href=#the-placeholder-attribute><span class=secno>4.10.5.3.10</span> The <code>placeholder</code> attribute</a></ol><li><a href=#common-input-element-apis><span class=secno>4.10.5.4</span> Common <code>input</code> element APIs</a><li><a href=#common-input-element-events><span class=secno>4.10.5.5</span> Common event behaviors</a></ol><li><a href=#the-button-element><span class=secno>4.10.6</span> The <code>button</code> element</a><li><a href=#the-select-element><span class=secno>4.10.7</span> The <code>select</code> element</a><li><a href=#the-datalist-element><span class=secno>4.10.8</span> The <code>datalist</code> element</a><li><a href=#the-optgroup-element><span class=secno>4.10.9</span> The <code>optgroup</code> element</a><li><a href=#the-option-element><span class=secno>4.10.10</span> The <code>option</code> element</a><li><a href=#the-textarea-element><span class=secno>4.10.11</span> The <code>textarea</code> element</a><li><a href=#the-output-element><span class=secno>4.10.12</span> The <code>output</code> element</a><li><a href=#the-progress-element><span class=secno>4.10.13</span> The <code>progress</code> element</a><li><a href=#the-meter-element><span class=secno>4.10.14</span> The <code>meter</code> element</a><li><a href=#the-fieldset-element><span class=secno>4.10.15</span> The <code>fieldset</code> element</a><li><a href=#the-legend-element><span class=secno>4.10.16</span> The <code>legend</code> element</a><li><a href=#form-control-infrastructure><span class=secno>4.10.17</span> Form control infrastructure</a><ol><li><a href="#a-form-control's-value"><span class=secno>4.10.17.1</span> A form control's value</a><li><a href=#mutability><span class=secno>4.10.17.2</span> Mutability</a><li><a href=#association-of-controls-and-forms><span class=secno>4.10.17.3</span> Association of controls and forms</a></ol><li><a href=#attributes-common-to-form-controls><span class=secno>4.10.18</span> Attributes common to form controls</a><ol><li><a href=#naming-form-controls:-the-name-attribute><span class=secno>4.10.18.1</span> Naming form controls: the <code>name</code> attribute</a><li><a href=#submitting-element-directionality:-the-dirname-attribute><span class=secno>4.10.18.2</span> Submitting element directionality: the <code>dirname</code> attribute</a><li><a href=#limiting-user-input-length:-the-maxlength-attribute><span class=secno>4.10.18.3</span> Limiting user input length: the <code>maxlength</code> attribute</a><li><a href=#setting-minimum-input-length-requirements:-the-minlength-attribute><span class=secno>4.10.18.4</span> Setting minimum input length requirements: the <code>minlength</code> attribute</a><li><a href=#enabling-and-disabling-form-controls:-the-disabled-attribute><span class=secno>4.10.18.5</span> Enabling and disabling form controls: the <code>disabled</code> attribute</a><li><a href=#form-submission-attributes><span class=secno>4.10.18.6</span> Form submission attributes</a><li><a href=#autofill><span class=secno>4.10.18.7</span> Autofill</a><ol><li><a href=#autofilling-form-controls:-the-autocomplete-attribute><span class=secno>4.10.18.7.1</span> Autofilling form controls: the <code>autocomplete</code> attribute</a><li><a href=#autofill-processing-model><span class=secno>4.10.18.7.2</span> Processing model</a></ol></ol><li><a href=#textFieldSelection><span class=secno>4.10.19</span> APIs for the text control selections</a><li><a href=#constraints><span class=secno>4.10.20</span> Constraints</a><ol><li><a href=#definitions><span class=secno>4.10.20.1</span> Definitions</a><li><a href=#constraint-validation><span class=secno>4.10.20.2</span> Constraint validation</a><li><a href=#the-constraint-validation-api><span class=secno>4.10.20.3</span> The constraint validation API</a><li><a href=#security-forms><span class=secno>4.10.20.4</span> Security</a></ol><li><a href=#form-submission-2><span class=secno>4.10.21</span> Form submission</a><ol><li><a href=#introduction-5><span class=secno>4.10.21.1</span> Introduction</a><li><a href=#implicit-submission><span class=secno>4.10.21.2</span> Implicit submission</a><li><a href=#form-submission-algorithm><span class=secno>4.10.21.3</span> Form submission algorithm</a><li><a href=#constructing-form-data-set><span class=secno>4.10.21.4</span> Constructing the entry list</a><li><a href=#selecting-a-form-submission-encoding><span class=secno>4.10.21.5</span> Selecting a form submission encoding</a><li><a href=#url-encoded-form-data><span class=secno>4.10.21.6</span> URL-encoded form data</a><li><a href=#multipart-form-data><span class=secno>4.10.21.7</span> Multipart form data</a><li><a href=#plain-text-form-data><span class=secno>4.10.21.8</span> Plain text form data</a><li><a href=#the-submitevent-interface><span class=secno>4.10.21.9</span> The <code>SubmitEvent</code> interface</a><li><a href=#the-formdataevent-interface><span class=secno>4.10.21.10</span> The <code>FormDataEvent</code> interface</a></ol><li><a href=#resetting-a-form><span class=secno>4.10.22</span> Resetting a form</a></ol><li><a href=#interactive-elements><span class=secno>4.11</span> Interactive elements</a><ol><li><a href=#the-details-element><span class=secno>4.11.1</span> The <code>details</code> element</a><li><a href=#the-summary-element><span class=secno>4.11.2</span> The <code>summary</code> element</a><li><a href=#commands><span class=secno>4.11.3</span> Commands</a><ol><li><a href=#facets-2><span class=secno>4.11.3.1</span> Facets</a><li><a href=#using-the-a-element-to-define-a-command><span class=secno>4.11.3.2</span> Using the <code>a</code> element to define a command</a><li><a href=#using-the-button-element-to-define-a-command><span class=secno>4.11.3.3</span> Using the <code>button</code> element to define a command</a><li><a href=#using-the-input-element-to-define-a-command><span class=secno>4.11.3.4</span> Using the <code>input</code> element to define a command</a><li><a href=#using-the-option-element-to-define-a-command><span class=secno>4.11.3.5</span> Using the <code>option</code> element to define a command</a><li><a href=#using-the-accesskey-attribute-on-a-legend-element-to-define-a-command><span class=secno>4.11.3.6</span> Using the <code>accesskey</code> attribute
  on a <code>legend</code> element to define a command</a><li><a href=#using-the-accesskey-attribute-to-define-a-command-on-other-elements><span class=secno>4.11.3.7</span> Using the <code>accesskey</code>
  attribute to define a command on other elements</a></ol><li><a href=#the-dialog-element><span class=secno>4.11.4</span> The <code>dialog</code> element</a></ol><li><a href=#scripting-3><span class=secno>4.12</span> Scripting</a><ol><li><a href=#the-script-element><span class=secno>4.12.1</span> The <code>script</code> element</a><ol><li><a href=#script-processing-model><span class=secno>4.12.1.1</span> Processing model</a><li><a href=#scriptingLanguages><span class=secno>4.12.1.2</span> Scripting languages</a><li><a href=#restrictions-for-contents-of-script-elements><span class=secno>4.12.1.3</span> Restrictions for contents of <code>script</code> elements</a><li><a href=#inline-documentation-for-external-scripts><span class=secno>4.12.1.4</span> Inline documentation for external scripts</a><li><a href=#scriptTagXSLT><span class=secno>4.12.1.5</span> Interaction of <code>script</code> elements and XSLT</a></ol><li><a href=#the-noscript-element><span class=secno>4.12.2</span> The <code>noscript</code> element</a><li><a href=#the-template-element><span class=secno>4.12.3</span> The <code>template</code> element</a><ol><li><a href=#template-XSLT-XPath><span class=secno>4.12.3.1</span> Interaction of <code>template</code> elements with XSLT and XPath</a></ol><li><a href=#the-slot-element><span class=secno>4.12.4</span> The <code>slot</code> element</a><li><a href=#the-canvas-element><span class=secno>4.12.5</span> The <code>canvas</code> element</a><ol><li><a href=#2dcontext><span class=secno>4.12.5.1</span> The 2D rendering context</a><ol><li><a href=#implementation-notes><span class=secno>4.12.5.1.1</span> Implementation notes</a><li><a href=#the-canvas-state><span class=secno>4.12.5.1.2</span> The canvas state</a><li><a href=#line-styles><span class=secno>4.12.5.1.3</span> Line styles</a><li><a href=#text-styles><span class=secno>4.12.5.1.4</span> Text styles</a><li><a href=#building-paths><span class=secno>4.12.5.1.5</span> Building paths</a><li><a href=#path2d-objects><span class=secno>4.12.5.1.6</span> <code>Path2D</code> objects</a><li><a href=#transformations><span class=secno>4.12.5.1.7</span> Transformations</a><li><a href=#image-sources-for-2d-rendering-contexts><span class=secno>4.12.5.1.8</span> Image sources for 2D rendering contexts</a><li><a href=#fill-and-stroke-styles><span class=secno>4.12.5.1.9</span> Fill and stroke styles</a><li><a href=#drawing-rectangles-to-the-bitmap><span class=secno>4.12.5.1.10</span> Drawing rectangles to the bitmap</a><li><a href=#drawing-text-to-the-bitmap><span class=secno>4.12.5.1.11</span> Drawing text to the bitmap</a><li><a href=#drawing-paths-to-the-canvas><span class=secno>4.12.5.1.12</span> Drawing paths to the canvas</a><li><a href=#drawing-focus-rings-and-scrolling-paths-into-view><span class=secno>4.12.5.1.13</span> Drawing focus rings and scrolling paths into view</a><li><a href=#drawing-images><span class=secno>4.12.5.1.14</span> Drawing images</a><li><a href=#pixel-manipulation><span class=secno>4.12.5.1.15</span> Pixel manipulation</a><li><a href=#compositing><span class=secno>4.12.5.1.16</span> Compositing</a><li><a href=#image-smoothing><span class=secno>4.12.5.1.17</span> Image smoothing</a><li><a href=#shadows><span class=secno>4.12.5.1.18</span> Shadows</a><li><a href=#filters><span class=secno>4.12.5.1.19</span> Filters</a><li><a href=#working-with-externally-defined-svg-filters><span class=secno>4.12.5.1.20</span> Working with externally-defined SVG filters</a><li><a href=#drawing-model><span class=secno>4.12.5.1.21</span> Drawing model</a><li><a href=#best-practices><span class=secno>4.12.5.1.22</span> Best practices</a><li><a href=#examples><span class=secno>4.12.5.1.23</span> Examples</a></ol><li><a href=#the-imagebitmap-rendering-context><span class=secno>4.12.5.2</span> The <code>ImageBitmap</code> rendering context</a><ol><li><a href=#introduction-6><span class=secno>4.12.5.2.1</span> Introduction</a><li><a href=#the-imagebitmaprenderingcontext-interface><span class=secno>4.12.5.2.2</span> The <code>ImageBitmapRenderingContext</code> interface</a></ol><li><a href=#the-offscreencanvas-interface><span class=secno>4.12.5.3</span> The <code>OffscreenCanvas</code> interface</a><ol><li><a href=#the-offscreen-2d-rendering-context><span class=secno>4.12.5.3.1</span> The offscreen 2D rendering context</a></ol><li><a href=#colour-spaces-and-colour-correction><span class=secno>4.12.5.4</span> Color spaces and color correction</a><li><a href=#serialising-bitmaps-to-a-file><span class=secno>4.12.5.5</span> Serializing bitmaps to a file</a><li><a href=#security-with-canvas-elements><span class=secno>4.12.5.6</span> Security with <code>canvas</code> elements</a></ol></ol><li><a href=#custom-elements><span class=secno>4.13</span> Custom elements</a><ol><li><a href=#custom-elements-intro><span class=secno>4.13.1</span> Introduction</a><ol><li><a href=#custom-elements-autonomous-example><span class=secno>4.13.1.1</span> Creating an autonomous custom element</a><li><a href=#custom-elements-face-example><span class=secno>4.13.1.2</span> Creating a form-associated custom element</a><li><a href=#custom-elements-accessibility-example><span class=secno>4.13.1.3</span> Creating a custom element with default accessible roles, states, and properties</a><li><a href=#custom-elements-customized-builtin-example><span class=secno>4.13.1.4</span> Creating a customized built-in element</a><li><a href=#custom-elements-autonomous-drawbacks><span class=secno>4.13.1.5</span> Drawbacks of autonomous custom elements</a><li><a href=#custom-elements-upgrades-examples><span class=secno>4.13.1.6</span> Upgrading elements after their creation</a></ol><li><a href=#custom-element-conformance><span class=secno>4.13.2</span> Requirements for custom element constructors and
  reactions</a><li><a href=#custom-elements-core-concepts><span class=secno>4.13.3</span> Core concepts</a><li><a href=#custom-elements-api><span class=secno>4.13.4</span> The <code>CustomElementRegistry</code> interface</a><li><a href=#upgrades><span class=secno>4.13.5</span> Upgrades</a><li><a href=#custom-element-reactions><span class=secno>4.13.6</span> Custom element reactions</a><li><a href=#element-internals><span class=secno>4.13.7</span> Element internals</a><ol><li><a href=#the-elementinternals-interface><span class=secno>4.13.7.1</span> The <code>ElementInternals</code> interface</a><li><a href=#shadow-root-access><span class=secno>4.13.7.2</span> Shadow root access</a><li><a href=#form-associated-custom-elements><span class=secno>4.13.7.3</span> Form-associated custom elements</a><li><a href=#accessibility-semantics><span class=secno>4.13.7.4</span> Accessibility semantics</a></ol></ol><li><a href=#common-idioms><span class=secno>4.14</span> Common idioms without dedicated elements</a><ol><li><a href=#rel-up><span class=secno>4.14.1</span> Bread crumb navigation</a><li><a href=#tag-clouds><span class=secno>4.14.2</span> Tag clouds</a><li><a href=#conversations><span class=secno>4.14.3</span> Conversations</a><li><a href=#footnotes><span class=secno>4.14.4</span> Footnotes</a></ol><li><a href=#disabled-elements><span class=secno>4.15</span> Disabled elements</a><li><a href=#selectors><span class=secno>4.16</span> Matching HTML elements using selectors and CSS</a><ol><li><a href="#case-sensitivity-of-the-css-'attr()'-function"><span class=secno>4.16.1</span> Case-sensitivity of the CSS 'attr()' function</a><li><a href=#case-sensitivity-of-selectors><span class=secno>4.16.2</span> Case-sensitivity of selectors</a><li><a href=#pseudo-classes><span class=secno>4.16.3</span> Pseudo-classes</a></ol></ol><li id=toc-microdata><a href=#microdata><span class=secno>5</span> Microdata</a><ol><li><a href=#introduction-7><span class=secno>5.1</span> Introduction</a><ol><li><a href=#overview><span class=secno>5.1.1</span> Overview</a><li><a href=#the-basic-syntax><span class=secno>5.1.2</span> The basic syntax</a><li><a href=#typed-items><span class=secno>5.1.3</span> Typed items</a><li><a href=#global-identifiers-for-items><span class=secno>5.1.4</span> Global identifiers for items</a><li><a href=#selecting-names-when-defining-vocabularies><span class=secno>5.1.5</span> Selecting names when defining vocabularies</a></ol><li><a href=#encoding-microdata><span class=secno>5.2</span> Encoding microdata</a><ol><li><a href=#the-microdata-model><span class=secno>5.2.1</span> The microdata model</a><li><a href=#items><span class=secno>5.2.2</span> Items</a><li><a href=#names:-the-itemprop-attribute><span class=secno>5.2.3</span> Names: the <code>itemprop</code> attribute</a><li><a href=#values><span class=secno>5.2.4</span> Values</a><li><a href=#associating-names-with-items><span class=secno>5.2.5</span> Associating names with items</a><li><a href=#microdata-and-other-namespaces><span class=secno>5.2.6</span> Microdata and other namespaces</a></ol><li><a href=#mdvocabs><span class=secno>5.3</span> Sample microdata vocabularies</a><ol><li><a href=#vcard><span class=secno>5.3.1</span> vCard</a><ol><li><a href=#conversion-to-vcard><span class=secno>5.3.1.1</span> Conversion to vCard</a><li><a href=#examples-2><span class=secno>5.3.1.2</span> Examples</a></ol><li><a href=#vevent><span class=secno>5.3.2</span> vEvent</a><ol><li><a href=#conversion-to-icalendar><span class=secno>5.3.2.1</span> Conversion to iCalendar</a><li><a href=#examples-3><span class=secno>5.3.2.2</span> Examples</a></ol><li><a href=#licensing-works><span class=secno>5.3.3</span> Licensing works</a><ol><li><a href=#examples-4><span class=secno>5.3.3.1</span> Examples</a></ol></ol><li><a href=#converting-html-to-other-formats><span class=secno>5.4</span> Converting HTML to other formats</a><ol><li><a href=#json><span class=secno>5.4.1</span> JSON</a></ol></ol><li id=toc-editing><a href=#editing><span class=secno>6</span> User interaction</a><ol><li><a href=#the-hidden-attribute><span class=secno>6.1</span> The <code>hidden</code> attribute</a><li><a href=#inert-subtrees><span class=secno>6.2</span> Inert subtrees</a><li><a href=#tracking-user-activation><span class=secno>6.3</span> Tracking user activation</a><ol><li><a href=#user-activation-data-model><span class=secno>6.3.1</span> Data model</a><li><a href=#user-activation-processing-model><span class=secno>6.3.2</span> Processing model</a><li><a href=#user-activation-gated-apis><span class=secno>6.3.3</span> APIs gated by user activation</a></ol><li><a href=#activation><span class=secno>6.4</span> Activation behavior of elements</a><li><a href=#focus><span class=secno>6.5</span> Focus</a><ol><li><a href=#introduction-8><span class=secno>6.5.1</span> Introduction</a><li><a href=#data-model><span class=secno>6.5.2</span> Data model</a><li><a href=#the-tabindex-attribute><span class=secno>6.5.3</span> The <code>tabindex</code> attribute</a><li><a href=#focus-processing-model><span class=secno>6.5.4</span> Processing model</a><li><a href=#sequential-focus-navigation><span class=secno>6.5.5</span> Sequential focus navigation</a><li><a href=#focus-management-apis><span class=secno>6.5.6</span> Focus management APIs</a><li><a href=#the-autofocus-attribute><span class=secno>6.5.7</span> The <code>autofocus</code> attribute</a></ol><li><a href=#assigning-keyboard-shortcuts><span class=secno>6.6</span> Assigning keyboard shortcuts</a><ol><li><a href=#introduction-9><span class=secno>6.6.1</span> Introduction</a><li><a href=#the-accesskey-attribute><span class=secno>6.6.2</span> The <code>accesskey</code>
  attribute</a><li><a href=#keyboard-shortcuts-processing-model><span class=secno>6.6.3</span> Processing
  model</a></ol><li><a href=#editing-2><span class=secno>6.7</span> Editing</a><ol><li><a href=#contenteditable><span class=secno>6.7.1</span> Making document regions editable: The <code>contenteditable</code> content attribute</a><li><a href=#making-entire-documents-editable:-the-designmode-idl-attribute><span class=secno>6.7.2</span> Making entire documents
  editable: the <code>designMode</code> getter and setter</a><li><a href=#best-practices-for-in-page-editors><span class=secno>6.7.3</span> Best practices for in-page editors</a><li><a href=#editing-apis><span class=secno>6.7.4</span> Editing APIs</a><li><a href=#spelling-and-grammar-checking><span class=secno>6.7.5</span> Spelling and grammar checking</a><li><a href=#autocapitalization><span class=secno>6.7.6</span> Autocapitalization</a><li><a href=#input-modalities:-the-inputmode-attribute><span class=secno>6.7.7</span> Input modalities: the <code>inputmode</code> attribute</a><li><a href=#input-modalities:-the-enterkeyhint-attribute><span class=secno>6.7.8</span> Input modalities: the <code>enterkeyhint</code>
  attribute</a></ol><li><a href=#find-in-page><span class=secno>6.8</span> Find-in-page</a><ol><li><a href=#introduction-10><span class=secno>6.8.1</span> Introduction</a><li><a href=#interaction-with-selection><span class=secno>6.8.2</span> Interaction with selection</a></ol><li><a href=#dnd><span class=secno>6.9</span> Drag and drop</a><ol><li><a href=#event-drag><span class=secno>6.9.1</span> Introduction</a><li><a href=#the-drag-data-store><span class=secno>6.9.2</span> The drag data store</a><li><a href=#the-datatransfer-interface><span class=secno>6.9.3</span> The <code>DataTransfer</code> interface</a><ol><li><a href=#the-datatransferitemlist-interface><span class=secno>6.9.3.1</span> The <code>DataTransferItemList</code> interface</a><li><a href=#the-datatransferitem-interface><span class=secno>6.9.3.2</span> The <code>DataTransferItem</code> interface</a></ol><li><a href=#the-dragevent-interface><span class=secno>6.9.4</span> The <code>DragEvent</code> interface</a><li><a href=#drag-and-drop-processing-model><span class=secno>6.9.5</span> Processing model</a><li><a href=#dndevents><span class=secno>6.9.6</span> Events summary</a><li><a href=#the-draggable-attribute><span class=secno>6.9.7</span> The <code>draggable</code> attribute</a><li><a href=#security-risks-in-the-drag-and-drop-model><span class=secno>6.9.8</span> Security risks in the drag-and-drop model</a></ol></ol><li id=toc-browsers><a href=#browsers><span class=secno>7</span> Loading web pages</a><ol><li><a href=#windows><span class=secno>7.1</span> Browsing contexts</a><ol><li><a href=#creating-browsing-contexts><span class=secno>7.1.1</span> Creating browsing contexts</a><li><a href=#nested-browsing-contexts><span class=secno>7.1.2</span> Related browsing contexts</a><ol><li><a href=#navigating-nested-browsing-contexts-in-the-dom><span class=secno>7.1.2.1</span> Navigating related browsing contexts in
  the DOM</a></ol><li><a href=#security-nav><span class=secno>7.1.3</span> Security</a><li><a href=#groupings-of-browsing-contexts><span class=secno>7.1.4</span> Groupings of browsing contexts</a><li><a href=#browsing-context-names><span class=secno>7.1.5</span> Browsing context names</a></ol><li><a href=#cross-origin-objects><span class=secno>7.2</span> Security infrastructure for <code>Window</code>,
  <code>WindowProxy</code>, and <code>Location</code> objects</a><ol><li><a href=#integration-with-idl><span class=secno>7.2.1</span> Integration with IDL</a><li><a href=#shared-internal-slot:-crossoriginpropertydescriptormap><span class=secno>7.2.2</span> Shared internal slot: [[CrossOriginPropertyDescriptorMap]]</a><li><a href=#shared-abstract-operations><span class=secno>7.2.3</span> Shared abstract operations</a><ol><li><a href=#crossoriginproperties-(-o-)><span class=secno>7.2.3.1</span> CrossOriginProperties ( <var>O</var> )</a><li><a href=#crossoriginpropertyfallback-(-p-)><span class=secno>7.2.3.2</span> CrossOriginPropertyFallback ( <var>P</var> )</a><li><a href=#isplatformobjectsameorigin-(-o-)><span class=secno>7.2.3.3</span> IsPlatformObjectSameOrigin ( <var>O</var> )</a><li><a href=#crossorigingetownpropertyhelper-(-o,-p-)><span class=secno>7.2.3.4</span> CrossOriginGetOwnPropertyHelper ( <var>O</var>, <var>P</var> )</a><li><a href=#crossoriginget-(-o,-p,-receiver-)><span class=secno>7.2.3.5</span> CrossOriginGet ( <var>O</var>, <var>P</var>, <var>Receiver</var> )</a><li><a href=#crossoriginset-(-o,-p,-v,-receiver-)><span class=secno>7.2.3.6</span> CrossOriginSet ( <var>O</var>, <var>P</var>, <var>V</var>,
  <var>Receiver</var> )</a><li><a href=#crossoriginownpropertykeys-(-o-)><span class=secno>7.2.3.7</span> CrossOriginOwnPropertyKeys ( <var>O</var> )</a></ol></ol><li><a href=#the-window-object><span class=secno>7.3</span> The <code>Window</code> object</a><ol><li><a href=#apis-for-creating-and-navigating-browsing-contexts-by-name><span class=secno>7.3.1</span> APIs for creating and navigating browsing contexts by name</a><li><a href=#accessing-other-browsing-contexts><span class=secno>7.3.2</span> Accessing other browsing contexts</a><li><a href=#named-access-on-the-window-object><span class=secno>7.3.3</span> Named access on the <code>Window</code> object</a><li><a href=#garbage-collection-and-browsing-contexts><span class=secno>7.3.4</span> Discarding browsing contexts</a><li><a href=#closing-browsing-contexts><span class=secno>7.3.5</span> Closing browsing contexts</a><li><a href=#browser-interface-elements><span class=secno>7.3.6</span> Browser interface elements</a><li><a href=#script-settings-for-window-objects><span class=secno>7.3.7</span> Script settings for <code>Window</code> objects</a></ol><li><a href=#the-windowproxy-exotic-object><span class=secno>7.4</span> The <code>WindowProxy</code> exotic object</a><ol><li><a href=#windowproxy-getprototypeof><span class=secno>7.4.1</span> [[GetPrototypeOf]] ( )</a><li><a href=#windowproxy-setprototypeof><span class=secno>7.4.2</span> [[SetPrototypeOf]] ( <var>V</var> )</a><li><a href=#windowproxy-isextensible><span class=secno>7.4.3</span> [[IsExtensible]] ( )</a><li><a href=#windowproxy-preventextensions><span class=secno>7.4.4</span> [[PreventExtensions]] ( )</a><li><a href=#windowproxy-getownproperty><span class=secno>7.4.5</span> [[GetOwnProperty]] ( <var>P</var> )</a><li><a href=#windowproxy-defineownproperty><span class=secno>7.4.6</span> [[DefineOwnProperty]] ( <var>P</var>, <var>Desc</var>
  )</a><li><a href=#windowproxy-get><span class=secno>7.4.7</span> [[Get]] ( <var>P</var>, <var>Receiver</var> )</a><li><a href=#windowproxy-set><span class=secno>7.4.8</span> [[Set]] ( <var>P</var>, <var>V</var>, <var>Receiver</var> )</a><li><a href=#windowproxy-delete><span class=secno>7.4.9</span> [[Delete]] ( <var>P</var> )</a><li><a href=#windowproxy-ownpropertykeys><span class=secno>7.4.10</span> [[OwnPropertyKeys]] ( )</a></ol><li><a href=#origin><span class=secno>7.5</span> Origin</a><ol><li><a href=#sites><span class=secno>7.5.1</span> Sites</a><li><a href=#relaxing-the-same-origin-restriction><span class=secno>7.5.2</span> Relaxing the same-origin restriction</a><li><a href=#origin-isolation><span class=secno>7.5.3</span> Origin isolation</a></ol><li><a href=#sandboxing><span class=secno>7.6</span> Sandboxing</a><li><a href=#cross-origin-opener-policies><span class=secno>7.7</span> Cross-origin opener policies</a><ol><li><a href=#the-headers><span class=secno>7.7.1</span> The headers</a><li><a href=#browsing-context-group-switches-due-to-cross-origin-opener-policy><span class=secno>7.7.2</span> Browsing context group switches due to cross-origin opener policy</a><li><a href=#reporting><span class=secno>7.7.3</span> Reporting</a></ol><li><a href=#coep><span class=secno>7.8</span> Cross-origin embedder policies</a><ol><li><a href=#the-headers-2><span class=secno>7.8.1</span> The headers</a><li><a href=#embedder-policy-checks><span class=secno>7.8.2</span> Embedder policy checks</a></ol><li><a href=#history><span class=secno>7.9</span> Session history and navigation</a><ol><li><a href=#browsing-sessions><span class=secno>7.9.1</span> Browsing sessions</a><li><a href=#the-session-history-of-browsing-contexts><span class=secno>7.9.2</span> The session history of browsing contexts</a><li><a href=#the-history-interface><span class=secno>7.9.3</span> The <code>History</code> interface</a><li><a href=#history-notes><span class=secno>7.9.4</span> Implementation notes for session history</a><li><a href=#the-location-interface><span class=secno>7.9.5</span> The <code>Location</code> interface</a><ol><li><a href=#location-getprototypeof><span class=secno>7.9.5.1</span> [[GetPrototypeOf]] ( )</a><li><a href=#location-setprototypeof><span class=secno>7.9.5.2</span> [[SetPrototypeOf]] ( <var>V</var> )</a><li><a href=#location-isextensible><span class=secno>7.9.5.3</span> [[IsExtensible]] ( )</a><li><a href=#location-preventextensions><span class=secno>7.9.5.4</span> [[PreventExtensions]] ( )</a><li><a href=#location-getownproperty><span class=secno>7.9.5.5</span> [[GetOwnProperty]] ( <var>P</var> )</a><li><a href=#location-defineownproperty><span class=secno>7.9.5.6</span> [[DefineOwnProperty]] ( <var>P</var>, <var>Desc</var> )</a><li><a href=#location-get><span class=secno>7.9.5.7</span> [[Get]] ( <var>P</var>, <var>Receiver</var> )</a><li><a href=#location-set><span class=secno>7.9.5.8</span> [[Set]] ( <var>P</var>, <var>V</var>, <var>Receiver</var> )</a><li><a href=#location-delete><span class=secno>7.9.5.9</span> [[Delete]] ( <var>P</var> )</a><li><a href=#location-ownpropertykeys><span class=secno>7.9.5.10</span> [[OwnPropertyKeys]] ( )</a></ol></ol><li><a href=#browsing-the-web><span class=secno>7.10</span> Browsing the web</a><ol><li><a href=#navigating-across-documents><span class=secno>7.10.1</span> Navigating across documents</a><li><a href=#read-html><span class=secno>7.10.2</span> Page load processing model for HTML files</a><li><a href=#read-xml><span class=secno>7.10.3</span> Page load processing model for XML files</a><li><a href=#read-text><span class=secno>7.10.4</span> Page load processing model for text files</a><li><a href=#read-multipart-x-mixed-replace><span class=secno>7.10.5</span> Page load processing model for <code>multipart/x-mixed-replace</code> resources</a><li><a href=#read-media><span class=secno>7.10.6</span> Page load processing model for media</a><li><a href=#read-plugin><span class=secno>7.10.7</span> Page load processing model for content that uses plugins</a><li><a href=#read-ua-inline><span class=secno>7.10.8</span> Page load processing model for inline
  content that doesn't have a DOM</a><li><a href=#scroll-to-fragid><span class=secno>7.10.9</span> Navigating to a fragment</a><li><a href=#history-traversal><span class=secno>7.10.10</span> History traversal</a><ol><li><a href=#persisted-user-state-restoration><span class=secno>7.10.10.1</span> Persisted history entry state</a><li><a href=#the-popstateevent-interface><span class=secno>7.10.10.2</span> The <code>PopStateEvent</code> interface</a><li><a href=#the-hashchangeevent-interface><span class=secno>7.10.10.3</span> The <code>HashChangeEvent</code> interface</a><li><a href=#the-pagetransitionevent-interface><span class=secno>7.10.10.4</span> The <code>PageTransitionEvent</code> interface</a></ol><li><a href=#loading-documents><span class=secno>7.10.11</span> Loading documents</a><li><a href=#unloading-documents><span class=secno>7.10.12</span> Unloading documents</a><ol><li><a href=#the-beforeunloadevent-interface><span class=secno>7.10.12.1</span> The <code>BeforeUnloadEvent</code> interface</a></ol><li><a href=#aborting-a-document-load><span class=secno>7.10.13</span> Aborting a document load</a><li><a href=#the-x-frame-options-header><span class=secno>7.10.14</span> The `<code>X-Frame-Options</code>` header</a></ol><li><a href=#offline><span class=secno>7.11</span> Offline web applications</a><ol><li><a href=#introduction-11><span class=secno>7.11.1</span> Introduction</a><ol><li><a href=#supporting-offline-caching-for-legacy-applications><span class=secno>7.11.1.1</span> Supporting offline caching for legacy applications</a><li><a href=#appcacheevents><span class=secno>7.11.1.2</span> Events summary</a></ol><li><a href=#appcache><span class=secno>7.11.2</span> Application caches</a><li><a href=#manifests><span class=secno>7.11.3</span> The cache manifest syntax</a><ol><li><a href=#some-sample-manifests><span class=secno>7.11.3.1</span> Some sample manifests</a><li><a href=#writing-cache-manifests><span class=secno>7.11.3.2</span> Writing cache manifests</a><li><a href=#parsing-cache-manifests><span class=secno>7.11.3.3</span> Parsing cache manifests</a></ol><li><a href=#downloading-or-updating-an-application-cache><span class=secno>7.11.4</span> Downloading or updating an application cache</a><li><a href=#the-application-cache-selection-algorithm><span class=secno>7.11.5</span> The application cache selection algorithm</a><li><a href=#changesToNetworkingModel><span class=secno>7.11.6</span> Changes to the networking model</a><li><a href=#expiring-application-caches><span class=secno>7.11.7</span> Expiring application caches</a><li><a href=#disk-space><span class=secno>7.11.8</span> Disk space</a><li><a href=#security-concerns-with-offline-applications-caches><span class=secno>7.11.9</span> Security concerns with offline applications caches</a><li><a href=#application-cache-api><span class=secno>7.11.10</span> Application cache API</a><li><a href=#navigator.online><span class=secno>7.11.11</span> Browser state</a></ol></ol><li id=toc-webappapis><a href=#webappapis><span class=secno>8</span> Web application APIs</a><ol><li><a href=#scripting><span class=secno>8.1</span> Scripting</a><ol><li><a href=#introduction-12><span class=secno>8.1.1</span> Introduction</a><li><a href=#agents-and-agent-clusters><span class=secno>8.1.2</span> Agents and agent clusters</a><ol><li><a href=#integration-with-the-javascript-agent-formalism><span class=secno>8.1.2.1</span> Integration with the JavaScript agent formalism</a><li><a href=#integration-with-the-javascript-agent-cluster-formalism><span class=secno>8.1.2.2</span> Integration with the JavaScript agent cluster formalism</a></ol><li><a href=#realms-and-their-counterparts><span class=secno>8.1.3</span> Realms and their counterparts</a><ol><li><a href=#environments><span class=secno>8.1.3.1</span> Environments</a><li><a href=#environment-settings-objects><span class=secno>8.1.3.2</span> Environment settings objects</a><li><a href=#realms-settings-objects-global-objects><span class=secno>8.1.3.3</span> Realms, settings objects, and global objects</a><ol><li><a href=#entry><span class=secno>8.1.3.3.1</span> Entry</a><li><a href=#incumbent><span class=secno>8.1.3.3.2</span> Incumbent</a><li><a href=#current><span class=secno>8.1.3.3.3</span> Current</a><li><a href=#relevant><span class=secno>8.1.3.3.4</span> Relevant</a></ol><li><a href=#enabling-and-disabling-scripting><span class=secno>8.1.3.4</span> Enabling and disabling scripting</a><li><a href=#secure-contexts><span class=secno>8.1.3.5</span> Secure contexts</a></ol><li><a href=#scripting-processing-model><span class=secno>8.1.4</span> Script processing
  model</a><ol><li><a href=#script-structs><span class=secno>8.1.4.1</span> Scripts</a><li><a href=#fetching-scripts><span class=secno>8.1.4.2</span> Fetching scripts</a><li><a href=#creating-scripts><span class=secno>8.1.4.3</span> Creating scripts</a><li><a href=#calling-scripts><span class=secno>8.1.4.4</span> Calling scripts</a><li><a href=#killing-scripts><span class=secno>8.1.4.5</span> Killing scripts</a><li><a href=#runtime-script-errors><span class=secno>8.1.4.6</span> Runtime script errors</a><li><a href=#unhandled-promise-rejections><span class=secno>8.1.4.7</span> Unhandled promise rejections</a></ol><li><a href=#javascript-specification-host-hooks><span class=secno>8.1.5</span> JavaScript specification host hooks</a><ol><li><a href=#hostenqueuepromisejob><span class=secno>8.1.5.1</span> HostEnqueuePromiseJob(<var>job</var>,
  <var>realm</var>)</a><li><a href=#hostensurecancompilestrings(callerrealm,-calleerealm)><span class=secno>8.1.5.2</span> HostEnsureCanCompileStrings(<var>callerRealm</var>, <var>calleeRealm</var>)</a><li><a href=#the-hostpromiserejectiontracker-implementation><span class=secno>8.1.5.3</span> HostPromiseRejectionTracker(<var>promise</var>, <var>operation</var>)</a><li><a href=#integration-with-the-javascript-module-system><span class=secno>8.1.5.4</span> Module-related host hooks</a><ol><li><a href=#hostgetimportmetaproperties><span class=secno>8.1.5.4.1</span> HostGetImportMetaProperties(<var>moduleRecord</var>)</a><li><a href=#hostimportmoduledynamically(referencingscriptormodule,-specifier,-promisecapability)><span class=secno>8.1.5.4.2</span> HostImportModuleDynamically(<var>referencingScriptOrModule</var>,
  <var>specifier</var>, <var>promiseCapability</var>)</a><li><a href=#hostresolveimportedmodule(referencingscriptormodule,-specifier)><span class=secno>8.1.5.4.3</span> HostResolveImportedModule(<var>referencingScriptOrModule</var>,
  <var>specifier</var>)</a></ol></ol><li><a href=#event-loops><span class=secno>8.1.6</span> Event loops</a><ol><li><a href=#definitions-3><span class=secno>8.1.6.1</span> Definitions</a><li><a href=#queuing-tasks><span class=secno>8.1.6.2</span> Queuing tasks</a><li><a href=#event-loop-processing-model><span class=secno>8.1.6.3</span> Processing model</a><li><a href=#generic-task-sources><span class=secno>8.1.6.4</span> Generic task sources</a><li><a href=#event-loop-for-spec-authors><span class=secno>8.1.6.5</span> Dealing with the event loop from other specifications</a></ol><li><a href=#events><span class=secno>8.1.7</span> Events</a><ol><li><a href=#event-handler-attributes><span class=secno>8.1.7.1</span> Event handlers</a><li><a href=#event-handlers-on-elements,-document-objects,-and-window-objects><span class=secno>8.1.7.2</span> Event handlers on elements, <code>Document</code> objects, and <code>Window</code> objects</a><ol><li><a href=#idl-definitions><span class=secno>8.1.7.2.1</span> IDL definitions</a></ol><li><a href=#event-firing><span class=secno>8.1.7.3</span> Event firing</a></ol></ol><li><a href=#windoworworkerglobalscope-mixin><span class=secno>8.2</span> The <code>WindowOrWorkerGlobalScope</code> mixin</a><li><a href=#atob><span class=secno>8.3</span> Base64 utility methods</a><li><a href=#dynamic-markup-insertion><span class=secno>8.4</span> Dynamic markup insertion</a><ol><li><a href=#opening-the-input-stream><span class=secno>8.4.1</span> Opening the input stream</a><li><a href=#closing-the-input-stream><span class=secno>8.4.2</span> Closing the input stream</a><li><a href=#document.write()><span class=secno>8.4.3</span> <code>document.write()</code></a><li><a href=#document.writeln()><span class=secno>8.4.4</span> <code>document.writeln()</code></a></ol><li><a href=#dom-parsing-and-serialization><span class=secno>8.5</span> DOM parsing</a><li><a href=#timers><span class=secno>8.6</span> Timers</a><li><a href=#microtask-queuing><span class=secno>8.7</span> Microtask queuing</a><li><a href=#user-prompts><span class=secno>8.8</span> User prompts</a><ol><li><a href=#simple-dialogs><span class=secno>8.8.1</span> Simple dialogs</a><li><a href=#printing><span class=secno>8.8.2</span> Printing</a></ol><li><a href=#system-state-and-capabilities><span class=secno>8.9</span> System state and capabilities</a><ol><li><a href=#the-navigator-object><span class=secno>8.9.1</span> The <code>Navigator</code> object</a><ol><li><a href=#client-identification><span class=secno>8.9.1.1</span> Client identification</a><li><a href=#language-preferences><span class=secno>8.9.1.2</span> Language preferences</a><li><a href=#custom-handlers><span class=secno>8.9.1.3</span> Custom scheme handlers: the <code>registerProtocolHandler()</code> method</a><ol><li><a href=#security-and-privacy><span class=secno>8.9.1.3.1</span> Security and privacy</a></ol><li><a href=#cookies><span class=secno>8.9.1.4</span> Cookies</a><li><a href=#plugins-2><span class=secno>8.9.1.5</span> Plugins</a></ol></ol><li><a href=#images-2><span class=secno>8.10</span> Images</a><li><a href=#animation-frames><span class=secno>8.11</span> Animation frames</a></ol><li id=toc-comms><a href=#comms><span class=secno>9</span> Communication</a><ol><li><a href=#the-messageevent-interface><span class=secno>9.1</span> The <code>MessageEvent</code> interface</a><li><a href=#server-sent-events><span class=secno>9.2</span> Server-sent events</a><ol><li><a href=#server-sent-events-intro><span class=secno>9.2.1</span> Introduction</a><li><a href=#the-eventsource-interface><span class=secno>9.2.2</span> The <code>EventSource</code> interface</a><li><a href=#sse-processing-model><span class=secno>9.2.3</span> Processing model</a><li><a href=#parsing-an-event-stream><span class=secno>9.2.4</span> Parsing an event stream</a><li><a href=#event-stream-interpretation><span class=secno>9.2.5</span> Interpreting an event stream</a><li><a href=#authoring-notes><span class=secno>9.2.6</span> Authoring notes</a><li><a href=#eventsource-push><span class=secno>9.2.7</span> Connectionless push and other features</a><li><a href=#garbage-collection><span class=secno>9.2.8</span> Garbage collection</a><li><a href=#implementation-advice><span class=secno>9.2.9</span> Implementation advice</a></ol><li><a href=#network><span class=secno>9.3</span> Web sockets</a><ol><li><a href=#network-intro><span class=secno>9.3.1</span> Introduction</a><li><a href=#the-websocket-interface><span class=secno>9.3.2</span> The <code>WebSocket</code> interface</a><li><a href=#feedback-from-the-protocol><span class=secno>9.3.3</span> Feedback from the protocol</a><li><a href=#ping-and-pong-frames><span class=secno>9.3.4</span> Ping and Pong frames</a><li><a href=#the-closeevent-interface><span class=secno>9.3.5</span> The <code>CloseEvent</code> interface</a><li><a href=#garbage-collection-2><span class=secno>9.3.6</span> Garbage collection</a></ol><li><a href=#web-messaging><span class=secno>9.4</span> Cross-document messaging</a><ol><li><a href=#introduction-13><span class=secno>9.4.1</span> Introduction</a><li><a href=#security-postmsg><span class=secno>9.4.2</span> Security</a><ol><li><a href=#authors><span class=secno>9.4.2.1</span> Authors</a><li><a href=#user-agents><span class=secno>9.4.2.2</span> User agents</a></ol><li><a href=#posting-messages><span class=secno>9.4.3</span> Posting messages</a></ol><li><a href=#channel-messaging><span class=secno>9.5</span> Channel messaging</a><ol><li><a href=#introduction-14><span class=secno>9.5.1</span> Introduction</a><ol><li><a href=#examples-5><span class=secno>9.5.1.1</span> Examples</a><li><a href=#ports-as-the-basis-of-an-object-capability-model-on-the-web><span class=secno>9.5.1.2</span> Ports as the basis of an object-capability model on the web</a><li><a href=#ports-as-the-basis-of-abstracting-out-service-implementations><span class=secno>9.5.1.3</span> Ports as the basis of abstracting out service implementations</a></ol><li><a href=#message-channels><span class=secno>9.5.2</span> Message channels</a><li><a href=#message-ports><span class=secno>9.5.3</span> Message ports</a><li><a href=#broadcasting-to-many-ports><span class=secno>9.5.4</span> Broadcasting to many ports</a><li><a href=#ports-and-garbage-collection><span class=secno>9.5.5</span> Ports and garbage collection</a></ol><li><a href=#broadcasting-to-other-browsing-contexts><span class=secno>9.6</span> Broadcasting to other browsing contexts</a></ol><li id=toc-workers><a href=#workers><span class=secno>10</span> Web workers</a><ol><li><a href=#introduction-15><span class=secno>10.1</span> Introduction</a><ol><li><a href=#scope-2><span class=secno>10.1.1</span> Scope</a><li><a href=#examples-6><span class=secno>10.1.2</span> Examples</a><ol><li><a href=#a-background-number-crunching-worker><span class=secno>10.1.2.1</span> A background number-crunching worker</a><li><a href=#module-worker-example><span class=secno>10.1.2.2</span> Using a JavaScript module as a worker</a><li><a href=#shared-workers-introduction><span class=secno>10.1.2.3</span> Shared workers introduction</a><li><a href=#shared-state-using-a-shared-worker><span class=secno>10.1.2.4</span> Shared state using a shared worker</a><li><a href=#delegation><span class=secno>10.1.2.5</span> Delegation</a><li><a href=#providing-libraries><span class=secno>10.1.2.6</span> Providing libraries</a></ol><li><a href=#tutorials><span class=secno>10.1.3</span> Tutorials</a><ol><li><a href=#creating-a-dedicated-worker><span class=secno>10.1.3.1</span> Creating a dedicated worker</a><li><a href=#communicating-with-a-dedicated-worker><span class=secno>10.1.3.2</span> Communicating with a dedicated worker</a><li><a href=#shared-workers><span class=secno>10.1.3.3</span> Shared workers</a></ol></ol><li><a href=#infrastructure-2><span class=secno>10.2</span> Infrastructure</a><ol><li><a href=#the-global-scope><span class=secno>10.2.1</span> The global scope</a><ol><li><a href=#the-workerglobalscope-common-interface><span class=secno>10.2.1.1</span> The <code>WorkerGlobalScope</code> common interface</a><li><a href=#dedicated-workers-and-the-dedicatedworkerglobalscope-interface><span class=secno>10.2.1.2</span> Dedicated workers and the <code>DedicatedWorkerGlobalScope</code> interface</a><li><a href=#shared-workers-and-the-sharedworkerglobalscope-interface><span class=secno>10.2.1.3</span> Shared workers and the <code>SharedWorkerGlobalScope</code> interface</a></ol><li><a href=#worker-event-loop><span class=secno>10.2.2</span> The event loop</a><li><a href="#the-worker's-lifetime"><span class=secno>10.2.3</span> The worker's lifetime</a><li><a href=#worker-processing-model><span class=secno>10.2.4</span> Processing model</a><li><a href=#runtime-script-errors-2><span class=secno>10.2.5</span> Runtime script errors</a><li><a href=#creating-workers><span class=secno>10.2.6</span> Creating workers</a><ol><li><a href=#the-abstractworker-mixin><span class=secno>10.2.6.1</span> The
  <code>AbstractWorker</code> mixin</a><li><a href=#script-settings-for-workers><span class=secno>10.2.6.2</span> Script settings for workers</a><li><a href=#dedicated-workers-and-the-worker-interface><span class=secno>10.2.6.3</span> Dedicated workers and the <code>Worker</code> interface</a><li><a href=#shared-workers-and-the-sharedworker-interface><span class=secno>10.2.6.4</span> Shared workers and the <code>SharedWorker</code> interface</a></ol><li><a href=#navigator.hardwareconcurrency><span class=secno>10.2.7</span> Concurrent hardware capabilities</a></ol><li><a href=#apis-available-to-workers><span class=secno>10.3</span> APIs available to workers</a><ol><li><a href=#importing-scripts-and-libraries><span class=secno>10.3.1</span> Importing scripts and libraries</a><li><a href=#the-workernavigator-object><span class=secno>10.3.2</span> The <code>WorkerNavigator</code> interface</a><li><a href=#worker-locations><span class=secno>10.3.3</span> The <code>WorkerLocation</code> interface</a></ol></ol><li id=toc-worklets><a href=#worklets><span class=secno>11</span> Worklets</a><ol><li><a href=#worklets-intro><span class=secno>11.1</span> Introduction</a><ol><li><a href=#worklets-motivations><span class=secno>11.1.1</span> Motivations</a><li><a href=#worklets-idempotent><span class=secno>11.1.2</span> Code idempotence</a><li><a href=#worklets-speculative><span class=secno>11.1.3</span> Speculative evaluation</a></ol><li><a href=#worklets-examples><span class=secno>11.2</span> Examples</a><ol><li><a href=#worklets-examples-loading><span class=secno>11.2.1</span> Loading scripts</a><li><a href=#worklets-example-registering><span class=secno>11.2.2</span> Registering a class and invoking its methods</a></ol><li><a href=#worklets-infrastructure><span class=secno>11.3</span> Infrastructure</a><ol><li><a href=#worklets-global><span class=secno>11.3.1</span> The global scope</a><ol><li><a href=#worklet-agents-and-event-loops><span class=secno>11.3.1.1</span> Agents and event loops</a><li><a href=#worklets-creation-termination><span class=secno>11.3.1.2</span> Creation and termination</a><li><a href=#script-settings-for-worklets><span class=secno>11.3.1.3</span> Script settings for worklets</a></ol><li><a href=#worklets-worklet><span class=secno>11.3.2</span> The <code>Worklet</code> class</a><li><a href=#worklets-lifetime><span class=secno>11.3.3</span> The worklet's lifetime</a></ol></ol><li id=toc-webstorage><a href=#webstorage><span class=secno>12</span> Web storage</a><ol><li><a href=#introduction-16><span class=secno>12.1</span> Introduction</a><li><a href=#storage><span class=secno>12.2</span> The API</a><ol><li><a href=#the-storage-interface><span class=secno>12.2.1</span> The <code>Storage</code> interface</a><li><a href=#the-sessionstorage-attribute><span class=secno>12.2.2</span> The <code>sessionStorage</code> getter</a><li><a href=#the-localstorage-attribute><span class=secno>12.2.3</span> The <code>localStorage</code> getter</a><li><a href=#the-storageevent-interface><span class=secno>12.2.4</span> The <code>StorageEvent</code> interface</a></ol><li><a href=#privacy><span class=secno>12.3</span> Privacy</a><ol><li><a href=#user-tracking><span class=secno>12.3.1</span> User tracking</a><li><a href=#sensitivity-of-data><span class=secno>12.3.2</span> Sensitivity of data</a></ol><li><a href=#security-storage><span class=secno>12.4</span> Security</a><ol><li><a href=#dns-spoofing-attacks><span class=secno>12.4.1</span> DNS spoofing attacks</a><li><a href=#cross-directory-attacks><span class=secno>12.4.2</span> Cross-directory attacks</a><li><a href=#implementation-risks><span class=secno>12.4.3</span> Implementation risks</a></ol></ol><li id=toc-syntax><a href=#syntax><span class=secno>13</span> The HTML syntax</a><ol><li><a href=#writing><span class=secno>13.1</span> Writing HTML documents</a><ol><li><a href=#the-doctype><span class=secno>13.1.1</span> The DOCTYPE</a><li><a href=#elements-2><span class=secno>13.1.2</span> Elements</a><ol><li><a href=#start-tags><span class=secno>13.1.2.1</span> Start tags</a><li><a href=#end-tags><span class=secno>13.1.2.2</span> End tags</a><li><a href=#attributes-2><span class=secno>13.1.2.3</span> Attributes</a><li><a href=#optional-tags><span class=secno>13.1.2.4</span> Optional tags</a><li><a href=#element-restrictions><span class=secno>13.1.2.5</span> Restrictions on content models</a><li><a href=#cdata-rcdata-restrictions><span class=secno>13.1.2.6</span> Restrictions on the contents of raw text and escapable raw text elements</a></ol><li><a href=#text-2><span class=secno>13.1.3</span> Text</a><ol><li><a href=#newlines><span class=secno>13.1.3.1</span> Newlines</a></ol><li><a href=#character-references><span class=secno>13.1.4</span> Character references</a><li><a href=#cdata-sections><span class=secno>13.1.5</span> CDATA sections</a><li><a href=#comments><span class=secno>13.1.6</span> Comments</a></ol><li><a href=#parsing><span class=secno>13.2</span> Parsing HTML documents</a><ol><li><a href=#overview-of-the-parsing-model><span class=secno>13.2.1</span> Overview of the parsing model</a><li><a href=#parse-errors><span class=secno>13.2.2</span> Parse errors</a><li><a href=#the-input-byte-stream><span class=secno>13.2.3</span> The input byte stream</a><ol><li><a href=#parsing-with-a-known-character-encoding><span class=secno>13.2.3.1</span> Parsing with a known character encoding</a><li><a href=#determining-the-character-encoding><span class=secno>13.2.3.2</span> Determining the character encoding</a><li><a href=#character-encodings><span class=secno>13.2.3.3</span> Character encodings</a><li><a href=#changing-the-encoding-while-parsing><span class=secno>13.2.3.4</span> Changing the encoding while parsing</a><li><a href=#preprocessing-the-input-stream><span class=secno>13.2.3.5</span> Preprocessing the input stream</a></ol><li><a href=#parse-state><span class=secno>13.2.4</span> Parse state</a><ol><li><a href=#the-insertion-mode><span class=secno>13.2.4.1</span> The insertion mode</a><li><a href=#the-stack-of-open-elements><span class=secno>13.2.4.2</span> The stack of open elements</a><li><a href=#the-list-of-active-formatting-elements><span class=secno>13.2.4.3</span> The list of active formatting elements</a><li><a href=#the-element-pointers><span class=secno>13.2.4.4</span> The element pointers</a><li><a href=#other-parsing-state-flags><span class=secno>13.2.4.5</span> Other parsing state flags</a></ol><li><a href=#tokenization><span class=secno>13.2.5</span> Tokenization</a><ol><li><a href=#data-state><span class=secno>13.2.5.1</span> Data state</a><li><a href=#rcdata-state><span class=secno>13.2.5.2</span> RCDATA state</a><li><a href=#rawtext-state><span class=secno>13.2.5.3</span> RAWTEXT state</a><li><a href=#script-data-state><span class=secno>13.2.5.4</span> Script data state</a><li><a href=#plaintext-state><span class=secno>13.2.5.5</span> PLAINTEXT state</a><li><a href=#tag-open-state><span class=secno>13.2.5.6</span> Tag open state</a><li><a href=#end-tag-open-state><span class=secno>13.2.5.7</span> End tag open state</a><li><a href=#tag-name-state><span class=secno>13.2.5.8</span> Tag name state</a><li><a href=#rcdata-less-than-sign-state><span class=secno>13.2.5.9</span> RCDATA less-than sign state</a><li><a href=#rcdata-end-tag-open-state><span class=secno>13.2.5.10</span> RCDATA end tag open state</a><li><a href=#rcdata-end-tag-name-state><span class=secno>13.2.5.11</span> RCDATA end tag name state</a><li><a href=#rawtext-less-than-sign-state><span class=secno>13.2.5.12</span> RAWTEXT less-than sign state</a><li><a href=#rawtext-end-tag-open-state><span class=secno>13.2.5.13</span> RAWTEXT end tag open state</a><li><a href=#rawtext-end-tag-name-state><span class=secno>13.2.5.14</span> RAWTEXT end tag name state</a><li><a href=#script-data-less-than-sign-state><span class=secno>13.2.5.15</span> Script data less-than sign state</a><li><a href=#script-data-end-tag-open-state><span class=secno>13.2.5.16</span> Script data end tag open state</a><li><a href=#script-data-end-tag-name-state><span class=secno>13.2.5.17</span> Script data end tag name state</a><li><a href=#script-data-escape-start-state><span class=secno>13.2.5.18</span> Script data escape start state</a><li><a href=#script-data-escape-start-dash-state><span class=secno>13.2.5.19</span> Script data escape start dash state</a><li><a href=#script-data-escaped-state><span class=secno>13.2.5.20</span> Script data escaped state</a><li><a href=#script-data-escaped-dash-state><span class=secno>13.2.5.21</span> Script data escaped dash state</a><li><a href=#script-data-escaped-dash-dash-state><span class=secno>13.2.5.22</span> Script data escaped dash dash state</a><li><a href=#script-data-escaped-less-than-sign-state><span class=secno>13.2.5.23</span> Script data escaped less-than sign state</a><li><a href=#script-data-escaped-end-tag-open-state><span class=secno>13.2.5.24</span> Script data escaped end tag open state</a><li><a href=#script-data-escaped-end-tag-name-state><span class=secno>13.2.5.25</span> Script data escaped end tag name state</a><li><a href=#script-data-double-escape-start-state><span class=secno>13.2.5.26</span> Script data double escape start state</a><li><a href=#script-data-double-escaped-state><span class=secno>13.2.5.27</span> Script data double escaped state</a><li><a href=#script-data-double-escaped-dash-state><span class=secno>13.2.5.28</span> Script data double escaped dash state</a><li><a href=#script-data-double-escaped-dash-dash-state><span class=secno>13.2.5.29</span> Script data double escaped dash dash state</a><li><a href=#script-data-double-escaped-less-than-sign-state><span class=secno>13.2.5.30</span> Script data double escaped less-than sign state</a><li><a href=#script-data-double-escape-end-state><span class=secno>13.2.5.31</span> Script data double escape end state</a><li><a href=#before-attribute-name-state><span class=secno>13.2.5.32</span> Before attribute name state</a><li><a href=#attribute-name-state><span class=secno>13.2.5.33</span> Attribute name state</a><li><a href=#after-attribute-name-state><span class=secno>13.2.5.34</span> After attribute name state</a><li><a href=#before-attribute-value-state><span class=secno>13.2.5.35</span> Before attribute value state</a><li><a href=#attribute-value-(double-quoted)-state><span class=secno>13.2.5.36</span> Attribute value (double-quoted) state</a><li><a href=#attribute-value-(single-quoted)-state><span class=secno>13.2.5.37</span> Attribute value (single-quoted) state</a><li><a href=#attribute-value-(unquoted)-state><span class=secno>13.2.5.38</span> Attribute value (unquoted) state</a><li><a href=#after-attribute-value-(quoted)-state><span class=secno>13.2.5.39</span> After attribute value (quoted) state</a><li><a href=#self-closing-start-tag-state><span class=secno>13.2.5.40</span> Self-closing start tag state</a><li><a href=#bogus-comment-state><span class=secno>13.2.5.41</span> Bogus comment state</a><li><a href=#markup-declaration-open-state><span class=secno>13.2.5.42</span> Markup declaration open state</a><li><a href=#comment-start-state><span class=secno>13.2.5.43</span> Comment start state</a><li><a href=#comment-start-dash-state><span class=secno>13.2.5.44</span> Comment start dash state</a><li><a href=#comment-state><span class=secno>13.2.5.45</span> Comment state</a><li><a href=#comment-less-than-sign-state><span class=secno>13.2.5.46</span> Comment less-than sign state</a><li><a href=#comment-less-than-sign-bang-state><span class=secno>13.2.5.47</span> Comment less-than sign bang state</a><li><a href=#comment-less-than-sign-bang-dash-state><span class=secno>13.2.5.48</span> Comment less-than sign bang dash state</a><li><a href=#comment-less-than-sign-bang-dash-dash-state><span class=secno>13.2.5.49</span> Comment less-than sign bang dash dash state</a><li><a href=#comment-end-dash-state><span class=secno>13.2.5.50</span> Comment end dash state</a><li><a href=#comment-end-state><span class=secno>13.2.5.51</span> Comment end state</a><li><a href=#comment-end-bang-state><span class=secno>13.2.5.52</span> Comment end bang state</a><li><a href=#doctype-state><span class=secno>13.2.5.53</span> DOCTYPE state</a><li><a href=#before-doctype-name-state><span class=secno>13.2.5.54</span> Before DOCTYPE name state</a><li><a href=#doctype-name-state><span class=secno>13.2.5.55</span> DOCTYPE name state</a><li><a href=#after-doctype-name-state><span class=secno>13.2.5.56</span> After DOCTYPE name state</a><li><a href=#after-doctype-public-keyword-state><span class=secno>13.2.5.57</span> After DOCTYPE public keyword state</a><li><a href=#before-doctype-public-identifier-state><span class=secno>13.2.5.58</span> Before DOCTYPE public identifier state</a><li><a href=#doctype-public-identifier-(double-quoted)-state><span class=secno>13.2.5.59</span> DOCTYPE public identifier (double-quoted) state</a><li><a href=#doctype-public-identifier-(single-quoted)-state><span class=secno>13.2.5.60</span> DOCTYPE public identifier (single-quoted) state</a><li><a href=#after-doctype-public-identifier-state><span class=secno>13.2.5.61</span> After DOCTYPE public identifier state</a><li><a href=#between-doctype-public-and-system-identifiers-state><span class=secno>13.2.5.62</span> Between DOCTYPE public and system identifiers state</a><li><a href=#after-doctype-system-keyword-state><span class=secno>13.2.5.63</span> After DOCTYPE system keyword state</a><li><a href=#before-doctype-system-identifier-state><span class=secno>13.2.5.64</span> Before DOCTYPE system identifier state</a><li><a href=#doctype-system-identifier-(double-quoted)-state><span class=secno>13.2.5.65</span> DOCTYPE system identifier (double-quoted) state</a><li><a href=#doctype-system-identifier-(single-quoted)-state><span class=secno>13.2.5.66</span> DOCTYPE system identifier (single-quoted) state</a><li><a href=#after-doctype-system-identifier-state><span class=secno>13.2.5.67</span> After DOCTYPE system identifier state</a><li><a href=#bogus-doctype-state><span class=secno>13.2.5.68</span> Bogus DOCTYPE state</a><li><a href=#cdata-section-state><span class=secno>13.2.5.69</span> CDATA section state</a><li><a href=#cdata-section-bracket-state><span class=secno>13.2.5.70</span> CDATA section bracket state</a><li><a href=#cdata-section-end-state><span class=secno>13.2.5.71</span> CDATA section end state</a><li><a href=#character-reference-state><span class=secno>13.2.5.72</span> Character reference state</a><li><a href=#named-character-reference-state><span class=secno>13.2.5.73</span> Named character reference state</a><li><a href=#ambiguous-ampersand-state><span class=secno>13.2.5.74</span> Ambiguous ampersand state</a><li><a href=#numeric-character-reference-state><span class=secno>13.2.5.75</span> Numeric character reference state</a><li><a href=#hexadecimal-character-reference-start-state><span class=secno>13.2.5.76</span> Hexadecimal character reference start state</a><li><a href=#decimal-character-reference-start-state><span class=secno>13.2.5.77</span> Decimal character reference start state</a><li><a href=#hexadecimal-character-reference-state><span class=secno>13.2.5.78</span> Hexadecimal character reference state</a><li><a href=#decimal-character-reference-state><span class=secno>13.2.5.79</span> Decimal character reference state</a><li><a href=#numeric-character-reference-end-state><span class=secno>13.2.5.80</span> Numeric character reference end state</a></ol><li><a href=#tree-construction><span class=secno>13.2.6</span> Tree construction</a><ol><li><a href=#creating-and-inserting-nodes><span class=secno>13.2.6.1</span> Creating and inserting nodes</a><li><a href=#parsing-elements-that-contain-only-text><span class=secno>13.2.6.2</span> Parsing elements that contain only text</a><li><a href=#closing-elements-that-have-implied-end-tags><span class=secno>13.2.6.3</span> Closing elements that have implied end tags</a><li><a href=#parsing-main-inhtml><span class=secno>13.2.6.4</span> The rules for parsing tokens in HTML content</a><ol><li><a href=#the-initial-insertion-mode><span class=secno>13.2.6.4.1</span> The "initial" insertion mode</a><li><a href=#the-before-html-insertion-mode><span class=secno>13.2.6.4.2</span> The "before html" insertion mode</a><li><a href=#the-before-head-insertion-mode><span class=secno>13.2.6.4.3</span> The "before head" insertion mode</a><li><a href=#parsing-main-inhead><span class=secno>13.2.6.4.4</span> The "in head" insertion mode</a><li><a href=#parsing-main-inheadnoscript><span class=secno>13.2.6.4.5</span> The "in head noscript" insertion mode</a><li><a href=#the-after-head-insertion-mode><span class=secno>13.2.6.4.6</span> The "after head" insertion mode</a><li><a href=#parsing-main-inbody><span class=secno>13.2.6.4.7</span> The "in body" insertion mode</a><li><a href=#parsing-main-incdata><span class=secno>13.2.6.4.8</span> The "text" insertion mode</a><li><a href=#parsing-main-intable><span class=secno>13.2.6.4.9</span> The "in table" insertion mode</a><li><a href=#parsing-main-intabletext><span class=secno>13.2.6.4.10</span> The "in table text" insertion mode</a><li><a href=#parsing-main-incaption><span class=secno>13.2.6.4.11</span> The "in caption" insertion mode</a><li><a href=#parsing-main-incolgroup><span class=secno>13.2.6.4.12</span> The "in column group" insertion mode</a><li><a href=#parsing-main-intbody><span class=secno>13.2.6.4.13</span> The "in table body" insertion mode</a><li><a href=#parsing-main-intr><span class=secno>13.2.6.4.14</span> The "in row" insertion mode</a><li><a href=#parsing-main-intd><span class=secno>13.2.6.4.15</span> The "in cell" insertion mode</a><li><a href=#parsing-main-inselect><span class=secno>13.2.6.4.16</span> The "in select" insertion mode</a><li><a href=#parsing-main-inselectintable><span class=secno>13.2.6.4.17</span> The "in select in table" insertion mode</a><li><a href=#parsing-main-intemplate><span class=secno>13.2.6.4.18</span> The "in template" insertion mode</a><li><a href=#parsing-main-afterbody><span class=secno>13.2.6.4.19</span> The "after body" insertion mode</a><li><a href=#parsing-main-inframeset><span class=secno>13.2.6.4.20</span> The "in frameset" insertion mode</a><li><a href=#parsing-main-afterframeset><span class=secno>13.2.6.4.21</span> The "after frameset" insertion mode</a><li><a href=#the-after-after-body-insertion-mode><span class=secno>13.2.6.4.22</span> The "after after body" insertion mode</a><li><a href=#the-after-after-frameset-insertion-mode><span class=secno>13.2.6.4.23</span> The "after after frameset" insertion mode</a></ol><li><a href=#parsing-main-inforeign><span class=secno>13.2.6.5</span> The rules for parsing tokens in foreign content</a></ol><li><a href=#the-end><span class=secno>13.2.7</span> The end</a><li><a href=#coercing-an-html-dom-into-an-infoset><span class=secno>13.2.8</span> Coercing an HTML DOM into an infoset</a><li><a href=#an-introduction-to-error-handling-and-strange-cases-in-the-parser><span class=secno>13.2.9</span> An introduction to error handling and strange cases in the parser</a><ol><li><a href=#misnested-tags:-b-i-/b-/i><span class=secno>13.2.9.1</span> Misnested tags: &lt;b>&lt;i>&lt;/b>&lt;/i></a><li><a href=#misnested-tags:-b-p-/b-/p><span class=secno>13.2.9.2</span> Misnested tags: &lt;b>&lt;p>&lt;/b>&lt;/p></a><li><a href=#unexpected-markup-in-tables><span class=secno>13.2.9.3</span> Unexpected markup in tables</a><li><a href=#scripts-that-modify-the-page-as-it-is-being-parsed><span class=secno>13.2.9.4</span> Scripts that modify the page as it is being parsed</a><li><a href=#the-execution-of-scripts-that-are-moving-across-multiple-documents><span class=secno>13.2.9.5</span> The execution of scripts that are moving across multiple documents</a><li><a href=#unclosed-formatting-elements><span class=secno>13.2.9.6</span> Unclosed formatting elements</a></ol></ol><li><a href=#serialising-html-fragments><span class=secno>13.3</span> Serializing HTML fragments</a><li><a href=#parsing-html-fragments><span class=secno>13.4</span> Parsing HTML fragments</a><li><a href=#named-character-references><span class=secno>13.5</span> Named character references</a></ol><li id=toc-the-xhtml-syntax><a href=#the-xhtml-syntax><span class=secno>14</span> The XML syntax</a><ol><li><a href=#writing-xhtml-documents><span class=secno>14.1</span> Writing documents in the XML syntax</a><li><a href=#parsing-xhtml-documents><span class=secno>14.2</span> Parsing XML documents</a><li><a href=#serialising-xhtml-fragments><span class=secno>14.3</span> Serializing XML fragments</a><li><a href=#parsing-xhtml-fragments><span class=secno>14.4</span> Parsing XML fragments</a></ol><li id=toc-rendering><a href=#rendering><span class=secno>15</span> Rendering</a><ol><li><a href=#introduction-17><span class=secno>15.1</span> Introduction</a><li><a href=#the-css-user-agent-style-sheet-and-presentational-hints><span class=secno>15.2</span> The CSS user agent style sheet and presentational hints</a><li><a href=#non-replaced-elements><span class=secno>15.3</span> Non-replaced elements</a><ol><li><a href=#hidden-elements><span class=secno>15.3.1</span> Hidden elements</a><li><a href=#the-page><span class=secno>15.3.2</span> The page</a><li><a href=#flow-content-3><span class=secno>15.3.3</span> Flow content</a><li><a href=#phrasing-content-3><span class=secno>15.3.4</span> Phrasing content</a><li><a href=#bidi-rendering><span class=secno>15.3.5</span> Bidirectional text</a><li><a href=#sections-and-headings><span class=secno>15.3.6</span> Sections and headings</a><li><a href=#lists><span class=secno>15.3.7</span> Lists</a><li><a href=#tables-2><span class=secno>15.3.8</span> Tables</a><li><a href=#margin-collapsing-quirks><span class=secno>15.3.9</span> Margin collapsing quirks</a><li><a href=#form-controls><span class=secno>15.3.10</span> Form controls</a><li><a href=#the-hr-element-2><span class=secno>15.3.11</span> The <code>hr</code> element</a><li><a href=#the-fieldset-and-legend-elements><span class=secno>15.3.12</span> The <code>fieldset</code> and <code>legend</code> elements</a></ol><li><a href=#replaced-elements><span class=secno>15.4</span> Replaced elements</a><ol><li><a href=#embedded-content-rendering-rules><span class=secno>15.4.1</span> Embedded content</a><li><a href=#images-3><span class=secno>15.4.2</span> Images</a><li><a href=#attributes-for-embedded-content-and-images><span class=secno>15.4.3</span> Attributes for embedded content and images</a><li><a href=#image-maps-2><span class=secno>15.4.4</span> Image maps</a></ol><li><a href=#widgets><span class=secno>15.5</span> Widgets</a><ol><li><a href=#introduction-18><span class=secno>15.5.1</span> Introduction</a><li><a href=#button-layout><span class=secno>15.5.2</span> Button layout</a><li><a href=#the-button-element-2><span class=secno>15.5.3</span> The <code>button</code> element</a><li><a href=#the-details-and-summary-elements><span class=secno>15.5.4</span> The <code>details</code> and <code>summary</code> elements</a><li><a href=#the-input-element-as-a-text-entry-widget><span class=secno>15.5.5</span> The <code>input</code> element as a text entry widget</a><li><a href=#the-input-element-as-domain-specific-widgets><span class=secno>15.5.6</span> The <code>input</code> element as domain-specific widgets</a><li><a href=#the-input-element-as-a-range-control><span class=secno>15.5.7</span> The <code>input</code> element as a range control</a><li><a href=#the-input-element-as-a-colour-well><span class=secno>15.5.8</span> The <code>input</code> element as a color
  well</a><li><a href=#the-input-element-as-a-checkbox-and-radio-button-widgets><span class=secno>15.5.9</span> The <code>input</code> element as a checkbox and radio button widgets</a><li><a href=#the-input-element-as-a-file-upload-control><span class=secno>15.5.10</span> The <code>input</code> element as a file upload control</a><li><a href=#the-input-element-as-a-button><span class=secno>15.5.11</span> The <code>input</code> element as a button</a><li><a href=#the-marquee-element-2><span class=secno>15.5.12</span> The <code>marquee</code> element</a><li><a href=#the-meter-element-2><span class=secno>15.5.13</span> The <code>meter</code> element</a><li><a href=#the-progress-element-2><span class=secno>15.5.14</span> The <code>progress</code> element</a><li><a href=#the-select-element-2><span class=secno>15.5.15</span> The <code>select</code> element</a><li><a href=#the-textarea-element-2><span class=secno>15.5.16</span> The <code>textarea</code> element</a></ol><li><a href=#frames-and-framesets><span class=secno>15.6</span> Frames and framesets</a><li><a href=#interactive-media><span class=secno>15.7</span> Interactive media</a><ol><li><a href=#links,-forms,-and-navigation><span class=secno>15.7.1</span> Links, forms, and navigation</a><li><a href=#the-title-attribute-2><span class=secno>15.7.2</span> The <code>title</code> attribute</a><li><a href=#editing-hosts><span class=secno>15.7.3</span> Editing hosts</a><li><a href=#text-rendered-in-native-user-interfaces><span class=secno>15.7.4</span> Text rendered in native user interfaces</a></ol><li><a href=#print-media><span class=secno>15.8</span> Print media</a><li><a href=#unstyled-xml-documents><span class=secno>15.9</span> Unstyled XML documents</a></ol><li id=toc-obsolete><a href=#obsolete><span class=secno>16</span> Obsolete features</a><ol><li><a href=#obsolete-but-conforming-features><span class=secno>16.1</span> Obsolete but conforming features</a><ol><li><a href=#warnings-for-obsolete-but-conforming-features><span class=secno>16.1.1</span> Warnings for obsolete but conforming features</a></ol><li><a href=#non-conforming-features><span class=secno>16.2</span> Non-conforming features</a><li><a href=#requirements-for-implementations><span class=secno>16.3</span> Requirements for implementations</a><ol><li><a href=#the-marquee-element><span class=secno>16.3.1</span> The <code>marquee</code> element</a><li><a href=#frames><span class=secno>16.3.2</span> Frames</a><li><a href=#other-elements,-attributes-and-apis><span class=secno>16.3.3</span> Other elements, attributes and APIs</a></ol></ol><li id=toc-iana><a href=#iana><span class=secno>17</span> IANA considerations</a><ol><li><a href=#text/html><span class=secno>17.1</span> <code>text/html</code></a><li><a href=#multipart/x-mixed-replace><span class=secno>17.2</span> <code>multipart/x-mixed-replace</code></a><li><a href=#application/xhtml+xml><span class=secno>17.3</span> <code>application/xhtml+xml</code></a><li><a href=#text/cache-manifest><span class=secno>17.4</span> <code>text/cache-manifest</code></a><li><a href=#text/ping><span class=secno>17.5</span> <code>text/ping</code></a><li><a href=#application/microdata+json><span class=secno>17.6</span> <code>application/microdata+json</code></a><li><a href=#text/event-stream><span class=secno>17.7</span> <code>text/event-stream</code></a><li><a href=#cross-origin-embedder-policy><span class=secno>17.8</span> `<code>Cross-Origin-Embedder-Policy</code>`</a><li><a href=#cross-origin-embedder-policy-report-only><span class=secno>17.9</span> `<code>Cross-Origin-Embedder-Policy-Report-Only</code>`</a><li><a href=#cross-origin-opener-policy-2><span class=secno>17.10</span> `<code>Cross-Origin-Opener-Policy</code>`</a><li><a href=#cross-origin-opener-policy-report-only><span class=secno>17.11</span> `<code>Cross-Origin-Opener-Policy-Report-Only</code>`</a><li><a href=#origin-isolation-2><span class=secno>17.12</span> `<code>Origin-Isolation</code>`</a><li><a href=#ping-from><span class=secno>17.13</span> `<code>Ping-From</code>`</a><li><a href=#ping-to><span class=secno>17.14</span> `<code>Ping-To</code>`</a><li><a href=#refresh><span class=secno>17.15</span> `<code>Refresh</code>`</a><li><a href=#last-event-id><span class=secno>17.16</span> `<code>Last-Event-ID</code>`</a><li><a href=#x-frame-options><span class=secno>17.17</span> `<code>X-Frame-Options</code>`</a><li><a href=#web+-scheme-prefix><span class=secno>17.18</span> <code>web+</code> scheme prefix</a></ol><li id=toc-index><a href=#index>Index</a><ol><li><a href=#elements-3>Elements</a><li><a href=#element-content-categories>Element content categories</a><li><a href=#attributes-3>Attributes</a><li><a href=#element-interfaces>Element Interfaces</a><li><a href=#all-interfaces>All Interfaces</a><li><a href=#events-2>Events</a><li><a href=#mime-types-2>MIME Types</a></ol><li id=toc-references><a href=#references>References</a><li id=toc-acknowledgments><a href=#acknowledgments>Acknowledgments</a><li id=toc-ipr><a href=#ipr>Intellectual property rights</a></ol>

  

  <h2 id=introduction><span class=secno>1</span> Introduction<a href=#introduction class=self-link></a></h2>

  

  <h3 id=abstract><span class=secno>1.1</span> Where does this specification fit?<a href=#abstract class=self-link></a></h3>

  <p>This specification defines a big part of the web platform, in lots of detail. Its place in the
  web platform specification stack relative to other specifications can be best summed up as
  follows:</p>

  <svg id=abstractimg width=398 role=img viewBox="0 0 398 359" height=359 aria-label="It consists of everything else, above such core technologies as HTTP, TLS, DOM, Unicode, Web IDL, MIME, URL, XML, JavaScript, and Encoding; below higher-level technologies like CSS, SVG, MathML, and Service Workers; and to the side of technologies like IndexedDB, Fetch, CSP, AV1, Opus, and PNG.">
   <rect width=398 height=80></rect>
   <text class=horizontal x=199 y=45>CSS SVG MathML Service Workers</text>
   <rect width=67 y=85 height=177></rect>
   <text transform="translate(25 173.5) rotate(-90)" class=left>IDB Fetch CSP</text>
   <text transform="translate(50 173.5) rotate(-90)" class=left>AV1 Opus PNG</text>
   
   <image xlink:href=/images/abstract.jpeg width=326 x=72 y=85 height=177></image>
   <text class=right x=130 y=250>THIS SPECIFICATION</text>
   <rect width=398 y=267 height=92></rect>
   <text class=horizontal x=199 y=300>HTTP TLS DOM Unicode Web IDL</text>
   <text class=horizontal x=199 y=330>MIME URL XML JavaScript Encoding</text>
  </svg>

  


  <h3 id=is-this-html5?><span class=secno>1.2</span> Is this HTML5?<a href=#is-this-html5? class=self-link></a></h3>

  <p><i>This section is non-normative.</i></p>

  <p>In short: Yes.</p>

  <p>In more length: the term "HTML5" is widely used as a buzzword to refer to modern web
  technologies, many of which (though by no means all) are developed at the WHATWG. This document is
  one such; others are available from <a href=https://spec.whatwg.org/>the WHATWG Standards
  overview</a>.</p>


  <h3 id=background><span class=secno>1.3</span> Background<a href=#background class=self-link></a></h3>

  <p><i>This section is non-normative.</i></p>

  <p>HTML is the World Wide Web's core markup language. Originally, HTML was primarily designed as a
  language for semantically describing scientific documents. Its general design, however, has
  enabled it to be adapted, over the subsequent years, to describe a number of other types of
  documents and even applications.</p>


  <h3 id=audience><span class=secno>1.4</span> Audience<a href=#audience class=self-link></a></h3>

  <p><i>This section is non-normative.</i></p>

  <p>This specification is intended for authors of documents and scripts that use the features
  defined in this specification, implementers of tools that operate on pages that
  use the features defined in this specification, and individuals wishing to establish the
  correctness of documents or implementations with respect to the requirements of this
  specification.</p>

  <p>This document is probably not suited to readers who do not already have at least a passing
  familiarity with web technologies, as in places it sacrifices clarity for precision, and brevity
  for completeness. More approachable tutorials and authoring guides can provide a gentler
  introduction to the topic.</p>

  <p>In particular, familiarity with the basics of DOM is necessary for a complete understanding of
  some of the more technical parts of this specification. An understanding of Web IDL, HTTP, XML,
  Unicode, character encodings, JavaScript, and CSS will also be helpful in places but is not
  essential.</p>


  <h3 id=scope><span class=secno>1.5</span> Scope<a href=#scope class=self-link></a></h3>

  <p><i>This section is non-normative.</i></p>

  <p>This specification is limited to providing a semantic-level markup language and associated
  semantic-level scripting APIs for authoring accessible pages on the web ranging from static
  documents to dynamic applications.</p>

  <p>The scope of this specification does not include providing mechanisms for media-specific
  customization of presentation (although default rendering rules for web browsers are included at
  the end of this specification, and several mechanisms for hooking into CSS are provided as part of
  the language).</p>

  <p>The scope of this specification is not to describe an entire operating system. In particular,
  hardware configuration software, image manipulation tools, and applications that users would be
  expected to use with high-end workstations on a daily basis are out of scope. In terms of
  applications, this specification is targeted specifically at applications that would be expected
  to be used by users on an occasional basis, or regularly but from disparate locations, with low
  CPU requirements. Examples of such applications include online purchasing systems, searching
  systems, games (especially multiplayer online games), public telephone books or address books,
  communications software (email clients, instant messaging clients, discussion software), document
  editing software, etc.</p>


  <h3 id=history-2><span class=secno>1.6</span> History<a href=#history-2 class=self-link></a></h3>

  <p><i>This section is non-normative.</i></p>

  <p>For its first five years (1990-1995), HTML went through a number of revisions and experienced a
  number of extensions, primarily hosted first at CERN, and then at the IETF.</p>

  <p>With the creation of the W3C, HTML's development changed venue again. A first abortive attempt
  at extending HTML in 1995 known as HTML 3.0 then made way to a more pragmatic approach known as
  HTML 3.2, which was completed in 1997. HTML4 quickly followed later that same year.</p>

  <p>The following year, the W3C membership decided to stop evolving HTML and instead begin work on
  an XML-based equivalent, called XHTML.  This
  effort started with a reformulation of HTML4 in XML, known as XHTML 1.0, which added no new
  features except the new serialization, and which was completed in 2000. After XHTML 1.0, the W3C's
  focus turned to making it easier for other working groups to extend XHTML, under the banner of
  XHTML Modularization. In parallel with this, the W3C also worked on a new language that was not
  compatible with the earlier HTML and XHTML languages, calling it XHTML2.</p>

  <p>Around the time that HTML's evolution was stopped in 1998, parts of the API for HTML developed
  by browser vendors were specified and published under the name DOM Level 1 (in 1998) and DOM Level
  2 Core and DOM Level 2 HTML (starting in 2000 and culminating in 2003). These efforts then petered
  out, with some DOM Level 3 specifications published in 2004 but the working group being closed
  before all the Level 3 drafts were completed.</p>

  <p>In 2003, the publication of XForms, a technology which was positioned as the next generation of
  web forms, sparked a renewed interest in evolving HTML itself, rather than finding replacements
  for it. This interest was borne from the realization that XML's deployment as a web technology was
  limited to entirely new technologies (like RSS and later Atom), rather than as a replacement for
  existing deployed technologies (like HTML).</p>

  <p>A proof of concept to show that it was possible to extend HTML4's forms to provide many of the
  features that XForms 1.0 introduced, without requiring browsers to implement rendering engines
  that were incompatible with existing HTML web pages, was the first result of this renewed
  interest. At this early stage, while the draft was already publicly available, and input was
  already being solicited from all sources, the specification was only under Opera Software's
  copyright.</p>

  <p>The idea that HTML's evolution should be reopened was tested at a W3C workshop in 2004, where
  some of the principles that underlie the HTML5 work (described below), as well as the
  aforementioned early draft proposal covering just forms-related features, were presented to the
  W3C jointly by Mozilla and Opera. The proposal was rejected on the grounds that the proposal
  conflicted with the previously chosen direction for the web's evolution; the W3C staff and
  membership voted to continue developing XML-based replacements instead.</p>

  <p>Shortly thereafter, Apple, Mozilla, and Opera jointly announced their intent to continue
  working on the effort under the umbrella of a new venue called the WHATWG. A public mailing list
  was created, and the draft was moved to the WHATWG site. The copyright was subsequently amended to
  be jointly owned by all three vendors, and to allow reuse of the specification.</p>

  <p>The WHATWG was based on several core principles, in particular that technologies need to be
  backwards compatible, that specifications and implementations need to match even if this means
  changing the specification rather than the implementations, and that specifications need to be
  detailed enough that implementations can achieve complete interoperability without
  reverse-engineering each other.</p>

  <p>The latter requirement in particular required that the scope of the HTML5 specification include
  what had previously been specified in three separate documents: HTML4, XHTML1, and DOM2 HTML. It
  also meant including significantly more detail than had previously been considered the norm.</p>

  <p>In 2006, the W3C indicated an interest to participate in the development of HTML5 after all,
  and in 2007 formed a working group chartered to work with the WHATWG on the development of the
  HTML5 specification. Apple, Mozilla, and Opera allowed the W3C to publish the specification under
  the W3C copyright, while keeping a version with the less restrictive license on the WHATWG
  site.</p>

  <p>For a number of years, both groups then worked together. In 2011, however, the groups came to
  the conclusion that they had different goals: the W3C wanted to publish a "finished" version of
  "HTML5", while the WHATWG wanted to continue working on a Living Standard for HTML, continuously
  maintaining the specification rather than freezing it in a state with known problems, and adding
  new features as needed to evolve the platform.</p>

  <p>In 2019, the WHATWG and W3C <a href=https://www.w3.org/blog/news/archives/7753>signed an
  agreement</a> to collaborate on a single version of HTML going forward: this document.</p>



  <h3 id=design-notes><span class=secno>1.7</span> Design notes<a href=#design-notes class=self-link></a></h3>

  <p><i>This section is non-normative.</i></p>

  <p>It must be admitted that many aspects of HTML appear at first glance to be nonsensical and
  inconsistent.</p>

  <p>HTML, its supporting DOM APIs, as well as many of its supporting technologies, have been
  developed over a period of several decades by a wide array of people with different priorities
  who, in many cases, did not know of each other's existence.</p>

  <p>Features have thus arisen from many sources, and have not always been designed in especially
  consistent ways. Furthermore, because of the unique characteristics of the web, implementation
  bugs have often become de-facto, and now de-jure, standards, as content is often unintentionally
  written in ways that rely on them before they can be fixed.</p>

  <p>Despite all this, efforts have been made to adhere to certain design goals. These are described
  in the next few subsections.</p>


  

  <h4 id=serialisability-of-script-execution><span class=secno>1.7.1</span> Serializability of script execution<a href=#serialisability-of-script-execution class=self-link></a></h4>

  <p><i>This section is non-normative.</i></p>

  <p>To avoid exposing web authors to the complexities of multithreading, the HTML and DOM APIs are
  designed such that no script can ever detect the simultaneous execution of other scripts. Even
  with <a href=#worker id=serialisability-of-script-execution:worker>workers</a>, the intent is that the behavior of implementations can
  be thought of as completely serializing the execution of all scripts in all <a href=#browsing-context id=serialisability-of-script-execution:browsing-context>browsing contexts</a>.</p>

  <p>The exception to this general design principle is the JavaScript <code id=serialisability-of-script-execution:sharedarraybuffer><a data-x-internal=sharedarraybuffer href=https://tc39.es/ecma262/#sec-sharedarraybuffer-objects>SharedArrayBuffer</a></code>
  class. Using <code id=serialisability-of-script-execution:sharedarraybuffer-2><a data-x-internal=sharedarraybuffer href=https://tc39.es/ecma262/#sec-sharedarraybuffer-objects>SharedArrayBuffer</a></code> objects, it can in fact be observed that scripts in
  other <a href=https://tc39.es/ecma262/#sec-agents id=serialisability-of-script-execution:agent data-x-internal=agent>agents</a> are executing simultaneously. Furthermore, due to the
  JavaScript memory model, there are situations which not only are un-representable via serialized
  <em>script</em> execution, but also un-representable via serialized <em>statement</em> execution
  among those scripts.</p>

  



  <h4 id=compliance-with-other-specifications><span class=secno>1.7.2</span> Compliance with other specifications<a href=#compliance-with-other-specifications class=self-link></a></h4>

  <p><i>This section is non-normative.</i></p>

  <p>This specification interacts with and relies on a wide variety of other specifications. In
  certain circumstances, unfortunately, conflicting needs have led to this specification violating
  the requirements of these other specifications. Whenever this has occurred, the transgressions
  have each been noted as a "<dfn id=willful-violation>willful violation</dfn>", and the reason for the violation has
  been noted.</p>



  <h4 id=extensibility><span class=secno>1.7.3</span> Extensibility<a href=#extensibility class=self-link></a></h4>

  <p><i>This section is non-normative.</i></p>

  <p>HTML has a wide array of extensibility mechanisms that can be used for adding semantics in a
  safe manner:</p>

  <ul><li><p>Authors can use the <code id=extensibility:classes><a href=#classes>class</a></code> attribute to extend elements,
   effectively creating their own elements, while using the most applicable existing "real" HTML
   element, so that browsers and other tools that don't know of the extension can still support it
   somewhat well. This is the tack used by microformats, for example.<li><p>Authors can include data for inline client-side scripts or server-side site-wide scripts
   to process using the <code id=extensibility:attr-data-*><a href=#attr-data-*>data-*=""</a></code> attributes. These are guaranteed
   to never be touched by browsers, and allow scripts to include data on HTML elements that scripts
   can then look for and process.<li><p>Authors can use the <code id=extensibility:the-meta-element><a href=#the-meta-element>&lt;meta name="" content=""></a></code> mechanism to
   include page-wide metadata.<li><p>Authors can use the <code id=extensibility:attr-hyperlink-rel><a href=#attr-hyperlink-rel>rel=""</a></code> mechanism to annotate
   links with specific meanings by registering <a href=#concept-rel-extensions id=extensibility:concept-rel-extensions>extensions to
   the predefined set of link types</a>. This is also used by microformats.<li><p>Authors can embed raw data using the <code id=extensibility:the-script-element><a href=#the-script-element>&lt;script type=""></a></code>
   mechanism with a custom type, for further handling by inline or server-side scripts.<li><p>Authors can create <a href=#plugin id=extensibility:plugin>plugins</a> and invoke them using the
   <code id=extensibility:the-embed-element><a href=#the-embed-element>embed</a></code> element. This is how Flash works.<li><p>Authors can extend APIs using the JavaScript prototyping mechanism. This is widely used by
   script libraries, for instance.<li><p>Authors can use the microdata feature (the <code id=extensibility:attr-itemscope><a href=#attr-itemscope>itemscope=""</a></code> and <code id=extensibility:names:-the-itemprop-attribute><a href=#names:-the-itemprop-attribute>itemprop=""</a></code>
   attributes) to embed nested name-value pairs of data to be shared with other applications and
   sites.</ul>




  <h3 id=html-vs-xhtml><span class=secno>1.8</span> HTML vs XML syntax<a href=#html-vs-xhtml class=self-link></a></h3>

  <p><i>This section is non-normative.</i></p>

  <p>This specification defines an abstract language for describing documents and applications, and
  some APIs for interacting with in-memory representations of resources that use this language.</p>

  <p>The in-memory representation is known as "DOM HTML", or "the DOM" for short.</p>

  <p>There are various concrete syntaxes that can be used to transmit resources that use this
  abstract language, two of which are defined in this specification.</p>

  <p>The first such concrete syntax is the HTML syntax. This is the format suggested for most
  authors. It is compatible with most legacy web browsers. If a document is transmitted with the
  <code id=html-vs-xhtml:text/html><a href=#text/html>text/html</a></code> <a id=html-vs-xhtml:mime-type href=https://mimesniff.spec.whatwg.org/#mime-type data-x-internal=mime-type>MIME type</a>, then it will be processed as an HTML document by
  web browsers. This specification defines the latest HTML syntax, known simply as "HTML".</p>

  <p>The second concrete syntax is XML. When a document is transmitted with an <a id=html-vs-xhtml:xml-mime-type href=https://mimesniff.spec.whatwg.org/#xml-mime-type data-x-internal=xml-mime-type>XML MIME
  type</a>, such as <code id=html-vs-xhtml:application/xhtml+xml><a href=#application/xhtml+xml>application/xhtml+xml</a></code>, then it is treated as an XML document by
  web browsers, to be parsed by an XML processor. Authors are reminded that the processing for XML
  and HTML differs; in particular, even minor syntax errors will prevent a document labeled as XML
  from being rendered fully, whereas they would be ignored in the HTML syntax.</p>

  <p class=note>The XML syntax for HTML was formerly referred to as "XHTML", but this
  specification does not use that term (among other reasons, because no such term is used for the
  HTML syntaxes of MathML and SVG).</p>

  <p>The DOM, the HTML syntax, and the XML syntax cannot all represent the same content. For
  example, namespaces cannot be represented using the HTML syntax, but they are supported in the DOM
  and in the XML syntax. Similarly, documents that use the <code id=html-vs-xhtml:the-noscript-element><a href=#the-noscript-element>noscript</a></code> feature can be
  represented using the HTML syntax, but cannot be represented with the DOM or in the XML syntax.
  Comments that contain the string "<code>--></code>" can only be represented in the
  DOM, not in the HTML and XML syntaxes.</p>


  <h3 id=structure-of-this-specification><span class=secno>1.9</span> Structure of this specification<a href=#structure-of-this-specification class=self-link></a></h3>

  <p><i>This section is non-normative.</i></p>

  <p>This specification is divided into the following major sections:</p>

  <dl><dt><a href=#introduction>Introduction</a><dd>Non-normative materials providing a context for the HTML standard.<dt><a href=#infrastructure>Common infrastructure</a><dd>The conformance classes, algorithms, definitions, and the common underpinnings of the rest of
   the specification.<dt><a href=#dom>Semantics, structure, and APIs of HTML documents</a><dd>Documents are built from elements. These elements form a tree using the DOM. This section
   defines the features of this DOM, as well as introducing the features common to all elements, and
   the concepts used in defining elements.<dt><a href=#semantics>The elements of HTML</a><dd>Each element has a predefined meaning, which is explained in this section. Rules for authors
   on how to use the element, along with user agent requirements for how to
   handle each element, are also given. This includes large signature features of HTML such
   as video playback and subtitles, form controls and form submission, and a 2D graphics API known
   as the HTML canvas.<dt><a href=#microdata>Microdata</a><dd>This specification introduces a mechanism for adding machine-readable annotations to
   documents, so that tools can extract trees of name-value pairs from the document. This section
   describes this mechanism and some algorithms that can be used to convert HTML
   documents into other formats. This section also defines some sample Microdata vocabularies
   for contact information, calendar events, and licensing works.<dt><a href=#editing>User interaction</a><dd>HTML documents can provide a number of mechanisms for users to interact with and modify
   content, which are described in this section, such as how focus works, and drag-and-drop.<dt><a href=#browsers>Loading web pages</a><dd>HTML documents do not exist in a vacuum — this section defines many of the features
   that affect environments that deal with multiple pages, such as web browsers and offline
   caching of web applications.<dt><a href=#webappapis>Web application APIs</a><dd>This section introduces basic features for scripting of applications in HTML.<dt><a href=#workers>Web workers</a><dd>This section defines an API for background threads in JavaScript.<dt><a href=#comms>The communication APIs</a><dd>This section describes some mechanisms that applications written in HTML can use to
   communicate with other applications from different domains running on the same client. It also
   introduces a server-push event stream mechanism known as Server Sent Events or
   <code id=structure-of-this-specification:eventsource><a href=#eventsource>EventSource</a></code>, and a two-way full-duplex socket protocol for scripts known as Web
   Sockets.
   <dt><a href=#webstorage>Web storage</a><dd>This section defines a client-side storage mechanism based on name-value pairs.<dt><a href=#syntax>The HTML syntax</a><dt><a href=#xhtml>The XML syntax</a><dd>All of these features would be for naught if they couldn't be represented in a serialized
   form and sent to other people, and so these sections define the syntaxes of HTML and XML, along with rules for how to parse content using those syntaxes.<dt><a href=#rendering>Rendering</a><dd>This section defines the default rendering rules for web browsers.</dl>

  <p>There are also some appendices, listing <a href=#obsolete>obsolete features</a> and <a href=#iana>IANA considerations</a>, and several indices.</p>


  

  <h4 id=how-to-read-this-specification><span class=secno>1.9.1</span> How to read this specification<a href=#how-to-read-this-specification class=self-link></a></h4>

  <p>This specification should be read like all other specifications. First, it should be read
  cover-to-cover, multiple times. Then, it should be read backwards at least once. Then it should be
  read by picking random sections from the contents list and following all the cross-references.</p>

  <p>As described in the conformance requirements section below, this specification describes
  conformance criteria for a variety of conformance classes. In particular, there are conformance
  requirements that apply to <em>producers</em>, for example authors and the documents they create,
  and there are conformance requirements that apply to <em>consumers</em>, for example web browsers.
  They can be distinguished by what they are requiring: a requirement on a producer states what is
  allowed, while a requirement on a consumer states how software is to act.</p>

  <div class=example>

   <p>For example, "the <code>foo</code> attribute's value must be a <a href=#valid-integer id=how-to-read-this-specification:valid-integer>valid
   integer</a>" is a requirement on producers, as it lays out the allowed values; in contrast,
   the requirement "the <code>foo</code> attribute's value must be parsed using the
   <a href=#rules-for-parsing-integers id=how-to-read-this-specification:rules-for-parsing-integers>rules for parsing integers</a>" is a requirement on consumers, as it describes how to
   process the content.</p>

  </div>

  <p><strong>Requirements on producers have no bearing whatsoever on consumers.</strong></p>

  <div class=example>

   <p>Continuing the above example, a requirement stating that a particular attribute's value is
   constrained to being a <a href=#valid-integer id=how-to-read-this-specification:valid-integer-2>valid integer</a> emphatically does <em>not</em> imply anything
   about the requirements on consumers. It might be that the consumers are in fact required to treat
   the attribute as an opaque string, completely unaffected by whether the value conforms to the
   requirements or not. It might be (as in the previous example) that the consumers are required to
   parse the value using specific rules that define how invalid (non-numeric in this case) values
   are to be processed.</p>

  </div>

  



  <h4 id=typographic-conventions><span class=secno>1.9.2</span> Typographic conventions<a href=#typographic-conventions class=self-link></a></h4>

  <p>This is a definition, requirement, or explanation.</p>

  <p class=note>This is a note.</p>

  <p class=example>This is an example.</p>

  <p class=XXX>This is an open issue.</p>

  <p class=warning>This is a warning.</p>

  <pre class=extract><code class='idl'>[<c- g>Exposed</c->=<c- n>Window</c->]
<c- b>interface</c-> <dfn><c- g>Example</c-></dfn> {
  // this is an IDL definition
};</code></pre>

  <dl class=domintro><dt><var>variable</var> = <var>object</var> . <code id=typographic-conventions:x-that><a href=#x-that>method</a></code>( [ <var>optionalArgument</var> ] )<dd>
    <p>This is a note to authors describing the usage of an interface.</p>
   </dl>

  <pre><code class='css'><c- c>/* this is a CSS fragment */</c-></code></pre>

  <p>The defining instance of a term is marked up like <dfn id=x-this>this</dfn>. Uses of that
  term are marked up like <a href=#x-this id=typographic-conventions:x-this>this</a> or like <i id=typographic-conventions:x-this-2><a href=#x-this>this</a></i>.</p>

  <p>The defining instance of an element, attribute, or API is marked up like <dfn id=x-that><code>this</code></dfn>. References to that element, attribute, or API are marked up
  like <code id=typographic-conventions:x-that-2><a href=#x-that>this</a></code>.</p>

  <p>Other code fragments are marked up <code>like this</code>.</p>

  <p>Variables are marked up like <var>this</var>.</p>

  <p>In an algorithm, steps in <a href=#synchronous-section id=typographic-conventions:synchronous-section>synchronous
  sections</a> are marked with ⌛.</p>

  <p>In some cases, requirements are given in the form of lists with conditions and corresponding
  requirements. In such cases, the requirements that apply to a condition are always the first set
  of requirements that follow the condition, even in the case of there being multiple sets of
  conditions for those requirements. Such cases are presented as follows:</p>

  <dl class=switch><dt>This is a condition
   <dt>This is another condition
   <dd>This is the requirement that applies to the conditions above.

   <dt>This is a third condition
   <dd>This is the requirement that applies to the third condition.
  </dl>




  <h3 id=a-quick-introduction-to-html><span class=secno>1.10</span> A quick introduction to HTML<a href=#a-quick-introduction-to-html class=self-link></a></h3>

  <p><i>This section is non-normative.</i></p>

  <p>A basic HTML document looks like this:</p>

  <pre id=intro-early-example><code class='html'><c- cp>&lt;!DOCTYPE html&gt;</c->
<c- p>&lt;</c-><c- f>html</c-> <c- e>lang</c-><c- o>=</c-><c- s>&quot;en&quot;</c-><c- p>&gt;</c->
 <c- p>&lt;</c-><c- f>head</c-><c- p>&gt;</c->
  <c- p>&lt;</c-><c- f>title</c-><c- p>&gt;</c->Sample page<c- p>&lt;/</c-><c- f>title</c-><c- p>&gt;</c->
 <c- p>&lt;/</c-><c- f>head</c-><c- p>&gt;</c->
 <c- p>&lt;</c-><c- f>body</c-><c- p>&gt;</c->
  <c- p>&lt;</c-><c- f>h1</c-><c- p>&gt;</c->Sample page<c- p>&lt;/</c-><c- f>h1</c-><c- p>&gt;</c->
  <c- p>&lt;</c-><c- f>p</c-><c- p>&gt;</c->This is a <c- p>&lt;</c-><c- f>a</c-> <c- e>href</c-><c- o>=</c-><c- s>&quot;demo.html&quot;</c-><c- p>&gt;</c->simple<c- p>&lt;/</c-><c- f>a</c-><c- p>&gt;</c-> sample.<c- p>&lt;/</c-><c- f>p</c-><c- p>&gt;</c->
  <c- c>&lt;!-- this is a comment --&gt;</c->
 <c- p>&lt;/</c-><c- f>body</c-><c- p>&gt;</c->
<c- p>&lt;/</c-><c- f>html</c-><c- p>&gt;</c-></code></pre>

  <p>HTML documents consist of a tree of elements and text. Each element is denoted in the source by
  a <a href=#syntax-start-tag id=a-quick-introduction-to-html:syntax-start-tag>start tag</a>, such as "<code>&lt;body></code>", and
  an <a href=#syntax-end-tag id=a-quick-introduction-to-html:syntax-end-tag>end tag</a>, such as "<code>&lt;/body></code>".
  (Certain start tags and end tags can in certain cases be <a href=#syntax-tag-omission id=a-quick-introduction-to-html:syntax-tag-omission>omitted</a> and are implied by other tags.)</p>

  <p>Tags have to be nested such that elements are all completely within each other, without
  overlapping:</p>

  <pre class=bad><code class='html'><c- p>&lt;</c-><c- f>p</c-><c- p>&gt;</c->This is <c- p>&lt;</c-><c- f>em</c-><c- p>&gt;</c->very <c- p>&lt;</c-><c- f>strong</c-><c- p>&gt;</c->wrong<c- p>&lt;/</c-><c- f>em</c-><c- p>&gt;</c->!<c- p>&lt;/</c-><c- f>strong</c-><c- p>&gt;&lt;/</c-><c- f>p</c-><c- p>&gt;</c-></code></pre>
  <pre><code class='html'><c- p>&lt;</c-><c- f>p</c-><c- p>&gt;</c->This <c- p>&lt;</c-><c- f>em</c-><c- p>&gt;</c->is <c- p>&lt;</c-><c- f>strong</c-><c- p>&gt;</c->correct<c- p>&lt;/</c-><c- f>strong</c-><c- p>&gt;</c->.<c- p>&lt;/</c-><c- f>em</c-><c- p>&gt;&lt;/</c-><c- f>p</c-><c- p>&gt;</c-></code></pre>

  <p>This specification defines a set of elements that can be used in HTML, along with rules about
  the ways in which the elements can be nested.</p>

  <p>Elements can have attributes, which control how the elements work. In the example below, there
  is a <a href=#hyperlink id=a-quick-introduction-to-html:hyperlink>hyperlink</a>, formed using the <code id=a-quick-introduction-to-html:the-a-element><a href=#the-a-element>a</a></code> element and its <code id=a-quick-introduction-to-html:attr-hyperlink-href><a href=#attr-hyperlink-href>href</a></code> attribute:</p>

  <pre><code class='html'><c- p>&lt;</c-><c- f>a</c-> <c- e>href</c-><c- o>=</c-><c- s>&quot;demo.html&quot;</c-><c- p>&gt;</c->simple<c- p>&lt;/</c-><c- f>a</c-><c- p>&gt;</c-></code></pre>

  <p><a href=#syntax-attributes id=a-quick-introduction-to-html:syntax-attributes>Attributes</a> are placed inside the start tag, and consist
  of a <a href=#syntax-attribute-name id=a-quick-introduction-to-html:syntax-attribute-name>name</a> and a <a href=#syntax-attribute-value id=a-quick-introduction-to-html:syntax-attribute-value>value</a>, separated by an "<code>=</code>" character.
  The attribute value can remain <a href=#unquoted>unquoted</a> if it doesn't contain <a id=a-quick-introduction-to-html:space-characters href=https://infra.spec.whatwg.org/#ascii-whitespace data-x-internal=space-characters>ASCII
  whitespace</a> or any of <code>"</code> <code>'</code> <code>`</code> <code>=</code> <code>&lt;</code> or <code>></code>. Otherwise, it has to be quoted using either single or double quotes. The
  value, along with the "<code>=</code>" character, can be omitted altogether if the value
  is the empty string.</p>

  <pre><code class='html'><c- c>&lt;!-- empty attributes --&gt;</c->
<c- p>&lt;</c-><c- f>input</c-> <c- e>name</c-><c- o>=</c-><c- s>address</c-> <c- e>disabled</c-><c- p>&gt;</c->
<c- p>&lt;</c-><c- f>input</c-> <c- e>name</c-><c- o>=</c-><c- s>address</c-> <c- e>disabled</c-><c- o>=</c-><c- s>&quot;&quot;</c-><c- p>&gt;</c->

<c- c>&lt;!-- attributes with a value --&gt;</c->
<c- p>&lt;</c-><c- f>input</c-> <c- e>name</c-><c- o>=</c-><c- s>address</c-> <c- e>maxlength</c-><c- o>=</c-><c- s>200</c-><c- p>&gt;</c->
<c- p>&lt;</c-><c- f>input</c-> <c- e>name</c-><c- o>=</c-><c- s>address</c-> <c- e>maxlength</c-><c- o>=</c-><c- s>&apos;200&apos;</c-><c- p>&gt;</c->
<c- p>&lt;</c-><c- f>input</c-> <c- e>name</c-><c- o>=</c-><c- s>address</c-> <c- e>maxlength</c-><c- o>=</c-><c- s>&quot;200&quot;</c-><c- p>&gt;</c-></code></pre>

  <p>HTML user agents (e.g., web browsers) then <i>parse</i> this markup, turning it into a DOM
  (Document Object Model) tree. A DOM tree is an in-memory representation of a document.</p>

  <p>DOM trees contain several kinds of nodes, in particular a <code id=a-quick-introduction-to-html:documenttype><a data-x-internal=documenttype href=https://dom.spec.whatwg.org/#interface-documenttype>DocumentType</a></code> node,
  <code id=a-quick-introduction-to-html:element><a data-x-internal=element href=https://dom.spec.whatwg.org/#interface-element>Element</a></code> nodes, <code id=a-quick-introduction-to-html:text><a data-x-internal=text href=https://dom.spec.whatwg.org/#interface-text>Text</a></code> nodes, <code id=a-quick-introduction-to-html:comment-2><a data-x-internal=comment-2 href=https://dom.spec.whatwg.org/#interface-comment>Comment</a></code> nodes, and in some cases
  <code id=a-quick-introduction-to-html:processinginstruction><a data-x-internal=processinginstruction href=https://dom.spec.whatwg.org/#interface-processinginstruction>ProcessingInstruction</a></code> nodes.</p>

  <p>The <a href=#intro-early-example>markup snippet at the top of this section</a> would be
  turned into the following DOM tree:</p>

  <ul class=domTree><li class=t10>DOCTYPE: <code>html</code><li class=t1><code id=a-quick-introduction-to-html:the-html-element><a href=#the-html-element>html</a></code> <span class=t2><code id=a-quick-introduction-to-html:attr-lang class="attribute name"><a href=#attr-lang>lang</a></code>="<code class="attribute value">en</code>"</span><ul><li class=t1><code id=a-quick-introduction-to-html:the-head-element><a href=#the-head-element>head</a></code><ul><li class=t3><code id=a-quick-introduction-to-html:text-2><a data-x-internal=text href=https://dom.spec.whatwg.org/#interface-text>#text</a></code>: <span>⏎␣␣</span><li class=t1><code id=a-quick-introduction-to-html:the-title-element><a href=#the-title-element>title</a></code><ul><li class=t3><code id=a-quick-introduction-to-html:text-3><a data-x-internal=text href=https://dom.spec.whatwg.org/#interface-text>#text</a></code>: <span>Sample page</span></ul><li class=t3><code id=a-quick-introduction-to-html:text-4><a data-x-internal=text href=https://dom.spec.whatwg.org/#interface-text>#text</a></code>: <span>⏎␣</span></ul><li class=t3><code id=a-quick-introduction-to-html:text-5><a data-x-internal=text href=https://dom.spec.whatwg.org/#interface-text>#text</a></code>: <span>⏎␣</span><li class=t1><code id=a-quick-introduction-to-html:the-body-element><a href=#the-body-element>body</a></code><ul><li class=t3><code id=a-quick-introduction-to-html:text-6><a data-x-internal=text href=https://dom.spec.whatwg.org/#interface-text>#text</a></code>: <span>⏎␣␣</span><li class=t1><code id=a-quick-introduction-to-html:the-h1,-h2,-h3,-h4,-h5,-and-h6-elements><a href=#the-h1,-h2,-h3,-h4,-h5,-and-h6-elements>h1</a></code><ul><li class=t3><code id=a-quick-introduction-to-html:text-7><a data-x-internal=text href=https://dom.spec.whatwg.org/#interface-text>#text</a></code>: <span>Sample page</span></ul><li class=t3><code id=a-quick-introduction-to-html:text-8><a data-x-internal=text href=https://dom.spec.whatwg.org/#interface-text>#text</a></code>: <span>⏎␣␣</span><li class=t1><code id=a-quick-introduction-to-html:the-p-element><a href=#the-p-element>p</a></code><ul><li class=t3><code id=a-quick-introduction-to-html:text-9><a data-x-internal=text href=https://dom.spec.whatwg.org/#interface-text>#text</a></code>: <span>This is a </span><li class=t1><code id=a-quick-introduction-to-html:the-a-element-2><a href=#the-a-element>a</a></code> <span class=t2><code id=a-quick-introduction-to-html:attr-hyperlink-href-2 class="attribute name"><a href=#attr-hyperlink-href>href</a></code>="<code class="attribute value">demo.html</code>"</span><ul><li class=t3><code id=a-quick-introduction-to-html:text-10><a data-x-internal=text href=https://dom.spec.whatwg.org/#interface-text>#text</a></code>: <span>simple</span></ul><li class=t3><code id=a-quick-introduction-to-html:text-11><a data-x-internal=text href=https://dom.spec.whatwg.org/#interface-text>#text</a></code>: <span> sample.</span></ul><li class=t3><code id=a-quick-introduction-to-html:text-12><a data-x-internal=text href=https://dom.spec.whatwg.org/#interface-text>#text</a></code>: <span>⏎␣␣</span><li class=t8><code id=a-quick-introduction-to-html:comment-2-2><a data-x-internal=comment-2 href=https://dom.spec.whatwg.org/#interface-comment>#comment</a></code>: <span> this is a comment </span><li class=t3><code id=a-quick-introduction-to-html:text-13><a data-x-internal=text href=https://dom.spec.whatwg.org/#interface-text>#text</a></code>: <span>⏎␣⏎</span></ul></ul></ul>

  <p>The <a id=a-quick-introduction-to-html:document-element href=https://dom.spec.whatwg.org/#document-element data-x-internal=document-element>document element</a> of this tree is the <code id=a-quick-introduction-to-html:the-html-element-2><a href=#the-html-element>html</a></code> element, which is the
  element always found in that position in HTML documents. It contains two elements,
  <code id=a-quick-introduction-to-html:the-head-element-2><a href=#the-head-element>head</a></code> and <code id=a-quick-introduction-to-html:the-body-element-2><a href=#the-body-element>body</a></code>, as well as a <code id=a-quick-introduction-to-html:text-14><a data-x-internal=text href=https://dom.spec.whatwg.org/#interface-text>Text</a></code> node between them.</p>

  <p>There are many more <code id=a-quick-introduction-to-html:text-15><a data-x-internal=text href=https://dom.spec.whatwg.org/#interface-text>Text</a></code> nodes in the DOM tree than one would initially expect,
  because the source contains a number of spaces (represented here by "␣") and line breaks
  ("⏎") that all end up as <code id=a-quick-introduction-to-html:text-16><a data-x-internal=text href=https://dom.spec.whatwg.org/#interface-text>Text</a></code> nodes in the DOM. However, for historical
  reasons not all of the spaces and line breaks in the original markup appear in the DOM. In
  particular, all the whitespace before <code id=a-quick-introduction-to-html:the-head-element-3><a href=#the-head-element>head</a></code> start tag ends up being dropped silently,
  and all the whitespace after the <code id=a-quick-introduction-to-html:the-body-element-3><a href=#the-body-element>body</a></code> end tag ends up placed at the end of the
  <code id=a-quick-introduction-to-html:the-body-element-4><a href=#the-body-element>body</a></code>.</p>

  <p>The <code id=a-quick-introduction-to-html:the-head-element-4><a href=#the-head-element>head</a></code> element contains a <code id=a-quick-introduction-to-html:the-title-element-2><a href=#the-title-element>title</a></code> element, which itself contains a
  <code id=a-quick-introduction-to-html:text-17><a data-x-internal=text href=https://dom.spec.whatwg.org/#interface-text>Text</a></code> node with the text "Sample page". Similarly, the <code id=a-quick-introduction-to-html:the-body-element-5><a href=#the-body-element>body</a></code> element
  contains an <code id=a-quick-introduction-to-html:the-h1,-h2,-h3,-h4,-h5,-and-h6-elements-2><a href=#the-h1,-h2,-h3,-h4,-h5,-and-h6-elements>h1</a></code> element, a <code id=a-quick-introduction-to-html:the-p-element-2><a href=#the-p-element>p</a></code> element, and a comment.</p>

  <hr>

  <p>This DOM tree can be manipulated from scripts in the page. Scripts (typically in JavaScript)
  are small programs that can be embedded using the <code id=a-quick-introduction-to-html:the-script-element><a href=#the-script-element>script</a></code> element or using <a href=#event-handler-content-attributes id=a-quick-introduction-to-html:event-handler-content-attributes>event
  handler content attributes</a>. For example, here is a form with a script that sets the value
  of the form's <code id=a-quick-introduction-to-html:the-output-element><a href=#the-output-element>output</a></code> element to say "Hello World":</p>

  <pre><code class='html'><c- p>&lt;</c-><a href='#the-form-element' id='a-quick-introduction-to-html:the-form-element'><c- f>form</c-></a> <a href='#attr-form-name' id='a-quick-introduction-to-html:attr-form-name'><c- e>name</c-></a><c- o>=</c-><c- s>&quot;main&quot;</c-><c- p>&gt;</c->
 Result: <c- p>&lt;</c-><a href='#the-output-element' id='a-quick-introduction-to-html:the-output-element-2'><c- f>output</c-></a> <a href='#attr-fe-name' id='a-quick-introduction-to-html:attr-fe-name'><c- e>name</c-></a><c- o>=</c-><c- s>&quot;result&quot;</c-><c- p>&gt;&lt;/</c-><c- f>output</c-><c- p>&gt;</c->
 <c- p>&lt;</c-><a href='#the-script-element' id='a-quick-introduction-to-html:the-script-element-2'><c- f>script</c-></a><c- p>&gt;</c->
  <a href='#document' id='a-quick-introduction-to-html:document'>document</a><c- p>.</c-><a href='#dom-document-forms' id='a-quick-introduction-to-html:dom-document-forms'>forms</a><c- p>.</c->main<c- p>.</c-><a href='#dom-form-elements' id='a-quick-introduction-to-html:dom-form-elements'>elements</a><c- p>.</c->result<c- p>.</c-><a href='#dom-output-value' id='a-quick-introduction-to-html:dom-output-value'>value</a> <c- o>=</c-> <c- t>&apos;Hello World&apos;</c-><c- p>;</c->
 <c- p>&lt;/</c-><c- f>script</c-><c- p>&gt;</c->
<c- p>&lt;/</c-><c- f>form</c-><c- p>&gt;</c-></code></pre>

  <p>Each element in the DOM tree is represented by an object, and these objects have APIs so that
  they can be manipulated. For instance, a link (e.g. the <code id=a-quick-introduction-to-html:the-a-element-3><a href=#the-a-element>a</a></code> element in the tree above)
  can have its "<code id=a-quick-introduction-to-html:attr-hyperlink-href-3><a href=#attr-hyperlink-href>href</a></code>" attribute changed in several
  ways:</p>

  <pre><code class='js'><c- a>var</c-> a <c- o>=</c-> <a href='#document' id='a-quick-introduction-to-html:document-2'>document</a><c- p>.</c-><a href='#dom-document-links' id='a-quick-introduction-to-html:dom-document-links'>links</a><c- p>[</c-><c- mi>0</c-><c- p>];</c-> <c- c1>// obtain the first link in the document</c->
a<c- p>.</c-><a href='#dom-hyperlink-href' id='a-quick-introduction-to-html:dom-hyperlink-href'>href</a> <c- o>=</c-> <c- t>&apos;sample.html&apos;</c-><c- p>;</c-> <c- c1>// change the destination URL of the link</c->
a<c- p>.</c-><a href='#dom-hyperlink-protocol' id='a-quick-introduction-to-html:dom-hyperlink-protocol'>protocol</a> <c- o>=</c-> <c- t>&apos;https&apos;</c-><c- p>;</c-> <c- c1>// change just the scheme part of the URL</c->
a<c- p>.</c->setAttribute<c- p>(</c-><c- t>&apos;href&apos;</c-><c- p>,</c-> <c- t>&apos;https://example.com/&apos;</c-><c- p>);</c-> <c- c1>// change the content attribute directly</c-></code></pre>

  <p>Since DOM trees are used as the way to represent HTML documents when they are processed and
  presented by implementations (especially interactive implementations like web browsers), this
  specification is mostly phrased in terms of DOM trees, instead of the markup described above.</p>

  <hr>

  <p>HTML documents represent a media-independent description of interactive content. HTML documents
  might be rendered to a screen, or through a speech synthesizer, or on a braille display. To
  influence exactly how such rendering takes place, authors can use a styling language such as
  CSS.</p>

  <p>In the following example, the page has been made yellow-on-blue using CSS.</p>

  <pre><code class='html'><c- cp>&lt;!DOCTYPE html&gt;</c->
<c- p>&lt;</c-><c- f>html</c-> <c- e>lang</c-><c- o>=</c-><c- s>&quot;en&quot;</c-><c- p>&gt;</c->
 <c- p>&lt;</c-><c- f>head</c-><c- p>&gt;</c->
  <c- p>&lt;</c-><c- f>title</c-><c- p>&gt;</c->Sample styled page<c- p>&lt;/</c-><c- f>title</c-><c- p>&gt;</c->
  <c- p>&lt;</c-><c- f>style</c-><c- p>&gt;</c->
   <c- f>body</c-> <c- p>{</c-> <c- k>background</c-><c- p>:</c-> <c- kc>navy</c-><c- p>;</c-> <c- k>color</c-><c- p>:</c-> <c- kc>yellow</c-><c- p>;</c-> <c- p>}</c->
  <c- p>&lt;/</c-><c- f>style</c-><c- p>&gt;</c->
 <c- p>&lt;/</c-><c- f>head</c-><c- p>&gt;</c->
 <c- p>&lt;</c-><c- f>body</c-><c- p>&gt;</c->
  <c- p>&lt;</c-><c- f>h1</c-><c- p>&gt;</c->Sample styled page<c- p>&lt;/</c-><c- f>h1</c-><c- p>&gt;</c->
  <c- p>&lt;</c-><c- f>p</c-><c- p>&gt;</c->This page is just a demo.<c- p>&lt;/</c-><c- f>p</c-><c- p>&gt;</c->
 <c- p>&lt;/</c-><c- f>body</c-><c- p>&gt;</c->
<c- p>&lt;/</c-><c- f>html</c-><c- p>&gt;</c-></code></pre>

  <p>For more details on how to use HTML, authors are encouraged to consult tutorials and guides.
  Some of the examples included in this specification might also be of use, but the novice author is
  cautioned that this specification, by necessity, defines the language with a level of detail that
  might be difficult to understand at first.</p>




  <h4 id=writing-secure-applications-with-html><span class=secno>1.10.1</span> Writing secure applications with HTML<a href=#writing-secure-applications-with-html class=self-link></a></h4>

  <p><i>This section is non-normative.</i></p>

  <p>When HTML is used to create interactive sites, care needs to be taken to avoid introducing
  vulnerabilities through which attackers can compromise the integrity of the site itself or of the
  site's users.</p>

  <p>A comprehensive study of this matter is beyond the scope of this document, and authors are
  strongly encouraged to study the matter in more detail. However, this section attempts to provide
  a quick introduction to some common pitfalls in HTML application development.</p>

  <p>The security model of the web is based on the concept of "origins", and correspondingly many of
  the potential attacks on the web involve cross-origin actions. <a href=#refsORIGIN>[ORIGIN]</a></p>

  <dl><dt>Not validating user input<dt>Cross-site scripting (XSS)<dt>SQL injection<dd>
    <p>When accepting untrusted input, e.g. user-generated content such as text comments, values in
    URL parameters, messages from third-party sites, etc, it is imperative that the data be
    validated before use, and properly escaped when displayed. Failing to do this can allow a
    hostile user to perform a variety of attacks, ranging from the potentially benign, such as
    providing bogus user information like a negative age, to the serious, such as running scripts
    every time a user looks at a page that includes the information, potentially propagating the
    attack in the process, to the catastrophic, such as deleting all data in the server.</p>

    <p>When writing filters to validate user input, it is imperative that filters always be
    safelist-based, allowing known-safe constructs and disallowing all other input. Blocklist-based
    filters that disallow known-bad inputs and allow everything else are not secure, as not
    everything that is bad is yet known (for example, because it might be invented in the
    future).</p>

    <div class=example>

     <p>For example, suppose a page looked at its URL's query string to determine what to display,
     and the site then redirected the user to that page to display a message, as in:</p>

     <pre><code class='html'><c- p>&lt;</c-><c- f>ul</c-><c- p>&gt;</c->
 <c- p>&lt;</c-><c- f>li</c-><c- p>&gt;&lt;</c-><c- f>a</c-> <c- e>href</c-><c- o>=</c-><c- s>&quot;message.cgi?say=Hello&quot;</c-><c- p>&gt;</c->Say Hello<c- p>&lt;/</c-><c- f>a</c-><c- p>&gt;</c->
 <c- p>&lt;</c-><c- f>li</c-><c- p>&gt;&lt;</c-><c- f>a</c-> <c- e>href</c-><c- o>=</c-><c- s>&quot;message.cgi?say=Welcome&quot;</c-><c- p>&gt;</c->Say Welcome<c- p>&lt;/</c-><c- f>a</c-><c- p>&gt;</c->
 <c- p>&lt;</c-><c- f>li</c-><c- p>&gt;&lt;</c-><c- f>a</c-> <c- e>href</c-><c- o>=</c-><c- s>&quot;message.cgi?say=Kittens&quot;</c-><c- p>&gt;</c->Say Kittens<c- p>&lt;/</c-><c- f>a</c-><c- p>&gt;</c->
<c- p>&lt;/</c-><c- f>ul</c-><c- p>&gt;</c-></code></pre>

     <p>If the message was just displayed to the user without escaping, a hostile attacker could
     then craft a URL that contained a script element:</p>

     <pre>https://example.com/message.cgi?say=%3Cscript%3Ealert%28%27Oh%20no%21%27%29%3C/script%3E</pre>

     <p>If the attacker then convinced a victim user to visit this page, a script of the attacker's
     choosing would run on the page. Such a script could do any number of hostile actions, limited
     only by what the site offers: if the site is an e-commerce shop, for instance, such a script
     could cause the user to unknowingly make arbitrarily many unwanted purchases.</p>

     <p>This is called a cross-site scripting attack.</p>

    </div>

    <p>There are many constructs that can be used to try to trick a site into executing code. Here
    are some that authors are encouraged to consider when writing safelist filters:</p>

    <ul><li>When allowing harmless-seeming elements like <code id=writing-secure-applications-with-html:the-img-element><a href=#the-img-element>img</a></code>, it is important to safelist
     any provided attributes as well. If one allowed all attributes then an attacker could, for
     instance, use the <code id=writing-secure-applications-with-html:handler-onload><a href=#handler-onload>onload</a></code> attribute to run arbitrary
     script.<li>When allowing URLs to be provided (e.g. for links), the scheme of each URL also needs to be
     explicitly safelisted, as there are many schemes that can be abused. The most prominent
     example is "<code id=writing-secure-applications-with-html:javascript-protocol><a href=#javascript-protocol>javascript:</a></code>", but user agents can
     implement (and indeed, have historically implemented) others.<li>Allowing a <code id=writing-secure-applications-with-html:the-base-element><a href=#the-base-element>base</a></code> element to be inserted means any <code id=writing-secure-applications-with-html:the-script-element><a href=#the-script-element>script</a></code> elements
     in the page with relative links can be hijacked, and similarly that any form submissions can
     get redirected to a hostile site.</ul>
   <dt>Cross-site request forgery (CSRF)<dd>
    <p>If a site allows a user to make form submissions with user-specific side-effects, for example
    posting messages on a forum under the user's name, making purchases, or applying for a passport,
    it is important to verify that the request was made by the user intentionally, rather than by
    another site tricking the user into making the request unknowingly.</p>

    <p>This problem exists because HTML forms can be submitted to other origins.</p>

    <p>Sites can prevent such attacks by populating forms with user-specific hidden tokens, or by
    checking `<code id=writing-secure-applications-with-html:http-origin><a data-x-internal=http-origin href=https://fetch.spec.whatwg.org/#http-origin>Origin</a></code>` headers on all requests.</p>
   <dt>Clickjacking<dd>
    <p>A page that provides users with an interface to perform actions that the user might not wish
    to perform needs to be designed so as to avoid the possibility that users can be tricked into
    activating the interface.</p>

    <p>One way that a user could be so tricked is if a hostile site places the victim site in a
    small <code id=writing-secure-applications-with-html:the-iframe-element><a href=#the-iframe-element>iframe</a></code> and then convinces the user to click, for instance by having the user
    play a reaction game. Once the user is playing the game, the hostile site can quickly position
    the iframe under the mouse cursor just as the user is about to click, thus tricking the user
    into clicking the victim site's interface.</p>

    <p>To avoid this, sites that do not expect to be used in frames are encouraged to only enable
    their interface if they detect that they are not in a frame (e.g. by comparing the <code id=writing-secure-applications-with-html:dom-window><a href=#dom-window>window</a></code> object to the value of the <code id=writing-secure-applications-with-html:dom-top><a href=#dom-top>top</a></code>
    attribute).</p>
   </dl>



  <h4 id=common-pitfalls-to-avoid-when-using-the-scripting-apis><span class=secno>1.10.2</span> Common pitfalls to avoid when using the scripting APIs<a href=#common-pitfalls-to-avoid-when-using-the-scripting-apis class=self-link></a></h4>

  <p><i>This section is non-normative.</i></p>

  <p>Scripts in HTML have "run-to-completion" semantics, meaning that the browser will generally run
  the script uninterrupted before doing anything else, such as firing further events or continuing
  to parse the document.</p>

  <p>On the other hand, parsing of HTML files happens incrementally, meaning that
  the parser can pause at any point to let scripts run. This is generally a good thing, but it does
  mean that authors need to be careful to avoid hooking event handlers after the events could have
  possibly fired.</p>

  <p>There are two techniques for doing this reliably: use <a href=#event-handler-content-attributes id=common-pitfalls-to-avoid-when-using-the-scripting-apis:event-handler-content-attributes>event handler content
  attributes</a>, or create the element and add the event handlers in the same script. The latter
  is safe because, as mentioned earlier, scripts are run to completion before further events can
  fire.</p>

  <div class=example>

   <p>One way this could manifest itself is with <code id=common-pitfalls-to-avoid-when-using-the-scripting-apis:the-img-element><a href=#the-img-element>img</a></code> elements and the <code id=common-pitfalls-to-avoid-when-using-the-scripting-apis:event-load><a href=#event-load>load</a></code> event. The event could fire as soon as the element has been
   parsed, especially if the image has already been cached (which is common).</p>

   <p>Here, the author uses the <code id=common-pitfalls-to-avoid-when-using-the-scripting-apis:handler-onload><a href=#handler-onload>onload</a></code> handler on an
   <code id=common-pitfalls-to-avoid-when-using-the-scripting-apis:the-img-element-2><a href=#the-img-element>img</a></code> element to catch the <code id=common-pitfalls-to-avoid-when-using-the-scripting-apis:event-load-2><a href=#event-load>load</a></code> event:</p>

   <pre><code class='html'><c- p>&lt;</c-><c- f>img</c-> <c- e>src</c-><c- o>=</c-><c- s>&quot;games.png&quot;</c-> <c- e>alt</c-><c- o>=</c-><c- s>&quot;Games&quot;</c-> <c- e>onload</c-><c- o>=</c-><c- s>&quot;gamesLogoHasLoaded(event)&quot;</c-><c- p>&gt;</c-></code></pre>

   <p>If the element is being added by script, then so long as the event handlers are added in the
   same script, the event will still not be missed:</p>

   <pre><code class='html'><c- p>&lt;</c-><c- f>script</c-><c- p>&gt;</c->
 <c- a>var</c-> img <c- o>=</c-> <c- k>new</c-> Image<c- p>();</c->
 img<c- p>.</c->src <c- o>=</c-> <c- t>&apos;games.png&apos;</c-><c- p>;</c->
 img<c- p>.</c->alt <c- o>=</c-> <c- t>&apos;Games&apos;</c-><c- p>;</c->
 img<c- p>.</c->onload <c- o>=</c-> gamesLogoHasLoaded<c- p>;</c->
 <c- c1>// img.addEventListener(&apos;load&apos;, gamesLogoHasLoaded, false); // would work also</c->
<c- p>&lt;/</c-><c- f>script</c-><c- p>&gt;</c-></code></pre>

   <p>However, if the author first created the <code id=common-pitfalls-to-avoid-when-using-the-scripting-apis:the-img-element-3><a href=#the-img-element>img</a></code> element and then in a separate
   script added the event listeners, there's a chance that the <code id=common-pitfalls-to-avoid-when-using-the-scripting-apis:event-load-3><a href=#event-load>load</a></code>
   event would be fired in between, leading it to be missed:</p>

   <pre class=bad><code class='html'><c- c>&lt;!-- Do not use this style, it has a race condition! --&gt;</c->
 <c- p>&lt;</c-><c- f>img</c-> <c- e>id</c-><c- o>=</c-><c- s>&quot;games&quot;</c-> <c- e>src</c-><c- o>=</c-><c- s>&quot;games.png&quot;</c-> <c- e>alt</c-><c- o>=</c-><c- s>&quot;Games&quot;</c-><c- p>&gt;</c->
 <c- c>&lt;!-- the &apos;load&apos; event might fire here while the parser is taking a</c->
<c- c>      break, in which case you will not see it! --&gt;</c->
 <c- p>&lt;</c-><c- f>script</c-><c- p>&gt;</c->
  <c- a>var</c-> img <c- o>=</c-> document<c- p>.</c->getElementById<c- p>(</c-><c- t>&apos;games&apos;</c-><c- p>);</c->
  img<c- p>.</c->onload <c- o>=</c-> gamesLogoHasLoaded<c- p>;</c-> <c- c1>// might never fire!</c->
 <c- p>&lt;/</c-><c- f>script</c-><c- p>&gt;</c-></code></pre>

  </div>



  <h4 id=how-to-catch-mistakes-when-writing-html:-validators-and-conformance-checkers><span class=secno>1.10.3</span> How to catch mistakes when writing HTML: validators and conformance checkers<a href=#how-to-catch-mistakes-when-writing-html:-validators-and-conformance-checkers class=self-link></a></h4>

  <p><i>This section is non-normative.</i></p>

  <p>Authors are encouraged to make use of conformance checkers (also known as <i>validators</i>) to
  catch common mistakes. The WHATWG maintains a list of such tools at: <a href=https://whatwg.org/validator/>https://whatwg.org/validator/</a></p>



  <h3 id=conformance-requirements-for-authors><span class=secno>1.11</span> Conformance requirements for authors<a href=#conformance-requirements-for-authors class=self-link></a></h3>

  <p><i>This section is non-normative.</i></p>

  <p>Unlike previous versions of the HTML specification, this specification defines in some detail
  the required processing for invalid documents as well as valid documents.</p> 

  <p>However, even though the processing of invalid content is in most cases well-defined,
  conformance requirements for documents are still important: in practice, interoperability (the
  situation in which all implementations process particular content in a reliable and identical or
  equivalent way) is not the only goal of document conformance requirements. This section details
  some of the more common reasons for still distinguishing between a conforming document and one
  with errors.</p>


  <h4 id=presentational-markup><span class=secno>1.11.1</span> Presentational markup<a href=#presentational-markup class=self-link></a></h4>

  <p><i>This section is non-normative.</i></p>

  <p>The majority of presentational features from previous versions of HTML are no longer allowed.
  Presentational markup in general has been found to have a number of problems:</p>

  <dl><dt>The use of presentational elements leads to poorer accessibility<dd>
    <p>While it is possible to use presentational markup in a way that provides users of assistive
    technologies (ATs) with an acceptable experience (e.g. using ARIA), doing so is significantly
    more difficult than doing so when using semantically-appropriate markup. Furthermore, even using
    such techniques doesn't help make pages accessible for non-AT non-graphical users, such as users
    of text-mode browsers.</p>

    <p>Using media-independent markup, on the other hand, provides an easy way for documents to be
    authored in such a way that they work for more users (e.g. users of text browsers).</p>
   <dt>Higher cost of maintenance<dd>
    <p>It is significantly easier to maintain a site written in such a way that the markup is
    style-independent. For example, changing the color of a site that uses <code>&lt;font color=""></code> throughout requires changes across the entire site,
    whereas a similar change to a site based on CSS can be done by changing a single file.</p>
   <dt>Larger document sizes<dd>
    <p>Presentational markup tends to be much more redundant, and thus results in larger document
    sizes.</p>
   </dl>

  <p>For those reasons, presentational markup has been removed from HTML in this version. This
  change should not come as a surprise; HTML4 deprecated presentational markup many years ago and
  provided a mode (HTML4 Transitional) to help authors move away from presentational markup; later,
  XHTML 1.1 went further and obsoleted those features altogether.</p>

  <p>The only remaining presentational markup features in HTML are the <code id=presentational-markup:attr-style><a href=#attr-style>style</a></code> attribute and the <code id=presentational-markup:the-style-element><a href=#the-style-element>style</a></code> element. Use of the <code id=presentational-markup:attr-style-2><a href=#attr-style>style</a></code> attribute is somewhat discouraged in production environments, but
  it can be useful for rapid prototyping (where its rules can be directly moved into a separate
  style sheet later) and for providing specific styles in unusual cases where a separate style sheet
  would be inconvenient. Similarly, the <code id=presentational-markup:the-style-element-2><a href=#the-style-element>style</a></code> element can be useful in syndication or
  for page-specific styles, but in general an external style sheet is likely to be more convenient
  when the styles apply to multiple pages.</p>

  <p>It is also worth noting that some elements that were previously presentational have been
  redefined in this specification to be media-independent: <code id=presentational-markup:the-b-element><a href=#the-b-element>b</a></code>, <code id=presentational-markup:the-i-element><a href=#the-i-element>i</a></code>,
  <code id=presentational-markup:the-hr-element><a href=#the-hr-element>hr</a></code>, <code id=presentational-markup:the-s-element><a href=#the-s-element>s</a></code>, <code id=presentational-markup:the-small-element><a href=#the-small-element>small</a></code>, and <code id=presentational-markup:the-u-element><a href=#the-u-element>u</a></code>.</p>


  <h4 id=syntax-errors><span class=secno>1.11.2</span> Syntax errors<a href=#syntax-errors class=self-link></a></h4>

  <p><i>This section is non-normative.</i></p>

  <p>The syntax of HTML is constrained to avoid a wide variety of problems.</p>

  <dl><dt>Unintuitive error-handling behavior<dd>
    <p>Certain invalid syntax constructs, when parsed, result in DOM trees that are highly
    unintuitive.</p>

    <div class=example>

     <p>For example, the following markup fragment results in a DOM with an <code id=syntax-errors:the-hr-element><a href=#the-hr-element>hr</a></code> element
     that is an <em>earlier</em> sibling of the corresponding <code id=syntax-errors:the-table-element><a href=#the-table-element>table</a></code> element:</p>

     <pre class=bad><code class='html'><c- p>&lt;</c-><c- f>table</c-><c- p>&gt;&lt;</c-><c- f>hr</c-><c- p>&gt;</c->...</code></pre>

    </div>
   <dt>Errors with optional error recovery<dd>
    <p>To allow user agents to be used in controlled environments without having to implement the
    more bizarre and convoluted error handling rules, user agents are permitted to fail whenever
    encountering a <a href=#parse-errors id=syntax-errors:parse-errors>parse error</a>.</p>
   <dt>Errors where the error-handling behavior is not compatible with streaming user agents<dd>
    <p>Some error-handling behavior, such as the behavior for the <code>&lt;table>&lt;hr>...</code> example mentioned above, are incompatible with streaming
    user agents (user agents that process HTML files in one pass, without storing state). To avoid
    interoperability problems with such user agents, any syntax resulting in such behavior is
    considered invalid.</p>
   <dt>Errors that can result in infoset coercion<dd>
    <p>When a user agent based on XML is connected to an HTML parser, it is possible that certain
    invariants that XML enforces, such as element or attribute names never contain multiple colons,
    will be violated by an HTML file. Handling this can require that the parser coerce the HTML DOM
    into an XML-compatible infoset. Most syntax constructs that require such handling are considered
    invalid. (Comments containing two consecutive hyphens, or ending with a hyphen, are exceptions
    that are allowed in the HTML syntax.)</p>
   <dt>Errors that result in disproportionately poor performance<dd>
    <p>Certain syntax constructs can result in disproportionately poor performance. To discourage the
    use of such constructs, they are typically made non-conforming.</p>

    <div class=example>

     <p>For example, the following markup results in poor performance, since all the unclosed
     <code id=syntax-errors:the-i-element><a href=#the-i-element>i</a></code> elements have to be reconstructed in each paragraph, resulting in progressively
     more elements in each paragraph:</p>

     <pre class=bad><code class='html'><c- p>&lt;</c-><c- f>p</c-><c- p>&gt;&lt;</c-><c- f>i</c-><c- p>&gt;</c->She dreamt.
<c- p>&lt;</c-><c- f>p</c-><c- p>&gt;&lt;</c-><c- f>i</c-><c- p>&gt;</c->She dreamt that she ate breakfast.
<c- p>&lt;</c-><c- f>p</c-><c- p>&gt;&lt;</c-><c- f>i</c-><c- p>&gt;</c->Then lunch.
<c- p>&lt;</c-><c- f>p</c-><c- p>&gt;&lt;</c-><c- f>i</c-><c- p>&gt;</c->And finally dinner.</code></pre>

     <p>The resulting DOM for this fragment would be:</p>

     <ul class=domTree><li class=t1><code id=syntax-errors:the-p-element><a href=#the-p-element>p</a></code><ul><li class=t1><code id=syntax-errors:the-i-element-2><a href=#the-i-element>i</a></code><ul><li class=t3><code id=syntax-errors:text><a data-x-internal=text href=https://dom.spec.whatwg.org/#interface-text>#text</a></code>: <span>She dreamt.</span></ul></ul><li class=t1><code id=syntax-errors:the-p-element-2><a href=#the-p-element>p</a></code><ul><li class=t1><code id=syntax-errors:the-i-element-3><a href=#the-i-element>i</a></code><ul><li class=t1><code id=syntax-errors:the-i-element-4><a href=#the-i-element>i</a></code><ul><li class=t3><code id=syntax-errors:text-2><a data-x-internal=text href=https://dom.spec.whatwg.org/#interface-text>#text</a></code>: <span>She dreamt that she ate breakfast.</span></ul></ul></ul><li class=t1><code id=syntax-errors:the-p-element-3><a href=#the-p-element>p</a></code><ul><li class=t1><code id=syntax-errors:the-i-element-5><a href=#the-i-element>i</a></code><ul><li class=t1><code id=syntax-errors:the-i-element-6><a href=#the-i-element>i</a></code><ul><li class=t1><code id=syntax-errors:the-i-element-7><a href=#the-i-element>i</a></code><ul><li class=t3><code id=syntax-errors:text-3><a data-x-internal=text href=https://dom.spec.whatwg.org/#interface-text>#text</a></code>: <span>Then lunch.</span></ul></ul></ul></ul><li class=t1><code id=syntax-errors:the-p-element-4><a href=#the-p-element>p</a></code><ul><li class=t1><code id=syntax-errors:the-i-element-8><a href=#the-i-element>i</a></code><ul><li class=t1><code id=syntax-errors:the-i-element-9><a href=#the-i-element>i</a></code><ul><li class=t1><code id=syntax-errors:the-i-element-10><a href=#the-i-element>i</a></code><ul><li class=t1><code id=syntax-errors:the-i-element-11><a href=#the-i-element>i</a></code><ul><li class=t3><code id=syntax-errors:text-4><a data-x-internal=text href=https://dom.spec.whatwg.org/#interface-text>#text</a></code>: <span>And finally dinner.</span></ul></ul></ul></ul></ul></ul>

    </div>
   <dt>Errors involving fragile syntax constructs<dd>
    <p>There are syntax constructs that, for historical reasons, are relatively fragile. To help
    reduce the number of users who accidentally run into such problems, they are made
    non-conforming.</p>

    <div class=example>

     <p>For example, the parsing of certain named character references in attributes happens even
     with the closing semicolon being omitted. It is safe to include an ampersand followed by
     letters that do not form a named character reference, but if the letters are changed to a
     string that <em>does</em> form a named character reference, they will be interpreted as that
     character instead.</p>

     <p>In this fragment, the attribute's value is "<code>?bill&amp;ted</code>":</p>

     <pre class=bad><code class='html'><c- p>&lt;</c-><c- f>a</c-> <c- e>href</c-><c- o>=</c-><c- s>&quot;?bill&amp;ted&quot;</c-><c- p>&gt;</c->Bill and Ted<c- p>&lt;/</c-><c- f>a</c-><c- p>&gt;</c-></code></pre>

     <p>In the following fragment, however, the attribute's value is actually "<code>?art©</code>", <em>not</em> the intended "<code>?art&amp;copy</code>",
     because even without the final semicolon, "<code>&amp;copy</code>" is handled the same
     as "<code>&amp;copy;</code>" and thus gets interpreted as "<code>©</code>":</p>

     <pre class=bad><code class='html'><c- p>&lt;</c-><c- f>a</c-> <c- e>href</c-><c- o>=</c-><c- s>&quot;?art&amp;copy&quot;</c-><c- p>&gt;</c->Art and Copy<c- p>&lt;/</c-><c- f>a</c-><c- p>&gt;</c-></code></pre>

     <p>To avoid this problem, all named character references are required to end with a semicolon,
     and uses of named character references without a semicolon are flagged as errors.</p>

     <p>Thus, the correct way to express the above cases is as
     follows:</p>

     <pre><code class='html'><c- p>&lt;</c-><c- f>a</c-> <c- e>href</c-><c- o>=</c-><c- s>&quot;?bill&amp;ted&quot;</c-><c- p>&gt;</c->Bill and Ted<c- p>&lt;/</c-><c- f>a</c-><c- p>&gt;</c-> <c- c>&lt;!-- &amp;ted is ok, since it&apos;s not a named character reference --&gt;</c-></code></pre>
     <pre><code class='html'><c- p>&lt;</c-><c- f>a</c-> <c- e>href</c-><c- o>=</c-><c- s>&quot;?art&amp;amp;copy&quot;</c-><c- p>&gt;</c->Art and Copy<c- p>&lt;/</c-><c- f>a</c-><c- p>&gt;</c-> <c- c>&lt;!-- the &amp; has to be escaped, since &amp;copy </c-><em><c- c>is</c-></em><c- c> a named character reference --&gt;</c-></code></pre>

    </div>
   <dt>Errors involving known interoperability problems in legacy user agents<dd>
    <p>Certain syntax constructs are known to cause especially subtle or serious problems in legacy
    user agents, and are therefore marked as non-conforming to help authors avoid them.</p>

    <div class=example>

     <p>For example, this is why the U+0060 GRAVE ACCENT character (`) is not allowed in unquoted
     attributes. In certain legacy user agents,  it is sometimes treated as a
     quote character.</p>

    </div>

    <div class=example>

     <p>Another example of this is the DOCTYPE, which is required to trigger <a id=syntax-errors:no-quirks-mode href=https://dom.spec.whatwg.org/#concept-document-no-quirks data-x-internal=no-quirks-mode>no-quirks
     mode</a>, because the behavior of legacy user agents in <a id=syntax-errors:quirks-mode href=https://dom.spec.whatwg.org/#concept-document-quirks data-x-internal=quirks-mode>quirks mode</a> is often
     largely undocumented.</p>

    </div>
   <dt>Errors that risk exposing authors to security attacks<dd>
    <p>Certain restrictions exist purely to avoid known security problems.</p>

    <div class=example>

     <p>For example, the restriction on using UTF-7 exists purely to avoid authors falling prey to a
     known cross-site-scripting attack using UTF-7. <a href=#refsUTF7>[UTF7]</a></p>

    </div>
   <dt>Cases where the author's intent is unclear<dd>
    <p>Markup where the author's intent is very unclear is often made non-conforming. Correcting
    these errors early makes later maintenance easier.</p>

    <div class=example>

     <p>For example, it is unclear whether the author intended the following to be an
     <code id=syntax-errors:the-h1,-h2,-h3,-h4,-h5,-and-h6-elements><a href=#the-h1,-h2,-h3,-h4,-h5,-and-h6-elements>h1</a></code> heading or an <code id=syntax-errors:the-h1,-h2,-h3,-h4,-h5,-and-h6-elements-2><a href=#the-h1,-h2,-h3,-h4,-h5,-and-h6-elements>h2</a></code> heading:</p>

     <pre class=bad><code class='html'><c- p>&lt;</c-><c- f>h1</c-><c- p>&gt;</c->Contact details<c- p>&lt;/</c-><c- f>h2</c-><c- p>&gt;</c-></code></pre>

    </div>
   <dt>Cases that are likely to be typos<dd>
    <p>When a user makes a simple typo, it is helpful if the error can be caught early, as this can
    save the author a lot of debugging time. This specification therefore usually considers it an
    error to use element names, attribute names, and so forth, that do not match the names defined
    in this specification.</p>

    <div class=example>

     <p>For example, if the author typed <code>&lt;capton></code> instead of <code>&lt;caption></code>, this would be flagged as an error and the author could correct
     the typo immediately.</p>

    </div>
   <dt>Errors that could interfere with new syntax in the future<dd>
    <p>In order to allow the language syntax to be extended in the future, certain otherwise
    harmless features are disallowed.</p>

    <div class=example>

     <p>For example, "attributes" in end tags are ignored currently, but they are invalid, in case a
     future change to the language makes use of that syntax feature without conflicting with
     already-deployed (and valid!) content.</p>

    </div>
   </dl>

  <p>Some authors find it helpful to be in the practice of always quoting all attributes and always
  including all optional tags, preferring the consistency derived from such custom over the minor
  benefits of terseness afforded by making use of the flexibility of the HTML syntax. To aid such
  authors, conformance checkers can provide modes of operation wherein such conventions are
  enforced.</p>



  <h4 id=restrictions-on-content-models-and-on-attribute-values><span class=secno>1.11.3</span> Restrictions on content models and on attribute values<a href=#restrictions-on-content-models-and-on-attribute-values class=self-link></a></h4>

  <p><i>This section is non-normative.</i></p>

  <p>Beyond the syntax of the language, this specification also places restrictions on how elements
  and attributes can be specified. These restrictions are present for similar reasons:</p>

  <dl><dt>Errors involving content with dubious semantics<dd>
    <p>To avoid misuse of elements with defined meanings, content models are defined that restrict
    how elements can be nested when such nestings would be of dubious value.</p>

    <p class=example>For example, this specification disallows nesting a <code id=restrictions-on-content-models-and-on-attribute-values:the-section-element><a href=#the-section-element>section</a></code>
    element inside a <code id=restrictions-on-content-models-and-on-attribute-values:the-kbd-element><a href=#the-kbd-element>kbd</a></code> element, since it is highly unlikely for an author to indicate
    that an entire section should be keyed in.</p>
   <dt>Errors that involve a conflict in expressed semantics<dd>
    <p>Similarly, to draw the author's attention to mistakes in the use of elements, clear
    contradictions in the semantics expressed are also considered conformance errors.</p>

    <div class=example>

     <p>In the fragments below, for example, the semantics are nonsensical: a separator cannot
     simultaneously be a cell, nor can a radio button be a progress bar.</p>

     <pre class=bad><code class='html'><c- p>&lt;</c-><c- f>hr</c-> <c- e>role</c-><c- o>=</c-><c- s>&quot;cell&quot;</c-><c- p>&gt;</c-></code></pre>
     <pre class=bad><code class='html'><c- p>&lt;</c-><c- f>input</c-> <c- e>type</c-><c- o>=</c-><c- s>radio</c-> <c- e>role</c-><c- o>=</c-><c- s>progressbar</c-><c- p>&gt;</c-></code></pre>

    </div>

    <p class=example>Another example is the restrictions on the content models of the
    <code id=restrictions-on-content-models-and-on-attribute-values:the-ul-element><a href=#the-ul-element>ul</a></code> element, which only allows <code id=restrictions-on-content-models-and-on-attribute-values:the-li-element><a href=#the-li-element>li</a></code> element children. Lists by definition
    consist just of zero or more list items, so if a <code id=restrictions-on-content-models-and-on-attribute-values:the-ul-element-2><a href=#the-ul-element>ul</a></code> element contains something
    other than an <code id=restrictions-on-content-models-and-on-attribute-values:the-li-element-2><a href=#the-li-element>li</a></code> element, it's not clear what was meant.</p>
   <dt>Cases where the default styles are likely to lead to confusion<dd>
    <p>Certain elements have default styles or behaviors that make certain combinations likely to
    lead to confusion. Where these have equivalent alternatives without this problem, the confusing
    combinations are disallowed.</p>

    <p class=example>For example, <code id=restrictions-on-content-models-and-on-attribute-values:the-div-element><a href=#the-div-element>div</a></code> elements are rendered as <a href=https://drafts.csswg.org/css2/#block-boxes%E2%91%A0 id=restrictions-on-content-models-and-on-attribute-values:block-box data-x-internal=block-box>block boxes</a>, and <code id=restrictions-on-content-models-and-on-attribute-values:the-span-element><a href=#the-span-element>span</a></code> elements as <a href=https://drafts.csswg.org/css2/#inline-box id=restrictions-on-content-models-and-on-attribute-values:inline-box data-x-internal=inline-box>inline boxes</a>. Putting a <a id=restrictions-on-content-models-and-on-attribute-values:block-box-2 href=https://drafts.csswg.org/css2/#block-boxes%E2%91%A0 data-x-internal=block-box>block box</a> in an
    <a id=restrictions-on-content-models-and-on-attribute-values:inline-box-2 href=https://drafts.csswg.org/css2/#inline-box data-x-internal=inline-box>inline box</a> is unnecessarily confusing; since either nesting just <code id=restrictions-on-content-models-and-on-attribute-values:the-div-element-2><a href=#the-div-element>div</a></code>
    elements, or nesting just <code id=restrictions-on-content-models-and-on-attribute-values:the-span-element-2><a href=#the-span-element>span</a></code> elements, or nesting <code id=restrictions-on-content-models-and-on-attribute-values:the-span-element-3><a href=#the-span-element>span</a></code> elements
    inside <code id=restrictions-on-content-models-and-on-attribute-values:the-div-element-3><a href=#the-div-element>div</a></code> elements all serve the same purpose as nesting a <code id=restrictions-on-content-models-and-on-attribute-values:the-div-element-4><a href=#the-div-element>div</a></code>
    element in a <code id=restrictions-on-content-models-and-on-attribute-values:the-span-element-4><a href=#the-span-element>span</a></code> element, but only the latter involves a <a id=restrictions-on-content-models-and-on-attribute-values:block-box-3 href=https://drafts.csswg.org/css2/#block-boxes%E2%91%A0 data-x-internal=block-box>block box</a> in
    an <a id=restrictions-on-content-models-and-on-attribute-values:inline-box-3 href=https://drafts.csswg.org/css2/#inline-box data-x-internal=inline-box>inline box</a>, the latter combination is disallowed.</p>

    <p class=example>Another example would be the way <a href=#interactive-content-2 id=restrictions-on-content-models-and-on-attribute-values:interactive-content-2>interactive content</a> cannot be
    nested. For example, a <code id=restrictions-on-content-models-and-on-attribute-values:the-button-element><a href=#the-button-element>button</a></code> element cannot contain a <code id=restrictions-on-content-models-and-on-attribute-values:the-textarea-element><a href=#the-textarea-element>textarea</a></code>
    element. This is because the default behavior of such nesting interactive elements would be
    highly confusing to users. Instead of nesting these elements, they can be placed side by
    side.</p>
   <dt>Errors that indicate a likely misunderstanding of the specification<dd>
    <p>Sometimes, something is disallowed because allowing it would likely cause author
    confusion.</p>

    <p class=example>For example, setting the <code id=restrictions-on-content-models-and-on-attribute-values:attr-fe-disabled><a href=#attr-fe-disabled>disabled</a></code>
    attribute to the value "<code>false</code>" is disallowed, because despite the
    appearance of meaning that the element is enabled, it in fact means that the element is
    <em>disabled</em> (what matters for implementations is the presence of the attribute, not its
    value).</p>
   <dt>Errors involving limits that have been imposed merely to simplify the language<dd>
    <p>Some conformance errors simplify the language that authors need to learn.</p>

    <p class=example>For example, the <code id=restrictions-on-content-models-and-on-attribute-values:the-area-element><a href=#the-area-element>area</a></code> element's <code id=restrictions-on-content-models-and-on-attribute-values:attr-area-shape><a href=#attr-area-shape>shape</a></code> attribute, despite accepting both <code id=restrictions-on-content-models-and-on-attribute-values:attr-area-shape-keyword-circ><a href=#attr-area-shape-keyword-circ>circ</a></code> and <code id=restrictions-on-content-models-and-on-attribute-values:attr-area-shape-keyword-circle><a href=#attr-area-shape-keyword-circle>circle</a></code> values in practice as synonyms, disallows
    the use of the <code id=restrictions-on-content-models-and-on-attribute-values:attr-area-shape-keyword-circ-2><a href=#attr-area-shape-keyword-circ>circ</a></code> value, so as to simplify
    tutorials and other learning aids. There would be no benefit to allowing both, but it would
    cause extra confusion when teaching the language.</p>
   <dt>Errors that involve peculiarities of the parser<dd>
    <p>Certain elements are parsed in somewhat eccentric ways (typically for historical reasons),
    and their content model restrictions are intended to avoid exposing the author to these
    issues.</p>

    <div class=example>

     <p>For example, a <code id=restrictions-on-content-models-and-on-attribute-values:the-form-element><a href=#the-form-element>form</a></code> element isn't allowed inside <a href=#phrasing-content-2 id=restrictions-on-content-models-and-on-attribute-values:phrasing-content-2>phrasing content</a>,
     because when parsed as HTML, a <code id=restrictions-on-content-models-and-on-attribute-values:the-form-element-2><a href=#the-form-element>form</a></code> element's start tag will imply a
     <code id=restrictions-on-content-models-and-on-attribute-values:the-p-element><a href=#the-p-element>p</a></code> element's end tag. Thus, the following markup results in two <a href=#paragraph id=restrictions-on-content-models-and-on-attribute-values:paragraph>paragraphs</a>, not one:</p>

     <pre><code class='html'><c- p>&lt;</c-><c- f>p</c-><c- p>&gt;</c->Welcome. <c- p>&lt;</c-><c- f>form</c-><c- p>&gt;&lt;</c-><c- f>label</c-><c- p>&gt;</c->Name:<c- p>&lt;/</c-><c- f>label</c-><c- p>&gt;</c-> <c- p>&lt;</c-><c- f>input</c-><c- p>&gt;&lt;/</c-><c- f>form</c-><c- p>&gt;</c-></code></pre>

     <p>It is parsed exactly like the following:</p>

     <pre><code class='html'><c- p>&lt;</c-><c- f>p</c-><c- p>&gt;</c->Welcome. <c- p>&lt;/</c-><c- f>p</c-><c- p>&gt;&lt;</c-><c- f>form</c-><c- p>&gt;&lt;</c-><c- f>label</c-><c- p>&gt;</c->Name:<c- p>&lt;/</c-><c- f>label</c-><c- p>&gt;</c-> <c- p>&lt;</c-><c- f>input</c-><c- p>&gt;&lt;/</c-><c- f>form</c-><c- p>&gt;</c-></code></pre>

    </div>
   <dt>Errors that would likely result in scripts failing in hard-to-debug ways<dd>
    <p>Some errors are intended to help prevent script problems that would be hard to debug.</p>

    <p class=example>This is why, for instance, it is non-conforming to have two <code id=restrictions-on-content-models-and-on-attribute-values:the-id-attribute><a href=#the-id-attribute>id</a></code> attributes with the same value. Duplicate IDs lead to the wrong
    element being selected, with sometimes disastrous effects whose cause is hard to determine.</p>
   <dt>Errors that waste authoring time<dd>
    <p>Some constructs are disallowed because historically they have been the cause of a lot of
    wasted authoring time, and by encouraging authors to avoid making them, authors can save time in
    future efforts.</p>

    <p class=example>For example, a <code id=restrictions-on-content-models-and-on-attribute-values:the-script-element><a href=#the-script-element>script</a></code> element's <code id=restrictions-on-content-models-and-on-attribute-values:attr-script-src><a href=#attr-script-src>src</a></code> attribute causes the element's contents to be ignored.
    However, this isn't obvious, especially if the element's contents appear to be executable script
    — which can lead to authors spending a lot of time trying to debug the inline script
    without realizing that it is not executing. To reduce this problem, this specification makes it
    non-conforming to have executable script in a <code id=restrictions-on-content-models-and-on-attribute-values:the-script-element-2><a href=#the-script-element>script</a></code> element when the <code id=restrictions-on-content-models-and-on-attribute-values:attr-script-src-2><a href=#attr-script-src>src</a></code> attribute is present. This means that authors who are
    validating their documents are less likely to waste time with this kind of mistake.</p>
   <dt>Errors that involve areas that affect authors migrating between the HTML and XML
   syntaxes<dd>
    <p>Some authors like to write files that can be interpreted as both XML and HTML with similar
    results. Though this practice is discouraged in general due to the myriad of subtle
    complications involved (especially when involving scripting, styling, or any kind of automated
    serialization), this specification has a few restrictions intended to at least somewhat mitigate
    the difficulties. This makes it easier for authors to use this as a transitionary step when
    migrating between the HTML and XML syntaxes.</p>

    <p class=example>For example, there are somewhat complicated rules surrounding the <code id=restrictions-on-content-models-and-on-attribute-values:attr-lang><a href=#attr-lang>lang</a></code> and <code id=restrictions-on-content-models-and-on-attribute-values:attr-xml-lang><a data-x-internal=attr-xml-lang href=https://www.w3.org/TR/xml/#sec-lang-tag>xml:lang</a></code> attributes
    intended to keep the two synchronized.</p>

    <p class=example>Another example would be the restrictions on the values of <code>xmlns</code> attributes in the HTML serialization, which are intended to ensure that
    elements in conforming documents end up in the same namespaces whether processed as HTML or
    XML.</p>
   <dt>Errors that involve areas reserved for future expansion<dd>
    <p>As with the restrictions on the syntax intended to allow for new syntax in future revisions
    of the language, some restrictions on the content models of elements and values of attributes
    are intended to allow for future expansion of the HTML vocabulary.</p>

    <p class=example>For example, limiting the values of the <code id=restrictions-on-content-models-and-on-attribute-values:attr-hyperlink-target><a href=#attr-hyperlink-target>target</a></code> attribute that start with an U+005F LOW LINE
    character (_) to only specific predefined values allows new predefined values to be introduced
    at a future time without conflicting with author-defined values.</p>
   <dt>Errors that indicate a mis-use of other specifications<dd>
    <p>Certain restrictions are intended to support the restrictions made by other
    specifications.</p>

    <p class=example>For example, requiring that attributes that take media query lists use only
    <em>valid</em> media query lists reinforces the importance of following the conformance rules of
    that specification.</p>
   </dl>



  <h3 id=suggested-reading><span class=secno>1.12</span> Suggested reading<a href=#suggested-reading class=self-link></a></h3>

  <p><i>This section is non-normative.</i></p>

  <p>The following documents might be of interest to readers of this specification.</p>

  <dl><dt><cite>Character Model for the World Wide Web 1.0: Fundamentals</cite> <a href=#refsCHARMOD>[CHARMOD]</a><dd><blockquote><p>This Architectural Specification provides authors of specifications, software
   developers, and content developers with a common reference for interoperable text manipulation on
   the World Wide Web, building on the Universal Character Set, defined jointly by the Unicode
   Standard and ISO/IEC 10646. Topics addressed include use of the terms 'character', 'encoding' and
   'string', a reference processing model, choice and identification of character encodings,
   character escaping, and string indexing.</blockquote><dt><cite>Unicode Security Considerations</cite> <a href=#refsUTR36>[UTR36]</a><dd><blockquote><p>Because Unicode contains such a large number of characters and incorporates
   the varied writing systems of the world, incorrect usage can expose programs or systems to
   possible security attacks. This is especially important as more and more products are
   internationalized. This document describes some of the security considerations that programmers,
   system analysts, standards developers, and users should take into account, and provides specific
   recommendations to reduce the risk of problems.</blockquote><dt><cite>Web Content Accessibility Guidelines (WCAG)</cite> <a href=#refsWCAG>[WCAG]</a><dd><blockquote><p>Web Content Accessibility Guidelines (WCAG) covers a wide range of
   recommendations for making web content more accessible. Following these guidelines will make
   content accessible to a wider range of people with disabilities, including blindness and low
   vision, deafness and hearing loss, learning disabilities, cognitive limitations, limited
   movement, speech disabilities, photosensitivity and combinations of these. Following these
   guidelines will also often make your web content more usable to users in
   general.</blockquote><dt><cite>Authoring Tool Accessibility Guidelines (ATAG) 2.0</cite> <a href=#refsATAG>[ATAG]</a><dd><blockquote><p>This specification provides guidelines for designing web content
   authoring tools that are more accessible for people with disabilities. An authoring tool that
   conforms to these guidelines will promote accessibility by providing an accessible user interface
   to authors with disabilities as well as by enabling, supporting, and promoting the production of
   accessible web content by all authors.</blockquote><dt><cite>User Agent Accessibility Guidelines (UAAG) 2.0</cite> <a href=#refsUAAG>[UAAG]</a><dd><blockquote><p>This document provides guidelines for designing user agents that
   lower barriers to web accessibility for people with disabilities. User agents include browsers
   and other types of software that retrieve and render web content. A user agent that conforms to
   these guidelines will promote accessibility through its own user interface and through other
   internal facilities, including its ability to communicate with other technologies (especially
   assistive technologies). Furthermore, all users, not just users with disabilities, should find
   conforming user agents to be more usable.</blockquote></dl>



  <h2 id=infrastructure><span class=secno>2</span> Common infrastructure<a href=#infrastructure class=self-link></a></h2>

  <p>This specification depends on <cite>Infra</cite>. <a href=#refsINFRA>[INFRA]</a></p>

  <h3 id=terminology><span class=secno>2.1</span> Terminology<a href=#terminology class=self-link></a></h3>

  <p>This specification refers to both HTML and XML attributes and IDL attributes, often in the same
  context. When it is not clear which is being referred to, they are referred to as <dfn>content attributes</dfn> for HTML and XML attributes, and <dfn>IDL
  attributes</dfn> for those defined on IDL interfaces. Similarly, the term "properties" is used for
  both JavaScript object properties and CSS properties. When these are ambiguous they are qualified
  as <dfn>object properties</dfn> and <dfn>CSS properties</dfn> respectively.</p>

  <p>Generally, when the specification states that a feature applies to <a href=#syntax id=terminology:syntax>the HTML syntax</a>
  or <a href=#the-xhtml-syntax id=terminology:the-xhtml-syntax>the XML syntax</a>, it also includes the other. When a feature specifically only
  applies to one of the two languages, it is called out by explicitly stating that it does not apply
  to the other format, as in "for HTML, ... (this does not apply to XML)".</p>

  <p>This specification uses the term <dfn>document</dfn> to refer to any use of HTML,
  ranging from short static documents to long essays or reports with rich multimedia, as well as to
  fully-fledged interactive applications. The term is used to refer both to <code id=terminology:document><a href=#document>Document</a></code>
  objects and their descendant DOM trees, and to serialized byte streams using the <a href=#syntax id=terminology:syntax-2>HTML syntax</a> or the <a href=#the-xhtml-syntax id=terminology:the-xhtml-syntax-2>XML syntax</a>, depending
  on context.</p>

  <p>In the context of the DOM structures, the terms <a href=https://dom.spec.whatwg.org/#html-document id=terminology:html-documents data-x-internal=html-documents>HTML
  document</a> and <a href=https://dom.spec.whatwg.org/#xml-document id=terminology:xml-documents data-x-internal=xml-documents>XML document</a> are used as defined in
  <cite>DOM</cite>, and refer specifically to two different modes that <code id=terminology:document-2><a href=#document>Document</a></code> objects
  can find themselves in. <a href=#refsDOM>[DOM]</a> (Such uses are always hyperlinked to their
  definition.)</p>

  <p>In the context of byte streams, the term HTML document refers to resources labeled as
  <code id=terminology:text/html><a href=#text/html>text/html</a></code>, and the term XML document refers to resources labeled with an <a id=terminology:xml-mime-type href=https://mimesniff.spec.whatwg.org/#xml-mime-type data-x-internal=xml-mime-type>XML
  MIME type</a>.</p>

  <hr>

  <p>For simplicity, terms such as <dfn>shown</dfn>, <dfn>displayed</dfn>, and
  <dfn>visible</dfn> might sometimes be used when referring to the way a document is
  rendered to the user. These terms are not meant to imply a visual medium; they must be considered
  to apply to other media in equivalent ways.</p>


  

  <h4 id=parallelism><span class=secno>2.1.1</span> Parallelism<a href=#parallelism class=self-link></a></h4>

  <p>To run steps <dfn id=in-parallel data-export="">in parallel</dfn> means those steps are to be run, one after another,
  at the same time as other logic in the standard (e.g., at the same time as the <a href=#event-loop id=parallelism:event-loop>event
  loop</a>). This standard does not define the precise mechanism by which this is achieved, be it
  time-sharing cooperative multitasking, fibers, threads, processes, using different hyperthreads,
  cores, CPUs, machines, etc. By contrast, an operation that is to run <dfn id=immediately>immediately</dfn> must
  interrupt the currently running task, run itself, and then resume the previously running task.</p>

  <p class=note>For guidance on writing specifications that leverage parallelism, see <a href=#event-loop-for-spec-authors>Dealing with the event loop from other specifications</a>.</p>

  <p>To avoid race conditions between different <a href=#in-parallel id=parallelism:in-parallel>in parallel</a> algorithms that operate on
  the same data, a <a href=#parallel-queue id=parallelism:parallel-queue>parallel queue</a> can be used.</p>

  <p>A <dfn id=parallel-queue data-export="">parallel queue</dfn> represents a queue of algorithm steps that must be run in
  series.</p>

  <p>A <a href=#parallel-queue id=parallelism:parallel-queue-2>parallel queue</a> has an <dfn id=algorithm-queue>algorithm queue</dfn> (a <a id=parallelism:queue href=https://infra.spec.whatwg.org/#queue data-x-internal=queue>queue</a>),
  initially empty.</p>

  <p>To <dfn data-dfn-for="parallel queue" id=enqueue-the-following-steps data-lt="enqueue steps|enqueue the following
  steps" data-export="">enqueue steps</dfn> to a <a href=#parallel-queue id=parallelism:parallel-queue-3>parallel queue</a>,
  <a id=parallelism:enqueue href=https://infra.spec.whatwg.org/#queue-enqueue data-x-internal=enqueue>enqueue</a> the algorithm steps to the <a href=#parallel-queue id=parallelism:parallel-queue-4>parallel queue</a>'s <a href=#algorithm-queue id=parallelism:algorithm-queue>algorithm
  queue</a>.</p>

  <p>To <dfn id=starting-a-new-parallel-queue data-lt="start a new parallel queue|starting a
  new parallel queue" data-export="">start a new parallel queue</dfn>, run the following steps:</p>

  <ol><li><p>Let <var>parallelQueue</var> be a new <a href=#parallel-queue id=parallelism:parallel-queue-5>parallel queue</a>.<li>
    <p>Run the following steps <a href=#in-parallel id=parallelism:in-parallel-2>in parallel</a>:</p>

    <ol><li>
      <p>While true:</p>

      <ol><li><p>Let <var>steps</var> be the result of <a href=https://infra.spec.whatwg.org/#queue-dequeue id=parallelism:dequeue data-x-internal=dequeue>dequeueing</a> from
       <var>parallelQueue</var>'s <a href=#algorithm-queue id=parallelism:algorithm-queue-2>algorithm queue</a>.<li><p>If <var>steps</var> is not nothing, then run <var>steps</var>.<li><p>Assert: running <var>steps</var> did not throw an exception, as steps running <a href=#in-parallel id=parallelism:in-parallel-3>in
       parallel</a> are not allowed to throw.</ol>

      <p class=note>Implementations are not expected to implement this as a continuously running
      loop. Algorithms in standards are to be easy to understand and are not necessarily great for
      battery life or performance.</p>
     </ol>
   <li><p>Return <var>parallelQueue</var>.</ol>

  <p class=note>Steps running <a href=#in-parallel id=parallelism:in-parallel-4>in parallel</a> can themselves run other steps in <a href=#in-parallel id=parallelism:in-parallel-5>in
  parallel</a>. E.g., inside a <a href=#parallel-queue id=parallelism:parallel-queue-6>parallel queue</a> it can be useful to run a series of
  steps in parallel with the queue.</p>

  <div class=example>
   <p>Imagine a standard defined <var>nameList</var> (a <a id=parallelism:list href=https://infra.spec.whatwg.org/#list data-x-internal=list>list</a>), along with a method to
   add a <var>name</var> to <var>nameList</var>, unless <var>nameList</var> already <a href=https://infra.spec.whatwg.org/#list-contain id=parallelism:list-contains data-x-internal=list-contains>contains</a> <var>name</var>, in which case it rejects.</p>

   <p>The following solution suffers from race conditions:</p>

   <ol><li><p>Let <var>p</var> be a new promise.<li>
     <p>Run the following steps <a href=#in-parallel id=parallelism:in-parallel-6>in parallel</a>:</p>

     <ol><li><p>If <var>nameList</var> <a href=https://infra.spec.whatwg.org/#list-contain id=parallelism:list-contains-2 data-x-internal=list-contains>contains</a> <var>name</var>,
      reject <var>p</var> with a <code id=parallelism:typeerror><a data-x-internal=typeerror href=https://tc39.es/ecma262/#sec-native-error-types-used-in-this-standard-typeerror>TypeError</a></code> and abort these steps.<li><p>Do some potentially lengthy work.<li><p><a href=https://infra.spec.whatwg.org/#list-append id=parallelism:list-append data-x-internal=list-append>Append</a> <var>name</var> to
      <var>nameList</var>.<li><p>Resolve <var>p</var> with undefined.</ol>
    <li><p>Return <var>p</var>.</ol>

   <p>Two invocations of the above could run simultaneously, meaning <var>name</var> isn't in
   <var>nameList</var> during step 2.1, but it <em>might be added</em> before step 2.3 runs,
   meaning <var>name</var> ends up in <var>nameList</var> twice.</p>

   <p>Parallel queues solve this. The standard would let <var>nameListQueue</var> be the result of
   <a href=#starting-a-new-parallel-queue id=parallelism:starting-a-new-parallel-queue>starting a new parallel queue</a>, then:</p>

   <ol><li><p>Let <var>p</var> be a new promise.<li>
     <p><mark><a href=#enqueue-the-following-steps id=parallelism:enqueue-the-following-steps>Enqueue the following steps</a> to <var>nameListQueue</var>:</mark></p>

     <ol><li><p>If <var>nameList</var> <a href=https://infra.spec.whatwg.org/#list-contain id=parallelism:list-contains-3 data-x-internal=list-contains>contains</a> <var>name</var>,
      reject <var>p</var> with a <code id=parallelism:typeerror-2><a data-x-internal=typeerror href=https://tc39.es/ecma262/#sec-native-error-types-used-in-this-standard-typeerror>TypeError</a></code> and abort these steps.<li><p>Do some potentially lengthy work.<li><p><a href=https://infra.spec.whatwg.org/#list-append id=parallelism:list-append-2 data-x-internal=list-append>Append</a> <var>name</var> to
      <var>nameList</var>.<li><p>Resolve <var>p</var> with undefined.</ol>
    <li><p>Return <var>p</var>.</ol>

   <p>The steps would now queue and the race is avoided.</p>
  </div>

  


  <h4 id=resources><span class=secno>2.1.2</span> Resources<a href=#resources class=self-link></a></h4>

  <p>The specification uses the term <dfn>supported</dfn> when referring to whether a user
  agent has an implementation capable of decoding the semantics of an external resource. A format or
  type is said to be <i>supported</i> if the implementation can process an external resource of that
  format or type without critical aspects of the resource being ignored. Whether a specific resource
  is <i>supported</i> can depend on what features of the resource's format are in use.</p>

  <p class=example>For example, a PNG image would be considered to be in a supported format if its
  pixel data could be decoded and rendered, even if, unbeknownst to the implementation, the image
  also contained animation data.</p>

  <p class=example>An MPEG-4 video file would not be considered to be in a supported format if the
  compression format used was not supported, even if the implementation could determine the
  dimensions of the movie from the file's metadata.</p>

  <p>What some specifications, in particular the HTTP specifications, refer to as a
  <i>representation</i> is referred to in this specification as a <dfn>resource</dfn>.
  <a href=#refsHTTP>[HTTP]</a></p>

  <p>A resource's <dfn id=critical-subresources>critical subresources</dfn> are those that the resource needs to have
  available to be correctly processed. Which resources are considered critical or not is defined by
  the specification that defines the resource's format.</p>

  <p>For <a href=https://drafts.csswg.org/cssom/#css-style-sheet id=resources:css-style-sheet data-x-internal=css-style-sheet>CSS style sheets</a>, we tentatively define here that
  their critical subresources are other style sheets imported via <code>@import</code>
  rules, including those indirectly imported by other imported style sheets.</p>

  <p class=XXX>This definition is not fully interoperable; furthermore, some user agents seem to
  count resources like background images or web fonts as critical subresources. Ideally, the CSS
  Working Group would define this; see <a href=https://github.com/w3c/csswg-drafts/issues/1088>w3c/csswg-drafts issue #1088</a> to track
  progress on that front.</p>


  <h4 id=xml><span class=secno>2.1.3</span> XML compatibility<a href=#xml class=self-link></a></h4>

  <p id=html-namespace>To ease migration from HTML to XML, UAs conforming to this specification
  will place elements in HTML in the <code id=xml:html-namespace-2><a data-x-internal=html-namespace-2 href=https://infra.spec.whatwg.org/#html-namespace>http://www.w3.org/1999/xhtml</a></code> namespace, at least for the purposes of the DOM and
  CSS. The term "<dfn id=html-elements data-export="">HTML elements</dfn>" refers to any element in that namespace, even in
  XML documents.</p>

  <p>Except where otherwise stated, all elements defined or mentioned in this specification are in
  the <a id=xml:html-namespace-2-2 href=https://infra.spec.whatwg.org/#html-namespace data-x-internal=html-namespace-2>HTML namespace</a> ("<code>http://www.w3.org/1999/xhtml</code>"), and all
  attributes defined or mentioned in this specification have no namespace.</p>

  <p>The term <dfn id=element-type>element type</dfn> is used to refer to the set of elements that have a given
  local name and namespace. For example, <code id=xml:the-button-element><a href=#the-button-element>button</a></code> elements are elements with the element
  type <code id=xml:the-button-element-2><a href=#the-button-element>button</a></code>, meaning they have the local name "<code>button</code>" and
  (implicitly as defined above) the <a id=xml:html-namespace-2-3 href=https://infra.spec.whatwg.org/#html-namespace data-x-internal=html-namespace-2>HTML namespace</a>.</p>

  <p>Attribute names are said to be <dfn id=xml-compatible>XML-compatible</dfn> if they match the <code id=xml:xml-name><a data-x-internal=xml-name href=https://www.w3.org/TR/xml/#NT-Name>Name</a></code> production defined in XML and they contain no U+003A COLON
  characters (:). <a href=#refsXML>[XML]</a></p>


  <h4 id=dom-trees><span class=secno>2.1.4</span> DOM trees<a href=#dom-trees class=self-link></a></h4>

  <p>When it is stated that some element or attribute is <dfn id=ignore>ignored</dfn>, or
  treated as some other value, or handled as if it was something else, this refers only to the
  processing of the node after it is in the DOM. A user agent must not mutate the
  DOM in such situations.</p>

  <p>A content attribute is said to <dfn>change</dfn> value only if its new value is
  different than its previous value; setting an attribute to a value it already has does not change
  it.</p>

  <p>The term <dfn>empty</dfn>, when used for an attribute value, <code id=dom-trees:text><a data-x-internal=text href=https://dom.spec.whatwg.org/#interface-text>Text</a></code> node,
  or string, means that the <a id=dom-trees:length href=https://infra.spec.whatwg.org/#string-length data-x-internal=length>length</a> of the text is zero (i.e., not even containing <a href=https://infra.spec.whatwg.org/#control id=dom-trees:control data-x-internal=control>controls</a> or U+0020 SPACE).</p>

  <p>A <dfn id=nodes-are-inserted>node <var>A</var> is inserted</dfn> into a node <var>B</var>
  when the <a href=https://dom.spec.whatwg.org/#concept-node-insert-ext id=dom-trees:concept-node-insert-ext data-x-internal=concept-node-insert-ext>insertion steps</a> are invoked with
  <var>A</var> as the argument and <var>A</var>'s new parent is <var>B</var>. Similarly, a <dfn id=nodes-are-removed>node <var>A</var> is removed</dfn> from a node <var>B</var> when the
  <a href=https://dom.spec.whatwg.org/#concept-node-remove-ext id=dom-trees:concept-node-remove-ext data-x-internal=concept-node-remove-ext>removing steps</a> are invoked with <var>A</var> as the
  <var>removedNode</var> argument and <var>B</var> as the <var>oldParent</var> argument.</p>

  <p>A <dfn id=insert-an-element-into-a-document data-lt="inserted into a document|node is inserted into a document" data-export="">node is inserted into a
  document</dfn> when the <a href=https://dom.spec.whatwg.org/#concept-node-insert-ext id=dom-trees:concept-node-insert-ext-2 data-x-internal=concept-node-insert-ext>insertion steps</a> are invoked
  with it as the argument and it is now <a id=dom-trees:in-a-document-tree href=https://dom.spec.whatwg.org/#in-a-document-tree data-x-internal=in-a-document-tree>in a document tree</a>.  Analogously, a <dfn id=remove-an-element-from-a-document data-lt="removed
  from a document|node is removed from a document" data-export="">node is removed from a document</dfn> when
  the <a href=https://dom.spec.whatwg.org/#concept-node-remove-ext id=dom-trees:concept-node-remove-ext-2 data-x-internal=concept-node-remove-ext>removing steps</a> are invoked with it as the
  argument and it is now no longer <a id=dom-trees:in-a-document-tree-2 href=https://dom.spec.whatwg.org/#in-a-document-tree data-x-internal=in-a-document-tree>in a document tree</a>.</p>

  <p>A node <dfn id=becomes-connected data-export="">becomes connected</dfn> when the <a href=https://dom.spec.whatwg.org/#concept-node-insert-ext id=dom-trees:concept-node-insert-ext-3 data-x-internal=concept-node-insert-ext>insertion steps</a> are invoked with it as the argument and it
  is now <a id=dom-trees:connected href=https://dom.spec.whatwg.org/#connected data-x-internal=connected>connected</a>. Analogously, a node <dfn id=becomes-disconnected data-lt="become disconnected" data-export="">becomes disconnected</dfn> when the <a href=https://dom.spec.whatwg.org/#concept-node-remove-ext id=dom-trees:concept-node-remove-ext-3 data-x-internal=concept-node-remove-ext>removing
  steps</a> are invoked with it as the argument and it is now no longer
  <a id=dom-trees:connected-2 href=https://dom.spec.whatwg.org/#connected data-x-internal=connected>connected</a>.</p>

  <p>A node is <dfn id=browsing-context-connected data-export="">browsing-context connected</dfn> when it is <a id=dom-trees:connected-3 href=https://dom.spec.whatwg.org/#connected data-x-internal=connected>connected</a> and
  its <a href=https://dom.spec.whatwg.org/#concept-shadow-including-root id=dom-trees:shadow-including-root data-x-internal=shadow-including-root>shadow-including root</a>'s <a href=#concept-document-bc id=dom-trees:concept-document-bc>browsing context</a> is non-null. A node <dfn id=becomes-browsing-context-connected data-lt="become
  browsing-context connected" data-export="">becomes browsing-context connected</dfn> when the <a href=https://dom.spec.whatwg.org/#concept-node-insert-ext id=dom-trees:concept-node-insert-ext-4 data-x-internal=concept-node-insert-ext>insertion steps</a> are invoked with it as the argument and it
  is now <a href=#browsing-context-connected id=dom-trees:browsing-context-connected>browsing-context connected</a>. A node <dfn id=becomes-browsing-context-disconnected data-lt="become browsing-context
  disconnected" data-export="">becomes browsing-context disconnected</dfn> either when the <a href=https://dom.spec.whatwg.org/#concept-node-remove-ext id=dom-trees:concept-node-remove-ext-4 data-x-internal=concept-node-remove-ext>removing steps</a> are invoked with it as the argument and it
  is now no longer <a href=#browsing-context-connected id=dom-trees:browsing-context-connected-2>browsing-context connected</a>, or when its <a id=dom-trees:shadow-including-root-2 href=https://dom.spec.whatwg.org/#concept-shadow-including-root data-x-internal=shadow-including-root>shadow-including
  root</a>'s <a href=#concept-document-bc id=dom-trees:concept-document-bc-2>browsing context</a> becomes null.


  <h4 id=scripting-2><span class=secno>2.1.5</span> Scripting<a href=#scripting-2 class=self-link></a></h4>

  <p>The construction "a <code>Foo</code> object", where <code>Foo</code> is
  actually an interface, is sometimes used instead of the more accurate "an object implementing the
  interface <code>Foo</code>".</p>

  <p>An IDL attribute is said to be <dfn>getting</dfn> when its value is being retrieved
  (e.g. by author script), and is said to be <dfn>setting</dfn> when a new value is
  assigned to it.</p>

  <p>If a DOM object is said to be <dfn id=live>live</dfn>, then the attributes and methods on that object
  must operate on the actual underlying data, not a snapshot of the
  data.</p>


  <h4 id=plugins><span class=secno>2.1.6</span> Plugins<a href=#plugins class=self-link></a></h4>

  <p>The term <dfn id=plugin>plugin</dfn> refers to an <a id=plugins:implementation-defined href=https://infra.spec.whatwg.org/#implementation-defined data-x-internal=implementation-defined>implementation-defined</a> set of content
  handlers used by the user agent that can take part in the user agent's rendering of a
  <code id=plugins:document><a href=#document>Document</a></code> object, but that neither act as <a href=#child-browsing-context id=plugins:child-browsing-context>child
  browsing contexts</a> of the <code id=plugins:document-2><a href=#document>Document</a></code> nor introduce any <code id=plugins:node><a data-x-internal=node href=https://dom.spec.whatwg.org/#interface-node>Node</a></code> objects
  to the <code id=plugins:document-3><a href=#document>Document</a></code>'s DOM.</p>

  <p>Typically such content handlers are provided by third parties, though a user agent can also
  designate built-in content handlers as plugins.</p>

  

  <p>A user agent must not consider the types <code id=plugins:text/plain><a data-x-internal=text/plain href=https://tools.ietf.org/html/rfc2046#section-4.1.3>text/plain</a></code> and
  <code id=plugins:application/octet-stream><a data-x-internal=application/octet-stream href=https://tools.ietf.org/html/rfc2046#section-4.5.1>application/octet-stream</a></code> as having a registered <a href=#plugin id=plugins:plugin>plugin</a>.</p> 

  

  <p class=example>One example of a plugin would be a PDF viewer that is instantiated in a
  <a href=#browsing-context id=plugins:browsing-context>browsing context</a> when the user navigates to a PDF file. This would count as a plugin
  regardless of whether the party that implemented the PDF viewer component was the same as that
  which implemented the user agent itself. However, a PDF viewer application that launches separate
  from the user agent (as opposed to using the same interface) is not a plugin by this
  definition.</p>

  <p class=note>This specification does not define a mechanism for interacting with plugins, as it
  is expected to be user-agent- and platform-specific. Some UAs might opt to support a plugin
  mechanism such as the Netscape Plugin API; others might use remote content converters or have
  built-in support for certain types. Indeed, this specification doesn't require user agents to
  support plugins at all. <a href=#refsNPAPI>[NPAPI]</a></p>

  <p>A plugin can be <dfn id=concept-plugin-secure>secured</dfn> if it honors the semantics of
  the <code id=plugins:attr-iframe-sandbox><a href=#attr-iframe-sandbox>sandbox</a></code> attribute.</p>

  <p class=example>For example, a secured plugin would prevent its contents from creating popups
  when the plugin is instantiated inside a sandboxed <code id=plugins:the-iframe-element><a href=#the-iframe-element>iframe</a></code>.</p>

  

  <p class=warning>Browsers should take extreme care when interacting with external content
  intended for <a href=#plugin id=plugins:plugin-2>plugins</a>. When third-party software is run with the same
  privileges as the user agent itself, vulnerabilities in the third-party software become as
  dangerous as those in the user agent.</p>

  <p>
  <a id=plugins:tracking-vector title="There is a tracking vector here." href=https://infra.spec.whatwg.org/#tracking-vector class=tracking-vector data-x-internal=tracking-vector><img width=46 alt="(This is a tracking vector.)" crossorigin="" src=https://resources.whatwg.org/tracking-vector.svg height=64></a>
  Since different users having different sets of <a href=#plugin id=plugins:plugin-3>plugins</a> provides a
  tracking vector that increases the chances of users being uniquely identified, user agents are
  encouraged to support the exact same set of <a href=#plugin id=plugins:plugin-4>plugins</a> for each
  user.</p>

  



  <h4 id=encoding-terminology><span class=secno>2.1.7</span> Character encodings<a href=#encoding-terminology class=self-link></a></h4>

  <p>A <dfn id=encoding><a href=https://encoding.spec.whatwg.org/#encoding>character
  encoding</a></dfn>, or just <i>encoding</i> where that is not ambiguous, is a defined way to convert
  between byte streams and Unicode strings, as defined in <cite>Encoding</cite>. An
  <a id=encoding-terminology:encoding href=https://encoding.spec.whatwg.org/#encoding data-x-internal=encoding>encoding</a> has an <dfn id=encoding-name><a href=https://encoding.spec.whatwg.org/#name>encoding
  name</a></dfn> and one or more <dfn id=encoding-label><a href=https://encoding.spec.whatwg.org/#label>encoding labels</a></dfn>, referred to as the
  encoding's <i>name</i> and <i>labels</i> in the Encoding standard. <a href=#refsENCODING>[ENCODING]</a></p>



  

  <h4 id=conformance-classes><span class=secno>2.1.8</span> Conformance classes<a href=#conformance-classes class=self-link></a></h4>

  <p>This specification describes the conformance criteria for user agents
  (relevant to implementers) and documents (relevant to authors and
  authoring tool implementers).</p>

  <p><dfn id=conforming-documents>Conforming documents</dfn> are those that comply with all the conformance criteria for
  documents. For readability, some of these conformance requirements are phrased as conformance
  requirements on authors; such requirements are implicitly requirements on documents: by
  definition, all documents are assumed to have had an author. (In some cases, that author may
  itself be a user agent — such user agents are subject to additional rules, as explained
  below.)</p>

  <p class=example>For example, if a requirement states that "authors must not
  use the <code>foobar</code> element", it would imply that documents are not allowed to
  contain elements named <code>foobar</code>.</p>

  <p class="note impl">There is no implied relationship between document conformance requirements
  and implementation conformance requirements. User agents are not free to handle non-conformant
  documents as they please; the processing model described in this specification applies to
  implementations regardless of the conformity of the input documents.</p>

  <p>User agents fall into several (overlapping) categories with different conformance
  requirements.</p>

  <dl><dt id=interactive>Web browsers and other interactive user agents<dd>
    <p>Web browsers that support <a href=#the-xhtml-syntax id=conformance-classes:the-xhtml-syntax>the XML syntax</a> must process elements and attributes
    from the <a id=conformance-classes:html-namespace-2 href=https://infra.spec.whatwg.org/#html-namespace data-x-internal=html-namespace-2>HTML namespace</a> found in XML documents as described in this specification,
    so that users can interact with them, unless the semantics of those elements have been
    overridden by other specifications.</p>

    <p class=example>A conforming web browser would, upon finding a <code id=conformance-classes:the-script-element><a href=#the-script-element>script</a></code> element in
    an XML document, execute the script contained in that element. However, if the element is found
    within a transformation expressed in XSLT (assuming the user agent also supports XSLT), then the
    processor would instead treat the <code id=conformance-classes:the-script-element-2><a href=#the-script-element>script</a></code> element as an opaque element that forms
    part of the transform.</p>

    <p>Web browsers that support <a href=#syntax id=conformance-classes:syntax>the HTML syntax</a> must process documents labeled with an
    <a id=conformance-classes:html-mime-type href=https://mimesniff.spec.whatwg.org/#html-mime-type data-x-internal=html-mime-type>HTML MIME type</a> as described in this specification, so that users can interact with
    them.</p>

    <p>User agents that support scripting must also be conforming implementations of the IDL
    fragments in this specification, as described in <cite>Web IDL</cite>. <a href=#refsWEBIDL>[WEBIDL]</a></p>

    <p class=note>Unless explicitly stated, specifications that override the semantics of HTML
    elements do not override the requirements on DOM objects representing those elements. For
    example, the <code id=conformance-classes:the-script-element-3><a href=#the-script-element>script</a></code> element in the example above would still implement the
    <code id=conformance-classes:htmlscriptelement><a href=#htmlscriptelement>HTMLScriptElement</a></code> interface.</p>
   <dt id=non-interactive>Non-interactive presentation user agents<dd>
    <p>User agents that process HTML and XML documents purely to render non-interactive versions of
    them must comply to the same conformance criteria as web browsers, except that they are exempt
    from requirements regarding user interaction.</p>

    <p class=note>Typical examples of non-interactive presentation user agents are printers
    (static UAs) and overhead displays (dynamic UAs). It is expected that most static
    non-interactive presentation user agents will also opt to <a href=#non-scripted>lack scripting
    support</a>.</p>

    <p class=example>A non-interactive but dynamic presentation UA would still execute scripts,
    allowing forms to be dynamically submitted, and so forth. However, since the concept of "focus"
    is irrelevant when the user cannot interact with the document, the UA would not need to support
    any of the focus-related DOM APIs.</p>
   <dt id=renderingUA>Visual user agents that support the suggested default rendering<dd>
    <p>User agents, whether interactive or not, may be designated (possibly as a user option) as
    supporting the suggested default rendering defined by this specification.</p>

    <p>This is not required. In particular, even user agents that do implement the suggested default
    rendering are encouraged to offer settings that override this default to improve the experience
    for the user, e.g. changing the color contrast, using different focus styles, or otherwise
    making the experience more accessible and usable to the user.</p>

    <p>User agents that are designated as supporting the suggested default rendering must, while so
    designated, implement the rules <a href=#rendering>the rendering section</a> defines as the
    behavior that user agents are <em>expected</em> to implement.</p>
   <dt id=non-scripted>User agents with no scripting support<dd>
    <p>Implementations that do not support scripting (or which have their scripting features
    disabled entirely) are exempt from supporting the events and DOM interfaces mentioned in this
    specification. For the parts of this specification that are defined in terms of an events model
    or in terms of the DOM, such user agents must still act as if events and the DOM were
    supported.</p>

    <p class=note>Scripting can form an integral part of an application. Web browsers that do not
    support scripting, or that have scripting disabled, might be unable to fully convey the author's
    intent.</p>
   <dt>Conformance checkers<dd id=conformance-checkers>

    <p>Conformance checkers must verify that a document conforms to the applicable conformance
    criteria described in this specification. Automated conformance checkers are exempt from
    detecting errors that require interpretation of the author's intent (for example, while a
    document is non-conforming if the content of a <code id=conformance-classes:the-blockquote-element><a href=#the-blockquote-element>blockquote</a></code> element is not a quote,
    conformance checkers running without the input of human judgement do not have to check that
    <code id=conformance-classes:the-blockquote-element-2><a href=#the-blockquote-element>blockquote</a></code> elements only contain quoted material).</p>

    <p>Conformance checkers must check that the input document conforms when parsed without a
    <a href=#concept-document-bc id=conformance-classes:concept-document-bc>browsing context</a> (meaning that no scripts are run, and
    that the parser's <a href=#scripting-flag id=conformance-classes:scripting-flag>scripting flag</a> is disabled), and should also check that the input
    document conforms when parsed with a <a href=#concept-document-bc id=conformance-classes:concept-document-bc-2>browsing context</a>
    in which scripts execute, and that the scripts never cause non-conforming states to occur other
    than transiently during script execution itself. (This is only a "SHOULD" and not a "MUST"
    requirement because it has been proven to be impossible. <a href=#refsCOMPUTABLE>[COMPUTABLE]</a>)</p>

    <p>The term "HTML validator" can be used to refer to a conformance checker that itself conforms
    to the applicable requirements of this specification.</p>

    <div class=note>

     <p>XML DTDs cannot express all the conformance requirements of this specification. Therefore, a
     validating XML processor and a DTD cannot constitute a conformance checker. Also, since neither
     of the two authoring formats defined in this specification are applications of SGML, a
     validating SGML system cannot constitute a conformance checker either.</p>

     <p>To put it another way, there are three types of conformance criteria:</p>

     <ol><li>Criteria that can be expressed in a DTD.<li>Criteria that cannot be expressed by a DTD, but can still be checked by a machine.<li>Criteria that can only be checked by a human.</ol>

     <p>A conformance checker must check for the first two. A simple DTD-based validator only checks
     for the first class of errors and is therefore not a conforming conformance checker according
     to this specification.</p>

    </div>
   <dt>Data mining tools<dd id=data-mining>

    <p>Applications and tools that process HTML and XML documents for reasons other than to either
    render the documents or check them for conformance should act in accordance with the semantics
    of the documents that they process.</p>

    <p class=example>A tool that generates <a href=#outline id=conformance-classes:outline>document outlines</a> but
    increases the nesting level for each paragraph and does not increase the nesting level for each
    section would not be conforming.</p>
   <dt id=editors>Authoring tools and markup generators<dd>
    <p>Authoring tools and markup generators must generate <a href=#conforming-documents id=conformance-classes:conforming-documents>conforming documents</a>.
    Conformance criteria that apply to authors also apply to authoring tools, where appropriate.</p>

    <p>Authoring tools are exempt from the strict requirements of using elements only for their
    specified purpose, but only to the extent that authoring tools are not yet able to determine
    author intent. However, authoring tools must not automatically misuse elements or encourage
    their users to do so.</p>

    <p class=example>For example, it is not conforming to use an <code id=conformance-classes:the-address-element><a href=#the-address-element>address</a></code> element for
    arbitrary contact information; that element can only be used for marking up contact information
    for its nearest <code id=conformance-classes:the-article-element><a href=#the-article-element>article</a></code> or <code id=conformance-classes:the-body-element><a href=#the-body-element>body</a></code> element ancestor. However, since an
    authoring tool is likely unable to determine the difference, an authoring tool is exempt from
    that requirement. This does not mean, though, that authoring tools can use <code id=conformance-classes:the-address-element-2><a href=#the-address-element>address</a></code>
    elements for any block of italics text (for instance); it just means that the authoring tool
    doesn't have to verify that when the user uses a tool for inserting contact information for an
    <code id=conformance-classes:the-article-element-2><a href=#the-article-element>article</a></code> element, that the user really is doing that and not inserting something
    else instead.</p>

    <p class=note>In terms of conformance checking, an editor has to output documents that conform
    to the same extent that a conformance checker will verify.</p>

    <p>When an authoring tool is used to edit a non-conforming document, it may preserve the
    conformance errors in sections of the document that were not edited during the editing session
    (i.e. an editing tool is allowed to round-trip erroneous content). However, an authoring tool
    must not claim that the output is conformant if errors have been so preserved.</p>

    <p>Authoring tools are expected to come in two broad varieties: tools that work from structure
    or semantic data, and tools that work on a What-You-See-Is-What-You-Get media-specific editing
    basis (WYSIWYG).</p>

    <p>The former is the preferred mechanism for tools that author HTML, since the structure in the
    source information can be used to make informed choices regarding which HTML elements and
    attributes are most appropriate.</p>

    <p>However, WYSIWYG tools are legitimate. WYSIWYG tools should use elements they know are
    appropriate, and should not use elements that they do not know to be appropriate. This might in
    certain extreme cases mean limiting the use of flow elements to just a few elements, like
    <code id=conformance-classes:the-div-element><a href=#the-div-element>div</a></code>, <code id=conformance-classes:the-b-element><a href=#the-b-element>b</a></code>, <code id=conformance-classes:the-i-element><a href=#the-i-element>i</a></code>, and <code id=conformance-classes:the-span-element><a href=#the-span-element>span</a></code> and making liberal use
    of the <code id=conformance-classes:attr-style><a href=#attr-style>style</a></code> attribute.</p>

    <p>All authoring tools, whether WYSIWYG or not, should make a best effort attempt at enabling
    users to create well-structured, semantically rich, media-independent content.</p>
   </dl>

  <p id=hardwareLimitations>
  <a id=conformance-classes:tracking-vector title="There is a tracking vector here." href=https://infra.spec.whatwg.org/#tracking-vector class=tracking-vector data-x-internal=tracking-vector><img width=46 alt="(This is a tracking vector.)" crossorigin="" src=https://resources.whatwg.org/tracking-vector.svg height=64></a>
  User agents may impose implementation-specific limits on otherwise unconstrained inputs, e.g., to
  prevent denial of service attacks, to guard against running out of memory, or to work around
  platform-specific limitations.</p>

  <p>For compatibility with existing content and prior specifications, this specification describes
  two authoring formats: one based on <a href=#the-xhtml-syntax id=conformance-classes:the-xhtml-syntax-2>XML</a>, and one using a <a href=#writing>custom format</a> inspired by SGML (referred to as <a href=#syntax id=conformance-classes:syntax-2>the HTML syntax</a>).
  Implementations must support at least one of these two formats, although supporting both is
  encouraged.</p>

  <p>Some conformance requirements are phrased as requirements on elements, attributes, methods or
  objects. Such requirements fall into two categories: those describing content model restrictions,
  and those describing implementation behavior. Those in the former category are requirements on
  documents and authoring tools. Those in the second category are requirements on user agents.
  Similarly, some conformance requirements are phrased as requirements on authors; such requirements
  are to be interpreted as conformance requirements on the documents that authors produce. (In other
  words, this specification does not distinguish between conformance criteria on authors and
  conformance criteria on documents.)</p>

  


  <h4 id=dependencies><span class=secno>2.1.9</span> Dependencies<a href=#dependencies class=self-link></a></h4>

  
  <div class="mdn-anno wrapped before"><button onclick=toggleStatus(this) class=mdn-anno-btn><b title="Support in all current engines." class=all-engines-flag>✔</b><span>MDN</span></button><div class=feature><p><a href=https://developer.mozilla.org/en-US/docs/Web/API/MessageEvent/MessageEvent title="The MessageEvent() constructor creates a new MessageEvent object instance.">MessageEvent/MessageEvent</a><p class=all-engines-text>Support in all current engines.<div class=support><span class="firefox yes"><span>Firefox</span><span>4+</span></span><span class="safari yes"><span>Safari</span><span>4+</span></span><span class="chrome yes"><span>Chrome</span><span>1+</span></span><hr><span class="opera unknown"><span>Opera</span><span>?</span></span><span class="edge_blink yes"><span>Edge</span><span>79+</span></span><hr><span class="edge yes"><span>Edge (Legacy)</span><span>12+</span></span><span class="ie yes"><span>Internet Explorer</span><span>9+</span></span><hr><span class="firefox_android unknown"><span>Firefox Android</span><span>?</span></span><span class="safari_ios yes"><span>Safari iOS</span><span>3+</span></span><span class="chrome_android yes"><span>Chrome Android</span><span>18+</span></span><span class="webview_android yes"><span>WebView Android</span><span>37+</span></span><span class="samsunginternet_android yes"><span>Samsung Internet</span><span>1.0+</span></span><span class="opera_android unknown"><span>Opera Android</span><span>?</span></span></div></div></div><div data-noexport="">

  <p>This specification relies on several other underlying specifications.</p>

  <dl><dt>Infra<dd>
    <p>The following terms are defined in <cite>Infra</cite>: <a href=#refsINFRA>[INFRA]</a></p>

    <ul class=brief><li>The general iteration terms <dfn id=while><a href=https://infra.spec.whatwg.org/#iteration-while>while</a></dfn>,
             <dfn id=continue><a href=https://infra.spec.whatwg.org/#iteration-continue>continue</a></dfn>, and
             <dfn id=break><a href=https://infra.spec.whatwg.org/#iteration-break>break</a></dfn>.<li><dfn id=implementation-defined><a href=https://infra.spec.whatwg.org/#implementation-defined>implementation-defined</a></dfn><li id=fingerprint><span id=fingerprinting-vector></span>
     
     <dfn id=tracking-vector><a href=https://infra.spec.whatwg.org/#tracking-vector>tracking vector</a></dfn><li><dfn id=code-point><a href=https://infra.spec.whatwg.org/#code-point>code point</a></dfn> and its synonym
         <dfn id=character><a href=https://infra.spec.whatwg.org/#code-point>character</a></dfn><li><dfn id=surrogate><a href=https://infra.spec.whatwg.org/#surrogate>surrogate</a></dfn><li><dfn id=scalar-value><a href=https://infra.spec.whatwg.org/#scalar-value>scalar value</a></dfn><li><dfn id=tuple><a href=https://infra.spec.whatwg.org/#tuple>tuple</a></dfn><li><dfn id=noncharacter><a href=https://infra.spec.whatwg.org/#noncharacter>noncharacter</a></dfn><li><dfn id=string><a href=https://infra.spec.whatwg.org/#string>string</a></dfn>,
         <dfn id=code-unit><a href=https://infra.spec.whatwg.org/#code-unit>code unit</a></dfn>,
         <dfn id=length><a href=https://infra.spec.whatwg.org/#string-length>length</a></dfn>, and
         <dfn id=code-point-length><a href=https://infra.spec.whatwg.org/#string-code-point-length>code point length</a></dfn><li id=case-sensitive>The string equality operations <dfn id=is><a href=https://infra.spec.whatwg.org/#string-is>is</a></dfn> and
         <dfn id=identical-to><a href=https://infra.spec.whatwg.org/#string-is>identical to</a></dfn><li><dfn id=scalar-value-string><a href=https://infra.spec.whatwg.org/#scalar-value-string>scalar value string</a></dfn><li><dfn id=space-characters><a href=https://infra.spec.whatwg.org/#ascii-whitespace>ASCII whitespace</a></dfn><li><dfn id=control><a href=https://infra.spec.whatwg.org/#control>control</a></dfn><li><dfn id=ascii-digits><a href=https://infra.spec.whatwg.org/#ascii-digit>ASCII digit</a></dfn><li><dfn id=uppercase-ascii-hex-digits><a href=https://infra.spec.whatwg.org/#ascii-upper-hex-digit>ASCII upper hex digit</a></dfn><li><dfn id=lowercase-ascii-hex-digits><a href=https://infra.spec.whatwg.org/#ascii-lower-hex-digit>ASCII lower hex digit</a></dfn><li><dfn id=ascii-hex-digits><a href=https://infra.spec.whatwg.org/#ascii-hex-digit>ASCII hex digit</a></dfn><li><dfn id=uppercase-ascii-letters><a href=https://infra.spec.whatwg.org/#ascii-upper-alpha>ASCII upper alpha</a></dfn><li><dfn id=lowercase-ascii-letters><a href=https://infra.spec.whatwg.org/#ascii-lower-alpha>ASCII lower alpha</a></dfn><li><dfn id=ascii-letters><a href=https://infra.spec.whatwg.org/#ascii-alpha>ASCII alpha</a></dfn><li><dfn id=alphanumeric-ascii-characters><a href=https://infra.spec.whatwg.org/#ascii-alphanumeric>ASCII alphanumeric</a></dfn><li><dfn id=isomorphic-decode><a href=https://infra.spec.whatwg.org/#isomorphic-decode>isomorphic decode</a></dfn><li><dfn id=converted-to-ascii-lowercase><a href=https://infra.spec.whatwg.org/#ascii-lowercase>ASCII lowercase</a></dfn><li><dfn id=converted-to-ascii-uppercase><a href=https://infra.spec.whatwg.org/#ascii-uppercase>ASCII uppercase</a></dfn><li><dfn id=ascii-case-insensitive><a href=https://infra.spec.whatwg.org/#ascii-case-insensitive>ASCII case-insensitive</a></dfn><li><dfn id=strip-newlines><a href=https://infra.spec.whatwg.org/#strip-newlines>strip newlines</a></dfn><li><dfn id=normalize-newlines><a href=https://infra.spec.whatwg.org/#normalize-newlines>normalize newlines</a></dfn><li><dfn id=strip-leading-and-trailing-ascii-whitespace><a href=https://infra.spec.whatwg.org/#strip-leading-and-trailing-ascii-whitespace>strip leading and trailing ASCII whitespace</a></dfn><li><dfn id=strip-and-collapse-ascii-whitespace><a href=https://infra.spec.whatwg.org/#strip-and-collapse-ascii-whitespace>strip and collapse ASCII whitespace</a></dfn><li><dfn id=split-a-string-on-spaces><a href=https://infra.spec.whatwg.org/#split-on-ascii-whitespace>split a string on ASCII whitespace</a></dfn><li><dfn id=split-a-string-on-commas><a href=https://infra.spec.whatwg.org/#split-on-commas>split a string on commas</a></dfn><li><dfn id=collect-a-sequence-of-code-points><a href=https://infra.spec.whatwg.org/#collect-a-sequence-of-code-points>collect a sequence of code points</a></dfn> and its associated
         <dfn id=position-variable><a href=https://infra.spec.whatwg.org/#string-position-variable>position variable</a></dfn><li><dfn id=skip-ascii-whitespace><a href=https://infra.spec.whatwg.org/#skip-ascii-whitespace>skip ASCII whitespace</a></dfn><li>The <dfn id=ordered-map><a href=https://infra.spec.whatwg.org/#ordered-map>ordered map</a></dfn> data structure and the associated definitions for
             <dfn id=map-value><a href=https://infra.spec.whatwg.org/#map-value>value</a></dfn>,
             <dfn id=map-entry><a href=https://infra.spec.whatwg.org/#map-entry>entry</a></dfn>,
             <dfn id=map-exists><a href=https://infra.spec.whatwg.org/#map-exists>exists</a></dfn>,
             <dfn id=map-get><a href=https://infra.spec.whatwg.org/#map-get>getting the value of an entry</a></dfn>,
             <dfn id=map-set><a href=https://infra.spec.whatwg.org/#map-set>setting the value of an entry</a></dfn>,
             <dfn id=map-remove><a href=https://infra.spec.whatwg.org/#map-remove>removing an entry</a></dfn>,
             <dfn id=map-clear><a href=https://infra.spec.whatwg.org/#map-clear>clear</a></dfn>,
             <dfn id=map-get-the-keys><a href=https://infra.spec.whatwg.org/#map-getting-the-keys>getting the keys</a></dfn>,
             <dfn id=map-size><a href=https://infra.spec.whatwg.org/#map-size>size</a></dfn>, and
             <dfn id=map-iterate><a href=https://infra.spec.whatwg.org/#map-iterate>iterate</a></dfn><li>The <dfn id=list><a href=https://infra.spec.whatwg.org/#list>list</a></dfn> data structure and the associated definitions for
             <dfn id=list-append><a href=https://infra.spec.whatwg.org/#list-append>append</a></dfn>,
             <dfn id=list-extend><a href=https://infra.spec.whatwg.org/#list-extend>extend</a></dfn>,
             <dfn id=list-replace><a href=https://infra.spec.whatwg.org/#list-remove>replace</a></dfn>,
             <dfn id=list-remove><a href=https://infra.spec.whatwg.org/#list-remove>remove</a></dfn>,
             <dfn id=list-empty><a href=https://infra.spec.whatwg.org/#list-empty>empty</a></dfn>,
             <dfn id=list-contains><a href=https://infra.spec.whatwg.org/#list-contain>contains</a></dfn>,
             <dfn id=list-size><a href=https://infra.spec.whatwg.org/#list-size>size</a></dfn>,
             <dfn id=list-is-empty><a href=https://infra.spec.whatwg.org/#list-is-empty>is empty</a></dfn>,
             <dfn id=list-iterate><a href=https://infra.spec.whatwg.org/#list-iterate>iterate</a></dfn>, and
             <dfn id=list-clone><a href=https://infra.spec.whatwg.org/#list-clone>clone</a></dfn><li>The <dfn id=stack><a href=https://infra.spec.whatwg.org/#stack>stack</a></dfn> data structure and the associated definitions for
             <dfn id=stack-push><a href=https://infra.spec.whatwg.org/#stack-push>push</a></dfn> and
             <dfn id=stack-pop><a href=https://infra.spec.whatwg.org/#stack-pop>pop</a></dfn><li>The <dfn id=queue><a href=https://infra.spec.whatwg.org/#queue>queue</a></dfn> data structure and the associated definitions for
             <dfn id=enqueue><a href=https://infra.spec.whatwg.org/#queue-enqueue>enqueue</a></dfn> and
             <dfn id=dequeue><a href=https://infra.spec.whatwg.org/#queue-dequeue>dequeue</a></dfn><li>The <dfn id=set><a href=https://infra.spec.whatwg.org/#ordered-set>ordered set</a></dfn> data structure and the associated definition for
             <dfn id=set-append><a href=https://infra.spec.whatwg.org/#set-append>append</a></dfn> and
             <dfn id=set-union><a href=https://infra.spec.whatwg.org/#set-union>union</a></dfn><li>The <dfn id=struct><a href=https://infra.spec.whatwg.org/#struct>struct</a></dfn> specification type and the associated definition for
             <dfn id=struct-item><a href=https://infra.spec.whatwg.org/#struct-item>item</a></dfn><li>The <dfn id=forgiving-base64-encode><a href=https://infra.spec.whatwg.org/#forgiving-base64-encode>forgiving-base64 encode</a></dfn> and
             <dfn id=forgiving-base64-decode><a href=https://infra.spec.whatwg.org/#forgiving-base64-decode>forgiving-base64 decode</a></dfn> algorithms<li><dfn id=html-namespace-2><a href=https://infra.spec.whatwg.org/#html-namespace>HTML namespace</a></dfn><li><dfn id=mathml-namespace><a href=https://infra.spec.whatwg.org/#mathml-namespace>MathML namespace</a></dfn><li><dfn id=svg-namespace><a href=https://infra.spec.whatwg.org/#svg-namespace>SVG namespace</a></dfn><li><dfn id=xlink-namespace><a href=https://infra.spec.whatwg.org/#xlink-namespace>XLink namespace</a></dfn><li><dfn id=xml-namespace><a href=https://infra.spec.whatwg.org/#xml-namespace>XML namespace</a></dfn><li><dfn id=xmlns-namespace><a href=https://infra.spec.whatwg.org/#xmlns-namespace>XMLNS namespace</a></dfn></ul>
   <dt>Unicode and Encoding<dd>
    <p>The Unicode character set is used to represent textual data, and <cite>Encoding</cite>
    defines requirements around <a href=https://encoding.spec.whatwg.org/#encoding id=dependencies:encoding data-x-internal=encoding>character encodings</a>. <a href=#refsUNICODE>[UNICODE]</a></p>

    <p class=note>This specification <a href=#encoding-terminology>introduces terminology</a>
    based on the terms defined in those specifications, as described earlier.</p>

    <p>The following terms are used as defined in <cite>Encoding</cite>: <a href=#refsENCODING>[ENCODING]</a></p>

    <ul class=brief><li><dfn id=getting-an-encoding><a href=https://encoding.spec.whatwg.org/#concept-encoding-get>Getting an
     encoding</a></dfn><li><dfn id=get-an-output-encoding><a href=https://encoding.spec.whatwg.org/#get-an-output-encoding>Get an output
     encoding</a></dfn><li>The generic <dfn id=decode><a href=https://encoding.spec.whatwg.org/#decode>decode</a></dfn>
     algorithm which takes a byte stream and an encoding and returns a character stream<li>The <dfn id=utf-8-decode><a href=https://encoding.spec.whatwg.org/#utf-8-decode>UTF-8 decode</a></dfn>
     algorithm which takes a byte stream and returns a character stream, additionally stripping one
     leading UTF-8 Byte Order Mark (BOM), if any<li>The <dfn id=utf-8-decode-without-bom><a href=https://encoding.spec.whatwg.org/#utf-8-decode-without-bom>UTF-8
     decode without BOM</a></dfn> algorithm which is identical to <a id=dependencies:utf-8-decode href=https://encoding.spec.whatwg.org/#utf-8-decode data-x-internal=utf-8-decode>UTF-8 decode</a> except that
     it does not strip one leading UTF-8 Byte Order Mark (BOM)<li>The <dfn id=encode><a href=https://encoding.spec.whatwg.org/#encode>encode</a></dfn> algorithm
     which takes a character stream and an encoding and returns a byte stream<li>The <dfn id=utf-8-encode><a href=https://encoding.spec.whatwg.org/#utf-8-encode>UTF-8 encode</a></dfn>
     algorithm which takes a character stream and returns a byte stream<li>The <dfn id=bom-sniff><a href=https://encoding.spec.whatwg.org/#bom-sniff>BOM sniff</a></dfn>
     algorithm which takes a byte stream and returns an encoding or null.</ul>
   <dt>XML and related specifications<dd>
    <p>Implementations that support <a href=#the-xhtml-syntax id=dependencies:the-xhtml-syntax>the XML syntax</a> for HTML must support some version
    of XML, as well as its corresponding namespaces specification, because that syntax uses an XML
    serialization with namespaces. <a href=#refsXML>[XML]</a> <a href=#refsXMLNS>[XMLNS]</a></p>

    <p>Data mining tools and other user agents that perform operations on content without running
    scripts, evaluating CSS or XPath expressions, or otherwise exposing the resulting DOM to
    arbitrary content, may "support namespaces" by just asserting that their DOM node analogues are
    in certain namespaces, without actually exposing the namespace strings.</p>

    <p class=note>In <a href=#syntax id=dependencies:syntax>the HTML syntax</a>, namespace prefixes and namespace declarations
    do not have the same effect as in XML. For instance, the colon has no special meaning in HTML
    element names.</p>

    <hr>

    <p>The attribute with the name <dfn id=attr-xml-space><a href=https://www.w3.org/TR/xml/#sec-white-space><code>space</code></a></dfn> in the <a id=dependencies:xml-namespace href=https://infra.spec.whatwg.org/#xml-namespace data-x-internal=xml-namespace>XML namespace</a> is defined by
    <cite>Extensible Markup Language</cite> (<cite>XML</cite>). <a href=#refsXML>[XML]</a></p>

    <p>The <dfn id=xml-name><a href=https://www.w3.org/TR/xml/#NT-Name><code>Name</code></a></dfn> production is defined in <cite>XML</cite>. <a href=#refsXML>[XML]</a></p>

    <p>This specification also references the <dfn id=xml-stylesheet><a href=https://www.w3.org/TR/xml-stylesheet/#the-xml-stylesheet-processing-instruction><code>&lt;?xml-stylesheet?></code></a></dfn>
    processing instruction, defined in <cite>Associating Style Sheets with XML documents</cite>.
    <a href=#refsXMLSSPI>[XMLSSPI]</a></p>

    <p>This specification also non-normatively mentions the <dfn id=xsltprocessor><code>XSLTProcessor</code></dfn>
    interface and its <dfn id=dom-xsltprocessor-transformtofragment><code>transformToFragment()</code></dfn> and <dfn id=dom-xsltprocessor-transformtodocument><code>transformToDocument()</code></dfn> methods. <a href=#refsXSLTP>[XSLTP]</a></p>
   <dt>URLs<dd>
    <p>The following terms are defined in <cite>URL</cite>: <a href=#refsURL>[URL]</a></p>

    <ul class=brief><li><dfn id=concept-host><a href=https://url.spec.whatwg.org/#concept-host>host</a></dfn><li><dfn id=public-suffix><a href=https://url.spec.whatwg.org/#host-public-suffix>public suffix</a></dfn><li><dfn id=concept-domain><a href=https://url.spec.whatwg.org/#concept-domain>domain</a></dfn><li><dfn id=ipv4-address><a href=https://url.spec.whatwg.org/#concept-ipv4>IPv4 address</a></dfn><li><dfn id=ipv6-address><a href=https://url.spec.whatwg.org/#concept-ipv6>IPv6 address</a></dfn><li><dfn id=url><a href=https://url.spec.whatwg.org/#concept-url>URL</a></dfn><li><dfn id=concept-url-origin><a href=https://url.spec.whatwg.org/#concept-url-origin>Origin</a></dfn> of URLs<li><dfn id=absolute-url><a href=https://url.spec.whatwg.org/#syntax-url-absolute>Absolute URL</a></dfn><li><dfn id=relative-url><a href=https://url.spec.whatwg.org/#syntax-url-relative>Relative URL</a></dfn><li><dfn id=registrable-domain><a href=https://url.spec.whatwg.org/#host-registrable-domain>registrable domain</a></dfn><li>
      The <dfn id=url-parser><a href=https://url.spec.whatwg.org/#concept-url-parser>URL parser</a></dfn> and
      <dfn id=basic-url-parser><a href=https://url.spec.whatwg.org/#concept-basic-url-parser>basic URL
      parser</a></dfn> as well as these parser states:
      <ul class=brief><li><dfn id=scheme-start-state><a href=https://url.spec.whatwg.org/#scheme-start-state>scheme start state</a></dfn><li><dfn id=host-state><a href=https://url.spec.whatwg.org/#host-state>host state</a></dfn><li><dfn id=hostname-state><a href=https://url.spec.whatwg.org/#hostname-state>hostname state</a></dfn><li><dfn id=port-state><a href=https://url.spec.whatwg.org/#port-state>port state</a></dfn><li><dfn id=path-start-state><a href=https://url.spec.whatwg.org/#path-start-state>path start state</a></dfn><li><dfn id=query-state><a href=https://url.spec.whatwg.org/#query-state>query state</a></dfn><li><dfn id=fragment-state><a href=https://url.spec.whatwg.org/#fragment-state>fragment state</a></dfn></ul>
     <li>
      <dfn id=url-record><a href=https://url.spec.whatwg.org/#concept-url>URL record</a></dfn>, as well as its
      individual components:
      <ul class=brief><li><dfn id=concept-url-scheme><a href=https://url.spec.whatwg.org/#concept-url-scheme>scheme</a></dfn><li><dfn id=concept-url-username><a href=https://url.spec.whatwg.org/#concept-url-username>username</a></dfn><li><dfn id=concept-url-password><a href=https://url.spec.whatwg.org/#concept-url-password>password</a></dfn><li><dfn id=concept-url-host><a href=https://url.spec.whatwg.org/#concept-url-host>host</a></dfn><li><dfn id=concept-url-port><a href=https://url.spec.whatwg.org/#concept-url-port>port</a></dfn><li><dfn id=concept-url-path><a href=https://url.spec.whatwg.org/#concept-url-path>path</a></dfn><li><dfn id=concept-url-query><a href=https://url.spec.whatwg.org/#concept-url-query>query</a></dfn><li><dfn id=concept-url-fragment><a href=https://url.spec.whatwg.org/#concept-url-fragment>fragment</a></dfn><li><dfn id=cannot-be-a-base-url-flag><a href=https://url.spec.whatwg.org/#url-cannot-be-a-base-url-flag>cannot-be-a-base-URL flag</a></dfn><li><dfn id=concept-url-object><a href=https://url.spec.whatwg.org/#concept-url-object>object</a></dfn></ul>
     <li><dfn id=valid-url-string><a href=https://url.spec.whatwg.org/#valid-url-string>valid URL string</a></dfn><li>The <dfn id=cannot-have-a-username/password/port><a href=https://url.spec.whatwg.org/#cannot-have-a-username-password-port>cannot have a username/password/port</a></dfn> concept<li>The <dfn id=concept-url-serializer><a href=https://url.spec.whatwg.org/#concept-url-serializer>URL serializer</a></dfn><li>The <dfn id=host-parser><a href=https://url.spec.whatwg.org/#concept-host-parser>host parser</a></dfn><li>The <dfn id=host-serializer><a href=https://url.spec.whatwg.org/#concept-host-serializer>host serializer</a></dfn><li><dfn id=host-equals><a href=https://url.spec.whatwg.org/#concept-host-equals>Host equals</a></dfn><li><dfn id=concept-url-equals><a href=https://url.spec.whatwg.org/#concept-url-equals>URL equals</a></dfn><li><dfn id=serialize-an-integer><a href=https://url.spec.whatwg.org/#serialize-an-integer>serialize an integer</a></dfn><li><dfn id=default-encode-set><a href=https://url.spec.whatwg.org/#default-encode-set>Default encode set</a></dfn><li><dfn id=component-percent-encode-set><a href=https://url.spec.whatwg.org/#component-percent-encode-set>component percent-encode set</a></dfn><li><dfn id=utf-8-percent-encode><a href=https://url.spec.whatwg.org/#string-utf-8-percent-encode>UTF-8 percent-encode</a></dfn><li><dfn id=percent-decode><a href=https://url.spec.whatwg.org/#string-percent-decode>percent-decode</a></dfn><li><dfn id=set-the-username><a href=https://url.spec.whatwg.org/#set-the-username>set the username</a></dfn><li><dfn id=set-the-password><a href=https://url.spec.whatwg.org/#set-the-password>set the password</a></dfn><li>The <dfn id=application/x-www-form-urlencoded><a href=https://url.spec.whatwg.org/#concept-urlencoded><code>application/x-www-form-urlencoded</code></a></dfn> format<li>The <dfn id=application/x-www-form-urlencoded-serializer><a href=https://url.spec.whatwg.org/#concept-urlencoded-serializer><code>application/x-www-form-urlencoded</code> serializer</a></dfn></ul>

    <p>A number of schemes and protocols are referenced by this specification also:</p>

    <ul class=brief><li>The <dfn id=about-protocol><a href=https://tools.ietf.org/html/rfc6694#section-2><code>about:</code></a></dfn> scheme <a href=#refsABOUT>[ABOUT]</a><li>The <dfn id=blob-protocol><a href=https://w3c.github.io/FileAPI/#DefinitionOfScheme><code>blob:</code></a></dfn> scheme <a href=#refsFILEAPI>[FILEAPI]</a><li>The <dfn id=data-protocol><a href=https://tools.ietf.org/html/rfc2397#section-2><code>data:</code></a></dfn> scheme <a href=#refsRFC2397>[RFC2397]</a><li>The <dfn id=http-protocol><a href=https://tools.ietf.org/html/rfc7230#section-2.7.1><code>http:</code></a></dfn> scheme <a href=#refsHTTP>[HTTP]</a><li>The <dfn id=https-protocol><a href=https://tools.ietf.org/html/rfc7230#section-2.7.2><code>https:</code></a></dfn> scheme <a href=#refsHTTP>[HTTP]</a><li>The <dfn id=mailto-protocol><a href=https://tools.ietf.org/html/rfc6068#section-2><code>mailto:</code></a></dfn> scheme <a href=#refsMAILTO>[MAILTO]</a><li>The <dfn id=sms-protocol><a href=https://tools.ietf.org/html/rfc5724#section-2><code>sms:</code></a></dfn> scheme <a href=#refsSMS>[SMS]</a><li>The <dfn id=urn-protocol><a href=https://tools.ietf.org/html/rfc2141#section-2><code>urn:</code></a></dfn> scheme <a href=#refsURN>[URN]</a></ul>

    <p><dfn id=media-fragment-syntax><a href=https://www.w3.org/TR/media-frags/#media-fragment-syntax>Media fragment
    syntax</a></dfn> is defined in <cite>Media Fragments URI</cite>. <a href=#refsMEDIAFRAG>[MEDIAFRAG]</a></p>
   <dt>HTTP and related specifications<dd>
    <p>The following terms are defined in the HTTP specifications: <a href=#refsHTTP>[HTTP]</a></p>

    <ul class=brief><li>`<dfn id=http-accept><a href=https://tools.ietf.org/html/rfc7231#section-5.3.2><code>Accept</code></a></dfn>` header<li>`<dfn id=http-accept-language><a href=https://tools.ietf.org/html/rfc7231#section-5.3.5><code>Accept-Language</code></a></dfn>` header<li>`<dfn id=http-cache-control><a href=https://tools.ietf.org/html/rfc7234#section-5.2><code>Cache-Control</code></a></dfn>` header<li>`<dfn id=http-content-disposition><a href=https://tools.ietf.org/html/rfc6266><code>Content-Disposition</code></a></dfn>` header<li>`<dfn id=http-content-language><a href=https://tools.ietf.org/html/rfc7231#section-3.1.3.2><code>Content-Language</code></a></dfn>` header<li>`<dfn id=http-last-modified><a href=https://tools.ietf.org/html/rfc7232#section-2.2><code>Last-Modified</code></a></dfn>` header<li>`<dfn id=http-referer><a href=https://tools.ietf.org/html/rfc7231#section-5.5.2><code>Referer</code></a></dfn>` header</ul>

    <p>The following terms are defined in <cite>HTTP State Management Mechanism</cite>: <a href=#refsCOOKIES>[COOKIES]</a></p>

    <ul class=brief><li><dfn id=cookie-string>cookie-string</dfn><li><dfn id=receives-a-set-cookie-string>receives a set-cookie-string</dfn><li>`<dfn id=http-cookie><code>Cookie</code></dfn>` header</ul>

    <p>The following term is defined in <cite>Web Linking</cite>: <a href=#refsWEBLINK>[WEBLINK]</a></p>

    <ul class=brief><li>`<dfn id=http-link><a href=https://tools.ietf.org/html/rfc8288#section-3><code>Link</code></a></dfn>` header</ul>

    <p>The following terms are defined in <cite>Structured Field Values for HTTP</cite>: <a href=#refsSTRUCTURED-FIELDS>[STRUCTURED-FIELDS]</a></p>

    <ul class=brief><li><dfn id=http-structured-header><a href=https://httpwg.org/http-extensions/draft-ietf-httpbis-header-structure.html>structured header</a></dfn><li><dfn id=http-structured-header-boolean><a href=https://httpwg.org/http-extensions/draft-ietf-httpbis-header-structure.html#boolean>boolean</a></dfn><li><dfn id=http-structured-header-token><a href=https://httpwg.org/http-extensions/draft-ietf-httpbis-header-structure.html#token>token</a></dfn><li><dfn id=http-structured-header-parameters><a href=https://httpwg.org/http-extensions/draft-ietf-httpbis-header-structure.html#param>parameters</a></dfn></ul>

    <p>The following terms are defined in <cite>MIME Sniffing</cite>: <a href=#refsMIMESNIFF>[MIMESNIFF]</a></p>

    <ul class=brief><li><dfn id=mime-type><a href=https://mimesniff.spec.whatwg.org/#mime-type>MIME type</a></dfn><li><dfn id=valid-mime-type-string><a href=https://mimesniff.spec.whatwg.org/#valid-mime-type>valid MIME type string</a></dfn><li><dfn id=valid-mime-type-string-with-no-parameters><a href=https://mimesniff.spec.whatwg.org/#valid-mime-type-with-no-parameters>valid MIME type string with no parameters</a></dfn><li><dfn id=html-mime-type><a href=https://mimesniff.spec.whatwg.org/#html-mime-type>HTML MIME type</a></dfn><li><dfn id=javascript-mime-type><a href=https://mimesniff.spec.whatwg.org/#javascript-mime-type>JavaScript MIME type</a></dfn> and
         <dfn id=javascript-mime-type-essence-match><a href=https://mimesniff.spec.whatwg.org/#javascript-mime-type-essence-match>JavaScript MIME type essence match</a></dfn><li><dfn id=json-mime-type><a href=https://mimesniff.spec.whatwg.org/#json-mime-type>JSON MIME type</a></dfn><li><dfn id=xml-mime-type><a href=https://mimesniff.spec.whatwg.org/#xml-mime-type>XML MIME type</a></dfn></ul>
   <dt>Fetch<dd>
    <p>The following terms are defined in <cite>Fetch</cite>: <a href=#refsFETCH>[FETCH]</a></p>

    <ul class=brief><li><dfn id=header-abnf><a href=https://fetch.spec.whatwg.org/#abnf>ABNF</a></dfn><li><dfn id=about:blank><code>about:blank</code></dfn><li>An <dfn id=http(s)-scheme><a href=https://fetch.spec.whatwg.org/#http-scheme>HTTP(S) scheme</a></dfn><li>A <dfn id=local-scheme><a href=https://fetch.spec.whatwg.org/#local-scheme>local scheme</a></dfn><li>A <dfn id=network-scheme><a href=https://fetch.spec.whatwg.org/#network-scheme>network scheme</a></dfn><li>A <dfn id=fetch-scheme><a href=https://fetch.spec.whatwg.org/#fetch-scheme>fetch scheme</a></dfn><li><dfn id=cors-protocol><a href=https://fetch.spec.whatwg.org/#http-cors-protocol>CORS protocol</a></dfn><li><dfn id=default-user-agent-value><a href=https://fetch.spec.whatwg.org/#default-user-agent-value>default `<code>User-Agent</code>` value</a></dfn><li><dfn id=extract-a-mime-type><a href=https://fetch.spec.whatwg.org/#concept-header-extract-mime-type>extract a MIME type</a></dfn><li><dfn id=concept-fetch><a href=https://fetch.spec.whatwg.org/#concept-fetch>fetch</a></dfn><li><dfn id=http-redirect-fetch><a href=https://fetch.spec.whatwg.org/#concept-http-redirect-fetch>HTTP-redirect fetch</a></dfn><li><dfn id=ok-status><a href=https://fetch.spec.whatwg.org/#ok-status>ok status</a></dfn><li><dfn id=navigation-request><a href=https://fetch.spec.whatwg.org/#navigation-request>navigation request</a></dfn><li><dfn id=network-error><a href=https://fetch.spec.whatwg.org/#concept-network-error>network error</a></dfn><li>`<dfn id=http-origin><a href=https://fetch.spec.whatwg.org/#http-origin><code>Origin</code></a></dfn>` header<li>`<dfn id=cross-origin-resource-policy><a href=https://fetch.spec.whatwg.org/#http-cross-origin-resource-policy><code>Cross-Origin-Resource-Policy</code></a></dfn>` header<li><dfn id=process-response><a href=https://fetch.spec.whatwg.org/#process-response>process response</a></dfn><li><dfn id=getting-a-structured-field-value><a href=https://fetch.spec.whatwg.org/#concept-header-list-get-structured-header>getting a structured field value</a></dfn><li><dfn id=concept-header-list-set><a href=https://fetch.spec.whatwg.org/#concept-header-list-set>set</a></dfn><li><dfn id=concept-header-list-get-decode-split><a href=https://fetch.spec.whatwg.org/#concept-header-list-get-decode-split>get, decode, and split</a></dfn><li><dfn id=concept-fetch-terminate><a href=https://fetch.spec.whatwg.org/#concept-fetch-terminate>terminate</a></dfn><li><dfn id=cross-origin-resource-policy-check><a href=https://fetch.spec.whatwg.org/#cross-origin-resource-policy-check>cross-origin resource policy check</a></dfn><li>the <dfn id=requestcredentials><a href=https://fetch.spec.whatwg.org/#requestcredentials><code>RequestCredentials</code></a></dfn> enumeration<li>the <dfn id=requestdestination><a href=https://fetch.spec.whatwg.org/#requestdestination><code>RequestDestination</code></a></dfn> enumeration<li>the <dfn id=fetch()><a href=https://fetch.spec.whatwg.org/#dom-global-fetch><code>fetch()</code></a></dfn> method<li><dfn id=serialize-a-response-url-for-reporting><a href=https://fetch.spec.whatwg.org/#serialize-a-response-url-for-reporting>serialize a response URL for reporting</a></dfn><li>
      <dfn id=concept-response><a href=https://fetch.spec.whatwg.org/#concept-response>response</a></dfn> and its
      associated:
      <ul class=brief><li><dfn id=concept-response-type><a href=https://fetch.spec.whatwg.org/#concept-response-type>type</a></dfn><li><dfn id=concept-response-url><a href=https://fetch.spec.whatwg.org/#concept-response-url>url</a></dfn><li><dfn id=concept-response-url-list><a href=https://fetch.spec.whatwg.org/#concept-response-url-list>url list</a></dfn><li><dfn id=concept-response-status><a href=https://fetch.spec.whatwg.org/#concept-response-status>status</a></dfn><li><dfn id=concept-response-header-list><a href=https://fetch.spec.whatwg.org/#concept-response-header-list>header list</a></dfn><li><dfn id=concept-response-body><a href=https://fetch.spec.whatwg.org/#concept-response-body>body</a></dfn><li><dfn id=concept-internal-response><a href=https://fetch.spec.whatwg.org/#concept-internal-response>internal response</a></dfn><li><dfn id=concept-response-csp-list><a href=https://fetch.spec.whatwg.org/#concept-response-csp-list>CSP list</a></dfn><li><dfn id=concept-response-location-url><a href=https://fetch.spec.whatwg.org/#concept-response-location-url>location URL</a></dfn></ul>
     <li>
      <dfn id=concept-request><a href=https://fetch.spec.whatwg.org/#concept-request>request</a></dfn> and its associated:
      <ul class=brief><li><dfn id=concept-request-url><a href=https://fetch.spec.whatwg.org/#concept-request-url>url</a></dfn><li><dfn id=concept-request-method><a href=https://fetch.spec.whatwg.org/#concept-request-method>method</a></dfn><li><dfn id=concept-request-header-list><a href=https://fetch.spec.whatwg.org/#concept-request-header-list>header list</a></dfn><li><dfn id=concept-request-body><a href=https://fetch.spec.whatwg.org/#concept-request-body>body</a></dfn><li><dfn id=concept-request-client><a href=https://fetch.spec.whatwg.org/#concept-request-client>client</a></dfn><li><dfn id=concept-request-url-list><a href=https://fetch.spec.whatwg.org/#concept-request-url-list>URL list</a></dfn><li><dfn id=concept-request-current-url><a href=https://fetch.spec.whatwg.org/#concept-request-current-url>current URL</a></dfn><li><dfn id=concept-request-reserved-client><a href=https://fetch.spec.whatwg.org/#concept-request-reserved-client>reserved client</a></dfn><li><dfn id=concept-request-replaces-client-id><a href=https://fetch.spec.whatwg.org/#concept-request-replaces-client-id>replaces client id</a></dfn><li><dfn id=concept-request-initiator><a href=https://fetch.spec.whatwg.org/#concept-request-initiator>initiator</a></dfn><li><dfn id=concept-request-destination><a href=https://fetch.spec.whatwg.org/#concept-request-destination>destination</a></dfn><li><dfn id=concept-potential-destination><a href=https://fetch.spec.whatwg.org/#concept-potential-destination>potential destination</a></dfn><li><dfn id=concept-potential-destination-translate><a href=https://fetch.spec.whatwg.org/#concept-potential-destination-translate>translating</a></dfn> a <a href=https://fetch.spec.whatwg.org/#concept-potential-destination id=dependencies:concept-potential-destination data-x-internal=concept-potential-destination>potential destination</a><li><dfn id=concept-script-like-destination><a href=https://fetch.spec.whatwg.org/#request-destination-script-like>script-like</a></dfn> <a href=https://fetch.spec.whatwg.org/#concept-request-destination id=dependencies:concept-request-destination data-x-internal=concept-request-destination>destinations</a><li><dfn id=concept-request-priority><a href=https://fetch.spec.whatwg.org/#concept-request-priority>priority</a></dfn><li><dfn id=concept-request-origin><a href=https://fetch.spec.whatwg.org/#concept-request-origin>origin</a></dfn><li><dfn id=concept-request-referrer><a href=https://fetch.spec.whatwg.org/#concept-request-referrer>referrer</a></dfn><li><dfn id=synchronous-flag><a href=https://fetch.spec.whatwg.org/#synchronous-flag>synchronous flag</a></dfn><li><dfn id=concept-request-mode><a href=https://fetch.spec.whatwg.org/#concept-request-mode>mode</a></dfn><li><dfn id=concept-request-credentials-mode><a href=https://fetch.spec.whatwg.org/#concept-request-credentials-mode>credentials mode</a></dfn><li><dfn id=use-url-credentials-flag><a href=https://fetch.spec.whatwg.org/#concept-request-use-url-credentials-flag>use-URL-credentials flag</a></dfn><li><dfn id=unsafe-request-flag><a href=https://fetch.spec.whatwg.org/#unsafe-request-flag>unsafe-request flag</a></dfn><li><dfn id=concept-request-cache-mode><a href=https://fetch.spec.whatwg.org/#concept-request-cache-mode>cache mode</a></dfn><li><dfn id=concept-request-redirect-mode><a href=https://fetch.spec.whatwg.org/#concept-request-redirect-mode>redirect mode</a></dfn><li><dfn id=concept-request-referrer-policy><a href=https://fetch.spec.whatwg.org/#concept-request-referrer-policy>referrer policy</a></dfn><li><dfn id=concept-request-nonce-metadata><a href=https://fetch.spec.whatwg.org/#concept-request-nonce-metadata>cryptographic nonce metadata</a></dfn><li><dfn id=concept-request-integrity-metadata><a href=https://fetch.spec.whatwg.org/#concept-request-integrity-metadata>integrity metadata</a></dfn><li><dfn id=concept-request-parser-metadata><a href=https://fetch.spec.whatwg.org/#concept-request-parser-metadata>parser metadata</a></dfn><li><dfn id=concept-request-reload-navigation-flag><a href=https://fetch.spec.whatwg.org/#concept-request-reload-navigation-flag>reload-navigation flag</a></dfn><li><dfn id=concept-request-history-navigation-flag><a href=https://fetch.spec.whatwg.org/#concept-request-history-navigation-flag>history-navigation flag</a></dfn></ul>
     </ul>

    <p>The following terms are defined in <cite>Referrer Policy</cite>: <a href=#refsREFERRERPOLICY>[REFERRERPOLICY]</a></p>

    <ul class=brief><li><dfn id=referrer-policy><a href=https://w3c.github.io/webappsec-referrer-policy/#referrer-policy>referrer policy</a></dfn><li>The `<dfn id=http-referrer-policy><a href=https://w3c.github.io/webappsec-referrer-policy/#referrer-policy-header-dfn><code>Referrer-Policy</code></a></dfn>` HTTP header<li>The <dfn id=parse-referrer-policy-header><a href=https://w3c.github.io/webappsec-referrer-policy/#parse-referrer-policy-from-header>parse a referrer policy from a `<code>Referrer-Policy</code>` header</a></dfn> algorithm<li>The "<dfn id=referrer-policy-no-referrer><a href=https://w3c.github.io/webappsec-referrer-policy/#referrer-policy-no-referrer><code>no-referrer</code></a></dfn>",
             "<dfn id=referrer-policy-no-referrer-when-downgrade><a href=https://w3c.github.io/webappsec-referrer-policy/#referrer-policy-no-referrer-when-downgrade><code>no-referrer-when-downgrade</code></a></dfn>",
             "<dfn id=referrer-policy-origin-when-cross-origin><a href=https://w3c.github.io/webappsec-referrer-policy/#referrer-policy-origin-when-cross-origin><code>origin-when-cross-origin</code></a></dfn>", and
             "<dfn id=referrer-policy-unsafe-url><a href=https://w3c.github.io/webappsec-referrer-policy/#referrer-policy-unsafe-url><code>unsafe-url</code></a></dfn>" referrer policies</ul>

    <p>The following terms are defined in <cite>Mixed Content</cite>: <a href=#refsMIX>[MIX]</a></p>

    <ul class=brief><li><dfn id=a-priori-authenticated-url><a href=https://w3c.github.io/webappsec-mixed-content/#a-priori-authenticated-url><i>a priori</i> authenticated URL</a></dfn></ul>
   <dt>Paint Timing<dd>
    <p>The following terms are defined in <cite>Paint Timing</cite>: <a href=#refsPAINTTIMING>[PAINTTIMING]</a></p>

    <ul class=brief><li><dfn id=mark-paint-timing><a href=https://w3c.github.io/paint-timing/#mark-paint-timing>mark paint timing</a></dfn></ul>
   <dt>Long Tasks<dd>
    <p>The following terms are defined in <cite>Long Tasks</cite>: <a href=#refsLONGTASKS>[LONGTASKS]</a></p>

    <ul class=brief><li><dfn id=report-long-tasks><a href=https://w3c.github.io/longtasks/#report-long-tasks>report long tasks</a></dfn></ul>
   <dt>Web IDL<dd>
    <p>The IDL fragments in this specification must be interpreted as required for conforming IDL
    fragments, as described in <cite>Web IDL</cite>. <a href=#refsWEBIDL>[WEBIDL]</a></p>

    <p>The following terms are defined in <cite>Web IDL</cite>:</p>

    <ul class=brief><li><dfn id=this><a href=https://heycam.github.io/webidl/#this>this</a></dfn><li><dfn id=extended-attribute><a href=https://heycam.github.io/webidl/#dfn-extended-attribute>extended attribute</a></dfn><li><dfn id=named-constructor><a href=https://heycam.github.io/webidl/#dfn-named-constructor>named constructor</a></dfn><li><dfn id=constructor-operation><a href=https://heycam.github.io/webidl/#idl-constructors>constructor operation</a></dfn><li><dfn id=overridden-constructor-steps><a href=https://heycam.github.io/webidl/#overridden-constructor-steps>overridden constructor steps</a></dfn><li><dfn id=internally-create-a-new-object-implementing-the-interface><a href=https://heycam.github.io/webidl/#internally-create-a-new-object-implementing-the-interface>internally create a new object implementing the interface</a></dfn><li><dfn id=array-index-property-name><a href=https://heycam.github.io/webidl/#dfn-array-index-property-name>array index property name</a></dfn><li><dfn id=supported-property-indices><a href=https://heycam.github.io/webidl/#dfn-supported-property-indices>supported property indices</a></dfn><li><dfn id=determine-the-value-of-an-indexed-property><a href=https://heycam.github.io/webidl/#dfn-determine-the-value-of-an-indexed-property>determine the value of an indexed property</a></dfn><li><dfn id=set-the-value-of-an-existing-indexed-property><a href=https://heycam.github.io/webidl/#dfn-set-the-value-of-an-existing-indexed-property>set the value of an existing indexed property</a></dfn><li><dfn id=set-the-value-of-a-new-indexed-property><a href=https://heycam.github.io/webidl/#dfn-set-the-value-of-a-new-indexed-property>set the value of a new indexed property</a></dfn><li><dfn id=support-named-properties><a href=https://heycam.github.io/webidl/#dfn-support-named-properties>support named properties</a></dfn><li><dfn id=supported-property-names><a href=https://heycam.github.io/webidl/#dfn-supported-property-names>supported property names</a></dfn><li><dfn id=determine-the-value-of-a-named-property><a href=https://heycam.github.io/webidl/#dfn-determine-the-value-of-a-named-property>determine the value of a named property</a></dfn><li><dfn id=set-the-value-of-an-existing-named-property><a href=https://heycam.github.io/webidl/#dfn-set-the-value-of-an-existing-named-property>set the value of an existing named property</a></dfn><li><dfn id=set-the-value-of-a-new-named-property><a href=https://heycam.github.io/webidl/#dfn-set-the-value-of-a-new-named-property>set the value of a new named property</a></dfn><li><dfn id=delete-an-existing-named-property><a href=https://heycam.github.io/webidl/#dfn-delete-an-existing-named-property>delete an existing named property</a></dfn><li><dfn id=perform-a-security-check><a href=https://heycam.github.io/webidl/#dfn-perform-a-security-check>perform a security check</a></dfn><li><dfn id=platform-object><a href=https://heycam.github.io/webidl/#dfn-platform-object>platform object</a></dfn><li><dfn id=legacy-platform-object><a href=https://heycam.github.io/webidl/#dfn-legacy-platform-object>legacy platform object</a></dfn><li><dfn id=primary-interface><a href=https://heycam.github.io/webidl/#dfn-primary-interface>primary interface</a></dfn><li><dfn id=interface-object><a href=https://heycam.github.io/webidl/#dfn-interface-object>interface object</a></dfn><li><dfn id=include><a href=https://heycam.github.io/webidl/#include>include</a></dfn><li><dfn id=inherit><a href=https://heycam.github.io/webidl/#dfn-inherit>inherit</a></dfn><li><dfn id=interface-prototype-object><a href=https://heycam.github.io/webidl/#dfn-interface-prototype-object>interface prototype object</a></dfn><li><dfn id=realm-field-of-a-platform-object><a href=https://heycam.github.io/webidl/#es-platform-objects>[[Realm]] field of a platform object</a></dfn><li><dfn id=callback-context><a href=https://heycam.github.io/webidl/#dfn-callback-context>callback context</a></dfn><li><dfn id=frozen-array><a href=https://heycam.github.io/webidl/#dfn-frozen-array-type>frozen array</a></dfn> and
         <dfn id=creating-a-frozen-array><a href=https://heycam.github.io/webidl/#dfn-create-frozen-array>creating a frozen array</a></dfn><li><dfn id=new><a href=https://heycam.github.io/webidl/#new>create a new object implementing the interface</a></dfn><li><dfn id=dfn-callback-this-value><a href=https://heycam.github.io/webidl/#dfn-callback-this-value>callback this value</a></dfn><li><dfn id=concept-idl-convert><a href=https://heycam.github.io/webidl/#es-type-mapping>converting</a></dfn> between Web IDL types and JS types<li><dfn id=es-invoking-callback-functions><a href=https://heycam.github.io/webidl/#invoke-a-callback-function>invoking</a></dfn> and
         <dfn id=es-constructing-callback-functions><a href=https://heycam.github.io/webidl/#construct-a-callback-function>constructing</a></dfn> callback functions<li><dfn id=converting-to-a-sequence-of-unicode-scalar-values><a href=https://heycam.github.io/webidl/#dfn-obtain-unicode>converting to a
     sequence of Unicode scalar values</a></dfn><li><dfn id=overload-resolution-algorithm><a href=https://heycam.github.io/webidl/#dfn-overload-resolution-algorithm>overload resolution algorithm</a></dfn><li><dfn id=idl-exposed><a href=https://heycam.github.io/webidl/#dfn-exposed>exposed</a></dfn><li><dfn id=legacyfactoryfunction><a href=https://heycam.github.io/webidl/#LegacyFactoryFunction><code>[LegacyFactoryFunction]</code></a></dfn><li><dfn id=legacylenientthis><a href=https://heycam.github.io/webidl/#LegacyLenientThis><code>[LegacyLenientThis]</code></a></dfn><li><dfn id=legacynulltoemptystring><a href=https://heycam.github.io/webidl/#LegacyNullToEmptyString><code>[LegacyNullToEmptyString]</code></a></dfn><li><dfn id=legacyoverridebuiltins><a href=https://heycam.github.io/webidl/#LegacyOverrideBuiltIns><code>[LegacyOverrideBuiltIns]</code></a></dfn><li><dfn id=legacytreatnonobjectasnull><a href=https://heycam.github.io/webidl/#LegacyTreatNonObjectAsNull><code>[LegacyTreatNonObjectAsNull]</code></a></dfn><li><dfn id=legacyunenumerablenamedproperties><a href=https://heycam.github.io/webidl/#LegacyUnenumerableNamedProperties><code>[LegacyUnenumerableNamedProperties]</code></a></dfn><li><dfn id=legacyunforgeable><a href=https://heycam.github.io/webidl/#LegacyUnforgeable><code>[LegacyUnforgeable]</code></a></dfn></ul>

    <p>The <cite>Web IDL</cite> also defines the following types that are used in Web IDL fragments
    in this specification:</p>

    <ul class=brief><li><dfn id=idl-arraybuffer><a href=https://heycam.github.io/webidl/#idl-ArrayBuffer><code>ArrayBuffer</code></a></dfn><li><dfn id=idl-arraybufferview><a href=https://heycam.github.io/webidl/#common-ArrayBufferView><code>ArrayBufferView</code></a></dfn><li><dfn id=idl-boolean><a href=https://heycam.github.io/webidl/#idl-boolean><code>boolean</code></a></dfn><li><dfn id=idl-domstring><a href=https://heycam.github.io/webidl/#idl-DOMString><code>DOMString</code></a></dfn><li><dfn id=idl-double><a href=https://heycam.github.io/webidl/#idl-double><code>double</code></a></dfn><li><dfn id=idl-enumeration><a href=https://heycam.github.io/webidl/#idl-enums>enumeration</a></dfn><li><dfn id=idl-error><a href=https://heycam.github.io/webidl/#idl-Error><code>Error</code></a></dfn><li><dfn id=idl-function><a href=https://heycam.github.io/webidl/#common-Function><code>Function</code></a></dfn><li><dfn id=idl-long><a href=https://heycam.github.io/webidl/#idl-long><code>long</code></a></dfn><li><dfn id=idl-object><a href=https://heycam.github.io/webidl/#idl-object><code>object</code></a></dfn><li><dfn id=idl-uint8clampedarray><a href=https://heycam.github.io/webidl/#idl-Uint8ClampedArray><code>Uint8ClampedArray</code></a></dfn><li><dfn id=idl-unrestricted-double><a href=https://heycam.github.io/webidl/#idl-unrestricted-double><code>unrestricted double</code></a></dfn><li><dfn id=idl-unsigned-long><a href=https://heycam.github.io/webidl/#idl-unsigned-long><code>unsigned long</code></a></dfn><li><dfn id=idl-usvstring><a href=https://heycam.github.io/webidl/#idl-USVString><code>USVString</code></a></dfn><li><dfn id=idl-voidfunction><a href=https://heycam.github.io/webidl/#VoidFunction><code>VoidFunction</code></a></dfn></ul>

    <p>The term <dfn id=throw><a href=https://heycam.github.io/webidl/#dfn-throw>throw</a></dfn> in this
    specification is used as defined in <cite>Web IDL</cite>. The <dfn id=domexception><a href=https://heycam.github.io/webidl/#dfn-DOMException><code>DOMException</code></a></dfn>
    type and the following exception names are defined by Web IDL and used by this
    specification:</p>

    <ul class=brief><li><dfn id=indexsizeerror><a href=https://heycam.github.io/webidl/#indexsizeerror>"<code>IndexSizeError</code>"</a></dfn><li><dfn id=hierarchyrequesterror><a href=https://heycam.github.io/webidl/#hierarchyrequesterror>"<code>HierarchyRequestError</code>"</a></dfn><li><dfn id=invalidcharactererror><a href=https://heycam.github.io/webidl/#invalidcharactererror>"<code>InvalidCharacterError</code>"</a></dfn><li><dfn id=notfounderror><a href=https://heycam.github.io/webidl/#notfounderror>"<code>NotFoundError</code>"</a></dfn><li><dfn id=notsupportederror><a href=https://heycam.github.io/webidl/#notsupportederror>"<code>NotSupportedError</code>"</a></dfn><li><dfn id=invalidstateerror><a href=https://heycam.github.io/webidl/#invalidstateerror>"<code>InvalidStateError</code>"</a></dfn><li><dfn id=syntaxerror><a href=https://heycam.github.io/webidl/#syntaxerror>"<code>SyntaxError</code>"</a></dfn><li><dfn id=invalidaccesserror><a href=https://heycam.github.io/webidl/#invalidaccesserror>"<code>InvalidAccessError</code>"</a></dfn><li><dfn id=securityerror><a href=https://heycam.github.io/webidl/#securityerror>"<code>SecurityError</code>"</a></dfn><li><dfn id=networkerror><a href=https://heycam.github.io/webidl/#networkerror>"<code>NetworkError</code>"</a></dfn><li><dfn id=aborterror><a href=https://heycam.github.io/webidl/#aborterror>"<code>AbortError</code>"</a></dfn><li><dfn id=quotaexceedederror><a href=https://heycam.github.io/webidl/#quotaexceedederror>"<code>QuotaExceededError</code>"</a></dfn><li><dfn id=datacloneerror><a href=https://heycam.github.io/webidl/#datacloneerror>"<code>DataCloneError</code>"</a></dfn><li><dfn id=encodingerror><a href=https://heycam.github.io/webidl/#encodingerror>"<code>EncodingError</code>"</a></dfn><li><dfn id=notallowederror><a href=https://heycam.github.io/webidl/#notallowederror>"<code>NotAllowedError</code>"</a></dfn></ul>

    <p>When this specification requires a user agent to <dfn id=create-a-date-object>create a <code>Date</code> object</dfn>
    representing a particular time (which could be the special value Not-a-Number), the milliseconds
    component of that time, if any, must be truncated to an integer, and the time value of the newly
    created <code id=dependencies:date><a data-x-internal=date href=https://tc39.es/ecma262/#sec-date-objects>Date</a></code> object must represent the resulting truncated time.</p>

    <p class=example>For instance, given the time 23045 millionths of a second after 01:00 UTC on
    January 1st 2000, i.e. the time 2000-01-01T00:00:00.023045Z, then the <code id=dependencies:date-2><a data-x-internal=date href=https://tc39.es/ecma262/#sec-date-objects>Date</a></code> object
    created representing that time would represent the same time as that created representing the
    time 2000-01-01T00:00:00.023Z, 45 millionths earlier. If the given time is NaN, then the result
    is a <code id=dependencies:date-3><a data-x-internal=date href=https://tc39.es/ecma262/#sec-date-objects>Date</a></code> object that represents a time value NaN (indicating that the object does
    not represent a specific instant of time).</p>
   <dt>JavaScript<dd>
    <p>Some parts of the language described by this specification only support JavaScript as the
    underlying scripting language. <a href=#refsJAVASCRIPT>[JAVASCRIPT]</a></p>

    <p class=note>The term "JavaScript" is used to refer to ECMA-262, rather than the official
    term ECMAScript, since the term JavaScript is more widely known. Similarly, the <a id=dependencies:mime-type href=https://mimesniff.spec.whatwg.org/#mime-type data-x-internal=mime-type>MIME
    type</a> used to refer to JavaScript in this specification is <code>text/javascript</code>, since that is the most commonly used type, <a href=#willful-violation id=dependencies:willful-violation>despite it being an officially obsoleted type</a> according to RFC
    4329. <a href=#refsRFC4329>[RFC4329]</a></p>

    <p>The following terms are defined in the JavaScript specification and used in this
    specification:</p>

    <ul class=brief><li><dfn id=active-function-object><a href=https://tc39.es/ecma262/#active-function-object>active function object</a></dfn><li><dfn id=agent><a href=https://tc39.es/ecma262/#sec-agents>agent</a></dfn> and
         <dfn id=agent-cluster><a href=https://tc39.es/ecma262/#sec-agent-clusters>agent cluster</a></dfn><li><dfn id=automatic-semicolon-insertion><a href=https://tc39.es/ecma262/#sec-automatic-semicolon-insertion>automatic semicolon insertion</a></dfn><li><dfn id=candidate-execution><a href=https://tc39.es/ecma262/#sec-candidate-executions>candidate execution</a></dfn><li>The <dfn id=current-realm-record><a href=https://tc39.es/ecma262/#current-realm>current Realm Record</a></dfn><li><dfn id=early-error><a href=https://tc39.es/ecma262/#early-error-rule>early error</a></dfn><li><dfn id=forward-progress><a href=https://tc39.es/ecma262/#sec-forward-progress>forward progress</a></dfn><li><dfn id=invariants-of-the-essential-internal-methods><a href=https://tc39.es/ecma262/#sec-invariants-of-the-essential-internal-methods>invariants of the essential internal methods</a></dfn><li><dfn id=javascript-execution-context><a href=https://tc39.es/ecma262/#sec-execution-contexts>JavaScript execution context</a></dfn><li><dfn id=javascript-execution-context-stack><a href=https://tc39.es/ecma262/#execution-context-stack>JavaScript execution context stack</a></dfn><li><dfn id=javascript-realm><a href=https://tc39.es/ecma262/#sec-code-realms>JavaScript realm</a></dfn><li><dfn id=environmentrecord><a href=https://tc39.es/ecma262/#sec-lexical-environments>EnvironmentRecord</a></dfn><li><dfn id=newtarget><a href=https://tc39.es/ecma262/#sec-built-in-function-objects>NewTarget</a></dfn><li><dfn id=running-javascript-execution-context><a href=https://tc39.es/ecma262/#running-execution-context>running JavaScript execution context</a></dfn><li><dfn id=surrounding-agent><a href=https://tc39.es/ecma262/#surrounding-agent>surrounding agent</a></dfn><li><dfn id=abstract-closure><a href=https://tc39.es/ecma262/#sec-abstract-closure>abstract closure</a></dfn><li><dfn id=immutable-prototype-exotic-object><a href=https://tc39.es/ecma262/#immutable-prototype-exotic-object>immutable prototype exotic object</a></dfn><li><dfn id=well-known-symbols><a href=https://tc39.es/ecma262/#sec-well-known-symbols>Well-Known Symbols</a></dfn>, including
         <dfn id=@@hasinstance>@@hasInstance</dfn>,
         <dfn id=@@isconcatspreadable>@@isConcatSpreadable</dfn>,
         <dfn id=@@toprimitive>@@toPrimitive</dfn>, and
         <dfn id=@@tostringtag>@@toStringTag</dfn><li><dfn id=well-known-intrinsic-objects><a href=https://tc39.es/ecma262/#sec-well-known-intrinsic-objects>Well-Known Intrinsic Objects</a></dfn>, including
         <dfn id=array.prototype><a href=https://tc39.es/ecma262/#sec-properties-of-the-array-prototype-object>%Array.prototype%</a></dfn>,
         <dfn id=error.prototype><a href=https://tc39.es/ecma262/#sec-properties-of-the-error-prototype-object>%Error.prototype%</a></dfn>,
         <dfn id=evalerror.prototype>%EvalError.prototype%</dfn>,
         <dfn id=function.prototype><a href=https://tc39.es/ecma262/#sec-properties-of-the-function-prototype-object>%Function.prototype%</a></dfn>,
         <dfn id=json.parse><a href=https://tc39.es/ecma262/#sec-json.parse>%JSON.parse%</a></dfn>,
         <dfn id=object.prototype><a href=https://tc39.es/ecma262/#sec-properties-of-the-object-prototype-object>%Object.prototype%</a></dfn>,
         <dfn id=object.prototype.valueof><a href=https://tc39.es/ecma262/#sec-object.prototype.valueof>%Object.prototype.valueOf%</a></dfn>,
         <dfn id=rangeerror.prototype>%RangeError.prototype%</dfn>,
         <dfn id=referenceerror.prototype>%ReferenceError.prototype%</dfn>,
         <dfn id=syntaxerror.prototype>%SyntaxError.prototype%</dfn>,
         <dfn id=typeerror.prototype>%TypeError.prototype%</dfn>, and
         <dfn id=urierror.prototype>%URIError.prototype%</dfn><li>The <dfn id=js-prod-functionbody><a href=https://tc39.es/ecma262/#prod-FunctionBody><i>FunctionBody</i></a></dfn> production<li>The <dfn id=js-prod-module><a href=https://tc39.es/ecma262/#prod-Module><i>Module</i></a></dfn> production<li>The <dfn id=js-prod-pattern><a href=https://tc39.es/ecma262/#prod-Pattern><i>Pattern</i></a></dfn> production<li>The <dfn id=js-prod-script><a href=https://tc39.es/ecma262/#prod-Script><i>Script</i></a></dfn> production<li>The <dfn id=js-type><a href=https://tc39.es/ecma262/#sec-ecmascript-data-types-and-values>Type</a></dfn> notation<li>The <dfn id=abrupt-completion><a href=https://tc39.es/ecma262/#sec-completion-record-specification-type>Completion Record</a></dfn> specification type<li>The <dfn id=js-list><a href=https://tc39.es/ecma262/#sec-list-and-record-specification-type>List</a></dfn> and
             <dfn id=record><a href=https://tc39.es/ecma262/#sec-list-and-record-specification-type>Record</a></dfn> specification types<li>The <dfn id=propertydescriptor><a href=https://tc39.es/ecma262/#sec-property-descriptor-specification-type>Property Descriptor</a></dfn> specification type<li>The <dfn id=script-record><a href=https://tc39.es/ecma262/#sec-script-records>Script Record</a></dfn> specification type<li>The <dfn id=cyclic-module-record><a href=https://tc39.es/ecma262/#sec-cyclic-module-records>Cyclic Module Record</a></dfn> specification type<li>The <dfn id=source-text-module-record><a href=https://tc39.es/ecma262/#sec-source-text-module-records>Source Text Module Record</a></dfn> specification type and its
             <dfn id=js-evaluate><a href=https://tc39.es/ecma262/#sec-moduleevaluation>Evaluate</a></dfn> and
             <dfn id=js-link><a href=https://tc39.es/ecma262/#sec-moduledeclarationlinking>Link</a></dfn> methods<li>The <dfn id=arraycreate><a href=https://tc39.es/ecma262/#sec-arraycreate>ArrayCreate</a></dfn> abstract operation<li>The <dfn id=call><a href=https://tc39.es/ecma262/#sec-call>Call</a></dfn> abstract operation<li>The <dfn id=construct><a href=https://tc39.es/ecma262/#sec-construct>Construct</a></dfn> abstract operation<li>The <dfn id=copydatablockbytes><a href=https://tc39.es/ecma262/#sec-copydatablockbytes>CopyDataBlockBytes</a></dfn> abstract operation<li>The <dfn id=createbytedatablock><a href=https://tc39.es/ecma262/#sec-createbytedatablock>CreateByteDataBlock</a></dfn> abstract operation<li>The <dfn id=createdataproperty><a href=https://tc39.es/ecma262/#sec-createdataproperty>CreateDataProperty</a></dfn> abstract operation<li>The <dfn id=detacharraybuffer><a href=https://tc39.es/ecma262/#sec-detacharraybuffer>DetachArrayBuffer</a></dfn> abstract operation<li>The <dfn id=enumerableownpropertynames><a href=https://tc39.es/ecma262/#sec-enumerableownpropertynames>EnumerableOwnPropertyNames</a></dfn> abstract operation<li>The <dfn id=finishdynamicimport><a href=https://tc39.es/ecma262/#sec-finishdynamicimport>FinishDynamicImport</a></dfn> abstract operation<li>The <dfn id=js-ordinaryfunctioncreate><a href=https://tc39.es/ecma262/#sec-ordinaryfunctioncreate>OrdinaryFunctionCreate</a></dfn> abstract operation<li>The <dfn id=js-get><a href=https://tc39.es/ecma262/#sec-get-o-p>Get</a></dfn> abstract operation<li>The <dfn id=getactivescriptormodule><a href=https://tc39.es/ecma262/#sec-getactivescriptormodule>GetActiveScriptOrModule</a></dfn> abstract operation<li>The <dfn id=getfunctionrealm><a href=https://tc39.es/ecma262/#sec-getfunctionrealm>GetFunctionRealm</a></dfn> abstract operation<li>The <dfn id=hasownproperty><a href=https://tc39.es/ecma262/#sec-hasownproperty>HasOwnProperty</a></dfn> abstract operation<li>The <dfn id=js-hostenqueuepromisejob><a href=https://tc39.es/ecma262/#sec-hostenqueuepromisejob>HostEnqueuePromiseJob</a></dfn> abstract operation<li>The <dfn id=js-hostensurecancompilestrings><a href=https://tc39.es/ecma262/#sec-hostensurecancompilestrings>HostEnsureCanCompileStrings</a></dfn> abstract operation<li>The <dfn id=js-hostimportmoduledynamically><a href=https://tc39.es/proposal-dynamic-import/#sec-hostimportmoduledynamically>HostImportModuleDynamically</a></dfn> abstract operation<li>The <dfn id=js-hostpromiserejectiontracker><a href=https://tc39.es/ecma262/#sec-host-promise-rejection-tracker>HostPromiseRejectionTracker</a></dfn> abstract operation<li>The <dfn id=js-hostresolveimportedmodule><a href=https://tc39.es/ecma262/#sec-hostresolveimportedmodule>HostResolveImportedModule</a></dfn> abstract operation<li>The <dfn id=js-initializehostdefinedrealm><a href=https://tc39.es/ecma262/#sec-initializehostdefinedrealm>InitializeHostDefinedRealm</a></dfn> abstract operation<li>The <dfn id=isaccessordescriptor><a href=https://tc39.es/ecma262/#sec-isaccessordescriptor>IsAccessorDescriptor</a></dfn> abstract operation<li>The <dfn id=iscallable><a href=https://tc39.es/ecma262/#sec-iscallable>IsCallable</a></dfn> abstract operation<li>The <dfn id=isconstructor><a href=https://tc39.es/ecma262/#sec-isconstructor>IsConstructor</a></dfn> abstract operation<li>The <dfn id=isdatadescriptor><a href=https://tc39.es/ecma262/#sec-isdatadescriptor>IsDataDescriptor</a></dfn> abstract operation<li>The <dfn id=isdetachedbuffer><a href=https://tc39.es/ecma262/#sec-isdetachedbuffer>IsDetachedBuffer</a></dfn> abstract operation<li>The <dfn id=issharedarraybuffer><a href=https://tc39.es/ecma262/#sec-issharedarraybuffer>IsSharedArrayBuffer</a></dfn> abstract operation<li>The <dfn id=js-newobjectenvironment><a href=https://tc39.es/ecma262/#sec-newobjectenvironment>NewObjectEnvironment</a></dfn> abstract operation<li>The <dfn id=normalcompletion><a href=https://tc39.es/ecma262/#sec-normalcompletion>NormalCompletion</a></dfn> abstract operation<li>The <dfn id=ordinarygetprototypeof><a href=https://tc39.es/ecma262/#sec-ordinarygetprototypeof>OrdinaryGetPrototypeOf</a></dfn> abstract operation<li>The <dfn id=ordinarysetprototypeof><a href=https://tc39.es/ecma262/#sec-ordinarysetprototypeof>OrdinarySetPrototypeOf</a></dfn> abstract operation<li>The <dfn id=ordinaryisextensible><a href=https://tc39.es/ecma262/#sec-ordinaryisextensible>OrdinaryIsExtensible</a></dfn> abstract operation<li>The <dfn id=ordinarypreventextensions><a href=https://tc39.es/ecma262/#sec-ordinarypreventextensions>OrdinaryPreventExtensions</a></dfn> abstract operation<li>The <dfn id=ordinarygetownproperty><a href=https://tc39.es/ecma262/#sec-ordinarygetownproperty>OrdinaryGetOwnProperty</a></dfn> abstract operation<li>The <dfn id=ordinarydefineownproperty><a href=https://tc39.es/ecma262/#sec-ordinarydefineownproperty>OrdinaryDefineOwnProperty</a></dfn> abstract operation<li>The <dfn id=ordinaryget><a href=https://tc39.es/ecma262/#sec-ordinaryget>OrdinaryGet</a></dfn> abstract operation<li>The <dfn id=ordinaryset><a href=https://tc39.es/ecma262/#sec-ordinaryset>OrdinarySet</a></dfn> abstract operation<li>The <dfn id=ordinarydelete><a href=https://tc39.es/ecma262/#sec-ordinarydelete>OrdinaryDelete</a></dfn> abstract operation<li>The <dfn id=ordinaryownpropertykeys><a href=https://tc39.es/ecma262/#sec-ordinaryownpropertykeys>OrdinaryOwnPropertyKeys</a></dfn> abstract operation<li>The <dfn id=objectcreate><a href=https://tc39.es/ecma262/#sec-objectcreate>ObjectCreate</a></dfn> abstract operation<li>The <dfn id=js-parsemodule><a href=https://tc39.es/ecma262/#sec-parsemodule>ParseModule</a></dfn> abstract operation<li>The <dfn id=js-parsescript><a href=https://tc39.es/ecma262/#sec-parse-script>ParseScript</a></dfn> abstract operation<li>The <dfn id=newpromisereactionjob><a href=https://tc39.es/ecma262/#sec-newpromisereactionjob>NewPromiseReactionJob</a></dfn> abstract operation<li>The <dfn id=newpromiseresolvethenablejob><a href=https://tc39.es/ecma262/#sec-newpromiseresolvethenablejob>NewPromiseResolveThenableJob</a></dfn> abstract operation<li>The <dfn id=regexpbuiltinexec><a href=https://tc39.es/ecma262/#sec-regexpbuiltinexec>RegExpBuiltinExec</a></dfn> abstract operation<li>The <dfn id=regexpcreate><a href=https://tc39.es/ecma262/#sec-regexpcreate>RegExpCreate</a></dfn> abstract operation<li>The <dfn id=runjobs><a href=https://tc39.es/ecma262/#sec-runjobs>RunJobs</a></dfn> abstract operation<li>The <dfn id=samevalue><a href=https://tc39.es/ecma262/#sec-samevalue>SameValue</a></dfn> abstract operation<li>The <dfn id=js-scriptevaluation><a href=https://tc39.es/ecma262/#sec-runtime-semantics-scriptevaluation>ScriptEvaluation</a></dfn> abstract operation<li>The <dfn id=setimmutableprototype><a href=https://tc39.es/ecma262/#sec-set-immutable-prototype>SetImmutablePrototype</a></dfn> abstract operation<li>The <dfn id=js-toboolean><a href=https://tc39.es/ecma262/#sec-toboolean>ToBoolean</a></dfn> abstract operation<li>The <dfn id=tostring><a href=https://tc39.es/ecma262/#sec-tostring>ToString</a></dfn> abstract operation<li>The <dfn id=touint32><a href=https://tc39.es/ecma262/#sec-touint32>ToUint32</a></dfn> abstract operation<li>The <dfn id=typedarraycreate><a href=https://tc39.es/ecma262/#typedarray-create>TypedArrayCreate</a></dfn> abstract operation<li>The <dfn id=js-abstract-equality><a href=https://tc39.es/ecma262/#sec-abstract-equality-comparison>Abstract Equality Comparison</a></dfn> algorithm<li>The <dfn id=js-strict-equality><a href=https://tc39.es/ecma262/#sec-strict-equality-comparison>Strict Equality Comparison</a></dfn> algorithm<li>The <dfn id=atomics><a href=https://tc39.es/ecma262/#sec-atomics-object><code>Atomics</code></a></dfn> object<li>The <dfn id=date><a href=https://tc39.es/ecma262/#sec-date-objects><code>Date</code></a></dfn> class<li>The <dfn id=regexp><a href=https://tc39.es/ecma262/#sec-regexp-regular-expression-objects><code>RegExp</code></a></dfn> class<li>The <dfn id=sharedarraybuffer><a href=https://tc39.es/ecma262/#sec-sharedarraybuffer-objects><code>SharedArrayBuffer</code></a></dfn> class<li>The <dfn id=typeerror><a href=https://tc39.es/ecma262/#sec-native-error-types-used-in-this-standard-typeerror><code>TypeError</code></a></dfn> class<li>The <dfn id=js-rangeerror><a href=https://tc39.es/ecma262/#sec-native-error-types-used-in-this-standard-rangeerror><code>RangeError</code></a></dfn> class<li>The <dfn id=eval()><a href=https://tc39.es/ecma262/#sec-eval-x><code>eval()</code></a></dfn> function<li>The <dfn id=ishtmldda><a href=https://tc39.es/ecma262/#sec-IsHTMLDDA-internal-slot>[[IsHTMLDDA]]</a></dfn> internal slot<li><dfn id=import()><a href=https://tc39.es/ecma262/#sec-import-calls><code>import()</code></a></dfn><li><dfn id=import.meta><a href=https://tc39.es/ecma262/#sec-meta-properties><code>import.meta</code></a></dfn><li>The <dfn id=js-hostgetimportmetaproperties><a href=https://tc39.es/ecma262/#sec-hostgetimportmetaproperties>HostGetImportMetaProperties</a></dfn> abstract operation<li>The <dfn id=js-typeof><a href=https://tc39.es/ecma262/#sec-typeof-operator><code>typeof</code></a></dfn> operator<li>The <dfn id=delete><a href=https://tc39.es/ecma262/#sec-delete-operator><code>delete</code></a></dfn> operator<li><dfn id=the-typedarray-constructors><a href=https://tc39.es/ecma262/#table-49>The <var>TypedArray</var> Constructors</a></dfn> table</ul>

    <p>Users agents that support JavaScript must also implement <cite>ECMAScript
    Internationalization API</cite>. <a href=#refsJSINTL>[JSINTL]</a></p>
   <dt>WebAssembly<dd>
    <p>The following term is defined in <cite>WebAssembly JavaScript Interface</cite>: <a href=#refsWASMJS>[WASMJS]</a></p>

    <ul class=brief><li><dfn id=webassembly.module><a href=https://webassembly.github.io/spec/js-api/#module><code>WebAssembly.Module</code></a></dfn></ul>
   <dt>DOM<dd>
    <p>The Document Object Model (DOM) is a representation — a model — of a document and
    its content. The DOM is not just an API; the conformance criteria of HTML implementations are
    defined, in this specification, in terms of operations on the DOM. <a href=#refsDOM>[DOM]</a></p>

    <p>Implementations must support DOM and the events defined in UI Events, because this
    specification is defined in terms of the DOM, and some of the features are defined as extensions
    to the DOM interfaces. <a href=#refsDOM>[DOM]</a> <a href=#refsUIEVENTS>[UIEVENTS]</a></p>

    <p>In particular, the following features are defined in <cite>DOM</cite>: <a href=#refsDOM>[DOM]</a></p>

    <ul class=brief><li><dfn id=attr><a href=https://dom.spec.whatwg.org/#interface-attr><code>Attr</code></a></dfn> interface<li><dfn id=comment-2><a href=https://dom.spec.whatwg.org/#interface-comment><code>Comment</code></a></dfn> interface<li><dfn id=domimplementation><a href=https://dom.spec.whatwg.org/#interface-domimplementation><code>DOMImplementation</code></a></dfn> interface<li><dfn id=dom-document><a href=https://dom.spec.whatwg.org/#interface-document><code>Document</code></a></dfn> interface<li><dfn id=dom-documentorshadowroot><a href=https://dom.spec.whatwg.org/#documentorshadowroot><code>DocumentOrShadowRoot</code></a></dfn> interface<li><dfn id=documentfragment><a href=https://dom.spec.whatwg.org/#interface-documentfragment><code>DocumentFragment</code></a></dfn> interface<li><dfn id=documenttype><a href=https://dom.spec.whatwg.org/#interface-documenttype><code>DocumentType</code></a></dfn> interface<li><dfn id=childnode><a href=https://dom.spec.whatwg.org/#interface-childnode><code>ChildNode</code></a></dfn> interface<li><dfn id=element><a href=https://dom.spec.whatwg.org/#interface-element><code>Element</code></a></dfn> interface<li><dfn id=dom-element-attachshadow><a href=https://dom.spec.whatwg.org/#dom-element-attachshadow><code>attachShadow()</code></a></dfn> method.<li>An element's <dfn id=concept-element-shadow-root><a href=https://dom.spec.whatwg.org/#concept-element-shadow-root>shadow root</a></dfn><li>The <dfn id=dom-retarget><a href=https://dom.spec.whatwg.org/#retarget>retargeting algorithm</a></dfn><li><dfn id=node><a href=https://dom.spec.whatwg.org/#interface-node><code>Node</code></a></dfn> interface<li><dfn id=nodelist><a href=https://dom.spec.whatwg.org/#interface-nodelist><code>NodeList</code></a></dfn> interface<li><dfn id=processinginstruction><a href=https://dom.spec.whatwg.org/#interface-processinginstruction><code>ProcessingInstruction</code></a></dfn> interface<li><dfn id=shadowroot><a href=https://dom.spec.whatwg.org/#interface-shadowroot><code>ShadowRoot</code></a></dfn> interface<li><dfn id=text><a href=https://dom.spec.whatwg.org/#interface-text><code>Text</code></a></dfn> interface<li><dfn id=node-document><a href=https://dom.spec.whatwg.org/#concept-node-document>node document</a></dfn> concept<li><dfn id=concept-document-type><a href=https://dom.spec.whatwg.org/#concept-document-type>document type</a></dfn> concept<li><dfn id=concept-documentfragment-host><a href=https://dom.spec.whatwg.org/#concept-documentfragment-host>host</a></dfn> concept<li>The <dfn id=shadow-root><a href=https://dom.spec.whatwg.org/#concept-shadow-root>shadow root</a></dfn> concept, and its <dfn id=delegates-focus><a href=https://dom.spec.whatwg.org/#shadowroot-delegates-focus>delegates focus</a></dfn> and <dfn id=available-to-element-internals><a href=https://dom.spec.whatwg.org/#shadowroot-available-to-element-internals>available to element internals</a></dfn>.<li>The <dfn id=shadow-host><a href=https://dom.spec.whatwg.org/#element-shadow-host>shadow host</a></dfn> concept<li><dfn id=htmlcollection><a href=https://dom.spec.whatwg.org/#interface-htmlcollection><code>HTMLCollection</code></a></dfn> interface, its
         <dfn id=dom-htmlcollection-length><a href=https://dom.spec.whatwg.org/#dom-htmlcollection-length><code>length</code></a></dfn> attribute, and its
         <dfn id=dom-htmlcollection-item><a href=https://dom.spec.whatwg.org/#dom-htmlcollection-item><code>item()</code></a></dfn> and
         <dfn id=dom-htmlcollection-nameditem><a href=https://dom.spec.whatwg.org/#dom-htmlcollection-nameditem><code>namedItem()</code></a></dfn> methods<li>The terms <dfn id=concept-collection><a href=https://dom.spec.whatwg.org/#concept-collection>collection</a></dfn> and <dfn id=represented-by-the-collection><a href=https://dom.spec.whatwg.org/#represented-by-the-collection>represented by the collection</a></dfn><li><dfn id=domtokenlist><a href=https://dom.spec.whatwg.org/#interface-domtokenlist><code>DOMTokenList</code></a></dfn> interface, and its
         <dfn id=dom-domtokenlist-value><a href=https://dom.spec.whatwg.org/#dom-domtokenlist-value><code>value</code></a></dfn> attribute<li><dfn id=dom-domimplementation-createdocument><a href=https://dom.spec.whatwg.org/#dom-domimplementation-createdocument><code>createDocument()</code></a></dfn> method<li><dfn id=dom-domimplementation-createhtmldocument><a href=https://dom.spec.whatwg.org/#dom-domimplementation-createhtmldocument><code>createHTMLDocument()</code></a></dfn> method<li><dfn id=dom-document-createelement><a href=https://dom.spec.whatwg.org/#dom-document-createelement><code>createElement()</code></a></dfn> method<li><dfn id=dom-document-createelementns><a href=https://dom.spec.whatwg.org/#dom-document-createelementns><code>createElementNS()</code></a></dfn> method<li><dfn id=dom-document-getelementbyid><a href=https://dom.spec.whatwg.org/#dom-nonelementparentnode-getelementbyid><code>getElementById()</code></a></dfn> method<li><dfn id=dom-document-getelementsbyclassname><a href=https://dom.spec.whatwg.org/#dom-document-getelementsbyclassname><code>getElementsByClassName()</code></a></dfn> method<li><dfn id=dom-node-appendchild><a href=https://dom.spec.whatwg.org/#dom-node-appendchild><code>appendChild()</code></a></dfn> method<li><dfn id=dom-node-clonenode><a href=https://dom.spec.whatwg.org/#dom-node-clonenode><code>cloneNode()</code></a></dfn> method<li><dfn id=dom-document-importnode><a href=https://dom.spec.whatwg.org/#dom-document-importnode><code>importNode()</code></a></dfn> method<li><dfn id=dom-event-preventdefault><a href=https://dom.spec.whatwg.org/#dom-event-preventdefault><code>preventDefault()</code></a></dfn> method<li><dfn id=dom-element-id><a href=https://dom.spec.whatwg.org/#dom-element-id><code>id</code></a></dfn> attribute<li><dfn id=dom-element-setattribute><a href=https://dom.spec.whatwg.org/#dom-element-setAttribute><code>setAttribute()</code></a></dfn> method<li><dfn id=textcontent><a href=https://dom.spec.whatwg.org/#dom-node-textcontent><code>textContent</code></a></dfn> attribute<li>The <dfn id=tree><a href=https://dom.spec.whatwg.org/#concept-tree>tree</a></dfn>, <dfn id=shadow-tree><a href=https://dom.spec.whatwg.org/#concept-shadow-tree>shadow tree</a></dfn>, and <dfn id=node-tree><a href=https://dom.spec.whatwg.org/#concept-node-tree>node tree</a></dfn> concepts<li>The <dfn id=tree-order><a href=https://dom.spec.whatwg.org/#concept-tree-order>tree order</a></dfn> and <dfn id=shadow-including-tree-order><a href=https://dom.spec.whatwg.org/#concept-shadow-including-tree-order>shadow-including tree order</a></dfn> concepts<li>The <dfn id=concept-tree-child><a href=https://dom.spec.whatwg.org/#concept-tree-child>child</a></dfn> concept<li>The <dfn id=root><a href=https://dom.spec.whatwg.org/#concept-tree-root>root</a></dfn> and <dfn id=shadow-including-root><a href=https://dom.spec.whatwg.org/#concept-shadow-including-root>shadow-including root</a></dfn> concepts<li>The <dfn id=inclusive-ancestor><a href=https://dom.spec.whatwg.org/#concept-tree-inclusive-ancestor>inclusive ancestor</a></dfn>,
             <dfn id=shadow-including-descendant><a href=https://dom.spec.whatwg.org/#concept-shadow-including-descendant>shadow-including descendant</a></dfn>,
             <dfn id=shadow-including-inclusive-descendant><a href=https://dom.spec.whatwg.org/#concept-shadow-including-inclusive-descendant>shadow-including inclusive descendant</a></dfn>, and
             <dfn id=shadow-including-inclusive-ancestor><a href=https://dom.spec.whatwg.org/#concept-shadow-including-inclusive-ancestor>shadow-including inclusive ancestor</a></dfn> concepts<li>The <dfn id=first-child><a href=https://dom.spec.whatwg.org/#concept-tree-first-child>first child</a></dfn> and <dfn id=next-sibling><a href=https://dom.spec.whatwg.org/#concept-tree-next-sibling>next sibling</a></dfn> concepts<li>The <dfn id=document-element><a href=https://dom.spec.whatwg.org/#document-element>document element</a></dfn> concept<li>The <dfn id=in-a-document-tree><a href=https://dom.spec.whatwg.org/#in-a-document-tree>in a document tree</a></dfn>, <dfn id=in-a-document><a href=https://dom.spec.whatwg.org/#in-a-document>in a document</a></dfn> (legacy), and <dfn id=connected><a href=https://dom.spec.whatwg.org/#connected>connected</a></dfn> concepts<li>The <dfn id=concept-slot><a href=https://dom.spec.whatwg.org/#concept-slot>slot</a></dfn> concept, and its <dfn id=slot-name><a href=https://dom.spec.whatwg.org/#slot-name>name</a></dfn> and <dfn id=assigned-nodes><a href=https://dom.spec.whatwg.org/#slot-assigned-nodes>assigned nodes</a></dfn><li>The <dfn id=assigned-slot><a href=https://dom.spec.whatwg.org/#slotable-assigned-slot>assigned slot</a></dfn> concept.
     <li>The <dfn id=finding-flattened-slottables><a href=https://dom.spec.whatwg.org/#find-flattened-slotables>find flattened slottables</a></dfn> algorithm<li>The <dfn id=assign-a-slot><a href=https://dom.spec.whatwg.org/#assign-a-slot>assign a slot</a></dfn> algorithm<li>The <dfn id=pre-insert><a href=https://dom.spec.whatwg.org/#concept-node-pre-insert>pre-insert</a></dfn>, <dfn id=concept-node-insert><a href=https://dom.spec.whatwg.org/#concept-node-insert>insert</a></dfn>, <dfn id=concept-node-append><a href=https://dom.spec.whatwg.org/#concept-node-append>append</a></dfn>, <dfn id=concept-node-replace><a href=https://dom.spec.whatwg.org/#concept-node-replace>replace</a></dfn>, <dfn id=concept-node-replace-all><a href=https://dom.spec.whatwg.org/#concept-node-replace-all>replace all</a></dfn>, <dfn id=string-replace-all><a href=https://dom.spec.whatwg.org/#string-replace-all>string replace all</a></dfn>, <dfn id=concept-node-remove><a href=https://dom.spec.whatwg.org/#concept-node-remove>remove</a></dfn>, and <dfn id=concept-node-adopt><a href=https://dom.spec.whatwg.org/#concept-node-adopt>adopt</a></dfn> algorithms for nodes<li>The <dfn id=concept-node-insert-ext><a href=https://dom.spec.whatwg.org/#concept-node-insert-ext>insertion steps</a></dfn>,
             <dfn id=concept-node-remove-ext><a href=https://dom.spec.whatwg.org/#concept-node-remove-ext>removing steps</a></dfn>,
             <dfn id=concept-node-adopt-ext><a href=https://dom.spec.whatwg.org/#concept-node-adopt-ext>adopting steps</a></dfn>, and
             <dfn id=children-changed-steps><a href=https://dom.spec.whatwg.org/#concept-node-children-changed-ext>children changed steps</a></dfn> hooks for elements<li>The <dfn id=concept-element-attributes-change><a href=https://dom.spec.whatwg.org/#concept-element-attributes-change>change</a></dfn>, <dfn id=concept-element-attributes-append><a href=https://dom.spec.whatwg.org/#concept-element-attributes-append>append</a></dfn>, <dfn id=concept-element-attributes-remove><a href=https://dom.spec.whatwg.org/#concept-element-attributes-remove>remove</a></dfn>, <dfn id=concept-element-attributes-replace><a href=https://dom.spec.whatwg.org/#concept-element-attributes-replace>replace</a></dfn>, and <dfn id=concept-element-attributes-set-value><a href=https://dom.spec.whatwg.org/#concept-element-attributes-set-value>set value</a></dfn> algorithms for attributes<li>The <dfn id=concept-element-attributes-change-ext><a href=https://dom.spec.whatwg.org/#concept-element-attributes-change-ext>attribute change steps</a></dfn> hook for attributes<li>The <dfn id=attribute-list><a href=https://dom.spec.whatwg.org/#concept-element-attribute>attribute list</a></dfn> concept<li>The <dfn id=concept-cd-data><a href=https://dom.spec.whatwg.org/#concept-cd-data>data</a></dfn> of a text node<li>The <dfn id=child-text-content><a href=https://dom.spec.whatwg.org/#concept-child-text-content>child text content</a></dfn> of a node<li>The <dfn id=descendant-text-content><a href=https://dom.spec.whatwg.org/#concept-descendant-text-content>descendant text content</a></dfn> of a node<li><dfn id=event><a href=https://dom.spec.whatwg.org/#interface-event><code>Event</code></a></dfn> interface<li><dfn id=dom-event-constructor><a href=https://dom.spec.whatwg.org/#concept-event-constructor><code>Event</code> and derived interfaces constructor behavior</a></dfn><li><dfn id=eventtarget><a href=https://dom.spec.whatwg.org/#interface-eventtarget><code>EventTarget</code></a></dfn> interface<li>The <dfn id=activation-behaviour><a href=https://dom.spec.whatwg.org/#eventtarget-activation-behavior>activation behavior</a></dfn> hook<li>The <dfn id=legacy-pre-activation-behavior><a href=https://dom.spec.whatwg.org/#eventtarget-legacy-pre-activation-behavior>legacy-pre-activation behavior</a></dfn> hook<li>The <dfn id=legacy-canceled-activation-behavior><a href=https://dom.spec.whatwg.org/#eventtarget-legacy-canceled-activation-behavior>legacy-canceled-activation behavior</a></dfn> hook<li>The <dfn id=creating-an-event><a href=https://dom.spec.whatwg.org/#concept-event-create>create an event</a></dfn> algorithm<li>The <dfn id=concept-event-fire><a href=https://dom.spec.whatwg.org/#concept-event-fire>fire an event</a></dfn> algorithm<li>The <dfn id=canceled-flag><a href=https://dom.spec.whatwg.org/#canceled-flag>canceled flag</a></dfn><li>The <dfn id=concept-event-dispatch><a href=https://dom.spec.whatwg.org/#concept-event-dispatch>dispatch</a></dfn> algorithm<li><dfn id=eventinit><a href=https://dom.spec.whatwg.org/#dictdef-eventinit><code>EventInit</code></a></dfn> dictionary type<li><dfn id=dom-event-type><a href=https://dom.spec.whatwg.org/#dom-event-type><code>type</code></a></dfn> attribute<li><dfn id=dom-event-target><a href=https://dom.spec.whatwg.org/#dom-event-target><code>target</code></a></dfn> attribute<li><dfn id=dom-event-currenttarget><a href=https://dom.spec.whatwg.org/#dom-event-currenttarget><code>currentTarget</code></a></dfn> attribute<li><dfn id=dom-event-bubbles><a href=https://dom.spec.whatwg.org/#dom-event-bubbles><code>bubbles</code></a></dfn> attribute<li><dfn id=dom-event-cancelable><a href=https://dom.spec.whatwg.org/#dom-event-cancelable><code>cancelable</code></a></dfn> attribute<li><dfn id=dom-event-composed><a href=https://dom.spec.whatwg.org/#dom-event-composed><code>composed</code></a></dfn> attribute<li><dfn id=composed-flag><a href=https://dom.spec.whatwg.org/#composed-flag>composed flag</a></dfn><li><dfn id=dom-event-istrusted><a href=https://dom.spec.whatwg.org/#dom-event-istrusted><code>isTrusted</code></a></dfn> attribute<li><dfn id=dom-event-initevent><a href=https://dom.spec.whatwg.org/#dom-event-initevent><code>initEvent()</code></a></dfn> method<li><dfn id=add-an-event-listener><a href=https://dom.spec.whatwg.org/#add-an-event-listener>add an event listener</a></dfn><li><dfn id=dom-eventtarget-addeventlistener><a href=https://dom.spec.whatwg.org/#dom-eventtarget-addeventlistener><code>addEventListener()</code></a></dfn> method<li>The <dfn id=remove-an-event-listener><a href=https://dom.spec.whatwg.org/#remove-an-event-listener>remove an event listener</a></dfn> and
             <dfn id=remove-all-event-listeners><a href=https://dom.spec.whatwg.org/#remove-all-event-listeners>remove all event listeners</a></dfn> algorithms<li><dfn id=dom-eventlistener><a href=https://dom.spec.whatwg.org/#callbackdef-eventlistener><code>EventListener</code></a></dfn> callback interface<li>The <dfn id=concept-event-type><a href=https://dom.spec.whatwg.org/#dom-event-type>type</a></dfn> of an event<li>An <dfn id=event-listener><a href=https://dom.spec.whatwg.org/#concept-event-listener>event listener</a></dfn> and its
            <dfn id=event-listener-type><a href=https://dom.spec.whatwg.org/#event-listener-type>type</a></dfn> and
            <dfn id=event-listener-callback><a href=https://dom.spec.whatwg.org/#event-listener-callback>callback</a></dfn><li>The <dfn id="document's-character-encoding"><a href=https://dom.spec.whatwg.org/#concept-document-encoding>encoding</a></dfn> (herein the <i>character encoding</i>),
             <dfn id=concept-document-mode><a href=https://dom.spec.whatwg.org/#concept-document-mode>mode</a></dfn>, and
             <dfn id=concept-document-content-type><a href=https://dom.spec.whatwg.org/#concept-document-content-type>content type</a></dfn> of a <code id=dependencies:document><a href=#document>Document</a></code><li>The distinction between <dfn id=xml-documents><a href=https://dom.spec.whatwg.org/#xml-document>XML documents</a></dfn> and
                                 <dfn id=html-documents><a href=https://dom.spec.whatwg.org/#html-document>HTML documents</a></dfn><li>The terms <dfn id=quirks-mode><a href=https://dom.spec.whatwg.org/#concept-document-quirks>quirks mode</a></dfn>,
                   <dfn id=limited-quirks-mode><a href=https://dom.spec.whatwg.org/#concept-document-limited-quirks>limited-quirks mode</a></dfn>, and
                   <dfn id=no-quirks-mode><a href=https://dom.spec.whatwg.org/#concept-document-no-quirks>no-quirks mode</a></dfn><li>The algorithm to <dfn id=concept-node-clone><a href=https://dom.spec.whatwg.org/#concept-node-clone>clone</a></dfn> a <code id=dependencies:node><a data-x-internal=node href=https://dom.spec.whatwg.org/#interface-node>Node</a></code>, and the concept of
                          <dfn id=concept-node-clone-ext><a href=https://dom.spec.whatwg.org/#concept-node-clone-ext>cloning steps</a></dfn> used by that algorithm<li>The concept of <dfn id=base-url-change-steps>base URL change steps</dfn> and the definition of what happens when an element is <dfn id=affected-by-a-base-url-change>affected by a base URL change</dfn><li>The concept of an element's <dfn id=concept-id><a href=https://dom.spec.whatwg.org/#concept-id>unique identifier (ID)</a></dfn><li>The concept of an element's <dfn id=concept-class><a href=https://dom.spec.whatwg.org/#concept-class>classes</a></dfn><li>The term <dfn id=concept-supported-tokens><a href=https://dom.spec.whatwg.org/#concept-supported-tokens>supported tokens</a></dfn><li>The concept of a DOM <dfn id=concept-range><a href=https://dom.spec.whatwg.org/#concept-range>range</a></dfn>, and the terms
                              <dfn id=concept-range-start><a href=https://dom.spec.whatwg.org/#concept-range-start>start</a></dfn>,
                              <dfn id=concept-range-end><a href=https://dom.spec.whatwg.org/#concept-range-end>end</a></dfn>, and
                              <dfn id=concept-range-bp><a href=https://dom.spec.whatwg.org/#concept-range-bp>boundary point</a></dfn> as applied to ranges.<li>The <dfn id=create-an-element><a href=https://dom.spec.whatwg.org/#concept-create-element>create an element</a></dfn> algorithm<li>The <dfn id=element-interface><a href=https://dom.spec.whatwg.org/#concept-element-interface>element interface</a></dfn> concept<li>The concepts of <dfn id=custom-element-state><a href=https://dom.spec.whatwg.org/#concept-element-custom-element-state>custom element state</a></dfn>, and of
                         <dfn id=concept-element-defined><a href=https://dom.spec.whatwg.org/#concept-element-defined>defined</a></dfn> and
                         <dfn id=concept-element-custom><a href=https://dom.spec.whatwg.org/#concept-element-custom>custom</a></dfn> elements<li>An element's <dfn id=concept-element-namespace><a href=https://dom.spec.whatwg.org/#concept-element-namespace>namespace</a></dfn>,
                      <dfn id=concept-element-namespace-prefix><a href=https://dom.spec.whatwg.org/#concept-element-namespace-prefix>namespace prefix</a></dfn>,
                      <dfn id=concept-element-local-name><a href=https://dom.spec.whatwg.org/#concept-element-local-name>local name</a></dfn>,
                      <dfn id=concept-element-custom-element-definition><a href=https://dom.spec.whatwg.org/#concept-element-custom-element-definition>custom element definition</a></dfn>, and
                      <dfn id=concept-element-is-value><a href=https://dom.spec.whatwg.org/#concept-element-is-value><code>is</code> value</a></dfn><li><dfn id=mutationobserver><a href=https://dom.spec.whatwg.org/#mutationobserver><code>MutationObserver</code></a></dfn> interface and <dfn id=mutation-observers><a href=https://dom.spec.whatwg.org/#mutation-observers>mutation observers</a></dfn> in general</ul>

    <p>The following features are defined in <cite>UI Events</cite>: <a href=#refsUIEVENTS>[UIEVENTS]</a></p>

    <ul class=brief><li>The <dfn id=mouseevent><a href=https://w3c.github.io/uievents/#mouseevent><code>MouseEvent</code></a></dfn> interface<li>The <code id=dependencies:mouseevent><a data-x-internal=mouseevent href=https://w3c.github.io/uievents/#mouseevent>MouseEvent</a></code> interface's <dfn id=dom-mouseevent-relatedtarget><a href=https://w3c.github.io/uievents/#dom-mouseevent-relatedtarget><code>relatedTarget</code></a></dfn> attribute<li><dfn id=mouseeventinit><a href=https://w3c.github.io/uievents/#dictdef-mouseeventinit><code>MouseEventInit</code></a></dfn> dictionary type<li>The <dfn id=focusevent><a href=https://w3c.github.io/uievents/#focusevent><code>FocusEvent</code></a></dfn> interface<li>The <code id=dependencies:focusevent><a data-x-internal=focusevent href=https://w3c.github.io/uievents/#focusevent>FocusEvent</a></code> interface's <dfn id=dom-focusevent-relatedtarget><a href=https://w3c.github.io/uievents/#dom-focusevent-relatedtarget><code>relatedTarget</code></a></dfn> attribute<li>The <dfn id=uievent><a href=https://w3c.github.io/uievents/#uievent><code>UIEvent</code></a></dfn> interface<li>The <code id=dependencies:uievent><a data-x-internal=uievent href=https://w3c.github.io/uievents/#uievent>UIEvent</a></code> interface's <dfn id=dom-uievent-view><a href=https://w3c.github.io/uievents/#dom-uievent-view><code>view</code></a></dfn> attribute<li><dfn id=event-auxclick><a href=https://w3c.github.io/uievents/#event-type-auxclick><code>auxclick</code></a></dfn> event<li><dfn id=event-click><a href=https://w3c.github.io/uievents/#event-type-click><code>click</code></a></dfn> event<li><dfn id=event-dblclick><a href=https://w3c.github.io/uievents/#event-type-dblclick><code>dblclick</code></a></dfn> event<li><dfn id=event-mousedown><a href=https://w3c.github.io/uievents/#event-type-mousedown><code>mousedown</code></a></dfn> event<li><dfn id=event-mouseenter><a href=https://w3c.github.io/uievents/#event-type-mouseenter><code>mouseenter</code></a></dfn> event<li><dfn id=event-mouseleave><a href=https://w3c.github.io/uievents/#event-type-mouseleave><code>mouseleave</code></a></dfn> event<li><dfn id=event-mousemove><a href=https://w3c.github.io/uievents/#event-type-mousemove><code>mousemove</code></a></dfn> event<li><dfn id=event-mouseout><a href=https://w3c.github.io/uievents/#event-type-mouseout><code>mouseout</code></a></dfn> event<li><dfn id=event-mouseover><a href=https://w3c.github.io/uievents/#event-type-mouseover><code>mouseover</code></a></dfn> event<li><dfn id=event-mouseup><a href=https://w3c.github.io/uievents/#event-type-mouseup><code>mouseup</code></a></dfn> event<li><dfn id=event-wheel><a href=https://w3c.github.io/uievents/#event-type-wheel><code>wheel</code></a></dfn> event<li><dfn id=event-keydown><a href=https://w3c.github.io/uievents/#event-type-keydown><code>keydown</code></a></dfn> event<li><dfn id=event-keypress><a href=https://w3c.github.io/uievents/#event-type-keypress><code>keypress</code></a></dfn> event<li><dfn id=event-keyup><a href=https://w3c.github.io/uievents/#event-type-keyup><code>keyup</code></a></dfn> event</ul>

    <p>The following features are defined in <cite>Touch Events</cite>: <a href=#refsTOUCH>[TOUCH]</a></p>

    <ul class=brief><li><dfn id=touch><a href=https://w3c.github.io/touch-events/#touch-interface><code>Touch</code></a></dfn> interface<li><dfn id=touch-point><a href=https://w3c.github.io/touch-events/#dfn-touch-point>Touch point</a></dfn> concept<li><dfn id=event-touchend><a href=https://w3c.github.io/touch-events/#event-touchend><code>touchend</code></a></dfn> event</ul>

    <p>The following features are defined in <cite>Pointer Events</cite>: <a href=#refsPOINTEREVENTS>[POINTEREVENTS]</a></p>

    <ul class=brief><li><dfn id=pointerevent><a href=https://w3c.github.io/pointerevents/#pointerevent-interface><code>PointerEvent</code></a></dfn> interface<li><dfn id=event-pointerup><a href=https://w3c.github.io/pointerevents/#the-pointerup-event><code>pointerup</code></a></dfn> event</ul>

    <p>This specification sometimes uses the term <dfn>name</dfn> to refer to the event's
    <a href=https://dom.spec.whatwg.org/#dom-event-type id=dependencies:concept-event-type data-x-internal=concept-event-type>type</a>; as in, "an event named <code>click</code>" or "if the event name is <code>keypress</code>". The terms
    "name" and "type" for events are synonymous.</p>

    <p>The following features are defined in <cite>DOM Parsing and Serialization</cite>: <a href=#refsDOMPARSING>[DOMPARSING]</a></p>

    <ul class=brief><li><dfn id=dom-innerhtml><a href=https://w3c.github.io/DOM-Parsing/#dom-element-innerhtml><code>innerHTML</code></a></dfn><li><dfn id=dom-outerhtml><a href=https://w3c.github.io/DOM-Parsing/#dom-element-outerhtml><code>outerHTML</code></a></dfn></ul>

    <p>The following features are defined in <cite>Selection API</cite>: <a href=#refsSELECTION>[SELECTION]</a></p>

    <ul class=brief><li><dfn id=document-selection><a href=https://w3c.github.io/selection-api/#dfn-selection>selection</a></dfn><li><dfn id=selection><a href=https://w3c.github.io/selection-api/#selection-interface><code>Selection</code></a></dfn></ul>

    <p class=note>User agents are encouraged to implement the features described in
    <cite>execCommand</cite>. <a href=#refsEXECCOMMAND>[EXECCOMMAND]</a></p>

    <p>The following parts of <cite>Fullscreen API</cite> are referenced from this
    specification, in part to define the rendering of <code id=dependencies:the-dialog-element><a href=#the-dialog-element>dialog</a></code> elements, and also to
    define how the Fullscreen API interacts with HTML: <a href=#refsFULLSCREEN>[FULLSCREEN]</a></p>

    <ul class=brief><li><dfn id=top-layer><a href=https://fullscreen.spec.whatwg.org/#top-layer>top layer</a></dfn> (an
         <a href=https://infra.spec.whatwg.org/#ordered-set id=dependencies:set data-x-internal=set>ordered set</a>) and its
         <dfn id=top-layer-add><a href=https://fullscreen.spec.whatwg.org/#top-layer-add>add</a></dfn>
         operation<li><dfn id=dom-element-requestfullscreen><a href=https://fullscreen.spec.whatwg.org/#dom-element-requestfullscreen><code>requestFullscreen()</code></a></dfn><li><dfn id=run-the-fullscreen-steps><a href=https://fullscreen.spec.whatwg.org/#run-the-fullscreen-steps>run the fullscreen steps</a></dfn></ul>

    <p><cite>High Resolution Time</cite> provides the <dfn id=current-high-resolution-time><a href=https://w3c.github.io/hr-time/#dfn-current-high-resolution-time>current high
    resolution time</a></dfn> and the <dfn id=domhighrestimestamp><a href=https://w3c.github.io/hr-time/#dom-domhighrestimestamp><code>DOMHighResTimeStamp</code></a></dfn>
    typedef. <a href=#refsHRT>[HRT]</a></p>
   <dt>File API<dd>
    <p>This specification uses the following features defined in <cite>File API</cite>: <a href=#refsFILEAPI>[FILEAPI]</a></p>

    <ul class=brief><li>The <dfn id=blob><a href=https://w3c.github.io/FileAPI/#dfn-Blob><code>Blob</code></a></dfn> interface and its
             <dfn id=dom-blob-type><a href=https://w3c.github.io/FileAPI/#dfn-type><code>type</code></a></dfn> attribute<li>The <dfn id=file><a href=https://w3c.github.io/FileAPI/#dfn-file><code>File</code></a></dfn> interface and its
             <dfn id=dom-file-name><a href=https://w3c.github.io/FileAPI/#dfn-name><code>name</code></a></dfn> and
             <dfn id=dom-file-lastmodified><a href=https://w3c.github.io/FileAPI/#dfn-lastModified><code>lastModified</code></a></dfn> attributes<li>The <dfn id=filelist><a href=https://w3c.github.io/FileAPI/#filelist-section><code>FileList</code></a></dfn> interface<li>The concept of a <code id=dependencies:blob><a data-x-internal=blob href=https://w3c.github.io/FileAPI/#dfn-Blob>Blob</a></code>'s <dfn id=snapshot-state><a href=https://w3c.github.io/FileAPI/#snapshot-state>snapshot state</a></dfn><li>The concept of <dfn id=file-error-read>read errors</dfn><li><dfn id=blob-url-store><a href=https://w3c.github.io/FileAPI/#BlobURLStore>Blob URL Store</a></dfn></ul>
   <dt>Indexed Database API<dd><p>This specification uses <dfn id=cleanup-indexed-database-transactions><a href=https://w3c.github.io/IndexedDB/#cleanup-indexed-database-transactions>cleanup
   Indexed Database transactions</a></dfn> defined by <cite>Indexed Database API</cite>. <a href=#refsINDEXEDDB>[INDEXEDDB]</a></p>



   <dt>Media Source Extensions<dd>
    <p>The following terms are defined in <cite>Media Source Extensions</cite>: <a href=#refsMEDIASOURCE>[MEDIASOURCE]</a></p>

    <ul class=brief><li><dfn id=mediasource><a href=https://w3c.github.io/media-source/#idl-def-mediasource><code>MediaSource</code></a></dfn> interface<li><dfn id=detaching-from-a-media-element><a href=https://w3c.github.io/media-source/#mediasource-detach>detaching from a media element</a></dfn></ul>
   <dt>Media Capture and Streams<dd>
    <p>The following terms are defined in <cite>Media Capture and Streams</cite>: <a href=#refsMEDIASTREAM>[MEDIASTREAM]</a></p>

    <ul class=brief><li><dfn id=mediastream><a href=https://w3c.github.io/mediacapture-main/getusermedia.html#idl-def-mediastream><code>MediaStream</code></a></dfn> interface</ul>
   <dt>Reporting<dd>
    <p>The following terms are defined in <cite>Reporting</cite>: <a href=#refsREPORTING>[REPORTING]</a></p>

    <ul class=brief><li><dfn id=queue-a-report><a href=https://w3c.github.io/reporting/#queue-report>Queue a report</a></dfn><li><dfn id=report-type><a href=https://w3c.github.io/reporting/#report-type>report type</a></dfn><li><dfn id=visible-to-reportingobservers><a href=https://w3c.github.io/reporting/#visible-to-reportingobservers>visible to <code>ReportingObserver</code>s</a></dfn></ul>
   <dt>XMLHttpRequest<dd>
    <p>The following features and terms are defined in <cite>XMLHttpRequest</cite>: <a href=#refsXHR>[XHR]</a></p>

    <ul class=brief><li>The <dfn id=xmlhttprequest><a href=https://xhr.spec.whatwg.org/#xmlhttprequest><code>XMLHttpRequest</code></a></dfn> interface, and its
             <dfn id=dom-xmlhttprequest-responsexml><a href=https://xhr.spec.whatwg.org/#dom-xmlhttprequest-responsexml><code>responseXML</code></a></dfn> attribute<li>The <dfn id=progressevent><a href=https://xhr.spec.whatwg.org/#interface-progressevent><code>ProgressEvent</code></a></dfn> interface, and its
             <dfn id=dom-progressevent-lengthcomputable><a href=https://xhr.spec.whatwg.org/#dom-progressevent-lengthcomputable><code>lengthComputable</code></a></dfn>,
             <dfn id=dom-progressevent-loaded><a href=https://xhr.spec.whatwg.org/#dom-progressevent-loaded><code>loaded</code></a></dfn>, and
             <dfn id=dom-progressevent-total><a href=https://xhr.spec.whatwg.org/#dom-progressevent-total><code>total</code></a></dfn> attributes<li>The <dfn id=formdata><a href=https://xhr.spec.whatwg.org/#formdata><code>FormData</code></a></dfn> interface<li>The <dfn id=formdataentryvalue><a href=https://xhr.spec.whatwg.org/#formdataentryvalue><code>FormDataEntryValue</code></a></dfn> type<li><dfn id=formdata-entry><a href=https://xhr.spec.whatwg.org/#concept-formdata-entry>entry</a></dfn><li><dfn id=create-an-entry><a href=https://xhr.spec.whatwg.org/#create-an-entry>create an entry</a></dfn></ul>
   <dt>Battery Status<dd>
    <p>The following features are defined in <cite>Battery Status API</cite>: <a href=#refsBATTERY>[BATTERY]</a></p>

    <ul class=brief><li><dfn id=dom-navigator-getbattery><a href=https://w3c.github.io/battery/#widl-Navigator-getBattery-Promise-BatteryManager><code>getBattery()</code></a></dfn> method</ul>
   <dt>Media Queries<dd>
    <p>Implementations must support <cite>Media Queries</cite>. The <dfn id=media-condition><a href=https://drafts.csswg.org/mediaqueries/#typedef-media-condition>&lt;media-condition></a></dfn>
    feature is defined therein. <a href=#refsMQ>[MQ]</a></p>
   <dt>CSS modules<dd>
    <p>While support for CSS as a whole is not required of implementations of this specification
    (though it is encouraged, at least for web browsers), some features are defined in terms of
    specific CSS requirements.</p>

    <p>When this specification requires that something be <dfn id=parse-something-according-to-a-css-grammar><a href=https://drafts.csswg.org/css-syntax/#parse-grammar>parsed according
    to a particular CSS grammar</a></dfn>, the relevant algorithm in <cite>CSS Syntax</cite> must be followed,
    including error handling rules. <a href=#refsCSSSYNTAX>[CSSSYNTAX]</a></p>

    <p class=example>For example, user agents are required to close all open constructs upon
    finding the end of a style sheet unexpectedly. Thus, when parsing the string "<code>rgb(0,0,0</code>" (with a missing close-parenthesis) for a color value, the close
    parenthesis is implied by this error handling rule, and a value is obtained (the color 'black').
    However, the similar construct "<code>rgb(0,0,</code>" (with both a missing
    parenthesis and a missing "blue" value) cannot be parsed, as closing the open construct does not
    result in a viable value.</p>

    <p>To <dfn id=parsed-as-a-css-color-value>parse a CSS &lt;color> value</dfn>, given a string
    <var>input</var> with an optional element <var>element</var>, run these steps:</p>

    <ol><li><p>Let <var>color</var> be the result of <a href=https://drafts.csswg.org/css-syntax/#parse-grammar id=dependencies:parse-something-according-to-a-css-grammar data-x-internal=parse-something-according-to-a-css-grammar>parsing</a> <var>input</var> as a CSS <a id=dependencies:color href=https://drafts.csswg.org/css-color/#typedef-color data-x-internal=color>&lt;color></a>. <a href=#refsCSSCOLOR>[CSSCOLOR]</a><li><p>If <var>color</var> is failure, then return failure.<li>
      <p>If <var>color</var> is <a id=dependencies:currentcolor href=https://drafts.csswg.org/css-color/#valdef-color-currentcolor data-x-internal=currentcolor>'currentcolor'</a>, then:</p>

      <ol><li><p>If <var>element</var> is not given, then set <var>color</var> to <a id=dependencies:opaque-black href=https://drafts.csswg.org/css-color/#opaque-black data-x-internal=opaque-black>opaque
       black</a>.<li><p>Otherwise, set <var>color</var> to the computed value of the <a id="dependencies:'color'" href=https://drafts.csswg.org/css-color/#the-color-property data-x-internal="'color'">'color'</a>
       property of <var>element</var>.</ol>
     <li><p>Return <var>color</var>.</ol>

    <p>The following terms and features are defined in <cite>Cascading Style Sheets</cite>
    (<cite>CSS</cite>): <a href=#refsCSS>[CSS]</a></p>

    <ul class=brief><li><dfn id=viewport><a href=https://drafts.csswg.org/css2/#viewport>viewport</a></dfn><li><dfn id=line-box><a href=https://drafts.csswg.org/css2/#line-box>line box</a></dfn><li><dfn id=out-of-flow><a href=https://drafts.csswg.org/css2/#out-of-flow>out-of-flow</a></dfn><li><dfn id=in-flow><a href=https://drafts.csswg.org/css2/#in-flow>in-flow</a></dfn><li><dfn id=content-area><a href=https://drafts.csswg.org/css2/#box-content-area>content area</a></dfn><li><dfn id=content-box><a href=https://drafts.csswg.org/css2/#content-box>content box</a></dfn><li><dfn id=border-box><a href=https://drafts.csswg.org/css2/#border-box>border box</a></dfn><li><dfn id=margin-box><a href=https://drafts.csswg.org/css2/#margin-box>margin box</a></dfn><li><dfn id=border-edge><a href=https://drafts.csswg.org/css2/#border-edge>border edge</a></dfn><li><dfn id=margin-edge><a href=https://drafts.csswg.org/css2/#margin-edge>margin edge</a></dfn><li><dfn id=collapsing-margins><a href=https://drafts.csswg.org/css2/#collapsing-margins>collapsing margins</a></dfn><li><dfn id=containing-block><a href=https://drafts.csswg.org/css2/#containing-block-details>containing block</a></dfn><li><dfn id=inline-box><a href=https://drafts.csswg.org/css2/#inline-box>inline box</a></dfn><li><dfn id=block-box><a href=https://drafts.csswg.org/css2/#block-boxes%E2%91%A0>block box</a></dfn><li>The <dfn id="'margin-top'"><a href=https://drafts.csswg.org/css2/#propdef-margin-top>'margin-top'</a></dfn>,
              <dfn id="'margin-bottom'"><a href=https://drafts.csswg.org/css2/#propdef-margin-bottom>'margin-bottom'</a></dfn>,
              <dfn id="'margin-left'"><a href=https://drafts.csswg.org/css2/#propdef-margin-left>'margin-left'</a></dfn>, and
              <dfn id="'margin-right'"><a href=https://drafts.csswg.org/css2/#propdef-margin-right>'margin-right'</a></dfn> properties<li>The <dfn id="'padding-top'"><a href=https://drafts.csswg.org/css2/#propdef-padding-top>'padding-top'</a></dfn>,
              <dfn id="'padding-bottom'"><a href=https://drafts.csswg.org/css2/#propdef-padding-bottom>'padding-bottom'</a></dfn>,
              <dfn id="'padding-left'"><a href=https://drafts.csswg.org/css2/#propdef-padding-left>'padding-left'</a></dfn>, and
              <dfn id="'padding-right'"><a href=https://drafts.csswg.org/css2/#propdef-padding-right>'padding-right'</a></dfn> properties<li>The <dfn id="'top'"><a href=https://drafts.csswg.org/css2/#propdef-top>'top'</a></dfn>,
              <dfn id="'bottom'"><a href=https://drafts.csswg.org/css2/#propdef-bottom>'bottom'</a></dfn>,
              <dfn id="'left'"><a href=https://drafts.csswg.org/css2/#propdef-left>'left'</a></dfn>, and
              <dfn id="'right'"><a href=https://drafts.csswg.org/css2/#propdef-right>'right'</a></dfn> properties<li>The <dfn id="'float'"><a href=https://drafts.csswg.org/css2/#float-position>'float'</a></dfn> property<li>The <dfn id="'clear'"><a href=https://drafts.csswg.org/css2/#flow-control>'clear'</a></dfn> property<li>The <dfn id="'width'"><a href=https://drafts.csswg.org/css2/#the-width-property>'width'</a></dfn> property<li>The <dfn id="'height'"><a href=https://drafts.csswg.org/css2/#the-height-property>'height'</a></dfn> property<li>The <dfn id="'max-width'"><a href=https://drafts.csswg.org/css2/#min-max-widths>'max-width'</a></dfn> property<li>The <dfn id="'max-height'"><a href=https://drafts.csswg.org/css2/#min-max-heights>'max-height'</a></dfn> property<li>The <dfn id="'line-height'"><a href=https://drafts.csswg.org/css2/#propdef-line-height>'line-height'</a></dfn> property<li>The <dfn id="'vertical-align'"><a href=https://drafts.csswg.org/css2/#propdef-vertical-align>'vertical-align'</a></dfn> property<li>The <dfn id="'content'"><a href=https://drafts.csswg.org/css2/#content%E2%91%A0>'content'</a></dfn> property<li>The <dfn id="'inline-block'"><a href=https://drafts.csswg.org/css2/#value-def-inline-block>'inline-block'</a></dfn> value of the <a id="dependencies:'display'" href=https://drafts.csswg.org/css2/#display-prop data-x-internal="'display'">'display'</a> property<li>The <dfn id="'visibility'"><a href=https://drafts.csswg.org/css2/#propdef-visibility>'visibility'</a></dfn> property</ul>

    <p><cite>CSS</cite> also defines the following border properties: <a href=#refsCSS>[CSS]</a></p>

    <table><caption>Border properties
     </caption><tr><td>
       <th>Top
       <th>Bottom
       <th>Left
       <th>Right
      <tr><th>Width
       <td><dfn id="'border-top-width'"><a href=https://drafts.csswg.org/css2/#propdef-border-top-width>'border-top-width'</a></dfn>
       <td><dfn id="'border-bottom-width'"><a href=https://drafts.csswg.org/css2/#propdef-border-bottom-width>'border-bottom-width'</a></dfn>
       <td><dfn id="'border-left-width'"><a href=https://drafts.csswg.org/css2/#propdef-border-left-width>'border-left-width'</a></dfn>
       <td><dfn id="'border-right-width'"><a href=https://drafts.csswg.org/css2/#propdef-border-right-width>'border-right-width'</a></dfn>
      <tr><th>Style
       <td><dfn id="'border-top-style'"><a href=https://drafts.csswg.org/css2/#propdef-border-top-style>'border-top-style'</a></dfn>
       <td><dfn id="'border-bottom-style'"><a href=https://drafts.csswg.org/css2/#propdef-border-bottom-style>'border-bottom-style'</a></dfn>
       <td><dfn id="'border-left-style'"><a href=https://drafts.csswg.org/css2/#propdef-border-left-style>'border-left-style'</a></dfn>
       <td><dfn id="'border-right-style'"><a href=https://drafts.csswg.org/css2/#propdef-border-right-style>'border-right-style'</a></dfn>
      <tr><th>Color
       <td><dfn id="'border-top-color'"><a href=https://drafts.csswg.org/css2/#propdef-border-top-color>'border-top-color'</a></dfn>
       <td><dfn id="'border-bottom-color'"><a href=https://drafts.csswg.org/css2/#propdef-border-bottom-color>'border-bottom-color'</a></dfn>
       <td><dfn id="'border-left-color'"><a href=https://drafts.csswg.org/css2/#propdef-border-left-color>'border-left-color'</a></dfn>
       <td><dfn id="'border-right-color'"><a href=https://drafts.csswg.org/css2/#propdef-border-right-color>'border-right-color'</a></dfn>
    </table>

    <p>The basic version of the <dfn id="'display'"><a href=https://drafts.csswg.org/css2/#display-prop>'display'</a></dfn> property
    is defined in <cite>CSS</cite>, and the property is extended by other CSS
    modules. <a href=#refsCSS>[CSS]</a> <a href=#refsCSSRUBY>[CSSRUBY]</a> <a href=#refsCSSTABLE>[CSSTABLE]</a></p>

    <p>The following terms and features are defined in <cite>CSS Logical Properties</cite>: <a href=#refsCSSLOGICAL>[CSSLOGICAL]</a></p>

    <ul class=brief><li>The <dfn id="'margin-block-start'"><a href=https://drafts.csswg.org/css-logical/#propdef-margin-block-start>'margin-block-start'</a></dfn>,
              <dfn id="'margin-block-end'"><a href=https://drafts.csswg.org/css-logical/#propdef-margin-block-end>'margin-block-end'</a></dfn>,
              <dfn id="'margin-inline-start'"><a href=https://drafts.csswg.org/css-logical/#propdef-margin-inline-start>'margin-inline-start'</a></dfn>, and
              <dfn id="'margin-inline-end'"><a href=https://drafts.csswg.org/css-logical/#propdef-margin-inline-end>'margin-inline-end'</a></dfn> properties<li>The <dfn id="'padding-block-start'"><a href=https://drafts.csswg.org/css-logical/#propdef-padding-block-start>'padding-block-start'</a></dfn>,
              <dfn id="'padding-block-end'"><a href=https://drafts.csswg.org/css-logical/#propdef-padding-block-end>'padding-block-end'</a></dfn>,
              <dfn id="'padding-inline-start'"><a href=https://drafts.csswg.org/css-logical/#propdef-padding-inline-start>'padding-inline-start'</a></dfn>, and
              <dfn id="'padding-inline-end'"><a href=https://drafts.csswg.org/css-logical/#propdef-padding-inline-end>'padding-inline-end'</a></dfn> properties<li>The <dfn id="'border-block-start-width'"><a href=https://drafts.csswg.org/css-logical/#propdef-border-block-start-width>'border-block-start-width'</a></dfn> property<li>The <dfn id="'block-size'"><a href=https://drafts.csswg.org/css-logical/#propdef-block-size>'block-size'</a></dfn> property<li>The <dfn id="'inline-size'"><a href=https://drafts.csswg.org/css-logical/#propdef-inline-size>'inline-size'</a></dfn> property<li>The <dfn id="'inset-block-start'"><a href=https://drafts.csswg.org/css-logical/#propdef-inset-block-start>'inset-block-start'</a></dfn> property<li>The <dfn id="'inset-block-end'"><a href=https://drafts.csswg.org/css-logical/#propdef-inset-block-end>'inset-block-end'</a></dfn> property</ul>

    <p>The following terms and features are defined in <cite>CSS Color</cite>: <a href=#refsCSSCOLOR>[CSSCOLOR]</a></p>

    <ul class=brief><li><dfn id=named-color><a href=https://drafts.csswg.org/css-color/#named-color>named color</a></dfn><li><dfn id=color><a href=https://drafts.csswg.org/css-color/#typedef-color>&lt;color></a></dfn><li>The <dfn id="'color'"><a href=https://drafts.csswg.org/css-color/#the-color-property>'color'</a></dfn> property<li>The <dfn id=currentcolor><a href=https://drafts.csswg.org/css-color/#valdef-color-currentcolor>'currentcolor'</a></dfn> value<li><dfn id=opaque-black><a href=https://drafts.csswg.org/css-color/#opaque-black>opaque black</a></dfn><li><dfn id=transparent-black><a href=https://drafts.csswg.org/css-color/#transparent-black>transparent black</a></dfn></ul>

    <p>The following terms are defined in <cite>CSS Images</cite>: <a href=#refsCSSIMAGES>[CSSIMAGES]</a></p>

    <ul class=brief><li><dfn id=default-object-size><a href=https://drafts.csswg.org/css-images/#default-object-size>default object size</a></dfn><li><dfn id=intrinsic-dimensions><a href=https://drafts.csswg.org/css-images/#intrinsic-dimensions>intrinsic dimensions</a></dfn><li><dfn id=intrinsic-aspect-ratio><a href=https://drafts.csswg.org/css-images/#intrinsic-aspect-ratio>intrinsic aspect ratio</a></dfn><li><dfn id=intrinsic-height><a href=https://drafts.csswg.org/css-images/#intrinsic-height>intrinsic height</a></dfn><li><dfn id=intrinsic-width><a href=https://drafts.csswg.org/css-images/#intrinsic-width>intrinsic width</a></dfn><li>The <dfn id="'image-orientation'"><a href=https://drafts.csswg.org/css-images-3/#the-image-orientation>'image-orientation'</a></dfn> property<li>The <dfn id="'object-fit'"><a href=https://drafts.csswg.org/css-images/#the-object-fit>'object-fit'</a></dfn> property</ul>

    <p>The term <dfn id=paint-source><a href=https://drafts.csswg.org/css-images-4/#paint-source>paint
    source</a></dfn> is used as defined in <cite>CSS Images Level 4</cite>
    to define the interaction of certain HTML elements with the CSS 'element()'
    function. <a href=#refsCSSIMAGES4>[CSSIMAGES4]</a></p>

    <p>The following features are defined in <cite>CSS Backgrounds and Borders</cite>: <a href=#refsCSSBG>[CSSBG]</a></p>

    <ul class=brief><li>The <dfn id="'background-color'"><a href=https://drafts.csswg.org/css-backgrounds/#the-background-color>'background-color'</a></dfn> property<li>The <dfn id="'background-image'"><a href=https://drafts.csswg.org/css-backgrounds/#the-background-image>'background-image'</a></dfn> property<li>The <dfn id="'border-radius'"><a href=https://drafts.csswg.org/css-backgrounds/#propdef-border-radius>'border-radius'</a></dfn> property</ul>

    <p>The following features are defined in <cite>CSS Box Alignment</cite>: <a href=#refsCSSALIGN>[CSSALIGN]</a></p>

    <ul class=brief><li>The <dfn id="'align-content'"><a href=https://drafts.csswg.org/css-align/#propdef-align-content>'align-content'</a></dfn> property<li>The <dfn id="'align-items'"><a href=https://drafts.csswg.org/css-align/#propdef-align-items>'align-items'</a></dfn> property<li>The <dfn id="'align-self'"><a href=https://drafts.csswg.org/css-align/#propdef-align-self>'align-self'</a></dfn> property<li>The <dfn id="'justify-self'"><a href=https://drafts.csswg.org/css-align/#propdef-justify-self>'justify-self'</a></dfn> property<li>The <dfn id="'justify-content'"><a href=https://drafts.csswg.org/css-align/#propdef-propdef-justify-content>'justify-content'</a></dfn> property<li>The <dfn id="'justify-items'"><a href=https://drafts.csswg.org/css-align/#propdef-propdef-justify-items>'justify-items'</a></dfn> property</ul>

    <p>The following terms and features are defined in <cite>CSS Display</cite>: <a href=#refsCSSDISPLAY>[CSSDISPLAY]</a></p>

    <ul class=brief><li><dfn id=outer-display-type><a href=https://drafts.csswg.org/css-display/#outer-display-type>outer display type</a></dfn><li><dfn id=block-level><a href=https://drafts.csswg.org/css-display/#block-level>block-level</a></dfn><li><dfn id=block-container><a href=https://drafts.csswg.org/css-display/#block-container>block container</a></dfn><li><dfn id=formatting-context><a href=https://drafts.csswg.org/css-display/#formatting-context>formatting context</a></dfn><li><dfn id=block-formatting-context><a href=https://drafts.csswg.org/css-display/#block-formatting-context>block formatting context</a></dfn><li><dfn id=inline-formatting-context><a href=https://drafts.csswg.org/css-display/#inline-formatting-context>inline formatting context</a></dfn><li><dfn id=absolutely-positioned><a href=https://drafts.csswg.org/css-display/#absolutely-positioned>absolutely positioned</a></dfn><li><dfn id=replaced-element><a href=https://drafts.csswg.org/css-display/#replaced-element>replaced element</a></dfn><li><dfn id=css-box><a href=https://drafts.csswg.org/css-display/#css-box>CSS box</a></dfn></ul>

    <p>The following features are defined in <cite>CSS Flexible Box Layout</cite>: <a href=#refsCSSFLEXBOX>[CSSFLEXBOX]</a></p>

    <ul class=brief><li>The <dfn id="'flex-direction'"><a href=https://drafts.csswg.org/css-flexbox/#propdef-flex-direction>'flex-direction'</a></dfn> property<li>The <dfn id="'flex-wrap'"><a href=https://drafts.csswg.org/css-flexbox/#propdef-flex-wrap>'flex-wrap'</a></dfn> property</ul>

    <p>The following terms and features are defined in <cite>CSS Fonts</cite>: <a href=#refsCSSFONTS>[CSSFONTS]</a></p>

    <ul class=brief><li><dfn id=first-available-font><a href=https://drafts.csswg.org/css-fonts/#first-available-font>first available font</a></dfn><li>The <dfn id="'font-family'"><a href=https://drafts.csswg.org/css-fonts/#font-family-prop>'font-family'</a></dfn> property<li>The <dfn id="'font-weight'"><a href=https://drafts.csswg.org/css-fonts/#font-weight-prop>'font-weight'</a></dfn> property<li>The <dfn id="'font-size'"><a href=https://drafts.csswg.org/css-fonts/#font-size-prop>'font-size'</a></dfn> property<li>The <dfn id="'font'"><a href=https://drafts.csswg.org/css-fonts/#font-prop>'font'</a></dfn> property</ul>

    <p>The following features are defined in <cite>CSS Grid Layout</cite>: <a href=#refsCSSGRID>[CSSGRID]</a></p>

    <ul class=brief><li>The <dfn id="'grid-auto-columns'"><a href=https://drafts.csswg.org/css-grid/#propdef-grid-auto-columns>'grid-auto-columns'</a></dfn> property<li>The <dfn id="'grid-auto-flow'"><a href=https://drafts.csswg.org/css-grid/#propdef-grid-auto-flow>'grid-auto-flow'</a></dfn> property<li>The <dfn id="'grid-auto-rows'"><a href=https://drafts.csswg.org/css-grid/#propdef-grid-auto-rows>'grid-auto-rows'</a></dfn> property<li>The <dfn id="'grid-column-gap'"><a href=https://drafts.csswg.org/css-grid/#propdef-grid-column-gap>'grid-column-gap'</a></dfn> property<li>The <dfn id="'grid-row-gap'"><a href=https://drafts.csswg.org/css-grid/#propdef-grid-row-gap>'grid-row-gap'</a></dfn> property<li>The <dfn id="'grid-template-areas'"><a href=https://drafts.csswg.org/css-grid/#propdef-grid-template-areas>'grid-template-areas'</a></dfn> property<li>The <dfn id="'grid-template-columns'"><a href=https://drafts.csswg.org/css-grid/#propdef-grid-template-columns>'grid-template-columns'</a></dfn> property<li>The <dfn id="'grid-template-rows'"><a href=https://drafts.csswg.org/css-grid/#propdef-grid-template-rows>'grid-template-rows'</a></dfn> property</ul>

    <p>The following terms are defined in <cite>CSS Inline Layout</cite>: <a href=#refsCSSINLINE>[CSSINLINE]</a></p>

    <ul class=brief><li><dfn id=alphabetic-baseline><a href=https://drafts.csswg.org/css-inline/#alphabetic-baseline>alphabetic baseline</a></dfn><li><dfn id=ascent-metric><a href=https://drafts.csswg.org/css-inline/#ascent-metric>ascent metric</a></dfn><li><dfn id=descent-metric><a href=https://drafts.csswg.org/css-inline/#descent-metric>descent metric</a></dfn><li><dfn id=hanging-baseline><a href=https://drafts.csswg.org/css-inline/#hanging-baseline>hanging baseline</a></dfn><li><dfn id=ideographic-under-baseline><a href=https://drafts.csswg.org/css-inline/#ideographic-under-baseline>ideographic-under baseline</a></dfn></ul>

    <p>The following terms and features are defined in <cite>CSS Intrinsic &amp; Extrinsic
    Sizing</cite>: <a href=#refsCSSSIZING>[CSSSIZING]</a></p>

    <ul class=brief><li><dfn id=fit-content-inline-size><a href=https://drafts.csswg.org/css-sizing/#fit-content-inline-size>fit-content inline size</a></dfn></ul>

    <p>The <dfn id="'list-style-type'"><a href=https://drafts.csswg.org/css-lists/#propdef-list-style-type>'list-style-type'</a></dfn>
    property is defined in <cite>CSS Lists and Counters</cite>. <a href=#refsCSSLISTS>[CSSLISTS]</a></p>

    <p>The following features are defined in <cite>CSS Overflow</cite>. <a href=#refsCSSOVERFLOW>[CSSOVERFLOW]</a></p>

    <ul class=brief><li>The <dfn id="'overflow'"><a href=https://drafts.csswg.org/css-overflow/#propdef-overflow>'overflow'</a></dfn> property and its <dfn id="'hidden'"><a href=https://drafts.csswg.org/css-overflow/#valdef-overflow-hidden>'hidden'</a></dfn> value<li>The <dfn id="'text-overflow'"><a href=https://drafts.csswg.org/css-overflow/#propdef-text-overflow>'text-overflow'</a></dfn> property</ul>

    <p>The following features are defined in <cite>CSS Positioned Layout</cite>: <a href=#refsCSSPOSITION>[CSSPOSITION]</a></p>

    <ul class=brief><li>The <dfn id="'position'"><a href=https://drafts.csswg.org/css-position/#position-property>'position'</a></dfn> property and its
              <dfn id="'static'"><a href=https://drafts.csswg.org/css-position/#valdef-position-static>'static'</a></dfn> value</ul>

    <p>The following features are defined in <cite>CSS Multi-column Layout</cite>. <a href=#refsCSSMULTICOL>[CSSMULTICOL]</a></p>

    <ul class=brief><li>The <dfn id="'column-count'"><a href=https://drafts.csswg.org/css-multicol/#propdef-column-count>'column-count'</a></dfn> property<li>The <dfn id="'column-fill'"><a href=https://drafts.csswg.org/css-multicol/#propdef-column-fill>'column-fill'</a></dfn> property<li>The <dfn id="'column-gap'"><a href=https://drafts.csswg.org/css-multicol/#propdef-column-gap>'column-gap'</a></dfn> property<li>The <dfn id="'column-rule'"><a href=https://drafts.csswg.org/css-multicol/#propdef-column-rule>'column-rule'</a></dfn> property<li>The <dfn id="'column-width'"><a href=https://drafts.csswg.org/css-multicol/#propdef-column-width>'column-width'</a></dfn> property</ul>

    <p>The <dfn id="'ruby-base'"><a href=https://drafts.csswg.org/css-ruby/#valdef-display-ruby-base>'ruby-base'</a></dfn>
    value of the <a id="dependencies:'display'-2" href=https://drafts.csswg.org/css2/#display-prop data-x-internal="'display'">'display'</a> property is defined in <cite>CSS Ruby Layout</cite>. <a href=#refsCSSRUBY>[CSSRUBY]</a></p>

    <p>The following features are defined in <cite>CSS Table</cite>: <a href=#refsCSSTABLE>[CSSTABLE]</a></p>

    <ul class=brief><li>The <dfn id="'border-spacing'"><a href=https://drafts.csswg.org/css-tables/#propdef-border-spacing>'border-spacing'</a></dfn> property<li>The <dfn id="'border-collapse'"><a href=https://drafts.csswg.org/css-tables/#border-collapse-property>'border-collapse'</a></dfn> property<li>The <dfn id="'table-cell'"><a href=https://drafts.csswg.org/css-tables/#table-cell>'table-cell'</a></dfn>,
              <dfn id="'table-row'"><a href=https://drafts.csswg.org/css-tables/#table-row>'table-row'</a></dfn>,
              <dfn id="'table-caption'"><a href=https://drafts.csswg.org/css-tables/#table-caption>'table-caption'</a></dfn>, and
              <dfn id="'table'"><a href=https://drafts.csswg.org/css-tables/#table>'table'</a></dfn> values of the <a id="dependencies:'display'-3" href=https://drafts.csswg.org/css2/#display-prop data-x-internal="'display'">'display'</a> property</ul>

    <p>The following features are defined in <cite>CSS Text</cite>: <a href=#refsCSSTEXT>[CSSTEXT]</a></p>

    <ul class=brief><li>The <dfn id="'text-transform'"><a href=https://drafts.csswg.org/css-text/#text-transform-property>'text-transform'</a></dfn> property<li>The <dfn id="'white-space'"><a href=https://drafts.csswg.org/css-text/#white-space-property>'white-space'</a></dfn> property<li>The <dfn id="'text-align'"><a href=https://drafts.csswg.org/css-text/#text-align-property>'text-align'</a></dfn> property<li>The <dfn id="'letter-spacing'"><a href=https://drafts.csswg.org/css-text/#letter-spacing-property>'letter-spacing'</a></dfn> property</ul>

    <p>The following features are defined in <cite>CSS Writing Modes</cite>: <a href=#refsCSSWM>[CSSWM]</a></p>

    <ul class=brief><li>The <dfn id="'direction'"><a href=https://drafts.csswg.org/css-writing-modes/#direction>'direction'</a></dfn> property<li>The <dfn id="'unicode-bidi'"><a href=https://drafts.csswg.org/css-writing-modes/#unicode-bidi>'unicode-bidi'</a></dfn> property<li>The <dfn id=block-flow-direction><a href=https://drafts.csswg.org/css-writing-modes/#block-flow-direction>block flow direction</a></dfn>,
              <dfn id=block-size><a href=https://drafts.csswg.org/css-writing-modes/#block-size>block size</a></dfn>,
              <dfn id=inline-size><a href=https://drafts.csswg.org/css-writing-modes/#inline-size>inline size</a></dfn>,
              <dfn id=block-start><a href=https://drafts.csswg.org/css-writing-modes/#block-start>block-start</a></dfn>,
              <dfn id=block-end><a href=https://drafts.csswg.org/css-writing-modes/#block-end>block-end</a></dfn>,
              <dfn id=inline-start><a href=https://drafts.csswg.org/css-writing-modes/#inline-start>inline-start</a></dfn>,
              <dfn id=inline-end><a href=https://drafts.csswg.org/css-writing-modes/#inline-end>inline-end</a></dfn>,
              <dfn id=line-left><a href=https://drafts.csswg.org/css-writing-modes/#line-left>line-left</a></dfn>, and
              <dfn id=line-right><a href=https://drafts.csswg.org/css-writing-modes/#line-right>line-right</a></dfn> concepts</ul>

    <p>The following features are defined in <cite>CSS Basic User Interface</cite>: <a href=#refsCSSUI>[CSSUI]</a></p>

    <ul class=brief><li>The <dfn id="'outline'"><a href=https://drafts.csswg.org/css-ui/#outline>'outline'</a></dfn> property<li>The <dfn id="'cursor'"><a href=https://drafts.csswg.org/css-ui/#cursor>'cursor'</a></dfn> property<li>The <dfn id="'appearance'"><a href=https://drafts.csswg.org/css-ui-4/#appearance-switching>'appearance'</a></dfn> property</ul>

    <p>The algorithm to <dfn id=update-animations-and-send-events><a href=https://drafts.csswg.org/web-animations/#update-animations-and-send-events>update
    animations and send events</a></dfn> is defined in <cite>Web Animations</cite>. <a href=#refsWEBANIMATIONS>[WEBANIMATIONS]</a>.</p>

    <p>Implementations that support scripting must support the CSS Object Model. The following
    features and terms are defined in the CSSOM specifications: <a href=#refsCSSOM>[CSSOM]</a> <a href=#refsCSSOMVIEW>[CSSOMVIEW]</a>

    <ul class=brief><li><dfn id=screen><a href=https://drafts.csswg.org/cssom-view/#the-screen-interface><code>Screen</code></a></dfn> interface<li><dfn id=linkstyle><a href=https://drafts.csswg.org/cssom/#the-linkstyle-interface><code>LinkStyle</code></a></dfn> interface<li><dfn id=cssstyledeclaration><a href=https://drafts.csswg.org/cssom/#the-cssstyledeclaration-interface><code>CSSStyleDeclaration</code></a></dfn> interface<li><dfn id=dom-style><a href=https://drafts.csswg.org/cssom/#dom-elementcssinlinestyle-style><code>style</code></a></dfn> IDL attribute<li><dfn id=dom-cssstyledeclaration-csstext><a href=https://drafts.csswg.org/cssom/#dom-cssstyledeclaration-csstext><code>cssText</code></a></dfn> attribute of <code id=dependencies:cssstyledeclaration><a data-x-internal=cssstyledeclaration href=https://drafts.csswg.org/cssom/#the-cssstyledeclaration-interface>CSSStyleDeclaration</a></code><li><dfn id=stylesheet><a href=https://drafts.csswg.org/cssom/#the-stylesheet-interface><code>StyleSheet</code></a></dfn> interface<li><dfn id=create-a-css-style-sheet><a href=https://drafts.csswg.org/cssom/#create-a-css-style-sheet>create a CSS style sheet</a></dfn><li><dfn id=remove-a-css-style-sheet><a href=https://drafts.csswg.org/cssom/#remove-a-css-style-sheet>remove a CSS style sheet</a></dfn><li><dfn id=associated-css-style-sheet><a href=https://drafts.csswg.org/cssom/#associated-css-style-sheet>associated CSS style sheet</a></dfn><li>
      <dfn id=css-style-sheet><a href=https://drafts.csswg.org/cssom/#css-style-sheet>CSS style sheets</a></dfn> and their
      properties:
      <ul class=brief><li><dfn id=concept-css-style-sheet-type><a href=https://drafts.csswg.org/cssom/#concept-css-style-sheet-type>type</a></dfn><li><dfn id=concept-css-style-sheet-location><a href=https://drafts.csswg.org/cssom/#concept-css-style-sheet-location>location</a></dfn><li><dfn id=concept-css-style-sheet-parent-css-style-sheet><a href=https://drafts.csswg.org/cssom/#concept-css-style-sheet-parent-css-style-sheet>parent CSS style sheet</a></dfn><li><dfn id=concept-css-style-sheet-owner-node><a href=https://drafts.csswg.org/cssom/#concept-css-style-sheet-owner-node>owner node</a></dfn><li><dfn id=concept-css-style-sheet-owner-css-rule><a href=https://drafts.csswg.org/cssom/#concept-css-style-sheet-owner-css-rule>owner CSS rule</a></dfn><li><dfn id=concept-css-style-sheet-media><a href=https://drafts.csswg.org/cssom/#concept-css-style-sheet-media>media</a></dfn><li><dfn id=concept-css-style-sheet-title><a href=https://drafts.csswg.org/cssom/#concept-css-style-sheet-title>title</a></dfn><li><dfn id=concept-css-style-sheet-alternate-flag><a href=https://drafts.csswg.org/cssom/#concept-css-style-sheet-alternate-flag>alternate flag</a></dfn><li><dfn id=concept-css-style-sheet-disabeld-flag><a href=https://drafts.csswg.org/cssom/#concept-css-style-sheet-disabled-flag>disabled flag</a></dfn><li><dfn id=concept-css-style-sheet-css-rules><a href=https://drafts.csswg.org/cssom/#concept-css-style-sheet-css-rules>CSS rules</a></dfn><li><dfn id=concept-css-style-sheet-origin-clean-flag><a href=https://drafts.csswg.org/cssom/#concept-css-style-sheet-origin-clean-flag>origin-clean flag</a></dfn></ul>
     <li><dfn id=css-style-sheet-set><a href=https://drafts.csswg.org/cssom/#css-style-sheet-set>CSS style sheet set</a></dfn><li><dfn id=css-style-sheet-set-name><a href=https://drafts.csswg.org/cssom/#css-style-sheet-set-name>CSS style sheet set name</a></dfn><li><dfn id=preferred-css-style-sheet-set-name><a href=https://drafts.csswg.org/cssom/#preferred-css-style-sheet-set-name>preferred CSS style sheet set name</a></dfn><li><dfn id=change-the-preferred-css-style-sheet-set-name><a href=https://drafts.csswg.org/cssom/#change-the-preferred-css-style-sheet-set-name>change the preferred CSS style sheet set name</a></dfn><li><dfn id=serializing-a-css-value><a href=https://drafts.csswg.org/cssom/#serialize-a-css-value>Serializing a CSS value</a></dfn><li><dfn id=run-the-resize-steps><a href=https://drafts.csswg.org/cssom-view/#run-the-resize-steps>run the resize steps</a></dfn><li><dfn id=run-the-scroll-steps><a href=https://drafts.csswg.org/cssom-view/#run-the-scroll-steps>run the scroll steps</a></dfn><li><dfn id=evaluate-media-queries-and-report-changes><a href=https://drafts.csswg.org/cssom-view/#evaluate-media-queries-and-report-changes>evaluate media queries and report changes</a></dfn><li><dfn id=scroll-an-element-into-view><a href=https://drafts.csswg.org/cssom-view/#scroll-an-element-into-view>Scroll an element into view</a></dfn><li><dfn id=scroll-to-the-beginning-of-the-document><a href=https://drafts.csswg.org/cssom-view/#scroll-to-the-beginning-of-the-document>Scroll to the beginning of the document</a></dfn><li>The <dfn id=event-resize><a href=https://drafts.csswg.org/cssom-view/#eventdef-window-resize><code>resize</code></a></dfn> event<li>The <dfn id=event-scroll><a href=https://drafts.csswg.org/cssom-view/#eventdef-document-scroll><code>scroll</code></a></dfn> event<li><dfn id=set-up-browsing-context-features><a href=https://drafts.csswg.org/cssom-view/#set-up-browsing-context-features>set up browsing context features</a></dfn></ul>

    <p>The following features and terms are defined in <cite>CSS Syntax</cite>: <a href=#refsCSSSYNTAX>[CSSSYNTAX]</a></p>

    <ul class=brief><li><dfn id=conformant-style-sheet><a href=https://drafts.csswg.org/css-syntax/#conform-classes>conformant style sheet</a></dfn><li><dfn id=parse-a-list-of-component-values><a href=https://drafts.csswg.org/css-syntax/#parse-a-list-of-component-values>parse a list of component values</a></dfn><li><dfn id=parse-a-comma-separated-list-of-component-values><a href=https://drafts.csswg.org/css-syntax/#parse-a-comma-separated-list-of-component-values>parse a comma-separated list of component values</a></dfn><li><dfn id=component-value><a href=https://drafts.csswg.org/css-syntax/#component-value>component value</a></dfn><li><dfn id=environment-encoding><a href=https://drafts.csswg.org/css-syntax/#environment-encoding>environment encoding</a></dfn><li><dfn id=whitespace-token><a href=https://drafts.csswg.org/css-syntax/#typedef-whitespace-token>&lt;whitespace-token></a></dfn></ul>

    <p>The following terms are defined in <cite>Selectors</cite>: <a href=#refsSELECTORS>[SELECTORS]</a></p>

    <ul class=brief><li><dfn id=type-selector><a href=https://drafts.csswg.org/selectors/#type-selector>type selector</a></dfn><li><dfn id=attribute-selector><a href=https://drafts.csswg.org/selectors/#attribute-selector>attribute selector</a></dfn><li><dfn id=pseudo-class><a href=https://drafts.csswg.org/selectors/#pseudo-class>pseudo-class</a></dfn></ul>

    <p>The following features are defined in <cite>CSS Values and Units</cite>: <a href=#refsCSSVALUES>[CSSVALUES]</a></p>

    <ul class=brief><li><dfn id=length-2><a href=https://drafts.csswg.org/css-values/#lengths>&lt;length></a></dfn><li>The <dfn id="'em'"><a href=https://drafts.csswg.org/css-values/#em>'em'</a></dfn> unit<li>The <dfn id="'ex'"><a href=https://drafts.csswg.org/css-values/#ex>'ex'</a></dfn> unit<li>The <dfn id="'vw'"><a href=https://drafts.csswg.org/css-values/#vw>'vw'</a></dfn> unit<li>The <dfn id="'in'"><a href=https://drafts.csswg.org/css-values/#in>'in'</a></dfn> unit<li>The <dfn id="'px'"><a href=https://drafts.csswg.org/css-values/#px>'px'</a></dfn> unit<li>The <dfn id="'attr()'"><a href=https://drafts.csswg.org/css-values/#funcdef-attr>'attr()'</a></dfn> function<li>The <dfn id=math-functions><a href=https://drafts.csswg.org/css-values/#math-function>math functions</a></dfn></ul>

    <p>The term <dfn id=css-styling-attribute><a href=https://drafts.csswg.org/css-style-attr/#style-attribute>style attribute</a></dfn> is
    defined in <cite>CSS Style Attributes</cite>. <a href=#refsCSSATTR>[CSSATTR]</a></p>

    <p>The following terms are defined in the <cite>CSS Cascading and Inheritance</cite>: <a href=#refsCSSCASCADE>[CSSCASCADE]</a></p>

    <ul class=brief><li><dfn id=specified-value><a href=https://drafts.csswg.org/css-cascade/#specified-value>specified value</a></dfn><li><dfn id=computed-value><a href=https://drafts.csswg.org/css-cascade/#computed-value>computed value</a></dfn><li><dfn id=used-value><a href=https://drafts.csswg.org/css-cascade/#used-value>used value</a></dfn></ul>

    <p>The <code id=dependencies:canvasrenderingcontext2d><a href=#canvasrenderingcontext2d>CanvasRenderingContext2D</a></code> object's use of fonts depends on the features
    described in the CSS <cite>Fonts</cite> and <cite>Font Loading</cite> specifications, including
    in particular <dfn id=fontface><code>FontFace</code></dfn> objects and the <dfn id=font-source><a href=https://drafts.csswg.org/css-font-loading/#font-source>font source</a></dfn> concept.
    <a href=#refsCSSFONTS>[CSSFONTS]</a> <a href=#refsCSSFONTLOAD>[CSSFONTLOAD]</a></p>

    <p>The following interfaces and terms are defined in <cite>Geometry Interfaces</cite>: <a href=#refsGEOMETRY>[GEOMETRY]</a></p>

    <ul class=brief><li><dfn id=dommatrix><a href=https://drafts.fxtf.org/geometry/#dommatrix><code>DOMMatrix</code></a></dfn> interface, and associated
         <dfn id=m11-element><a href=https://drafts.fxtf.org/geometry/#matrix-m11-element>m11 element</a></dfn>,
         <dfn id=m12-element><a href=https://drafts.fxtf.org/geometry/#matrix-m12-element>m12 element</a></dfn>,
         <dfn id=m21-element><a href=https://drafts.fxtf.org/geometry/#matrix-m21-element>m21 element</a></dfn>,
         <dfn id=m22-element><a href=https://drafts.fxtf.org/geometry/#matrix-m22-element>m22 element</a></dfn>,
         <dfn id=m41-element><a href=https://drafts.fxtf.org/geometry/#matrix-m41-element>m41 element</a></dfn>, and
         <dfn id=m42-element><a href=https://drafts.fxtf.org/geometry/#matrix-m42-element>m42 element</a></dfn><li><dfn id=dommatrix2dinit><a href=https://drafts.fxtf.org/geometry/#dictdef-dommatrix2dinit><code>DOMMatrix2DInit</code></a></dfn> and
         <dfn id=dommatrixinit><a href=https://drafts.fxtf.org/geometry/#dictdef-dommatrixinit><code>DOMMatrixInit</code></a></dfn> dictionaries<li>The <dfn id=create-a-dommatrix-from-a-dictionary><a href=https://drafts.fxtf.org/geometry/#create-a-dommatrix-from-the-dictionary>create a <code>DOMMatrix</code> from a dictionary</a></dfn>
         and <dfn id=create-a-dommatrix-from-a-2d-dictionary><a href=https://drafts.fxtf.org/geometry/#create-a-dommatrix-from-the-2d-dictionary>create a <code>DOMMatrix</code> from a 2D dictionary</a></dfn>
         algorithms for <code id=dependencies:dommatrix2dinit><a data-x-internal=dommatrix2dinit href=https://drafts.fxtf.org/geometry/#dictdef-dommatrix2dinit>DOMMatrix2DInit</a></code> or <code id=dependencies:dommatrixinit><a data-x-internal=dommatrixinit href=https://drafts.fxtf.org/geometry/#dictdef-dommatrixinit>DOMMatrixInit</a></code></ul>

    <p>The following terms are defined in the <cite>CSS Scoping</cite>: <a href=#refsCSSSCOPING>[CSSSCOPING]</a></p>

    <ul class=brief><li><dfn id=flat-tree><a href=https://drafts.csswg.org/css-scoping/#flat-tree>flat tree</a></dfn></ul>

    <p>The following terms and features are defined in <cite>CSS Color Adjustment</cite>: <a href=#refsCSSCOLORADJUST>[CSSCOLORADJUST]</a></p>

    <ul class=brief><li><dfn id="'color-scheme'"><a href=https://drafts.csswg.org/css-color-adjust/#color-scheme-prop>'color-scheme'</a></dfn></ul>
   <dt>Intersection Observer<dd>
    <p>The following term is defined in <cite>Intersection Observer</cite>: <a href=#refsINTERSECTIONOBSERVER>[INTERSECTIONOBSERVER]</a></p>

    <ul class=brief><li><dfn id=run-the-update-intersection-observations-steps><a href=https://w3c.github.io/IntersectionObserver/#run-the-update-intersection-observations-steps>run the update intersection observations steps</a></dfn><li><dfn id=intersectionobserver><a href=https://w3c.github.io/IntersectionObserver/#intersectionobserver><code>IntersectionObserver</code></a></dfn><li><dfn id=intersectionobserverinit><a href=https://w3c.github.io/IntersectionObserver/#dictdef-intersectionobserverinit><code>IntersectionObserverInit</code></a></dfn><li><dfn id=dom-intersectionobserver-observe><a href=https://w3c.github.io/IntersectionObserver/#dom-intersectionobserver-observe><code>observe</code></a></dfn><li><dfn id=dom-intersectionobserver-unobserve><a href=https://w3c.github.io/IntersectionObserver/#dom-intersectionobserver-unobserve><code>unobserve</code></a></dfn><li><dfn id=dom-intersectionobserverentry-isintersecting><a href=https://w3c.github.io/IntersectionObserver/#dom-intersectionobserverentry-isintersecting><code>isIntersecting</code></a></dfn><li><dfn id=dom-intersectionobserverentry-target><a href=https://w3c.github.io/IntersectionObserver/#dom-intersectionobserverentry-target><code>target</code></a></dfn></ul>
   <dt>WebGL<dd>
    <p>The following interfaces are defined in the WebGL specifications: <a href=#refsWEBGL>[WEBGL]</a></p>

    <ul class=brief><li><dfn id=webglrenderingcontext><a href=https://www.khronos.org/registry/webgl/specs/latest/1.0/#WebGLRenderingContext><code>WebGLRenderingContext</code></a></dfn> interface<li><dfn id=webgl2renderingcontext><a href=https://www.khronos.org/registry/webgl/specs/latest/2.0/#WebGL2RenderingContext><code>WebGL2RenderingContext</code></a></dfn> interface<li><dfn id=webglcontextattributes><a href=https://www.khronos.org/registry/webgl/specs/latest/1.0/#WebGLContextAttributes><code>WebGLContextAttributes</code></a></dfn> dictionary</ul>
   <dt>WebVTT<dd>
    <p>Implementations may support WebVTT as a text track format for subtitles, captions, metadata,
    etc., for media resources. <a href=#refsWEBVTT>[WEBVTT]</a></p>

    <p>The following terms, used in this specification, are defined in <cite>WebVTT</cite>:</p>

    <ul class=brief><li><dfn id=webvtt-file><a href=https://w3c.github.io/webvtt/#webvtt-file>WebVTT file</a></dfn><li><dfn id=webvtt-file-using-cue-text><a href=https://w3c.github.io/webvtt/#webvtt-file-using-cue-text>WebVTT file using cue text</a></dfn><li><dfn id=webvtt-file-using-only-nested-cues><a href=https://w3c.github.io/webvtt/#webvtt-file-using-only-nested-cues>WebVTT file using only nested cues</a></dfn><li><dfn id=webvtt-parser><a href=https://w3c.github.io/webvtt/#webvtt-parser>WebVTT parser</a></dfn><li>The <dfn id=rules-for-updating-the-display-of-webvtt-text-tracks><a href=https://w3c.github.io/webvtt/#rules-for-updating-the-display-of-webvtt-text-tracks>rules for updating the display of WebVTT text tracks</a></dfn><li>The WebVTT <dfn id=text-track-cue-writing-direction><a href=https://w3c.github.io/webvtt/#webvtt-cue-writing-direction>text track cue writing direction</a></dfn><li><dfn id=vttcue><a href=https://w3c.github.io/webvtt/#vttcue><code>VTTCue</code></a></dfn> interface</ul>
   <dt>The WebSocket protocol<dd>
    <p>The following terms are defined in <cite>Fetch</cite>: <a href=#refsFETCH>[FETCH]</a></p>

    <ul class=brief><li><dfn id=concept-websocket-establish><a href=https://fetch.spec.whatwg.org/#concept-websocket-establish>establish a WebSocket connection</a></dfn></ul>

    <p>The following terms are defined in <cite>The WebSocket protocol</cite>: <a href=#refsWSP>[WSP]</a></p>

    <ul class=brief><li><dfn id=concept-websocket-established>the WebSocket connection is established</dfn><li><dfn id=concept-websockets-active-extensions>extensions in use</dfn><li><dfn id=concept-websocket-subprotocol>subprotocol in use</dfn><li><dfn id=concept-websocket-message-received>a WebSocket message has been received</dfn><li><dfn id=concept-websocket-send>send a WebSocket Message</dfn><li><dfn id=concept-websocket-fail>fail the WebSocket connection</dfn><li><dfn id=concept-websocket-close>close the WebSocket connection</dfn><li><dfn id=concept-websocket-start-closing-handshake>start the WebSocket closing handshake</dfn><li><dfn id=concept-websocket-closing-handshake>the WebSocket closing handshake is started</dfn><li><dfn id=concept-websocket-closed>the WebSocket connection is closed</dfn> (possibly <i>cleanly</i>)<li><dfn id=concept-websocket-close-code>the WebSocket connection close code</dfn><li><dfn id=concept-websocket-close-reason>the WebSocket connection close reason</dfn><li><dfn id=http-sec-websocket-protocol><code>Sec-WebSocket-Protocol</code></dfn> field</ul>
   <dt>ARIA<dd>
    <p>The <dfn id=attr-aria-role data-dfn-type=element-attr><code>role</code></dfn> attribute is defined in
    <cite>Accessible Rich Internet Applications</cite> (<cite>ARIA</cite>), as are the following
    roles: <a href=#refsARIA>[ARIA]</a></p>

    <ul class=brief><li><dfn id=attr-aria-role-button><a href=https://w3c.github.io/aria/#button><code>button</code></a></dfn><li><dfn id=attr-aria-role-presentation><a href=https://w3c.github.io/aria/#presentation><code>presentation</code></a></dfn></ul>

    <p>In addition, the following <dfn id=attr-aria-*><code>aria-*</code></dfn> content
    attributes are defined in <cite>ARIA</cite>: <a href=#refsARIA>[ARIA]</a></p>

    <ul class=brief><li><dfn id=attr-aria-checked><a href=https://w3c.github.io/aria/#aria-checked><code>aria-checked</code></a></dfn><li><dfn id=attr-aria-describedby><a href=https://w3c.github.io/aria/#aria-describedby><code>aria-describedby</code></a></dfn><li><dfn id=attr-aria-disabled><a href=https://w3c.github.io/aria/#aria-disabled><code>aria-disabled</code></a></dfn><li><dfn id=attr-aria-label><a href=https://w3c.github.io/aria/#aria-label><code>aria-label</code></a></dfn></ul>

    <p>Finally, the following terms are defined <cite>ARIA</cite>: <a href=#refsARIA>[ARIA]</a></p>

    <ul class=brief><li><dfn id=role><a href=https://w3c.github.io/aria/#dfn-role>role</a></dfn><li><dfn id=concept-accessible-name><a href=https://w3c.github.io/aria/#dfn-accessible-name>accessible name</a></dfn><li>The <dfn id=ariamixin><a href=https://w3c.github.io/aria/#ARIAMixin><code>ARIAMixin</code></a></dfn> interface, with its associated
             <dfn id=ariamixin-getter-steps><a href=https://w3c.github.io/aria/#dfn-ariamixin-getter-steps><code>ARIAMixin</code> getter steps</a></dfn> and
             <dfn id=ariamixin-setter-steps><a href=https://w3c.github.io/aria/#dfn-ariamixin-setter-steps><code>ARIAMixin</code> setter steps</a></dfn> hooks</ul>
   <dt>Content Security Policy<dd>
    <p>The following terms are defined in <cite>Content Security Policy</cite>: <a href=#refsCSP>[CSP]</a></p>

    <ul class=brief><li><dfn id=content-security-policy><a href=https://w3c.github.io/webappsec-csp/#content-security-policy-object>Content Security Policy</a></dfn><li><dfn id=csp-disposition><a href=https://w3c.github.io/webappsec-csp/#policy-disposition>disposition</a></dfn><li><dfn id=csp-directive-set><a href=https://w3c.github.io/webappsec-csp/#policy-directive-set>directive set</a></dfn><li><dfn id=content-security-policy-directive><a href=https://w3c.github.io/webappsec-csp/#directives>Content Security Policy directive</a></dfn><li><dfn id=concept-csp-list><a href=https://w3c.github.io/webappsec-csp/#csp-list>CSP list</a></dfn><li>The <dfn id=content-security-policy-syntax><a href=https://w3c.github.io/webappsec-csp/#grammardef-serialized-policy>Content Security Policy syntax</a></dfn><li><dfn id=enforce-the-policy><a href=https://w3c.github.io/webappsec-csp/#enforced>enforce the policy</a></dfn><li>The <dfn id=parse-a-serialized-content-security-policy><a href=https://w3c.github.io/webappsec-csp/#parse-serialized-policy>parse a serialized Content Security Policy</a></dfn> algorithm<li>The <dfn id="initialize-a-global-object's-csp-list"><a href=https://w3c.github.io/webappsec-csp/#initialize-global-object-csp>Initialize a global object's CSP list</a></dfn> algorithm<li>The <dfn id="initialize-a-document's-csp-list"><a href=https://w3c.github.io/webappsec-csp/#initialize-document-csp>Initialize a Document's CSP list</a></dfn> algorithm<li>The <dfn id="should-element's-inline-behavior-be-blocked-by-content-security-policy"><a href=https://w3c.github.io/webappsec-csp/#should-block-inline>Should element's inline behavior be blocked by Content Security Policy?</a></dfn> algorithm<li>The <dfn id=should-navigation-request-of-type-be-blocked-by-content-security-policy><a href=https://w3c.github.io/webappsec-csp/#should-block-navigation-request>Should navigation request of type be blocked by Content Security Policy?</a></dfn> algorithm<li>The <dfn id=should-navigation-response-to-navigation-request-of-type-in-target-be-blocked-by-content-security-policy><a href=https://w3c.github.io/webappsec-csp/#should-block-navigation-response>Should navigation response to navigation request of type in target be blocked by Content Security Policy?</a></dfn> algorithm<li>The <dfn id=report-uri-directive><a href=https://w3c.github.io/webappsec-csp/#report-uri><code>report-uri</code> directive</a></dfn><li>The <dfn id=csp-ensurecspdoesnotblockstringcompilation><a href=https://w3c.github.io/webappsec-csp/#can-compile-strings>EnsureCSPDoesNotBlockStringCompilation</a></dfn> abstract operation<li>The <dfn id=is-base-allowed-for-document><a href=https://w3c.github.io/webappsec-csp/#allow-base-for-document>Is base allowed for Document?</a></dfn> algorithm<li>The <dfn id=frame-ancestors-directive><a href=https://w3c.github.io/webappsec-csp/#frame-ancestors><code>frame-ancestors</code> directive</a></dfn><li>The <dfn id=sandbox-directive><a href=https://w3c.github.io/webappsec-csp/#sandbox><code>sandbox</code> directive</a></dfn><li>The <dfn id=should-plugin-element-be-blocked-a-priori-by-content-security-policy><a href=https://w3c.github.io/webappsec-csp/#should-plugin-element-be-blocked-a-priori-by-content-security-policy>Should plugin element be blocked <i lang=la>a priori</i> by Content Security Policy?</a></dfn> algorithm<li>The <dfn id=contains-a-header-delivered-content-security-policy><a href=https://w3c.github.io/webappsec-csp/#contains-a-header-delivered-content-security-policy>contains a header-delivered Content Security Policy</a></dfn> property.</ul>
   <dt>Service Workers<dd>
    <p>The following terms are defined in <cite>Service Workers</cite>: <a href=#refsSW>[SW]</a></p>

    <ul class=brief><li><dfn id=dfn-active-worker><a href=https://w3c.github.io/ServiceWorker/#dfn-active-worker>active worker</a></dfn><li><dfn id=dfn-client-message-queue><a href=https://w3c.github.io/ServiceWorker/#dfn-client-message-queue>client message queue</a></dfn><li><dfn id=dfn-control><a href=https://w3c.github.io/ServiceWorker/#dfn-control>control</a></dfn><li><dfn id=on-fetch-request-algorithm><a href=https://w3c.github.io/ServiceWorker/#on-fetch-request-algorithm>handle fetch</a></dfn><li><dfn id=scope-match-algorithm><a href=https://w3c.github.io/ServiceWorker/#scope-match-algorithm>match service worker registration</a></dfn><li><dfn id=dfn-service-worker><a href=https://w3c.github.io/ServiceWorker/#dfn-service-worker>service worker</a></dfn><li><dfn id=serviceworkercontainer-service-worker-client><a href=https://w3c.github.io/ServiceWorker/#serviceworkercontainer-service-worker-client>service worker client</a></dfn><li><dfn id=serviceworker><a href=https://w3c.github.io/ServiceWorker/#serviceworker><code>ServiceWorker</code></a></dfn> interface<li><dfn id=serviceworkercontainer><a href=https://w3c.github.io/ServiceWorker/#serviceworkercontainer><code>ServiceWorkerContainer</code></a></dfn> interface<li><dfn id=serviceworkerglobalscope><a href=https://w3c.github.io/ServiceWorker/#serviceworkerglobalscope><code>ServiceWorkerGlobalScope</code></a></dfn> interface</ul>
   <dt>Secure Contexts<dd>
    <p>The following algorithms are defined in <cite>Secure Contexts</cite>: <a href=#refsSECURE-CONTEXTS>[SECURE-CONTEXTS]</a></p>

    <ul class=brief><li><dfn id=is-url-potentially-trustworthy><a href=https://w3c.github.io/webappsec-secure-contexts/#potentially-trustworthy-url>Is url potentially trustworthy?</a></dfn></ul>
   <dt>Permissions Policy<dd>
    <p>The following terms are defined in <cite>Permissions Policy</cite>: <a href=#refsPERMISSIONSPOLICY>[PERMISSIONSPOLICY]</a></p>

    <ul class=brief><li><dfn id=concept-permissions-policy><a href=https://w3c.github.io/webappsec-feature-policy/#permissions-policy>permissions policy</a></dfn><li><dfn id=concept-policy-controlled-feature><a href=https://w3c.github.io/webappsec-feature-policy/#policy-controlled-feature>policy-controlled feature</a></dfn><li><dfn id=concept-container-policy><a href=https://w3c.github.io/webappsec-feature-policy/#container-policy>container policy</a></dfn><li><dfn id=concept-serialized-permissions-policy><a href=https://w3c.github.io/webappsec-feature-policy/#serialized-permissions-policy>serialized permissions policy</a></dfn><li><dfn id=concept-default-allowlist><a href=https://w3c.github.io/webappsec-feature-policy/#default-allowlist>default allowlist</a></dfn><li>The <dfn id=creating-a-permissions-policy><a href=https://w3c.github.io/webappsec-feature-policy/#create-for-browsingcontext>creating a permissions policy</a></dfn> algorithm<li>The <dfn id=creating-a-permissions-policy-from-a-response><a href=https://w3c.github.io/webappsec-feature-policy/#create-from-response>creating a permissions policy from a response</a></dfn> algorithm<li>The <dfn id=is-feature-enabled><a href=https://w3c.github.io/webappsec-feature-policy/#is-feature-enabled>is feature enabled by policy for origin</a></dfn> algorithm<li>The <dfn id=process-permissions-policy-attributes><a href=https://w3c.github.io/webappsec-feature-policy/#process-permissions-policy-attributes>process permissions policy attributes</a></dfn> algorithm</ul>
   <dt>Payment Request API<dd>
    <p>The following feature is defined in <cite>Payment Request API</cite>:
    <a href=#refsPAYMENTREQUEST>[PAYMENTREQUEST]</a></p>

    <ul class=brief><li><dfn id=paymentrequest><a href=https://w3c.github.io/payment-request/#dom-paymentrequest><code>PaymentRequest</code></a></dfn> interface</ul>
   <dt>MathML<dd>
    <p>While support for MathML as a whole is not required by this specification (though it is
    encouraged, at least for web browsers), certain features depend upon small parts of MathML being
    implemented. <a href=#refsMATHML>[MATHML]</a></p>

    <p>The following features are defined in <cite>Mathematical Markup Language</cite>
    (<cite>MathML</cite>):</p>

    <ul class=brief><li><dfn id=mathml-annotation-xml><a href=https://www.w3.org/Math/draft-spec/chapter5.html#mixing.elements.annotation.xml>MathML <code>annotation-xml</code></a></dfn> element<li><dfn id=mathml-math><a href=https://www.w3.org/Math/draft-spec/chapter2.html#interf.toplevel>MathML <code>math</code></a></dfn> element<li><dfn id=mathml-merror><a href=https://www.w3.org/Math/draft-spec/chapter3.html#presm.merror>MathML <code>merror</code></a></dfn> element<li><dfn id=mathml-mi><a href=https://www.w3.org/Math/draft-spec/chapter3.html#presm.mi>MathML <code>mi</code></a></dfn> element<li><dfn id=mathml-mn><a href=https://www.w3.org/Math/draft-spec/chapter3.html#presm.mn>MathML <code>mn</code></a></dfn> element<li><dfn id=mathml-mo><a href=https://www.w3.org/Math/draft-spec/chapter3.html#presm.mo>MathML <code>mo</code></a></dfn> element<li><dfn id=mathml-ms><a href=https://www.w3.org/Math/draft-spec/chapter3.html#presm.ms>MathML <code>ms</code></a></dfn> element<li><dfn id=mathml-mtext><a href=https://www.w3.org/Math/draft-spec/chapter3.html#presm.mtext>MathML <code>mtext</code></a></dfn> element</ul>
   <dt>SVG<dd>
    <p>While support for SVG as a whole is not required by this specification (though it is
    encouraged, at least for web browsers), certain features depend upon parts of SVG being
    implemented.</p>

    <p>User agents that implement SVG must implement the <cite>SVG 2</cite> specification, and not
    any earlier revisions.</p>

    <p>The following features are defined in the <cite>SVG 2</cite> specification: <a href=#refsSVG>[SVG]</a></p>

    <ul class=brief><li><dfn id=svgelement><a href=https://svgwg.org/svg2-draft/types.html#InterfaceSVGElement><code>SVGElement</code></a></dfn> interface<li><dfn id=svgimageelement><a href=https://svgwg.org/svg2-draft/embedded.html#InterfaceSVGImageElement><code>SVGImageElement</code></a></dfn> interface<li><dfn id=svgscriptelement><a href=https://svgwg.org/svg2-draft/interact.html#InterfaceSVGScriptElement><code>SVGScriptElement</code></a></dfn> interface<li><dfn id=svgsvgelement><a href=https://svgwg.org/svg2-draft/struct.html#InterfaceSVGSVGElement><code>SVGSVGElement</code></a></dfn> interface<li><dfn id=svg-a><a href=https://svgwg.org/svg2-draft/linking.html#AElement>SVG <code>a</code></a></dfn> element<li><dfn id=svg-desc><a href=https://svgwg.org/svg2-draft/struct.html#DescElement>SVG <code>desc</code></a></dfn> element<li><dfn id=svg-foreignobject><a href=https://svgwg.org/svg2-draft/embedded.html#ForeignObjectElement>SVG <code>foreignObject</code></a></dfn> element<li><dfn id=svg-image><a href=https://svgwg.org/svg2-draft/embedded.html#ImageElement>SVG <code>image</code></a></dfn> element<li><dfn id=svg-script><a href=https://svgwg.org/svg2-draft/interact.html#ScriptElement>SVG <code>script</code></a></dfn> element<li><dfn id=svg-svg><a href=https://svgwg.org/svg2-draft/struct.html#SVGElement>SVG <code>svg</code></a></dfn> element<li><dfn id=svg-title><a href=https://svgwg.org/svg2-draft/struct.html#TitleElement>SVG <code>title</code></a></dfn> element<li><dfn id=svg-use><a href=https://svgwg.org/svg2-draft/struct.html#UseElement>SVG <code>use</code></a></dfn> element</ul>
   <dt>Filter Effects<dd>
    <p>The following feature is defined in <cite>Filter Effects</cite>:
    <a href=#refsFILTERS>[FILTERS]</a></p>

    <ul class=brief><li><dfn id=filter-function-list><a href=https://drafts.fxtf.org/filter-effects/#typedef-filter-function-list>&lt;filter-function-list></a></dfn></ul>
   <dt>Cooperative Scheduling of Background Tasks<dd>
    <p>The following features are defined in <cite>Cooperative Scheduling of Background Tasks</cite>:
    <a href=#refsREQUESTIDLECALLBACK>[REQUESTIDLECALLBACK]</a></p>

    <ul class=brief><li><dfn id=requestidlecallback()><a href=https://w3c.github.io/requestidlecallback/#the-requestidlecallback-method><code>requestIdleCallback()</code></a></dfn><li><dfn id=start-an-idle-period-algorithm><a href=https://w3c.github.io/requestidlecallback/#start-an-idle-period-algorithm>start an idle period algorithm</a></dfn></ul>
   <dt>Storage<dd>
    <p>The following terms are defined in <cite>Storage</cite>: <a href=#refsSTORAGE>[STORAGE]</a></p>

    <ul class=brief><li><dfn id=obtain-a-local-storage-bottle-map><a href=https://storage.spec.whatwg.org/#obtain-a-local-storage-bottle-map>obtain a local storage bottle map</a></dfn><li><dfn id=obtain-a-session-storage-bottle-map><a href=https://storage.spec.whatwg.org/#obtain-a-session-storage-bottle-map>obtain a session storage bottle map</a></dfn><li><dfn id=storage-proxy-map><a href=https://storage.spec.whatwg.org/#storage-proxy-map>storage proxy map</a></dfn><li><dfn id=legacy-clone-a-browsing-session-storage-shed><a href=https://storage.spec.whatwg.org/#legacy-clone-a-browsing-session-storage-shed>legacy-clone a browsing session storage shed</a></dfn></ul>
   <dt>Web App Manifest<dd>
    <p>The following features are defined in <cite>Web App Manifest</cite>: <a href=#refsMANIFEST>[MANIFEST]</a></p>

    <ul class=brief><li><dfn id=application-manifest><a href=https://w3c.github.io/manifest/#dfn-manifest>application manifest</a></dfn><li><dfn id=installed-web-application><a href=https://w3c.github.io/manifest/#dfn-installed-web-application>installed web application</a></dfn><li><dfn id=process-the-manifest><a href=https://w3c.github.io/manifest/#dfn-processing-a-manifest>process the manifest</a></dfn></ul>
   </dl>

  <hr>

  <p>This specification does not <em>require</em> support of any particular network protocol, style
  sheet language, scripting language, or any of the DOM specifications beyond those required in the
  list above. However, the language described by this specification is biased towards CSS as the
  styling language, JavaScript as the scripting language, and HTTP as the network protocol, and
  several features assume that those languages and protocols are in use.</p>

  <p>A user agent that implements the HTTP protocol must implement <cite>HTTP State Management
  Mechanism</cite> (Cookies) as well. <a href=#refsHTTP>[HTTP]</a> <a href=#refsCOOKIES>[COOKIES]</a></p>

  <p class=note>This specification might have certain additional requirements on character
  encodings, image formats, audio formats, and video formats in the respective sections.</p>

  </div>




  <h4 id=extensibility-2><span class=secno>2.1.10</span> Extensibility<a href=#extensibility-2 class=self-link></a></h4>

  <p>Vendor-specific proprietary user agent extensions to this specification are strongly
  discouraged. Documents must not use such extensions, as doing so reduces interoperability and
  fragments the user base, allowing only users of specific user agents to access the content in
  question.</p>

  

  <p>All extensions must be defined so that the use of extensions neither contradicts nor causes the
  non-conformance of functionality defined in the specification.</p> 

  <div class=example>

   <p>For example, while strongly discouraged from doing so, an implementation could add a new IDL
   attribute "<code>typeTime</code>" to a control that returned the time it took the user
   to select the current value of a control (say). On the other hand, defining a new control that
   appears in a form's <code id=extensibility-2:dom-form-elements><a href=#dom-form-elements>elements</a></code> array would be in violation
   of the above requirement, as it would violate the definition of <code id=extensibility-2:dom-form-elements-2><a href=#dom-form-elements>elements</a></code> given in this specification.</p>

  </div>

  

  <hr>

  <p>When vendor-neutral extensions to this specification are needed, either this specification can
  be updated accordingly, or an extension specification can be written that overrides the
  requirements in this specification. When someone applying this specification to their activities
  decides that they will recognize the requirements of such an extension specification, it becomes
  an <dfn id=other-applicable-specifications>applicable specification</dfn> for the purposes
  of conformance requirements in this specification.</p>

  <p class=note>Someone could write a specification that defines any arbitrary byte stream as
  conforming, and then claim that their random junk is conforming. However, that does not mean that
  their random junk actually is conforming for everyone's purposes: if someone else decides that
  that specification does not apply to their work, then they can quite legitimately say that the
  aforementioned random junk is just that, junk, and not conforming at all. As far as conformance
  goes, what matters in a particular community is what that community <em>agrees</em> is
  applicable.</p>

  

  <hr>

  <p>User agents must treat elements and attributes that they do not understand as semantically
  neutral; leaving them in the DOM (for DOM processors), and styling them according to CSS (for CSS
  processors), but not inferring any meaning from them.</p>


  <p>When support for a feature is disabled (e.g. as an emergency measure to mitigate a security
  problem, or to aid in development, or for performance reasons), user agents must act as if they
  had no support for the feature whatsoever, and as if the feature was not mentioned in this
  specification. For example, if a particular feature is accessed via an attribute in a Web IDL
  interface, the attribute itself would be omitted from the objects that implement that interface
  — leaving the attribute on the object but making it return null or throw an exception is
  insufficient.</p>


  


  

  <h4 id=interactions-with-xpath-and-xslt><span class=secno>2.1.11</span> Interactions with XPath and XSLT<a href=#interactions-with-xpath-and-xslt class=self-link></a></h4>

  <p id=xpath-1.0-processors>Implementations of XPath 1.0 that operate on <a id=interactions-with-xpath-and-xslt:html-documents href=https://dom.spec.whatwg.org/#html-document data-x-internal=html-documents>HTML
  documents</a> parsed or created in the manners described in this specification (e.g. as part of
  the <code>document.evaluate()</code> API) must act as if the following edit was applied
  to the XPath 1.0 specification.</p>

  <p>First, remove this paragraph:</p>

  <blockquote cite=https://www.w3.org/TR/1999/REC-xpath-19991116/#node-tests>

   <p>A <a href=https://www.w3.org/TR/REC-xml-names/#NT-QName>QName</a> in the node test is expanded
   into an <a href=https://www.w3.org/TR/1999/REC-xpath-19991116/#dt-expanded-name>expanded-name</a>
   using the namespace declarations from the expression context. This is the same way expansion is
   done for element type names in start and end-tags except that the default namespace declared with
   <code>xmlns</code> is not used: if the <a href=https://www.w3.org/TR/REC-xml-names/#NT-QName>QName</a> does not have a prefix, then the
   namespace URI is null (this is the same way attribute names are expanded). It is an error if the
   <a href=https://www.w3.org/TR/REC-xml-names/#NT-QName>QName</a> has a prefix for which there is
   no namespace declaration in the expression context.</p>

  </blockquote>

  <p>Then, insert in its place the following:</p>

  <blockquote cite="https://www.w3.org/Bugs/Public/show_bug.cgi?id=7059#c37">

   <p>A QName in the node test is expanded into an expanded-name using the namespace declarations
   from the expression context. If the QName has a prefix, then there must be a namespace declaration for this prefix in
   the expression context, and the corresponding namespace URI is the one that is
   associated with this prefix. It is an error if the QName has a prefix for which there is no
   namespace declaration in the expression context. </p>

   <p>If the QName has no prefix and the principal node type of the axis is element, then the
   default element namespace is used. Otherwise if the QName has no prefix, the namespace URI is
   null. The default element namespace is a member of the context for the XPath expression. The
   value of the default element namespace when executing an XPath expression through the DOM3 XPath
   API is determined in the following way:</p>

   <ol><li>If the context node is from an HTML DOM, the default element namespace is
    "http://www.w3.org/1999/xhtml".<li>Otherwise, the default element namespace URI is null.</ol>

   <p class=note>This is equivalent to adding the default element namespace feature of XPath 2.0
   to XPath 1.0, and using the HTML namespace as the default element namespace for HTML documents.
   It is motivated by the desire to have implementations be compatible with legacy HTML content
   while still supporting the changes that this specification introduces to HTML regarding the
   namespace used for HTML elements, and by the desire to use XPath 1.0 rather than XPath 2.0.</p>

  </blockquote>

  <p class=note>This change is a <a href=#willful-violation id=interactions-with-xpath-and-xslt:willful-violation>willful violation</a> of the XPath 1.0 specification,
  motivated by desire to have implementations be compatible with legacy content while still
  supporting the changes that this specification introduces to HTML regarding which namespace is
  used for HTML elements. <a href=#refsXPATH10>[XPATH10]</a></p> 

  <hr>

  <p id=dom-based-xslt-1.0-processors>XSLT 1.0 processors outputting to a DOM when the output
  method is "html" (either explicitly or via the defaulting rule in XSLT 1.0) are affected as
  follows:</p>

  <p>If the transformation program outputs an element in no namespace, the processor must, prior to
  constructing the corresponding DOM element node, change the namespace of the element to the
  <a id=interactions-with-xpath-and-xslt:html-namespace-2 href=https://infra.spec.whatwg.org/#html-namespace data-x-internal=html-namespace-2>HTML namespace</a>, <a href=https://infra.spec.whatwg.org/#ascii-lowercase id=interactions-with-xpath-and-xslt:converted-to-ascii-lowercase data-x-internal=converted-to-ascii-lowercase>ASCII-lowercase</a> the
  element's local name, and <a href=https://infra.spec.whatwg.org/#ascii-lowercase id=interactions-with-xpath-and-xslt:converted-to-ascii-lowercase-2 data-x-internal=converted-to-ascii-lowercase>ASCII-lowercase</a> the
  names of any non-namespaced attributes on the element.</p>

  <p class=note>This requirement is a <a href=#willful-violation id=interactions-with-xpath-and-xslt:willful-violation-2>willful violation</a> of the XSLT 1.0
  specification, required because this specification changes the namespaces and case-sensitivity
  rules of HTML in a manner that would otherwise be incompatible with DOM-based XSLT
  transformations. (Processors that serialize the output are unaffected.) <a href=#refsXSLT10>[XSLT10]</a></p> 

  <hr>

  <p>This specification does not specify precisely how XSLT processing interacts with the <a href=#html-parser id=interactions-with-xpath-and-xslt:html-parser>HTML
  parser</a> infrastructure (for example, whether an XSLT processor acts as if it puts any
  elements into a <a href=#stack-of-open-elements id=interactions-with-xpath-and-xslt:stack-of-open-elements>stack of open elements</a>). However, XSLT processors must <a href=#stop-parsing id=interactions-with-xpath-and-xslt:stop-parsing>stop
  parsing</a> if they successfully complete, and must set the <a href=#current-document-readiness id=interactions-with-xpath-and-xslt:current-document-readiness>current document
  readiness</a> first to "<code>interactive</code>" and then to "<code>complete</code>" if they are aborted.</p>

  <hr>

  <p>This specification does not specify how XSLT interacts with the <a href=#navigate id=interactions-with-xpath-and-xslt:navigate>navigation</a> algorithm, how it fits in with the <a href=#event-loop id=interactions-with-xpath-and-xslt:event-loop>event loop</a>, nor
  how error pages are to be handled (e.g. whether XSLT errors are to replace an incremental XSLT
  output, or are rendered inline, etc).</p>

  <p class=note>There are also additional non-normative comments regarding the interaction of XSLT
  and HTML <a href=#scriptTagXSLT>in the <code>script</code> element section</a>, and of
  XSLT, XPath, and HTML <a href=#template-XSLT-XPath>in the <code>template</code> element
  section</a>.</p>

  

  <h3 id=case-sensitivity-and-string-comparison><span class=secno>2.2</span> String comparison<a href=#case-sensitivity-and-string-comparison class=self-link></a></h3>

  <p>A string <var>pattern</var> is a <dfn id=prefix-match>prefix match</dfn> for a string <var>s</var> when
  <var>pattern</var> is not longer than <var>s</var> and truncating <var>s</var> to
  <var>pattern</var>'s length leaves the two strings as matches of each other.</p>


  <h3 id=policy-controlled-features><span class=secno>2.3</span> Policy-controlled features<a href=#policy-controlled-features class=self-link></a></h3>

  <p>This document defines the following <a href=https://w3c.github.io/webappsec-feature-policy/#policy-controlled-feature id=policy-controlled-features:concept-policy-controlled-feature data-x-internal=concept-policy-controlled-feature>policy-controlled features</a>:</p>

  <div class="mdn-anno wrapped before"><button onclick=toggleStatus(this) class=mdn-anno-btn><span>MDN</span></button><div class=feature><p><a href=https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Feature-Policy/document-domain title="The HTTP Feature-Policy header document-domain directive controls whether the current document is allowed to set document.domain. When this policy is enabled, attempting to set document.domain will fail and cause a SecurityError DOMException to be be thrown.">Headers/Feature-Policy/document-domain</a><div class=support><span class="firefox yes"><span>Firefox</span><span>74+</span></span><span class="safari no"><span>Safari</span><span>No</span></span><span class="chrome yes"><span>Chrome</span><span>77+</span></span><hr><span class="opera yes"><span>Opera</span><span>64+</span></span><span class="edge_blink yes"><span>Edge</span><span>79+</span></span><hr><span class="edge no"><span>Edge (Legacy)</span><span>No</span></span><span class="ie no"><span>Internet Explorer</span><span>No</span></span><hr><span class="firefox_android yes"><span>Firefox Android</span><span title="Requires setting a user preference or runtime flag.">🔰 65+</span></span><span class="safari_ios no"><span>Safari iOS</span><span>No</span></span><span class="chrome_android no"><span>Chrome Android</span><span>No</span></span><span class="webview_android no"><span>WebView Android</span><span>No</span></span><span class="samsunginternet_android no"><span>Samsung Internet</span><span>No</span></span><span class="opera_android no"><span>Opera Android</span><span>No</span></span></div></div></div><ul><li>"<dfn id=autoplay-feature><code>autoplay</code></dfn>", which has a <a href=https://w3c.github.io/webappsec-feature-policy/#default-allowlist id=policy-controlled-features:concept-default-allowlist data-x-internal=concept-default-allowlist>default allowlist</a> of <code>'self'</code>.<li>"<dfn id=cross-origin-isolated-feature><code>cross-origin-isolated</code></dfn>", which has a <a href=https://w3c.github.io/webappsec-feature-policy/#default-allowlist id=policy-controlled-features:concept-default-allowlist-2 data-x-internal=concept-default-allowlist>default allowlist</a> of <code>'self'</code>.<li>"<dfn id=document-domain-feature><code>document-domain</code></dfn>", which
    has a <a href=https://w3c.github.io/webappsec-feature-policy/#default-allowlist id=policy-controlled-features:concept-default-allowlist-3 data-x-internal=concept-default-allowlist>default allowlist</a> of <code>*</code>.</ul>


  <h3 id=common-microsyntaxes><span class=secno>2.4</span> Common microsyntaxes<a href=#common-microsyntaxes class=self-link></a></h3>

  <p>There are various places in HTML that accept particular data types, such as dates or numbers.
  This section describes what the conformance criteria for content in those formats is, and how to
  parse them.</p>

  

  <p class=note>Implementors are strongly urged to carefully examine any third-party libraries
  they might consider using to implement the parsing of syntaxes described below. For example, date
  libraries are likely to implement error handling behavior that differs from what is required in
  this specification, since error-handling behavior is often not defined in specifications that
  describe date syntaxes similar to those used in this specification, and thus implementations tend
  to vary greatly in how they handle errors.</p>

  


  

  <h4 id=common-parser-idioms><span class=secno>2.4.1</span> Common parser idioms<a href=#common-parser-idioms class=self-link></a></h4>

  <p>Some of the micro-parsers described below follow the pattern of having an <var>input</var>
  variable that holds the string being parsed, and having a <var>position</var> variable pointing at
  the next character to parse in <var>input</var>.</p>

  



  <h4 id=boolean-attributes><span class=secno>2.4.2</span> Boolean attributes<a href=#boolean-attributes class=self-link></a></h4>

  <p>A number of attributes are <dfn id=boolean-attribute>boolean attributes</dfn>. The
  presence of a boolean attribute on an element represents the true value, and the absence of the
  attribute represents the false value.</p>

  <p>If the attribute is present, its value must either be the empty string or a value that is an
  <a id=boolean-attributes:ascii-case-insensitive href=https://infra.spec.whatwg.org/#ascii-case-insensitive data-x-internal=ascii-case-insensitive>ASCII case-insensitive</a> match for the attribute's canonical name, with no leading or
  trailing whitespace.</p>

  <p class=note>The values "true" and "false" are not allowed on boolean attributes. To represent
  a false value, the attribute has to be omitted altogether.</p>

  <div class=example>

   <p>Here is an example of a checkbox that is checked and disabled. The <code id=boolean-attributes:attr-input-checked><a href=#attr-input-checked>checked</a></code> and <code id=boolean-attributes:attr-fe-disabled><a href=#attr-fe-disabled>disabled</a></code>
   attributes are the boolean attributes.</p>

   <pre><code class='html'><c- p>&lt;</c-><c- f>label</c-><c- p>&gt;&lt;</c-><c- f>input</c-> <c- e>type</c-><c- o>=</c-><c- s>checkbox</c-> <c- e>checked</c-> <c- e>name</c-><c- o>=</c-><c- s>cheese</c-> <c- e>disabled</c-><c- p>&gt;</c-> Cheese<c- p>&lt;/</c-><c- f>label</c-><c- p>&gt;</c-></code></pre>

   <p>This could be equivalently written as this:

   <pre><code class='html'><c- p>&lt;</c-><c- f>label</c-><c- p>&gt;&lt;</c-><c- f>input</c-> <c- e>type</c-><c- o>=</c-><c- s>checkbox</c-> <c- e>checked</c-><c- o>=</c-><c- s>checked</c-> <c- e>name</c-><c- o>=</c-><c- s>cheese</c-> <c- e>disabled</c-><c- o>=</c-><c- s>disabled</c-><c- p>&gt;</c-> Cheese<c- p>&lt;/</c-><c- f>label</c-><c- p>&gt;</c-></code></pre>

   <p>You can also mix styles; the following is still equivalent:</p>

   <pre><code class='html'><c- p>&lt;</c-><c- f>label</c-><c- p>&gt;&lt;</c-><c- f>input</c-> <c- e>type</c-><c- o>=</c-><c- s>&apos;checkbox&apos;</c-> <c- e>checked</c-> <c- e>name</c-><c- o>=</c-><c- s>cheese</c-> <c- e>disabled</c-><c- o>=</c-><c- s>&quot;&quot;</c-><c- p>&gt;</c-> Cheese<c- p>&lt;/</c-><c- f>label</c-><c- p>&gt;</c-></code></pre>

  </div>



  <h4 id=keywords-and-enumerated-attributes><span class=secno>2.4.3</span> Keywords and enumerated attributes<a href=#keywords-and-enumerated-attributes class=self-link></a></h4>

  <p>Some attributes are defined as taking one of a finite set of keywords. Such attributes are
  called <dfn id=enumerated-attribute>enumerated attributes</dfn>. The keywords are each
  defined to map to a particular <em>state</em> (several keywords might map to the same state, in
  which case some of the keywords are synonyms of each other; additionally, some of the keywords can
  be said to be non-conforming, and are only in the specification for historical reasons). In
  addition, two default states can be given. The first is the <dfn id=invalid-value-default><i>invalid value default</i></dfn>, the
  second is the <dfn id=missing-value-default><i>missing value default</i></dfn>.</p>

  <p>If an enumerated attribute is specified, the attribute's value must be an <a id=keywords-and-enumerated-attributes:ascii-case-insensitive href=https://infra.spec.whatwg.org/#ascii-case-insensitive data-x-internal=ascii-case-insensitive>ASCII
  case-insensitive</a> match for one of the given keywords that are not said to be
  non-conforming, with no leading or trailing whitespace.</p>

  <p>When the attribute is specified, if its value is an <a id=keywords-and-enumerated-attributes:ascii-case-insensitive-2 href=https://infra.spec.whatwg.org/#ascii-case-insensitive data-x-internal=ascii-case-insensitive>ASCII case-insensitive</a> match
  for one of the given keywords then that keyword's state is the state that the attribute
  represents. If the attribute value matches none of the given keywords, but the attribute has an
  <i id=keywords-and-enumerated-attributes:invalid-value-default><a href=#invalid-value-default>invalid value default</a></i>, then the attribute represents that
  state. Otherwise, there is no default, and invalid values mean that there is no state
  represented.</p>

  <p>When the attribute is <em>not</em> specified, if there is a <i id=keywords-and-enumerated-attributes:missing-value-default><a href=#missing-value-default>missing value default</a></i> state
  defined, then that is the state represented by the (missing) attribute. Otherwise, the absence of
  the attribute means that there is no state represented.</p>

  <p class=note>The empty string can be a valid keyword.</p>


  <h4 id=numbers><span class=secno>2.4.4</span> Numbers<a href=#numbers class=self-link></a></h4>

  <h5 id=signed-integers><span class=secno>2.4.4.1</span> Signed integers<a href=#signed-integers class=self-link></a></h5>

  <p>A string is a <dfn id=valid-integer>valid integer</dfn> if it consists of one or more <a id=signed-integers:ascii-digits href=https://infra.spec.whatwg.org/#ascii-digit data-x-internal=ascii-digits>ASCII digits</a>,
  optionally prefixed with a U+002D HYPHEN-MINUS character (-).</p>

  <p>A <a href=#valid-integer id=signed-integers:valid-integer>valid integer</a> without a U+002D HYPHEN-MINUS (-) prefix represents the number
  that is represented in base ten by that string of digits. A <a href=#valid-integer id=signed-integers:valid-integer-2>valid integer</a>
  <em>with</em> a U+002D HYPHEN-MINUS (-) prefix represents the number represented in base ten by
  the string of digits that follows the U+002D HYPHEN-MINUS, subtracted from zero.</p>

  

  <p>The <dfn id=rules-for-parsing-integers>rules for parsing integers</dfn> are as given in the following algorithm. When
  invoked, the steps must be followed in the order given, aborting at the first step that returns a
  value. This algorithm will return either an integer or an error.</p>

  <ol><li><p>Let <var>input</var> be the string being parsed.<li><p>Let <var>position</var> be a pointer into <var>input</var>, initially pointing at the
   start of the string.<li><p>Let <var>sign</var> have the value "positive".<li><p><a id=signed-integers:skip-ascii-whitespace href=https://infra.spec.whatwg.org/#skip-ascii-whitespace data-x-internal=skip-ascii-whitespace>Skip ASCII whitespace</a> within <var>input</var> given
   <var>position</var>.<li><p>If <var>position</var> is past the end of <var>input</var>, return an error.<li>
    <p>If the character indicated by <var>position</var> (the first character) is a U+002D
    HYPHEN-MINUS character (-):</p>

    <ol><li>Let <var>sign</var> be "negative".<li>Advance <var>position</var> to the next character.<li>If <var>position</var> is past the end of <var>input</var>, return an error.</ol>

    <p>Otherwise, if the character indicated by <var>position</var> (the first character) is a
    U+002B PLUS SIGN character (+):</p>

    <ol><li>Advance <var>position</var> to the next character. (The "<code>+</code>" is
     ignored, but it is not conforming.)<li>If <var>position</var> is past the end of <var>input</var>, return an error.</ol>
   <li><p>If the character indicated by <var>position</var> is not an <a href=https://infra.spec.whatwg.org/#ascii-digit id=signed-integers:ascii-digits-2 data-x-internal=ascii-digits>ASCII digit</a>, then return an error.<li><p><a id=signed-integers:collect-a-sequence-of-code-points href=https://infra.spec.whatwg.org/#collect-a-sequence-of-code-points data-x-internal=collect-a-sequence-of-code-points>Collect a sequence of code points</a> that are <a id=signed-integers:ascii-digits-3 href=https://infra.spec.whatwg.org/#ascii-digit data-x-internal=ascii-digits>ASCII digits</a> from
   <var>input</var> given <var>position</var>, and interpret the resulting sequence as a base-ten
   integer. Let <var>value</var> be that integer.<li><p>If <var>sign</var> is "positive", return <var>value</var>, otherwise return the result of
   subtracting <var>value</var> from zero.</ol>

  


  <h5 id=non-negative-integers><span class=secno>2.4.4.2</span> Non-negative integers<a href=#non-negative-integers class=self-link></a></h5>

  <p>A string is a <dfn id=valid-non-negative-integer>valid non-negative integer</dfn> if it consists of one or more <a id=non-negative-integers:ascii-digits href=https://infra.spec.whatwg.org/#ascii-digit data-x-internal=ascii-digits>ASCII
  digits</a>.</p>

  <p>A <a href=#valid-non-negative-integer id=non-negative-integers:valid-non-negative-integer>valid non-negative integer</a> represents the number that is represented in base ten
  by that string of digits.</p>

  

  <p>The <dfn id=rules-for-parsing-non-negative-integers>rules for parsing non-negative integers</dfn> are as given in the following algorithm.
  When invoked, the steps must be followed in the order given, aborting at the first step that
  returns a value. This algorithm will return either zero, a positive integer, or an error.</p>

  <ol><li><p>Let <var>input</var> be the string being parsed.<li><p>Let <var>value</var> be the result of parsing <var>input</var> using the
   <a href=#rules-for-parsing-integers id=non-negative-integers:rules-for-parsing-integers>rules for parsing integers</a>.<li><p>If <var>value</var> is an error, return an error.<li><p>If <var>value</var> is less than zero, return an error.<li><p>Return <var>value</var>.</ol>

  

  


  <h5 id=floating-point-numbers><span class=secno>2.4.4.3</span> Floating-point numbers<a href=#floating-point-numbers class=self-link></a></h5>

  <p>A string is a <dfn id=valid-floating-point-number>valid floating-point number</dfn> if it consists of:</p>

  <ol class=brief><li>Optionally, a U+002D HYPHEN-MINUS character (-).<li>One or both of the following, in the given order:

    <ol><li>A series of one or more <a id=floating-point-numbers:ascii-digits href=https://infra.spec.whatwg.org/#ascii-digit data-x-internal=ascii-digits>ASCII digits</a>.<li>Both of the following, in the given order:

      <ol><li>A single U+002E FULL STOP character (.).<li>A series of one or more <a id=floating-point-numbers:ascii-digits-2 href=https://infra.spec.whatwg.org/#ascii-digit data-x-internal=ascii-digits>ASCII digits</a>.</ol>
     </ol>
   <li>Optionally:

    <ol><li>Either a U+0065 LATIN SMALL LETTER E character (e) or a U+0045 LATIN CAPITAL LETTER E
     character (E).<li>Optionally, a U+002D HYPHEN-MINUS character (-) or U+002B PLUS SIGN character (+).<li>A series of one or more <a id=floating-point-numbers:ascii-digits-3 href=https://infra.spec.whatwg.org/#ascii-digit data-x-internal=ascii-digits>ASCII digits</a>.</ol>
   </ol>

  <p>A <a href=#valid-floating-point-number id=floating-point-numbers:valid-floating-point-number>valid floating-point number</a> represents the number obtained by multiplying the
  significand by ten raised to the power of the exponent, where the significand is the first number,
  interpreted as base ten (including the decimal point and the number after the decimal point, if
  any, and interpreting the significand as a negative number if the whole string starts with a
  U+002D HYPHEN-MINUS character (-) and the number is not zero), and where the exponent is the
  number after the E, if any (interpreted as a negative number if there is a U+002D HYPHEN-MINUS
  character (-) between the E and the number and the number is not zero, or else ignoring a U+002B
  PLUS SIGN character (+) between the E and the number if there is one). If there is no E, then the
  exponent is treated as zero.</p>

  <p class=note>The Infinity and Not-a-Number (NaN) values are not <a href=#valid-floating-point-number id=floating-point-numbers:valid-floating-point-number-2>valid floating-point numbers</a>.</p>

  

  <p class=note>The <a href=#valid-floating-point-number id=floating-point-numbers:valid-floating-point-number-3>valid floating-point number</a> concept is typically only used to
  restrict what is allowed for authors, while the user agent requirements use the <a href=#rules-for-parsing-floating-point-number-values id=floating-point-numbers:rules-for-parsing-floating-point-number-values>rules for
  parsing floating-point number values</a> below (e.g., the <code id=floating-point-numbers:attr-progress-max><a href=#attr-progress-max>max</a></code> attribute of the <code id=floating-point-numbers:the-progress-element><a href=#the-progress-element>progress</a></code> element). However, in
  some cases the user agent requirements include checking if a string is a <a href=#valid-floating-point-number id=floating-point-numbers:valid-floating-point-number-4>valid
  floating-point number</a> (e.g., the <a href=#value-sanitization-algorithm id=floating-point-numbers:value-sanitization-algorithm>value sanitization algorithm</a> for the <a href="#number-state-(type=number)" id="floating-point-numbers:number-state-(type=number)">Number</a> state of the <code id=floating-point-numbers:the-input-element><a href=#the-input-element>input</a></code> element, or the
  <a href=#parse-a-srcset-attribute id=floating-point-numbers:parse-a-srcset-attribute>parse a srcset attribute</a> algorithm).</p>

  <p>The <dfn id=best-representation-of-the-number-as-a-floating-point-number>best
  representation of the number <var>n</var> as a floating-point number</dfn> is the string
  obtained from running <a id=floating-point-numbers:tostring href=https://tc39.es/ecma262/#sec-tostring data-x-internal=tostring>ToString</a>(<var>n</var>). The abstract operation
  <a id=floating-point-numbers:tostring-2 href=https://tc39.es/ecma262/#sec-tostring data-x-internal=tostring>ToString</a> is not uniquely determined. When there are multiple possible strings that
  could be obtained from <a id=floating-point-numbers:tostring-3 href=https://tc39.es/ecma262/#sec-tostring data-x-internal=tostring>ToString</a> for a particular value, the user agent must always
  return the same string for that value (though it may differ from the value used by other user
  agents).</p>

  <p>The <dfn id=rules-for-parsing-floating-point-number-values data-export="">rules for parsing floating-point number values</dfn> are as given in the
  following algorithm. This algorithm must be aborted at the first step that returns something.
  This algorithm will return either a number or an error.</p>

  <ol><li><p>Let <var>input</var> be the string being parsed.<li><p>Let <var>position</var> be a pointer into <var>input</var>, initially pointing at the
   start of the string.<li><p>Let <var>value</var> have the value 1.<li><p>Let <var>divisor</var> have the value 1.<li><p>Let <var>exponent</var> have the value 1.<li><p><a id=floating-point-numbers:skip-ascii-whitespace href=https://infra.spec.whatwg.org/#skip-ascii-whitespace data-x-internal=skip-ascii-whitespace>Skip ASCII whitespace</a> within <var>input</var> given
   <var>position</var>.<li><p>If <var>position</var> is past the end of <var>input</var>, return an error.<li>
    <p>If the character indicated by <var>position</var> is a U+002D HYPHEN-MINUS character (-):</p>

    <ol><li>Change <var>value</var> and <var>divisor</var> to −1.<li>Advance <var>position</var> to the next character.<li>If <var>position</var> is past the end of <var>input</var>, return an error.</ol>

    <p>Otherwise, if the character indicated by <var>position</var> (the first character) is a
    U+002B PLUS SIGN character (+):</p>

    <ol><li>Advance <var>position</var> to the next character. (The "<code>+</code>"
     is ignored, but it is not conforming.)<li>If <var>position</var> is past the end of <var>input</var>, return an error.</ol>
   <li><p>If the character indicated by <var>position</var> is a U+002E FULL STOP (.), and that is
   not the last character in <var>input</var>, and the character after the character indicated by
   <var>position</var> is an <a href=https://infra.spec.whatwg.org/#ascii-digit id=floating-point-numbers:ascii-digits-4 data-x-internal=ascii-digits>ASCII digit</a>, then set
   <var>value</var> to zero and jump to the step labeled <i>fraction</i>.</p> 

   <li><p>If the character indicated by <var>position</var> is not an <a href=https://infra.spec.whatwg.org/#ascii-digit id=floating-point-numbers:ascii-digits-5 data-x-internal=ascii-digits>ASCII digit</a>, then return an error.<li><p><a id=floating-point-numbers:collect-a-sequence-of-code-points href=https://infra.spec.whatwg.org/#collect-a-sequence-of-code-points data-x-internal=collect-a-sequence-of-code-points>Collect a sequence of code points</a> that are <a id=floating-point-numbers:ascii-digits-6 href=https://infra.spec.whatwg.org/#ascii-digit data-x-internal=ascii-digits>ASCII digits</a> from
   <var>input</var> given <var>position</var>, and interpret the resulting sequence as a base-ten
   integer. Multiply <var>value</var> by that integer.<li>If <var>position</var> is past the end of <var>input</var>, jump to the step labeled
   <i>conversion</i>.<li><p><i>Fraction</i>: If the character indicated by <var>position</var> is a U+002E
   FULL STOP (.), run these substeps:</p>

    <ol><li><p>Advance <var>position</var> to the next character.<li><p>If <var>position</var> is past the end of <var>input</var>, or if the character
     indicated by <var>position</var> is not an <a href=https://infra.spec.whatwg.org/#ascii-digit id=floating-point-numbers:ascii-digits-7 data-x-internal=ascii-digits>ASCII digit</a>,
     U+0065 LATIN SMALL LETTER E (e), or U+0045 LATIN CAPITAL LETTER E (E), then jump to the step
     labeled <i>conversion</i>.<li><p>If the character indicated by <var>position</var> is a U+0065 LATIN SMALL
     LETTER E character (e) or a U+0045 LATIN CAPITAL LETTER E character (E), skip the remainder of
     these substeps.</p>

     <li><p><i>Fraction loop</i>: Multiply <var>divisor</var> by ten.<li>Add the value of the character indicated by <var>position</var>, interpreted as a
     base-ten digit (0..9) and divided by <var>divisor</var>, to <var>value</var>.<li><p>Advance <var>position</var> to the next character.<li><p>If <var>position</var> is past the end of <var>input</var>, then jump to the step
     labeled <i>conversion</i>.<li><p>If the character indicated by <var>position</var> is an <a href=https://infra.spec.whatwg.org/#ascii-digit id=floating-point-numbers:ascii-digits-8 data-x-internal=ascii-digits>ASCII digit</a>, jump back to the step labeled <i>fraction loop</i> in these
     substeps.</ol>
   <li><p>If the character indicated by <var>position</var> is U+0065 (e) or a U+0045 (E), then:</p>

    <ol><li><p>Advance <var>position</var> to the next character.<li><p>If <var>position</var> is past the end of <var>input</var>, then jump to the step
     labeled <i>conversion</i>.<li>
      <p>If the character indicated by <var>position</var> is a U+002D HYPHEN-MINUS character
      (-):</p>

      <ol><li>Change <var>exponent</var> to −1.<li>Advance <var>position</var> to the next character.<li><p>If <var>position</var> is past the end of <var>input</var>, then jump to the step
       labeled <i>conversion</i>.</ol>

      <p>Otherwise, if the character indicated by <var>position</var> is a U+002B PLUS SIGN
      character (+):</p>

      <ol><li>Advance <var>position</var> to the next character.<li><p>If <var>position</var> is past the end of <var>input</var>, then jump to the step
       labeled <i>conversion</i>.</ol>
     <li><p>If the character indicated by <var>position</var> is not an <a href=https://infra.spec.whatwg.org/#ascii-digit id=floating-point-numbers:ascii-digits-9 data-x-internal=ascii-digits>ASCII digit</a>, then jump to the step labeled <i>conversion</i>.<li><p><a id=floating-point-numbers:collect-a-sequence-of-code-points-2 href=https://infra.spec.whatwg.org/#collect-a-sequence-of-code-points data-x-internal=collect-a-sequence-of-code-points>Collect a sequence of code points</a> that are <a id=floating-point-numbers:ascii-digits-10 href=https://infra.spec.whatwg.org/#ascii-digit data-x-internal=ascii-digits>ASCII digits</a> from
     <var>input</var> given <var>position</var>, and interpret the resulting sequence as a base-ten
     integer. Multiply <var>exponent</var> by that integer.<li><p>Multiply <var>value</var> by ten raised to the <var>exponent</var>th power.</ol>
   <li><p><i>Conversion</i>: Let <var>S</var> be the set of finite IEEE 754 double-precision
   floating-point values except −0, but with two special values added: 2<sup>1024</sup> and −2<sup>1024</sup>.<li><p>Let <var>rounded-value</var> be the number in <var>S</var> that is closest to
   <var>value</var>, selecting the number with an even significand if there are two equally close
   values. (The two special values 2<sup>1024</sup> and −2<sup>1024</sup>
   are considered to have even significands for this purpose.)<li><p>If <var>rounded-value</var> is 2<sup>1024</sup> or −2<sup>1024</sup>, return an error.<li><p>Return <var>rounded-value</var>.</ol>

  



  <h5 id=percentages-and-dimensions><span class=secno>2.4.4.4</span> Percentages and lengths<a href=#percentages-and-dimensions class=self-link></a></h5>

  <p>The <dfn id=rules-for-parsing-dimension-values>rules for parsing dimension values</dfn> are as given in the following algorithm. When
  invoked, the steps must be followed in the order given, aborting at the first step that returns a
  value. This algorithm will return either a number greater than or equal to 0.0, or failure; if a
  number is returned, then it is further categorized as either a percentage or a length.</p>

  <ol><li><p>Let <var>input</var> be the string being parsed.<li><p>Let <var>position</var> be a <a id=percentages-and-dimensions:position-variable href=https://infra.spec.whatwg.org/#string-position-variable data-x-internal=position-variable>position variable</a> for <var>input</var>,
   initially pointing at the start of <var>input</var>.<li><p><a id=percentages-and-dimensions:skip-ascii-whitespace href=https://infra.spec.whatwg.org/#skip-ascii-whitespace data-x-internal=skip-ascii-whitespace>Skip ASCII whitespace</a> within <var>input</var> given
   <var>position</var>.<li><p>If <var>position</var> is past the end of <var>input</var> or the code point at
   <var>position</var> within <var>input</var> is not an <a href=https://infra.spec.whatwg.org/#ascii-digit id=percentages-and-dimensions:ascii-digits data-x-internal=ascii-digits>ASCII
   digit</a>, then return failure.<li><p><a id=percentages-and-dimensions:collect-a-sequence-of-code-points href=https://infra.spec.whatwg.org/#collect-a-sequence-of-code-points data-x-internal=collect-a-sequence-of-code-points>Collect a sequence of code points</a> that are <a id=percentages-and-dimensions:ascii-digits-2 href=https://infra.spec.whatwg.org/#ascii-digit data-x-internal=ascii-digits>ASCII digits</a> from
   <var>input</var> given <var>position</var>, and interpret the resulting sequence as a base-ten
   integer. Let <var>value</var> be that number.<li><p>If <var>position</var> is past the end of <var>input</var>, then return <var>value</var>
   as a length.<li>
    <p>If the code point at <var>position</var> within <var>input</var> is U+002E (.), then:</p>

    <ol><li><p>Advance <var>position</var> by 1.<li><p>If <var>position</var> is past the end of <var>input</var> or the code point at
     <var>position</var> within <var>input</var> is not an <a href=https://infra.spec.whatwg.org/#ascii-digit id=percentages-and-dimensions:ascii-digits-3 data-x-internal=ascii-digits>ASCII
     digit</a>, then return the <a href=#current-dimension-value id=percentages-and-dimensions:current-dimension-value>current dimension value</a> with <var>value</var>,
     <var>input</var>, and <var>position</var>.<li><p>Let <var>divisor</var> have the value 1.<li>
      <p>While true:</p>

      <ol><li><p>Multiply <var>divisor</var> by ten.<li><p>Add the value of the code point at <var>position</var> within <var>input</var>,
       interpreted as a base-ten digit (0..9) and divided by <var>divisor</var>, to
       <var>value</var>.<li><p>Advance <var>position</var> by 1.<li><p>If <var>position</var> is past the end of <var>input</var>, then return
       <var>value</var> as a length.<li><p>If the code point at <var>position</var> within <var>input</var> is not an <a href=https://infra.spec.whatwg.org/#ascii-digit id=percentages-and-dimensions:ascii-digits-4 data-x-internal=ascii-digits>ASCII digit</a>, then <a id=percentages-and-dimensions:break href=https://infra.spec.whatwg.org/#iteration-break data-x-internal=break>break</a>.</ol>
     </ol>
   <li><p>Return the <a href=#current-dimension-value id=percentages-and-dimensions:current-dimension-value-2>current dimension value</a> with <var>value</var>, <var>input</var>,
   and <var>position</var>.</ol>

  <p>The <dfn id=current-dimension-value>current dimension value</dfn>, given <var>value</var>, <var>input</var>, and
  <var>position</var>, is determined as follows:</p>

  <ol><li><p>If <var>position</var> is past the end of <var>input</var>, then return <var>value</var>
   as a length.<li><p>If the code point at <var>position</var> within <var>input</var> is U+0025 (%), then
   return <var>value</var> as a percentage.<li><p>Return <var>value</var> as a length.</ol>

  <h5 id=non-zero-percentages-and-lengths><span class=secno>2.4.4.5</span> Non-zero percentages and lengths<a href=#non-zero-percentages-and-lengths class=self-link></a></h5>

  <p>The <dfn id=rules-for-parsing-non-zero-dimension-values>rules for parsing nonzero dimension
  values</dfn> are as given in the following algorithm. When invoked, the steps must be followed in
  the order given, aborting at the first step that returns a value. This algorithm will return
  either a number greater than 0.0, or an error; if a number is returned, then it is further
  categorized as either a percentage or a length.</p>

  <ol><li><p>Let <var>input</var> be the string being parsed.<li><p>Let <var>value</var> be the result of parsing <var>input</var> using the <a href=#rules-for-parsing-dimension-values id=non-zero-percentages-and-lengths:rules-for-parsing-dimension-values>rules for
   parsing dimension values</a>.<li><p>If <var>value</var> is an error, return an error.<li><p>If <var>value</var> is zero, return an error.<li><p>If <var>value</var> is a percentage, return <var>value</var> as a percentage.<li><p>Return <var>value</var> as a length.</ol>

  


  <h5 id=lists-of-floating-point-numbers><span class=secno>2.4.4.6</span> Lists of floating-point numbers<a href=#lists-of-floating-point-numbers class=self-link></a></h5>

  <p>A <dfn id=valid-list-of-floating-point-numbers>valid list of floating-point numbers</dfn> is a number of <a href=#valid-floating-point-number id=lists-of-floating-point-numbers:valid-floating-point-number>valid floating-point numbers</a> separated by U+002C COMMA characters,
  with no other characters (e.g. no <a id=lists-of-floating-point-numbers:space-characters href=https://infra.spec.whatwg.org/#ascii-whitespace data-x-internal=space-characters>ASCII whitespace</a>). In addition, there might be
  restrictions on the number of floating-point numbers that can be given, or on the range of values
  allowed.</p>

  

  <p>The <dfn id=rules-for-parsing-a-list-of-floating-point-numbers>rules for parsing a list of floating-point numbers</dfn> are as follows:</p>

  <ol><li><p>Let <var>input</var> be the string being parsed.<li><p>Let <var>position</var> be a pointer into <var>input</var>, initially pointing at the
   start of the string.<li><p>Let <var>numbers</var> be an initially empty list of floating-point numbers. This list
   will be the result of this algorithm.<li><p><a id=lists-of-floating-point-numbers:collect-a-sequence-of-code-points href=https://infra.spec.whatwg.org/#collect-a-sequence-of-code-points data-x-internal=collect-a-sequence-of-code-points>Collect a sequence of code points</a> that are <a id=lists-of-floating-point-numbers:space-characters-2 href=https://infra.spec.whatwg.org/#ascii-whitespace data-x-internal=space-characters>ASCII whitespace</a>,
   U+002C COMMA, or U+003B SEMICOLON characters from <var>input</var> given <var>position</var>.
   This skips past any leading delimiters.<li><p>While <var>position</var> is not past the end of <var>input</var>:</p>

    <ol><li><p><a id=lists-of-floating-point-numbers:collect-a-sequence-of-code-points-2 href=https://infra.spec.whatwg.org/#collect-a-sequence-of-code-points data-x-internal=collect-a-sequence-of-code-points>Collect a sequence of code points</a> that are not <a id=lists-of-floating-point-numbers:space-characters-3 href=https://infra.spec.whatwg.org/#ascii-whitespace data-x-internal=space-characters>ASCII
     whitespace</a>, U+002C COMMA, U+003B SEMICOLON, <a id=lists-of-floating-point-numbers:ascii-digits href=https://infra.spec.whatwg.org/#ascii-digit data-x-internal=ascii-digits>ASCII digits</a>, U+002E FULL STOP,
     or U+002D HYPHEN-MINUS characters from <var>input</var> given <var>position</var>. This skips
     past leading garbage.<li><p><a id=lists-of-floating-point-numbers:collect-a-sequence-of-code-points-3 href=https://infra.spec.whatwg.org/#collect-a-sequence-of-code-points data-x-internal=collect-a-sequence-of-code-points>Collect a sequence of code points</a> that are not <a id=lists-of-floating-point-numbers:space-characters-4 href=https://infra.spec.whatwg.org/#ascii-whitespace data-x-internal=space-characters>ASCII
     whitespace</a>, U+002C COMMA, or U+003B SEMICOLON characters from <var>input</var> given
     <var>position</var>, and let <var>unparsed number</var> be the result.<li><p>Let <var>number</var> be the result of parsing <var>unparsed number</var> using the
     <a href=#rules-for-parsing-floating-point-number-values id=lists-of-floating-point-numbers:rules-for-parsing-floating-point-number-values>rules for parsing floating-point number values</a>.<li><p>If <var>number</var> is an error, set <var>number</var> to zero.<li><p>Append <var>number</var> to <var>numbers</var>.<li><p><a id=lists-of-floating-point-numbers:collect-a-sequence-of-code-points-4 href=https://infra.spec.whatwg.org/#collect-a-sequence-of-code-points data-x-internal=collect-a-sequence-of-code-points>Collect a sequence of code points</a> that are <a id=lists-of-floating-point-numbers:space-characters-5 href=https://infra.spec.whatwg.org/#ascii-whitespace data-x-internal=space-characters>ASCII whitespace</a>,
     U+002C COMMA, or U+003B SEMICOLON characters from <var>input</var> given <var>position</var>.
     This skips past the delimiter.</ol>
   <li><p>Return <var>numbers</var>.</ol>

  

  

  <h5 id=lists-of-dimensions><span class=secno>2.4.4.7</span> Lists of dimensions<a href=#lists-of-dimensions class=self-link></a></h5>

  

  <p>The <dfn id=rules-for-parsing-a-list-of-dimensions>rules for parsing a list of dimensions</dfn> are as follows. These rules return a list
  of zero or more pairs consisting of a number and a unit, the unit being one of <i>percentage</i>,
  <i>relative</i>, and <i>absolute</i>.</p>

  <ol><li><p>Let <var>raw input</var> be the string being parsed.<li><p>If the last character in <var>raw input</var> is a U+002C COMMA character (,),
   then remove that character from <var>raw input</var>.<li><p><a href=https://infra.spec.whatwg.org/#split-on-commas id=lists-of-dimensions:split-a-string-on-commas data-x-internal=split-a-string-on-commas>Split the string <var>raw input</var> on
   commas</a>. Let <var>raw tokens</var> be the resulting list of tokens.<li><p>Let <var>result</var> be an empty list of number/unit pairs.<li>
    <p>For each token in <var>raw tokens</var>, run the following substeps:</p>

    <ol><li><p>Let <var>input</var> be the token.<li><p>Let <var>position</var> be a pointer into <var>input</var>,
     initially pointing at the start of the string.<li><p>Let <var>value</var> be the number 0.<li><p>Let <var>unit</var> be <i>absolute</i>.<li><p>If <var>position</var> is past the end of <var>input</var>, set <var>unit</var> to <i>relative</i> and jump to the last substep.<li><p>If the character at <var>position</var> is an <a href=https://infra.spec.whatwg.org/#ascii-digit id=lists-of-dimensions:ascii-digits data-x-internal=ascii-digits>ASCII digit</a>, <a id=lists-of-dimensions:collect-a-sequence-of-code-points href=https://infra.spec.whatwg.org/#collect-a-sequence-of-code-points data-x-internal=collect-a-sequence-of-code-points>collect a sequence of code points</a> that are <a id=lists-of-dimensions:ascii-digits-2 href=https://infra.spec.whatwg.org/#ascii-digit data-x-internal=ascii-digits>ASCII
     digits</a> from <var>input</var> given <var>position</var>, interpret the resulting sequence
     as an integer in base ten, and increment <var>value</var> by that integer.<li>
      <p>If the character at <var>position</var> is U+002E (.), then:</p>

      <ol><li><p><a id=lists-of-dimensions:collect-a-sequence-of-code-points-2 href=https://infra.spec.whatwg.org/#collect-a-sequence-of-code-points data-x-internal=collect-a-sequence-of-code-points>Collect a sequence of code points</a> consisting of <a id=lists-of-dimensions:space-characters href=https://infra.spec.whatwg.org/#ascii-whitespace data-x-internal=space-characters>ASCII
       whitespace</a> and <a id=lists-of-dimensions:ascii-digits-3 href=https://infra.spec.whatwg.org/#ascii-digit data-x-internal=ascii-digits>ASCII digits</a> from <var>input</var> given
       <var>position</var>. Let <var>s</var> be the resulting sequence.<li><p>Remove all <a id=lists-of-dimensions:space-characters-2 href=https://infra.spec.whatwg.org/#ascii-whitespace data-x-internal=space-characters>ASCII whitespace</a> in <var>s</var>.<li>
        <p>If <var>s</var> is not the empty string, then:</p>

        <ol><li><p>Let <var>length</var> be the number of characters in <var>s</var> (after the spaces were removed).<li><p>Let <var>fraction</var> be the result of interpreting <var>s</var> as a base-ten integer, and then dividing that number by <span>10<sup><var>length</var></sup></span>.<li><p>Increment <var>value</var> by <var>fraction</var>.</ol>
       </ol>
     <li><p><a id=lists-of-dimensions:skip-ascii-whitespace href=https://infra.spec.whatwg.org/#skip-ascii-whitespace data-x-internal=skip-ascii-whitespace>Skip ASCII whitespace</a> within <var>input</var> given
     <var>position</var>.<li>
      <p>If the character at <var>position</var> is a U+0025 PERCENT SIGN character (%),
      then set <var>unit</var> to <i>percentage</i>.</p>

      <p>Otherwise, if the character at <var>position</var> is a U+002A ASTERISK character
      (*), then set <var>unit</var> to <i>relative</i>.</p>
     <li><p>Add an entry to <var>result</var> consisting of the number given by <var>value</var> and the unit given by <var>unit</var>.</ol>
   <li><p>Return the list <var>result</var>.</ol>

  


  <h4 id=dates-and-times><span class=secno>2.4.5</span> Dates and times<a href=#dates-and-times class=self-link></a></h4>

  <p>In the algorithms below, the <dfn id=number-of-days-in-month-month-of-year-year>number of days in month <var>month</var> of year
  <var>year</var></dfn> is: <em>31</em> if <var>month</var> is 1, 3, 5, 7, 8,
  10, or 12; <em>30</em> if <var>month</var> is 4, 6, 9, or 11; <em>29</em> if <var>month</var> is 2 and <var>year</var> is a number divisible by 400, or if <var>year</var> is a number divisible by 4 but not by 100; and <em>28</em> otherwise. This
  takes into account leap years in the Gregorian calendar. <a href=#refsGREGORIAN>[GREGORIAN]</a></p>

  <p>When <a id=dates-and-times:ascii-digits href=https://infra.spec.whatwg.org/#ascii-digit data-x-internal=ascii-digits>ASCII digits</a> are used in the date and time syntaxes defined in this section,
  they express numbers in base ten.</p>

  

  <p class=note>While the formats described here are intended to be subsets of the corresponding
  ISO8601 formats, this specification defines parsing rules in much more detail than ISO8601.
  Implementors are therefore encouraged to carefully examine any date parsing libraries before using
  them to implement the parsing rules described below; ISO8601 libraries might not parse dates and
  times in exactly the same manner. <a href=#refsISO8601>[ISO8601]</a></p>

  

  <p>Where this specification refers to the <dfn id=proleptic-gregorian-calendar>proleptic Gregorian calendar</dfn>, it means the
  modern Gregorian calendar, extrapolated backwards to year 1. A date in the <a href=#proleptic-gregorian-calendar id=dates-and-times:proleptic-gregorian-calendar>proleptic
  Gregorian calendar</a>, sometimes explicitly referred to as a <dfn id=proleptic-gregorian-date>proleptic-Gregorian
  date</dfn>, is one that is described using that calendar even if that calendar was not in use at
  the time (or place) in question. <a href=#refsGREGORIAN>[GREGORIAN]</a></p>

  <p class=note>The use of the Gregorian calendar as the wire format in this specification is an
  arbitrary choice resulting from the cultural biases of those involved in the decision. See also
  the section discussing <a href=#input-author-notes>date, time, and number formats</a> in forms
  (for authors), <a href=#input-impl-notes>implementation notes regarding
  localization of form controls</a>, and the <code id=dates-and-times:the-time-element><a href=#the-time-element>time</a></code> element.</p>


  <h5 id=months><span class=secno>2.4.5.1</span> Months<a href=#months class=self-link></a></h5>

  <p>A <dfn id=concept-month>month</dfn> consists of a specific <a href=#proleptic-gregorian-date id=months:proleptic-gregorian-date>proleptic-Gregorian
  date</a> with no time-zone information and no date information beyond a year and a month. <a href=#refsGREGORIAN>[GREGORIAN]</a></p>

  <p>A string is a <dfn id=valid-month-string>valid month string</dfn> representing a year <var>year</var> and
  month <var>month</var> if it consists of the following components in the given order:</p>

  <ol><li>Four or more <a id=months:ascii-digits href=https://infra.spec.whatwg.org/#ascii-digit data-x-internal=ascii-digits>ASCII digits</a>, representing <var>year</var>, where <var>year</var> > 0<li>A U+002D HYPHEN-MINUS character (-)<li>Two <a id=months:ascii-digits-2 href=https://infra.spec.whatwg.org/#ascii-digit data-x-internal=ascii-digits>ASCII digits</a>, representing the month <var>month</var>, in the range
   1 ≤ <var>month</var> ≤ 12</ol>

  

  <p>The rules to <dfn id=parse-a-month-string>parse a month string</dfn> are as follows. This will return either a year and
  month, or nothing. If at any point the algorithm says that it "fails", this means that it is
  aborted at that point and returns nothing.</p>

  <ol><li><p>Let <var>input</var> be the string being parsed.<li><p>Let <var>position</var> be a pointer into <var>input</var>, initially
   pointing at the start of the string.<li><p><a href=#parse-a-month-component id=months:parse-a-month-component>Parse a month component</a> to obtain <var>year</var> and <var>month</var>. If this returns nothing, then fail.</p>

   <li><p>If <var>position</var> is <em>not</em> beyond the
   end of <var>input</var>, then fail.<li><p>Return <var>year</var> and <var>month</var>.</ol>

  <p>The rules to <dfn id=parse-a-month-component>parse a month component</dfn>, given an <var>input</var> string and
  a <var>position</var>, are as follows. This will return either a year and a month, or
  nothing. If at any point the algorithm says that it "fails", this means that it is aborted at that
  point and returns nothing.</p>

  <ol><li><p><a id=months:collect-a-sequence-of-code-points href=https://infra.spec.whatwg.org/#collect-a-sequence-of-code-points data-x-internal=collect-a-sequence-of-code-points>Collect a sequence of code points</a> that are <a id=months:ascii-digits-3 href=https://infra.spec.whatwg.org/#ascii-digit data-x-internal=ascii-digits>ASCII digits</a> from
   <var>input</var> given <var>position</var>. If the collected sequence is not at least four
   characters long, then fail. Otherwise, interpret the resulting sequence as a base-ten integer.
   Let that number be the <var>year</var>.<li><p>If <var>year</var> is not a number greater than zero, then fail.<li><p>If <var>position</var> is beyond the end of <var>input</var> or if the
   character at <var>position</var> is not a U+002D HYPHEN-MINUS character, then fail.
   Otherwise, move <var>position</var> forwards one character.<li><p><a id=months:collect-a-sequence-of-code-points-2 href=https://infra.spec.whatwg.org/#collect-a-sequence-of-code-points data-x-internal=collect-a-sequence-of-code-points>Collect a sequence of code points</a> that are <a id=months:ascii-digits-4 href=https://infra.spec.whatwg.org/#ascii-digit data-x-internal=ascii-digits>ASCII digits</a> from
   <var>input</var> given <var>position</var>. If the collected sequence is not exactly two
   characters long, then fail. Otherwise, interpret the resulting sequence as a base-ten integer.
   Let that number be the <var>month</var>.<li><p>If <var>month</var> is not a number in the range 1 ≤ <var>month</var> ≤ 12, then fail.<li><p>Return <var>year</var> and <var>month</var>.</ol>

  


  <h5 id=dates><span class=secno>2.4.5.2</span> Dates<a href=#dates class=self-link></a></h5>

  <p>A <dfn id=concept-date>date</dfn> consists of a specific <a href=#proleptic-gregorian-date id=dates:proleptic-gregorian-date>proleptic-Gregorian
  date</a> with no time-zone information, consisting of a year, a month, and a day. <a href=#refsGREGORIAN>[GREGORIAN]</a></p>

  <p>A string is a <dfn id=valid-date-string>valid date string</dfn> representing a year <var>year</var>, month
  <var>month</var>, and day <var>day</var> if it consists of the following
  components in the given order:</p>

  <ol><li>A <a href=#valid-month-string id=dates:valid-month-string>valid month string</a>, representing <var>year</var> and <var>month</var><li>A U+002D HYPHEN-MINUS character (-)<li>Two <a id=dates:ascii-digits href=https://infra.spec.whatwg.org/#ascii-digit data-x-internal=ascii-digits>ASCII digits</a>, representing <var>day</var>, in the range
   1 ≤ <var>day</var> ≤ <var>maxday</var> where <var>maxday</var> is the <a href=#number-of-days-in-month-month-of-year-year id=dates:number-of-days-in-month-month-of-year-year>number of
   days in the month <var>month</var> and year <var>year</var></a></ol>

  

  <p>The rules to <dfn id=parse-a-date-string>parse a date string</dfn> are as follows. This will return either a date, or
  nothing. If at any point the algorithm says that it "fails", this means that it is aborted at that
  point and returns nothing.</p>

  <ol><li><p>Let <var>input</var> be the string being parsed.<li><p>Let <var>position</var> be a pointer into <var>input</var>, initially
   pointing at the start of the string.<li><p><a href=#parse-a-date-component id=dates:parse-a-date-component>Parse a date component</a> to obtain <var>year</var>, <var>month</var>, and <var>day</var>. If this returns nothing, then fail.</p>

   <li><p>If <var>position</var> is <em>not</em> beyond the end of <var>input</var>, then fail.<li><p>Let <var>date</var> be the date with year <var>year</var>, month <var>month</var>, and day <var>day</var>.<li><p>Return <var>date</var>.</ol>

  <p>The rules to <dfn id=parse-a-date-component>parse a date component</dfn>, given an <var>input</var> string and a
  <var>position</var>, are as follows. This will return either a year, a month, and a day,
  or nothing. If at any point the algorithm says that it "fails", this means that it is aborted at
  that point and returns nothing.</p>

  <ol><li><p><a href=#parse-a-month-component id=dates:parse-a-month-component>Parse a month component</a> to obtain <var>year</var> and <var>month</var>. If this returns nothing, then fail.<li><p>Let <var>maxday</var> be the <a href=#number-of-days-in-month-month-of-year-year id=dates:number-of-days-in-month-month-of-year-year-2>number of days in month <var>month</var> of year <var>year</var></a>.<li><p>If <var>position</var> is beyond the end of <var>input</var> or if the
   character at <var>position</var> is not a U+002D HYPHEN-MINUS character, then fail.
   Otherwise, move <var>position</var> forwards one character.<li><p><a id=dates:collect-a-sequence-of-code-points href=https://infra.spec.whatwg.org/#collect-a-sequence-of-code-points data-x-internal=collect-a-sequence-of-code-points>Collect a sequence of code points</a> that are <a id=dates:ascii-digits-2 href=https://infra.spec.whatwg.org/#ascii-digit data-x-internal=ascii-digits>ASCII digits</a> from
   <var>input</var> given <var>position</var>. If the collected sequence is not exactly two
   characters long, then fail. Otherwise, interpret the resulting sequence as a base-ten integer.
   Let that number be the <var>day</var>.<li><p>If <var>day</var> is not a number in the range 1 ≤ <var>day</var> ≤ <var>maxday</var>, then fail.<li><p>Return <var>year</var>, <var>month</var>, and <var>day</var>.</ol>

  


  <h5 id=yearless-dates><span class=secno>2.4.5.3</span> Yearless dates<a href=#yearless-dates class=self-link></a></h5>

  <p>A <dfn id=concept-yearless-date>yearless date</dfn> consists of a Gregorian month and a
  day within that month, but with no associated year. <a href=#refsGREGORIAN>[GREGORIAN]</a></p>

  <p>A string is a <dfn id=valid-yearless-date-string>valid yearless date string</dfn> representing a month <var>month</var> and a day <var>day</var> if it consists of the following components
  in the given order:</p>

  <ol><li>Optionally, two U+002D HYPHEN-MINUS characters (-)<li>Two <a id=yearless-dates:ascii-digits href=https://infra.spec.whatwg.org/#ascii-digit data-x-internal=ascii-digits>ASCII digits</a>, representing the month <var>month</var>, in the range
   1 ≤ <var>month</var> ≤ 12<li>A U+002D HYPHEN-MINUS character (-)<li>Two <a id=yearless-dates:ascii-digits-2 href=https://infra.spec.whatwg.org/#ascii-digit data-x-internal=ascii-digits>ASCII digits</a>, representing <var>day</var>, in the range
   1 ≤ <var>day</var> ≤ <var>maxday</var> where <var>maxday</var> is the <a href=#number-of-days-in-month-month-of-year-year id=yearless-dates:number-of-days-in-month-month-of-year-year>number of
   days</a> in the month <var>month</var> and any arbitrary leap year (e.g. 4 or
   2000)</ol>

  <p class=note>In other words, if the <var>month</var> is "<code>02</code>",
  meaning February, then the day can be 29, as if the year was a leap year.</p>

  

  <p>The rules to <dfn id=parse-a-yearless-date-string>parse a yearless date string</dfn> are as follows. This will return either a
  month and a day, or nothing. If at any point the algorithm says that it "fails", this means that
  it is aborted at that point and returns nothing.</p>

  <ol><li><p>Let <var>input</var> be the string being parsed.<li><p>Let <var>position</var> be a pointer into <var>input</var>, initially
   pointing at the start of the string.<li><p><a href=#parse-a-yearless-date-component id=yearless-dates:parse-a-yearless-date-component>Parse a yearless date component</a> to obtain <var>month</var> and <var>day</var>. If this returns nothing, then fail.</p>

   <li><p>If <var>position</var> is <em>not</em> beyond the end of <var>input</var>, then fail.<li><p>Return <var>month</var> and <var>day</var>.</ol>

  <p>The rules to <dfn id=parse-a-yearless-date-component>parse a yearless date component</dfn>, given an <var>input</var>
  string and a <var>position</var>, are as follows. This will return either a month and a
  day, or nothing. If at any point the algorithm says that it "fails", this means that it is aborted
  at that point and returns nothing.</p>

  <ol><li><p><a id=yearless-dates:collect-a-sequence-of-code-points href=https://infra.spec.whatwg.org/#collect-a-sequence-of-code-points data-x-internal=collect-a-sequence-of-code-points>Collect a sequence of code points</a> that are U+002D HYPHEN-MINUS characters (-)
   from <var>input</var> given <var>position</var>. If the collected sequence is not exactly zero or
   two characters long, then fail.<li><p><a id=yearless-dates:collect-a-sequence-of-code-points-2 href=https://infra.spec.whatwg.org/#collect-a-sequence-of-code-points data-x-internal=collect-a-sequence-of-code-points>Collect a sequence of code points</a> that are <a id=yearless-dates:ascii-digits-3 href=https://infra.spec.whatwg.org/#ascii-digit data-x-internal=ascii-digits>ASCII digits</a> from
   <var>input</var> given <var>position</var>. If the collected sequence is not exactly two
   characters long, then fail. Otherwise, interpret the resulting sequence as a base-ten integer.
   Let that number be the <var>month</var>.<li><p>If <var>month</var> is not a number in the range 1 ≤ <var>month</var> ≤ 12, then fail.<li><p>Let <var>maxday</var> be the <a href=#number-of-days-in-month-month-of-year-year id=yearless-dates:number-of-days-in-month-month-of-year-year-2>number of days</a> in month <var>month</var> of any arbitrary leap year (e.g. 4
   or 2000).<li><p>If <var>position</var> is beyond the end of <var>input</var> or if the
   character at <var>position</var> is not a U+002D HYPHEN-MINUS character, then fail.
   Otherwise, move <var>position</var> forwards one character.<li><p><a id=yearless-dates:collect-a-sequence-of-code-points-3 href=https://infra.spec.whatwg.org/#collect-a-sequence-of-code-points data-x-internal=collect-a-sequence-of-code-points>Collect a sequence of code points</a> that are <a id=yearless-dates:ascii-digits-4 href=https://infra.spec.whatwg.org/#ascii-digit data-x-internal=ascii-digits>ASCII digits</a> from
   <var>input</var> given <var>position</var>. If the collected sequence is not exactly two
   characters long, then fail. Otherwise, interpret the resulting sequence as a base-ten integer.
   Let that number be the <var>day</var>.<li><p>If <var>day</var> is not a number in the range 1 ≤ <var>day</var> ≤ <var>maxday</var>, then fail.<li><p>Return <var>month</var> and <var>day</var>.</ol>

  


  <h5 id=times><span class=secno>2.4.5.4</span> Times<a href=#times class=self-link></a></h5>

  <p>A <dfn id=concept-time>time</dfn> consists of a specific time with no time-zone
  information, consisting of an hour, a minute, a second, and a fraction of a second.</p>

  <p>A string is a <dfn id=valid-time-string>valid time string</dfn> representing an hour <var>hour</var>, a
  minute <var>minute</var>, and a second <var>second</var> if it consists of the
  following components in the given order:</p>

  <ol><li>Two <a id=times:ascii-digits href=https://infra.spec.whatwg.org/#ascii-digit data-x-internal=ascii-digits>ASCII digits</a>, representing <var>hour</var>, in the range
   0 ≤ <var>hour</var> ≤ 23<li>A U+003A COLON character (:)<li>Two <a id=times:ascii-digits-2 href=https://infra.spec.whatwg.org/#ascii-digit data-x-internal=ascii-digits>ASCII digits</a>, representing <var>minute</var>, in the range
   0 ≤ <var>minute</var> ≤ 59<li>If <var>second</var> is nonzero, or optionally if <var>second</var> is
   zero:

    <ol><li>A U+003A COLON character (:)<li>Two <a id=times:ascii-digits-3 href=https://infra.spec.whatwg.org/#ascii-digit data-x-internal=ascii-digits>ASCII digits</a>, representing the integer part of <var>second</var>,
     in the range 0 ≤ <var>s</var> ≤ 59<li>If <var>second</var> is not an integer, or optionally if <var>second</var> is an integer:

      <ol><li>A U+002E FULL STOP character (.)<li>One, two, or three <a id=times:ascii-digits-4 href=https://infra.spec.whatwg.org/#ascii-digit data-x-internal=ascii-digits>ASCII digits</a>, representing the fractional part of <var>second</var></ol>
     </ol>
   </ol>

  <p class=note>The <var>second</var> component cannot be 60 or 61; leap seconds cannot
  be represented.</p>

  

  <p>The rules to <dfn id=parse-a-time-string>parse a time string</dfn> are as follows. This will return either a time, or
  nothing. If at any point the algorithm says that it "fails", this means that it is aborted at that
  point and returns nothing.</p>

  <ol><li><p>Let <var>input</var> be the string being parsed.<li><p>Let <var>position</var> be a pointer into <var>input</var>, initially
   pointing at the start of the string.<li><p><a href=#parse-a-time-component id=times:parse-a-time-component>Parse a time component</a> to obtain <var>hour</var>, <var>minute</var>, and <var>second</var>. If this returns nothing, then fail.</p>

   <li><p>If <var>position</var> is <em>not</em> beyond the end of <var>input</var>, then fail.<li><p>Let <var>time</var> be the time with hour <var>hour</var>, minute <var>minute</var>, and second <var>second</var>.<li><p>Return <var>time</var>.</ol>

  <p>The rules to <dfn id=parse-a-time-component>parse a time component</dfn>, given an <var>input</var> string and a
  <var>position</var>, are as follows. This will return either an hour, a minute, and a
  second, or nothing. If at any point the algorithm says that it "fails", this means that it is
  aborted at that point and returns nothing.</p>

  <ol><li><p><a id=times:collect-a-sequence-of-code-points href=https://infra.spec.whatwg.org/#collect-a-sequence-of-code-points data-x-internal=collect-a-sequence-of-code-points>Collect a sequence of code points</a> that are <a id=times:ascii-digits-5 href=https://infra.spec.whatwg.org/#ascii-digit data-x-internal=ascii-digits>ASCII digits</a> from
   <var>input</var> given <var>position</var>. If the collected sequence is not exactly two
   characters long, then fail. Otherwise, interpret the resulting sequence as a base-ten integer.
   Let that number be the <var>hour</var>.<li>If <var>hour</var> is not a number in the range 0 ≤ <var>hour</var> ≤ 23, then fail.<li><p>If <var>position</var> is beyond the end of <var>input</var> or if the
   character at <var>position</var> is not a U+003A COLON character, then fail. Otherwise,
   move <var>position</var> forwards one character.<li><p><a id=times:collect-a-sequence-of-code-points-2 href=https://infra.spec.whatwg.org/#collect-a-sequence-of-code-points data-x-internal=collect-a-sequence-of-code-points>Collect a sequence of code points</a> that are <a id=times:ascii-digits-6 href=https://infra.spec.whatwg.org/#ascii-digit data-x-internal=ascii-digits>ASCII digits</a> from
   <var>input</var> given <var>position</var>. If the collected sequence is not exactly two
   characters long, then fail. Otherwise, interpret the resulting sequence as a base-ten integer.
   Let that number be the <var>minute</var>.<li>If <var>minute</var> is not a number in the range 0 ≤ <var>minute</var> ≤ 59, then fail.<li><p>Let <var>second</var> be 0.<li>
    <p>If <var>position</var> is not beyond the end of <var>input</var> and the character at
    <var>position</var> is U+003A (:), then:</p>

    <ol><li><p>Advance <var>position</var> to the next character in <var>input</var>.<li><p>If <var>position</var> is beyond the end of <var>input</var>, or at
     the last character in <var>input</var>, or if the next <em>two</em> characters in <var>input</var> starting at <var>position</var> are not both <a id=times:ascii-digits-7 href=https://infra.spec.whatwg.org/#ascii-digit data-x-internal=ascii-digits>ASCII
     digits</a>, then fail.<li><p><a id=times:collect-a-sequence-of-code-points-3 href=https://infra.spec.whatwg.org/#collect-a-sequence-of-code-points data-x-internal=collect-a-sequence-of-code-points>Collect a sequence of code points</a> that are either <a id=times:ascii-digits-8 href=https://infra.spec.whatwg.org/#ascii-digit data-x-internal=ascii-digits>ASCII digits</a>
     or U+002E FULL STOP characters from <var>input</var> given <var>position</var>. If the
     collected sequence is three characters long, or if it is longer than three characters long and
     the third character is not a U+002E FULL STOP character, or if it has more than one U+002E FULL
     STOP character, then fail. Otherwise, interpret the resulting sequence as a base-ten number
     (possibly with a fractional part). Set <var>second</var> to that number.<li><p>If <var>second</var> is not a number in the range
     0 ≤ <var>second</var> &lt; 60, then fail.</ol>
   <li><p>Return <var>hour</var>, <var>minute</var>, and <var>second</var>.</ol>

  


  <h5 id=local-dates-and-times><span class=secno>2.4.5.5</span> Local dates and times<a href=#local-dates-and-times class=self-link></a></h5>

  <p>A <dfn id=concept-datetime-local>local date and time</dfn> consists of a specific
  <a href=#proleptic-gregorian-date id=local-dates-and-times:proleptic-gregorian-date>proleptic-Gregorian date</a>, consisting of a year, a month, and a day, and a time,
  consisting of an hour, a minute, a second, and a fraction of a second, but expressed without a
  time zone. <a href=#refsGREGORIAN>[GREGORIAN]</a></p>

  <p>A string is a <dfn id=valid-local-date-and-time-string>valid local date and time string</dfn> representing a date and time if it
  consists of the following components in the given order:</p>

  <ol><li>A <a href=#valid-date-string id=local-dates-and-times:valid-date-string>valid date string</a> representing the date<li>A U+0054 LATIN CAPITAL LETTER T character (T) or a U+0020 SPACE character<li>A <a href=#valid-time-string id=local-dates-and-times:valid-time-string>valid time string</a> representing the time</ol>

  <p>A string is a <dfn id=valid-normalised-local-date-and-time-string>valid
  normalized local date and time string</dfn> representing a date and time if it consists of the
  following components in the given order:</p>

  <ol><li>A <a href=#valid-date-string id=local-dates-and-times:valid-date-string-2>valid date string</a> representing the date<li>A U+0054 LATIN CAPITAL LETTER T character (T)<li>A <a href=#valid-time-string id=local-dates-and-times:valid-time-string-2>valid time string</a> representing the time, expressed as the shortest possible
   string for the given time (e.g. omitting the seconds component entirely if the given time is zero
   seconds past the minute)</ol>

  

  <p>The rules to <dfn id=parse-a-local-date-and-time-string>parse a local date and time string</dfn> are as follows. This will return
  either a date and time, or nothing. If at any point the algorithm says that it "fails", this means
  that it is aborted at that point and returns nothing.</p>

  <ol><li><p>Let <var>input</var> be the string being parsed.<li><p>Let <var>position</var> be a pointer into <var>input</var>, initially
   pointing at the start of the string.<li><p><a href=#parse-a-date-component id=local-dates-and-times:parse-a-date-component>Parse a date component</a> to obtain <var>year</var>, <var>month</var>, and <var>day</var>. If this returns nothing, then fail.</p>

   <li><p>If <var>position</var> is beyond the end of <var>input</var> or if the
   character at <var>position</var> is neither a U+0054 LATIN CAPITAL LETTER T character
   (T) nor a U+0020 SPACE character, then fail. Otherwise, move <var>position</var>
   forwards one character.<li><p><a href=#parse-a-time-component id=local-dates-and-times:parse-a-time-component>Parse a time component</a> to obtain <var>hour</var>, <var>minute</var>, and <var>second</var>. If this returns nothing, then fail.</p>

   <li><p>If <var>position</var> is <em>not</em> beyond the end of <var>input</var>, then fail.<li><p>Let <var>date</var> be the date with year <var>year</var>, month <var>month</var>, and day <var>day</var>.<li><p>Let <var>time</var> be the time with hour <var>hour</var>, minute <var>minute</var>, and second <var>second</var>.<li><p>Return <var>date</var> and <var>time</var>.</ol>

  


  <h5 id=time-zones><span class=secno>2.4.5.6</span> Time zones<a href=#time-zones class=self-link></a></h5>

  <p>A <dfn id=concept-timezone>time-zone offset</dfn> consists of a signed number of hours and
  minutes.</p>

  <p>A string is a <dfn id=valid-time-zone-offset-string>valid time-zone offset string</dfn> representing a time-zone offset if it
  consists of either:</p>


  <ul><li><p>A U+005A LATIN CAPITAL LETTER Z character (Z), allowed only if the time zone is
   UTC<li>
    <p>Or, the following components, in the given order:</p>

    <ol><li>Either a U+002B PLUS SIGN character (+) or, if the time-zone offset is not zero, a U+002D
     HYPHEN-MINUS character (-), representing the sign of the time-zone offset<li>Two <a id=time-zones:ascii-digits href=https://infra.spec.whatwg.org/#ascii-digit data-x-internal=ascii-digits>ASCII digits</a>, representing the hours component <var>hour</var> of
     the time-zone offset, in the range 0 ≤ <var>hour</var> ≤ 23<li>Optionally, a U+003A COLON character (:)<li>Two <a id=time-zones:ascii-digits-2 href=https://infra.spec.whatwg.org/#ascii-digit data-x-internal=ascii-digits>ASCII digits</a>, representing the minutes component <var>minute</var> of the time-zone offset, in the range 0 ≤ <var>minute</var> ≤ 59</ol>
   </ul>

  <p class=note>This format allows for time-zone offsets from -23:59 to +23:59. Right now, in
  practice, the range of offsets of actual time zones is -12:00 to +14:00, and the minutes component
  of offsets of actual time zones is always either 00, 30, or 45. There is no guarantee that this
  will remain so forever, however, since time zones are used as political footballs and are thus
  subject to very whimsical policy decisions.</p>

  <p class=note>See also the usage notes and examples in the <a href=#concept-datetime id=time-zones:concept-datetime>global
  date and time</a> section below for details on using time-zone offsets with historical times
  that predate the formation of formal time zones.</p>

  

  <p>The rules to <dfn id=parse-a-time-zone-offset-string>parse a time-zone offset string</dfn> are as follows. This will return either
  a time-zone offset, or nothing. If at any point the algorithm says that it "fails", this means
  that it is aborted at that point and returns nothing.</p>

  <ol><li><p>Let <var>input</var> be the string being parsed.<li><p>Let <var>position</var> be a pointer into <var>input</var>, initially
   pointing at the start of the string.<li><p><a href=#parse-a-time-zone-offset-component id=time-zones:parse-a-time-zone-offset-component>Parse a time-zone offset component</a> to obtain <var>timezone<sub>hours</sub></var> and <var>timezone<sub>minutes</sub></var>. If this
   returns nothing, then fail.</p>

   <li><p>If <var>position</var> is <em>not</em> beyond the end of <var>input</var>, then fail.<li><p>Return the time-zone offset that is <var>timezone<sub>hours</sub></var>
   hours and <var>timezone<sub>minutes</sub></var> minutes from UTC.</ol>

  <p>The rules to <dfn id=parse-a-time-zone-offset-component>parse a time-zone offset component</dfn>, given an <var>input</var>
  string and a <var>position</var>, are as follows. This will return either time-zone hours
  and time-zone minutes, or nothing. If at any point the algorithm says that it "fails", this means
  that it is aborted at that point and returns nothing.</p>

  <ol><li>
    <p>If the character at <var>position</var> is a U+005A LATIN CAPITAL LETTER Z character
    (Z), then:</p>

    <ol><li><p>Let <var>timezone<sub>hours</sub></var> be 0.<li><p>Let <var>timezone<sub>minutes</sub></var> be 0.<li><p>Advance <var>position</var> to the next character in <var>input</var>.</ol>

    <p>Otherwise, if the character at <var>position</var> is either a U+002B PLUS SIGN (+)
    or a U+002D HYPHEN-MINUS (-), then:</p>

    <ol><li><p>If the character at <var>position</var> is a U+002B PLUS SIGN (+), let <var>sign</var> be "positive". Otherwise, it's a U+002D HYPHEN-MINUS (-); let <var>sign</var> be "negative".<li><p>Advance <var>position</var> to the next character in <var>input</var>.<li><p><a id=time-zones:collect-a-sequence-of-code-points href=https://infra.spec.whatwg.org/#collect-a-sequence-of-code-points data-x-internal=collect-a-sequence-of-code-points>Collect a sequence of code points</a> that are <a id=time-zones:ascii-digits-3 href=https://infra.spec.whatwg.org/#ascii-digit data-x-internal=ascii-digits>ASCII digits</a> from
     <var>input</var> given <var>position</var>. Let <var>s</var> be the collected
     sequence.<li>
      <p>If <var>s</var> is exactly two characters long, then:</p>

      <ol><li><p>Interpret <var>s</var> as a base-ten integer. Let that number be the <var>timezone<sub>hours</sub></var>.<li><p>If <var>position</var> is beyond the end of <var>input</var> or if
       the character at <var>position</var> is not a U+003A COLON character, then fail.
       Otherwise, move <var>position</var> forwards one character.<li><p><a id=time-zones:collect-a-sequence-of-code-points-2 href=https://infra.spec.whatwg.org/#collect-a-sequence-of-code-points data-x-internal=collect-a-sequence-of-code-points>Collect a sequence of code points</a> that are <a id=time-zones:ascii-digits-4 href=https://infra.spec.whatwg.org/#ascii-digit data-x-internal=ascii-digits>ASCII digits</a> from
       <var>input</var> given <var>position</var>. If the collected sequence is not exactly two
       characters long, then fail. Otherwise, interpret the resulting sequence as a base-ten
       integer. Let that number be the <var>timezone<sub>minutes</sub></var>.</ol>

      <p>If <var>s</var> is exactly four characters long, then:</p>

      <ol><li><p>Interpret the first two characters of <var>s</var> as a base-ten integer. Let
       that number be the <var>timezone<sub>hours</sub></var>.<li><p>Interpret the last two characters of <var>s</var> as a base-ten integer. Let
       that number be the <var>timezone<sub>minutes</sub></var>.</ol>

      <p>Otherwise, fail.</p>
     <li>If <var>timezone<sub>hours</sub></var> is not a number in the range
     0 ≤ <var>timezone<sub>hours</sub></var> ≤ 23, then
     fail.<li>If <var>sign</var> is "negative", then negate <var>timezone<sub>hours</sub></var>.<li>If <var>timezone<sub>minutes</sub></var> is not a number in the range
     0 ≤ <var>timezone<sub>minutes</sub></var> ≤ 59,
     then fail.<li>If <var>sign</var> is "negative", then negate <var>timezone<sub>minutes</sub></var>.</ol>

    <p>Otherwise, fail.</p>
   <li><p>Return <var>timezone<sub>hours</sub></var> and <var>timezone<sub>minutes</sub></var>.</ol>

  


  <h5 id=global-dates-and-times><span class=secno>2.4.5.7</span> Global dates and times<a href=#global-dates-and-times class=self-link></a></h5>

  <p>A <dfn id=concept-datetime>global date and time</dfn> consists of a specific
  <a href=#proleptic-gregorian-date id=global-dates-and-times:proleptic-gregorian-date>proleptic-Gregorian date</a>, consisting of a year, a month, and a day, and a time,
  consisting of an hour, a minute, a second, and a fraction of a second, expressed with a time-zone
  offset, consisting of a signed number of hours and minutes. <a href=#refsGREGORIAN>[GREGORIAN]</a></p>

  <p>A string is a <dfn id=valid-global-date-and-time-string>valid global date and time string</dfn> representing a date, time, and a
  time-zone offset if it consists of the following components in the given order:</p>

  <ol><li>A <a href=#valid-date-string id=global-dates-and-times:valid-date-string>valid date string</a> representing the date<li>A U+0054 LATIN CAPITAL LETTER T character (T) or a U+0020 SPACE character<li>A <a href=#valid-time-string id=global-dates-and-times:valid-time-string>valid time string</a> representing the time<li>A <a href=#valid-time-zone-offset-string id=global-dates-and-times:valid-time-zone-offset-string>valid time-zone offset string</a> representing the time-zone offset</ol>

  <p>Times in dates before the formation of UTC in the mid twentieth century must be expressed and
  interpreted in terms of UT1 (contemporary Earth solar time at the 0° longitude), not UTC (the
  approximation of UT1 that ticks in SI seconds). Time before the formation of time zones must be
  expressed and interpreted as UT1 times with explicit time zones that approximate the contemporary
  difference between the appropriate local time and the time observed at the location of Greenwich,
  London.</p>

  <div class=example>

   <p>The following are some examples of dates written as <a href=#valid-global-date-and-time-string id=global-dates-and-times:valid-global-date-and-time-string>valid global date and time strings</a>.</p>

   <dl><dt>"<code>0037-12-13 00:00Z</code>"<dd>Midnight in areas using London time on the birthday of Nero (the Roman Emperor). See below
    for further discussion on which date this actually corresponds to.<dt>"<code>1979-10-14T12:00:00.001-04:00</code>"<dd>One millisecond after noon on October 14th 1979, in the time zone in use on the east coast
    of the USA during daylight saving time.<dt>"<code>8592-01-01T02:09+02:09</code>"<dd>Midnight UTC on the 1st of January, 8592. The time zone associated with that time is two
    hours and nine minutes ahead of UTC, which is not currently a real time zone, but is nonetheless
    allowed.</dl>

   <p>Several things are notable about these dates:</p>

   <ul><li>Years with fewer than four digits have to be zero-padded. The date "37-12-13" would not be a
    valid date.<li>If the "<code>T</code>" is replaced by a space, it must be a single space
    character. The string "<code>2001-12-21  12:00Z</code>" (with two spaces
    between the components) would not be parsed successfully.<li>To unambiguously identify a moment in time prior to the introduction of the Gregorian
    calendar (insofar as moments in time before the formation of UTC can be unambiguously
    identified), the date has to be first converted to the Gregorian calendar from the calendar in
    use at the time (e.g. from the Julian calendar). The date of Nero's birth is the 15th of
    December 37, in the Julian Calendar, which is the 13th of December 37 in the <a href=#proleptic-gregorian-calendar id=global-dates-and-times:proleptic-gregorian-calendar>proleptic
    Gregorian calendar</a>.<li>The time and time-zone offset components are not optional.<li>Dates before the year one can't be represented as a datetime in this version of HTML.<li>Times of specific events in ancient times are, at best, approximations, since time was not
    well coordinated or measured until relatively recent decades.<li>Time-zone offsets differ based on daylight saving time.</ul>

  </div>

  

  <p>The rules to <dfn id=parse-a-global-date-and-time-string>parse a global date and time string</dfn> are as follows. This will return
  either a time in UTC, with associated time-zone offset information for round-tripping or display
  purposes, or nothing. If at any point the algorithm says that it "fails", this means that it is
  aborted at that point and returns nothing.</p>

  <ol><li><p>Let <var>input</var> be the string being parsed.<li><p>Let <var>position</var> be a pointer into <var>input</var>, initially
   pointing at the start of the string.<li><p><a href=#parse-a-date-component id=global-dates-and-times:parse-a-date-component>Parse a date component</a> to obtain <var>year</var>, <var>month</var>, and <var>day</var>. If this returns nothing, then fail.</p>

   <li><p>If <var>position</var> is beyond the end of <var>input</var> or if the
   character at <var>position</var> is neither a U+0054 LATIN CAPITAL LETTER T character
   (T) nor a U+0020 SPACE character, then fail. Otherwise, move <var>position</var>
   forwards one character.<li><p><a href=#parse-a-time-component id=global-dates-and-times:parse-a-time-component>Parse a time component</a> to obtain <var>hour</var>, <var>minute</var>, and <var>second</var>. If this returns nothing, then fail.</p>

   <li><p>If <var>position</var> is beyond the end of <var>input</var>, then
   fail.<li><p><a href=#parse-a-time-zone-offset-component id=global-dates-and-times:parse-a-time-zone-offset-component>Parse a time-zone offset component</a> to obtain <var>timezone<sub>hours</sub></var> and <var>timezone<sub>minutes</sub></var>. If this
   returns nothing, then fail.</p>

   <li><p>If <var>position</var> is <em>not</em> beyond the end of <var>input</var>, then fail.<li><p>Let <var>time</var> be the moment in time at year <var>year</var>, month
   <var>month</var>, day <var>day</var>, hours <var>hour</var>, minute
   <var>minute</var>, second <var>second</var>, subtracting <var>timezone<sub>hours</sub></var> hours and <var>timezone<sub>minutes</sub></var> minutes. That moment in time is a moment in the UTC time
   zone.<li><p>Let <var>timezone</var> be <var>timezone<sub>hours</sub></var>
   hours and <var>timezone<sub>minutes</sub></var> minutes from UTC.<li><p>Return <var>time</var> and <var>timezone</var>.</ol>

  


  <h5 id=weeks><span class=secno>2.4.5.8</span> Weeks<a href=#weeks class=self-link></a></h5>

  <p>A <dfn id=concept-week>week</dfn> consists of a week-year number and a week number
  representing a seven-day period starting on a Monday. Each week-year in this calendaring system
  has either 52 or 53 such seven-day periods, as defined below. The seven-day period starting on the
  Gregorian date Monday December 29th 1969 (1969-12-29) is defined as week number 1 in week-year
  1970. Consecutive weeks are numbered sequentially. The week before the number 1 week in a
  week-year is the last week in the previous week-year, and vice versa. <a href=#refsGREGORIAN>[GREGORIAN]</a></p>

  <p>A week-year with a number <var>year</var> has 53 weeks if it corresponds to either a
  year <var>year</var> in the <a href=#proleptic-gregorian-calendar id=weeks:proleptic-gregorian-calendar>proleptic Gregorian calendar</a> that has a Thursday
  as its first day (January 1st), or a year <var>year</var> in the <a href=#proleptic-gregorian-calendar id=weeks:proleptic-gregorian-calendar-2>proleptic
  Gregorian calendar</a> that has a Wednesday as its first day (January 1st) and where <var>year</var> is a number divisible by 400, or a number divisible by 4 but not by 100. All
  other week-years have 52 weeks.</p>

  <p>The <dfn id=week-number-of-the-last-day>week number of the last day</dfn> of a week-year with 53 weeks is 53; the week number
  of the last day of a week-year with 52 weeks is 52.</p>

  <p class=note>The week-year number of a particular day can be different than the number of the
  year that contains that day in the <a href=#proleptic-gregorian-calendar id=weeks:proleptic-gregorian-calendar-3>proleptic Gregorian calendar</a>. The first week in a
  week-year <var>y</var> is the week that contains the first Thursday of the Gregorian year
  <var>y</var>.</p>

  <p class=note>For modern purposes, a <a href=#concept-week id=weeks:concept-week>week</a> as defined here is
  equivalent to ISO weeks as defined in ISO 8601. <a href=#refsISO8601>[ISO8601]</a></p>

  <p>A string is a <dfn id=valid-week-string>valid week string</dfn> representing a week-year <var>year</var>
  and week <var>week</var> if it consists of the following components in the given
  order:</p>

  <ol><li>Four or more <a id=weeks:ascii-digits href=https://infra.spec.whatwg.org/#ascii-digit data-x-internal=ascii-digits>ASCII digits</a>, representing <var>year</var>, where <var>year</var> > 0<li>A U+002D HYPHEN-MINUS character (-)<li>A U+0057 LATIN CAPITAL LETTER W character (W)<li>Two <a id=weeks:ascii-digits-2 href=https://infra.spec.whatwg.org/#ascii-digit data-x-internal=ascii-digits>ASCII digits</a>, representing the week <var>week</var>, in the range
   1 ≤ <var>week</var> ≤ <var>maxweek</var>, where <var>maxweek</var> is the <a href=#week-number-of-the-last-day id=weeks:week-number-of-the-last-day>week number of the last day</a> of week-year <var>year</var></ol>

  

  <p>The rules to <dfn id=parse-a-week-string>parse a week string</dfn> are as follows. This will return either a week-year
  number and week number, or nothing. If at any point the algorithm says that it "fails", this means
  that it is aborted at that point and returns nothing.</p>

  <ol><li><p>Let <var>input</var> be the string being parsed.<li><p>Let <var>position</var> be a pointer into <var>input</var>, initially
   pointing at the start of the string.<li><p><a id=weeks:collect-a-sequence-of-code-points href=https://infra.spec.whatwg.org/#collect-a-sequence-of-code-points data-x-internal=collect-a-sequence-of-code-points>Collect a sequence of code points</a> that are <a id=weeks:ascii-digits-3 href=https://infra.spec.whatwg.org/#ascii-digit data-x-internal=ascii-digits>ASCII digits</a> from
   <var>input</var> given <var>position</var>. If the collected sequence is not at least four
   characters long, then fail. Otherwise, interpret the resulting sequence as a base-ten integer.
   Let that number be the <var>year</var>.<li><p>If <var>year</var> is not a number greater than zero, then fail.<li><p>If <var>position</var> is beyond the end of <var>input</var> or if the
   character at <var>position</var> is not a U+002D HYPHEN-MINUS character, then fail.
   Otherwise, move <var>position</var> forwards one character.<li><p>If <var>position</var> is beyond the end of <var>input</var> or if the
   character at <var>position</var> is not a U+0057 LATIN CAPITAL LETTER W character (W),
   then fail. Otherwise, move <var>position</var> forwards one character.<li><p><a id=weeks:collect-a-sequence-of-code-points-2 href=https://infra.spec.whatwg.org/#collect-a-sequence-of-code-points data-x-internal=collect-a-sequence-of-code-points>Collect a sequence of code points</a> that are <a id=weeks:ascii-digits-4 href=https://infra.spec.whatwg.org/#ascii-digit data-x-internal=ascii-digits>ASCII digits</a> from
   <var>input</var> given <var>position</var>. If the collected sequence is not exactly two
   characters long, then fail. Otherwise, interpret the resulting sequence as a base-ten integer.
   Let that number be the <var>week</var>.<li><p>Let <var>maxweek</var> be the <a href=#week-number-of-the-last-day id=weeks:week-number-of-the-last-day-2>week number of the last day</a> of year
   <var>year</var>.<li><p>If <var>week</var> is not a number in the range 1 ≤ <var>week</var> ≤ <var>maxweek</var>, then fail.<li><p>If <var>position</var> is <em>not</em> beyond the end of <var>input</var>, then fail.<li><p>Return the week-year number <var>year</var> and the week number <var>week</var>.</ol>

  


  <h5 id=durations><span class=secno>2.4.5.9</span> Durations<a href=#durations class=self-link></a></h5>

  

  <p>A <dfn id=concept-duration>duration</dfn> consists of  a number of seconds.</p>

  <p class=note>Since months and seconds are not comparable (a month is not a precise number of
  seconds, but is instead a period whose exact length depends on the precise day from which it is
  measured) a <a href=#concept-duration id=durations:concept-duration>duration</a> as defined in this specification cannot
   include months (or years, which are equivalent to
  twelve months). Only durations that describe a specific number of seconds can be described.</p>

  <p>A string is a <dfn id=valid-duration-string>valid duration string</dfn> representing a <a href=#concept-duration id=durations:concept-duration-2>duration</a> <var>t</var> if it consists of either of the
  following:</p>

  <ul><li>
    

    

    <p>A literal U+0050 LATIN CAPITAL LETTER P character followed by one or more of the following
    subcomponents, in the order given, where the number of days, hours, minutes, and
    seconds corresponds to the same number of seconds as in <var>t</var>:</p>

    <ol><li><p>One or more <a id=durations:ascii-digits href=https://infra.spec.whatwg.org/#ascii-digit data-x-internal=ascii-digits>ASCII digits</a> followed by a U+0044 LATIN CAPITAL LETTER D
     character, representing a number of days.<li>
      <p>A U+0054 LATIN CAPITAL LETTER T character followed by one or more of the following
      subcomponents, in the order given:</p>

      <ol><li><p>One or more <a id=durations:ascii-digits-2 href=https://infra.spec.whatwg.org/#ascii-digit data-x-internal=ascii-digits>ASCII digits</a> followed by a U+0048 LATIN CAPITAL LETTER H
       character, representing a number of hours.<li><p>One or more <a id=durations:ascii-digits-3 href=https://infra.spec.whatwg.org/#ascii-digit data-x-internal=ascii-digits>ASCII digits</a> followed by a U+004D LATIN CAPITAL LETTER M
       character, representing a number of minutes.<li>
        <p>The following components:</p>

        <ol><li><p>One or more <a id=durations:ascii-digits-4 href=https://infra.spec.whatwg.org/#ascii-digit data-x-internal=ascii-digits>ASCII digits</a>, representing a number of seconds.<li><p>Optionally, a U+002E FULL STOP character (.) followed by one, two, or three
         <a id=durations:ascii-digits-5 href=https://infra.spec.whatwg.org/#ascii-digit data-x-internal=ascii-digits>ASCII digits</a>, representing a fraction of a second.<li><p>A U+0053 LATIN CAPITAL LETTER S character.</ol>
       </ol>
     </ol>

    <p class=note>This, as with a number of other date- and time-related microsyntaxes defined in
    this specification, is based on one of the formats defined in ISO 8601. <a href=#refsISO8601>[ISO8601]</a></p>
   <li>
    <p>One or more <a href=#duration-time-component id=durations:duration-time-component>duration time components</a>, each with
    a different <a href=#duration-time-component-scale id=durations:duration-time-component-scale>duration time component scale</a>, in any order; the sum of the represented
    seconds being equal to the number of seconds in <var>t</var>.</p>

    <p>A <dfn id=duration-time-component>duration time component</dfn> is a string consisting of the following components:</p>

    <ol><li><p>Zero or more <a id=durations:space-characters href=https://infra.spec.whatwg.org/#ascii-whitespace data-x-internal=space-characters>ASCII whitespace</a>.<li><p>One or more <a id=durations:ascii-digits-6 href=https://infra.spec.whatwg.org/#ascii-digit data-x-internal=ascii-digits>ASCII digits</a>, representing a number of time units, scaled by
     the <a href=#duration-time-component-scale id=durations:duration-time-component-scale-2>duration time component scale</a> specified (see below) to represent a number of
     seconds.<li><p>If the <a href=#duration-time-component-scale id=durations:duration-time-component-scale-3>duration time component scale</a> specified is 1 (i.e. the units are
     seconds), then, optionally, a U+002E FULL STOP character (.) followed by one, two, or three
     <a id=durations:ascii-digits-7 href=https://infra.spec.whatwg.org/#ascii-digit data-x-internal=ascii-digits>ASCII digits</a>, representing a fraction of a second.<li><p>Zero or more <a id=durations:space-characters-2 href=https://infra.spec.whatwg.org/#ascii-whitespace data-x-internal=space-characters>ASCII whitespace</a>.<li>
      <p>One of the following characters, representing the <dfn id=duration-time-component-scale>duration time component scale</dfn>
      of the time unit used in the numeric part of the <a href=#duration-time-component id=durations:duration-time-component-2>duration time component</a>:</p>

      <dl><dt>U+0057 LATIN CAPITAL LETTER W character<dt>U+0077 LATIN SMALL LETTER W character<dd>Weeks. The scale is 604800.<dt>U+0044 LATIN CAPITAL LETTER D character<dt>U+0064 LATIN SMALL LETTER D character<dd>Days. The scale is 86400.<dt>U+0048 LATIN CAPITAL LETTER H character<dt>U+0068 LATIN SMALL LETTER H character<dd>Hours. The scale is 3600.<dt>U+004D LATIN CAPITAL LETTER M character<dt>U+006D LATIN SMALL LETTER M character<dd>Minutes. The scale is 60.<dt>U+0053 LATIN CAPITAL LETTER S character<dt>U+0073 LATIN SMALL LETTER S character<dd>Seconds. The scale is 1.</dl>
     <li><p>Zero or more <a id=durations:space-characters-3 href=https://infra.spec.whatwg.org/#ascii-whitespace data-x-internal=space-characters>ASCII whitespace</a>.</ol>

    <p class=note>This is not based on any of the formats in ISO 8601. It is intended to be a more
    human-readable alternative to the ISO 8601 duration format.</p>
   </ul>

  

  <p>The rules to <dfn id=parse-a-duration-string>parse a duration string</dfn> are as follows. This will return either a <a href=#concept-duration id=durations:concept-duration-3>duration</a> or nothing. If at any point the algorithm says that it
  "fails", this means that it is aborted at that point and returns nothing.</p>

  <ol><li><p>Let <var>input</var> be the string being parsed.<li><p>Let <var>position</var> be a pointer into <var>input</var>, initially
   pointing at the start of the string.<li><p>Let <var>months</var>, <var>seconds</var>, and <var>component
   count</var> all be zero.<li>
    <p>Let <var>M-disambiguator</var> be <i>minutes</i>.</p>

    
    <p class=note>This flag's other value is <i>months</i>. It is used to disambiguate the "M"
    unit in ISO8601 durations, which use the same unit for months and minutes. Months are not
    allowed, but are parsed for future compatibility and to avoid misinterpreting ISO8601 durations
    that would be valid in other contexts.</p>
   <li><p><a id=durations:skip-ascii-whitespace href=https://infra.spec.whatwg.org/#skip-ascii-whitespace data-x-internal=skip-ascii-whitespace>Skip ASCII whitespace</a> within <var>input</var> given
   <var>position</var>.<li><p>If <var>position</var> is past the end of <var>input</var>, then
   fail.<li><p>If the character in <var>input</var> pointed to by <var>position</var> is a U+0050 LATIN
   CAPITAL LETTER P character, then advance <var>position</var> to the next character, set
   <var>M-disambiguator</var> to <i>months</i>, and <a id=durations:skip-ascii-whitespace-2 href=https://infra.spec.whatwg.org/#skip-ascii-whitespace data-x-internal=skip-ascii-whitespace>skip ASCII whitespace</a> within
   <var>input</var> given <var>position</var>.<li>
    <p>While true:</p>

    <ol><li><p>Let <var>units</var> be undefined. It will be assigned one of the following
     values: <i>years</i>, <i>months</i>, <i>weeks</i>, <i>days</i>, <i>hours</i>, <i>minutes</i>,
     and <i>seconds</i>.<li><p>Let <var>next character</var> be undefined. It is used to process characters
     from the <var>input</var>.<li><p>If <var>position</var> is past the end of <var>input</var>, then break.<li><p>If the character in <var>input</var> pointed to by <var>position</var> is a U+0054 LATIN
     CAPITAL LETTER T character, then advance <var>position</var> to the next character, set
     <var>M-disambiguator</var> to <i>minutes</i>, <a id=durations:skip-ascii-whitespace-3 href=https://infra.spec.whatwg.org/#skip-ascii-whitespace data-x-internal=skip-ascii-whitespace>skip ASCII whitespace</a> within
     <var>input</var> given <var>position</var>, and continue.<li><p>Set <var>next character</var> to the character in <var>input</var>
     pointed to by <var>position</var>.<li>
      <p>If <var>next character</var> is a U+002E FULL STOP character (.), then let <var>N</var> equal zero. (Do not advance <var>position</var>. That is taken care
      of below.)</p>

      <p>Otherwise, if <var>next character</var> is an <a href=https://infra.spec.whatwg.org/#ascii-digit id=durations:ascii-digits-8 data-x-internal=ascii-digits>ASCII
      digit</a>, then <a id=durations:collect-a-sequence-of-code-points href=https://infra.spec.whatwg.org/#collect-a-sequence-of-code-points data-x-internal=collect-a-sequence-of-code-points>collect a sequence of code points</a> that are <a id=durations:ascii-digits-9 href=https://infra.spec.whatwg.org/#ascii-digit data-x-internal=ascii-digits>ASCII
      digits</a> from <var>input</var> given <var>position</var>, interpret the resulting
      sequence as a base-ten integer, and let <var>N</var> be that number.</p>

      <p>Otherwise <var>next character</var> is not part of a number; fail.</p>
     <li><p>If <var>position</var> is past the end of <var>input</var>, then
     fail.<li><p>Set <var>next character</var> to the character in <var>input</var>
     pointed to by <var>position</var>, and this time advance <var>position</var>
     to the next character. (If <var>next character</var> was a U+002E FULL STOP character
     (.) before, it will still be that character this time.)<li>
      <p>If <var>next character</var> is U+002E (.), then:</p>

      <ol><li><p><a id=durations:collect-a-sequence-of-code-points-2 href=https://infra.spec.whatwg.org/#collect-a-sequence-of-code-points data-x-internal=collect-a-sequence-of-code-points>Collect a sequence of code points</a> that are <a id=durations:ascii-digits-10 href=https://infra.spec.whatwg.org/#ascii-digit data-x-internal=ascii-digits>ASCII digits</a> from
       <var>input</var> given <var>position</var>. Let <var>s</var> be the resulting
       sequence.<li><p>If <var>s</var> is the empty string, then fail.<li><p>Let <var>length</var> be the number of characters in <var>s</var>.<li><p>Let <var>fraction</var> be the result of interpreting <var>s</var>
       as a base-ten integer, and then dividing that number by <span>10<sup><var>length</var></sup></span>.<li><p>Increment <var>N</var> by <var>fraction</var>.<li><p><a id=durations:skip-ascii-whitespace-4 href=https://infra.spec.whatwg.org/#skip-ascii-whitespace data-x-internal=skip-ascii-whitespace>Skip ASCII whitespace</a> within <var>input</var> given
       <var>position</var>.<li><p>If <var>position</var> is past the end of <var>input</var>, then
       fail.<li><p>Set <var>next character</var> to the character in <var>input</var>
       pointed to by <var>position</var>, and advance <var>position</var> to the
       next character.<li><p>If <var>next character</var> is neither a U+0053 LATIN CAPITAL LETTER S
       character nor a U+0073 LATIN SMALL LETTER S character, then fail.<li><p>Set <var>units</var> to <i>seconds</i>.</ol>

      <p>Otherwise:</p>

      <ol><li><p>If <var>next character</var> is <a id=durations:space-characters-4 href=https://infra.spec.whatwg.org/#ascii-whitespace data-x-internal=space-characters>ASCII whitespace</a>, then <a id=durations:skip-ascii-whitespace-5 href=https://infra.spec.whatwg.org/#skip-ascii-whitespace data-x-internal=skip-ascii-whitespace>skip ASCII
       whitespace</a> within <var>input</var> given <var>position</var>, set <var>next
       character</var> to the character in <var>input</var> pointed to by <var>position</var>, and
       advance <var>position</var> to the next character.<li>
        

        <p>If <var>next character</var> is a U+0059 LATIN CAPITAL LETTER Y character, or a
        U+0079 LATIN SMALL LETTER Y character, set <var>units</var> to <i>years</i> and set
        <var>M-disambiguator</var> to <i>months</i>.</p>

        <p>If <var>next character</var> is a U+004D LATIN CAPITAL LETTER M character or a
        U+006D LATIN SMALL LETTER M character, and <var>M-disambiguator</var> is
        <i>months</i>, then set <var>units</var> to <i>months</i>.</p>

        <p>If <var>next character</var> is a U+0057 LATIN CAPITAL LETTER W character or a
        U+0077 LATIN SMALL LETTER W character, set <var>units</var> to <i>weeks</i> and set
        <var>M-disambiguator</var> to <i>minutes</i>.</p>

        <p>If <var>next character</var> is a U+0044 LATIN CAPITAL LETTER D character or a
        U+0064 LATIN SMALL LETTER D character, set <var>units</var> to <i>days</i> and set
        <var>M-disambiguator</var> to <i>minutes</i>.</p>

        <p>If <var>next character</var> is a U+0048 LATIN CAPITAL LETTER H character or a
        U+0068 LATIN SMALL LETTER H character, set <var>units</var> to <i>hours</i> and set
        <var>M-disambiguator</var> to <i>minutes</i>.</p>

        <p>If <var>next character</var> is a U+004D LATIN CAPITAL LETTER M character or a
        U+006D LATIN SMALL LETTER M character, and <var>M-disambiguator</var> is
        <i>minutes</i>, then set <var>units</var> to <i>minutes</i>.</p>

        <p>If <var>next character</var> is a U+0053 LATIN CAPITAL LETTER S character or a
        U+0073 LATIN SMALL LETTER S character, set <var>units</var> to <i>seconds</i> and
        set <var>M-disambiguator</var> to <i>minutes</i>.</p>

        <p>Otherwise if <var>next character</var> is none of the above characters, then
        fail.</p>
       </ol>
     <li><p>Increment <var>component count</var>.<li><p>Let <var>multiplier</var> be 1.<li><p>If <var>units</var> is <i>years</i>, multiply <var>multiplier</var> by
     12 and set <var>units</var> to <i>months</i>.<li>
      <p>If <var>units</var> is <i>months</i>, add the product of <var>N</var> and
      <var>multiplier</var> to <var>months</var>.</p>

      <p>Otherwise:</p>

      <ol><li><p>If <var>units</var> is <i>weeks</i>, multiply <var>multiplier</var>
       by 7 and set <var>units</var> to <i>days</i>.<li><p>If <var>units</var> is <i>days</i>, multiply <var>multiplier</var>
       by 24 and set <var>units</var> to <i>hours</i>.<li><p>If <var>units</var> is <i>hours</i>, multiply <var>multiplier</var>
       by 60 and set <var>units</var> to <i>minutes</i>.<li><p>If <var>units</var> is <i>minutes</i>, multiply <var>multiplier</var> by 60 and set <var>units</var> to <i>seconds</i>.<li><p>Forcibly, <var>units</var> is now <i>seconds</i>. Add the product of <var>N</var> and <var>multiplier</var> to <var>seconds</var>.</ol>
     <li><p><a id=durations:skip-ascii-whitespace-6 href=https://infra.spec.whatwg.org/#skip-ascii-whitespace data-x-internal=skip-ascii-whitespace>Skip ASCII whitespace</a> within <var>input</var> given
     <var>position</var>.</ol>
   <li><p>If <var>component count</var> is zero,
   fail.<li><p>If <var>months</var> is not zero, fail.<li><p>Return the <a href=#concept-duration id=durations:concept-duration-4>duration</a> consisting of <var>seconds</var> seconds.</ol>

  


  <h5 id=vaguer-moments-in-time><span class=secno>2.4.5.10</span> Vaguer moments in time<a href=#vaguer-moments-in-time class=self-link></a></h5>

  <p>A string is a <dfn id=valid-date-string-with-optional-time>valid date string with optional time</dfn> if
  it is also one of the following:</p>

  <ul><li>A <a href=#valid-date-string id=vaguer-moments-in-time:valid-date-string>valid date string</a><li>A <a href=#valid-global-date-and-time-string id=vaguer-moments-in-time:valid-global-date-and-time-string>valid global date and time string</a></ul>

  

  <hr>

  <p>The rules to <dfn id=parse-a-date-or-time-string>parse a date or time string</dfn> are as follows. The algorithm will return
  either a <a href=#concept-date id=vaguer-moments-in-time:concept-date>date</a>, a <a href=#concept-time id=vaguer-moments-in-time:concept-time>time</a>, a <a href=#concept-datetime id=vaguer-moments-in-time:concept-datetime>global date and time</a>, or nothing. If at any point the algorithm
  says that it "fails", this means that it is aborted at that point and returns nothing.</p>

  <ol><li><p>Let <var>input</var> be the string being parsed.<li><p>Let <var>position</var> be a pointer into <var>input</var>, initially
   pointing at the start of the string.<li><p>Set <var>start position</var> to the same position as <var>position</var>.<li><p>Set the <var>date present</var> and <var>time present</var> flags to
   true.<li><p><a href=#parse-a-date-component id=vaguer-moments-in-time:parse-a-date-component>Parse a date component</a> to obtain <var>year</var>, <var>month</var>, and <var>day</var>. If this fails, then set the <var>date
   present</var> flag to false.</p>

   <li>
    <p>If <var>date present</var> is true, and <var>position</var> is not beyond
    the end of <var>input</var>, and the character at <var>position</var> is
    either a U+0054 LATIN CAPITAL LETTER T character (T) or a U+0020 SPACE character, then advance
    <var>position</var> to the next character in <var>input</var>.</p>

    <p>Otherwise, if <var>date present</var> is true, and either <var>position</var> is beyond the end of <var>input</var> or the character at <var>position</var> is neither a U+0054 LATIN CAPITAL LETTER T character (T) nor a U+0020
    SPACE character, then set <var>time present</var> to false.</p>

    <p>Otherwise, if <var>date present</var> is false, set <var>position</var>
    back to the same position as <var>start position</var>.</p>
   <li><p>If the <var>time present</var> flag is true, then <a href=#parse-a-time-component id=vaguer-moments-in-time:parse-a-time-component>parse a time
   component</a> to obtain <var>hour</var>, <var>minute</var>, and <var>second</var>. If this returns nothing, then fail.<li><p>If the <var>date present</var> and <var>time present</var> flags are
   both true, but <var>position</var> is beyond the end of <var>input</var>, then
   fail.<li><p>If the <var>date present</var> and <var>time present</var> flags are
   both true, <a href=#parse-a-time-zone-offset-component id=vaguer-moments-in-time:parse-a-time-zone-offset-component>parse a time-zone offset component</a> to obtain <var>timezone<sub>hours</sub></var> and <var>timezone<sub>minutes</sub></var>. If this
   returns nothing, then fail.</p>

   <li><p>If <var>position</var> is <em>not</em> beyond the end of <var>input</var>, then fail.<li>
    <p>If the <var>date present</var> flag is true and the <var>time present</var>
    flag is false, then let <var>date</var> be the date with year <var>year</var>,
    month <var>month</var>, and day <var>day</var>, and return <var>date</var>.</p>

    <p>Otherwise, if the <var>time present</var> flag is true and the <var>date
    present</var> flag is false, then let <var>time</var> be the time with hour <var>hour</var>, minute <var>minute</var>, and second <var>second</var>,
    and return <var>time</var>.</p>

    <p>Otherwise, let <var>time</var> be the moment in time at year <var>year</var>, month <var>month</var>, day <var>day</var>, hours <var>hour</var>, minute <var>minute</var>, second <var>second</var>,
    subtracting <var>timezone<sub>hours</sub></var> hours and <var>timezone<sub>minutes</sub></var> minutes, that moment in time being a moment
    in the UTC time zone; let <var>timezone</var> be <var>timezone<sub>hours</sub></var> hours and <var>timezone<sub>minutes</sub></var>
    minutes from UTC; and return <var>time</var> and <var>timezone</var>.</p>
   </ol>

  


  <h4 id=colours><span class=secno>2.4.6</span> Colors<a href=#colours class=self-link></a></h4>

  <p>A <dfn id=simple-colour>simple color</dfn> consists of three 8-bit numbers in the
  range 0..255, representing the red, green, and blue components of the color respectively, in the
  sRGB color space. <a href=#refsSRGB>[SRGB]</a></p>

  <p>A string is a <dfn id=valid-simple-colour>valid simple color</dfn> if it is
  exactly seven characters long, and the first character is a U+0023 NUMBER SIGN character (#), and
  the remaining six characters are all <a id=colours:ascii-hex-digits href=https://infra.spec.whatwg.org/#ascii-hex-digit data-x-internal=ascii-hex-digits>ASCII hex digits</a>, with the first two digits
  representing the red component, the middle two digits representing the green component, and the
  last two digits representing the blue component, in hexadecimal.</p>

  <p>A string is a <dfn id=valid-lowercase-simple-colour>valid lowercase simple
  color</dfn> if it is a <a href=#valid-simple-colour id=colours:valid-simple-colour>valid simple color</a> and doesn't use any characters in the range
  U+0041 LATIN CAPITAL LETTER A to U+0046 LATIN CAPITAL LETTER F.</p>

  

  <p>The <dfn id=rules-for-parsing-simple-colour-values>rules for parsing simple color
  values</dfn> are as given in the following algorithm. When invoked, the steps must be followed in
  the order given, aborting at the first step that returns a value. This algorithm will return
  either a <a href=#simple-colour id=colours:simple-colour>simple color</a> or an error.</p>

  <ol><li><p>Let <var>input</var> be the string being parsed.<li><p>If <var>input</var> is not exactly seven characters long, then return an
   error.<li><p>If the first character in <var>input</var> is not a U+0023 NUMBER SIGN character
   (#), then return an error.<li><p>If the last six characters of <var>input</var> are not all <a id=colours:ascii-hex-digits-2 href=https://infra.spec.whatwg.org/#ascii-hex-digit data-x-internal=ascii-hex-digits>ASCII hex
   digits</a>, then return an error.<li><p>Let <var>result</var> be a <a href=#simple-colour id=colours:simple-colour-2>simple color</a>.</p>

   <li><p>Interpret the second and third characters as a hexadecimal number and let the result be
   the red component of <var>result</var>.</p>

   <li><p>Interpret the fourth and fifth characters as a hexadecimal number and let the result be
   the green component of <var>result</var>.</p>

   <li><p>Interpret the sixth and seventh characters as a hexadecimal number and let the result be
   the blue component of <var>result</var>.</p>

   <li><p>Return <var>result</var>.</p>
  </ol>

  <p>The <dfn id=rules-for-serialising-simple-colour-values>rules for serializing
  simple color values</dfn> given a <a href=#simple-colour id=colours:simple-colour-3>simple color</a> are as given in the following
  algorithm:</p>

  <ol><li><p>Let <var>result</var> be a string consisting of a single U+0023 NUMBER SIGN
   character (#).<li><p>Convert the red, green, and blue components in turn to two-digit hexadecimal numbers using
   <a href=https://infra.spec.whatwg.org/#ascii-lower-hex-digit id=colours:lowercase-ascii-hex-digits data-x-internal=lowercase-ascii-hex-digits>ASCII lower hex digits</a>, zero-padding if necessary,
   and append these numbers to <var>result</var>, in the order red, green, blue.</p>

   <li><p>Return <var>result</var>, which will be a <a href=#valid-lowercase-simple-colour id=colours:valid-lowercase-simple-colour>valid lowercase simple
   color</a>.</ol>

  <hr>

  <p>Some obsolete legacy attributes parse colors in a more complicated manner, using the
  <dfn id=rules-for-parsing-a-legacy-colour-value>rules for parsing a legacy color
  value</dfn>, which are given in the following algorithm. When invoked, the steps must be followed
  in the order given, aborting at the first step that returns a value. This algorithm will return
  either a <a href=#simple-colour id=colours:simple-colour-4>simple color</a> or an error.</p>

  <ol><li><p>Let <var>input</var> be the string being parsed.<li><p>If <var>input</var> is the empty string, then return an error.<li><p><a id=colours:strip-leading-and-trailing-ascii-whitespace href=https://infra.spec.whatwg.org/#strip-leading-and-trailing-ascii-whitespace data-x-internal=strip-leading-and-trailing-ascii-whitespace>Strip leading and trailing ASCII whitespace</a> from <var>input</var>.<li><p>If <var>input</var> is an <a id=colours:ascii-case-insensitive href=https://infra.spec.whatwg.org/#ascii-case-insensitive data-x-internal=ascii-case-insensitive>ASCII case-insensitive</a> match for the
   string "<code>transparent</code>", then return an error.<li>
    <p>If <var>input</var> is an <a id=colours:ascii-case-insensitive-2 href=https://infra.spec.whatwg.org/#ascii-case-insensitive data-x-internal=ascii-case-insensitive>ASCII case-insensitive</a> match for one of the
    <a href=https://drafts.csswg.org/css-color/#named-color id=colours:named-color data-x-internal=named-color>named colors</a>, then return the <a href=#simple-colour id=colours:simple-colour-5>simple
    color</a> corresponding to that keyword. <a href=#refsCSSCOLOR>[CSSCOLOR]</a></p>

    <p class=note><a href=https://www.w3.org/TR/css3-color/#css2-system>CSS2 System
    Colors</a> are not recognized.</p>
   <li>
    <p>If <var>input</var>'s <a id=colours:code-point-length href=https://infra.spec.whatwg.org/#string-code-point-length data-x-internal=code-point-length>code point length</a> is four, and the first character in
    <var>input</var> is U+0023 (#), and the last three characters of <var>input</var> are all
    <a id=colours:ascii-hex-digits-3 href=https://infra.spec.whatwg.org/#ascii-hex-digit data-x-internal=ascii-hex-digits>ASCII hex digits</a>, then:</p>

    <ol><li><p>Let <var>result</var> be a <a href=#simple-colour id=colours:simple-colour-6>simple color</a>.</p>

     <li><p>Interpret the second character of <var>input</var> as a hexadecimal digit; let
     the red component of <var>result</var> be the resulting number multiplied by 17.</p>

     <li><p>Interpret the third character of <var>input</var> as a hexadecimal digit; let
     the green component of <var>result</var> be the resulting number multiplied by 17.</p>

     <li><p>Interpret the fourth character of <var>input</var> as a hexadecimal digit; let
     the blue component of <var>result</var> be the resulting number multiplied by 17.</p>

     <li><p>Return <var>result</var>.</p>
    </ol>
   <li><p>Replace any <a href=https://infra.spec.whatwg.org/#code-point id=colours:code-point data-x-internal=code-point>code points</a> greater than U+FFFF in
   <var>input</var> (i.e., any characters that are not in the basic multilingual plane) with the
   two-character string "<code>00</code>".<li><p>If <var>input</var>'s <a id=colours:code-point-length-2 href=https://infra.spec.whatwg.org/#string-code-point-length data-x-internal=code-point-length>code point length</a> is greater than 128, truncate
   <var>input</var>, leaving only the first 128 characters.<li><p>If the first character in <var>input</var> is a U+0023 NUMBER SIGN character (#),
   remove it.<li><p>Replace any character in <var>input</var> that is not an <a href=https://infra.spec.whatwg.org/#ascii-hex-digit id=colours:ascii-hex-digits-4 data-x-internal=ascii-hex-digits>ASCII hex digit</a> with the character U+0030 DIGIT ZERO (0).<li><p>While <var>input</var>'s <a id=colours:code-point-length-3 href=https://infra.spec.whatwg.org/#string-code-point-length data-x-internal=code-point-length>code point length</a> is zero or not a multiple of
   three, append a U+0030 DIGIT ZERO (0) character to <var>input</var>.<li><p>Split <var>input</var> into three strings of equal <a id=colours:code-point-length-4 href=https://infra.spec.whatwg.org/#string-code-point-length data-x-internal=code-point-length>code point length</a>, to
   obtain three components. Let <var>length</var> be the <a id=colours:code-point-length-5 href=https://infra.spec.whatwg.org/#string-code-point-length data-x-internal=code-point-length>code point length</a> that all of
   those components have (one third the <a id=colours:code-point-length-6 href=https://infra.spec.whatwg.org/#string-code-point-length data-x-internal=code-point-length>code point length</a> of <var>input</var>).<li><p>If <var>length</var> is greater than 8, then remove the leading <span><var>length</var>-8</span> characters in each component, and let <var>length</var> be 8.<li><p>While <var>length</var> is greater than two and the first character in each
   component is a U+0030 DIGIT ZERO (0) character, remove that character and reduce <var>length</var> by one.<li><p>If <var>length</var> is <em>still</em> greater than two, truncate each component,
   leaving only the first two characters in each.<li><p>Let <var>result</var> be a <a href=#simple-colour id=colours:simple-colour-7>simple color</a>.</p>

   <li><p>Interpret the first component as a hexadecimal number; let the red component of <var>result</var> be the resulting number.<li><p>Interpret the second component as a hexadecimal number; let the green component of <var>result</var> be the resulting number.<li><p>Interpret the third component as a hexadecimal number; let the blue component of <var>result</var> be the resulting number.<li><p>Return <var>result</var>.</p>
  </ol>

  

  <hr>

  <p class=note>The <a href=#canvas-context-2d id=colours:canvas-context-2d>2D graphics context</a> has a separate
  color syntax that also handles opacity.</p>



  <h4 id=space-separated-tokens><span class=secno>2.4.7</span> Space-separated tokens<a href=#space-separated-tokens class=self-link></a></h4>

  <p>A <dfn id=set-of-space-separated-tokens data-export="">set of space-separated tokens</dfn> is a string containing zero or more words
  (known as tokens) separated by one or more <a id=space-separated-tokens:space-characters href=https://infra.spec.whatwg.org/#ascii-whitespace data-x-internal=space-characters>ASCII whitespace</a>, where words consist of
  any string of one or more characters, none of which are <a id=space-separated-tokens:space-characters-2 href=https://infra.spec.whatwg.org/#ascii-whitespace data-x-internal=space-characters>ASCII whitespace</a>.</p>

  <p>A string containing a <a href=#set-of-space-separated-tokens id=space-separated-tokens:set-of-space-separated-tokens>set of space-separated tokens</a> may have leading or trailing
  <a id=space-separated-tokens:space-characters-3 href=https://infra.spec.whatwg.org/#ascii-whitespace data-x-internal=space-characters>ASCII whitespace</a>.</p>

  <p>An <dfn id=unordered-set-of-unique-space-separated-tokens data-export="">unordered set of unique space-separated tokens</dfn> is a <a href=#set-of-space-separated-tokens id=space-separated-tokens:set-of-space-separated-tokens-2>set of
  space-separated tokens</a> where none of the tokens are duplicated.</p>

  <p>An <dfn id=ordered-set-of-unique-space-separated-tokens>ordered set of unique space-separated tokens</dfn> is a <a href=#set-of-space-separated-tokens id=space-separated-tokens:set-of-space-separated-tokens-3>set of space-separated
  tokens</a> where none of the tokens are duplicated but where the order of the tokens is
  meaningful.</p>

  <p><a href=#set-of-space-separated-tokens id=space-separated-tokens:set-of-space-separated-tokens-4>Sets of space-separated tokens</a> sometimes
  have a defined set of allowed values. When a set of allowed values is defined, the tokens must all
  be from that list of allowed values; other values are non-conforming. If no such set of allowed
  values is provided, then all values are conforming.</p>

  <p class=note>How tokens in a <a href=#set-of-space-separated-tokens id=space-separated-tokens:set-of-space-separated-tokens-5>set of space-separated tokens</a> are to be compared
  (e.g. case-sensitively or not) is defined on a per-set basis.</p>



  <h4 id=comma-separated-tokens><span class=secno>2.4.8</span> Comma-separated tokens<a href=#comma-separated-tokens class=self-link></a></h4>

  <p>A <dfn id=set-of-comma-separated-tokens>set of comma-separated tokens</dfn> is a string containing zero or more tokens each
  separated from the next by a single U+002C COMMA character (,), where tokens consist of any string
  of zero or more characters, neither beginning nor ending with <a id=comma-separated-tokens:space-characters href=https://infra.spec.whatwg.org/#ascii-whitespace data-x-internal=space-characters>ASCII whitespace</a>, nor
  containing any U+002C COMMA characters (,), and optionally surrounded by <a id=comma-separated-tokens:space-characters-2 href=https://infra.spec.whatwg.org/#ascii-whitespace data-x-internal=space-characters>ASCII
  whitespace</a>.</p>

  <p class=example>For instance, the string "<code> a ,b,,d d </code>" consists of four tokens: "a", "b", the empty
  string, and "d d". Leading and trailing whitespace around each token doesn't count as part of
  the token, and the empty string can be a token.</p>

  <p><a href=#set-of-comma-separated-tokens id=comma-separated-tokens:set-of-comma-separated-tokens>Sets of comma-separated tokens</a> sometimes
  have further restrictions on what consists a valid token. When such restrictions are defined, the
  tokens must all fit within those restrictions; other values are non-conforming. If no such
  restrictions are specified, then all values are conforming.</p>



  <h4 id=syntax-references><span class=secno>2.4.9</span> References<a href=#syntax-references class=self-link></a></h4>

  <p>A <dfn id=valid-hash-name-reference>valid hash-name reference</dfn> to an element of type <var>type</var> is a
  string consisting of a U+0023 NUMBER SIGN character (#) followed by a string which exactly matches
  the value of the <code>name</code> attribute of an element with type <var>type</var> in
  the same <a id=syntax-references:tree href=https://dom.spec.whatwg.org/#concept-tree data-x-internal=tree>tree</a>.</p>

  

  <p>The <dfn id=rules-for-parsing-a-hash-name-reference>rules for parsing a hash-name reference</dfn> to an element of type <var>type</var>,
  given a context node <var>scope</var>, are as follows:</p>

  <ol><li><p>If the string being parsed does not contain a U+0023 NUMBER SIGN character, or if the
   first such character in the string is the last character in the string, then return null.<li><p>Let <var>s</var> be the string from the character immediately after the first
   U+0023 NUMBER SIGN character in the string being parsed up to the end of that string.<li>
    <p>Return the first element of type <var>type</var> in <var>scope</var>'s <a id=syntax-references:tree-2 href=https://dom.spec.whatwg.org/#concept-tree data-x-internal=tree>tree</a>, in
    <a id=syntax-references:tree-order href=https://dom.spec.whatwg.org/#concept-tree-order data-x-internal=tree-order>tree order</a>, that has an <code id=syntax-references:the-id-attribute><a href=#the-id-attribute>id</a></code> or <code>name</code> attribute whose value is <var>s</var>, or null if there is no such
    element.</p>

    <p class=note>Although <code id=syntax-references:the-id-attribute-2><a href=#the-id-attribute>id</a></code> attributes are accounted for when
    parsing, they are not used in determining whether a value is a <a href=#valid-hash-name-reference id=syntax-references:valid-hash-name-reference><em>valid</em> hash-name
    reference</a>. That is, a hash-name reference that refers to an element based on <code id=syntax-references:the-id-attribute-3><a href=#the-id-attribute>id</a></code> is a conformance error (unless that element also has a <code>name</code> attribute with the same value).</p>
   </ol>

  


  <h4 id=mq><span class=secno>2.4.10</span> Media queries<a href=#mq class=self-link></a></h4>

  <p>A string is a <dfn id=valid-media-query-list>valid media query list</dfn> if it matches the <code>&lt;media-query-list></code> production of <cite>Media Queries</cite>. <a href=#refsMQ>[MQ]</a></p>

  <p>A string <dfn id=matches-the-environment>matches the environment</dfn> of the user if it is the empty string, a string
  consisting of only <a id=mq:space-characters href=https://infra.spec.whatwg.org/#ascii-whitespace data-x-internal=space-characters>ASCII whitespace</a>, or is a media query list that matches the user's
  environment according to the definitions given in <cite>Media Queries</cite>. <a href=#refsMQ>[MQ]</a></p>




  <h3 id=urls><span class=secno>2.5</span> URLs<a href=#urls class=self-link></a></h3>

  <h4 id=terminology-2><span class=secno>2.5.1</span> Terminology<a href=#terminology-2 class=self-link></a></h4>

  <p>A string is a <dfn id=valid-non-empty-url>valid non-empty URL</dfn> if it is a <a id=terminology-2:valid-url-string href=https://url.spec.whatwg.org/#valid-url-string data-x-internal=valid-url-string>valid URL string</a> but it is
  not the empty string.</p>

  <p>A string is a <dfn id=valid-url-potentially-surrounded-by-spaces>valid URL potentially surrounded by spaces</dfn> if, after <a href=https://infra.spec.whatwg.org/#strip-leading-and-trailing-ascii-whitespace id=terminology-2:strip-leading-and-trailing-ascii-whitespace data-x-internal=strip-leading-and-trailing-ascii-whitespace>stripping leading and trailing ASCII
  whitespace</a> from it, it is a <a id=terminology-2:valid-url-string-2 href=https://url.spec.whatwg.org/#valid-url-string data-x-internal=valid-url-string>valid URL string</a>.</p>

  <p>A string is a <dfn id=valid-non-empty-url-potentially-surrounded-by-spaces>valid non-empty URL potentially surrounded by spaces</dfn> if, after <a href=https://infra.spec.whatwg.org/#strip-leading-and-trailing-ascii-whitespace id=terminology-2:strip-leading-and-trailing-ascii-whitespace-2 data-x-internal=strip-leading-and-trailing-ascii-whitespace>stripping leading and trailing ASCII
  whitespace</a> from it, it is a <a href=#valid-non-empty-url id=terminology-2:valid-non-empty-url>valid non-empty URL</a>.</p>

  <p>This specification defines the URL <dfn id=about:legacy-compat><code>about:legacy-compat</code></dfn> as a reserved,
  though unresolvable, <code id=terminology-2:about-protocol><a data-x-internal=about-protocol href=https://tools.ietf.org/html/rfc6694#section-2>about:</a></code> URL, for use in <a href=#syntax-doctype id=terminology-2:syntax-doctype>DOCTYPE</a>s in <a id=terminology-2:html-documents href=https://dom.spec.whatwg.org/#html-document data-x-internal=html-documents>HTML documents</a> when needed for
  compatibility with XML tools. <a href=#refsABOUT>[ABOUT]</a></p>

  <p>This specification defines the URL <dfn id=about:html-kind><code>about:html-kind</code></dfn> as a reserved,
  though unresolvable, <code id=terminology-2:about-protocol-2><a data-x-internal=about-protocol href=https://tools.ietf.org/html/rfc6694#section-2>about:</a></code> URL, that is used as an
  identifier for kinds of media tracks. <a href=#refsABOUT>[ABOUT]</a></p>

  <p>This specification defines the URL <dfn id=about:srcdoc><code>about:srcdoc</code></dfn> as a reserved, though
  unresolvable, <code id=terminology-2:about-protocol-3><a data-x-internal=about-protocol href=https://tools.ietf.org/html/rfc6694#section-2>about:</a></code> URL, that is used as the <a href=https://dom.spec.whatwg.org/#concept-document-url id="terminology-2:the-document's-address" data-x-internal="the-document's-address">URL</a> of <a href=#an-iframe-srcdoc-document id=terminology-2:an-iframe-srcdoc-document><code>iframe</code> <code>srcdoc</code> documents</a>.
  <a href=#refsABOUT>[ABOUT]</a></p>

  <p>The <dfn id=fallback-base-url>fallback base URL</dfn> of a <code id=terminology-2:document><a href=#document>Document</a></code> object <var>document</var> is the
  <a id=terminology-2:url-record href=https://url.spec.whatwg.org/#concept-url data-x-internal=url-record>URL record</a> obtained by running these steps:</p>

  <ol><li><p>If <var>document</var> is <a href=#an-iframe-srcdoc-document id=terminology-2:an-iframe-srcdoc-document-2>an <code>iframe</code> <code>srcdoc</code> document</a>, then return the <a href=#document-base-url id=terminology-2:document-base-url>document base
   URL</a> of <var>document</var>'s <a href=#concept-document-bc id=terminology-2:concept-document-bc>browsing context</a>'s
   <a href=#bc-container-document id=terminology-2:bc-container-document>container document</a>.<li>
    

    

    <p>If <var>document</var>'s <a href=https://dom.spec.whatwg.org/#concept-document-url id="terminology-2:the-document's-address-2" data-x-internal="the-document's-address">URL</a> is
    <code id=terminology-2:about:blank><a href=#about:blank>about:blank</a></code>, and <var>document</var>'s <a href=#concept-document-bc id=terminology-2:concept-document-bc-2>browsing
    context</a>'s <a href=#creator-base-url id=terminology-2:creator-base-url>creator base URL</a> is non-null, then return that <a href=#creator-base-url id=terminology-2:creator-base-url-2>creator base
    URL</a>.</p>
   <li><p>Return <var>document</var>'s <a href=https://dom.spec.whatwg.org/#concept-document-url id="terminology-2:the-document's-address-3" data-x-internal="the-document's-address">URL</a>.</ol>

  <p>The <dfn id=document-base-url data-export="">document base URL</dfn> of a <code id=terminology-2:document-2><a href=#document>Document</a></code> object is the
  <a id=terminology-2:absolute-url href=https://url.spec.whatwg.org/#syntax-url-absolute data-x-internal=absolute-url>absolute URL</a> obtained by running these steps:</p>

  <ol><li><p>If there is no <code id=terminology-2:the-base-element><a href=#the-base-element>base</a></code> element that has an <code id=terminology-2:attr-base-href><a href=#attr-base-href>href</a></code> attribute in the <code id=terminology-2:document-3><a href=#document>Document</a></code>, then return the
   <code id=terminology-2:document-4><a href=#document>Document</a></code>'s <a href=#fallback-base-url id=terminology-2:fallback-base-url>fallback base URL</a>.<li><p>Otherwise, return the <a href=#frozen-base-url id=terminology-2:frozen-base-url>frozen base URL</a> of the first <code id=terminology-2:the-base-element-2><a href=#the-base-element>base</a></code> element
   in the <code id=terminology-2:document-5><a href=#document>Document</a></code> that has an <code id=terminology-2:attr-base-href-2><a href=#attr-base-href>href</a></code> attribute, in
   <a id=terminology-2:tree-order href=https://dom.spec.whatwg.org/#concept-tree-order data-x-internal=tree-order>tree order</a>.</ol>


  

  <h4 id=resolving-urls><span class=secno>2.5.2</span> Parsing URLs<a href=#resolving-urls class=self-link></a></h4>

  <p>Parsing a URL is the process of taking a string and obtaining the <a id=resolving-urls:url-record href=https://url.spec.whatwg.org/#concept-url data-x-internal=url-record>URL record</a> that
  it represents. While this process is defined in <cite>URL</cite>, the HTML standard defines
  a wrapper for convenience. <a href=#refsURL>[URL]</a></p>

  <p class=note>This wrapper is only useful when the character encoding for the URL parser has to
  match that of the document or environment settings object for legacy reasons. When that is not the
  case the <a id=resolving-urls:url-parser href=https://url.spec.whatwg.org/#concept-url-parser data-x-internal=url-parser>URL parser</a> can be used directly.</p>

  <p id=resolve-a-url>To <dfn id=parse-a-url>parse a URL</dfn> <var>url</var>, relative to either a
  <var>document</var> or <var>environment settings object</var>, the user agent must use the
  following steps. Parsing a URL either results in failure or a <a href=#resulting-url-string id=resolving-urls:resulting-url-string>resulting URL string</a>
  and <a href=#resulting-url-record id=resolving-urls:resulting-url-record>resulting URL record</a>.</p>

  <ol><li><p>Let <var>encoding</var> be <var>document</var>'s <a href=https://dom.spec.whatwg.org/#concept-document-encoding id="resolving-urls:document's-character-encoding" data-x-internal="document's-character-encoding">character encoding</a>, if <var>document</var> was given, and <var>environment
   settings object</var>'s <a href=#api-url-character-encoding id=resolving-urls:api-url-character-encoding>API URL character encoding</a> otherwise.<li><p>Let <var>baseURL</var> be <var>document</var>'s <a href=#document-base-url id=resolving-urls:document-base-url>base
   URL</a>, if <var>document</var> was given, and <var>environment settings object</var>'s
   <a href=#api-base-url id=resolving-urls:api-base-url>API base URL</a> otherwise.<li><p>Let <var>urlRecord</var> be the result of applying the <a id=resolving-urls:url-parser-2 href=https://url.spec.whatwg.org/#concept-url-parser data-x-internal=url-parser>URL parser</a> to
   <var>url</var>, with <var>baseURL</var> and <var>encoding</var>.<li><p>If <var>urlRecord</var> is failure, then return failure.<li><p>Let <var>urlString</var> be the result of applying the <a href=https://url.spec.whatwg.org/#concept-url-serializer id=resolving-urls:concept-url-serializer data-x-internal=concept-url-serializer>URL serializer</a> to <var>urlRecord</var>.<li><p>Return <var>urlString</var> as the <dfn id=resulting-url-string>resulting URL string</dfn> and
   <var>urlRecord</var> as the <dfn id=resulting-url-record>resulting URL record</dfn>.</ol>

  



  

  <h4 id=dynamic-changes-to-base-urls><span class=secno>2.5.3</span> Dynamic changes to base URLs<a href=#dynamic-changes-to-base-urls class=self-link></a></h4>

  <p>When a document's <a href=#document-base-url id=dynamic-changes-to-base-urls:document-base-url>document base URL</a> changes, all elements in that document are
  <a href=#affected-by-a-base-url-change id=dynamic-changes-to-base-urls:affected-by-a-base-url-change>affected by a base URL change</a>.</p>

  <p>The following are <a href=#base-url-change-steps id=dynamic-changes-to-base-urls:base-url-change-steps>base URL change steps</a>, which run when an element is
  <a href=#affected-by-a-base-url-change id=dynamic-changes-to-base-urls:affected-by-a-base-url-change-2>affected by a base URL change</a> (as defined by <cite>DOM</cite>):</p>

  <dl class=switch><dt>If the element creates a <a href=#hyperlink id=dynamic-changes-to-base-urls:hyperlink>hyperlink</a><dd>
    <p>If the <a id=dynamic-changes-to-base-urls:url href=https://url.spec.whatwg.org/#concept-url data-x-internal=url>URL</a> identified by the hyperlink is being shown to the user, or if any
    data derived from that <a id=dynamic-changes-to-base-urls:url-2 href=https://url.spec.whatwg.org/#concept-url data-x-internal=url>URL</a> is affecting the display, then the <code id=dynamic-changes-to-base-urls:attr-hyperlink-href><a href=#attr-hyperlink-href>href</a></code> attribute should be <a href=#parse-a-url id=dynamic-changes-to-base-urls:parse-a-url>reparsed</a> relative to the element's <a id=dynamic-changes-to-base-urls:node-document href=https://dom.spec.whatwg.org/#concept-node-document data-x-internal=node-document>node document</a> and the UI updated
    appropriately.</p>

    <p class=example>For example, the CSS <code id=dynamic-changes-to-base-urls:selector-link><a href=#selector-link>:link</a></code>/<code id=dynamic-changes-to-base-urls:selector-visited><a href=#selector-visited>:visited</a></code> <a href=https://drafts.csswg.org/selectors/#pseudo-class id=dynamic-changes-to-base-urls:pseudo-class data-x-internal=pseudo-class>pseudo-classes</a>
    might have been affected.</p>

    <p>If the hyperlink has a <code id=dynamic-changes-to-base-urls:ping><a href=#ping>ping</a></code> attribute and its
    <a href=https://url.spec.whatwg.org/#concept-url id=dynamic-changes-to-base-urls:url-3 data-x-internal=url>URL(s)</a> are being shown to the user, then the <code id=dynamic-changes-to-base-urls:ping-2><a href=#ping>ping</a></code> attribute's tokens should be <a href=#parse-a-url id=dynamic-changes-to-base-urls:parse-a-url-2>reparsed</a> relative to the element's <a id=dynamic-changes-to-base-urls:node-document-2 href=https://dom.spec.whatwg.org/#concept-node-document data-x-internal=node-document>node document</a> and the UI updated
    appropriately.</p>
   <dt>If the element is a <code id=dynamic-changes-to-base-urls:the-q-element><a href=#the-q-element>q</a></code>, <code id=dynamic-changes-to-base-urls:the-blockquote-element><a href=#the-blockquote-element>blockquote</a></code>, <code id=dynamic-changes-to-base-urls:the-ins-element><a href=#the-ins-element>ins</a></code>, or
   <code id=dynamic-changes-to-base-urls:the-del-element><a href=#the-del-element>del</a></code> element with a <code>cite</code> attribute<dd>
    <p>If the <a id=dynamic-changes-to-base-urls:url-4 href=https://url.spec.whatwg.org/#concept-url data-x-internal=url>URL</a> identified by the <code>cite</code> attribute is being
    shown to the user, or if any data derived from that <a id=dynamic-changes-to-base-urls:url-5 href=https://url.spec.whatwg.org/#concept-url data-x-internal=url>URL</a> is affecting the display,
    then the <a id=dynamic-changes-to-base-urls:url-6 href=https://url.spec.whatwg.org/#concept-url data-x-internal=url>URL</a> should be <a href=#parse-a-url id=dynamic-changes-to-base-urls:parse-a-url-3>reparsed</a> relative to the
    element's <a id=dynamic-changes-to-base-urls:node-document-3 href=https://dom.spec.whatwg.org/#concept-node-document data-x-internal=node-document>node document</a> and the UI updated appropriately.</p>
   <dt>Otherwise<dd>
    <p>The element is not directly affected.</p>

    <p class=example>For instance, changing the base URL doesn't affect the image displayed by
    <code id=dynamic-changes-to-base-urls:the-img-element><a href=#the-img-element>img</a></code> elements, although subsequent accesses of the <code id=dynamic-changes-to-base-urls:dom-img-src><a href=#dom-img-src>src</a></code> IDL attribute from script will return a new <a id=dynamic-changes-to-base-urls:absolute-url href=https://url.spec.whatwg.org/#syntax-url-absolute data-x-internal=absolute-url>absolute
    URL</a> that might no longer correspond to the image being shown.</p>
   </dl>

  


  

  <h3 id=fetching-resources><span class=secno>2.6</span> Fetching resources<a href=#fetching-resources class=self-link></a></h3>

  <h4 id=terminology-3><span class=secno>2.6.1</span> Terminology<a href=#terminology-3 class=self-link></a></h4>

  <p>A <a href=https://fetch.spec.whatwg.org/#concept-response id=terminology-3:concept-response data-x-internal=concept-response>response</a> whose <a href=https://fetch.spec.whatwg.org/#concept-response-type id=terminology-3:concept-response-type data-x-internal=concept-response-type>type</a> is "<code>basic</code>", "<code>cors</code>", or "<code>default</code>" is <dfn id=cors-same-origin data-export="">CORS-same-origin</dfn>. <a href=#refsFETCH>[FETCH]</a></p>

  <p>A <a href=https://fetch.spec.whatwg.org/#concept-response id=terminology-3:concept-response-2 data-x-internal=concept-response>response</a> whose <a href=https://fetch.spec.whatwg.org/#concept-response-type id=terminology-3:concept-response-type-2 data-x-internal=concept-response-type>type</a> is "<code>opaque</code>" or "<code>opaqueredirect</code>" is <dfn id=cors-cross-origin>CORS-cross-origin</dfn>.</p>

  <p>A <a href=https://fetch.spec.whatwg.org/#concept-response id=terminology-3:concept-response-3 data-x-internal=concept-response>response</a>'s <dfn id=unsafe-response data-export="">unsafe response</dfn> is its
  <a href=https://fetch.spec.whatwg.org/#concept-internal-response id=terminology-3:concept-internal-response data-x-internal=concept-internal-response>internal response</a> if it has one, and the <a href=https://fetch.spec.whatwg.org/#concept-response id=terminology-3:concept-response-4 data-x-internal=concept-response>response</a> itself otherwise.</p>

  <p>To <dfn id=create-a-potential-cors-request>create a potential-CORS request</dfn>, given a <var>url</var>, <var>destination</var>,
  <var>corsAttributeState</var>, and an optional <i>same-origin fallback flag</i>, run these
  steps:</p>

  <ol><li><p>Let <var>mode</var> be "<code>no-cors</code>" if <var>corsAttributeState</var>
   is <a href=#attr-crossorigin-none id=terminology-3:attr-crossorigin-none>No CORS</a>, and "<code>cors</code>"
   otherwise.<li><p>If <i>same-origin fallback flag</i> is set and <var>mode</var> is "<code>no-cors</code>", set <var>mode</var> to "<code>same-origin</code>".<li><p>Let <var>credentialsMode</var> be "<code>include</code>".<li><p>If <var>corsAttributeState</var> is <a href=#attr-crossorigin-anonymous id=terminology-3:attr-crossorigin-anonymous>Anonymous</a>, set <var>credentialsMode</var> to "<code>same-origin</code>".<li><p>Let <var>request</var> be a new <a href=https://fetch.spec.whatwg.org/#concept-request id=terminology-3:concept-request data-x-internal=concept-request>request</a> whose
   <a href=https://fetch.spec.whatwg.org/#concept-request-url id=terminology-3:concept-request-url data-x-internal=concept-request-url>url</a> is <var>url</var>, <a href=https://fetch.spec.whatwg.org/#concept-request-destination id=terminology-3:concept-request-destination data-x-internal=concept-request-destination>destination</a> is <var>destination</var>,
   <a href=https://fetch.spec.whatwg.org/#concept-request-mode id=terminology-3:concept-request-mode data-x-internal=concept-request-mode>mode</a> is <var>mode</var>, <a href=https://fetch.spec.whatwg.org/#concept-request-credentials-mode id=terminology-3:concept-request-credentials-mode data-x-internal=concept-request-credentials-mode>credentials mode</a> is
   <var>credentialsMode</var>, and whose <a id=terminology-3:use-url-credentials-flag href=https://fetch.spec.whatwg.org/#concept-request-use-url-credentials-flag data-x-internal=use-url-credentials-flag>use-URL-credentials flag</a> is set.</ol>


  <h4 id=content-type-sniffing><span class=secno>2.6.2</span> Determining the type of a resource<a href=#content-type-sniffing class=self-link></a></h4>

  <p>The <dfn id=content-type>Content-Type metadata</dfn> of a resource must be obtained and
  interpreted in a manner consistent with the requirements of <cite>MIME Sniffing</cite>.
  <a href=#refsMIMESNIFF>[MIMESNIFF]</a></p>

  <p>The <dfn id=content-type-sniffing-2><a href=https://mimesniff.spec.whatwg.org/#computed-mime-type>computed MIME type</a></dfn> of a
  resource must be found in a manner consistent with the requirements given in <cite>MIME
  Sniffing</cite>. <a href=#refsMIMESNIFF>[MIMESNIFF]</a></p>

  <p>The <dfn id=content-type-sniffing:-image><a href=https://mimesniff.spec.whatwg.org/#rules-for-sniffing-images-specifically>rules for
  sniffing images specifically</a></dfn>, the <dfn id=content-type-sniffing:-text-or-binary><a href=https://mimesniff.spec.whatwg.org/#rules-for-text-or-binary>rules for distinguishing
  if a resource is text or binary</a></dfn>, and the <dfn id=content-type-sniffing:-video><a href=https://mimesniff.spec.whatwg.org/#rules-for-sniffing-audio-and-video-specifically>rules
  for sniffing audio and video specifically</a></dfn> are also defined in <cite>MIME Sniffing</cite>. These rules return a <a id=content-type-sniffing:mime-type href=https://mimesniff.spec.whatwg.org/#mime-type data-x-internal=mime-type>MIME type</a> as their result. <a href=#refsMIMESNIFF>[MIMESNIFF]</a></p>

  <p class=warning>It is imperative that the rules in <cite>MIME Sniffing</cite> be
  followed exactly. When a user agent uses different heuristics for content type detection than the
  server expects, security problems can occur. For more details, see <cite>MIME Sniffing</cite>. <a href=#refsMIMESNIFF>[MIMESNIFF]</a></p>


  <h4 id=extracting-character-encodings-from-meta-elements><span class=secno>2.6.3</span> Extracting character encodings from <code id=extracting-character-encodings-from-meta-elements:the-meta-element><a href=#the-meta-element>meta</a></code> elements<a href=#extracting-character-encodings-from-meta-elements class=self-link></a></h4>

  <p>The <dfn id=algorithm-for-extracting-a-character-encoding-from-a-meta-element>algorithm for extracting a character encoding from a <code>meta</code> element</dfn>,
  given a string <var>s</var>, is as follows. It either returns a character encoding or
  nothing.</p>

  <ol><li><p>Let <var>position</var> be a pointer into <var>s</var>, initially
   pointing at the start of the string.<li><p><i>Loop</i>: Find the first seven characters in <var>s</var> after <var>position</var> that are an <a id=extracting-character-encodings-from-meta-elements:ascii-case-insensitive href=https://infra.spec.whatwg.org/#ascii-case-insensitive data-x-internal=ascii-case-insensitive>ASCII case-insensitive</a> match for the word "<code>charset</code>". If no such match is found, return nothing.<li><p>Skip any <a id=extracting-character-encodings-from-meta-elements:space-characters href=https://infra.spec.whatwg.org/#ascii-whitespace data-x-internal=space-characters>ASCII whitespace</a> that immediately follow the word "<code>charset</code>" (there might not be any).<li><p>If the next character is not a U+003D EQUALS SIGN (=), then move <var>position</var> to point just before that next character, and jump back to the step
   labeled <i>loop</i>.<li><p>Skip any <a id=extracting-character-encodings-from-meta-elements:space-characters-2 href=https://infra.spec.whatwg.org/#ascii-whitespace data-x-internal=space-characters>ASCII whitespace</a> that immediately follow the equals sign (there
   might not be any).<li>
    <p>Process the next character as follows:</p>

    <dl class=switch><dt>If it is a U+0022 QUOTATION MARK character (") and there is a later U+0022 QUOTATION MARK character (") in <var>s</var><dt>If it is a U+0027 APOSTROPHE character (') and there is a later U+0027 APOSTROPHE character (') in <var>s</var><dd>Return the result of <a id=extracting-character-encodings-from-meta-elements:getting-an-encoding href=https://encoding.spec.whatwg.org/#concept-encoding-get data-x-internal=getting-an-encoding>getting an encoding</a> from the substring that is between
     this character and the next earliest occurrence of this character.<dt>If it is an unmatched U+0022 QUOTATION MARK character (")<dt>If it is an unmatched U+0027 APOSTROPHE character (')<dt>If there is no next character<dd>Return nothing.<dt>Otherwise<dd>Return the result of <a id=extracting-character-encodings-from-meta-elements:getting-an-encoding-2 href=https://encoding.spec.whatwg.org/#concept-encoding-get data-x-internal=getting-an-encoding>getting an encoding</a> from the substring that consists of
     this character up to but not including the first <a id=extracting-character-encodings-from-meta-elements:space-characters-3 href=https://infra.spec.whatwg.org/#ascii-whitespace data-x-internal=space-characters>ASCII whitespace</a> or U+003B
     SEMICOLON character (;), or the end of <var>s</var>, whichever comes first.</dl>
   </ol>

  <p class=note>This algorithm is distinct from those in the HTTP specifications (for example,
  HTTP doesn't allow the use of single quotes and requires supporting a backslash-escape mechanism
  that is not supported by this algorithm). While the
  algorithm is used in contexts that, historically, were related to HTTP, the syntax as supported by
  implementations diverged some time ago. <a href=#refsHTTP>[HTTP]</a></p>

  


  <h4 id=cors-settings-attributes><span class=secno>2.6.4</span> CORS settings attributes<a href=#cors-settings-attributes class=self-link></a></h4><div class="mdn-anno wrapped"><button onclick=toggleStatus(this) class=mdn-anno-btn><b title="Support in all current engines." class=all-engines-flag>✔</b><span>MDN</span></button><div class=feature><p><a href=https://developer.mozilla.org/en-US/docs/Web/HTML/Attributes/crossorigin title="The crossorigin attribute, valid on the <audio>, <img>, <link>, <script>, and <video> elements, provides support for CORS, defining how the element handles crossorigin requests, thereby enabling the configuration of the CORS requests for the element's fetched data. Depending on the element, the attribute can be a CORS settings attribute.">Attributes/crossorigin</a><p class=all-engines-text>Support in all current engines.<div class=support><span class="firefox yes"><span>Firefox</span><span>Yes</span></span><span class="safari yes"><span>Safari</span><span>Yes</span></span><span class="chrome yes"><span>Chrome</span><span>Yes</span></span><hr><span class="opera yes"><span>Opera</span><span>Yes</span></span><span class="edge_blink yes"><span>Edge</span><span>Yes</span></span><hr><span class="edge yes"><span>Edge (Legacy)</span><span>12+</span></span><span class="ie yes"><span>Internet Explorer</span><span>Yes</span></span><hr><span class="firefox_android yes"><span>Firefox Android</span><span>Yes</span></span><span class="safari_ios yes"><span>Safari iOS</span><span>Yes</span></span><span class="chrome_android yes"><span>Chrome Android</span><span>Yes</span></span><span class="webview_android yes"><span>WebView Android</span><span>Yes</span></span><span class="samsunginternet_android yes"><span>Samsung Internet</span><span>Yes</span></span><span class="opera_android yes"><span>Opera Android</span><span>Yes</span></span></div></div></div>

  <p>A <dfn id=cors-settings-attribute>CORS settings attribute</dfn> is an <a href=#enumerated-attribute id=cors-settings-attributes:enumerated-attribute>enumerated attribute</a>. The following
  table lists the keywords and states for the attribute — the states given in the first cell
  of the rows with keywords give the states to which those keywords map.</p>

  <table><thead><tr><th> State
     <th> Keywords
     <th> Brief description
   <tbody><tr><td rowspan=2><dfn id=attr-crossorigin-anonymous>Anonymous</dfn>
     <td><dfn id=attr-crossorigin-anonymous-keyword><code>anonymous</code></dfn>
     <td rowspan=2><a href=https://fetch.spec.whatwg.org/#concept-request id=cors-settings-attributes:concept-request data-x-internal=concept-request>Requests</a> for the element will have their
     <a href=https://fetch.spec.whatwg.org/#concept-request-mode id=cors-settings-attributes:concept-request-mode data-x-internal=concept-request-mode>mode</a> set to "<code>cors</code>" and their
     <a href=https://fetch.spec.whatwg.org/#concept-request-credentials-mode id=cors-settings-attributes:concept-request-credentials-mode data-x-internal=concept-request-credentials-mode>credentials mode</a> set to "<code>same-origin</code>".
    <tr><td>(the empty string)
    <tr><td><dfn id=attr-crossorigin-use-credentials>Use Credentials</dfn>
     <td><dfn id=attr-crossorigin-use-credentials-keyword><code>use-credentials</code></dfn>
     <td><a href=https://fetch.spec.whatwg.org/#concept-request id=cors-settings-attributes:concept-request-2 data-x-internal=concept-request>Requests</a> for the element will have their <a href=https://fetch.spec.whatwg.org/#concept-request-mode id=cors-settings-attributes:concept-request-mode-2 data-x-internal=concept-request-mode>mode</a> set to "<code>cors</code>" and their <a href=https://fetch.spec.whatwg.org/#concept-request-credentials-mode id=cors-settings-attributes:concept-request-credentials-mode-2 data-x-internal=concept-request-credentials-mode>credentials mode</a> set to "<code>include</code>".
  </table>

  <p>The attribute's <i id=cors-settings-attributes:invalid-value-default><a href=#invalid-value-default>invalid value default</a></i> is the <a href=#attr-crossorigin-anonymous id=cors-settings-attributes:attr-crossorigin-anonymous>Anonymous</a> state, and its <i id=cors-settings-attributes:missing-value-default><a href=#missing-value-default>missing value default</a></i> is the <dfn id=attr-crossorigin-none>No CORS</dfn> state.
  For the purposes of <a href=#reflect id=cors-settings-attributes:reflect>reflection</a>, the <a href=#canonical-keyword id=cors-settings-attributes:canonical-keyword>canonical keyword</a>
  for the <a href=#attr-crossorigin-anonymous id=cors-settings-attributes:attr-crossorigin-anonymous-2>Anonymous</a> state is the <code id=cors-settings-attributes:attr-crossorigin-anonymous-keyword><a href=#attr-crossorigin-anonymous-keyword>anonymous</a></code> keyword.</p>

  <p>The majority of fetches governed by <a href=#cors-settings-attribute id=cors-settings-attributes:cors-settings-attribute>CORS settings
  attributes</a> will be done via the <a href=#create-a-potential-cors-request id=cors-settings-attributes:create-a-potential-cors-request>create a potential-CORS request</a> algorithm.</p>

  <p id=module-script-credentials-mode>For more modern features, where the request's <a href=https://fetch.spec.whatwg.org/#concept-request-mode id=cors-settings-attributes:concept-request-mode-3 data-x-internal=concept-request-mode>mode</a> is always "<code>cors</code>", certain <a href=#cors-settings-attribute id=cors-settings-attributes:cors-settings-attribute-2>CORS settings attributes</a> have been repurposed to have a
  slightly different meaning, wherein they only impact the <a href=https://fetch.spec.whatwg.org/#concept-request id=cors-settings-attributes:concept-request-3 data-x-internal=concept-request>request</a>'s <a href=https://fetch.spec.whatwg.org/#concept-request-credentials-mode id=cors-settings-attributes:concept-request-credentials-mode-3 data-x-internal=concept-request-credentials-mode>credentials mode</a>. To perform this translation, we
  define the <dfn id=cors-settings-attribute-credentials-mode>CORS settings attribute credentials mode</dfn> for a given <a href=#cors-settings-attribute id=cors-settings-attributes:cors-settings-attribute-3>CORS settings
  attribute</a> to be determined by switching on the attribute's state:</p>

  <dl class=switch><dt><a href=#attr-crossorigin-none id=cors-settings-attributes:attr-crossorigin-none>No CORS</a><dt><a href=#attr-crossorigin-anonymous id=cors-settings-attributes:attr-crossorigin-anonymous-3>Anonymous</a><dd>"<code>same-origin</code>"<dt><a href=#attr-crossorigin-none id=cors-settings-attributes:attr-crossorigin-none-2>Use Credentials</a><dd>"<code>include</code>"</dl>


  <h4 id=referrer-policy-attributes><span class=secno>2.6.5</span> Referrer policy attributes<a href=#referrer-policy-attributes class=self-link></a></h4>

  <p>A <dfn id=referrer-policy-attribute data-export="">referrer policy attribute</dfn> is an <a href=#enumerated-attribute id=referrer-policy-attributes:enumerated-attribute>enumerated attribute</a>.  Each
  <a id=referrer-policy-attributes:referrer-policy href=https://w3c.github.io/webappsec-referrer-policy/#referrer-policy data-x-internal=referrer-policy>referrer policy</a>, including the empty string, is a keyword for this attribute, mapping
  to a state of the same name.</p>

  <p>The attribute's <i id=referrer-policy-attributes:invalid-value-default><a href=#invalid-value-default>invalid value default</a></i> and <i id=referrer-policy-attributes:missing-value-default><a href=#missing-value-default>missing value default</a></i> are both the empty string state.</p>

  <p>The impact of these states on the processing model of various <a href=https://fetch.spec.whatwg.org/#concept-fetch id=referrer-policy-attributes:concept-fetch data-x-internal=concept-fetch>fetches</a> is defined in more detail throughout this specification, in
  <cite>Fetch</cite>, and in <cite>Referrer Policy</cite>. <a href=#refsFETCH>[FETCH]</a> <a href=#refsREFERRERPOLICY>[REFERRERPOLICY]</a></p>

  <div class=note>
   <p>Several signals can contribute to which processing model is used for a given <a href=https://fetch.spec.whatwg.org/#concept-fetch id=referrer-policy-attributes:concept-fetch-2 data-x-internal=concept-fetch>fetch</a>; a <a href=#referrer-policy-attribute id=referrer-policy-attributes:referrer-policy-attribute>referrer policy attribute</a> is only one of
   them. In general, the order in which these signals are processed are:</p>

   <ol><li><p>First, the presence of a <code id=referrer-policy-attributes:link-type-noreferrer><a href=#link-type-noreferrer>noreferrer</a></code> link
    type;<li><p>Then, the value of a <a href=#referrer-policy-attribute id=referrer-policy-attributes:referrer-policy-attribute-2>referrer policy attribute</a>;<li><p>Then, the presence of any <code id=referrer-policy-attributes:the-meta-element><a href=#the-meta-element>meta</a></code> element with <code id=referrer-policy-attributes:attr-meta-name><a href=#attr-meta-name>name</a></code> attribute set to <code id=referrer-policy-attributes:meta-referrer><a href=#meta-referrer>referrer</a></code>.<li><p>Finally, the `<code id=referrer-policy-attributes:http-referrer-policy><a data-x-internal=http-referrer-policy href=https://w3c.github.io/webappsec-referrer-policy/#referrer-policy-header-dfn>Referrer-Policy</a></code>` HTTP
    header.</ol>
  </div>

  <h4 id=nonce-attributes><span class=secno>2.6.6</span> Nonce attributes<a href=#nonce-attributes class=self-link></a></h4>

  <div class="mdn-anno wrapped before"><button onclick=toggleStatus(this) class=mdn-anno-btn><b title="Support in all current engines." class=all-engines-flag>✔</b><span>MDN</span></button><div class=feature><p><a href=https://developer.mozilla.org/en-US/docs/Web/API/HTMLOrForeignElement/nonce title="The nonce property of the HTMLOrForeignElement interface returns the cryptographic number used once that is used by Content Security Policy to determine whether a given fetch will be allowed to proceed.">HTMLOrForeignElement/nonce</a><p class=all-engines-text>Support in all current engines.<div class=support><span class="firefox yes"><span>Firefox</span><span>75+</span></span><span class="safari yes"><span>Safari</span><span>10+</span></span><span class="chrome yes"><span>Chrome</span><span>61+</span></span><hr><span class="opera yes"><span>Opera</span><span>Yes</span></span><span class="edge_blink yes"><span>Edge</span><span>79+</span></span><hr><span class="edge no"><span>Edge (Legacy)</span><span>No</span></span><span class="ie no"><span>Internet Explorer</span><span>No</span></span><hr><span class="firefox_android no"><span>Firefox Android</span><span>No</span></span><span class="safari_ios yes"><span>Safari iOS</span><span>10+</span></span><span class="chrome_android yes"><span>Chrome Android</span><span>61+</span></span><span class="webview_android yes"><span>WebView Android</span><span>61+</span></span><span class="samsunginternet_android yes"><span>Samsung Internet</span><span>8.0+</span></span><span class="opera_android yes"><span>Opera Android</span><span>Yes</span></span></div></div></div><p>A <dfn data-dfn-for=htmlsvg-global id=attr-nonce data-dfn-type=element-attr><code>nonce</code></dfn> content
  attribute represents a cryptographic nonce ("number used once") which can be used by <cite>Content
  Security Policy</cite> to determine whether or not a given fetch will be allowed to proceed. The
  value is text. <a href=#refsCSP>[CSP]</a></p>

  <p>Elements that have a <code id=nonce-attributes:attr-nonce><a href=#attr-nonce>nonce</a></code> content attribute ensure that the
  crytographic nonce is only exposed to script (and not to side-channels like CSS attribute
  selectors) by taking the value from the content attribute, moving it into an internal slot
  named <dfn data-dfn-for=HTMLOrSVGElement id=cryptographicnonce data-dfn-type=attribute>[[CryptographicNonce]]</dfn>, exposing it to script
  via the <code id=nonce-attributes:htmlorsvgelement><a href=#htmlorsvgelement>HTMLOrSVGElement</a></code> interface mixin, and setting the content attribute to the
  empty string. Unless otherwise specified, the slot's value is the empty string.</p>

  <dl class=domintro><dt><var>element</var> . <code>nonce</code><dd>
    <p>Returns the value of the element's <a href=#cryptographicnonce id=nonce-attributes:cryptographicnonce>[[CryptographicNonce]]</a> internal slot.</p>
    <p>Can be set, to update that slot's value.</p>
   </dl>

  <p>The <dfn data-dfn-for=HTMLOrSVGElement id=dom-noncedelement-nonce data-dfn-type=attribute><code>nonce</code></dfn> IDL attribute must, on getting, return the
  value of this element's <a href=#cryptographicnonce id=nonce-attributes:cryptographicnonce-2>[[CryptographicNonce]]</a>; and on setting, set this element's
  <a href=#cryptographicnonce id=nonce-attributes:cryptographicnonce-3>[[CryptographicNonce]]</a> to the given value.</p>

  <p id=nonce-does-not-update-dom class=note>Note how the setter for the <code id=nonce-attributes:dom-noncedelement-nonce><a href=#dom-noncedelement-nonce>nonce</a></code> IDL attribute does not update the corresponding
  content attribute. This, as well as the below setting of the <code id=nonce-attributes:attr-nonce-2><a href=#attr-nonce>nonce</a></code> content attribute to the empty string when an element
  <a href=#becomes-browsing-context-connected id=nonce-attributes:becomes-browsing-context-connected>becomes browsing-context connected</a>, is meant to prevent exfiltration of the nonce
  value through mechanisms that can easily read content attributes, such as selectors. Learn more in
  <a href=https://github.com/whatwg/html/issues/2369>issue #2369</a>, where this behavior was
  introduced.</p>

  <p>The following <a href=https://dom.spec.whatwg.org/#concept-element-attributes-change-ext id=nonce-attributes:concept-element-attributes-change-ext data-x-internal=concept-element-attributes-change-ext>attribute change
  steps</a> are used for the <code id=nonce-attributes:attr-nonce-3><a href=#attr-nonce>nonce</a></code> content attribute:

  <ol><li><p>If <var>element</var> does not <a id=nonce-attributes:include href=https://heycam.github.io/webidl/#include data-x-internal=include>include</a> <code id=nonce-attributes:htmlorsvgelement-2><a href=#htmlorsvgelement>HTMLOrSVGElement</a></code>, then
   return.<li><p>If <var>localName</var> is not <code id=nonce-attributes:attr-nonce-4><a href=#attr-nonce>nonce</a></code> or
   <var>namespace</var> is not null, then return.<li><p>If <var>value</var> is null, then set <var>element</var>'s
   <a href=#cryptographicnonce id=nonce-attributes:cryptographicnonce-4>[[CryptographicNonce]]</a> to the empty string.<li><p>Otherwise, set <var>element</var>'s <a href=#cryptographicnonce id=nonce-attributes:cryptographicnonce-5>[[CryptographicNonce]]</a> to
   <var>value</var>.</ol>

  <p>Whenever an element <a href=https://heycam.github.io/webidl/#include id=nonce-attributes:include-2 data-x-internal=include>including</a> <code id=nonce-attributes:htmlorsvgelement-3><a href=#htmlorsvgelement>HTMLOrSVGElement</a></code>
  <a href=#becomes-browsing-context-connected id=nonce-attributes:becomes-browsing-context-connected-2>becomes browsing-context connected</a>, the user agent must execute the following steps
  on the <var>element</var>:</p>

  <ol><li><p>Let <var>CSP list</var> be <var>element</var>'s <a href=https://dom.spec.whatwg.org/#concept-shadow-including-root id=nonce-attributes:shadow-including-root data-x-internal=shadow-including-root>shadow-including root</a>'s <a href=#concept-document-csp-list id=nonce-attributes:concept-document-csp-list>CSP
   list</a>.<li>
    <p>If <var>CSP list</var> <a id=nonce-attributes:contains-a-header-delivered-content-security-policy href=https://w3c.github.io/webappsec-csp/#contains-a-header-delivered-content-security-policy data-x-internal=contains-a-header-delivered-content-security-policy>contains a header-delivered Content Security Policy</a>, and
    <var>element</var> has a <code id=nonce-attributes:attr-nonce-5><a href=#attr-nonce>nonce</a></code> content attribute
    <var>attr</var> whose value is not the empty string, then:</p>

    <ol><li><p>Let <var>nonce</var> be <var>element</var>'s
     <a href=#cryptographicnonce id=nonce-attributes:cryptographicnonce-6>[[CryptographicNonce]]</a>.<li><p><a href=https://dom.spec.whatwg.org/#concept-element-attributes-set-value id=nonce-attributes:concept-element-attributes-set-value data-x-internal=concept-element-attributes-set-value>Set an attribute value</a> for
     <var>element</var> using "<code id=nonce-attributes:attr-nonce-6><a href=#attr-nonce>nonce</a></code>" and the empty
     string.<li><p>Set <var>element</var>'s <a href=#cryptographicnonce id=nonce-attributes:cryptographicnonce-7>[[CryptographicNonce]]</a> to
     <var>nonce</var>.</ol>

    <p class=note>If <var>element</var>'s <a href=#cryptographicnonce id=nonce-attributes:cryptographicnonce-8>[[CryptographicNonce]]</a> were not restored it
    would be the empty string at this point.</p>
   </ol>

  <p class=note>As each <code id=nonce-attributes:document><a href=#document>Document</a></code>'s <a href=#concept-document-csp-list id=nonce-attributes:concept-document-csp-list-2>CSP
  list</a> is append-only, user agents can optimize away the <a id=nonce-attributes:contains-a-header-delivered-content-security-policy-2 href=https://w3c.github.io/webappsec-csp/#contains-a-header-delivered-content-security-policy data-x-internal=contains-a-header-delivered-content-security-policy>contains a header-delivered
  Content Security Policy</a> check by, for example, holding a flag on the <code id=nonce-attributes:document-2><a href=#document>Document</a></code>,
  set during <a href=#initialise-the-document-object id=nonce-attributes:initialise-the-document-object><code>Document</code>
  creation and initialization</a>.</p>

  <p>The <a href=https://dom.spec.whatwg.org/#concept-node-clone-ext id=nonce-attributes:concept-node-clone-ext data-x-internal=concept-node-clone-ext>cloning steps</a> for elements that
  <a id=nonce-attributes:include-3 href=https://heycam.github.io/webidl/#include data-x-internal=include>include</a> <code id=nonce-attributes:htmlorsvgelement-4><a href=#htmlorsvgelement>HTMLOrSVGElement</a></code> must set the
  <a href=#cryptographicnonce id=nonce-attributes:cryptographicnonce-9>[[CryptographicNonce]]</a> slot on the copy to the value of the slot on the element being
  cloned.</p>

  <h4 id=lazy-loading-attributes><span class=secno>2.6.7</span> Lazy loading attributes<a href=#lazy-loading-attributes class=self-link></a></h4><div class="mdn-anno wrapped"><button onclick=toggleStatus(this) class=mdn-anno-btn><span>MDN</span></button><div class=feature><p><a href=https://developer.mozilla.org/en-US/docs/Web/Performance/Lazy_loading title="Lazy loading is a strategy to identify resources as non-blocking (non-critical) and load these only when needed. It's a way to shorten the length of the critical rendering path, which translates into reduced page load times.">Lazy_loading</a><div class=support><span class="firefox yes"><span>Firefox</span><span>75+</span></span><span class="safari no"><span>Safari</span><span>No</span></span><span class="chrome yes"><span>Chrome</span><span>76+</span></span><hr><span class="opera yes"><span>Opera</span><span>63+</span></span><span class="edge_blink yes"><span>Edge</span><span>79+</span></span><hr><span class="edge no"><span>Edge (Legacy)</span><span>No</span></span><span class="ie no"><span>Internet Explorer</span><span>No</span></span><hr><span class="firefox_android no"><span>Firefox Android</span><span>No</span></span><span class="safari_ios no"><span>Safari iOS</span><span>No</span></span><span class="chrome_android yes"><span>Chrome Android</span><span>76+</span></span><span class="webview_android no"><span>WebView Android</span><span>No</span></span><span class="samsunginternet_android no"><span>Samsung Internet</span><span>No</span></span><span class="opera_android yes"><span>Opera Android</span><span>54+</span></span><hr><span class=caniuse><span><a href="https://caniuse.com/#feat=loading-lazy-attr" title="Lazy loading via attribute for images &amp; iframes">caniuse.com table</a></span></span></div></div></div>

  <p>A <dfn id=lazy-loading-attribute>lazy loading attribute</dfn> is an <a href=#enumerated-attribute id=lazy-loading-attributes:enumerated-attribute>enumerated attribute</a>. The following
  table lists the keywords and states for the attribute — the keywords in the left column map
  to the states in the cell in the second column on the same row as the keyword.</p>

  <p>The attribute directs the user agent to fetch a resource immediately or to defer fetching until
  some conditions associated with the element are met, according to the attribute's current
  state.</p>

  <table><thead><tr><th>Keyword
     <th>State
     <th>Description
   <tbody><tr><td><dfn id=attr-loading-lazy><code>lazy</code></dfn>
     <td><dfn id=attr-loading-lazy-state>Lazy</dfn>
     <td>Used to defer fetching a resource until some conditions are met.
    <tr><td><dfn id=attr-loading-eager><code>eager</code></dfn>
     <td><dfn id=attr-loading-eager-state>Eager</dfn>
     <td>Used to fetch a resource immediately; the default state.
  </table>

  <p>The attribute's <i id=lazy-loading-attributes:missing-value-default><a href=#missing-value-default>missing value default</a></i> and <i id=lazy-loading-attributes:invalid-value-default><a href=#invalid-value-default>invalid value default</a></i> are both the <a href=#attr-loading-eager-state id=lazy-loading-attributes:attr-loading-eager-state>Eager</a> state.</p>

  <hr>

  <p>The <dfn id=will-lazy-load-element-steps>will lazy load element steps</dfn>, given an element <var>element</var>,
  are as follows:</p>

  <ol><li>
    <p>If <a href=#concept-n-noscript id=lazy-loading-attributes:concept-n-noscript>scripting is disabled</a> for <var>element</var>,
    then return false.</p>

    <p class=note>This is an anti-tracking measure, because if a user agent supported lazy loading
    when scripting is disabled, it would still be possible for a site to track a user's approximate
    scroll position throughout a session, by strategically placing images in a page's markup such
    that a server can track how many images are requested and when.</p>
   <li><p>If <var>element</var>'s <a href=#lazy-loading-attribute id=lazy-loading-attributes:lazy-loading-attribute>lazy loading attribute</a> is in the <a href=#attr-loading-lazy-state id=lazy-loading-attributes:attr-loading-lazy-state>Lazy</a> state, then return true.<li><p>Return false.</ol>

  <p>Each <code id=lazy-loading-attributes:the-img-element><a href=#the-img-element>img</a></code> and <code id=lazy-loading-attributes:the-iframe-element><a href=#the-iframe-element>iframe</a></code> element has associated <dfn id=lazy-load-resumption-steps>lazy load resumption
  steps</dfn>, initially null.</p>

  <p class=note>For <code id=lazy-loading-attributes:the-img-element-2><a href=#the-img-element>img</a></code> and <code id=lazy-loading-attributes:the-iframe-element-2><a href=#the-iframe-element>iframe</a></code> elements that <a href=#will-lazy-load-element-steps id=lazy-loading-attributes:will-lazy-load-element-steps>will lazy load</a>, these steps are run from the <a href=#lazy-load-intersection-observer id=lazy-loading-attributes:lazy-load-intersection-observer>lazy load
  intersection observer</a>'s callback or when their <a href=#lazy-loading-attribute id=lazy-loading-attributes:lazy-loading-attribute-2>lazy loading attribute</a> is set
  to the <a href=#attr-loading-eager-state id=lazy-loading-attributes:attr-loading-eager-state-2>Eager</a> state. This causes the element to
  continue loading.</p>

  <p>Each <code id=lazy-loading-attributes:document><a href=#document>Document</a></code> has a <dfn id=lazy-load-intersection-observer>lazy load intersection observer</dfn>, initially set to
  null but can be set to an <code id=lazy-loading-attributes:intersectionobserver><a data-x-internal=intersectionobserver href=https://w3c.github.io/IntersectionObserver/#intersectionobserver>IntersectionObserver</a></code> instance.</p>

  <p>To <dfn id=start-intersection-observing-a-lazy-loading-element>start intersection-observing a lazy loading element</dfn> <var>element</var>, run these
  steps:</p>

  <ol><li><p>Let <var>doc</var> be <var>element</var>'s <a id=lazy-loading-attributes:node-document href=https://dom.spec.whatwg.org/#concept-node-document data-x-internal=node-document>node document</a>.<li>
    <p>If <var>doc</var>'s <a href=#lazy-load-intersection-observer id=lazy-loading-attributes:lazy-load-intersection-observer-2>lazy load intersection observer</a> is null, set it to a new
    <code id=lazy-loading-attributes:intersectionobserver-2><a data-x-internal=intersectionobserver href=https://w3c.github.io/IntersectionObserver/#intersectionobserver>IntersectionObserver</a></code> instance, initialized as follows:</p>

    <p class=XXX>The intention is to use the original value of the
    <code id=lazy-loading-attributes:intersectionobserver-3><a data-x-internal=intersectionobserver href=https://w3c.github.io/IntersectionObserver/#intersectionobserver>IntersectionObserver</a></code> constructor. However, we're forced to use the
    JavaScript-exposed constructor in this specification, until <cite>Intersection Observer</cite>
    exposes low-level hooks for use in specifications. See bug <a href=https://github.com/w3c/IntersectionObserver/issues/427>w3c/IntersectionObserver#427</a>
    which tracks this. <a href=#refsINTERSECTIONOBSERVER>[INTERSECTIONOBSERVER]</a></p>

    <ul><li>
      <p>The <var>callback</var> is these steps, with arguments <var>entries</var> and
      <var>observer</var>:</p>

      <ol><li>
        <p>For each <var>entry</var> in <var>entries</var> <span class=XXX>using a method of
        iteration which does not trigger developer-modifiable array accessors or
        iteration hooks</span>:</p>

        <ol><li><p>Let <var>resumptionSteps</var> be null.<li><p>If <var>entry</var>.<code id=lazy-loading-attributes:dom-intersectionobserverentry-isintersecting><a data-x-internal=dom-intersectionobserverentry-isintersecting href=https://w3c.github.io/IntersectionObserver/#dom-intersectionobserverentry-isintersecting>isIntersecting</a></code> is true, then
         set <var>resumptionSteps</var> to <var>entry</var>.<code id=lazy-loading-attributes:dom-intersectionobserverentry-target><a data-x-internal=dom-intersectionobserverentry-target href=https://w3c.github.io/IntersectionObserver/#dom-intersectionobserverentry-target>target</a></code>'s
         <a href=#lazy-load-resumption-steps id=lazy-loading-attributes:lazy-load-resumption-steps>lazy load resumption steps</a>.<li><p>If <var>resumptionSteps</var> is null, then return.<li><p><a href=#stop-intersection-observing-a-lazy-loading-element id=lazy-loading-attributes:stop-intersection-observing-a-lazy-loading-element>Stop intersection-observing a lazy loading element</a> for
         <var>entry</var>.<code id=lazy-loading-attributes:dom-intersectionobserverentry-target-2><a data-x-internal=dom-intersectionobserverentry-target href=https://w3c.github.io/IntersectionObserver/#dom-intersectionobserverentry-target>target</a></code>.<li><p>Set <var>entry</var>.<code id=lazy-loading-attributes:dom-intersectionobserverentry-target-3><a data-x-internal=dom-intersectionobserverentry-target href=https://w3c.github.io/IntersectionObserver/#dom-intersectionobserverentry-target>target</a></code>'s <a href=#lazy-load-resumption-steps id=lazy-loading-attributes:lazy-load-resumption-steps-2>lazy load resumption
         steps</a> to null.<li><p>Invoke <var>resumptionSteps</var>.</ol>

        <p class=XXX>The intention is to use the original value of the
        <code id=lazy-loading-attributes:dom-intersectionobserverentry-isintersecting-2><a data-x-internal=dom-intersectionobserverentry-isintersecting href=https://w3c.github.io/IntersectionObserver/#dom-intersectionobserverentry-isintersecting>isIntersecting</a></code> and
        <code id=lazy-loading-attributes:dom-intersectionobserverentry-target-4><a data-x-internal=dom-intersectionobserverentry-target href=https://w3c.github.io/IntersectionObserver/#dom-intersectionobserverentry-target>target</a></code> getters. See <a href=https://github.com/w3c/IntersectionObserver/issues/427>w3c/IntersectionObserver#427</a>.
        <a href=#refsINTERSECTIONOBSERVER>[INTERSECTIONOBSERVER]</a></p>
       </ol>
     <li>
      <p>The <var>options</var> is an <code id=lazy-loading-attributes:intersectionobserverinit><a data-x-internal=intersectionobserverinit href=https://w3c.github.io/IntersectionObserver/#dictdef-intersectionobserverinit>IntersectionObserverInit</a></code> dictionary with the
      following dictionary members: «[ "<code>rootMargin</code>" → <a href=#lazy-load-root-margin id=lazy-loading-attributes:lazy-load-root-margin>lazy load root
      margin</a> ]»</p>

      <p class=note>This allows for fetching the image during scrolling, when it does not yet —
      but is about to — intersect the viewport.</p>

      <p class=XXX>The <a href=#lazy-load-root-margin id=lazy-loading-attributes:lazy-load-root-margin-2>lazy load root margin</a> suggestions imply dynamic changes to the
      value, but the <code id=lazy-loading-attributes:intersectionobserver-4><a data-x-internal=intersectionobserver href=https://w3c.github.io/IntersectionObserver/#intersectionobserver>IntersectionObserver</a></code> API does not support changing the root
      margin. See issue <a href=https://github.com/w3c/IntersectionObserver/issues/428>w3c/IntersectionObserver#428</a>.</p>
     </ul>
   <li>
    <p>Call <var>doc</var>'s <a href=#lazy-load-intersection-observer id=lazy-loading-attributes:lazy-load-intersection-observer-3>lazy load intersection observer</a>'s <code id=lazy-loading-attributes:dom-intersectionobserver-observe><a data-x-internal=dom-intersectionobserver-observe href=https://w3c.github.io/IntersectionObserver/#dom-intersectionobserver-observe>observe</a></code> method with <var>element</var> as the
    argument.</p>

    <p class=XXX>The intention is to use the original value of the <code id=lazy-loading-attributes:dom-intersectionobserver-observe-2><a data-x-internal=dom-intersectionobserver-observe href=https://w3c.github.io/IntersectionObserver/#dom-intersectionobserver-observe>observe</a></code> method. See <a href=https://github.com/w3c/IntersectionObserver/issues/427>w3c/IntersectionObserver#427</a>.
    <a href=#refsINTERSECTIONOBSERVER>[INTERSECTIONOBSERVER]</a></p>
   </ol>

  <p>To <dfn id=stop-intersection-observing-a-lazy-loading-element>stop intersection-observing a lazy loading element</dfn> <var>element</var>, run these
  steps:</p>

  <ol><li><p>Let <var>doc</var> be <var>element</var>'s <a id=lazy-loading-attributes:node-document-2 href=https://dom.spec.whatwg.org/#concept-node-document data-x-internal=node-document>node document</a>.<li><p>Assert: <var>doc</var>'s <a href=#lazy-load-intersection-observer id=lazy-loading-attributes:lazy-load-intersection-observer-4>lazy load intersection observer</a> is not
   null.<li>
    <p>Call <var>doc</var>'s <a href=#lazy-load-intersection-observer id=lazy-loading-attributes:lazy-load-intersection-observer-5>lazy load intersection observer</a> <code id=lazy-loading-attributes:dom-intersectionobserver-unobserve><a data-x-internal=dom-intersectionobserver-unobserve href=https://w3c.github.io/IntersectionObserver/#dom-intersectionobserver-unobserve>unobserve</a></code> method with <var>element</var> as
    the argument.</p>

    <p class=XXX>The intention is to use the original value of the <code id=lazy-loading-attributes:dom-intersectionobserver-unobserve-2><a data-x-internal=dom-intersectionobserver-unobserve href=https://w3c.github.io/IntersectionObserver/#dom-intersectionobserver-unobserve>unobserve</a></code> method. See <a href=https://github.com/w3c/IntersectionObserver/issues/427>w3c/IntersectionObserver#427</a>.
    <a href=#refsINTERSECTIONOBSERVER>[INTERSECTIONOBSERVER]</a></p>
   </ol>

  <p>
  <a id=lazy-loading-attributes:tracking-vector title="There is a tracking vector here." href=https://infra.spec.whatwg.org/#tracking-vector class=tracking-vector data-x-internal=tracking-vector><img width=46 alt="(This is a tracking vector.)" crossorigin="" src=https://resources.whatwg.org/tracking-vector.svg height=64></a>
  The <dfn id=lazy-load-root-margin>lazy load root margin</dfn> is an <a id=lazy-loading-attributes:implementation-defined href=https://infra.spec.whatwg.org/#implementation-defined data-x-internal=implementation-defined>implementation-defined</a> value, but with
  the following suggestions to consider:</p>

  <ul><li><p>Set a minimum value that most often results in the resources being loaded before they
   intersect the viewport under normal usage patterns for the given device.<li><p>The typical scrolling speed: increase the value for devices with faster typical scrolling
   speeds.<li><p>The current scrolling speed or momentum: the UA can attempt to predict where the scrolling
   will likely stop, and adjust the value accordingly.<li><p>The network quality: increase the value for slow or high-latency connections.<li><p>User preferences can influence the value.</ul>

  <p class=note>It is important <a href=https://infra.spec.whatwg.org/#tracking-vector id=lazy-loading-attributes:tracking-vector-2 class=no-backref data-x-internal=tracking-vector>for
  privacy</a> that the <a href=#lazy-load-root-margin id=lazy-loading-attributes:lazy-load-root-margin-3>lazy load root margin</a> not leak additional information. For
  example, the typical scrolling speed on the current device could be imprecise so as to not
  introduce a new fingerprinting vector.</p>

  <h3 id=common-dom-interfaces><span class=secno>2.7</span> Common DOM interfaces<a href=#common-dom-interfaces class=self-link></a></h3>

  <h4 id=reflecting-content-attributes-in-idl-attributes><span class=secno>2.7.1</span> Reflecting content attributes in IDL attributes<a href=#reflecting-content-attributes-in-idl-attributes class=self-link></a></h4>

  <p>Some IDL attributes are defined to <dfn id=reflect>reflect</dfn> a particular content attribute. This
  means that on getting, the IDL attribute returns the current value of the content attribute, and
  on setting, the IDL attribute changes the value of the content attribute to the given value.</p>

  

  <p>In general, on getting, if the content attribute is not present, the IDL attribute must act as
  if the content attribute's value is the empty string; and on setting, if the content attribute is
  not present, it must first be added.</p>

  <p>If a reflecting IDL attribute is a <code id=reflecting-content-attributes-in-idl-attributes:idl-usvstring><a data-x-internal=idl-usvstring href=https://heycam.github.io/webidl/#idl-USVString>USVString</a></code> attribute
  whose content attribute is defined to contain a <a id=reflecting-content-attributes-in-idl-attributes:url href=https://url.spec.whatwg.org/#concept-url data-x-internal=url>URL</a>, then on getting, if the
  content attribute is absent, the IDL attribute must return the empty string. Otherwise, the IDL
  attribute must <a href=#parse-a-url id=reflecting-content-attributes-in-idl-attributes:parse-a-url>parse</a> the value of the content attribute relative
  to the element's <a id=reflecting-content-attributes-in-idl-attributes:node-document href=https://dom.spec.whatwg.org/#concept-node-document data-x-internal=node-document>node document</a> and if that is successful, return the <a href=#resulting-url-string id=reflecting-content-attributes-in-idl-attributes:resulting-url-string>resulting
  URL string</a>. If parsing fails, then the value of the content attribute must be returned
  instead, <a href=https://heycam.github.io/webidl/#es-type-mapping id=reflecting-content-attributes-in-idl-attributes:concept-idl-convert data-x-internal=concept-idl-convert>converted</a> to a <code id=reflecting-content-attributes-in-idl-attributes:idl-usvstring-2><a data-x-internal=idl-usvstring href=https://heycam.github.io/webidl/#idl-USVString>USVString</a></code>. On setting, the content attribute must be set to the
  specified new value.</p>

  <p>If a reflecting IDL attribute is a <code id=reflecting-content-attributes-in-idl-attributes:idl-domstring><a data-x-internal=idl-domstring href=https://heycam.github.io/webidl/#idl-DOMString>DOMString</a></code> attribute
  whose content attribute is an <a href=#enumerated-attribute id=reflecting-content-attributes-in-idl-attributes:enumerated-attribute>enumerated attribute</a>, and the IDL attribute is
  <dfn id=limited-to-only-known-values>limited to only known values</dfn>, then, on getting, the IDL attribute must return the
  keyword value associated with the state the attribute is in, if any, or the empty string if the
  attribute is in a state that has no associated keyword value or if the attribute is not in a
  defined state (e.g. the attribute is missing and there is no <i id=reflecting-content-attributes-in-idl-attributes:missing-value-default><a href=#missing-value-default>missing value default</a></i>). If there are multiple keyword values for the state, then
  return the conforming one. If there are multiple conforming keyword values, then one will be
  designated the <dfn id=canonical-keyword>canonical keyword</dfn>; choose that one. On setting, the content attribute
  must be set to the specified new value.</p>

  <p>If a reflecting IDL attribute is a nullable <code id=reflecting-content-attributes-in-idl-attributes:idl-domstring-2><a data-x-internal=idl-domstring href=https://heycam.github.io/webidl/#idl-DOMString>DOMString</a></code>
  attribute whose content attribute is an <a href=#enumerated-attribute id=reflecting-content-attributes-in-idl-attributes:enumerated-attribute-2>enumerated attribute</a>, then, on getting, if
  the corresponding content attribute is in its <i>missing value default</i> state then the IDL
  attribute must return null, otherwise, the IDL attribute must return the keyword value associated
  with the state the attribute is in. If there are multiple keyword values for the state, then
  return the conforming one. If there are multiple conforming keyword values, then one will be
  designated the <a href=#canonical-keyword id=reflecting-content-attributes-in-idl-attributes:canonical-keyword>canonical keyword</a>; choose that one. On setting, if the new value is
  null, the content attribute must be removed, and otherwise, the content attribute must be set to
  the specified new value.</p>

  <p>If a reflecting IDL attribute is a <code id=reflecting-content-attributes-in-idl-attributes:idl-domstring-3><a data-x-internal=idl-domstring href=https://heycam.github.io/webidl/#idl-DOMString>DOMString</a></code> or <code id=reflecting-content-attributes-in-idl-attributes:idl-usvstring-3><a data-x-internal=idl-usvstring href=https://heycam.github.io/webidl/#idl-USVString>USVString</a></code> attribute but doesn't fall into any of the above
  categories, then the getting and setting must be done in a transparent, case-preserving manner.</p>

  <p>If a reflecting IDL attribute is a <code id=reflecting-content-attributes-in-idl-attributes:idl-boolean><a data-x-internal=idl-boolean href=https://heycam.github.io/webidl/#idl-boolean>boolean</a></code> attribute, then on
  getting the IDL attribute must return true if the content attribute is set, and false if it is
  absent. On setting, the content attribute must be removed if the IDL attribute is set to false,
  and must be set to the empty string if the IDL attribute is set to true. (This corresponds to the
  rules for <a href=#boolean-attribute id=reflecting-content-attributes-in-idl-attributes:boolean-attribute>boolean content attributes</a>.)</p>

  <p>If a reflecting IDL attribute has a signed integer type (<code id=reflecting-content-attributes-in-idl-attributes:idl-long><a data-x-internal=idl-long href=https://heycam.github.io/webidl/#idl-long>long</a></code>)
  then, on getting, the content attribute must be parsed according to the <a href=#rules-for-parsing-integers id=reflecting-content-attributes-in-idl-attributes:rules-for-parsing-integers>rules for parsing signed integers</a>, and if that is successful, and the
  value is in the range of the IDL attribute's type, the resulting value must be returned. If, on
  the other hand, it fails or returns an out of range value, or if the attribute is absent, then the
  default value must be returned instead, or 0 if there is no default value. On setting, the given
  value must be converted to the shortest possible string representing the number as a <a href=#valid-integer id=reflecting-content-attributes-in-idl-attributes:valid-integer>valid
  integer</a> and then that string must be used as the new content attribute value.</p>

  <p>If a reflecting IDL attribute has a signed integer type (<code id=reflecting-content-attributes-in-idl-attributes:idl-long-2><a data-x-internal=idl-long href=https://heycam.github.io/webidl/#idl-long>long</a></code>)
  that is <dfn id=limited-to-only-non-negative-numbers>limited to only non-negative numbers</dfn> then, on getting, the content attribute
  must be parsed according to the <a href=#rules-for-parsing-non-negative-integers id=reflecting-content-attributes-in-idl-attributes:rules-for-parsing-non-negative-integers>rules for parsing non-negative integers</a>, and if that
  is successful, and the value is in the range of the IDL attribute's type, the resulting value must
  be returned. If, on the other hand, it fails or returns an out of range value, or if the attribute
  is absent, the default value must be returned instead, or −1 if there is no default value.
  On setting, if the value is negative, the user agent must throw an
  <a id=reflecting-content-attributes-in-idl-attributes:indexsizeerror href=https://heycam.github.io/webidl/#indexsizeerror data-x-internal=indexsizeerror>"<code>IndexSizeError</code>"</a> <code id=reflecting-content-attributes-in-idl-attributes:domexception><a data-x-internal=domexception href=https://heycam.github.io/webidl/#dfn-DOMException>DOMException</a></code>. Otherwise, the given value
  must be converted to the shortest possible string representing the number as a <a href=#valid-non-negative-integer id=reflecting-content-attributes-in-idl-attributes:valid-non-negative-integer>valid
  non-negative integer</a> and then that string must be used as the new content attribute
  value.</p>

  <p>If a reflecting IDL attribute has an <em>unsigned</em> integer type (<code id=reflecting-content-attributes-in-idl-attributes:idl-unsigned-long><a data-x-internal=idl-unsigned-long href=https://heycam.github.io/webidl/#idl-unsigned-long>unsigned long</a></code>) then, on getting, the content attribute must be
  parsed according to the <a href=#rules-for-parsing-non-negative-integers id=reflecting-content-attributes-in-idl-attributes:rules-for-parsing-non-negative-integers-2>rules for parsing non-negative integers</a>, and if that is
  successful, and the value is in the range 0 to 2147483647 inclusive, the resulting value must be
  returned. If, on the other hand, it fails or returns an out of range value, or if the attribute is
  absent, the default value must be returned instead, or 0 if there is no default value. On setting,
  first, if the new value is in the range 0 to 2147483647, then let <var>n</var> be the new value,
  otherwise let <var>n</var> be the default value, or 0 if there is no default value; then,
  <var>n</var> must be converted to the shortest possible string representing the number as a
  <a href=#valid-non-negative-integer id=reflecting-content-attributes-in-idl-attributes:valid-non-negative-integer-2>valid non-negative integer</a> and that string must be used as the new content attribute
  value.</p>
  

  <p>If a reflecting IDL attribute has an unsigned integer type (<code id=reflecting-content-attributes-in-idl-attributes:idl-unsigned-long-2><a data-x-internal=idl-unsigned-long href=https://heycam.github.io/webidl/#idl-unsigned-long>unsigned long</a></code>) that is <dfn id=limited-to-only-non-negative-numbers-greater-than-zero>limited to only non-negative numbers
  greater than zero</dfn>, then the behavior is similar to the previous case, but zero is not
  allowed. On getting, the content attribute must first be parsed according to the <a href=#rules-for-parsing-non-negative-integers id=reflecting-content-attributes-in-idl-attributes:rules-for-parsing-non-negative-integers-3>rules for
  parsing non-negative integers</a>, and if that is successful, and the value is in the range 1
  to 2147483647 inclusive, the resulting value must be returned. If, on the other hand, it fails or
  returns an out of range value, or if the attribute is absent, the default value must be returned
  instead, or 1 if there is no default value. On setting, if the value is zero, the user agent must
  throw an <a id=reflecting-content-attributes-in-idl-attributes:indexsizeerror-2 href=https://heycam.github.io/webidl/#indexsizeerror data-x-internal=indexsizeerror>"<code>IndexSizeError</code>"</a> <code id=reflecting-content-attributes-in-idl-attributes:domexception-2><a data-x-internal=domexception href=https://heycam.github.io/webidl/#dfn-DOMException>DOMException</a></code>. Otherwise, first,
  if the new value is in the range 1 to 2147483647, then let <var>n</var> be the new value,
  otherwise let <var>n</var> be the default value, or 1 if there is no default value; then,
  <var>n</var> must be converted to the shortest possible string representing the number as a
  <a href=#valid-non-negative-integer id=reflecting-content-attributes-in-idl-attributes:valid-non-negative-integer-3>valid non-negative integer</a> and that string must be used as the new content attribute
  value.</p> 

  <p>If a reflecting IDL attribute has an unsigned integer type (<code id=reflecting-content-attributes-in-idl-attributes:idl-unsigned-long-3><a data-x-internal=idl-unsigned-long href=https://heycam.github.io/webidl/#idl-unsigned-long>unsigned long</a></code>) that is <dfn id=limited-to-only-non-negative-numbers-greater-than-zero-with-fallback>limited to only non-negative numbers
  greater than zero with fallback</dfn>, then the behavior is similar to the previous case, but
  disallowed values are converted to the default value. On getting, the content attribute must first
  be parsed according to the <a href=#rules-for-parsing-non-negative-integers id=reflecting-content-attributes-in-idl-attributes:rules-for-parsing-non-negative-integers-4>rules for parsing non-negative integers</a>, and if that is
  successful, and the value is in the range 1 to 2147483647 inclusive, the resulting value must be
  returned. If, on the other hand, it fails or returns an out of range value, or if the attribute is
  absent, the default value must be returned instead. On setting, first, if the new value is in the
  range 1 to 2147483647, then let <var>n</var> be the new value, otherwise let <var>n</var> be the
  default value; then, <var>n</var> must be converted to the shortest possible string representing
  the number as a <a href=#valid-non-negative-integer id=reflecting-content-attributes-in-idl-attributes:valid-non-negative-integer-4>valid non-negative integer</a> and that string must be used as the new
  content attribute value.</p> 

  <p>If a reflecting IDL attribute has an unsigned integer type (<code id=reflecting-content-attributes-in-idl-attributes:idl-unsigned-long-4><a data-x-internal=idl-unsigned-long href=https://heycam.github.io/webidl/#idl-unsigned-long>unsigned long</a></code>) that is <dfn id=clamped-to-the-range>clamped to the range</dfn>
  [<var>min</var>, <var>max</var>], then on getting, the content attribute must first be parsed
  according to the <a href=#rules-for-parsing-non-negative-integers id=reflecting-content-attributes-in-idl-attributes:rules-for-parsing-non-negative-integers-5>rules for parsing non-negative integers</a>, and if that is successful,
  and the value is between <var>min</var> and <var>max</var> inclusive, the resulting value must be
  returned. If it fails, the default value must be returned. If it succeeds but the value is less
  than <var>min</var>, <var>min</var> must be returned. If it succeeds but the value is greater than
  <var>max</var>, <var>max</var> must be returned. On setting, it behaves the same as setting a
  regular reflected unsigned integer.</p>

  <p>If a reflecting IDL attribute has a floating-point number type (<code id=reflecting-content-attributes-in-idl-attributes:idl-double><a data-x-internal=idl-double href=https://heycam.github.io/webidl/#idl-double>double</a></code> or <code id=reflecting-content-attributes-in-idl-attributes:idl-unrestricted-double><a data-x-internal=idl-unrestricted-double href=https://heycam.github.io/webidl/#idl-unrestricted-double>unrestricted
  double</a></code>), then, on getting, the content attribute must be parsed according to the
  <a href=#rules-for-parsing-floating-point-number-values id=reflecting-content-attributes-in-idl-attributes:rules-for-parsing-floating-point-number-values>rules for parsing floating-point number values</a>, and if that is successful, the
  resulting value must be returned. If, on the other hand, it fails, or if the attribute is absent,
  the default value must be returned instead, or 0.0 if there is no default value. On setting, the
  given value must be converted to the <a href=#best-representation-of-the-number-as-a-floating-point-number id=reflecting-content-attributes-in-idl-attributes:best-representation-of-the-number-as-a-floating-point-number>best representation of the number as a floating-point
  number</a> and then that string must be used as the new content attribute value.</p>

  <p>If a reflecting IDL attribute has a floating-point number type (<code id=reflecting-content-attributes-in-idl-attributes:idl-double-2><a data-x-internal=idl-double href=https://heycam.github.io/webidl/#idl-double>double</a></code> or <code id=reflecting-content-attributes-in-idl-attributes:idl-unrestricted-double-2><a data-x-internal=idl-unrestricted-double href=https://heycam.github.io/webidl/#idl-unrestricted-double>unrestricted
  double</a></code>) that is <dfn id=limited-to-numbers-greater-than-zero>limited to numbers greater than zero</dfn>, then the behavior is
  similar to the previous case, but zero and negative values are not allowed. On getting, the
  content attribute must be parsed according to the <a href=#rules-for-parsing-floating-point-number-values id=reflecting-content-attributes-in-idl-attributes:rules-for-parsing-floating-point-number-values-2>rules for parsing floating-point number
  values</a>, and if that is successful and the value is greater than 0.0, the resulting value
  must be returned. If, on the other hand, it fails or returns an out of range value, or if the
  attribute is absent, the default value must be returned instead, or 0.0 if there is no default
  value. On setting, if the value is less than or equal to zero, then the value must be ignored.
  Otherwise, the given value must be converted to the <a href=#best-representation-of-the-number-as-a-floating-point-number id=reflecting-content-attributes-in-idl-attributes:best-representation-of-the-number-as-a-floating-point-number-2>best representation of the number as a
  floating-point number</a> and then that string must be used as the new content attribute
  value.</p>

  <p class=note>The values Infinity and Not-a-Number (NaN) values throw an exception on setting,
  as defined in <cite>Web IDL</cite>. <a href=#refsWEBIDL>[WEBIDL]</a></p>

  <p>If a reflecting IDL attribute has the type <code id=reflecting-content-attributes-in-idl-attributes:domtokenlist><a data-x-internal=domtokenlist href=https://dom.spec.whatwg.org/#interface-domtokenlist>DOMTokenList</a></code>, then on getting it must
  return a <code id=reflecting-content-attributes-in-idl-attributes:domtokenlist-2><a data-x-internal=domtokenlist href=https://dom.spec.whatwg.org/#interface-domtokenlist>DOMTokenList</a></code> object whose associated element is the element in question and
  whose associated attribute's local name is the name of the attribute in question.</p>

  


  <h4 id=collections><span class=secno>2.7.2</span> Collections<a href=#collections class=self-link></a></h4>

  <p>The <code id=collections:htmlformcontrolscollection><a href=#htmlformcontrolscollection>HTMLFormControlsCollection</a></code> and <code id=collections:htmloptionscollection><a href=#htmloptionscollection>HTMLOptionsCollection</a></code> interfaces
  are <a href=https://dom.spec.whatwg.org/#concept-collection id=collections:concept-collection data-x-internal=concept-collection>collections</a> derived from the
  <code id=collections:htmlcollection><a data-x-internal=htmlcollection href=https://dom.spec.whatwg.org/#interface-htmlcollection>HTMLCollection</a></code> interface. The <code id=collections:htmlallcollection><a href=#htmlallcollection>HTMLAllCollection</a></code> interface is a <a href=https://dom.spec.whatwg.org/#concept-collection id=collections:concept-collection-2 data-x-internal=concept-collection>collection</a>, but is not so derived.</p>


  <h5 id=the-htmlallcollection-interface><span class=secno>2.7.2.1</span> The <code id=the-htmlallcollection-interface:htmlallcollection><a href=#htmlallcollection>HTMLAllCollection</a></code> interface<a href=#the-htmlallcollection-interface class=self-link></a></h5>

  <p>The <code id=the-htmlallcollection-interface:htmlallcollection-2><a href=#htmlallcollection>HTMLAllCollection</a></code> interface is used for the legacy <code id=the-htmlallcollection-interface:dom-document-all><a href=#dom-document-all>document.all</a></code> attribute. It operates similarly to
  <code id=the-htmlallcollection-interface:htmlcollection><a data-x-internal=htmlcollection href=https://dom.spec.whatwg.org/#interface-htmlcollection>HTMLCollection</a></code>; the main differences are that it allows a staggering variety of
  different (ab)uses of its methods to all end up returning something, and that it can be called as
  a function as an alternative to property access.</p>

  <p class=note>All <code id=the-htmlallcollection-interface:htmlallcollection-3><a href=#htmlallcollection>HTMLAllCollection</a></code> objects are rooted at a <code id=the-htmlallcollection-interface:document><a href=#document>Document</a></code>
  and have a filter that matches all elements, so the elements <a id=the-htmlallcollection-interface:represented-by-the-collection href=https://dom.spec.whatwg.org/#represented-by-the-collection data-x-internal=represented-by-the-collection>represented by the
  collection</a> of an <code id=the-htmlallcollection-interface:htmlallcollection-4><a href=#htmlallcollection>HTMLAllCollection</a></code> object consist of all the descendant
  elements of the root <code id=the-htmlallcollection-interface:document-2><a href=#document>Document</a></code>.</p>

  <p>Objects that implement the <code id=the-htmlallcollection-interface:htmlallcollection-5><a href=#htmlallcollection>HTMLAllCollection</a></code> interface are <a href=https://heycam.github.io/webidl/#dfn-legacy-platform-object id=the-htmlallcollection-interface:legacy-platform-object data-x-internal=legacy-platform-object>legacy platform objects</a> with an additonal [[Call]] internal
  method described in the <a href=#HTMLAllCollection-call>section below</a>. They also have an
  <a id=the-htmlallcollection-interface:ishtmldda href=https://tc39.es/ecma262/#sec-IsHTMLDDA-internal-slot data-x-internal=ishtmldda>[[IsHTMLDDA]]</a> internal slot.</p>

  <div class=note>
   <p>Objects that implement the <code id=the-htmlallcollection-interface:htmlallcollection-6><a href=#htmlallcollection>HTMLAllCollection</a></code> interface have several unusual
   behaviors, due of the fact that they have an <a id=the-htmlallcollection-interface:ishtmldda-2 href=https://tc39.es/ecma262/#sec-IsHTMLDDA-internal-slot data-x-internal=ishtmldda>[[IsHTMLDDA]]</a> internal slot:</p>

   <ul><li><p>The <a href=https://tc39.es/ecma262/#sec-toboolean id=the-htmlallcollection-interface:js-toboolean data-x-internal=js-toboolean>ToBoolean</a> abstract operation in JavaScript returns
    false when given objects implementing the <code id=the-htmlallcollection-interface:htmlallcollection-7><a href=#htmlallcollection>HTMLAllCollection</a></code> interface.<li><p>The <a href=https://tc39.es/ecma262/#sec-abstract-equality-comparison id=the-htmlallcollection-interface:js-abstract-equality data-x-internal=js-abstract-equality>Abstract Equality Comparison</a> algorithm,
    when given objects implementing the <code id=the-htmlallcollection-interface:htmlallcollection-8><a href=#htmlallcollection>HTMLAllCollection</a></code> interface, returns true when
    compared to the <code>undefined</code> and <code>null</code> values.
    (Comparisons using the <a href=https://tc39.es/ecma262/#sec-strict-equality-comparison id=the-htmlallcollection-interface:js-strict-equality data-x-internal=js-strict-equality>Strict Equality Comparison</a>
    algorithm, and Abstract Equality comparisons to other values such as strings or objects, are
    unaffected.)<li><p>The <code id=the-htmlallcollection-interface:js-typeof><a data-x-internal=js-typeof href=https://tc39.es/ecma262/#sec-typeof-operator>typeof</a></code> operator in JavaScript returns the string
    <code>"undefined"</code> when applied to objects implementing the
    <code id=the-htmlallcollection-interface:htmlallcollection-9><a href=#htmlallcollection>HTMLAllCollection</a></code> interface.</ul>

   <p>These special behaviors are motivated by a desire for compatibility with two classes of legacy
   content: one that uses the presence of <code id=the-htmlallcollection-interface:dom-document-all-2><a href=#dom-document-all>document.all</a></code> as a
   way to detect legacy user agents, and one that only supports those legacy user agents and uses
   the <code id=the-htmlallcollection-interface:dom-document-all-3><a href=#dom-document-all>document.all</a></code> object without testing for its presence
   first. <a href=#refsJAVASCRIPT>[JAVASCRIPT]</a></p>
  </div>

  <pre><code class='idl'>[<c- g>Exposed</c->=<c- n>Window</c->,
 <c- g>LegacyUnenumerableNamedProperties</c->]
<c- b>interface</c-> <dfn id='htmlallcollection'><c- g>HTMLAllCollection</c-></dfn> {
  <c- b>readonly</c-> <c- b>attribute</c-> <c- b>unsigned</c-> <c- b>long</c-> <a href='#dom-htmlallcollection-length' id='the-htmlallcollection-interface:dom-htmlallcollection-length'><c- g>length</c-></a>;
  <c- b>getter</c-> <a id='the-htmlallcollection-interface:element' href='https://dom.spec.whatwg.org/#interface-element' data-x-internal='element'><c- n>Element</c-></a> (<c- b>unsigned</c-> <c- b>long</c-> <c- g>index</c->);
  <c- b>getter</c-> (<a id='the-htmlallcollection-interface:htmlcollection-2' href='https://dom.spec.whatwg.org/#interface-htmlcollection' data-x-internal='htmlcollection'><c- n>HTMLCollection</c-></a> <c- b>or</c-> <a id='the-htmlallcollection-interface:element-2' href='https://dom.spec.whatwg.org/#interface-element' data-x-internal='element'><c- n>Element</c-></a>)? <a href='#dom-htmlallcollection-nameditem' id='the-htmlallcollection-interface:dom-htmlallcollection-nameditem'><c- g>namedItem</c-></a>(<c- b>DOMString</c-> <c- g>name</c->);
  (<a id='the-htmlallcollection-interface:htmlcollection-3' href='https://dom.spec.whatwg.org/#interface-htmlcollection' data-x-internal='htmlcollection'><c- n>HTMLCollection</c-></a> <c- b>or</c-> <a id='the-htmlallcollection-interface:element-3' href='https://dom.spec.whatwg.org/#interface-element' data-x-internal='element'><c- n>Element</c-></a>)? <a href='#dom-htmlallcollection-item' id='the-htmlallcollection-interface:dom-htmlallcollection-item'><c- g>item</c-></a>(<c- b>optional</c-> <c- b>DOMString</c-> <c- g>nameOrIndex</c->);

  // Note: HTMLAllCollection objects have a <a href='#HTMLAllCollection-call'>custom [[Call]] internal method</a> and an <a id='the-htmlallcollection-interface:ishtmldda-3' href='https://tc39.es/ecma262/#sec-IsHTMLDDA-internal-slot' data-x-internal='ishtmldda'>[[IsHTMLDDA]]</a> internal slot.
};</code></pre>

  <dl class=domintro><dt><var>collection</var> . <code id=dom-htmlallcollection-length-dev><a href=#dom-htmlallcollection-length>length</a></code><dd>
    <p>Returns the number of elements in the collection.</p>
   <dt><var>element</var> = <var>collection</var> . <code id=dom-htmlallcollection-item-dev><a href=#dom-htmlallcollection-item>item</a></code>(<var>index</var>)<dt><var>element</var> = <var>collection</var>(<var>index</var>)<dt><var>element</var> = <var>collection</var>[<var>index</var>]<dd>
    <p>Returns the item with index <var>index</var> from the collection (determined by <a id=the-htmlallcollection-interface:tree-order href=https://dom.spec.whatwg.org/#concept-tree-order data-x-internal=tree-order>tree order</a>).</p>
   <dt><var>element</var> = <var>collection</var> . <code id=the-htmlallcollection-interface:dom-htmlallcollection-item-2><a href=#dom-htmlallcollection-item>item</a></code>(<var>name</var>)<dt><var>collection</var> = <var>collection</var> . <code id=the-htmlallcollection-interface:dom-htmlallcollection-item-3><a href=#dom-htmlallcollection-item>item</a></code>(<var>name</var>)<dt><var>element</var> = <var>collection</var> . <code id=dom-htmlallcollection-nameditem-dev><a href=#dom-htmlallcollection-nameditem>namedItem</a></code>(<var>name</var>)<dt><var>collection</var> = <var>collection</var> . <code id=the-htmlallcollection-interface:dom-htmlallcollection-nameditem-2><a href=#dom-htmlallcollection-nameditem>namedItem</a></code>(<var>name</var>)<dt><var>element</var> = <var>collection</var>(<var>name</var>)<dt><var>collection</var> = <var>collection</var>(<var>name</var>)<dt><var>element</var> = <var>collection</var>[<var>name</var>]<dt><var>collection</var> = <var>collection</var>[<var>name</var>]<dd>
    <p>Returns the item with <a href=https://dom.spec.whatwg.org/#concept-id id=the-htmlallcollection-interface:concept-id data-x-internal=concept-id>ID</a> or name <var>name</var> from the collection.</p>
    <p>If there are multiple matching items, then an <code id=the-htmlallcollection-interface:htmlcollection-4><a data-x-internal=htmlcollection href=https://dom.spec.whatwg.org/#interface-htmlcollection>HTMLCollection</a></code> object containing all those elements is returned.</p>
    
    <p>Only <code id=the-htmlallcollection-interface:the-button-element><a href=#the-button-element>button</a></code>, <code id=the-htmlallcollection-interface:the-form-element><a href=#the-form-element>form</a></code>, <code id=the-htmlallcollection-interface:the-iframe-element><a href=#the-iframe-element>iframe</a></code>,
    <code id=the-htmlallcollection-interface:the-input-element><a href=#the-input-element>input</a></code>, <code id=the-htmlallcollection-interface:the-map-element><a href=#the-map-element>map</a></code>, <code id=the-htmlallcollection-interface:the-meta-element><a href=#the-meta-element>meta</a></code>, <code id=the-htmlallcollection-interface:the-object-element><a href=#the-object-element>object</a></code>,
    <code id=the-htmlallcollection-interface:the-select-element><a href=#the-select-element>select</a></code>, and <code id=the-htmlallcollection-interface:the-textarea-element><a href=#the-textarea-element>textarea</a></code> elements can have a name for the purpose of this
    method; their name is given by the value of their <code>name</code> attribute.</p>
   </dl>

  

  <p>The object's <a id=the-htmlallcollection-interface:supported-property-indices href=https://heycam.github.io/webidl/#dfn-supported-property-indices data-x-internal=supported-property-indices>supported property indices</a> are as defined for
  <code id=the-htmlallcollection-interface:htmlcollection-5><a data-x-internal=htmlcollection href=https://dom.spec.whatwg.org/#interface-htmlcollection>HTMLCollection</a></code> objects.</p>

  <p>The <a id=the-htmlallcollection-interface:supported-property-names href=https://heycam.github.io/webidl/#dfn-supported-property-names data-x-internal=supported-property-names>supported property names</a> consist of the non-empty values of all the <code id=the-htmlallcollection-interface:the-id-attribute><a href=#the-id-attribute>id</a></code> attributes of all the elements <a id=the-htmlallcollection-interface:represented-by-the-collection-2 href=https://dom.spec.whatwg.org/#represented-by-the-collection data-x-internal=represented-by-the-collection>represented by the
  collection</a>, and the non-empty values of all the <code>name</code> attributes of
  all the <a href=#all-named-elements id=the-htmlallcollection-interface:all-named-elements>"all"-named elements</a> <a id=the-htmlallcollection-interface:represented-by-the-collection-3 href=https://dom.spec.whatwg.org/#represented-by-the-collection data-x-internal=represented-by-the-collection>represented by the collection</a>, in
  <a id=the-htmlallcollection-interface:tree-order-2 href=https://dom.spec.whatwg.org/#concept-tree-order data-x-internal=tree-order>tree order</a>, ignoring later duplicates, with the <code id=the-htmlallcollection-interface:the-id-attribute-2><a href=#the-id-attribute>id</a></code> of
  an element preceding its <code>name</code> if it contributes both, they differ from each
  other, and neither is the duplicate of an earlier entry.</p>

  <p>On getting, the <dfn id=dom-htmlallcollection-length><code>length</code></dfn>
  attribute must return the number of nodes <a id=the-htmlallcollection-interface:represented-by-the-collection-4 href=https://dom.spec.whatwg.org/#represented-by-the-collection data-x-internal=represented-by-the-collection>represented by the collection</a>.</p>

  <p>The indexed property getter must return the result of <a href=#concept-get-all-indexed id=the-htmlallcollection-interface:concept-get-all-indexed>getting the "all"-indexed element</a> from this
  <code id=the-htmlallcollection-interface:htmlallcollection-10><a href=#htmlallcollection>HTMLAllCollection</a></code> given the passed index.</p>

  <p>The <dfn id=dom-htmlallcollection-nameditem><code>namedItem(<var>name</var>)</code></dfn>
  method must return the result of <a href=#concept-get-all-named id=the-htmlallcollection-interface:concept-get-all-named>getting the "all"-named
  element(s)</a> from this <code id=the-htmlallcollection-interface:htmlallcollection-11><a href=#htmlallcollection>HTMLAllCollection</a></code> given <var>name</var>.</p>

  <p>The <dfn id=dom-htmlallcollection-item><code>item(<var>nameOrIndex</var>)</code></dfn>
  method must perform the following steps:</p>

  <ol><li><p>If <var>nameOrIndex</var> was not provided, return null.<li><p>Return the result of <a href=#concept-get-all-indexed-or-named id=the-htmlallcollection-interface:concept-get-all-indexed-or-named>getting the
   "all"-indexed or named element(s)</a> from this <code id=the-htmlallcollection-interface:htmlallcollection-12><a href=#htmlallcollection>HTMLAllCollection</a></code>, given
   <var>nameOrIndex</var>.</ol>

  <hr>

  
  <p>The following elements are <dfn id=all-named-elements>"all"-named elements</dfn>:
   <code id=the-htmlallcollection-interface:the-a-element><a href=#the-a-element>a</a></code>, 
   <code id=the-htmlallcollection-interface:the-button-element-2><a href=#the-button-element>button</a></code>, 
   <code id=the-htmlallcollection-interface:the-embed-element><a href=#the-embed-element>embed</a></code>,
   <code id=the-htmlallcollection-interface:the-form-element-2><a href=#the-form-element>form</a></code>,
   <code id=the-htmlallcollection-interface:frame><a href=#frame>frame</a></code>,
   <code id=the-htmlallcollection-interface:frameset><a href=#frameset>frameset</a></code>,
   <code id=the-htmlallcollection-interface:the-iframe-element-2><a href=#the-iframe-element>iframe</a></code>, 
   <code id=the-htmlallcollection-interface:the-img-element><a href=#the-img-element>img</a></code>,
   <code id=the-htmlallcollection-interface:the-input-element-2><a href=#the-input-element>input</a></code>,
   
   <code id=the-htmlallcollection-interface:the-map-element-2><a href=#the-map-element>map</a></code>, 
   <code id=the-htmlallcollection-interface:the-meta-element-2><a href=#the-meta-element>meta</a></code>, 
   <code id=the-htmlallcollection-interface:the-object-element-2><a href=#the-object-element>object</a></code>,
   <code id=the-htmlallcollection-interface:the-select-element-2><a href=#the-select-element>select</a></code>, and
   <code id=the-htmlallcollection-interface:the-textarea-element-2><a href=#the-textarea-element>textarea</a></code> 
  </p>

  <p>To <dfn id=concept-get-all-indexed>get the "all"-indexed element</dfn> from an
  <code id=the-htmlallcollection-interface:htmlallcollection-13><a href=#htmlallcollection>HTMLAllCollection</a></code> <var>collection</var> given an index <var>index</var>, return the
  <var>index</var><sup>th</sup> element in <var>collection</var>, or null if there is no such
  <var>index</var><sup>th</sup> element.</p>

  <p>To <dfn id=concept-get-all-named>get the "all"-named element(s)</dfn> from an
  <code id=the-htmlallcollection-interface:htmlallcollection-14><a href=#htmlallcollection>HTMLAllCollection</a></code> <var>collection</var> given a name <var>name</var>, perform the
  following steps:</p>

  <ol><li><p>If <var>name</var> is the empty string, return null.<li>
    <p>Let <var>subCollection</var> be an <code id=the-htmlallcollection-interface:htmlcollection-6><a data-x-internal=htmlcollection href=https://dom.spec.whatwg.org/#interface-htmlcollection>HTMLCollection</a></code> object rooted at the same
    <code id=the-htmlallcollection-interface:document-3><a href=#document>Document</a></code> as <var>collection</var>, whose filter matches only elements that are
    either:</p>

    <ul><li><p><a href=#all-named-elements id=the-htmlallcollection-interface:all-named-elements-2>"all"-named elements</a> with a <code>name</code> attribute equal to
     <var>name</var>, or,<li><p>elements with an <a href=https://dom.spec.whatwg.org/#concept-id id=the-htmlallcollection-interface:concept-id-2 data-x-internal=concept-id>ID</a> equal to <var>name</var>.</ul>
   <li><p>If there is exactly one element in <var>subCollection</var>, then return that
   element.<li><p>Otherwise, if <var>subCollection</var> is empty, return null.<li><p>Otherwise, return <var>subCollection</var>.</ol>

  <p>To <dfn id=concept-get-all-indexed-or-named>get the "all"-indexed or named
  element(s)</dfn> from an <code id=the-htmlallcollection-interface:htmlallcollection-15><a href=#htmlallcollection>HTMLAllCollection</a></code> <var>collection</var> given
  <var>nameOrIndex</var>:

  <ol><li><p>If <var>nameOrIndex</var>, <a href=https://heycam.github.io/webidl/#es-type-mapping id=the-htmlallcollection-interface:concept-idl-convert data-x-internal=concept-idl-convert>converted</a> to a
   JavaScript String value, is an <a id=the-htmlallcollection-interface:array-index-property-name href=https://heycam.github.io/webidl/#dfn-array-index-property-name data-x-internal=array-index-property-name>array index property name</a>, return the result of <a href=#concept-get-all-indexed id=the-htmlallcollection-interface:concept-get-all-indexed-2>getting the "all"-indexed element</a> from this
   <code id=the-htmlallcollection-interface:htmlallcollection-16><a href=#htmlallcollection>HTMLAllCollection</a></code> given the number represented by <var>nameOrIndex</var>.<li><p>Return the result of <a href=#concept-get-all-named id=the-htmlallcollection-interface:concept-get-all-named-2>getting the "all"-named
   element(s)</a> from this <code id=the-htmlallcollection-interface:htmlallcollection-17><a href=#htmlallcollection>HTMLAllCollection</a></code> given
   <var>nameOrIndex</var>.</ol>

  <h6 id=HTMLAllCollection-call><span class=secno>2.7.2.1.1</span> [[Call]] ( <var>thisArgument</var>,  <var>argumentsList</var> )<a href=#HTMLAllCollection-call class=self-link></a></h6>

  <ol><li><p>If <var>argumentsList</var>'s <a href=https://infra.spec.whatwg.org/#list-size id=HTMLAllCollection-call:list-size data-x-internal=list-size>size</a> is zero, or if
   <var>argumentsList</var>[0] is undefined, return null.<li><p>Let <var>nameOrIndex</var> be the result of <a href=https://heycam.github.io/webidl/#es-type-mapping id=HTMLAllCollection-call:concept-idl-convert data-x-internal=concept-idl-convert>converting</a> <var>argumentsList</var>[0] to a <code id=HTMLAllCollection-call:idl-domstring><a data-x-internal=idl-domstring href=https://heycam.github.io/webidl/#idl-DOMString>DOMString</a></code>.<li><p>Let <var>result</var> be the result of <a href=#concept-get-all-indexed-or-named id=HTMLAllCollection-call:concept-get-all-indexed-or-named>getting the "all"-indexed or named element(s)</a>
   from this <code id=HTMLAllCollection-call:htmlallcollection><a href=#htmlallcollection>HTMLAllCollection</a></code> given <var>nameOrIndex</var>.<li><p>Return the result of <a href=https://heycam.github.io/webidl/#es-type-mapping id=HTMLAllCollection-call:concept-idl-convert-2 data-x-internal=concept-idl-convert>converting</a>
   <var>result</var> to an ECMAScript value.</ol>

  <p class=note>The <var>thisArgument</var> is ignored, and thus code such as <code>Function.prototype.call.call(document.all, null, "x")</code> will still search for
  elements. (<code>document.all.call</code> does not exist, since <code>document.all</code> does not inherit from <code>Function.prototype</code>.)</p>

  


  <h5 id=the-htmlformcontrolscollection-interface><span class=secno>2.7.2.2</span> The <code id=the-htmlformcontrolscollection-interface:htmlformcontrolscollection><a href=#htmlformcontrolscollection>HTMLFormControlsCollection</a></code> interface<a href=#the-htmlformcontrolscollection-interface class=self-link></a></h5>

  <p>The <code id=the-htmlformcontrolscollection-interface:htmlformcontrolscollection-2><a href=#htmlformcontrolscollection>HTMLFormControlsCollection</a></code> interface is used for
  <a href=https://dom.spec.whatwg.org/#concept-collection id=the-htmlformcontrolscollection-interface:concept-collection data-x-internal=concept-collection>collections</a> of <a href=#category-listed id=the-htmlformcontrolscollection-interface:category-listed>listed
  elements</a> in <code id=the-htmlformcontrolscollection-interface:the-form-element><a href=#the-form-element>form</a></code> elements.</p>

  <div class="mdn-anno wrapped before"><button onclick=toggleStatus(this) class=mdn-anno-btn><b title="Support in all current engines." class=all-engines-flag>✔</b><span>MDN</span></button><div class=feature><p><a href=https://developer.mozilla.org/en-US/docs/Web/API/HTMLFormControlsCollection title="The HTMLFormControlsCollection interface represents a collection of HTML form control elements. ">HTMLFormControlsCollection</a><p class=all-engines-text>Support in all current engines.<div class=support><span class="firefox yes"><span>Firefox</span><span>27+</span></span><span class="safari yes"><span>Safari</span><span>Yes</span></span><span class="chrome yes"><span>Chrome</span><span>Yes</span></span><hr><span class="opera yes"><span>Opera</span><span>Yes</span></span><span class="edge_blink yes"><span>Edge</span><span>Yes</span></span><hr><span class="edge no"><span>Edge (Legacy)</span><span>No</span></span><span class="ie no"><span>Internet Explorer</span><span>No</span></span><hr><span class="firefox_android yes"><span>Firefox Android</span><span>27+</span></span><span class="safari_ios yes"><span>Safari iOS</span><span>Yes</span></span><span class="chrome_android yes"><span>Chrome Android</span><span>Yes</span></span><span class="webview_android yes"><span>WebView Android</span><span>Yes</span></span><span class="samsunginternet_android yes"><span>Samsung Internet</span><span>Yes</span></span><span class="opera_android yes"><span>Opera Android</span><span>Yes</span></span></div></div><div class=feature><p><a href=https://developer.mozilla.org/en-US/docs/Web/API/RadioNodeList title="The RadioNodeList interface represents a collection of radio elements in a <form> or a <fieldset> element.">RadioNodeList</a><p class=all-engines-text>Support in all current engines.<div class=support><span class="firefox yes"><span>Firefox</span><span>33+</span></span><span class="safari yes"><span>Safari</span><span>10+</span></span><span class="chrome yes"><span>Chrome</span><span>34+</span></span><hr><span class="opera yes"><span>Opera</span><span>Yes</span></span><span class="edge_blink yes"><span>Edge</span><span>79+</span></span><hr><span class="edge yes"><span>Edge (Legacy)</span><span>12+</span></span><span class="ie yes"><span>Internet Explorer</span><span>Yes</span></span><hr><span class="firefox_android yes"><span>Firefox Android</span><span>33+</span></span><span class="safari_ios yes"><span>Safari iOS</span><span>9+</span></span><span class="chrome_android yes"><span>Chrome Android</span><span>Yes</span></span><span class="webview_android yes"><span>WebView Android</span><span>Yes</span></span><span class="samsunginternet_android yes"><span>Samsung Internet</span><span>Yes</span></span><span class="opera_android yes"><span>Opera Android</span><span>Yes</span></span></div></div></div><pre><code class='idl'>[<c- g>Exposed</c->=<c- n>Window</c->]
<c- b>interface</c-> <dfn id='htmlformcontrolscollection'><c- g>HTMLFormControlsCollection</c-></dfn> : <a id='the-htmlformcontrolscollection-interface:htmlcollection' href='https://dom.spec.whatwg.org/#interface-htmlcollection' data-x-internal='htmlcollection'><c- n>HTMLCollection</c-></a> {
  // inherits <a href='https://dom.spec.whatwg.org/#dom-htmlcollection-length' id='the-htmlformcontrolscollection-interface:dom-htmlcollection-length' data-x-internal='dom-htmlcollection-length'>length</a> and <a href='https://dom.spec.whatwg.org/#dom-htmlcollection-item' id='the-htmlformcontrolscollection-interface:dom-htmlcollection-item' data-x-internal='dom-htmlcollection-item'>item</a>()
  <c- b>getter</c-> (<a href='#radionodelist' id='the-htmlformcontrolscollection-interface:radionodelist'><c- n>RadioNodeList</c-></a> <c- b>or</c-> <a id='the-htmlformcontrolscollection-interface:element' href='https://dom.spec.whatwg.org/#interface-element' data-x-internal='element'><c- n>Element</c-></a>)? <a href='#dom-htmlformcontrolscollection-nameditem' id='the-htmlformcontrolscollection-interface:dom-htmlformcontrolscollection-nameditem'><c- g>namedItem</c-></a>(<c- b>DOMString</c-> <c- g>name</c->); // shadows inherited <a href='https://dom.spec.whatwg.org/#dom-htmlcollection-nameditem' id='the-htmlformcontrolscollection-interface:dom-htmlcollection-nameditem' data-x-internal='dom-htmlcollection-nameditem'>namedItem()</a>
};

[<c- g>Exposed</c->=<c- n>Window</c->]
<c- b>interface</c-> <dfn id='radionodelist'><c- g>RadioNodeList</c-></dfn> : <a id='the-htmlformcontrolscollection-interface:nodelist' href='https://dom.spec.whatwg.org/#interface-nodelist' data-x-internal='nodelist'><c- n>NodeList</c-></a> {
  <c- b>attribute</c-> <c- b>DOMString</c-> <a href='#dom-radionodelist-value' id='the-htmlformcontrolscollection-interface:dom-radionodelist-value'><c- g>value</c-></a>;
};</code></pre>

  <dl class=domintro><dt><var>collection</var> . <code id=dom-htmlcollection-length-dev><a data-x-internal=dom-htmlcollection-length href=https://dom.spec.whatwg.org/#dom-htmlcollection-length>length</a></code><dd>
    <p>Returns the number of elements in the collection.</p>
   <dt><var>element</var> = <var>collection</var> . <code id=dom-htmlcollection-item-dev><a data-x-internal=dom-htmlcollection-item href=https://dom.spec.whatwg.org/#dom-htmlcollection-item>item</a></code>(<var>index</var>)<dt><var>element</var> = <var>collection</var>[<var>index</var>]<dd>
    <p>Returns the item with index <var>index</var> from the collection. The items are sorted in <a id=the-htmlformcontrolscollection-interface:tree-order href=https://dom.spec.whatwg.org/#concept-tree-order data-x-internal=tree-order>tree order</a>.</p>
   <dt><var>element</var> = <var>collection</var> . <code id=dom-htmlformcontrolscollection-nameditem-dev><a href=#dom-htmlformcontrolscollection-nameditem>namedItem</a></code>(<var>name</var>)<dt><var>radioNodeList</var> = <var>collection</var> . <code id=the-htmlformcontrolscollection-interface:dom-htmlformcontrolscollection-nameditem-2><a href=#dom-htmlformcontrolscollection-nameditem>namedItem</a></code>(<var>name</var>)<dt><var>element</var> = <var>collection</var>[<var>name</var>]<dt><var>radioNodeList</var> = <var>collection</var>[<var>name</var>]<dd>
    <p>Returns the item with <a href=https://dom.spec.whatwg.org/#concept-id id=the-htmlformcontrolscollection-interface:concept-id data-x-internal=concept-id>ID</a> or <code id=the-htmlformcontrolscollection-interface:attr-fe-name><a href=#attr-fe-name>name</a></code> <var>name</var> from the collection.</p>
    <p>If there are multiple matching items, then a <code id=the-htmlformcontrolscollection-interface:radionodelist-2><a href=#radionodelist>RadioNodeList</a></code> object containing all those elements is returned.</p>
   <dt><var>radioNodeList</var> . <var>value</var> [ = <var>value</var> ]<dd>
    <p>Returns the value of the first checked radio button represented by the object.</p>
    <p>Can be set, to check the first radio button with the given value represented by the object.</p>
   </dl>

  

  <p>The object's <a id=the-htmlformcontrolscollection-interface:supported-property-indices href=https://heycam.github.io/webidl/#dfn-supported-property-indices data-x-internal=supported-property-indices>supported property indices</a> are as defined for
  <code id=the-htmlformcontrolscollection-interface:htmlcollection-2><a data-x-internal=htmlcollection href=https://dom.spec.whatwg.org/#interface-htmlcollection>HTMLCollection</a></code> objects.</p>

  <p>The <a id=the-htmlformcontrolscollection-interface:supported-property-names href=https://heycam.github.io/webidl/#dfn-supported-property-names data-x-internal=supported-property-names>supported property names</a> consist of the non-empty values of all the <code id=the-htmlformcontrolscollection-interface:the-id-attribute><a href=#the-id-attribute>id</a></code> and <code id=the-htmlformcontrolscollection-interface:attr-fe-name-2><a href=#attr-fe-name>name</a></code> attributes of all the
  elements <a id=the-htmlformcontrolscollection-interface:represented-by-the-collection href=https://dom.spec.whatwg.org/#represented-by-the-collection data-x-internal=represented-by-the-collection>represented by the collection</a>, in <a id=the-htmlformcontrolscollection-interface:tree-order-2 href=https://dom.spec.whatwg.org/#concept-tree-order data-x-internal=tree-order>tree order</a>, ignoring later
  duplicates, with the <code id=the-htmlformcontrolscollection-interface:the-id-attribute-2><a href=#the-id-attribute>id</a></code> of an element preceding its <code id=the-htmlformcontrolscollection-interface:attr-fe-name-3><a href=#attr-fe-name>name</a></code> if it contributes both, they differ from each other, and neither is the
  duplicate of an earlier entry.</p>

  <div class="mdn-anno wrapped before"><button onclick=toggleStatus(this) class=mdn-anno-btn><b title="Support in all current engines." class=all-engines-flag>✔</b><span>MDN</span></button><div class=feature><p><a href=https://developer.mozilla.org/en-US/docs/Web/API/HTMLFormControlsCollection/namedItem title="The HTMLFormControlsCollection.namedItem() method returns the RadioNodeList or the Element in the collection whose name or id match the specified name, or null if no node matches.">HTMLFormControlsCollection/namedItem</a><p class=all-engines-text>Support in all current engines.<div class=support><span class="firefox yes"><span>Firefox</span><span>33+</span></span><span class="safari yes"><span>Safari</span><span>Yes</span></span><span class="chrome yes"><span>Chrome</span><span>Yes</span></span><hr><span class="opera yes"><span>Opera</span><span>Yes</span></span><span class="edge_blink yes"><span>Edge</span><span>Yes</span></span><hr><span class="edge no"><span>Edge (Legacy)</span><span>No</span></span><span class="ie no"><span>Internet Explorer</span><span>No</span></span><hr><span class="firefox_android yes"><span>Firefox Android</span><span>33+</span></span><span class="safari_ios yes"><span>Safari iOS</span><span>Yes</span></span><span class="chrome_android yes"><span>Chrome Android</span><span>Yes</span></span><span class="webview_android yes"><span>WebView Android</span><span>Yes</span></span><span class="samsunginternet_android yes"><span>Samsung Internet</span><span>Yes</span></span><span class="opera_android yes"><span>Opera Android</span><span>Yes</span></span></div></div></div><p>The <dfn id=dom-htmlformcontrolscollection-nameditem><code>namedItem(<var>name</var>)</code></dfn> method must act according to the following algorithm:</p>

  <ol><li>If <var>name</var> is the empty string, return null and stop the algorithm.<li>If, at the time the method is called, there is exactly one node in the collection that has
   either an <code id=the-htmlformcontrolscollection-interface:the-id-attribute-3><a href=#the-id-attribute>id</a></code> attribute or a <code id=the-htmlformcontrolscollection-interface:attr-fe-name-4><a href=#attr-fe-name>name</a></code>
   attribute equal to <var>name</var>, then return that node and stop the algorithm.<li>Otherwise, if there are no nodes in the collection that have either an <code id=the-htmlformcontrolscollection-interface:the-id-attribute-4><a href=#the-id-attribute>id</a></code> attribute or a <code id=the-htmlformcontrolscollection-interface:attr-fe-name-5><a href=#attr-fe-name>name</a></code> attribute equal
   to <var>name</var>, then return null and stop the algorithm.<li>Otherwise, create a new <code id=the-htmlformcontrolscollection-interface:radionodelist-3><a href=#radionodelist>RadioNodeList</a></code> object representing a <a href=#live id=the-htmlformcontrolscollection-interface:live>live</a>
   view of the <code id=the-htmlformcontrolscollection-interface:htmlformcontrolscollection-3><a href=#htmlformcontrolscollection>HTMLFormControlsCollection</a></code> object, further filtered so that the only
   nodes in the <code id=the-htmlformcontrolscollection-interface:radionodelist-4><a href=#radionodelist>RadioNodeList</a></code> object are those that have either an <code id=the-htmlformcontrolscollection-interface:the-id-attribute-5><a href=#the-id-attribute>id</a></code> attribute or a <code id=the-htmlformcontrolscollection-interface:attr-fe-name-6><a href=#attr-fe-name>name</a></code> attribute equal
   to <var>name</var>. The nodes in the <code id=the-htmlformcontrolscollection-interface:radionodelist-5><a href=#radionodelist>RadioNodeList</a></code> object must be sorted in
   <a id=the-htmlformcontrolscollection-interface:tree-order-3 href=https://dom.spec.whatwg.org/#concept-tree-order data-x-internal=tree-order>tree order</a>.<li>Return that <code id=the-htmlformcontrolscollection-interface:radionodelist-6><a href=#radionodelist>RadioNodeList</a></code> object.</ol>

  <hr>

  <p>Members of the <code id=the-htmlformcontrolscollection-interface:radionodelist-7><a href=#radionodelist>RadioNodeList</a></code> interface inherited from the <code id=the-htmlformcontrolscollection-interface:nodelist-2><a data-x-internal=nodelist href=https://dom.spec.whatwg.org/#interface-nodelist>NodeList</a></code>
  interface must behave as they would on a <code id=the-htmlformcontrolscollection-interface:nodelist-3><a data-x-internal=nodelist href=https://dom.spec.whatwg.org/#interface-nodelist>NodeList</a></code> object.</p>

  <div class="mdn-anno wrapped before"><button onclick=toggleStatus(this) class=mdn-anno-btn><b title="Support in all current engines." class=all-engines-flag>✔</b><span>MDN</span></button><div class=feature><p><a href=https://developer.mozilla.org/en-US/docs/Web/API/RadioNodeList/value title="If the underlying element collection contains radio buttons, the RadioNodeList.value property represents the checked radio button. On retrieving the value property, the value of the currently checked radio button is returned as a string. If the collection does not contain any radio buttons or none of the radio buttons in the collection is in checked state, the empty string is returned. On setting the value property, the first radio button input element whose value property is equal to the new value will be set to checked.">RadioNodeList/value</a><p class=all-engines-text>Support in all current engines.<div class=support><span class="firefox yes"><span>Firefox</span><span>33+</span></span><span class="safari yes"><span>Safari</span><span>10+</span></span><span class="chrome yes"><span>Chrome</span><span>34+</span></span><hr><span class="opera yes"><span>Opera</span><span>Yes</span></span><span class="edge_blink yes"><span>Edge</span><span>79+</span></span><hr><span class="edge yes"><span>Edge (Legacy)</span><span>18</span></span><span class="ie yes"><span>Internet Explorer</span><span>Yes</span></span><hr><span class="firefox_android yes"><span>Firefox Android</span><span>33+</span></span><span class="safari_ios yes"><span>Safari iOS</span><span>9+</span></span><span class="chrome_android yes"><span>Chrome Android</span><span>Yes</span></span><span class="webview_android yes"><span>WebView Android</span><span>Yes</span></span><span class="samsunginternet_android yes"><span>Samsung Internet</span><span>Yes</span></span><span class="opera_android yes"><span>Opera Android</span><span>Yes</span></span></div></div></div><p>The <dfn id=dom-radionodelist-value><code>value</code></dfn> IDL attribute on the
  <code id=the-htmlformcontrolscollection-interface:radionodelist-8><a href=#radionodelist>RadioNodeList</a></code> object, on getting, must return the value returned by running the
  following steps:</p>

  <ol><li><p>Let <var>element</var> be the first element in <a id=the-htmlformcontrolscollection-interface:tree-order-4 href=https://dom.spec.whatwg.org/#concept-tree-order data-x-internal=tree-order>tree order</a>
   represented by the <code id=the-htmlformcontrolscollection-interface:radionodelist-9><a href=#radionodelist>RadioNodeList</a></code> object that is an <code id=the-htmlformcontrolscollection-interface:the-input-element><a href=#the-input-element>input</a></code> element whose
   <code id=the-htmlformcontrolscollection-interface:attr-input-type><a href=#attr-input-type>type</a></code> attribute is in the <a href="#radio-button-state-(type=radio)" id="the-htmlformcontrolscollection-interface:radio-button-state-(type=radio)">Radio Button</a> state and whose <a href=#concept-fe-checked id=the-htmlformcontrolscollection-interface:concept-fe-checked>checkedness</a> is true. Otherwise, let it be null.<li><p>If <var>element</var> is null, return the empty string.<li><p>If <var>element</var> is an element with no <code id=the-htmlformcontrolscollection-interface:attr-input-value><a href=#attr-input-value>value</a></code> attribute, return the string "<code>on</code>".<li><p>Otherwise, return the value of <var>element</var>'s <code id=the-htmlformcontrolscollection-interface:attr-input-value-2><a href=#attr-input-value>value</a></code> attribute.</ol>

  <p>On setting, the <code id=the-htmlformcontrolscollection-interface:dom-radionodelist-value-2><a href=#dom-radionodelist-value>value</a></code> IDL attribute must run the
  following steps:</p>

  <ol><li><p>If the new value is the string "<code>on</code>": let <var>element</var> be the first element in <a id=the-htmlformcontrolscollection-interface:tree-order-5 href=https://dom.spec.whatwg.org/#concept-tree-order data-x-internal=tree-order>tree order</a>
   represented by the <code id=the-htmlformcontrolscollection-interface:radionodelist-10><a href=#radionodelist>RadioNodeList</a></code> object that is an <code id=the-htmlformcontrolscollection-interface:the-input-element-2><a href=#the-input-element>input</a></code> element whose
   <code id=the-htmlformcontrolscollection-interface:attr-input-type-2><a href=#attr-input-type>type</a></code> attribute is in the <a href="#radio-button-state-(type=radio)" id="the-htmlformcontrolscollection-interface:radio-button-state-(type=radio)-2">Radio Button</a> state and whose <code id=the-htmlformcontrolscollection-interface:attr-input-value-3><a href=#attr-input-value>value</a></code> content attribute is either absent, or present and equal to the new value, if any. If no such element exists, then instead let <var>element</var> be null.</p>

   <p>Otherwise: let <var>element</var> be the first element in <a id=the-htmlformcontrolscollection-interface:tree-order-6 href=https://dom.spec.whatwg.org/#concept-tree-order data-x-internal=tree-order>tree order</a>
   represented by the <code id=the-htmlformcontrolscollection-interface:radionodelist-11><a href=#radionodelist>RadioNodeList</a></code> object that is an <code id=the-htmlformcontrolscollection-interface:the-input-element-3><a href=#the-input-element>input</a></code> element whose
   <code id=the-htmlformcontrolscollection-interface:attr-input-type-3><a href=#attr-input-type>type</a></code> attribute is in the <a href="#radio-button-state-(type=radio)" id="the-htmlformcontrolscollection-interface:radio-button-state-(type=radio)-3">Radio Button</a> state and whose <code id=the-htmlformcontrolscollection-interface:attr-input-value-4><a href=#attr-input-value>value</a></code> content attribute is present and equal to the new value, if
   any. If no such element exists, then instead let <var>element</var> be null.<li><p>If <var>element</var> is not null, then set its <a href=#concept-fe-checked id=the-htmlformcontrolscollection-interface:concept-fe-checked-2>checkedness</a> to true.</p>
  </ol>



  


  <h5 id=the-htmloptionscollection-interface><span class=secno>2.7.2.3</span> The <code id=the-htmloptionscollection-interface:htmloptionscollection><a href=#htmloptionscollection>HTMLOptionsCollection</a></code> interface<a href=#the-htmloptionscollection-interface class=self-link></a></h5><div class="mdn-anno wrapped"><button onclick=toggleStatus(this) class=mdn-anno-btn><b title="Support in all current engines." class=all-engines-flag>✔</b><span>MDN</span></button><div class=feature><p><a href=https://developer.mozilla.org/en-US/docs/Web/API/HTMLOptionsCollection title="This interface inherits the methods of its parent, HTMLCollection.">HTMLOptionsCollection</a><p class=all-engines-text>Support in all current engines.<div class=support><span class="firefox yes"><span>Firefox</span><span>Yes</span></span><span class="safari yes"><span>Safari</span><span>Yes</span></span><span class="chrome yes"><span>Chrome</span><span>Yes</span></span><hr><span class="opera yes"><span>Opera</span><span>Yes</span></span><span class="edge_blink yes"><span>Edge</span><span>Yes</span></span><hr><span class="edge yes"><span>Edge (Legacy)</span><span>12+</span></span><span class="ie yes"><span>Internet Explorer</span><span>Yes</span></span><hr><span class="firefox_android yes"><span>Firefox Android</span><span>Yes</span></span><span class="safari_ios yes"><span>Safari iOS</span><span>Yes</span></span><span class="chrome_android yes"><span>Chrome Android</span><span>Yes</span></span><span class="webview_android yes"><span>WebView Android</span><span>Yes</span></span><span class="samsunginternet_android yes"><span>Samsung Internet</span><span>Yes</span></span><span class="opera_android yes"><span>Opera Android</span><span>Yes</span></span></div></div></div>

  <p>The <code id=the-htmloptionscollection-interface:htmloptionscollection-2><a href=#htmloptionscollection>HTMLOptionsCollection</a></code> interface is used for <a href=https://dom.spec.whatwg.org/#concept-collection id=the-htmloptionscollection-interface:concept-collection data-x-internal=concept-collection>collections</a> of <code id=the-htmloptionscollection-interface:the-option-element><a href=#the-option-element>option</a></code> elements. It is always
  rooted on a <code id=the-htmloptionscollection-interface:the-select-element><a href=#the-select-element>select</a></code> element and has attributes and methods that manipulate that
  element's descendants.</p>

  <pre><code class='idl'>[<c- g>Exposed</c->=<c- n>Window</c->]
<c- b>interface</c-> <dfn id='htmloptionscollection'><c- g>HTMLOptionsCollection</c-></dfn> : <a id='the-htmloptionscollection-interface:htmlcollection' href='https://dom.spec.whatwg.org/#interface-htmlcollection' data-x-internal='htmlcollection'><c- n>HTMLCollection</c-></a> {
  // inherits <a href='https://dom.spec.whatwg.org/#dom-htmlcollection-item' id='the-htmloptionscollection-interface:dom-htmlcollection-item' data-x-internal='dom-htmlcollection-item'>item</a>(), <a href='https://dom.spec.whatwg.org/#dom-htmlcollection-nameditem' id='the-htmloptionscollection-interface:dom-htmlcollection-nameditem' data-x-internal='dom-htmlcollection-nameditem'>namedItem</a>()
  [<a href='#cereactions' id='the-htmloptionscollection-interface:cereactions'><c- g>CEReactions</c-></a>] <c- b>attribute</c-> <c- b>unsigned</c-> <c- b>long</c-> <a href='#dom-htmloptionscollection-length' id='the-htmloptionscollection-interface:dom-htmloptionscollection-length'><c- g>length</c-></a>; // shadows inherited <a href='https://dom.spec.whatwg.org/#dom-htmlcollection-length' id='the-htmloptionscollection-interface:dom-htmlcollection-length' data-x-internal='dom-htmlcollection-length'>length</a>
  [<a href='#cereactions' id='the-htmloptionscollection-interface:cereactions-2'><c- g>CEReactions</c-></a>] <a href='#dom-htmloptionscollection-setter'><c- b>setter</c-></a> <c- b>undefined</c-> (<c- b>unsigned</c-> <c- b>long</c-> <c- g>index</c->, <a href='#htmloptionelement' id='the-htmloptionscollection-interface:htmloptionelement'><c- n>HTMLOptionElement</c-></a>? <c- g>option</c->);
  [<a href='#cereactions' id='the-htmloptionscollection-interface:cereactions-3'><c- g>CEReactions</c-></a>] <c- b>undefined</c-> <a href='#dom-htmloptionscollection-add' id='the-htmloptionscollection-interface:dom-htmloptionscollection-add'><c- g>add</c-></a>((<a href='#htmloptionelement' id='the-htmloptionscollection-interface:htmloptionelement-2'><c- n>HTMLOptionElement</c-></a> <c- b>or</c-> <a href='#htmloptgroupelement' id='the-htmloptionscollection-interface:htmloptgroupelement'><c- n>HTMLOptGroupElement</c-></a>) <c- g>element</c->, <c- b>optional</c-> (<a href='#htmlelement' id='the-htmloptionscollection-interface:htmlelement'><c- n>HTMLElement</c-></a> <c- b>or</c-> <c- b>long</c->)? <c- g>before</c-> = <c- b>null</c->);
  [<a href='#cereactions' id='the-htmloptionscollection-interface:cereactions-4'><c- g>CEReactions</c-></a>] <c- b>undefined</c-> <a href='#dom-htmloptionscollection-remove' id='the-htmloptionscollection-interface:dom-htmloptionscollection-remove'><c- g>remove</c-></a>(<c- b>long</c-> <c- g>index</c->);
  <c- b>attribute</c-> <c- b>long</c-> <a href='#dom-htmloptionscollection-selectedindex' id='the-htmloptionscollection-interface:dom-htmloptionscollection-selectedindex'><c- g>selectedIndex</c-></a>;
};</code></pre>

  <dl class=domintro><dt><var>collection</var> . <code id=dom-htmloptionscollection-length-dev><a href=#dom-htmloptionscollection-length>length</a></code> [ = <var>value</var> ]<dd>
    <p>Returns the number of elements in the collection.</p>
    <p>When set to a smaller number, truncates the number of <code id=the-htmloptionscollection-interface:the-option-element-2><a href=#the-option-element>option</a></code> elements in the corresponding container.</p>
    <p>When set to a greater number, adds new blank <code id=the-htmloptionscollection-interface:the-option-element-3><a href=#the-option-element>option</a></code> elements to that container.</p>
   <dt><var>element</var> = <var>collection</var> . <code id=the-htmloptionscollection-interface:dom-htmlcollection-item-2><a data-x-internal=dom-htmlcollection-item href=https://dom.spec.whatwg.org/#dom-htmlcollection-item>item</a></code>(<var>index</var>)<dt><var>element</var> = <var>collection</var>[<var>index</var>]<dd>
    <p>Returns the item with index <var>index</var> from the collection. The items are sorted in <a id=the-htmloptionscollection-interface:tree-order href=https://dom.spec.whatwg.org/#concept-tree-order data-x-internal=tree-order>tree order</a>.</p>
   <dt><var>collection</var>[<var>index</var>] = <var>element</var><dd>
    <p>When <var>index</var> is a greater number than the number of items in the collection, adds
    new blank <code id=the-htmloptionscollection-interface:the-option-element-4><a href=#the-option-element>option</a></code> elements in the corresponding container.
    <p>When set to null, removes the item at index <var>index</var> from the collection.</p>
    <p>When set to an <code id=the-htmloptionscollection-interface:the-option-element-5><a href=#the-option-element>option</a></code> element, adds or replaces it at index <var>index</var>
    from the collection.</p>
   <dt><var>element</var> = <var>collection</var> . <code id=dom-htmlcollection-nameditem-dev><a data-x-internal=dom-htmlcollection-nameditem href=https://dom.spec.whatwg.org/#dom-htmlcollection-nameditem>namedItem</a></code>(<var>name</var>)<dt><var>element</var> = <var>collection</var>[<var>name</var>]<dd>
    <p>Returns the item with <a href=https://dom.spec.whatwg.org/#concept-id id=the-htmloptionscollection-interface:concept-id data-x-internal=concept-id>ID</a> or <code id=the-htmloptionscollection-interface:attr-option-name><a href=#attr-option-name>name</a></code> <var>name</var> from the collection.</p>
    <p>If there are multiple matching items, then the first is returned.</p>
   <dt><var>collection</var> . <code id=dom-htmloptionscollection-add-dev><a href=#dom-htmloptionscollection-add>add</a></code>(<var>element</var> [, <var>before</var> ] )<dd>
    <p>Inserts <var>element</var> before the node given by <var>before</var>.</p>
    <p>The <var>before</var> argument can be a number, in which case <var>element</var> is inserted before the item with that number, or an element from the
    collection, in which case <var>element</var> is inserted before that element.</p>
    <p>If <var>before</var> is omitted, null, or a number out of range, then <var>element</var> will be added at the end of the list.</p>
    <p>This method will throw a <a id=the-htmloptionscollection-interface:hierarchyrequesterror href=https://heycam.github.io/webidl/#hierarchyrequesterror data-x-internal=hierarchyrequesterror>"<code>HierarchyRequestError</code>"</a> <code id=the-htmloptionscollection-interface:domexception><a data-x-internal=domexception href=https://heycam.github.io/webidl/#dfn-DOMException>DOMException</a></code> if
    <var>element</var> is an ancestor of the element into which it is to be inserted.</p>
   <dt><var>collection</var> . <code id=dom-htmloptionscollection-remove-dev><a href=#dom-htmloptionscollection-remove>remove</a></code>(<var>index</var>)<dd>
    <p>Removes the item with index <var>index</var> from the collection.</p>
   <dt><var>collection</var> . <code id=dom-htmloptionscollection-selectedindex-dev><a href=#dom-htmloptionscollection-selectedindex>selectedIndex</a></code> [ = <var>value</var> ]<dd>
    <p>Returns the index of the first selected item, if any, or −1 if there is no selected
    item.</p>

    <p>Can be set, to change the selection.</p>
   </dl>

  

  <p>The object's <a id=the-htmloptionscollection-interface:supported-property-indices href=https://heycam.github.io/webidl/#dfn-supported-property-indices data-x-internal=supported-property-indices>supported property indices</a> are as defined for
  <code id=the-htmloptionscollection-interface:htmlcollection-2><a data-x-internal=htmlcollection href=https://dom.spec.whatwg.org/#interface-htmlcollection>HTMLCollection</a></code> objects.</p>

  <p>On getting, the <dfn id=dom-htmloptionscollection-length><code>length</code></dfn>
  attribute must return the number of nodes <a id=the-htmloptionscollection-interface:represented-by-the-collection href=https://dom.spec.whatwg.org/#represented-by-the-collection data-x-internal=represented-by-the-collection>represented by the collection</a>.</p>

  <p>On setting, the behavior depends on whether the new value is equal to, greater than, or less
  than the number of nodes <a id=the-htmloptionscollection-interface:represented-by-the-collection-2 href=https://dom.spec.whatwg.org/#represented-by-the-collection data-x-internal=represented-by-the-collection>represented by the collection</a> at that time. If the number is
  the same, then setting the attribute must do nothing. If the new value is greater, then <var>n</var> new <code id=the-htmloptionscollection-interface:the-option-element-6><a href=#the-option-element>option</a></code> elements with no attributes and no child nodes must be
  appended to the <code id=the-htmloptionscollection-interface:the-select-element-2><a href=#the-select-element>select</a></code> element on which the <code id=the-htmloptionscollection-interface:htmloptionscollection-3><a href=#htmloptionscollection>HTMLOptionsCollection</a></code> is
  rooted, where <var>n</var> is the difference between the two numbers (new value minus old
  value). Mutation events must be fired as if a <code id=the-htmloptionscollection-interface:documentfragment><a data-x-internal=documentfragment href=https://dom.spec.whatwg.org/#interface-documentfragment>DocumentFragment</a></code> containing the new
  <code id=the-htmloptionscollection-interface:the-option-element-7><a href=#the-option-element>option</a></code> elements had been inserted. If the new value is lower, then the last <var>n</var> nodes in the collection must be removed from their parent nodes, where <var>n</var> is the difference between the two numbers (old value minus new value).</p>

  <p class=note>Setting <code id=the-htmloptionscollection-interface:dom-htmloptionscollection-length-2><a href=#dom-htmloptionscollection-length>length</a></code> never removes
  or adds any <code id=the-htmloptionscollection-interface:the-optgroup-element><a href=#the-optgroup-element>optgroup</a></code> elements, and never adds new children to existing
  <code id=the-htmloptionscollection-interface:the-optgroup-element-2><a href=#the-optgroup-element>optgroup</a></code> elements (though it can remove children from them).</p>

  <p>The <a id=the-htmloptionscollection-interface:supported-property-names href=https://heycam.github.io/webidl/#dfn-supported-property-names data-x-internal=supported-property-names>supported property names</a> consist of the non-empty values of all the <code id=the-htmloptionscollection-interface:the-id-attribute><a href=#the-id-attribute>id</a></code> and <code id=the-htmloptionscollection-interface:attr-option-name-2><a href=#attr-option-name>name</a></code> attributes of all the
  elements <a id=the-htmloptionscollection-interface:represented-by-the-collection-3 href=https://dom.spec.whatwg.org/#represented-by-the-collection data-x-internal=represented-by-the-collection>represented by the collection</a>, in <a id=the-htmloptionscollection-interface:tree-order-2 href=https://dom.spec.whatwg.org/#concept-tree-order data-x-internal=tree-order>tree order</a>, ignoring later
  duplicates, with the <code id=the-htmloptionscollection-interface:the-id-attribute-2><a href=#the-id-attribute>id</a></code> of an element preceding its <code id=the-htmloptionscollection-interface:attr-option-name-3><a href=#attr-option-name>name</a></code> if it contributes both, they differ from each other, and neither is
  the duplicate of an earlier entry.</p>

  <p id=dom-htmloptionscollection-setter>When the user agent is to <a id=the-htmloptionscollection-interface:set-the-value-of-a-new-indexed-property href=https://heycam.github.io/webidl/#dfn-set-the-value-of-a-new-indexed-property data-x-internal=set-the-value-of-a-new-indexed-property>set the value of a new
  indexed property</a> or <a id=the-htmloptionscollection-interface:set-the-value-of-an-existing-indexed-property href=https://heycam.github.io/webidl/#dfn-set-the-value-of-an-existing-indexed-property data-x-internal=set-the-value-of-an-existing-indexed-property>set the value of an existing indexed property</a> for a given
  property index <var>index</var> to a new value <var>value</var>, it must run the following
  algorithm:</p>

  <ol><li><p>If <var>value</var> is null, invoke the steps for the <code id=the-htmloptionscollection-interface:dom-htmloptionscollection-remove-2><a href=#dom-htmloptionscollection-remove>remove</a></code> method with <var>index</var> as
   the argument, and return.<li><p>Let <var>length</var> be the number of nodes <a id=the-htmloptionscollection-interface:represented-by-the-collection-4 href=https://dom.spec.whatwg.org/#represented-by-the-collection data-x-internal=represented-by-the-collection>represented by the
   collection</a>.<li><p>Let <var>n</var> be <var>index</var> minus <var>length</var>.<li><p>If <var>n</var> is greater than zero, then <a href=https://dom.spec.whatwg.org/#concept-node-append id=the-htmloptionscollection-interface:concept-node-append data-x-internal=concept-node-append>append</a> a <code id=the-htmloptionscollection-interface:documentfragment-2><a data-x-internal=documentfragment href=https://dom.spec.whatwg.org/#interface-documentfragment>DocumentFragment</a></code> consisting of <span><var>n</var>-1</span> new <code id=the-htmloptionscollection-interface:the-option-element-8><a href=#the-option-element>option</a></code> elements with no attributes and
   no child nodes to the <code id=the-htmloptionscollection-interface:the-select-element-3><a href=#the-select-element>select</a></code> element on which the <code id=the-htmloptionscollection-interface:htmloptionscollection-4><a href=#htmloptionscollection>HTMLOptionsCollection</a></code>
   is rooted.<li><p>If <var>n</var> is greater than or equal to zero, <a href=https://dom.spec.whatwg.org/#concept-node-append id=the-htmloptionscollection-interface:concept-node-append-2 data-x-internal=concept-node-append>append</a> <var>value</var> to the <code id=the-htmloptionscollection-interface:the-select-element-4><a href=#the-select-element>select</a></code>
   element. Otherwise, <a href=https://dom.spec.whatwg.org/#concept-node-replace id=the-htmloptionscollection-interface:concept-node-replace data-x-internal=concept-node-replace>replace</a> the <var>index</var>th element in the collection by <var>value</var>.</ol>

  <p>The <dfn id=dom-htmloptionscollection-add><code>add(<var>element</var>, <var>before</var>)</code></dfn> method must act according to the following algorithm:</p>

  <ol><li><p>If <var>element</var> is an ancestor of the <code id=the-htmloptionscollection-interface:the-select-element-5><a href=#the-select-element>select</a></code> element on which
   the <code id=the-htmloptionscollection-interface:htmloptionscollection-5><a href=#htmloptionscollection>HTMLOptionsCollection</a></code> is rooted, then throw a
   <a id=the-htmloptionscollection-interface:hierarchyrequesterror-2 href=https://heycam.github.io/webidl/#hierarchyrequesterror data-x-internal=hierarchyrequesterror>"<code>HierarchyRequestError</code>"</a> <code id=the-htmloptionscollection-interface:domexception-2><a data-x-internal=domexception href=https://heycam.github.io/webidl/#dfn-DOMException>DOMException</a></code>.<li><p>If <var>before</var> is an element, but that element isn't a descendant of the
   <code id=the-htmloptionscollection-interface:the-select-element-6><a href=#the-select-element>select</a></code> element on which the <code id=the-htmloptionscollection-interface:htmloptionscollection-6><a href=#htmloptionscollection>HTMLOptionsCollection</a></code> is rooted, then throw
   a <a id=the-htmloptionscollection-interface:notfounderror href=https://heycam.github.io/webidl/#notfounderror data-x-internal=notfounderror>"<code>NotFoundError</code>"</a> <code id=the-htmloptionscollection-interface:domexception-3><a data-x-internal=domexception href=https://heycam.github.io/webidl/#dfn-DOMException>DOMException</a></code>.<li><p>If <var>element</var> and <var>before</var> are the same element, then
   return.<li><p>If <var>before</var> is a node, then let <var>reference</var> be that
   node. Otherwise, if <var>before</var> is an integer, and there is a <var>before</var>th node in the collection, let <var>reference</var> be that node.
   Otherwise, let <var>reference</var> be null.<li><p>If <var>reference</var> is not null, let <var>parent</var> be the parent
   node of <var>reference</var>. Otherwise, let <var>parent</var> be the
   <code id=the-htmloptionscollection-interface:the-select-element-7><a href=#the-select-element>select</a></code> element on which the <code id=the-htmloptionscollection-interface:htmloptionscollection-7><a href=#htmloptionscollection>HTMLOptionsCollection</a></code> is rooted.<li><p><a id=the-htmloptionscollection-interface:pre-insert href=https://dom.spec.whatwg.org/#concept-node-pre-insert data-x-internal=pre-insert>Pre-insert</a> <var>element</var> into <var>parent</var> node before
   <var>reference</var>.</p>
  </ol>

  <p>The <dfn id=dom-htmloptionscollection-remove><code>remove(<var>index</var>)</code></dfn> method must act according to the following algorithm:</p>

  <ol><li><p>If the number of nodes <a id=the-htmloptionscollection-interface:represented-by-the-collection-5 href=https://dom.spec.whatwg.org/#represented-by-the-collection data-x-internal=represented-by-the-collection>represented by the collection</a> is zero, return.<li><p>If <var>index</var> is not a number greater than or equal to 0 and less than the
   number of nodes <a id=the-htmloptionscollection-interface:represented-by-the-collection-6 href=https://dom.spec.whatwg.org/#represented-by-the-collection data-x-internal=represented-by-the-collection>represented by the collection</a>, return.<li><p>Let <var>element</var> be the <var>index</var>th element in the
   collection.<li><p>Remove <var>element</var> from its parent node.</ol>

  <p>The <dfn id=dom-htmloptionscollection-selectedindex><code>selectedIndex</code></dfn> IDL
  attribute must act like the identically named attribute on the <code id=the-htmloptionscollection-interface:the-select-element-8><a href=#the-select-element>select</a></code> element on
  which the <code id=the-htmloptionscollection-interface:htmloptionscollection-8><a href=#htmloptionscollection>HTMLOptionsCollection</a></code> is rooted</p>

  

  


  <h4 id=the-domstringlist-interface><span class=secno>2.7.3</span> The <code id=the-domstringlist-interface:domstringlist><a href=#domstringlist>DOMStringList</a></code> interface<a href=#the-domstringlist-interface class=self-link></a></h4><div class="mdn-anno wrapped"><button onclick=toggleStatus(this) class=mdn-anno-btn><b title="Support in all current engines." class=all-engines-flag>✔</b><span>MDN</span></button><div class=feature><p><a href=https://developer.mozilla.org/en-US/docs/Web/API/DOMStringList title="A type returned by some APIs which contains a list of DOMString (strings).">DOMStringList</a><p class=all-engines-text>Support in all current engines.<div class=support><span class="firefox yes"><span>Firefox</span><span>Yes</span></span><span class="safari yes"><span>Safari</span><span>Yes</span></span><span class="chrome yes"><span>Chrome</span><span>Yes</span></span><hr><span class="opera yes"><span>Opera</span><span>Yes</span></span><span class="edge_blink yes"><span>Edge</span><span>Yes</span></span><hr><span class="edge yes"><span>Edge (Legacy)</span><span>12+</span></span><span class="ie yes"><span>Internet Explorer</span><span>10+</span></span><hr><span class="firefox_android yes"><span>Firefox Android</span><span>Yes</span></span><span class="safari_ios yes"><span>Safari iOS</span><span>Yes</span></span><span class="chrome_android yes"><span>Chrome Android</span><span>Yes</span></span><span class="webview_android yes"><span>WebView Android</span><span>Yes</span></span><span class="samsunginternet_android yes"><span>Samsung Internet</span><span>Yes</span></span><span class="opera_android yes"><span>Opera Android</span><span>Yes</span></span></div></div></div>

  <p>The <code id=the-domstringlist-interface:domstringlist-2><a href=#domstringlist>DOMStringList</a></code> interface is a non-fashionable retro way of representing a list
  of strings.</p>

  <pre><code class='idl'>[<c- g>Exposed</c->=(<c- n>Window</c->,<c- n>Worker</c->)]
<c- b>interface</c-> <dfn id='domstringlist'><c- g>DOMStringList</c-></dfn> {
  <c- b>readonly</c-> <c- b>attribute</c-> <c- b>unsigned</c-> <c- b>long</c-> <a href='#dom-domstringlist-length' id='the-domstringlist-interface:dom-domstringlist-length'><c- g>length</c-></a>;
  <c- b>getter</c-> <c- b>DOMString</c->? <a href='#dom-domstringlist-item' id='the-domstringlist-interface:dom-domstringlist-item'><c- g>item</c-></a>(<c- b>unsigned</c-> <c- b>long</c-> <c- g>index</c->);
  <c- b>boolean</c-> <a href='#dom-domstringlist-contains' id='the-domstringlist-interface:dom-domstringlist-contains'><c- g>contains</c-></a>(<c- b>DOMString</c-> <c- g>string</c->);
};</code></pre>

  <p class=warning>New APIs must use <code>sequence&lt;DOMString></code> or
  equivalent rather than <code id=the-domstringlist-interface:domstringlist-3><a href=#domstringlist>DOMStringList</a></code>.</p>

  <dl class=domintro><dt><var>strings</var> . <code id=dom-domstringlist-length-dev><a href=#dom-domstringlist-length>length</a></code><dd><p>Returns the number of strings in <var>strings</var>.<dt><var>strings</var>[<var>index</var>]<dt><var>strings</var> . <code id=dom-domstringlist-item-dev><a href=#dom-domstringlist-item>item</a></code>(<var>index</var>)<dd><p>Returns the string with index <var>index</var> from <var>strings</var>.<dt><var>strings</var> . <code id=dom-domstringlist-contains-dev><a href=#dom-domstringlist-contains>contains</a></code>(<var>string</var>)<dd><p>Returns true if <var>strings</var> contains <var>string</var>, and false
   otherwise.</dl>

  

  <p>Each <code id=the-domstringlist-interface:domstringlist-4><a href=#domstringlist>DOMStringList</a></code> object has an associated <a id=the-domstringlist-interface:list href=https://infra.spec.whatwg.org/#list data-x-internal=list>list</a>.</p>

  <p>The <a id=the-domstringlist-interface:supported-property-indices href=https://heycam.github.io/webidl/#dfn-supported-property-indices data-x-internal=supported-property-indices>supported property indices</a> for a <code id=the-domstringlist-interface:domstringlist-5><a href=#domstringlist>DOMStringList</a></code> object are the
  numbers zero to the associated list's <a href=https://infra.spec.whatwg.org/#list-size id=the-domstringlist-interface:list-size data-x-internal=list-size>size</a> minus one. If its
  associated list <a href=https://infra.spec.whatwg.org/#list-is-empty id=the-domstringlist-interface:list-is-empty data-x-internal=list-is-empty>is empty</a>, it has no <a id=the-domstringlist-interface:supported-property-indices-2 href=https://heycam.github.io/webidl/#dfn-supported-property-indices data-x-internal=supported-property-indices>supported property
  indices</a>.</p>

  <p>The <dfn id=dom-domstringlist-length><code>length</code></dfn> attribute's getter must
  return this <code id=the-domstringlist-interface:domstringlist-6><a href=#domstringlist>DOMStringList</a></code> object's associated list's <a href=https://infra.spec.whatwg.org/#list-size id=the-domstringlist-interface:list-size-2 data-x-internal=list-size>size</a>.</p>

  <p>The <dfn id=dom-domstringlist-item><code>item(<var>index</var>)</code></dfn> method, when
  invoked, must return the <var>index</var>th item in this <code id=the-domstringlist-interface:domstringlist-7><a href=#domstringlist>DOMStringList</a></code> object's
  associated list, or null if <var>index</var> plus one is greater than this <code id=the-domstringlist-interface:domstringlist-8><a href=#domstringlist>DOMStringList</a></code>
  object's associated list's <a href=https://infra.spec.whatwg.org/#list-size id=the-domstringlist-interface:list-size-3 data-x-internal=list-size>size</a>.</p>

  <p>The <dfn id=dom-domstringlist-contains><code>contains(<var>string</var>)</code></dfn>
  method, when invoked, must return true if this <code id=the-domstringlist-interface:domstringlist-9><a href=#domstringlist>DOMStringList</a></code> object's associated list
  <a href=https://infra.spec.whatwg.org/#list-contain id=the-domstringlist-interface:list-contains data-x-internal=list-contains>contains</a> <var>string</var>, and false otherwise.</p>

  


  

  <h3 id=safe-passing-of-structured-data><span class=secno>2.8</span> Safe passing of structured data<a href=#safe-passing-of-structured-data class=self-link></a></h3>

  <p>This section uses the terminology and typographic conventions from the JavaScript
  specification. <a href=#refsJAVASCRIPT>[JAVASCRIPT]</a></p>

  <h4 id=serializable-objects><span class=secno>2.8.1</span> <dfn>Serializable objects</dfn><a href=#serializable-objects class=self-link></a></h4>

  <p><a href=#serializable-objects id=serializable-objects:serializable-objects>Serializable objects</a> support being serialized, and later deserialized, in a way
  that is independent of any given <a id=serializable-objects:javascript-realm href=https://tc39.es/ecma262/#sec-code-realms data-x-internal=javascript-realm>JavaScript Realm</a>. This allows them to be stored on
  disk and later restored, or cloned across <a id=serializable-objects:agent href=https://tc39.es/ecma262/#sec-agents data-x-internal=agent>agent</a> and even <a id=serializable-objects:agent-cluster href=https://tc39.es/ecma262/#sec-agent-clusters data-x-internal=agent-cluster>agent cluster</a>
  boundaries.</p>

  <p>Not all objects are <a href=#serializable-objects id=serializable-objects:serializable-objects-2>serializable objects</a>, and not all aspects of objects that are
  <a href=#serializable-objects id=serializable-objects:serializable-objects-3>serializable objects</a> are necessarily preserved when they are serialized.</p>

  <p><a href=https://heycam.github.io/webidl/#dfn-platform-object id=serializable-objects:platform-object data-x-internal=platform-object>Platform objects</a> can be <a href=#serializable-objects id=serializable-objects:serializable-objects-4>serializable objects</a>
  if their <a id=serializable-objects:primary-interface href=https://heycam.github.io/webidl/#dfn-primary-interface data-x-internal=primary-interface>primary interface</a> is decorated with the <dfn id=serializable data-dfn-type=extended-attribute data-lt=Serializable><code>[Serializable]</code></dfn> IDL <a id=serializable-objects:extended-attribute href=https://heycam.github.io/webidl/#dfn-extended-attribute data-x-internal=extended-attribute>extended
  attribute</a>. Such interfaces must also define the following algorithms:</p>

  <dl><dt><dfn id=serialization-steps data-export="">serialization steps</dfn>, taking a <a id=serializable-objects:platform-object-2 href=https://heycam.github.io/webidl/#dfn-platform-object data-x-internal=platform-object>platform object</a>
   <var>value</var>, a <a id=serializable-objects:record href=https://tc39.es/ecma262/#sec-list-and-record-specification-type data-x-internal=record>Record</a> <var>serialized</var>, and a boolean
   <var>forStorage</var><dd>
    <p>A set of steps that serializes the data in <var>value</var> into fields of
    <var>serialized</var>. The resulting data serialized into <var>serialized</var> must be
    independent of any <a id=serializable-objects:javascript-realm-2 href=https://tc39.es/ecma262/#sec-code-realms data-x-internal=javascript-realm>JavaScript Realm</a>.</p>

    <p>These steps may throw an exception if serialization is not possible.</p>

    <p>These steps may perform a <a href=#sub-serialization id=serializable-objects:sub-serialization>sub-serialization</a> to serialize nested data
    structures. They should not call <a href=#structuredserialize id=serializable-objects:structuredserialize>StructuredSerialize</a> directly, as doing so will
    omit the important <var>memory</var> argument.</p>

    <p>The introduction of these steps should omit mention of the <var>forStorage</var> argument if
    it is not relevant to the algorithm.</p>
   <dt><dfn id=deserialization-steps data-export="">deserialization steps</dfn>, taking a <a id=serializable-objects:record-2 href=https://tc39.es/ecma262/#sec-list-and-record-specification-type data-x-internal=record>Record</a>
   <var>serialized</var> and a <a id=serializable-objects:platform-object-3 href=https://heycam.github.io/webidl/#dfn-platform-object data-x-internal=platform-object>platform object</a> <var>value</var><dd>
    <p>A set of steps that deserializes the data in <var>serialized</var>, using it to set up
    <var>value</var> as appropriate. <var>value</var> will be a newly-created instance of the
    <a id=serializable-objects:platform-object-4 href=https://heycam.github.io/webidl/#dfn-platform-object data-x-internal=platform-object>platform object</a> type in question, with none of its internal data set up; setting
    that up is the job of these steps.</p>

    <p>These steps may throw an exception if deserialization is not possible.</p>

    <p>These steps may perform a <a href=#sub-deserialization id=serializable-objects:sub-deserialization>sub-deserialization</a> to deserialize nested data
    structures. They should not call <a href=#structureddeserialize id=serializable-objects:structureddeserialize>StructuredDeserialize</a> directly, as doing so will
    omit the important <var>targetRealm</var> and <var>memory</var> arguments.</p>
   </dl>

  <p>It is up to the definition of individual platform objects to determine what data is serialized
  and deserialized by these steps. Typically the steps are very symmetric.</p>

  <p>The <code id=serializable-objects:serializable><a href=#serializable>[Serializable]</a></code> extended attribute must take no
  arguments, and must only appear on an interface. It must not appear more than once on an
  interface.</p>

  <p>For a given <a id=serializable-objects:platform-object-5 href=https://heycam.github.io/webidl/#dfn-platform-object data-x-internal=platform-object>platform object</a>, only the object's <a id=serializable-objects:primary-interface-2 href=https://heycam.github.io/webidl/#dfn-primary-interface data-x-internal=primary-interface>primary interface</a> is
  considered during the (de)serialization process. Thus, if inheritance is involved in defining the
  interface, each <code id=serializable-objects:serializable-2><a href=#serializable>[Serializable]</a></code>-annotated interface in the
  inheritance chain needs to define standalone <a href=#serialization-steps id=serializable-objects:serialization-steps>serialization steps</a> and
  <a href=#deserialization-steps id=serializable-objects:deserialization-steps>deserialization steps</a>, including taking into account any important data that might
  come from inherited interfaces.</p>

  <div class=example>
   <p>Let's say we were defining a platform object <code>Person</code>, which had
   associated with it two pieces of associated data:</p>

   <ul class=brief><li>a name value, which is a string;<li>and a best friend value, which is either another <code>Person</code> instance or
    null</ul>

   <p>We could then define <code>Person</code> instances to be <a href=#serializable-objects id=serializable-objects:serializable-objects-5>serializable
   objects</a> by annotating the <code>Person</code> interface with the <code id=serializable-objects:serializable-3><a href=#serializable>[Serializable]</a></code> <a id=serializable-objects:extended-attribute-2 href=https://heycam.github.io/webidl/#dfn-extended-attribute data-x-internal=extended-attribute>extended attribute</a>, and defining the
   following accompanying algorithms:</p>

   <dl><dt><a href=#serialization-steps id=serializable-objects:serialization-steps-2>serialization steps</a><dd>
     <ol><li><p>Set <var>serialized</var>.[[Name]] to <var>value</var>'s associated name
      value.<li><p>Let <var>serializedBestFriend</var> be the <a href=#sub-serialization id=serializable-objects:sub-serialization-2>sub-serialization</a> of
      <var>value</var>'s associated best friend value.<li><p>Set <var>serialized</var>.[[BestFriend]] to <var>serializedBestFriend</var>.</ol>
    <dt><a href=#deserialization-steps id=serializable-objects:deserialization-steps-2>deserialization steps</a><dd>
     <ol><li><p>Set <var>value</var>'s associated name value to
      <var>serialized</var>.[[Name]].<li><p>Let <var>deserializedBestFriend</var> be the <a href=#sub-deserialization id=serializable-objects:sub-deserialization-2>sub-deserialization</a> of
      <var>serialized</var>.[[BestFriend]].<li><p>Set <var>value</var>'s associated best friend value to
      <var>deserializedBestFriend</var>.</ol>
    </dl>
  </div>

  <p>Objects defined in the JavaScript specification are handled by the
  <a href=#structuredserialize id=serializable-objects:structuredserialize-2>StructuredSerialize</a> abstract operation directly.</p>

  <p id=cloneable-objects class=note>Originally, this specification defined the concept of
  "cloneable objects", which could be cloned from one <a id=serializable-objects:javascript-realm-3 href=https://tc39.es/ecma262/#sec-code-realms data-x-internal=javascript-realm>JavaScript Realm</a> to another.
  However, to better specify the behavior of certain more complex situations, the model was updated
  to make the serialization and deserialization explicit.</p>

  <h4 id=transferable-objects data-lt="transferable object" data-export=""><span class=secno>2.8.2</span> <dfn>Transferable objects</dfn><a href=#transferable-objects class=self-link></a></h4><div class="mdn-anno wrapped"><button onclick=toggleStatus(this) class=mdn-anno-btn><b title="Support in all current engines." class=all-engines-flag>✔</b><span>MDN</span></button><div class=feature><p><a href=https://developer.mozilla.org/en-US/docs/Web/API/Transferable title="The Transferable interface represents an object that can be transfered between different execution contexts, like the main thread and Web workers.">Transferable</a><p class=all-engines-text>Support in all current engines.<div class=support><span class="firefox yes"><span>Firefox</span><span>4+</span></span><span class="safari yes"><span>Safari</span><span>Yes</span></span><span class="chrome yes"><span>Chrome</span><span>Yes</span></span><hr><span class="opera yes"><span>Opera</span><span>Yes</span></span><span class="edge_blink yes"><span>Edge</span><span>Yes</span></span><hr><span class="edge yes"><span>Edge (Legacy)</span><span>12+</span></span><span class="ie yes"><span>Internet Explorer</span><span>10+</span></span><hr><span class="firefox_android yes"><span>Firefox Android</span><span>4+</span></span><span class="safari_ios yes"><span>Safari iOS</span><span>Yes</span></span><span class="chrome_android yes"><span>Chrome Android</span><span>Yes</span></span><span class="webview_android yes"><span>WebView Android</span><span>Yes</span></span><span class="samsunginternet_android yes"><span>Samsung Internet</span><span>Yes</span></span><span class="opera_android yes"><span>Opera Android</span><span>Yes</span></span></div></div></div>

  <p><a href=#transferable-objects id=transferable-objects:transferable-objects>Transferable objects</a> support being transferred across <a href=https://tc39.es/ecma262/#sec-agents id=transferable-objects:agent data-x-internal=agent>agents</a>. Transferring is effectively recreating the object while sharing a
  reference to the underlying data and then detaching the object being transferred. This is useful
  to transfer ownership of expensive resources. Not all objects are <a href=#transferable-objects id=transferable-objects:transferable-objects-2>transferable
  objects</a> and not all aspects of objects that are <a href=#transferable-objects id=transferable-objects:transferable-objects-3>transferable objects</a> are
  necessarily preserved when transferred.</p>

  <p class=note>Transferring is an irreversible and non-idempotent operation. Once an object has
  been transferred, it cannot be transferred, or indeed used, again.</p>

  <p><a href=https://heycam.github.io/webidl/#dfn-platform-object id=transferable-objects:platform-object data-x-internal=platform-object>Platform objects</a> can be <a href=#transferable-objects id=transferable-objects:transferable-objects-4>transferable objects</a>
  if their <a id=transferable-objects:primary-interface href=https://heycam.github.io/webidl/#dfn-primary-interface data-x-internal=primary-interface>primary interface</a> is decorated with the <dfn id=transferable data-dfn-type=extended-attribute data-lt=Transferable><code>[Transferable]</code></dfn> IDL <a id=transferable-objects:extended-attribute href=https://heycam.github.io/webidl/#dfn-extended-attribute data-x-internal=extended-attribute>extended
  attribute</a>. Such interfaces must also define the following algorithms:</p>

  <dl><dt><dfn id=transfer-steps data-export="">transfer steps</dfn>, taking a <a id=transferable-objects:platform-object-2 href=https://heycam.github.io/webidl/#dfn-platform-object data-x-internal=platform-object>platform object</a>
   <var>value</var> and a <a id=transferable-objects:record href=https://tc39.es/ecma262/#sec-list-and-record-specification-type data-x-internal=record>Record</a> <var>dataHolder</var><dd>
    <p>A set of steps that transfers the data in <var>value</var> into fields of
    <var>dataHolder</var>. The resulting data held in <var>dataHolder</var> must be
    independent of any <a id=transferable-objects:javascript-realm href=https://tc39.es/ecma262/#sec-code-realms data-x-internal=javascript-realm>JavaScript Realm</a>.</p>

    <p>These steps may throw an exception if transferral is not possible.</p>
   <dt><dfn id=transfer-receiving-steps data-export="">transfer-receiving steps</dfn>, taking a <a id=transferable-objects:record-2 href=https://tc39.es/ecma262/#sec-list-and-record-specification-type data-x-internal=record>Record</a>
   <var>dataHolder</var> and a <a id=transferable-objects:platform-object-3 href=https://heycam.github.io/webidl/#dfn-platform-object data-x-internal=platform-object>platform object</a> <var>value</var><dd>
    <p>A set of steps that receives the data in <var>dataHolder</var>, using it to set up
    <var>value</var> as appropriate. <var>value</var> will be a newly-created instance of the
    <a id=transferable-objects:platform-object-4 href=https://heycam.github.io/webidl/#dfn-platform-object data-x-internal=platform-object>platform object</a> type in question, with none of its internal data set up; setting
    that up is the job of these steps.</p>

    <p>These steps may throw an exception if it is not possible to receive the transfer.</p>
   </dl>

  <p>It is up to the definition of individual platform objects to determine what data is transferred
  by these steps. Typically the steps are very symmetric.</p>

  <p>The <code id=transferable-objects:transferable><a href=#transferable>[Transferable]</a></code> extended attribute must take no
  arguments, and must only appear on an interface. It must not appear more than once on an
  interface.</p>

  <p>For a given <a id=transferable-objects:platform-object-5 href=https://heycam.github.io/webidl/#dfn-platform-object data-x-internal=platform-object>platform object</a>, only the object's <a id=transferable-objects:primary-interface-2 href=https://heycam.github.io/webidl/#dfn-primary-interface data-x-internal=primary-interface>primary interface</a> is
  considered during the transferring process. Thus, if inheritance is involved in defining the
  interface, each <code id=transferable-objects:transferable-2><a href=#transferable>[Transferable]</a></code>-annotated interface in the
  inheritance chain needs to define standalone <a href=#transfer-steps id=transferable-objects:transfer-steps>transfer steps</a> and
  <a href=#transfer-receiving-steps id=transferable-objects:transfer-receiving-steps>transfer-receiving steps</a>, including taking into account any important data that might
  come from inherited interfaces.</p>

  <p><a href=https://heycam.github.io/webidl/#dfn-platform-object id=transferable-objects:platform-object-6 data-x-internal=platform-object>Platform objects</a> that are <a href=#transferable-objects id=transferable-objects:transferable-objects-5>transferable
  objects</a> have a <dfn data-dfn-for="platform object" id=detached data-dfn-type=attribute>[[Detached]]</dfn> internal slot. This
  is used to ensure that once a platform object has been transferred, it cannot be transferred
  again.</p>

  <p>Objects defined in the JavaScript specification are handled by the
  <a href=#structuredserializewithtransfer id=transferable-objects:structuredserializewithtransfer>StructuredSerializeWithTransfer</a> abstract operation directly.</p>

  <h4 id=structuredserializeinternal data-dfn-type=abstract-op data-noexport="" data-lt=StructuredSerializeInternal><span class=secno>2.8.3</span> <dfn>StructuredSerializeInternal</dfn> ( <var>value</var>, <var>forStorage</var> [ ,
  <var>memory</var> ] )<a href=#structuredserializeinternal class=self-link></a></h4>

  <p>The <a href=#structuredserializeinternal id=structuredserializeinternal:structuredserializeinternal>StructuredSerializeInternal</a> abstract operation takes as input a JavaScript
  value <var>value</var> and serializes it to a <a href=https://tc39.es/ecma262/#sec-code-realms id=structuredserializeinternal:javascript-realm data-x-internal=javascript-realm>Realm</a>-independent form, represented here as a <a id=structuredserializeinternal:record href=https://tc39.es/ecma262/#sec-list-and-record-specification-type data-x-internal=record>Record</a>. This serialized
  form has all the information necessary to later deserialize into a new JavaScript value in a
  different Realm.</p>

  <p>This process can throw an exception, for example when trying to serialize un-serializable
  objects.</p>

  <ol><li>
    <p>If <var>memory</var> was not supplied, let <var>memory</var> be an empty <a href=https://infra.spec.whatwg.org/#ordered-map id=structuredserializeinternal:ordered-map data-x-internal=ordered-map>map</a>.</p>

    <p class=note>The purpose of the <var>memory</var> map is to avoid serializing objects twice.
    This ends up preserving cycles and the identity of duplicate objects in graphs.</p>
   <li><p>If <var>memory</var>[<var>value</var>] <a href=https://infra.spec.whatwg.org/#map-exists id=structuredserializeinternal:map-exists data-x-internal=map-exists>exists</a>, then
   return <var>memory</var>[<var>value</var>].<li><p>Let <var>deep</var> be false.<li><p>If <a href=https://tc39.es/ecma262/#sec-ecmascript-data-types-and-values id=structuredserializeinternal:js-type data-x-internal=js-type>Type</a>(<var>value</var>) is Undefined, Null, Boolean,
   Number, BigInt, or String, then return { [[Type]]: "primitive",
   [[Value]]: <var>value</var> }.<li><p>If <a href=https://tc39.es/ecma262/#sec-ecmascript-data-types-and-values id=structuredserializeinternal:js-type-2 data-x-internal=js-type>Type</a>(<var>value</var>) is Symbol, then throw a
   <a id=structuredserializeinternal:datacloneerror href=https://heycam.github.io/webidl/#datacloneerror data-x-internal=datacloneerror>"<code>DataCloneError</code>"</a> <code id=structuredserializeinternal:domexception><a data-x-internal=domexception href=https://heycam.github.io/webidl/#dfn-DOMException>DOMException</a></code>.<li><p>Let <var>serialized</var> be an uninitialized value.<li><p>If <var>value</var> has a [[BooleanData]] internal slot, then set <var>serialized</var> to
   { [[Type]]: "Boolean", [[BooleanData]]: <var>value</var>.[[BooleanData]] }.<li><p>Otherwise, if <var>value</var> has a [[NumberData]] internal slot, then set
   <var>serialized</var> to { [[Type]]: "Number", [[NumberData]]: <var>value</var>.[[NumberData]]
   }.<li><p>Otherwise, if <var>value</var> has a [[BigIntData]]
   internal slot, then set <var>serialized</var> to { [[Type]]: "BigInt", [[BigIntData]]:
   <var>value</var>.[[BigIntData]] }.<li><p>Otherwise, if <var>value</var> has a [[StringData]] internal slot, then set
   <var>serialized</var> to { [[Type]]: "String", [[StringData]]: <var>value</var>.[[StringData]]
   }.<li><p>Otherwise, if <var>value</var> has a [[DateValue]] internal slot, then set
   <var>serialized</var> to { [[Type]]: "Date", [[DateValue]]: <var>value</var>.[[DateValue]]
   }.<li><p>Otherwise, if <var>value</var> has a [[RegExpMatcher]] internal slot, then set
   <var>serialized</var> to { [[Type]]: "RegExp", [[RegExpMatcher]]:
   <var>value</var>.[[RegExpMatcher]], [[OriginalSource]]: <var>value</var>.[[OriginalSource]],
   [[OriginalFlags]]: <var>value</var>.[[OriginalFlags]] }.<li>
    <p>Otherwise, if <var>value</var> has an [[ArrayBufferData]] internal slot, then:</p>

    <ol><li><p>Let <var>size</var> be <var>value</var>.[[ArrayBufferByteLength]].<li>
      <p>If ! <a id=structuredserializeinternal:issharedarraybuffer href=https://tc39.es/ecma262/#sec-issharedarraybuffer data-x-internal=issharedarraybuffer>IsSharedArrayBuffer</a>(<var>value</var>) is true, then:

      <ol><li><p>Let <var>agentCluster</var> be the <a id=structuredserializeinternal:surrounding-agent href=https://tc39.es/ecma262/#surrounding-agent data-x-internal=surrounding-agent>surrounding agent</a>'s
       <a id=structuredserializeinternal:agent-cluster href=https://tc39.es/ecma262/#sec-agent-clusters data-x-internal=agent-cluster>agent cluster</a>.<li>
        <p>If <var>agentCluster</var>'s <a href=#cross-origin-isolated id=structuredserializeinternal:cross-origin-isolated>cross-origin isolated</a> is false, then throw a
        <a id=structuredserializeinternal:datacloneerror-2 href=https://heycam.github.io/webidl/#datacloneerror data-x-internal=datacloneerror>"<code>DataCloneError</code>"</a> <code id=structuredserializeinternal:domexception-2><a data-x-internal=domexception href=https://heycam.github.io/webidl/#dfn-DOMException>DOMException</a></code>.</p>

        <p class=note>This check is only needed when serializing (and not when deserializing) as
        <a href=#cross-origin-isolated id=structuredserializeinternal:cross-origin-isolated-2>cross-origin isolated</a> cannot change over time and a
        <code id=structuredserializeinternal:sharedarraybuffer><a data-x-internal=sharedarraybuffer href=https://tc39.es/ecma262/#sec-sharedarraybuffer-objects>SharedArrayBuffer</a></code> cannot leave an <a id=structuredserializeinternal:agent-cluster-2 href=https://tc39.es/ecma262/#sec-agent-clusters data-x-internal=agent-cluster>agent cluster</a>.</p>
       <li><p>If <var>forStorage</var> is true, then throw a
       <a id=structuredserializeinternal:datacloneerror-3 href=https://heycam.github.io/webidl/#datacloneerror data-x-internal=datacloneerror>"<code>DataCloneError</code>"</a> <code id=structuredserializeinternal:domexception-3><a data-x-internal=domexception href=https://heycam.github.io/webidl/#dfn-DOMException>DOMException</a></code>.<li><p>Set <var>serialized</var> to { [[Type]]: "SharedArrayBuffer", [[ArrayBufferData]]:
       <var>value</var>.[[ArrayBufferData]], [[ArrayBufferByteLength]]: <var>size</var>,
       [[AgentCluster]]: <var>agentCluster</var> }.</ol>
     <li>
      <p>Otherwise:</p>

      <ol><li><p>If ! <a id=structuredserializeinternal:isdetachedbuffer href=https://tc39.es/ecma262/#sec-isdetachedbuffer data-x-internal=isdetachedbuffer>IsDetachedBuffer</a>(<var>value</var>) is true, then throw a
       <a id=structuredserializeinternal:datacloneerror-4 href=https://heycam.github.io/webidl/#datacloneerror data-x-internal=datacloneerror>"<code>DataCloneError</code>"</a> <code id=structuredserializeinternal:domexception-4><a data-x-internal=domexception href=https://heycam.github.io/webidl/#dfn-DOMException>DOMException</a></code>.<li>
        <p>Let <var>dataCopy</var> be ? <a id=structuredserializeinternal:createbytedatablock href=https://tc39.es/ecma262/#sec-createbytedatablock data-x-internal=createbytedatablock>CreateByteDataBlock</a>(<var>size</var>).</p>

        <p class=note>This can throw a <code id=structuredserializeinternal:js-rangeerror><a data-x-internal=js-rangeerror href=https://tc39.es/ecma262/#sec-native-error-types-used-in-this-standard-rangeerror>RangeError</a></code> exception
        upon allocation failure.</p>
       <li><p>Perform ! <a id=structuredserializeinternal:copydatablockbytes href=https://tc39.es/ecma262/#sec-copydatablockbytes data-x-internal=copydatablockbytes>CopyDataBlockBytes</a>(<var>dataCopy</var>, 0,
       <var>value</var>.[[ArrayBufferData]], 0, <var>size</var>).<li><p>Set <var>serialized</var> to { [[Type]]: "ArrayBuffer", [[ArrayBufferData]]:
       <var>dataCopy</var>, [[ArrayBufferByteLength]]: <var>size</var> }.</ol>
     </ol>
   <li>
    <p>Otherwise, if <var>value</var> has a [[ViewedArrayBuffer]] internal slot, then:</p>

    <ol><li><p>Let <var>buffer</var> be the value of <var>value</var>'s [[ViewedArrayBuffer]] internal
     slot.<li><p>Let <var>bufferSerialized</var> be ?
     <a href=#structuredserializeinternal id=structuredserializeinternal:structuredserializeinternal-2>StructuredSerializeInternal</a>(<var>buffer</var>, <var>forStorage</var>,
     <var>memory</var>).<li><p>Assert: <var>bufferSerialized</var>.[[Type]] is "ArrayBuffer".<li><p>If <var>value</var> has a [[DataView]] internal slot, then set <var>serialized</var> to
     { [[Type]]: "ArrayBufferView", [[Constructor]]: "DataView", [[ArrayBufferSerialized]]:
     <var>bufferSerialized</var>, [[ByteLength]]: <var>value</var>.[[ByteLength]], [[ByteOffset]]:
     <var>value</var>.[[ByteOffset]] }.<li>
      <p>Otherwise:</p>

      <ol><li><p>Assert: <var>value</var> has a [[TypedArrayName]] internal slot.<li><p>Set <var>serialized</var> to { [[Type]]: "ArrayBufferView", [[Constructor]]:
       <var>value</var>.[[TypedArrayName]], [[ArrayBufferSerialized]]: <var>bufferSerialized</var>,
       [[ByteLength]]: <var>value</var>.[[ByteLength]], [[ByteOffset]]:
       <var>value</var>.[[ByteOffset]], [[ArrayLength]]: <var>value</var>.[[ArrayLength]] }.</ol>
     </ol>
   <li>
    <p>Otherwise, if <var>value</var> has [[MapData]] internal slot, then:</p>

    <ol><li><p>Set <var>serialized</var> to { [[Type]]: "Map", [[MapData]]: a new empty <a href=https://tc39.es/ecma262/#sec-list-and-record-specification-type id=structuredserializeinternal:js-list data-x-internal=js-list>List</a> }.<li><p>Set <var>deep</var> to true.</ol>
   <li>
    <p>Otherwise, if <var>value</var> has [[SetData]] internal slot, then:</p>

    <ol><li><p>Set <var>serialized</var> to { [[Type]]: "Set", [[SetData]]: a new empty <a href=https://tc39.es/ecma262/#sec-list-and-record-specification-type id=structuredserializeinternal:js-list-2 data-x-internal=js-list>List</a> }.<li><p>Set <var>deep</var> to true.</ol>
   <li>
    <p>Otherwise, if <var>value</var> has an [[ErrorData]] internal slot and <var>value</var> is not
    a <a id=structuredserializeinternal:platform-object href=https://heycam.github.io/webidl/#dfn-platform-object data-x-internal=platform-object>platform object</a>, then:</p>
    

    <ol><li><p>Let <var>name</var> be ? <a href=https://tc39.es/ecma262/#sec-get-o-p id=structuredserializeinternal:js-get data-x-internal=js-get>Get</a>(<var>value</var>,
     "name").<li><p>If <var>name</var> is not one of "Error", "EvalError", "RangeError", "ReferenceError",
     "SyntaxError", "TypeError", or "URIError", then set <var>name</var> to "Error".<li><p>Let <var>valueMessageDesc</var> be ? <var>value</var>.[[GetOwnProperty]]("<code>message</code>").<li><p>Let <var>message</var> be undefined if
     <a id=structuredserializeinternal:isdatadescriptor href=https://tc39.es/ecma262/#sec-isdatadescriptor data-x-internal=isdatadescriptor>IsDataDescriptor</a>(<var>valueMessageDesc</var>) is false, and
     ? <a id=structuredserializeinternal:tostring href=https://tc39.es/ecma262/#sec-tostring data-x-internal=tostring>ToString</a>(<var>valueMessageDesc</var>.[[Value]]) otherwise.<li><p>Set <var>serialized</var> to { [[Type]]: "Error", [[Name]]: <var>name</var>,
     [[Message]]: <var>message</var> }.<li>
      <p>User agents should attach a serialized representation of any interesting accompanying
      data which are not yet specified, notably the <code>stack</code> property, to
      <var>serialized</var>.</p>

      <p class=note>See the <cite>Error Stacks</cite> proposal for in-progress work on
      specifying this data. <a href=#refsJSERRORSTACKS>[JSERRORSTACKS]</a></p>
     </ol>
   <li>
    <p>Otherwise, if <var>value</var> is an Array exotic object, then:</p>
    

    <ol><li><p>Let <var>valueLenDescriptor</var> be ?
     <a id=structuredserializeinternal:ordinarygetownproperty href=https://tc39.es/ecma262/#sec-ordinarygetownproperty data-x-internal=ordinarygetownproperty>OrdinaryGetOwnProperty</a>(<var>value</var>, "<code>length</code>").<li><p>Let <var>valueLen</var> be <var>valueLenDescriptor</var>.[[Value]].<li><p>Set <var>serialized</var> to { [[Type]]: "Array", [[Length]]: <var>valueLen</var>,
     [[Properties]]: a new empty <a href=https://tc39.es/ecma262/#sec-list-and-record-specification-type id=structuredserializeinternal:js-list-3 data-x-internal=js-list>List</a> }.<li><p>Set <var>deep</var> to true.</ol>
   <li>
    <p>Otherwise, if <var>value</var> is a <a id=structuredserializeinternal:platform-object-2 href=https://heycam.github.io/webidl/#dfn-platform-object data-x-internal=platform-object>platform object</a> that is a <a href=#serializable-objects id=structuredserializeinternal:serializable-objects>serializable object</a>:</p>

    <ol><li><p>If <var>value</var> has a <a href=#detached id=structuredserializeinternal:detached>[[Detached]]</a> internal slot whose value is true,
     then throw a <a id=structuredserializeinternal:datacloneerror-5 href=https://heycam.github.io/webidl/#datacloneerror data-x-internal=datacloneerror>"<code>DataCloneError</code>"</a> <code id=structuredserializeinternal:domexception-5><a data-x-internal=domexception href=https://heycam.github.io/webidl/#dfn-DOMException>DOMException</a></code>.<li><p>Let <var>typeString</var> be the identifier of the <a id=structuredserializeinternal:primary-interface href=https://heycam.github.io/webidl/#dfn-primary-interface data-x-internal=primary-interface>primary interface</a> of
     <var>value</var>.<li><p>Set <var>serialized</var> to { [[Type]]: <var>typeString</var> }.<li><p>Set <var>deep</var> to true.</ol>
   <li><p>Otherwise, if <var>value</var> is a <a id=structuredserializeinternal:platform-object-3 href=https://heycam.github.io/webidl/#dfn-platform-object data-x-internal=platform-object>platform object</a>, then throw a
   <a id=structuredserializeinternal:datacloneerror-6 href=https://heycam.github.io/webidl/#datacloneerror data-x-internal=datacloneerror>"<code>DataCloneError</code>"</a> <code id=structuredserializeinternal:domexception-6><a data-x-internal=domexception href=https://heycam.github.io/webidl/#dfn-DOMException>DOMException</a></code>.<li><p>Otherwise, if <a id=structuredserializeinternal:iscallable href=https://tc39.es/ecma262/#sec-iscallable data-x-internal=iscallable>IsCallable</a>(<var>value</var>) is true, then throw a
   <a id=structuredserializeinternal:datacloneerror-7 href=https://heycam.github.io/webidl/#datacloneerror data-x-internal=datacloneerror>"<code>DataCloneError</code>"</a> <code id=structuredserializeinternal:domexception-7><a data-x-internal=domexception href=https://heycam.github.io/webidl/#dfn-DOMException>DOMException</a></code>.<li>
    <p>Otherwise, if <var>value</var> has any internal slot other than [[Prototype]] or
    [[Extensible]], then throw a <a id=structuredserializeinternal:datacloneerror-8 href=https://heycam.github.io/webidl/#datacloneerror data-x-internal=datacloneerror>"<code>DataCloneError</code>"</a>
    <code id=structuredserializeinternal:domexception-8><a data-x-internal=domexception href=https://heycam.github.io/webidl/#dfn-DOMException>DOMException</a></code>.</p>

    <p class=example>For instance, a [[PromiseState]] or [[WeakMapData]] internal slot.</p>
   <li>
    <p>Otherwise, if <var>value</var> is an exotic object and <var>value</var> is not the
    <a id=structuredserializeinternal:object.prototype href=https://tc39.es/ecma262/#sec-properties-of-the-object-prototype-object data-x-internal=object.prototype>%Object.prototype%</a> intrinsic object associated with any <a id=structuredserializeinternal:javascript-realm-2 href=https://tc39.es/ecma262/#sec-code-realms data-x-internal=javascript-realm>JavaScript
    realm</a>, then throw a <a id=structuredserializeinternal:datacloneerror-9 href=https://heycam.github.io/webidl/#datacloneerror data-x-internal=datacloneerror>"<code>DataCloneError</code>"</a>
    <code id=structuredserializeinternal:domexception-9><a data-x-internal=domexception href=https://heycam.github.io/webidl/#dfn-DOMException>DOMException</a></code>.</p>

    <p class=example>For instance, a proxy object.</p>
   <li>
    <p>Otherwise:</p>

    <ol><li><p>Set <var>serialized</var> to { [[Type]]: "Object", [[Properties]]: a new empty <a href=https://tc39.es/ecma262/#sec-list-and-record-specification-type id=structuredserializeinternal:js-list-4 data-x-internal=js-list>List</a> }.<li><p>Set <var>deep</var> to true.</ol>

    <p class=note><a id=structuredserializeinternal:object.prototype-2 href=https://tc39.es/ecma262/#sec-properties-of-the-object-prototype-object data-x-internal=object.prototype>%Object.prototype%</a> will end up being handled via this step and
    subsequent steps. The end result is that its exoticness is ignored, and after deserialization
    the result will be an empty object (not an <a id=structuredserializeinternal:immutable-prototype-exotic-object href=https://tc39.es/ecma262/#immutable-prototype-exotic-object data-x-internal=immutable-prototype-exotic-object>immutable prototype exotic object</a>).
   <li><p><a href=https://infra.spec.whatwg.org/#map-set id=structuredserializeinternal:map-set data-x-internal=map-set>Set</a> <var>memory</var>[<var>value</var>] to
   <var>serialized</var>.<li>
    <p>If <var>deep</var> is true, then:</p>

    <ol><li>
      <p>If <var>value</var> has a [[MapData]] internal slot, then:

      <ol><li><p>Let <var>copiedList</var> be a new empty <a href=https://tc39.es/ecma262/#sec-list-and-record-specification-type id=structuredserializeinternal:js-list-5 data-x-internal=js-list>List</a>.

       <li>
        <p><a href=https://infra.spec.whatwg.org/#list-iterate id=structuredserializeinternal:list-iterate data-x-internal=list-iterate>For each</a> <a id=structuredserializeinternal:record-2 href=https://tc39.es/ecma262/#sec-list-and-record-specification-type data-x-internal=record>Record</a> { [[Key]], [[Value]] }
        <var>entry</var> of <var>value</var>.[[MapData]]:</p>

        <ol><li><p>Let <var>copiedEntry</var> be a new <a id=structuredserializeinternal:record-3 href=https://tc39.es/ecma262/#sec-list-and-record-specification-type data-x-internal=record>Record</a> { [[Key]]:
         <var>entry</var>.[[Key]], [[Value]]: <var>entry</var>.[[Value]] }.<li><p>If <var>copiedEntry</var>.[[Key]] is not the special value <i>empty</i>, <a href=https://infra.spec.whatwg.org/#list-append id=structuredserializeinternal:list-append data-x-internal=list-append>append</a> <var>copiedEntry</var> to <var>copiedList</var>.</ol>
       <li>
        <p><a href=https://infra.spec.whatwg.org/#list-iterate id=structuredserializeinternal:list-iterate-2 data-x-internal=list-iterate>For each</a> <a id=structuredserializeinternal:record-4 href=https://tc39.es/ecma262/#sec-list-and-record-specification-type data-x-internal=record>Record</a> { [[Key]], [[Value]] }
        <var>entry</var> of <var>copiedList</var>:</p>

        <ol><li><p>Let <var>serializedKey</var> be ?
         <a href=#structuredserializeinternal id=structuredserializeinternal:structuredserializeinternal-3>StructuredSerializeInternal</a>(<var>entry</var>.[[Key]], <var>forStorage</var>,
         <var>memory</var>).<li><p>Let <var>serializedValue</var> be ?
         <a href=#structuredserializeinternal id=structuredserializeinternal:structuredserializeinternal-4>StructuredSerializeInternal</a>(<var>entry</var>.[[Value]], <var>forStorage</var>,
         <var>memory</var>).<li><p><a href=https://infra.spec.whatwg.org/#list-append id=structuredserializeinternal:list-append-2 data-x-internal=list-append>Append</a> { [[Key]]: <var>serializedKey</var>,
         [[Value]]: <var>serializedValue</var> } to <var>serialized</var>.[[MapData]].</ol>
       </ol>
     <li>
      <p>Otherwise, if <var>value</var> has a [[SetData]] internal slot, then:</p>

      <ol><li><p>Let <var>copiedList</var> be a new empty <a href=https://tc39.es/ecma262/#sec-list-and-record-specification-type id=structuredserializeinternal:js-list-6 data-x-internal=js-list>List</a>.

       <li>
        <p><a href=https://infra.spec.whatwg.org/#list-iterate id=structuredserializeinternal:list-iterate-3 data-x-internal=list-iterate>For each</a> <var>entry</var> of <var>value</var>.[[SetData]]:</p>

        <ol><li><p>If <var>entry</var> is not the special value <i>empty</i>, <a href=https://infra.spec.whatwg.org/#list-append id=structuredserializeinternal:list-append-3 data-x-internal=list-append>append</a> <var>entry</var> to <var>copiedList</var>.</ol>
       <li>
        <p><a href=https://infra.spec.whatwg.org/#list-iterate id=structuredserializeinternal:list-iterate-4 data-x-internal=list-iterate>For each</a> <var>entry</var> of <var>copiedList</var>:</p>

        <ol><li><p>Let <var>serializedEntry</var> be ?
         <a href=#structuredserializeinternal id=structuredserializeinternal:structuredserializeinternal-5>StructuredSerializeInternal</a>(<var>entry</var>, <var>forStorage</var>,
         <var>memory</var>).<li><p><a href=https://infra.spec.whatwg.org/#list-append id=structuredserializeinternal:list-append-4 data-x-internal=list-append>Append</a> <var>serializedEntry</var> to
         <var>serialized</var>.[[SetData]].</ol>
       </ol>
     <li>
      <p>Otherwise, if <var>value</var> is a <a id=structuredserializeinternal:platform-object-4 href=https://heycam.github.io/webidl/#dfn-platform-object data-x-internal=platform-object>platform object</a> that is a <a href=#serializable-objects id=structuredserializeinternal:serializable-objects-2>serializable object</a>, then perform the <a href=#serialization-steps id=structuredserializeinternal:serialization-steps>serialization
      steps</a> for <var>value</var>'s <a id=structuredserializeinternal:primary-interface-2 href=https://heycam.github.io/webidl/#dfn-primary-interface data-x-internal=primary-interface>primary interface</a>, given <var>value</var>,
      <var>serialized</var>, and <var>forStorage</var>.</p>

      <p>The <a href=#serialization-steps id=structuredserializeinternal:serialization-steps-2>serialization steps</a> may need to perform a <dfn id=sub-serialization data-export="">sub-serialization</dfn>. This is an operation which takes as input a value
      <var>subValue</var>, and returns <a href=#structuredserializeinternal id=structuredserializeinternal:structuredserializeinternal-6>StructuredSerializeInternal</a>(<var>subValue</var>,
      <var>forStorage</var>, <var>memory</var>). (In other words, a <a href=#sub-serialization id=structuredserializeinternal:sub-serialization>sub-serialization</a>
      is a specialization of <a href=#structuredserializeinternal id=structuredserializeinternal:structuredserializeinternal-7>StructuredSerializeInternal</a> to be consistent within this
      invocation.)</p>
     <li>
      <p>Otherwise, for each <var>key</var> in !
      <a id=structuredserializeinternal:enumerableownpropertynames href=https://tc39.es/ecma262/#sec-enumerableownpropertynames data-x-internal=enumerableownpropertynames>EnumerableOwnPropertyNames</a>(<var>value</var>, <var>key</var>):</p>

      <ol><li>
        <p>If ! <a id=structuredserializeinternal:hasownproperty href=https://tc39.es/ecma262/#sec-hasownproperty data-x-internal=hasownproperty>HasOwnProperty</a>(<var>value</var>, <var>key</var>) is true, then:</p>

        <ol><li><p>Let <var>inputValue</var> be ? <var>value</var>.[[Get]](<var>key</var>,
         <var>value</var>).<li><p>Let <var>outputValue</var> be ?
         <a href=#structuredserializeinternal id=structuredserializeinternal:structuredserializeinternal-8>StructuredSerializeInternal</a>(<var>inputValue</var>, <var>forStorage</var>,
         <var>memory</var>).<li><p><a href=https://infra.spec.whatwg.org/#list-append id=structuredserializeinternal:list-append-5 data-x-internal=list-append>Append</a> { [[Key]]: <var>key</var>, [[Value]]:
         <var>outputValue</var> } to <var>serialized</var>.[[Properties]].</ol>
       </ol>
     </ol>
   <li><p>Return <var>serialized</var>.</ol>

  <div class=example>
   <p>It's important to realize that the <a href=https://tc39.es/ecma262/#sec-list-and-record-specification-type id=structuredserializeinternal:record-5 data-x-internal=record>Records</a>
   produced by <a href=#structuredserializeinternal id=structuredserializeinternal:structuredserializeinternal-9>StructuredSerializeInternal</a> might contain "pointers" to other records
   that create circular references. For example, when we pass the following JavaScript object into
   <a href=#structuredserializeinternal id=structuredserializeinternal:structuredserializeinternal-10>StructuredSerializeInternal</a>:</p>

  <pre><code class='js'><c- kr>const</c-> o <c- o>=</c-> <c- p>{};</c->
o<c- p>.</c->myself <c- o>=</c-> o<c- p>;</c-></code></pre>

  <p>it produces the following result:</p>

  <pre><samp>{
  [[Type]]: "Object",
  [[Properties]]: «
    {
      [[Key]]: "myself",
      [[Value]]: <i>&lt;a pointer to this whole structure></i>
    }
  »
}</samp></pre>
  </div>


  <h4 id=structuredserialize data-dfn-type=abstract-op data-lt=StructuredSerialize><span class=secno>2.8.4</span> <dfn>StructuredSerialize</dfn> ( <var>value</var> )<a href=#structuredserialize class=self-link></a></h4>

  <ol><li><p>Return ? <a href=#structuredserializeinternal id=structuredserialize:structuredserializeinternal>StructuredSerializeInternal</a>(<var>value</var>, false).</ol>


  <h4 id=structuredserializeforstorage data-dfn-type=abstract-op data-lt=StructuredSerializeForStorage><span class=secno>2.8.5</span> <dfn>StructuredSerializeForStorage</dfn> ( <var>value</var> )<a href=#structuredserializeforstorage class=self-link></a></h4>

  <ol><li><p>Return ? <a href=#structuredserializeinternal id=structuredserializeforstorage:structuredserializeinternal>StructuredSerializeInternal</a>(<var>value</var>, true).</ol>


  <h4 id=structureddeserialize data-dfn-type=abstract-op data-lt=StructuredDeserialize><span class=secno>2.8.6</span> <dfn>StructuredDeserialize</dfn> ( <var>serialized</var>, <var>targetRealm</var> [ ,
  <var>memory</var> ] )<a href=#structureddeserialize class=self-link></a></h4>

  <p>The <a href=#structureddeserialize id=structureddeserialize:structureddeserialize>StructuredDeserialize</a> abstract operation takes as input a <a id=structureddeserialize:record href=https://tc39.es/ecma262/#sec-list-and-record-specification-type data-x-internal=record>Record</a>
  <var>serialized</var>, which was previously produced by <a href=#structuredserialize id=structureddeserialize:structuredserialize>StructuredSerialize</a> or
  <a href=#structuredserializeforstorage id=structureddeserialize:structuredserializeforstorage>StructuredSerializeForStorage</a>, and deserializes it into a new JavaScript value,
  created in <var>targetRealm</var>.</p>

  <p>This process can throw an exception, for example when trying to allocate memory for the new
  objects (especially <code>ArrayBuffer</code> objects).</p>

  <ol><li>
    <p>If <var>memory</var> was not supplied, let <var>memory</var> be an empty <a href=https://infra.spec.whatwg.org/#ordered-map id=structureddeserialize:ordered-map data-x-internal=ordered-map>map</a>.</p>

    <p class=note>The purpose of the <var>memory</var> map is to avoid deserializing objects
    twice. This ends up preserving cycles and the identity of duplicate objects in graphs.</p>
   <li><p>If <var>memory</var>[<var>serialized</var>] <a href=https://infra.spec.whatwg.org/#map-exists id=structureddeserialize:map-exists data-x-internal=map-exists>exists</a>, then
   return <var>memory</var>[<var>serialized</var>].<li><p>Let <var>deep</var> be false.<li><p>Let <var>value</var> be an uninitialized value.<li><p>If <var>serialized</var>.[[Type]] is "primitive", then set <var>value</var> to
   <var>serialized</var>.[[Value]].</p>

   <li><p>Otherwise, if <var>serialized</var>.[[Type]] is "Boolean", then set <var>value</var> to a
   new Boolean object in <var>targetRealm</var> whose [[BooleanData]] internal slot value is
   <var>serialized</var>.[[BooleanData]].<li><p>Otherwise, if <var>serialized</var>.[[Type]] is "Number", then set <var>value</var> to a
   new Number object in <var>targetRealm</var> whose [[NumberData]] internal slot value is
   <var>serialized</var>.[[NumberData]].<li><p>Otherwise, if <var>serialized</var>.[[Type]] is "BigInt", then set <var>value</var> to a
   new BigInt object in <var>targetRealm</var> whose [[BigIntData]] internal slot value is
   <var>serialized</var>.[[BigIntData]].<li><p>Otherwise, if <var>serialized</var>.[[Type]] is "String", then set <var>value</var> to a
   new String object in <var>targetRealm</var> whose [[StringData]] internal slot value is
   <var>serialized</var>.[[StringData]].<li><p>Otherwise, if <var>serialized</var>.[[Type]] is "Date", then set <var>value</var> to a new
   Date object in <var>targetRealm</var> whose [[DateValue]] internal slot value is
   <var>serialized</var>.[[DateValue]].<li><p>Otherwise, if <var>serialized</var>.[[Type]] is "RegExp", then set <var>value</var> to a
   new RegExp object in <var>targetRealm</var> whose [[RegExpMatcher]] internal slot value is
   <var>serialized</var>.[[RegExpMatcher]], whose [[OriginalSource]] internal slot value is
   <var>serialized</var>.[[OriginalSource]], and whose [[OriginalFlags]] internal slot value is
   <var>serialized</var>.[[OriginalFlags]].<li>
    <p>Otherwise, if <var>serialized</var>.[[Type]] is "SharedArrayBuffer", then:</p>

    <ol><li><p>If <var>targetRealm</var>'s corresponding <a id=structureddeserialize:agent-cluster href=https://tc39.es/ecma262/#sec-agent-clusters data-x-internal=agent-cluster>agent cluster</a> is not
     <var>serialized</var>.[[AgentCluster]], then then throw a
     <a id=structureddeserialize:datacloneerror href=https://heycam.github.io/webidl/#datacloneerror data-x-internal=datacloneerror>"<code>DataCloneError</code>"</a> <code id=structureddeserialize:domexception><a data-x-internal=domexception href=https://heycam.github.io/webidl/#dfn-DOMException>DOMException</a></code>.<li><p>Otherwise, set <var>value</var> to a new SharedArrayBuffer object in
     <var>targetRealm</var> whose [[ArrayBufferData]] internal slot value is
     <var>serialized</var>.[[ArrayBufferData]] and whose [[ArrayBufferByteLength]] internal slot
     value is <var>serialized</var>.[[ArrayBufferByteLength]].</ol>
   <li>
    <p>Otherwise, if <var>serialized</var>.[[Type]] is "ArrayBuffer", then set <var>value</var> to a
    new ArrayBuffer object in <var>targetRealm</var> whose [[ArrayBufferData]] internal slot value
    is <var>serialized</var>.[[ArrayBufferData]], and whose [[ArrayBufferByteLength]] internal slot
    value is <var>serialized</var>.[[ArrayBufferByteLength]].</p>

    <p>If this throws an exception, catch it, and then throw a
    <a id=structureddeserialize:datacloneerror-2 href=https://heycam.github.io/webidl/#datacloneerror data-x-internal=datacloneerror>"<code>DataCloneError</code>"</a> <code id=structureddeserialize:domexception-2><a data-x-internal=domexception href=https://heycam.github.io/webidl/#dfn-DOMException>DOMException</a></code>.</p>

    <p class=note>This step might throw an exception if there is not enough memory available to
    create such an ArrayBuffer object.</p>
   <li>
    <p>Otherwise, if <var>serialized</var>.[[Type]] is "ArrayBufferView", then:</p>

    <ol><li><p>Let <var>deserializedArrayBuffer</var> be ?
     <a href=#structureddeserialize id=structureddeserialize:structureddeserialize-2>StructuredDeserialize</a>(<var>serialized</var>.[[ArrayBufferSerialized]],
     <var>targetRealm</var>, <var>memory</var>).<li><p>If <var>serialized</var>.[[Constructor]] is "DataView", then set <var>value</var> to a
     new DataView object in <var>targetRealm</var> whose [[ViewedArrayBuffer]] internal slot value
     is <var>deserializedArrayBuffer</var>, whose [[ByteLength]] internal slot value is
     <var>serialized</var>.[[ByteLength]], and whose [[ByteOffset]] internal slot value is
     <var>serialized</var>.[[ByteOffset]].<li><p>Otherwise, set <var>value</var> to a new typed array object in <var>targetRealm</var>,
     using the constructor given by <var>serialized</var>.[[Constructor]], whose
     [[ViewedArrayBuffer]] internal slot value is <var>deserializedArrayBuffer</var>, whose
     [[TypedArrayName]] internal slot value is <var>serialized</var>.[[Constructor]], whose
     [[ByteLength]] internal slot value is <var>serialized</var>.[[ByteLength]], whose
     [[ByteOffset]] internal slot value is <var>serialized</var>.[[ByteOffset]], and whose
     [[ArrayLength]] internal slot value is <var>serialized</var>.[[ArrayLength]].</ol>
   <li>
    <p>Otherwise, if <var>serialized</var>.[[Type]] is "Map", then:</p>

    <ol><li><p>Set <var>value</var> to a new Map object in <var>targetRealm</var> whose [[MapData]]
     internal slot value is a new empty <a href=https://tc39.es/ecma262/#sec-list-and-record-specification-type id=structureddeserialize:js-list data-x-internal=js-list>List</a>.<li><p>Set <var>deep</var> to true.</ol>
   <li>
    <p>Otherwise, if <var>serialized</var>.[[Type]] is "Set", then:</p>

    <ol><li><p>Set <var>value</var> to a new Set object in <var>targetRealm</var> whose [[SetData]]
     internal slot value is a new empty <a href=https://tc39.es/ecma262/#sec-list-and-record-specification-type id=structureddeserialize:js-list-2 data-x-internal=js-list>List</a>.<li><p>Set <var>deep</var> to true.</ol>
   <li>
    <p>Otherwise, if <var>serialized</var>.[[Type]] is "Array", then:</p>

    <ol><li><p>Let <var>outputProto</var> be
     <var>targetRealm</var>.[[Intrinsics]].[[<a id=structureddeserialize:array.prototype href=https://tc39.es/ecma262/#sec-properties-of-the-array-prototype-object data-x-internal=array.prototype>%Array.prototype%</a>]].<li><p>Set <var>value</var> to ! <a id=structureddeserialize:arraycreate href=https://tc39.es/ecma262/#sec-arraycreate data-x-internal=arraycreate>ArrayCreate</a>(<var>serialized</var>.[[Length]],
     <var>outputProto</var>).<li><p>Set <var>deep</var> to true.</ol>
   <li>
    <p>Otherwise, if <var>serialized</var>.[[Type]] is "Object", then:</p>

    <ol><li><p>Set <var>value</var> to a new Object in <var>targetRealm</var>.<li><p>Set <var>deep</var> to true.</ol>
   <li>
    <p>Otherwise, if <var>serialized</var>.[[Type]] is "Error", then:</p>

    <ol><li><p>Let <var>prototype</var> be <a id=structureddeserialize:error.prototype href=https://tc39.es/ecma262/#sec-properties-of-the-error-prototype-object data-x-internal=error.prototype>%Error.prototype%</a>.<li><p>If <var>serialized</var>.[[Name]] is "EvalError", then set <var>prototype</var> to
     <a href=#evalerror.prototype id=structureddeserialize:evalerror.prototype>%EvalError.prototype%</a>.<li><p>If <var>serialized</var>.[[Name]] is "RangeError", then set <var>prototype</var>
     to <a href=#rangeerror.prototype id=structureddeserialize:rangeerror.prototype>%RangeError.prototype%</a>.<li><p>If <var>serialized</var>.[[Name]] is "ReferenceError", then set
     <var>prototype</var> to <a href=#referenceerror.prototype id=structureddeserialize:referenceerror.prototype>%ReferenceError.prototype%</a>.<li><p>If <var>serialized</var>.[[Name]] is "SyntaxError", then set <var>prototype</var>
     to <a href=#syntaxerror.prototype id=structureddeserialize:syntaxerror.prototype>%SyntaxError.prototype%</a>.<li><p>If <var>serialized</var>.[[Name]] is "TypeError", then set <var>prototype</var> to
     <a href=#typeerror.prototype id=structureddeserialize:typeerror.prototype>%TypeError.prototype%</a>.<li><p>If <var>serialized</var>.[[Name]] is "URIError", then set <var>prototype</var> to
     <a href=#urierror.prototype id=structureddeserialize:urierror.prototype>%URIError.prototype%</a>.<li><p>Let <var>message</var> be <var>serialized</var>.[[Message]].<li><p>Set <var>value</var> to ! <a id=structureddeserialize:objectcreate href=https://tc39.es/ecma262/#sec-objectcreate data-x-internal=objectcreate>ObjectCreate</a>(<var>prototype</var>, «
     [[ErrorData]] »).<li><p>Let <var>messageDesc</var> be <a id=structureddeserialize:propertydescriptor href=https://tc39.es/ecma262/#sec-property-descriptor-specification-type data-x-internal=propertydescriptor>PropertyDescriptor</a>{ [[Value]]:
     <var>message</var>, [[Writable]]: true, [[Enumerable]]: false, [[Configurable]]: true
     }.<li><p>If <var>message</var> is not undefined, then perform !
     <a id=structureddeserialize:ordinarydefineownproperty href=https://tc39.es/ecma262/#sec-ordinarydefineownproperty data-x-internal=ordinarydefineownproperty>OrdinaryDefineOwnProperty</a>(<var>value</var>, "<code>message</code>",
     <var>messageDesc</var>).<li><p>Any interesting accompanying data attached to <var>serialized</var> should be
     deserialized and attached to <var>value</var>.</ol>
   <li>
    <p>Otherwise:</p>

    <ol><li><p>Let <var>interfaceName</var> be <var>serialized</var>.[[Type]].<li><p>If the interface identified by <var>interfaceName</var> is not
     <a href=https://heycam.github.io/webidl/#dfn-exposed id=structureddeserialize:idl-exposed data-x-internal=idl-exposed>exposed</a> in <var>targetRealm</var>, then throw a
     <a id=structureddeserialize:datacloneerror-3 href=https://heycam.github.io/webidl/#datacloneerror data-x-internal=datacloneerror>"<code>DataCloneError</code>"</a> <code id=structureddeserialize:domexception-3><a data-x-internal=domexception href=https://heycam.github.io/webidl/#dfn-DOMException>DOMException</a></code>.<li><p>Set <var>value</var> to a new instance of the interface identified by
     <var>interfaceName</var>, created in <var>targetRealm</var>.</p>

     <li><p>Set <var>deep</var> to true.</ol>
   <li><p><a href=https://infra.spec.whatwg.org/#map-set id=structureddeserialize:map-set data-x-internal=map-set>Set</a> <var>memory</var>[<var>serialized</var>] to
   <var>value</var>.<li>
    <p>If <var>deep</var> is true, then:</p>

    <ol><li>
      <p>If <var>serialized</var>.[[Type]] is "Map", then:</p>

      <ol><li>
        <p><a href=https://infra.spec.whatwg.org/#list-iterate id=structureddeserialize:list-iterate data-x-internal=list-iterate>For each</a> <a id=structureddeserialize:record-2 href=https://tc39.es/ecma262/#sec-list-and-record-specification-type data-x-internal=record>Record</a> { [[Key]], [[Value]] }
        <var>entry</var> of <var>serialized</var>.[[MapData]]:</p>

        <ol><li><p>Let <var>deserializedKey</var> be ?
         <a href=#structureddeserialize id=structureddeserialize:structureddeserialize-3>StructuredDeserialize</a>(<var>entry</var>.[[Key]], <var>targetRealm</var>,
         <var>memory</var>).<li><p>Let <var>deserializedValue</var> be ?
         <a href=#structureddeserialize id=structureddeserialize:structureddeserialize-4>StructuredDeserialize</a>(<var>entry</var>.[[Value]], <var>targetRealm</var>,
         <var>memory</var>).<li><p><a href=https://infra.spec.whatwg.org/#list-append id=structureddeserialize:list-append data-x-internal=list-append>Append</a> { [[Key]]: <var>deserializedKey</var>,
         [[Value]]: <var>deserializedValue</var> } to <var>value</var>.[[MapData]].</ol>
       </ol>
     <li>
      <p>Otherwise, if <var>serialized</var>.[[Type]] is "Set", then:</p>

      <ol><li>
        <p><a href=https://infra.spec.whatwg.org/#list-iterate id=structureddeserialize:list-iterate-2 data-x-internal=list-iterate>For each</a> <var>entry</var> of
        <var>serialized</var>.[[SetData]]:</p>

        <ol><li><p>Let <var>deserializedEntry</var> be ?
         <a href=#structureddeserialize id=structureddeserialize:structureddeserialize-5>StructuredDeserialize</a>(<var>entry</var>, <var>targetRealm</var>,
         <var>memory</var>).<li><p><a href=https://infra.spec.whatwg.org/#list-append id=structureddeserialize:list-append-2 data-x-internal=list-append>Append</a> <var>deserializedEntry</var> to
         <var>value</var>.[[SetData]].</ol>
       </ol>
     <li>
      <p>Otherwise, if <var>serialized</var>.[[Type]] is "Array" or "Object", then:</p>

      <ol><li>
        <p><a href=https://infra.spec.whatwg.org/#list-iterate id=structureddeserialize:list-iterate-3 data-x-internal=list-iterate>For each</a> <a id=structureddeserialize:record-3 href=https://tc39.es/ecma262/#sec-list-and-record-specification-type data-x-internal=record>Record</a> { [[Key]], [[Value]] }
        <var>entry</var> of <var>serialized</var>.[[Properties]]:</p>

        <ol><li><p>Let <var>deserializedValue</var> be ?
         <a href=#structureddeserialize id=structureddeserialize:structureddeserialize-6>StructuredDeserialize</a>(<var>entry</var>.[[Value]], <var>targetRealm</var>,
         <var>memory</var>).<li><p>Let <var>result</var> be ! <a id=structureddeserialize:createdataproperty href=https://tc39.es/ecma262/#sec-createdataproperty data-x-internal=createdataproperty>CreateDataProperty</a>(<var>value</var>,
         <var>entry</var>.[[Key]], <var>deserializedValue</var>).<li><p>Assert: <var>result</var> is true.</ol>
       </ol>
     <li>
      <p>Otherwise:</p>

      <ol><li>
        <p>Perform the appropriate <a href=#deserialization-steps id=structureddeserialize:deserialization-steps>deserialization steps</a> for the interface identified
        by <var>serialized</var>.[[Type]], given <var>serialized</var> and <var>value</var>.</p>

        <p>The <a href=#deserialization-steps id=structureddeserialize:deserialization-steps-2>deserialization steps</a> may need to perform a <dfn id=sub-deserialization data-export="">sub-deserialization</dfn>. This is an operation which takes as input a
        previously-serialized <a id=structureddeserialize:record-4 href=https://tc39.es/ecma262/#sec-list-and-record-specification-type data-x-internal=record>Record</a> <var>subSerialized</var>, and returns
        <a href=#structureddeserialize id=structureddeserialize:structureddeserialize-7>StructuredDeserialize</a>(<var>subSerialized</var>, <var>targetRealm</var>,
        <var>memory</var>). (In other words, a <a href=#sub-deserialization id=structureddeserialize:sub-deserialization>sub-deserialization</a> is a specialization
        of <a href=#structureddeserialize id=structureddeserialize:structureddeserialize-8>StructuredDeserialize</a> to be consistent within this invocation.)</p>
       </ol>
    </ol>
   <li><p>Return <var>value</var>.</ol>

  <h4 id=structuredserializewithtransfer data-dfn-type=abstract-op data-lt=StructuredSerializeWithTransfer><span class=secno>2.8.7</span> <dfn>StructuredSerializeWithTransfer</dfn> ( <var>value</var>, <var>transferList</var>
  )<a href=#structuredserializewithtransfer class=self-link></a></h4>

  <ol><li>
    <p>Let <var>memory</var> be an empty <a href=https://infra.spec.whatwg.org/#ordered-map id=structuredserializewithtransfer:ordered-map data-x-internal=ordered-map>map</a>.</p>

    <p class=note>In addition to how it is used normally by
    <a href=#structuredserializeinternal id=structuredserializewithtransfer:structuredserializeinternal>StructuredSerializeInternal</a>, in this algorithm <var>memory</var> is also used to
    ensure that <a href=#structuredserializeinternal id=structuredserializewithtransfer:structuredserializeinternal-2>StructuredSerializeInternal</a> ignores items in <var>transferList</var>,
    and let us do our own handling instead.</p>
   <li>
    <p><a href=https://infra.spec.whatwg.org/#list-iterate id=structuredserializewithtransfer:list-iterate data-x-internal=list-iterate>For each</a> <var>transferable</var> of
    <var>transferList</var>:</p>

    <ol><li><p>If <var>transferable</var> has neither an [[ArrayBufferData]] internal slot nor a
     <a href=#detached id=structuredserializewithtransfer:detached>[[Detached]]</a> internal slot, then throw a
     <a id=structuredserializewithtransfer:datacloneerror href=https://heycam.github.io/webidl/#datacloneerror data-x-internal=datacloneerror>"<code>DataCloneError</code>"</a> <code id=structuredserializewithtransfer:domexception><a data-x-internal=domexception href=https://heycam.github.io/webidl/#dfn-DOMException>DOMException</a></code>.<li><p>If <var>transferable</var> has an [[ArrayBufferData]] internal slot and !
     <a id=structuredserializewithtransfer:issharedarraybuffer href=https://tc39.es/ecma262/#sec-issharedarraybuffer data-x-internal=issharedarraybuffer>IsSharedArrayBuffer</a>(<var>transferable</var>) is true, then throw a
     <a id=structuredserializewithtransfer:datacloneerror-2 href=https://heycam.github.io/webidl/#datacloneerror data-x-internal=datacloneerror>"<code>DataCloneError</code>"</a> <code id=structuredserializewithtransfer:domexception-2><a data-x-internal=domexception href=https://heycam.github.io/webidl/#dfn-DOMException>DOMException</a></code>.<li><p>If <var>memory</var>[<var>transferable</var>] <a href=https://infra.spec.whatwg.org/#map-exists id=structuredserializewithtransfer:map-exists data-x-internal=map-exists>exists</a>,
     then throw a <a id=structuredserializewithtransfer:datacloneerror-3 href=https://heycam.github.io/webidl/#datacloneerror data-x-internal=datacloneerror>"<code>DataCloneError</code>"</a> <code id=structuredserializewithtransfer:domexception-3><a data-x-internal=domexception href=https://heycam.github.io/webidl/#dfn-DOMException>DOMException</a></code>.<li>
      <p><a href=https://infra.spec.whatwg.org/#map-set id=structuredserializewithtransfer:map-set data-x-internal=map-set>Set</a> <var>memory</var>[<var>transferable</var>] to {
      [[Type]]: an uninitialized value }.</p>

      <p class=note><var>transferable</var> is not transferred yet as transferring has side
      effects and <a href=#structuredserializeinternal id=structuredserializewithtransfer:structuredserializeinternal-3>StructuredSerializeInternal</a> needs to be able to throw first.</p>
     </ol>
   <li><p>Let <var>serialized</var> be ? <a href=#structuredserializeinternal id=structuredserializewithtransfer:structuredserializeinternal-4>StructuredSerializeInternal</a>(<var>value</var>,
   false, <var>memory</var>).<li><p>Let <var>transferDataHolders</var> be a new empty <a href=https://tc39.es/ecma262/#sec-list-and-record-specification-type id=structuredserializewithtransfer:js-list data-x-internal=js-list>List</a>.<li>
    <p><a href=https://infra.spec.whatwg.org/#list-iterate id=structuredserializewithtransfer:list-iterate-2 data-x-internal=list-iterate>For each</a> <var>transferable</var> of
    <var>transferList</var>:</p>

    <ol><li><p>If <var>transferable</var> has an [[ArrayBufferData]] internal slot and !
     <a id=structuredserializewithtransfer:isdetachedbuffer href=https://tc39.es/ecma262/#sec-isdetachedbuffer data-x-internal=isdetachedbuffer>IsDetachedBuffer</a>(<var>transferable</var>) is true, then throw a
     <a id=structuredserializewithtransfer:datacloneerror-4 href=https://heycam.github.io/webidl/#datacloneerror data-x-internal=datacloneerror>"<code>DataCloneError</code>"</a> <code id=structuredserializewithtransfer:domexception-4><a data-x-internal=domexception href=https://heycam.github.io/webidl/#dfn-DOMException>DOMException</a></code>.<li><p>If <var>transferable</var> has a <a href=#detached id=structuredserializewithtransfer:detached-2>[[Detached]]</a> internal slot and
     <var>transferable</var>.<a href=#detached id=structuredserializewithtransfer:detached-3>[[Detached]]</a> is true, then throw a
     <a id=structuredserializewithtransfer:datacloneerror-5 href=https://heycam.github.io/webidl/#datacloneerror data-x-internal=datacloneerror>"<code>DataCloneError</code>"</a> <code id=structuredserializewithtransfer:domexception-5><a data-x-internal=domexception href=https://heycam.github.io/webidl/#dfn-DOMException>DOMException</a></code>.<li><p>Let <var>dataHolder</var> be <var>memory</var>[<var>transferable</var>].<li>
      <p>If <var>transferable</var> has an [[ArrayBufferData]] internal slot, then:</p>

      <ol><li><p>Set <var>dataHolder</var>.[[Type]] to "ArrayBuffer".<li><p>Set <var>dataHolder</var>.[[ArrayBufferData]] to
       <var>transferable</var>.[[ArrayBufferData]].<li><p>Set <var>dataHolder</var>.[[ArrayBufferByteLength]] to
       <var>transferable</var>.[[ArrayBufferByteLength]].<li>
        <p>Perform ? <a id=structuredserializewithtransfer:detacharraybuffer href=https://tc39.es/ecma262/#sec-detacharraybuffer data-x-internal=detacharraybuffer>DetachArrayBuffer</a>(<var>transferable</var>).</p>

        <p class=note>Specifications can use the [[ArrayBufferDetachKey]] internal slot to prevent
        <code id=structuredserializewithtransfer:idl-arraybuffer><a data-x-internal=idl-arraybuffer href=https://heycam.github.io/webidl/#idl-ArrayBuffer>ArrayBuffer</a></code>s from being detached. This is used in
        <cite>WebAssembly JavaScript Interface</cite>, for example. <a href=#refsWASMJS>[WASMJS]</a></p>
       </ol>
     <li>
      <p>Otherwise:</p>

      <ol><li><p>Assert: <var>transferable</var> is a <a id=structuredserializewithtransfer:platform-object href=https://heycam.github.io/webidl/#dfn-platform-object data-x-internal=platform-object>platform object</a> that is a <a href=#transferable-objects id=structuredserializewithtransfer:transferable-objects>transferable object</a>.<li><p>Let <var>interfaceName</var> be the identifier of the <a id=structuredserializewithtransfer:primary-interface href=https://heycam.github.io/webidl/#dfn-primary-interface data-x-internal=primary-interface>primary interface</a>
       of <var>transferable</var>.<li><p>Set <var>dataHolder</var>.[[Type]] to <var>interfaceName</var>.<li><p>Perform the appropriate <a href=#transfer-steps id=structuredserializewithtransfer:transfer-steps>transfer steps</a> for the interface identified by
       <var>interfaceName</var>, given <var>transferable</var> and <var>dataHolder</var>.<li><p>Set <var>transferable</var>.<a href=#detached id=structuredserializewithtransfer:detached-4>[[Detached]]</a> to true.</ol>
     <li><p><a href=https://infra.spec.whatwg.org/#list-append id=structuredserializewithtransfer:list-append data-x-internal=list-append>Append</a> <var>dataHolder</var> to
     <var>transferDataHolders</var>.</ol>
   <li><p>Return { [[Serialized]]: <var>serialized</var>, [[TransferDataHolders]]:
   <var>transferDataHolders</var> }.</ol>

  <h4 id=structureddeserializewithtransfer data-dfn-type=abstract-op data-lt=StructuredDeserializeWithTransfer><span class=secno>2.8.8</span> <dfn>StructuredDeserializeWithTransfer</dfn> ( <var>serializeWithTransferResult</var>,
  <var>targetRealm</var> )<a href=#structureddeserializewithtransfer class=self-link></a></h4>

  <ol><li>
    <p>Let <var>memory</var> be an empty <a href=https://infra.spec.whatwg.org/#ordered-map id=structureddeserializewithtransfer:ordered-map data-x-internal=ordered-map>map</a>.</p>

    <p class=note>Analogous to <a href=#structuredserializewithtransfer id=structureddeserializewithtransfer:structuredserializewithtransfer>StructuredSerializeWithTransfer</a>, in addition to how it
    is used normally by <a href=#structureddeserialize id=structureddeserializewithtransfer:structureddeserialize>StructuredDeserialize</a>, in this algorithm <var>memory</var> is
    also used to ensure that <a href=#structureddeserialize id=structureddeserializewithtransfer:structureddeserialize-2>StructuredDeserialize</a> ignores items in
    <var>serializeWithTransferResult</var>.[[TransferDataHolders]], and let us do our own handling
    instead.</p>
   <li><p>Let <var>transferredValues</var> be a new empty <a href=https://tc39.es/ecma262/#sec-list-and-record-specification-type id=structureddeserializewithtransfer:js-list data-x-internal=js-list>List</a>.<li>
    <p><a href=https://infra.spec.whatwg.org/#list-iterate id=structureddeserializewithtransfer:list-iterate data-x-internal=list-iterate>For each</a> <var>transferDataHolder</var> of
    <var>serializeWithTransferResult</var>.[[TransferDataHolders]]:</p>

    <ol><li><p>Let <var>value</var> be an uninitialized value.<li>
      <p>If <var>transferDataHolder</var>.[[Type]] is "ArrayBuffer", then set <var>value</var> to a
      new ArrayBuffer object in <var>targetRealm</var> whose [[ArrayBufferData]] internal slot value
      is <var>transferDataHolder</var>.[[ArrayBufferData]], and whose [[ArrayBufferByteLength]]
      internal slot value is <var>transferDataHolder</var>.[[ArrayBufferByteLength]].</p>

      <p class=note>In cases where the original memory occupied by [[ArrayBufferData]] is
      accessible during the deserialization, this step is unlikely to throw an exception, as no new
      memory needs to be allocated: the memory occupied by [[ArrayBufferData]] is instead just
      getting transferred into the new ArrayBuffer. This could be true, for example, when both the
      source and target Realms are in the same process.</p>
     <li>
      <p>Otherwise:</p>

      <ol><li><p>Let <var>interfaceName</var> be <var>transferDataHolder</var>.[[Type]].<li><p>If the interface identified by <var>interfaceName</var> is not exposed in
       <var>targetRealm</var>, then throw a <a id=structureddeserializewithtransfer:datacloneerror href=https://heycam.github.io/webidl/#datacloneerror data-x-internal=datacloneerror>"<code>DataCloneError</code>"</a>
       <code id=structureddeserializewithtransfer:domexception><a data-x-internal=domexception href=https://heycam.github.io/webidl/#dfn-DOMException>DOMException</a></code>.<li><p>Set <var>value</var> to a new instance of the interface identified by
       <var>interfaceName</var>, created in <var>targetRealm</var>.<li><p>Perform the appropriate <a href=#transfer-receiving-steps id=structureddeserializewithtransfer:transfer-receiving-steps>transfer-receiving steps</a> for the interface
       identified by <var>interfaceName</var> given <var>transferDataHolder</var> and
       <var>value</var>.</ol>
     <li><p><a href=https://infra.spec.whatwg.org/#map-set id=structureddeserializewithtransfer:map-set data-x-internal=map-set>Set</a> <var>memory</var>[<var>transferDataHolder</var>] to
     <var>value</var>.<li><p><a href=https://infra.spec.whatwg.org/#list-append id=structureddeserializewithtransfer:list-append data-x-internal=list-append>Append</a> <var>value</var> to
     <var>transferredValues</var>.</ol>
   <li><p>Let <var>deserialized</var> be ?
   <a href=#structureddeserialize id=structureddeserializewithtransfer:structureddeserialize-3>StructuredDeserialize</a>(<var>serializeWithTransferResult</var>.[[Serialized]],
   <var>targetRealm</var>, <var>memory</var>).<li><p>Return { [[Deserialized]]: <var>deserialized</var>, [[TransferredValues]]:
   <var>transferredValues</var> }.</ol>


  <h4 id=performing-structured-clones-from-other-specifications><span class=secno>2.8.9</span> Performing serialization and
  transferring from other specifications<a href=#performing-structured-clones-from-other-specifications class=self-link></a></h4>

  <p>Other specifications may use the abstract operations defined here. The following provides
  some guidance on when each abstract operation is typically useful, with examples.</p>

  <dl><dt><a href=#structuredserializewithtransfer id=performing-structured-clones-from-other-specifications:structuredserializewithtransfer>StructuredSerializeWithTransfer</a><dt><a href=#structureddeserializewithtransfer id=performing-structured-clones-from-other-specifications:structureddeserializewithtransfer>StructuredDeserializeWithTransfer</a><dd>
    <p>Cloning a value to another <a id=performing-structured-clones-from-other-specifications:javascript-realm href=https://tc39.es/ecma262/#sec-code-realms data-x-internal=javascript-realm>JavaScript Realm</a>, with a transfer list, but where the
    target Realm is not known ahead of time. In this case the serialization step can be performed
    immediately, with the deserialization step delayed until the target Realm becomes known.</p>

    <p class=example><code id=performing-structured-clones-from-other-specifications:dom-messageport-postmessage><a href=#dom-messageport-postmessage>messagePort.postMessage()</a></code>
    uses this pair of abstract operations, as the destination Realm is not known until the
    <code id=performing-structured-clones-from-other-specifications:messageport><a href=#messageport>MessagePort</a></code> <a href=#has-been-shipped id=performing-structured-clones-from-other-specifications:has-been-shipped>has been shipped</a>.</p>
   <dt><a href=#structuredserialize id=performing-structured-clones-from-other-specifications:structuredserialize>StructuredSerialize</a><dt><a href=#structuredserializeforstorage id=performing-structured-clones-from-other-specifications:structuredserializeforstorage>StructuredSerializeForStorage</a><dt><a href=#structureddeserialize id=performing-structured-clones-from-other-specifications:structureddeserialize>StructuredDeserialize</a><dd>
    <p>Creating a <a id=performing-structured-clones-from-other-specifications:javascript-realm-2 href=https://tc39.es/ecma262/#sec-code-realms data-x-internal=javascript-realm>JavaScript Realm</a>-independent snapshot of a given value which can be
    saved for an indefinite amount of time, and then reified back into a JavaScript value later,
    possibly multiple times.</p>

    <p><a href=#structuredserializeforstorage id=performing-structured-clones-from-other-specifications:structuredserializeforstorage-2>StructuredSerializeForStorage</a> can be used for situations where the serialization
    is anticipated to be stored in a persistent manner, instead of passed between Realms. It throws
    when attempting to serialize <code id=performing-structured-clones-from-other-specifications:sharedarraybuffer><a data-x-internal=sharedarraybuffer href=https://tc39.es/ecma262/#sec-sharedarraybuffer-objects>SharedArrayBuffer</a></code> objects, since storing shared memory
    does not make sense. Similarly, it can throw or possibly have different behavior when given a
    <a id=performing-structured-clones-from-other-specifications:platform-object href=https://heycam.github.io/webidl/#dfn-platform-object data-x-internal=platform-object>platform object</a> with custom <a href=#serialization-steps id=performing-structured-clones-from-other-specifications:serialization-steps>serialization steps</a> when the
    <var>forStorage</var> argument is true.</p>

    <p class=example><code id=performing-structured-clones-from-other-specifications:dom-history-pushstate><a href=#dom-history-pushstate>history.pushState()</a></code> and <code id=performing-structured-clones-from-other-specifications:dom-history-replacestate><a href=#dom-history-replacestate>history.replaceState()</a></code> use
    <a href=#structuredserializeforstorage id=performing-structured-clones-from-other-specifications:structuredserializeforstorage-3>StructuredSerializeForStorage</a> on author-supplied state objects, storing them as
    <a href=#serialized-state id=performing-structured-clones-from-other-specifications:serialized-state>serialized state</a> in the appropriate <a href=#session-history-entry id=performing-structured-clones-from-other-specifications:session-history-entry>session history entry</a>. Then,
    <a href=#structureddeserialize id=performing-structured-clones-from-other-specifications:structureddeserialize-2>StructuredDeserialize</a> is used so that the <code id=performing-structured-clones-from-other-specifications:dom-history-state><a href=#dom-history-state>history.state</a></code> property can return a clone of the
    originally-supplied state object.</p>

    <p class=example><code id=performing-structured-clones-from-other-specifications:dom-broadcastchannel-postmessage><a href=#dom-broadcastchannel-postmessage>broadcastChannel.postMessage()</a></code> uses
    <a href=#structuredserialize id=performing-structured-clones-from-other-specifications:structuredserialize-2>StructuredSerialize</a> on its input, then uses <a href=#structureddeserialize id=performing-structured-clones-from-other-specifications:structureddeserialize-3>StructuredDeserialize</a>
    multiple times on the result to produce a fresh clone for each destination being broadcast
    to. Note that transferring does not make sense in multi-destination situations.</p>

    <p class=example>Any API for persisting JavaScript values to the filesystem would also use
    <a href=#structuredserializeforstorage id=performing-structured-clones-from-other-specifications:structuredserializeforstorage-4>StructuredSerializeForStorage</a> on its input and <a href=#structureddeserialize id=performing-structured-clones-from-other-specifications:structureddeserialize-4>StructuredDeserialize</a>
    on its output.</p>
   </dl>

  <p>In general, call sites may pass in Web IDL values instead of JavaScript values; this is to be
  understood to perform an implicit <a href=https://heycam.github.io/webidl/#es-type-mapping id=performing-structured-clones-from-other-specifications:concept-idl-convert data-x-internal=concept-idl-convert>conversion</a> to the
  JavaScript value before invoking these algorithms.</p>

  <p id=structuredclone class=note><span id=structured-clone></span>This specification used
  to define a "structured clone" algorithm, and more recently a StructuredClone abstract operation.
  However, in practice all known uses of it were better served by separate serialization and
  deserialization steps, so it was removed.</p>

  <hr>

  <p>Call sites that are not invoked as a result of author code synchronously calling into a user
  agent method must take care to properly <a href=#prepare-to-run-script id=performing-structured-clones-from-other-specifications:prepare-to-run-script>prepare to run script</a> and <a href=#prepare-to-run-a-callback id=performing-structured-clones-from-other-specifications:prepare-to-run-a-callback>prepare to
  run a callback</a> before invoking <a href=#structuredserialize id=performing-structured-clones-from-other-specifications:structuredserialize-3>StructuredSerialize</a>,
  <a href=#structuredserializeforstorage id=performing-structured-clones-from-other-specifications:structuredserializeforstorage-5>StructuredSerializeForStorage</a>, or <a href=#structuredserializewithtransfer id=performing-structured-clones-from-other-specifications:structuredserializewithtransfer-2>StructuredSerializeWithTransfer</a>
  abstract operations, if they are being performed on arbitrary objects. This is necessary because
  the serialization process can invoke author-defined accessors as part of its final
  deep-serialization steps, and these accessors could call into operations that rely on the <a href=#concept-entry-everything id=performing-structured-clones-from-other-specifications:concept-entry-everything>entry</a> and <a href=#concept-incumbent-everything id=performing-structured-clones-from-other-specifications:concept-incumbent-everything>incumbent</a> concepts being properly set up.</p>

  <p class=example><code id=performing-structured-clones-from-other-specifications:dom-window-postmessage><a href=#dom-window-postmessage>window.postMessage()</a></code> performs
  <a href=#structuredserializewithtransfer id=performing-structured-clones-from-other-specifications:structuredserializewithtransfer-3>StructuredSerializeWithTransfer</a> on its arguments, but is careful to do so
  immediately, inside the synchronous portion of its algorithm. Thus it is able to use the
  algorithms without needing to <a href=#prepare-to-run-script id=performing-structured-clones-from-other-specifications:prepare-to-run-script-2>prepare to run script</a> and <a href=#prepare-to-run-a-callback id=performing-structured-clones-from-other-specifications:prepare-to-run-a-callback-2>prepare to run a
  callback</a>.</p>

  <p class=example>In contrast, a hypothetical API that used <a href=#structuredserialize id=performing-structured-clones-from-other-specifications:structuredserialize-4>StructuredSerialize</a> to
  serialize some author-supplied object periodically, directly from a <a href=#concept-task id=performing-structured-clones-from-other-specifications:concept-task>task</a> on the <a href=#event-loop id=performing-structured-clones-from-other-specifications:event-loop>event loop</a>, would need to ensure it performs
  the appropriate preparations beforehand. As of this time, we know of no such APIs on the platform;
  usually it is simpler to perform the serialization ahead of time, as a synchronous consequence of
  author code.</p>

  


  <h2 id=dom><span class=secno>3</span> Semantics, structure, and APIs of HTML documents<a href=#dom class=self-link></a></h2>

  <h3 id=documents><span class=secno>3.1</span> Documents<a href=#documents class=self-link></a></h3>

  <p>Every XML and HTML document in an HTML UA is represented by a <code id=documents:document><a href=#document>Document</a></code> object. <a href=#refsDOM>[DOM]</a></p>

  <p>The <code id=documents:document-2><a href=#document>Document</a></code> object's <dfn data-x-for=Document id="the-document's-address"><a href=https://dom.spec.whatwg.org/#concept-document-url>URL</a></dfn> is defined in <cite>DOM</cite>. It is initially set when
  the <code id=documents:document-3><a href=#document>Document</a></code> object is created, but can change during the lifetime of the
  <code id=documents:document-4><a href=#document>Document</a></code> object; for example, it changes when the user <a href=#navigate id=documents:navigate>navigates</a> to a <a href=#navigate-fragid id=documents:navigate-fragid>fragment</a> on the page
  and when the <code id=documents:dom-history-pushstate><a href=#dom-history-pushstate>pushState()</a></code> method is called with a new
  <a id=documents:url href=https://url.spec.whatwg.org/#concept-url data-x-internal=url>URL</a>. <a href=#refsDOM>[DOM]</a></p>
  

  <p class=warning>Interactive user agents typically expose the <code id=documents:document-5><a href=#document>Document</a></code> object's
  <a href=https://dom.spec.whatwg.org/#concept-document-url id="documents:the-document's-address" data-x-internal="the-document's-address">URL</a> in their user interface. This is the primary
  mechanism by which a user can tell if a site is attempting to impersonate another.</p>

  <p>When a <code id=documents:document-6><a href=#document>Document</a></code> is created by a <a href=#concept-script id=documents:concept-script>script</a> using
  the <code id=documents:dom-domimplementation-createdocument><a data-x-internal=dom-domimplementation-createdocument href=https://dom.spec.whatwg.org/#dom-domimplementation-createdocument>createDocument()</a></code> or <code id=documents:dom-domimplementation-createhtmldocument><a data-x-internal=dom-domimplementation-createhtmldocument href=https://dom.spec.whatwg.org/#dom-domimplementation-createhtmldocument>createHTMLDocument()</a></code> the
  <code id=documents:document-7><a href=#document>Document</a></code> is <a href=#ready-for-post-load-tasks id=documents:ready-for-post-load-tasks>ready for post-load tasks</a> immediately.</p>

  <p><dfn id="the-document's-referrer" data-export="">The document's referrer</dfn> is a string (representing a <a id=documents:url-2 href=https://url.spec.whatwg.org/#concept-url data-x-internal=url>URL</a>) that
  can be set when the <code id=documents:document-8><a href=#document>Document</a></code> is created. If it is not explicitly set, then its value
  is the empty string.</p>


  <h4 id=the-document-object><span class=secno>3.1.1</span> The <code id=the-document-object:document><a href=#document>Document</a></code> object<a href=#the-document-object class=self-link></a></h4><div class="mdn-anno wrapped"><button onclick=toggleStatus(this) class=mdn-anno-btn><b title="Support in all current engines." class=all-engines-flag>✔</b><span>MDN</span></button><div class=feature><p><a href=https://developer.mozilla.org/en-US/docs/Web/API/Document title="The Document interface represents any web page loaded in the browser and serves as an entry point into the web page's content, which is the DOM tree.">Document</a><p class=all-engines-text>Support in all current engines.<div class=support><span class="firefox yes"><span>Firefox</span><span>1+</span></span><span class="safari yes"><span>Safari</span><span>1+</span></span><span class="chrome yes"><span>Chrome</span><span>1+</span></span><hr><span class="opera yes"><span>Opera</span><span>3+</span></span><span class="edge_blink yes"><span>Edge</span><span>79+</span></span><hr><span class="edge yes"><span>Edge (Legacy)</span><span>12+</span></span><span class="ie yes"><span>Internet Explorer</span><span>4+</span></span><hr><span class="firefox_android yes"><span>Firefox Android</span><span>4+</span></span><span class="safari_ios yes"><span>Safari iOS</span><span>1+</span></span><span class="chrome_android yes"><span>Chrome Android</span><span>18+</span></span><span class="webview_android yes"><span>WebView Android</span><span>1+</span></span><span class="samsunginternet_android yes"><span>Samsung Internet</span><span>1.0+</span></span><span class="opera_android yes"><span>Opera Android</span><span>10.1+</span></span></div></div></div>

  <p><cite>DOM</cite> defines a <code id=the-document-object:dom-document><a data-x-internal=dom-document href=https://dom.spec.whatwg.org/#interface-document>Document</a></code> interface, which
  this specification extends significantly.</p>

  <pre><code class='idl'><c- b>enum</c-> <dfn id='documentreadystate'><c- g>DocumentReadyState</c-></dfn> { <c- s>&quot;loading&quot;</c->, <c- s>&quot;interactive&quot;</c->, <c- s>&quot;complete&quot;</c-> };
<c- b>typedef</c-> (<a href='#htmlscriptelement' id='the-document-object:htmlscriptelement'><c- n>HTMLScriptElement</c-></a> <c- b>or</c-> <a id='the-document-object:svgscriptelement' href='https://svgwg.org/svg2-draft/interact.html#InterfaceSVGScriptElement' data-x-internal='svgscriptelement'><c- n>SVGScriptElement</c-></a>) <dfn id='htmlorsvgscriptelement'><c- g>HTMLOrSVGScriptElement</c-></dfn>;

[<a id='the-document-object:legacyoverridebuiltins' href='https://heycam.github.io/webidl/#LegacyOverrideBuiltIns' data-x-internal='legacyoverridebuiltins'><c- g>LegacyOverrideBuiltIns</c-></a>]
<c- b>partial</c-> <c- b>interface</c-> <dfn data-lt id='document'><c- g>Document</c-></dfn> {
  // <a href='#resource-metadata-management' id='the-document-object:resource-metadata-management'>resource metadata management</a>
  [<c- g>PutForwards</c->=<a href='#dom-location-href' id='the-document-object:dom-location-href'><c- n>href</c-></a>, <a id='the-document-object:legacyunforgeable' href='https://heycam.github.io/webidl/#LegacyUnforgeable' data-x-internal='legacyunforgeable'><c- g>LegacyUnforgeable</c-></a>] <c- b>readonly</c-> <c- b>attribute</c-> <a href='#location' id='the-document-object:location'><c- n>Location</c-></a>? <a href='#dom-document-location' id='the-document-object:dom-document-location'><c- g>location</c-></a>;
  <c- b>attribute</c-> <c- b>USVString</c-> <a href='#dom-document-domain' id='the-document-object:dom-document-domain'><c- g>domain</c-></a>;
  <c- b>readonly</c-> <c- b>attribute</c-> <c- b>USVString</c-> <a href='#dom-document-referrer' id='the-document-object:dom-document-referrer'><c- g>referrer</c-></a>;
  <c- b>attribute</c-> <c- b>USVString</c-> <a href='#dom-document-cookie' id='the-document-object:dom-document-cookie'><c- g>cookie</c-></a>;
  <c- b>readonly</c-> <c- b>attribute</c-> <c- b>DOMString</c-> <a href='#dom-document-lastmodified' id='the-document-object:dom-document-lastmodified'><c- g>lastModified</c-></a>;
  <c- b>readonly</c-> <c- b>attribute</c-> <a href='#documentreadystate' id='the-document-object:documentreadystate'><c- n>DocumentReadyState</c-></a> <a href='#dom-document-readystate' id='the-document-object:dom-document-readystate'><c- g>readyState</c-></a>;

  // <a href='#dom-tree-accessors' id='the-document-object:dom-tree-accessors'>DOM tree accessors</a>
  <a href='#dom-document-nameditem'><c- b>getter</c-></a> <c- b>object</c-> (<c- b>DOMString</c-> <c- g>name</c->);
  [<a href='#cereactions' id='the-document-object:cereactions'><c- g>CEReactions</c-></a>] <c- b>attribute</c-> <c- b>DOMString</c-> <a href='#document.title' id='the-document-object:document.title'><c- g>title</c-></a>;
  [<a href='#cereactions' id='the-document-object:cereactions-2'><c- g>CEReactions</c-></a>] <c- b>attribute</c-> <c- b>DOMString</c-> <a href='#dom-document-dir' id='the-document-object:dom-document-dir'><c- g>dir</c-></a>;
  [<a href='#cereactions' id='the-document-object:cereactions-3'><c- g>CEReactions</c-></a>] <c- b>attribute</c-> <a href='#htmlelement' id='the-document-object:htmlelement'><c- n>HTMLElement</c-></a>? <a href='#dom-document-body' id='the-document-object:dom-document-body'><c- g>body</c-></a>;
  <c- b>readonly</c-> <c- b>attribute</c-> <a href='#htmlheadelement' id='the-document-object:htmlheadelement'><c- n>HTMLHeadElement</c-></a>? <a href='#dom-document-head' id='the-document-object:dom-document-head'><c- g>head</c-></a>;
  [<c- g>SameObject</c->] <c- b>readonly</c-> <c- b>attribute</c-> <a id='the-document-object:htmlcollection' href='https://dom.spec.whatwg.org/#interface-htmlcollection' data-x-internal='htmlcollection'><c- n>HTMLCollection</c-></a> <a href='#dom-document-images' id='the-document-object:dom-document-images'><c- g>images</c-></a>;
  [<c- g>SameObject</c->] <c- b>readonly</c-> <c- b>attribute</c-> <a id='the-document-object:htmlcollection-2' href='https://dom.spec.whatwg.org/#interface-htmlcollection' data-x-internal='htmlcollection'><c- n>HTMLCollection</c-></a> <a href='#dom-document-embeds' id='the-document-object:dom-document-embeds'><c- g>embeds</c-></a>;
  [<c- g>SameObject</c->] <c- b>readonly</c-> <c- b>attribute</c-> <a id='the-document-object:htmlcollection-3' href='https://dom.spec.whatwg.org/#interface-htmlcollection' data-x-internal='htmlcollection'><c- n>HTMLCollection</c-></a> <a href='#dom-document-plugins' id='the-document-object:dom-document-plugins'><c- g>plugins</c-></a>;
  [<c- g>SameObject</c->] <c- b>readonly</c-> <c- b>attribute</c-> <a id='the-document-object:htmlcollection-4' href='https://dom.spec.whatwg.org/#interface-htmlcollection' data-x-internal='htmlcollection'><c- n>HTMLCollection</c-></a> <a href='#dom-document-links' id='the-document-object:dom-document-links'><c- g>links</c-></a>;
  [<c- g>SameObject</c->] <c- b>readonly</c-> <c- b>attribute</c-> <a id='the-document-object:htmlcollection-5' href='https://dom.spec.whatwg.org/#interface-htmlcollection' data-x-internal='htmlcollection'><c- n>HTMLCollection</c-></a> <a href='#dom-document-forms' id='the-document-object:dom-document-forms'><c- g>forms</c-></a>;
  [<c- g>SameObject</c->] <c- b>readonly</c-> <c- b>attribute</c-> <a id='the-document-object:htmlcollection-6' href='https://dom.spec.whatwg.org/#interface-htmlcollection' data-x-internal='htmlcollection'><c- n>HTMLCollection</c-></a> <a href='#dom-document-scripts' id='the-document-object:dom-document-scripts'><c- g>scripts</c-></a>;
  <c- n>NodeList</c-> <a href='#dom-document-getelementsbyname' id='the-document-object:dom-document-getelementsbyname'><c- g>getElementsByName</c-></a>(<c- b>DOMString</c-> <c- g>elementName</c->);
  <c- b>readonly</c-> <c- b>attribute</c-> <a href='#htmlorsvgscriptelement' id='the-document-object:htmlorsvgscriptelement'><c- n>HTMLOrSVGScriptElement</c-></a>? <a href='#dom-document-currentscript' id='the-document-object:dom-document-currentscript'><c- g>currentScript</c-></a>; // classic scripts in a document tree only

  // <a href='#dynamic-markup-insertion' id='the-document-object:dynamic-markup-insertion'>dynamic markup insertion</a>
  [<a href='#cereactions' id='the-document-object:cereactions-4'><c- g>CEReactions</c-></a>] <a href='#document' id='the-document-object:document-2'><c- n>Document</c-></a> <a href='#dom-document-open' id='the-document-object:dom-document-open'><c- g>open</c-></a>(<c- b>optional</c-> <c- b>DOMString</c-> <c- g>unused1</c->, <c- b>optional</c-> <c- b>DOMString</c-> <c- g>unused2</c->); // both arguments are <a href='#dom-document-open-unused-arguments'>ignored</a>
  <a href='#windowproxy' id='the-document-object:windowproxy'><c- n>WindowProxy</c-></a>? <a href='#dom-document-open-window' id='the-document-object:dom-document-open-window'><c- g>open</c-></a>(<c- b>USVString</c-> <c- g>url</c->, <c- b>DOMString</c-> <c- g>name</c->, <c- b>DOMString</c-> <c- g>features</c->);
  [<a href='#cereactions' id='the-document-object:cereactions-5'><c- g>CEReactions</c-></a>] <c- b>undefined</c-> <a href='#dom-document-close' id='the-document-object:dom-document-close'><c- g>close</c-></a>();
  [<a href='#cereactions' id='the-document-object:cereactions-6'><c- g>CEReactions</c-></a>] <c- b>undefined</c-> <a href='#dom-document-write' id='the-document-object:dom-document-write'><c- g>write</c-></a>(<c- b>DOMString</c->... <c- g>text</c->);
  [<a href='#cereactions' id='the-document-object:cereactions-7'><c- g>CEReactions</c-></a>] <c- b>undefined</c-> <a href='#dom-document-writeln' id='the-document-object:dom-document-writeln'><c- g>writeln</c-></a>(<c- b>DOMString</c->... <c- g>text</c->);

  // <a href='#editing' id='the-document-object:editing'>user interaction</a>
  <c- b>readonly</c-> <c- b>attribute</c-> <a href='#windowproxy' id='the-document-object:windowproxy-2'><c- n>WindowProxy</c-></a>? <a href='#dom-document-defaultview' id='the-document-object:dom-document-defaultview'><c- g>defaultView</c-></a>;
  <c- b>boolean</c-> <a href='#dom-document-hasfocus' id='the-document-object:dom-document-hasfocus'><c- g>hasFocus</c-></a>();
  [<a href='#cereactions' id='the-document-object:cereactions-8'><c- g>CEReactions</c-></a>] <c- b>attribute</c-> <c- b>DOMString</c-> <a href='#designMode' id='the-document-object:designMode'><c- g>designMode</c-></a>;
  [<a href='#cereactions' id='the-document-object:cereactions-9'><c- g>CEReactions</c-></a>] <c- b>boolean</c-> <a href='https://w3c.github.io/editing/docs/execCommand/#execcommand%28%29' id='the-document-object:execCommand' data-x-internal='execCommand'><c- g>execCommand</c-></a>(<c- b>DOMString</c-> <c- g>commandId</c->, <c- b>optional</c-> <c- b>boolean</c-> <c- g>showUI</c-> = <c- b>false</c->, <c- b>optional</c-> <c- b>DOMString</c-> <c- g>value</c-> = &quot;&quot;);
  <c- b>boolean</c-> <a href='https://w3c.github.io/editing/docs/execCommand/#querycommandenabled%28%29' id='the-document-object:dom-document-querycommandenabled' data-x-internal='dom-document-querycommandenabled'><c- g>queryCommandEnabled</c-></a>(<c- b>DOMString</c-> <c- g>commandId</c->);
  <c- b>boolean</c-> <a href='https://w3c.github.io/editing/docs/execCommand/#querycommandindeterm%28%29' id='the-document-object:dom-document-querycommandindeterm' data-x-internal='dom-document-querycommandindeterm'><c- g>queryCommandIndeterm</c-></a>(<c- b>DOMString</c-> <c- g>commandId</c->);
  <c- b>boolean</c-> <a href='https://w3c.github.io/editing/docs/execCommand/#querycommandstate%28%29' id='the-document-object:dom-document-querycommandstate' data-x-internal='dom-document-querycommandstate'><c- g>queryCommandState</c-></a>(<c- b>DOMString</c-> <c- g>commandId</c->);
  <c- b>boolean</c-> <a href='https://w3c.github.io/editing/docs/execCommand/#querycommandsupported%28%29' id='the-document-object:dom-document-querycommandsupported' data-x-internal='dom-document-querycommandsupported'><c- g>queryCommandSupported</c-></a>(<c- b>DOMString</c-> <c- g>commandId</c->);
  <c- b>DOMString</c-> <a href='https://w3c.github.io/editing/docs/execCommand/#querycommandvalue%28%29' id='the-document-object:dom-document-querycommandvalue' data-x-internal='dom-document-querycommandvalue'><c- g>queryCommandValue</c-></a>(<c- b>DOMString</c-> <c- g>commandId</c->);

  // special <a href='#event-handler-idl-attributes' id='the-document-object:event-handler-idl-attributes'>event handler IDL attributes</a> that only apply to Document objects
  [<a id='the-document-object:legacylenientthis' href='https://heycam.github.io/webidl/#LegacyLenientThis' data-x-internal='legacylenientthis'><c- g>LegacyLenientThis</c-></a>] <c- b>attribute</c-> <a href='#eventhandler' id='the-document-object:eventhandler'><c- n>EventHandler</c-></a> <a href='#handler-onreadystatechange' id='the-document-object:handler-onreadystatechange'><c- g>onreadystatechange</c-></a>;

  // <a href='#Document-partial'>also has obsolete members</a>
};
<a href='#document' id='the-document-object:document-3'><c- n>Document</c-></a> <c- b>includes</c-> <a href='#globaleventhandlers' id='the-document-object:globaleventhandlers'><c- n>GlobalEventHandlers</c-></a>;
<a href='#document' id='the-document-object:document-4'><c- n>Document</c-></a> <c- b>includes</c-> <a href='#documentandelementeventhandlers' id='the-document-object:documentandelementeventhandlers'><c- n>DocumentAndElementEventHandlers</c-></a>;</code></pre>

  <p>The <code id=the-document-object:document-5><a href=#document>Document</a></code> has a <dfn data-dfn-for=Document id=concept-document-referrer-policy data-export="">referrer policy</dfn> (a <a id=the-document-object:referrer-policy href=https://w3c.github.io/webappsec-referrer-policy/#referrer-policy data-x-internal=referrer-policy>referrer policy</a>), initially the empty string,
  which represents the default <a id=the-document-object:referrer-policy-2 href=https://w3c.github.io/webappsec-referrer-policy/#referrer-policy data-x-internal=referrer-policy>referrer policy</a> used by <a href=https://fetch.spec.whatwg.org/#concept-fetch id=the-document-object:concept-fetch data-x-internal=concept-fetch>fetches</a> initiated by the <code id=the-document-object:document-6><a href=#document>Document</a></code>.</p>

  <p>The <code id=the-document-object:document-7><a href=#document>Document</a></code> has an <dfn id=concept-document-embedder-policy>embedder
  policy</dfn> (an <a href=#embedder-policy id=the-document-object:embedder-policy>embedder policy</a>).</p>

  <p>The <code id=the-document-object:document-8><a href=#document>Document</a></code> has a <dfn data-dfn-for=Document id=concept-document-csp-list data-export="">CSP list</dfn>, which is a <a href=https://w3c.github.io/webappsec-csp/#csp-list id=the-document-object:concept-csp-list data-x-internal=concept-csp-list>CSP list</a>
  containing all of the <a id=the-document-object:content-security-policy href=https://w3c.github.io/webappsec-csp/#content-security-policy-object data-x-internal=content-security-policy>Content Security Policy</a> objects active for the document. The
  list is empty unless otherwise specified.</p>

  <p id=concept-document-feature-policy>The <code id=the-document-object:document-9><a href=#document>Document</a></code> has a <dfn data-dfn-for=Document id=concept-document-permissions-policy data-export="">permissions policy</dfn>, which
  is a <a href=https://w3c.github.io/webappsec-feature-policy/#permissions-policy id=the-document-object:concept-permissions-policy data-x-internal=concept-permissions-policy>permissions policy</a>, which is initially
  empty.</p>

  <p>The <code id=the-document-object:document-10><a href=#document>Document</a></code> has a <dfn id=concept-document-module-map>module map</dfn>,
  which is a <a href=#module-map id=the-document-object:module-map>module map</a>, initially empty.</p>

  <p>The <code id=the-document-object:document-11><a href=#document>Document</a></code> has a <dfn id=concept-document-coop>cross-origin opener
  policy</dfn>, which is a <a href=#cross-origin-opener-policy id=the-document-object:cross-origin-opener-policy>cross-origin opener policy</a>.</p>

  <h4 id=the-documentorshadowroot-interface><span class=secno>3.1.2</span> The <code id=the-documentorshadowroot-interface:documentorshadowroot><a href=#documentorshadowroot>DocumentOrShadowRoot</a></code> interface<a href=#the-documentorshadowroot-interface class=self-link></a></h4>

  <p><cite>DOM</cite> defines the <code id=the-documentorshadowroot-interface:dom-documentorshadowroot><a data-x-internal=dom-documentorshadowroot href=https://dom.spec.whatwg.org/#documentorshadowroot>DocumentOrShadowRoot</a></code> mixin, which this specification
  extends.</p>

  <pre><code class='idl'><c- b>partial</c-> <c- b>interface</c-> <c- b>mixin</c-> <dfn data-lt id='documentorshadowroot'><c- g>DocumentOrShadowRoot</c-></dfn> {
  <c- b>readonly</c-> <c- b>attribute</c-> <a id='the-documentorshadowroot-interface:element' href='https://dom.spec.whatwg.org/#interface-element' data-x-internal='element'><c- n>Element</c-></a>? <a href='#dom-documentorshadowroot-activeelement' id='the-documentorshadowroot-interface:dom-documentorshadowroot-activeelement'><c- g>activeElement</c-></a>;
};</code></pre>

  <h4 id=resource-metadata-management><span class=secno>3.1.3</span> <dfn>Resource metadata management</dfn><a href=#resource-metadata-management class=self-link></a></h4>

  <dl class=domintro><dt><var>document</var> . <code id=dom-document-referrer-dev><a href=#dom-document-referrer>referrer</a></code><div class="mdn-anno wrapped before"><button onclick=toggleStatus(this) class=mdn-anno-btn><b title="Support in all current engines." class=all-engines-flag>✔</b><span>MDN</span></button><div class=feature><p><a href=https://developer.mozilla.org/en-US/docs/Web/API/Document/referrer title="The Document.referrer property returns the URI of the page that linked to this page.">Document/referrer</a><p class=all-engines-text>Support in all current engines.<div class=support><span class="firefox yes"><span>Firefox</span><span>1+</span></span><span class="safari yes"><span>Safari</span><span>1+</span></span><span class="chrome yes"><span>Chrome</span><span>1+</span></span><hr><span class="opera yes"><span>Opera</span><span>3+</span></span><span class="edge_blink yes"><span>Edge</span><span>79+</span></span><hr><span class="edge yes"><span>Edge (Legacy)</span><span>12+</span></span><span class="ie yes"><span>Internet Explorer</span><span>4+</span></span><hr><span class="firefox_android yes"><span>Firefox Android</span><span>4+</span></span><span class="safari_ios yes"><span>Safari iOS</span><span>1+</span></span><span class="chrome_android yes"><span>Chrome Android</span><span>18+</span></span><span class="webview_android yes"><span>WebView Android</span><span>1+</span></span><span class="samsunginternet_android yes"><span>Samsung Internet</span><span>1.0+</span></span><span class="opera_android yes"><span>Opera Android</span><span>10.1+</span></span></div></div></div><dd>
    <p>Returns the <a href=https://dom.spec.whatwg.org/#concept-document-url id="resource-metadata-management:the-document's-address" data-x-internal="the-document's-address">URL</a> of the <code id=resource-metadata-management:document><a href=#document>Document</a></code>
    from which the user navigated to this one, unless it was blocked or there was no such document,
    in which case it returns the empty string.</p>

    <p>The <code id=resource-metadata-management:link-type-noreferrer><a href=#link-type-noreferrer>noreferrer</a></code> link type can be used to block the
    referrer.</p>
   </dl>

  

  <p>The <dfn data-dfn-for=Document id=dom-document-referrer data-dfn-type=attribute><code>referrer</code></dfn>
  attribute must return <a href="#the-document's-referrer" id="resource-metadata-management:the-document's-referrer">the document's referrer</a>.</p>

  

  <hr>

  <dl class=domintro><dt><var>document</var> . <code id=dom-document-cookie-dev><a href=#dom-document-cookie>cookie</a></code> [ = <var>value</var> ]<dd>
    <p>Returns the HTTP cookies that apply to the <code id=resource-metadata-management:document-2><a href=#document>Document</a></code>. If there are no cookies or
    cookies can't be applied to this resource, the empty string will be returned.</p>

    <p>Can be set, to add a new cookie to the element's set of HTTP cookies.</p>

    <p>If the contents are <a href=#sandboxed-origin-browsing-context-flag id=resource-metadata-management:sandboxed-origin-browsing-context-flag>sandboxed into a
    unique origin</a> (e.g. in an <code id=resource-metadata-management:the-iframe-element><a href=#the-iframe-element>iframe</a></code> with the <code id=resource-metadata-management:attr-iframe-sandbox><a href=#attr-iframe-sandbox>sandbox</a></code> attribute), a
    <a id=resource-metadata-management:securityerror href=https://heycam.github.io/webidl/#securityerror data-x-internal=securityerror>"<code>SecurityError</code>"</a> <code id=resource-metadata-management:domexception><a data-x-internal=domexception href=https://heycam.github.io/webidl/#dfn-DOMException>DOMException</a></code> will be thrown on getting
    and setting.</p>
   </dl>

  

  <p>The <dfn data-dfn-for=Document id=dom-document-cookie data-dfn-type=attribute><code>cookie</code></dfn>
  attribute represents the cookies of the resource identified by the document's <a href=https://dom.spec.whatwg.org/#concept-document-url id="resource-metadata-management:the-document's-address-2" data-x-internal="the-document's-address">URL</a>.</p>

  <p>A <code id=resource-metadata-management:document-3><a href=#document>Document</a></code> object that falls into one of the following conditions is a
  <dfn id=cookie-averse-document-object>cookie-averse <code>Document</code> object</dfn>:</p>

  <ul><li>A <code id=resource-metadata-management:document-4><a href=#document>Document</a></code> object whose <a href=#concept-document-bc id=resource-metadata-management:concept-document-bc>browsing
   context</a> is null.<li>A <code id=resource-metadata-management:document-5><a href=#document>Document</a></code> whose <a href=https://dom.spec.whatwg.org/#concept-document-url id="resource-metadata-management:the-document's-address-3" data-x-internal="the-document's-address">URL</a>'s <a href=https://url.spec.whatwg.org/#concept-url-scheme id=resource-metadata-management:concept-url-scheme data-x-internal=concept-url-scheme>scheme</a> is not a <a id=resource-metadata-management:network-scheme href=https://fetch.spec.whatwg.org/#network-scheme data-x-internal=network-scheme>network scheme</a>.</ul>

  

  <p id=sandboxCookies>
  <a id=resource-metadata-management:tracking-vector title="There is a tracking vector here." href=https://infra.spec.whatwg.org/#tracking-vector class=tracking-vector data-x-internal=tracking-vector><img width=46 alt="(This is a tracking vector.)" crossorigin="" src=https://resources.whatwg.org/tracking-vector.svg height=64></a>
  On getting, if the document is a <a href=#cookie-averse-document-object id=resource-metadata-management:cookie-averse-document-object>cookie-averse <code>Document</code> object</a>, then the
  user agent must return the empty string. Otherwise, if the <code id=resource-metadata-management:document-6><a href=#document>Document</a></code>'s
  <a href=#concept-origin id=resource-metadata-management:concept-origin>origin</a> is an <a href=#concept-origin-opaque id=resource-metadata-management:concept-origin-opaque>opaque origin</a>, the user
  agent must throw a <a id=resource-metadata-management:securityerror-2 href=https://heycam.github.io/webidl/#securityerror data-x-internal=securityerror>"<code>SecurityError</code>"</a> <code id=resource-metadata-management:domexception-2><a data-x-internal=domexception href=https://heycam.github.io/webidl/#dfn-DOMException>DOMException</a></code>. Otherwise,
  the user agent must return the <a href=#cookie-string id=resource-metadata-management:cookie-string>cookie-string</a> for the document's <a href=https://dom.spec.whatwg.org/#concept-document-url id="resource-metadata-management:the-document's-address-4" data-x-internal="the-document's-address">URL</a> for a "non-HTTP" API, decoded using <a id=resource-metadata-management:utf-8-decode-without-bom href=https://encoding.spec.whatwg.org/#utf-8-decode-without-bom data-x-internal=utf-8-decode-without-bom>UTF-8 decode
  without BOM</a>. <a href=#refsCOOKIES>[COOKIES]</a></p>

  <p>On setting, if the document is a <a href=#cookie-averse-document-object id=resource-metadata-management:cookie-averse-document-object-2>cookie-averse <code>Document</code> object</a>, then
  the user agent must do nothing. Otherwise, if the <code id=resource-metadata-management:document-7><a href=#document>Document</a></code>'s <a href=#concept-origin id=resource-metadata-management:concept-origin-2>origin</a> is
  an <a href=#concept-origin-opaque id=resource-metadata-management:concept-origin-opaque-2>opaque origin</a>, the user agent must throw a
  <a id=resource-metadata-management:securityerror-3 href=https://heycam.github.io/webidl/#securityerror data-x-internal=securityerror>"<code>SecurityError</code>"</a> <code id=resource-metadata-management:domexception-3><a data-x-internal=domexception href=https://heycam.github.io/webidl/#dfn-DOMException>DOMException</a></code>. Otherwise, the user agent
  must act as it would when <a href=#receives-a-set-cookie-string id=resource-metadata-management:receives-a-set-cookie-string>receiving a
  set-cookie-string</a> for the document's <a href=https://dom.spec.whatwg.org/#concept-document-url id="resource-metadata-management:the-document's-address-5" data-x-internal="the-document's-address">URL</a> via a
  "non-HTTP" API, consisting of the new value <a href=https://encoding.spec.whatwg.org/#utf-8-encode id=resource-metadata-management:utf-8-encode data-x-internal=utf-8-encode>encoded as UTF-8</a>.
  <a href=#refsCOOKIES>[COOKIES]</a> <a href=#refsENCODING>[ENCODING]</a></p>

  <p class=note>Since the <code id=resource-metadata-management:dom-document-cookie><a href=#dom-document-cookie>cookie</a></code> attribute is accessible
  across frames, the path restrictions on cookies are only a tool to help manage which cookies are
  sent to which parts of the site, and are not in any way a security feature.</p>

  <p class=warning>The <code id=resource-metadata-management:dom-document-cookie-2><a href=#dom-document-cookie>cookie</a></code> attribute's getter and
  setter synchronously access shared state. Since there is no locking mechanism, other browsing
  contexts in a multiprocess user agent can modify cookies while scripts are running. A site could,
  for instance, try to read a cookie, increment its value, then write it back out, using the new
  value of the cookie as a unique identifier for the session; if the site does this twice in two
  different browser windows at the same time, it might end up using the same "unique" identifier for
  both sessions, with potentially disastrous effects.</p>

  <hr>

  

  <dl class=domintro><dt><var>document</var> . <code id=dom-document-lastmodified-dev><a href=#dom-document-lastmodified>lastModified</a></code><dd>
    <p>Returns the date of the last modification to the document, as reported by the server, in the
    form "<code>MM/DD/YYYY hh:mm:ss</code>", in the user's local time zone.</p>
    <p>If the last modification date is not known, the current time is returned instead.</p>
   </dl>

  

  <div class="mdn-anno wrapped before"><button onclick=toggleStatus(this) class=mdn-anno-btn><b title="Support in all current engines." class=all-engines-flag>✔</b><span>MDN</span></button><div class=feature><p><a href=https://developer.mozilla.org/en-US/docs/Web/API/Document/lastModified title="The lastModified property of the Document interface returns a string containing the date and time on which the current document was last modified.">Document/lastModified</a><p class=all-engines-text>Support in all current engines.<div class=support><span class="firefox yes"><span>Firefox</span><span>1+</span></span><span class="safari yes"><span>Safari</span><span>1+</span></span><span class="chrome yes"><span>Chrome</span><span>1+</span></span><hr><span class="opera yes"><span>Opera</span><span>12.1+</span></span><span class="edge_blink yes"><span>Edge</span><span>79+</span></span><hr><span class="edge yes"><span>Edge (Legacy)</span><span>12+</span></span><span class="ie yes"><span>Internet Explorer</span><span>4+</span></span><hr><span class="firefox_android yes"><span>Firefox Android</span><span>4+</span></span><span class="safari_ios yes"><span>Safari iOS</span><span>1+</span></span><span class="chrome_android yes"><span>Chrome Android</span><span>18+</span></span><span class="webview_android yes"><span>WebView Android</span><span>1+</span></span><span class="samsunginternet_android yes"><span>Samsung Internet</span><span>1.0+</span></span><span class="opera_android yes"><span>Opera Android</span><span>12.1+</span></span></div></div></div><p>The <dfn id=dom-document-lastmodified><code>lastModified</code></dfn> attribute, on
  getting, must return the date and time of the <code id=resource-metadata-management:document-8><a href=#document>Document</a></code>'s source file's last
  modification, in the user's local time zone, in the following format:</p>

  <ol><li> The month component of the date. <li> A U+002F SOLIDUS character (/). <li> The day component of the date. <li> A U+002F SOLIDUS character (/). <li> The year component of the date. <li> A U+0020 SPACE character. <li> The hours component of the time. <li> A U+003A COLON character (:). <li> The minutes component of the time. <li> A U+003A COLON character (:). <li> The seconds component of the time. </ol>

  <p>All the numeric components above, other than the year, must be given as two <a id=resource-metadata-management:ascii-digits href=https://infra.spec.whatwg.org/#ascii-digit data-x-internal=ascii-digits>ASCII
  digits</a> representing the number in base ten, zero-padded if necessary. The year must be
  given as the shortest possible string of four or more <a id=resource-metadata-management:ascii-digits-2 href=https://infra.spec.whatwg.org/#ascii-digit data-x-internal=ascii-digits>ASCII digits</a> representing the
  number in base ten, zero-padded if necessary.</p>

  <p>The <code id=resource-metadata-management:document-9><a href=#document>Document</a></code>'s source file's last modification date and time must be derived from
  relevant features of the networking protocols used, e.g. from the value of the HTTP `<code id=resource-metadata-management:http-last-modified><a data-x-internal=http-last-modified href=https://tools.ietf.org/html/rfc7232#section-2.2>Last-Modified</a></code>` header of the document, or from metadata in the
  file system for local files. If the last modification date and time are not known, the attribute
  must return the current date and time in the above format.</p>

  <hr>

  

  <dl class=domintro><dt><var>document</var> . <code id=dom-document-readystate-dev><a href=#dom-document-readystate>readyState</a></code><dd>
    <p>Returns "<code>loading</code>" while the <code id=resource-metadata-management:document-10><a href=#document>Document</a></code> is loading, "<code>interactive</code>" once it is finished parsing but still loading subresources, and
    "<code>complete</code>" once it has loaded.</p>

    <p>The <code id=resource-metadata-management:event-readystatechange><a href=#event-readystatechange>readystatechange</a></code> event fires on the
    <code id=resource-metadata-management:document-11><a href=#document>Document</a></code> object when this value changes.</p>

    <p>The <code id=resource-metadata-management:event-domcontentloaded><a href=#event-domcontentloaded>DOMContentLoaded</a></code> event fires after the transition to
    "<code>interactive</code>" but before the transition to "<code>complete</code>", at the point where all subresources apart from <code id=resource-metadata-management:attr-script-async><a href=#attr-script-async>async</a></code> <code id=resource-metadata-management:the-script-element><a href=#the-script-element>script</a></code> elements have loaded.</p>
   </dl>

  

  <div class="mdn-anno wrapped before"><button onclick=toggleStatus(this) class=mdn-anno-btn><b title="Support in all current engines." class=all-engines-flag>✔</b><span>MDN</span></button><div class=feature><p><a href=https://developer.mozilla.org/en-US/docs/Web/API/Document/readyState title="The Document.readyState property describes the loading state of the document.">Document/readyState</a><p class=all-engines-text>Support in all current engines.<div class=support><span class="firefox yes"><span>Firefox</span><span>4+</span></span><span class="safari yes"><span>Safari</span><span>1+</span></span><span class="chrome yes"><span>Chrome</span><span>1+</span></span><hr><span class="opera yes"><span>Opera</span><span>11+</span></span><span class="edge_blink yes"><span>Edge</span><span>79+</span></span><hr><span class="edge yes"><span>Edge (Legacy)</span><span>12+</span></span><span class="ie yes"><span>Internet Explorer</span><span>11</span></span><hr><span class="firefox_android yes"><span>Firefox Android</span><span>4+</span></span><span class="safari_ios yes"><span>Safari iOS</span><span>1+</span></span><span class="chrome_android yes"><span>Chrome Android</span><span>18+</span></span><span class="webview_android yes"><span>WebView Android</span><span>1+</span></span><span class="samsunginternet_android yes"><span>Samsung Internet</span><span>1.0+</span></span><span class="opera_android yes"><span>Opera Android</span><span>11+</span></span></div></div></div><p>Each document has a <dfn id=current-document-readiness>current document readiness</dfn>. When a <code id=resource-metadata-management:document-12><a href=#document>Document</a></code> object
  is created, it must have its <a href=#current-document-readiness id=resource-metadata-management:current-document-readiness>current document readiness</a> set to the string "<code>loading</code>" if the document is associated with an <a href=#html-parser id=resource-metadata-management:html-parser>HTML parser</a>, an
  <a href=#xml-parser id=resource-metadata-management:xml-parser>XML parser</a>, or an XSLT processor, and to the string "<code>complete</code>"
  otherwise. Various algorithms during page loading affect this value. When the value is set, the
  user agent must <a href=https://dom.spec.whatwg.org/#concept-event-fire id=resource-metadata-management:concept-event-fire data-x-internal=concept-event-fire>fire an event</a> named <code id=resource-metadata-management:event-readystatechange-2><a href=#event-readystatechange>readystatechange</a></code> at the <code id=resource-metadata-management:document-13><a href=#document>Document</a></code> object.</p>

  <p>A <code id=resource-metadata-management:document-14><a href=#document>Document</a></code> is said to have an <dfn id=active-parser>active parser</dfn> if it is associated with an
  <a href=#html-parser id=resource-metadata-management:html-parser-2>HTML parser</a> or an <a href=#xml-parser id=resource-metadata-management:xml-parser-2>XML parser</a> that has not yet been <a href=#stop-parsing id=resource-metadata-management:stop-parsing>stopped</a> or <a href=#abort-a-parser id=resource-metadata-management:abort-a-parser>aborted</a>.</p>

  <p>The <dfn id=dom-document-readystate><code>readyState</code></dfn> IDL attribute must, on
  getting, return the <a href=#current-document-readiness id=resource-metadata-management:current-document-readiness-2>current document readiness</a>.</p>

  



  <h4 id=dom-tree-accessors><span class=secno>3.1.4</span> <dfn>DOM tree accessors</dfn><a href=#dom-tree-accessors class=self-link></a></h4>

  <p><dfn id=the-html-element-2>The <code>html</code> element</dfn> of a document is its <a id=dom-tree-accessors:document-element href=https://dom.spec.whatwg.org/#document-element data-x-internal=document-element>document element</a>,
  if it's an <code id=dom-tree-accessors:the-html-element><a href=#the-html-element>html</a></code> element, and null otherwise.</p>

  <hr>

  <dl class=domintro><dt><var>document</var> . <code id=dom-document-head-dev><a href=#dom-document-head>head</a></code><dd>
    <p>Returns <a href=#the-head-element-2 id=dom-tree-accessors:the-head-element-2>the <code>head</code> element</a>.</p>
   </dl>

  <p><dfn id=the-head-element-2>The <code>head</code> element</dfn> of a document is the first <code id=dom-tree-accessors:the-head-element><a href=#the-head-element>head</a></code> element
  that is a child of <a href=#the-html-element-2 id=dom-tree-accessors:the-html-element-2>the <code>html</code> element</a>, if there is one, or null
  otherwise.</p>

  

  <div class="mdn-anno wrapped before"><button onclick=toggleStatus(this) class=mdn-anno-btn><b title="Support in all current engines." class=all-engines-flag>✔</b><span>MDN</span></button><div class=feature><p><a href=https://developer.mozilla.org/en-US/docs/Web/API/Document/head title="The head read-only property of the Document interface returns the <head> element of the current document.">Document/head</a><p class=all-engines-text>Support in all current engines.<div class=support><span class="firefox yes"><span>Firefox</span><span>4+</span></span><span class="safari yes"><span>Safari</span><span>5+</span></span><span class="chrome yes"><span>Chrome</span><span>4+</span></span><hr><span class="opera yes"><span>Opera</span><span>11+</span></span><span class="edge_blink yes"><span>Edge</span><span>79+</span></span><hr><span class="edge yes"><span>Edge (Legacy)</span><span>12+</span></span><span class="ie yes"><span>Internet Explorer</span><span>9+</span></span><hr><span class="firefox_android yes"><span>Firefox Android</span><span>4+</span></span><span class="safari_ios yes"><span>Safari iOS</span><span>Yes</span></span><span class="chrome_android yes"><span>Chrome Android</span><span>Yes</span></span><span class="webview_android yes"><span>WebView Android</span><span>Yes</span></span><span class="samsunginternet_android yes"><span>Samsung Internet</span><span>Yes</span></span><span class="opera_android yes"><span>Opera Android</span><span>Yes</span></span><hr><span class=caniuse><span><a href="https://caniuse.com/#feat=documenthead" title=document.head>caniuse.com table</a></span></span></div></div></div><p>The <dfn data-dfn-for=Document id=dom-document-head data-dfn-type=attribute><code>head</code></dfn> attribute,
  on getting, must return <a href=#the-head-element-2 id=dom-tree-accessors:the-head-element-2-2>the <code>head</code> element</a> of the document (a
  <code id=dom-tree-accessors:the-head-element-3><a href=#the-head-element>head</a></code> element or null).</p>

  

  <hr>

  <dl class=domintro><dt><var>document</var> . <code id=dom-document-title-dev><a href=#document.title>title</a></code> [ = <var>value</var> ]<dd>
    <p>Returns the document's title, as given by <a href=#the-title-element-2 id=dom-tree-accessors:the-title-element-2>the <code>title</code> element</a> for
    HTML and as given by the <a id=dom-tree-accessors:svg-title href=https://svgwg.org/svg2-draft/struct.html#TitleElement data-x-internal=svg-title>SVG <code>title</code></a> element for SVG.</p>

    <p>Can be set, to update the document's title. If there is no appropriate element to update, the
    new value is ignored.</p>
   </dl>

  <p><dfn id=the-title-element-2>The <code>title</code> element</dfn> of a document is the first <code id=dom-tree-accessors:the-title-element><a href=#the-title-element>title</a></code> element
  in the document (in <a id=dom-tree-accessors:tree-order href=https://dom.spec.whatwg.org/#concept-tree-order data-x-internal=tree-order>tree order</a>), if there is one, or null otherwise.</p>

  

  <div class="mdn-anno wrapped before"><button onclick=toggleStatus(this) class=mdn-anno-btn><b title="Support in all current engines." class=all-engines-flag>✔</b><span>MDN</span></button><div class=feature><p><a href=https://developer.mozilla.org/en-US/docs/Web/API/Document/title title="The document.title property gets or sets the current title of the document.">Document/title</a><p class=all-engines-text>Support in all current engines.<div class=support><span class="firefox yes"><span>Firefox</span><span>1+</span></span><span class="safari yes"><span>Safari</span><span>1+</span></span><span class="chrome yes"><span>Chrome</span><span>1+</span></span><hr><span class="opera yes"><span>Opera</span><span>12.1+</span></span><span class="edge_blink yes"><span>Edge</span><span>79+</span></span><hr><span class="edge yes"><span>Edge (Legacy)</span><span>12+</span></span><span class="ie yes"><span>Internet Explorer</span><span>4+</span></span><hr><span class="firefox_android yes"><span>Firefox Android</span><span>4+</span></span><span class="safari_ios yes"><span>Safari iOS</span><span>1+</span></span><span class="chrome_android yes"><span>Chrome Android</span><span>18+</span></span><span class="webview_android yes"><span>WebView Android</span><span>1+</span></span><span class="samsunginternet_android yes"><span>Samsung Internet</span><span>1.0+</span></span><span class="opera_android yes"><span>Opera Android</span><span>12.1+</span></span></div></div></div><p>The <dfn data-dfn-for=Document id=document.title data-dfn-type=attribute><code>title</code></dfn> attribute must, on getting, run the following
  algorithm:</p>

  <ol><li><p>If the <a id=dom-tree-accessors:document-element-2 href=https://dom.spec.whatwg.org/#document-element data-x-internal=document-element>document element</a> is an <a id=dom-tree-accessors:svg-svg href=https://svgwg.org/svg2-draft/struct.html#SVGElement data-x-internal=svg-svg>SVG <code>svg</code></a> element, then
   let <var>value</var> be the <a id=dom-tree-accessors:child-text-content href=https://dom.spec.whatwg.org/#concept-child-text-content data-x-internal=child-text-content>child text content</a> of the first <a id=dom-tree-accessors:svg-title-2 href=https://svgwg.org/svg2-draft/struct.html#TitleElement data-x-internal=svg-title>SVG
   <code>title</code></a> element that is a child of the <a id=dom-tree-accessors:document-element-3 href=https://dom.spec.whatwg.org/#document-element data-x-internal=document-element>document element</a>.<li><p>Otherwise, let <var>value</var> be the <a id=dom-tree-accessors:child-text-content-2 href=https://dom.spec.whatwg.org/#concept-child-text-content data-x-internal=child-text-content>child text content</a> of <a href=#the-title-element-2 id=dom-tree-accessors:the-title-element-2-2>the
   <code>title</code> element</a>, or the empty string if <a href=#the-title-element-2 id=dom-tree-accessors:the-title-element-2-3>the <code>title</code>
   element</a> is null.<li><p><a id=dom-tree-accessors:strip-and-collapse-ascii-whitespace href=https://infra.spec.whatwg.org/#strip-and-collapse-ascii-whitespace data-x-internal=strip-and-collapse-ascii-whitespace>Strip and collapse ASCII whitespace</a> in <var>value</var>.<li><p>Return <var>value</var>.</ol>

  <p>On setting, the steps corresponding to the first matching condition in the following list must
  be run:</p>

  <dl class=switch><dt>If the <a id=dom-tree-accessors:document-element-4 href=https://dom.spec.whatwg.org/#document-element data-x-internal=document-element>document element</a> is an <a id=dom-tree-accessors:svg-svg-2 href=https://svgwg.org/svg2-draft/struct.html#SVGElement data-x-internal=svg-svg>SVG <code>svg</code></a> element<dd>
    <ol><li><p>If there is an <a id=dom-tree-accessors:svg-title-3 href=https://svgwg.org/svg2-draft/struct.html#TitleElement data-x-internal=svg-title>SVG <code>title</code></a> element that is a child of the
     <a id=dom-tree-accessors:document-element-5 href=https://dom.spec.whatwg.org/#document-element data-x-internal=document-element>document element</a>, let <var>element</var> be the first such element.<li>
      <p>Otherwise:</p>

      <ol><li><p>Let <var>element</var> be the result of <a href=https://dom.spec.whatwg.org/#concept-create-element id=dom-tree-accessors:create-an-element data-x-internal=create-an-element>creating an
       element</a> given the <a id=dom-tree-accessors:document-element-6 href=https://dom.spec.whatwg.org/#document-element data-x-internal=document-element>document element</a>'s <a id=dom-tree-accessors:node-document href=https://dom.spec.whatwg.org/#concept-node-document data-x-internal=node-document>node document</a>, <code id=dom-tree-accessors:svg-title-4><a data-x-internal=svg-title href=https://svgwg.org/svg2-draft/struct.html#TitleElement>title</a></code>, and the <a id=dom-tree-accessors:svg-namespace href=https://infra.spec.whatwg.org/#svg-namespace data-x-internal=svg-namespace>SVG namespace</a>.</p>

       <li><p>Insert <var>element</var> as the <a id=dom-tree-accessors:first-child href=https://dom.spec.whatwg.org/#concept-tree-first-child data-x-internal=first-child>first child</a> of the <a id=dom-tree-accessors:document-element-7 href=https://dom.spec.whatwg.org/#document-element data-x-internal=document-element>document
       element</a>.</ol>
     <li><p><a id=dom-tree-accessors:string-replace-all href=https://dom.spec.whatwg.org/#string-replace-all data-x-internal=string-replace-all>String replace all</a> with the given value within <var>element</var>.</ol>
   <dt>If the <a id=dom-tree-accessors:document-element-8 href=https://dom.spec.whatwg.org/#document-element data-x-internal=document-element>document element</a> is in the <a id=dom-tree-accessors:html-namespace-2 href=https://infra.spec.whatwg.org/#html-namespace data-x-internal=html-namespace-2>HTML namespace</a><dd>
    <ol><li><p>If <a href=#the-title-element-2 id=dom-tree-accessors:the-title-element-2-4>the <code>title</code> element</a> is null and <a href=#the-head-element-2 id=dom-tree-accessors:the-head-element-2-3>the <code>head</code>
     element</a> is null, then return.<li><p>If <a href=#the-title-element-2 id=dom-tree-accessors:the-title-element-2-5>the <code>title</code> element</a> is non-null, let <var>element</var> be
     <a href=#the-title-element-2 id=dom-tree-accessors:the-title-element-2-6>the <code>title</code> element</a>.<li>
      <p>Otherwise:</p>

      <ol><li><p>Let <var>element</var> be the result of <a href=https://dom.spec.whatwg.org/#concept-create-element id=dom-tree-accessors:create-an-element-2 data-x-internal=create-an-element>creating an
       element</a> given the <a id=dom-tree-accessors:document-element-9 href=https://dom.spec.whatwg.org/#document-element data-x-internal=document-element>document element</a>'s <a id=dom-tree-accessors:node-document-2 href=https://dom.spec.whatwg.org/#concept-node-document data-x-internal=node-document>node document</a>,
       <code id=dom-tree-accessors:the-title-element-3><a href=#the-title-element>title</a></code>, and the <a id=dom-tree-accessors:html-namespace-2-2 href=https://infra.spec.whatwg.org/#html-namespace data-x-internal=html-namespace-2>HTML namespace</a>.</p>

       <li><p><a href=https://dom.spec.whatwg.org/#concept-node-append id=dom-tree-accessors:concept-node-append data-x-internal=concept-node-append>Append</a> <var>element</var> to <a href=#the-head-element-2 id=dom-tree-accessors:the-head-element-2-4>the
       <code>head</code> element</a>.</ol>
     <li><p><a id=dom-tree-accessors:string-replace-all-2 href=https://dom.spec.whatwg.org/#string-replace-all data-x-internal=string-replace-all>String replace all</a> with the given value within <var>element</var>.</ol>
   <dt>Otherwise<dd>
    <p>Do nothing.</p>
   </dl>

  

  <hr>

  <dl class=domintro><dt><var>document</var> . <code id=dom-document-body-dev><a href=#dom-document-body>body</a></code> [ = <var>value</var> ]<dd>
    <p>Returns <a href=#the-body-element-2 id=dom-tree-accessors:the-body-element-2>the body element</a>.</p>

    <p>Can be set, to replace <a href=#the-body-element-2 id=dom-tree-accessors:the-body-element-2-2>the body element</a>.</p>

    <p>If the new value is not a <code id=dom-tree-accessors:the-body-element><a href=#the-body-element>body</a></code> or <code id=dom-tree-accessors:frameset><a href=#frameset>frameset</a></code> element, this will throw
    a <a id=dom-tree-accessors:hierarchyrequesterror href=https://heycam.github.io/webidl/#hierarchyrequesterror data-x-internal=hierarchyrequesterror>"<code>HierarchyRequestError</code>"</a> <code id=dom-tree-accessors:domexception><a data-x-internal=domexception href=https://heycam.github.io/webidl/#dfn-DOMException>DOMException</a></code>.</p>
   </dl>

  <p><dfn id=the-body-element-2>The body element</dfn> of a document is the first of <a href=#the-html-element-2 id=dom-tree-accessors:the-html-element-2-2>the <code>html</code>
  element</a>'s children that is either a <code id=dom-tree-accessors:the-body-element-3><a href=#the-body-element>body</a></code> element or a <code id=dom-tree-accessors:frameset-2><a href=#frameset>frameset</a></code>
  element, or null if there is no such element.</p>

  

  <div class="mdn-anno wrapped before"><button onclick=toggleStatus(this) class=mdn-anno-btn><b title="Support in all current engines." class=all-engines-flag>✔</b><span>MDN</span></button><div class=feature><p><a href=https://developer.mozilla.org/en-US/docs/Web/API/Document/body title="The Document.body property represents the <body> or <frameset> node of the current document, or null if no such element exists.">Document/body</a><p class=all-engines-text>Support in all current engines.<div class=support><span class="firefox yes"><span>Firefox</span><span>2+</span></span><span class="safari yes"><span>Safari</span><span>4+</span></span><span class="chrome yes"><span>Chrome</span><span>1+</span></span><hr><span class="opera yes"><span>Opera</span><span>9.6+</span></span><span class="edge_blink yes"><span>Edge</span><span>79+</span></span><hr><span class="edge yes"><span>Edge (Legacy)</span><span>12+</span></span><span class="ie yes"><span>Internet Explorer</span><span>6+</span></span><hr><span class="firefox_android yes"><span>Firefox Android</span><span>4+</span></span><span class="safari_ios yes"><span>Safari iOS</span><span>5+</span></span><span class="chrome_android yes"><span>Chrome Android</span><span>18+</span></span><span class="webview_android yes"><span>WebView Android</span><span>37+</span></span><span class="samsunginternet_android yes"><span>Samsung Internet</span><span>1.0+</span></span><span class="opera_android yes"><span>Opera Android</span><span>10.1+</span></span></div></div></div><p>The <dfn data-dfn-for=Document id=dom-document-body data-dfn-type=attribute><code>body</code></dfn> attribute,
  on getting, must return <a href=#the-body-element-2 id=dom-tree-accessors:the-body-element-2-3>the body element</a> of the document (either a <code id=dom-tree-accessors:the-body-element-4><a href=#the-body-element>body</a></code>
  element, a <code id=dom-tree-accessors:frameset-3><a href=#frameset>frameset</a></code> element, or null). On setting, the following algorithm must be
  run:</p>

  <ol><li>If the new value is not a <code id=dom-tree-accessors:the-body-element-5><a href=#the-body-element>body</a></code> or <code id=dom-tree-accessors:frameset-4><a href=#frameset>frameset</a></code> element, then throw a
   <a id=dom-tree-accessors:hierarchyrequesterror-2 href=https://heycam.github.io/webidl/#hierarchyrequesterror data-x-internal=hierarchyrequesterror>"<code>HierarchyRequestError</code>"</a> <code id=dom-tree-accessors:domexception-2><a data-x-internal=domexception href=https://heycam.github.io/webidl/#dfn-DOMException>DOMException</a></code>.<li>Otherwise, if the new value is the same as <a href=#the-body-element-2 id=dom-tree-accessors:the-body-element-2-4>the body element</a>, return.<li>Otherwise, if <a href=#the-body-element-2 id=dom-tree-accessors:the-body-element-2-5>the body element</a> is not null, then <a href=https://dom.spec.whatwg.org/#concept-node-replace id=dom-tree-accessors:concept-node-replace data-x-internal=concept-node-replace>replace</a> <a href=#the-body-element-2 id=dom-tree-accessors:the-body-element-2-6>the body element</a> with the new value
   within <a href=#the-body-element-2 id=dom-tree-accessors:the-body-element-2-7>the body element</a>'s parent and return.<li>Otherwise, if there is no <a id=dom-tree-accessors:document-element-10 href=https://dom.spec.whatwg.org/#document-element data-x-internal=document-element>document element</a>, throw a
   <a id=dom-tree-accessors:hierarchyrequesterror-3 href=https://heycam.github.io/webidl/#hierarchyrequesterror data-x-internal=hierarchyrequesterror>"<code>HierarchyRequestError</code>"</a> <code id=dom-tree-accessors:domexception-3><a data-x-internal=domexception href=https://heycam.github.io/webidl/#dfn-DOMException>DOMException</a></code>.<li>Otherwise, <a href=#the-body-element-2 id=dom-tree-accessors:the-body-element-2-8>the body element</a> is null, but there's a
   <a id=dom-tree-accessors:document-element-11 href=https://dom.spec.whatwg.org/#document-element data-x-internal=document-element>document element</a>. <a href=https://dom.spec.whatwg.org/#concept-node-append id=dom-tree-accessors:concept-node-append-2 data-x-internal=concept-node-append>Append</a> the new value to
   the <a id=dom-tree-accessors:document-element-12 href=https://dom.spec.whatwg.org/#document-element data-x-internal=document-element>document element</a>.</ol>

  <p class=note>The value returned by the <code id=dom-tree-accessors:dom-document-body><a href=#dom-document-body>body</a></code> getter is
  not always the one passed to the setter.</p>

  <div class=example>
   <p>In this example, the setter successfully inserts a <code id=dom-tree-accessors:the-body-element-6><a href=#the-body-element>body</a></code> element (though this is
   non-conforming since SVG does not allow a <code id=dom-tree-accessors:the-body-element-7><a href=#the-body-element>body</a></code> as child of <a id=dom-tree-accessors:svg-svg-3 href=https://svgwg.org/svg2-draft/struct.html#SVGElement data-x-internal=svg-svg>SVG
   <code>svg</code></a>). However the getter will return null because the document element is not
   <code id=dom-tree-accessors:the-html-element-3><a href=#the-html-element>html</a></code>.</p>

   <pre><code class='html'><c- p>&lt;</c-><c- f>svg</c-> <c- e>xmlns</c-><c- o>=</c-><c- s>&quot;http://www.w3.org/2000/svg&quot;</c-><c- p>&gt;</c->
 <c- p>&lt;</c-><c- f>script</c-><c- p>&gt;</c->
  document<c- p>.</c->body <c- o>=</c-> document<c- p>.</c->createElementNS<c- p>(</c-><c- u>&quot;http://www.w3.org/1999/xhtml&quot;</c-><c- p>,</c-> <c- u>&quot;body&quot;</c-><c- p>);</c->
  console<c- p>.</c->assert<c- p>(</c->document<c- p>.</c->body <c- o>===</c-> <c- kc>null</c-><c- p>);</c->
 <c- p>&lt;/</c-><c- f>script</c-><c- p>&gt;</c->
<c- p>&lt;/</c-><c- f>svg</c-><c- p>&gt;</c-></code></pre>
  </div>

  

  <hr>

  <dl class=domintro><dt><var>document</var> . <code id=dom-document-images-dev><a href=#dom-document-images>images</a></code><dd>
    <p>Returns an <code id=dom-tree-accessors:htmlcollection><a data-x-internal=htmlcollection href=https://dom.spec.whatwg.org/#interface-htmlcollection>HTMLCollection</a></code> of the <code id=dom-tree-accessors:the-img-element><a href=#the-img-element>img</a></code> elements in the <code id=dom-tree-accessors:document><a href=#document>Document</a></code>.</p>
   <dt><var>document</var> . <code id=dom-document-embeds-dev><a href=#dom-document-embeds>embeds</a></code><dt><var>document</var> . <code id=dom-document-plugins-dev><a href=#dom-document-plugins>plugins</a></code><dd>
    <p>Return an <code id=dom-tree-accessors:htmlcollection-2><a data-x-internal=htmlcollection href=https://dom.spec.whatwg.org/#interface-htmlcollection>HTMLCollection</a></code> of the <code id=dom-tree-accessors:the-embed-element><a href=#the-embed-element>embed</a></code> elements in the <code id=dom-tree-accessors:document-2><a href=#document>Document</a></code>.</p>
   <dt><var>document</var> . <code id=dom-document-links-dev><a href=#dom-document-links>links</a></code><dd>
    <p>Returns an <code id=dom-tree-accessors:htmlcollection-3><a data-x-internal=htmlcollection href=https://dom.spec.whatwg.org/#interface-htmlcollection>HTMLCollection</a></code> of the <code id=dom-tree-accessors:the-a-element><a href=#the-a-element>a</a></code> and <code id=dom-tree-accessors:the-area-element><a href=#the-area-element>area</a></code> elements
    in the <code id=dom-tree-accessors:document-3><a href=#document>Document</a></code> that have <code id=dom-tree-accessors:attr-hyperlink-href><a href=#attr-hyperlink-href>href</a></code>
    attributes.</p>
   <dt><var>document</var> . <code id=dom-document-forms-dev><a href=#dom-document-forms>forms</a></code><dd>
    <p>Return an <code id=dom-tree-accessors:htmlcollection-4><a data-x-internal=htmlcollection href=https://dom.spec.whatwg.org/#interface-htmlcollection>HTMLCollection</a></code> of the <code id=dom-tree-accessors:the-form-element><a href=#the-form-element>form</a></code> elements in the <code id=dom-tree-accessors:document-4><a href=#document>Document</a></code>.</p>
   <dt><var>document</var> . <code id=dom-document-scripts-dev><a href=#dom-document-scripts>scripts</a></code><dd>
    <p>Return an <code id=dom-tree-accessors:htmlcollection-5><a data-x-internal=htmlcollection href=https://dom.spec.whatwg.org/#interface-htmlcollection>HTMLCollection</a></code> of the <code id=dom-tree-accessors:the-script-element><a href=#the-script-element>script</a></code> elements in the <code id=dom-tree-accessors:document-5><a href=#document>Document</a></code>.</p>
   </dl>

  

  

  <div class="mdn-anno wrapped before"><button onclick=toggleStatus(this) class=mdn-anno-btn><b title="Support in all current engines." class=all-engines-flag>✔</b><span>MDN</span></button><div class=feature><p><a href=https://developer.mozilla.org/en-US/docs/Web/API/Document/images title="The images read-only property of the Document interface returns a collection of the images in the current HTML document.">Document/images</a><p class=all-engines-text>Support in all current engines.<div class=support><span class="firefox yes"><span>Firefox</span><span>1+</span></span><span class="safari yes"><span>Safari</span><span>1+</span></span><span class="chrome yes"><span>Chrome</span><span>1+</span></span><hr><span class="opera yes"><span>Opera</span><span>12.1+</span></span><span class="edge_blink yes"><span>Edge</span><span>79+</span></span><hr><span class="edge yes"><span>Edge (Legacy)</span><span>12+</span></span><span class="ie yes"><span>Internet Explorer</span><span>4+</span></span><hr><span class="firefox_android yes"><span>Firefox Android</span><span>4+</span></span><span class="safari_ios yes"><span>Safari iOS</span><span>1+</span></span><span class="chrome_android yes"><span>Chrome Android</span><span>18+</span></span><span class="webview_android yes"><span>WebView Android</span><span>1+</span></span><span class="samsunginternet_android yes"><span>Samsung Internet</span><span>1.0+</span></span><span class="opera_android yes"><span>Opera Android</span><span>12.1+</span></span></div></div></div><p>The <dfn data-dfn-for=Document id=dom-document-images data-dfn-type=attribute><code>images</code></dfn>
  attribute must return an <code id=dom-tree-accessors:htmlcollection-6><a data-x-internal=htmlcollection href=https://dom.spec.whatwg.org/#interface-htmlcollection>HTMLCollection</a></code> rooted at the <code id=dom-tree-accessors:document-6><a href=#document>Document</a></code> node,
  whose filter matches only <code id=dom-tree-accessors:the-img-element-2><a href=#the-img-element>img</a></code> elements.</p>

  <div class="mdn-anno wrapped before"><button onclick=toggleStatus(this) class=mdn-anno-btn><b title="Support in all current engines." class=all-engines-flag>✔</b><span>MDN</span></button><div class=feature><p><a href=https://developer.mozilla.org/en-US/docs/Web/API/Document/embeds title="The embeds read-only property of the Document interface returns a list of the embedded <object> elements within the current document.">Document/embeds</a><p class=all-engines-text>Support in all current engines.<div class=support><span class="firefox yes"><span>Firefox</span><span>1+</span></span><span class="safari yes"><span>Safari</span><span>2+</span></span><span class="chrome yes"><span>Chrome</span><span>45+</span></span><hr><span class="opera yes"><span>Opera</span><span>12.1+</span></span><span class="edge_blink yes"><span>Edge</span><span>79+</span></span><hr><span class="edge yes"><span>Edge (Legacy)</span><span>12+</span></span><span class="ie yes"><span>Internet Explorer</span><span>4+</span></span><hr><span class="firefox_android yes"><span>Firefox Android</span><span>4+</span></span><span class="safari_ios yes"><span>Safari iOS</span><span>1+</span></span><span class="chrome_android yes"><span>Chrome Android</span><span>45+</span></span><span class="webview_android yes"><span>WebView Android</span><span>45+</span></span><span class="samsunginternet_android yes"><span>Samsung Internet</span><span>5.0+</span></span><span class="opera_android yes"><span>Opera Android</span><span>12.1+</span></span></div></div></div><p>The <dfn data-dfn-for=Document id=dom-document-embeds data-dfn-type=attribute><code>embeds</code></dfn>
  attribute must return an <code id=dom-tree-accessors:htmlcollection-7><a data-x-internal=htmlcollection href=https://dom.spec.whatwg.org/#interface-htmlcollection>HTMLCollection</a></code> rooted at the <code id=dom-tree-accessors:document-7><a href=#document>Document</a></code> node,
  whose filter matches only <code id=dom-tree-accessors:the-embed-element-2><a href=#the-embed-element>embed</a></code> elements.</p>

  <div class="mdn-anno wrapped before"><button onclick=toggleStatus(this) class=mdn-anno-btn><b title="Support in all current engines." class=all-engines-flag>✔</b><span>MDN</span></button><div class=feature><p><a href=https://developer.mozilla.org/en-US/docs/Web/API/Document/plugins title="The plugins read-only property of the Document interface returns an HTMLCollection object containing one or more HTMLEmbedElements representing the <embed> elements in the current document.">Document/plugins</a><p class=all-engines-text>Support in all current engines.<div class=support><span class="firefox yes"><span>Firefox</span><span>1+</span></span><span class="safari yes"><span>Safari</span><span>3+</span></span><span class="chrome yes"><span>Chrome</span><span>45+</span></span><hr><span class="opera yes"><span>Opera</span><span>12.1+</span></span><span class="edge_blink yes"><span>Edge</span><span>79+</span></span><hr><span class="edge yes"><span>Edge (Legacy)</span><span>12+</span></span><span class="ie yes"><span>Internet Explorer</span><span>4+</span></span><hr><span class="firefox_android yes"><span>Firefox Android</span><span>4+</span></span><span class="safari_ios yes"><span>Safari iOS</span><span>1+</span></span><span class="chrome_android yes"><span>Chrome Android</span><span>45+</span></span><span class="webview_android yes"><span>WebView Android</span><span>45+</span></span><span class="samsunginternet_android yes"><span>Samsung Internet</span><span>5.0+</span></span><span class="opera_android yes"><span>Opera Android</span><span>12.1+</span></span></div></div></div><p>The <dfn data-dfn-for=Document id=dom-document-plugins data-dfn-type=attribute><code>plugins</code></dfn>
  attribute must return the same object as that returned by the <code id=dom-tree-accessors:dom-document-embeds><a href=#dom-document-embeds>embeds</a></code> attribute.</p>

  <div class="mdn-anno wrapped before"><button onclick=toggleStatus(this) class=mdn-anno-btn><b title="Support in all current engines." class=all-engines-flag>✔</b><span>MDN</span></button><div class=feature><p><a href=https://developer.mozilla.org/en-US/docs/Web/API/Document/links title="The links read-only property of the Document interface returns a collection of all <area> elements and <a> elements in a document with a value for the href attribute.">Document/links</a><p class=all-engines-text>Support in all current engines.<div class=support><span class="firefox yes"><span>Firefox</span><span>1+</span></span><span class="safari yes"><span>Safari</span><span>1+</span></span><span class="chrome yes"><span>Chrome</span><span>1+</span></span><hr><span class="opera yes"><span>Opera</span><span>12.1+</span></span><span class="edge_blink yes"><span>Edge</span><span>79+</span></span><hr><span class="edge yes"><span>Edge (Legacy)</span><span>12+</span></span><span class="ie yes"><span>Internet Explorer</span><span>4+</span></span><hr><span class="firefox_android yes"><span>Firefox Android</span><span>4+</span></span><span class="safari_ios yes"><span>Safari iOS</span><span>1+</span></span><span class="chrome_android yes"><span>Chrome Android</span><span>18+</span></span><span class="webview_android yes"><span>WebView Android</span><span>1+</span></span><span class="samsunginternet_android yes"><span>Samsung Internet</span><span>1.0+</span></span><span class="opera_android yes"><span>Opera Android</span><span>12.1+</span></span></div></div></div><p>The <dfn data-dfn-for=Document id=dom-document-links data-dfn-type=attribute><code>links</code></dfn>
  attribute must return an <code id=dom-tree-accessors:htmlcollection-8><a data-x-internal=htmlcollection href=https://dom.spec.whatwg.org/#interface-htmlcollection>HTMLCollection</a></code> rooted at the <code id=dom-tree-accessors:document-8><a href=#document>Document</a></code> node,
  whose filter matches only <code id=dom-tree-accessors:the-a-element-2><a href=#the-a-element>a</a></code> elements with <code id=dom-tree-accessors:attr-hyperlink-href-2><a href=#attr-hyperlink-href>href</a></code> attributes and <code id=dom-tree-accessors:the-area-element-2><a href=#the-area-element>area</a></code> elements with <code id=dom-tree-accessors:attr-hyperlink-href-3><a href=#attr-hyperlink-href>href</a></code> attributes.</p>

  <div class="mdn-anno wrapped before"><button onclick=toggleStatus(this) class=mdn-anno-btn><b title="Support in all current engines." class=all-engines-flag>✔</b><span>MDN</span></button><div class=feature><p><a href=https://developer.mozilla.org/en-US/docs/Web/API/Document/forms title="The forms read-only property of the Document interface returns an HTMLCollection listing all the <form> elements contained in the document.">Document/forms</a><p class=all-engines-text>Support in all current engines.<div class=support><span class="firefox yes"><span>Firefox</span><span>1+</span></span><span class="safari yes"><span>Safari</span><span>1+</span></span><span class="chrome yes"><span>Chrome</span><span>1+</span></span><hr><span class="opera yes"><span>Opera</span><span>12.1+</span></span><span class="edge_blink yes"><span>Edge</span><span>79+</span></span><hr><span class="edge yes"><span>Edge (Legacy)</span><span>12+</span></span><span class="ie yes"><span>Internet Explorer</span><span>4+</span></span><hr><span class="firefox_android yes"><span>Firefox Android</span><span>4+</span></span><span class="safari_ios yes"><span>Safari iOS</span><span>1+</span></span><span class="chrome_android yes"><span>Chrome Android</span><span>18+</span></span><span class="webview_android yes"><span>WebView Android</span><span>1+</span></span><span class="samsunginternet_android yes"><span>Samsung Internet</span><span>1.0+</span></span><span class="opera_android yes"><span>Opera Android</span><span>12.1+</span></span></div></div></div><p>The <dfn data-dfn-for=Document id=dom-document-forms data-dfn-type=attribute><code>forms</code></dfn>
  attribute must return an <code id=dom-tree-accessors:htmlcollection-9><a data-x-internal=htmlcollection href=https://dom.spec.whatwg.org/#interface-htmlcollection>HTMLCollection</a></code> rooted at the <code id=dom-tree-accessors:document-9><a href=#document>Document</a></code> node,
  whose filter matches only <code id=dom-tree-accessors:the-form-element-2><a href=#the-form-element>form</a></code> elements.</p>

  <div class="mdn-anno wrapped before"><button onclick=toggleStatus(this) class=mdn-anno-btn><b title="Support in all current engines." class=all-engines-flag>✔</b><span>MDN</span></button><div class=feature><p><a href=https://developer.mozilla.org/en-US/docs/Web/API/Document/scripts title="The scripts property of the Document interface returns a list of the <script> elements in the document. The returned object is an HTMLCollection.">Document/scripts</a><p class=all-engines-text>Support in all current engines.<div class=support><span class="firefox yes"><span>Firefox</span><span>9+</span></span><span class="safari yes"><span>Safari</span><span>3+</span></span><span class="chrome yes"><span>Chrome</span><span>1+</span></span><hr><span class="opera yes"><span>Opera</span><span>12.1+</span></span><span class="edge_blink yes"><span>Edge</span><span>79+</span></span><hr><span class="edge yes"><span>Edge (Legacy)</span><span>12+</span></span><span class="ie yes"><span>Internet Explorer</span><span>4+</span></span><hr><span class="firefox_android yes"><span>Firefox Android</span><span>9+</span></span><span class="safari_ios yes"><span>Safari iOS</span><span>1+</span></span><span class="chrome_android yes"><span>Chrome Android</span><span>18+</span></span><span class="webview_android yes"><span>WebView Android</span><span>1+</span></span><span class="samsunginternet_android yes"><span>Samsung Internet</span><span>1.0+</span></span><span class="opera_android yes"><span>Opera Android</span><span>12.1+</span></span></div></div></div><p>The <dfn data-dfn-for=Document id=dom-document-scripts data-dfn-type=attribute><code>scripts</code></dfn>
  attribute must return an <code id=dom-tree-accessors:htmlcollection-10><a data-x-internal=htmlcollection href=https://dom.spec.whatwg.org/#interface-htmlcollection>HTMLCollection</a></code> rooted at the <code id=dom-tree-accessors:document-10><a href=#document>Document</a></code> node,
  whose filter matches only <code id=dom-tree-accessors:the-script-element-2><a href=#the-script-element>script</a></code> elements.</p>

  <hr>

  

  <dl class=domintro><dt><var>collection</var> = <var>document</var> . <code id=dom-document-getelementsbyname-dev><a href=#dom-document-getelementsbyname>getElementsByName</a></code>(<var>name</var>)<dd>
    <p>Returns a <code id=dom-tree-accessors:nodelist><a data-x-internal=nodelist href=https://dom.spec.whatwg.org/#interface-nodelist>NodeList</a></code> of elements in the <code id=dom-tree-accessors:document-11><a href=#document>Document</a></code> that have a <code>name</code> attribute with the value <var>name</var>.</p>
   </dl>

  

  <div class="mdn-anno wrapped before"><button onclick=toggleStatus(this) class=mdn-anno-btn><b title="Support in all current engines." class=all-engines-flag>✔</b><span>MDN</span></button><div class=feature><p><a href=https://developer.mozilla.org/en-US/docs/Web/API/Document/getElementsByName title="The getElementsByName() method of the Document object returns a NodeList Collection of elements with a given name in the document.">Document/getElementsByName</a><p class=all-engines-text>Support in all current engines.<div class=support><span class="firefox yes"><span>Firefox</span><span>1+</span></span><span class="safari yes"><span>Safari</span><span>1+</span></span><span class="chrome yes"><span>Chrome</span><span>1+</span></span><hr><span class="opera yes"><span>Opera</span><span>5+</span></span><span class="edge_blink yes"><span>Edge</span><span>79+</span></span><hr><span class="edge yes"><span>Edge (Legacy)</span><span>12+</span></span><span class="ie yes"><span>Internet Explorer</span><span>5+</span></span><hr><span class="firefox_android yes"><span>Firefox Android</span><span>4+</span></span><span class="safari_ios yes"><span>Safari iOS</span><span>1+</span></span><span class="chrome_android yes"><span>Chrome Android</span><span>18+</span></span><span class="webview_android yes"><span>WebView Android</span><span>1+</span></span><span class="samsunginternet_android yes"><span>Samsung Internet</span><span>1.0+</span></span><span class="opera_android yes"><span>Opera Android</span><span>10.1+</span></span></div></div></div><p>The <dfn id=dom-document-getelementsbyname><code>getElementsByName(<var>name</var>)</code></dfn> method
  steps are to return a <a href=#live id=dom-tree-accessors:live>live</a> <code id=dom-tree-accessors:nodelist-2><a data-x-internal=nodelist href=https://dom.spec.whatwg.org/#interface-nodelist>NodeList</a></code> containing all the <a href=#html-elements id=dom-tree-accessors:html-elements>HTML
  elements</a> in that document that have a <code>name</code> attribute whose value is
  <a id=dom-tree-accessors:identical-to href=https://infra.spec.whatwg.org/#string-is data-x-internal=identical-to>identical to</a> the <var>name</var> argument, in <a id=dom-tree-accessors:tree-order-2 href=https://dom.spec.whatwg.org/#concept-tree-order data-x-internal=tree-order>tree order</a>. When the
  method is invoked on a <code id=dom-tree-accessors:document-12><a href=#document>Document</a></code> object again with the same argument, the user agent
  may return the same as the object returned by the earlier call. In other cases, a new
  <code id=dom-tree-accessors:nodelist-3><a data-x-internal=nodelist href=https://dom.spec.whatwg.org/#interface-nodelist>NodeList</a></code> object must be returned.</p>

  

  <hr>

  <dl class=domintro><dt><var>document</var> . <code id=dom-document-currentscript-dev><a href=#dom-document-currentscript>currentScript</a></code><dd>
    <p>Returns the <code id=dom-tree-accessors:the-script-element-3><a href=#the-script-element>script</a></code> element, or the <a id=dom-tree-accessors:svg-script href=https://svgwg.org/svg2-draft/interact.html#ScriptElement data-x-internal=svg-script>SVG <code>script</code></a> element,
    that is currently executing, as long as the element represents a <a href=#classic-script id=dom-tree-accessors:classic-script>classic script</a>.
    In the case of reentrant script execution, returns the one that most recently started executing
    amongst those that have not yet finished executing.</p>

    <p>Returns null if the <code id=dom-tree-accessors:document-13><a href=#document>Document</a></code> is not currently executing a <code id=dom-tree-accessors:the-script-element-4><a href=#the-script-element>script</a></code>
    or <a id=dom-tree-accessors:svg-script-2 href=https://svgwg.org/svg2-draft/interact.html#ScriptElement data-x-internal=svg-script>SVG <code>script</code></a> element (e.g., because the running script is an event
    handler, or a timeout), or if the currently executing <code id=dom-tree-accessors:the-script-element-5><a href=#the-script-element>script</a></code> or <a id=dom-tree-accessors:svg-script-3 href=https://svgwg.org/svg2-draft/interact.html#ScriptElement data-x-internal=svg-script>SVG
    <code>script</code></a> element represents a <a href=#module-script id=dom-tree-accessors:module-script>module script</a>.</p>
   </dl>

  

  <div class="mdn-anno wrapped before"><button onclick=toggleStatus(this) class=mdn-anno-btn><b title="Support in all current engines." class=all-engines-flag>✔</b><span>MDN</span></button><div class=feature><p><a href=https://developer.mozilla.org/en-US/docs/Web/API/Document/currentScript title="The Document.currentScript property returns the <script> element whose script is currently being processed and isn't a JavaScript module. (For modules use import.meta instead.)">Document/currentScript</a><p class=all-engines-text>Support in all current engines.<div class=support><span class="firefox yes"><span>Firefox</span><span>4+</span></span><span class="safari yes"><span>Safari</span><span>8+</span></span><span class="chrome yes"><span>Chrome</span><span>29+</span></span><hr><span class="opera yes"><span>Opera</span><span>16+</span></span><span class="edge_blink yes"><span>Edge</span><span>79+</span></span><hr><span class="edge yes"><span>Edge (Legacy)</span><span>12+</span></span><span class="ie no"><span>Internet Explorer</span><span>No</span></span><hr><span class="firefox_android yes"><span>Firefox Android</span><span>Yes</span></span><span class="safari_ios yes"><span>Safari iOS</span><span>8+</span></span><span class="chrome_android yes"><span>Chrome Android</span><span>29+</span></span><span class="webview_android yes"><span>WebView Android</span><span>Yes</span></span><span class="samsunginternet_android yes"><span>Samsung Internet</span><span>2.0+</span></span><span class="opera_android yes"><span>Opera Android</span><span>Yes</span></span><hr><span class=caniuse><span><a href="https://caniuse.com/#feat=document-currentscript" title=document.currentScript>caniuse.com table</a></span></span></div></div></div><p>The <dfn id=dom-document-currentscript><code>currentScript</code></dfn> attribute, on
  getting, must return the value to which it was most recently set. When the <code id=dom-tree-accessors:document-14><a href=#document>Document</a></code>
  is created, the <code id=dom-tree-accessors:dom-document-currentscript><a href=#dom-document-currentscript>currentScript</a></code> must be
  initialized to null.</p>

  

  <p class=note>This API has fallen out of favor in the implementer and standards community, as
  it globally exposes <code id=dom-tree-accessors:the-script-element-6><a href=#the-script-element>script</a></code> or <a id=dom-tree-accessors:svg-script-4 href=https://svgwg.org/svg2-draft/interact.html#ScriptElement data-x-internal=svg-script>SVG <code>script</code></a> elements. As such,
  it is not available in newer contexts, such as when running <a href=#module-script id=dom-tree-accessors:module-script-2>module
  scripts</a> or when running scripts in a <a id=dom-tree-accessors:shadow-tree href=https://dom.spec.whatwg.org/#concept-shadow-tree data-x-internal=shadow-tree>shadow tree</a>. We are looking into creating
  a new solution for identifying the running script in such contexts, which does not make it
  globally available: see <a href=https://github.com/whatwg/html/issues/1013>issue #1013</a>.</p>

  

  <hr>

  <p id=dom-document-namedItem-which>The <code id=dom-tree-accessors:document-15><a href=#document>Document</a></code> interface <a href=https://heycam.github.io/webidl/#dfn-support-named-properties id=dom-tree-accessors:support-named-properties data-x-internal=support-named-properties>supports named properties</a>. The <a id=dom-tree-accessors:supported-property-names href=https://heycam.github.io/webidl/#dfn-supported-property-names data-x-internal=supported-property-names>supported property names</a> of a
  <code id=dom-tree-accessors:document-16><a href=#document>Document</a></code> object <var>document</var> at any moment consist of the following, in
  <a id=dom-tree-accessors:tree-order-3 href=https://dom.spec.whatwg.org/#concept-tree-order data-x-internal=tree-order>tree order</a> according to the element that contributed them, ignoring later duplicates,
  and with values from <code id=dom-tree-accessors:the-id-attribute><a href=#the-id-attribute>id</a></code> attributes coming before values from <code>name</code> attributes when the same element contributes both:</p>

  
  <ul><li><p>the value of the <code>name</code> content attribute for all
   <a href=#exposed id=dom-tree-accessors:exposed>exposed</a> <code id=dom-tree-accessors:the-embed-element-3><a href=#the-embed-element>embed</a></code>, <code id=dom-tree-accessors:the-form-element-3><a href=#the-form-element>form</a></code>, <code id=dom-tree-accessors:the-iframe-element><a href=#the-iframe-element>iframe</a></code>,
   <code id=dom-tree-accessors:the-img-element-3><a href=#the-img-element>img</a></code>, and <a href=#exposed id=dom-tree-accessors:exposed-2>exposed</a> <code id=dom-tree-accessors:the-object-element><a href=#the-object-element>object</a></code> elements that have a non-empty
   <code>name</code> content attribute and are <a id=dom-tree-accessors:in-a-document-tree href=https://dom.spec.whatwg.org/#in-a-document-tree data-x-internal=in-a-document-tree>in a document tree</a> with
   <var>document</var> as their <a id=dom-tree-accessors:root href=https://dom.spec.whatwg.org/#concept-tree-root data-x-internal=root>root</a>;<p><li><p>the value of the <code id=dom-tree-accessors:the-id-attribute-2><a href=#the-id-attribute>id</a></code> content attribute for all
   <a href=#exposed id=dom-tree-accessors:exposed-3>exposed</a> <code id=dom-tree-accessors:the-object-element-2><a href=#the-object-element>object</a></code> elements that have a non-empty
   <code id=dom-tree-accessors:the-id-attribute-3><a href=#the-id-attribute>id</a></code> content attribute and are <a id=dom-tree-accessors:in-a-document-tree-2 href=https://dom.spec.whatwg.org/#in-a-document-tree data-x-internal=in-a-document-tree>in a document tree</a> with
   <var>document</var> as their <a id=dom-tree-accessors:root-2 href=https://dom.spec.whatwg.org/#concept-tree-root data-x-internal=root>root</a>; and<li><p>the value of the <code id=dom-tree-accessors:the-id-attribute-4><a href=#the-id-attribute>id</a></code> content attribute for all
   <code id=dom-tree-accessors:the-img-element-4><a href=#the-img-element>img</a></code> elements that have both a non-empty <code id=dom-tree-accessors:the-id-attribute-5><a href=#the-id-attribute>id</a></code> content
   attribute and a non-empty <code>name</code> content attribute, and are <a id=dom-tree-accessors:in-a-document-tree-3 href=https://dom.spec.whatwg.org/#in-a-document-tree data-x-internal=in-a-document-tree>in a
   document tree</a> with <var>document</var> as their <a id=dom-tree-accessors:root-3 href=https://dom.spec.whatwg.org/#concept-tree-root data-x-internal=root>root</a>.</ul>

  <p id=dom-document-nameditem>To <a id=dom-tree-accessors:determine-the-value-of-a-named-property href=https://heycam.github.io/webidl/#dfn-determine-the-value-of-a-named-property data-x-internal=determine-the-value-of-a-named-property>determine the value of a named property</a>
  <var>name</var> for a <code id=dom-tree-accessors:document-17><a href=#document>Document</a></code>, the user agent must return the value obtained using
  the following steps:</p>

  <ol><li>
    <p>Let <var>elements</var> be the list of <a href=#dom-document-nameditem-filter id=dom-tree-accessors:dom-document-nameditem-filter>named
    elements</a> with the name <var>name</var> that are <a id=dom-tree-accessors:in-a-document-tree-4 href=https://dom.spec.whatwg.org/#in-a-document-tree data-x-internal=in-a-document-tree>in a document tree</a> with the
    <code id=dom-tree-accessors:document-18><a href=#document>Document</a></code> as their <a id=dom-tree-accessors:root-4 href=https://dom.spec.whatwg.org/#concept-tree-root data-x-internal=root>root</a>.</p>

    <p class=note>There will be at least one such element, by definition.</p>
   <li>
    <p>If <var>elements</var> has only one element, and that element is an <code id=dom-tree-accessors:the-iframe-element-2><a href=#the-iframe-element>iframe</a></code>
    element, and that <code id=dom-tree-accessors:the-iframe-element-3><a href=#the-iframe-element>iframe</a></code> element's <a href=#nested-browsing-context id=dom-tree-accessors:nested-browsing-context>nested browsing context</a> is not
    null, then return the <code id=dom-tree-accessors:windowproxy><a href=#windowproxy>WindowProxy</a></code> object of the element's <a href=#nested-browsing-context id=dom-tree-accessors:nested-browsing-context-2>nested browsing
    context</a>.</p>
   <li>
    <p>Otherwise, if <var>elements</var> has only one element, return that element.</p>
   <li>
    <p>Otherwise return an <code id=dom-tree-accessors:htmlcollection-11><a data-x-internal=htmlcollection href=https://dom.spec.whatwg.org/#interface-htmlcollection>HTMLCollection</a></code> rooted at the <code id=dom-tree-accessors:document-19><a href=#document>Document</a></code> node,
    whose filter matches only <a href=#dom-document-nameditem-filter id=dom-tree-accessors:dom-document-nameditem-filter-2>named elements</a> with
    the name <var>name</var>.</p> 
   </ol>

  <p><dfn id=dom-document-nameditem-filter>Named elements</dfn> with the name <var>name</var>, for the purposes of the above algorithm, are those that are either:</p>

  
  <ul><li><a href=#exposed id=dom-tree-accessors:exposed-4>Exposed</a> <code id=dom-tree-accessors:the-embed-element-4><a href=#the-embed-element>embed</a></code>, <code id=dom-tree-accessors:the-form-element-4><a href=#the-form-element>form</a></code>, <code id=dom-tree-accessors:the-iframe-element-4><a href=#the-iframe-element>iframe</a></code>,
   <code id=dom-tree-accessors:the-img-element-5><a href=#the-img-element>img</a></code>, or <a href=#exposed id=dom-tree-accessors:exposed-5>exposed</a> <code id=dom-tree-accessors:the-object-element-3><a href=#the-object-element>object</a></code> elements that have a <code>name</code> content attribute whose value is <var>name</var>, or<li><a href=#exposed id=dom-tree-accessors:exposed-6>Exposed</a> <code id=dom-tree-accessors:the-object-element-4><a href=#the-object-element>object</a></code> elements that have an <code id=dom-tree-accessors:the-id-attribute-6><a href=#the-id-attribute>id</a></code> content attribute whose value is <var>name</var>, or<li><code id=dom-tree-accessors:the-img-element-6><a href=#the-img-element>img</a></code> elements that have an <code id=dom-tree-accessors:the-id-attribute-7><a href=#the-id-attribute>id</a></code> content attribute
   whose value is <var>name</var>, and that have a non-empty <code>name</code>
   content attribute present also.</ul>

  <p>An <code id=dom-tree-accessors:the-embed-element-5><a href=#the-embed-element>embed</a></code> or <code id=dom-tree-accessors:the-object-element-5><a href=#the-object-element>object</a></code> element is said to be <dfn id=exposed>exposed</dfn> if it has
  no <a href=#exposed id=dom-tree-accessors:exposed-7>exposed</a> <code id=dom-tree-accessors:the-object-element-6><a href=#the-object-element>object</a></code> ancestor, and, for <code id=dom-tree-accessors:the-object-element-7><a href=#the-object-element>object</a></code> elements, is
  additionally either not showing its <a href=#fallback-content id=dom-tree-accessors:fallback-content>fallback content</a> or has no <code id=dom-tree-accessors:the-object-element-8><a href=#the-object-element>object</a></code> or
  <code id=dom-tree-accessors:the-embed-element-6><a href=#the-embed-element>embed</a></code> descendants.</p>

  

  <hr>

  <p class=note>The <code id=dom-tree-accessors:dom-document-dir><a href=#dom-document-dir>dir</a></code> attribute on the
  <code id=dom-tree-accessors:document-20><a href=#document>Document</a></code> interface is defined along with the <code id=dom-tree-accessors:attr-dir><a href=#attr-dir>dir</a></code>
  content attribute.</p>



  <h3 id=elements><span class=secno>3.2</span> Elements<a href=#elements class=self-link></a></h3>

  <h4 id=semantics-2><span class=secno>3.2.1</span> Semantics<a href=#semantics-2 class=self-link></a></h4>

  <p>Elements, attributes, and attribute values in HTML are defined (by this specification) to have
  certain meanings (semantics). For example, the <code id=semantics-2:the-ol-element><a href=#the-ol-element>ol</a></code> element represents an ordered list,
  and the <code id=semantics-2:attr-lang><a href=#attr-lang>lang</a></code> attribute represents the language of the content.</p>

  <p>These definitions allow HTML processors, such as web browsers or search engines, to present and
  use documents and applications in a wide variety of contexts that the author might not have
  considered.</p>

  <div class=example>

   <p>As a simple example, consider a web page written by an author who only considered desktop
   computer web browsers:</p>

   <pre><code class='html'><c- cp>&lt;!DOCTYPE HTML&gt;</c->
<c- p>&lt;</c-><c- f>html</c-> <c- e>lang</c-><c- o>=</c-><c- s>&quot;en&quot;</c-><c- p>&gt;</c->
 <c- p>&lt;</c-><c- f>head</c-><c- p>&gt;</c->
  <c- p>&lt;</c-><c- f>title</c-><c- p>&gt;</c->My Page<c- p>&lt;/</c-><c- f>title</c-><c- p>&gt;</c->
 <c- p>&lt;/</c-><c- f>head</c-><c- p>&gt;</c->
 <c- p>&lt;</c-><c- f>body</c-><c- p>&gt;</c->
  <c- p>&lt;</c-><c- f>h1</c-><c- p>&gt;</c->Welcome to my page<c- p>&lt;/</c-><c- f>h1</c-><c- p>&gt;</c->
  <c- p>&lt;</c-><c- f>p</c-><c- p>&gt;</c->I like cars and lorries and have a big Jeep!<c- p>&lt;/</c-><c- f>p</c-><c- p>&gt;</c->
  <c- p>&lt;</c-><c- f>h2</c-><c- p>&gt;</c->Where I live<c- p>&lt;/</c-><c- f>h2</c-><c- p>&gt;</c->
  <c- p>&lt;</c-><c- f>p</c-><c- p>&gt;</c->I live in a small hut on a mountain!<c- p>&lt;/</c-><c- f>p</c-><c- p>&gt;</c->
 <c- p>&lt;/</c-><c- f>body</c-><c- p>&gt;</c->
<c- p>&lt;/</c-><c- f>html</c-><c- p>&gt;</c-></code></pre>

   <p>Because HTML conveys <em>meaning</em>, rather than presentation, the same
   page can also be used by a small browser on a mobile phone, without any change to the page.
   Instead of headings being in large letters as on the desktop, for example, the browser on the
   mobile phone might use the same size text for the whole page, but with the headings in bold.</p>

   <p>But it goes further than just differences in screen size: the same page could equally be used
   by a blind user using a browser based around speech synthesis, which instead of displaying the
   page on a screen, reads the page to the user, e.g. using headphones. Instead of large text for
   the headings, the speech browser might use a different volume or a slower voice.</p>

   <p>That's not all, either. Since the browsers know which parts of the page are the headings, they
   can create a document outline that the user can use to quickly navigate around the document,
   using keys for "jump to next heading" or "jump to previous heading". Such features are especially
   common with speech browsers, where users would otherwise find quickly navigating a page quite
   difficult.</p>

   <p>Even beyond browsers, software can make use of this information. Search engines can use the
   headings to more effectively index a page, or to provide quick links to subsections of the page
   from their results. Tools can use the headings to create a table of contents (that is in fact how
   this very specification's table of contents is generated).</p>

   <p>This example has focused on headings, but the same principle applies to all of the semantics
   in HTML.</p>

  </div>

  <p>Authors must not use elements, attributes, or attribute values for purposes other than their
  appropriate intended semantic purpose, as doing so prevents software from correctly processing the
  page.</p>

  <div class=example>

   <p>For example, the following snippet, intended to represent the heading of a
   corporate site, is non-conforming because the second line is not intended to
   be a heading of a subsection, but merely a subheading or subtitle (a
   subordinate heading for the same section).</p>

   <pre class=bad><code class='html'><c- p>&lt;</c-><c- f>body</c-><c- p>&gt;</c->
 <c- p>&lt;</c-><c- f>h1</c-><c- p>&gt;</c->ACME Corporation<c- p>&lt;/</c-><c- f>h1</c-><c- p>&gt;</c->
 <c- p>&lt;</c-><c- f>h2</c-><c- p>&gt;</c->The leaders in arbitrary fast delivery since 1920<c- p>&lt;/</c-><c- f>h2</c-><c- p>&gt;</c->
 ...</code></pre>

   <p>The <code id=semantics-2:the-hgroup-element><a href=#the-hgroup-element>hgroup</a></code> element is intended for these kinds of situations:</p>

   <pre><code class='html'><c- p>&lt;</c-><c- f>body</c-><c- p>&gt;</c->
 <c- p>&lt;</c-><c- f>hgroup</c-><c- p>&gt;</c->
  <c- p>&lt;</c-><c- f>h1</c-><c- p>&gt;</c->ACME Corporation<c- p>&lt;/</c-><c- f>h1</c-><c- p>&gt;</c->
  <c- p>&lt;</c-><c- f>h2</c-><c- p>&gt;</c->The leaders in arbitrary fast delivery since 1920<c- p>&lt;/</c-><c- f>h2</c-><c- p>&gt;</c->
 <c- p>&lt;/</c-><c- f>hgroup</c-><c- p>&gt;</c->
 ...</code></pre>

  </div>

  <div class=example>

   <p>The document in this next example is similarly non-conforming, despite
   being syntactically correct, because the data placed in the cells is clearly
   not tabular data, and the <code id=semantics-2:the-cite-element><a href=#the-cite-element>cite</a></code> element mis-used:</p>

   <pre lang=en-GB class=bad><code class='html'><c- cp>&lt;!DOCTYPE HTML&gt;</c->
<c- p>&lt;</c-><c- f>html</c-> <c- e>lang</c-><c- o>=</c-><c- s>&quot;en-GB&quot;</c-><c- p>&gt;</c->
 <c- p>&lt;</c-><c- f>head</c-><c- p>&gt;</c-> <c- p>&lt;</c-><c- f>title</c-><c- p>&gt;</c-> Demonstration <c- p>&lt;/</c-><c- f>title</c-><c- p>&gt;</c-> <c- p>&lt;/</c-><c- f>head</c-><c- p>&gt;</c->
 <c- p>&lt;</c-><c- f>body</c-><c- p>&gt;</c->
  <c- p>&lt;</c-><c- f>table</c-><c- p>&gt;</c->
   <c- p>&lt;</c-><c- f>tr</c-><c- p>&gt;</c-> <c- p>&lt;</c-><c- f>td</c-><c- p>&gt;</c-> My favourite animal is the cat. <c- p>&lt;/</c-><c- f>td</c-><c- p>&gt;</c-> <c- p>&lt;/</c-><c- f>tr</c-><c- p>&gt;</c->
   <c- p>&lt;</c-><c- f>tr</c-><c- p>&gt;</c->
    <c- p>&lt;</c-><c- f>td</c-><c- p>&gt;</c->
     —<c- p>&lt;</c-><c- f>a</c-> <c- e>href</c-><c- o>=</c-><c- s>&quot;https://example.org/~ernest/&quot;</c-><c- p>&gt;&lt;</c-><c- f>cite</c-><c- p>&gt;</c->Ernest<c- p>&lt;/</c-><c- f>cite</c-><c- p>&gt;&lt;/</c-><c- f>a</c-><c- p>&gt;</c->,
     in an essay from 1992
    <c- p>&lt;/</c-><c- f>td</c-><c- p>&gt;</c->
   <c- p>&lt;/</c-><c- f>tr</c-><c- p>&gt;</c->
  <c- p>&lt;/</c-><c- f>table</c-><c- p>&gt;</c->
 <c- p>&lt;/</c-><c- f>body</c-><c- p>&gt;</c->
<c- p>&lt;/</c-><c- f>html</c-><c- p>&gt;</c-></code></pre>

   <p>This would make software that relies on these semantics fail: for example,
   a speech browser that allowed a blind user to navigate tables in the document
   would report the quote above as a table, confusing the user; similarly, a
   tool that extracted titles of works from pages would extract "Ernest" as the
   title of a work, even though it's actually a person's name, not a title.</p>

   <p>A corrected version of this document might be:</p>

   <pre lang=en-GB><code class='html'><c- cp>&lt;!DOCTYPE HTML&gt;</c->
<c- p>&lt;</c-><c- f>html</c-> <c- e>lang</c-><c- o>=</c-><c- s>&quot;en-GB&quot;</c-><c- p>&gt;</c->
 <c- p>&lt;</c-><c- f>head</c-><c- p>&gt;</c-> <c- p>&lt;</c-><c- f>title</c-><c- p>&gt;</c-> Demonstration <c- p>&lt;/</c-><c- f>title</c-><c- p>&gt;</c-> <c- p>&lt;/</c-><c- f>head</c-><c- p>&gt;</c->
 <c- p>&lt;</c-><c- f>body</c-><c- p>&gt;</c->
  <c- p>&lt;</c-><c- f>blockquote</c-><c- p>&gt;</c->
   <c- p>&lt;</c-><c- f>p</c-><c- p>&gt;</c-> My favourite animal is the cat. <c- p>&lt;/</c-><c- f>p</c-><c- p>&gt;</c->
  <c- p>&lt;/</c-><c- f>blockquote</c-><c- p>&gt;</c->
  <c- p>&lt;</c-><c- f>p</c-><c- p>&gt;</c->
   —<c- p>&lt;</c-><c- f>a</c-> <c- e>href</c-><c- o>=</c-><c- s>&quot;https://example.org/~ernest/&quot;</c-><c- p>&gt;</c->Ernest<c- p>&lt;/</c-><c- f>a</c-><c- p>&gt;</c->,
   in an essay from 1992
  <c- p>&lt;/</c-><c- f>p</c-><c- p>&gt;</c->
 <c- p>&lt;/</c-><c- f>body</c-><c- p>&gt;</c->
<c- p>&lt;/</c-><c- f>html</c-><c- p>&gt;</c-></code></pre>

  </div>

  <p>Authors must not use elements, attributes, or attribute values that are not permitted by this
  specification or <a href=#other-applicable-specifications id=semantics-2:other-applicable-specifications>other applicable specifications</a>, as doing so makes it significantly
  harder for the language to be extended in the future.</p>

  <div class=example>

   <p>In the next example, there is a non-conforming attribute value ("carpet") and a non-conforming
   attribute ("texture"), which is not permitted by this specification:</p>

   <pre class=bad><code class='html'><c- p>&lt;</c-><c- f>label</c-><c- p>&gt;</c->Carpet: <c- p>&lt;</c-><c- f>input</c-> <c- e>type</c-><c- o>=</c-><c- s>&quot;carpet&quot;</c-> <c- e>name</c-><c- o>=</c-><c- s>&quot;c&quot;</c-> <c- e>texture</c-><c- o>=</c-><c- s>&quot;deep pile&quot;</c-><c- p>&gt;&lt;/</c-><c- f>label</c-><c- p>&gt;</c-></code></pre>

   <p>Here would be an alternative and correct way to mark this up:</p>

   <pre><code class='html'><c- p>&lt;</c-><c- f>label</c-><c- p>&gt;</c->Carpet: <c- p>&lt;</c-><c- f>input</c-> <c- e>type</c-><c- o>=</c-><c- s>&quot;text&quot;</c-> <c- e>class</c-><c- o>=</c-><c- s>&quot;carpet&quot;</c-> <c- e>name</c-><c- o>=</c-><c- s>&quot;c&quot;</c-> <c- e>data-texture</c-><c- o>=</c-><c- s>&quot;deep pile&quot;</c-><c- p>&gt;&lt;/</c-><c- f>label</c-><c- p>&gt;</c-></code></pre>

  </div>

  <p id=no-browsing-context>DOM nodes whose <a id=semantics-2:node-document href=https://dom.spec.whatwg.org/#concept-node-document data-x-internal=node-document>node document</a>'s <a href=#concept-document-bc id=semantics-2:concept-document-bc>browsing context</a> is null are exempt from all document
  conformance requirements other than the <a href=#writing>HTML syntax</a> requirements and <a href=#writing-xhtml-documents>XML syntax</a> requirements.</p>

  <div class=example>
   <p>In particular, the <code id=semantics-2:the-template-element><a href=#the-template-element>template</a></code> element's <a href=#template-contents id=semantics-2:template-contents>template contents</a>'s <a id=semantics-2:node-document-2 href=https://dom.spec.whatwg.org/#concept-node-document data-x-internal=node-document>node
   document</a>'s <a href=#concept-document-bc id=semantics-2:concept-document-bc-2>browsing context</a> is null. For
   example, the <a href=#concept-element-content-model id=semantics-2:concept-element-content-model>content model</a> requirements and
   attribute value microsyntax requirements do not apply to a <code id=semantics-2:the-template-element-2><a href=#the-template-element>template</a></code> element's
   <a href=#template-contents id=semantics-2:template-contents-2>template contents</a>. In this example an <code id=semantics-2:the-img-element><a href=#the-img-element>img</a></code> element has attribute values
   that are placeholders that would be invalid outside a <code id=semantics-2:the-template-element-3><a href=#the-template-element>template</a></code> element.</p>

   <pre><code class='html'><c- p>&lt;</c-><c- f>template</c-><c- p>&gt;</c->
 <c- p>&lt;</c-><c- f>article</c-><c- p>&gt;</c->
  <c- p>&lt;</c-><c- f>img</c-> <mark><c- e>src</c-><c- o>=</c-><c- s>&quot;{{src}}&quot;</c-> <c- e>alt</c-><c- o>=</c-><c- s>&quot;{{alt}}&quot;</c-></mark><c- p>&gt;</c->
  <c- p>&lt;</c-><c- f>h1</c-><c- p>&gt;&lt;/</c-><c- f>h1</c-><c- p>&gt;</c->
 <c- p>&lt;/</c-><c- f>article</c-><c- p>&gt;</c->
<c- p>&lt;/</c-><c- f>template</c-><c- p>&gt;</c-></code></pre>

   <p>However, if the above markup were to omit the <code>&lt;/h1></code> end tag, that
   would be a violation of the <a href=#writing>HTML syntax</a>, and would thus be flagged as an
   error by conformance checkers.</p>
  </div>

  <p>Through scripting and using other mechanisms, the values of attributes, text, and indeed the
  entire structure of the document may change dynamically while a user agent is processing it. The
  semantics of a document at an instant in time are those represented by the state of the document
  at that instant in time, and the semantics of a document can therefore change over time. User
  agents must update their presentation of the document as this
  occurs.</p>

  <p class=example>HTML has a <code id=semantics-2:the-progress-element><a href=#the-progress-element>progress</a></code> element that describes a progress bar. If its
  "value" attribute is dynamically updated by a script, the UA would update the rendering to show
  the progress changing.</p>



  <h4 id=elements-in-the-dom><span class=secno>3.2.2</span> Elements in the DOM<a href=#elements-in-the-dom class=self-link></a></h4>

  <p>The nodes representing <a href=#html-elements id=elements-in-the-dom:html-elements>HTML elements</a> in the DOM must
  implement, and expose to scripts, the interfaces listed for them in the relevant sections of this
  specification. This includes <a href=#html-elements id=elements-in-the-dom:html-elements-2>HTML elements</a> in <a id=elements-in-the-dom:xml-documents href=https://dom.spec.whatwg.org/#xml-document data-x-internal=xml-documents>XML documents</a>, even when
  those documents are in another context (e.g. inside an XSLT transform).</p>

  <p>Elements in the DOM <dfn id=represents>represent</dfn> things; that is, they have
  intrinsic <em>meaning</em>, also known as semantics.</p>

  <p class=example>For example, an <code id=elements-in-the-dom:the-ol-element><a href=#the-ol-element>ol</a></code> element represents an ordered list.</p>

  <p>Elements can be <dfn id=referenced>referenced</dfn> (referred to) in some way, either
  explicitly or implicitly. One way that an element in the DOM can be explicitly referenced is by
  giving an <code id=elements-in-the-dom:the-id-attribute><a href=#the-id-attribute>id</a></code> attribute to the element, and then creating a
  <a href=#hyperlink id=elements-in-the-dom:hyperlink>hyperlink</a> with that <code id=elements-in-the-dom:the-id-attribute-2><a href=#the-id-attribute>id</a></code> attribute's value as the <a href=#navigate-fragid id=elements-in-the-dom:navigate-fragid>fragment</a> for the <a href=#hyperlink id=elements-in-the-dom:hyperlink-2>hyperlink</a>'s <code id=elements-in-the-dom:attr-hyperlink-href><a href=#attr-hyperlink-href>href</a></code> attribute value. Hyperlinks are not necessary for a
  reference, however; any manner of referring to the element in question will suffice.</p>

  <div class=example>
   <p>Consider the following <code id=elements-in-the-dom:the-figure-element><a href=#the-figure-element>figure</a></code> element, which is given an <code id=elements-in-the-dom:the-id-attribute-3><a href=#the-id-attribute>id</a></code> attribute:</p>

   <pre><code class='html'><c- p>&lt;</c-><c- f>figure</c-> <c- e>id</c-><c- o>=</c-><c- s>&quot;module-script-graph&quot;</c-><c- p>&gt;</c->
  <c- p>&lt;</c-><c- f>img</c-> <c- e>src</c-><c- o>=</c-><c- s>&quot;module-script-graph.svg&quot;</c->
       <c- e>alt</c-><c- o>=</c-><c- s>&quot;Module A depends on module B, which depends</c->
<c- s>            on modules C and D.&quot;</c-><c- p>&gt;</c->
  <c- p>&lt;</c-><c- f>figcaption</c-><c- p>&gt;</c->Figure 27: a simple module graph<c- p>&lt;/</c-><c- f>figcaption</c-><c- p>&gt;</c->
<c- p>&lt;/</c-><c- f>figure</c-><c- p>&gt;</c-></code></pre>

   <p>A <a href=#hyperlink id=elements-in-the-dom:hyperlink-3>hyperlink</a>-based <a href=#referenced id=elements-in-the-dom:referenced>reference</a> could be created
   using the <code id=elements-in-the-dom:the-a-element><a href=#the-a-element>a</a></code> element, like so:</p>

   <pre><code class='html'>As we can see in <c- p>&lt;</c-><c- f>a</c-> <c- e>href</c-><c- o>=</c-><c- s>&quot;#module-script-graph&quot;</c-><c- p>&gt;</c->figure 27<c- p>&lt;/</c-><c- f>a</c-><c- p>&gt;</c->, ...</code></pre>

   <p>However, there are many other ways of <a href=#referenced id=elements-in-the-dom:referenced-2>referencing</a> the
   <code id=elements-in-the-dom:the-figure-element-2><a href=#the-figure-element>figure</a></code> element, such as:</p>

   <ul><li><p>"As depicted in the figure of modules A, B, C, and D..."<li><p>"In Figure 27..." (without a hyperlink)<li><p>"From the contents of the 'simple module graph' figure..."<li><p>"In the figure below..." (but <a href=#figure-note-about-references>this is
    discouraged</a>)</ul>
  </div>

  <p>The basic interface, from which all the <a href=#html-elements id=elements-in-the-dom:html-elements-3>HTML elements</a>' interfaces inherit, and which must be used by elements that have no additional requirements, is
  the <code id=elements-in-the-dom:htmlelement><a href=#htmlelement>HTMLElement</a></code> interface.</p>

  <div class="mdn-anno wrapped before"><button onclick=toggleStatus(this) class=mdn-anno-btn><b title="Support in all current engines." class=all-engines-flag>✔</b><span>MDN</span></button><div class=feature><p><a href=https://developer.mozilla.org/en-US/docs/Web/API/HTMLElement title="The HTMLElement interface represents any HTML element. Some elements directly implement this interface, while others implement it via an interface that inherits it.">HTMLElement</a><p class=all-engines-text>Support in all current engines.<div class=support><span class="firefox yes"><span>Firefox</span><span>1+</span></span><span class="safari yes"><span>Safari</span><span>3+</span></span><span class="chrome yes"><span>Chrome</span><span>1+</span></span><hr><span class="opera yes"><span>Opera</span><span>8+</span></span><span class="edge_blink yes"><span>Edge</span><span>79+</span></span><hr><span class="edge yes"><span>Edge (Legacy)</span><span>12+</span></span><span class="ie yes"><span>Internet Explorer</span><span>5.5+</span></span><hr><span class="firefox_android yes"><span>Firefox Android</span><span>4+</span></span><span class="safari_ios yes"><span>Safari iOS</span><span>1+</span></span><span class="chrome_android yes"><span>Chrome Android</span><span>18+</span></span><span class="webview_android yes"><span>WebView Android</span><span>1+</span></span><span class="samsunginternet_android yes"><span>Samsung Internet</span><span>1.0+</span></span><span class="opera_android yes"><span>Opera Android</span><span>10.1+</span></span></div></div><div class=feature><p><a href=https://developer.mozilla.org/en-US/docs/Web/API/HTMLUnknownElement title="The HTMLUnknownElement interface represents an invalid HTML element and derives from the HTMLElement interface, but without implementing any additional properties or methods.">HTMLUnknownElement</a><p class=all-engines-text>Support in all current engines.<div class=support><span class="firefox yes"><span>Firefox</span><span>Yes</span></span><span class="safari yes"><span>Safari</span><span>Yes</span></span><span class="chrome yes"><span>Chrome</span><span>Yes</span></span><hr><span class="opera yes"><span>Opera</span><span>Yes</span></span><span class="edge_blink yes"><span>Edge</span><span>Yes</span></span><hr><span class="edge yes"><span>Edge (Legacy)</span><span>12+</span></span><span class="ie yes"><span>Internet Explorer</span><span>Yes</span></span><hr><span class="firefox_android yes"><span>Firefox Android</span><span>Yes</span></span><span class="safari_ios yes"><span>Safari iOS</span><span>Yes</span></span><span class="chrome_android yes"><span>Chrome Android</span><span>Yes</span></span><span class="webview_android yes"><span>WebView Android</span><span>Yes</span></span><span class="samsunginternet_android yes"><span>Samsung Internet</span><span>Yes</span></span><span class="opera_android yes"><span>Opera Android</span><span>Yes</span></span></div></div></div><pre><code class='idl'>[<c- g>Exposed</c->=<c- n>Window</c->]
<c- b>interface</c-> <dfn id='htmlelement'><c- g>HTMLElement</c-></dfn> : <a id='elements-in-the-dom:element' href='https://dom.spec.whatwg.org/#interface-element' data-x-internal='element'><c- n>Element</c-></a> {
  [<a href='#htmlconstructor' id='elements-in-the-dom:htmlconstructor'><c- g>HTMLConstructor</c-></a>] <c- g>constructor</c->();

  // metadata attributes
  [<a href='#cereactions' id='elements-in-the-dom:cereactions'><c- g>CEReactions</c-></a>] <c- b>attribute</c-> <c- b>DOMString</c-> <a href='#dom-title' id='elements-in-the-dom:dom-title'><c- g>title</c-></a>;
  [<a href='#cereactions' id='elements-in-the-dom:cereactions-2'><c- g>CEReactions</c-></a>] <c- b>attribute</c-> <c- b>DOMString</c-> <a href='#dom-lang' id='elements-in-the-dom:dom-lang'><c- g>lang</c-></a>;
  [<a href='#cereactions' id='elements-in-the-dom:cereactions-3'><c- g>CEReactions</c-></a>] <c- b>attribute</c-> <c- b>boolean</c-> <a href='#dom-translate' id='elements-in-the-dom:dom-translate'><c- g>translate</c-></a>;
  [<a href='#cereactions' id='elements-in-the-dom:cereactions-4'><c- g>CEReactions</c-></a>] <c- b>attribute</c-> <c- b>DOMString</c-> <a href='#dom-dir' id='elements-in-the-dom:dom-dir'><c- g>dir</c-></a>;

  // <a href='#editing' id='elements-in-the-dom:editing'>user interaction</a>
  [<a href='#cereactions' id='elements-in-the-dom:cereactions-5'><c- g>CEReactions</c-></a>] <c- b>attribute</c-> <c- b>boolean</c-> <a href='#dom-hidden' id='elements-in-the-dom:dom-hidden'><c- g>hidden</c-></a>;
  <c- b>undefined</c-> <a href='#dom-click' id='elements-in-the-dom:dom-click'><c- g>click</c-></a>();
  [<a href='#cereactions' id='elements-in-the-dom:cereactions-6'><c- g>CEReactions</c-></a>] <c- b>attribute</c-> <c- b>DOMString</c-> <a href='#dom-accesskey' id='elements-in-the-dom:dom-accesskey'><c- g>accessKey</c-></a>;
  <c- b>readonly</c-> <c- b>attribute</c-> <c- b>DOMString</c-> <a href='#dom-accesskeylabel' id='elements-in-the-dom:dom-accesskeylabel'><c- g>accessKeyLabel</c-></a>;
  [<a href='#cereactions' id='elements-in-the-dom:cereactions-7'><c- g>CEReactions</c-></a>] <c- b>attribute</c-> <c- b>boolean</c-> <a href='#dom-draggable' id='elements-in-the-dom:dom-draggable'><c- g>draggable</c-></a>;
  [<a href='#cereactions' id='elements-in-the-dom:cereactions-8'><c- g>CEReactions</c-></a>] <c- b>attribute</c-> <c- b>boolean</c-> <a href='#dom-spellcheck' id='elements-in-the-dom:dom-spellcheck'><c- g>spellcheck</c-></a>;
  [<a href='#cereactions' id='elements-in-the-dom:cereactions-9'><c- g>CEReactions</c-></a>] <c- b>attribute</c-> <c- b>DOMString</c-> <a href='#dom-autocapitalize' id='elements-in-the-dom:dom-autocapitalize'><c- g>autocapitalize</c-></a>;

  [<a href='#cereactions' id='elements-in-the-dom:cereactions-10'><c- g>CEReactions</c-></a>] <c- b>attribute</c-> [<a id='elements-in-the-dom:legacynulltoemptystring' href='https://heycam.github.io/webidl/#LegacyNullToEmptyString' data-x-internal='legacynulltoemptystring'><c- g>LegacyNullToEmptyString</c-></a>] <c- b>DOMString</c-> <a href='#dom-innertext' id='elements-in-the-dom:dom-innertext'><c- g>innerText</c-></a>;

  <a href='#elementinternals' id='elements-in-the-dom:elementinternals'><c- n>ElementInternals</c-></a> <a href='#dom-attachinternals' id='elements-in-the-dom:dom-attachinternals'><c- g>attachInternals</c-></a>();
};

<a href='#htmlelement' id='elements-in-the-dom:htmlelement-2'><c- n>HTMLElement</c-></a> <c- b>includes</c-> <a href='#globaleventhandlers' id='elements-in-the-dom:globaleventhandlers'><c- n>GlobalEventHandlers</c-></a>;
<a href='#htmlelement' id='elements-in-the-dom:htmlelement-3'><c- n>HTMLElement</c-></a> <c- b>includes</c-> <a href='#documentandelementeventhandlers' id='elements-in-the-dom:documentandelementeventhandlers'><c- n>DocumentAndElementEventHandlers</c-></a>;
<a href='#htmlelement' id='elements-in-the-dom:htmlelement-4'><c- n>HTMLElement</c-></a> <c- b>includes</c-> <a href='#elementcontenteditable' id='elements-in-the-dom:elementcontenteditable'><c- n>ElementContentEditable</c-></a>;
<a href='#htmlelement' id='elements-in-the-dom:htmlelement-5'><c- n>HTMLElement</c-></a> <c- b>includes</c-> <a href='#htmlorsvgelement' id='elements-in-the-dom:htmlorsvgelement'><c- n>HTMLOrSVGElement</c-></a>;

[<c- g>Exposed</c->=<c- n>Window</c->]
<c- b>interface</c-> <dfn id='htmlunknownelement'><c- g>HTMLUnknownElement</c-></dfn> : <a href='#htmlelement' id='elements-in-the-dom:htmlelement-6'><c- n>HTMLElement</c-></a> {
  // Note: <a href='#customized-built-in-element-restrictions'>intentionally</a> no [<a href='#htmlconstructor' id='elements-in-the-dom:htmlconstructor-2'>HTMLConstructor</a>]
};</code></pre>

  <p>The <code id=elements-in-the-dom:htmlelement-7><a href=#htmlelement>HTMLElement</a></code> interface holds methods and attributes related to a number of
  disparate features, and the members of this interface are therefore described in various different
  sections of this specification.</p>

  

  <hr>

  <p>The <a id=elements-in-the-dom:element-interface href=https://dom.spec.whatwg.org/#concept-element-interface data-x-internal=element-interface>element interface</a> for an element with name <var>name</var> in the
  <a id=elements-in-the-dom:html-namespace-2 href=https://infra.spec.whatwg.org/#html-namespace data-x-internal=html-namespace-2>HTML namespace</a> is determined as follows:</p>

  <ol><li><p>If <var>name</var> is <code id=elements-in-the-dom:applet><a href=#applet>applet</a></code>, <code id=elements-in-the-dom:bgsound><a href=#bgsound>bgsound</a></code>, <code id=elements-in-the-dom:blink><a href=#blink>blink</a></code>,
   <code id=elements-in-the-dom:isindex><a href=#isindex>isindex</a></code>, <code id=elements-in-the-dom:keygen><a href=#keygen>keygen</a></code>, <code id=elements-in-the-dom:multicol><a href=#multicol>multicol</a></code>, <code id=elements-in-the-dom:nextid><a href=#nextid>nextid</a></code>, or
   <code id=elements-in-the-dom:spacer><a href=#spacer>spacer</a></code>, then return <code id=elements-in-the-dom:htmlunknownelement><a href=#htmlunknownelement>HTMLUnknownElement</a></code>.<li><p>If <var>name</var> is <code id=elements-in-the-dom:acronym><a href=#acronym>acronym</a></code>, <code id=elements-in-the-dom:basefont><a href=#basefont>basefont</a></code>, <code id=elements-in-the-dom:big><a href=#big>big</a></code>,
   <code id=elements-in-the-dom:center><a href=#center>center</a></code>, <code id=elements-in-the-dom:nobr><a href=#nobr>nobr</a></code>, <code id=elements-in-the-dom:noembed><a href=#noembed>noembed</a></code>, <code id=elements-in-the-dom:noframes><a href=#noframes>noframes</a></code>,
   <code id=elements-in-the-dom:plaintext><a href=#plaintext>plaintext</a></code>, <code id=elements-in-the-dom:rb><a href=#rb>rb</a></code>, <code id=elements-in-the-dom:rtc><a href=#rtc>rtc</a></code>, <code id=elements-in-the-dom:strike><a href=#strike>strike</a></code>, or
   <code id=elements-in-the-dom:tt><a href=#tt>tt</a></code>, then return <code id=elements-in-the-dom:htmlelement-8><a href=#htmlelement>HTMLElement</a></code>.<li><p>If <var>name</var> is <code id=elements-in-the-dom:listing><a href=#listing>listing</a></code> or <code id=elements-in-the-dom:xmp><a href=#xmp>xmp</a></code>, then return
   <code id=elements-in-the-dom:htmlpreelement><a href=#htmlpreelement>HTMLPreElement</a></code>.<li><p>Otherwise, if this specification defines an interface appropriate for the <a href=#element-type id=elements-in-the-dom:element-type>element
   type</a> corresponding to the local name <var>name</var>, then return that interface.<li><p>If <a href=#other-applicable-specifications id=elements-in-the-dom:other-applicable-specifications>other applicable specifications</a> define an appropriate interface for
   <var>name</var>, then return the interface they define.<li><p>If <var>name</var> is a <a href=#valid-custom-element-name id=elements-in-the-dom:valid-custom-element-name>valid custom element name</a>, then return
   <code id=elements-in-the-dom:htmlelement-9><a href=#htmlelement>HTMLElement</a></code>.</p>

   <li><p>Return <code id=elements-in-the-dom:htmlunknownelement-2><a href=#htmlunknownelement>HTMLUnknownElement</a></code>.</ol>

  <p class=note>The use of <code id=elements-in-the-dom:htmlelement-10><a href=#htmlelement>HTMLElement</a></code> instead of <code id=elements-in-the-dom:htmlunknownelement-3><a href=#htmlunknownelement>HTMLUnknownElement</a></code> in
  the case of <a href=#valid-custom-element-name id=elements-in-the-dom:valid-custom-element-name-2>valid custom element names</a> is done to
  ensure that any potential future <a href=#upgrades id=elements-in-the-dom:upgrades>upgrades</a> only cause
  a linear transition of the element's prototype chain, from <code id=elements-in-the-dom:htmlelement-11><a href=#htmlelement>HTMLElement</a></code> to a subclass,
  instead of a lateral one, from <code id=elements-in-the-dom:htmlunknownelement-4><a href=#htmlunknownelement>HTMLUnknownElement</a></code> to an unrelated subclass.</p>

  <p>Features shared between HTML and SVG elements use the <code id=elements-in-the-dom:htmlorsvgelement-2><a href=#htmlorsvgelement>HTMLOrSVGElement</a></code> interface
  mixin: <a href=#refsSVG>[SVG]</a></p>

  <pre><code class='idl'><c- b>interface</c-> <c- b>mixin</c-> <dfn id='htmlorsvgelement'><c- g>HTMLOrSVGElement</c-></dfn> {
  [<c- g>SameObject</c->] <c- b>readonly</c-> <c- b>attribute</c-> <a href='#domstringmap' id='elements-in-the-dom:domstringmap'><c- n>DOMStringMap</c-></a> <a href='#dom-dataset' id='elements-in-the-dom:dom-dataset'><c- g>dataset</c-></a>;
  <c- b>attribute</c-> <c- b>DOMString</c-> <a href='#dom-noncedelement-nonce' id='elements-in-the-dom:dom-noncedelement-nonce'><c- g>nonce</c-></a>; // <a href='#nonce-does-not-update-dom'>intentionally no [CEReactions]</a>

  [<a href='#cereactions' id='elements-in-the-dom:cereactions-11'><c- g>CEReactions</c-></a>] <c- b>attribute</c-> <c- b>boolean</c-> <a href='#dom-fe-autofocus' id='elements-in-the-dom:dom-fe-autofocus'><c- g>autofocus</c-></a>;
  [<a href='#cereactions' id='elements-in-the-dom:cereactions-12'><c- g>CEReactions</c-></a>] <c- b>attribute</c-> <c- b>long</c-> <a href='#dom-tabindex' id='elements-in-the-dom:dom-tabindex'><c- g>tabIndex</c-></a>;
  <c- b>undefined</c-> <a href='#dom-focus' id='elements-in-the-dom:dom-focus'><c- g>focus</c-></a>(<c- b>optional</c-> <a href='#focusoptions' id='elements-in-the-dom:focusoptions'><c- n>FocusOptions</c-></a> <c- g>options</c-> = {});
  <c- b>undefined</c-> <a href='#dom-blur' id='elements-in-the-dom:dom-blur'><c- g>blur</c-></a>();
};</code></pre>

  

  <div class=example>
   <p>An example of an element that is neither an HTML nor SVG element is one created as
   follows:</p>

   <pre><code class='html'>const el = document.createElementNS(&quot;some namespace&quot;, &quot;example&quot;);
console.assert(el.constructor === Element);</code></pre>
  </div>

  

  <h4 id=html-element-constructors><span class=secno>3.2.3</span> HTML element constructors<a href=#html-element-constructors class=self-link></a></h4>

  <p>To support the <a href=#custom-elements>custom elements</a> feature, all HTML elements have
  special constructor behavior. This is indicated via the <dfn id=htmlconstructor data-dfn-type=extended-attribute data-lt=HTMLConstructor><code>[HTMLConstructor]</code></dfn> IDL
  <a id=html-element-constructors:extended-attribute href=https://heycam.github.io/webidl/#dfn-extended-attribute data-x-internal=extended-attribute>extended attribute</a>.  It indicates that the interface object for the given interface
  will have a specific behavior when called, as defined in detail below.</p>

  <p>The <code id=html-element-constructors:htmlconstructor><a href=#htmlconstructor>[HTMLConstructor]</a></code> extended attribute must take no
  arguments, and must only appear on <a href=https://heycam.github.io/webidl/#idl-constructors id=html-element-constructors:constructor-operation data-x-internal=constructor-operation>constructor
  operations</a>. It must appear only once on a constructor operation, and the interface must
  contain only the single, annotated constructor operation, and no others. The annotated
  constructor operation must be declared to take no arguments.</p>

  <p>Interfaces declared with constructor operations that are annotated with the <code id=html-element-constructors:htmlconstructor-2><a href=#htmlconstructor>[HTMLConstructor]</a></code> extended attribute have the following
  <a id=html-element-constructors:overridden-constructor-steps href=https://heycam.github.io/webidl/#overridden-constructor-steps data-x-internal=overridden-constructor-steps>overridden constructor steps</a>:</p>

  <ol><li><p>Let <var>registry</var> be the <a href=#current-global-object id=html-element-constructors:current-global-object>current global object</a>'s
   <code id=html-element-constructors:customelementregistry><a href=#customelementregistry>CustomElementRegistry</a></code> object.<li>
    <p>If <a id=html-element-constructors:newtarget href=https://tc39.es/ecma262/#sec-built-in-function-objects data-x-internal=newtarget>NewTarget</a> is equal to the <a id=html-element-constructors:active-function-object href=https://tc39.es/ecma262/#active-function-object data-x-internal=active-function-object>active function
    object</a>, then throw a <code id=html-element-constructors:typeerror><a data-x-internal=typeerror href=https://tc39.es/ecma262/#sec-native-error-types-used-in-this-standard-typeerror>TypeError</a></code>.</p>

    <div class="example no-backref">
     <p>This can occur when a custom element is defined using an <a id=html-element-constructors:element-interface href=https://dom.spec.whatwg.org/#concept-element-interface data-x-internal=element-interface>element interface</a> as
     its constructor:</p>

     <pre><code class='js'>customElements<c- p>.</c->define<c- p>(</c-><c- u>&quot;bad-1&quot;</c-><c- p>,</c-> HTMLButtonElement<c- p>);</c->
<c- k>new</c-> HTMLButtonElement<c- p>();</c->          <c- c1>// (1)</c->
document<c- p>.</c->createElement<c- p>(</c-><c- u>&quot;bad-1&quot;</c-><c- p>);</c->  <c- c1>// (2)</c-></code></pre>

     <p>In this case, during the execution of <code id=html-element-constructors:htmlbuttonelement><a href=#htmlbuttonelement>HTMLButtonElement</a></code> (either explicitly, as
     in (1), or implicitly, as in (2)), both the <a id=html-element-constructors:active-function-object-2 href=https://tc39.es/ecma262/#active-function-object data-x-internal=active-function-object>active function object</a> and
     <a id=html-element-constructors:newtarget-2 href=https://tc39.es/ecma262/#sec-built-in-function-objects data-x-internal=newtarget>NewTarget</a> are <code id=html-element-constructors:htmlbuttonelement-2><a href=#htmlbuttonelement>HTMLButtonElement</a></code>. If this check was not present, it
     would be possible to create an instance of <code id=html-element-constructors:htmlbuttonelement-3><a href=#htmlbuttonelement>HTMLButtonElement</a></code> whose local name was
     <code>bad-1</code>.</p>
    </div>
   <li>
    <p>Let <var>definition</var> be the entry in <var>registry</var> with <a href=#concept-custom-element-definition-constructor id=html-element-constructors:concept-custom-element-definition-constructor>constructor</a> equal to
    <a id=html-element-constructors:newtarget-3 href=https://tc39.es/ecma262/#sec-built-in-function-objects data-x-internal=newtarget>NewTarget</a>. If there is no such definition, then throw a <code id=html-element-constructors:typeerror-2><a data-x-internal=typeerror href=https://tc39.es/ecma262/#sec-native-error-types-used-in-this-standard-typeerror>TypeError</a></code>.</p>

    <p class=note>Since there can be no entry in <var>registry</var> with a <a href=#concept-custom-element-definition-constructor id=html-element-constructors:concept-custom-element-definition-constructor-2>constructor</a> of undefined, this
    step also prevents HTML element constructors from being called as functions (since in that case
    <a id=html-element-constructors:newtarget-4 href=https://tc39.es/ecma262/#sec-built-in-function-objects data-x-internal=newtarget>NewTarget</a> will be undefined).</p>
   <li><p>Let <var>is value</var> be null.<li>
    <p>If <var>definition</var>'s <a href=#concept-custom-element-definition-local-name id=html-element-constructors:concept-custom-element-definition-local-name>local
    name</a> is equal to <var>definition</var>'s <a href=#concept-custom-element-definition-name id=html-element-constructors:concept-custom-element-definition-name>name</a> (i.e., <var>definition</var> is for
    an <a href=#autonomous-custom-element id=html-element-constructors:autonomous-custom-element>autonomous custom element</a>), then:</p>

    <ol><li>
      <p>If the <a id=html-element-constructors:active-function-object-3 href=https://tc39.es/ecma262/#active-function-object data-x-internal=active-function-object>active function object</a> is not <code id=html-element-constructors:htmlelement><a href=#htmlelement>HTMLElement</a></code>, then throw a
      <code id=html-element-constructors:typeerror-3><a data-x-internal=typeerror href=https://tc39.es/ecma262/#sec-native-error-types-used-in-this-standard-typeerror>TypeError</a></code>.</p>

      <div class="example no-backref">
       <p>This can occur when a custom element is defined to not extend any local names, but
       inherits from a non-<code id=html-element-constructors:htmlelement-2><a href=#htmlelement>HTMLElement</a></code> class:</p>

       <pre><code class='js'>customElements<c- p>.</c->define<c- p>(</c-><c- u>&quot;bad-2&quot;</c-><c- p>,</c-> <c- kr>class</c-> Bad2 <c- kr>extends</c-> HTMLParagraphElement <c- p>{});</c-></code></pre>

       <p>In this case, during the (implicit) <code>super()</code> call that occurs when
       constructing an instance of <code>Bad2</code>, the <a id=html-element-constructors:active-function-object-4 href=https://tc39.es/ecma262/#active-function-object data-x-internal=active-function-object>active function
       object</a> is <code id=html-element-constructors:htmlparagraphelement><a href=#htmlparagraphelement>HTMLParagraphElement</a></code>, not <code id=html-element-constructors:htmlelement-3><a href=#htmlelement>HTMLElement</a></code>.</p>
      </div>
     </ol>
   <li>
    <p>Otherwise (i.e., if <var>definition</var> is for a <a href=#customized-built-in-element id=html-element-constructors:customized-built-in-element>customized built-in
    element</a>):</p>

    <ol><li><p>Let <var>valid local names</var> be the list of local names for elements defined in this
     specification or in <a href=#other-applicable-specifications id=html-element-constructors:other-applicable-specifications>other applicable specifications</a> that use the <a id=html-element-constructors:active-function-object-5 href=https://tc39.es/ecma262/#active-function-object data-x-internal=active-function-object>active
     function object</a> as their <a id=html-element-constructors:element-interface-2 href=https://dom.spec.whatwg.org/#concept-element-interface data-x-internal=element-interface>element interface</a>.<li>
      <p>If <var>valid local names</var> does not contain <var>definition</var>'s <a href=#concept-custom-element-definition-local-name id=html-element-constructors:concept-custom-element-definition-local-name-2>local name</a>, then throw a
      <code id=html-element-constructors:typeerror-4><a data-x-internal=typeerror href=https://tc39.es/ecma262/#sec-native-error-types-used-in-this-standard-typeerror>TypeError</a></code>.</p>

      <div class="example no-backref">
       <p>This can occur when a custom element is defined to extend a given local name but inherits
       from the wrong class:</p>

       <pre><code class='js'>customElements<c- p>.</c->define<c- p>(</c-><c- u>&quot;bad-3&quot;</c-><c- p>,</c-> <c- kr>class</c-> Bad3 <c- kr>extends</c-> HTMLQuoteElement <c- p>{},</c-> <c- p>{</c-> <c- kr>extends</c-><c- o>:</c-> <c- u>&quot;p&quot;</c-> <c- p>});</c-></code></pre>

       <p>In this case, during the (implicit) <code>super()</code> call that occurs when
       constructing an instance of <code>Bad3</code>, <var>valid local names</var> is the
       list containing <code id=html-element-constructors:the-q-element><a href=#the-q-element>q</a></code> and <code id=html-element-constructors:the-blockquote-element><a href=#the-blockquote-element>blockquote</a></code>, but <var>definition</var>'s <a href=#concept-custom-element-definition-local-name id=html-element-constructors:concept-custom-element-definition-local-name-3>local name</a> is <code id=html-element-constructors:the-p-element><a href=#the-p-element>p</a></code>,
       which is not in that list.</p>
      </div>
     <li><p>Set <var>is value</var> to <var>definition</var>'s <a href=#concept-custom-element-definition-name id=html-element-constructors:concept-custom-element-definition-name-2>name</a>.</ol>
   <li>
    <p>If <var>definition</var>'s <a href=#concept-custom-element-definition-construction-stack id=html-element-constructors:concept-custom-element-definition-construction-stack>construction stack</a> is
    empty, then:</p>

    <ol><li><p>Let <var>element</var> be the result of <a href=https://heycam.github.io/webidl/#internally-create-a-new-object-implementing-the-interface id=html-element-constructors:internally-create-a-new-object-implementing-the-interface data-x-internal=internally-create-a-new-object-implementing-the-interface>internally creating a new object implementing the interface</a>
     to which the <a id=html-element-constructors:active-function-object-6 href=https://tc39.es/ecma262/#active-function-object data-x-internal=active-function-object>active function object</a> corresponds, given the <a id=html-element-constructors:current-realm-record href=https://tc39.es/ecma262/#current-realm data-x-internal=current-realm-record>current Realm
     Record</a> and <a id=html-element-constructors:newtarget-5 href=https://tc39.es/ecma262/#sec-built-in-function-objects data-x-internal=newtarget>NewTarget</a>.<li><p>Set <var>element</var>'s <a id=html-element-constructors:node-document href=https://dom.spec.whatwg.org/#concept-node-document data-x-internal=node-document>node document</a> to the <a href=#current-global-object id=html-element-constructors:current-global-object-2>current global
     object</a>'s <a href=#concept-document-window id=html-element-constructors:concept-document-window>associated
     <code>Document</code></a>.<li><p>Set <var>element</var>'s <a href=https://dom.spec.whatwg.org/#concept-element-namespace id=html-element-constructors:concept-element-namespace data-x-internal=concept-element-namespace>namespace</a> to
     the <a id=html-element-constructors:html-namespace-2 href=https://infra.spec.whatwg.org/#html-namespace data-x-internal=html-namespace-2>HTML namespace</a>.<li><p>Set <var>element</var>'s <a href=https://dom.spec.whatwg.org/#concept-element-namespace-prefix id=html-element-constructors:concept-element-namespace-prefix data-x-internal=concept-element-namespace-prefix>namespace
     prefix</a> to null.<li><p>Set <var>element</var>'s <a href=https://dom.spec.whatwg.org/#concept-element-local-name id=html-element-constructors:concept-element-local-name data-x-internal=concept-element-local-name>local name</a> to
     <var>definition</var>'s <a href=#concept-custom-element-definition-local-name id=html-element-constructors:concept-custom-element-definition-local-name-4>local
     name</a>.<li><p>Set <var>element</var>'s <a id=html-element-constructors:custom-element-state href=https://dom.spec.whatwg.org/#concept-element-custom-element-state data-x-internal=custom-element-state>custom element state</a> to "<code>custom</code>".<li><p>Set <var>element</var>'s <a href=https://dom.spec.whatwg.org/#concept-element-custom-element-definition id=html-element-constructors:concept-element-custom-element-definition data-x-internal=concept-element-custom-element-definition>custom
     element definition</a> to <var>definition</var>.<li><p>Set <var>element</var>'s <a href=https://dom.spec.whatwg.org/#concept-element-is-value id=html-element-constructors:concept-element-is-value data-x-internal=concept-element-is-value><code>is</code> value</a> to <var>is value</var>.<li><p>Return <var>element</var>.</ol>

    <p class=note>This occurs when author script constructs a new custom element directly, e.g.
    via <code>new MyCustomElement()</code>.</p>
   <li><p>Let <var>prototype</var> be <a href=https://tc39.es/ecma262/#sec-get-o-p id=html-element-constructors:js-get data-x-internal=js-get>Get</a>(<a id=html-element-constructors:newtarget-6 href=https://tc39.es/ecma262/#sec-built-in-function-objects data-x-internal=newtarget>NewTarget</a>,
   "prototype"). Rethrow any exceptions.<li>
    <p>If <a href=https://tc39.es/ecma262/#sec-ecmascript-data-types-and-values id=html-element-constructors:js-type data-x-internal=js-type>Type</a>(<var>prototype</var>) is not Object, then:</p>

    <ol><li><p>Let <var>realm</var> be <a id=html-element-constructors:getfunctionrealm href=https://tc39.es/ecma262/#sec-getfunctionrealm data-x-internal=getfunctionrealm>GetFunctionRealm</a>(<a id=html-element-constructors:newtarget-7 href=https://tc39.es/ecma262/#sec-built-in-function-objects data-x-internal=newtarget>NewTarget</a>).<li><p>Set <var>prototype</var> to the <a id=html-element-constructors:interface-prototype-object href=https://heycam.github.io/webidl/#dfn-interface-prototype-object data-x-internal=interface-prototype-object>interface prototype object</a> of
     <var>realm</var> whose interface is the same as the interface of the <a id=html-element-constructors:active-function-object-7 href=https://tc39.es/ecma262/#active-function-object data-x-internal=active-function-object>active function
     object</a>.</ol>

    <p class=note>The realm of the <a id=html-element-constructors:active-function-object-8 href=https://tc39.es/ecma262/#active-function-object data-x-internal=active-function-object>active function object</a> might not be
    <var>realm</var>, so we are using the more general concept of "the same interface" across
    realms; we are not looking for equality of <a href=https://heycam.github.io/webidl/#dfn-interface-object id=html-element-constructors:interface-object data-x-internal=interface-object>interface
    objects</a>. This fallback behavior, including using the realm of <a id=html-element-constructors:newtarget-8 href=https://tc39.es/ecma262/#sec-built-in-function-objects data-x-internal=newtarget>NewTarget</a> and
    looking up the appropriate prototype there, is designed to match analogous behavior for the
    JavaScript built-ins and Web IDL's <a id=html-element-constructors:internally-create-a-new-object-implementing-the-interface-2 href=https://heycam.github.io/webidl/#internally-create-a-new-object-implementing-the-interface data-x-internal=internally-create-a-new-object-implementing-the-interface>internally create a new object implementing the
    interface</a> algorithm.</p>
   <li><p>Let <var>element</var> be the last entry in <var>definition</var>'s <a href=#concept-custom-element-definition-construction-stack id=html-element-constructors:concept-custom-element-definition-construction-stack-2>construction stack</a>.<li>
    <p>If <var>element</var> is an <a href=#concept-already-constructed-marker id=html-element-constructors:concept-already-constructed-marker><i>already
    constructed</i> marker</a>, then throw an <a id=html-element-constructors:invalidstateerror href=https://heycam.github.io/webidl/#invalidstateerror data-x-internal=invalidstateerror>"<code>InvalidStateError</code>"</a>
    <code id=html-element-constructors:domexception><a data-x-internal=domexception href=https://heycam.github.io/webidl/#dfn-DOMException>DOMException</a></code>.</p>

    <div class=example>
     <p>This can occur when the author code inside the <a href=#custom-element-constructor id=html-element-constructors:custom-element-constructor>custom element
     constructor</a> <a href=#custom-element-conformance>non-conformantly</a> creates another
     instance of the class being constructed, before calling <code>super()</code>:</p>

     <pre><code class='js'><c- a>let</c-> doSillyThing <c- o>=</c-> <c- kc>false</c-><c- p>;</c->

<c- kr>class</c-> DontDoThis <c- kr>extends</c-> HTMLElement <c- p>{</c->
  constructor<c- p>()</c-> <c- p>{</c->
    <c- k>if</c-> <c- p>(</c->doSillyThing<c- p>)</c-> <c- p>{</c->
      doSillyThing <c- o>=</c-> <c- kc>false</c-><c- p>;</c->
      <c- k>new</c-> DontDoThis<c- p>();</c->
      <c- c1>// Now the construction stack will contain an </c-><i><c- c1>already constructed</c-></i><c- c1> marker.</c->
    <c- p>}</c->

    <c- c1>// This will then fail with an &quot;InvalidStateError&quot; DOMException:</c->
    <c- kr>super</c-><c- p>();</c->
  <c- p>}</c->
<c- p>}</c-></code></pre>
    </div>

    <div class=example>
     <p>This can also occur when author code inside the <a href=#custom-element-constructor id=html-element-constructors:custom-element-constructor-2>custom element constructor</a> <a href=#custom-element-conformance>non-conformantly</a> calls <code>super()</code>
     twice, since per the JavaScript specification, this actually executes the superclass
     constructor (i.e. this algorithm) twice, before throwing an error:

     <pre><code class='js'><c- kr>class</c-> DontDoThisEither <c- kr>extends</c-> HTMLElement <c- p>{</c->
  constructor<c- p>()</c-> <c- p>{</c->
    <c- kr>super</c-><c- p>();</c->

    <c- c1>// This will throw, but not until it has already called into the HTMLElement constructor</c->
    <c- kr>super</c-><c- p>();</c->
  <c- p>}</c->
<c- p>}</c-></code></pre>
    </div>
   <li><p>Perform <var>element</var>.[[SetPrototypeOf]](<var>prototype</var>). Rethrow any
   exceptions.<li><p>Replace the last entry in <var>definition</var>'s <a href=#concept-custom-element-definition-construction-stack id=html-element-constructors:concept-custom-element-definition-construction-stack-3>construction stack</a> with an
   <a href=#concept-already-constructed-marker id=html-element-constructors:concept-already-constructed-marker-2><i>already constructed</i>
   marker</a>.<li>
    <p>Return <var>element</var>.</p>

    <p class=note>This step is normally reached when <a href=#upgrades id=html-element-constructors:upgrades>upgrading</a> a custom element; the existing element is
    returned, so that the <code>super()</code> call inside the <a href=#custom-element-constructor id=html-element-constructors:custom-element-constructor-3>custom element
    constructor</a> assigns that existing element to <b>this</b>.</p>
   </ol>

  <hr>

  <p>In addition to the constructor behavior implied by <code id=html-element-constructors:htmlconstructor-3><a href=#htmlconstructor>[HTMLConstructor]</a></code>, some elements also have <a href=https://heycam.github.io/webidl/#dfn-named-constructor id=html-element-constructors:named-constructor data-x-internal=named-constructor>named constructors</a> (which are really factory functions with a modified <code>prototype</code> property).

  <div class=example>
   <p>Named constructors for HTML elements can also be used in an <code>extends</code>
   clause when defining a <a href=#custom-element-constructor id=html-element-constructors:custom-element-constructor-4>custom element constructor</a>:</p>

   <pre><code class='js'><c- kr>class</c-> AutoEmbiggenedImage <c- kr>extends</c-> Image <c- p>{</c->
  constructor<c- p>(</c->width<c- p>,</c-> height<c- p>)</c-> <c- p>{</c->
    <c- kr>super</c-><c- p>(</c->width <c- o>*</c-> <c- mi>10</c-><c- p>,</c-> height <c- o>*</c-> <c- mi>10</c-><c- p>);</c->
  <c- p>}</c->
<c- p>}</c->

customElements<c- p>.</c->define<c- p>(</c-><c- u>&quot;auto-embiggened&quot;</c-><c- p>,</c-> AutoEmbiggenedImage<c- p>,</c-> <c- p>{</c-> <c- kr>extends</c-><c- o>:</c-> <c- u>&quot;img&quot;</c-> <c- p>});</c->

<c- kr>const</c-> image <c- o>=</c-> <c- k>new</c-> AutoEmbiggenedImage<c- p>(</c-><c- mi>15</c-><c- p>,</c-> <c- mi>20</c-><c- p>);</c->
console<c- p>.</c->assert<c- p>(</c->image<c- p>.</c->width <c- o>===</c-> <c- mi>150</c-><c- p>);</c->
console<c- p>.</c->assert<c- p>(</c->image<c- p>.</c->height <c- o>===</c-> <c- mi>200</c-><c- p>);</c-></code></pre>
  </div>

  



  <h4 id=element-definitions><span class=secno>3.2.4</span> Element definitions<a href=#element-definitions class=self-link></a></h4>

  <p>Each element in this specification has a definition that includes the following
  information:</p>

  <dl><dt><dfn id=concept-element-categories>Categories</dfn><dd><p>A list of <a href=#content-categories id=element-definitions:content-categories>categories</a> to which the element belongs.
   These are used when defining the <a href=#content-models id=element-definitions:content-models>content models</a> for each element.<dt><dfn id=concept-element-contexts>Contexts in which this element can be used</dfn><dd>
    <p>A <em>non-normative</em> description of where the element can be used. This information is
    redundant with the content models of elements that allow this one as a child, and is provided
    only as a convenience.</p>

    <div class=note>
     <p>For simplicity, only the most specific expectations are listed.</p>

     <p>For example, all <a href=#phrasing-content-2 id=element-definitions:phrasing-content-2>phrasing content</a> is <a href=#flow-content-2 id=element-definitions:flow-content-2>flow content</a>. Thus, elements
     that are <a href=#phrasing-content-2 id=element-definitions:phrasing-content-2-2>phrasing content</a> will only be listed as "where <a href=#phrasing-content-2 id=element-definitions:phrasing-content-2-3>phrasing
     content</a> is expected", since this is the more-specific expectation. Anywhere that expects
     <a href=#flow-content-2 id=element-definitions:flow-content-2-2>flow content</a> also expects <a href=#phrasing-content-2 id=element-definitions:phrasing-content-2-4>phrasing content</a>, and thus also meets this
     expectation.</p>
    </div>
   <dt><dfn id=concept-element-content-model>Content model</dfn><dd><p>A normative description of what content must be included as children and descendants of
   the element.<dt><dfn id=concept-element-tag-omission>Tag omission in text/html</dfn><dd><p>A <em>non-normative</em> description of whether, in the <code id=element-definitions:text/html><a href=#text/html>text/html</a></code> syntax, the
   <a href=#syntax-start-tag id=element-definitions:syntax-start-tag>start</a> and <a href=#syntax-end-tag id=element-definitions:syntax-end-tag>end</a> tags can
   be omitted. This information is redundant with the normative requirements given in the <a href=#syntax-tag-omission id=element-definitions:syntax-tag-omission>optional tags</a> section, and is provided in the element
   definitions only as a convenience.<dt><dfn id=concept-element-attributes>Content attributes</dfn><dd><p>A normative list of attributes that may be specified on the element (except where
   otherwise disallowed), along with non-normative descriptions of those attributes. (The content to
   the left of the dash is normative, the content to the right of the dash is not.)<dt><dfn id=concept-element-accessibility-considerations>Accessibility considerations</dfn><dd>
    <p>For authors: Conformance requirements for use of <cite>ARIA</cite> <code id=element-definitions:attr-aria-role><a href=#attr-aria-role>role</a></code> and <code id=element-definitions:attr-aria-*><a href=#attr-aria-*>aria-*</a></code> attributes
    are defined in <cite>ARIA in HTML</cite>. <a href=#refsARIA>[ARIA]</a> <a href=#refsARIAHTML>[ARIAHTML]</a></p>

    <p>For implementers: User agent requirements for implementing accessibility API semantics
    are defined in <cite>HTML Accessibility API Mappings</cite>. <a href=#refsHTMLAAM>[HTMLAAM]</a></p>
   <dt><dfn id=concept-element-dom>DOM interface</dfn><dd><p>A normative definition of a DOM interface that such elements must implement.</dl>

  <p>This is then followed by a description of what the element <a href=#represents id=element-definitions:represents>represents</a>, along with
  any additional normative conformance criteria that may apply to authors and implementations. Examples are sometimes also included.</p>


  <h5 id=attributes><span class=secno>3.2.4.1</span> Attributes<a href=#attributes class=self-link></a></h5>

  <p id=attribute-text>An attribute value is a string. Except where otherwise specified,
  attribute values on <a href=#html-elements id=attributes:html-elements>HTML elements</a> may be any string value, including the empty
  string, and there is no restriction on what text can be specified in such attribute values.</p>



  <h4 id=content-models><span class=secno>3.2.5</span> <dfn>Content models</dfn><a href=#content-models class=self-link></a></h4>

  <p>Each element defined in this specification has a content model: a description of the element's
  expected <a href=#concept-html-contents id=content-models:concept-html-contents>contents</a>. An <a href=#html-elements id=content-models:html-elements>HTML
  element</a> must have contents that match the requirements described in the element's content
  model. The <dfn id=concept-html-contents>contents</dfn> of an element are its children in the
  DOM.</p>

  <p><a id=content-models:space-characters href=https://infra.spec.whatwg.org/#ascii-whitespace data-x-internal=space-characters>ASCII whitespace</a> is always allowed between elements. User agents represent these
  characters between elements in the source markup as <code id=content-models:text><a data-x-internal=text href=https://dom.spec.whatwg.org/#interface-text>Text</a></code> nodes in the DOM. Empty <code id=content-models:text-2><a data-x-internal=text href=https://dom.spec.whatwg.org/#interface-text>Text</a></code> nodes and
  <code id=content-models:text-3><a data-x-internal=text href=https://dom.spec.whatwg.org/#interface-text>Text</a></code> nodes consisting of just sequences of those characters are considered
  <dfn id=inter-element-whitespace>inter-element whitespace</dfn>.</p>

  <p><a href=#inter-element-whitespace id=content-models:inter-element-whitespace>Inter-element whitespace</a>, comment nodes, and processing instruction nodes must be
  ignored when establishing whether an element's contents match the element's content model or not,
  and must be ignored when following algorithms that define document and element semantics.</p>

  <p class=note>Thus, an element <var>A</var> is said to be <i>preceded or followed</i>
  by a second element <var>B</var> if <var>A</var> and <var>B</var> have
  the same parent node and there are no other element nodes or <code id=content-models:text-4><a data-x-internal=text href=https://dom.spec.whatwg.org/#interface-text>Text</a></code> nodes (other than
  <a href=#inter-element-whitespace id=content-models:inter-element-whitespace-2>inter-element whitespace</a>) between them. Similarly, a node is the <i>only child</i> of
  an element if that element contains no other nodes other than <a href=#inter-element-whitespace id=content-models:inter-element-whitespace-3>inter-element
  whitespace</a>, comment nodes, and processing instruction nodes.</p>

  <p>Authors must not use <a href=#html-elements id=content-models:html-elements-2>HTML elements</a> anywhere except where they are explicitly
  allowed, as defined for each element, or as explicitly required by other specifications. For XML
  compound documents, these contexts could be inside elements from other namespaces, if those
  elements are defined as providing the relevant contexts.</p>

  <p class=example><cite>The Atom Syndication Format</cite> defines a <code>content</code> element. When its <code>type</code> attribute has the value
  <code>xhtml</code>, <cite>The Atom Syndication Format</cite> requires that it contain a
  single HTML <code id=content-models:the-div-element><a href=#the-div-element>div</a></code> element. Thus, a <code id=content-models:the-div-element-2><a href=#the-div-element>div</a></code> element is allowed in that context,
  even though this is not explicitly normatively stated by this specification. <a href=#refsATOM>[ATOM]</a></p>

  <p>In addition, <a href=#html-elements id=content-models:html-elements-3>HTML elements</a> may be orphan nodes (i.e. without a parent node).</p>

  <div class=example>

   <p>For example, creating a <code id=content-models:the-td-element><a href=#the-td-element>td</a></code> element and storing it in a global variable in a
   script is conforming, even though <code id=content-models:the-td-element-2><a href=#the-td-element>td</a></code> elements are otherwise only supposed to be used
   inside <code id=content-models:the-tr-element><a href=#the-tr-element>tr</a></code> elements.</p>

   <pre><code class='js'><c- a>var</c-> data <c- o>=</c-> <c- p>{</c->
  name<c- o>:</c-> <c- u>&quot;Banana&quot;</c-><c- p>,</c->
  cell<c- o>:</c-> document<c- p>.</c->createElement<c- p>(</c-><c- t>&apos;td&apos;</c-><c- p>),</c->
<c- p>};</c-></code></pre>

  </div>


  <h5 id=the-nothing-content-model><span class=secno>3.2.5.1</span> The "nothing" content model<a href=#the-nothing-content-model class=self-link></a></h5>

  <p>When an element's content model is <dfn id=concept-content-nothing>nothing</dfn>, the
  element must contain no <code id=the-nothing-content-model:text><a data-x-internal=text href=https://dom.spec.whatwg.org/#interface-text>Text</a></code> nodes (other than <a href=#inter-element-whitespace id=the-nothing-content-model:inter-element-whitespace>inter-element whitespace</a>)
  and no element nodes.</p>

  <p class=note>Most HTML elements whose content model is "nothing" are also, for convenience,
  <a href=#void-elements id=the-nothing-content-model:void-elements>void elements</a> (elements that have no <a href=#syntax-end-tag id=the-nothing-content-model:syntax-end-tag>end tag</a> in
  the <a href=#syntax>HTML syntax</a>). However, these are entirely separate concepts.</p>


  <h5 id=kinds-of-content><span class=secno>3.2.5.2</span> Kinds of content<a href=#kinds-of-content class=self-link></a></h5>

  <p>Each element in HTML falls into zero or more <dfn id=content-categories>categories</dfn>
  that group elements with similar characteristics together. The following broad categories are used
  in this specification:</p>

  <ul class=brief><li><a href=#metadata-content-2 id=kinds-of-content:metadata-content-2>Metadata content</a><li><a href=#flow-content-2 id=kinds-of-content:flow-content-2>Flow content</a><li><a href=#sectioning-content-2 id=kinds-of-content:sectioning-content-2>Sectioning content</a><li><a href=#heading-content-2 id=kinds-of-content:heading-content-2>Heading content</a><li><a href=#phrasing-content-2 id=kinds-of-content:phrasing-content-2>Phrasing content</a><li><a href=#embedded-content-category id=kinds-of-content:embedded-content-category>Embedded content</a><li><a href=#interactive-content-2 id=kinds-of-content:interactive-content-2>Interactive content</a></ul>

  <p class=note>Some elements also fall into other categories, which are defined in other parts of
  this specification.</p>

  <p>These categories are related as follows:</p>

  <p><object width=1000 height=288 data=/images/content-venn.svg></object></p>

  <p>Sectioning content, heading content, phrasing content, embedded content, and interactive
  content are all types of flow content. Metadata is sometimes flow content. Metadata and
  interactive content are sometimes phrasing content. Embedded content is also a type of phrasing
  content, and sometimes is interactive content.</p>

  <p>Other categories are also used for specific purposes, e.g. form controls are specified using a
  number of categories to define common requirements. Some elements have unique requirements and do
  not fit into any particular category.</p>


  <h6 id=metadata-content><span class=secno>3.2.5.2.1</span> Metadata content<a href=#metadata-content class=self-link></a></h6>

  <p><dfn id=metadata-content-2>Metadata content</dfn> is content that sets up the presentation or behavior of the rest of
  the content, or that sets up the relationship of the document with other documents, or that
  conveys other "out of band" information.</p>

  
  <ul class="brief category-list"><li><code id=metadata-content:the-base-element><a href=#the-base-element>base</a></code><li><code id=metadata-content:the-link-element><a href=#the-link-element>link</a></code><li><code id=metadata-content:the-meta-element><a href=#the-meta-element>meta</a></code><li><code id=metadata-content:the-noscript-element><a href=#the-noscript-element>noscript</a></code><li><code id=metadata-content:the-script-element><a href=#the-script-element>script</a></code><li><code id=metadata-content:the-style-element><a href=#the-style-element>style</a></code><li><code id=metadata-content:the-template-element><a href=#the-template-element>template</a></code><li><code id=metadata-content:the-title-element><a href=#the-title-element>title</a></code></ul>

  <p>Elements from other namespaces whose semantics are primarily metadata-related (e.g. RDF) are
  also <a href=#metadata-content-2 id=metadata-content:metadata-content-2>metadata content</a>.</p>

  <div class=example>

   <p>Thus, in the XML serialization, one can use RDF, like this:</p>

   <pre><code class='html'><c- p>&lt;</c-><c- f>html</c-> <c- e>xmlns</c-><c- o>=</c-><c- s>&quot;http://www.w3.org/1999/xhtml&quot;</c->
      <c- e>xmlns:r</c-><c- o>=</c-><c- s>&quot;http://www.w3.org/1999/02/22-rdf-syntax-ns#&quot;</c-> <c- e>xml:lang</c-><c- o>=</c-><c- s>&quot;en&quot;</c-><c- p>&gt;</c->
 <c- p>&lt;</c-><c- f>head</c-><c- p>&gt;</c->
  <c- p>&lt;</c-><c- f>title</c-><c- p>&gt;</c->Hedral&apos;s Home Page<c- p>&lt;/</c-><c- f>title</c-><c- p>&gt;</c->
  <c- p>&lt;</c-><c- f>r:RDF</c-><c- p>&gt;</c->
   <c- p>&lt;</c-><c- f>Person</c-> <c- e>xmlns</c-><c- o>=</c-><c- s>&quot;http://www.w3.org/2000/10/swap/pim/contact#&quot;</c->
           <c- e>r:about</c-><c- o>=</c-><c- s>&quot;https://hedral.example.com/#&quot;</c-><c- p>&gt;</c->
    <c- p>&lt;</c-><c- f>fullName</c-><c- p>&gt;</c->Cat Hedral<c- p>&lt;/</c-><c- f>fullName</c-><c- p>&gt;</c->
    <c- p>&lt;</c-><c- f>mailbox</c-> <c- e>r:resource</c-><c- o>=</c-><c- s>&quot;mailto:hedral@damowmow.com&quot;</c-><c- p>/&gt;</c->
    <c- p>&lt;</c-><c- f>personalTitle</c-><c- p>&gt;</c->Sir<c- p>&lt;/</c-><c- f>personalTitle</c-><c- p>&gt;</c->
   <c- p>&lt;/</c-><c- f>Person</c-><c- p>&gt;</c->
  <c- p>&lt;/</c-><c- f>r:RDF</c-><c- p>&gt;</c->
 <c- p>&lt;/</c-><c- f>head</c-><c- p>&gt;</c->
 <c- p>&lt;</c-><c- f>body</c-><c- p>&gt;</c->
  <c- p>&lt;</c-><c- f>h1</c-><c- p>&gt;</c->My home page<c- p>&lt;/</c-><c- f>h1</c-><c- p>&gt;</c->
  <c- p>&lt;</c-><c- f>p</c-><c- p>&gt;</c->I like playing with string, I guess. Sister says squirrels are fun
  too so sometimes I follow her to play with them.<c- p>&lt;/</c-><c- f>p</c-><c- p>&gt;</c->
 <c- p>&lt;/</c-><c- f>body</c-><c- p>&gt;</c->
<c- p>&lt;/</c-><c- f>html</c-><c- p>&gt;</c-></code></pre>

   <p>This isn't possible in the HTML serialization, however.</p>

  </div>


  <h6 id=flow-content><span class=secno>3.2.5.2.2</span> Flow content<a href=#flow-content class=self-link></a></h6>

  <p>Most elements that are used in the body of documents and applications are categorized as
  <dfn id=flow-content-2 data-export="">flow content</dfn>.</p>

  
  <ul class="brief category-list"><li><code id=flow-content:the-a-element><a href=#the-a-element>a</a></code><li><code id=flow-content:the-abbr-element><a href=#the-abbr-element>abbr</a></code><li><code id=flow-content:the-address-element><a href=#the-address-element>address</a></code><li><code id=flow-content:the-area-element><a href=#the-area-element>area</a></code> (if it is a descendant of a <code id=flow-content:the-map-element><a href=#the-map-element>map</a></code> element)<li><code id=flow-content:the-article-element><a href=#the-article-element>article</a></code><li><code id=flow-content:the-aside-element><a href=#the-aside-element>aside</a></code><li><code id=flow-content:the-audio-element><a href=#the-audio-element>audio</a></code><li><code id=flow-content:the-b-element><a href=#the-b-element>b</a></code><li><code id=flow-content:the-bdi-element><a href=#the-bdi-element>bdi</a></code><li><code id=flow-content:the-bdo-element><a href=#the-bdo-element>bdo</a></code><li><code id=flow-content:the-blockquote-element><a href=#the-blockquote-element>blockquote</a></code><li><code id=flow-content:the-br-element><a href=#the-br-element>br</a></code><li><code id=flow-content:the-button-element><a href=#the-button-element>button</a></code><li><code id=flow-content:the-canvas-element><a href=#the-canvas-element>canvas</a></code><li><code id=flow-content:the-cite-element><a href=#the-cite-element>cite</a></code><li><code id=flow-content:the-code-element><a href=#the-code-element>code</a></code><li><code id=flow-content:the-data-element><a href=#the-data-element>data</a></code><li><code id=flow-content:the-datalist-element><a href=#the-datalist-element>datalist</a></code><li><code id=flow-content:the-del-element><a href=#the-del-element>del</a></code><li><code id=flow-content:the-details-element><a href=#the-details-element>details</a></code><li><code id=flow-content:the-dfn-element><a href=#the-dfn-element>dfn</a></code><li><code id=flow-content:the-dialog-element><a href=#the-dialog-element>dialog</a></code><li><code id=flow-content:the-div-element><a href=#the-div-element>div</a></code><li><code id=flow-content:the-dl-element><a href=#the-dl-element>dl</a></code><li><code id=flow-content:the-em-element><a href=#the-em-element>em</a></code><li><code id=flow-content:the-embed-element><a href=#the-embed-element>embed</a></code><li><code id=flow-content:the-fieldset-element><a href=#the-fieldset-element>fieldset</a></code><li><code id=flow-content:the-figure-element><a href=#the-figure-element>figure</a></code><li><code id=flow-content:the-footer-element><a href=#the-footer-element>footer</a></code><li><code id=flow-content:the-form-element><a href=#the-form-element>form</a></code><li><code id=flow-content:the-h1,-h2,-h3,-h4,-h5,-and-h6-elements><a href=#the-h1,-h2,-h3,-h4,-h5,-and-h6-elements>h1</a></code><li><code id=flow-content:the-h1,-h2,-h3,-h4,-h5,-and-h6-elements-2><a href=#the-h1,-h2,-h3,-h4,-h5,-and-h6-elements>h2</a></code><li><code id=flow-content:the-h1,-h2,-h3,-h4,-h5,-and-h6-elements-3><a href=#the-h1,-h2,-h3,-h4,-h5,-and-h6-elements>h3</a></code><li><code id=flow-content:the-h1,-h2,-h3,-h4,-h5,-and-h6-elements-4><a href=#the-h1,-h2,-h3,-h4,-h5,-and-h6-elements>h4</a></code><li><code id=flow-content:the-h1,-h2,-h3,-h4,-h5,-and-h6-elements-5><a href=#the-h1,-h2,-h3,-h4,-h5,-and-h6-elements>h5</a></code><li><code id=flow-content:the-h1,-h2,-h3,-h4,-h5,-and-h6-elements-6><a href=#the-h1,-h2,-h3,-h4,-h5,-and-h6-elements>h6</a></code><li><code id=flow-content:the-header-element><a href=#the-header-element>header</a></code><li><code id=flow-content:the-hgroup-element><a href=#the-hgroup-element>hgroup</a></code><li><code id=flow-content:the-hr-element><a href=#the-hr-element>hr</a></code><li><code id=flow-content:the-i-element><a href=#the-i-element>i</a></code><li><code id=flow-content:the-iframe-element><a href=#the-iframe-element>iframe</a></code><li><code id=flow-content:the-img-element><a href=#the-img-element>img</a></code><li><code id=flow-content:the-input-element><a href=#the-input-element>input</a></code><li><code id=flow-content:the-ins-element><a href=#the-ins-element>ins</a></code><li><code id=flow-content:the-kbd-element><a href=#the-kbd-element>kbd</a></code><li><code id=flow-content:the-label-element><a href=#the-label-element>label</a></code><li><code id=flow-content:the-link-element><a href=#the-link-element>link</a></code> (if it is <a href=#allowed-in-the-body id=flow-content:allowed-in-the-body>allowed in the body</a>)<li><code id=flow-content:the-main-element><a href=#the-main-element>main</a></code> (if it is a <a href=#hierarchically-correct-main-element id=flow-content:hierarchically-correct-main-element>hierarchically correct <code>main</code> element</a>)<li><code id=flow-content:the-map-element-2><a href=#the-map-element>map</a></code><li><code id=flow-content:the-mark-element><a href=#the-mark-element>mark</a></code><li><a id=flow-content:mathml-math href=https://www.w3.org/Math/draft-spec/chapter2.html#interf.toplevel data-x-internal=mathml-math>MathML <code>math</code></a><li><code id=flow-content:the-menu-element><a href=#the-menu-element>menu</a></code><li><code id=flow-content:the-meta-element><a href=#the-meta-element>meta</a></code> (if the <code id=flow-content:names:-the-itemprop-attribute><a href=#names:-the-itemprop-attribute>itemprop</a></code> attribute is present)<li><code id=flow-content:the-meter-element><a href=#the-meter-element>meter</a></code><li><code id=flow-content:the-nav-element><a href=#the-nav-element>nav</a></code><li><code id=flow-content:the-noscript-element><a href=#the-noscript-element>noscript</a></code><li><code id=flow-content:the-object-element><a href=#the-object-element>object</a></code><li><code id=flow-content:the-ol-element><a href=#the-ol-element>ol</a></code><li><code id=flow-content:the-output-element><a href=#the-output-element>output</a></code><li><code id=flow-content:the-p-element><a href=#the-p-element>p</a></code><li><code id=flow-content:the-picture-element><a href=#the-picture-element>picture</a></code><li><code id=flow-content:the-pre-element><a href=#the-pre-element>pre</a></code><li><code id=flow-content:the-progress-element><a href=#the-progress-element>progress</a></code><li><code id=flow-content:the-q-element><a href=#the-q-element>q</a></code><li><code id=flow-content:the-ruby-element><a href=#the-ruby-element>ruby</a></code><li><code id=flow-content:the-s-element><a href=#the-s-element>s</a></code><li><code id=flow-content:the-samp-element><a href=#the-samp-element>samp</a></code><li><code id=flow-content:the-script-element><a href=#the-script-element>script</a></code><li><code id=flow-content:the-section-element><a href=#the-section-element>section</a></code><li><code id=flow-content:the-select-element><a href=#the-select-element>select</a></code><li><code id=flow-content:the-slot-element><a href=#the-slot-element>slot</a></code><li><code id=flow-content:the-small-element><a href=#the-small-element>small</a></code><li><code id=flow-content:the-span-element><a href=#the-span-element>span</a></code><li><code id=flow-content:the-strong-element><a href=#the-strong-element>strong</a></code><li><code id=flow-content:the-sub-and-sup-elements><a href=#the-sub-and-sup-elements>sub</a></code><li><code id=flow-content:the-sub-and-sup-elements-2><a href=#the-sub-and-sup-elements>sup</a></code><li><a id=flow-content:svg-svg href=https://svgwg.org/svg2-draft/struct.html#SVGElement data-x-internal=svg-svg>SVG <code>svg</code></a><li><code id=flow-content:the-table-element><a href=#the-table-element>table</a></code><li><code id=flow-content:the-template-element><a href=#the-template-element>template</a></code><li><code id=flow-content:the-textarea-element><a href=#the-textarea-element>textarea</a></code><li><code id=flow-content:the-time-element><a href=#the-time-element>time</a></code><li><code id=flow-content:the-u-element><a href=#the-u-element>u</a></code><li><code id=flow-content:the-ul-element><a href=#the-ul-element>ul</a></code><li><code id=flow-content:the-var-element><a href=#the-var-element>var</a></code><li><code id=flow-content:the-video-element><a href=#the-video-element>video</a></code><li><code id=flow-content:the-wbr-element><a href=#the-wbr-element>wbr</a></code><li><a href=#autonomous-custom-element id=flow-content:autonomous-custom-element>autonomous custom elements</a><li><a href=#text-content id=flow-content:text-content>text</a></ul>


  <h6 id=sectioning-content><span class=secno>3.2.5.2.3</span> Sectioning content<a href=#sectioning-content class=self-link></a></h6>

  <p><dfn id=sectioning-content-2>Sectioning content</dfn> is content that defines the scope of <a href=#heading-content-2 id=sectioning-content:heading-content-2>headings</a> and <a href=#the-footer-element id=sectioning-content:the-footer-element>footers</a>.</p>

  
  <ul class="brief category-list"><li><code id=sectioning-content:the-article-element><a href=#the-article-element>article</a></code><li><code id=sectioning-content:the-aside-element><a href=#the-aside-element>aside</a></code><li><code id=sectioning-content:the-nav-element><a href=#the-nav-element>nav</a></code><li><code id=sectioning-content:the-section-element><a href=#the-section-element>section</a></code></ul>

  <p>Each <a href=#sectioning-content-2 id=sectioning-content:sectioning-content-2>sectioning content</a> element potentially has a heading and an
  <a href=#outline id=sectioning-content:outline>outline</a>. See the section on <a href=#headings-and-sections id=sectioning-content:headings-and-sections>headings and sections</a> for further
  details.</p>

  <p class=note>There are also certain elements that are <a href=#sectioning-root id=sectioning-content:sectioning-root>sectioning
  roots</a>. These are distinct from <a href=#sectioning-content-2 id=sectioning-content:sectioning-content-2-2>sectioning content</a>, but they can also have an
  <a href=#outline id=sectioning-content:outline-2>outline</a>.</p>


  <h6 id=heading-content><span class=secno>3.2.5.2.4</span> Heading content<a href=#heading-content class=self-link></a></h6>

  <p><dfn id=heading-content-2>Heading content</dfn> defines the header of a section (whether explicitly marked up using
  <a href=#sectioning-content-2 id=heading-content:sectioning-content-2>sectioning content</a> elements, or implied by the heading content itself).</p>

  
  <ul class="brief category-list"><li><code id=heading-content:the-h1,-h2,-h3,-h4,-h5,-and-h6-elements><a href=#the-h1,-h2,-h3,-h4,-h5,-and-h6-elements>h1</a></code><li><code id=heading-content:the-h1,-h2,-h3,-h4,-h5,-and-h6-elements-2><a href=#the-h1,-h2,-h3,-h4,-h5,-and-h6-elements>h2</a></code><li><code id=heading-content:the-h1,-h2,-h3,-h4,-h5,-and-h6-elements-3><a href=#the-h1,-h2,-h3,-h4,-h5,-and-h6-elements>h3</a></code><li><code id=heading-content:the-h1,-h2,-h3,-h4,-h5,-and-h6-elements-4><a href=#the-h1,-h2,-h3,-h4,-h5,-and-h6-elements>h4</a></code><li><code id=heading-content:the-h1,-h2,-h3,-h4,-h5,-and-h6-elements-5><a href=#the-h1,-h2,-h3,-h4,-h5,-and-h6-elements>h5</a></code><li><code id=heading-content:the-h1,-h2,-h3,-h4,-h5,-and-h6-elements-6><a href=#the-h1,-h2,-h3,-h4,-h5,-and-h6-elements>h6</a></code><li><code id=heading-content:the-hgroup-element><a href=#the-hgroup-element>hgroup</a></code></ul>


  <h6 id=phrasing-content><span class=secno>3.2.5.2.5</span> Phrasing content<a href=#phrasing-content class=self-link></a></h6>

  <p><dfn id=phrasing-content-2>Phrasing content</dfn> is the text of the document, as well as elements that mark up that
  text at the intra-paragraph level. Runs of <a href=#phrasing-content-2 id=phrasing-content:phrasing-content-2>phrasing content</a> form <a href=#paragraph id=phrasing-content:paragraph>paragraphs</a>.</p>

  
  <ul class="brief category-list"><li><code id=phrasing-content:the-a-element><a href=#the-a-element>a</a></code><li><code id=phrasing-content:the-abbr-element><a href=#the-abbr-element>abbr</a></code><li><code id=phrasing-content:the-area-element><a href=#the-area-element>area</a></code> (if it is a descendant of a <code id=phrasing-content:the-map-element><a href=#the-map-element>map</a></code> element)<li><code id=phrasing-content:the-audio-element><a href=#the-audio-element>audio</a></code><li><code id=phrasing-content:the-b-element><a href=#the-b-element>b</a></code><li><code id=phrasing-content:the-bdi-element><a href=#the-bdi-element>bdi</a></code><li><code id=phrasing-content:the-bdo-element><a href=#the-bdo-element>bdo</a></code><li><code id=phrasing-content:the-br-element><a href=#the-br-element>br</a></code><li><code id=phrasing-content:the-button-element><a href=#the-button-element>button</a></code><li><code id=phrasing-content:the-canvas-element><a href=#the-canvas-element>canvas</a></code><li><code id=phrasing-content:the-cite-element><a href=#the-cite-element>cite</a></code><li><code id=phrasing-content:the-code-element><a href=#the-code-element>code</a></code><li><code id=phrasing-content:the-data-element><a href=#the-data-element>data</a></code><li><code id=phrasing-content:the-datalist-element><a href=#the-datalist-element>datalist</a></code><li><code id=phrasing-content:the-del-element><a href=#the-del-element>del</a></code><li><code id=phrasing-content:the-dfn-element><a href=#the-dfn-element>dfn</a></code><li><code id=phrasing-content:the-em-element><a href=#the-em-element>em</a></code><li><code id=phrasing-content:the-embed-element><a href=#the-embed-element>embed</a></code><li><code id=phrasing-content:the-i-element><a href=#the-i-element>i</a></code><li><code id=phrasing-content:the-iframe-element><a href=#the-iframe-element>iframe</a></code><li><code id=phrasing-content:the-img-element><a href=#the-img-element>img</a></code><li><code id=phrasing-content:the-input-element><a href=#the-input-element>input</a></code><li><code id=phrasing-content:the-ins-element><a href=#the-ins-element>ins</a></code><li><code id=phrasing-content:the-kbd-element><a href=#the-kbd-element>kbd</a></code><li><code id=phrasing-content:the-label-element><a href=#the-label-element>label</a></code><li><code id=phrasing-content:the-link-element><a href=#the-link-element>link</a></code> (if it is <a href=#allowed-in-the-body id=phrasing-content:allowed-in-the-body>allowed in the body</a>)<li><code id=phrasing-content:the-map-element-2><a href=#the-map-element>map</a></code><li><code id=phrasing-content:the-mark-element><a href=#the-mark-element>mark</a></code><li><a id=phrasing-content:mathml-math href=https://www.w3.org/Math/draft-spec/chapter2.html#interf.toplevel data-x-internal=mathml-math>MathML <code>math</code></a><li><code id=phrasing-content:the-meta-element><a href=#the-meta-element>meta</a></code> (if the <code id=phrasing-content:names:-the-itemprop-attribute><a href=#names:-the-itemprop-attribute>itemprop</a></code> attribute is present)<li><code id=phrasing-content:the-meter-element><a href=#the-meter-element>meter</a></code><li><code id=phrasing-content:the-noscript-element><a href=#the-noscript-element>noscript</a></code><li><code id=phrasing-content:the-object-element><a href=#the-object-element>object</a></code><li><code id=phrasing-content:the-output-element><a href=#the-output-element>output</a></code><li><code id=phrasing-content:the-picture-element><a href=#the-picture-element>picture</a></code><li><code id=phrasing-content:the-progress-element><a href=#the-progress-element>progress</a></code><li><code id=phrasing-content:the-q-element><a href=#the-q-element>q</a></code><li><code id=phrasing-content:the-ruby-element><a href=#the-ruby-element>ruby</a></code><li><code id=phrasing-content:the-s-element><a href=#the-s-element>s</a></code><li><code id=phrasing-content:the-samp-element><a href=#the-samp-element>samp</a></code><li><code id=phrasing-content:the-script-element><a href=#the-script-element>script</a></code><li><code id=phrasing-content:the-select-element><a href=#the-select-element>select</a></code><li><code id=phrasing-content:the-slot-element><a href=#the-slot-element>slot</a></code><li><code id=phrasing-content:the-small-element><a href=#the-small-element>small</a></code><li><code id=phrasing-content:the-span-element><a href=#the-span-element>span</a></code><li><code id=phrasing-content:the-strong-element><a href=#the-strong-element>strong</a></code><li><code id=phrasing-content:the-sub-and-sup-elements><a href=#the-sub-and-sup-elements>sub</a></code><li><code id=phrasing-content:the-sub-and-sup-elements-2><a href=#the-sub-and-sup-elements>sup</a></code><li><a id=phrasing-content:svg-svg href=https://svgwg.org/svg2-draft/struct.html#SVGElement data-x-internal=svg-svg>SVG <code>svg</code></a><li><code id=phrasing-content:the-template-element><a href=#the-template-element>template</a></code><li><code id=phrasing-content:the-textarea-element><a href=#the-textarea-element>textarea</a></code><li><code id=phrasing-content:the-time-element><a href=#the-time-element>time</a></code><li><code id=phrasing-content:the-u-element><a href=#the-u-element>u</a></code><li><code id=phrasing-content:the-var-element><a href=#the-var-element>var</a></code><li><code id=phrasing-content:the-video-element><a href=#the-video-element>video</a></code><li><code id=phrasing-content:the-wbr-element><a href=#the-wbr-element>wbr</a></code><li><a href=#autonomous-custom-element id=phrasing-content:autonomous-custom-element>autonomous custom elements</a><li><a href=#text-content id=phrasing-content:text-content>text</a></ul>

  <p class=note>Most elements that are categorized as phrasing content can only contain elements
  that are themselves categorized as phrasing content, not any flow content.</p>

  <p><dfn id=text-content>Text</dfn>, in the context of content models, means either nothing,
  or <code id=phrasing-content:text><a data-x-internal=text href=https://dom.spec.whatwg.org/#interface-text>Text</a></code> nodes. <a href=#text-content id=phrasing-content:text-content-2>Text</a> is sometimes used as a content
  model on its own, but is also <a href=#phrasing-content-2 id=phrasing-content:phrasing-content-2-2>phrasing content</a>, and can be <a href=#inter-element-whitespace id=phrasing-content:inter-element-whitespace>inter-element
  whitespace</a> (if the <code id=phrasing-content:text-2><a data-x-internal=text href=https://dom.spec.whatwg.org/#interface-text>Text</a></code> nodes are empty or contain just <a id=phrasing-content:space-characters href=https://infra.spec.whatwg.org/#ascii-whitespace data-x-internal=space-characters>ASCII
  whitespace</a>).</p>

  <p><code id=phrasing-content:text-3><a data-x-internal=text href=https://dom.spec.whatwg.org/#interface-text>Text</a></code> nodes and attribute values must consist of <a href=https://infra.spec.whatwg.org/#scalar-value id=phrasing-content:scalar-value data-x-internal=scalar-value>scalar
  values</a>, excluding <a href=https://infra.spec.whatwg.org/#noncharacter id=phrasing-content:noncharacter data-x-internal=noncharacter>noncharacters</a>, and <a href=https://infra.spec.whatwg.org/#control id=phrasing-content:control data-x-internal=control>controls</a> other than <a id=phrasing-content:space-characters-2 href=https://infra.spec.whatwg.org/#ascii-whitespace data-x-internal=space-characters>ASCII whitespace</a>.

   
  

  This specification includes extra constraints on the exact value of <code id=phrasing-content:text-4><a data-x-internal=text href=https://dom.spec.whatwg.org/#interface-text>Text</a></code> nodes and
  attribute values depending on their precise context.

  

  </p>



  <h6 id=embedded-content-2><span class=secno>3.2.5.2.6</span> Embedded content<a href=#embedded-content-2 class=self-link></a></h6>

  <p><dfn id=embedded-content-category>Embedded content</dfn> is content that imports another
  resource into the document, or content from another vocabulary that
  is inserted into the document.</p>

  
  <ul class="brief category-list"><li><code id=embedded-content-2:the-audio-element><a href=#the-audio-element>audio</a></code><li><code id=embedded-content-2:the-canvas-element><a href=#the-canvas-element>canvas</a></code><li><code id=embedded-content-2:the-embed-element><a href=#the-embed-element>embed</a></code><li><code id=embedded-content-2:the-iframe-element><a href=#the-iframe-element>iframe</a></code><li><code id=embedded-content-2:the-img-element><a href=#the-img-element>img</a></code><li><a id=embedded-content-2:mathml-math href=https://www.w3.org/Math/draft-spec/chapter2.html#interf.toplevel data-x-internal=mathml-math>MathML <code>math</code></a><li><code id=embedded-content-2:the-object-element><a href=#the-object-element>object</a></code><li><code id=embedded-content-2:the-picture-element><a href=#the-picture-element>picture</a></code><li><a id=embedded-content-2:svg-svg href=https://svgwg.org/svg2-draft/struct.html#SVGElement data-x-internal=svg-svg>SVG <code>svg</code></a><li><code id=embedded-content-2:the-video-element><a href=#the-video-element>video</a></code></ul>

  <p>Elements that are from namespaces other than the <a id=embedded-content-2:html-namespace-2 href=https://infra.spec.whatwg.org/#html-namespace data-x-internal=html-namespace-2>HTML namespace</a> and that convey
  content but not metadata, are <a href=#embedded-content-category id=embedded-content-2:embedded-content-category>embedded content</a> for the purposes of the content models
  defined in this specification. (For example, MathML, or SVG.)</p>

  <p>Some embedded content elements can have <dfn id=fallback-content>fallback content</dfn>: content that is to be used
  when the external resource cannot be used (e.g. because it is of an unsupported format). The
  element definitions state what the fallback is, if any.</p>
  


  <h6 id=interactive-content><span class=secno>3.2.5.2.7</span> Interactive content<a href=#interactive-content class=self-link></a></h6>



  <p><dfn id=interactive-content-2>Interactive content</dfn> is content that is specifically intended for user
  interaction.</p>

  
  <ul class="brief category-list"><li><code id=interactive-content:the-a-element><a href=#the-a-element>a</a></code> (if the <code id=interactive-content:attr-hyperlink-href><a href=#attr-hyperlink-href>href</a></code> attribute is present)<li><code id=interactive-content:the-audio-element><a href=#the-audio-element>audio</a></code> (if the <code id=interactive-content:attr-media-controls><a href=#attr-media-controls>controls</a></code> attribute is present)<li><code id=interactive-content:the-button-element><a href=#the-button-element>button</a></code><li><code id=interactive-content:the-details-element><a href=#the-details-element>details</a></code><li><code id=interactive-content:the-embed-element><a href=#the-embed-element>embed</a></code><li><code id=interactive-content:the-iframe-element><a href=#the-iframe-element>iframe</a></code><li><code id=interactive-content:the-img-element><a href=#the-img-element>img</a></code> (if the <code id=interactive-content:attr-hyperlink-usemap><a href=#attr-hyperlink-usemap>usemap</a></code> attribute is present)<li><code id=interactive-content:the-input-element><a href=#the-input-element>input</a></code> (if the <code id=interactive-content:attr-input-type><a href=#attr-input-type>type</a></code> attribute is <em>not</em> in the <a href="#hidden-state-(type=hidden)" id="interactive-content:hidden-state-(type=hidden)">Hidden</a> state)<li><code id=interactive-content:the-label-element><a href=#the-label-element>label</a></code><li><code id=interactive-content:the-object-element><a href=#the-object-element>object</a></code> (if the <code id=interactive-content:attr-hyperlink-usemap-2><a href=#attr-hyperlink-usemap>usemap</a></code> attribute is present)<li><code id=interactive-content:the-select-element><a href=#the-select-element>select</a></code><li><code id=interactive-content:the-textarea-element><a href=#the-textarea-element>textarea</a></code><li><code id=interactive-content:the-video-element><a href=#the-video-element>video</a></code> (if the <code id=interactive-content:attr-media-controls-2><a href=#attr-media-controls>controls</a></code> attribute is present)</ul>



  <h6 id=palpable-content><span class=secno>3.2.5.2.8</span> Palpable content<a href=#palpable-content class=self-link></a></h6>

  <p>As a general rule, elements whose content model allows any <a href=#flow-content-2 id=palpable-content:flow-content-2>flow content</a> or
  <a href=#phrasing-content-2 id=palpable-content:phrasing-content-2>phrasing content</a> should have at least one node in its <a href=#concept-html-contents id=palpable-content:concept-html-contents>contents</a> that is <dfn id=palpable-content-2>palpable content</dfn> and that does
  not have the <code id=palpable-content:attr-hidden><a href=#attr-hidden>hidden</a></code> attribute specified.</p>

  <p class=note><a href=#palpable-content-2 id=palpable-content:palpable-content-2>Palpable content</a> makes an element non-empty by providing either
  some descendant non-empty <a href=#text-content id=palpable-content:text-content>text</a>, or else something users can
  hear (<code id=palpable-content:the-audio-element><a href=#the-audio-element>audio</a></code> elements) or view (<code id=palpable-content:the-video-element><a href=#the-video-element>video</a></code> or <code id=palpable-content:the-img-element><a href=#the-img-element>img</a></code> or
  <code id=palpable-content:the-canvas-element><a href=#the-canvas-element>canvas</a></code> elements) or otherwise interact with (for example, interactive form
  controls).</p>

  <p>This requirement is not a hard requirement, however, as there are many cases where an element
  can be empty legitimately, for example when it is used as a placeholder which will later be filled
  in by a script, or when the element is part of a template and would on most pages be filled in but
  on some pages is not relevant.</p>

  <p>Conformance checkers are encouraged to provide a mechanism for authors to find elements that
  fail to fulfill this requirement, as an authoring aid.</p>

  <p>The following elements are palpable content:</p>

  
  <ul class="brief category-list"><li><code id=palpable-content:the-a-element><a href=#the-a-element>a</a></code><li><code id=palpable-content:the-abbr-element><a href=#the-abbr-element>abbr</a></code><li><code id=palpable-content:the-address-element><a href=#the-address-element>address</a></code><li><code id=palpable-content:the-article-element><a href=#the-article-element>article</a></code><li><code id=palpable-content:the-aside-element><a href=#the-aside-element>aside</a></code><li><code id=palpable-content:the-audio-element-2><a href=#the-audio-element>audio</a></code> (if the <code id=palpable-content:attr-media-controls><a href=#attr-media-controls>controls</a></code> attribute is present)<li><code id=palpable-content:the-b-element><a href=#the-b-element>b</a></code><li><code id=palpable-content:the-bdi-element><a href=#the-bdi-element>bdi</a></code><li><code id=palpable-content:the-bdo-element><a href=#the-bdo-element>bdo</a></code><li><code id=palpable-content:the-blockquote-element><a href=#the-blockquote-element>blockquote</a></code><li><code id=palpable-content:the-button-element><a href=#the-button-element>button</a></code><li><code id=palpable-content:the-canvas-element-2><a href=#the-canvas-element>canvas</a></code><li><code id=palpable-content:the-cite-element><a href=#the-cite-element>cite</a></code><li><code id=palpable-content:the-code-element><a href=#the-code-element>code</a></code><li><code id=palpable-content:the-data-element><a href=#the-data-element>data</a></code><li><code id=palpable-content:the-details-element><a href=#the-details-element>details</a></code><li><code id=palpable-content:the-dfn-element><a href=#the-dfn-element>dfn</a></code><li><code id=palpable-content:the-div-element><a href=#the-div-element>div</a></code><li><code id=palpable-content:the-dl-element><a href=#the-dl-element>dl</a></code> (if the element's children include at least one name-value group)<li><code id=palpable-content:the-em-element><a href=#the-em-element>em</a></code><li><code id=palpable-content:the-embed-element><a href=#the-embed-element>embed</a></code><li><code id=palpable-content:the-fieldset-element><a href=#the-fieldset-element>fieldset</a></code><li><code id=palpable-content:the-figure-element><a href=#the-figure-element>figure</a></code><li><code id=palpable-content:the-footer-element><a href=#the-footer-element>footer</a></code><li><code id=palpable-content:the-form-element><a href=#the-form-element>form</a></code><li><code id=palpable-content:the-h1,-h2,-h3,-h4,-h5,-and-h6-elements><a href=#the-h1,-h2,-h3,-h4,-h5,-and-h6-elements>h1</a></code><li><code id=palpable-content:the-h1,-h2,-h3,-h4,-h5,-and-h6-elements-2><a href=#the-h1,-h2,-h3,-h4,-h5,-and-h6-elements>h2</a></code><li><code id=palpable-content:the-h1,-h2,-h3,-h4,-h5,-and-h6-elements-3><a href=#the-h1,-h2,-h3,-h4,-h5,-and-h6-elements>h3</a></code><li><code id=palpable-content:the-h1,-h2,-h3,-h4,-h5,-and-h6-elements-4><a href=#the-h1,-h2,-h3,-h4,-h5,-and-h6-elements>h4</a></code><li><code id=palpable-content:the-h1,-h2,-h3,-h4,-h5,-and-h6-elements-5><a href=#the-h1,-h2,-h3,-h4,-h5,-and-h6-elements>h5</a></code><li><code id=palpable-content:the-h1,-h2,-h3,-h4,-h5,-and-h6-elements-6><a href=#the-h1,-h2,-h3,-h4,-h5,-and-h6-elements>h6</a></code><li><code id=palpable-content:the-header-element><a href=#the-header-element>header</a></code><li><code id=palpable-content:the-hgroup-element><a href=#the-hgroup-element>hgroup</a></code><li><code id=palpable-content:the-i-element><a href=#the-i-element>i</a></code><li><code id=palpable-content:the-iframe-element><a href=#the-iframe-element>iframe</a></code><li><code id=palpable-content:the-img-element-2><a href=#the-img-element>img</a></code><li><code id=palpable-content:the-input-element><a href=#the-input-element>input</a></code> (if the <code id=palpable-content:attr-input-type><a href=#attr-input-type>type</a></code> attribute is <em>not</em> in the <a href="#hidden-state-(type=hidden)" id="palpable-content:hidden-state-(type=hidden)">Hidden</a> state)<li><code id=palpable-content:the-ins-element><a href=#the-ins-element>ins</a></code><li><code id=palpable-content:the-kbd-element><a href=#the-kbd-element>kbd</a></code><li><code id=palpable-content:the-label-element><a href=#the-label-element>label</a></code><li><code id=palpable-content:the-main-element><a href=#the-main-element>main</a></code><li><code id=palpable-content:the-map-element><a href=#the-map-element>map</a></code><li><code id=palpable-content:the-mark-element><a href=#the-mark-element>mark</a></code><li><a id=palpable-content:mathml-math href=https://www.w3.org/Math/draft-spec/chapter2.html#interf.toplevel data-x-internal=mathml-math>MathML <code>math</code></a><li><code id=palpable-content:the-menu-element><a href=#the-menu-element>menu</a></code> (if the element's children include at least one <code id=palpable-content:the-li-element><a href=#the-li-element>li</a></code> element)<li><code id=palpable-content:the-meter-element><a href=#the-meter-element>meter</a></code><li><code id=palpable-content:the-nav-element><a href=#the-nav-element>nav</a></code><li><code id=palpable-content:the-object-element><a href=#the-object-element>object</a></code><li><code id=palpable-content:the-ol-element><a href=#the-ol-element>ol</a></code> (if the element's children include at least one <code id=palpable-content:the-li-element-2><a href=#the-li-element>li</a></code> element)<li><code id=palpable-content:the-output-element><a href=#the-output-element>output</a></code><li><code id=palpable-content:the-p-element><a href=#the-p-element>p</a></code><li><code id=palpable-content:the-pre-element><a href=#the-pre-element>pre</a></code><li><code id=palpable-content:the-progress-element><a href=#the-progress-element>progress</a></code><li><code id=palpable-content:the-q-element><a href=#the-q-element>q</a></code><li><code id=palpable-content:the-ruby-element><a href=#the-ruby-element>ruby</a></code><li><code id=palpable-content:the-s-element><a href=#the-s-element>s</a></code><li><code id=palpable-content:the-samp-element><a href=#the-samp-element>samp</a></code><li><code id=palpable-content:the-section-element><a href=#the-section-element>section</a></code><li><code id=palpable-content:the-select-element><a href=#the-select-element>select</a></code><li><code id=palpable-content:the-small-element><a href=#the-small-element>small</a></code><li><code id=palpable-content:the-span-element><a href=#the-span-element>span</a></code><li><code id=palpable-content:the-strong-element><a href=#the-strong-element>strong</a></code><li><code id=palpable-content:the-sub-and-sup-elements><a href=#the-sub-and-sup-elements>sub</a></code><li><code id=palpable-content:the-sub-and-sup-elements-2><a href=#the-sub-and-sup-elements>sup</a></code><li><a id=palpable-content:svg-svg href=https://svgwg.org/svg2-draft/struct.html#SVGElement data-x-internal=svg-svg>SVG <code>svg</code></a><li><code id=palpable-content:the-table-element><a href=#the-table-element>table</a></code><li><code id=palpable-content:the-textarea-element><a href=#the-textarea-element>textarea</a></code><li><code id=palpable-content:the-time-element><a href=#the-time-element>time</a></code><li><code id=palpable-content:the-u-element><a href=#the-u-element>u</a></code><li><code id=palpable-content:the-ul-element><a href=#the-ul-element>ul</a></code> (if the element's children include at least one <code id=palpable-content:the-li-element-3><a href=#the-li-element>li</a></code> element)<li><code id=palpable-content:the-var-element><a href=#the-var-element>var</a></code><li><code id=palpable-content:the-video-element-2><a href=#the-video-element>video</a></code><li><a href=#autonomous-custom-element id=palpable-content:autonomous-custom-element>autonomous custom elements</a><li><a href=#text-content id=palpable-content:text-content-2>text</a> that is not <a href=#inter-element-whitespace id=palpable-content:inter-element-whitespace>inter-element whitespace</a></ul>


  <h6 id=script-supporting-elements><span class=secno>3.2.5.2.9</span> Script-supporting elements<a href=#script-supporting-elements class=self-link></a></h6>

  <p><dfn id=script-supporting-elements-2>Script-supporting elements</dfn> are those that do not <a href=#represents id=script-supporting-elements:represents>represent</a> anything themselves (i.e. they are not rendered), but are
  used to support scripts, e.g. to provide functionality for the user.</p>

  <p>The following elements are script-supporting elements:</p>

  
  <ul class="brief category-list"><li><code id=script-supporting-elements:the-script-element><a href=#the-script-element>script</a></code><li><code id=script-supporting-elements:the-template-element><a href=#the-template-element>template</a></code></ul>



  <h5 id=transparent-content-models><span class=secno>3.2.5.3</span> Transparent content models<a href=#transparent-content-models class=self-link></a></h5>

  <p>Some elements are described as <dfn id=transparent>transparent</dfn>; they have "transparent" in the
  description of their content model. The content model of a <a href=#transparent id=transparent-content-models:transparent>transparent</a> element is
  derived from the content model of its parent element: the elements required in the part of the
  content model that is "transparent" are the same elements as required in the part of the content
  model of the parent of the transparent element in which the transparent element finds itself.</p>

  <div class=example>

   <p>For instance, an <code id=transparent-content-models:the-ins-element><a href=#the-ins-element>ins</a></code> element inside a <code id=transparent-content-models:the-ruby-element><a href=#the-ruby-element>ruby</a></code> element cannot contain an
   <code id=transparent-content-models:the-rt-element><a href=#the-rt-element>rt</a></code> element, because the part of the <code id=transparent-content-models:the-ruby-element-2><a href=#the-ruby-element>ruby</a></code> element's content model that
   allows <code id=transparent-content-models:the-ins-element-2><a href=#the-ins-element>ins</a></code> elements is the part that allows <a href=#phrasing-content-2 id=transparent-content-models:phrasing-content-2>phrasing content</a>, and the
   <code id=transparent-content-models:the-rt-element-2><a href=#the-rt-element>rt</a></code> element is not <a href=#phrasing-content-2 id=transparent-content-models:phrasing-content-2-2>phrasing content</a>.</p>

  </div>



  <p class=note>In some cases, where transparent elements are nested in each other, the process
  has to be applied iteratively.</p>

  <div class=example>

   <p>Consider the following markup fragment:</p>

   <pre><code class='html'><c- p>&lt;</c-><c- f>p</c-><c- p>&gt;&lt;</c-><c- f>object</c-><c- p>&gt;&lt;</c-><c- f>param</c-><c- p>&gt;&lt;</c-><c- f>ins</c-><c- p>&gt;&lt;</c-><c- f>map</c-><c- p>&gt;&lt;</c-><c- f>a</c-> <c- e>href</c-><c- o>=</c-><c- s>&quot;/&quot;</c-><c- p>&gt;</c->Apples<c- p>&lt;/</c-><c- f>a</c-><c- p>&gt;&lt;/</c-><c- f>map</c-><c- p>&gt;&lt;/</c-><c- f>ins</c-><c- p>&gt;&lt;/</c-><c- f>object</c-><c- p>&gt;&lt;/</c-><c- f>p</c-><c- p>&gt;</c-></code></pre>

   <p>To check whether "Apples" is allowed inside the <code id=transparent-content-models:the-a-element><a href=#the-a-element>a</a></code> element, the content models are
   examined. The <code id=transparent-content-models:the-a-element-2><a href=#the-a-element>a</a></code> element's content model is transparent, as is the <code id=transparent-content-models:the-map-element><a href=#the-map-element>map</a></code>
   element's, as is the <code id=transparent-content-models:the-ins-element-3><a href=#the-ins-element>ins</a></code> element's, as is the part of the <code id=transparent-content-models:the-object-element><a href=#the-object-element>object</a></code>
   element's in which the <code id=transparent-content-models:the-ins-element-4><a href=#the-ins-element>ins</a></code> element is found. The <code id=transparent-content-models:the-object-element-2><a href=#the-object-element>object</a></code> element is
   found in the <code id=transparent-content-models:the-p-element><a href=#the-p-element>p</a></code> element, whose content model is <a href=#phrasing-content-2 id=transparent-content-models:phrasing-content-2-3>phrasing content</a>. Thus,
   "Apples" is allowed, as text is phrasing content.</p>

  </div>

  <p>When a transparent element has no parent, then the part of its content model that is
  "transparent" must instead be treated as accepting any <a href=#flow-content-2 id=transparent-content-models:flow-content-2>flow content</a>.</p>


  <h5 id=paragraphs><span class=secno>3.2.5.4</span> Paragraphs<a href=#paragraphs class=self-link></a></h5>

  <p class=note>The term <a href=#paragraph id=paragraphs:paragraph>paragraph</a> as defined in this section is used for more than
  just the definition of the <code id=paragraphs:the-p-element><a href=#the-p-element>p</a></code> element. The <a href=#paragraph id=paragraphs:paragraph-2>paragraph</a> concept defined here
  is used to describe how to interpret documents. The <code id=paragraphs:the-p-element-2><a href=#the-p-element>p</a></code> element is merely one of
  several ways of marking up a <a href=#paragraph id=paragraphs:paragraph-3>paragraph</a>.</p>

  
  <p>A <dfn id=paragraph>paragraph</dfn> is typically a run of <a href=#phrasing-content-2 id=paragraphs:phrasing-content-2>phrasing content</a> that forms a block
  of text with one or more sentences that discuss a particular topic, as in typography, but can also
  be used for more general thematic grouping. For instance, an address is also a paragraph, as is a
  part of a form, a byline, or a stanza in a poem.</p>

  <div class=example>

   <p>In the following example, there are two paragraphs in a section. There is also a heading,
   which contains phrasing content that is not a paragraph. Note how the comments and
   <a href=#inter-element-whitespace id=paragraphs:inter-element-whitespace>inter-element whitespace</a> do not form paragraphs.</p>

   <pre><code class='html'><c- p>&lt;</c-><c- f>section</c-><c- p>&gt;</c->
  <c- p>&lt;</c-><c- f>h1</c-><c- p>&gt;</c->Example of paragraphs<c- p>&lt;/</c-><c- f>h1</c-><c- p>&gt;</c->
  This is the <c- p>&lt;</c-><c- f>em</c-><c- p>&gt;</c->first<c- p>&lt;/</c-><c- f>em</c-><c- p>&gt;</c-> paragraph in this example.
  <c- p>&lt;</c-><c- f>p</c-><c- p>&gt;</c->This is the second.<c- p>&lt;/</c-><c- f>p</c-><c- p>&gt;</c->
  <c- c>&lt;!-- This is not a paragraph. --&gt;</c->
<c- p>&lt;/</c-><c- f>section</c-><c- p>&gt;</c-></code></pre>

  </div>

  <p>Paragraphs in <a href=#flow-content-2 id=paragraphs:flow-content-2>flow content</a> are defined relative to what the document looks like
  without the <code id=paragraphs:the-a-element><a href=#the-a-element>a</a></code>, <code id=paragraphs:the-ins-element><a href=#the-ins-element>ins</a></code>, <code id=paragraphs:the-del-element><a href=#the-del-element>del</a></code>, and <code id=paragraphs:the-map-element><a href=#the-map-element>map</a></code> elements
  complicating matters, since those elements, with their hybrid content models, can straddle
  paragraph boundaries, as shown in the first two examples below.</p>

  <p class=note>Generally, having elements straddle paragraph boundaries is best avoided.
  Maintaining such markup can be difficult.</p>

  <div class=example>

   <p>The following example takes the markup from the earlier example and puts <code id=paragraphs:the-ins-element-2><a href=#the-ins-element>ins</a></code> and
   <code id=paragraphs:the-del-element-2><a href=#the-del-element>del</a></code> elements around some of the markup to show that the text was changed (though in
   this case, the changes admittedly don't make much sense). Notice how this example has exactly the
   same paragraphs as the previous one, despite the <code id=paragraphs:the-ins-element-3><a href=#the-ins-element>ins</a></code> and <code id=paragraphs:the-del-element-3><a href=#the-del-element>del</a></code> elements
   — the <code id=paragraphs:the-ins-element-4><a href=#the-ins-element>ins</a></code> element straddles the heading and the first paragraph, and the
   <code id=paragraphs:the-del-element-4><a href=#the-del-element>del</a></code> element straddles the boundary between the two paragraphs.</p>

   <pre><code class='html'><c- p>&lt;</c-><c- f>section</c-><c- p>&gt;</c->
  <c- p>&lt;</c-><c- f>ins</c-><c- p>&gt;&lt;</c-><c- f>h1</c-><c- p>&gt;</c->Example of paragraphs<c- p>&lt;/</c-><c- f>h1</c-><c- p>&gt;</c->
  This is the <c- p>&lt;</c-><c- f>em</c-><c- p>&gt;</c->first<c- p>&lt;/</c-><c- f>em</c-><c- p>&gt;</c-> paragraph in<c- p>&lt;/</c-><c- f>ins</c-><c- p>&gt;</c-> this example<c- p>&lt;</c-><c- f>del</c-><c- p>&gt;</c->.
  <c- p>&lt;</c-><c- f>p</c-><c- p>&gt;</c->This is the second.<c- p>&lt;/</c-><c- f>p</c-><c- p>&gt;&lt;/</c-><c- f>del</c-><c- p>&gt;</c->
  <c- c>&lt;!-- This is not a paragraph. --&gt;</c->
<c- p>&lt;/</c-><c- f>section</c-><c- p>&gt;</c-></code></pre>

  </div>

  

  <p>Let <var>view</var> be a view of the DOM that replaces all <code id=paragraphs:the-a-element-2><a href=#the-a-element>a</a></code>,
  <code id=paragraphs:the-ins-element-5><a href=#the-ins-element>ins</a></code>, <code id=paragraphs:the-del-element-5><a href=#the-del-element>del</a></code>, and <code id=paragraphs:the-map-element-2><a href=#the-map-element>map</a></code> elements in the document with their <a href=#concept-html-contents id=paragraphs:concept-html-contents>contents</a>. Then, in <var>view</var>, for each run
  of sibling <a href=#phrasing-content-2 id=paragraphs:phrasing-content-2-2>phrasing content</a> nodes uninterrupted by other types of content, in an
  element that accepts content other than <a href=#phrasing-content-2 id=paragraphs:phrasing-content-2-3>phrasing content</a> as well as <a href=#phrasing-content-2 id=paragraphs:phrasing-content-2-4>phrasing
  content</a>, let <var>first</var> be the first node of the run, and let <var>last</var> be the last node of the run. For each such run that consists of at least one
  node that is neither <a href=#embedded-content-category id=paragraphs:embedded-content-category>embedded content</a> nor <a href=#inter-element-whitespace id=paragraphs:inter-element-whitespace-2>inter-element whitespace</a>, a
  paragraph exists in the original DOM from immediately before <var>first</var> to
  immediately after <var>last</var>. (Paragraphs can thus span across <code id=paragraphs:the-a-element-3><a href=#the-a-element>a</a></code>,
  <code id=paragraphs:the-ins-element-6><a href=#the-ins-element>ins</a></code>, <code id=paragraphs:the-del-element-6><a href=#the-del-element>del</a></code>, and <code id=paragraphs:the-map-element-3><a href=#the-map-element>map</a></code> elements.)</p>

  <p>Conformance checkers may warn authors of cases where they have paragraphs that overlap each
  other (this can happen with <code id=paragraphs:the-object-element><a href=#the-object-element>object</a></code>, <code id=paragraphs:the-video-element><a href=#the-video-element>video</a></code>, <code id=paragraphs:the-audio-element><a href=#the-audio-element>audio</a></code>, and
  <code id=paragraphs:the-canvas-element><a href=#the-canvas-element>canvas</a></code> elements, and indirectly through elements in other namespaces that allow HTML
  to be further embedded therein, like <a id=paragraphs:svg-svg href=https://svgwg.org/svg2-draft/struct.html#SVGElement data-x-internal=svg-svg>SVG <code>svg</code></a> or <a id=paragraphs:mathml-math href=https://www.w3.org/Math/draft-spec/chapter2.html#interf.toplevel data-x-internal=mathml-math>MathML
  <code>math</code></a>).</p>
  

  

  <p>A <a href=#paragraph id=paragraphs:paragraph-4>paragraph</a> is also formed explicitly by <code id=paragraphs:the-p-element-3><a href=#the-p-element>p</a></code> elements.</p>

  <p class=note>The <code id=paragraphs:the-p-element-4><a href=#the-p-element>p</a></code> element can be used to wrap individual paragraphs when there
  would otherwise not be any content other than phrasing content to separate the paragraphs from
  each other.</p>

  <div class=example>

   <p>In the following example, the link spans half of the first paragraph, all of the heading
   separating the two paragraphs, and half of the second paragraph. It straddles the paragraphs and
   the heading.</p>

   <pre><code class='html'><c- p>&lt;</c-><c- f>header</c-><c- p>&gt;</c->
 Welcome!
 <c- p>&lt;</c-><c- f>a</c-> <c- e>href</c-><c- o>=</c-><c- s>&quot;about.html&quot;</c-><c- p>&gt;</c->
  This is home of...
  <c- p>&lt;</c-><c- f>h1</c-><c- p>&gt;</c->The Falcons!<c- p>&lt;/</c-><c- f>h1</c-><c- p>&gt;</c->
  The Lockheed Martin multirole jet fighter aircraft!
 <c- p>&lt;/</c-><c- f>a</c-><c- p>&gt;</c->
 This page discusses the F-16 Fighting Falcon&apos;s innermost secrets.
<c- p>&lt;/</c-><c- f>header</c-><c- p>&gt;</c-></code></pre>

   <p>Here is another way of marking this up, this time showing the paragraphs explicitly, and
   splitting the one link element into three:</p>

   <pre><code class='html'><c- p>&lt;</c-><c- f>header</c-><c- p>&gt;</c->
 <c- p>&lt;</c-><c- f>p</c-><c- p>&gt;</c->Welcome! <c- p>&lt;</c-><c- f>a</c-> <c- e>href</c-><c- o>=</c-><c- s>&quot;about.html&quot;</c-><c- p>&gt;</c->This is home of...<c- p>&lt;/</c-><c- f>a</c-><c- p>&gt;&lt;/</c-><c- f>p</c-><c- p>&gt;</c->
 <c- p>&lt;</c-><c- f>h1</c-><c- p>&gt;&lt;</c-><c- f>a</c-> <c- e>href</c-><c- o>=</c-><c- s>&quot;about.html&quot;</c-><c- p>&gt;</c->The Falcons!<c- p>&lt;/</c-><c- f>a</c-><c- p>&gt;&lt;/</c-><c- f>h1</c-><c- p>&gt;</c->
 <c- p>&lt;</c-><c- f>p</c-><c- p>&gt;&lt;</c-><c- f>a</c-> <c- e>href</c-><c- o>=</c-><c- s>&quot;about.html&quot;</c-><c- p>&gt;</c->The Lockheed Martin multirole jet
 fighter aircraft!<c- p>&lt;/</c-><c- f>a</c-><c- p>&gt;</c-> This page discusses the F-16 Fighting
 Falcon&apos;s innermost secrets.<c- p>&lt;/</c-><c- f>p</c-><c- p>&gt;</c->
<c- p>&lt;/</c-><c- f>header</c-><c- p>&gt;</c-></code></pre>

  </div>

  <div class=example>

   

   <p>It is possible for paragraphs to overlap when using certain elements that define fallback
   content. For example, in the following section:</p>

   <pre><code class='html'><c- p>&lt;</c-><c- f>section</c-><c- p>&gt;</c->
 <c- p>&lt;</c-><c- f>h1</c-><c- p>&gt;</c->My Cats<c- p>&lt;/</c-><c- f>h1</c-><c- p>&gt;</c->
 You can play with my cat simulator.
 <c- p>&lt;</c-><c- f>object</c-> <c- e>data</c-><c- o>=</c-><c- s>&quot;cats.sim&quot;</c-><c- p>&gt;</c->
  To see the cat simulator, use one of the following links:
  <c- p>&lt;</c-><c- f>ul</c-><c- p>&gt;</c->
   <c- p>&lt;</c-><c- f>li</c-><c- p>&gt;&lt;</c-><c- f>a</c-> <c- e>href</c-><c- o>=</c-><c- s>&quot;cats.sim&quot;</c-><c- p>&gt;</c->Download simulator file<c- p>&lt;/</c-><c- f>a</c-><c- p>&gt;</c->
   <c- p>&lt;</c-><c- f>li</c-><c- p>&gt;&lt;</c-><c- f>a</c-> <c- e>href</c-><c- o>=</c-><c- s>&quot;https://sims.example.com/watch?v=LYds5xY4INU&quot;</c-><c- p>&gt;</c->Use online simulator<c- p>&lt;/</c-><c- f>a</c-><c- p>&gt;</c->
  <c- p>&lt;/</c-><c- f>ul</c-><c- p>&gt;</c->
  Alternatively, upgrade to the Mellblom Browser.
 <c- p>&lt;/</c-><c- f>object</c-><c- p>&gt;</c->
 I&apos;m quite proud of it.
<c- p>&lt;/</c-><c- f>section</c-><c- p>&gt;</c-></code></pre>

   <p>There are five paragraphs:</p>

   <ol class=brief><li>The paragraph that says "You can play with my cat simulator. <i>object</i> I'm
    quite proud of it.", where <i>object</i> is the <code id=paragraphs:the-object-element-2><a href=#the-object-element>object</a></code> element.<li>The paragraph that says "To see the cat simulator, use one of the following links:".<li>The paragraph that says "Download simulator file".<li>The paragraph that says "Use online simulator".<li>The paragraph that says "Alternatively, upgrade to the Mellblom Browser.".</ol>

   <p>The first paragraph is overlapped by the other four. A user agent that supports the "cats.sim"
   resource will only show the first one, but a user agent that shows the fallback will confusingly
   show the first sentence of the first paragraph as if it was in the same paragraph as the second
   one, and will show the last paragraph as if it was at the start of the second sentence of the
   first paragraph.</p>

   <p>To avoid this confusion, explicit <code id=paragraphs:the-p-element-5><a href=#the-p-element>p</a></code> elements can be used. For example:</p>

   
   <pre><code class='html'><c- p>&lt;</c-><c- f>section</c-><c- p>&gt;</c->
 <c- p>&lt;</c-><c- f>h1</c-><c- p>&gt;</c->My Cats<c- p>&lt;/</c-><c- f>h1</c-><c- p>&gt;</c->
 <c- p>&lt;</c-><c- f>p</c-><c- p>&gt;</c->You can play with my cat simulator.<c- p>&lt;/</c-><c- f>p</c-><c- p>&gt;</c->
 <c- p>&lt;</c-><c- f>object</c-> <c- e>data</c-><c- o>=</c-><c- s>&quot;cats.sim&quot;</c-><c- p>&gt;</c->
  <c- p>&lt;</c-><c- f>p</c-><c- p>&gt;</c->To see the cat simulator, use one of the following links:<c- p>&lt;/</c-><c- f>p</c-><c- p>&gt;</c->
  <c- p>&lt;</c-><c- f>ul</c-><c- p>&gt;</c->
   <c- p>&lt;</c-><c- f>li</c-><c- p>&gt;&lt;</c-><c- f>a</c-> <c- e>href</c-><c- o>=</c-><c- s>&quot;cats.sim&quot;</c-><c- p>&gt;</c->Download simulator file<c- p>&lt;/</c-><c- f>a</c-><c- p>&gt;</c->
   <c- p>&lt;</c-><c- f>li</c-><c- p>&gt;&lt;</c-><c- f>a</c-> <c- e>href</c-><c- o>=</c-><c- s>&quot;https://sims.example.com/watch?v=LYds5xY4INU&quot;</c-><c- p>&gt;</c->Use online simulator<c- p>&lt;/</c-><c- f>a</c-><c- p>&gt;</c->
  <c- p>&lt;/</c-><c- f>ul</c-><c- p>&gt;</c->
  <c- p>&lt;</c-><c- f>p</c-><c- p>&gt;</c->Alternatively, upgrade to the Mellblom Browser.<c- p>&lt;/</c-><c- f>p</c-><c- p>&gt;</c->
 <c- p>&lt;/</c-><c- f>object</c-><c- p>&gt;</c->
 <c- p>&lt;</c-><c- f>p</c-><c- p>&gt;</c->I&apos;m quite proud of it.<c- p>&lt;/</c-><c- f>p</c-><c- p>&gt;</c->
<c- p>&lt;/</c-><c- f>section</c-><c- p>&gt;</c-></code></pre>

  </div>


  <h4 id=global-attributes><span class=secno>3.2.6</span> <dfn>Global attributes</dfn><a href=#global-attributes class=self-link></a></h4><div class="mdn-anno wrapped"><button onclick=toggleStatus(this) class=mdn-anno-btn><span>MDN</span></button><div class=feature><p><a href=https://developer.mozilla.org/en-US/docs/Web/HTML/Global_attributes title="Global attributes are attributes common to all HTML elements; they can be used on all elements, though they may have no effect on some elements.">Global_attributes</a></div></div>

  <p>The following attributes are common to and may be specified on all <a href=#html-elements id=global-attributes:html-elements>HTML
  elements</a> (even those not defined in this specification):</p>

  <ul class=brief><li><code id=global-attributes:the-accesskey-attribute><a href=#the-accesskey-attribute>accesskey</a></code><li><code id=global-attributes:attr-autocapitalize><a href=#attr-autocapitalize>autocapitalize</a></code><li><code id=global-attributes:attr-fe-autofocus><a href=#attr-fe-autofocus>autofocus</a></code><li><code id=global-attributes:attr-contenteditable><a href=#attr-contenteditable>contenteditable</a></code><li><code id=global-attributes:attr-dir><a href=#attr-dir>dir</a></code><li><code id=global-attributes:attr-draggable><a href=#attr-draggable>draggable</a></code><li><code id=global-attributes:attr-enterkeyhint><a href=#attr-enterkeyhint>enterkeyhint</a></code><li><code id=global-attributes:attr-hidden><a href=#attr-hidden>hidden</a></code><li><code id=global-attributes:attr-inputmode><a href=#attr-inputmode>inputmode</a></code><li><code id=global-attributes:attr-is><a href=#attr-is>is</a></code><li><code id=global-attributes:attr-itemid><a href=#attr-itemid>itemid</a></code><li><code id=global-attributes:names:-the-itemprop-attribute><a href=#names:-the-itemprop-attribute>itemprop</a></code><li><code id=global-attributes:attr-itemref><a href=#attr-itemref>itemref</a></code><li><code id=global-attributes:attr-itemscope><a href=#attr-itemscope>itemscope</a></code><li><code id=global-attributes:attr-itemtype><a href=#attr-itemtype>itemtype</a></code><li><code id=global-attributes:attr-lang><a href=#attr-lang>lang</a></code><li><code id=global-attributes:attr-nonce><a href=#attr-nonce>nonce</a></code><li><code id=global-attributes:attr-spellcheck><a href=#attr-spellcheck>spellcheck</a></code><li><code id=global-attributes:attr-style><a href=#attr-style>style</a></code><li><code id=global-attributes:attr-tabindex><a href=#attr-tabindex>tabindex</a></code><li><code id=global-attributes:attr-title><a href=#attr-title>title</a></code><li><code id=global-attributes:attr-translate><a href=#attr-translate>translate</a></code></ul>

  

  <p>These attributes are only defined by this specification as attributes for <a href=#html-elements id=global-attributes:html-elements-2>HTML
  elements</a>. When this specification refers to elements having these attributes, elements from
  namespaces that are not defined as having these attributes must not be considered as being
  elements with these attributes.</p>

  <div class=example>

   <p>For example, in the following XML fragment, the "<code>bogus</code>" element does not
   have a <code id=global-attributes:attr-dir-2><a href=#attr-dir>dir</a></code> attribute as defined in this specification, despite
   having an attribute with the literal name "<code>dir</code>". Thus, <a href=#the-directionality id=global-attributes:the-directionality>the
   directionality</a> of the inner-most <code id=global-attributes:the-span-element><a href=#the-span-element>span</a></code> element is '<a href=#concept-rtl id=global-attributes:concept-rtl>rtl</a>', inherited from the <code id=global-attributes:the-div-element><a href=#the-div-element>div</a></code> element indirectly through
   the "<code>bogus</code>" element.</p>

   <pre class=bad><code class='html'><c- p>&lt;</c-><c- f>div</c-> <c- e>xmlns</c-><c- o>=</c-><c- s>&quot;http://www.w3.org/1999/xhtml&quot;</c-> <c- e>dir</c-><c- o>=</c-><c- s>&quot;rtl&quot;</c-><c- p>&gt;</c->
 <c- p>&lt;</c-><c- f>bogus</c-> <c- e>xmlns</c-><c- o>=</c-><c- s>&quot;https://example.net/ns&quot;</c-> <c- e>dir</c-><c- o>=</c-><c- s>&quot;ltr&quot;</c-><c- p>&gt;</c->
  <c- p>&lt;</c-><c- f>span</c-> <c- e>xmlns</c-><c- o>=</c-><c- s>&quot;http://www.w3.org/1999/xhtml&quot;</c-><c- p>&gt;</c->
  <c- p>&lt;/</c-><c- f>span</c-><c- p>&gt;</c->
 <c- p>&lt;/</c-><c- f>bogus</c-><c- p>&gt;</c->
<c- p>&lt;/</c-><c- f>div</c-><c- p>&gt;</c-></code></pre>

  </div>

  

  <hr>

  <div class="mdn-anno wrapped before"><button onclick=toggleStatus(this) class=mdn-anno-btn><b title="Support in all current engines." class=all-engines-flag>✔</b><span>MDN</span></button><div class=feature><p><a href=https://developer.mozilla.org/en-US/docs/Web/HTML/Global_attributes/class title="The class global attribute is a space-separated list of the case-sensitive classes of the element. Classes allow CSS and Javascript to select and access specific elements via the class selectors or functions like the DOM method document.getElementsByClassName.">Global_attributes/class</a><p class=all-engines-text>Support in all current engines.<div class=support><span class="firefox yes"><span>Firefox</span><span>32+</span></span><span class="safari yes"><span>Safari</span><span>Yes</span></span><span class="chrome yes"><span>Chrome</span><span>Yes</span></span><hr><span class="opera yes"><span>Opera</span><span>Yes</span></span><span class="edge_blink yes"><span>Edge</span><span>Yes</span></span><hr><span class="edge yes"><span>Edge (Legacy)</span><span>12+</span></span><span class="ie yes"><span>Internet Explorer</span><span>Yes</span></span><hr><span class="firefox_android yes"><span>Firefox Android</span><span>32+</span></span><span class="safari_ios yes"><span>Safari iOS</span><span>Yes</span></span><span class="chrome_android yes"><span>Chrome Android</span><span>Yes</span></span><span class="webview_android yes"><span>WebView Android</span><span>Yes</span></span><span class="samsunginternet_android yes"><span>Samsung Internet</span><span>Yes</span></span><span class="opera_android yes"><span>Opera Android</span><span>Yes</span></span></div></div><div class=feature><p><a href=https://developer.mozilla.org/en-US/docs/Web/HTML/Global_attributes/id title="The id global attribute defines an identifier (ID) which must be unique in the whole document. Its purpose is to identify the element when linking (using a fragment identifier), scripting, or styling (with CSS).">Global_attributes/id</a><p class=all-engines-text>Support in all current engines.<div class=support><span class="firefox yes"><span>Firefox</span><span>32+</span></span><span class="safari yes"><span>Safari</span><span>Yes</span></span><span class="chrome yes"><span>Chrome</span><span>Yes</span></span><hr><span class="opera yes"><span>Opera</span><span>Yes</span></span><span class="edge_blink yes"><span>Edge</span><span>Yes</span></span><hr><span class="edge yes"><span>Edge (Legacy)</span><span>12+</span></span><span class="ie yes"><span>Internet Explorer</span><span>Yes</span></span><hr><span class="firefox_android yes"><span>Firefox Android</span><span>32+</span></span><span class="safari_ios yes"><span>Safari iOS</span><span>Yes</span></span><span class="chrome_android yes"><span>Chrome Android</span><span>Yes</span></span><span class="webview_android yes"><span>WebView Android</span><span>Yes</span></span><span class="samsunginternet_android yes"><span>Samsung Internet</span><span>Yes</span></span><span class="opera_android yes"><span>Opera Android</span><span>Yes</span></span></div></div><div class=feature><p><a href=https://developer.mozilla.org/en-US/docs/Web/HTML/Global_attributes/slot title="The slot global attribute assigns a slot in a shadow DOM shadow tree to an element: An element with a slot attribute is assigned to the slot created by the <slot> element whose name attribute's value matches that slot attribute's value.">Global_attributes/slot</a><p class=all-engines-text>Support in all current engines.<div class=support><span class="firefox yes"><span>Firefox</span><span>63+</span></span><span class="safari yes"><span>Safari</span><span>10+</span></span><span class="chrome yes"><span>Chrome</span><span>53+</span></span><hr><span class="opera yes"><span>Opera</span><span>40+</span></span><span class="edge_blink yes"><span>Edge</span><span>79+</span></span><hr><span class="edge no"><span>Edge (Legacy)</span><span>No</span></span><span class="ie unknown"><span>Internet Explorer</span><span>?</span></span><hr><span class="firefox_android yes"><span>Firefox Android</span><span>63+</span></span><span class="safari_ios yes"><span>Safari iOS</span><span>10+</span></span><span class="chrome_android yes"><span>Chrome Android</span><span>53+</span></span><span class="webview_android yes"><span>WebView Android</span><span>53+</span></span><span class="samsunginternet_android yes"><span>Samsung Internet</span><span>6.0+</span></span><span class="opera_android yes"><span>Opera Android</span><span>41+</span></span></div></div></div><p><cite>DOM</cite> defines the user agent requirements for the <dfn data-dfn-for=global id=classes data-dfn-type=element-attr><code>class</code></dfn>, <dfn data-dfn-for=global id=the-id-attribute data-dfn-type=element-attr><code>id</code></dfn>, and <dfn data-dfn-for=global id=attr-slot data-dfn-type=element-attr><code>slot</code></dfn> attributes for any element in any namespace. <a href=#refsDOM>[DOM]</a></p>

  <p>The <code id=global-attributes:classes><a href=#classes>class</a></code>, <code id=global-attributes:the-id-attribute><a href=#the-id-attribute>id</a></code>, and <code id=global-attributes:attr-slot><a href=#attr-slot>slot</a></code> attributes may be specified on all <a href=#html-elements id=global-attributes:html-elements-3>HTML elements</a>.</p>

  <p>When specified on <a href=#html-elements id=global-attributes:html-elements-4>HTML elements</a>, the <code id=global-attributes:classes-2><a href=#classes>class</a></code>
  attribute must have a value that is a <a href=#set-of-space-separated-tokens id=global-attributes:set-of-space-separated-tokens>set of space-separated tokens</a> representing the
  various classes that the element belongs to.</p>

  <div class=note>

  <p>Assigning classes to an element affects class matching in selectors in CSS, the <code id=global-attributes:dom-document-getelementsbyclassname><a data-x-internal=dom-document-getelementsbyclassname href=https://dom.spec.whatwg.org/#dom-document-getelementsbyclassname>getElementsByClassName()</a></code> method in the DOM,
  and other such features.</p>

  <p>There are no additional restrictions on the tokens authors can use in the <code id=global-attributes:classes-3><a href=#classes>class</a></code> attribute, but authors are encouraged to use values that describe
  the nature of the content, rather than values that describe the desired presentation of the
  content.</p>

  </div>

  <p>When specified on <a href=#html-elements id=global-attributes:html-elements-5>HTML elements</a>, the <code id=global-attributes:the-id-attribute-2><a href=#the-id-attribute>id</a></code> attribute
  value must be unique amongst all the <a href=https://dom.spec.whatwg.org/#concept-id id=global-attributes:concept-id data-x-internal=concept-id>IDs</a> in the element's
  <a id=global-attributes:tree href=https://dom.spec.whatwg.org/#concept-tree data-x-internal=tree>tree</a> and must contain at least one character. The value must not contain any
  <a id=global-attributes:space-characters href=https://infra.spec.whatwg.org/#ascii-whitespace data-x-internal=space-characters>ASCII whitespace</a>.</p>

  

  <div class=note>

  <p>The <code id=global-attributes:the-id-attribute-3><a href=#the-id-attribute>id</a></code> attribute specifies its element's <a href=https://dom.spec.whatwg.org/#concept-id id=global-attributes:concept-id-2 data-x-internal=concept-id>unique identifier (ID)</a>.</p>

  <p>There are no other restrictions on what form an ID can take; in particular, IDs can consist of
  just digits, start with a digit, start with an underscore, consist of just punctuation, etc.</p>

  <p>An element's <a href=https://dom.spec.whatwg.org/#concept-id id=global-attributes:concept-id-3 data-x-internal=concept-id>unique identifier</a> can be used for a variety of
  purposes, most notably as a way to link to specific parts of a document using <a href=https://url.spec.whatwg.org/#concept-url-fragment id=global-attributes:concept-url-fragment data-x-internal=concept-url-fragment>fragments</a>, as a way to target an element when scripting, and
  as a way to style a specific element from CSS.</p>

  </div>

  

  <p>Identifiers are opaque strings. Particular meanings should not be derived from the value of the
  <code id=global-attributes:the-id-attribute-4><a href=#the-id-attribute>id</a></code> attribute.</p>

  

  <p>There are no conformance requirements for the <code id=global-attributes:attr-slot-2><a href=#attr-slot>slot</a></code> attribute
  specific to <a href=#html-elements id=global-attributes:html-elements-6>HTML elements</a>.</p>

  <p class=note>The <code id=global-attributes:attr-slot-3><a href=#attr-slot>slot</a></code> attribute is used to <a id=global-attributes:assign-a-slot href=https://dom.spec.whatwg.org/#assign-a-slot data-x-internal=assign-a-slot>assign a
  slot</a> to an element: an element with a <code id=global-attributes:attr-slot-4><a href=#attr-slot>slot</a></code> attribute is
  <a href=https://dom.spec.whatwg.org/#assign-a-slot id=global-attributes:assign-a-slot-2 data-x-internal=assign-a-slot>assigned</a> to the <a href=https://dom.spec.whatwg.org/#concept-slot id=global-attributes:concept-slot data-x-internal=concept-slot>slot</a>
  created by the <code id=global-attributes:the-slot-element><a href=#the-slot-element>slot</a></code> element whose <a href=#attr-slot-name id=global-attributes:attr-slot-name>name</a>
  attribute's value matches that <code id=global-attributes:attr-slot-5><a href=#attr-slot>slot</a></code> attribute's value — but only
  if that <code id=global-attributes:the-slot-element-2><a href=#the-slot-element>slot</a></code> element finds itself in the <a id=global-attributes:shadow-tree href=https://dom.spec.whatwg.org/#concept-shadow-tree data-x-internal=shadow-tree>shadow tree</a> whose
  <a id=global-attributes:root href=https://dom.spec.whatwg.org/#concept-tree-root data-x-internal=root>root</a>'s <a href=https://dom.spec.whatwg.org/#concept-documentfragment-host id=global-attributes:concept-documentfragment-host data-x-internal=concept-documentfragment-host>host</a> has the corresponding
  <code id=global-attributes:attr-slot-6><a href=#attr-slot>slot</a></code> attribute value.</p>

  <hr>

  <p>To enable assistive technology products to expose a more fine-grained interface than is
  otherwise possible with HTML elements and attributes, a set of <a href=#wai-aria>annotations
  for assistive technology products</a> can be specified (the ARIA <code id=global-attributes:attr-aria-role><a href=#attr-aria-role>role</a></code> and <code id=global-attributes:attr-aria-*><a href=#attr-aria-*>aria-*</a></code> attributes). <a href=#refsARIA>[ARIA]</a></p>

  <hr>

  <p>The following <a href=#event-handler-content-attributes id=global-attributes:event-handler-content-attributes>event handler content attributes</a> may be specified on any <a href=#html-elements id=global-attributes:html-elements-7>HTML element</a>:</p>

  <ul class=brief><li><code id=global-attributes:handler-onabort><a href=#handler-onabort>onabort</a></code><li><code id=global-attributes:handler-onauxclick><a href=#handler-onauxclick>onauxclick</a></code><li><code id=global-attributes:handler-onblur><a href=#handler-onblur>onblur</a></code>*<li><code id=global-attributes:handler-oncancel><a href=#handler-oncancel>oncancel</a></code><li><code id=global-attributes:handler-oncanplay><a href=#handler-oncanplay>oncanplay</a></code><li><code id=global-attributes:handler-oncanplaythrough><a href=#handler-oncanplaythrough>oncanplaythrough</a></code><li><code id=global-attributes:handler-onchange><a href=#handler-onchange>onchange</a></code><li><code id=global-attributes:handler-onclick><a href=#handler-onclick>onclick</a></code><li><code id=global-attributes:handler-onclose><a href=#handler-onclose>onclose</a></code><li><code id=global-attributes:handler-oncontextmenu><a href=#handler-oncontextmenu>oncontextmenu</a></code><li><code id=global-attributes:handler-oncopy><a href=#handler-oncopy>oncopy</a></code><li><code id=global-attributes:handler-oncuechange><a href=#handler-oncuechange>oncuechange</a></code><li><code id=global-attributes:handler-oncut><a href=#handler-oncut>oncut</a></code><li><code id=global-attributes:handler-ondblclick><a href=#handler-ondblclick>ondblclick</a></code><li><code id=global-attributes:handler-ondrag><a href=#handler-ondrag>ondrag</a></code><li><code id=global-attributes:handler-ondragend><a href=#handler-ondragend>ondragend</a></code><li><code id=global-attributes:handler-ondragenter><a href=#handler-ondragenter>ondragenter</a></code><li><code id=global-attributes:handler-ondragleave><a href=#handler-ondragleave>ondragleave</a></code><li><code id=global-attributes:handler-ondragover><a href=#handler-ondragover>ondragover</a></code><li><code id=global-attributes:handler-ondragstart><a href=#handler-ondragstart>ondragstart</a></code><li><code id=global-attributes:handler-ondrop><a href=#handler-ondrop>ondrop</a></code><li><code id=global-attributes:handler-ondurationchange><a href=#handler-ondurationchange>ondurationchange</a></code><li><code id=global-attributes:handler-onemptied><a href=#handler-onemptied>onemptied</a></code><li><code id=global-attributes:handler-onended><a href=#handler-onended>onended</a></code><li><code id=global-attributes:handler-onerror><a href=#handler-onerror>onerror</a></code>*<li><code id=global-attributes:handler-onfocus><a href=#handler-onfocus>onfocus</a></code>*<li><code id=global-attributes:handler-onformdata><a href=#handler-onformdata>onformdata</a></code><li><code id=global-attributes:handler-oninput><a href=#handler-oninput>oninput</a></code><li><code id=global-attributes:handler-oninvalid><a href=#handler-oninvalid>oninvalid</a></code><li><code id=global-attributes:handler-onkeydown><a href=#handler-onkeydown>onkeydown</a></code><li><code id=global-attributes:handler-onkeypress><a href=#handler-onkeypress>onkeypress</a></code><li><code id=global-attributes:handler-onkeyup><a href=#handler-onkeyup>onkeyup</a></code><li><code id=global-attributes:handler-onload><a href=#handler-onload>onload</a></code>*<li><code id=global-attributes:handler-onloadeddata><a href=#handler-onloadeddata>onloadeddata</a></code><li><code id=global-attributes:handler-onloadedmetadata><a href=#handler-onloadedmetadata>onloadedmetadata</a></code><li><code id=global-attributes:handler-onloadstart><a href=#handler-onloadstart>onloadstart</a></code><li><code id=global-attributes:handler-onmousedown><a href=#handler-onmousedown>onmousedown</a></code><li><code id=global-attributes:handler-onmouseenter><a href=#handler-onmouseenter>onmouseenter</a></code><li><code id=global-attributes:handler-onmouseleave><a href=#handler-onmouseleave>onmouseleave</a></code><li><code id=global-attributes:handler-onmousemove><a href=#handler-onmousemove>onmousemove</a></code><li><code id=global-attributes:handler-onmouseout><a href=#handler-onmouseout>onmouseout</a></code><li><code id=global-attributes:handler-onmouseover><a href=#handler-onmouseover>onmouseover</a></code><li><code id=global-attributes:handler-onmouseup><a href=#handler-onmouseup>onmouseup</a></code><li><code id=global-attributes:handler-onpaste><a href=#handler-onpaste>onpaste</a></code><li><code id=global-attributes:handler-onpause><a href=#handler-onpause>onpause</a></code><li><code id=global-attributes:handler-onplay><a href=#handler-onplay>onplay</a></code><li><code id=global-attributes:handler-onplaying><a href=#handler-onplaying>onplaying</a></code><li><code id=global-attributes:handler-onprogress><a href=#handler-onprogress>onprogress</a></code><li><code id=global-attributes:handler-onratechange><a href=#handler-onratechange>onratechange</a></code><li><code id=global-attributes:handler-onreset><a href=#handler-onreset>onreset</a></code><li><code id=global-attributes:handler-onresize><a href=#handler-onresize>onresize</a></code>*<li><code id=global-attributes:handler-onscroll><a href=#handler-onscroll>onscroll</a></code>*<li><code id=global-attributes:handler-onsecuritypolicyviolation><a href=#handler-onsecuritypolicyviolation>onsecuritypolicyviolation</a></code><li><code id=global-attributes:handler-onseeked><a href=#handler-onseeked>onseeked</a></code><li><code id=global-attributes:handler-onseeking><a href=#handler-onseeking>onseeking</a></code><li><code id=global-attributes:handler-onselect><a href=#handler-onselect>onselect</a></code><li><code id=global-attributes:handler-onslotchange><a href=#handler-onslotchange>onslotchange</a></code><li><code id=global-attributes:handler-onstalled><a href=#handler-onstalled>onstalled</a></code><li><code id=global-attributes:handler-onsubmit><a href=#handler-onsubmit>onsubmit</a></code><li><code id=global-attributes:handler-onsuspend><a href=#handler-onsuspend>onsuspend</a></code><li><code id=global-attributes:handler-ontimeupdate><a href=#handler-ontimeupdate>ontimeupdate</a></code><li><code id=global-attributes:handler-ontoggle><a href=#handler-ontoggle>ontoggle</a></code><li><code id=global-attributes:handler-onvolumechange><a href=#handler-onvolumechange>onvolumechange</a></code><li><code id=global-attributes:handler-onwaiting><a href=#handler-onwaiting>onwaiting</a></code><li><code id=global-attributes:handler-onwheel><a href=#handler-onwheel>onwheel</a></code></ul>

  <p class=note>The attributes marked with an asterisk have a different meaning when specified on
  <code id=global-attributes:the-body-element><a href=#the-body-element>body</a></code> elements as those elements expose <a href=#event-handlers id=global-attributes:event-handlers>event handlers</a> of the
  <code id=global-attributes:window><a href=#window>Window</a></code> object with the same names.</p>

  <p class=note>While these attributes apply to all elements, they are not useful on all elements.
  For example, only <a href=#media-element id=global-attributes:media-element>media elements</a> will ever receive a <code id=global-attributes:event-media-volumechange><a href=#event-media-volumechange>volumechange</a></code> event fired by the user agent.</p>

  <hr>

  <p><a href=#custom-data-attribute id=global-attributes:custom-data-attri