package HTML::ParagraphSplit;

use strict;
use warnings;

our $VERSION = '1.05';

require Exporter;

our @ISA = qw( Exporter );

our @EXPORT_OK = qw( split_paragraphs split_paragraphs_to_text );

use HTML::Entities;
use HTML::TreeBuilder;
use HTML::Tagset;
use Scalar::Util qw/ blessed /;

use vars qw( %p_content );
*p_content = *HTML::Tagset::is_Possible_Strict_P_Content;


=head1 NAME

HTML::ParagraphSplit - Change text containing HTML into a formatted HTML fragment

=head1 SYNOPSIS

  use HTML::ParagraphSplit qw( split_paragraphs_to_text split_paragraphs );

  # Read in from a file handle, output text
  print split_paragraphs_to_text(\*ARGV);

  # Convert text to nicely split text
  print split_paragraphs_to_text(<<END_OF_MARKUP);
  This is one paragraph.

  This is a another paragraph.
  END_OF_MARKUP

  # Convert to an HTML::Element object instead
  my $tree = split_paragraphs($html_input);
  print $tree->as_HTML;

  # Create your own HTML::Element object and split it
  my $tree = HTML::TreeBuilder->new;
  $tree->parse($text);
  $tree->eof;

  split_paragraphs($tree);

  my $html_fragment = $tree->guts->as_HTML;
  $tree->delete;

=head1 DESCRIPTION

The purpose of this library is to provide methods for converting double line-breaks in text to HTML paragraphs (i.e., wrap in C<E<lt>PE<gt>E<lt>/PE<gt>> tags). It can also convert single line breaks into C<E<lt>BRE<gt>> tags. In addition, markup can be mixed in as well and this library will DoTheRightThing(tm). There are a number of additional options that can modify how the paragraph splits are performed.

For example, given this input (the initial text was generated by DadaDodo L<http://www.jwz.org/dadadodo/dadadodo.cgi>, btw):

  I see over the <strong>noise</strong> but I don't understand sometimes.

  <ol><li>One</li><li>Two</li><li>Three</li><ol>

  Fortunately, we've traded the club you can't skimp on the do because This
  week! Presented by code Lounge: except, for controlling Knox video cameras
  Linux well that the reason, the runlevel to run some reason number of coming 
  back next server; sees you Control <a href="blah.html">display</a> a steep 
  and I tagged with specifications of six feet, moving to Code, flyer main room
  motel balcony, <p>and airflow in which define the ability to run a common. We
  need to current in a manner <pre>than six months and that already gotten a
  webcast</pre> is roughly long and bulk: and up the src page: and updates on a:
  user will probably does this.

This would be converted into the following:

  <p>I see over the <strong>noise</strong> but I don't understand sometimes.</p>

  <ol><li>One</li><li>Two</li><li>Three</li><ol>

  <p>Fortunately, we've traded the club you can't skimp on the do because This
  week! Presented by code Lounge: except, for controlling Knox video cameras
  Linux well that the reason, the runlevel to run some reason number of coming 
  back next server; sees you Control <a href="blah.html">display</a> a steep 
  and I tagged with specifications of six feet, moving to Code, flyer main room
  motel balcony,</p>
  <p>and airflow in which define the ability to run a common. We need to
  current in a manner</p>
  <pre>than six months and that already gotten a
  webcast</pre>
  <p>is roughly long and bulk: and up the src page: and updates on a: user will 
  probably does this.</p>

This allows authors to use HTML markup some without having to cope with getting their paragraph tags right.

This library depends upon L<HTML::TreeBuilder> and L<HTML::Tagset>. You may wish to see the documentation for those libraries for additional details.

=head1 METHODS

The primary method of this library is C<split_paragraphs()>. An additional method, C<split_paragraphs_to_text()> is provided to simplify the task of generating output without having to fuss with L<HTML::TreeBuilder>.

=head2 split_paragraphs

=over

=item $element = split_paragraphs($handle, \%options)

=item $element = split_paragraphs($text, \%options)

=item $element = split_paragraphs($element, \%options)

=back

This method has three forms, which vary only in the input they receive. If the first argument is a file handle, C<$handle>, then that handle will be read, parsed, and split. If the first argument is a scalar, C<$text>, then that text will parsed and split. If the first argument is a subclass of L<HTML::Element>, C<$element>, then the tree represented by the node will be traversed and split. 

If you use the third form, your tree will be modified in place and the same tree will be returned. You will want to clone the tree ahead of time if you need to preserve the old tree.

All forms take an optional second parameter, C<\%options>, which is a reference to a hash of options which modify the default behavior. See below for details.

The first two forms perform an extra step, but are handled essentially the same after the input is parsed into an L<HTML::Element> using L<HTML::TreeBuilder>. This is done using the defaults, except that C<no_space_compacting()> is set to a true value (otherwise, we lose any double returns that were in the original text). If you parse your own trees, you'll probably want to do the same.

This method will search down the element tree and find the first node with non-implicit child ndoes and use that as the root of operations. 

The C<split_paragraphs()> method then walks the tree and wraps any undecorated text node in a paragraph. Any double line break discovered will result in multiple paragraphs. Any paragraph content elements (as defined by C<%is_Possible_Strict_P_Content> of L<HTML::Tagset>) will be inserted into the paragraph elements as if they were text. Any block level tags (i.e., not in C<%is_Possible_Strict_P_Content>) cause a paragraph break immediately before and after such elements.

Any text found within a block-level node may also be paragraphified. Those blocks of text will not be wrapped in paragraphs unless they contain a double-line break (that way we're not inserting C<P>-tags without an explicit need for them).

Note also that this will insert C<P>-tags conservatively. If more than two line-breaks are present, even if they are mixed with other white space, all of that whitespace will be treated as the same paragraph break. No empty C<P>-tags or C<P>-tags containing only whitespace will be inserted (mostly). The only exception is when the white space is created by white space entities, such as C<&nbsp;>.

All of that is the default behavior. That behavior may be modified by the second parameter, which is used to specify options that modify that behavior.

Here's the list of options and what they do:

=over

=item p_on_breaks_only =E<gt> 1

If this option is used, then paragrpahs will not be added to your text unless there is at least one double-line break. This option is used internally to make sure nested elements do not have extra C<P>-tags unnecessarily.

=item single_line_breaks_to_br =E<gt> 1

If this option is given, then single line breaks will also be converted to C<BR>-tags.

=item br_only_if_can_tighten =E<gt> 1

This option modifies the C<single_line_breaks_to_br> option by specifying that C<BR>-tags are not added within blocks that cannot be tightened (i.e., aren't set in C<%canTighten> of L<HTML::Tagset>). This can be useful for preventing double-line breaks from appearing inside C<PRE>-tags or C<TEXTAREA>-tags because of added C<BR>-tags.

=item use_br_instead_of_p =E<gt> 1

As an alternative to using C<P>-tags at all, this can also just place C<BR>-tags everywhere instead. Instead of inserting C<P>-tags whenever a double line-break is enountered, two C<BR>-tags will be inserted instead.

This option is independant of C<single_line_breaks_to_br> as single line-breaks are not dealt with unless that option is turned on. Also note that, like C<P>-tag insertion, it inserts C<BR>-tags conservatively. Multiple consecutive line-breaks (even mixed with whitespace) will be treated just as if they were only two. Thus, given the default stylesheet of your typical browser, the rendered output will appear pretty much the same in most circumstances.

=item add_attrs_to_p =E<gt> \%attrs

This can be used to insert a static set of attributes to each inserted C<P>-element. For example:

  # Give each newly added paragraph the "generated" class.
  split_paragraphs($tree, {
      add_attrs_to_p => { class => 'generated' },
  });

=item add_attrs_to_br =E<gt> \%attrs

Same as above, but for the inserted C<BR>-tags.

=item filter_added_nodes =E<gt> \&sub

This can be used to run a small subroutine on each added paragraph or line-break tag as it is added. For example:

  # Give each newly added paragraph a unique ID
  split_paragraphs($tree, {
      filter_added_nodes => sub {
          my ($element) = @_;
          $element->idf();
      },
  });

Many, if not all, of the other options can be simulated using this method, by the way.

=item use_instead_of_p =E<gt> $tag

Rather than using C<P>-tags to break everything, use a different tag. This example uses C<DIV>-tags instead of C<P>-tags:

  split_paragraphs($tree, {
      use_instead_of_p => 'div',
  });

=back

=cut

sub _split_paragraphs_phrases {
    my ($h, $phrase, $options) = @_;

    # Trim the phrase of any extra whitespace at the edges
    $phrase =~ s/^\s+//; $phrase =~ s/\s+$//;

    # Adding a space at the end fixes a problem I have with the parser dropping
    # the last word in the text. *shrug*
    #my @paragraphs = map { "$_ " } split /\r?\n\s*(?:\r?\n\s*)+/, $phrase;
    my @paragraphs = split /\r?\n\s*(?:\r?\n\s*)+/, $phrase;

    # If there's only one paragraph and we're not allowed to P without a break,
    # push the text back in and skip paragraphification.
    if ($options->{p_on_breaks_only} && @paragraphs == 1) {
        my $parsed_paragraph = _nice_tree_builder();
        $parsed_paragraph->parse($paragraphs[0]);
        $parsed_paragraph->eof;

        $h->push_content($parsed_paragraph->guts);
        return;
    }

    # Loop over the paragraph splits
    my $iteration = 0;
    for my $paragraph (@paragraphs) {
        $iteration++;
        
        # Add BR tags within the paragraph, if necessary
        $paragraph =~ s[\r?\n][<br/>]g 
            if $options->{single_line_breaks_to_br};

        # Parse the paragraph text to recover the phrase elements that were
        # converted to text
        my $parsed_paragraph = _nice_tree_builder();
        $parsed_paragraph->parse($paragraph);
        $parsed_paragraph->eof;
        
        # Don't use P-tags, use BR-tags
        if ($options->{use_br_instead_of_p}) {
            $h->push_content($parsed_paragraph->guts);
            if ($iteration < @paragraphs) {
                my $new_br_1 =
                    HTML::Element->new('br', %{ $options->{add_attrs_to_br} });
                my $new_br_2 =
                    HTML::Element->new('br', %{ $options->{add_attrs_to_br} });

                $h->push_content($new_br_1);
                $h->push_content($new_br_2);

                if ($options->{filter_added_nodes}) {
                    $options->{filter_added_nodes}->($new_br_1);
                    $options->{filter_added_nodes}->($new_br_2);
                }

            }
        }

        # Use P-tags
        else {
            # Create a new P-tag to contain the text
            my $p = HTML::Element->new(
                $options->{use_instead_of_p}, 
                %{ $options->{add_attrs_to_p} }
            );

            # Now add the paragraph back to the parent
            $p->push_content($parsed_paragraph->guts);
            $h->push_content($p);

            if ($options->{filter_added_nodes}) {
                $options->{filter_added_nodes}->($p);
            }
        }
    }
}

sub _split_paragraphs_guts {
    my ($h, $options) = @_;

    # Rip the content out of the parent so we can add it back in a piece at a
    # time as we process it
    my @content = $h->detach_content;

    # This is a temporary variable used to collect text nodes and phrase-level
    # elements to be split into paragraphs
    my $unparsed_phrase;

    # Loop over the child nodes
    for my $content (@content) {

        # Handle nested elements.
        if (blessed $content) {

            # Get the tag name
            my $tag = $content->tag;

            # It's a phrase level tag, append it as text to the accumulator
            if ($p_content{$tag}) {
                $unparsed_phrase .= $content->as_HTML(undef, undef, {});

                # HTML::TreeBuilder adds a new line after tags, kill it. This
                # new line will mess things up with single_line_breaks_to_br is
                # used.
                chomp $unparsed_phrase;
            }

            # It's not a phrase level tag
            else {

                if (defined $unparsed_phrase) {
                    # Process the accumulator and empty it
                    _split_paragraphs_phrases($h, $unparsed_phrase, $options);
                    $unparsed_phrase = undef;
                }

                # Recurse downward to find sub-elements to split, if any
                _split_paragraphs_guts($content, { %$options,
                    p_on_breaks_only => 1,
                });

                # Push this tag into the list
                $h->push_content($content);
            }
        }

        # Text node, just append to the accumulator
        else {
            # Need to make sure entities are escaped first
            $content = encode_entities($content);
            $unparsed_phrase .= $content;
        }
    }
    
    # If the accumulator still has something in it, split it.
    if (defined $unparsed_phrase) {
        _split_paragraphs_phrases($h, $unparsed_phrase, $options);
    }
}

sub _nice_tree_builder {
    # Create a HTML::TreeBuilder that's good for splitting
    my $tree = HTML::TreeBuilder->new;
    $tree->no_space_compacting(1);
    return $tree;
}

sub split_paragraphs {
    my ($input, $options) = @_;

    # Translate the first argument into a tree
    my $tree;

    # The first argument is a tree; use it.
    if (blessed $input && $input->isa('HTML::Element')) {
        $tree = $input;
    }

    # The first argument is a file handle (or bad stuff); parse the file.
    elsif (ref $input) {
        $tree = _nice_tree_builder();
        $tree->parse_file($input);
    }

    # The first argument is text; parse it.
    else {
        $tree = _nice_tree_builder();
        $tree->parse($input);
        $tree->eof;
    }

    # Don't use ->guts because they could give us an HTML::Element.
    # We objectify in case something goes wrong and the only non-implicit node
    # is text.
    $tree->objectify_text;
    my $guts = $tree->look_down(_implicit => undef)->parent;
    $tree->deobjectify_text;

    # Setup the default options
    $options = {
        add_attrs_to_p   => {},
        add_attrs_to_br  => {},
        use_instead_of_p => 'p',
        defined $options ? %$options : (),
    };

    # Split those guts
    _split_paragraphs_guts($guts, $options || {});

#    $tree->find('body')->delete_content;
#    $tree->find('body')->push_content($guts->content_list);

    return $tree;
}

=head2 split_paragraphs_to_text

=over

=item $html_text = split_paragraphs_to_text($handle, \%options)

=item $html_text = split_paragraphs_to_text($text, \%options)

=item $html_text = split_paragraphs_to_text($element, \%options)

=back

This method performs the exact same operation as the C<split_paragraphs()> method, but returns the text as a scalar value. This is helpful if you just want a quick method that takes in text and outputs text and you don't really need the HTML formatted in any particular way and don't need to modify the tree at all.

I created this method primarily as a way of outputing the tree to make testing easier. If the output isn't want you like, use C<split_paragraphs()> instead and use the output methods available in L<HTML::Element> directly to get the desired output.

=cut

sub split_paragraphs_to_text {
    # Use split_paragraphs() to do all the work
    my $tree = split_paragraphs(@_);

    # Get the top non-implicit node
    $tree->objectify_text;
    my $output_element = $tree->look_down(_implicit => undef)->parent;
    $tree->deobjectify_text;

    # Append each node as text
    my $text;
    for my $element ($output_element->content_list) {

        # Convert each element to HTML text
        if (ref $element) {
            $text .= $element->as_HTML(undef, undef, {});
        }

        # Add text nodes as-is
        else {
            $text .= $element;
        }
    }

    # Deallocate the tree to prevent memory leaks
    $tree->delete;

    return $text;
}

=head1 SEE ALSO

L<HTML::TreeBuilder>, L<HTML::Tagset>

=head1 BUGS AND TODO

I don't really have any explicit plans for this module, but if you find a bug or would like an additional feature or have another contribution, send me email at E<lt>hanenkamp@cpan.orgE<gt>.

=head1 NOTES

I tried to name this library HTML::Paragraphify first. After typing that a dozen times and looking at it for a few hours, my eyes felt like they were starting to bleed so I changed it to L<HTML::ParagraphSplit>.

=head1 AUTHOR

Andrew Sterling Hanenkamp, C<< <hanenkamp@cpan.org> >>

=head1 LICENSE AND COPYRIGHT

Copyright 2006 Andrew Sterling Hanenkamp C<< <hanenkamp@cpan.org> >>.  All 
Rights Reserved.

This module is free software; you can redistribute it and/or modify it under
the same terms as Perl itself. See L<perlartistic>.

This program is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE.

=cut

1
