#!perl
BEGIN
{
    use strict;
    use warnings;
    use lib './lib';
    use open ':std' => ':utf8';
    use vars qw( $DEBUG $TEST_ID );
    use utf8;
    use version;
    use Test::More;
    use DateTime;
    our $DEBUG = exists( $ENV{AUTHOR_TESTING} ) ? $ENV{AUTHOR_TESTING} : 0;
    $TEST_ID = $ENV{TEST_ID} if( exists( $ENV{TEST_ID} ) );
};

BEGIN
{
    use_ok( 'DateTime::Format::Intl' ) || BAIL_OUT( 'Unable to load DateTime::Format::Intl' );
};

use strict;
use warnings;
use utf8;

my $dt = DateTime->new(
    year => 2024,
    month => 9,
    day => 24,
    hour => 9,
    minute => 0,
    second => 10,
    time_zone => 'UTC',
);
my $date = $dt->iso8601;

my $tests = 
[
    # NOTE: Test #0
    {
        date1 => { day => 10, hour => 9, minute => 0, month => 9, second => 0, year => 2024 },
        date2 => { day => 10, hour => 16, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "9 in the morning – 4 in the afternoon",
        greatest_diff => "B",
        locale => "en",
        options => { calendar => "gregorian", dayPeriod => "short", hour => "numeric" },
        pattern => "h B – h B",
        skeleton => "Bh",
    },
    {
        date1 => { day => 10, hour => 10, minute => 0, month => 9, second => 0, year => 2024 },
        date2 => { day => 10, hour => 11, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "10 – 11 in the morning",
        greatest_diff => "h",
        locale => "en",
        options => { calendar => "gregorian", dayPeriod => "short", hour => "numeric" },
        pattern => "h – h B",
        skeleton => "Bh",
    },
    {
        date1 => { day => 10, hour => 9, minute => 30, month => 9, second => 0, year => 2024 },
        date2 => { day => 10, hour => 16, minute => 30, month => 9, second => 0, year => 2024 },
        expects => "9:30 in the morning – 4:30 in the afternoon",
        greatest_diff => "B",
        locale => "en",
        options => {
            calendar => "gregorian",
            dayPeriod => "short",
            hour => "numeric",
            minute => "numeric",
        },
        pattern => "h:mm B – h:mm B",
        skeleton => "Bhm",
    },
    {
        date1 => { day => 10, hour => 10, minute => 30, month => 9, second => 0, year => 2024 },
        date2 => { day => 10, hour => 11, minute => 30, month => 9, second => 0, year => 2024 },
        expects => "10:30 – 11:30 in the morning",
        greatest_diff => "h",
        locale => "en",
        options => {
            calendar => "gregorian",
            dayPeriod => "short",
            hour => "numeric",
            minute => "numeric",
        },
        pattern => "h:mm – h:mm B",
        skeleton => "Bhm",
    },
    {
        date1 => { day => 10, hour => 10, minute => 0, month => 9, second => 0, year => 2024 },
        date2 => { day => 10, hour => 10, minute => 30, month => 9, second => 0, year => 2024 },
        expects => "10:00 – 10:30 in the morning",
        greatest_diff => "m",
        locale => "en",
        options => {
            calendar => "gregorian",
            dayPeriod => "short",
            hour => "numeric",
            minute => "numeric",
        },
        pattern => "h:mm – h:mm B",
        skeleton => "Bhm",
    },
    {
        date1 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        date2 => { day => 11, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "10 – 11",
        greatest_diff => "d",
        locale => "en",
        options => { calendar => "gregorian", day => "numeric" },
        pattern => "d – d",
        skeleton => "d",
    },
    {
        date1 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => -2024 },
        date2 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "-2024 BC – 2024 AD",
        greatest_diff => "G",
        locale => "en",
        options => { calendar => "gregorian", era => "short", year => "numeric" },
        pattern => "y G – y G",
        skeleton => "Gy",
    },
    {
        date1 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => 2023 },
        date2 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "2023 – 2024 AD",
        greatest_diff => "y",
        locale => "en",
        options => { calendar => "gregorian", era => "short", year => "numeric" },
        pattern => "y – y G",
        skeleton => "Gy",
    },
    {
        date1 => { day => 10, hour => 0, minute => 0, month => 8, second => 0, year => -2024 },
        date2 => { day => 10, hour => 0, minute => 0, month => 8, second => 0, year => 2024 },
        expects => "8/-2024 BC – 8/2024 AD",
        greatest_diff => "G",
        locale => "en",
        options => {
            calendar => "gregorian",
            era => "short",
            month => "numeric",
            year => "numeric",
        },
        pattern => "M/y G – M/y G",
        skeleton => "GyM",
    },
    {
        date1 => { day => 10, hour => 0, minute => 0, month => 7, second => 0, year => 2024 },
        date2 => { day => 10, hour => 0, minute => 0, month => 8, second => 0, year => 2024 },
        expects => "7/2024 – 8/2024 AD",
        greatest_diff => "M",
        locale => "en",
        options => {
            calendar => "gregorian",
            era => "short",
            month => "numeric",
            year => "numeric",
        },
        pattern => "M/y – M/y G",
        skeleton => "GyM",
    },
    # NOTE: Test #10
    {
        date1 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => 2023 },
        date2 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "9/2023 – 9/2024 AD",
        greatest_diff => "y",
        locale => "en",
        options => {
            calendar => "gregorian",
            era => "short",
            month => "numeric",
            year => "numeric",
        },
        pattern => "M/y – M/y G",
        skeleton => "GyM",
    },
    {
        date1 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        date2 => { day => 11, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "9/10/2024 – 9/11/2024 AD",
        greatest_diff => "d",
        locale => "en",
        options => {
            calendar => "gregorian",
            day => "numeric",
            era => "short",
            month => "numeric",
            year => "numeric",
        },
        pattern => "M/d/y – M/d/y G",
        skeleton => "GyMd",
    },
    {
        date1 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => -2024 },
        date2 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "9/10/-2024 BC – 9/10/2024 AD",
        greatest_diff => "G",
        locale => "en",
        options => {
            calendar => "gregorian",
            day => "numeric",
            era => "short",
            month => "numeric",
            year => "numeric",
        },
        pattern => "M/d/y G – M/d/y G",
        skeleton => "GyMd",
    },
    {
        date1 => { day => 10, hour => 0, minute => 0, month => 8, second => 0, year => 2024 },
        date2 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "8/10/2024 – 9/10/2024 AD",
        greatest_diff => "M",
        locale => "en",
        options => {
            calendar => "gregorian",
            day => "numeric",
            era => "short",
            month => "numeric",
            year => "numeric",
        },
        pattern => "M/d/y – M/d/y G",
        skeleton => "GyMd",
    },
    {
        date1 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => 2023 },
        date2 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "9/10/2023 – 9/10/2024 AD",
        greatest_diff => "y",
        locale => "en",
        options => {
            calendar => "gregorian",
            day => "numeric",
            era => "short",
            month => "numeric",
            year => "numeric",
        },
        pattern => "M/d/y – M/d/y G",
        skeleton => "GyMd",
    },
    {
        date1 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        date2 => { day => 11, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "Tue, 9/10/2024 – Wed, 9/11/2024 AD",
        greatest_diff => "d",
        locale => "en",
        options => {
            calendar => "gregorian",
            day => "numeric",
            era => "short",
            month => "numeric",
            weekday => "short",
            year => "numeric",
        },
        pattern => "E, M/d/y – E, M/d/y G",
        skeleton => "GyMEd",
    },
    {
        date1 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => -2024 },
        date2 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "Fri, 9/10/-2024 BC – Tue, 9/10/2024 AD",
        greatest_diff => "G",
        locale => "en",
        options => {
            calendar => "gregorian",
            day => "numeric",
            era => "short",
            month => "numeric",
            weekday => "short",
            year => "numeric",
        },
        pattern => "E, M/d/y G – E, M/d/y G",
        skeleton => "GyMEd",
    },
    {
        date1 => { day => 10, hour => 0, minute => 0, month => 8, second => 0, year => 2024 },
        date2 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "Sat, 8/10/2024 – Tue, 9/10/2024 AD",
        greatest_diff => "M",
        locale => "en",
        options => {
            calendar => "gregorian",
            day => "numeric",
            era => "short",
            month => "numeric",
            weekday => "short",
            year => "numeric",
        },
        pattern => "E, M/d/y – E, M/d/y G",
        skeleton => "GyMEd",
    },
    {
        date1 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => 2023 },
        date2 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "Sun, 9/10/2023 – Tue, 9/10/2024 AD",
        greatest_diff => "y",
        locale => "en",
        options => {
            calendar => "gregorian",
            day => "numeric",
            era => "short",
            month => "numeric",
            weekday => "short",
            year => "numeric",
        },
        pattern => "E, M/d/y – E, M/d/y G",
        skeleton => "GyMEd",
    },
    {
        date1 => { day => 10, hour => 0, minute => 0, month => 8, second => 0, year => -2024 },
        date2 => { day => 10, hour => 0, minute => 0, month => 8, second => 0, year => 2024 },
        expects => "Aug -2024 BC – Aug 2024 AD",
        greatest_diff => "G",
        locale => "en",
        options => {
            calendar => "gregorian",
            era => "short",
            month => "short",
            year => "numeric",
        },
        pattern => "MMM y G – MMM y G",
        skeleton => "GyMMM",
    },
    # NOTE: Test #20
    {
        date1 => { day => 10, hour => 0, minute => 0, month => 7, second => 0, year => 2024 },
        date2 => { day => 10, hour => 0, minute => 0, month => 8, second => 0, year => 2024 },
        expects => "Jul – Aug 2024 AD",
        greatest_diff => "M",
        locale => "en",
        options => {
            calendar => "gregorian",
            era => "short",
            month => "short",
            year => "numeric",
        },
        pattern => "MMM – MMM y G",
        skeleton => "GyMMM",
    },
    {
        date1 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => 2023 },
        date2 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "Sep 2023 – Sep 2024 AD",
        greatest_diff => "y",
        locale => "en",
        options => {
            calendar => "gregorian",
            era => "short",
            month => "short",
            year => "numeric",
        },
        pattern => "MMM y – MMM y G",
        skeleton => "GyMMM",
    },
    {
        date1 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        date2 => { day => 11, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "Sep 10 – 11, 2024 AD",
        greatest_diff => "d",
        locale => "en",
        options => {
            calendar => "gregorian",
            day => "numeric",
            era => "short",
            month => "short",
            year => "numeric",
        },
        pattern => "MMM d – d, y G",
        skeleton => "GyMMMd",
    },
    {
        date1 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => -2024 },
        date2 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "Sep 10, -2024 BC – Sep 10, 2024 AD",
        greatest_diff => "G",
        locale => "en",
        options => {
            calendar => "gregorian",
            day => "numeric",
            era => "short",
            month => "short",
            year => "numeric",
        },
        pattern => "MMM d, y G – MMM d, y G",
        skeleton => "GyMMMd",
    },
    {
        date1 => { day => 10, hour => 0, minute => 0, month => 8, second => 0, year => 2024 },
        date2 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "Aug 10 – Sep 10, 2024 AD",
        greatest_diff => "M",
        locale => "en",
        options => {
            calendar => "gregorian",
            day => "numeric",
            era => "short",
            month => "short",
            year => "numeric",
        },
        pattern => "MMM d – MMM d, y G",
        skeleton => "GyMMMd",
    },
    {
        date1 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => 2023 },
        date2 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "Sep 10, 2023 – Sep 10, 2024 AD",
        greatest_diff => "y",
        locale => "en",
        options => {
            calendar => "gregorian",
            day => "numeric",
            era => "short",
            month => "short",
            year => "numeric",
        },
        pattern => "MMM d, y – MMM d, y G",
        skeleton => "GyMMMd",
    },
    {
        date1 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        date2 => { day => 11, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "Tue, Sep 10 – Wed, Sep 11, 2024 AD",
        greatest_diff => "d",
        locale => "en",
        options => {
            calendar => "gregorian",
            day => "numeric",
            era => "short",
            month => "short",
            weekday => "short",
            year => "numeric",
        },
        pattern => "E, MMM d – E, MMM d, y G",
        skeleton => "GyMMMEd",
    },
    {
        date1 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => -2024 },
        date2 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "Fri, Sep 10, -2024 BC – Tue, Sep 10, 2024 AD",
        greatest_diff => "G",
        locale => "en",
        options => {
            calendar => "gregorian",
            day => "numeric",
            era => "short",
            month => "short",
            weekday => "short",
            year => "numeric",
        },
        pattern => "E, MMM d, y G – E, MMM d, y G",
        skeleton => "GyMMMEd",
    },
    {
        date1 => { day => 10, hour => 0, minute => 0, month => 8, second => 0, year => 2024 },
        date2 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "Sat, Aug 10 – Tue, Sep 10, 2024 AD",
        greatest_diff => "M",
        locale => "en",
        options => {
            calendar => "gregorian",
            day => "numeric",
            era => "short",
            month => "short",
            weekday => "short",
            year => "numeric",
        },
        pattern => "E, MMM d – E, MMM d, y G",
        skeleton => "GyMMMEd",
    },
    {
        date1 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => 2023 },
        date2 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "Sun, Sep 10, 2023 – Tue, Sep 10, 2024 AD",
        greatest_diff => "y",
        locale => "en",
        options => {
            calendar => "gregorian",
            day => "numeric",
            era => "short",
            month => "short",
            weekday => "short",
            year => "numeric",
        },
        pattern => "E, MMM d, y – E, MMM d, y G",
        skeleton => "GyMMMEd",
    },
    # NOTE: Test #30
    {
        date1 => { day => 10, hour => 10, minute => 0, month => 9, second => 0, year => 2024 },
        date2 => { day => 10, hour => 13, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "10 AM – 1 PM",
        greatest_diff => "a",
        locale => "en",
        options => { calendar => "gregorian", hour => "numeric", hourCycle => "h12" },
        pattern => "h a – h a",
        skeleton => "h",
    },
    {
        date1 => { day => 10, hour => 9, minute => 0, month => 9, second => 0, year => 2024 },
        date2 => { day => 10, hour => 10, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "9 – 10 AM",
        greatest_diff => "h",
        locale => "en",
        options => { calendar => "gregorian", hour => "numeric", hourCycle => "h12" },
        pattern => "h – h a",
        skeleton => "h",
    },
    {
        date1 => { day => 10, hour => 13, minute => 0, month => 9, second => 0, year => 2024 },
        date2 => { day => 10, hour => 14, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "13 – 14",
        greatest_diff => "H",
        locale => "en",
        options => { calendar => "gregorian", hour => "numeric", hourCycle => "h24" },
        pattern => "HH – HH",
        skeleton => "H",
    },
    {
        date1 => { day => 10, hour => 9, minute => 30, month => 9, second => 0, year => 2024 },
        date2 => { day => 10, hour => 10, minute => 30, month => 9, second => 0, year => 2024 },
        expects => "9:30 – 10:30 AM",
        greatest_diff => "h",
        locale => "en",
        options => {
            calendar => "gregorian",
            hour => "numeric",
            hourCycle => "h12",
            minute => "numeric",
        },
        pattern => "h:mm a – h:mm a",
        skeleton => "hm",
    },
    {
        date1 => { day => 10, hour => 9, minute => 0, month => 9, second => 0, year => 2024 },
        date2 => { day => 10, hour => 10, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "9:00 – 10:00 AM",
        greatest_diff => "h",
        locale => "en",
        options => {
            calendar => "gregorian",
            hour => "numeric",
            hourCycle => "h12",
            minute => "numeric",
        },
        pattern => "h:mm – h:mm a",
        skeleton => "hm",
    },
    {
        date1 => { day => 10, hour => 10, minute => 0, month => 9, second => 0, year => 2024 },
        date2 => { day => 10, hour => 10, minute => 30, month => 9, second => 0, year => 2024 },
        expects => "10:00 – 10:30 AM",
        greatest_diff => "m",
        locale => "en",
        options => {
            calendar => "gregorian",
            hour => "numeric",
            hourCycle => "h12",
            minute => "numeric",
        },
        pattern => "h:mm – h:mm a",
        skeleton => "hm",
    },
    {
        date1 => { day => 10, hour => 9, minute => 0, month => 9, second => 0, year => 2024 },
        date2 => { day => 10, hour => 10, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "09:00 – 10:00",
        greatest_diff => "H",
        locale => "en",
        options => {
            calendar => "gregorian",
            hour => "numeric",
            hourCycle => "h24",
            minute => "numeric",
        },
        pattern => "HH:mm – HH:mm",
        skeleton => "Hm",
    },
    {
        date1 => { day => 10, hour => 10, minute => 0, month => 9, second => 0, year => 2024 },
        date2 => { day => 10, hour => 10, minute => 30, month => 9, second => 0, year => 2024 },
        expects => "10:00 – 10:30",
        greatest_diff => "m",
        locale => "en",
        options => {
            calendar => "gregorian",
            hour => "numeric",
            hourCycle => "h24",
            minute => "numeric",
        },
        pattern => "HH:mm – HH:mm",
        skeleton => "Hm",
    },
    {
        date1 => { day => 10, hour => 9, minute => 30, month => 9, second => 0, year => 2024 },
        date2 => { day => 10, hour => 10, minute => 30, month => 9, second => 0, year => 2024 },
        expects => "9:30 – 10:30 AM UTC Time",
        greatest_diff => "h",
        locale => "en",
        options => {
            calendar => "gregorian",
            hour => "numeric",
            hourCycle => "h12",
            minute => "numeric",
            timeZoneName => "short",
        },
        pattern => "h:mm a – h:mm a v",
        skeleton => "hmv",
    },
    {
        date1 => { day => 10, hour => 9, minute => 0, month => 9, second => 0, year => 2024 },
        date2 => { day => 10, hour => 10, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "9:00 – 10:00 AM UTC Time",
        greatest_diff => "h",
        locale => "en",
        options => {
            calendar => "gregorian",
            hour => "numeric",
            hourCycle => "h12",
            minute => "numeric",
            timeZoneName => "short",
        },
        pattern => "h:mm – h:mm a v",
        skeleton => "hmv",
    },
    # NOTE: Test #40
    {
        date1 => { day => 10, hour => 10, minute => 0, month => 9, second => 0, year => 2024 },
        date2 => { day => 10, hour => 10, minute => 30, month => 9, second => 0, year => 2024 },
        expects => "10:00 – 10:30 AM UTC Time",
        greatest_diff => "m",
        locale => "en",
        options => {
            calendar => "gregorian",
            hour => "numeric",
            hourCycle => "h12",
            minute => "numeric",
            timeZoneName => "short",
        },
        pattern => "h:mm – h:mm a v",
        skeleton => "hmv",
    },
    {
        date1 => { day => 10, hour => 9, minute => 0, month => 9, second => 0, year => 2024 },
        date2 => { day => 10, hour => 10, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "910:00 – UTC Time",
        greatest_diff => "H",
        locale => "en",
        options => {
            calendar => "gregorian",
            hour => "numeric",
            hourCycle => "h24",
            minute => "numeric",
            timeZoneName => "short",
        },
        pattern => "HH:mm – HH:mm v",
        skeleton => "Hmv",
    },
    {
        date1 => { day => 10, hour => 10, minute => 0, month => 9, second => 0, year => 2024 },
        date2 => { day => 10, hour => 10, minute => 30, month => 9, second => 0, year => 2024 },
        expects => "10:030 – UTC Time",
        greatest_diff => "m",
        locale => "en",
        options => {
            calendar => "gregorian",
            hour => "numeric",
            hourCycle => "h24",
            minute => "numeric",
            timeZoneName => "short",
        },
        pattern => "HH:mm – HH:mm v",
        skeleton => "Hmv",
    },
    {
        date1 => { day => 10, hour => 9, minute => 30, month => 9, second => 0, year => 2024 },
        date2 => { day => 10, hour => 10, minute => 30, month => 9, second => 0, year => 2024 },
        expects => "9 – 10 AM UTC Time",
        greatest_diff => "h",
        locale => "en",
        options => {
            calendar => "gregorian",
            hour => "numeric",
            hourCycle => "h12",
            timeZoneName => "short",
        },
        pattern => "h a – h a v",
        skeleton => "hv",
    },
    {
        date1 => { day => 10, hour => 9, minute => 0, month => 9, second => 0, year => 2024 },
        date2 => { day => 10, hour => 10, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "9 – 10 AM UTC Time",
        greatest_diff => "h",
        locale => "en",
        options => {
            calendar => "gregorian",
            hour => "numeric",
            hourCycle => "h12",
            timeZoneName => "short",
        },
        pattern => "h – h a v",
        skeleton => "hv",
    },
    {
        date1 => { day => 10, hour => 9, minute => 0, month => 9, second => 0, year => 2024 },
        date2 => { day => 10, hour => 10, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "910 – UTC Time",
        greatest_diff => "H",
        locale => "en",
        options => {
            calendar => "gregorian",
            hour => "numeric",
            hourCycle => "h24",
            timeZoneName => "short",
        },
        pattern => "HH – HH v",
        skeleton => "Hv",
    },
    {
        date1 => { day => 10, hour => 0, minute => 0, month => 6, second => 0, year => 2024 },
        date2 => { day => 10, hour => 0, minute => 0, month => 7, second => 0, year => 2024 },
        expects => "6 – 7",
        greatest_diff => "M",
        locale => "en",
        options => { calendar => "gregorian", month => "numeric" },
        pattern => "M – M",
        skeleton => "M",
    },
    {
        date1 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        date2 => { day => 11, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "9/10 – 9/11",
        greatest_diff => "d",
        locale => "en",
        options => { calendar => "gregorian", day => "numeric", month => "numeric" },
        pattern => "M/d – M/d",
        skeleton => "Md",
    },
    {
        date1 => { day => 10, hour => 0, minute => 0, month => 8, second => 0, year => 2024 },
        date2 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "8/10 – 9/10",
        greatest_diff => "M",
        locale => "en",
        options => { calendar => "gregorian", day => "numeric", month => "numeric" },
        pattern => "M/d – M/d",
        skeleton => "Md",
    },
    {
        date1 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        date2 => { day => 11, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "Tue, 9/10 – Wed, 9/11",
        greatest_diff => "d",
        locale => "en",
        options => {
            calendar => "gregorian",
            day => "numeric",
            month => "numeric",
            weekday => "short",
        },
        pattern => "E, M/d – E, M/d",
        skeleton => "MEd",
    },
    # NOTE: Test #50
    {
        date1 => { day => 10, hour => 0, minute => 0, month => 8, second => 0, year => 2024 },
        date2 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "Sat, 8/10 – Tue, 9/10",
        greatest_diff => "M",
        locale => "en",
        options => {
            calendar => "gregorian",
            day => "numeric",
            month => "numeric",
            weekday => "short",
        },
        pattern => "E, M/d – E, M/d",
        skeleton => "MEd",
    },
    {
        date1 => { day => 10, hour => 0, minute => 0, month => 6, second => 0, year => 2024 },
        date2 => { day => 10, hour => 0, minute => 0, month => 7, second => 0, year => 2024 },
        expects => "Jun – Jul",
        greatest_diff => "M",
        locale => "en",
        options => { calendar => "gregorian", month => "short" },
        pattern => "MMM – MMM",
        skeleton => "MMM",
    },
    {
        date1 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        date2 => { day => 11, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "Sep 10 – 11",
        greatest_diff => "d",
        locale => "en",
        options => { calendar => "gregorian", day => "numeric", month => "short" },
        pattern => "MMM d – d",
        skeleton => "MMMd",
    },
    {
        date1 => { day => 10, hour => 0, minute => 0, month => 8, second => 0, year => 2024 },
        date2 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "Aug 10 – Sep 10",
        greatest_diff => "M",
        locale => "en",
        options => { calendar => "gregorian", day => "numeric", month => "short" },
        pattern => "MMM d – MMM d",
        skeleton => "MMMd",
    },
    {
        date1 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        date2 => { day => 11, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "Tue, Sep 10 – Wed, Sep 11",
        greatest_diff => "d",
        locale => "en",
        options => {
            calendar => "gregorian",
            day => "numeric",
            month => "short",
            weekday => "short",
        },
        pattern => "E, MMM d – E, MMM d",
        skeleton => "MMMEd",
    },
    {
        date1 => { day => 10, hour => 0, minute => 0, month => 8, second => 0, year => 2024 },
        date2 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "Sat, Aug 10 – Tue, Sep 10",
        greatest_diff => "M",
        locale => "en",
        options => {
            calendar => "gregorian",
            day => "numeric",
            month => "short",
            weekday => "short",
        },
        pattern => "E, MMM d – E, MMM d",
        skeleton => "MMMEd",
    },
    {
        date1 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => 2023 },
        date2 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "2023 – 2024",
        greatest_diff => "y",
        locale => "en",
        options => { calendar => "gregorian", year => "numeric" },
        pattern => "y – y",
        skeleton => "y",
    },
    {
        date1 => { day => 1, hour => 0, minute => 0, month => 6, second => 0, year => 2024 },
        date2 => { day => 1, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "6/2024 – 9/2024",
        greatest_diff => "M",
        locale => "en",
        options => { calendar => "gregorian", month => "numeric", year => "numeric" },
        pattern => "M/y – M/y",
        skeleton => "yM",
    },
    {
        date1 => { day => 1, hour => 0, minute => 0, month => 9, second => 0, year => 2023 },
        date2 => { day => 1, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "9/2023 – 9/2024",
        greatest_diff => "y",
        locale => "en",
        options => { calendar => "gregorian", month => "numeric", year => "numeric" },
        pattern => "M/y – M/y",
        skeleton => "yM",
    },
    {
        date1 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        date2 => { day => 11, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "9/10/2024 – 9/11/2024",
        greatest_diff => "d",
        locale => "en",
        options => {
            calendar => "gregorian",
            day => "numeric",
            month => "numeric",
            year => "numeric",
        },
        pattern => "M/d/y – M/d/y",
        skeleton => "yMd",
    },
    # NOTE: Test #60
    {
        date1 => { day => 10, hour => 0, minute => 0, month => 8, second => 0, year => 2024 },
        date2 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "8/10/2024 – 9/10/2024",
        greatest_diff => "M",
        locale => "en",
        options => {
            calendar => "gregorian",
            day => "numeric",
            month => "numeric",
            year => "numeric",
        },
        pattern => "M/d/y – M/d/y",
        skeleton => "yMd",
    },
    {
        date1 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => 2023 },
        date2 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "9/10/2023 – 9/10/2024",
        greatest_diff => "y",
        locale => "en",
        options => {
            calendar => "gregorian",
            day => "numeric",
            month => "numeric",
            year => "numeric",
        },
        pattern => "M/d/y – M/d/y",
        skeleton => "yMd",
    },
    {
        date1 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        date2 => { day => 11, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "Tue, 9/10/2024 – Wed, 9/11/2024",
        greatest_diff => "d",
        locale => "en",
        options => {
            calendar => "gregorian",
            day => "numeric",
            month => "numeric",
            weekday => "short",
            year => "numeric",
        },
        pattern => "E, M/d/y – E, M/d/y",
        skeleton => "yMEd",
    },
    {
        date1 => { day => 10, hour => 0, minute => 0, month => 8, second => 0, year => 2024 },
        date2 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "Sat, 8/10/2024 – Tue, 9/10/2024",
        greatest_diff => "M",
        locale => "en",
        options => {
            calendar => "gregorian",
            day => "numeric",
            month => "numeric",
            weekday => "short",
            year => "numeric",
        },
        pattern => "E, M/d/y – E, M/d/y",
        skeleton => "yMEd",
    },
    {
        date1 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => 2023 },
        date2 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "Sun, 9/10/2023 – Tue, 9/10/2024",
        greatest_diff => "y",
        locale => "en",
        options => {
            calendar => "gregorian",
            day => "numeric",
            month => "numeric",
            weekday => "short",
            year => "numeric",
        },
        pattern => "E, M/d/y – E, M/d/y",
        skeleton => "yMEd",
    },
    {
        date1 => { day => 1, hour => 0, minute => 0, month => 6, second => 0, year => 2024 },
        date2 => { day => 1, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "Jun – Sep 2024",
        greatest_diff => "M",
        locale => "en",
        options => { calendar => "gregorian", month => "short", year => "numeric" },
        pattern => "MMM – MMM y",
        skeleton => "yMMM",
    },
    {
        date1 => { day => 1, hour => 0, minute => 0, month => 9, second => 0, year => 2023 },
        date2 => { day => 1, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "Sep 2023 – Sep 2024",
        greatest_diff => "y",
        locale => "en",
        options => { calendar => "gregorian", month => "short", year => "numeric" },
        pattern => "MMM y – MMM y",
        skeleton => "yMMM",
    },
    {
        date1 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        date2 => { day => 11, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "Sep 10 – 11, 2024",
        greatest_diff => "d",
        locale => "en",
        options => {
            calendar => "gregorian",
            day => "numeric",
            month => "short",
            year => "numeric",
        },
        pattern => "MMM d – d, y",
        skeleton => "yMMMd",
    },
    {
        date1 => { day => 10, hour => 0, minute => 0, month => 8, second => 0, year => 2024 },
        date2 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "Aug 10 – Sep 10, 2024",
        greatest_diff => "M",
        locale => "en",
        options => {
            calendar => "gregorian",
            day => "numeric",
            month => "short",
            year => "numeric",
        },
        pattern => "MMM d – MMM d, y",
        skeleton => "yMMMd",
    },
    {
        date1 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => 2023 },
        date2 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "Sep 10, 2023 – Sep 10, 2024",
        greatest_diff => "y",
        locale => "en",
        options => {
            calendar => "gregorian",
            day => "numeric",
            month => "short",
            year => "numeric",
        },
        pattern => "MMM d, y – MMM d, y",
        skeleton => "yMMMd",
    },
    # NOTE: Test #70
    {
        date1 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        date2 => { day => 11, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "Tue, Sep 10 – Wed, Sep 11, 2024",
        greatest_diff => "d",
        locale => "en",
        options => {
            calendar => "gregorian",
            day => "numeric",
            month => "short",
            weekday => "short",
            year => "numeric",
        },
        pattern => "E, MMM d – E, MMM d, y",
        skeleton => "yMMMEd",
    },
    {
        date1 => { day => 10, hour => 0, minute => 0, month => 8, second => 0, year => 2024 },
        date2 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "Sat, Aug 10 – Tue, Sep 10, 2024",
        greatest_diff => "M",
        locale => "en",
        options => {
            calendar => "gregorian",
            day => "numeric",
            month => "short",
            weekday => "short",
            year => "numeric",
        },
        pattern => "E, MMM d – E, MMM d, y",
        skeleton => "yMMMEd",
    },
    {
        date1 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => 2023 },
        date2 => { day => 10, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "Sun, Sep 10, 2023 – Tue, Sep 10, 2024",
        greatest_diff => "y",
        locale => "en",
        options => {
            calendar => "gregorian",
            day => "numeric",
            month => "short",
            weekday => "short",
            year => "numeric",
        },
        pattern => "E, MMM d, y – E, MMM d, y",
        skeleton => "yMMMEd",
    },
    {
        date1 => { day => 1, hour => 0, minute => 0, month => 6, second => 0, year => 2024 },
        date2 => { day => 1, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "June – September 2024",
        greatest_diff => "M",
        locale => "en",
        options => { calendar => "gregorian", month => "long", year => "numeric" },
        pattern => "MMMM – MMMM y",
        skeleton => "yMMMM",
    },
    {
        date1 => { day => 1, hour => 0, minute => 0, month => 9, second => 0, year => 2023 },
        date2 => { day => 1, hour => 0, minute => 0, month => 9, second => 0, year => 2024 },
        expects => "September 2023 – September 2024",
        greatest_diff => "y",
        locale => "en",
        options => { calendar => "gregorian", month => "long", year => "numeric" },
        pattern => "MMMM y – MMMM y",
        skeleton => "yMMMM",
    },
];

my $failed = [];
for( my $i = 0; $i < scalar( @$tests ); $i++ )
{
    if( defined( $TEST_ID ) )
    {
        next unless( $i == $TEST_ID );
        last if( $i > $TEST_ID );
    }
    my $test = $tests->[$i];
    my @keys = sort( keys( %{$test->{options}} ) );
    my $dt1 = DateTime->new( %{$test->{date1}} );
    my $dt2 = DateTime->new( %{$test->{date2}} );
    my $date1 = $dt1->iso8601;
    my $date2 = $dt2->iso8601;
    local $" = ', ';
    subtest 'DateTime::Format::Intl->new( ' . ( ref( $test->{locale} ) eq 'ARRAY' ? "[@{$test->{locale}}]" : $test->{locale} ) . ", \{@keys\} )->format_range( $date1, $date2 )" => sub
    {
        local $SIG{__DIE__} = sub
        {
            diag( "Test No ${i} died: ", join( '', @_ ) );
        };
        my $fmt = DateTime::Format::Intl->new( $test->{locale}, $test->{options} );
        SKIP:
        {
            isa_ok( $fmt => 'DateTime::Format::Intl' );
            if( !defined( $fmt ) )
            {
                diag( "Error instantiating the DateTime::Format::Intl object: ", DateTime::Format::Intl->error );
                skip( "Unable to instantiate a new DateTime::Format::Intl object.", 1 );
            }
            my $str = $fmt->format_range( $dt1, $dt2 );
            if( !defined( $str ) )
            {
                diag( "Error formatting date ${date1} to ${date2}: ", $fmt->error );
            }
            $str =~ s/[[:blank:]\h]/ /g;
            my $best_pattern = $fmt->interval_pattern;
            my $skeleton = $fmt->interval_skeleton;
            my $diff = $fmt->greatest_diff;
            if( !defined( $best_pattern ) )
            {
                diag( "Error getting the best skeleton: ", $fmt->error );
            }
            if( !is( $diff => $test->{greatest_diff}, "\$fmt->greatest_diff -> '" . ( $test->{greatest_diff} // 'undef' ) . "'" ) )
            {
                # push( @$failed, { test => $i, skeleton => $test->{expects}, diff => $test->{greatest_diff} } );
                push( @$failed, { test => $i, interval_pattern => $best_pattern, interval_skeleton => $skeleton, got => $str, got_diff => $test->{greatest_diff}, %$test } );
            }
            if( !is( $str => $test->{expects}, "\$fmt->format_range( $date1, $date2 ) -> '$test->{expects}'" ) )
            {
                push( @$failed, { test => $i, interval_pattern => $best_pattern, interval_skeleton => $skeleton, got => $str, got_diff => $test->{greatest_diff}, %$test } );
            }
        };
    };
}


done_testing();

__END__
