/* GNOME DB library
 * Copyright (C) 1999-2000 Rodrigo Moya
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "config.h"
#include <gnome-db.h>
#include <gnome-db-help.h>
#if defined(HAVE_GTKHTML)
#  include <gtkhtml/gtkhtml.h>
#endif

static void load_url (GnomeDbHelp *help, const gchar *url);

enum
{
  PAGE_LOADED,
  LAST_SIGNAL
};

static gint db_help_signals[LAST_SIGNAL] = { 0 };

/*
 * Callbacks
 */
static void
menu_item_selected_cb (GtkMenuItem *menu_item, GnomeDbHelp *help)
{
  gchar* url;
  
  g_return_if_fail(GTK_IS_MENU_ITEM(menu_item));
  g_return_if_fail(GNOME_DB_IS_HELP(help));
  
  url = (gchar *) gtk_object_get_data(GTK_OBJECT(menu_item), "GNOME_DB_Help_MenuItemLabel");
  if (url)
    {
      load_url(help, url);
    }
}

#if defined(HAVE_GTKHTML)
static void
link_clicked_cb (GtkHTML *html, const gchar *url, GnomeDbHelp *help)
{
  g_print("url = %s\n", url);
}
#endif

/*
 * Private functions
 */
static void
fill_table_of_contents (GnomeDbHelp *help)
{
  GtkWidget* menu;
  GList*     entries;
  GList*     node;

  g_return_if_fail(GNOME_DB_IS_HELP(help));
  g_return_if_fail(GTK_IS_OPTION_MENU(help->index_list));
  
  gtk_option_menu_remove_menu(GTK_OPTION_MENU(help->index_list));
  menu = gtk_menu_new();
  gtk_option_menu_set_menu(GTK_OPTION_MENU(help->index_list), menu);
  
  /* search for installed documentation */
  entries = gda_config_list_keys(GNOME_DB_CONFIG_SECTION_HELP_VIEWER_DOCS);
  for (node = g_list_first(entries); node != NULL; node = node->next)
    {
      gchar* title = (gchar *) node->data;
      if (title)
        {
          gchar* url;
          gchar* entry = g_strdup_printf("%s/%s", GNOME_DB_CONFIG_SECTION_HELP_VIEWER_DOCS, title);
          url = gda_config_get_string(entry);
          if (url)
            {
              GtkWidget* menu_item = gtk_menu_item_new_with_label(title);
              gtk_signal_connect(GTK_OBJECT(menu_item),
                                 "activate",
                                 GTK_SIGNAL_FUNC(menu_item_selected_cb),
                                 (gpointer) help);
              gtk_object_set_data(GTK_OBJECT(menu_item), "GNOME_DB_Help_MenuItemLabel", url);
              gtk_widget_show(menu_item);
              gtk_menu_append(GTK_MENU(menu), menu_item);
            }
          g_free((gpointer) entry);
        }
    }
  gda_config_free_list(entries);
  gtk_widget_show(menu);
}

static void
load_url (GnomeDbHelp *help, const gchar *url)
{
#if defined(HAVE_GTKHTML)
  GtkHTMLStream* handle;
#endif
  gchar*         contents;
  
  g_return_if_fail(GNOME_DB_IS_HELP(help));
  g_return_if_fail(url != NULL);
  
#if defined(HAVE_GTKHTML)
  handle = gtk_html_begin(GTK_HTML(help->html_viewer));
#endif
  contents = gda_util_load_file(url);
  if (contents)
    {
#if defined(HAVE_GTKHTML)
      gtk_html_write(GTK_HTML(help->html_viewer), handle, contents, strlen(contents));
#endif
      g_free((gpointer) contents);
      gtk_signal_emit(GTK_OBJECT(help), db_help_signals[PAGE_LOADED], url);
    }
#if defined(HAVE_GTKHTML)
  gtk_html_end(GTK_HTML(help->html_viewer), handle, GTK_HTML_STREAM_OK);
#endif
}

/*
 * Public functions
 */
static void
gnome_db_help_destroy (GnomeDbHelp *help)
{
}

static void
gnome_db_help_class_init (GnomeDbHelpClass *klass)
{
  GtkObjectClass* object_class = GTK_OBJECT_CLASS(klass);
  
  /* install default docs if it's the first time */
  if (!gda_config_has_section(GNOME_DB_CONFIG_SECTION_HELP_VIEWER))
    {
      gchar* file;
      
      file = gnome_help_file_find_file("gnome-db", "index.html");
      if (file)
        gda_config_set_string(GNOME_DB_CONFIG_SECTION_HELP_VIEWER_DOCS "/GNOME-DB-Manual", file);
      
      file = gnome_help_file_find_file("libgda", "index.html");
      if (file)
        gda_config_set_string(GNOME_DB_CONFIG_SECTION_HELP_VIEWER_DOCS "/GDA-Manual", file);
    }
  
  /* add signals */
  db_help_signals[PAGE_LOADED] =
           gtk_signal_new("page_loaded", GTK_RUN_LAST, object_class->type,
                          GTK_SIGNAL_OFFSET(GnomeDbHelpClass, page_loaded),
                          gtk_marshal_NONE__POINTER,
                          GTK_TYPE_NONE, 1, GTK_TYPE_POINTER);
  gtk_object_class_add_signals(object_class, db_help_signals, LAST_SIGNAL);
}

static void
gnome_db_help_init (GnomeDbHelp *help)
{
  GtkWidget* table;
  GtkWidget* scroll;
  GtkWidget* frame;

  gtk_signal_connect(GTK_OBJECT(help),
                     "destroy",
                     GTK_SIGNAL_FUNC(gnome_db_help_destroy),
                     NULL);

  table = gnome_db_new_table_widget(1, 2, FALSE);
  gtk_box_pack_start(GTK_BOX(help), table, 1, 1, 0);
  
  /* add table of contents */
  help->index_list = gnome_db_new_option_menu_widget();
  gtk_table_attach(GTK_TABLE(table), help->index_list, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 3, 3);
  fill_table_of_contents(help);
  
  /* add HTML viewer */
  frame = gnome_db_new_frame_widget(NULL);
  gtk_table_attach(GTK_TABLE(table), frame, 0, 1, 1, 2,
                   GTK_FILL | GTK_SHRINK | GTK_EXPAND,
                   GTK_FILL | GTK_SHRINK | GTK_EXPAND,
                   3, 3);
  scroll = gnome_db_new_scrolled_window_widget();
  gtk_container_add(GTK_CONTAINER(frame), scroll);
#if defined(HAVE_GTKHTML)
  help->html_viewer = gtk_html_new();
  //gtk_html_set_editable(GTK_HTML(help->html_viewer), FALSE);
  gtk_html_allow_selection(GTK_HTML(help->html_viewer), TRUE);
  gtk_widget_show(help->html_viewer);
  gtk_container_add(GTK_CONTAINER(scroll), help->html_viewer);
  gtk_signal_connect(GTK_OBJECT(help->html_viewer),
                     "link_clicked",
                     GTK_SIGNAL_FUNC(link_clicked_cb),
                     (gpointer) help);
#endif
}

GtkType
gnome_db_help_get_type (void)
{
  static guint db_help_type = 0;

  if (!db_help_type)
    {
      GtkTypeInfo db_help_info =
      {
        "GnomeDbHelp",
        sizeof (GnomeDbHelp),
        sizeof (GnomeDbHelpClass),
        (GtkClassInitFunc) gnome_db_help_class_init,
        (GtkObjectInitFunc) gnome_db_help_init,
        (GtkArgSetFunc) NULL,
        (GtkArgGetFunc) NULL
      };
      db_help_type = gtk_type_unique(gtk_vbox_get_type(), &db_help_info);
    }
  return (db_help_type);
}

/**
 * gnome_db_help_new
 */
GtkWidget *
gnome_db_help_new (void)
{
  GtkWidget* help;
  
  help = GTK_WIDGET(gtk_type_new(gnome_db_help_get_type()));
  gnome_db_help_go_home(GNOME_DB_HELP(help));
  return help;
}

/**
 * gnome_db_help_show_url
 */
void
gnome_db_help_show_url (GnomeDbHelp *help, const gchar *url)
{
  g_return_if_fail(GNOME_DB_IS_HELP(help));
  g_return_if_fail(url != NULL);
  
  load_url(help, url);
}

/**
 * gnome_db_help_go_home
 */
void
gnome_db_help_go_home (GnomeDbHelp *help)
{
  g_return_if_fail(GNOME_DB_IS_HELP(help));
  load_url(help, gnome_db_help_get_home());
}

/**
 * gnome_db_help_go_back
 */
void
gnome_db_help_go_back (GnomeDbHelp *help)
{
  g_return_if_fail(GNOME_DB_IS_HELP(help));
}

/**
 * gnome_db_help_go_forward
 */
void
gnome_db_help_go_forward (GnomeDbHelp *help)
{
}

/**
 * gnome_db_help_get_home
 *
 * Return the URL for the help system home page
 */
const gchar *
gnome_db_help_get_home (void)
{
  return gda_config_get_string(GNOME_DB_CONFIG_KEY_HELP_VIEWER_HOME);
}

