/* GNOME DB libary
 * Copyright (C) 1999 Rodrigo Moya
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gda-postgres-command.h"
#include "gda-postgres-recset.h"
#include "gda-postgres-error.h"
#include "gda-postgres-connection.h"
#include "gda-postgres-field.h"

/* create a new command object */
Gda_POSTGRES_Command *
gda_postgres_cmd_new (void)
{
  Gda_POSTGRES_Command *rc = g_new0(Gda_POSTGRES_Command, 1);
  return (rc);
}

void
gda_postgres_cmd_free (Gda_POSTGRES_Command *cmd)
{
  if (cmd->cmd != 0)
    g_free((gpointer) cmd->cmd);
  cmd->cmd = 0;
  /* free rowsets? */
  g_free((gpointer) cmd);
}

Gda_POSTGRES_Connection *
gda_postgres_cmd_set_connection (Gda_POSTGRES_Command *cmd,
                                 Gda_POSTGRES_Connection *cnc)
{
  Gda_POSTGRES_Connection *rc = cmd->cnc;
  cmd->cnc = cnc;
  return (rc);
}

static Gda_POSTGRES_Recordset *
init_recset_fields (Gda_POSTGRES_Recordset *recset)
{
  register gint cnt;
  guint ncols;

  if (recset->pq_data)
    ncols = PQnfields(recset->pq_data);
  else
    ncols = Gda_Builtin_Result_get_nbfields(recset->btin_res);
  /* check parameters */
  g_return_val_if_fail(recset != NULL, NULL);
  for (cnt = 0; cnt < ncols; cnt++)
    {
      Gda_POSTGRES_Field *f = gda_postgres_field_new();
      if (recset->pq_data)
	{
	  f->name = g_strdup(PQfname(recset->pq_data, cnt));
	  f->sql_type = PQftype(recset->pq_data, cnt);
	}
      else
	{
	  f->name = g_strdup(Gda_Builtin_Result_get_fname
			     (recset->btin_res, cnt));
	  f->sql_type = Gda_Builtin_Result_get_ftype(recset->btin_res, cnt);
	}

      f->c_type = 
	gda_postgres_connection_get_c_type(recset->cnc, f->sql_type);
      /* f->nullable can not be set because no way of getting it
	 from the beckend! */
      f->value = NULL;
      f->precision = 0; 
      f->num_scale = 0;
      if (recset->pq_data)
	f->defined_length = f->actual_length = PQfsize(recset->pq_data, cnt);
      else
	f->defined_length = f->actual_length = Gda_Builtin_Result_get_fsize
	  (recset->btin_res, cnt);;
      f->malloced = 0;
      /* add field definition to list */
      recset->fields = g_list_append(recset->fields, f);
    }
  return (recset);
}

Gda_POSTGRES_Recordset *
gda_postgres_cmd_execute (Gda_POSTGRES_Command *cmd,
                          Gda_POSTGRES_Error *error,
                          const GDA_CmdParameterSeq *params,
			  gulong *affected,
                          gulong options)
{
  Gda_POSTGRES_Recordset *recset;

  /* create recordset object */
  recset = gda_postgres_recset_new();
  recset->cnc = cmd->cnc;
  fprintf(stderr, "gda_postgres_cmd_execute(): enter\n");
  
  /* FIXME: parameters stuff, ask Michael */
  /* execute query */
  recset->pq_data = PQexec(cmd->cnc->pq_conn, cmd->cmd);
  if (recset->pq_data != 0)
    {
      switch (PQresultStatus(recset->pq_data))
	{
	case PGRES_EMPTY_QUERY :
	case PGRES_TUPLES_OK :
	  recset->at_end = FALSE;
	  recset->at_begin = TRUE;
	  recset->pos = 0;
	  *affected = PQntuples(recset->pq_data);
	  /*recset->affected = *affected */
	  return (init_recset_fields(recset));
	case PGRES_COMMAND_OK :
	case PGRES_COPY_OUT :
	case PGRES_COPY_IN :
	  recset->at_begin = recset->at_end = TRUE;
	  recset->fields = 0;
	  recset->pos = 0;
	  *affected = 0;
	  return (recset);
	default :
	  gda_postgres_recset_free(recset);
	  gda_postgres_error_make(error, recset, cmd->cnc, "PQexec");
	  *affected = 0;
	  return (0);
	}
    }
  else 
    gda_postgres_error_make(error, recset, cmd->cnc, "PQexec");
  return (0);
}

Gda_POSTGRES_Recordset *
gda_postgres_cmd_execute_from_builtin (Gda_POSTGRES_Command *cmd,
				       Gda_Builtin_Result *res,
				       gulong *affected)
{
  Gda_POSTGRES_Recordset *recset;

  /* create recordset object */
  recset = gda_postgres_recset_new();
  recset->cnc = cmd->cnc;
  fprintf(stderr, "gda_postgres_cmd_execute_from_builtin(): enter\n");
  
  /* execute query */
  recset->pq_data = NULL;
  recset->btin_res = res;
  recset->at_end = FALSE;
  recset->at_begin = TRUE;
  recset->pos = 0;
  *affected = Gda_Builtin_Result_get_nbtuples(res);

  return (init_recset_fields(recset));
}
