/* GNOME DB
 * Copyright (C) 1998 Michael Lausch
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>

#include <mysql/mysql.h>
#include <glib.h>

#include "gda-mysql-connection.h"
#include "gda-mysql-recset.h"

typedef Gda_MYSQL_Recordset* (*schema_ops_fn) (Gda_MYSQL_Recordset*,
                                               Gda_MYSQL_Connection*,
                                               GDA_Connection_Constraint* constraints,
                                               gint length);

schema_ops_fn schema_ops[GDA_Connection_GDCN_SCHEMA_LAST] = { 0 };

static Gda_MYSQL_Recordset* schema_tables(Gda_MYSQL_Recordset* recset,
                                          Gda_MYSQL_Connection*,
                                          GDA_Connection_Constraint* constraints,
                                          gint length);

static Gda_MYSQL_Recordset* schema_views(Gda_MYSQL_Recordset* recset,
                                         Gda_MYSQL_Connection*,
                                         GDA_Connection_Constraint* constraints,
                                         gint length);
static Gda_MYSQL_Recordset* schema_cols(Gda_MYSQL_Recordset* recset,
                                        Gda_MYSQL_Connection*,
                                        GDA_Connection_Constraint* constraints,
                                        gint length);

void           initialize_schema_ops(void);

void
initialize_schema_ops(void)
{
  memset((gpointer) schema_ops, 0, sizeof(schema_ops));

  schema_ops[GDA_Connection_GDCN_SCHEMA_TABLES] = schema_tables;
  schema_ops[GDA_Connection_GDCN_SCHEMA_VIEWS]  = schema_views;
  schema_ops[GDA_Connection_GDCN_SCHEMA_COLS]   = schema_cols;
}


gint SQLTABLES_order[]={2};

static MYSQL_FIELD SQLTABLES_fields[] = {
  {"Table_qualifer","Catalog",NULL,FIELD_TYPE_VAR_STRING,NAME_LEN,0},
  {"Table_owner","Catalog",NULL,FIELD_TYPE_VAR_STRING,NAME_LEN,0},
  {"Table_name","Catalog",NULL,FIELD_TYPE_VAR_STRING,NAME_LEN,NAME_LEN,0},
  {"Table_type","Catalog",NULL,FIELD_TYPE_VAR_STRING,NAME_LEN,5,0},
  {"Remarks","Catalog",NULL,FIELD_TYPE_VAR_STRING,NAME_LEN,11}};

char *SQLTABLES_values[]={"","",NULL,"TABLE","MySQL table"};

static MYSQL_ROW
fix_fields_copy(Gda_MYSQL_Recordset* rs, MYSQL_ROW row)
{
  gint i;
  for (i = 0 ; i < rs->order_count; i++)
    rs->array[rs->order[i]]=row[i];
  return rs->array;
}

static void
link_fields(MYSQL_RES* result, MYSQL_FIELD* fields, gint field_count)
{
  result->fields = fields;
  result->field_count = field_count;
  result->current_field = 0;
}

static Gda_MYSQL_Recordset*
schema_tables(Gda_MYSQL_Recordset* recset, Gda_MYSQL_Connection* cnc,
	      GDA_Connection_Constraint* constraints, gint length)
{
  recset->cnc = cnc;
  
  recset->mysql_res = mysql_list_tables(recset->cnc->mysql, 0);
  link_fields(recset->mysql_res, SQLTABLES_fields, 5);
  recset->order = SQLTABLES_order;
  recset->order_count = 1;
  recset->array = g_new0(gchar*, 5);
  memcpy(recset->array, SQLTABLES_values, sizeof(SQLTABLES_values));
  recset->fix_fields = fix_fields_copy;
  recset->mysql_res->lengths = g_new0(gint, 5);
  return recset;
}

static Gda_MYSQL_Recordset*
schema_views(Gda_MYSQL_Recordset* recset, Gda_MYSQL_Connection* cnc,
	      GDA_Connection_Constraint* constraints, gint length)
{
  return recset;
}

static Gda_MYSQL_Recordset*
schema_cols(Gda_MYSQL_Recordset* recset, Gda_MYSQL_Connection* cnc,
	      GDA_Connection_Constraint* constraints, gint length)
{
  return recset;
}


Gda_MYSQL_Connection*
gda_mysql_connection_new()
{
  static gint initialized = 0;
  
  Gda_MYSQL_Connection* c = g_new0(Gda_MYSQL_Connection, 1);

  if (!initialized)
    {
      initialize_schema_ops();
      initialized = 1;
    }
  return c;
}


void
gda_mysql_connection_free(Gda_MYSQL_Connection* c)
{
  if (c->host)
    g_free(c->host);
  if (c->db)
    g_free(c->db);
  if (c->port)
    g_free(c->port);
  if (c->unix_socket)
    g_free(c->unix_socket);
  if (c->flags)
    g_free(c->flags);
  if (c->mysql)
    {
      gda_mysql_connection_close(c);
      g_free(c->mysql);
    }
  if (c->user)
    g_free(c->user);
  if (c->password)
    g_free(c->password);
  g_free(c);
}


static gchar*
get_value(gchar* ptr)
{
  while (*ptr && *ptr != '=')
    ptr++;
  
  if (!*ptr)
    return 0;
  ptr++;
  if (!*ptr)
    return 0;
  
  while (*ptr && isspace(*ptr))
    ptr++;

  return g_strdup(ptr);
}


gint
gda_mysql_connection_open(Gda_MYSQL_Connection* cnc,
			  const gchar*          cnc_string,
			  const gchar*          user,
			  const gchar*          passwd)
{
  gchar* ptr_s;
  gchar* ptr_e;
  MYSQL* rc;
  int    err;
  
  ptr_s = (gchar*)cnc_string;
  while (ptr_s && *ptr_s)
    {
      ptr_e = strchr(ptr_s, ';');

      if (ptr_e)
	*ptr_e = '\0';
      if (strncasecmp(ptr_s, "HOST", strlen("HOST")) == 0)
        cnc->host = get_value(ptr_s);
      else if (strncasecmp(ptr_s, "DATABASE", strlen("DATABASE")) == 0)
        cnc->db = get_value(ptr_s);
      else if (strncasecmp(ptr_s, "PORT", strlen("PORT")) == 0)
        cnc->port = get_value(ptr_s);
      else if (strncasecmp(ptr_s, "UNIX_SOCKET", strlen("UNIX_SOCKET")) == 0)
        cnc->unix_socket = get_value(ptr_s);
      else if (strncasecmp(ptr_s, "FLAGS", strlen("FLAGS")) == 0)
        cnc->flags = get_value(ptr_s);
      ptr_s = ptr_e;
      if (ptr_s)
	ptr_s++;
    }
  cnc->mysql = g_new0(MYSQL, 1);
  fprintf(stderr,"cnc->mysql = %p\n", cnc->mysql);
  fprintf(stderr,"** Calling mysql_real_connect\n");

#if MYSQL_VERSION_ID >= 32200
  rc = mysql_real_connect(cnc->mysql, cnc->host,
			  cnc->user,
			  cnc->password,
			  cnc->db,
			  cnc->port ? atoi(cnc->port) : 0,
			  cnc->unix_socket,
			  cnc->flags ? atoi(cnc->flags) : 0);
#else
  rc = mysql_real_connect(cnc->mysql, cnc->host,
			  cnc->user,
			  cnc->password,
			  cnc->port ? atoi(cnc->port) : 0,
			  cnc->unix_socket,
			  cnc->flags ? atoi(cnc->flags) : 0);
#endif

  fprintf(stderr,"cnc->mysql after connect = %p\n", cnc->mysql);
  
  if (!rc)
    {
      gda_mysql_error_make(cnc, "gda_mysql_connection_open");
      return -1;
    }

#if MYSQL_VERSION_ID < 32200
  fprintf(stderr,"before mysql_select_db: cnc->db = '%s'\n", cnc->db);
  fprintf(stderr,"cnc->mysql = %p\n", cnc->mysql);
  fprintf(stderr,"*** Calling mysql_select_db\n");
  err = mysql_select_db(cnc->mysql, cnc->db);
  fprintf(stderr,"mysql_select_db returns %d\n", err);
  if (err != 0)
    {
      fprintf(stderr,"mysql errno  = %d\n",mysql_errno(cnc->mysql));
      fprintf(stderr,"mysql errmsg = '%s'\n",
	      mysql_error(cnc->mysql));
      return -1;
    }
#endif

  fprintf(stderr,"gda_mysql_connection_open: returning\n");
  return 0;
}

Gda_MYSQL_Recordset*
gda_mysql_connection_open_schema (Gda_MYSQL_Connection* cnc,
				  Gda_MYSQL_Error* e,
				  GDA_Connection_QType t,
				  GDA_Connection_Constraint* constraints,
				  gint length)
{
  Gda_MYSQL_Recordset* rc = gda_mysql_recset_new();
  schema_ops_fn        fn = schema_ops[t];

  if (fn)
    {
      return fn(rc, cnc, constraints, length);
    }
  else
    {
      g_warning("connection_open_schema: Unhandled SCHEMA_QTYPE %d\n", t);
    }
  return rc;
}


gint
gda_mysql_connection_close(Gda_MYSQL_Connection* c)
{
  fprintf(stderr, "Calling mysql_close\n");
  mysql_close(c->mysql);
  return 0;
}

gboolean
gda_mysql_connection_supports (Gda_MYSQL_Connection *cnc, GDA_Connection_Feature feature)
{
  g_return_val_if_fail(cnc != 0, FALSE);

  switch (feature)
    {
    case GDA_Connection_FEATURE_TRANSACTIONS : return FALSE;
    case GDA_Connection_FEATURE_SEQUENCES : return TRUE; /* FIXME: ??? */
    case GDA_Connection_FEATURE_INHERITANCE : return FALSE;
    case GDA_Connection_FEATURE_PROCS : return FALSE;
    default :
      fprintf(stderr, "Invalid GDA_Connection_Feature %d", feature);
    }
  return FALSE;
}