/* GNOME DB library
 * Copyright (C) 1999, 2000 Rodrigo Moya
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef __gda_xml_h__
#define __gda_xml_h__ 1

#include <gda-connection.h>

#include <gnome-xml/parser.h>
#include <gnome-xml/tree.h>

/*
 * Data structures and utility functions for reading and writing
 * GDA objects from/to XML
 *
 * GDA xml uses a custom DTD to represent the entire database in
 * XML format. Currently, the format is as follows:
 *
 * <database>
 *   <table name="TABLE_NAME" [command="CREATE...]">
 *     <field name="" EMPTY>
 *     ...
 *     <row>
 *       <value EMPTY>
 *       ...
 *     </row>
 *     ...
 *   </table>
 *   ...
 *   <view name="VIEW_NAME" [command="CREATE...]">
 *   ...
 * </database>
 */
 
typedef struct _Gda_XmlDatabase      Gda_XmlDatabase;
typedef struct _Gda_XmlDatabaseClass Gda_XmlDatabaseClass;

#define GDA_TYPE_XML_DATABASE            (gda_xml_database_get_type())
#define GDA_XML_DATABASE(obj)            GTK_CHECK_CAST(obj, GDA_TYPE_XML_DATABASE, Gda_XmlDatabase)
#define GDA_XML_DATABASE_CLASS(klass)    GTK_CHECK_CLASS_CAST(klass, GDA_TYPE_XML_DATABASE, Gda_XmlDatabaseClass)
#define IS_GDA_XML_DATABASE(obj)         GTK_CHECK_TYPE(obj, GDA_TYPE_XML_DATABASE)
#define IS_GDA_XML_DATABASE_CLASS(klass) (GTK_CHECK_CLASS_TYPE((klass), GDA_TYPE_XML_DATABASE))

typedef struct _Gda_XmlObject
{
  Gda_XmlDatabase*       db;
  xmlNodePtr             xmlnode;
  GList*                 children;
  struct _Gda_XmlObject* parent;
} Gda_XmlObject;

struct _Gda_XmlDatabase
{
  GtkObject object;
  xmlDocPtr xmldoc;
  gchar*    filename;
  GList*    objects;
};

struct _Gda_XmlDatabaseClass
{
  GtkObjectClass parent_class;
  void           (*changed)   (Gda_XmlDatabase *db);
  void           (*add_object)(Gda_XmlDatabase *db, Gda_XmlObject *obj);
};

#define GDA_XML_OBJECT_TYPE_TABLE "TABLE"
#define GDA_XML_OBJECT_TYPE_FIELD "FIELD"
#define GDA_XML_OBJECT_TYPE_ROW   "ROW"
#define GDA_XML_OBJECT_TYPE_VIEW  "VIEW"

#define GDA_XML_PROPERTY_NAME "NAME"
#define GDA_XML_PROPERTY_SIZE "SIZE"
#define GDA_XML_PROPERTY_SQL  "SQL"

typedef gboolean (*Gda_XmlEnumFunc)    (Gda_XmlDatabase *db, Gda_XmlObject *obj, gpointer user_data);
typedef gboolean (*Gda_XmlEnumPropFunc)(Gda_XmlObject *obj, const gchar *prop, gpointer user_data);

guint            gda_xml_database_get_type      (void);
Gda_XmlDatabase* gda_xml_database_new           (const gchar *name);
Gda_XmlDatabase* gda_xml_database_new_from_file (const gchar *filename);
void             gda_xml_database_free          (Gda_XmlDatabase *db);

gboolean         gda_xml_database_save          (Gda_XmlDatabase *db, const gchar *filename);

void             gda_xml_database_changed       (Gda_XmlDatabase *db);
const gchar*     gda_xml_database_get_name      (Gda_XmlDatabase *db);
void             gda_xml_database_set_name      (Gda_XmlDatabase *db, const gchar *name);
const gchar*     gda_xml_database_get_date      (Gda_XmlDatabase *db);
void             gda_xml_database_set_date      (Gda_XmlDatabase *db, const gchar *dt);
const gchar*     gda_xml_database_get_filename  (Gda_XmlDatabase *db);
void             gda_xml_database_set_filename  (Gda_XmlDatabase *db, const gchar *filename);

/*
 * Gda_XmlObject management functions
 * They are for internal use, so although you can use them in your applications,
 * its use is not recommended
 */
Gda_XmlObject*   gda_xml_database_add_object    (Gda_XmlDatabase *db,
                                                 const gchar *name,
                                                 const gchar *type);
void             gda_xml_database_delete_object (Gda_XmlDatabase *db, Gda_XmlObject *obj);
Gda_XmlObject*   gda_xml_database_find_object   (Gda_XmlDatabase *db,
                                                 const gchar *name,
                                                 const gchar *type);
gboolean         gda_xml_database_enum_objects  (Gda_XmlDatabase *db,
                                                 Gda_XmlEnumFunc func,
                                                 const gchar *type,
                                                 gpointer user_data);
Gda_XmlObject*   gda_xml_object_add_child       (Gda_XmlObject *obj, const gchar *type);
void             gda_xml_object_delete_child    (Gda_XmlObject *obj, Gda_XmlObject* child);
gboolean         gda_xml_object_enum_children   (Gda_XmlObject *obj,
                                                 Gda_XmlEnumFunc func,
                                                 const gchar *type,
                                                 gpointer user_data);

const gchar*     gda_xml_object_get_property    (Gda_XmlObject *obj, const gchar *prop);
void             gda_xml_object_set_property    (Gda_XmlObject *obj, const gchar *prop, const gchar *value);
gboolean         gda_xml_object_enum_properties (Gda_XmlObject *obj, Gda_XmlEnumPropFunc func, gpointer user_data);
const gchar*     gda_xml_object_type            (Gda_XmlObject *obj);

/*
 * Tables
 */
Gda_XmlObject*   gda_xml_table_new              (Gda_XmlDatabase *db, const gchar *name);
Gda_XmlObject*   gda_xml_table_new_from_gda_recordset (Gda_XmlDatabase *db,
                                                       const gchar *name,
                                                       Gda_Recordset *recset,
                                                       gboolean export_data);
void             gda_xml_table_delete           (Gda_XmlDatabase *db, Gda_XmlObject *table);
void             gda_xml_table_delete_by_name   (Gda_XmlDatabase *db, const gchar *name);
const gchar*     gda_xml_table_get_name         (Gda_XmlObject *table);
void             gda_xml_table_set_name         (Gda_XmlObject *table, const gchar *name);

#define gda_xml_database_find_table(_db_, _name_) gda_xml_database_find_object(GDA_XML_DATABASE(_db_), (const gchar *) _name_, GDA_XML_OBJECT_TYPE_TABLE)
#define gda_xml_database_enum_tables(_db_, _func_, _user_data_) gda_xml_database_enum_objects(GDA_XML_DATABASE(_db_), (Gda_XmlEnumFunc) _func_, GDA_XML_OBJECT_TYPE_TABLE, _user_data_)

/*
 * Views
 */
#define gda_xml_database_find_view(_db_, _name_) gda_xml_database_find_object(GDA_XML_DATABASE(_db_), (const gchar *) _name_, GDA_XML_OBJECT_TYPE_VIEW)
#define gda_xml_database_enum_views(_db_, _func_, _user_data_) gda_xml_database_enum_objects(GDA_XML_DATABASE(_db_), (Gda_XmlEnumFunc) _func_, GDA_XML_OBJECT_TYPE_VIEW, _user_data_)

#endif
