/* GNOME DB libary
 * Copyright (C) 1999-2000 Rodrigo Moya
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gda-xml.h>

enum
{
  GDA_XML_DATABASE_CHANGED,
  GDA_XML_DATABASE_ADD_OBJECT,
  GDA_XML_LAST_SIGNAL
};

static gint gda_xml_database_signals[GDA_XML_LAST_SIGNAL] = { 0, };

static void gda_xml_database_init       (Gda_XmlDatabase *db);
static void gda_xml_database_class_init (Gda_XmlDatabaseClass *klass);

/* private functions */
static void
database_add_object_cb (Gda_XmlDatabase *db, Gda_XmlObject *obj)
{
  g_return_if_fail(IS_GDA_XML_DATABASE(db));
  g_return_if_fail(obj != 0);

  gtk_signal_emit(GTK_OBJECT(db),
                  gda_xml_database_signals[GDA_XML_DATABASE_CHANGED]);
}

static void
database_changed_cb (Gda_XmlDatabase *db)
{
  g_return_if_fail(IS_GDA_XML_DATABASE(db));
  
  /* update the modification time */
  gda_xml_database_set_date(db, 0);
}

static Gda_XmlObject *
find_object_by_name_and_type (Gda_XmlDatabase *db, const gchar *name, const gchar *type)
{
  GList* node;

  g_return_val_if_fail(IS_GDA_XML_DATABASE(db), 0);
  g_return_val_if_fail(name != 0, 0);
  g_return_val_if_fail(type != 0, 0);

  /* traverse the list of objects (already loaded) */
  node = g_list_first(db->objects);
  while (node != 0)
    {
      gchar*         tmp_name, *tmp_type;
      Gda_XmlObject* obj = (Gda_XmlObject *) node->data;

      tmp_name = xmlGetProp(obj->xmlnode, GDA_XML_PROPERTY_NAME);
      tmp_type = obj->xmlnode->name;
      if (tmp_name && tmp_type)
        {
          if (!g_strcasecmp(name, tmp_name) &&
              !g_strcasecmp(type, tmp_type))
            {
              return (obj);
            }
        }
      node = g_list_next(node);
    }
  return (0); /* not found */
}

static const gchar *
get_current_date (void)
{
  static gchar buf[256];
  time_t current = time(0);
  strftime(buf, sizeof(buf), "%c", localtime(&current));
  return ((const gchar *) buf);
}

static GList*
get_node_children (Gda_XmlObject *obj)
{
  xmlNodePtr xmlnode;
  GList*     children = 0;

  g_return_val_if_fail(obj != 0, 0);
  g_return_val_if_fail(IS_GDA_XML_DATABASE(obj->db), 0);

  xmlnode = obj->xmlnode->childs;
  while (xmlnode != 0)
    {
      Gda_XmlObject* child;
          
      child = g_new0(Gda_XmlObject, 1);
      child->db = obj->db;
      child->xmlnode = xmlnode;
      child->parent = obj;
      /* recursively call this function to retrieve children of children */
      child->children = get_node_children(child);

      children = g_list_append(children, (gpointer) child);
      xmlnode = xmlnode->next; /* next item in list */
    }
  return (children);
  
}

static void
gda_xml_database_class_init (Gda_XmlDatabaseClass *klass)
{
  GtkObjectClass *object_class = (GtkObjectClass *) klass;
  
  gda_xml_database_signals[GDA_XML_DATABASE_CHANGED] =
          gtk_signal_new("changed",
                         GTK_RUN_FIRST,
                         object_class->type,
                         GTK_SIGNAL_OFFSET(Gda_XmlDatabaseClass, changed),
                         gtk_marshal_NONE__POINTER,
                         GTK_TYPE_NONE, 0);
  gda_xml_database_signals[GDA_XML_DATABASE_ADD_OBJECT] =
          gtk_signal_new("add_object",
                         GTK_RUN_FIRST,
                         object_class->type,
                         GTK_SIGNAL_OFFSET(Gda_XmlDatabaseClass, add_object),
                         gtk_marshal_NONE__POINTER,
                         GTK_TYPE_NONE, 1, GTK_TYPE_POINTER);
  gtk_object_class_add_signals(object_class, gda_xml_database_signals,
                               GDA_XML_LAST_SIGNAL);

  klass->changed = database_changed_cb;
  klass->add_object = database_add_object_cb;
}

static void
gda_xml_database_init (Gda_XmlDatabase *db)
{
  db->xmldoc = 0;
  db->filename = 0;
  db->objects = 0;
}

/*
 * Gda_XmlDatabase interface
 */
guint
gda_xml_database_get_type (void)
{
  static guint gda_xml_database_type = 0;
  if (!gda_xml_database_type)
    {
      GtkTypeInfo gda_xml_database_info =
      {
        "Gda_XmlDatabase",
        sizeof(Gda_XmlDatabase),
        sizeof(Gda_XmlDatabaseClass),
        (GtkClassInitFunc) gda_xml_database_class_init,
        (GtkObjectInitFunc) gda_xml_database_init,
        (GtkArgSetFunc) 0,
        (GtkArgSetFunc) 0
      };
      gda_xml_database_type = gtk_type_unique(gtk_object_get_type(), 
                                              &gda_xml_database_info);
    }
  return (gda_xml_database_type);
}

/**
 * gda_xml_database_new
 * @name: The name to represent the database
 *
 * Allocates memory for a new Gda_XmlDatabase object
 *
 * Returns: a pointer to the newly allocated object, or NULL on error
 */
Gda_XmlDatabase *
gda_xml_database_new (const gchar *name)
{
  Gda_XmlDatabase *db;
  
  db = gtk_type_new(gda_xml_database_get_type());
  db->xmldoc = xmlNewDoc("1.0");
  db->xmldoc->root = xmlNewDocNode(db->xmldoc, 0, "DATABASE", 0);
  if (name != 0) xmlSetProp(db->xmldoc->root, "NAME", name);
  xmlSetProp(db->xmldoc->root, "DATE", get_current_date());
  db->filename = 0;
  return (db);
}

/**
 * gda_xml_database_new_from_file
 * @filename: path to an existing file
 *
 * Loads a XML file containing the representation of a database
 */
Gda_XmlDatabase *
gda_xml_database_new_from_file (const gchar *filename)
{
  Gda_XmlDatabase* db;
  
  g_return_val_if_fail(filename != 0, 0);
  
  db = gtk_type_new(gda_xml_database_get_type());
  db->xmldoc = xmlParseFile(filename);
  if (db->xmldoc)
    {
      xmlNodePtr xmlnode;
      
      db->filename = g_strdup(filename);
      
      /* add all objects to list of nodes */
      xmlnode = db->xmldoc->root->childs;
      while (xmlnode != 0)
        {
          Gda_XmlObject* obj;
          
          obj = g_new0(Gda_XmlObject, 1);
          obj->db = db;
          obj->xmlnode = xmlnode;
          obj->parent = 0;
          obj->children = get_node_children(obj);

          db->objects = g_list_append(db->objects, (gpointer) obj);
          xmlnode = xmlnode->next; /* next item in list */
        }
      return (db);
    }
    
  /* free memory on error */
  gda_xml_database_free(db);
  return (0);
}

/**
 * gda_xml_database_free
 * @db: a @Gda_XmlDatabase object
 *
 * Cleans up all memory associated with the XML representation of the
 * given database
 */
void
gda_xml_database_free (Gda_XmlDatabase *db)
{
  GList* node;

  g_return_if_fail(IS_GDA_XML_DATABASE(db));
  
  while ((node = g_list_first(db->objects)))
    {
      Gda_XmlObject* obj = (Gda_XmlObject *) node->data;
      db->objects = g_list_remove(db->objects, (gpointer) obj);
      gda_xml_database_delete_object(db, obj);
    }

  if (db->xmldoc) xmlFreeDoc(db->xmldoc);
  if (db->filename) g_free((gpointer) db->filename);
  g_free((gpointer) db);
}

/**
 * gda_xml_database_save
 * @db: a @Gda_XmlDatabase object
 * @filename: file name
 *
 * Saves the given XML database to a disk file. You may specify NULL for
 * the @filename parameter, in which case the current file name for the
 * database is used
 *
 * Returns: TRUE if successful, FALSE on error
 */
gboolean
gda_xml_database_save (Gda_XmlDatabase *db, const gchar *filename)
{
  gchar* filename_to_use = 0;

  g_return_val_if_fail(IS_GDA_XML_DATABASE(db), FALSE);
  g_return_val_if_fail(db->xmldoc != 0, FALSE);

  if (filename) filename_to_use = filename;
  else if (!(filename_to_use = gda_xml_database_get_filename(db)))
    {
      g_warning("no file name given for this database");
      return (FALSE);
    }

  return (xmlSaveFile(filename_to_use, db->xmldoc));
}

/**
 * gda_xml_database_changed
 * @db: a @Gda_XmlDatabase object
 *
 * Emits the "changed" signal for the given XML database. Applications do not need
 * to call this function, since the library emits any needed signal for the
 * Gda_XmlDatabase object.
 */
void
gda_xml_database_changed (Gda_XmlDatabase *db)
{
  g_return_if_fail(IS_GDA_XML_DATABASE(db));
  gtk_signal_emit(GTK_OBJECT(db),
                  gda_xml_database_signals[GDA_XML_DATABASE_CHANGED]);
}

/**
 * gda_xml_database_get_name
 * @db: a @Gda_XmlDatabase object
 *
 * Returns the name of the database
 */
const gchar *
gda_xml_database_get_name (Gda_XmlDatabase *db)
{
  g_return_val_if_fail(IS_GDA_XML_DATABASE(db), 0);
  g_return_val_if_fail(db->xmldoc != 0, 0);
  
  return ((const gchar *) xmlGetProp(db->xmldoc->root, "NAME"));
}

/**
 * gda_xml_database_set_name
 * @db: a @Gda_XmlDatabase object
 * @name: new name for the database
 *
 * Sets the name for the given database
 */
void
gda_xml_database_set_name (Gda_XmlDatabase *db, const gchar *name)
{
  g_return_if_fail(IS_GDA_XML_DATABASE(db));
  g_return_if_fail(name != 0);
  
  xmlSetProp(db->xmldoc->root, GDA_XML_PROPERTY_NAME, name);
  gda_xml_database_changed(db);
}

/**
 * gda_xml_database_get_date
 * @db: a @Gda_XmlDatabase object
 *
 * Returns a string representation of the last modification date. This date
 * is updated each time the XML tree is modified
 */
const gchar *
gda_xml_database_get_date (Gda_XmlDatabase *db)
{
  g_return_val_if_fail(IS_GDA_XML_DATABASE(db), 0);
  return ((const gchar *) xmlGetProp(db->xmldoc->root, "DATE"));
}

/**
 * gda_xml_database_set_date
 * @db: a @Gda_XmlDatabase object
 * @dt: date string
 *
 * Sets the modification time of the given XML database. The format used for this
 * date is up to the programmer, as no further operations are done on this date.
 * Anyway, if you want to follow the same rules as the library, use the format as
 * returned by strftime()
 */
void
gda_xml_database_set_date (Gda_XmlDatabase *db, const gchar *dt)
{
  g_return_if_fail(IS_GDA_XML_DATABASE(db));
  g_return_if_fail(dt != 0);
  xmlSetProp(db->xmldoc->root, "DATE", dt);
}

/**
 * gda_xml_database_get_filename
 * db: a @Gda_XmlDatabase object
 *
 * Returns the filename associated with the given XML database representation.
 * This filename will then be used when calling @gda_xml_database_save
 */
const gchar *
gda_xml_database_get_filename (Gda_XmlDatabase *db)
{
  g_return_val_if_fail(IS_GDA_XML_DATABASE(db), 0);
  return (db->filename);
}

/**
 * gda_xml_database_set_filename
 * @db: a @Gda_XmlDatabase object
 *
 * Associates a file name with the given XML database representation
 */
void
gda_xml_database_set_filename (Gda_XmlDatabase *db, const gchar *filename)
{
  g_return_if_fail(IS_GDA_XML_DATABASE(db));
  
  if (db->filename) g_free((gpointer) db->filename);
  
  if (filename) db->filename = g_strdup(filename);
  else db->filename = 0;
}

/**
 * gda_xml_database_add_object
 * @db: a @Gda_XmlDatabase object
 * @name: name of the object
 * @type: type of the object
 *
 * Adds an object of the given type. It is better for application programmers to
 * call the specific-object creation functions (@gda_xml_table_new, 
 * @gda_xml_view_new, ...), than calling this function directly. In any case, this
 * function is public to allow you the choice
 */
Gda_XmlObject *
gda_xml_database_add_object (Gda_XmlDatabase *db, const gchar *name, const gchar *type)
{
  Gda_XmlObject* obj;

  /* first look for an existing object with the given name and type */
  obj = find_object_by_name_and_type(db, name, type);
  if (obj)
    {
      g_warning("found existing object %s of type %s", name, type);
    }
  else
    {
      /* create the object */
      obj = g_new0(Gda_XmlObject, 1);
      obj->db = db;
      obj->xmlnode = xmlNewChild(db->xmldoc->root, 0, name, 0);
      if (!obj->xmlnode)
        {
          g_warning("XML error!");
          g_free((gpointer) obj);
          return (0);
        }
      obj->parent = 0;
      obj->children = 0;
      db->objects = g_list_append(db->objects, (gpointer) obj);
      gda_xml_database_changed(db);
    }
  return (obj);
}

/**
 * gda_xml_database_delete_object
 * @db: a @Gda_XmlDatabase object
 * @obj: the object to be deleted
 *
 * Deletes the given object from the XML database
 */
void
gda_xml_database_delete_object (Gda_XmlDatabase *db, Gda_XmlObject *obj)
{
  GList* node;

  g_return_if_fail(IS_GDA_XML_DATABASE(db));
  g_return_if_fail(obj != 0);

  node = g_list_first(db->objects);
  while (node != 0)
    {
      if (obj == (Gda_XmlObject *) node->data)
        {
          GList* ptr;

          /* remove all children */
          while ((ptr = g_list_first(obj->children)))
            {
              Gda_XmlObject* child = (Gda_XmlObject *) ptr->data;
              obj->children = g_list_remove(obj->children, (gpointer) child);
              gda_xml_object_delete_child(obj, child);
            }

          /* remove this object */
          db->objects = g_list_remove(db->objects, (gpointer) obj);
          xmlUnlinkNode(obj->xmlnode);
          xmlFreeNode(obj->xmlnode);
          g_free((gpointer) obj);
          gda_xml_database_changed(db);
          return;
        }
      node = g_list_next(node);
    }
  g_warning("unknown object in database %s", gda_xml_database_get_name(db));
}
  
/**
 * gda_xml_database_find_object
 * @db: a @Gda_XmlDatabase object
 * @name: name of the object
 * @type: type of the object
 *
 * Searches for a given object of a given type in the XML database specified.
 *
 * Returns: a pointer to the object if found, or NULL
 */
Gda_XmlObject *
gda_xml_database_find_object (Gda_XmlDatabase *db, const gchar *name, const gchar *type)
{
  g_return_val_if_fail(IS_GDA_XML_DATABASE(db), 0);
  g_return_val_if_fail(name != 0, 0);
  g_return_val_if_fail(type != 0, 0);
  return (find_object_by_name_and_type(db, name, type));
}

/**
 * gda_xml_database_enum_objects
 * @db: a @Gda_XmlDatabase object
 * @func: user function
 * @type: object type to be enumerated
 * @data: user data
 *
 * This function calls the user-provided function for each object contained
 * in the given XML database representation. You can specify which type of
 * objects you want to get (by using the GDA_XML_OBJECT_TYPE_* #defines) or
 * all objects of any type (by supplying NULL as the @type parameter.
 *
 * Returns: TRUE if successful, FALSE on error
 */
gboolean
gda_xml_database_enum_objects (Gda_XmlDatabase *db,
                               Gda_XmlEnumFunc func,
                               const gchar *type,
                               gpointer user_data)
{
  GList* node;
  
  g_return_val_if_fail(IS_GDA_XML_DATABASE(db), FALSE);
  g_return_val_if_fail(func != 0, FALSE);
  
  /* traverse the list of objects */
  node = g_list_first(db->objects);
  while (node != 0)
    {
      Gda_XmlObject* obj = (Gda_XmlObject *) node->data;
      if (obj != 0 && obj->xmlnode != 0)
        {
          if (!type) (*func)(db, obj, user_data);
          else if (type != 0 && !g_strcasecmp(type, gda_xml_object_type(obj)))
            (*func)(db, obj, user_data);
        }
      node = g_list_next(node);
    }
  return (TRUE);
}

/**
 * gda_xml_object_add_child
 * @obj: a #Gda_XmlObject
 * @type: type of the child
 *
 * Adds a new child to the given #Gda_XmlObject. This function is mainly
 * for library internal use, you should better use the specific object
 * management functions
 */
Gda_XmlObject *
gda_xml_object_add_child (Gda_XmlObject *obj, const gchar *type)
{
  Gda_XmlObject* child;

  g_return_val_if_fail(obj, 0);
  g_return_val_if_fail(obj->xmlnode, 0);
  g_return_val_if_fail(type, 0);

  child = g_new0(Gda_XmlObject, 1);
  child->parent = obj;
  child->children = 0;
  child->xmlnode = xmlNewChild(obj->xmlnode, 0, type, 0);
  if (!child->xmlnode)
    {
      g_free((gpointer) obj);
      return (0);
    }

  obj->children = g_list_append(obj->children, (gpointer) obj);
  gda_xml_database_changed(obj->db);
  return (obj);
}

/**
 * gda_xml_object_delete_child
 * @obj: a #Gda_XmlObject
 * @child: child object
 *
 * Removes the given #Gda_XmlObject child, freeing all associated
 * memory. This function is recursive in that, it calls itself to
 * remove the child object's children.
 *
 * Again, you shouldn't use this function, but the specific
 * object management functions instead
 */
void
gda_xml_object_delete_child (Gda_XmlObject *obj, Gda_XmlObject *child)
{
  GList* node;

  g_return_if_fail(obj != NULL);
  g_return_if_fail(child != NULL);

  while ((node = g_list_first(child->children)))
    {
      child->children = g_list_remove(child->children, node);
      gda_xml_object_delete_child(child, (Gda_XmlObject *) node);
    }
  obj->children = g_list_remove(obj->children, (gpointer) child);
  xmlUnlinkNode(child->xmlnode);
  xmlFreeNode(child->xmlnode);
  g_free((gpointer) child);
  gda_xml_database_changed(obj->db);
}

/**
 * gda_xml_object_enum_children
 * @obj: a #Gda_XmlObject
 * @func: callback function
 * @type: type of the children
 * @user_data: data to pass to the callback
 *
 * This function enumerates all the children (or the children of @type, if not NULL)
 * by calling the given callback function for each child object found
 *
 * Returns: TRUE if successful, FALSE on error
 */
gboolean
gda_xml_object_enum_children (Gda_XmlObject *obj,
                              Gda_XmlEnumFunc func,
                              const gchar *type,
                              gpointer user_data)
{
  xmlNodePtr xmlnode;
  GList*     node;

  g_return_val_if_fail(obj != 0, FALSE);
  g_return_val_if_fail(IS_GDA_XML_DATABASE(obj->db), FALSE);
  g_return_val_if_fail(func != 0, FALSE);

  node = g_list_first(obj->children);
  while (node != 0)
    {
      Gda_XmlObject* child = (Gda_XmlObject *) node->data;
      if (child != 0)
        {
          if (!type) (*func)(obj->db, child, user_data);
          else if (type != 0 && !g_strcasecmp(type, gda_xml_object_type(obj)))
            (*func)(obj->db, child, user_data);
        }
      node = g_list_next(node);
    }
  return TRUE;
}

/**
 * gda_xml_object_get_property
 * @obj: a #Gda_XmlObject
 * @prop: property name
 *
 * Returns the value for the given property on the specified
 * object
 */
const gchar *
gda_xml_object_get_property (Gda_XmlObject *obj, const gchar *prop)
{
  g_return_val_if_fail(obj != NULL, NULL);
  g_return_val_if_fail(prop != NULL, NULL);

  return (xmlGetProp(obj->xmlnode, prop));
}

/**
 * gda_xml_object_set_property
 * @obj: a #Gda_XmlObject
 * @prop: property name
 * @value: value to be set
 */
void
gda_xml_object_set_property (Gda_XmlObject *obj, const gchar *prop, const gchar *value)
{
  g_return_if_fail(obj != NULL);
  g_return_if_fail(prop != NULL);
  g_return_if_fail(value != NULL);

  xmlSetProp(obj->xmlnode, prop, value);
  gda_xml_database_changed(obj->db);
}

/**
 * gda_xml_object_enum_properties
 * @obj: a #Gda_XmlObject
 * @func: callback function
 * @user_data: data to be passed to @callback
 *
 * Enums the properties defined for the given object by calling a user-supplied
 * callback function
 *
 * Returns: TRUE if success, FALSE on error
 */
gboolean
gda_xml_object_enum_properties (Gda_XmlObject *obj, Gda_XmlEnumPropFunc func, gpointer user_data)
{
  xmlAttrPtr attr;
  gboolean   res = TRUE;

  g_return_val_if_fail(obj != 0, FALSE);
  g_return_val_if_fail(obj->xmlnode != 0, FALSE);
  g_return_val_if_fail(func != 0, FALSE);

  attr = obj->xmlnode->properties;
  while (attr)
    {
      res = (*func)(obj, (const gchar *) attr->name, user_data);
      if (!res) break;
      attr = attr->next;
    }
  return (res);
}

/**
 * gda_xml_object_type
 * @obj: a #Gda_XmlObject
 *
 * Returns the type of the given object
 */
const gchar *
gda_xml_object_type (Gda_XmlObject *obj)
{
  g_return_val_if_fail(obj != 0, 0);
  g_return_val_if_fail(obj->xmlnode != 0, 0);
  return (obj->xmlnode->name);
}

/**
 * gda_xml_table_new
 * @db: a @Gda_XmlDatabase object
 * @name: name of the table to be created
 *
 * Adds a new table to the given XML database representation. This function
 * just creates the XML node. If you want to set properties for it, you should
 * call the gda_xml_table_set_* functions on the @Gda_XmlTable object returned
 * by this function
 */
Gda_XmlObject *
gda_xml_table_new (Gda_XmlDatabase *db, const gchar *name)
{
  g_return_val_if_fail(IS_GDA_XML_DATABASE(db), 0);
  g_return_val_if_fail(name != 0, 0);
  return (gda_xml_database_add_object(db, name, GDA_XML_OBJECT_TYPE_TABLE));
}

/**
 * gda_xml_table_new_from_gda_recordset
 * @db: a @Gda_XmlDatabase object
 * @name: name of the table to be created
 * @recset: an existing #Gda_Recordset object
 *
 * Adds a new table to the given XML database from an existing
 * #Gda_Recordset object. This function adds a <FIELD> tag for each
 * field in the recordset, and a <ROW> tag for each row fetched
 * from the recordset
 */
Gda_XmlObject *
gda_xml_table_new_from_gda_recordset (Gda_XmlDatabase *db,
                                      const gchar *name,
                                      Gda_Recordset *recset,
                                      gboolean export_data)
{
  gulong         position;
  Gda_XmlObject* table;

  g_return_val_if_fail(IS_GDA_XML_DATABASE(db), 0);
  g_return_val_if_fail(name, 0);
  g_return_val_if_fail(IS_GDA_RECORDSET(recset), 0);

  /* create the XML object */
  table = gda_xml_database_add_object(db, name, GDA_XML_OBJECT_TYPE_TABLE);
  if (table)
    {
      gboolean first_time = TRUE;

      position = gda_recordset_move_first(recset);
      while (position != GDA_RECORDSET_INVALID_POSITION && !gda_recordset_eof(recset))
        {
          gint cnt;

          for (cnt = 0; cnt < gda_recordset_rowsize(recset); cnt++)
            {
              Gda_XmlObject* obj;
              Gda_Field*     f = gda_recordset_field_idx(recset, cnt);

              /* only first time, add field descriptions */
              if (first_time)
                {
                  obj = gda_xml_object_add_child(table, GDA_XML_OBJECT_TYPE_FIELD);
                  gda_xml_object_set_property(obj,
                                              GDA_XML_PROPERTY_NAME,
                                              gda_field_name(f));
                  gda_xml_object_set_property(obj,
                                             GDA_XML_PROPERTY_SIZE,
                                             gda_field_defined_size(f));
                }

              /* if !export_data, we've finished */
              if (!export_data) return (table);
            }
          position = gda_recordset_move(recset, 1, 0);
          first_time = FALSE;
        }
      return (table);
    }
  return (0);
}

/**
 * gda_xml_table_delete
 * @db: a @Gda_XmlDatabase object
 * @table: table to be deleted
 */
void
gda_xml_table_delete (Gda_XmlDatabase *db, Gda_XmlObject *table)
{
  g_return_if_fail(IS_GDA_XML_DATABASE(db));
  g_return_if_fail(table != 0);
  gda_xml_database_delete_object(db, table);
}

/**
 * gda_xml_table_delete_by_name
 * @db: a @Gda_XmlDatabase object
 * @name: table name
 */
void
gda_xml_table_delete_by_name (Gda_XmlDatabase *db, const gchar *name)
{
  Gda_XmlObject* table;

  g_return_if_fail(IS_GDA_XML_DATABASE(db));
  g_return_if_fail(name != 0);

  table = find_object_by_name_and_type(db, name, GDA_XML_OBJECT_TYPE_TABLE);
  if (table) gda_xml_database_delete_object(db, table);
  else g_warning("table %s not found in database", name);
}

/**
 * gda_xml_table_get_name
 * @table: table object
 *
 * Returns the name of the given table
 */
const gchar *
gda_xml_table_get_name (Gda_XmlObject *table)
{
  g_return_val_if_fail(table != 0, 0);
  return ((const gchar *) xmlGetProp(table->xmlnode, GDA_XML_PROPERTY_NAME));
}

/**
 * gda_xml_table_set_name
 * @table: table object
 * @name: table name
 *
 * Sets the name of the given table
 */
void
gda_xml_table_set_name (Gda_XmlObject *table, const gchar *name)
{
  g_return_if_fail(table != 0);
  g_return_if_fail(name != 0);

  xmlSetProp(table->xmlnode, GDA_XML_PROPERTY_NAME, name);
  gda_xml_database_changed(table->db);
}
