/* GNOME DB libary
 * Copyright (C) 1998,1999 Michael Lausch
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gnome.h>
#include "gda-server.h"

/**
 * gda_server_new:
 *
 * Allocates memory for a new Gda_Server object and initializes struct 
 * members.
 *
 * Returns: a pointer to a new Gda_Server object.
 */
Gda_Server *
gda_server_new (void)
{
  Gda_Server* retval;

  retval = g_new0(Gda_Server, 1);
  return retval;
}

/**
 * gda_server_copy:
 *
 * @server: the server to be copies.
 *
 * Make a deep copy of all the data needed to describe a gGDA server.
 *
 * Returns: a pointer to the newly allocated server object
 */
Gda_Server *
gda_server_copy (Gda_Server* server)
{
  Gda_Server* retval;

  retval = gda_server_new();
  
  if (server->name)
    retval->name = g_strdup(server->name);

  if (server->comment)
    retval->comment = g_strdup(server->comment);

  if (server->location)
    retval->location = g_strdup(server->location);
  if (server->repo_id)
    retval->repo_id = g_strdup(server->repo_id);
  
  return retval;
}

/**
 * gda_server_free:
 *
 * @server: the server to de-allocate.
 *
 * Frees the memory allocated with gda_server_new() and the memory
 * allocated to struct members.
 */
void
gda_server_free (Gda_Server* server)
{
  if (server->name)
    g_free(server->name);

  if (server->comment)
    g_free(server->comment);

  if (server->location)
    g_free(server->location);

  g_free(server);
}

/**
 * gda_server_list:
 *
 * Searches the GNORBA database for GDA servers and returns a Glist of 
 * Gda_Server structs. 
 *
 * Returns: a GList of GDA servers structs
 */
GList *
gda_server_list (void)
{
  GList*          retval = 0;
  GoadServerList* servlist;
  GoadServer*     slist;
  gint            i;
  Gda_Server*     server;
  
  servlist = goad_server_list_get();
  slist = servlist->list;
  for (i = 0; slist[i].repo_id; i++)
    {
      if (strcmp(*slist[i].repo_id, "IDL:GDA/ConnectionFactory:1.0")  == 0)
	{
	  fprintf(stderr,"Found server '%s'\n", slist[i].server_id);
	  server = gda_server_new();
	  server->name     = g_strdup(slist[i].server_id);
	  server->location = g_strdup(slist[i].location_info);
	  server->comment  = g_strdup(slist[i].description);
	  server->repo_id  = g_strdup(*(slist[i].repo_id));
	  server->type     = slist[i].type;
	  retval = g_list_append(retval, server);
	}
    }
  goad_server_list_free(servlist);
  return retval;
}

/**
 * gda_server_free_list
 * @list: list of #Gda_Server structures
 *
 * Frees a list of #Gda_Server structures previously returned by
 * a call to #gda_server_list
 */
void
gda_server_free_list (GList *list)
{
  GList* node;

  g_return_if_fail(list);

  while ((node = g_list_first(list)))
    {
      Gda_Server* server = (Gda_Server *) node->data;
      list = g_list_remove(list, (gpointer) server);
      gda_server_free(server);
    }
}

/**
 * gda_server_find_by_name
 * @provider: provider name
 *
 * Returns a #Gda_Server structure describing the given provider. This function
 * searches all the servers present on your system
 * and tries to find the specified provider.
 *
 * Returns: a pointer to the server structure, or NULL on error
 */
Gda_Server *
gda_server_find_by_name (const gchar *provider)
{
  GList*      list;
  GList*      node;
  Gda_Server* server = NULL;

  g_return_val_if_fail(provider, NULL);

  list = gda_server_list();
  node = g_list_first(list);
  while (node)
    {
      if (!strcmp(provider, GDA_SERVER_NAME((Gda_Server *) node->data)))
        {
          server = gda_server_copy((Gda_Server *) node->data);
          break;
        }
      node = g_list_next(node);
    }
  gda_server_free_list(list);
  return (server);
}

/**
 * gda_list_datasources:
 *
 * Lists all datasources configured on the system.
 *
 * Returns a GList with the names of all data sources configured.
 */
GList *
gda_list_datasources (void)
{
  gpointer gda_iterator;
  GList*   datasources = 0;
  gchar*   gda_name;
  gchar*   section_name;
  
  gda_iterator = gnome_config_init_iterator("/gdalib/Datasources");
  while ((gda_iterator = gnome_config_iterator_next(gda_iterator, &gda_name, &section_name)))
    {
      datasources = g_list_append(datasources, gda_name);
    }
  return datasources;
}

/**
 * gda_list_datasources_for_provider:
 * @provider: the provider which should be used to look for datasources
 *
 * Returns: a GList of all datasources available to a specific @provider.
 */
GList *
gda_list_datasources_for_provider (gchar* provider)
{
  gpointer gda_iterator;
  GList*   datasources = 0;
  gchar*   gda_name;
  gchar*   section_name;
  gchar*   dsn;
  GString* path;
  gchar*   provider_name;
  
  path = g_string_new("");
  gda_iterator = gnome_config_init_iterator("/gdalib/Datasources");
  while ((gda_iterator = gnome_config_iterator_next(gda_iterator, &gda_name, &section_name)))
    {
      
      g_string_sprintf(path, "/gdalib/%s/", gda_name);
      gnome_config_push_prefix(path->str);
      
      provider_name = gnome_config_get_string("Provider");
      if (strcmp(provider_name, provider) == 0)
        {
          dsn = gnome_config_get_string("DSN");
          datasources = g_list_append(datasources, dsn);
        }
    }
  return datasources;
}
