/* GNOME-DB - GUI front-end
 * Copyright (c) 1998-2000 by Rodrigo Moya
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gdafe.h"

static void clear_sql          (GtkWidget *w, gpointer data);
static void cut_sql            (GtkWidget *w, gpointer data);
static void copy_sql           (GtkWidget *w, gpointer data);
static void grid_loaded        (GnomeDbGrid *grid);
static void load_sql           (GtkWidget *w, gpointer data);
static void paste_sql          (GtkWidget *w, gpointer data);
static void real_disconnect    (GtkWidget *w, gpointer data);
static void refresh_grid       (GtkWidget *w, gpointer data);
static void run_script         (GtkWidget *w, gpointer data);
static void run_sql            (GtkWidget *w, gpointer data);
static void save_sql           (GtkWidget *w, gpointer data);
static void select_row         (GnomeDbGrid *grid, gint row, gpointer data);
static void select_sql_command (GtkWidget *w, gpointer data);

GnomeUIInfo sqltoolbar[] =
{
  { GNOME_APP_UI_ITEM, N_("Run"), N_("Execute SQL command"),
    run_sql, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_EXEC, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Run Script"), N_("Execute buffer as SQL script"),
    run_script, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_EXEC, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, "SQL", N_("Select SQL Command"),
    select_sql_command, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_UP, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Refresh"), N_("Refresh grid"),
    refresh_grid, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_REFRESH, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Clear"), N_("Clear SQL window"),
    clear_sql, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_TRASH, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Print"), N_("Print grid"),
    NULL, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_PRINT, 0, 0, NULL },
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("Open"), N_("Load file into buffer"),
    load_sql, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_OPEN, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Save"), N_("Save current buffer"),
    save_sql, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_SAVE, 0, 0, NULL },
    GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("Cut"), N_("Cut selected text to clipboard"),
    cut_sql, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_CUT, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Copy"), N_("Copy selected text to clipboard"),
    copy_sql, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_COPY, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Paste"), N_("Paste text from clipboard"),
    paste_sql, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_PASTE, 0, 0, NULL },
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("Disconnect"), N_("Close current connection"),
    fe_disconnect, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_STOP, 0, 0, NULL },
  GNOMEUIINFO_END
};

/*
 * Private functions
 */
static void
clear_sql (GtkWidget *w, gpointer data)
{
  FE_ConnectionInfo *cnc_info = fe_get_current_connection();
  if (cnc_info != NULL)
    {
      gtk_editable_delete_text(GTK_EDITABLE(cnc_info->sql_window), 0,
                               gtk_text_get_length(GTK_TEXT(cnc_info->sql_window)));
      gnome_db_grid_clear(GNOME_DB_GRID(cnc_info->grid));
    }
}

static void
cut_sql (GtkWidget *w, gpointer data)
{
  FE_ConnectionInfo *cnc_info = fe_get_current_connection();
  if (cnc_info != NULL)
    {
      gtk_editable_cut_clipboard(GTK_EDITABLE(cnc_info->sql_window));
    }
}

static void
copy_sql (GtkWidget *w, gpointer data)
{
  FE_ConnectionInfo *cnc_info = fe_get_current_connection();
  if (cnc_info != NULL)
    {
      gtk_editable_copy_clipboard(GTK_EDITABLE(cnc_info->sql_window));
    }
}

static void
grid_loaded (GnomeDbGrid *grid)
{
  g_return_if_fail(GNOME_DB_IS_GRID(grid));
  fe_status_bar_message(_("%d rows"), gnome_db_grid_get_row_count(grid));
}

static void
load_sql (GtkWidget *w, gpointer data)
{
  FE_ConnectionInfo *cnc_info = fe_get_current_connection();
  if (cnc_info != NULL)
    {
      gchar *file_name = fe_get_file_name(_("Select SQL Script"));
      if (file_name != NULL)
        {
          fe_load_file_in_text(GTK_TEXT(cnc_info->sql_window), file_name);
          g_free((gchar *) file_name);
        }
    }
}

static void
paste_sql (GtkWidget *w, gpointer data)
{
  FE_ConnectionInfo *cnc_info = fe_get_current_connection();
  if (cnc_info != NULL)
    {
      gtk_editable_paste_clipboard(GTK_EDITABLE(cnc_info->sql_window));
    }
}

static void
real_disconnect (GtkWidget *view, gpointer data)
{
  FE_ConnectionInfo *cnc_info = (FE_ConnectionInfo *) data;
  g_print("real_disconnect() called!\n");
  if (cnc_info != NULL)
    {
      /* close all related windows */

      /* disconnect from database */
      gda_connection_close(cnc_info->cnc);
      gda_connection_free(cnc_info->cnc);
      
      if (cnc_info->dsn != NULL) g_free((gpointer) cnc_info->dsn);
      if (cnc_info->trace_file != NULL) g_free((gpointer) cnc_info->trace_file);
      g_free((gpointer) cnc_info);
    }
}

static void
refresh_grid (GtkWidget *w, gpointer data)
{
  FE_ConnectionInfo *cnc_info = fe_get_current_connection();
  if (cnc_info != NULL)
    {
      gnome_db_grid_refresh(GNOME_DB_GRID(cnc_info->grid));
    }
}

static void
run_script (GtkWidget *w, gpointer data)
{
  FE_ConnectionInfo *cnc_info = fe_get_current_connection();
  if (cnc_info != NULL)
    {
      gchar *sql = gtk_editable_get_chars(GTK_EDITABLE(cnc_info->sql_window),
                             0,
                             gtk_text_get_length(GTK_TEXT(cnc_info->sql_window)));
      if (sql)
        {
          Gda_Command* cmd;
          gchar*       tmp;
          
          /* create command object */
          cmd = gda_command_new();
          gda_command_set_connection(cmd, cnc_info->cnc);
          
          tmp = strtok(sql, ";");
          while (tmp)
            {
              Gda_Recordset* recset;
              gulong         count;
              
              /* execute command */
              gda_command_set_text(cmd, tmp);
              fe_status_bar_message(_("Executing '%s'"), tmp);
              recset = gda_command_execute(cmd, &count, 0);
              if (recset)
                {
                  fe_status_bar_message(_("Command succeeded"));
                  gda_recordset_free(recset);
                }
              else fe_status_bar_message(_("Command failed"));
              
              fe_add_recent_command(tmp);
              tmp = strtok(NULL, ";");
            }
          gda_command_free(cmd);
        }
    }
}

static void
run_sql (GtkWidget *w, gpointer data)
{
  FE_ConnectionInfo *cnc_info = fe_get_current_connection();
  if (cnc_info != NULL)
    {
      gchar *sql = gtk_editable_get_chars(GTK_EDITABLE(cnc_info->sql_window),
                             0,
                             gtk_text_get_length(GTK_TEXT(cnc_info->sql_window)));
      if (sql)
        {
          gulong reccount;
          Gda_Recordset *rs = gda_connection_execute(cnc_info->cnc, sql,
                                                     &reccount, 0);
          if (rs != 0)
            {
              fe_status_bar_message(_("Command succeeded"));
              gnome_db_grid_set_recordset(GNOME_DB_GRID(cnc_info->grid), rs);
            }
          fe_add_recent_command(sql);
        }
    }
}

static void
save_sql (GtkWidget *w, gpointer data)
{
  FE_ConnectionInfo *cnc_info = fe_get_current_connection();
  if (cnc_info != NULL)
    {
      gchar *file_name = fe_get_file_name(_("Save SQL Script"));
      if (file_name != NULL)
        {
          fe_save_text_to_file(GTK_TEXT(cnc_info->sql_window), file_name);
          g_free((gpointer) file_name);
        }
    }
}

static void
select_row (GnomeDbGrid *grid, gint row, gpointer data)
{
  fe_status_bar_message(_("Row %d - %d rows selected"), row,
                        gnome_db_grid_get_selected_rows(grid));
}

static void
select_sql_command (GtkWidget *w, gpointer data)
{
  GtkWidget *dialog, *scrolled_window, *list;
  gint selected;
  FE_ConnectionInfo *cnc_info = fe_get_current_connection();
  if (cnc_info != NULL)
    {
      GList *cmds = fe_get_recent_commands();
      dialog = gnome_dialog_new(_("Select SQL Command"), GNOME_STOCK_BUTTON_OK,
                                GNOME_STOCK_BUTTON_CANCEL, NULL);
      gtk_widget_set_usize(dialog, 400, 400);
      gnome_dialog_set_default(GNOME_DIALOG(dialog), 0);
 
      /* create scrolled window to put list inside */
      scrolled_window = gtk_scrolled_window_new(NULL, NULL);
      gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window),
                                     GTK_POLICY_AUTOMATIC,
                                     GTK_POLICY_AUTOMATIC);
      gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), scrolled_window,
                         TRUE, TRUE, GNOME_PAD);
      gtk_widget_show(scrolled_window);

      /* create list of commands */
      list = gtk_list_new();
      if (cmds != NULL)
        fe_fill_gtk_list_with_glist(GTK_LIST(list), cmds);
      gtk_list_set_selection_mode(GTK_LIST(list), GTK_SELECTION_SINGLE);
      gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrolled_window), list);
      gtk_widget_show(list);

      selected = gnome_dialog_run(GNOME_DIALOG(dialog));
      if (selected == 0)	/* OK button */
        {
          gchar *sql = fe_get_list_current_string(GTK_LIST(list));
          /* set buffer text */
          gtk_editable_delete_text(GTK_EDITABLE(cnc_info->sql_window), 0,
                                   gtk_text_get_length(GTK_TEXT(cnc_info->sql_window)));
          gtk_text_freeze(GTK_TEXT(cnc_info->sql_window));
          gtk_text_insert(GTK_TEXT(cnc_info->sql_window), NULL, NULL, NULL,
                          sql, strlen(sql));
          gtk_text_thaw(GTK_TEXT(cnc_info->sql_window));
        }
      gnome_dialog_close(GNOME_DIALOG(dialog));
    }
}

/*
 * Public functions
 */
void
fe_disconnect (GtkWidget *w, gpointer data)
{
  FE_ConnectionInfo *cnc_info = fe_get_current_connection();
  if (cnc_info != 0)
    {
      fe_destroy_view(cnc_info->view_window);
    }
}

void
fe_disconnect_all (GtkWidget *w, gpointer data)
{
}

void
fe_open_sql_window (Gda_Connection *cnc, gchar *dsn)
{
  GtkWidget *scrolled_window;
  gchar *title;
  FE_ConnectionInfo *cnc_info;

  g_return_if_fail(cnc != NULL);
  g_return_if_fail(cnc->is_open);

  /* allocate memory for this window's context */
  cnc_info = (FE_ConnectionInfo *) g_new0(FE_ConnectionInfo, 1);
  
  cnc_info->cnc = cnc;
  cnc_info->dsn = dsn ? g_strdup(dsn) : NULL;
  cnc_info->trace_file = NULL;

  /* create SQL window */
  title = g_strdup_printf("SQL: %s@%s",gda_connection_get_user(cnc_info->cnc),
                          cnc_info->dsn);
  cnc_info->view_window = fe_new_view(title, 1, 2, sqltoolbar);
  g_free((gpointer) title);
  fe_set_view_destroy_func(cnc_info->view_window, (FE_DestroyViewFunc) real_disconnect,
                           (gpointer) cnc_info);

  /* create children */
  cnc_info->pane = gtk_vpaned_new();
  fe_add_widget_to_view(cnc_info->view_window, 0, 1, 1, 2, cnc_info->pane);

  scrolled_window = gda_ui_new_scrolled_window_widget();
  cnc_info->sql_window = fe_new_text_widget();
  gtk_container_add(GTK_CONTAINER(scrolled_window), cnc_info->sql_window);

  cnc_info->grid = fe_new_grid_widget(0);
  gtk_signal_connect(GTK_OBJECT(cnc_info->grid), "loaded",
                     GTK_SIGNAL_FUNC(grid_loaded), 0);
  gtk_signal_connect(GTK_OBJECT(cnc_info->grid), "select_row",
                     GTK_SIGNAL_FUNC(select_row), 0);
  gtk_signal_connect(GTK_OBJECT(cnc_info->grid), "unselect_row",
                     GTK_SIGNAL_FUNC(select_row), 0);

  gtk_paned_add1(GTK_PANED(cnc_info->pane), scrolled_window);
  gtk_paned_add2(GTK_PANED(cnc_info->pane), cnc_info->grid);
  fe_set_view_data(cnc_info->view_window, "FE_ConnectionInfo", 
                   (gpointer) cnc_info);
}
