/* GNOME DB
 * Copyright (C) 1999 Rodrigo Moya
 * Copyright (C) 1999 Stephan Heinze
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gda-oracle-types.h"
#include "gda-oracle-recset.h"
#include "gda-oracle-command.h"
#include "gda-oracle-connection.h"
#include "gda-oracle-error.h"

typedef Gda_ORACLE_Recordset* (*schema_ops_fn)(Gda_ORACLE_Error *,
                               Gda_ORACLE_Connection *,
                               GDA_Connection_Constraint *,
                               gint );

schema_ops_fn schema_ops[GDA_Connection_GDNC_SCHEMA_LAST] =
{
  0,
};

/*
 * Private functions
 */
void
add_error_to_connection (Gda_ORACLE_Connection *cnc)
{
  Gda_ORACLE_Error* o_error = NULL;

  g_return_if_fail(cnc != NULL);

  o_error = gda_oracle_error_new();
  gda_oracle_error_make( o_error, NULL, cnc, "");
}

static Gda_ORACLE_Recordset *
schema_columns (Gda_ORACLE_Error *e,
                Gda_ORACLE_Connection *cnc,
                GDA_Connection_Constraint *constraints,
                gint length)
{
  Gda_ORACLE_Command *cmd;
  
  /* create the command object */
  cmd = gda_oracle_cmd_new();
  cmd->cnc = cnc;
  cmd->cmd = g_strdup_printf("SELECT 'default' AS \"Catalog\", "
                             "       owner AS \"Schema\", "
                             "       table_name AS \"Table\", "
                             "       column_name AS \"Column\", "
                             "       data_type AS \"Type\", "
                             "       data_length AS \"Length\", "
                             "       data_precision AS \"Precision\", "
                             "       nullable AS \"nullable\" "
                             " FROM all_tab_columns "
                             " WHERE table_name = '%s' "
                             " ORDER BY column_name ",
                             g_strstrip(constraints->value)
                      );
   g_print(cmd->cmd);
                      
  return (gda_oracle_cmd_execute(cmd, e, 0, 0));
}

static Gda_ORACLE_Recordset *
schema_procedures (Gda_ORACLE_Error *e,
                   Gda_ORACLE_Connection *cnc,
                   GDA_Connection_Constraint *constraints,
                   gint length)
{
  return (0);
}

static Gda_ORACLE_Recordset *
schema_tables (Gda_ORACLE_Error *e,
               Gda_ORACLE_Connection *cnc,
               GDA_Connection_Constraint *constraints,
               gint length)
{
  Gda_ORACLE_Command *cmd;
  
  /* create the command object */
  cmd = gda_oracle_cmd_new();
  cmd->cnc = cnc;
  cmd->cmd = g_strdup("SELECT a.table_name, b.owner, a.table_name, 'TABLE', d.comments"
                      " FROM all_tables b, user_tables a, dictionary d"
                      " WHERE a.table_name = b.table_name"
                      "   AND d.table_name (+)= a.table_name"
                      );
                      
  return (gda_oracle_cmd_execute(cmd, e, 0, 0));
}

static Gda_ORACLE_Recordset *
schema_views (Gda_ORACLE_Error *e,
              Gda_ORACLE_Connection *cnc,
              GDA_Connection_Constraint *constraints,
              gint length)
{
  Gda_ORACLE_Command *cmd;
  
  /* create the command object */
  cmd = gda_oracle_cmd_new();
  cmd->cnc = cnc;
  cmd->cmd = g_strdup("SELECT a.view_name, b.owner, a.view_name, 'VIEW', NULL as \"comments\" "
                      " from user_views a, all_views b "
                      " where a.view_name = b.view_name "
                     );
                      
  return (gda_oracle_cmd_execute(cmd, e, 0, 0));
}

static void
initialize_schema_ops (void)
{
  schema_ops[GDCN_SCHEMA_TABLES] = schema_tables;
  schema_ops[GDCN_SCHEMA_COLS] = schema_columns;
  schema_ops[GDCN_SCHEMA_PROCS] = schema_procedures;
  schema_ops[GDCN_SCHEMA_VIEWS] = schema_views;
}

/*
 * Public functions
 */
Gda_ORACLE_Connection *
gda_oracle_connection_new (void)
{
  static gint initialized = 0;
  Gda_ORACLE_Connection *cnc;
  if (!initialized)
    {
      /* initialize OCI library */
      if (OCI_SUCCESS !=  OCIInitialize((ub4) OCI_THREADED | OCI_OBJECT,
                                        (dvoid*) 0,
                                        (dvoid*(*)(dvoid*, size_t)) 0,
                                        (dvoid* (*)(dvoid*, dvoid*, size_t)) 0,
                                        (void (*)(dvoid*, dvoid*))0))
        {
          g_error("error in OCIInitialize()");
          return (0);
        }
      /* initialize schema functions */
      initialize_schema_ops();
    }

  /* create Gda_ORACLE_Connection object */
  cnc = g_new0(Gda_ORACLE_Connection, 1);
  
  if (OCI_SUCCESS == OCIEnvInit((OCIEnv **) &cnc->henv, OCI_DEFAULT, (size_t) 0,
                                (dvoid **) 0))
  {
    if (OCI_SUCCESS == OCIHandleAlloc((dvoid *) cnc->henv,
                                      (dvoid **) &cnc->herr,
                                      OCI_HTYPE_ERROR,
                                      (size_t) 0,
                                      (dvoid **) 0))
    {
      /* we use the Multiple Sessions/Connections OCI paradigm for this server */
      if (OCI_SUCCESS == OCIHandleAlloc((dvoid *) cnc->henv,
                                        (dvoid **) &cnc->hserver,
                                        OCI_HTYPE_SERVER, (size_t) 0,
                                        (dvoid **) 0))
      {
        if (OCI_SUCCESS == OCIHandleAlloc((dvoid *) cnc->henv,
                                          (dvoid **) &cnc->hservice,
                                          OCI_HTYPE_SVCCTX, (size_t) 0,
                                          (dvoid **) 0))
        {
          return (cnc);
        }
        else
          g_print("error in OCIHandleAlloc() HTYPE_SVCCTX\n");
        OCIHandleFree((dvoid *) cnc->hserver, OCI_HTYPE_SERVER);
      }
      else
        g_print("error in OCIHandleAlloc() HTYPE_SERVER\n");
      OCIHandleFree((dvoid *) cnc->herr, OCI_HTYPE_ERROR);
    }
    else
      g_print("error in OCIHandleAlloc() HTYPE_ERROR\n");
    OCIHandleFree((dvoid *) cnc->henv, OCI_HTYPE_ENV);
  }
  else
    g_print("error in OCIEnvInit()");
  g_free((gpointer) cnc);
  return (0);
}

void
gda_oracle_connection_free (Gda_ORACLE_Connection *cnc)
{
  g_return_if_fail(cnc != NULL);
  if (0 == gda_oracle_connection_close(cnc))
    {
      if (0 != cnc->user)
        g_free((gpointer) cnc->user);
      g_free((gpointer) cnc);
    }
  else
    add_error_to_connection(cnc);
}

gint
gda_oracle_connection_close (Gda_ORACLE_Connection *cnc)
{
  g_return_val_if_fail(cnc != NULL, -1);

  if (OCI_SUCCESS == OCISessionEnd(cnc->hservice,
                                   cnc->herr,
                                   cnc->hsession,
                                   OCI_DEFAULT))
    {
      OCIHandleFree((dvoid *) cnc->hsession, OCI_HTYPE_SESSION);
      OCIServerDetach(cnc->hserver, cnc->herr, OCI_DEFAULT);
      OCIHandleFree((dvoid *) cnc->hservice, OCI_HTYPE_SVCCTX);
      OCIHandleFree((dvoid *) cnc->hserver, OCI_HTYPE_SERVER);
      OCIHandleFree((dvoid *) cnc->herr, OCI_HTYPE_ERROR);
      OCIHandleFree((dvoid *) cnc->henv, OCI_HTYPE_ENV);
      return (0);
    }
  add_error_to_connection(cnc);
  return (-1);
}

gint 
gda_oracle_connection_open(Gda_ORACLE_Connection *cnc, 
                           gchar *cnc_string, 
                           gchar* username, 
                           gchar* password)
{
  gchar *ptr, *database;
  
  g_return_val_if_fail(cnc != 0, -1);

  ptr = cnc_string;
  
  /* get database name */
  while (*ptr && *ptr != '=')
    ptr++;
  if (!*ptr)
    database = cnc_string;
  else
    database = ptr + 1;
  
  /* attach to server */
  if (OCI_SUCCESS == OCIServerAttach(cnc->hserver,
                                     cnc->herr, 
                                     (text *) database,
                                     strlen(database),
                                     OCI_DEFAULT))
    {
      OCIAttrSet((dvoid *) cnc->hservice, OCI_HTYPE_SVCCTX,
                 (dvoid *) cnc->hserver, (ub4) 0, OCI_ATTR_SERVER, cnc->herr);

      /* create the session handle */
      if (OCI_SUCCESS == OCIHandleAlloc((dvoid *) cnc->henv, 
                                        (dvoid **) &cnc->hsession,
                                        OCI_HTYPE_SESSION, 
                                        (size_t) 0, 
                                        (dvoid **) 0))
        {
          OCIAttrSet((dvoid *) cnc->hsession, OCI_HTYPE_SESSION, 
                     (dvoid *) username, (ub4) strlen(username),
                     OCI_ATTR_USERNAME, cnc->herr);
          OCIAttrSet((dvoid *) cnc->hsession, OCI_HTYPE_SESSION, 
                     (dvoid *) password, (ub4) strlen(password), 
                     OCI_ATTR_PASSWORD, cnc->herr);
       
          /* inititalize connection */
          if (OCI_SUCCESS == OCISessionBegin(cnc->hservice,
                                             cnc->herr,
                                             cnc->hsession,
                                             OCI_CRED_RDBMS,
                                             OCI_DEFAULT))
            {
              OCIAttrSet((dvoid *) cnc->hservice, OCI_HTYPE_SVCCTX,
                         (dvoid *) cnc->hsession, (ub4) 0, OCI_ATTR_SESSION,
                         cnc->herr);
                         
              cnc->user = g_strdup(username);
              return (0);
            }
          else
            {
              add_error_to_connection(cnc);
              OCIHandleFree((dvoid *) cnc->hsession, OCI_HTYPE_SESSION);
            }
        }
      else
          add_error_to_connection(cnc);
      OCIServerDetach(cnc->hserver, cnc->herr, OCI_DEFAULT);
    }
  else
      add_error_to_connection(cnc);
  return (-1);
}

Gda_ORACLE_Recordset *
gda_oracle_connection_open_schema (Gda_ORACLE_Connection *cnc,
                                   Gda_ORACLE_Error *e,
                                   GDA_Connection_QType t,
                                   GDA_Connection_Constraint *constraints,
                                   gint length)
{
  schema_ops_fn fn = schema_ops[(gint) t];

  if (fn)
    return fn(e, cnc, constraints, length);
  else
    g_log("GDA ORACLE", G_LOG_LEVEL_INFO,
	  "schema_open_schema: Unhandled SCHEMA_QTYPE %d\n", (gint) t);
  return 0;
}

gint
gda_oracle_connection_start_logging (Gda_ORACLE_Connection *cnc, gchar *file_name)
{
  g_return_val_if_fail(cnc != 0, -1);
  g_return_val_if_fail(file_name != 0, -1);
  
  return (0);
}

gint
gda_oracle_connection_stop_logging (Gda_ORACLE_Connection *cnc)
{
}

gint
gda_oracle_connection_begin_transaction (Gda_ORACLE_Connection *cnc)
{
  g_return_val_if_fail(cnc != NULL, -1);

  if (OCITransStart(cnc->hservice, cnc->herr, 0, OCI_TRANS_NEW) != OCI_SUCCESS)
    {
      add_error_to_connection(cnc);
      return (-1);
    }
  return (0);
}

gint
gda_oracle_connection_commit_transaction (Gda_ORACLE_Connection *cnc)
{
  g_return_val_if_fail(cnc != NULL, -1);

  if (OCITransCommit(cnc->hservice, cnc->herr, OCI_DEFAULT) != OCI_SUCCESS)
    {
      add_error_to_connection(cnc);
      return (-1);
    }
  return (0);
}

gint
gda_oracle_connection_rollback_transaction (Gda_ORACLE_Connection *cnc)
{
  g_return_val_if_fail(cnc != NULL, -1);

  if (OCITransRollback((dvoid *) cnc->hservice, cnc->herr, OCI_DEFAULT) != OCI_SUCCESS)
    {
      add_error_to_connection(cnc);
      return (-1);
    }
  return (0);
}
