/* GNOME DB
 * Copyright (C) 1998 Michael Lausch
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gda-odbc-misc.h"
#include "odbc.h"
#include "sqldriver.h"

GDA_ValueType
gda_odbc_sql2gdatype(gint sql_type)
{
  switch (sql_type)
    {
    case SQL_BIT:
    case SQL_BIGINT:
      return GDA_TypeBigint;
    case SQL_SMALLINT:
      return GDA_TypeSmallint;
    case SQL_TINYINT:
      return GDA_TypeTinyint;
    case SQL_INTEGER:
      return GDA_TypeInteger;
    case SQL_CHAR:
      return GDA_TypeChar;
    case SQL_VARCHAR:
      return GDA_TypeVarchar;
    case SQL_VARBINARY:
      return GDA_TypeVarbin;
    case SQL_BINARY:
      return GDA_TypeBinary;
    case SQL_LONGVARBINARY:
      return GDA_TypeLongvarbin;
    case SQL_LONGVARCHAR:
      return GDA_TypeLongvarchar;
    case SQL_DECIMAL:
      return GDA_TypeDecimal;
    case SQL_NUMERIC:
      return GDA_TypeNumeric;
    case SQL_REAL:
      return GDA_TypeSingle;
    case SQL_DOUBLE:
      return GDA_TypeDouble;
    case SQL_FLOAT:
      return GDA_TypeSingle;
    case SQL_DATE:
      return GDA_TypeDbDate;
    case SQL_TIME:
      return GDA_TypeDbTime;
    case SQL_TIMESTAMP:
      return GDA_TypeDbTimestamp;
    default:
      return GDA_TypeNull;
    }
}

gint
gda_odbc_sql2ctype(gint sql_type)
{
  switch (sql_type)
    {
    case SQL_BIT:
      return SQL_C_BIT;
    case SQL_BIGINT:
      return SQL_C_CHAR;
    case SQL_SMALLINT:
      return SQL_C_SSHORT;
    case SQL_TINYINT:
      return SQL_C_STINYINT;
    case SQL_INTEGER:
      return SQL_C_SLONG;
    case SQL_CHAR:
      return SQL_C_CHAR;
    case SQL_VARCHAR:
      return SQL_C_CHAR;
    case SQL_VARBINARY:
      return SQL_C_BINARY;
    case SQL_BINARY:
      return SQL_C_BINARY;
    case SQL_LONGVARBINARY:
      return SQL_C_BINARY;
    case SQL_LONGVARCHAR:
      return SQL_C_CHAR;
    case SQL_DECIMAL:
      return SQL_C_CHAR;
    case SQL_NUMERIC:
      return SQL_C_CHAR;
    case SQL_REAL:
      return SQL_C_FLOAT;
    case SQL_DOUBLE:
      return SQL_C_DOUBLE;
    case SQL_FLOAT:
      return SQL_C_DOUBLE;
    case SQL_DATE:
      return SQL_C_DATE;
    case SQL_TIME:
      return SQL_C_TIME;
    case SQL_TIMESTAMP:
      return SQL_C_TIMESTAMP;
    default:
      return GDA_TypeNull;
    }
}

gint
gda_odbc_gda2ctype(GDA_ValueType tag)
{
  
  switch (tag)
    {
    case GDA_TypeNull:
      g_warning("GOT GDA_TYPE_NULL");
      abort();
      break;
    case GDA_TypeBigint:
      return SQL_C_CHAR;	/* unsigned char* */
      break;
    case GDA_TypeBinary:
      return SQL_C_BINARY;	/* unsigned char*  */
      break;
    case GDA_TypeBoolean:	/* unsigned char */
      return SQL_C_BIT;
      break;
    case GDA_TypeBstr:
      return SQL_C_BINARY;	/* unsigned char* */
      break;
    case GDA_TypeChar:
      return SQL_C_CHAR;	/* unsigned char*  */
      break;
    case GDA_TypeCurrency:
      return SQL_C_CHAR;	/* unsigned char* */
      break;
    case GDA_TypeDate:
      return SQL_C_LONG;	/* time_t */
      break;
    case GDA_TypeDbDate:
      return SQL_C_DATE;	/* struct SQLDATE */
      break;
    case GDA_TypeDbTime:
      return SQL_C_TIME;	/* struct SQLTIME */
      break;
    case GDA_TypeDbTimestamp:
      return SQL_C_TIMESTAMP;	/* struxct sqltimestamp */
      break;
    case GDA_TypeDecimal:	
      return SQL_C_CHAR;	/* unsigned char* */
      break;
    case GDA_TypeDouble:
      return SQL_C_DOUBLE;	/* double */
    case GDA_TypeError:
      return SQL_C_CHAR;	/* unsigned char* */
      break;
    case GDA_TypeInteger:
      return SQL_C_SLONG;	/* unsigned long */
    case GDA_TypeLongvarbin:
      return SQL_C_BINARY;	/* unsigned char* */
    case GDA_TypeLongvarchar:
      return SQL_C_CHAR;	/* unsigned char* */
    case GDA_TypeLongvarwchar:
      return SQL_C_CHAR;	/* unsigned char* */
    case GDA_TypeNumeric:
      return SQL_C_CHAR;	/* unsigned char* */
    case GDA_TypeSingle:
      return SQL_C_FLOAT;	/* float */
    case GDA_TypeSmallint:
      return SQL_C_SSHORT;	/* short */
    case GDA_TypeTinyint:
      return SQL_C_STINYINT;	/* unsigned char */
    case GDA_TypeUBigint:
      return SQL_C_CHAR;	/* unsigned char* */
    case GDA_TypeUSmallint:
      return SQL_C_USHORT;	/* unsigned short */
    case GDA_TypeVarchar:
      return SQL_C_CHAR;	/* unsigned char* */
    case GDA_TypeVarbin:
      return SQL_C_BINARY;	/* unsigned char* */
    case GDA_TypeVarwchar:
      return SQL_C_CHAR;	/* unsigned char* */
    case GDA_TypeFixchar:
      return SQL_C_CHAR;	/* unsigned char* */
    case GDA_TypeFixbin:
      return SQL_C_CHAR;	/* unsigned char* */
    case GDA_TypeFixwchar:
      return SQL_C_CHAR;	/* unsigned char* */
    }
  g_warning("gda_odbc_gda2ctype: reached end of function, no matching GDA type for %d found\n", tag);
  abort();
  return -1;
}

void*
value_2_ptr(GDA_Value* v)
{
  
  switch (v->_d)
    {
    case GDA_TypeNull:
      g_warning("GOT GDA_TYPE_NULL");
      abort();
      break;
    case GDA_TypeBigint:
      return v->_u.lvc;		/* unsigned char */
      break;
    case GDA_TypeBinary:
      return v->_u.lvb._buffer;	/* unsigned char*  */
      break;
    case GDA_TypeBoolean:	/* unsigned char */
      return &v->_u.b;
      break;
    case GDA_TypeBstr:
      return v->_u.lvb._buffer;	/* unsigned char* */
      break;
    case GDA_TypeChar:
      return v->_u.lvc;		/* unsigned char*  */
      break;
    case GDA_TypeCurrency:
      return v->_u.lvc;		/* unsigned char* */
      break;
    case GDA_TypeDate:
      return &v->_u.d; /* struct SQLDATE */
      break;
    case GDA_TypeDbDate:
      return &v->_u.dbd; /* struct SQLDATE */
      break;
    case GDA_TypeDbTime:
      return &v->_u.dbt;	/* struct SQLTIME */
      break;
    case GDA_TypeDbTimestamp:
      return &v->_u.dbtstamp;	/* struct SQLTIMESTAMP */
      break;
    case GDA_TypeDecimal:	
      return v->_u.lvc;		/* unsigned char* */
      break;
    case GDA_TypeDouble:
      return &v->_u.dp;		/* double */
    case GDA_TypeError:
      return &v->_u.lvc;	/* unsigned char* */
      break;
    case GDA_TypeInteger:
      return &v->_u.i;		/* CORBA_long */
    case GDA_TypeLongvarbin:
      return v->_u.lvb._buffer; /* unsigned char* */
    case GDA_TypeLongvarchar:
      return v->_u.lvc;		/* unsigned char* */
    case GDA_TypeLongvarwchar:
      return v->_u.lvc;		/* unsigned char* */
    case GDA_TypeNumeric:
      return v->_u.lvc;		/* unsigned char* */
    case GDA_TypeSingle:
      return &v->_u.f;		/* float */
    case GDA_TypeSmallint:
      return &v->_u.si;		/* short */
    case GDA_TypeTinyint:
      return &v->_u.c;		/* unsigned char */
    case GDA_TypeUBigint:
      return v->_u.lvc;		/* unsigned char* */
    case GDA_TypeUSmallint:
      return &v->_u.us;		/* unsigned short */
    case GDA_TypeVarchar:
      return v->_u.lvc;		/* unsigned char* */
    case GDA_TypeVarbin:
      return v->_u.lvb._buffer;	/* unsigned char* */
    case GDA_TypeVarwchar:
      return v->_u.lvc;		/* unsigned char* */
    case GDA_TypeFixchar:
      return v->_u.lvc;		/* unsigned char* */
    case GDA_TypeFixbin:
      return v->_u.lvb._buffer; /* unsigned char* */
    case GDA_TypeFixwchar:
      return v->_u.lvc;		/* unsigned char* */
    }
  g_warning("value_2_ptr: reached end of function, no matching GDA type found\n");
  abort();
  return (void*)0;
}
      
long
value_2_inputsize(GDA_Value* v)
{
  
  switch (v->_d)
    {
    case GDA_TypeNull:
      g_warning("GOT GDA_TYPE_NULL");
      abort();
      break;
    case GDA_TypeBigint:
      return SQL_NTS;		/* unsigned char */
      break;
    case GDA_TypeBinary:
      return v->_u.lvb._length;	/* unsigned char*  */
      break;
    case GDA_TypeBoolean:	/* unsigned char */
      return sizeof(v->_u.b);
      break;
    case GDA_TypeBstr:
      return v->_u.lvb._length;	/* unsigned char* */
      break;
    case GDA_TypeChar:
      return SQL_NTS;		/* unsigned char*  */
      break;
    case GDA_TypeCurrency:
      return SQL_NTS;		/* unsigned char* */
      break;
    case GDA_TypeDate:
      return sizeof(v->_u.d); /* struct SQLDATE */
      break;
    case GDA_TypeDbDate:
      return sizeof(v->_u.dbd);	/* struct SQLDATE */
      break;
    case GDA_TypeDbTime:
      return sizeof(v->_u.dbt);	/* struct SQLTIME */
      break;
    case GDA_TypeDbTimestamp:
      return sizeof(v->_u.dbtstamp); /* struct SQLTIMESTAMP */
      break;
    case GDA_TypeDecimal:	
      return SQL_NTS;		/* unsigned char* */
      break;
    case GDA_TypeDouble:
      return sizeof(v->_u.dp);		/* double */
    case GDA_TypeError:
      return SQL_NTS;	/* unsigned char* */
      break;
    case GDA_TypeInteger:
      return sizeof(v->_u.i);		/* CORBA_long */
    case GDA_TypeLongvarbin:
      return v->_u.lvb._length; /* unsigned char* */
    case GDA_TypeLongvarchar:
      return SQL_NTS;		/* unsigned char* */
    case GDA_TypeLongvarwchar:
      return SQL_NTS;		/* unsigned char* */
    case GDA_TypeNumeric:
      return SQL_NTS;		/* unsigned char* */
    case GDA_TypeSingle:
      return sizeof(v->_u.f);	/* float */
    case GDA_TypeSmallint:
      return sizeof(v->_u.si);	/* short */
    case GDA_TypeTinyint:
      return sizeof(v->_u.c);	/* unsigned char */
    case GDA_TypeUBigint:
      return SQL_NTS;		/* unsigned char* */
    case GDA_TypeUSmallint:
      return sizeof(v->_u.us);	/* unsigned short */
    case GDA_TypeVarchar:
      return SQL_NTS;		/* unsigned char* */
    case GDA_TypeVarbin:
      return v->_u.lvb._length;	/* unsigned char* */
    case GDA_TypeVarwchar:
      return SQL_NTS;		/* unsigned char* */
    case GDA_TypeFixchar:
      return SQL_NTS;		/* unsigned char* */
    case GDA_TypeFixbin:
      return v->_u.lvb._length; /* unsigned char* */
    case GDA_TypeFixwchar:
      return SQL_NTS;		/* unsigned char* */
    }
  g_warning("value_2_inputsize: reached end of function, no matching GDA type found\n");
  abort();
  return 0;
}
      
gint
gda_odbc_gda2sqltype(GDA_ValueType gda_type)
{
  switch (gda_type)
    {
    case GDA_TypeBigint:
      return SQL_BIGINT;
	
    case GDA_TypeSmallint:
      return SQL_SMALLINT;

    case GDA_TypeTinyint:
      return SQL_TINYINT;
      
    case GDA_TypeInteger:
      return SQL_INTEGER;

    case GDA_TypeChar:
      return SQL_CHAR;

    case GDA_TypeVarchar:
      return SQL_VARCHAR;

    case GDA_TypeVarbin:
      return SQL_VARBINARY;

    case GDA_TypeBinary:
      return SQL_BINARY;

    case GDA_TypeLongvarbin:
      return SQL_LONGVARBINARY;

    case GDA_TypeLongvarchar:
      return SQL_LONGVARCHAR;

    case GDA_TypeDecimal:
      return SQL_DECIMAL;

    case GDA_TypeNumeric:
      return SQL_NUMERIC;

    case GDA_TypeSingle:
      return SQL_REAL;

    case GDA_TypeDouble:
      return SQL_DOUBLE;

    case GDA_TypeDbDate:
      return SQL_DATE;

    case GDA_TypeDbTime:
      return SQL_TIME;

    case GDA_TypeDbTimestamp:
      return SQL_TIMESTAMP;

    default:
      g_error("gda_odbc_gda2sqltype: Unhandled gda type %d\n", gda_type);
    }
  return 0;
}
