/* GNOME DB libary
 * Copyright (C) 1999 Rodrigo Moya
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gnome.h>

#include <gda.h>
#include "gda-xml.h"

#define GDA_XML_NODE_RECORDSET "RECORDSET"
#define GDA_XML_NODE_VIEW      "VIEW"

enum
{
  GDA_XML_DATABASE_CHANGED,
  GDA_XML_LAST_SIGNAL
};

static gint gda_xml_database_signals[GDA_XML_LAST_SIGNAL] = { 0, };

static void gda_xml_database_class_init (Gda_XmlDatabaseClass *klass);
static void gda_xml_database_init (Gda_XmlDatabase *db);

/* private functions */
static void
database_changed_cb (Gda_XmlDatabase *db)
{
  g_return_if_fail(IS_GDA_XML_DATABASE(db));
  
  /* update the modification time */
  gda_xml_database_set_date(db, 0);
}

static xmlNodePtr
find_field_by_name (xmlNodePtr xmlrs, const gchar *name)
{
  xmlNodePtr node;
  g_return_val_if_fail(xmlrs != 0, 0);
  g_return_val_if_fail(name != 0, 0);
  for (node = xmlrs->childs; node != 0; node = node->next)
    {
      gchar *prop = xmlGetProp(node, "NAME");
      if ((prop != 0) && !strcmp(node->name, "FIELD") && !strcmp(prop, name))
        return (node);
    }
  return (0);
}

static xmlNodePtr
find_node_by_name (xmlDocPtr doc, const gchar *name, const gchar *type)
{
  xmlNodePtr node;
  
  g_return_val_if_fail(doc != 0, 0);
  g_return_val_if_fail(name != 0, 0);
  g_return_val_if_fail(type != 0, 0);
  
  for (node = doc->root->childs; node != 0; node = node->next)
    {
      gchar *prop = xmlGetProp(node, "NAME");
      if ((prop != 0) && !strcmp(node->name, type) && !strcmp(prop, name))
        return (node);
    }
  return (0);
}

static const gchar *
get_current_date (void)
{
  static gchar buf[256];
  time_t current = time(0);
  strftime(buf, sizeof(buf), "%c", localtime(&current));
  return ((const gchar *) buf);
}

static xmlNodePtr
recordset_add_gda_field (xmlNodePtr xmlrs, Gda_Field *field)
{
  xmlNodePtr xmlfld;
  gchar *tmp, bfr[256];
  
  g_return_val_if_fail(xmlrs != 0, 0);
  g_return_val_if_fail(field != 0, xmlrs);
  
  /* add the field */
  xmlfld = xmlNewChild(xmlrs, 0, "FIELD", 0);
  xmlSetProp(xmlfld, "NAME", gda_field_name(field));
  xmlSetProp(xmlfld, "SIZE",
             (tmp = g_strdup_printf("%d", gda_field_defined_size(field))));
  g_free((gpointer) tmp);
  xmlSetProp(xmlfld, "SCALE", 
             (tmp = g_strdup_printf("%d", gda_field_scale(field))));
  g_free((gpointer) tmp);
  
  /* for the type, call gda_fieldtype_2_string() */
  gda_fieldtype_2_string(bfr, sizeof(bfr), gda_field_type(field));
  xmlSetProp(xmlfld, "TYPE", bfr);
  return (xmlrs);
}

static xmlNodePtr
recordset_to_xml (xmlDocPtr doc, Gda_Recordset *recset, const gchar *name)
{
  xmlNodePtr xmlrs;
  gint total_fields, cnt;
  gulong position;
  gchar *create_cmd;

  g_return_val_if_fail(doc != 0, 0);
  g_return_val_if_fail(recset != 0, 0);
  
  /* if there is already a recordset with that name, do nothing */
  if ((xmlrs = find_node_by_name(doc, name, GDA_XML_NODE_RECORDSET)))
    return (xmlrs);
  
  /* create the root document if 0 */
  if (doc->root == 0)
    {
      doc->root = xmlNewDocNode(doc, 0, "DATABASE", 0);
      xmlSetProp(doc->root, "DATE", get_current_date());
    }
  
  /* create the RECORDSET item */
  xmlrs = xmlNewChild(doc->root, 0, "RECORDSET", 0);
  xmlSetProp(xmlrs, "NAME", name);
  
  /*
  create_cmd = gda_connection_create_recordset(recset->cnc, recset);
  if (create_cmd != 0)
    {
      xmlSetProp(xmlrs, "COMMAND", create_cmd);
      g_free((gpointer) create_cmd);
    }
  */

  /* add the fields */
  total_fields = gda_recordset_rowsize(recset);
  for (cnt = 0; cnt < total_fields; cnt++)
    {
      xmlrs = recordset_add_gda_field(xmlrs, gda_recordset_field_idx(recset, cnt));
    }

  /* and now, add recordset data */
  position = gda_recordset_move(recset, 1, 0);
  while (position != GDA_RECORDSET_INVALID_POSITION &&
         !gda_recordset_eof(recset))
    {
      xmlNodePtr row, cell;

      /* add row entity */
      row = xmlNewChild(xmlrs, 0, "ROW", 0);

      /* add cells */
      for (cnt = 0; cnt < total_fields; cnt++)
        {
          Gda_Field *field = gda_recordset_field_idx(recset, cnt);
          if (field != 0)
            cell = xmlNewChild(row, 0, "VALUE", gda_stringify_value(0, 0, field));
          else cell = xmlNewChild(row, 0, "VALUE", 0);
        }
      position = gda_recordset_move(recset, 1, 0);
    }

  /* finally, return the node pointer */
  return (xmlrs);
}

static void
gda_xml_database_class_init (Gda_XmlDatabaseClass *klass)
{
  GtkObjectClass *object_class = (GtkObjectClass *) klass;
  
  gda_xml_database_signals[GDA_XML_DATABASE_CHANGED] =
          gtk_signal_new("changed",
                         GTK_RUN_FIRST,
                         object_class->type,
                         GTK_SIGNAL_OFFSET(Gda_XmlDatabaseClass, changed),
                         gtk_marshal_NONE__POINTER,
                         GTK_TYPE_NONE, 0);
  gtk_object_class_add_signals(object_class, gda_xml_database_signals,
                               GDA_XML_LAST_SIGNAL);
  klass->changed = database_changed_cb;
}

static void
gda_xml_database_init (Gda_XmlDatabase *db)
{
  db->xmldoc = 0;
  db->filename = 0;
}

/*
 * Gda_XmlDatabase interface
 */
guint
gda_xml_database_get_type (void)
{
  static guint gda_xml_database_type = 0;
  if (!gda_xml_database_type)
    {
      GtkTypeInfo gda_xml_database_info =
      {
        "Gda_XmlDatabase",
        sizeof(Gda_XmlDatabase),
        sizeof(Gda_XmlDatabaseClass),
        (GtkClassInitFunc) gda_xml_database_class_init,
        (GtkObjectInitFunc) gda_xml_database_init,
        (GtkArgSetFunc) 0,
        (GtkArgSetFunc) 0
      };
      gda_xml_database_type = gtk_type_unique(gtk_object_get_type(), 
                                              &gda_xml_database_info);
    }
  return (gda_xml_database_type);
}

/**
 * gda_xml_database_new
 * @name: The name to represent the database
 *
 * Allocates memory for a new Gda_XmlDatabase object
 *
 * Returns: a pointer to the newly allocated object, or NULL on error
 */
Gda_XmlDatabase *
gda_xml_database_new (const gchar *name)
{
  Gda_XmlDatabase *db;
  if ((db = gtk_type_new(gda_xml_database_get_type())))
    {
      db->xmldoc = xmlNewDoc("1.0");
      db->xmldoc->root = xmlNewDocNode(db->xmldoc, 0, "DATABASE", 0);
      if (name != 0) xmlSetProp(db->xmldoc->root, "NAME", name);
      xmlSetProp(db->xmldoc->root, "DATE", get_current_date());
      db->filename = 0;
      return (db);
    }
  else g_error("gda_xml_database_new(): cannot create Gda_XmlDatabase object");
  return (0);
}

/**
 * gda_xml_database_load_from_file
 * @filename: The path to the file to be loaded
 *
 * Loads a XML file and creates a Gda_XmlDatabase object to
 * represent the file's contents
 *
 * Returns: a pointer to the newly created object, or NULL on error
 */
Gda_XmlDatabase *
gda_xml_database_load_from_file (const gchar *filename)
{
  Gda_XmlDatabase *db;
  xmlDocPtr doc;
  
  g_return_val_if_fail(filename != 0, 0);
  if ((doc = xmlParseFile(filename)))
    {
      /* allocate Gda_XmlDatabase structure */
      if ((db = gtk_type_new(gda_xml_database_get_type())))
        {
          db->xmldoc = doc;
          db->filename = g_strdup(filename);
          return (db);
        }
      else g_warning("cannot create Gda_XmlDatabase object");
    }
  else g_warning("error while parsing file %s", filename);
  return (0);
}

/**
 * gda_xml_database_free
 * @db: Pointer to the object to be freed
 *
 * Cleans up all memory resources occupied by the Gda_XmlDatabase object
 */
void
gda_xml_database_free (Gda_XmlDatabase *db)
{
  g_return_if_fail(db != 0);
  xmlFreeDoc(db->xmldoc);
  if (db->filename != 0)
    g_free((gpointer) db->filename);
  g_free((gpointer) db);
}

/**
 * gda_xml_database_add_recordset
 * @db: Pointer to the XML database object
 * @recset: The Gda_Recordset object to be added
 * @name: Name to represent the recordset
 *
 * Adds a recordset from an existing Gda_Recordset object
 *
 * Returns: 0 on success, -1 on error
 */
gint
gda_xml_database_add_recordset (Gda_XmlDatabase *db, Gda_Recordset *recset,
                                const gchar *name)
{
  xmlNodePtr new_recset;
  
  g_return_val_if_fail(db != 0, -1);
  g_return_val_if_fail(recset != 0, -1);
  
  /* if there is already a recordset with that name, do nothing */
  if ((new_recset = find_node_by_name(db->xmldoc, name, GDA_XML_NODE_RECORDSET)))
    return (0);
    
  /* create xml recordset representation */
  if ((new_recset = recordset_to_xml(db->xmldoc, recset, name)))
    {
      gtk_signal_emit(GTK_OBJECT(db), gda_xml_database_signals[GDA_XML_DATABASE_CHANGED]);
      return (0);
    }
  return (-1);
}

/**
 * gda_xml_database_save_to_file
 * @db: Pointer to the XML database object
 * @filename File name to be saved
 *
 * Saves the Gda_XmlDatabase object to a file in disk
 *
 * Returns: 0 on success, -1 on error
 */
gint
gda_xml_database_save_to_file (Gda_XmlDatabase *db, const gchar *filename)
{
  g_return_val_if_fail(db != 0, -1);
  g_return_val_if_fail(filename != 0, -1);
  if (filename != 0)
    {
      if (db->filename != 0)
        g_free((gpointer) db->filename);
      db->filename = g_strdup(filename);
    }
  gda_xml_database_set_date(db, get_current_date());
  return (xmlSaveFile(db->filename, db->xmldoc));
}

/**
 * gda_xml_database_get_name
 * @db: Pointer to the XML database object
 *
 * Returns the name of the database object
 */
gchar *
gda_xml_database_get_name (Gda_XmlDatabase *db)
{
  g_return_val_if_fail(db != 0, 0);
  return (xmlGetProp(db->xmldoc->root, "NAME"));
}

/**
 * gda_xml_database_set_name
 * @db: the Gda_XmlDatabase object
 * @new_name: new name of the database
 *
 * Sets the name to represent the XML database object
 */
void
gda_xml_database_set_name (Gda_XmlDatabase *db, const gchar *new_name)
{
  g_return_if_fail(db != 0);
  g_return_if_fail(new_name != 0);
  xmlSetProp(db->xmldoc->root, "NAME", new_name);
}

/**
 * gda_xml_database_get_filename
 * @db: the Gda_XmlDatabase object
 *
 * Returns the file name associated with this object
 */
gchar *
gda_xml_database_get_filename (Gda_XmlDatabase *db)
{
  g_return_val_if_fail(db != 0, 0);
  return (db->filename);
}

/**
 * gda_xml_database_set_filename
 * @db: the Gda_XmlDatabase object
 * @new_name: file name
 *
 * Associates a file in disk with this object
 */
void
gda_xml_database_set_filename (Gda_XmlDatabase *db, const gchar *new_name)
{
  g_return_if_fail(db != 0);
  g_return_if_fail(new_name != 0);
  if (db->filename != 0)
    g_free((gpointer) db->filename);
  db->filename = g_strdup(new_name);
}

/**
 * gda_xml_database_get_date
 * @db: the Gda_XmlDatabase object
 *
 * Returns the last-modified time
 */
gchar *
gda_xml_database_get_date (Gda_XmlDatabase *db)
{
  g_return_val_if_fail(db != 0, 0);
  return (xmlGetProp(db->xmldoc->root, "DATE"));
}

/**
 * gda_xml_database_set_date
 * @db: the Gda_XmlDatabase object
 * @new_date: string representing the date
 *
 * Sets the last-modified time
 */
void
gda_xml_database_set_date (Gda_XmlDatabase *db, const gchar *new_date)
{
  g_return_if_fail(db != 0);
  xmlSetProp(db->xmldoc->root, "DATE", new_date != 0 
             ? new_date : get_current_date());
}


/**
 * gda_xml_database_list_recordsets
 * @db: the Gda_XmlDatabase object
 * @callback: user-defined callback function
 *
 * Lists all recordsets in the given database. To do so, it calls the
 * callback function specified as the second argument for each recordset
 * found.
 * The callback function should look like the following:
 * 	void callback_function (Gda_XmlDatabase *db, const gchar *recset_name)
 */
void
gda_xml_database_list_recordsets (Gda_XmlDatabase *db, GFunc callback)
{
  xmlNodePtr node;
  
  g_return_if_fail(IS_GDA_XML_DATABASE(db));
  g_return_if_fail(db->xmldoc != 0);
  g_return_if_fail(callback != 0);
  
  for (node = db->xmldoc->root->childs; node != 0; node = node->next)
    {
      if (!strcmp(node->name, GDA_XML_NODE_RECORDSET))
        {
          gchar *prop = xmlGetProp(node, "NAME");
          (*callback)((gpointer) db, (gpointer) prop);
        }
    }
}

/*
 * Adding recordsets
 */
 
/**
 * gda_xml_recordset_new
 * @db: the Gda_XmlDatabase object
 * @name: name of the new recordset
 *
 * Creates a new recordset from scratch
 *
 * Returns: 0 on success, -1 on error
 */
gint
gda_xml_recordset_new (Gda_XmlDatabase *db, const gchar *name)
{
  xmlNodePtr xmlrs;
  
  g_return_val_if_fail(db != 0, -1);
  g_return_val_if_fail(name != 0, -1);
  
  /* if there is already a recordset with that name, do nothing */
  if ((xmlrs = find_node_by_name(db->xmldoc, name, GDA_XML_NODE_RECORDSET)))
    return (0);

  /* create the RECORDSET item */
  xmlrs = xmlNewChild(db->xmldoc->root, 0, "RECORDSET", 0);
  xmlSetProp(xmlrs, "NAME", name);
  gtk_signal_emit(GTK_OBJECT(db), gda_xml_database_signals[GDA_XML_DATABASE_CHANGED]);
  return (0);
}

/**
 * gda_xml_recordset_remove
 * @db: the Gda_XmlDatabase object
 * @name: recordset name
 *
 * Removes an existing recordset
 */
gint
gda_xml_recordset_remove (Gda_XmlDatabase *db, const gchar *name)
{
  xmlNodePtr xmlrs;
  
  g_return_val_if_fail(db != 0, -1);
  g_return_val_if_fail(name != 0, -1);
  
  if ((xmlrs = find_node_by_name(db->xmldoc, name, GDA_XML_NODE_RECORDSET)))
    {
      xmlUnlinkNode(xmlrs);
      xmlFreeNode(xmlrs);
      gtk_signal_emit(GTK_OBJECT(db), gda_xml_database_signals[GDA_XML_DATABASE_CHANGED]);
      return (0);
    }
  return (-1);
}

/**
 * gda_xml_recordset_add_field
 *
 * Adds a new field to an existing recordset
 */
gint
gda_xml_recordset_add_field (Gda_XmlDatabase *db, const gchar *recset,
                             const gchar *fieldname, gint size,
                             gint scale, gint type)
{
  xmlNodePtr xmlrs, xmlfld;
  gchar *tmp;
  
  g_return_val_if_fail(db != 0, -1);
  g_return_val_if_fail(recset != 0, -1);
  g_return_val_if_fail(fieldname != 0, -1);
  
  if ((xmlrs = find_node_by_name(db->xmldoc, recset, GDA_XML_NODE_RECORDSET)))
    {
      if ((xmlfld = find_field_by_name(xmlrs, fieldname)))	/* already exists */
        return (0);

      /* add the field */
      xmlfld = xmlNewChild(xmlrs, 0, "FIELD", 0);
      xmlSetProp(xmlfld, "NAME", fieldname);
      xmlSetProp(xmlfld, "SIZE", (tmp = g_strdup_printf("%d", size)));
      g_free((gpointer) tmp);
      xmlSetProp(xmlfld, "SCALE", (tmp = g_strdup_printf("%d", scale)));
      g_free((gpointer) tmp);
      xmlSetProp(xmlfld, "TYPE", (tmp = g_strdup_printf("%d", type)));
      g_free((gpointer) tmp);
      gtk_signal_emit(GTK_OBJECT(db), gda_xml_database_signals[GDA_XML_DATABASE_CHANGED]);
      return (0);
    }
  return (-1);
}

/**
 * gda_xml_recordset_add_gda_field
 *
 * Adds a field to an existing recordset from a Gda_Field object
 */
gint
gda_xml_recordset_add_gda_field (Gda_XmlDatabase *db, 
                                 const gchar *recset, 
                                 Gda_Field *field)
{
  xmlNodePtr xmlrs;
  
  g_return_val_if_fail(db != 0, -1);
  g_return_val_if_fail(field != 0, -1);
  
  if ((xmlrs = find_node_by_name(db->xmldoc, recset, GDA_XML_NODE_RECORDSET)))
    {
      if (find_field_by_name(xmlrs, gda_field_name(field)))
        return (0);
      xmlrs = recordset_add_gda_field(xmlrs, field);
      gtk_signal_emit(GTK_OBJECT(db), gda_xml_database_signals[GDA_XML_DATABASE_CHANGED]);
      return (0);
    }
  return (-1);
}

/**
 * gda_xml_recordset_remove_field
 *
 * Removes a field from an existing recordset
 */
gint
gda_xml_recordset_remove_field (Gda_XmlDatabase *db, 
                                const gchar *recset, 
                                const gchar *fieldname)
{
  xmlNodePtr rs, fld;
  
  g_return_val_if_fail(db != 0, -1);
  g_return_val_if_fail(recset != 0, -1);
  g_return_val_if_fail(fieldname != 0, -1);
  
  if ((rs = find_node_by_name(db->xmldoc, recset, GDA_XML_NODE_RECORDSET)))
    {
      if ((fld = find_field_by_name(rs, fieldname)))
        {
          xmlUnlinkNode(fld);
          xmlFreeNode(fld);
          gtk_signal_emit(GTK_OBJECT(db), gda_xml_database_signals[GDA_XML_DATABASE_CHANGED]);
          return (0);
        }
    }
  return (-1);
}

/**
 * gda_xml_recordset_list_fields
 * @db The database stored in XML format
 * @recset Name of the recordset to describe
 * @callback User-defined callback function
 *
 * Sends the description of the given recordset by calling
 * a user-defined callback function for each field in the
 * recordset
 */
void
gda_xml_recordset_list_fields (Gda_XmlDatabase *db, const gchar *recset, 
                              Gda_XmlListFieldFunc callback, gpointer user_data)
{
  xmlNodePtr rs, fld;
  
  g_return_if_fail(IS_GDA_XML_DATABASE(db));
  g_return_if_fail(db->xmldoc != 0);
  g_return_if_fail(recset != 0);
  g_return_if_fail(callback != 0);
  
  rs = find_node_by_name(db->xmldoc, recset, GDA_XML_NODE_RECORDSET);
  if (rs != 0)
    {
      for (fld = rs->childs; fld != 0; fld = fld->next)
        {
          if (!strcmp(fld->name, "FIELD"))
            {
              gchar *name = xmlGetProp(fld, "NAME");
              gchar *type = xmlGetProp(fld, "TYPE");
              gchar *size = xmlGetProp(fld, "SIZE");
              gchar *scale = xmlGetProp(fld, "SCALE");
              if (name != 0)
                {
                  (*callback)(db, recset, name,
                              size != 0 ? atoi(size) : 0,
                              scale != 0 ? atoi(scale) : 0,
                              type, user_data);
                }
            }
        }
    }
}

/**
 * gda_xml_view_new
 * @db the database stored in XML format
 * @view view name
 * @cmd command needed to create the given view
 *
 * Adds a new SQL view to the given database. The 'cmd' parameter must
 * contain a SQL command needed to create the view
 */
gint
gda_xml_view_new (Gda_XmlDatabase *db, const gchar *view, const gchar *cmd)
{
  xmlNodePtr xmlv;
  
  g_return_val_if_fail(IS_GDA_XML_DATABASE(db), -1);
  g_return_val_if_fail(view != 0, -1);
  g_return_val_if_fail(cmd != 0, -1);
  
  /* look for already existing view */
  xmlv = find_node_by_name(db->xmldoc, view, GDA_XML_NODE_VIEW);
  
  /* create new node if not found */
  if (xmlv == 0)
    {
      xmlv = xmlNewChild(db->xmldoc->root, 0, GDA_XML_NODE_VIEW, 0);
    }
  xmlSetProp(xmlv, "NAME", view);
  xmlSetProp(xmlv, "COMMAND", cmd);
  gtk_signal_emit(GTK_OBJECT(db), gda_xml_database_signals[GDA_XML_DATABASE_CHANGED]);
  return (0);
}

/**
 * gda_xml_view_remove
 * @db the database stored in XML format
 * @view view name
 *
 * Removes the given view from the XML database definition.
 */
gint
gda_xml_view_remove (Gda_XmlDatabase *db, const gchar *view)
{
  xmlNodePtr xmlv;
  
  g_return_val_if_fail(IS_GDA_XML_DATABASE(db), -1);
  g_return_val_if_fail(view != 0, -1);
  
  xmlv = find_node_by_name(db->xmldoc, view, GDA_XML_NODE_VIEW);
  if (xmlv != 0)
    {
      xmlUnlinkNode(xmlv);
      xmlFreeNode(xmlv);
      gtk_signal_emit(GTK_OBJECT(db), gda_xml_database_signals[GDA_XML_DATABASE_CHANGED]);
      return (0);
    }
  return (-1);
}
