/* GNOME DB libary
 * Copyright (C) 1999 Rodrigo Moya
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
 
#include "gnome-db-browser.h"
#include "gnome-db-grid.h"
#include "gnome-db-list.h"
#include "gda-ui.h"

static void gnome_db_browser_class_init (GnomeDbBrowserClass *);
static void gnome_db_browser_init       (GnomeDbBrowser *);
static void gnome_db_browser_destroy    (GnomeDbBrowser *);

static void browse_data_cb (GtkWidget *w, gpointer data);
static void show_tables_cb (GtkWidget *w, gpointer data);
static void show_procs_cb  (GtkWidget *w, gpointer data);
static void show_views_cb  (GtkWidget *w, gpointer data);

/*
 * GnomeDbBrowser widget signals
 */
enum
{
  GNOME_DB_BROWSER_CONNECTION_CHANGED,
  GNOME_DB_BROWSER_LAST_SIGNAL
};

static gint gnome_db_browser_signals[GNOME_DB_BROWSER_LAST_SIGNAL] = { 0, };

/*
 * Private functions
 */
static void
show_object_list (GnomeDbBrowser *brw, Gda_Recordset *recset, gint idx)
{
  static GtkWidget *pixmap = 0;
  gchar  bfr[128];
  gulong position;
  gint cnt = 0;
  
  g_return_if_fail(GNOME_DB_IS_BROWSER(brw));
  g_return_if_fail(recset != 0);

  position = gda_recordset_move(recset, 1, 0);
  while (position != GDA_RECORDSET_INVALID_POSITION &&
         !gda_recordset_eof(recset))
    {
      Gda_Field *field = gda_recordset_field_idx(recset, idx);
      if (field != 0)
        {
          gchar *row[2], empty_string[] = "       ";
          
          gda_stringify_value(bfr, sizeof(bfr) - 1, field);
          row[0] = empty_string;
          row[1] = bfr;
          gtk_clist_append(GTK_CLIST(brw->object_list), row);
          
          /* create shared pixmap */
          if (!pixmap)
            {
              pixmap = gnome_stock_pixmap_widget(brw->object_list, GNOME_STOCK_MENU_BOOK_BLUE);
              if (!pixmap) return;
              gtk_widget_show(pixmap);
            }
          gtk_clist_set_pixmap(GTK_CLIST(brw->object_list), cnt, 0,
                               GNOME_PIXMAP(pixmap)->pixmap,
                               GNOME_PIXMAP(pixmap)->mask);
        }
      position = gda_recordset_move(recset, 1, 0);
      cnt++;
    }
  /* free the recordset */
  gda_recordset_close(recset);
  gda_recordset_free(recset);
}

/*
 * Callbacks
 */
static void
browse_data_cb (GtkWidget *w, gpointer data)
{
  GList *selected;
  GnomeDbBrowser *brw = GNOME_DB_BROWSER(data);
  
  g_return_if_fail(brw != 0);
  g_return_if_fail(GNOME_DB_IS_BROWSER(brw));
  
  selected = GTK_CLIST(brw->object_list)->selection;
  if (selected)
    {
      gchar *name = 0;
      guint row = GPOINTER_TO_UINT(selected->data);
      gtk_clist_get_text(GTK_CLIST(brw->object_list), row, 1, &name);
      if (name)
        {
          gchar *str = 0;
          if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(brw->tables_button)) ||
              gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(brw->views_button)))
            {
              str = g_strdup_printf("select * from %s", name);
            }
          
          if (str != 0)
            {
              GtkWidget *dialog, *table, *grid;
              gulong reccount;
              Gda_Recordset *recset;
              
              dialog = gnome_dialog_new(name, GNOME_STOCK_BUTTON_CLOSE, NULL);
              gtk_widget_set_usize(dialog, 600, 450);
              
              table = gtk_table_new(3, 2, FALSE);
              gtk_widget_show(table);
              gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), table, 1, 1, 0);
              
              grid = gnome_db_grid_new(0);
              gtk_widget_show(grid);
              gtk_table_attach(GTK_TABLE(table), grid, 0, 2, 1, 2,
                               GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                               GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                               3, 3);
              
              /* show recordset */
              recset = gda_connection_execute(brw->cnc, str, &reccount, 0);
              gnome_db_grid_set_recordset(GNOME_DB_GRID(grid), recset);
              
              gnome_dialog_run_and_close(GNOME_DIALOG(dialog));
              
              g_free((gpointer) str);
            }
        }
    }
}

static void
select_object_cb (GtkCList *clist, gint row, gint column, GdkEvent *event, GnomeDbBrowser *brw)
{
  gchar *name = 0;
  
  g_return_if_fail(GNOME_DB_IS_BROWSER(brw));
  
  gtk_clist_get_text(GTK_CLIST(brw->object_list), row, 1, &name);
  if (name)
    {
      Gda_Recordset *recset;
      if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(brw->tables_button)) ||
          gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(brw->views_button)))
        {
          recset = gda_connection_open_schema(brw->cnc, 
                                              GDA_Connection_GDCN_SCHEMA_COLS,
                                              GDA_Connection_TABLE_NAME,
                                              name,
                                              GDA_Connection_no_CONSTRAINT);
        }
      gnome_db_grid_set_recordset(GNOME_DB_GRID(brw->object_description), recset);
      
      if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(brw->tables_button)) ||
          gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(brw->views_button)))
        {
          /* hide all non-table/view specific columns */
          gnome_db_grid_set_column_visibility(GNOME_DB_GRID(brw->object_description), 0, 0);
          gnome_db_grid_set_column_visibility(GNOME_DB_GRID(brw->object_description), 1, 0);
          gnome_db_grid_set_column_visibility(GNOME_DB_GRID(brw->object_description), 2, 0);
        }
    }
}

static void
show_tables_cb (GtkWidget *w, gpointer data)
{
  Gda_Recordset *recset;
  GnomeDbBrowser *brw = GNOME_DB_BROWSER(data);
  
  g_return_if_fail(GTK_IS_TOGGLE_BUTTON(w));
  g_return_if_fail(GNOME_DB_IS_BROWSER(brw));
  
  if (!gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(brw->tables_button)))
    {
      return;
    }
    
  /* clean up previous lists */
  gnome_db_browser_clear(brw);
  
  recset = gda_connection_open_schema(brw->cnc, GDA_Connection_GDCN_SCHEMA_TABLES,
                                      GDA_Connection_no_CONSTRAINT, 0);
  if (recset != 0)
    {
      show_object_list(brw, recset, 2);
    }
}

static void
show_procs_cb (GtkWidget *w, gpointer data)
{
  Gda_Recordset *recset;
  GnomeDbBrowser *brw = GNOME_DB_BROWSER(data);
  
  g_return_if_fail(GTK_IS_TOGGLE_BUTTON(w));
  g_return_if_fail(GNOME_DB_IS_BROWSER(brw));
  
  if (!gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(brw->procs_button)))
    {
      return;
    }
    
  /* clean up previous lists */
  gnome_db_browser_clear(brw);
  
  recset = gda_connection_open_schema(brw->cnc, GDA_Connection_GDCN_SCHEMA_PROCS,
                                      GDA_Connection_no_CONSTRAINT, 0);
  if (recset != 0)
    {
      show_object_list(brw, recset, 2);
    }
}

static void
show_views_cb (GtkWidget *w, gpointer data)
{
  Gda_Recordset *recset;
  GnomeDbBrowser *brw = GNOME_DB_BROWSER(data);
  
  g_return_if_fail(GTK_IS_TOGGLE_BUTTON(w));
  g_return_if_fail(GNOME_DB_IS_BROWSER(brw));
  
  if (!gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(brw->views_button)))
    {
      return;
    }
    
  /* clean up previous lists */
  gnome_db_browser_clear(brw);
  
  recset = gda_connection_open_schema(brw->cnc, GDA_Connection_GDCN_SCHEMA_VIEWS,
                                      GDA_Connection_no_CONSTRAINT, 0);
                                      
  if (recset != 0)
    {
      show_object_list(brw, recset, 0);
    }
}

/*
 * GnomeDbBrowser interface
 */
static void
gnome_db_browser_class_init (GnomeDbBrowserClass *klass)
{
  GtkObjectClass *object_class = (GtkObjectClass *) klass;

  gnome_db_browser_signals[GNOME_DB_BROWSER_CONNECTION_CHANGED] =
           gtk_signal_new("connection_changed", GTK_RUN_LAST, object_class->type,
                          GTK_SIGNAL_OFFSET(GnomeDbBrowserClass, connection_changed),
                          gtk_signal_default_marshaller,
                          GTK_TYPE_NONE, 0);

  gtk_object_class_add_signals(object_class, gnome_db_browser_signals,
                               GNOME_DB_BROWSER_LAST_SIGNAL);
  klass->connection_changed = 0;
  object_class->destroy = gnome_db_browser_destroy;
}

static void
gnome_db_browser_init (GnomeDbBrowser *brw)
{
  GtkWidget *table;
  GtkWidget *box;
  GtkWidget *button;
  GtkWidget *scroll;
  
  table = gtk_table_new(5, 8, FALSE);
  gtk_box_pack_start(GTK_BOX(brw), table, 1, 1, 0);
  gtk_widget_show(table);
  
  scroll = gtk_scrolled_window_new(0, 0);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scroll),
                                 GTK_POLICY_AUTOMATIC,
                                 GTK_POLICY_AUTOMATIC);
  brw->object_list = gda_ui_new_clist_widget(0, 2);
  gtk_signal_connect(GTK_OBJECT(brw->object_list), "select_row",
                     GTK_SIGNAL_FUNC(select_object_cb), (gpointer) brw);
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scroll), brw->object_list);
  gtk_widget_show(brw->object_list);
  gtk_widget_show(scroll);
  gtk_table_attach(GTK_TABLE(table), scroll, 0, 6, 0, 1,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   3, 3);

  brw->object_description = gnome_db_grid_new(0);
  gnome_db_grid_hide_row_numbers(GNOME_DB_GRID(brw->object_description));
  gtk_widget_show(brw->object_description);
  gtk_table_attach(GTK_TABLE(table), brw->object_description, 0, 8, 1, 5,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   3, 3);
  
  /* create tabs */                 
  box = gtk_vbox_new(FALSE, 0);
  brw->tables_button = gtk_radio_button_new_with_label(0, _("Tables"));
  gtk_signal_connect(GTK_OBJECT(brw->tables_button), "toggled",
                     GTK_SIGNAL_FUNC(show_tables_cb), (gpointer) brw);
  gtk_widget_show(brw->tables_button);
  gtk_box_pack_start(GTK_BOX(box), brw->tables_button, 0, 0, 0);

  brw->views_button = gtk_radio_button_new_with_label(
  				gtk_radio_button_group(GTK_RADIO_BUTTON(brw->tables_button)),
  				_("Views"));
  gtk_signal_connect(GTK_OBJECT(brw->views_button), "toggled",
                     GTK_SIGNAL_FUNC(show_views_cb), (gpointer) brw); 
  gtk_widget_show(brw->views_button);
  gtk_box_pack_start(GTK_BOX(box), brw->views_button, 0, 0, 0);
      
  brw->procs_button = gtk_radio_button_new_with_label(
  				gtk_radio_button_group(GTK_RADIO_BUTTON(brw->tables_button)),
  				_("Procedures"));
  gtk_signal_connect(GTK_OBJECT(brw->procs_button), "toggled",
                     GTK_SIGNAL_FUNC(show_procs_cb), (gpointer) brw);
  gtk_widget_show(brw->procs_button);
  gtk_box_pack_start(GTK_BOX(box), brw->procs_button, 0, 0, 0);
  
  gtk_table_attach(GTK_TABLE(table), box, 6, 7, 0, 1,
                   GTK_SHRINK, 
                   0,
                   3, 3);
  gtk_widget_show(box);
  
  /* create left button bar */
  box = gtk_vbox_new(FALSE, 0);
  gtk_widget_show(box);
  gtk_table_attach(GTK_TABLE(table), box, 7, 8, 0, 1,
                   GTK_SHRINK,
                   0,
                   3, 3);
                   
  button = gnome_pixmap_button(gnome_stock_new_with_icon(GNOME_STOCK_MENU_PROP),
                               _("Data"));
  gtk_button_set_relief(GTK_BUTTON(button), GTK_RELIEF_NONE);
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
                     GTK_SIGNAL_FUNC(browse_data_cb), (gpointer) brw);
  gtk_widget_show(button);
  gtk_box_pack_start(GTK_BOX(box), button, 0, 0, 0);
  
  button = gnome_pixmap_button(gnome_stock_new_with_icon(GNOME_STOCK_MENU_TRASH),
                               _("Remove"));
  gtk_button_set_relief(GTK_BUTTON(button), GTK_RELIEF_NONE);
  //gtk_signal_connect(GTK_OBJECT(button), "clicked",
  //                   GTK_SIGNAL_FUNC(browse_data_cb), (gpointer) brw);
  gtk_widget_show(button);
  gtk_box_pack_start(GTK_BOX(box), button, 0, 0, 0);
}

guint
gnome_db_browser_get_type (void)
{
  static guint db_browser_type = 0;

  if (!db_browser_type)
    {
      GtkTypeInfo db_browser_info =
      {
        "GnomeDbBrowser",
        sizeof (GnomeDbBrowser),
        sizeof (GnomeDbBrowserClass),
        (GtkClassInitFunc) gnome_db_browser_class_init,
        (GtkObjectInitFunc) gnome_db_browser_init,
        (GtkArgSetFunc) NULL,
        (GtkArgGetFunc) NULL
      };
      db_browser_type = gtk_type_unique(gtk_vbox_get_type(), &db_browser_info);
    }
  return (db_browser_type);
}

/**
 * gnome_db_browser_new
 * @cnc: the connection to be used for browsing
 *
 * Create a new GnomeDbBrowser widget, binding it to an existing GDA connection
 * if given
 *
 * Returns: a pointer to the new widget if success, NULL if error
 */
GtkWidget *
gnome_db_browser_new (Gda_Connection *cnc)
{
  GtkWidget *brw;

  brw = gtk_type_new(gnome_db_browser_get_type());
  GNOME_DB_BROWSER(brw)->cnc = cnc;
  return (brw);
}

/**
 * gnome_db_browser_destroy
 * @brw: the browser widget to destroy
 *
 * Destroy the widget pointed to by the @brw parameter
 */
static void
gnome_db_browser_destroy (GnomeDbBrowser *brw)
{
  g_return_if_fail(GNOME_DB_IS_BROWSER(brw));
  
  gtk_widget_destroy(brw->object_description);
}

/**
 * gnome_db_browser_get_connection
 * @brw: the browser widget
 *
 * Return the Gda_Connection object actually being used by this browser
 *
 * Returns: a pointer to a Gda_Connection object, or NULL on error
 */
Gda_Connection *
gnome_db_browser_get_connection (GnomeDbBrowser *brw)
{
  g_return_val_if_fail(GNOME_DB_IS_BROWSER(brw), 0);
  return (brw->cnc);
}

/**
 * gnome_db_browser_set_connection
 * @brw: the browser widget
 * @cnc: the Gda_Connection object to be used
 *
 * Dynamically change the Gda_Connection object being used by the GnomeDbBrowser
 * widget
 */
void
gnome_db_browser_set_connection (GnomeDbBrowser *brw, Gda_Connection *cnc)
{
  g_return_if_fail(GNOME_DB_IS_BROWSER(brw));
  g_return_if_fail(IS_GDA_CONNECTION(cnc));
  g_return_if_fail(gda_connection_is_open(cnc));
  
  brw->cnc = cnc;
  
  /* update object list */
  gnome_db_browser_clear(brw);
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(brw->tables_button)))
    show_tables_cb(brw, (gpointer) brw);
  else if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(brw->views_button)))
    show_views_cb(brw, (gpointer) brw);

  gtk_signal_emit(GTK_OBJECT(brw), 
                  gnome_db_browser_signals[GNOME_DB_BROWSER_CONNECTION_CHANGED]);
}

/**
 * gnome_db_browser_clear
 * @brw: the browser widget
 *
 * Clear all widgets in the browser
 */
void
gnome_db_browser_clear (GnomeDbBrowser *brw)
{
  g_return_if_fail(GNOME_DB_IS_BROWSER(brw));
  
  gda_ui_clear_clist(GTK_CLIST(brw->object_list));
  gnome_db_grid_set_recordset(GNOME_DB_GRID(brw->object_description), 0);
}

/**
 * gnome_db_browser_refresh
 * @brw: the browser widget
 *
 * Refresh the browser widgets' contents to display possible changes in
 * the underlying recordset
 */
void
gnome_db_browser_refresh (GnomeDbBrowser *brw)
{
  g_return_if_fail(brw != NULL);
  g_return_if_fail(GNOME_DB_IS_BROWSER(brw));
}
