/* GNOME-DB - GUI front-end
 * Copyright (c) 1998-2000 by Rodrigo Moya
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gdafe.h"

typedef struct
{
  gchar *gda_name;
  gchar *provider;
  gchar *dsn;
  gchar *description;
  gchar *configurator;
} FE_GdaDataSource;

static void dialog_clicked_cb (GnomeDialog *dialog, gint button_number, gpointer data);
static void new_source_cb (GtkWidget *w, gpointer data);
static void remove_source_cb (GtkWidget *w, gpointer data);
static void select_list_cb (GtkWidget *w, gpointer data);

static GtkWidget* l_dialog = NULL;
static GtkWidget* l_list = NULL;
static GtkWidget* l_GdaName = NULL;
static GtkWidget* l_provider = NULL;
static GtkWidget* l_DsnEntry = NULL;
static GtkWidget* l_description = NULL;
static GtkWidget* l_config = NULL;
static GList*     l_dsns = NULL;

/*
 * Private functions
 */
static void
fill_provider_list (void)
{
  const gchar *empty_string = "";
  if (l_dialog != NULL)
    {
      GList *providers = gda_server_list();
      if (providers != NULL)
        {
          GList *node, *strs = NULL;
          while ((node = g_list_first(providers)))
            {
              Gda_Server *server = (Gda_Server *) node->data;
              strs = g_list_append(strs, g_strdup(server->name));
              providers = g_list_remove(providers, (gpointer) server);
              gda_server_free(server);
            }
          gtk_combo_set_popdown_strings(GTK_COMBO(l_provider), strs);
        }
      gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(l_provider)->entry), empty_string);
    }
}

static FE_GdaDataSource *
find_source_by_name (gchar *name)
{
  GList *node = g_list_first(l_dsns);
  g_return_val_if_fail(name != NULL, NULL);
  g_return_val_if_fail(l_dialog != NULL, NULL);
  
  while (node != NULL)
    {
      FE_GdaDataSource *data = (FE_GdaDataSource *) node->data;
      if (data != NULL &&
          !strcmp(name, data->gda_name))
        {
          return (data);
        }
      node = g_list_next(node);
    }
  return (NULL);
}

static gchar*
trim_string(gchar* s)
{
  gint length = strlen(s);

  while (length && isspace(s[--length]))
    ;
  s[length+1] = '\0';
  return s;
}

static GList *
get_data_sources (void)
{
  GList *list = NULL;
  gchar *gda_name, *section_name;
  gpointer gda_iterator;

  gtk_list_clear_items(GTK_LIST(l_list), 0, -1);
  gda_iterator = gnome_config_init_iterator(FE_CONFIG_SECTION_DATASOURCES);
  while ((gda_iterator = gnome_config_iterator_next(gda_iterator, &gda_name, 
                                                    &section_name)))
    {
      gchar *str;
      FE_GdaDataSource *src = g_new0(FE_GdaDataSource, 1);

      str = g_strdup_printf("/gdalib/%s/", trim_string(gda_name));
      gnome_config_push_prefix(str);
      src->gda_name = gda_name;
      src->provider = gnome_config_get_string("Provider");
      src->dsn = gnome_config_get_string("DSN");
      src->description = gnome_config_get_string("Description");
      src->configurator = gnome_config_get_string("Configurator");
      list = g_list_append(list, (gpointer) src);
      fe_add_list_item(GTK_LIST(l_list), src->gda_name);
      gnome_config_pop_prefix();
      g_free((gpointer) str);
    }
  return (list);
}

static void
real_close (void)
{
  if (l_dialog != NULL)
    {
      if (l_dsns != NULL)
        {
          while (l_dsns != NULL)
            {
              gpointer item = l_dsns->data;
              l_dsns = g_list_remove(l_dsns, item);
              g_free(item);
            }
        }
      l_dsns = NULL;
      gnome_dialog_close(GNOME_DIALOG(l_dialog));
      l_dialog = NULL;
    }
}

static void
show_source_data (FE_GdaDataSource *data)
{
  const gchar *empty_string = "";
  
  g_return_if_fail(data != NULL);
  g_return_if_fail(l_dialog != NULL);
  
  gtk_entry_set_text(GTK_ENTRY(l_GdaName), data->gda_name ? data->gda_name : empty_string);
  gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(l_provider)->entry), data->provider ? data->provider : empty_string);
  gtk_entry_set_text(GTK_ENTRY(l_DsnEntry), data->dsn ? data->dsn : empty_string);
  gtk_entry_set_text(GTK_ENTRY(l_description), data->description ? data->description : empty_string);
  gtk_entry_set_text(GTK_ENTRY(l_config), data->configurator ? data->configurator : empty_string);
}

/*
 * Callbacks
 */
static void
dialog_clicked_cb (GnomeDialog * dialog, gint button_number, gpointer data)
{
  switch (button_number) 
    {
    case 0: /* OK button */
      gnome_config_sync();
      real_close();
      break;
    case 1: /* Apply button */
      {
        gchar *gda_name, *provider, *dsn, *desc, *config, *str;
        gda_name = gtk_entry_get_text(GTK_ENTRY(l_GdaName));
        if (gda_name != NULL && strlen(gda_name) > 1)
          {
            provider = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(l_provider)->entry));
            dsn = gtk_entry_get_text(GTK_ENTRY(l_DsnEntry));
            desc = gtk_entry_get_text(GTK_ENTRY(l_description));
            config = gtk_entry_get_text(GTK_ENTRY(l_config));
            
            str = g_strdup_printf("%s/%s", FE_CONFIG_SECTION_DATASOURCES, gda_name);
            gnome_config_set_string(str, gda_name);
            g_free((gpointer) str);
            
            str = g_strdup_printf("/gdalib/%s/Provider", gda_name);
            if (provider) gnome_config_set_string(str, provider);
            g_free((gpointer) str);
            
            str = g_strdup_printf("/gdalib/%s/DSN", gda_name);
            if (dsn) gnome_config_set_string(str, dsn);
            g_free((gpointer) str);
            
            str = g_strdup_printf("/gdalib/%s/Description", gda_name);
            if (desc) gnome_config_set_string(str, desc);
            g_free((gpointer) str);
            
            str = g_strdup_printf("/gdalib/%s/Configurator", gda_name);
            if (config) gnome_config_set_string(str, config);
            g_free((gpointer) str);
            
            get_data_sources();
          }
      }
      break;
    case 2: /* Cancel button */
      gnome_config_drop_all();
      real_close();
      break;
    default:
      g_assert_not_reached();
    };
}
             
static void
new_source_cb (GtkWidget *w, gpointer data)
{
  const gchar *empty_string = "";
  
  g_return_if_fail(l_dialog != 0);
  
  gtk_entry_set_text(GTK_ENTRY(l_GdaName), empty_string);
  gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(l_provider)->entry), empty_string);
  gtk_entry_set_text(GTK_ENTRY(l_DsnEntry), empty_string);
  gtk_entry_set_text(GTK_ENTRY(l_description), empty_string);
  gtk_entry_set_text(GTK_ENTRY(l_config), empty_string);
}

static void
remove_source_cb (GtkWidget *w, gpointer data)
{
  gchar *src = gtk_entry_get_text(GTK_ENTRY(l_GdaName));
  FE_GdaDataSource *dsn = find_source_by_name(src);
  if (dsn != NULL)
    {
      gchar *tmp = g_strdup_printf("/gdalib/%s", dsn->gda_name);
      gnome_config_clean_section(tmp);
      l_dsns = g_list_remove(l_dsns, (gpointer) dsn);
      g_free((gpointer) tmp);
      tmp = g_strdup_printf("/gdalib/Datasources/%s", dsn->gda_name);
      gnome_config_clean_key(tmp);
      g_free((gpointer) tmp);
      
      if (dsn->gda_name != NULL) g_free((gpointer) dsn->gda_name);
      if (dsn->provider != NULL) g_free((gpointer) dsn->provider);
      if (dsn->dsn != NULL) g_free((gpointer) dsn->dsn);
      if (dsn->description != NULL) g_free((gpointer) dsn->description);
      if (dsn->configurator != NULL) g_free((gpointer) dsn->configurator);
      g_free((gpointer) dsn);
      
      get_data_sources();
      new_source_cb(0, 0);
    }
}

static void
select_list_cb (GtkWidget *w, gpointer data)
{
  if (l_dialog != NULL)
    {
      gchar *name = fe_get_list_current_string(GTK_LIST(l_list));
      if (name != NULL)
        {
          FE_GdaDataSource *data = find_source_by_name(name);
          if (data != NULL)
            show_source_data(data);
        }
    }
}

/*
 * Public functions
 */
void
fe_open_gda_config (GtkWidget *w, gpointer data)
{
  GtkWidget *box, *scrolled_window, *table, *label, *button, *frame;
  GList *node;
  if (l_dialog == NULL)
    {
      l_dialog = gnome_dialog_new(_("GDA Config"),
                                  GNOME_STOCK_BUTTON_OK,
                                  GNOME_STOCK_BUTTON_APPLY,
                                  GNOME_STOCK_BUTTON_CLOSE,
                                  NULL);
      gtk_signal_connect(GTK_OBJECT(l_dialog), "clicked",
                         GTK_SIGNAL_FUNC(dialog_clicked_cb),
                         NULL);

      frame = gtk_frame_new(_("Data Sources"));
      gtk_widget_show(frame);
      gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(l_dialog)->vbox), frame, 1, 1, GNOME_PAD);
      
      box = gtk_hbox_new(FALSE, 0);
      gtk_container_add(GTK_CONTAINER(frame), box);
      scrolled_window = gda_ui_new_scrolled_window_widget();
      l_list = fe_new_list_widget();
      gtk_signal_connect(GTK_OBJECT(l_list), "selection_changed",
                         GTK_SIGNAL_FUNC(select_list_cb), NULL);
      l_dsns = get_data_sources();
      gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrolled_window),
                                            l_list);
      gtk_box_pack_start(GTK_BOX(box), scrolled_window, 1, 1, GNOME_PAD);
      
      table = gtk_table_new(6, 2, FALSE);
      gtk_widget_show(table);
      label = gda_ui_new_label_widget(_("GDA Name"));
      gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, 
                       GTK_FILL | GTK_EXPAND, 
                       GTK_FILL, 3, 3);
      l_GdaName = gda_ui_new_entry_widget(0, TRUE);
      gtk_table_attach(GTK_TABLE(table), l_GdaName, 1, 2, 0, 1,
                       GTK_FILL | GTK_EXPAND,
                       GTK_FILL, 3, 3);
                       
      label = gda_ui_new_label_widget(_("Provider"));
      gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2, 
                       GTK_FILL | GTK_EXPAND, 
                       GTK_FILL, 3, 3);
      l_provider = gda_ui_new_combo_widget();
      gtk_table_attach(GTK_TABLE(table), l_provider, 1, 2, 1, 2, 
                       GTK_FILL | GTK_EXPAND, 
                       GTK_FILL, 3, 3);
      fill_provider_list();
      
      label = gda_ui_new_label_widget(_("DSN"));
      gtk_table_attach(GTK_TABLE(table), label, 0, 1, 2, 3,
                       GTK_FILL | GTK_EXPAND,
                       GTK_FILL, 3, 3);
      l_DsnEntry = gda_ui_new_entry_widget(0, TRUE);
      gtk_table_attach(GTK_TABLE(table), l_DsnEntry, 1, 2, 2, 3,
                       GTK_FILL | GTK_EXPAND,
                       GTK_FILL, 3, 3);
      label = gda_ui_new_label_widget(_("Description"));
      gtk_table_attach(GTK_TABLE(table), label, 0, 1, 3, 4,
                       GTK_FILL | GTK_EXPAND,
                       GTK_FILL, 3, 3);
      l_description = gda_ui_new_entry_widget(0, TRUE);
      gtk_table_attach(GTK_TABLE(table), l_description, 1, 2, 3, 4,
                       GTK_FILL | GTK_EXPAND,
                       GTK_FILL, 3, 3);
      label = gda_ui_new_label_widget(_("Config"));
      gtk_table_attach(GTK_TABLE(table), label, 0, 1, 4, 5,
                       GTK_FILL | GTK_EXPAND,
                       GTK_FILL, 3, 3);
      l_config = gda_ui_new_entry_widget(0, TRUE);
      gtk_table_attach(GTK_TABLE(table), l_config, 1, 2, 4, 5,
                       GTK_FILL | GTK_EXPAND,
                       GTK_FILL, 3, 3);
      gtk_box_pack_start(GTK_BOX(box), table, 1, 1, GNOME_PAD);
      gtk_widget_show(box);
      
      box = gtk_hbox_new(TRUE, 0);
      gtk_table_attach(GTK_TABLE(table), box, 0, 2, 5, 6,
                       GTK_FILL | GTK_EXPAND,
                       GTK_FILL, 3, 3);
      button = gda_ui_new_button_widget_with_pixmap(_("New"), GNOME_STOCK_MENU_NEW);
      gtk_signal_connect(GTK_OBJECT(button), "clicked",
                         GTK_SIGNAL_FUNC(new_source_cb), 0);
      gtk_box_pack_start(GTK_BOX(box), button, 0, 0, 0);
      gtk_widget_show(button);
      button = gda_ui_new_button_widget_with_pixmap(_("Delete"), GNOME_STOCK_MENU_TRASH);
      gtk_signal_connect(GTK_OBJECT(button), "clicked",
                         GTK_SIGNAL_FUNC(remove_source_cb), 0);
      gtk_box_pack_start(GTK_BOX(box), button, 0, 0, 0);
      gtk_widget_show(button);
      gtk_widget_show(box);
    }
  gtk_widget_set_usize(l_dialog, 450, 350);
  gtk_widget_show(l_dialog);
}
