/* GNOME-DB - GUI front-end
 * Copyright (c) 1998-2000 by Rodrigo Moya
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gdafe.h"

typedef struct
{
  gchar name[64];
  gchar type[64];
  gchar size[16];
  gchar precision[16];
} FE_FieldInfo;

static void     add_table (Gda_XmlDatabase *db, const gchar *recset_name);
static gint     edit_table (const gchar *table_name);
static gboolean edit_table_check_field (GnomeDialog *dialog);
static void     edit_table_list_fields (Gda_XmlDatabase *db, const gchar *recset,
                                        const gchar *fieldname, gint size, gint scale,
                                        const gchar *type, gpointer user_data);
static void     edit_table_update_field (GnomeDialog *dialog);
static void     fill_type_list (GtkCombo *combo);

static void edit_table_clear_cb (GtkButton *button, gpointer data);
static void edit_table_delete_cb (GtkButton *button, gpointer data);
static void edit_table_new_cb (GtkButton *button, gpointer data);
static void edit_table_next_field_cb (GtkButton *button, gpointer data);
static void edit_table_prev_field_cb (GtkButton *button, gpointer data);

static void clear_designer_cb (GtkWidget *w, gpointer data);
static void close_designer_cb (GtkWidget *w, gpointer data);
static void delete_table_cb (GtkWidget *w, gpointer data);
static void new_file_cb (GtkWidget *w, gpointer data);
static void new_table_cb (GtkWidget *w, gpointer data);
static void open_file_cb (GtkWidget *w, gpointer data);
static void open_table_cb (GtkWidget *w, gpointer data);
static void refresh_designer_cb (GtkWidget *w, gpointer data);
static void remove_field_cb (GtkWidget *w, gpointer data);
static void save_file_cb (GtkWidget *w, gpointer data);
static void select_list_cb (GtkCList *clist, gint row, gint column, GdkEvent *event, gpointer data);

static GtkWidget*       l_view = 0;
static GtkWidget*       l_list = 0;
static GtkWidget*       l_detail = 0;
static GtkWidget*       l_fileentry = 0;
static Gda_XmlDatabase* l_xmldb = 0;
static GnomeUIInfo designertoolbar[] =
{
  { GNOME_APP_UI_ITEM, N_("New"), N_("Create new database design"),
    new_file_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_NEW, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Open"), N_("Open database design from file"),
    open_file_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_OPEN, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Save"), N_("Save database design to file"),
    save_file_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_SAVE, 0, 0, NULL },
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("Clear"), N_("Clear all widgets"),
    clear_designer_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_TRASH, 0, 0, NULL },
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("Close"), N_("Close this window"),
    close_designer_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_CLOSE, 0, 0, NULL },
  GNOMEUIINFO_END
};
static GnomeUIInfo fieldmenu[] =
{
  { GNOME_APP_UI_ITEM, N_("Remove field"), N_("Remove selected fields"),
    remove_field_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_TRASH, 0, 0, NULL },
  GNOMEUIINFO_END
};

/*
 * Private functions
 */
static void
add_table (Gda_XmlDatabase *db, const gchar *recset_name)
{
  static GtkWidget *pixmap = 0;
  gchar *row[2], *empty_string = "";
  
  g_return_if_fail(l_view != 0);
  g_return_if_fail(IS_GDA_XML_DATABASE(db));
  g_return_if_fail(recset_name != 0);
  
  row[0] = empty_string;
  row[1] = recset_name;
  gtk_clist_append(GTK_CLIST(l_list), row);

  /* create shared pixmap */
  if (!pixmap)
    {
      pixmap = gnome_stock_pixmap_widget(l_list, GNOME_STOCK_MENU_BOOK_BLUE);
      if (!pixmap) return;
      gtk_widget_show(pixmap);
    }
  gtk_clist_set_pixmap(GTK_CLIST(l_list), GTK_CLIST(l_list)->rows - 1, 0,
                       GNOME_PIXMAP(pixmap)->pixmap,
                       GNOME_PIXMAP(pixmap)->mask);
}

static gint
edit_table (const gchar *table_name)
{
  GtkWidget *dialog, *table, *button, *entry, *label;
  gint btn, current_record = 0, total_records = 0;
  gchar *str;
  GList *node;

  /* create dialog box */
  dialog = gnome_dialog_new(table_name != 0 ? _("Open Table") : _("New Table"), 
                            GNOME_STOCK_BUTTON_OK,
                            GNOME_STOCK_BUTTON_CANCEL,
                            0);
  table = gtk_table_new(8, 4, FALSE);
  gtk_widget_show(table);
  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), table, 1, 1, GNOME_PAD);
  
  label = gda_ui_new_label_widget(_("Table name"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   GTK_FILL,
                   3, 3);
  entry = gda_ui_new_entry_widget(0, TRUE);
  gtk_object_set_data(GTK_OBJECT(dialog), "FE_Designer_TableName", (gpointer) entry);
  gtk_table_attach(GTK_TABLE(table), entry, 1, 3, 0, 1,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   GTK_FILL,
                   3, 3);
  if (table_name) gtk_entry_set_text(GTK_ENTRY(entry), table_name);

  /* add top buttons */
  button = gda_ui_new_button_widget_with_pixmap(_("Previous"), GNOME_STOCK_MENU_BACK);
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
                     GTK_SIGNAL_FUNC(edit_table_prev_field_cb), (gpointer) dialog);
  gtk_table_attach(GTK_TABLE(table), button, 1, 2, 1, 2,
                   GTK_FILL,
                   GTK_FILL,
                   3, 3);
                       
  button = gda_ui_new_button_widget_with_pixmap(_("Next"), GNOME_STOCK_MENU_FORWARD);
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
                     GTK_SIGNAL_FUNC(edit_table_next_field_cb), (gpointer) dialog);
  gtk_table_attach(GTK_TABLE(table), button, 2, 3, 1, 2,
                   GTK_FILL,
                   GTK_FILL,
                   3, 3);
      
  /* add field description widgets */
  label = gda_ui_new_label_widget(_("Field Name"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 2, 3,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   GTK_FILL,
                   3, 3);
  entry = gda_ui_new_entry_widget(0, TRUE);
  gtk_object_set_data(GTK_OBJECT(dialog), "FE_Designer_FieldName", (gpointer) entry);
  gtk_table_attach(GTK_TABLE(table), entry, 1, 4, 2, 3,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   GTK_FILL,
                   3, 3);
  label = gda_ui_new_label_widget(_("Type"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 3, 4,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   GTK_FILL,
                   3, 3);
  entry = gda_ui_new_combo_widget();
  fill_type_list(GTK_COMBO(entry));
  gtk_object_set_data(GTK_OBJECT(dialog), "FE_Designer_FieldType", (gpointer) entry);
  gtk_table_attach(GTK_TABLE(table), entry, 1, 4, 3, 4,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   GTK_FILL,
                   3, 3);
  label = gda_ui_new_label_widget(_("Size"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 4, 5,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   GTK_FILL,
                   3, 3);
  entry = gda_ui_new_entry_widget(0, TRUE);
  gtk_object_set_data(GTK_OBJECT(dialog), "FE_Designer_FieldSize", (gpointer) entry);
  gtk_table_attach(GTK_TABLE(table), entry, 1, 4, 4, 5,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   GTK_FILL,
                   3, 3);
  label = gda_ui_new_label_widget(_("Precision"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 5, 6,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   GTK_FILL,
                   3, 3);
  entry = gda_ui_new_entry_widget(0, TRUE);
  gtk_object_set_data(GTK_OBJECT(dialog), "FE_Designer_FieldPrecision", 
                      (gpointer) entry);
  gtk_table_attach(GTK_TABLE(table), entry, 1, 4, 5, 6,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   GTK_FILL,
                   3, 3);
                   
  /* add bottom button bar */
  button = gda_ui_new_button_widget_with_pixmap(_("New"), GNOME_STOCK_MENU_NEW);
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
                     GTK_SIGNAL_FUNC(edit_table_new_cb), (gpointer) dialog);
  gtk_table_attach(GTK_TABLE(table), button, 0, 1, 6, 7,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   GTK_FILL,
                   3, 3);
  button = gda_ui_new_button_widget_with_pixmap(_("Delete"), GNOME_STOCK_MENU_TRASH);
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
                     GTK_SIGNAL_FUNC(edit_table_delete_cb), (gpointer) dialog);
  gtk_table_attach(GTK_TABLE(table), button, 1, 2, 6, 7,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   GTK_FILL,
                   3, 3);
  button = gda_ui_new_button_widget_with_pixmap(_("Update"), GNOME_STOCK_MENU_REDO);
  gtk_table_attach(GTK_TABLE(table), button, 2, 3, 6, 7,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   GTK_FILL,
                   3, 3);
  button = gda_ui_new_button_widget_with_pixmap(_("Clear"), GNOME_STOCK_MENU_UNDO);
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
                     GTK_SIGNAL_FUNC(edit_table_clear_cb), (gpointer) dialog);
  gtk_table_attach(GTK_TABLE(table), button, 3, 4, 6, 7,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   GTK_FILL,
                   3, 3);
                   
  /* add information label */
  if (table_name != 0)
    {
      gda_xml_recordset_list_fields(l_xmldb, table_name, edit_table_list_fields, 
                                      (gpointer) dialog);
      current_record = 1;
    }
  str = g_strdup_printf(_("Record %d of %d"), current_record, total_records);
  label = gda_ui_new_label_widget(str);
  gtk_object_set_data(GTK_OBJECT(dialog), "FE_Designer_InfoLabel", (gpointer) label);
  gtk_misc_set_alignment(GTK_MISC(label), 0.5, 0.5);
  g_free((gpointer) str);
  gtk_table_attach(GTK_TABLE(table), label, 0, 4, 7, 8,
                   GTK_FILL | GTK_EXPAND | GTK_FILL,
                   GTK_FILL,
                   3, 3);
  
  /* finally run dialog box */
  edit_table_update_field(GNOME_DIALOG(dialog));
  btn = gnome_dialog_run(GNOME_DIALOG(dialog));
  if (btn == 0) /* 'Ok' button */
    {
      GtkWidget *entry;
      gchar *table_name;
      GList *node;
      
      entry = (GtkWidget *) gtk_object_get_data(GTK_OBJECT(dialog),
                                                "FE_Designer_TableName");
      table_name = gtk_entry_get_text(GTK_ENTRY(entry));
      if (entry != 0 || table_name != 0)
        {
          gda_xml_recordset_remove(l_xmldb, table_name);
      
          /* create again the recordset */
          gda_xml_recordset_new(l_xmldb, table_name);
          node = g_list_first((GList *) gtk_object_get_data(GTK_OBJECT(dialog),
                                                            "FE_Designer_FieldList"));
          while (node)
            {
              FE_FieldInfo *fld_info = (FE_FieldInfo *) node->data;
          
              if (fld_info)
                {
                  gda_xml_recordset_add_field(l_xmldb, table_name, fld_info->name,
                                              atoi(fld_info->size), atoi(fld_info->precision),
                                              gda_string_2_fieldtype(fld_info->type));
                }
              node = g_list_next(node);
            }
        }
        
      /* refresh lists */
      refresh_designer_cb(0, 0);
    }
  
  /* free fields list */
  node = g_list_first((GList *) gtk_object_get_data(GTK_OBJECT(dialog),
                                                    "FE_Designer_FieldList"));
  if (node)
    {
      g_list_foreach(node, (GFunc) g_free, 0);
      g_list_free(node);
    }
    
  /* close dialog and finish */
  gnome_dialog_close(GNOME_DIALOG(dialog));
  return (btn);
}

static gboolean
edit_table_check_field (GnomeDialog *dialog)
{
  gchar *str;
  GList *node;
  FE_FieldInfo *field_info;
  
  g_return_val_if_fail(GNOME_IS_DIALOG(dialog), FALSE);
  
  /* check current record */
  node = (GList *) gtk_object_get_data(GTK_OBJECT(dialog), "FE_Designer_FieldList");
  if (node == 0) return (FALSE);
  field_info = (FE_FieldInfo *) node->data;
    
  /* check and upgrade field name */
  str = gtk_entry_get_text(GTK_ENTRY(gtk_object_get_data(GTK_OBJECT(dialog),
                                                         "FE_Designer_FieldName")));
  if (str == 0 || strlen(str) < 1)
    {
      fe_status_bar_message(_("Field name cannot be empty"));
      return (FALSE);
    }
  strcpy(field_info->name, str);
  
  return (TRUE);
}

static void
edit_table_list_fields (Gda_XmlDatabase *db,
                        const gchar *recset,
                        const gchar *fieldname,
                        gint size,
                        gint scale,
                        const gchar *type,
                        gpointer user_data)
{
  GList *list;
  FE_FieldInfo *fld_info;
  GtkWidget *dialog = GTK_WIDGET(user_data);
  
  g_return_if_fail(GNOME_IS_DIALOG(dialog));
  
  list = (GList *) gtk_object_get_data(GTK_OBJECT(dialog), "FE_Designer_FieldList");
  fld_info = g_new0(FE_FieldInfo, 1);
  strcpy(fld_info->name, fieldname);
  strcpy(fld_info->type, type);
  sprintf(fld_info->size, "%d", size);
  sprintf(fld_info->precision, "%d", scale);
  list = g_list_append(list, (gpointer) fld_info);
  gtk_object_set_data(GTK_OBJECT(dialog), "FE_Designer_FieldList", 
                      (gpointer) g_list_first(list));
}

static void
edit_table_update_field (GnomeDialog *dialog)
{
  GList *node;
  gchar *str;
  GtkWidget *label;
  
  g_return_if_fail(GNOME_IS_DIALOG(dialog));
  
  node = (GList *) gtk_object_get_data(GTK_OBJECT(dialog), "FE_Designer_FieldList");
  if (node)
    {
      FE_FieldInfo *field_info = (FE_FieldInfo *) node->data;
      if (field_info != 0)
        {        
          gtk_entry_set_text(GTK_ENTRY(gtk_object_get_data(GTK_OBJECT(dialog),
                                                           "FE_Designer_FieldName")),
                                                           field_info->name);
          gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(gtk_object_get_data(GTK_OBJECT(dialog),
                                                           "FE_Designer_FieldType"))->entry),
                                                           field_info->type);
          gtk_entry_set_text(GTK_ENTRY(gtk_object_get_data(GTK_OBJECT(dialog),
                                                           "FE_Designer_FieldSize")),
                                                           field_info->size);
          gtk_entry_set_text(GTK_ENTRY(gtk_object_get_data(GTK_OBJECT(dialog),
                                                           "FE_Designer_FieldPrecision")),
                                                           field_info->precision);
        }
    }
  else
    {
      /* clean up all fields */
      edit_table_clear_cb(0, (gpointer) dialog);
    }
    
  /* update record-information label */
  label = gtk_object_get_data(GTK_OBJECT(dialog), "FE_Designer_InfoLabel");
  if (GTK_IS_LABEL(label))
    {
      str = g_strdup_printf(_("Record %d of %d"),
                            node ? g_list_index(g_list_first(node), node->data) + 1 : 0,
                            node ? g_list_length(g_list_first(node)) : 0);
      gtk_label_set(GTK_LABEL(label), str);
      g_free((gpointer) str);
    }
}

static void
fill_type_list (GtkCombo *combo)
{
  GList *list = 0;
  register gint cnt;
  
  g_return_if_fail(GTK_IS_COMBO(combo));
  
  for (cnt = GDA_TypeNull; cnt < GDA_TypeLastValue; cnt++)
    {
      gchar *str = 0;
      
      str = gda_fieldtype_2_string(str, 256, cnt);
      if (str != 0)
        list = g_list_append(list, (gpointer) str);
    }
  gtk_combo_set_popdown_strings(GTK_COMBO(combo), list);
}

static gchar *
get_current_table (void)
{
  GList *tmp;
  gchar *table = 0;
  guint row;
  
  g_return_val_if_fail(l_view != 0, 0);
  g_return_val_if_fail(l_list != 0, 0);
  
  tmp = GTK_CLIST(l_list)->selection;
  if (tmp)
    {
      row = GPOINTER_TO_UINT(tmp->data);
      gtk_clist_get_text(GTK_CLIST(l_list), row, 1, &table);
    }
  return (table);
}

static void
real_close (GtkWidget *w, gpointer data)
{
  g_return_if_fail(l_view != 0);
  
  if (l_xmldb != 0)
    {
      gda_xml_database_free(l_xmldb);
    }
  l_xmldb = 0;
  l_view = 0;
}

/*
 * Callbacks
 */
static void
clear_designer_cb (GtkWidget *w, gpointer data)
{
  gchar *empty_string = "";
  
  g_return_if_fail(l_view != 0);
  
  if (l_xmldb != 0)
    {
      gda_xml_database_free(l_xmldb);
      l_xmldb = 0;
    }
  gda_ui_clear_clist(GTK_CLIST(l_list));
  gda_ui_clear_clist(GTK_CLIST(l_detail));
  gtk_entry_set_text(GTK_ENTRY(l_fileentry), empty_string);
}

static void
close_designer_cb (GtkWidget *w, gpointer data)
{
  g_return_if_fail(l_view != 0);
  
  fe_destroy_view(l_view);
}

static void
delete_table_cb (GtkWidget *w, gpointer data)
{
  gchar *table = 0;
  
  g_return_if_fail(l_view != 0);
  
  table = get_current_table();
  if (table != 0 && IS_GDA_XML_DATABASE(l_xmldb))
    {
      gda_xml_recordset_remove(l_xmldb, (const gchar *) table);
      refresh_designer_cb(w, data);
    }
}

static void
edit_table_clear_cb (GtkButton *button, gpointer data)
{
  GnomeDialog *dialog = GNOME_DIALOG(data);
  if (GNOME_IS_DIALOG(dialog))
    {
      const gchar *empty_string = "";
      
      gtk_entry_set_text(GTK_ENTRY(gtk_object_get_data(GTK_OBJECT(dialog),
                                                       "FE_Designer_FieldName")),
                                                       empty_string);
      gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(gtk_object_get_data(GTK_OBJECT(dialog),
                                                       "FE_Designer_FieldType"))->entry),
                                                       empty_string);
      gtk_entry_set_text(GTK_ENTRY(gtk_object_get_data(GTK_OBJECT(dialog),
                                                       "FE_Designer_FieldSize")),
                                                       empty_string);
      gtk_entry_set_text(GTK_ENTRY(gtk_object_get_data(GTK_OBJECT(dialog),
                                                       "FE_Designer_FieldPrecision")),
                                                       empty_string);
    }
}

static void
edit_table_delete_cb (GtkButton *button, gpointer data)
{
  GnomeDialog *dialog = GNOME_DIALOG(data);
  if (GNOME_IS_DIALOG(dialog))
    {
      GList *node = (GList *) gtk_object_get_data(GTK_OBJECT(dialog),
                                                  "FE_Designer_FieldList");
      if (node != 0)
        {
          gpointer data = node->data;
          
          /* remove field from list */
          node = g_list_remove(g_list_first(node), data);
          g_free(data);
          
          gtk_object_set_data(GTK_OBJECT(dialog), "FE_Designer_FieldList", (gpointer) node);
          edit_table_update_field(dialog);
        }
    }
}

static void
edit_table_new_cb (GtkButton *button, gpointer data)
{
  GnomeDialog *dialog = GNOME_DIALOG(data);
  if (GNOME_IS_DIALOG(dialog))
    {
      FE_FieldInfo *field_info;
      GList *list = g_list_first((GList *) gtk_object_get_data(GTK_OBJECT(dialog),
                                                               "FE_Designer_FieldList"));
      
      /* allocate memory for new field */
      field_info = g_new0(FE_FieldInfo, 1);
      list = g_list_append(list, (gpointer) field_info);
      
      gtk_object_set_data(GTK_OBJECT(dialog), "FE_Designer_FieldList", 
                          (gpointer) g_list_last(list));
      edit_table_update_field(dialog);
    }
}

static void
edit_table_next_field_cb (GtkButton *button, gpointer data)
{
  GnomeDialog *dialog = GNOME_DIALOG(data);
  if (GNOME_IS_DIALOG(dialog))
    {
      GList *node = (GList *) gtk_object_get_data(GTK_OBJECT(dialog),
                                                  "FE_Designer_FieldList");
      if (node != 0 && edit_table_check_field(dialog))
        {
          node = g_list_next(node);
          if (node != 0)
            {
              gtk_object_set_data(GTK_OBJECT(dialog), "FE_Designer_FieldList",
                                  (gpointer) node);
            }
          edit_table_update_field(dialog);
        }
    }
}

static void
edit_table_prev_field_cb (GtkButton *button, gpointer data)
{
  GnomeDialog *dialog = GNOME_DIALOG(data);
  if (GNOME_IS_DIALOG(dialog))
    {
      GList *node = (GList *) gtk_object_get_data(GTK_OBJECT(dialog),
                                                  "FE_Designer_FieldList");
      if (node != 0 && edit_table_check_field(dialog))
        {
          node = g_list_previous(node);
          if (node != 0)
            {
              gtk_object_set_data(GTK_OBJECT(dialog), "FE_Designer_FieldList",
                                  (gpointer) node);
            }
          edit_table_update_field(dialog);
        }
    }
}

static void
new_file_cb (GtkWidget *w, gpointer data)
{
  g_return_if_fail(l_view != 0);
  
  /* first ask user to save current design */
  if (l_xmldb)
    {
      if (fe_yes_no_dialog(_("Do you want to save changes?")))
        {
          gchar *filename;
          gboolean free_mem = FALSE;
          
          filename = gtk_entry_get_text(GTK_ENTRY(l_fileentry));
          if (!filename)
            {
              filename = fe_get_file_name(_("Select File"));
              free_mem = TRUE;
            }
            
          /* save if user said so */
          if (filename)
            {
              if (gda_xml_database_save_to_file(l_xmldb, filename) == -1)
                {
                  gda_ui_show_error(_("Error saving database design to file %s"),
                                    filename);
                }
            }
          if (free_mem && filename) g_free((gpointer) filename);
          gda_xml_database_free(l_xmldb);
        }
        
      /* clear all widgets */
      clear_designer_cb(w, data);
    }
    
  /* create new XML database */
  l_xmldb = gda_xml_database_new(0);
}

static void
new_table_cb (GtkWidget *w, gpointer data)
{
  g_return_if_fail(l_view != 0);
  
  if (edit_table(0) == 0)
    {
    }
}

static void
open_file_cb (GtkWidget *w, gpointer data)
{
  gchar *filename;
  
  g_return_if_fail(l_view != 0);
  
  filename = fe_get_file_name(_("Open database"));
  if (filename != 0)
    {
      Gda_XmlDatabase *db = gda_xml_database_load_from_file(filename);
      if (db != 0)
        {
          if (l_xmldb != 0)
            {
              gda_xml_database_free(l_xmldb);
            }
          l_xmldb = db;

          refresh_designer_cb(0, 0);
          gtk_entry_set_text(GTK_ENTRY(l_fileentry), filename);
        }
      else gda_ui_show_error(_("Error opening file '%s'"), filename);
      g_free((gpointer) filename);
    }
}

static void
open_table_cb (GtkWidget *w, gpointer data)
{
  g_return_if_fail(l_view != 0);
  
  if (get_current_table() != 0)
    {
      if (edit_table(get_current_table()) == 0)
        {
        }
    }
}

static void
refresh_designer_cb (GtkWidget *w, gpointer data)
{
  g_return_if_fail(l_view != 0);
  
  gda_ui_clear_clist(GTK_CLIST(l_list));
  gda_ui_clear_clist(GTK_CLIST(l_detail));
  if (l_xmldb != 0)
    {
      gda_xml_database_list_recordsets(l_xmldb, (GFunc) add_table);
    }
}

static void
remove_field_cb (GtkWidget *w, gpointer data)
{
  GList *selected;
  gchar *table = get_current_table();
  
  g_return_if_fail(l_view != 0);
  g_return_if_fail(table != 0);
  g_return_if_fail(GTK_IS_CLIST(l_detail));
  g_return_if_fail(IS_GDA_XML_DATABASE(l_xmldb));
  
  selected = GTK_CLIST(l_detail)->selection;
  while (selected != 0)
    {
      guint row = GPOINTER_TO_UINT(selected->data);
      gchar *field = 0;
      
      /* first, remove the field from the XML database definition */
      gtk_clist_get_text(GTK_CLIST(l_detail), row, 1, &field);
      if (field != 0)
        {
          gda_xml_recordset_remove_field(l_xmldb, table, field);
          
          /* then, remove the item from the GtkCList */
          gtk_clist_remove(GTK_CLIST(l_detail), GPOINTER_TO_UINT(selected->data));
        }
      selected = g_list_next(selected);
    }
    
  /* refresh the sreen widgets */
  select_list_cb(GTK_CLIST(l_list), GPOINTER_TO_UINT(GTK_CLIST(l_list)->selection->data), 0, 0, 0);
}

static void
save_file_cb (GtkWidget *w, gpointer data)
{
  gchar *filename;
  gboolean free_mem = FALSE;
  
  g_return_if_fail(l_view != 0);
  
  if (!l_xmldb) gda_ui_show_error(_("First open or create a design"));
  
  filename = gtk_entry_get_text(GTK_ENTRY(l_fileentry));
  if (!filename)
    {
      filename = fe_get_file_name(_("Select File"));
      if (!filename) return;
      free_mem = TRUE;
    }
    
  /* save database design */
  if (gda_xml_database_save_to_file(l_xmldb, filename) == -1)
    {
      gda_ui_show_error(_("Error saving database design to file %s"), filename);
      return;
    }
  gtk_entry_set_text(GTK_ENTRY(l_fileentry), filename);
  if (free_mem) g_free((gpointer) filename);
}

static void
list_field_cb (Gda_XmlDatabase *db, const gchar *recset, const gchar *fieldname,
               gint size, gint scale, const gchar *type, gpointer user_data)
{
  static GtkWidget *pixmap = 0;
  gchar *cols[5], *empty_string = "";
  
  g_return_if_fail(l_view != 0);
  g_return_if_fail(fieldname != 0);
  
  cols[0] = empty_string;
  cols[1] = fieldname;
  cols[2] = type;
  cols[3] = g_strdup_printf("%d", size);
  cols[4] = g_strdup_printf("%d", scale);

  gtk_clist_append(GTK_CLIST(l_detail), cols);
  
  /* create shared pixmap */
  if (!pixmap)
    {
      pixmap = gnome_stock_pixmap_widget(l_detail, GNOME_STOCK_MENU_MIC);
      if (!pixmap) return;
      gtk_widget_show(pixmap);
    }
  gtk_clist_set_pixmap(GTK_CLIST(l_detail), GTK_CLIST(l_detail)->rows - 1, 0,
                       GNOME_PIXMAP(pixmap)->pixmap,
                       GNOME_PIXMAP(pixmap)->mask);
  
  g_free((gpointer) cols[3]);
  g_free((gpointer) cols[4]);
}

static void
select_list_cb (GtkCList *clist, gint row, gint column, GdkEvent *event, gpointer data)
{
  gchar *recset = 0;
  
  g_return_if_fail(l_view != 0);
  g_return_if_fail(l_xmldb != 0);
  
  gtk_clist_get_text(GTK_CLIST(l_list), row, 1, &recset);
  if (recset != 0)
    {
      gda_ui_clear_clist(GTK_CLIST(l_detail));
      gda_xml_recordset_list_fields(l_xmldb, recset, list_field_cb, 0);
      fe_status_bar_message(_("Got description for table '%s'"), recset);
    }
}

/*
 * Public functions
 */
void
fe_open_designer (GtkWidget *w, gpointer data)
{
  static const gchar *detail_cols[] =
  {
    "", N_("Name"), N_("Type"), N_("Size"), N_("Precision")
  };
  if (l_view == NULL)
    {
      GtkWidget *box, *button, *scroll, *box2, *paned, *table;
      
      l_view = fe_new_view(_("Designer"), 1, 2, designertoolbar);
      fe_set_view_destroy_func(l_view, (FE_DestroyViewFunc) real_close, NULL);
      
      /* create children */
      box = gtk_hbox_new(FALSE, 0);
      fe_add_widget_to_view(l_view, 0, 1, 1, 2, box);
      
      box2 = gtk_vbox_new(FALSE, 0);
      button = gda_ui_new_button_widget_with_pixmap(_("New"), GNOME_STOCK_MENU_SEARCH);
      gtk_button_set_relief(GTK_BUTTON(button), GTK_RELIEF_NONE);
      gtk_signal_connect(GTK_OBJECT(button), "clicked",
                         GTK_SIGNAL_FUNC(new_table_cb), (gpointer) 0);
      gtk_box_pack_start(GTK_BOX(box2), button, 0, 0, 0);

      button = gda_ui_new_button_widget_with_pixmap(_("Edit"), GNOME_STOCK_MENU_PROP);
      gtk_button_set_relief(GTK_BUTTON(button), GTK_RELIEF_NONE);
      gtk_signal_connect(GTK_OBJECT(button), "clicked",
                         GTK_SIGNAL_FUNC(open_table_cb), (gpointer) 0);
      gtk_box_pack_start(GTK_BOX(box2), button, 0, 0, 0);

      button = gda_ui_new_button_widget_with_pixmap(_("Delete"), GNOME_STOCK_MENU_TRASH);
      gtk_button_set_relief(GTK_BUTTON(button), GTK_RELIEF_NONE);
      gtk_signal_connect(GTK_OBJECT(button), "clicked",
                         GTK_SIGNAL_FUNC(delete_table_cb), (gpointer) 0);
      gtk_box_pack_start(GTK_BOX(box2), button, 0, 0, 0);
      
      button = gda_ui_new_button_widget_with_pixmap(_("Refresh"), GNOME_STOCK_MENU_REFRESH);
      gtk_button_set_relief(GTK_BUTTON(button), GTK_RELIEF_NONE);
      gtk_signal_connect(GTK_OBJECT(button), "clicked",
                         GTK_SIGNAL_FUNC(refresh_designer_cb), (gpointer) 0);
      gtk_box_pack_start(GTK_BOX(box2), button, 0, 0, 0);
      
      gtk_widget_show(box2);
      gtk_box_pack_start(GTK_BOX(box), box2, 0, 0, GNOME_PAD);
      
      paned = gtk_vpaned_new();
      gtk_widget_show(paned);
      gtk_box_pack_start(GTK_BOX(box), paned, 1, 1, GNOME_PAD);
      
      table = gtk_table_new(3, 4, FALSE);
      gtk_widget_show(table);
      gtk_paned_add1(GTK_PANED(paned), table);
      
      scroll = gda_ui_new_scrolled_window_widget();
      l_list = gda_ui_new_clist_widget(0, 2);
      gtk_signal_connect(GTK_OBJECT(l_list), "select_row",
                         GTK_SIGNAL_FUNC(select_list_cb), 0);
      gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scroll), l_list);
      gtk_table_attach(GTK_TABLE(table), scroll, 0, 2, 0, 3,
                       GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                       GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                       3, 3);
      
      l_fileentry = gda_ui_new_entry_widget(0, TRUE);
      gtk_table_attach(GTK_TABLE(table), l_fileentry, 2, 3, 0, 1,
                       GTK_FILL,
                       GTK_FILL,
                       3, 3);
      button = fe_new_browse_button_widget(GTK_ENTRY(l_fileentry));
      gtk_table_attach(GTK_TABLE(table), button, 3, 4, 0, 1,
                       0,
                       0,
                       3, 3);
      
      box2 = gtk_vbox_new(FALSE, 0);
      scroll = gda_ui_new_scrolled_window_widget();
      l_detail = gda_ui_new_clist_widget((gchar *) detail_cols,
                                         sizeof(detail_cols) / sizeof(detail_cols[0]));
      gda_ui_new_popup_menu(l_detail, fieldmenu, 0);
      gtk_container_add(GTK_CONTAINER(scroll), l_detail);
      gtk_box_pack_start(GTK_BOX(box2), scroll, 1, 1, GNOME_PAD);
      gtk_widget_show(box2);
      gtk_paned_add2(GTK_PANED(paned), box2);
    }

  /* create empty XML database */
  l_xmldb = gda_xml_database_new(0);

  fe_display_view(l_view);
}
