/* GNOME DB
 * Copyright (C) 1998 Michael Lausch
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <memory.h>

#include "odbc.h"
#include "sqldriver.h"

#include "gda-command.h"
#include "gda-recset.h"
#include "gda-error.h"
#include "gda-connection.h"
#include "gda-field.h"
#include "gda-error.h"
#include "gda-odbc-misc.h"

Gda_ODBC_Command*
gda_odbc_cmd_new(void)
{
  Gda_ODBC_Command* rc = g_new0(Gda_ODBC_Command, 1);
  return rc;
}


void
gda_odbc_cmd_free(Gda_ODBC_Command* cmd)
{
  GList* ptr;
  
  if (cmd->cmd)
    g_free(cmd->cmd);

  cmd->cmd = 0;

  ptr = cmd->recsets;
  while(ptr)
    {
      gda_odbc_recset_free(ptr->data);
      ptr = g_list_next(ptr);
    }
  g_list_free(cmd->recsets);
  cmd->recsets = 0;
  g_free(cmd);
	
}



Gda_ODBC_Connection*
gda_odbc_cmd_set_connection(Gda_ODBC_Command* cmd,
			    Gda_ODBC_Connection* cnc)
{
  Gda_ODBC_Connection* rc = cmd->cnc;
  fprintf(stderr,"gda_odbc_cmd_set_connection: cmd = %p, connection = %p\n", cmd, cnc);
  cmd->cnc = cnc;
  return rc;
}





Gda_ODBC_Recordset*
gda_odbc_describe_recset(Gda_ODBC_Recordset* rec)
{
  gint   rc;
  gshort ncols;
  gint   i;
  gchar  colname[128];
  gshort colnamelen;
  
  rc = SQLNumResultCols(rec->hstmt, &ncols);
  if (rc != SQL_SUCCESS)
    {
      gda_odbc_error_make(rec, 0, "SQLNumResultCols");
      g_log("GDA ODBC", G_LOG_LEVEL_DEBUG, "SQLNumResultCols error\n");
      return 0;
    }
  if (ncols == 0)
    return 0;

  for (i = 0; i < ncols; i++)
    {
      Gda_ODBC_Field* f;

      f = gda_odbc_field_new();
      f->defined_length = 0;
      rc = SQLDescribeCol(rec->hstmt,
			  i+1,
			  colname, sizeof(colname), &colnamelen,
			  &f->sql_type,
			  &f->defined_length,
			  &f->num_scale,
			  &f->nullable);
      fprintf(stderr,"Describing field idx %d: colname = '%s', def_size(precission) = %ld, scale = %d\n",
	      i, colname, f->defined_length, f->num_scale);
      if (rc == SQL_ERROR)
	{
	  g_log("GDA ODBC", G_LOG_LEVEL_DEBUG, "SQLDescribeCol error\n");
	  gda_odbc_error_make(rec, 0, "SQLDescribeCol");
	  return 0;
	}
      colname[colnamelen] = '\0';
      f->name = g_strdup(colname);
      rec->fields = g_list_append(rec->fields, f);
    }
  return rec;
}





Gda_ODBC_Recordset*
gda_odbc_cmd_execute(Gda_ODBC_Command* cmd, const GDA_CmdParameterSeq* params,
		     gulong* affected_rows, gulong options)
{
  gshort                 nparams;
  gint                   rc;
  int                    i;
  static long            sizes[1000];
  Gda_ODBC_Recordset*    recset;
  GDA_CmdParameter*      current_parameter;

  recset = gda_odbc_recset_new();
  recset->hstmt = 0;
  recset->cmd = cmd;
  
  rc = SQLAllocStmt(cmd->cnc->hdbc, &recset->hstmt);
  
  if (rc != SQL_SUCCESS)
    {
      fprintf(stderr,"gda_odbc_cmd_execute: SQLAllocStmt: hstmt is now %p\n", recset->hstmt);
      gda_odbc_error_make(recset, 0, "SQLAllocStmt");
      gda_odbc_recset_free(recset);
      return 0;
    }
#if 0
  rc = SQLSetScrollOptions(recset->hstmt, SQL_CONCUR_ROWVER,  SQL_SCROLL_KEYSET_DRIVEN, 20);
  if (rc != SQL_SUCCESS)
    {
      gda_odbc_error_make(error, recset, 0, "SQLPrepare");
      gda_odbc_recset_free(recset);
      return 0;
    }
#endif
  rc = SQLPrepare(recset->hstmt, cmd->cmd, SQL_NTS);
  if (rc == SQL_ERROR)
    {
      gda_odbc_error_make(recset, 0, "SQLPrepare");
      gda_odbc_recset_free(recset);
      return 0;
    }
  rc = SQLNumParams(recset->hstmt, &nparams);
  if (rc == SQL_ERROR)
    {
      gda_odbc_error_make(recset, 0, "SQLNumParams");
      gda_odbc_recset_free(recset);
      return 0;
    }
  if (params && ((nparams && params->_length == 0) || (!nparams && params->_length)))
    {
      if (nparams)
	{
	  g_log("GDA ODBC", G_LOG_LEVEL_ERROR,"Statement needs parameters, but none passed\n");
	}
      else
	{
	  g_log("GDA ODBC", G_LOG_LEVEL_ERROR, "Statement doesn't need parameters, but params availble\n");
	}
    }
  if (nparams)
    {
      
      for (i = 0; i < nparams; i++)
	{
	  gushort sqlType;
	  gulong precision;
	  gshort scale;
	  gshort nullable;
	  unsigned long ctype;

	  current_parameter = &(params->_buffer[i]);
	  
	  if (current_parameter->value._d)
	    {
	      g_warning ("NULL PARAMETER NYI");
	      abort();
	    }

	  ctype = gda_odbc_gda2ctype(current_parameter->value._u.v._d);
	  
	  rc = SQLDescribeParam(recset->hstmt,
				i + 1,
				&sqlType,
				&precision,
				&scale,
				&nullable);
	  if (rc != SQL_SUCCESS)
	    {
	      gda_odbc_error_make(recset, 0, "SQLNumParams");
	      gda_odbc_recset_free(recset);
	      return 0;
	    }
	  sizes[i] = value_2_inputsize(&current_parameter->value._u.v);
	  rc = SQLBindParameter(recset->hstmt,
				i + 1,
				SQL_PARAM_INPUT,
				ctype,
				sqlType,
				precision,
				scale,
				value_2_ptr(&current_parameter->value._u.v),
				0,
				&sizes[i]
				);
	  
	  if (rc != SQL_SUCCESS)
	    {
	      gda_odbc_error_make(recset, 0, "SQLNumParams");
	      gda_odbc_recset_free(recset);
	      return 0;
	    }
	}
    }
  rc = SQLExecute(recset->hstmt);
  for (i = 0; i < nparams; i++)
    {
      fprintf(stderr,"gda_odbc_cmd_execute: after sizes[%d] = %ld\n", i,
	      sizes[i]);
    }
  if (rc == SQL_ERROR)
    {
      gda_odbc_error_make(recset, 0, "SQLExecute");
      gda_odbc_recset_free(recset);
      return 0;
    }
  return gda_odbc_describe_recset(recset);
}




  
