/* GNOME-DB
 * Copyright (c) 1998 by Rodrigo Moya
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gda-postgres-connection.h"

/* private functions */
static void
initialize_schema_ops (void)
{
}

/* create new connection object */
Gda_POSTGRES_Connection *
gda_postgres_connection_new (void)
{
  static gint initialized = 0;
  Gda_POSTGRES_Connection *c = g_new0(Gda_POSTGRES_Connection, 1);
  if (!initialized)
    {
      initialize_schema_ops();
      initialize = 1;
    }
  /* initialize everything to NULL to avoid core dumped */
  c->pq_host = NULL;
  c->pq_port = NULL;
  c->pq_options = NULL;
  c->pq_tty = NULL;
  c->pq_db = NULL;
  c->pq_login = NULL;
  c->pq_pwd = NULL;
  return (c);
}

/* free given connection object */
void
gda_postgres_connection_free (Gda_POSTGRES_Connection *c)
{
  if (c->pq_host != NULL) g_free((gpointer) c->pq_host);
  if (c->pq_port != NULL) g_free((gpointer) c->pq_port);
  if (c->pq_options != NULL) g_free((gpointer) c->pq_options);
  if (c->pq_tty != NULL) g_free((gpointer) c->pq_tty);
  if (c->pq_db != NULL) g_free((gpointer) c->pq_db);
  if (c->pq_login != NULL) g_free((gpointer) c->pq_login);
  if (c->pq_pwd != NULL) g_free((gpointer) c->pq_pwd);
  g_free((gpointer) c);
}

/* open new connection to database server */
static gchar*
get_value(gchar* ptr)
{
  while (*ptr && *ptr != '=')
    ptr++;
  if (!*ptr)
    return 0;
  ptr++;
  if (!*ptr)
    return 0;
  while (*ptr && isspace(*ptr))
    ptr++;
  return (g_strdup(ptr));
}

gint
gda_postgres_connection_open (Gda_POSTGRES_Connection *cnc, const gchar *dsn,
			      const gchar *user, const gchar *password)
{
  gchar *ptr_s, *ptr_e;
  /* get options from connection string */
  ptr_s = dsn;
  while (ptr_s && *ptr_s)
    {
      ptr_e = strchr(ptr_s, ';');
      if (ptr_e) *ptr_e = '\0';
      if (strcasecmp(ptr_s, "HOST") == 0)
	cnc->pq_host = get_value(ptr_s);
      else if (strncasecmp(ptr_s, "DATABASE", strlen("DATABASE")) == 0)
	cnc->pq_db = get_value(ptr_s);
      else if (strncasecmp(ptr_s, "PORT", strlen("PORT")) == 0)
	cnc->pq_port = get_value(ptr_s);
      else if (strncasecmp(ptr_s, "OPTIONS", strlen("OPTIONS")) == 0)
	cnc->pq_options = get_value(ptr_s);
      else if (strncasecmp(ptr_s, "TTY", strlen("TTY")) == 0)
	cnc->pq_tty = get_value(ptr_s);
      else if (strncasecmp(ptr_s, "LOGIN", strlen("LOGIN")) == 0)
	cnc->pq_login = get_value(ptr_s);
      else if (strncasecmp(ptr_s, "PASSWORD", strlen("PASSWORD")) == 0)
	cnc->pq_pwd = get_value(ptr_s);
      ptr_s = ptr_e;
      if (ptr_s)
	ptr_s++;
    }
  if (cnc->pq_login == NULL) 
    cnc->pq_login = user != NULL ? g_strdup(user) : NULL;
  if (cnc->pq_pwd)
    cnc->pq_pwd = password != NULL ? g_strdup(password) : NULL;
  /* actually establish the connection */
  cnc->pq_conn = PQsetdbLogin(cnc->pq_host, cnc->pq_port, cnc->pq_options,
			      cnc->pq_tty, cnc->pq_db, cnc->pq_login, cnc->pq_pwd);
  if (PQstatus(cnc->pq_conn) != CONNECTION_OK)
    {
      fprintf(stderr, "PostgreSQL error: %s\n", PQerrorMessage(cnc->pq_conn));
      return (-1);
    }
  return (0);
}

/* close given connection to database */
gint
gda_postgres_connection_close (Gda_POSTGRES_Connection *c)
{
  /* check connection status */
  if (PQstatus(c->pq_conn) == CONNECTION_OK)
    {
      PQfinish(c->pq_conn);
      c->pq_conn = NULL;
      return (0);
    }
  return (-1);
}
