/* GNOME DB libary
 * Copyright (C) 1998,1999 Michael Lausch
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gnome.h>

#include <gda.h>
#include "gda-command.h"
#include "gda-recordset.h"


typedef struct _Parameter
{
  gchar* name;
  GDA_Value* value;
  GDA_ParameterDirection inout;
}Parameter;


static void
release_connection_object(Gda_Command* cmd, Gda_Connection* cnc)
{
  cmd->connection->commands = g_list_remove(cmd->connection->commands, cmd);
}

/*
 * this function will change as soon as the client side
 * error objects are here
 */


static gint
Exception(CORBA_Environment *ev)
{
  switch (ev->_major)
    {
    case CORBA_SYSTEM_EXCEPTION:
      fprintf(stderr,"CORBA system exception %s.\n", CORBA_exception_id(ev));
      return -1;
    case CORBA_USER_EXCEPTION:
      fprintf(stderr,"CORBA user exception %s.\n", CORBA_exception_id(ev));
      return -1;
    default:
    }
  return 0;
}

/**
 * gda_command_new:
 *
 * This function allocates the memory for a command object.
 *
 * Returns: a pointer to the command object
 */

Gda_Command*
gda_command_new(void)
{
  Gda_Command* cmd;

  cmd = g_new0(Gda_Command, 1);
  return cmd;
}


/**
 * gda_command_free:
 * @cmd: The command object which should be deallocated.
 *
 * This function frees the memory of command object and
 * cuts the association with its connection object.
 *
 */

void
gda_command_free(Gda_Command* cmd)
{
  CORBA_Environment ev;
  
  g_return_if_fail(cmd != NULL);

  if (cmd->connection && cmd->connection->commands)
    release_connection_object(cmd, cmd->connection);
  else
    {
      if (cmd->connection && !cmd->connection->commands)
	g_error("gda_command_free: connection object has no command list");
    }
  if (cmd->text)
    g_free(cmd->text);

  if (cmd->command != CORBA_OBJECT_NIL)
    {
      CORBA_exception_init(&ev);
      CORBA_Object_release(cmd->command, &ev);
      Exception(&ev);
    }
  g_free(cmd);
}

/**
 * gda_command_set_connection:
 * @cmd: The command object
 * @cnc: The connection object
 *
 * Associates a connection with a command. All functions with this
 * command will use the connection to talk to the data source. If the
 * command is already associated with a connection, this association
 * is destroyed.  
 *
 * Returns: -1 on error, 0 on success
 */
gint
gda_command_set_connection(Gda_Command* cmd, Gda_Connection* cnc)
{
  CORBA_Environment ev;

  g_return_val_if_fail(cmd != NULL, -1);
  g_return_val_if_fail(cnc != NULL, -1);
  g_return_val_if_fail(cnc->connection != NULL, -1);
  
  if (cmd->connection)
    {
      release_connection_object(cmd, cmd->connection);
    }
  cmd->connection = cnc;
  CORBA_exception_init(&ev);
  
  cmd->command = GDA_Connection_createCommand(cnc->connection, &ev);
  if (Exception(&ev) < 0)
    {
      cmd->connection = 0;
      return -1;
    }
  cmd->connection->commands = g_list_append(cmd->connection->commands, cmd);
  return 0;
}
  

/**
 * gda_command_get_connection:
 * @cmd: the command object
 *
 * Returns the #gda_Connection object which is used by the command.
 *
 * Returns: a pointer to the #gda_Connection object
 */
Gda_Connection*
gda_command_get_connection(Gda_Command* cmd)
{
  g_return_val_if_fail(cmd != NULL, 0);

  return cmd->connection;
}


/**
 * gda_command_set_cmd:
 * @cmd: the #Gda_Command object
 * @text: the command to perform. There are some special texts
 * which are reckognized by the servers. See the server documantation
 * for a list of special commands.
 *
 * Sets the command which is executed when the gda_command_execute()
 * function is called.
 *
 */

void
gda_command_set_text(Gda_Command* cmd, gchar* text)
{
  g_return_if_fail(cmd != NULL);
  
  if (cmd->text)
    g_free(cmd->text);
  
  cmd->text = g_strdup(text);
}

/**
 * gda_command_get_cmd:
 * @cmd: the #Gda_Command object
 *
 * Gets the command string which is executed when the gda_command_execute()
 * function is called.
 *
 * Returns: a reference to the command string.
 */

gchar*
gda_command_get_text(Gda_Command* cmd)
{
  g_return_val_if_fail(cmd != NULL, NULL);

  return cmd->text;
}


#define ENUM_TO_STR(e) case (e): strncpy(bfr, #e, length); break

gchar* 
gda_gdatype_2_string(gchar* bfr, gint length, GDA_ValueType t)
{
  if (!bfr)
    {
      bfr = g_new0(gchar, 20);
      length = 20;
    }
  switch (t)
    {
      ENUM_TO_STR(GDA_TypeNull);
      ENUM_TO_STR(GDA_TypeBigint);
      ENUM_TO_STR(GDA_TypeBinary);
      ENUM_TO_STR(GDA_TypeBoolean);
      ENUM_TO_STR(GDA_TypeBstr);
      ENUM_TO_STR(GDA_TypeChar);
      ENUM_TO_STR(GDA_TypeCurrency);
      ENUM_TO_STR(GDA_TypeDate);
      ENUM_TO_STR(GDA_TypeDbDate);
      ENUM_TO_STR(GDA_TypeDbTime);
      ENUM_TO_STR(GDA_TypeDbTimestamp);
      ENUM_TO_STR(GDA_TypeDecimal);
      ENUM_TO_STR(GDA_TypeDouble);
      ENUM_TO_STR(GDA_TypeError);
      ENUM_TO_STR(GDA_TypeInteger);
      ENUM_TO_STR(GDA_TypeLongvarbin);
      ENUM_TO_STR(GDA_TypeLongvarchar);
      ENUM_TO_STR(GDA_TypeLongvarwchar);
      ENUM_TO_STR(GDA_TypeNumeric);
      ENUM_TO_STR(GDA_TypeSingle);
      ENUM_TO_STR(GDA_TypeSmallint);
      ENUM_TO_STR(GDA_TypeTinyint);
      ENUM_TO_STR(GDA_TypeUBigint);
      ENUM_TO_STR(GDA_TypeUSmallint);
      ENUM_TO_STR(GDA_TypeVarchar);
      ENUM_TO_STR(GDA_TypeVarbin);
      ENUM_TO_STR(GDA_TypeVarwchar);
      ENUM_TO_STR(GDA_TypeFixchar);
      ENUM_TO_STR(GDA_TypeFixbin);
      ENUM_TO_STR(GDA_TypeFixwchar);
    }
  return bfr;
}


/**
 * gda_command_execute:
 * @cmd: the command object
 * @reccount: a pointer to a gulong variable. In this variable the number
 * of affected records is stored.
 * @flags: flags to categorize the command.
 *
 * This function executes the command which has been set with
 * gda_command_set_text(). It returns a #Gda_Recordset pointer which holds the
 * results from this query. If the command doesn't return any results, like
 * insert, or updaste statements in SQL, an empty result set is returnd.
 *
 * Returns: a pointer to a recordset or a NULL pointer if there was an error.
 */

/* Currently it's not possible to use command objects without an association
   with a connection oject. This conmstraint will be released if the cmd->text
   string is parsed to extract the provider, data source name, user, and
   password.
   Then a temporary connection object for this command object will be
   constructed and used.
*/

Gda_Recordset*
gda_command_execute(Gda_Command* cmd, gulong* reccount, gulong flags)
{
  CORBA_Environment         ev;
  GDA_CmdParameterSeq*      corba_parameters;
  gint                      parameter_count;
  CORBA_Object              corba_rs;
  CORBA_long                corba_reccount;
  Gda_Recordset*            rs;
  GList*                    ptr;
  GDA_CmdParameter*         corba_parameter;
  gint                      idx;
  
  g_return_val_if_fail(cmd != NULL, 0);
  g_return_val_if_fail(cmd->text != NULL, 0);
  g_return_val_if_fail(reccount != NULL, 0);
  g_return_val_if_fail(cmd->connection != NULL, 0);
  
  CORBA_exception_init(&ev);
  
  corba_parameters = GDA_CmdParameterSeq__alloc();
  parameter_count = cmd->parameters ? g_list_length(cmd->parameters) : 0;
  corba_parameters->_buffer = CORBA_sequence_GDA_CmdParameter_allocbuf(parameter_count);
  corba_parameters->_length = parameter_count;

  if (parameter_count)
    {
      ptr = cmd->parameters;
      idx = 0;
      while(ptr)
	{
	  GDA_Value* value;
	  Parameter* parameter;
	  
	  parameter = ptr->data;
	  corba_parameter = &(corba_parameters->_buffer[idx]);

	  corba_parameter->dir = parameter->inout;
	  if (parameter->name)
	    {
	      corba_parameter->name = CORBA_string_dup(parameter->name);
	    }
	  else
	    corba_parameter->name = 0;

	  value = parameter->value;
	  if (!(corba_parameter->value._d = value ? 0 : 1))
	    {
	      corba_parameter->value._u.v = *((*parameter).value);
	    }
	  else
	    {
	      fprintf(stderr,"Got NULL param value\n");
	    }
	  ptr = g_list_next(ptr);
	  idx++;
	}
    }
  corba_rs = GDA_Command_execute(cmd->command, cmd->text, (CORBA_long*)&corba_reccount,
				 corba_parameters, flags, &ev);
  if (!corba_rs)
    return 0;
  *reccount = corba_reccount;
  if (Exception(&ev) < 0)
    return 0;
  rs = gda_recordset_new();
  rs->external_cmd = cmd;
  rs->corba_rs = corba_rs;
  rs->eof = 0;
  rs->bof = 0;
  rs->open = 1;

  rs->field_attributes = GDA_Recordset_describe(corba_rs, &ev);
  if (Exception(&ev) || rs->field_attributes == CORBA_OBJECT_NIL)
    {
      fprintf(stderr," cannot get row description\n");
    }
#if 0
  for (idx = 0; idx  < rs->field_attributes->_length; idx++)
    {
      gchar type_bfr[120];
      fprintf(stderr,"Description for field '%s'\n", rs->field_attributes->_buffer[idx].name);
      fprintf(stderr,"     defined Size   = %d\n", rs->field_attributes->_buffer[idx].definedSize);
      fprintf(stderr,"     scale          = %d\n", rs->field_attributes->_buffer[idx].scale);
      fprintf(stderr,"     gdaType        = '%s'\n", gda_gdatype_2_string(type_bfr, sizeof(type_bfr),
									    rs->field_attributes->_buffer[idx].gdaType));
      fprintf(stderr,"     cType          = %d\n", rs->field_attributes->_buffer[idx].cType);
      fprintf(stderr,"     nativeType     = %d\n", rs->field_attributes->_buffer[idx].nativeType);
    }
#endif
  return rs;
}


/**
 * gda_command_create_parameter:
 * @cmd: the command object
 * @name: the name of the parameter
 * @inout: direction of the parameter
 * @value: value of the parameter
 *
 * This function creates a new parameter for the command. This is 
 * important if you want to use parameterized statements like
 * "select * from table where key = ?"
 * and substitute the value for key at a later time. It also comes
 * handy if you don't want to convert each parameter to it's string
 * representation (floating point values).
 *
 * The @inout parameter defines if the @value parameter is used to
 * pass a value to the statement or to receive a value. In the exampel 
 * abov the @input parameter will have a value of #PARAM_IN. The value 
 * #PARAM_OUT is used for output parameters when the command executes
 * a precoedure.
 *
 * @value points to a GDA_Value variable (there will be helper functions
 * to create such varbales from the basic C types). This variable must
 * hold a valid value when the gda_command_execute() function is called
 * and the @inout parameter is either #PARAM_IN or #PARAM_INOUT.
 * If the @inoput parameter is #PARAM_OUT, memory is allocated by the
 * library and the user must free it using CORBA_free().
 *
 * IMPORTANT: use the CORBA functions to allocate the memory for the
 * value parameter if it's not a simple data type (integer, float, ...)
 */
void
gda_command_create_parameter(Gda_Command* cmd, gchar* name, GDA_ParameterDirection inout,
			     GDA_Value* value)
{
  Parameter* param = g_new0(Parameter, 1);

  param->name = g_strdup(name);
  param->inout = inout;
  param->value = value;

  cmd->parameters = g_list_append(cmd->parameters, param);
}



  
      
