/* GnomENIUS Calculator
 * Copyright (C) 1997 George Lebl.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <string.h>
#include <gmp.h>
#include <glib.h>
#include <math.h>
#include "calc.h"
#include "util.h"
#include "mymath.h"

#include "mpwrap.h"

extern void (*errorout)(char *);
extern error_t error_num;

/*************************************************************************/
/*low level stuff prototypes                                             */
/*************************************************************************/

/*my own power function for floats, very simple :) */
static void mympf_pow_ui(mpf_t rop,mpf_t op,unsigned long int e);

/*clear extra variables of type type, if type=op->type nothing is done*/
static int mpwl_clear_extra_type(MpwRealNum *op,int type);

/*only set the type, don't free it, and don't set the type variable
  create an extra type of the variable for temporary use*/
static void mpwl_make_extra_type(MpwRealNum *op,int type);

static void mpwl_make_type(MpwRealNum *op,int type);

/*this only adds a value of that type but does nto clear the old one!
  retuns the new extra type set*/
static int mpwl_make_same_extra_type(MpwRealNum *op1,MpwRealNum *op2);

static void mympq_set_f(mpq_t rop,mpf_t op);
/*make new type and clear the old one*/
static void mpwl_make_same_type(MpwRealNum *op1,MpwRealNum *op2);

static void mpwl_clear(MpwRealNum *op);

static void mpwl_init_type(MpwRealNum *op,int type);

static void mpwl_free(MpwRealNum *op);

static int mpwl_sgn(MpwRealNum *op);

static void mpwl_set_si(MpwRealNum *rop,signed long int i);
static void mpwl_set_ui(MpwRealNum *rop,unsigned long int i);
static void mpwl_set_d(MpwRealNum *rop,double d);

static void mpwl_set(MpwRealNum *rop,MpwRealNum *op);

static void mpwl_add(MpwRealNum *rop,MpwRealNum *op1,MpwRealNum *op2);

static void mpwl_mul(MpwRealNum *rop,MpwRealNum *op1,MpwRealNum *op2);
static void mpwl_mul_ui(MpwRealNum *rop,MpwRealNum *op,unsigned long int i);

static void mpwl_div(MpwRealNum *rop,MpwRealNum *op1,MpwRealNum *op2);
static void mpwl_div_ui(MpwRealNum *rop,MpwRealNum *op,unsigned long int i);
static void mpwl_ui_div_ui(MpwRealNum *rop,unsigned long int i,MpwRealNum *op);

static void mpwl_mod(MpwRealNum *rop,MpwRealNum *op1,MpwRealNum *op2);

static void mpwl_neg(MpwRealNum *rop,MpwRealNum *op);

static void mpwl_fac_ui(MpwRealNum *rop,unsigned int op);

static void mpwl_fac(MpwRealNum *rop,MpwRealNum *op);

static void mpwl_pow_q(MpwRealNum *rop,MpwRealNum *op1,MpwRealNum *op2);

/*power to an unsigned long and optionaly invert the answer*/
static void mpwl_pow_ui(MpwRealNum *rop,MpwRealNum *op1,unsigned int e,
	int reverse);

static void mpwl_pow_z(MpwRealNum *rop,MpwRealNum *op1,MpwRealNum *op2);

static void mpwl_pow_f(MpwRealNum *rop,MpwRealNum *op1,MpwRealNum *op2);

static void mpwl_pow(MpwRealNum *rop,MpwRealNum *op1,MpwRealNum *op2);

static int mpwl_cmp(MpwRealNum *op1, MpwRealNum *op2);

static int mpwl_cmp_ui(MpwRealNum *op, unsigned long int i);

static void mpwl_make_int(MpwRealNum *rop, int floats);

/*make number into a float, this might be neccessary for unprecise
  calculations*/
static void mpwl_make_float(MpwRealNum *rop);

static void mpwl_set_str_float(MpwRealNum *rop,char *s,int base);
static void mpwl_set_str_int(MpwRealNum *rop,char *s,int base);

/**************/
/*output stuff*/

/*round off the number at some digits*/
static void str_make_max_digits(char *s,int digits);

/*formats a floating point with mantissa in p and exponent in e*/
static char * str_format_float(char *p,long int e,int scientific_notation);

static char * str_getstring_z(mpz_t num, int max_digits,
	int scientific_notation);
static char * str_getstring_q(mpq_t num, int max_digits,
	int scientific_notation);
static char * str_getstring_f(mpf_t num, int max_digits,
	int scientific_notation);

static char * mpwl_getstring(MpwRealNum * num, int max_digits,
	int scientific_notation, int results_as_floats);

/*************************************************************************/
/*low level stuff                                                        */
/*************************************************************************/

/*my own power function for floats, very simple :) */
static void
mympf_pow_ui(mpf_t rop,mpf_t op,unsigned long int e)
{
	mpf_t fr;

	if(rop==op) {
		mpf_init(fr);
		mpf_set_ui(fr,1);
		for(;e>0;e--)
			mpf_mul(fr,fr,op);
		mpf_set(rop,fr);
		mpf_clear(fr);
	} else { /*different numbers safe to do a quicker way*/
		mpf_set_ui(rop,1);
		for(;e>0;e--)
			mpf_mul(rop,rop,op);
	}
}


/*clear extra variables of type type, if type=op->type nothing is done*/
static int
mpwl_clear_extra_type(MpwRealNum *op,int type)
{
	if(op->type==type)
		return;
	switch(type) {
		case MPW_FLOAT: mpf_clear(op->data.fval); break;
		case MPW_RATIONAL: mpq_clear(op->data.rval); break;
		case MPW_INTEGER: mpz_clear(op->data.ival); break;
	}
}

static void
mympq_set_f(mpq_t rop,mpf_t op)
{
	char *s;
	long int e;

	s=mpf_get_str(NULL,&e,10,0,op);
	e-=strlen(s);
	if(e>0) {
		s=my_realloc(s,strlen(s)+1,strlen(s)+e+1);
		for(;e>0;e--)
			strcat(s,"0");
	}
	mpz_set_str(mpq_numref(rop),s,10);
	mpz_set_ui(mpq_denref(rop),10);
	mpz_pow_ui(mpq_denref(rop),mpq_denref(rop),-e);

	g_free(s);

	mpq_canonicalize(rop);

}

/*only set the type, don't free it, and don't set the type variable
  create an extra type of the variable for temporary use*/
static void
mpwl_make_extra_type(MpwRealNum *op,int type)
{
	mpz_t ir;
	if(op->type==type)
		return;
	switch(type) {
		case MPW_FLOAT:
			mpf_init(op->data.fval);
			if(op->type==MPW_INTEGER)
				mpf_set_z(op->data.fval,op->data.ival);
			else if(op->type==MPW_RATIONAL)
				mpf_set_q(op->data.fval,op->data.rval);
			break;
		case MPW_RATIONAL:
			mpq_init(op->data.rval);
			if(op->type==MPW_INTEGER)
				mpq_set_z(op->data.rval,op->data.ival);
			else if(op->type==MPW_FLOAT)
				mympq_set_f(op->data.rval,op->data.fval);
			break;
		case MPW_INTEGER:
			mpz_init(op->data.ival);
			if(op->type==MPW_FLOAT)
				mpz_set_f(op->data.ival,op->data.fval);
			else if(op->type==MPW_RATIONAL)
				mpz_set_q(op->data.ival,op->data.rval);
			break;
	}
}

static void
mpwl_make_type(MpwRealNum *op,int type)
{
	int t;

	if(op->type==type)
		return;
	t=op->type;
	mpwl_make_extra_type(op,type);
	op->type=type;
	mpwl_clear_extra_type(op,t);
}

/*this only adds a value of that type but does nto clear the old one!
  retuns the new extra type set*/
static int
mpwl_make_same_extra_type(MpwRealNum *op1,MpwRealNum *op2)
{
	if(op1->type==op2->type)
		return op1->type;
	else if(op1->type > op2->type) {
		mpwl_make_extra_type(op2,op1->type);
		return op1->type;
	} else { /*if(op1->type < op2->type)*/
		mpwl_make_extra_type(op1,op2->type);
		return op2->type;
	}
}

/*make new type and clear the old one*/
static void
mpwl_make_same_type(MpwRealNum *op1,MpwRealNum *op2)
{
	if(op1->type==op2->type)
		return;
	else if(op1->type > op2->type)
		mpwl_make_type(op2,op1->type);
	else /*if(op1->type < op2->type)*/
		mpwl_make_type(op1,op2->type);
}

static void
mpwl_clear(MpwRealNum *op)
{
	if(!op) return;

	switch(op->type) {
		case MPW_INTEGER: mpz_clear(op->data.ival); break;
		case MPW_RATIONAL: mpq_clear(op->data.rval); break;
		case MPW_FLOAT: mpf_clear(op->data.fval); break;
	}
}

static void
mpwl_init_type(MpwRealNum *op,int type)
{
	if(!op) return;

	op->type=type;

	switch(type) {
		case MPW_INTEGER:
			mpz_init(op->data.ival);
			break;
		case MPW_RATIONAL:
			mpq_init(op->data.rval);
			break;
		case MPW_FLOAT:
			mpf_init(op->data.fval);
			break;
	}
}

static void
mpwl_free(MpwRealNum *op)
{
	if(!op) return;
	mpwl_clear(op);
	g_free(op);
}

static int
mpwl_sgn(MpwRealNum *op)
{
	switch(op->type) {
		case MPW_FLOAT: return mpf_sgn(op->data.fval);
		case MPW_RATIONAL: return mpq_sgn(op->data.rval);
		case MPW_INTEGER: return mpz_sgn(op->data.ival);
	}
	return 0;
}

static int
mpwl_cmp(MpwRealNum *op1, MpwRealNum *op2)
{
	int r=0;
	int t;

	t=mpwl_make_same_extra_type(op1,op2);
	switch(t) {
		case MPW_FLOAT:
			r=mpf_cmp(op1->data.fval,op2->data.fval);
			break;
		case MPW_RATIONAL:
			r=mpq_cmp(op1->data.rval,op2->data.rval);
			break;
		case MPW_INTEGER:
			r=mpz_cmp(op1->data.ival,op2->data.ival);
			break;
	}
	mpwl_clear_extra_type(op1,t);
	mpwl_clear_extra_type(op2,t);
	return r;
}

static int
mpwl_cmp_ui(MpwRealNum *op, unsigned long int i)
{
	switch(op->type) {
		case MPW_FLOAT: return mpf_cmp_ui(op->data.fval,i);
		case MPW_RATIONAL: return mpq_cmp_ui(op->data.rval,i,1);
		case MPW_INTEGER: return mpz_cmp_ui(op->data.ival,i);
	}
	return 0;
}

static void
mpwl_set_d(MpwRealNum *rop,double d)
{
	switch(rop->type) {
		case MPW_FLOAT:
			mpf_set_d(rop->data.fval,d);
			break;
		case MPW_RATIONAL:
		case MPW_INTEGER:
			mpwl_clear(rop);
			mpwl_init_type(rop,MPW_FLOAT);
			mpf_set_d(rop->data.fval,d);
			break;
	}
}

static void
mpwl_set_si(MpwRealNum *rop,signed long int i)
{
	switch(rop->type) {
		case MPW_FLOAT:
			mpwl_clear(rop);
			mpwl_init_type(rop,MPW_INTEGER);
			mpz_set_si(rop->data.ival,i);
			break;
		case MPW_RATIONAL:
			mpq_set_si(rop->data.rval,i,1);
			break;
		case MPW_INTEGER:
			mpz_set_si(rop->data.ival,i);
			break;
	}
}

static void
mpwl_set_ui(MpwRealNum *rop,unsigned long int i)
{
	switch(rop->type) {
		case MPW_FLOAT:
			mpwl_clear(rop);
			mpwl_init_type(rop,MPW_INTEGER);
			mpz_set_ui(rop->data.ival,i);
			break;
		case MPW_RATIONAL:
			mpq_set_ui(rop->data.rval,i,1);
			break;
		case MPW_INTEGER:
			mpz_set_ui(rop->data.ival,i);
			break;
	}
}

static void
mpwl_set(MpwRealNum *rop,MpwRealNum *op)
{
	if(rop==op)
		return;
	else if(rop->type==op->type) {
		switch(op->type) {
			case MPW_FLOAT:
				mpf_set(rop->data.fval,op->data.fval);
				break;
			case MPW_RATIONAL:
				mpq_set(rop->data.rval,op->data.rval);
				mpwl_make_int(rop,FALSE);
				break;
			case MPW_INTEGER:
				mpz_set(rop->data.ival,op->data.ival);
				break;
		}
	} else {
		mpwl_clear(rop);
		mpwl_init_type(rop,op->type);
		mpwl_set(rop,op);
	}
}

static void
mpwl_add(MpwRealNum *rop,MpwRealNum *op1,MpwRealNum *op2)
{
	int t;
	MpwRealNum r;

	t=mpwl_make_same_extra_type(op1,op2);

	mpwl_init_type(&r,t);

	switch(t) {
		case MPW_FLOAT:
			mpf_add(r.data.fval,op1->data.fval,op2->data.fval);
			break;
		case MPW_RATIONAL:
			mpq_add(r.data.rval,op1->data.rval,op2->data.rval);
			mpwl_make_int(&r,FALSE);
			break;
		case MPW_INTEGER:
			mpz_add(r.data.ival,op1->data.ival,op2->data.ival);
			break;
	}
	mpwl_clear_extra_type(op1,t);
	mpwl_clear_extra_type(op2,t);
	mpwl_set(rop,&r);
	mpwl_clear(&r);
}

static void
mpwl_mul(MpwRealNum *rop,MpwRealNum *op1,MpwRealNum *op2)
{
	int t;
	MpwRealNum r;

	t=mpwl_make_same_extra_type(op1,op2);

	mpwl_init_type(&r,t);

	switch(t) {
		case MPW_FLOAT:
			mpf_mul(r.data.fval,op1->data.fval,op2->data.fval);
			break;
		case MPW_RATIONAL:
			mpq_mul(r.data.rval,op1->data.rval,op2->data.rval);
			mpwl_make_int(&r,FALSE);
			break;
		case MPW_INTEGER:
			mpz_mul(r.data.ival,op1->data.ival,op2->data.ival);
			break;
	}
	mpwl_clear_extra_type(op1,t);
	mpwl_clear_extra_type(op2,t);
	mpwl_set(rop,&r);
	mpwl_clear(&r);
}

static void
mpwl_mul_ui(MpwRealNum *rop,MpwRealNum *op,unsigned long int i)
{
	if(rop->type!=op->type) {
		mpwl_clear(rop);
		mpwl_init_type(rop,op->type);
	}

	switch(op->type) {
		case MPW_FLOAT:
			mpf_mul_ui(rop->data.fval,op->data.fval,i);
			break;
		case MPW_RATIONAL:
			mpz_mul_ui(mpq_numref(rop->data.rval),
				mpq_numref(op->data.rval),i);
			mpwl_make_int(rop,FALSE);
			break;
		case MPW_INTEGER:
			mpz_mul_ui(rop->data.ival,op->data.ival,i);
			break;
	}
}

static void
mpwl_div(MpwRealNum *rop,MpwRealNum *op1,MpwRealNum *op2)
{
	int t;
	MpwRealNum r;

	if(mpwl_sgn(op2)==0) {
		error_num=NUMERICAL_MPW_ERROR;
		return;
	}

	t=mpwl_make_same_extra_type(op1,op2);

	switch(t) {
		case MPW_FLOAT:
			mpwl_init_type(&r,t);
			mpf_div(r.data.fval,op1->data.fval,op2->data.fval);
			break;
		case MPW_RATIONAL:
			mpwl_init_type(&r,t);
			mpq_div(r.data.rval,op1->data.rval,op2->data.rval);
			mpwl_make_int(&r,FALSE);
			break;
		case MPW_INTEGER:
			mpwl_init_type(&r,MPW_RATIONAL);
			mpq_set_z(r.data.rval,op1->data.ival);
			mpz_set(mpq_denref(r.data.rval),
				op2->data.ival);
			break;
	}
	mpwl_clear_extra_type(op1,t);
	mpwl_clear_extra_type(op2,t);
	mpwl_set(rop,&r);
	mpwl_clear(&r);

}

static void
mpwl_div_ui(MpwRealNum *rop,MpwRealNum *op,unsigned long int i)
{
	int t;
	if(i==0) {
		error_num=NUMERICAL_MPW_ERROR;
		return;
	}

	switch(op->type) {
		case MPW_FLOAT:
			if(rop->type!=MPW_FLOAT) {
				mpwl_clear(rop);
				mpwl_init_type(rop,MPW_FLOAT);
			}
			mpf_div_ui(rop->data.fval,op->data.fval,i);
			break;
		case MPW_RATIONAL:
			if(rop->type!=MPW_RATIONAL) {
				mpwl_clear(rop);
				mpwl_init_type(rop,MPW_RATIONAL);
			}
			mpz_mul_ui(mpq_denref(rop->data.rval),
				mpq_denref(op->data.rval),i);
			mpwl_make_int(rop,FALSE);
			break;
		case MPW_INTEGER:
			t=rop->type;
			if(rop->type!=MPW_RATIONAL)
				mpwl_init_type(rop,MPW_RATIONAL);
			mpq_set_z(rop->data.rval,op->data.ival);
			mpz_set_ui(mpq_denref(rop->data.rval),i);
			mpwl_clear_extra_type(rop,t);
			break;
	}
}

static void
mpwl_ui_div(MpwRealNum *rop,unsigned long int i,MpwRealNum *op)
{
	int t;
	if(mpwl_sgn(op)==0) {
		error_num=NUMERICAL_MPW_ERROR;
		return;
	}

	switch(op->type) {
		case MPW_FLOAT:
			if(rop->type!=MPW_FLOAT) {
				mpwl_clear(rop);
				mpwl_init_type(rop,MPW_FLOAT);
			}
			mpf_ui_div(rop->data.fval,i,op->data.fval);
			break;
		case MPW_RATIONAL:
			if(rop->type!=MPW_RATIONAL) {
				mpwl_clear(rop);
				mpwl_init_type(rop,MPW_RATIONAL);
			}
			mpq_inv(rop->data.rval,op->data.rval);
			mpz_mul_ui(mpq_numref(rop->data.rval),
				mpq_numref(rop->data.rval),i);
			mpwl_make_int(rop,FALSE);
			break;
		case MPW_INTEGER:
			t=rop->type;
			if(rop->type!=MPW_RATIONAL)
				mpwl_init_type(rop,MPW_RATIONAL);
			mpz_set_ui(mpq_numref(rop->data.rval),i);
			mpz_set(mpq_denref(rop->data.rval),op->data.ival);
			mpwl_clear_extra_type(rop,t);
			break;
	}
}

static void
mpwl_mod(MpwRealNum *rop,MpwRealNum *op1,MpwRealNum *op2)
{
	if(mpwl_sgn(op2)==0) {
		error_num=NUMERICAL_MPW_ERROR;
		return;
	}

	if(op1->type==MPW_INTEGER && op2->type==MPW_INTEGER) {
			if(rop->type!=MPW_INTEGER) {
				mpwl_clear(rop);
				mpwl_init_type(rop,MPW_INTEGER);
			}
			mpz_mod(rop->data.ival,op1->data.ival,op2->data.ival);
	} else {
		(*errorout)("Can't do modulo of floats or rationals!");
		error_num=NUMERICAL_MPW_ERROR;
	}
}

static void
mpwl_neg(MpwRealNum *rop,MpwRealNum *op)
{
	if(rop->type!=op->type) {
		mpwl_clear(rop);
		mpwl_init_type(rop,op->type);
	}

	switch(op->type) {
		case MPW_FLOAT:
			mpf_neg(rop->data.fval,op->data.fval);
			break;
		case MPW_RATIONAL:
			mpq_neg(rop->data.rval,op->data.rval);
			break;
		case MPW_INTEGER:
			mpz_neg(rop->data.ival,op->data.ival);
			break;
	}
}

static void
mpwl_fac_ui(MpwRealNum *rop,unsigned int op)
{
	if(rop->type!=MPW_INTEGER) {
		mpwl_clear(rop);
		mpwl_init_type(rop,MPW_INTEGER);
	}
	mpz_fac_ui(rop->data.ival,op);
}

static void
mpwl_fac(MpwRealNum *rop,MpwRealNum *op)
{
	if(op->type!=MPW_INTEGER) {
		(*errorout)("Can't do factorials of rationals or floats!");
		error_num=NUMERICAL_MPW_ERROR;
		return;
	}
	if(mpz_cmp_ui(op->data.ival,ULONG_MAX)>0) {
		(*errorout)("Number too large to compute factorial!");
		error_num=NUMERICAL_MPW_ERROR;
		return;
	}
	if(mpz_sgn(op->data.ival)<0) {
		(*errorout)("Can't do factorials of negative numbers!");
		error_num=NUMERICAL_MPW_ERROR;
		return;
	}

	mpwl_fac_ui(rop,mpz_get_ui(op->data.ival));
}

static void
mpwl_pow_q(MpwRealNum *rop,MpwRealNum *op1,MpwRealNum *op2)
{
	mpf_t fr;
	mpf_t fr2;
	mpf_t frt;
	unsigned long ne;
	unsigned long de;
	int t;
	int reverse=FALSE;

	if(op2->type!=MPW_RATIONAL) {
		error_num=INTERNAL_MPW_ERROR;
		return;
	}

	if((mpz_cmp_ui(mpq_numref(op2->data.rval),ULONG_MAX)>0) ||
		(mpz_cmp_ui(mpq_denref(op2->data.rval),ULONG_MAX)>0))
	{
		(*errorout)("Exponent numerator/quotient too big!");
		error_num=NUMERICAL_MPW_ERROR;
		return;
	}

	ne=mpz_get_ui(mpq_numref(op2->data.rval));
	de=mpz_get_ui(mpq_denref(op2->data.rval));
				/*this will make it allways
					positive!*/
	if(mpwl_sgn(op1)<0 && ((de/2)*2)==de) {
		(*errorout)("Can't yet handle complex numbers!");
		error_num=NUMERICAL_MPW_ERROR;
		return;
	}

	if(mpq_sgn(op2->data.rval)<0)
		reverse=TRUE;

	mpwl_make_extra_type(op1,MPW_FLOAT);
	t=MPW_FLOAT;


	/*
	 * Newton's method: Xn+1 = Xn - f(Xn)/f'(Xn)
	 */
	
	mpf_init(fr);
	mpf_init(fr2);
	mpf_init(frt);
	mpf_div_ui(fr,op1->data.fval,2); /*use half the value
					     as an initial guess*/
	for(;;) {
		mympf_pow_ui(fr2,fr,de);
		mpf_sub(fr2,fr2,op1->data.fval);

		mympf_pow_ui(frt,fr,de-1);
		mpf_mul_ui(frt,frt,de);
		mpf_div(fr2,fr2,frt);
		mpf_neg(fr2,fr2);
		mpf_add(fr2,fr2,fr);

		
		if(mpf_cmp(fr2,fr)==0)
			break;
		mpf_set(fr,fr2);
	}
	mpf_clear(fr2);
	mpf_clear(frt);

	mympf_pow_ui(fr,fr,ne);
	if(reverse)
		mpf_ui_div(fr,1,fr);

	/*op1 might have equaled rop so clear extra type here*/
	mpwl_clear_extra_type(op1,t);

	mpwl_clear(rop);
	mpwl_init_type(rop,MPW_FLOAT);
	mpf_set(rop->data.fval,fr);
}

/*power to an unsigned long and optionaly invert the answer*/
static void
mpwl_pow_ui(MpwRealNum *rop,MpwRealNum *op1,unsigned int e, int reverse)
{
	MpwRealNum r;

	switch(op1->type) {
		case MPW_RATIONAL:
			mpwl_init_type(&r,MPW_RATIONAL);
			mpz_pow_ui(mpq_numref(r.data.rval),
				op1->data.ival,e);
			mpz_pow_ui(mpq_denref(r.data.rval),
				op1->data.ival,e);
			/*the exponent was negative! reverse the result!*/
			if(reverse)
				mpq_inv(r.data.rval,r.data.rval);
			break;
		case MPW_INTEGER:
			if(!reverse) {
				mpwl_init_type(&r,MPW_INTEGER);
				mpz_pow_ui(r.data.ival,
					op1->data.ival,e);
			} else {
				mpwl_init_type(&r,MPW_RATIONAL);
				mpz_pow_ui(mpq_denref(r.data.rval),
					op1->data.ival,e);
				mpz_pow_ui(mpq_numref(r.data.rval),
					op1->data.ival,e);
			}
			break;
		case MPW_FLOAT:
			mpwl_init_type(&r,MPW_FLOAT);
			mympf_pow_ui(r.data.fval,op1->data.fval,e);

			if(reverse)
				mpf_ui_div(r.data.fval,1,r.data.fval);
			break;
	}
	mpwl_set(rop,&r);
	mpwl_clear(&r);
}

static void
mpwl_pow_z(MpwRealNum *rop,MpwRealNum *op1,MpwRealNum *op2)
{
	if(op2->type!=MPW_INTEGER) {
		error_num=INTERNAL_MPW_ERROR;
		return;
	}

	if(mpz_cmp_ui(op2->data.ival,ULONG_MAX)>0) {
		(*errorout)("Exponent too big!");
		error_num=NUMERICAL_MPW_ERROR;
		return;
	}

	if(mpz_sgn(op2->data.ival)==0)
		mpwl_set_ui(rop,1);
	else if(mpz_sgn(op2->data.ival)>0)
		mpwl_pow_ui(rop,op1,mpz_get_ui(op2->data.ival),FALSE);
	else
		mpwl_pow_ui(rop,op1,mpz_get_ui(op2->data.ival),TRUE);
}

static void
mpwl_pow_f(MpwRealNum *rop,MpwRealNum *op1,MpwRealNum *op2)
{
	MpwRealNum r;

	if(op2->type!=MPW_FLOAT) {
		error_num=INTERNAL_MPW_ERROR;
		return;
	}

	mpwl_init_type(&r,MPW_FLOAT);
	mpwl_set(&r,op2);
	mpwl_make_type(&r,MPW_RATIONAL);
	mpwl_pow_q(rop,op1,&r);
	mpwl_clear(&r);
}

static void
mpwl_pow(MpwRealNum *rop,MpwRealNum *op1,MpwRealNum *op2)
{
	if(mpwl_sgn(op2)==0) {
		mpwl_set_ui(rop,1);
		return;
	}

	switch(op2->type) {
		case MPW_FLOAT: mpwl_pow_f(rop,op1,op2); break;
		case MPW_RATIONAL: mpwl_pow_q(rop,op1,op2); break;
		case MPW_INTEGER: mpwl_pow_z(rop,op1,op2); break;
	}
}

static void
mpwl_make_int(MpwRealNum *rop, int floats)
{
	mpf_t fr;
	switch(rop->type) {
		case MPW_INTEGER: return;
		case MPW_RATIONAL:
			mpq_canonicalize(rop->data.rval);
			if(mpz_cmp_ui(mpq_denref(rop->data.rval),1)==0) {
				mpz_init_set(rop->data.ival,
					mpq_numref(rop->data.rval));
				mpq_clear(rop->data.rval);
				rop->type=MPW_INTEGER;
			}
			break;
		case MPW_FLOAT:
			if(!floats) return;
			/*gotta find a better way of doing this!*/

			mpz_set_f(rop->data.ival,rop->data.fval);
			mpf_init(fr);
			mpf_set_z(fr,rop->data.ival);
			if(mpf_cmp(fr,rop->data.fval)==0) {
				rop->type=MPW_INTEGER;
				mpf_clear(rop->data.fval);
			} else
				mpz_clear(rop->data.ival);
			mpf_clear(fr);
			break;
	}

}

/*make number into a float, this might be neccessary for unprecise
  calculations*/
static void
mpwl_make_float(MpwRealNum *rop)
{
	mpwl_make_type(rop,MPW_FLOAT);
}

static void
mpwl_set_str_float(MpwRealNum *rop,char *s,int base)
{
	if(rop->type!=MPW_FLOAT) {
		mpwl_clear(rop);
		mpwl_init_type(rop,MPW_FLOAT);
	}
	mpf_set_str(rop->data.fval,s,base);
}

static void
mpwl_set_str_int(MpwRealNum *rop,char *s,int base)
{
	if(rop->type!=MPW_INTEGER) {
		mpwl_clear(rop);
		mpwl_init_type(rop,MPW_INTEGER);
	}
	mpz_set_str(rop->data.ival,s,base);
}


/**************/
/*output stuff*/

/*round off the number at some digits*/
static void
str_make_max_digits(char *s,int digits)
{
	int i;
	int sd=0; /*digit where the number starts*/

	if(s[0]=='-')
		sd=1;

	if(!s || digits<=0)
		return;

	digits+=sd;

	if(strlen(s)<=digits)
		return;

	if(s[digits]<'5') {
		s[digits]='\0';
		return;
	}
	s[digits]='\0';

	for(i=digits-1;i>=sd;i--) {
		if(s[i]<'9') {
			s[i]++;
			return;
		}
		s[i]='\0';
	}
	shiftstr(s,1);
	s[sd]='1';
}

/*formats a floating point with mantissa in p and exponent in e*/
static char *
str_format_float(char *p,long int e,int scientific_notation)
{
	long int len;
	int i;
	if(((e-1)<-8 || (e-1)>8) || scientific_notation) {
		p=my_realloc(p,strlen(p)+1,
			strlen(p)+1+((int)log10(abs(e))+2)+1);
		if(p[0]=='-') {
			if(strlen(p)>2) {
				shiftstr(p+2,1);
				p[2]='.';
			}
		} else {
			if(strlen(p)>1) {
				shiftstr(p+1,1);
				p[1]='.';
			}
		}
		sprintf(p,"%se%ld",p,e-1);
	} else if(e>0) {
		len=strlen(p);
		if(p[0]=='-')
			len--;
		if(e>len) {
			p=my_realloc(p,strlen(p)+1,
				strlen(p)+1+e-len);
			for(i=0;i<e-len;i++)
				strcat(p,"0");
		} else if(e<len) {
			if(p[0]=='-') {
				shiftstr(p+1+e,1);
				p[e+1]='.';
			} else {
				shiftstr(p+e,1);
				p[e]='.';
			}
		}
	} else { /*e<=0*/
		if(strlen(p)==0) {
			p=g_strdup("0");
		} else {
			p=my_realloc(p,strlen(p)+1,
				strlen(p)+1+(-e)+2);
			if(p[0]=='-') {
				shiftstr(p+1,2+(-e));
				p[1]='0';
				p[2]='.';
				for(i=0;i<(-e);i++)
					p[i+3]='0';
			} else {
				shiftstr(p,2+(-e));
				p[0]='0';
				p[1]='.';
				for(i=0;i<(-e);i++)
					p[i+2]='0';
			}
		}
	}
	return p;
}

static char *
str_getstring_z(mpz_t num, int max_digits,int scientific_notation)
{
	char *p,*p2;
	mpf_t fr;

	p=mpz_get_str(NULL,10,num);
	if(max_digits>0 && max_digits<strlen(p)) {
		mpf_init(fr);
		mpf_set_z(fr,num);
		p2=str_getstring_f(fr,max_digits,scientific_notation);
		mpf_clear(fr);
		if(strlen(p2)>=strlen(p)) {
			g_free(p2);
			return p;
		} else  {
			g_free(p);
			return p2;
		}
	}
	return p;
}

static char *
str_getstring_q(mpq_t num, int max_digits,int scientific_notation)
{
	char *p,*p2;
	mpf_t fr;

	p=mpz_get_str(NULL,10,mpq_numref(num));
	p=appendstr(p,"/");
	p2=mpz_get_str(NULL,10,mpq_denref(num));
	p=appendstr(p,p2);
	g_free(p2);
	if(max_digits>0 && max_digits<strlen(p)) {
		mpf_init(fr);
		mpf_set_q(fr,num);
		p2=str_getstring_f(fr,max_digits,scientific_notation);
		mpf_clear(fr);
		if(strlen(p2)>=strlen(p)) {
			g_free(p2);
			return p;
		} else  {
			g_free(p);
			return p2;
		}
	}
	return p;
}

static char *
str_getstring_f(mpf_t num, int max_digits,int scientific_notation)
{
	char *p;
	long e;

	p=mpf_get_str(NULL,&e,10,0,num);
	str_make_max_digits(p,max_digits);
	p=str_format_float(p,e,scientific_notation);

	return p;
}

static char *
mpwl_getstring(MpwRealNum * num, int max_digits,int scientific_notation,
	int results_as_floats)
{
	mpf_t fr;
	char *p;
	switch(num->type) {
		case MPW_RATIONAL:
			if(results_as_floats) {
				mpf_init(fr);
				mpf_set_q(fr,num->data.rval);
				p=str_getstring_f(fr,max_digits,
					scientific_notation);
				mpf_clear(fr);
				return p;
			}
			return str_getstring_q(num->data.rval,max_digits,
					scientific_notation);
		case MPW_INTEGER:
			if(results_as_floats) {
				mpf_init(fr);
				mpf_set_z(fr,num->data.ival);
				p=str_getstring_f(fr,max_digits,
					scientific_notation);
				mpf_clear(fr);
				return p;
			}
			return str_getstring_z(num->data.ival,max_digits,
					scientific_notation);
		case MPW_FLOAT:
			return str_getstring_f(num->data.fval,max_digits,
				scientific_notation);
	}
}



/*************************************************************************/
/*high level stuff                                                       */
/*************************************************************************/

/*set default precision*/
void
mpw_set_default_prec(unsigned long int i)
{
	mpf_set_default_prec(i);
}

/*initialize a number*/
void
mpw_init(mpw_ptr op)
{
	op->type=MPW_REAL;
	op->r=(MpwRealNum *)g_malloc(sizeof(MpwRealNum));
	op->r->type=MPW_INTEGER;
	mpz_init(op->r->data.ival);
	op->i=NULL;
}

/*clear memory held by number*/
void
mpw_clear(mpw_ptr op)
{
	mpwl_free(op->r);
	mpwl_free(op->i);
	op->type=0;
}

/*make them the same type without loosing information*/
void
mpw_make_same_type(mpw_ptr op1,mpw_ptr op2)
{
	if(op1->type==MPW_REAL && op2->type==MPW_REAL) {
		mpwl_make_same_type(op1->r,op2->r);
	} else {
		/*FIXME: complex numbers*/
		g_error("can't deal with complex numbers");
	}
}

void
mpw_set(mpw_ptr rop,mpw_ptr op)
{
	if(op->type==MPW_REAL) {
		if(rop->type==MPW_COMPLEX) {
			mpw_clear(rop);
			mpw_init(rop);
		}
		mpwl_set(rop->r,op->r);
	} else {
		/*FIXME: complex numbers*/
		g_error("can't deal with complex numbers");
	}
}

void
mpw_set_d(mpw_ptr rop,double d)
{
	if(rop->type==MPW_COMPLEX) {
		mpw_clear(rop);
		mpw_init(rop);
	}
	mpwl_set_d(rop->r,d);
}

void
mpw_set_si(mpw_ptr rop,signed long int i)
{
	if(rop->type==MPW_COMPLEX) {
		mpw_clear(rop);
		mpw_init(rop);
	}
	mpwl_set_si(rop->r,i);
}

void
mpw_set_ui(mpw_ptr rop,unsigned long int i)
{
	if(rop->type==MPW_COMPLEX) {
		mpw_clear(rop);
		mpw_init(rop);
	}
	mpwl_set_ui(rop->r,i);
}

int
mpw_sgn(mpw_ptr op)
{
	if(op->type==MPW_REAL) {
		return mpwl_sgn(op->r);
	} else {
		/*FIXME: complex numbers*/
		g_error("can't deal with complex numbers");
	}
	return 0;
}

void
mpw_neg(mpw_ptr rop,mpw_ptr op)
{
	if(op->type==MPW_REAL) {
		if(rop->type==MPW_COMPLEX) {
			mpw_clear(rop);
			mpw_init(rop);
		}
		mpwl_neg(rop->r,op->r);
	} else {
		/*FIXME: complex numbers*/
		g_error("can't deal with complex numbers");
	}
}

void
mpw_add(mpw_ptr rop,mpw_ptr op1, mpw_ptr op2)
{
	if(op1->type==MPW_REAL && op2->type==MPW_REAL) {
		if(rop->type==MPW_COMPLEX) {
			mpw_clear(rop);
			mpw_init(rop);
		}
		mpwl_add(rop->r,op1->r,op2->r);
	} else {
		/*FIXME: complex numbers*/
		g_error("can't deal with complex numbers");
	}
}

void
mpw_sub(mpw_ptr rop,mpw_ptr op1, mpw_ptr op2)
{
	/*well not the most optimized but at least we won't need to change
	  it when the time comes, plus I don't see sign changes doing that
	  much damage in the first place*/
	mpw_neg(op2,op2);
	mpw_add(rop,op1,op2);
	mpw_neg(op2,op2);
}

void
mpw_mul(mpw_ptr rop,mpw_ptr op1, mpw_ptr op2)
{
	if(op1->type==MPW_REAL && op2->type==MPW_REAL) {
		if(rop->type==MPW_COMPLEX) {
			mpw_clear(rop);
			mpw_init(rop);
		}
		mpwl_mul(rop->r,op1->r,op2->r);
	} else {
		/*FIXME: complex numbers*/
		g_error("can't deal with complex numbers");
	}
}

void
mpw_mul_ui(mpw_ptr rop,mpw_ptr op, unsigned int i)
{
	if(op->type==MPW_REAL) {
		if(rop->type==MPW_COMPLEX) {
			mpw_clear(rop);
			mpw_init(rop);
		}
		mpwl_mul_ui(rop->r,op->r,i);
	} else {
		/*FIXME: complex numbers*/
		g_error("can't deal with complex numbers");
	}
}

void
mpw_div(mpw_ptr rop,mpw_ptr op1, mpw_ptr op2)
{
	if(mpw_sgn(op2)==0) {
		(*errorout)("Division by zero!");
		error_num=NUMERICAL_MPW_ERROR;
		return;
	}
	if(op1->type==MPW_REAL && op2->type==MPW_REAL) {
		if(rop->type==MPW_COMPLEX) {
			mpw_clear(rop);
			mpw_init(rop);
		}
		mpwl_div(rop->r,op1->r,op2->r);
	} else {
		/*FIXME: complex numbers*/
		g_error("can't deal with complex numbers");
	}
}

void
mpw_div_ui(mpw_ptr rop,mpw_ptr op, unsigned int i)
{
	if(i==0) {
		(*errorout)("Division by zero!");
		error_num=NUMERICAL_MPW_ERROR;
		return;
	}
	if(op->type==MPW_REAL) {
		if(rop->type==MPW_COMPLEX) {
			mpw_clear(rop);
			mpw_init(rop);
		}
		mpwl_div_ui(rop->r,op->r,i);
	} else {
		/*FIXME: complex numbers*/
		g_error("can't deal with complex numbers");
	}
}

void
mpw_ui_div(mpw_ptr rop,unsigned int i,mpw_ptr op)
{
	if(mpw_sgn(op)==0) {
		(*errorout)("Division by zero!");
		error_num=NUMERICAL_MPW_ERROR;
		return;
	}
	if(op->type==MPW_REAL) {
		if(rop->type==MPW_COMPLEX) {
			mpw_clear(rop);
			mpw_init(rop);
		}
		mpwl_ui_div(rop->r,1,op->r);
	} else {
		/*FIXME: complex numbers*/
		g_error("can't deal with complex numbers");
	}
}

void
mpw_mod(mpw_ptr rop,mpw_ptr op1, mpw_ptr op2)
{
	if(mpw_sgn(op2)==0) {
		(*errorout)("Division by zero!");
		error_num=NUMERICAL_MPW_ERROR;
		return;
	}
	if(op1->type==MPW_REAL && op2->type==MPW_REAL) {
		if(rop->type==MPW_COMPLEX) {
			mpw_clear(rop);
			mpw_init(rop);
		}
		mpwl_mod(rop->r,op1->r,op2->r);
	} else {
		/*FIXME: complex numbers*/
		g_error("can't deal with complex numbers");
	}
}

void
mpw_pow(mpw_ptr rop,mpw_ptr op1, mpw_ptr op2)
{
	if(op1->type==MPW_REAL && op2->type==MPW_REAL) {
		if(rop->type==MPW_COMPLEX) {
			mpw_clear(rop);
			mpw_init(rop);
		}
		mpwl_pow(rop->r,op1->r,op2->r);
	} else {
		/*FIXME: complex numbers*/
		g_error("can't deal with complex numbers");
	}
}

void
mpw_pow_ui(mpw_ptr rop,mpw_ptr op, unsigned long int e)
{
	if(op->type==MPW_REAL) {
		if(rop->type==MPW_COMPLEX) {
			mpw_clear(rop);
			mpw_init(rop);
		}
		mpwl_pow_ui(rop->r,op->r,e,FALSE);
	} else {
		/*FIXME: complex numbers*/
		g_error("can't deal with complex numbers");
	}
}

int
mpw_cmp(mpw_ptr op1, mpw_ptr op2)
{
	if(op1->type==MPW_REAL && op2->type==MPW_REAL) {
		return mpwl_cmp(op1->r,op2->r);
	} else {
		/*FIXME: complex numbers*/
		g_error("can't deal with complex numbers");
	}
}

int
mpw_cmp_ui(mpw_ptr op, unsigned long int i)
{
	if(op->type==MPW_REAL) {
		return mpwl_cmp_ui(op->r,i);
	} else {
		/*FIXME: complex numbers*/
		g_error("can't deal with complex numbers");
	}
}

void
mpw_fac_ui(mpw_ptr rop,unsigned long int i)
{
	if(rop->type==MPW_COMPLEX) {
		mpw_clear(rop);
		mpw_init(rop);
	}
	mpwl_fac_ui(rop->r,i);
}

void
mpw_fac(mpw_ptr rop,mpw_ptr op)
{
	if(op->type==MPW_REAL) {
		if(rop->type==MPW_COMPLEX) {
			mpw_clear(rop);
			mpw_init(rop);
		}
		mpwl_fac(rop->r,op->r);
	} else {
		/*FIXME: complex numbers*/
		g_error("can't deal with complex numbers");
	}
}

/*make a number int if possible, if floats is true try to convert to an
  int if the two equal*/
void
mpw_make_int(mpw_ptr rop,int floats)
{
	if(rop->type==MPW_REAL) {
		mpwl_make_int(rop->r,floats);
	} else {
		/*FIXME: complex numbers*/
		g_error("can't deal with complex numbers");
	}
}

/*make number into a float, this might be neccessary for unprecise
  calculations*/
void
mpw_make_float(mpw_ptr rop)
{
	if(rop->type==MPW_REAL) {
		mpwl_make_float(rop->r);
	} else {
		/*FIXME: complex numbers*/
		g_error("can't deal with complex numbers");
	}
}

/*init the mp stuff*/
void
mpw_init_mp(void)
{
	mp_set_memory_functions(g_malloc,my_realloc,my_free);
}

char *
mpw_getstring(mpw_ptr num, int max_digits,int scientific_notation,
	int results_as_floats)
{
	if(num->type==MPW_REAL) {
		mpwl_getstring(num->r,max_digits,scientific_notation,
			results_as_floats);
	} else {
		/*FIXME: complex numbers*/
		g_error("can't deal with complex numbers");
	}
}

void
mpw_set_str_float(mpw_ptr rop,char *s,int base)
{
	if(rop->type==MPW_REAL) {
		mpwl_set_str_float(rop->r,s,base);
	} else {
		/*FIXME: complex numbers*/
		g_error("can't deal with complex numbers");
	}
}

void
mpw_set_str_int(mpw_ptr rop,char *s,int base)
{
	if(rop->type==MPW_REAL) {
		mpwl_set_str_int(rop->r,s,base);
	} else {
		/*FIXME: complex numbers*/
		g_error("can't deal with complex numbers");
	}
}
