/*	$NetBSD: wwadd.c,v 1.7 2003/08/07 11:17:35 agc Exp $	*/

/*
 * Copyright (c) 1983, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * This code is derived from software contributed to Berkeley by
 * Edward Wang at The University of California, Berkeley.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/cdefs.h>

#ifndef lint
__RCSID("$NetBSD: wwadd.c,v 1.7 2003/08/07 11:17:35 agc Exp $");
#endif /* not lint */

#include <sys/types.h>
#include <sys/param.h>
#include <sys/wait.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <sys/select.h>

#if !defined(OLD_TTY) && !defined(TIOCPKT_DATA)
#include <sys/ioctl.h>
#endif

#include <poll.h>
#include <fcntl.h>
#include <err.h>
#include <errno.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <util.h>

#include "char.h"
#include "ww.h"

static void
wwdelete1(winvars_t *winvars, ww_t *w, int t, int b, int l, int r)
{
	int i;
	int tt, bb, ll, rr;
	char hasglass;

again:
	hasglass = 0;
	tt = MAX(t, w->ww_i.t);
	bb = MIN(b, w->ww_i.b);
	ll = MAX(l, w->ww_i.l);
	rr = MIN(r, w->ww_i.r);
	if (tt >= bb || ll >= rr) {
		if ((w = w->ww_forw) == &winvars->wwhead)
			return;
		goto again;
	}
	for (i = tt; i < bb; i++) {
		int j;
		unsigned char *smap = winvars->wwsmap[i];
		union ww_char *ns = winvars->wwns[i];
		char *win = w->ww_win[i];
		union ww_char *buf = w->ww_buf[i];
		int nvis = w->ww_nvis[i];
		int nchanged = 0;

		for (j = ll; j < rr; j++) {
			if (smap[j] != WWX_NOBODY)
				continue;
			if (win[j] & WWM_GLS) {
				hasglass = 1;
				continue;
			}
			smap[j] = w->ww_index;
			setparts(&ns[j], getcharpart(&buf[j]), win[j]);
			nchanged++;
			if (win[j] == 0)
				nvis++;
		}
		if (nchanged > 0)
			winvars->wwtouched[i] |= WWU_TOUCHED;
		w->ww_nvis[i] = nvis;
	}
	if ((w = w->ww_forw) == &winvars->wwhead)
		return;
	if (hasglass)
		goto again;
	if (tt > t)
		wwdelete1(winvars, w, t, tt, l, r);
	if (bb < b)
		wwdelete1(winvars, w, bb, b, l, r);
	if (ll > l)
		wwdelete1(winvars, w, tt, bb, l, ll);
	if (rr < r)
		wwdelete1(winvars, w, tt, bb, rr, r);
}

static int
wwsetttysize(winvars_t *winvars, int d, int r, int c)
{
	struct winsize winsize;

	winsize.ws_row = r;
	winsize.ws_col = c;
	winsize.ws_xpixel = winsize.ws_ypixel = 0;
	if (ioctl(d, TIOCSWINSZ, &winsize) < 0) {
		winvars->wwerrno = WWE_SYS;
		return -1;
	}
	return 0;
}

/*
 * Set up the environment of this process to run in window 'wp'.
 */
static int
wwenviron(winvars_t *winvars, ww_t *wp)
{

#ifndef TIOCSCTTY
	int		pgrp = getpid();
#endif
	sigset_t	nsigset;
	char		buf[1024];
	int		fd;

#ifndef TIOCSCTTY
	if ((fd = open("/dev/tty", O_RDONLY)) < 0) {
		goto bad;
	}
	if (ioctl(fd, TIOCNOTTY, NULL) < 0) {
		goto bad;
	}
	(void) close(fd);
#endif
	if ((fd = wp->ww_socket) < 0) {
		/* open the slave end */
		if ((fd = open(wp->ww_ttyname, O_RDWR | O_NOCTTY)) < 0) {
			goto bad;
		}
		if (wwsettty(winvars, fd, &winvars->wwwintty) < 0) {
			goto bad;
		}
		if (wwsetttysize(winvars, fd, wp->ww_w.nr, wp->ww_w.nc) < 0) {
			goto bad;
		}
	}
	(void) dup2(fd, STDIN_FILENO);
	(void) dup2(fd, STDOUT_FILENO);
	(void) dup2(fd, STDERR_FILENO);
	(void) close(fd);
#ifdef TIOCSCTTY
	if (setsid() < 0) {
		wwprintf(wp, "setsid failed %d\r\n", errno);
	}
	if (ioctl(0, TIOCSCTTY, 0) < 0) {
		wwprintf(wp, "TIOCSCTTY ioctl failed %d\r\n", errno);
	}
#else
	(void) ioctl(0, TIOCSPGRP, (char *)&pgrp);
	(void) setpgid(pgrp, pgrp);
#endif
	/* SIGPIPE is the only one we ignore */
	(void) signal(SIGPIPE, SIG_DFL);
	sigemptyset(&nsigset);
	sigprocmask(SIG_SETMASK, &nsigset, NULL);
	/*
	 * Two conditions that make destructive setenv ok:
	 * 1. setenv() copies the string,
	 * 2. we've already called tgetent which copies the termcap entry.
	 */
	(void) snprintf(buf, sizeof(buf), "%sco#%d:li#%d:%s",
		WWT_TERMCAP, wp->ww_w.nc, wp->ww_w.nr, winvars->wwwintermcap);
	(void) setenv("TERMCAP", buf, 1);
	(void) snprintf(buf, sizeof(buf), "%d", wp->ww_id + 1);
	(void) setenv("WINDOW_ID", buf, 1);
	(void) snprintf(buf, sizeof(buf), "%d", wp->ww_w.nr);
	(void) setenv("LINES", buf, 1);
	(void) snprintf(buf, sizeof(buf), "%d", wp->ww_w.nc);
	(void) setenv("COLUMNS", buf, 1);
	return 0;
bad:
	winvars->wwerrno = WWE_SYS;
	return -1;
}

#if defined(__APPLE__)
/* returns with the slave name in w->ww_ttyname */
/* returns file descriptor for master side of pty */
int
MI_openpty(ww_t *w)
{
	void	(*sigchild)(int);
	char	 *slavename;

	if ((w->ww_pty = open("/dev/ptmx", O_RDWR | O_NOCTTY)) == -1) {
		return -1;
	}
	sigchild = signal(SIGCHLD, SIG_DFL);
	if ((slavename = ptsname(w->ww_pty)) == NULL || grantpt(w->ww_pty) || unlockpt(w->ww_pty)) {
		signal(SIGCHLD, sigchild);
		close(w->ww_pty);
		return -1;
	} 
	(void) signal(SIGCHLD, sigchild);
	(void) strlcpy(w->ww_ttyname, slavename, sizeof(w->ww_ttyname));
	tcflush(w->ww_pty, TCIOFLUSH);
	return w->ww_pty;
}
#else
int
MI_openpty(ww_t *w)
{
	int	tty;

	if (openpty(&w->ww_pty, &tty, w->ww_ttyname, NULL, NULL) < 0) {
		return -1;
	}
	return w->ww_pty;
}
#endif

int
wwgetpty(winvars_t *winvars, ww_t *w)
{
	int	masterfd;
	int	on = 1;

	if ((masterfd = MI_openpty(w)) < 0) {
		w->ww_pty = -1;
		winvars->wwerrno = WWE_NOPTY;
		return -1;
	}
	if (ioctl(w->ww_pty, TIOCPKT, &on) < 0) {
		printf("TIOCPKT mode failed\r\n");
	}
	if (fcntl(w->ww_pty, F_SETFD, 1) < 0) {
		printf("fcntl F_SETFD failed\n");
	}
	return 0;
}

/*
 * Stick w1 behind w2.
 */
void
wwadd(winvars_t *winvars, ww_t *w1, ww_t *w2)
{
	int i;
	ww_t *w;

	w1->ww_order = w2->ww_order + 1;
	w1->ww_back = w2;
	w1->ww_forw = w2->ww_forw;
	w2->ww_forw->ww_back = w1;
	w2->ww_forw = w1;

	for (w = w1->ww_forw; w != &winvars->wwhead; w = w->ww_forw)
		w->ww_order++;
	for (i = w1->ww_i.t; i < w1->ww_i.b; i++) {
		int j;
		unsigned char *smap = winvars->wwsmap[i];
		char *win = w1->ww_win[i];
		union ww_char *ns = winvars->wwns[i];
		union ww_char *buf = w1->ww_buf[i];
		int nvis = 0;
		int nchanged = 0;

		for (j = w1->ww_i.l; j < w1->ww_i.r; j++) {
			w = winvars->wwindex[smap[j]];
			if (w1->ww_order > w->ww_order)
				continue;
			if (win[j] & WWM_GLS)
				continue;
			if (w != &winvars->wwnobody && w->ww_win[i][j] == 0)
				w->ww_nvis[i]--;
			smap[j] = w1->ww_index;
			if (win[j] == 0)
				nvis++;
			setparts(&ns[j], getcharpart(&buf[j]), win[j]);
			nchanged++;
		}
		if (nchanged > 0)
			winvars->wwtouched[i] |= WWU_TOUCHED;
		w1->ww_nvis[i] = nvis;
	}
}

char **
wwalloc(winvars_t *winvars, int row, int col, int nrow, int ncol, int size)
{
	char *p, **pp;
	int i;

	/* fast, call malloc only once */
	pp = (char **)
		malloc((unsigned) sizeof (char **) * nrow + size * nrow * ncol);
	if (pp == 0) {
		winvars->wwerrno = WWE_NOMEM;
		return 0;
	}
	p = (char *)(void *)&pp[nrow];
	col *= size;
	size /= sizeof (char);		/* paranoid */
	size *= ncol;
	for (i = 0; i < nrow; i++) {
		pp[i] = p - col;
		p += size;
	}
	return pp - row;
}

void
wwfree(char **p, int row)
{
	free(p + row);
}

/* utility functions to access the various parts of ww_char union */
char
getcharpart(union ww_char *wwch)
{
	return wwch->c_un.C_c;
}

char
setcharpart(union ww_char *wwch, char newch)
{
	return wwch->c_un.C_c = newch;
}

char
getmodepart(union ww_char *wwch)
{
	return wwch->c_un.C_m;
}

char
setmodepart(union ww_char *wwch, char newmode)
{
	return wwch->c_un.C_m = newmode;
}

short
getwhole(union ww_char *wwch)
{
	return wwch->c_w;
}

short
setwhole(union ww_char *wwch, short newwhole)
{
	return wwch->c_w = newwhole;
}

short
setparts(union ww_char *wwch, char ch, char mode)
{
	setcharpart(wwch, ch);
	setmodepart(wwch, mode);
	return getwhole(wwch);
}

/* ARGSUSED0 */
void
wwchild(int dummy __unused)
{
	int olderrno;
	ww_t **wp;
	int status;
	int pid;
	char collected = 0;
	winvars_t	*winvars;

	winvars = get_winvars();
	olderrno = errno;
	while ((pid = wait3(&status, WNOHANG|WUNTRACED, NULL)) > 0) {
		for (wp = winvars->wwindex; wp < &winvars->wwindex[MAX_NUM_WINDOWS]; wp++) {
			if (*wp && (*wp)->ww_state == WWS_HASPROC
			    && (*wp)->ww_pid == pid) {
				(*wp)->ww_state = WWS_DEAD;
				collected = 1;
				break;
			}
		}
	}
	errno = olderrno;
	/* jump out of wwiomux when somebody dies */
	if (collected) {
		WWSETINTR(winvars);
	}
}

void
wwcursor(winvars_t *winvars, ww_t *w, int on)
{
	char *win;

	if (on) {
		if (ISSET(w->ww_wflags, WWW_HASCURSOR))
			return;
		SET(w->ww_wflags, WWW_HASCURSOR);
	} else {
		if (!ISSET(w->ww_wflags, WWW_HASCURSOR))
			return;
		CLR(w->ww_wflags, WWW_HASCURSOR);
	}
	if (winvars->wwcursormodes != 0) {
		win = &w->ww_win[w->ww_cur.r][w->ww_cur.c];
		*win ^= winvars->wwcursormodes;
		if (w->ww_cur.r < w->ww_i.t || w->ww_cur.r >= w->ww_i.b ||
		    w->ww_cur.c < w->ww_i.l || w->ww_cur.c >= w->ww_i.r)
			return;
		if (winvars->wwsmap[w->ww_cur.r][w->ww_cur.c] == w->ww_index) {
			if (*win == 0)
				w->ww_nvis[w->ww_cur.r]++;
			else if (*win == winvars->wwcursormodes)
				w->ww_nvis[w->ww_cur.r]--;
			setmodepart(&winvars->wwns[w->ww_cur.r][w->ww_cur.c],
				getmodepart(&winvars->wwns[w->ww_cur.r][w->ww_cur.c]) ^ winvars->wwcursormodes);
			winvars->wwtouched[w->ww_cur.r] |= WWU_TOUCHED;
		}
	}
}

void
wwsetcursormodes(winvars_t *winvars, int new)
{
	int i;
	ww_t *w;
	int old;

	old = winvars->wwcursormodes;
	new &= winvars->wwavailmodes;
	if (new == winvars->wwcursormodes)
		return;
	for (i = 0; i < MAX_NUM_WINDOWS; i++)
		if (winvars->wwindex[i] != 0 &&
		    ISSET((w = winvars->wwindex[i])->ww_wflags, WWW_HASCURSOR)) {
			wwcursor(winvars, w, 0);
			winvars->wwcursormodes = new;
			wwcursor(winvars, w, 1);
			winvars->wwcursormodes = old;
		}
	winvars->wwcursormodes = new;
}

/*
 * Pull w free from the cover list.
 */
void
wwdelete(winvars_t *winvars, ww_t *w)
{
	int i;

	for (i = w->ww_i.t; i < w->ww_i.b; i++) {
		int j;
		unsigned char *smap = winvars->wwsmap[i];
		union ww_char *ns = winvars->wwns[i];
		int nchanged = 0;

		for (j = w->ww_i.l; j < w->ww_i.r; j++) {
			if (smap[j] == w->ww_index) {
				smap[j] = WWX_NOBODY;
				setcharpart(&ns[j], ' ');
				nchanged++;
			}
		}
		if (nchanged > 0)
			winvars->wwtouched[i] |= WWU_TOUCHED;
	}

	{
		ww_t *wp;

		for (wp = w->ww_forw; wp != &winvars->wwhead; wp = wp->ww_forw)
			wp->ww_order--;
	}

	if (w->ww_forw != &winvars->wwhead)
		wwdelete1(winvars, w->ww_forw,
			w->ww_i.t, w->ww_i.b, w->ww_i.l, w->ww_i.r);

	w->ww_back->ww_forw = w->ww_forw;
	w->ww_forw->ww_back = w->ww_back;
	w->ww_forw = w->ww_back = 0;
}

/*
 * Clear w to the end of line.
 * If cleared is true, then the screen line has already been cleared.
 */
void
wwclreol1(winvars_t *winvars, ww_t *w, int row, int col, char cleared)
{
	int i;

	/*
	 * Clear the buffer right off
	 */
	{
		union ww_char *buf;

		buf = &w->ww_buf[row][col]; 
		for (i = w->ww_b.r - col; --i >= 0;)
			setcharpart(buf++, ' ');
	}

	/*
	 * If can't see it, just return.
	 */
	if (row < w->ww_i.t || row >= w->ww_i.b
	    || w->ww_i.r <= 0 || w->ww_i.r <= col)
		return;

	if (col < w->ww_i.l)
		col = w->ww_i.l;

	/*
	 * Now fix wwns.
	 */
	{
		union ww_char *s;
		unsigned char *smap;
		char *win;

		i = col;
		smap = &winvars->wwsmap[row][i];
		s = &winvars->wwns[row][i];
		win = &w->ww_win[row][i];
		for (i = w->ww_i.r - i; --i >= 0;)
			if (*smap++ == w->ww_index)
				setparts(s++, ' ', *win++);
			else
				s++, win++;
	}
	if (!cleared)
		winvars->wwtouched[row] |= WWU_TOUCHED;
}

void
wwdelline(winvars_t *winvars, ww_t *w, int row)
{
	int i;
	union ww_char **cpp, **cqq;
	union ww_char *cp;
	int row1, row2;
	char deleted;
	int visible;

	/*
	 * Scroll first.
	 */
	if ((row1 = row) < w->ww_i.t) {
		row1 = w->ww_i.t;
	}
	if ((row2 = w->ww_b.b) > w->ww_i.b) {
		row2 = w->ww_i.b;
		visible = 0;
	} else
		visible = 1;
	deleted = wwscroll1(winvars, w, row1, row2, 1, visible);

	/*
	 * Fix the buffer.
	 * But leave clearing the last line for WWCLREOL().
	 */
	cpp = &w->ww_buf[row];
	cqq = cpp + 1;
	cp = *cpp;
	for (i = w->ww_b.b - row; --i > 0;)
		*cpp++ = *cqq++;
	*cpp = cp;

	/*
	 * Now clear the last line.
	 */
	if (visible)
		wwclreol1(winvars, w, w->ww_b.b - 1, w->ww_b.l, deleted);
	else {
		cp += w->ww_b.l;
		for (i = w->ww_b.nc; --i >= 0;)
			setcharpart(cp++, ' ');
	}
}

const char *
wwerror(winvars_t *winvars)
{
	switch (winvars->wwerrno) {
	case WWE_NOERR:
		return "No error";
	case WWE_SYS:
		return strerror(errno);
	case WWE_NOMEM:
		return "Out of memory";
	case WWE_TOOMANY:
		return "Too many windows";
	case WWE_NOPTY:
		return "Out of pseudo-terminals";
	case WWE_SIZE:
		return "Bad window size";
	case WWE_BADTERM:
		return "Unknown terminal type";
	case WWE_CANTDO:
		return "Can't run window on this terminal";
	default:
		return "Unknown error";
	}
}

static void
rub(winvars_t *winvars, int c, ww_t *w)
{
	int i;

	for (i = strlen(UNCTRL(c)); --i >= 0;)
		(void) wwwrite(winvars, w, "\b \b", 3);
}

void
wwgets(winvars_t *winvars, char *buf, int n, ww_t *w)
{
	char *p = buf;
	int c;
	int uc = ISSET(w->ww_wflags, WWW_UNCTRL);

	CLR(w->ww_wflags, WWW_UNCTRL);
	for (;;) {
		WWCURTOWIN(winvars, w);
		while ((c = WWGETC(winvars)) < 0)
			wwiomux(winvars);
#ifdef OLD_TTY
		if (c == winvars->wwoldtty.ww_sgttyb.sg_erase)
#else
		if (c == winvars->wwoldtty.ww_termios.c_cc[VERASE])
#endif
		{
			if (p > buf)
				rub(winvars, *--p, w);
		} else
#ifdef OLD_TTY
		if (c == winvars->wwoldtty.ww_sgttyb.sg_kill)
#else
		if (c == winvars->wwoldtty.ww_termios.c_cc[VKILL])
#endif
		{
			while (p > buf)
				rub(winvars, *--p, w);
		} else
#ifdef OLD_TTY
		if (c == winvars->wwoldtty.ww_ltchars.t_werasc)
#else
		if (c == winvars->wwoldtty.ww_termios.c_cc[VWERASE])
#endif
		{
			while (--p >= buf && (*p == ' ' || *p == '\t'))
				rub(winvars, *p, w);
			while (p >= buf && *p != ' ' && *p != '\t')
				rub(winvars, *p--, w);
			p++;
		} else if (c == '\r' || c == '\n') {
			break;
		} else {
			if (p >= buf + n - 1)
				wwputc(CONTROL('g'), w);
			else
				wwputs(winvars, UNCTRL(*p++ = c), w);
		}
	}
	*p = 0;
	SET(w->ww_wflags, uc);
}

void
wwinsline(winvars_t *winvars, ww_t *w, int row)
{
	int i;
	union ww_char **cpp, **cqq;
	union ww_char *cp;
	int row1, row2;
	char deleted;
	int visible;

	/*
	 * Scroll first.
	 */
	if ((row1 = row) < w->ww_i.t) {
		row1 = w->ww_i.t;
		visible = 0;
	} else
		visible = 1;
	if ((row2 = w->ww_b.b) > w->ww_i.b) {
		row2 = w->ww_i.b;
	}
	deleted = wwscroll1(winvars, w, row1, row2, -1, visible);

	/*
	 * Fix the buffer.
	 * But leave clearing the last line for WWCLREOL().
	 */
	cpp = &w->ww_buf[w->ww_b.b];
	cqq = cpp - 1;
	cp = *cqq;
	for (i = w->ww_b.b - row; --i > 0;)
		*--cpp = *--cqq;
	*cqq = cp;

	/*
	 * Now clear the last line.
	 */
	if (visible)
		wwclreol1(winvars, w, row, w->ww_b.l, deleted);
	else {
		cp += w->ww_b.l;
		for (i = w->ww_b.nc; --i >= 0;)
			setcharpart(cp++, ' ');
	}
}

/*
 * It seems that Darwin's poll(2), in a bid to be unique, does not work
 * on devices, unlike every other implementation of poll out there.
 * So we emulate this using select(2) on Mac OS X.
 * Great work, guys, took me ages to work this one out.
 */
static int
realpoll(struct pollfd *pollfds, unsigned nfd, int millis)
{
#ifdef __APPLE__
	struct timeval	tv;
	static fd_set	reads;
	static fd_set	writes;
	static fd_set	errs;
	int		ret;
	int		i;

	(void) memset(&tv, 0x0, sizeof(tv));
	FD_ZERO(&reads);
	FD_ZERO(&writes);
	FD_ZERO(&errs);
	tv.tv_sec = (millis / 1000);
	tv.tv_usec = (millis - (tv.tv_sec * 1000)) * 1000;
	for (i = 0 ; i < nfd ; i++) {
		if (pollfds[i].events & POLLIN) {
			FD_SET(pollfds[i].fd, &reads);
		}
		if (pollfds[i].events & POLLOUT) {
			FD_SET(pollfds[i].fd, &writes);
		}
	}
	ret = select(MAX_NUM_WINDOWS, &reads, &writes, &errs, &tv);
	for (i = 0 ; i < nfd ; i++) {
		pollfds[i].revents = 0;
		if (FD_ISSET(pollfds[i].fd, &reads)) {
			pollfds[i].revents |= POLLIN;
		}
		if (FD_ISSET(pollfds[i].fd, &writes)) {
			pollfds[i].revents |= POLLOUT;
		}
	}
	return ret;
#else
	return poll(pollfds, nfd, millis);
#endif
}

/*
 * Multiple window output handler.
 * The idea is to copy window outputs to the terminal, via the
 * display package.  We try to give wwcurwin highest priority.
 * The only return conditions are when there is keyboard input
 * and when a child process dies.
 * When there's nothing to do, we sleep in a select().
 * The history of this routine is interesting.
 */
void
wwiomux(winvars_t *winvars)
{
	ww_t * volatile w;
	int nfd;
	int volatile dostdin;	/* avoid longjmp clobbering */
	char volatile c;	/* avoid longjmp clobbering */
	char *p;
	int millis;
	char noblock = 0;
	static struct pollfd *pfd = NULL;
	static size_t maxfds = 0;

	c = 0; 	/* XXXGCC -Wuninitialized */

	for (;;) {
		if (WWINTERRUPT(winvars)) {
			WWCLRINTR(winvars);
			return;
		}

		nfd = 0;
		for (w = winvars->wwhead.ww_forw; w != &winvars->wwhead; w = w->ww_forw) {
			if (w->ww_pty < 0 || w->ww_obq >= w->ww_obe) {
				continue;
			}
			nfd++;
		}

		if (maxfds <= ++nfd) {	/* One more for the fd=0 case below */
			struct pollfd *npfd;

			npfd = (pfd == NULL) ? malloc(sizeof(*pfd) * nfd) :
				realloc(pfd, sizeof(*pfd) * nfd);
			if (npfd == NULL) {
				warn("will retry");
				if (pfd) {
					free(pfd);
				}
				pfd = NULL;
				maxfds = 0;
				return;
			}
			pfd = npfd;
			maxfds = nfd;
		}

		nfd = 0;
		for (w = winvars->wwhead.ww_forw; w != &winvars->wwhead; w = w->ww_forw) {
			if (w->ww_pty < 0) {
				continue;
			}
			if (w->ww_obq < w->ww_obe) {
				pfd[nfd].fd = w->ww_pty;
				pfd[nfd++].events = POLLIN;
			}
			if (w->ww_obq > w->ww_obp &&
			    !ISSET(w->ww_pflags, WWP_STOPPED))
				noblock = 1;
		}
		if (winvars->wwibq < winvars->wwibe) {
			dostdin = nfd;
			pfd[nfd].fd = 0;
			pfd[nfd++].events = POLLIN;
		} else {
			dostdin = -1;
		}

		if (!noblock) {
			if (winvars->wwcurwin != NULL) {
				WWCURTOWIN(winvars, winvars->wwcurwin);
			}
			WWUPDATE(winvars);
			wwflush(winvars);
			(void) setjmp(winvars->wwjmpbuf);
			winvars->wwsetjmp = 1;
			if (WWINTERRUPT(winvars)) {
				winvars->wwsetjmp = 0;
				WWCLRINTR(winvars);
				return;
			}
			/* XXXX */
			millis = 30000;
		} else {
			millis = 10;
		}
		winvars->wwnselect++;
		nfd = realpoll(pfd, (unsigned)nfd, millis);
		winvars->wwsetjmp = 0;
		noblock = 0;

		if (nfd < 0) {
			winvars->wwnselecte++;
		} else if (nfd == 0) {
			winvars->wwnselectz++;
		} else {
			if (dostdin != -1 && (pfd[dostdin].revents & POLLIN) != 0) {
				wwrint(winvars);
			}

			nfd = 0;
			for (w = winvars->wwhead.ww_forw; w != &winvars->wwhead; w = w->ww_forw) {
				int n;

				if (w->ww_pty < 0) {
					continue;
				}
				if (w->ww_pty != pfd[nfd].fd) {
					continue;
				}
				if ((pfd[nfd++].revents & POLLIN) == 0) {
					continue;
				}
				winvars->wwnwread++;
				p = w->ww_obq;
				if (w->ww_type == WWT_PTY) {
					if (p == w->ww_ob) {
						w->ww_obp++;
						w->ww_obq++;
					} else
						p--;
					c = *p;
				}
				n = read(w->ww_pty, p,
					(unsigned)(w->ww_obe - p));
				if (n < 0) {
					winvars->wwnwreade++;
					(void) close(w->ww_pty);
					w->ww_pty = -1;
				} else if (n == 0) {
					winvars->wwnwreadz++;
					(void) close(w->ww_pty);
					w->ww_pty = -1;
				} else if (w->ww_type != WWT_PTY) {
					winvars->wwnwreadd++;
					winvars->wwnwreadc += n;
					w->ww_obq += n;
				} else if (*p == TIOCPKT_DATA) {
					n--;
					winvars->wwnwreadd++;
					winvars->wwnwreadc += n;
					w->ww_obq += n;
				} else {
					winvars->wwnwreadp++;
					if (*p & TIOCPKT_STOP) {
						SET(w->ww_pflags, WWP_STOPPED);
					}
					if (*p & TIOCPKT_START) {
						CLR(w->ww_pflags, WWP_STOPPED);
					}
					if (*p & TIOCPKT_FLUSHWRITE) {
						CLR(w->ww_pflags, WWP_STOPPED);
						w->ww_obq = w->ww_obp =
							w->ww_ob;
					}
				}
				if (w->ww_type == WWT_PTY) {
					*p = c;
				}
			}
		}
		/*
		 * Try the current window first, if there is output
		 * then process it and go back to the top to try again.
		 * This can lead to starvation of the other windows,
		 * but presumably that what we want.
		 * Update will eventually happen when output from wwcurwin
		 * dies down.
		 */
#define OUT_DATAC(w)	(int)((w)->ww_obq - (w)->ww_obp)
		if ((w = winvars->wwcurwin) != NULL && w->ww_pty >= 0 &&
		    OUT_DATAC(w) > 0 &&
		    !ISSET(w->ww_pflags, WWP_STOPPED)) {
			int n = wwwrite(winvars, w, w->ww_obp, OUT_DATAC(w));
			if ((w->ww_obp += n) == w->ww_obq) {
				w->ww_obq = w->ww_obp = w->ww_ob;
			}
			noblock = 1;
			continue;
		}
		for (w = winvars->wwhead.ww_forw; w != &winvars->wwhead; w = w->ww_forw) {
			if (w->ww_pty >= 0 &&
			    OUT_DATAC(w) > 0 &&
			    !ISSET(w->ww_pflags, WWP_STOPPED)) {
				int n = wwwrite(winvars, w, w->ww_obp, OUT_DATAC(w));
				if ((w->ww_obp += n) == w->ww_obq) {
					w->ww_obq = w->ww_obp = w->ww_ob;
				}
				if (WWINTERRUPT(winvars)) {
					break;
				}
			}
		}
	}
}

/*
 * Label window w on f,
 * at 1 line above w and 'where' columns from its left edge.
 * Gross, but it works.
 */
void
wwlabel(winvars_t *winvars, ww_t *w, ww_t *f, int where, char *l, int mode)
{
	int row;
	int j;
	int jj;
	char *win;
	union ww_char *buf;
	union ww_char *ns;
	char *fmap;
	unsigned char *smap;
	char touched;
	char *p;

	if (f->ww_fmap == 0)
		return;

	row = w->ww_w.t - 1;
	if (row < f->ww_i.t || row >= f->ww_i.b)
		return;
	win = f->ww_win[row];
	buf = f->ww_buf[row];
	fmap = f->ww_fmap[row];
	ns = winvars->wwns[row];
	smap = winvars->wwsmap[row];
	touched = winvars->wwtouched[row];

	jj = MIN(w->ww_i.r, f->ww_i.r);
	j = w->ww_i.l + where;
	while (j < jj && *l) {
		for (p = __UNCONST(UNCTRL(*l++)); j < jj && *p; j++, p++) {
			/* can't label if not already framed */
			if (win[j] & WWM_GLS) {
				continue;
			}
			if (smap[j] != f->ww_index) {
				setparts(&buf[j], *p, mode);
			} else {
				setparts(&buf[j], *p, mode);
				setparts(&ns[j], *p, mode ^ win[j]);
				touched |= WWU_TOUCHED;
			}
			fmap[j] |= WWF_LABEL;
		}
	}
	winvars->wwtouched[row] = touched;
}

/*
 * Move a window.  Should be unattached.
 */
void
wwmove(winvars_t *winvars, ww_t *w, int row, int col)
{
	int dr, dc;
	int i;

	dr = row - w->ww_w.t;
	dc = col - w->ww_w.l;

	w->ww_w.t += dr;
	w->ww_w.b += dr;
	w->ww_w.l += dc;
	w->ww_w.r += dc;

	w->ww_b.t += dr;
	w->ww_b.b += dr;
	w->ww_b.l += dc;
	w->ww_b.r += dc;

	w->ww_i.t = MAX(w->ww_w.t, 0);
	w->ww_i.b = MIN(w->ww_w.b, winvars->wwnrow);
	w->ww_i.nr = w->ww_i.b - w->ww_i.t;
	w->ww_i.l = MAX(w->ww_w.l, 0);
	w->ww_i.r = MIN(w->ww_w.r, winvars->wwncol);
	w->ww_i.nc = w->ww_i.r - w->ww_i.l;

	w->ww_cur.r += dr;
	w->ww_cur.c += dc;

	w->ww_win -= dr;
	for (i = w->ww_w.t; i < w->ww_w.b; i++)
		w->ww_win[i] -= dc;
	if (w->ww_fmap != 0) {
		w->ww_fmap -= dr;
		for (i = w->ww_w.t; i < w->ww_w.b; i++)
			w->ww_fmap[i] -= dc;
	}
	w->ww_nvis -= dr;
	for (i = w->ww_i.t; i < w->ww_i.b; i++) {
		int j = w->ww_i.l;
		char *win = &w->ww_win[i][j];
		unsigned char *smap = &winvars->wwsmap[i][j];
		int nvis = 0;

		for (; j < w->ww_i.r; j++, win++, smap++)
			if (*win == 0 && *smap == w->ww_index)
				nvis++;
		w->ww_nvis[i] = nvis;
	}
	w->ww_buf -= dr;
	for (i = w->ww_b.t; i < w->ww_b.b; i++)
		w->ww_buf[i] -= dc;
}

ww_t *
wwopen(winvars_t *winvars, int type, int oflags, int nrow, int ncol, int row, int col, int nline)
{
	ww_t *w;
	int i, j;
	char m;
	short nvis;

	w = (ww_t *)calloc(1, sizeof (ww_t));
	if (w == 0) {
		winvars->wwerrno = WWE_NOMEM;
		goto bad;
	}
	w->ww_pty = -1;
	w->ww_socket = -1;

	for (i = 0; i < MAX_NUM_WINDOWS && winvars->wwindex[i] != 0; i++)
		;
	if (i >= MAX_NUM_WINDOWS) {
		winvars->wwerrno = WWE_TOOMANY;
		goto bad;
	}
	w->ww_index = i;

	if (nline < nrow)
		nline = nrow;

	w->ww_w.t = row;
	w->ww_w.b = row + nrow;
	w->ww_w.l = col;
	w->ww_w.r = col + ncol;
	w->ww_w.nr = nrow;
	w->ww_w.nc = ncol;

	w->ww_b.t = row;
	w->ww_b.b = row + nline;
	w->ww_b.l = col;
	w->ww_b.r = col + ncol;
	w->ww_b.nr = nline;
	w->ww_b.nc = ncol;

	w->ww_i.t = MAX(w->ww_w.t, 0);
	w->ww_i.b = MIN(w->ww_w.b, winvars->wwnrow);
	w->ww_i.l = MAX(w->ww_w.l, 0);
	w->ww_i.r = MIN(w->ww_w.r, winvars->wwncol);
	w->ww_i.nr = w->ww_i.b - w->ww_i.t;
	w->ww_i.nc = w->ww_i.r - w->ww_i.l;

	w->ww_cur.r = w->ww_w.t;
	w->ww_cur.c = w->ww_w.l;

	w->ww_type = type;
	switch (type) {
	case WWT_PTY:
		if (wwgetpty(winvars, w) < 0)
			goto bad;
		break;
	case WWT_SOCKET:
	    {
		int d[2];
		if (socketpair(AF_LOCAL, SOCK_STREAM, PF_UNSPEC, d) < 0) {
			winvars->wwerrno = WWE_SYS;
			goto bad;
		}
		(void) fcntl(d[0], F_SETFD, 1);
		(void) fcntl(d[1], F_SETFD, 1);
		w->ww_pty = d[0];
		w->ww_socket = d[1];
		break;
	    }
	}
	if (type != WWT_INTERNAL) {
		if ((w->ww_ob = malloc(512)) == 0) {
			winvars->wwerrno = WWE_NOMEM;
			goto bad;
		}
		w->ww_obe = w->ww_ob + 512;
		w->ww_obp = w->ww_obq = w->ww_ob;
		if (w->ww_pty >= winvars->wwdtablesize)
			winvars->wwdtablesize = w->ww_pty + 1;
	}

	w->ww_win = wwalloc(winvars, w->ww_w.t, w->ww_w.l,
		w->ww_w.nr, w->ww_w.nc, sizeof (char));
	if (w->ww_win == NULL)
		goto bad;
	m = 0;
	if (oflags & WWO_GLASS)
		m |= WWM_GLS;
	if (oflags & WWO_REVERSE) {
		if (winvars->wwavailmodes & WWM_REV)
			m |= WWM_REV;
		else	oflags &= ~WWO_REVERSE;
	}
	for (i = w->ww_w.t; i < w->ww_w.b; i++)
		for (j = w->ww_w.l; j < w->ww_w.r; j++)
			w->ww_win[i][j] = m;

	if (oflags & WWO_FRAME) {
		w->ww_fmap = wwalloc(winvars, w->ww_w.t, w->ww_w.l,
			w->ww_w.nr, w->ww_w.nc, sizeof (char));
		if (w->ww_fmap == 0)
			goto bad;
		for (i = w->ww_w.t; i < w->ww_w.b; i++)
			for (j = w->ww_w.l; j < w->ww_w.r; j++)
				w->ww_fmap[i][j] = 0;
	}

	w->ww_buf = (union ww_char **)
		wwalloc(winvars, w->ww_b.t, w->ww_b.l,
			w->ww_b.nr, w->ww_b.nc, sizeof (union ww_char));
	if (w->ww_buf == 0)
		goto bad;
	for (i = w->ww_b.t; i < w->ww_b.b; i++)
		for (j = w->ww_b.l; j < w->ww_b.r; j++)
			setcharpart(&w->ww_buf[i][j], ' ');

	w->ww_nvis = (short *)malloc((unsigned) w->ww_w.nr * sizeof (short));
	if (w->ww_nvis == 0) {
		winvars->wwerrno = WWE_NOMEM;
		goto bad;
	}
	w->ww_nvis -= w->ww_w.t;
	nvis = m ? 0 : w->ww_w.nc;
	for (i = w->ww_w.t; i < w->ww_w.b; i++)
		w->ww_nvis[i] = nvis;

	w->ww_state = WWS_INITIAL;
	CLR(w->ww_oflags, WWO_ALLFLAGS);
	SET(w->ww_oflags, oflags);
	return winvars->wwindex[w->ww_index] = w;
bad:
	if (w != NULL) {
		if (w->ww_win != NULL)
			wwfree(w->ww_win, w->ww_w.t);
		if (w->ww_fmap != NULL)
			wwfree(w->ww_fmap, w->ww_w.t);
		if (w->ww_buf != NULL)
			wwfree((char **)w->ww_buf, w->ww_b.t);
		if (w->ww_nvis != NULL)
			free(w->ww_nvis + w->ww_w.t);
		if (w->ww_ob != NULL)
			free(w->ww_ob);
		if (w->ww_pty >= 0)
			(void) close(w->ww_pty);
		if (w->ww_socket >= 0)
			(void) close(w->ww_socket);
		free(w);
	}
	return NULL;
}

void
wwprintf(ww_t *w, const char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	(void) wwvprintf(w, fmt, ap);
	va_end(ap);
}

void
wwvprintf(ww_t *w, const char *fmt, va_list ap)
{
	char buf[1024];
	winvars_t	*winvars;

	winvars = get_winvars();
	(void) wwwrite(winvars, w, buf, vsnprintf(buf, sizeof(buf), fmt, ap));
}

void
wwputc(char c, ww_t *w)
{
	winvars_t	*winvars;

	winvars = get_winvars();
	(void) wwwrite(winvars, w, &c, sizeof(c));
}

void
wwputs(winvars_t *winvars, const char *s, ww_t *w)
{
	const char *p = s;

	while (*p++)
		;
	(void) wwwrite(winvars, w, s, p - s - 1);
}

void
wwredrawwin1(winvars_t *winvars, ww_t *w, int row1, int row2, int offset)
{
	int row;
	int col;
	unsigned char *smap;
	union ww_char *buf;
	char *win;
	union ww_char *ns;
	union ww_char x;
	int nchanged;

	for (row = row1; row < row2; row++) {
		col = w->ww_i.l;
		ns = winvars->wwns[row];
		smap = &winvars->wwsmap[row][col];
		buf = w->ww_buf[row + offset];
		win = w->ww_win[row];
		nchanged = 0;
		for (; col < w->ww_i.r; col++) {
			setparts(&x, getcharpart(&buf[col]),
				getmodepart(&buf[col]) ^ win[col]);
			if (*smap++ == w->ww_index &&
			    getwhole(&ns[col]) != getwhole(&x)) {
				setparts(&ns[col], getcharpart(&x), getmodepart(&x));
				nchanged++;
			}
		}
		if (nchanged > 0) {
			winvars->wwtouched[row] |= WWU_TOUCHED;
		}
	}
}

/*
 * Resize a window.  Should be unattached.
 */
int
wwsize(winvars_t *winvars, ww_t *w, int nrow, int ncol)
{
	int i, j;
	int nline = 0;
	union ww_char **buf = 0;
	char **win = 0;
	short *nvis = 0;
	char **fmap = 0;
	char m;

	/*
	 * First allocate new buffers.
	 */
	win = wwalloc(winvars, w->ww_w.t, w->ww_w.l, nrow, ncol, sizeof (char));
	if (win == 0)
		goto bad;
	if (w->ww_fmap != 0) {
		fmap = wwalloc(winvars, w->ww_w.t, w->ww_w.l, nrow, ncol, sizeof (char));
		if (fmap == 0)
			goto bad;
	}
	if (nrow > w->ww_b.nr || ncol > w->ww_b.nc) {
		nline = MAX(w->ww_b.nr, nrow);
		buf = (union ww_char **) wwalloc(winvars, w->ww_b.t, w->ww_b.l,
			nline, ncol, sizeof (union ww_char));
		if (buf == 0)
			goto bad;
	}
	nvis = (short *)malloc((unsigned) nrow * sizeof (short));
	if (nvis == 0) {
		winvars->wwerrno = WWE_NOMEM;
		goto bad;
	}
	nvis -= w->ww_w.t;
	/*
	 * Copy text buffer.
	 */
	if (buf != 0) {
		int b, r;

		b = w->ww_b.t + nline;
		r = w->ww_b.l + ncol;
		if (ncol < w->ww_b.nc)
			for (i = w->ww_b.t; i < w->ww_b.b; i++)
				for (j = w->ww_b.l; j < r; j++)
					buf[i][j] = w->ww_buf[i][j];
		else
			for (i = w->ww_b.t; i < w->ww_b.b; i++) {
				for (j = w->ww_b.l; j < w->ww_b.r; j++)
					buf[i][j] = w->ww_buf[i][j];
				for (; j < r; j++)
					setcharpart(&buf[i][j], ' ');
			}
		for (; i < b; i++)
			for (j = w->ww_b.l; j < r; j++)
				setcharpart(&buf[i][j], ' ');
	}
	/*
	 * Now free the old stuff.
	 */
	wwfree((char **)w->ww_win, w->ww_w.t);
	w->ww_win = win;
	if (buf != 0) {
		wwfree((char **)w->ww_buf, w->ww_b.t);
		w->ww_buf = buf;
	}
	if (w->ww_fmap != 0) {
		wwfree((char **)w->ww_fmap, w->ww_w.t);
		w->ww_fmap = fmap;
	}
	free(w->ww_nvis + w->ww_w.t);
	w->ww_nvis = nvis;
	/*
	 * Set new sizes.
	 */
		/* window */
	w->ww_w.b = w->ww_w.t + nrow;
	w->ww_w.r = w->ww_w.l + ncol;
	w->ww_w.nr = nrow;
	w->ww_w.nc = ncol;
		/* text buffer */
	if (buf != 0) {
		w->ww_b.b = w->ww_b.t + nline;
		w->ww_b.r = w->ww_b.l + ncol;
		w->ww_b.nr = nline;
		w->ww_b.nc = ncol;
	}
		/* scroll */
	if ((i = w->ww_b.b - w->ww_w.b) < 0 ||
	    (i = w->ww_cur.r - w->ww_w.b + 1) > 0) {
		w->ww_buf += i;
		w->ww_b.t -= i;
		w->ww_b.b -= i;
		w->ww_cur.r -= i;
	}
		/* interior */
	w->ww_i.b = MIN(w->ww_w.b, winvars->wwnrow);
	w->ww_i.r = MIN(w->ww_w.r, winvars->wwncol);
	w->ww_i.nr = w->ww_i.b - w->ww_i.t;
	w->ww_i.nc = w->ww_i.r - w->ww_i.l;
	/*
	 * Initialize new buffers.
	 */
		/* window */
	m = 0;
	if (w->ww_oflags & WWO_GLASS)
		m |= WWM_GLS;
	if (w->ww_oflags & WWO_REVERSE)
		m |= WWM_REV;
	for (i = w->ww_w.t; i < w->ww_w.b; i++)
		for (j = w->ww_w.l; j < w->ww_w.r; j++)
			w->ww_win[i][j] = m;
		/* frame map */
	if (fmap != 0)
		for (i = w->ww_w.t; i < w->ww_w.b; i++)
			for (j = w->ww_w.l; j < w->ww_w.r; j++)
				w->ww_fmap[i][j] = 0;
		/* visibility */
	j = m ? 0 : w->ww_w.nc;
	for (i = w->ww_w.t; i < w->ww_w.b; i++)
		w->ww_nvis[i] = j;
	/*
	 * Put cursor back.
	 */
	if (ISSET(w->ww_wflags, WWW_HASCURSOR)) {
		CLR(w->ww_wflags, WWW_HASCURSOR);
		wwcursor(winvars, w, 1);
	}
	/*
	 * Fool with pty.
	 */
	if (w->ww_type == WWT_PTY && w->ww_pty >= 0)
		(void) wwsetttysize(winvars, w->ww_pty, nrow, ncol);
	return 0;
bad:
	if (win != 0)
		wwfree(win, w->ww_w.t);
	if (fmap != 0)
		wwfree(fmap, w->ww_w.t);
	if (buf != 0)
		wwfree((char **)buf, w->ww_b.t);
	return -1;
}

/*
 * There is a dead lock with vfork and closing of pseudo-ports.
 * So we have to be sneaky about error reporting.
 */
int
wwspawn(winvars_t *winvars, ww_t *w, char *file, char **argv)
{
	int pid;
	int ret;
	int tty;
	char volatile erred;
	sigset_t nsigset, osigset;

	erred = 0;

	sigemptyset(&nsigset);
	sigaddset(&nsigset, SIGCHLD);
	sigprocmask(SIG_BLOCK, &nsigset, &osigset);

	switch (pid = vfork()) {
	case -1:
		winvars->wwerrno = WWE_SYS;
		ret = -1;
		break;
	case 0:
		if (wwenviron(winvars, w) >= 0) {
			execvp(file, argv);
		}
		erred = 1;
		_exit(1);
		/* NOTREACHED */
	default:
		if (erred) {
			winvars->wwerrno = WWE_SYS;
			ret = -1;
		} else {
			w->ww_pid = pid;
			w->ww_state = WWS_HASPROC;
			ret = pid;
		}
	}

	sigprocmask(SIG_SETMASK, &osigset, NULL);

	if (w->ww_socket >= 0) {
		(void) close(w->ww_socket);
		w->ww_socket = -1;
	}
	return ret;
}


int
wwgettty(winvars_t *winvars, int d, ww_tty_t *t)
{
#ifdef OLD_TTY
	if (ioctl(d, TIOCGETP, (char *)&t->ww_sgttyb) < 0)
		goto bad;
	if (ioctl(d, TIOCGETC, (char *)&t->ww_tchars) < 0)
		goto bad;
	if (ioctl(d, TIOCGLTC, (char *)&t->ww_ltchars) < 0)
		goto bad;
	if (ioctl(d, TIOCLGET, (char *)&t->ww_lmode) < 0)
		goto bad;
	if (ioctl(d, TIOCGETD, (char *)&t->ww_ldisc) < 0)
		goto bad;
#else
	if (tcgetattr(d, &t->ww_termios) < 0)
		goto bad;
#endif
	return 0;
bad:
	winvars->wwerrno = WWE_SYS;
	return -1;
}

/*
 * Set the modes of tty 'd' to 't'
 * 'o' is the current modes.  We set the line discipline only if
 * it changes, to avoid unnecessary flushing of typeahead.
 */
int
wwsettty(winvars_t *winvars, int d, ww_tty_t *t)
{
#ifdef OLD_TTY
	int i;

	/* XXX, for buggy tty drivers that don't wait for output to drain */
	while (ioctl(d, TIOCOUTQ, &i) >= 0 && i > 0)
		usleep(100000);
	if (ioctl(d, TIOCSETN, (char *)&t->ww_sgttyb) < 0)
		goto bad;
	if (ioctl(d, TIOCSETC, (char *)&t->ww_tchars) < 0)
		goto bad;
	if (ioctl(d, TIOCSLTC, (char *)&t->ww_ltchars) < 0)
		goto bad;
	if (ioctl(d, TIOCLSET, (char *)&t->ww_lmode) < 0)
		goto bad;
	if (ioctl(d, TIOCGETD, (char *)&i) < 0)
		goto bad;
	if (t->ww_ldisc != i &&
	    ioctl(d, TIOCSETD, (char *)&t->ww_ldisc) < 0)
		goto bad;
#elif defined(sun)
	/* XXX, for buggy tty drivers that don't wait for output to drain */
	(void) tcdrain(d);
#else
	if (tcsetattr(d, TCSADRAIN, &t->ww_termios) < 0)
		goto bad;
#endif
	return 0;
bad:
	winvars->wwerrno = WWE_SYS;
	return -1;
}

/*
 * The ttysize and stop-start routines must also work
 * on the control side of pseudoterminals.
 */

int
wwgetttysize(winvars_t *winvars, int d, int *r, int *c)
{
	struct winsize winsize;

	if (ioctl(d, TIOCGWINSZ, &winsize) < 0) {
		winvars->wwerrno = WWE_SYS;
		return -1;
	}
	if (winsize.ws_row != 0)
		*r = winsize.ws_row;
	if (winsize.ws_col != 0)
		*c = winsize.ws_col;
	return 0;
}

int
wwstoptty(winvars_t *winvars, int d)
{
#if !defined(OLD_TTY) && defined(TCOOFF)
	/* not guaranteed to work on the pty side */
	if (tcflow(d, TCOOFF) < 0)
#else
	if (ioctl(d, TIOCSTOP, NULL) < 0)
#endif
	{
		winvars->wwerrno = WWE_SYS;
		return -1;
	}
	return 0;
}

int
wwstarttty(winvars_t *winvars, int d)
{
#if !defined(OLD_TTY) && defined(TCOON)
	/* not guaranteed to work on the pty side */
	if (tcflow(d, TCOON) < 0)
#else
	if (ioctl(d, TIOCSTART, NULL) < 0)
#endif
	{
		winvars->wwerrno = WWE_SYS;
		return -1;
	}
	return 0;
}

void
wwunframe(winvars_t *winvars, ww_t *w)
{
	int i;

	for (i = w->ww_i.t; i < w->ww_i.b; i++) {
		int j;
		char *win = w->ww_win[i];
		char *fmap = w->ww_fmap ? w->ww_fmap[i] : 0;
		unsigned char *smap = winvars->wwsmap[i];
		union ww_char *ns = winvars->wwns[i];
		int nchanged = 0;

		for (j = w->ww_i.l; j < w->ww_i.r; j++) {
			if (win[j] & WWM_GLS)
				continue;
			win[j] |= WWM_GLS;
			if (fmap != 0)
				fmap[j] = 0;
			if (smap[j] == w->ww_index) {
				smap[j] = WWX_NOBODY;
				setcharpart(&ns[j], ' ');
				nchanged++;
			}
		}
		if (nchanged > 0)
			winvars->wwtouched[i] |= WWU_TOUCHED;
		w->ww_nvis[i] = 0;
	}

	if (w->ww_forw != &winvars->wwhead)
		wwdelete1(winvars, w->ww_forw,
			w->ww_i.t, w->ww_i.b, w->ww_i.l, w->ww_i.r);
}



#ifdef TERMINFO

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>

#include <stdio.h>
#include <stdlib.h>
#include <paths.h>
#include <unistd.h>
#include "local.h"
#include "ww.h"

/*
 * Terminfo support
 *
 * Written by Brian Buhrow
 *
 * Subsequently modified by Edward Wang
 */

/*
 * Initialize the working terminfo directory
 */
int
wwterminfoinit(winvars_t *winvars)
{
	FILE *fp;
	char buf[2048];

		/* make the directory */
	(void) snprintf(winvars->wwterminfopath, sizeof(winvars->wwterminfopath),
			"%swwinXXXXXX", _PATH_TMP);
	if (mkdtemp(winvars->wwterminfopath) == NULL) ||
	    chmod(winvars->wwterminfopath, 0755) < 0) {
		winvars->wwerrno = WWE_SYS;
		return -1;
	}
	(void) setenv("TERMINFO", winvars->wwterminfopath, 1);
		/* make a termcap entry and turn it into terminfo */
	(void) snprintf(buf, sizeof(buf), "%s/cap", winvars->wwterminfopath);
	if ((fp = fopen(buf, "w")) == NULL) {
		winvars->wwerrno = WWE_SYS;
		return -1;
	}
	(void) fprintf(fp, "%sco#%d:li#%d:%s\n",
		WWT_TERMCAP, winvars->wwncol, winvars->wwnrow, winvars->wwwintermcap);
	(void) fclose(fp);
	(void) snprintf(buf, sizeof(buf),
		"cd %s; %s cap >info 2>/dev/null; %s info >/dev/null 2>&1",
		winvars->wwterminfopath, _PATH_CAPTOINFO, _PATH_TIC);
	(void) system(buf);
	return 0;
}

/*
 * Delete the working terminfo directory at shutdown
 */
int
wwterminfoend(winvars_t *winvars)
{
	int pstat;
	pid_t pid;

	pid = vfork();
	switch (pid) {
	case -1:
		/* can't really do (or say) anything about errors */
		return -1;
	case 0:
		execl(_PATH_RM, _PATH_RM, "-rf", winvars->wwterminfopath, 0);
		_exit(1);
	}
	pid = waitpid(pid, &pstat, 0);
	if (pid == -1 || !WIFEXITED(pstat) || WEXITSTATUS(pstat) != 0)
		return -1;
	return 0;
}
#endif /* TERMINFO */
