/*
 * sun-rbootd   yamamori@kt.rim.or.jp
 */

#include "config.h"

#include <stdio.h>
#include <string.h>

#include <sys/fcntl.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <sys/ioctl.h>

#include <net/if.h>
#include <netinet/in.h>
#ifndef LINUX
# include <netinet/if_ether.h>
#else
# include <linux/if_ether.h>
#endif

#ifdef SUNOS4
# include <sys/stropts.h>
# include <net/nit_if.h>
# include <net/nit_pf.h>
# include <net/packetfilt.h>
#endif
#ifdef SOLARIS2
# include <sys/sockio.h>
# include <sys/stropts.h>
# include <sys/dlpi.h>
# include <sys/pfmod.h>
#endif
#ifdef HPUX
# include <sys/stropts.h>
# include <sys/dlpi.h>
# include <sys/dlpi_ext.h>
# include <sys/poll.h>
# include <sys/stat.h>
#endif
#ifdef HPUX_LLA
# include <sys/netio.h>
#endif
#if defined(NEWSOS_4) || defined(NETBSD)
# include <sys/errno.h>
# include <net/bpf.h>
#endif
#ifdef NETBSD
# include <net/if_dl.h>
# include <net/if_types.h>
#endif


#include "nitlib.h"

#define OFFSET(str,mem) ((int)&((struct str *)0)->mem)
#define NIT_OFFSET(str,mem) (OFFSET(str,mem)/sizeof(short))


extern int debug;

static const struct ether_addr rmp_addr = {RMP_ADDR};

#ifdef SUNOS4
# define PFMOD "pf"
# define X_IOCSETF NIOCSETF
#endif
#ifdef SOLARIS2
# define PFMOD "pfmod"
# define X_IOCSETF PFIOCSETF
#endif

#if defined(SOLARIS2) || defined(HPUX)
static int AttachDevice _P((int, int));
static int BindProtocol _P((int, int, int, int, int, int));
static int AddMultiAddress _P((int, const struct ether_addr *));
static int GetEthernetAddress _P((int, struct ether_addr *));
static int PromMode _P((int, int));
#endif
#ifdef HPUX
static int get_ppa _P((int fd, const char *device, int unit));
static int fd_w;
#endif

#if defined(NEWSOS_4) || defined(NETBSD)
static int bpf_len;
static int bpf_r_len;
static char *bpf_buf;
static char *bpf_cur;
#endif

#ifdef LINUX
static struct sockaddr linux_src;
#endif


#define MAXNUMIF 8

char *
ether_if _P((void))
{
  struct ifconf ifc;
  struct ifreq *ifrp;
  int sfd;
  static char if_name[IFNAMSIZ];
#ifdef NETBSD
  int ifr_size;
  struct ifreq cbuf[MAXNUMIF*2];
# define GET_IFR_SIZE(p) (                                 \
  ifr_size = (                                             \
    sizeof (struct ifreq) + (                              \
      (p)->ifr_addr.sa_len > sizeof (struct sockaddr) ?    \
      (p)->ifr_addr.sa_len - sizeof (struct sockaddr): 0   \
    )                                                      \
  )                                                        \
)
# define INC_IFRP(p) ((p) = (struct ifreq *)&((char *)(p))[ifr_size])
#else
  struct ifreq cbuf[MAXNUMIF];
# define GET_IFR_SIZE(p)
# define INC_IFRP(p) ((p)++)
#endif


  if ((sfd = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
    return 0;
  }
  ifc.ifc_len = sizeof cbuf;
  ifc.ifc_req = cbuf;
  if (ioctl(sfd, SIOCGIFCONF, &ifc) < 0) {
    close(sfd);
    return 0;
  }

  for (
    ifrp = cbuf;
    ifrp < (struct ifreq *)&((char *)cbuf)[ifc.ifc_len];
    INC_IFRP(ifrp)
  ) {
    GET_IFR_SIZE(ifrp);
    if (debug) {printf("%s ", ifrp->ifr_name);}
    if (ioctl(sfd, SIOCGIFFLAGS, ifrp) < 0) {
      continue;
    }
    if (
      (ifrp->ifr_flags & IFF_UP) &&
      !(ifrp->ifr_flags & (IFF_LOOPBACK|IFF_POINTOPOINT))
    ) {
      strcpy(if_name, ifrp->ifr_name);
      if (debug) {printf("-- use %s\n", if_name);}
      close(sfd);
      return if_name;
    }
  }
  close(sfd);
  return 0;
}


#ifdef NETBSD
int
get_ether_addr(char *name, struct ether_addr *ea)
{
  struct ifconf ifc;
  struct ifreq *ifrp;
  int sfd;
  int ifr_size;
  struct sockaddr_dl *sdl;
  struct ifreq cbuf[MAXNUMIF*2];

  if ((sfd = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
    return -1;
  }
  ifc.ifc_len = sizeof cbuf;
  ifc.ifc_req = cbuf;
  if (ioctl(sfd, SIOCGIFCONF, &ifc) < 0) {
    close(sfd);
    return -1;
  }
  close(sfd);

  for (
    ifrp = cbuf;
    ifrp < (struct ifreq *)&((char *)cbuf)[ifc.ifc_len];
    INC_IFRP(ifrp)
  ) {
    GET_IFR_SIZE(ifrp);
    sdl = (struct sockaddr_dl *)&(ifrp->ifr_addr);
    if (
      sdl->sdl_family == AF_LINK &&
      sdl->sdl_type == IFT_ETHER &&
      !strcmp(ifrp->ifr_name, name)
    ) {
      *ea = *(struct ether_addr *)LLADDR(sdl);
      return 0;
    }
  }
  return -1;
}
#endif


#if defined(SUNOS4) || defined(SOLARIS2)
static int
#ifdef __GNUC__
i_str(int fd, int cmd, char *ptr, int len)
#else
i_str(fd, cmd, ptr, len) char *ptr;
#endif
{
  struct strioctl si;

  si.ic_cmd = cmd;
  si.ic_timout = INFTIM;
  si.ic_len = len;
  si.ic_dp = ptr;
  return ioctl(fd, I_STR, (char *)&si);
}
#endif


int
#ifdef __GNUC__
ether_open(char *name, struct ether_addr *my_addr)
#else
ether_open(name, my_addr) char *name; struct ether_addr *my_addr;
#endif
{
  int fd;

#if defined(SUNOS4) || defined(NEWSOS_4) || defined(NETBSD) || defined(LINUX)
  struct ifreq ifr;
#endif


#ifdef SUNOS4
  if ((fd = open("/dev/nit", O_RDWR)) < 0) {
    return -1;
  }
  if (ioctl(fd, I_SRDOPT, (char *)RMSGD) < 0) {
    return -1;
  }
  strcpy(ifr.ifr_name, name);
  if (i_str(fd, NIOCBIND, (char *)&ifr, sizeof ifr) < 0) {
    return -1;
  }
#endif
#ifdef SOLARIS2
  {
    int num;
    char dev[32], *s;

    sprintf(dev, "/dev/%s", name);
    for (s = dev; *s; s++) {
      if ('0' <= *s && *s <= '9') {
	num = atoi(s);
	*s = '\0';
	break;
      }
    }
    if ((fd = open(dev, O_RDWR)) < 0) {
      return -1;
    }
    if (AttachDevice(fd, num) == 0) {
      return -1;
    }
  }
#endif
#ifdef HPUX
  {
    int num, ppa;
    char dev[32], *s;

    sprintf(dev, "/dev/%s", name);

    for (s = name; *s; s++) {
      if ('0' <= *s && *s <= '9') {
	num = atoi(s);
	break;
      }
    }
    if ((fd = open("/dev/dlpi", O_RDWR)) < 0) {
      return -1;
    }
    if ((ppa = get_ppa(fd, dev, num)) < 0) {
      perror(dev);
      return -1;
    }
    if (debug) {printf("ppa=%d\n", ppa);}
    if (AttachDevice(fd, ppa) == 0) {
      return -1;
    }
    if ((fd_w = open("/dev/dlpi", O_RDWR)) < 0) {
      return -1;
    }
    if ((ppa = get_ppa(fd_w, dev, num)) < 0) {
      perror(dev);
      return -1;
    }
    if (debug) {printf("ppa=%d\n", ppa);}
    if (AttachDevice(fd_w, ppa) == 0) {
      return -1;
    }
  }
#endif
#ifdef HPUX_LLA
  {
    char dev[32];
    sprintf(dev, "/dev/%s", name);
    if ((fd = open(dev, O_RDWR)) < 0) {
      perror(dev);
      return -1;
    }
  }
#endif
#if defined(NEWSOS_4) || defined(NETBSD)
  {
    int num;
    char dev[32];
    extern int errno;

    for (num = 0; ; num++) {
      sprintf(dev, "/dev/bpf%d", num);
      if (debug) {printf("%s ", dev);}
      if ((fd = open(dev, O_RDWR)) >= 0) {
	break;
      }
      if (errno != EBUSY && errno != EPERM) {
	return -1;
      }
    }
    if (debug) {printf("-- use %s\n", dev);}
  }
  strcpy(ifr.ifr_name, name);
  if (ioctl(fd, BIOCSETIF, &ifr) < 0) {
    return -1;
  }
#endif
#ifdef LINUX
  if ((fd = socket(PF_INET, SOCK_PACKET, htons(ETH_P_ALL))) < 0) {
    perror("SOCK_PACKET");
    return -1;
  }
  linux_src.sa_family = AF_INET;
  strcpy(linux_src.sa_data, name);
  if (bind(fd, &linux_src, sizeof(linux_src)) < 0) {
    perror("bind");
    return -1;
  }
#endif


#if defined(SOLARIS2)
  if (BindProtocol(fd, 0, 0, DL_CLDLS, 0, 0) == 0) {
    return -1;
  }
  if (PromMode(fd, DL_PROMISC_SAP) == 0) {
    return -1;
  }
  if (i_str(fd, DLIOCRAW, NULL, 0) < 0) {
    return -1;
  }
#endif
#if defined(HPUX)
  if (PromMode(fd, DL_PROMISC_PHYS) == 0) {
/*
    return -1;
*/
  }
  if (BindProtocol(fd, 22, 1, DL_HP_RAWDLS, 0, 0) == 0) {
    return -1;
  }
  if (BindProtocol(fd_w, 24, 1, DL_HP_RAWDLS, 0, 0) == 0) {
    return -1;
  }
#endif
#ifdef HPUX_LLA
  {
    struct fis s_fis;
    s_fis.reqtype = 801;
    s_fis.vtype = INTEGERTYPE;
    s_fis.value.i = HPEXT_DXSAP;
    if (ioctl(fd, NETCTRL, &s_fis) < 0) {
      perror("HPEXT_DXSAP");
      /* return -1; */
    }
    s_fis.reqtype = 802;
    s_fis.vtype = INTEGERTYPE;
    s_fis.value.i = HPEXT_SXSAP;
    if (ioctl(fd, NETCTRL, &s_fis) < 0) {
      perror("HPEXT_SXSAP");
      /* return -1; */
    }
  }
#endif
#if defined(NEWSOS_4) || defined(NETBSD)
  {
    int flag = 1;
    if (ioctl(fd, BIOCIMMEDIATE, &flag) < 0) {
      return -1;
    }
  }
  if (ioctl(fd, BIOCGBLEN, &bpf_len) < 0) {
    return -1;
  }
  if ((bpf_buf = (char *)malloc(bpf_len)) == NULL) {
    return -1;
  }
  bpf_cur = bpf_buf;
  bpf_r_len = 0;
  if (debug) {printf("bpf_len = %d\n", bpf_len);}
#endif


#if defined(SUNOS4) || defined(SOLARIS2)
  if (ioctl(fd, I_PUSH, PFMOD) < 0) {
    return -1;
  }
  {
    struct packetfilt filter;
    unsigned short *fptr;

    fptr = &filter.Pf_Filter[0];

    *fptr++ = ENF_PUSHWORD + NIT_OFFSET(ether_header, ether_type);
    *fptr++ = ENF_PUSHLIT | ENF_LE;
    *fptr++ = htons(1500);
    *fptr++ = ENF_PUSHZERO | ENF_CNOR;

    *fptr++ = ENF_PUSHWORD + NIT_OFFSET(rmp_packet, hp_llc.dsap);
    *fptr++ = ENF_PUSHLIT | ENF_CAND;
    *fptr++ = htons((IEEE_DSAP_HP<<8)|IEEE_SSAP_HP);

    *fptr++ = ENF_PUSHWORD + NIT_OFFSET(rmp_packet, hp_llc.cntrl);
    *fptr++ = ENF_PUSHLIT | ENF_CAND;
    *fptr++ = htons(IEEE_CNTL_HP);

    *fptr++ = ENF_PUSHWORD + NIT_OFFSET(rmp_packet, hp_llc.filler);
    *fptr++ = ENF_PUSHZERO | ENF_CAND;

    *fptr++ = ENF_PUSHWORD + NIT_OFFSET(rmp_packet, hp_llc.dxsap);
    *fptr++ = ENF_PUSHLIT | ENF_CAND;
    *fptr++ = htons(HPEXT_DXSAP);

    *fptr++ = ENF_PUSHWORD + NIT_OFFSET(rmp_packet, hp_llc.sxsap);
    *fptr++ = ENF_PUSHLIT | ENF_CAND;
    *fptr++ = htons(HPEXT_SXSAP);

    filter.Pf_FilterLen = fptr - &filter.Pf_Filter[0];

    if (i_str(fd, X_IOCSETF, (char *)&filter, sizeof filter) < 0) {
      return -1;
    }
  }
#endif
#if defined(NEWSOS_4) || defined(NETBSD)
  {
    static struct bpf_insn bpf_insn[] = {
# ifdef NEWSOS_4
      BPF_STMT(LdHOp, OFFSET(ether_header, ether_type)),
      BPF_JUMP(GTOp, 1500, 8, 1),
      BPF_STMT(LdOp, OFFSET(rmp_packet, hp_llc.dsap)),
      BPF_JUMP(EQOp, (IEEE_DSAP_HP<<24)|(IEEE_SSAP_HP<<16)|IEEE_CNTL_HP, 1, 6),
      BPF_STMT(LdHOp, OFFSET(rmp_packet, hp_llc.filler)),
      BPF_JUMP(EQOp, 0, 1, 4),
      BPF_STMT(LdOp, OFFSET(rmp_packet, hp_llc.dxsap)),
      BPF_JUMP(EQOp, (HPEXT_DXSAP<<16)|HPEXT_SXSAP, 1, 2),
      BPF_STMT(RetOp, 128),
      BPF_STMT(RetOp, 0)
# endif
# ifdef NETBSD
      BPF_STMT(BPF_LD|BPF_H|BPF_ABS, OFFSET(ether_header, ether_type)),
      BPF_JUMP(BPF_JMP|BPF_JEQ|BPF_K, 1500, 7, 0),
      BPF_STMT(BPF_LD|BPF_W|BPF_ABS, OFFSET(rmp_packet, hp_llc.dsap)),
      BPF_JUMP(BPF_JMP|BPF_JEQ|BPF_K,
	  (IEEE_DSAP_HP<<24)|(IEEE_SSAP_HP<<16)|IEEE_CNTL_HP, 0, 5),
      BPF_STMT(BPF_LD|BPF_H|BPF_ABS, OFFSET(rmp_packet, hp_llc.filler)),
      BPF_JUMP(BPF_JMP|BPF_JEQ|BPF_K, 0, 0, 3),
      BPF_STMT(BPF_LD|BPF_W|BPF_ABS, OFFSET(rmp_packet, hp_llc.dxsap)),
      BPF_JUMP(BPF_JMP|BPF_JEQ|BPF_K, (HPEXT_DXSAP<<16)|HPEXT_SXSAP, 0, 1),
      BPF_STMT(BPF_RET|BPF_K, 128),
      BPF_STMT(BPF_RET|BPF_K, 0)
# endif
    };
    static struct bpf_program bpf_pgm = {
      sizeof bpf_insn / sizeof (struct bpf_insn),
      bpf_insn
    };
    if (ioctl(fd, BIOCSETF, &bpf_pgm) < 0) {
      return -1;
    }
  }
#endif


#ifdef SUNOS4
  ifr.ifr_addr.sa_family = AF_UNSPEC;
  *(struct ether_addr *)ifr.ifr_addr.sa_data = rmp_addr;
  if (ioctl(fd, SIOCADDMULTI, &ifr) < 0) {
    perror("SIOCADDMULTI");
    return -1;
  }
#endif
#if defined(SOLARIS2) || defined(HPUX)
  if (AddMultiAddress(fd, &rmp_addr) == 0) {
    return -1;
  }
#endif
#ifdef HPUX_LLA
  {
    struct fis s_fis;
    s_fis.reqtype = DELETE_MULTICAST;
    s_fis.vtype = sizeof rmp_addr;
    *(struct ether_addr *)s_fis.value.s = rmp_addr;
    ioctl(fd, NETCTRL, &s_fis);
    s_fis.reqtype = ADD_MULTICAST;
    s_fis.vtype = sizeof rmp_addr;
    *(struct ether_addr *)s_fis.value.s = rmp_addr;
    if (ioctl(fd, NETCTRL, &s_fis) < 0) {
      perror("ADD_MULTICAST");
      /* return -1; */
    }
  }
#endif
#if defined(NEWSOS_4) || defined(NETBSD)
  if (ioctl(fd, BIOCPROMISC, NULL) < 0) {
    perror("BIOCPROMISC");
    return -1;
  }
#endif
#ifdef LINUX
  strcpy(ifr.ifr_name, name);
  if (ioctl(fd, SIOCGIFFLAGS, &ifr) < 0 ) {
    return -1;
  }
  ifr.ifr_flags |= IFF_PROMISC;
  if (ioctl(fd, SIOCSIFFLAGS, &ifr) < 0 ) {
    perror("IFF_PROMISC");
    return -1;
  }
#endif


#if defined(SUNOS4) || defined(NEWSOS_4)
  if (ioctl(fd, SIOCGIFADDR, &ifr) < 0) {
    perror("SIOCGIFADDR");
    return -1;
  }
  *my_addr = *(struct ether_addr *)ifr.ifr_addr.sa_data;
#endif
#if defined(SOLARIS2) || defined(HPUX)
  if (GetEthernetAddress(fd, my_addr) == 0) {
    return -1;
  }
#endif
#ifdef HPUX_LLA
  {
    struct fis s_fis;
    s_fis.reqtype = LOCAL_ADDRESS;
    ioctl(fd, NETSTAT, &s_fis);
    *my_addr = *(struct ether_addr *)s_fis.value.s;
  }
#endif
#ifdef NETBSD
  if (get_ether_addr(name, my_addr) < 0) {
    return -1;
  }
#endif
#ifdef LINUX
  strcpy(ifr.ifr_name, name);
  if (ioctl(fd, SIOCGIFHWADDR, &ifr) < 0 ) {
    perror("SIOCGIFHWADDR");
    return -1;
  }
  *my_addr = *(struct ether_addr *)ifr.ifr_addr.sa_data;
#endif


#if defined(SUNOS4) || defined(SOLARIS2) || defined(HPUX)
  ioctl(fd, I_FLUSH, (char *)FLUSHR);
#endif
#if defined(NEWSOS_4) || defined(NETBSD)
  ioctl(fd, BIOCFLUSH, NULL);
#endif

  return fd;
}


#ifdef SUNOS4
int
#ifdef __GNUC__
ether_write(int fd, char *pkt, int len)
#else
ether_write(fd, pkt, len) char *pkt;
#endif
{
  struct sockaddr sa;
  struct strbuf cbuf;
  struct strbuf dbuf;

  sa.sa_family = AF_UNSPEC;
  *(struct ether_header *)sa.sa_data = *(struct ether_header *)pkt;

  cbuf.maxlen = cbuf.len = sizeof sa;
  cbuf.buf = (char *)&sa;

  dbuf.maxlen = dbuf.len = len - sizeof (struct ether_header);
  dbuf.buf = pkt + sizeof (struct ether_header);

  return putmsg(fd, &cbuf, &dbuf, 0);
}
#endif


#if defined(NEWSOS_4) || defined(NETBSD)
int
#ifdef __GNUC__
ether_read(int fd, char *pkt, int len)
#else
ether_read(fd, pkt, len) char *pkt;
#endif
{
  if (bpf_cur - bpf_buf >= bpf_r_len) {
#ifdef __NetBSD__
    {
      fd_set r;
      FD_ZERO(&r);
      FD_SET(fd, &r);
      select(fd+1, &r, NULL, NULL, NULL);
    }
#endif
    bpf_r_len = read(fd, bpf_buf, bpf_len);
    bpf_cur = bpf_buf;
  }

  memcpy(pkt, bpf_cur + ((struct bpf_hdr *)bpf_cur)->bh_hdrlen, len);

  bpf_cur += BPF_WORDALIGN(
    ((struct bpf_hdr *)bpf_cur)->bh_hdrlen +
    ((struct bpf_hdr *)bpf_cur)->bh_caplen
  );

  return len;
}
#endif


#define HIGH_LOW(s) (((s)>>8)&0xff), ((s)&0xff)

#ifdef LINUX
int
ether_read(int fd, char *pkt, int len)
{
  int r_len;

  static const char read_filter[] = {
    IEEE_DSAP_HP, IEEE_SSAP_HP,
    HIGH_LOW(IEEE_CNTL_HP),
    HIGH_LOW(0),
    HIGH_LOW(HPEXT_DXSAP),
    HIGH_LOW(HPEXT_SXSAP)
  };

  while (
    !(
      (r_len = read(fd, pkt, len)) > 0 &&
      ntohs(((struct ether_header *)pkt)->ether_type) <= 1500 &&
      memcmp(
	&((struct rmp_packet *)pkt)->hp_llc,
	read_filter,
	sizeof (struct hp_llc)
      ) == 0
    )
  ) {}

  return r_len;
}
int
ether_write(int fd, char *pkt, int len)
{
  return sendto(fd, pkt, len, 0, &linux_src, sizeof linux_src);
}
#endif


#if defined(LINUX) || defined(HPUX) || defined(HPUX_LLA)
#define LB_SIZE 80
int
#ifdef __GNUC__
ether_ntohost(char *hostname, struct ether_addr *ea)
#else
ether_ntohost(hostname, ea) char *hostname; struct ether_addr *ea;
#endif
{
  int i, c;
  FILE *fd;
  int eab[6];
  char buf[LB_SIZE];

  if ((fd = fopen("/etc/ethers", "r")) == NULL) {
    perror("/etc/ethers");
    return -1;
  }

  for (;;) {
    for (i = 0; ; i++) {
      c = getc(fd);
      if (c == EOF) {
	fclose(fd);
	return -1;
      }
      if (c == '\n') {
	break;
      }
      if (c == '#' || i >= LB_SIZE-1) {
	while ((c = getchar()) != EOF && c != '\n') {}
	break;
      }
      buf[i] = c;
    }
    buf[i] = '\0';
    if (
      sscanf(
	buf, "%2x:%2x:%2x:%2x:%2x:%2x%*[ \t]%s",
	&eab[0], &eab[1], &eab[2], &eab[3], &eab[4], &eab[5],
	hostname
      ) == 7
    ) {
      for (i = 0; ea->ether_addr_octet[i] == eab[i];) {
	if (++i >= 6) {
	  fclose(fd);
	  return 0;
	}
      }
    }
  }
}
#endif


#ifdef NEWSOS_4
void
setsid _P((void))
{
  int fd;

  fd = open("/dev/tty", O_RDONLY);
  ioctl(fd, TIOCNOTTY, NULL);
  close(fd);
}
#endif


#if defined(HPUX)
#define MAXDLBUF 8192

static unsigned long ctlb[MAXDLBUF];
static struct strbuf ctls = {
  MAXDLBUF,
  0,
  (char *)ctlb
};

int
#ifdef __GNUC__
ether_read(int fd, char *pkt, int len)
#else
ether_read(fd, pkt, len) char *pkt;
#endif
{
  int flags;
  struct strbuf data;

  static const char read_filter[] = {
    IEEE_DSAP_HP, IEEE_SSAP_HP,
    HIGH_LOW(IEEE_CNTL_HP),
    HIGH_LOW(0),
    HIGH_LOW(HPEXT_DXSAP),
    HIGH_LOW(HPEXT_SXSAP)
  };

  do {
    flags = 0;
    data.buf = pkt;
    data.maxlen = MAXDLBUF;
    data.len = 0;
    if (getmsg(fd, &ctls, &data, &flags) < 0) {
      perror("getmsg");
      return -1;
    }
  } while (
    !(
      ntohs(((struct ether_header *)pkt)->ether_type) <= 1500 &&
      memcmp(
	&((struct rmp_packet *)pkt)->hp_llc,
	read_filter,
	sizeof (struct hp_llc)
      ) == 0
    )
  );

  return data.len;
}


int
#ifdef __GNUC__
ether_write(int fd, char *pkt, int len)
#else
ether_write(fd, pkt, len) char *pkt;
#endif
{
  struct strbuf data;

  ctls.len = sizeof (dl_hp_rawdata_req_t);
  ((dl_hp_rawdata_req_t *)ctlb)->dl_primitive = DL_HP_RAWDATA_REQ;
  data.buf = pkt;
  data.maxlen = data.len = len;
  if (putmsg(fd_w, &ctls, &data, 0) < 0) {
    perror("putmsg");
    return -1;
  }
}
#endif


#ifdef HPUX_LLA
int
#ifdef __GNUC__
ether_read(int fd, char *pkt, int len)
#else
ether_read(fd, pkt, len) char *pkt;
#endif
{
  int n;
  struct fis s_fis;

  n = read(fd, pkt+24, len-24);
  if (n <= 0) {
    perror("read");
    printf("read %d\n", n);
    exit(1);
  }

  s_fis.reqtype = FRAME_HEADER;
  ioctl(fd, NETSTAT, &s_fis);
  memcpy(pkt, s_fis.value.s, 24);

  return n + 24;
}

int
#ifdef __GNUC__
ether_write(int fd, char *pkt, int len)
#else
ether_write(fd, pkt, len) char *pkt;
#endif
{
  struct fis s_fis;

  s_fis.reqtype = LOG_DEST_ADDR;
  s_fis.vtype = 6;
  *(struct ether_addr *)s_fis.value.s = *(struct ether_addr *)pkt;
  if (ioctl(fd, NETCTRL, &s_fis) < 0) {
    perror("LOG_DEST_ADDR");
  }

  write(fd, pkt+24, len-24);
}
#endif

#if defined(HPUX) || defined(HPUX_LLA)
setlinebuf() {}
#endif


/* --- Following sources are partially taken from "cap60" ---*/

#if defined(SOLARIS2) || defined(HPUX)


/*
 * sdlpi.c - Simple "protocol" level interface to Streams based DLPI
 * (SunOS 5.x) (derived from snitp.c SunOS 4.x module)
 *
 *  Provides ability to read/write packets at ethernet level
 *
 *
 * Copyright (c) 1988 by The Trustees of Columbia University 
 *  in the City of New York.
 *
 * Permission is granted to any individual or institution to use,
 * copy, or redistribute this software so long as it is not sold for
 * profit, provided that this notice and the original copyright
 * notices are retained.  Columbia University nor the author make no
 * representations about the suitability of this software for any
 * purpose.  It is provided "as is" without express or implied
 * warranty.
 *
 * Edit History:
 *
 *  July 1988  CCKim Created
 *  April '91  djh   Add Phase 2 support
 *  May-June 93  montjoy@thor.ece.uc.EDU,
 *               appro@fy.chalmers.se        SunOS 5.x support
 *
 */

static char columbia_copyright[] = "Copyright (c) 1988 by The Trustees of \
Columbia University in the City of New York";

/*
 * DLPI Support Routines
 *
 */

static int
#ifdef __GNUC__
Acknoledge(union DL_primitives *dlp_p, int ack, char *msg)
#else
Acknoledge(dlp_p, ack, msg) union DL_primitives *dlp_p; char *msg;
#endif
{
        if (dlp_p->dl_primitive != ack) {
		fprintf(stderr,"dlpi: %s is nacked.\n",msg);
		if (dlp_p->dl_primitive == DL_ERROR_ACK)
			fprintf(stderr,	"dlpi: dlpi_errno %d\n"
					"dlpi: unix_errno %d\n",
			dlp_p->error_ack.dl_errno,
			dlp_p->error_ack.dl_unix_errno);
	   	else
			fprintf(stderr,"dlpi: spiritual primitive %d.\n",
			dlp_p->dl_primitive);
		return(0);
	}
	return(1);
}

static int
#ifdef __GNUC__
AttachDevice(int fd, int devno)
#else
AttachDevice(fd, devno)
#endif
{
  	int retval;
  	int flags = RS_HIPRI;
  	struct strbuf ctlbuf;
  	union DL_primitives rcvbuf;
  	dl_attach_req_t Request;


  	/* bind to underlying interface */
  	Request.dl_primitive 	= DL_ATTACH_REQ;
  	Request.dl_ppa 		= devno;
  	ctlbuf.len 		= sizeof(Request);
  	ctlbuf.buf 		= (caddr_t)&Request;

  	if (putmsg(fd, &ctlbuf ,NULL,0)  < 0) {
    		perror("Attach Device:");
    		return(0);
  	}

  	ctlbuf.maxlen = sizeof(union DL_primitives);
  	ctlbuf.len = 0;
  	ctlbuf.buf = (char *)&rcvbuf;
  	if ((retval = getmsg(fd, &ctlbuf ,NULL, &flags))  < 0) {
    		perror("Attach Device ack!");
    		return(0);
  	}

	return (Acknoledge(&rcvbuf, DL_OK_ACK, "DL_ATTACH_REQ"));
}

static int
#ifdef __GNUC__
BindProtocol(
  int fd, int sap, int max_conind,
  int service_mode, int conn_mgmt, int xidtest_flg
)
#else
BindProtocol(fd, sap, max_conind, service_mode, conn_mgmt, xidtest_flg)
#endif
{
  	int retval;
  	int flags = RS_HIPRI;
  	struct strbuf ctlbuf;
  	union DL_primitives rcvbuf;
  	dl_bind_req_t   BindRequest;


  	BindRequest.dl_primitive 		= DL_BIND_REQ;
  	BindRequest.dl_sap       		= sap;
  	BindRequest.dl_max_conind       	= max_conind;
  	BindRequest.dl_service_mode       	= service_mode;
  	BindRequest.dl_conn_mgmt       		= conn_mgmt;
  	BindRequest.dl_xidtest_flg       	= xidtest_flg;
	
  	ctlbuf.len = sizeof(BindRequest);
  	ctlbuf.buf = (caddr_t)&BindRequest;
	
  	if (putmsg(fd, &ctlbuf ,NULL,0)  < 0) {
    		perror("Bind Protocol:");
    		return(0);
  	}

  	ctlbuf.maxlen = sizeof(union DL_primitives);
  	ctlbuf.len = 0;
  	ctlbuf.buf = (char *)&rcvbuf;
  	if ((retval = getmsg(fd, &ctlbuf ,NULL, &flags))  < 0) {
    		perror("Bind Protocol ACK!");
    		return(0);
  	}

	return (Acknoledge(&rcvbuf, DL_BIND_ACK, "DL_BIND_REQ"));
}

static int
#ifdef __GNUC__
PromMode(int fd, int level)
#else
PromMode(fd, level)
#endif
{
  	int retval;
  	int flags = RS_HIPRI;
  	struct strbuf ctlbuf;
  	union DL_primitives rcvbuf;
	dl_promiscon_req_t PromRequest;


  	PromRequest.dl_primitive 		= DL_PROMISCON_REQ;
  	PromRequest.dl_level 			= level;

        ctlbuf.len = sizeof(PromRequest);
        ctlbuf.buf = (caddr_t)&PromRequest;

        if (putmsg(fd, &ctlbuf ,NULL,0)  < 0) {
                perror("Prom Mode:");
                return(0);
        }

        ctlbuf.maxlen = sizeof(union DL_primitives);
        ctlbuf.len = 0;
        ctlbuf.buf = (char *)&rcvbuf;
        if ((retval = getmsg(fd, &ctlbuf ,NULL, &flags))  < 0) {
                perror("Prom Mode ack!");
                return(0);
        }

	return (Acknoledge(&rcvbuf, DL_OK_ACK, "DL_PROMISCON_REQ"));
}


static int
#ifdef __GNUC__
GetEthernetAddress(int fd, struct ether_addr *EtherBuf)
#else
GetEthernetAddress(fd, EtherBuf) struct ether_addr *EtherBuf;
#endif
{
  	int retval;
  	int flags = RS_HIPRI;
  	char buf[80];
  	union DL_primitives rcvbuf;
  	dl_phys_addr_req_t PRequest;
  	struct strbuf ctlbuf;


  	PRequest.dl_primitive  = DL_PHYS_ADDR_REQ;
  	PRequest.dl_addr_type  = DL_CURR_PHYS_ADDR;
  	ctlbuf.len = sizeof(PRequest);
  	ctlbuf.buf = (caddr_t)&PRequest;

  	if (putmsg(fd, &ctlbuf, NULL,0)  < 0) {
    		perror("Ethernet Address:");
    		return(-1);
  	}

  	ctlbuf.maxlen = sizeof(union DL_primitives);
  	ctlbuf.len = 0;
  	ctlbuf.buf = (char *)&rcvbuf;
  	if ((retval = getmsg(fd, &ctlbuf ,NULL, &flags)) < 0) {
    		perror("Ethernet Address ack!");
    		return(-1);
  	}

	if (Acknoledge(&rcvbuf, DL_PHYS_ADDR_ACK, "DL_PHYS_ADDR_REQ")) {
	  *EtherBuf = *(struct ether_addr *)
	    &ctlbuf.buf[rcvbuf.physaddr_ack.dl_addr_offset];
		return(1);
	}
	return(0);
}


static int
#ifdef __GNUC__
AddMultiAddress(int fd, const struct ether_addr *multi)
#else
AddMultiAddress(fd, multi) struct ether_addr *multi;
#endif
{
  	int retval;
  	int flags = RS_HIPRI;
  	u_char buf[512];
  	union DL_primitives rcvbuf;
  	struct strbuf databuf;
  	struct strbuf ctlbuf;
  	dl_enabmulti_req_t *MultiRequest = (dl_enabmulti_req_t *)buf;

	
  	MultiRequest->dl_primitive 	= DL_ENABMULTI_REQ;
  	MultiRequest->dl_addr_length	= sizeof(struct ether_addr);
  	MultiRequest->dl_addr_offset	= DL_ENABMULTI_REQ_SIZE;

	*(struct ether_addr *)&buf[DL_ENABMULTI_REQ_SIZE] = *multi;
	
  	ctlbuf.maxlen = 0;
  	ctlbuf.len = DL_ENABMULTI_REQ_SIZE + sizeof(struct ether_addr);
  	ctlbuf.buf = (caddr_t)buf;

  	if ((retval =  putmsg(fd, &ctlbuf ,NULL, flags)) < 0) {
    		perror("bogus2");
    		return(0);
  	}

  	ctlbuf.maxlen = sizeof(rcvbuf);
  	ctlbuf.len = 0;
  	ctlbuf.buf = (char *)&rcvbuf;

  	databuf.maxlen = 512;
  	databuf.len    = 0;
  	databuf.buf    = (char *)buf;

  	if ((retval = getmsg(fd, &ctlbuf, &databuf, &flags)) < 0) {
    		perror("bogus!");
    		return(0);
  	}

	return (Acknoledge(&rcvbuf, DL_OK_ACK, "DL_ENABMULTI_REQ"));
}
#endif /* SOLARIS2 || HPUX */


/* --- Following code is partially taken from "libpcap" ---*/

#ifdef HPUX

/*
 * Copyright (c) 1993, 1994, 1995, 1996, 1997
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: (1) source code distributions
 * retain the above copyright notice and this paragraph in its entirety, (2)
 * distributions including binary code include the above copyright notice and
 * this paragraph in its entirety in the documentation or other materials
 * provided with the distribution, and (3) all advertising materials mentioning
 * features or use of this software display the following acknowledgement:
 * ``This product includes software developed by the University of California,
 * Lawrence Berkeley Laboratory and its contributors.'' Neither the name of
 * the University nor the names of its contributors may be used to endorse
 * or promote products derived from this software without specific prior
 * written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 * This code contributed by Atanu Ghosh (atanu@cs.ucl.ac.uk),
 * University College London.
 */



/* Determine ppa number that specifies ifname */
static int
#ifdef __GNUC__
get_ppa(int fd, const char *device, int unit)
#else
get_ppa(fd, device, unit) char *device;
#endif
{
	register dl_hp_ppa_ack_t *ap;
	register dl_hp_ppa_info_t *ip;
	register int i;
	register unsigned long majdev;
	dl_hp_ppa_req_t	req;
	struct stat statbuf;
	unsigned long buf[MAXDLBUF];
	int flags = 0;
	struct strbuf ctl;

	if (stat(device, &statbuf) < 0) {
	  perror(device);
	  return -1;
	}
	majdev = major(statbuf.st_rdev);

	memset((char *)&req, 0, sizeof(req));
	req.dl_primitive = DL_HP_PPA_REQ;
	ctl.maxlen = 0;
	ctl.len = sizeof(req);
	ctl.buf = (char *)&req;
	if (putmsg(fd, &ctl, (struct strbuf *) NULL, flags) < 0) {
	  perror("DL_HP_PPA_REQ");
	  return -1;
	}

	memset((char *)buf, 0, sizeof(buf));
	ctl.maxlen = MAXDLBUF;
	ctl.len = 0;
	ctl.buf = buf;
	if (getmsg(fd, &ctl, (struct strbuf*)NULL, &flags) < 0) {
	  perror("DL_HP_PPA_REQ ack");
	  return -1;
	}
	if (Acknoledge(&buf, DL_HP_PPA_ACK, "DL_HP_PPA_ACK") == 0) {
	  return -1;
	}

	ap = (dl_hp_ppa_ack_t *)buf;
	ip = (dl_hp_ppa_info_t *)((u_char *)ap + ap->dl_offset);

        for(i = 0; i < ap->dl_count; i++) {
	  if (ip->dl_mjr_num == majdev && ip->dl_instance_num == unit) {
	    break;
	  }
	  ip = (dl_hp_ppa_info_t *)((u_char *)ip + ip->dl_next_offset);
        }
        if (i == ap->dl_count) {
	  fputs("can't find PPA\n", stderr);
	  return -1;
        }
        if (ip->dl_hdw_state == HDW_DEAD) {
	  fputs("hardware state: DOWN\n", stderr);
	  return -1;
        }
        return ((int)ip->dl_ppa);
}
#endif /* HPUX */
