/*
 * Copyright 2009, Google Inc.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 *     * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above
 * copyright notice, this list of conditions and the following disclaimer
 * in the documentation and/or other materials provided with the
 * distribution.
 *     * Neither the name of Google Inc. nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */


#include "core/cross/client.h"
#include "core/cross/bitmap.h"
#include "core/cross/texture.h"
#include "tests/common/win/testing_common.h"
#include "base/file_path.h"
#include "utils/cross/file_path_utils.h"

namespace o3d {

class BitmapTest : public testing::Test {
};

// The first 128 bytes of tga-256x256-24bit.tga, converted to BGRX format.
static uint8 ktga256x256_24bit_BGRX[128] = {
  0x36, 0x87, 0xbc, 0xff, 0x36, 0x87, 0xbc, 0xff,
  0x35, 0x83, 0xb5, 0xff, 0x34, 0x83, 0xb5, 0xff,
  0x35, 0x83, 0xb5, 0xff, 0x35, 0x82, 0xb5, 0xff,
  0x35, 0x82, 0xb4, 0xff, 0x35, 0x82, 0xb4, 0xff,
  0x35, 0x82, 0xb4, 0xff, 0x35, 0x82, 0xb4, 0xff,
  0x35, 0x82, 0xb4, 0xff, 0x35, 0x82, 0xb4, 0xff,
  0x35, 0x82, 0xb4, 0xff, 0x35, 0x82, 0xb4, 0xff,
  0x35, 0x82, 0xb4, 0xff, 0x35, 0x82, 0xb4, 0xff,
  0x35, 0x82, 0xb4, 0xff, 0x35, 0x82, 0xb4, 0xff,
  0x35, 0x83, 0xb5, 0xff, 0x37, 0x85, 0xb8, 0xff,
  0x36, 0x85, 0xb8, 0xff, 0x35, 0x86, 0xb7, 0xff,
  0x35, 0x87, 0xb7, 0xff, 0x35, 0x88, 0xb7, 0xff,
  0x35, 0x89, 0xb7, 0xff, 0x34, 0x8a, 0xb7, 0xff,
  0x34, 0x8b, 0xb7, 0xff, 0x32, 0x8c, 0xb5, 0xff,
  0x31, 0x8c, 0xb1, 0xff, 0x2f, 0x8d, 0xaf, 0xff,
  0x1d, 0x8d, 0x96, 0xff, 0x1b, 0x8f, 0x90, 0xff,
};

static uint8 ktga256x256_32bit_BGRA[128] = {
  0x36, 0x87, 0xbc, 0x7d, 0x36, 0x87, 0xbc, 0x7c,
  0x35, 0x83, 0xb5, 0x78, 0x34, 0x83, 0xb5, 0x77,
  0x35, 0x83, 0xb5, 0x77, 0x35, 0x82, 0xb5, 0x76,
  0x35, 0x82, 0xb4, 0x76, 0x35, 0x82, 0xb4, 0x76,
  0x35, 0x82, 0xb4, 0x76, 0x35, 0x82, 0xb4, 0x77,
  0x35, 0x82, 0xb4, 0x76, 0x35, 0x82, 0xb4, 0x77,
  0x35, 0x82, 0xb4, 0x76, 0x35, 0x82, 0xb4, 0x76,
  0x35, 0x82, 0xb4, 0x76, 0x35, 0x82, 0xb4, 0x76,
  0x35, 0x82, 0xb4, 0x76, 0x35, 0x82, 0xb4, 0x76,
  0x35, 0x83, 0xb5, 0x77, 0x37, 0x85, 0xb8, 0x7a,
  0x36, 0x85, 0xb8, 0x7a, 0x35, 0x86, 0xb7, 0x7a,
  0x35, 0x87, 0xb7, 0x7a, 0x35, 0x88, 0xb7, 0x7c,
  0x35, 0x89, 0xb7, 0x7c, 0x34, 0x8a, 0xb7, 0x7d,
  0x34, 0x8b, 0xb7, 0x7e, 0x32, 0x8c, 0xb5, 0x7e,
  0x31, 0x8c, 0xb1, 0x7d, 0x2f, 0x8d, 0xaf, 0x7c,
  0x1d, 0x8d, 0x96, 0x75, 0x1b, 0x8f, 0x90, 0x75,
};

static uint8 kjpg256x256_BGRX[128] = {
  0x3a, 0x88, 0xbd, 0xff, 0x38, 0x86, 0xbb, 0xff,
  0x36, 0x85, 0xb8, 0xff, 0x34, 0x83, 0xb6, 0xff,
  0x36, 0x82, 0xb6, 0xff, 0x35, 0x82, 0xb3, 0xff,
  0x36, 0x82, 0xb3, 0xff, 0x36, 0x82, 0xb3, 0xff,
  0x35, 0x81, 0xb5, 0xff, 0x35, 0x81, 0xb5, 0xff,
  0x35, 0x81, 0xb5, 0xff, 0x35, 0x81, 0xb5, 0xff,
  0x35, 0x81, 0xb5, 0xff, 0x35, 0x81, 0xb5, 0xff,
  0x35, 0x81, 0xb5, 0xff, 0x35, 0x81, 0xb5, 0xff,
  0x34, 0x82, 0xb1, 0xff, 0x35, 0x82, 0xb3, 0xff,
  0x35, 0x84, 0xb7, 0xff, 0x36, 0x85, 0xb8, 0xff,
  0x36, 0x85, 0xb8, 0xff, 0x35, 0x87, 0xb7, 0xff,
  0x35, 0x87, 0xb7, 0xff, 0x36, 0x88, 0xb7, 0xff,
  0x32, 0x8a, 0xb3, 0xff, 0x34, 0x8c, 0xb5, 0xff,
  0x35, 0x8b, 0xb5, 0xff, 0x33, 0x8b, 0xb4, 0xff,
  0x32, 0x8d, 0xb2, 0xff, 0x2b, 0x8f, 0xa8, 0xff,
  0x21, 0x8d, 0x99, 0xff, 0x1a, 0x8f, 0x8e, 0xff,
};

static uint8 kpng256x256_24bit_BGRX[128] = {
  0x36, 0x87, 0xbc, 0xff, 0x36, 0x87, 0xbc, 0xff,
  0x35, 0x83, 0xb5, 0xff, 0x34, 0x83, 0xb5, 0xff,
  0x35, 0x83, 0xb5, 0xff, 0x35, 0x82, 0xb5, 0xff,
  0x35, 0x82, 0xb4, 0xff, 0x35, 0x82, 0xb4, 0xff,
  0x35, 0x82, 0xb4, 0xff, 0x35, 0x82, 0xb4, 0xff,
  0x35, 0x82, 0xb4, 0xff, 0x35, 0x82, 0xb4, 0xff,
  0x35, 0x82, 0xb4, 0xff, 0x35, 0x82, 0xb4, 0xff,
  0x35, 0x82, 0xb4, 0xff, 0x35, 0x82, 0xb4, 0xff,
  0x35, 0x82, 0xb4, 0xff, 0x35, 0x82, 0xb4, 0xff,
  0x35, 0x83, 0xb5, 0xff, 0x37, 0x85, 0xb8, 0xff,
  0x36, 0x85, 0xb8, 0xff, 0x35, 0x86, 0xb7, 0xff,
  0x35, 0x87, 0xb7, 0xff, 0x35, 0x88, 0xb7, 0xff,
  0x35, 0x89, 0xb7, 0xff, 0x34, 0x8a, 0xb7, 0xff,
  0x34, 0x8b, 0xb7, 0xff, 0x32, 0x8c, 0xb5, 0xff,
  0x31, 0x8c, 0xb1, 0xff, 0x2f, 0x8d, 0xaf, 0xff,
  0x1d, 0x8d, 0x96, 0xff, 0x1b, 0x8f, 0x90, 0xff,
};

static uint8 kpng256x256_24bit_interlaced_BGRX[128] = {
  0x36, 0x87, 0xbc, 0xff, 0x36, 0x87, 0xbc, 0xff,
  0x35, 0x83, 0xb5, 0xff, 0x34, 0x83, 0xb5, 0xff,
  0x35, 0x83, 0xb5, 0xff, 0x35, 0x82, 0xb5, 0xff,
  0x35, 0x82, 0xb4, 0xff, 0x35, 0x82, 0xb4, 0xff,
  0x35, 0x82, 0xb4, 0xff, 0x35, 0x82, 0xb4, 0xff,
  0x35, 0x82, 0xb4, 0xff, 0x35, 0x82, 0xb4, 0xff,
  0x35, 0x82, 0xb4, 0xff, 0x35, 0x82, 0xb4, 0xff,
  0x35, 0x82, 0xb4, 0xff, 0x35, 0x82, 0xb4, 0xff,
  0x35, 0x82, 0xb4, 0xff, 0x35, 0x82, 0xb4, 0xff,
  0x35, 0x83, 0xb5, 0xff, 0x37, 0x85, 0xb8, 0xff,
  0x36, 0x85, 0xb8, 0xff, 0x35, 0x86, 0xb7, 0xff,
  0x35, 0x87, 0xb7, 0xff, 0x35, 0x88, 0xb7, 0xff,
  0x35, 0x89, 0xb7, 0xff, 0x34, 0x8a, 0xb7, 0xff,
  0x34, 0x8b, 0xb7, 0xff, 0x32, 0x8c, 0xb5, 0xff,
  0x31, 0x8c, 0xb1, 0xff, 0x2f, 0x8d, 0xaf, 0xff,
  0x1d, 0x8d, 0x96, 0xff, 0x1b, 0x8f, 0x90, 0xff,
};

static uint8 kpng256x256_32bit_BGRA[128] = {
  0x36, 0x87, 0xbc, 0xd4, 0x36, 0x87, 0xbc, 0xfa,
  0x35, 0x83, 0xb5, 0xff, 0x34, 0x83, 0xb5, 0xfe,
  0x35, 0x83, 0xb5, 0xf3, 0x35, 0x82, 0xb5, 0xcf,
  0x35, 0x82, 0xb4, 0x9a, 0x35, 0x82, 0xb4, 0x75,
  0x35, 0x82, 0xb4, 0x67, 0x35, 0x82, 0xb4, 0x7d,
  0x35, 0x82, 0xb4, 0xa8, 0x35, 0x82, 0xb4, 0xdd,
  0x35, 0x82, 0xb4, 0xf7, 0x35, 0x82, 0xb4, 0xff,
  0x35, 0x82, 0xb4, 0xff, 0x35, 0x82, 0xb4, 0xf8,
  0x35, 0x82, 0xb4, 0xaf, 0x35, 0x82, 0xb4, 0x68,
  0x35, 0x83, 0xb5, 0x5a, 0x37, 0x85, 0xb8, 0x6a,
  0x36, 0x85, 0xb8, 0xaf, 0x35, 0x86, 0xb7, 0xf8,
  0x35, 0x87, 0xb7, 0xff, 0x35, 0x88, 0xb7, 0xff,
  0x35, 0x89, 0xb7, 0xff, 0x34, 0x8a, 0xb7, 0xff,
  0x34, 0x8b, 0xb7, 0xff, 0x32, 0x8c, 0xb5, 0xff,
  0x31, 0x8c, 0xb1, 0xff, 0x2f, 0x8d, 0xaf, 0xff,
  0x1d, 0x8d, 0x96, 0xff, 0x1b, 0x8f, 0x90, 0xff,
};

static uint8 kpng256x256_8bit_palette_BGRX[128] = {
  0x36, 0x89, 0xbb, 0xff, 0x36, 0x89, 0xbb, 0xff,
  0x35, 0x81, 0xb5, 0xff, 0x32, 0x84, 0xb5, 0xff,
  0x35, 0x81, 0xb5, 0xff, 0x35, 0x81, 0xb5, 0xff,
  0x35, 0x81, 0xb5, 0xff, 0x35, 0x81, 0xb5, 0xff,
  0x35, 0x81, 0xb5, 0xff, 0x35, 0x81, 0xb5, 0xff,
  0x35, 0x81, 0xb5, 0xff, 0x35, 0x81, 0xb5, 0xff,
  0x35, 0x81, 0xb5, 0xff, 0x35, 0x81, 0xb5, 0xff,
  0x35, 0x81, 0xb5, 0xff, 0x35, 0x81, 0xb5, 0xff,
  0x35, 0x81, 0xb5, 0xff, 0x35, 0x81, 0xb5, 0xff,
  0x35, 0x81, 0xb5, 0xff, 0x35, 0x85, 0xba, 0xff,
  0x35, 0x85, 0xba, 0xff, 0x36, 0x86, 0xb6, 0xff,
  0x36, 0x86, 0xb6, 0xff, 0x36, 0x86, 0xb6, 0xff,
  0x38, 0x8b, 0xb6, 0xff, 0x38, 0x8b, 0xb6, 0xff,
  0x38, 0x8b, 0xb6, 0xff, 0x33, 0x8b, 0xb2, 0xff,
  0x33, 0x8b, 0xb2, 0xff, 0x33, 0x8b, 0xb2, 0xff,
  0x1f, 0x8d, 0x91, 0xff, 0x1f, 0x8d, 0x91, 0xff,
};

static uint8 kpng20x14_4bit_palette_BGRX[128] = {
  0xed, 0xed, 0xed, 0xff, 0xff, 0xff, 0xff, 0xff,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0xff, 0xff, 0xff, 0xff, 0xed, 0xed, 0xed, 0xff,
  0xff, 0xff, 0xff, 0xff, 0x54, 0x54, 0x54, 0xff,
  0x33, 0x33, 0x33, 0xff, 0x33, 0x33, 0x33, 0xff,
  0x33, 0x33, 0x33, 0xff, 0x33, 0x33, 0x33, 0xff,
  0x33, 0x33, 0x33, 0xff, 0x33, 0x33, 0x33, 0xff,
  0x33, 0x33, 0x33, 0xff, 0x33, 0x33, 0x33, 0xff,
  0x54, 0x54, 0x54, 0xff, 0x33, 0x33, 0x33, 0xff,
};

static uint8 kdxt1_256x256[128] = {
  0x47, 0xbc, 0x06, 0xb4, 0x5a, 0x6a, 0x6a, 0xea,
  0x27, 0xb4, 0x06, 0xb4, 0x57, 0x57, 0x57, 0x5e,
  0x06, 0xb4, 0x07, 0xb4, 0xaa, 0xaa, 0xaa, 0xaa,
  0x06, 0xb4, 0x07, 0xb4, 0xaa, 0xaa, 0xaa, 0xaa,
  0x46, 0xbc, 0x07, 0xb4, 0xd5, 0xf5, 0xf5, 0xa5,
  0x46, 0xbc, 0x27, 0xb4, 0xfd, 0xbf, 0xbf, 0xaf,
  0x47, 0xbc, 0x65, 0xac, 0xaa, 0xea, 0xea, 0x7a,
  0x46, 0xac, 0x83, 0x8c, 0xf0, 0x7c, 0x7c, 0x5f,
  0xe3, 0x8c, 0x83, 0x8c, 0x29, 0x0b, 0x0b, 0x02,
  0x03, 0x8d, 0xe3, 0x8c, 0xfd, 0xfd, 0xfd, 0xfd,
  0xe3, 0x8c, 0x03, 0x8d, 0x56, 0x5a, 0x5a, 0xaa,
  0x03, 0x95, 0x23, 0x95, 0x55, 0x55, 0x55, 0x00,
  0xc4, 0x9c, 0x23, 0x95, 0x55, 0xf5, 0xf5, 0x2f,
  0x24, 0x9c, 0xe4, 0x94, 0xf5, 0xbd, 0xbd, 0x02,
  0x25, 0xa4, 0x84, 0x94, 0xa9, 0x2a, 0x2a, 0x00,
  0x06, 0xb4, 0x45, 0xa4, 0xfd, 0xaf, 0xaf, 0x0a,
};

static uint8 kdxt1_256x256_alpha[128] = {
  0x00, 0x00, 0x01, 0x00, 0xff, 0xff, 0xff, 0xff,
  0x00, 0x00, 0x01, 0x00, 0xff, 0xff, 0xff, 0xff,
  0x00, 0x00, 0x01, 0x00, 0xff, 0xff, 0xff, 0xff,
  0x00, 0x00, 0x01, 0x00, 0xff, 0xff, 0xff, 0xff,
  0x00, 0x00, 0x01, 0x00, 0xff, 0xff, 0xff, 0xff,
  0x00, 0x00, 0x01, 0x00, 0xff, 0xff, 0xff, 0xff,
  0x00, 0x00, 0x01, 0x00, 0xff, 0xff, 0xff, 0xff,
  0x00, 0x00, 0x01, 0x00, 0xff, 0xff, 0xff, 0xff,
  0x02, 0x85, 0xe3, 0x8c, 0xff, 0x7f, 0x7f, 0x7f,
  0xe3, 0x8c, 0x03, 0x8d, 0xa8, 0xa8, 0xa8, 0xa8,
  0xe3, 0x8c, 0x03, 0x8d, 0x56, 0x5a, 0x5a, 0xaa,
  0x03, 0x95, 0x23, 0x95, 0x55, 0x55, 0x55, 0x00,
  0xe4, 0x94, 0x23, 0x95, 0x95, 0xa5, 0xa5, 0xca,
  0xa4, 0x94, 0x04, 0x95, 0xf9, 0xfe, 0xfe, 0xff,
  0x00, 0x00, 0x01, 0x00, 0xff, 0xff, 0xff, 0xff,
  0x00, 0x00, 0x01, 0x00, 0xff, 0xff, 0xff, 0xff,
};

static uint8 kdxt1_256x256_mipmap[128] = {
  0x47, 0xbc, 0x06, 0xb4, 0x5a, 0x6a, 0x6a, 0xea,
  0x27, 0xb4, 0x06, 0xb4, 0x57, 0x57, 0x57, 0x5e,
  0x06, 0xb4, 0x07, 0xb4, 0xaa, 0xaa, 0xaa, 0xaa,
  0x06, 0xb4, 0x07, 0xb4, 0xaa, 0xaa, 0xaa, 0xaa,
  0x46, 0xbc, 0x07, 0xb4, 0xd5, 0xf5, 0xf5, 0xa5,
  0x46, 0xbc, 0x27, 0xb4, 0xfd, 0xbf, 0xbf, 0xaf,
  0x47, 0xbc, 0x65, 0xac, 0xaa, 0xea, 0xea, 0x7a,
  0x46, 0xac, 0x83, 0x8c, 0xf0, 0x7c, 0x7c, 0x5f,
  0xe3, 0x8c, 0x83, 0x8c, 0x29, 0x0b, 0x0b, 0x02,
  0x03, 0x8d, 0xe3, 0x8c, 0xfd, 0xfd, 0xfd, 0xfd,
  0xe3, 0x8c, 0x03, 0x8d, 0x56, 0x5a, 0x5a, 0xaa,
  0x03, 0x95, 0x23, 0x95, 0x55, 0x55, 0x55, 0x00,
  0xc4, 0x9c, 0x23, 0x95, 0x55, 0xf5, 0xf5, 0x2f,
  0x24, 0x9c, 0xe4, 0x94, 0xf5, 0xbd, 0xbd, 0x02,
  0x25, 0xa4, 0x84, 0x94, 0xa9, 0x2a, 0x2a, 0x00,
  0x06, 0xb4, 0x45, 0xa4, 0xfd, 0xaf, 0xaf, 0x0a,
};

static uint8 kdxt3_256x256_alpha[128] = {
  0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77,
  0x47, 0xbc, 0x06, 0xb4, 0x5a, 0x6a, 0x6a, 0xea,
  0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77,
  0x27, 0xb4, 0x06, 0xb4, 0x57, 0x57, 0x57, 0x5e,
  0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77,
  0x07, 0xb4, 0x06, 0xb4, 0xaa, 0xaa, 0xaa, 0xaa,
  0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77,
  0x07, 0xb4, 0x06, 0xb4, 0xaa, 0xaa, 0xaa, 0xaa,
  0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77,
  0x46, 0xbc, 0x07, 0xb4, 0xd5, 0xb5, 0xb5, 0xa5,
  0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77,
  0x46, 0xbc, 0x27, 0xb4, 0xfd, 0xbf, 0xbf, 0xaf,
  0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77,
  0x47, 0xbc, 0x65, 0xac, 0xaa, 0xea, 0xea, 0x7a,
  0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77,
  0x46, 0xac, 0x83, 0x8c, 0xf0, 0x7c, 0x7c, 0x5f,
};

static uint8 kdxt3_256x256_mipmap[128] = {
  0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77,
  0x47, 0xbc, 0x06, 0xb4, 0x5a, 0x6a, 0x6a, 0xea,
  0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77,
  0x27, 0xb4, 0x06, 0xb4, 0x57, 0x57, 0x57, 0x5e,
  0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77,
  0x07, 0xb4, 0x06, 0xb4, 0xaa, 0xaa, 0xaa, 0xaa,
  0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77,
  0x07, 0xb4, 0x06, 0xb4, 0xaa, 0xaa, 0xaa, 0xaa,
  0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77,
  0x46, 0xbc, 0x07, 0xb4, 0xd5, 0xb5, 0xb5, 0xa5,
  0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77,
  0x46, 0xbc, 0x27, 0xb4, 0xfd, 0xbf, 0xbf, 0xaf,
  0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77,
  0x47, 0xbc, 0x65, 0xac, 0xaa, 0xea, 0xea, 0x7a,
  0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77, 0x77,
  0x46, 0xac, 0x83, 0x8c, 0xf0, 0x7c, 0x7c, 0x5f,
};

static uint8 kdxt5_256x256_alpha[128] = {
  0x7e, 0x77, 0xda, 0x23, 0x00, 0x90, 0x0a, 0x00,
  0x47, 0xbc, 0x06, 0xb4, 0x5a, 0x6a, 0x6a, 0xea,
  0x78, 0x76, 0x4e, 0x02, 0x00, 0x70, 0x02, 0x00,
  0x27, 0xb4, 0x06, 0xb4, 0x57, 0x57, 0x57, 0x5e,
  0x77, 0x75, 0x86, 0x61, 0x00, 0xb0, 0x6d, 0x00,
  0x07, 0xb4, 0x06, 0xb4, 0xaa, 0xaa, 0xaa, 0xaa,
  0x77, 0x76, 0x49, 0x82, 0x00, 0x41, 0x12, 0x00,
  0x07, 0xb4, 0x06, 0xb4, 0xaa, 0xaa, 0xaa, 0xaa,
  0x7c, 0x76, 0xc9, 0x77, 0x00, 0xff, 0x60, 0x00,
  0x46, 0xbc, 0x07, 0xb4, 0xd5, 0xb5, 0xb5, 0xa5,
  0x7c, 0x7a, 0x49, 0x60, 0x00, 0x36, 0x60, 0x00,
  0x46, 0xbc, 0x27, 0xb4, 0xfd, 0xbf, 0xbf, 0xaf,
  0x7e, 0x7c, 0x31, 0x60, 0x00, 0x06, 0x6c, 0x00,
  0x47, 0xbc, 0x65, 0xac, 0xaa, 0xea, 0xea, 0x7a,
  0x7d, 0x75, 0x50, 0x82, 0x00, 0x7c, 0x98, 0x00,
  0x46, 0xac, 0x83, 0x8c, 0xf0, 0x7c, 0x7c, 0x5f,
};

static uint8 kdxt5_256x256_mipmap[128] = {
  0x7e, 0x77, 0xda, 0x23, 0x00, 0x90, 0x0a, 0x00,
  0x47, 0xbc, 0x06, 0xb4, 0x5a, 0x6a, 0x6a, 0xea,
  0x78, 0x76, 0x4e, 0x02, 0x00, 0x70, 0x02, 0x00,
  0x27, 0xb4, 0x06, 0xb4, 0x57, 0x57, 0x57, 0x5e,
  0x77, 0x75, 0x86, 0x61, 0x00, 0xb0, 0x6d, 0x00,
  0x07, 0xb4, 0x06, 0xb4, 0xaa, 0xaa, 0xaa, 0xaa,
  0x77, 0x76, 0x49, 0x82, 0x00, 0x41, 0x12, 0x00,
  0x07, 0xb4, 0x06, 0xb4, 0xaa, 0xaa, 0xaa, 0xaa,
  0x7c, 0x76, 0xc9, 0x77, 0x00, 0xff, 0x60, 0x00,
  0x46, 0xbc, 0x07, 0xb4, 0xd5, 0xb5, 0xb5, 0xa5,
  0x7c, 0x7a, 0x49, 0x60, 0x00, 0x36, 0x60, 0x00,
  0x46, 0xbc, 0x27, 0xb4, 0xfd, 0xbf, 0xbf, 0xaf,
  0x7e, 0x7c, 0x31, 0x60, 0x00, 0x06, 0x6c, 0x00,
  0x47, 0xbc, 0x65, 0xac, 0xaa, 0xea, 0xea, 0x7a,
  0x7d, 0x75, 0x50, 0x82, 0x00, 0x7c, 0x98, 0x00,
  0x46, 0xac, 0x83, 0x8c, 0xf0, 0x7c, 0x7c, 0x5f,
};

// Match the first 128 bytes of a loaded bitmap data againt known values.
bool TestBitmapData(const Bitmap &bitmap, uint8 reference[128]) {
  if (!bitmap.image_data()) return false;
  return std::memcmp(bitmap.image_data(), reference, sizeof(reference)) == 0;
}

// Prints the first 128 bytes of a loaded bitmap data. Helper function to
// generate the known data above.
bool PrintBitmapData(const Bitmap &bitmap, const char *name) {
  const uint8 *data = bitmap.image_data();
  if (!data) return false;
  printf("static uint8 %s[128] = {\n", name);
  for (int i = 0; i < 16; ++i) {
    printf(" ");
    for (int j = 0; j < 8; ++j) {
      printf(" 0x%02x,", data[i*8+j]);
    }
    printf("\n");
  }
  printf("};\n");
  return true;
}


// -----------------------------------------------------------------------------

TEST_F(BitmapTest, Basic) {
  EXPECT_EQ(static_cast<int>(Bitmap::FACE_NEGATIVE_X),
            static_cast<int>(TextureCUBE::FACE_NEGATIVE_X));
  EXPECT_EQ(static_cast<int>(Bitmap::FACE_NEGATIVE_Y),
            static_cast<int>(TextureCUBE::FACE_NEGATIVE_Y));
  EXPECT_EQ(static_cast<int>(Bitmap::FACE_NEGATIVE_Z),
            static_cast<int>(TextureCUBE::FACE_NEGATIVE_Z));
  EXPECT_EQ(static_cast<int>(Bitmap::FACE_POSITIVE_X),
            static_cast<int>(TextureCUBE::FACE_POSITIVE_X));
  EXPECT_EQ(static_cast<int>(Bitmap::FACE_POSITIVE_Y),
            static_cast<int>(TextureCUBE::FACE_POSITIVE_Y));
  EXPECT_EQ(static_cast<int>(Bitmap::FACE_POSITIVE_Z),
            static_cast<int>(TextureCUBE::FACE_POSITIVE_Z));
  Bitmap::Ref bitmap(new Bitmap(g_service_locator));
  ASSERT_FALSE(bitmap.IsNull());
  EXPECT_TRUE(bitmap->IsA(Bitmap::GetApparentClass()));
  EXPECT_TRUE(bitmap->IsA(ParamObject::GetApparentClass()));
  EXPECT_TRUE(bitmap->image_data() == NULL);
  EXPECT_EQ(Texture::UNKNOWN_FORMAT, bitmap->format());
  EXPECT_EQ(0u, bitmap->width());
  EXPECT_EQ(0u, bitmap->height());
  EXPECT_EQ(0u, bitmap->num_mipmaps());
  EXPECT_EQ(Bitmap::IMAGE, bitmap->semantic());
}

// Loads a 24 bit TGA file, checks it against the known data.
TEST_F(BitmapTest, LoadTGAFile24bit) {
  // Load the texture object from a file.
  String filename = *g_program_path + "/bitmap_test/tga-256x256-24bit.tga";
  FilePath filepath = UTF8ToFilePath(filename);
  BitmapRefArray bitmaps;
  EXPECT_TRUE(Bitmap::LoadFromFile(
      g_service_locator, filepath, image::TGA, &bitmaps));
  ASSERT_EQ(1u, bitmaps.size());
  Bitmap* bitmap = bitmaps[0].Get();
  EXPECT_TRUE(bitmap->image_data() != NULL);
  EXPECT_EQ(Texture::XRGB8, bitmap->format());
  EXPECT_EQ(256U, bitmap->width());
  EXPECT_EQ(256U, bitmap->height());
  EXPECT_EQ(1U, bitmap->num_mipmaps());
  EXPECT_TRUE(TestBitmapData(*bitmap, ktga256x256_24bit_BGRX));
}

// Loads a 32 bit TGA file, checks it against the known data.
TEST_F(BitmapTest, LoadTGAFile32bit) {
  String filename = *g_program_path + "/bitmap_test/tga-256x256-32bit.tga";
  FilePath filepath = UTF8ToFilePath(filename);
  BitmapRefArray bitmaps;
  EXPECT_TRUE(Bitmap::LoadFromFile(
      g_service_locator, filepath, image::TGA, &bitmaps));
  ASSERT_EQ(1u, bitmaps.size());
  Bitmap* bitmap = bitmaps[0].Get();
  EXPECT_TRUE(bitmap->image_data() != NULL);
  EXPECT_EQ(Texture::ARGB8, bitmap->format());
  EXPECT_EQ(256U, bitmap->width());
  EXPECT_EQ(256U, bitmap->height());
  EXPECT_EQ(1U, bitmap->num_mipmaps());
  EXPECT_TRUE(TestBitmapData(*bitmap, ktga256x256_32bit_BGRA));
}

// Tries to load a 5kx5k TGA file, which should fail.
TEST_F(BitmapTest, LoadTGAFileTooLarge) {
  // NOTE: the 5kx5k.tga file only has the first 4k bytes, to avoid
  // creating a 100MB test file. The code reads the header (first 18 bytes),
  // but bails before reading the actual image bytes.
  String filename = *g_program_path + "/bitmap_test/5kx5k.tga";
  FilePath filepath = UTF8ToFilePath(filename);
  BitmapRefArray bitmaps;
  EXPECT_FALSE(Bitmap::LoadFromFile(
      g_service_locator, filepath, image::TGA, &bitmaps));
  EXPECT_EQ(0u, bitmaps.size());
}

// Loads a JPEG file, checks it against the known data.
TEST_F(BitmapTest, LoadJPEGFile) {
  String filename = *g_program_path + "/bitmap_test/jpeg-256x256.jpg";
  FilePath filepath = UTF8ToFilePath(filename);
  BitmapRefArray bitmaps;
  EXPECT_TRUE(Bitmap::LoadFromFile(
      g_service_locator, filepath, image::JPEG, &bitmaps));
  ASSERT_EQ(1u, bitmaps.size());
  Bitmap* bitmap = bitmaps[0].Get();
  EXPECT_TRUE(bitmap->image_data() != NULL);
  EXPECT_EQ(Texture::XRGB8, bitmap->format());
  EXPECT_EQ(256U, bitmap->width());
  EXPECT_EQ(256U, bitmap->height());
  EXPECT_EQ(1U, bitmap->num_mipmaps());
  EXPECT_TRUE(TestBitmapData(*bitmap, kjpg256x256_BGRX));
}

// Tries to load a 5kx5k JPEG file, which should fail.
TEST_F(BitmapTest, LoadJPEGFileTooLarge) {
  String filename = *g_program_path + "/bitmap_test/5kx5k.jpg";
  FilePath filepath = UTF8ToFilePath(filename);
  BitmapRefArray bitmaps;
  EXPECT_FALSE(Bitmap::LoadFromFile(
      g_service_locator, filepath, image::JPEG, &bitmaps));
  EXPECT_EQ(0u, bitmaps.size());
}

// Loads a 24 bit PNG file, checks it against the known data.
TEST_F(BitmapTest, LoadPNGFile24bit) {
  String filename = *g_program_path + "/bitmap_test/png-256x256-24bit.png";
  FilePath filepath = UTF8ToFilePath(filename);
  BitmapRefArray bitmaps;
  EXPECT_TRUE(Bitmap::LoadFromFile(
      g_service_locator, filepath, image::PNG, &bitmaps));
  ASSERT_EQ(1u, bitmaps.size());
  Bitmap* bitmap = bitmaps[0].Get();
  EXPECT_TRUE(bitmap->image_data() != NULL);
  EXPECT_EQ(Texture::XRGB8, bitmap->format());
  EXPECT_EQ(256U, bitmap->width());
  EXPECT_EQ(256U, bitmap->height());
  EXPECT_EQ(1U, bitmap->num_mipmaps());
  EXPECT_TRUE(TestBitmapData(*bitmap, kpng256x256_24bit_BGRX));
}

// Loads a 24 bit interlaced PNG file, checks it against the known data.
TEST_F(BitmapTest, LoadPNGFile24bitInterlaced) {
  String filename = *g_program_path +
                    "/bitmap_test/png-256x256-24bit-interlaced.png";
  FilePath filepath = UTF8ToFilePath(filename);
  BitmapRefArray bitmaps;
  EXPECT_TRUE(Bitmap::LoadFromFile(
      g_service_locator, filepath, image::PNG, &bitmaps));
  ASSERT_EQ(1u, bitmaps.size());
  Bitmap* bitmap = bitmaps[0].Get();
  EXPECT_TRUE(bitmap->image_data() != NULL);
  EXPECT_EQ(Texture::XRGB8, bitmap->format());
  EXPECT_EQ(256U, bitmap->width());
  EXPECT_EQ(256U, bitmap->height());
  EXPECT_EQ(1U, bitmap->num_mipmaps());
  EXPECT_TRUE(TestBitmapData(*bitmap, kpng256x256_24bit_interlaced_BGRX));
}

TEST_F(BitmapTest, LoadPNGFile32bit) {
  String filename = *g_program_path + "/bitmap_test/png-256x256-32bit.png";
  FilePath filepath = UTF8ToFilePath(filename);
  BitmapRefArray bitmaps;
  EXPECT_TRUE(Bitmap::LoadFromFile(
      g_service_locator, filepath, image::PNG, &bitmaps));
  ASSERT_EQ(1u, bitmaps.size());
  Bitmap* bitmap = bitmaps[0].Get();
  EXPECT_TRUE(bitmap->image_data() != NULL);
  EXPECT_EQ(Texture::ARGB8, bitmap->format());
  EXPECT_EQ(256U, bitmap->width());
  EXPECT_EQ(256U, bitmap->height());
  EXPECT_EQ(1U, bitmap->num_mipmaps());
  EXPECT_TRUE(TestBitmapData(*bitmap, kpng256x256_32bit_BGRA));
}

// Loads a palettized PNG file, checks it against the known data.
TEST_F(BitmapTest, LoadPNGFile8bitPalette) {
  String filename = *g_program_path +
                    "/bitmap_test/png-256x256-8bit-palette.png";
  FilePath filepath = UTF8ToFilePath(filename);
  BitmapRefArray bitmaps;
  EXPECT_TRUE(Bitmap::LoadFromFile(
      g_service_locator, filepath, image::PNG, &bitmaps));
  ASSERT_EQ(1u, bitmaps.size());
  Bitmap* bitmap = bitmaps[0].Get();
  EXPECT_TRUE(bitmap->image_data() != NULL);
  EXPECT_EQ(Texture::XRGB8, bitmap->format());
  EXPECT_EQ(256U, bitmap->width());
  EXPECT_EQ(256U, bitmap->height());
  EXPECT_EQ(1U, bitmap->num_mipmaps());
  EXPECT_TRUE(TestBitmapData(*bitmap, kpng256x256_8bit_palette_BGRX));
}

// Loads a palettized PNG file, checks it against the known data.
TEST_F(BitmapTest, LoadPNGFile4bitPalette) {
  String filename = *g_program_path +
                    "/bitmap_test/png-20x14-4bit-palette.png";
  FilePath filepath = UTF8ToFilePath(filename);
  BitmapRefArray bitmaps;
  EXPECT_TRUE(Bitmap::LoadFromFile(
      g_service_locator, filepath, image::PNG, &bitmaps));
  ASSERT_EQ(1u, bitmaps.size());
  Bitmap* bitmap = bitmaps[0].Get();
  EXPECT_TRUE(bitmap->image_data() != NULL);
  EXPECT_EQ(Texture::XRGB8, bitmap->format());
  EXPECT_EQ(20U, bitmap->width());
  EXPECT_EQ(14U, bitmap->height());
  EXPECT_EQ(1U, bitmap->num_mipmaps());
  EXPECT_TRUE(TestBitmapData(*bitmap, kpng20x14_4bit_palette_BGRX));
}


// Tries to load a 5kx5k PNG file, which should fail.
TEST_F(BitmapTest, LoadPNGFileTooLarge) {
  String filename = *g_program_path + "/bitmap_test/5kx5k.png";
  FilePath filepath = UTF8ToFilePath(filename);
  BitmapRefArray bitmaps;
  EXPECT_FALSE(Bitmap::LoadFromFile(
      g_service_locator, filepath, image::PNG, &bitmaps));
  EXPECT_EQ(0u, bitmaps.size());
}

// NOTE: Having trouble recognising the alpha channel in a PNG
// exported from Photoshop.
/*
TEST_F(BitmapTest, LoadPNGFile8bitPaletteAlpha) {
  String filename = *g_program_path +
      "/bitmap_test/png-256x256-8bit-palette-alpha.png";
  FilePath filepath = UTF8ToFilePath(filename);
  BitmapRefArray bitmaps;
  EXPECT_TRUE(Bitmap::LoadFromFile(
      g_service_locator, filepath, image::PNG, &bitmaps));
  ASSERT_EQ(1u, bitmaps.size());
  Bitmap* bitmap = bitmaps[0].Get();
  EXPECT_TRUE(bitmap->image_data() != NULL);
  EXPECT_EQ(Texture::ARGB8, bitmap->format());
  EXPECT_EQ(256U, bitmap->width());
  EXPECT_EQ(256U, bitmap->height());
  EXPECT_EQ(1U, bitmap->num_mipmaps());
}
*/

// Loads a DXT1 DDS file, checks the format.
TEST_F(BitmapTest, LoadDDSFileDXT1) {
  String filename = *g_program_path + "/bitmap_test/dds-dxt1-256x256.dds";
  FilePath filepath = UTF8ToFilePath(filename);
  BitmapRefArray bitmaps;
  EXPECT_TRUE(Bitmap::LoadFromFile(
      g_service_locator, filepath, image::DDS, &bitmaps));
  ASSERT_EQ(1u, bitmaps.size());
  Bitmap* bitmap = bitmaps[0].Get();
  EXPECT_TRUE(bitmap->image_data() != NULL);
  EXPECT_EQ(Texture::DXT1, bitmap->format());
  EXPECT_EQ(256U, bitmap->width());
  EXPECT_EQ(256U, bitmap->height());
  EXPECT_EQ(1U, bitmap->num_mipmaps());
  EXPECT_TRUE(TestBitmapData(*bitmap, kdxt1_256x256));
}

// Loads a DXT1 DDS file with alpha, checks the format.
TEST_F(BitmapTest, LoadDDSFileDXT1Alpha) {
  String filename = *g_program_path +
      "/bitmap_test/dds-dxt1-256x256-alpha.dds";
  FilePath filepath = UTF8ToFilePath(filename);
  BitmapRefArray bitmaps;
  EXPECT_TRUE(Bitmap::LoadFromFile(
      g_service_locator, filepath, image::DDS, &bitmaps));
  ASSERT_EQ(1u, bitmaps.size());
  Bitmap* bitmap = bitmaps[0].Get();
  EXPECT_TRUE(bitmap->image_data() != NULL);
  EXPECT_EQ(Texture::DXT1, bitmap->format());
  EXPECT_EQ(256U, bitmap->width());
  EXPECT_EQ(256U, bitmap->height());
  EXPECT_EQ(1U, bitmap->num_mipmaps());
  EXPECT_TRUE(TestBitmapData(*bitmap, kdxt1_256x256_alpha));
}

// Loads a DXT1 DDS file with mipmaps, checks the format.
TEST_F(BitmapTest, LoadDDSFileDXT1Mipmap) {
  String filename = *g_program_path +
      "/bitmap_test/dds-dxt1-256x256-mipmap.dds";
  FilePath filepath = UTF8ToFilePath(filename);
  BitmapRefArray bitmaps;
  EXPECT_TRUE(Bitmap::LoadFromFile(
      g_service_locator, filepath, image::DDS, &bitmaps));
  ASSERT_EQ(1u, bitmaps.size());
  Bitmap* bitmap = bitmaps[0].Get();
  EXPECT_TRUE(bitmap->image_data() != NULL);
  EXPECT_EQ(Texture::DXT1, bitmap->format());
  EXPECT_EQ(256U, bitmap->width());
  EXPECT_EQ(256U, bitmap->height());
  EXPECT_EQ(9U, bitmap->num_mipmaps());
  for (unsigned int i = 0; i < bitmap->num_mipmaps(); ++i) {
    EXPECT_TRUE(bitmap->GetMipData(i) != NULL);
  }
  EXPECT_TRUE(TestBitmapData(*bitmap, kdxt1_256x256_mipmap));
}

// Loads a DXT3 DDS file, checks the format.
TEST_F(BitmapTest, LoadDDSFileDXT3) {
  String filename = *g_program_path +
      "/bitmap_test/dds-dxt3-256x256-alpha.dds";
  FilePath filepath = UTF8ToFilePath(filename);
  BitmapRefArray bitmaps;
  EXPECT_TRUE(Bitmap::LoadFromFile(
      g_service_locator, filepath, image::DDS, &bitmaps));
  ASSERT_EQ(1u, bitmaps.size());
  Bitmap* bitmap = bitmaps[0].Get();
  EXPECT_TRUE(bitmap->image_data() != NULL);
  EXPECT_EQ(Texture::DXT3, bitmap->format());
  EXPECT_EQ(256U, bitmap->width());
  EXPECT_EQ(256U, bitmap->height());
  EXPECT_EQ(1U, bitmap->num_mipmaps());
  EXPECT_TRUE(TestBitmapData(*bitmap, kdxt3_256x256_alpha));
}

// Loads a DXT3 DDS file with mipmaps, checks the format.
TEST_F(BitmapTest, LoadDDSFileDXT3Mipmap) {
  String filename = *g_program_path +
      "/bitmap_test/dds-dxt3-256x256-mipmap.dds";
  FilePath filepath = UTF8ToFilePath(filename);
  BitmapRefArray bitmaps;
  EXPECT_TRUE(Bitmap::LoadFromFile(
      g_service_locator, filepath, image::DDS, &bitmaps));
  ASSERT_EQ(1u, bitmaps.size());
  Bitmap* bitmap = bitmaps[0].Get();
  EXPECT_TRUE(bitmap->image_data() != NULL);
  EXPECT_EQ(Texture::DXT3, bitmap->format());
  EXPECT_EQ(256U, bitmap->width());
  EXPECT_EQ(256U, bitmap->height());
  EXPECT_EQ(9U, bitmap->num_mipmaps());
  for (unsigned int i = 0; i < bitmap->num_mipmaps(); ++i) {
    EXPECT_TRUE(bitmap->GetMipData(i) != NULL);
  }
  EXPECT_TRUE(TestBitmapData(*bitmap, kdxt3_256x256_mipmap));
}

// Loads a DXT5 DDS file, checks the format.
TEST_F(BitmapTest, LoadDDSFileDXT5) {
  String filename = *g_program_path +
      "/bitmap_test/dds-dxt5-256x256-alpha.dds";
  FilePath filepath = UTF8ToFilePath(filename);
  BitmapRefArray bitmaps;
  EXPECT_TRUE(Bitmap::LoadFromFile(
      g_service_locator, filepath, image::DDS, &bitmaps));
  ASSERT_EQ(1u, bitmaps.size());
  Bitmap* bitmap = bitmaps[0].Get();
  EXPECT_TRUE(bitmap->image_data() != NULL);
  EXPECT_EQ(Texture::DXT5, bitmap->format());
  EXPECT_EQ(256U, bitmap->width());
  EXPECT_EQ(256U, bitmap->height());
  EXPECT_EQ(1U, bitmap->num_mipmaps());
  EXPECT_TRUE(TestBitmapData(*bitmap, kdxt5_256x256_alpha));
}

// Loads a DXT5 DDS file with mipmaps, checks the format.
TEST_F(BitmapTest, LoadDDSFileDXT5Mipmap) {
  String filename = *g_program_path +
      "/bitmap_test/dds-dxt5-256x256-mipmap.dds";
  FilePath filepath = UTF8ToFilePath(filename);
  BitmapRefArray bitmaps;
  EXPECT_TRUE(Bitmap::LoadFromFile(
      g_service_locator, filepath, image::DDS, &bitmaps));
  ASSERT_EQ(1u, bitmaps.size());
  Bitmap* bitmap = bitmaps[0].Get();
  EXPECT_TRUE(bitmap->image_data() != NULL);
  EXPECT_EQ(Texture::DXT5, bitmap->format());
  EXPECT_EQ(256U, bitmap->width());
  EXPECT_EQ(256U, bitmap->height());
  EXPECT_EQ(9U, bitmap->num_mipmaps());
  for (unsigned int i = 0; i < bitmap->num_mipmaps(); ++i) {
    EXPECT_TRUE(bitmap->GetMipData(i) != NULL);
  }
  EXPECT_TRUE(TestBitmapData(*bitmap, kdxt5_256x256_mipmap));
}

// Tries to load a 5kx5k DDS file, which should fail.
TEST_F(BitmapTest, LoadDDSFileTooLarge) {
  // NOTE: the 5kx5k.dds file only has the first 4k bytes, to avoid
  // creating a 100MB test file. The code reads the header (first 128 bytes),
  // but bails before reading the actual image bytes.
  String filename = *g_program_path + "/bitmap_test/5kx5k.dds";
  FilePath filepath = UTF8ToFilePath(filename);
  BitmapRefArray bitmaps;
  EXPECT_FALSE(Bitmap::LoadFromFile(
      g_service_locator, filepath, image::DDS, &bitmaps));
  EXPECT_EQ(0u, bitmaps.size());
}

static uint8 kpng_8x4_drawImage[128] = {
  // Raw dest image used in drawimage test.
  0x30, 0x60, 0xc0, 0xff, 0x32, 0x64, 0xc8, 0xff,
  0x34, 0x68, 0xd0, 0xff, 0x36, 0x6c, 0xd8, 0xff,
  0x38, 0x70, 0xe0, 0xff, 0x3a, 0x74, 0xe8, 0xff,
  0x3c, 0x78, 0xf0, 0xff, 0x3e, 0x7c, 0xf8, 0xff,
  0x20, 0x40, 0x80, 0xff, 0x22, 0x44, 0x88, 0xff,
  0x24, 0x48, 0x90, 0xff, 0x26, 0x4c, 0x98, 0xff,
  0x28, 0x50, 0xa0, 0xff, 0x2a, 0x54, 0xa8, 0xff,
  0x2c, 0x58, 0xb0, 0xff, 0x2e, 0x5c, 0xb8, 0xff,
  0x10, 0x20, 0x40, 0xff, 0x12, 0x24, 0x48, 0xff,
  0x14, 0x28, 0x50, 0xff, 0x16, 0x2c, 0x58, 0xff,
  0x18, 0x30, 0x60, 0xff, 0x1a, 0x34, 0x68, 0xff,
  0x1c, 0x38, 0x70, 0xff, 0x1e, 0x3c, 0x78, 0xff,
  0x00, 0x00, 0x00, 0xff, 0x02, 0x04, 0x08, 0xff,
  0x04, 0x08, 0x10, 0xff, 0x06, 0x0c, 0x18, 0xff,
  0x08, 0x10, 0x20, 0xff, 0x0a, 0x14, 0x28, 0xff,
  0x0c, 0x18, 0x30, 0xff, 0x0e, 0x1c, 0x38, 0xff,
};

static uint8 kpng_8x4_drawImage_top_left[128] = {
  // expected result of drawimage on top left corner of dest image.
  0x28, 0x28, 0x28, 0xff, 0x29, 0x29, 0x29, 0xff,
  0x2a, 0x2a, 0x2a, 0xff, 0x36, 0x6c, 0xd8, 0xff,
  0x38, 0x70, 0xe0, 0xff, 0x3a, 0x74, 0xe8, 0xff,
  0x3c, 0x78, 0xf0, 0xff, 0x3e, 0x7c, 0xf8, 0xff,
  0x24, 0x24, 0x24, 0xff, 0x25, 0x25, 0x25, 0xff,
  0x26, 0x26, 0x26, 0xff, 0x26, 0x4c, 0x98, 0xff,
  0x28, 0x50, 0xa0, 0xff, 0x2a, 0x54, 0xa8, 0xff,
  0x2c, 0x58, 0xb0, 0xff, 0x2e, 0x5c, 0xb8, 0xff,
  0x20, 0x20, 0x20, 0xff, 0x21, 0x21, 0x21, 0xff,
  0x22, 0x22, 0x22, 0xff, 0x16, 0x2c, 0x58, 0xff,
  0x18, 0x30, 0x60, 0xff, 0x1a, 0x34, 0x68, 0xff,
  0x1c, 0x38, 0x70, 0xff, 0x1e, 0x3c, 0x78, 0xff,
  0x00, 0x00, 0x00, 0xff, 0x02, 0x04, 0x08, 0xff,
  0x04, 0x08, 0x10, 0xff, 0x06, 0x0c, 0x18, 0xff,
  0x08, 0x10, 0x20, 0xff, 0x0a, 0x14, 0x28, 0xff,
  0x0c, 0x18, 0x30, 0xff, 0x0e, 0x1c, 0x38, 0xff,
};

static uint8 kpng_8x4_drawImage_top[128] = {
  // expected result of drawimage on top bound of dest image.
  0x30, 0x60, 0xc0, 0xff, 0x32, 0x64, 0xc8, 0xff,
  0x34, 0x68, 0xd0, 0xff, 0x36, 0x6c, 0xd8, 0xff,
  0x38, 0x70, 0xe0, 0xff, 0x3a, 0x74, 0xe8, 0xff,
  0x3c, 0x78, 0xf0, 0xff, 0x3e, 0x7c, 0xf8, 0xff,
  0x20, 0x40, 0x80, 0xff, 0x22, 0x44, 0x88, 0xff,
  0x24, 0x48, 0x90, 0xff, 0x26, 0x4c, 0x98, 0xff,
  0x28, 0x50, 0xa0, 0xff, 0x2a, 0x54, 0xa8, 0xff,
  0x2c, 0x58, 0xb0, 0xff, 0x2e, 0x5c, 0xb8, 0xff,
  0x10, 0x20, 0x40, 0xff, 0x12, 0x24, 0x48, 0xff,
  0x2b, 0x2b, 0x2b, 0xff, 0x2c, 0x2c, 0x2c, 0xff,
  0x2d, 0x2d, 0x2d, 0xff, 0x2e, 0x2e, 0x2e, 0xff,
  0x1c, 0x38, 0x70, 0xff, 0x1e, 0x3c, 0x78, 0xff,
  0x00, 0x00, 0x00, 0xff, 0x02, 0x04, 0x08, 0xff,
  0x27, 0x27, 0x27, 0xff, 0x28, 0x28, 0x28, 0xff,
  0x29, 0x29, 0x29, 0xff, 0x2a, 0x2a, 0x2a, 0xff,
  0x0c, 0x18, 0x30, 0xff, 0x0e, 0x1c, 0x38, 0xff,
};

static uint8 kpng_8x4_drawImage_top_right[128] = {
  // expected result of drawimage on top right corner of dest image.
  0x30, 0x60, 0xc0, 0xff, 0x32, 0x64, 0xc8, 0xff,
  0x34, 0x68, 0xd0, 0xff, 0x36, 0x6c, 0xd8, 0xff,
  0x38, 0x70, 0xe0, 0xff, 0x3a, 0x74, 0xe8, 0xff,
  0x3c, 0x78, 0xf0, 0xff, 0x3e, 0x7c, 0xf8, 0xff,
  0x20, 0x40, 0x80, 0xff, 0x22, 0x44, 0x88, 0xff,
  0x24, 0x48, 0x90, 0xff, 0x26, 0x4c, 0x98, 0xff,
  0x28, 0x50, 0xa0, 0xff, 0x2b, 0x2b, 0x2b, 0xff,
  0x2c, 0x2c, 0x2c, 0xff, 0x2d, 0x2d, 0x2d, 0xff,
  0x10, 0x20, 0x40, 0xff, 0x12, 0x24, 0x48, 0xff,
  0x14, 0x28, 0x50, 0xff, 0x16, 0x2c, 0x58, 0xff,
  0x18, 0x30, 0x60, 0xff, 0x27, 0x27, 0x27, 0xff,
  0x28, 0x28, 0x28, 0xff, 0x29, 0x29, 0x29, 0xff,
  0x00, 0x00, 0x00, 0xff, 0x02, 0x04, 0x08, 0xff,
  0x04, 0x08, 0x10, 0xff, 0x06, 0x0c, 0x18, 0xff,
  0x08, 0x10, 0x20, 0xff, 0x23, 0x23, 0x23, 0xff,
  0x24, 0x24, 0x24, 0xff, 0x25, 0x25, 0x25, 0xff,
};

static uint8 kpng_8x4_drawImage_right[128] = {
  // expected result of drawimage on right bound of dest image.
  0x30, 0x60, 0xc0, 0xff, 0x32, 0x64, 0xc8, 0xff,
  0x34, 0x68, 0xd0, 0xff, 0x36, 0x6c, 0xd8, 0xff,
  0x38, 0x70, 0xe0, 0xff, 0x2b, 0x2b, 0x2b, 0xff,
  0x2c, 0x2c, 0x2c, 0xff, 0x2d, 0x2d, 0x2d, 0xff,
  0x20, 0x40, 0x80, 0xff, 0x22, 0x44, 0x88, 0xff,
  0x24, 0x48, 0x90, 0xff, 0x26, 0x4c, 0x98, 0xff,
  0x28, 0x50, 0xa0, 0xff, 0x27, 0x27, 0x27, 0xff,
  0x28, 0x28, 0x28, 0xff, 0x29, 0x29, 0x29, 0xff,
  0x10, 0x20, 0x40, 0xff, 0x12, 0x24, 0x48, 0xff,
  0x14, 0x28, 0x50, 0xff, 0x16, 0x2c, 0x58, 0xff,
  0x18, 0x30, 0x60, 0xff, 0x23, 0x23, 0x23, 0xff,
  0x24, 0x24, 0x24, 0xff, 0x25, 0x25, 0x25, 0xff,
  0x00, 0x00, 0x00, 0xff, 0x02, 0x04, 0x08, 0xff,
  0x04, 0x08, 0x10, 0xff, 0x06, 0x0c, 0x18, 0xff,
  0x08, 0x10, 0x20, 0xff, 0x1f, 0x1f, 0x1f, 0xff,
  0x20, 0x20, 0x20, 0xff, 0x21, 0x21, 0x21, 0xff,
};

static uint8 kpng_8x4_drawImage_bottom_right[128] = {
  // expected result of drawimage on bottom right corner of dest image.
  0x30, 0x60, 0xc0, 0xff, 0x32, 0x64, 0xc8, 0xff,
  0x34, 0x68, 0xd0, 0xff, 0x36, 0x6c, 0xd8, 0xff,
  0x38, 0x70, 0xe0, 0xff, 0x27, 0x27, 0x27, 0xff,
  0x28, 0x28, 0x28, 0xff, 0x29, 0x29, 0x29, 0xff,
  0x20, 0x40, 0x80, 0xff, 0x22, 0x44, 0x88, 0xff,
  0x24, 0x48, 0x90, 0xff, 0x26, 0x4c, 0x98, 0xff,
  0x28, 0x50, 0xa0, 0xff, 0x23, 0x23, 0x23, 0xff,
  0x24, 0x24, 0x24, 0xff, 0x25, 0x25, 0x25, 0xff,
  0x10, 0x20, 0x40, 0xff, 0x12, 0x24, 0x48, 0xff,
  0x14, 0x28, 0x50, 0xff, 0x16, 0x2c, 0x58, 0xff,
  0x18, 0x30, 0x60, 0xff, 0x1f, 0x1f, 0x1f, 0xff,
  0x20, 0x20, 0x20, 0xff, 0x21, 0x21, 0x21, 0xff,
  0x00, 0x00, 0x00, 0xff, 0x02, 0x04, 0x08, 0xff,
  0x04, 0x08, 0x10, 0xff, 0x06, 0x0c, 0x18, 0xff,
  0x08, 0x10, 0x20, 0xff, 0x0a, 0x14, 0x28, 0xff,
  0x0c, 0x18, 0x30, 0xff, 0x0e, 0x1c, 0x38, 0xff,
};

static uint8 kpng_8x4_drawImage_bottom[128] = {
  // expected result of drawimage on bottom bound of dest image.
  0x30, 0x60, 0xc0, 0xff, 0x32, 0x64, 0xc8, 0xff,
  0x27, 0x27, 0x27, 0xff, 0x28, 0x28, 0x28, 0xff,
  0x29, 0x29, 0x29, 0xff, 0x2a, 0x2a, 0x2a, 0xff,
  0x3c, 0x78, 0xf0, 0xff, 0x3e, 0x7c, 0xf8, 0xff,
  0x20, 0x40, 0x80, 0xff, 0x22, 0x44, 0x88, 0xff,
  0x23, 0x23, 0x23, 0xff, 0x24, 0x24, 0x24, 0xff,
  0x25, 0x25, 0x25, 0xff, 0x26, 0x26, 0x26, 0xff,
  0x2c, 0x58, 0xb0, 0xff, 0x2e, 0x5c, 0xb8, 0xff,
  0x10, 0x20, 0x40, 0xff, 0x12, 0x24, 0x48, 0xff,
  0x1f, 0x1f, 0x1f, 0xff, 0x20, 0x20, 0x20, 0xff,
  0x21, 0x21, 0x21, 0xff, 0x22, 0x22, 0x22, 0xff,
  0x1c, 0x38, 0x70, 0xff, 0x1e, 0x3c, 0x78, 0xff,
  0x00, 0x00, 0x00, 0xff, 0x02, 0x04, 0x08, 0xff,
  0x04, 0x08, 0x10, 0xff, 0x06, 0x0c, 0x18, 0xff,
  0x08, 0x10, 0x20, 0xff, 0x0a, 0x14, 0x28, 0xff,
  0x0c, 0x18, 0x30, 0xff, 0x0e, 0x1c, 0x38, 0xff,
};

static uint8 kpng_8x4_drawImage_bottom_left[128] = {
  // expected result of drawimage on bottom left corner of dest image.
  0x30, 0x60, 0xc0, 0xff, 0x32, 0x64, 0xc8, 0xff,
  0x34, 0x68, 0xd0, 0xff, 0x36, 0x6c, 0xd8, 0xff,
  0x38, 0x70, 0xe0, 0xff, 0x3a, 0x74, 0xe8, 0xff,
  0x3c, 0x78, 0xf0, 0xff, 0x3e, 0x7c, 0xf8, 0xff,
  0x2c, 0x2c, 0x2c, 0xff, 0x2d, 0x2d, 0x2d, 0xff,
  0x2e, 0x2e, 0x2e, 0xff, 0x26, 0x4c, 0x98, 0xff,
  0x28, 0x50, 0xa0, 0xff, 0x2a, 0x54, 0xa8, 0xff,
  0x2c, 0x58, 0xb0, 0xff, 0x2e, 0x5c, 0xb8, 0xff,
  0x28, 0x28, 0x28, 0xff, 0x29, 0x29, 0x29, 0xff,
  0x2a, 0x2a, 0x2a, 0xff, 0x16, 0x2c, 0x58, 0xff,
  0x18, 0x30, 0x60, 0xff, 0x1a, 0x34, 0x68, 0xff,
  0x1c, 0x38, 0x70, 0xff, 0x1e, 0x3c, 0x78, 0xff,
  0x24, 0x24, 0x24, 0xff, 0x25, 0x25, 0x25, 0xff,
  0x26, 0x26, 0x26, 0xff, 0x06, 0x0c, 0x18, 0xff,
  0x08, 0x10, 0x20, 0xff, 0x0a, 0x14, 0x28, 0xff,
  0x0c, 0x18, 0x30, 0xff, 0x0e, 0x1c, 0x38, 0xff,
};

static uint8 kpng_8x4_drawImage_left[128] = {
  // expected result of drawimage on left bound of dest image.
  0x2c, 0x2c, 0x2c, 0xff, 0x2d, 0x2d, 0x2d, 0xff,
  0x2e, 0x2e, 0x2e, 0xff, 0x36, 0x6c, 0xd8, 0xff,
  0x38, 0x70, 0xe0, 0xff, 0x3a, 0x74, 0xe8, 0xff,
  0x3c, 0x78, 0xf0, 0xff, 0x3e, 0x7c, 0xf8, 0xff,
  0x28, 0x28, 0x28, 0xff, 0x29, 0x29, 0x29, 0xff,
  0x2a, 0x2a, 0x2a, 0xff, 0x26, 0x4c, 0x98, 0xff,
  0x28, 0x50, 0xa0, 0xff, 0x2a, 0x54, 0xa8, 0xff,
  0x2c, 0x58, 0xb0, 0xff, 0x2e, 0x5c, 0xb8, 0xff,
  0x24, 0x24, 0x24, 0xff, 0x25, 0x25, 0x25, 0xff,
  0x26, 0x26, 0x26, 0xff, 0x16, 0x2c, 0x58, 0xff,
  0x18, 0x30, 0x60, 0xff, 0x1a, 0x34, 0x68, 0xff,
  0x1c, 0x38, 0x70, 0xff, 0x1e, 0x3c, 0x78, 0xff,
  0x20, 0x20, 0x20, 0xff, 0x21, 0x21, 0x21, 0xff,
  0x22, 0x22, 0x22, 0xff, 0x06, 0x0c, 0x18, 0xff,
  0x08, 0x10, 0x20, 0xff, 0x0a, 0x14, 0x28, 0xff,
  0x0c, 0x18, 0x30, 0xff, 0x0e, 0x1c, 0x38, 0xff,
};

static uint8 kpng_8x4_drawImage_scale_up[128] = {
  // expected result of scale up from 2x2 to 8x4.
  0x38, 0x38, 0x38, 0xff, 0x43, 0x43, 0x43, 0xff,
  0x52, 0x52, 0x52, 0xff, 0x63, 0x63, 0x63, 0xff,
  0x77, 0x77, 0x77, 0xff, 0x88, 0x88, 0x88, 0xff,
  0x96, 0x96, 0x96, 0xff, 0x90, 0x90, 0x90, 0xff,
  0x25, 0x25, 0x25, 0xff, 0x2c, 0x2c, 0x2c, 0xff,
  0x36, 0x36, 0x36, 0xff, 0x45, 0x45, 0x45, 0xff,
  0x54, 0x54, 0x54, 0xff, 0x63, 0x63, 0x63, 0xff,
  0x6e, 0x6e, 0x6e, 0xff, 0x69, 0x69, 0x69, 0xff,
  0x0b, 0x0b, 0x0b, 0xff, 0x0d, 0x0d, 0x0d, 0xff,
  0x13, 0x13, 0x13, 0xff, 0x1c, 0x1c, 0x1c, 0xff,
  0x28, 0x28, 0x28, 0xff, 0x31, 0x31, 0x31, 0xff,
  0x39, 0x39, 0x39, 0xff, 0x35, 0x35, 0x35, 0xff,
  0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0xff,
  0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0xff,
  0x05, 0x05, 0x05, 0xff, 0x0c, 0x0c, 0x0c, 0xff,
  0x11, 0x11, 0x11, 0xff, 0x0e, 0x0e, 0x0e, 0xff,
};

static uint8 kpng_8x4_drawImage_scale_down[128] = {
  // expected result of scale down from 8x8 to 4x4.
  0xa0, 0xa0, 0xa0, 0xff, 0xa7, 0xa7, 0xa7, 0xff,
  0xad, 0xad, 0xad, 0xff, 0xb3, 0xb3, 0xb3, 0xff,
  0x38, 0x70, 0xe0, 0xff, 0x3a, 0x74, 0xe8, 0xff,
  0x3c, 0x78, 0xf0, 0xff, 0x3e, 0x7c, 0xf8, 0xff,
  0x6d, 0x6d, 0x6d, 0xff, 0x74, 0x74, 0x74, 0xff,
  0x7a, 0x7a, 0x7a, 0xff, 0x80, 0x80, 0x80, 0xff,
  0x28, 0x50, 0xa0, 0xff, 0x2a, 0x54, 0xa8, 0xff,
  0x2c, 0x58, 0xb0, 0xff, 0x2e, 0x5c, 0xb8, 0xff,
  0x3d, 0x3d, 0x3d, 0xff, 0x44, 0x44, 0x44, 0xff,
  0x4a, 0x4a, 0x4a, 0xff, 0x50, 0x50, 0x50, 0xff,
  0x18, 0x30, 0x60, 0xff, 0x1a, 0x34, 0x68, 0xff,
  0x1c, 0x38, 0x70, 0xff, 0x1e, 0x3c, 0x78, 0xff,
  0x0a, 0x0a, 0x0a, 0xff, 0x11, 0x11, 0x11, 0xff,
  0x17, 0x17, 0x17, 0xff, 0x1d, 0x1d, 0x1d, 0xff,
  0x08, 0x10, 0x20, 0xff, 0x0a, 0x14, 0x28, 0xff,
  0x0c, 0x18, 0x30, 0xff, 0x0e, 0x1c, 0x38, 0xff,
};

static uint8 kpng_8x4_drawImage_scale_out[128] = {
  // expected result of scale src image larger than dest image.
  0x7c, 0x7c, 0x7c, 0xff, 0x7e, 0x7e, 0x7e, 0xff,
  0x80, 0x80, 0x80, 0xff, 0x82, 0x82, 0x82, 0xff,
  0x85, 0x85, 0x85, 0xff, 0x87, 0x87, 0x87, 0xff,
  0x89, 0x89, 0x89, 0xff, 0x8b, 0x8b, 0x8b, 0xff,
  0x70, 0x70, 0x70, 0xff, 0x72, 0x72, 0x72, 0xff,
  0x74, 0x74, 0x74, 0xff, 0x76, 0x76, 0x76, 0xff,
  0x79, 0x79, 0x79, 0xff, 0x7b, 0x7b, 0x7b, 0xff,
  0x7d, 0x7d, 0x7d, 0xff, 0x7f, 0x7f, 0x7f, 0xff,
  0x56, 0x56, 0x56, 0xff, 0x58, 0x58, 0x58, 0xff,
  0x5a, 0x5a, 0x5a, 0xff, 0x5c, 0x5c, 0x5c, 0xff,
  0x5f, 0x5f, 0x5f, 0xff, 0x61, 0x61, 0x61, 0xff,
  0x63, 0x63, 0x63, 0xff, 0x65, 0x65, 0x65, 0xff,
  0x4a, 0x4a, 0x4a, 0xff, 0x4c, 0x4c, 0x4c, 0xff,
  0x4e, 0x4e, 0x4e, 0xff, 0x50, 0x50, 0x50, 0xff,
  0x53, 0x53, 0x53, 0xff, 0x55, 0x55, 0x55, 0xff,
  0x57, 0x57, 0x57, 0xff, 0x59, 0x59, 0x59, 0xff,
};

static uint8 kpng_8x4_drawImage_flip[128] = {
  // expected result of flip src image.
  0x30, 0x60, 0xc0, 0xff, 0x32, 0x64, 0xc8, 0xff,
  0x22, 0x22, 0x22, 0xff, 0x21, 0x21, 0x21, 0xff,
  0x20, 0x20, 0x20, 0xff, 0x1f, 0x1f, 0x1f, 0xff,
  0x3c, 0x78, 0xf0, 0xff, 0x3e, 0x7c, 0xf8, 0xff,
  0x20, 0x40, 0x80, 0xff, 0x22, 0x44, 0x88, 0xff,
  0x26, 0x26, 0x26, 0xff, 0x25, 0x25, 0x25, 0xff,
  0x24, 0x24, 0x24, 0xff, 0x23, 0x23, 0x23, 0xff,
  0x2c, 0x58, 0xb0, 0xff, 0x2e, 0x5c, 0xb8, 0xff,
  0x10, 0x20, 0x40, 0xff, 0x12, 0x24, 0x48, 0xff,
  0x2a, 0x2a, 0x2a, 0xff, 0x29, 0x29, 0x29, 0xff,
  0x28, 0x28, 0x28, 0xff, 0x27, 0x27, 0x27, 0xff,
  0x1c, 0x38, 0x70, 0xff, 0x1e, 0x3c, 0x78, 0xff,
  0x00, 0x00, 0x00, 0xff, 0x02, 0x04, 0x08, 0xff,
  0x2e, 0x2e, 0x2e, 0xff, 0x2d, 0x2d, 0x2d, 0xff,
  0x2c, 0x2c, 0x2c, 0xff, 0x2b, 0x2b, 0x2b, 0xff,
  0x0c, 0x18, 0x30, 0xff, 0x0e, 0x1c, 0x38, 0xff,
};

static uint8 kpng_8x4_drawImage_argb8[128] = {
  // expected result of drawimage with rgb8 format.
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0xca, 0xca, 0xca, 0x3e, 0xd7, 0xd7, 0xd7, 0x9a,
  0x00, 0x00, 0x00, 0xee, 0x00, 0x00, 0x00, 0xff,
  0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0xff,
  0x00, 0x00, 0x00, 0x00, 0xca, 0xca, 0xca, 0x3e,
  0xe6, 0xe6, 0xe6, 0xdc, 0xfc, 0xfc, 0xfc, 0xfe,
  0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0xff,
  0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0xff,
  0xca, 0xca, 0xca, 0x3e, 0xe6, 0xe6, 0xe6, 0xdc,
  0xfe, 0xfe, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff,
  0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0xff,
  0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0xff,
  0xc7, 0xc7, 0xc7, 0x83, 0xf4, 0xf4, 0xf4, 0xfb,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0x00, 0x00, 0x00, 0xeb, 0x00, 0x00, 0x00, 0xff,
  0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0xff,
};

// DrawImage from another bitmap on different positions,
// compare with expected results.
TEST_F(BitmapTest, DrawImage) {
  // path of dest image.
  String fname_dst = *g_program_path +
                     "/bitmap_test/png-8x4-24bit-drawimage-dest.png";
  BitmapRefArray bitmaps;
  // load three src bitmaps in different sizes from files.
  String filename_2x2_src = *g_program_path +
      "/bitmap_test/png-2x2-24bit-drawimage-src.png";
  EXPECT_TRUE(Bitmap::LoadFromFile(
      g_service_locator, UTF8ToFilePath(filename_2x2_src),
      image::PNG, &bitmaps));
  ASSERT_EQ(1u, bitmaps.size());
  Bitmap::Ref bitmap_2x2_src(bitmaps[0]);

  String filename_4x4_src = *g_program_path +
      "/bitmap_test/png-4x4-24bit-drawimage-src.png";
  bitmaps.clear();
  EXPECT_TRUE(Bitmap::LoadFromFile(
      g_service_locator, UTF8ToFilePath(filename_4x4_src),
      image::PNG, &bitmaps));
  ASSERT_EQ(1u, bitmaps.size());
  Bitmap::Ref bitmap_4x4_src(bitmaps[0]);

  String filename_8x8_src = *g_program_path +
      "/bitmap_test/png-8x8-24bit-drawimage-src.png";
  bitmaps.clear();
  EXPECT_TRUE(Bitmap::LoadFromFile(
      g_service_locator, UTF8ToFilePath(filename_8x8_src),
      image::PNG, &bitmaps));
  ASSERT_EQ(1u, bitmaps.size());
  Bitmap::Ref bitmap_8x8_src(bitmaps[0]);

  // test draw image on top left boundary.
  bitmaps.clear();
  EXPECT_TRUE(Bitmap::LoadFromFile(
      g_service_locator, UTF8ToFilePath(fname_dst),
      image::PNG, &bitmaps));
  ASSERT_EQ(1u, bitmaps.size());
  Bitmap::Ref bitmap_dest_top_left(bitmaps[0]);
  // test whether the raw image is loaded correctly or not.
  EXPECT_TRUE(bitmap_dest_top_left->image_data() != NULL);
  EXPECT_TRUE(TestBitmapData(*bitmap_dest_top_left, kpng_8x4_drawImage));
  bitmap_dest_top_left->DrawImage(
      *bitmap_4x4_src, 0, 0, 0, 4, 4, 0, -1, -1, 4, 4);
  EXPECT_TRUE(TestBitmapData(*bitmap_dest_top_left,
                             kpng_8x4_drawImage_top_left));

  // test draw image on top boundary.
  bitmaps.clear();
  EXPECT_TRUE(Bitmap::LoadFromFile(
      g_service_locator, UTF8ToFilePath(fname_dst),
      image::PNG, &bitmaps));
  ASSERT_EQ(1u, bitmaps.size());
  Bitmap::Ref bitmap_dest_top(bitmaps[0]);
  bitmap_dest_top->DrawImage(*bitmap_4x4_src, 0, 0, 0, 4, 4, 0, 2, -2, 4, 4);
  EXPECT_TRUE(TestBitmapData(*bitmap_dest_top, kpng_8x4_drawImage_top));

  // test draw image on top right boundary.
  bitmaps.clear();
  EXPECT_TRUE(Bitmap::LoadFromFile(
      g_service_locator, UTF8ToFilePath(fname_dst),
      image::PNG, &bitmaps));
  ASSERT_EQ(1u, bitmaps.size());
  Bitmap::Ref bitmap_dest_top_right(bitmaps[0]);
  bitmap_dest_top_right->DrawImage(
      *bitmap_4x4_src, 0, 0, 0, 4, 4, 0, 5, -1, 4, 4);
  EXPECT_TRUE(TestBitmapData(*bitmap_dest_top_right,
                              kpng_8x4_drawImage_top_right));

  // test draw image on right boundary.
  bitmaps.clear();
  EXPECT_TRUE(Bitmap::LoadFromFile(
      g_service_locator, UTF8ToFilePath(fname_dst),
      image::PNG, &bitmaps));
  ASSERT_EQ(1u, bitmaps.size());
  Bitmap::Ref bitmap_dest_right(bitmaps[0]);
  bitmap_dest_right->DrawImage(*bitmap_4x4_src, 0, 0, 0, 4, 4, 0, 5, 0, 4, 4);
  EXPECT_TRUE(TestBitmapData(*bitmap_dest_right, kpng_8x4_drawImage_right));

  // test draw image on bottom right boundary.
  bitmaps.clear();
  EXPECT_TRUE(Bitmap::LoadFromFile(
      g_service_locator, UTF8ToFilePath(fname_dst),
      image::PNG, &bitmaps));
  ASSERT_EQ(1u, bitmaps.size());
  Bitmap::Ref bitmap_dest_bottom_right(bitmaps[0]);
  bitmap_dest_bottom_right->DrawImage(
      *bitmap_4x4_src, 0, 0, 0, 4, 4, 0, 5, 1, 4, 4);
  EXPECT_TRUE(TestBitmapData(*bitmap_dest_bottom_right,
                             kpng_8x4_drawImage_bottom_right));

  // test draw image on bottom boundary.
  bitmaps.clear();
  EXPECT_TRUE(Bitmap::LoadFromFile(
      g_service_locator, UTF8ToFilePath(fname_dst),
      image::PNG, &bitmaps));
  ASSERT_EQ(1u, bitmaps.size());
  Bitmap::Ref bitmap_dest_bottom(bitmaps[0]);
  bitmap_dest_bottom->DrawImage(
      *bitmap_4x4_src, 0, 0, 0, 4, 4, 0, 2, 1, 4, 4);
  EXPECT_TRUE(TestBitmapData(*bitmap_dest_bottom, kpng_8x4_drawImage_bottom));

  // test draw image on bottom left boundary.
  bitmaps.clear();
  EXPECT_TRUE(Bitmap::LoadFromFile(
      g_service_locator, UTF8ToFilePath(fname_dst),
      image::PNG, &bitmaps));
  ASSERT_EQ(1u, bitmaps.size());
  Bitmap::Ref bitmap_dest_bottom_left(bitmaps[0]);
  bitmap_dest_bottom_left->DrawImage(
      *bitmap_4x4_src, 0, 0, 0, 4, 4, 0, -1, 1, 4, 4);
  EXPECT_TRUE(TestBitmapData(*bitmap_dest_bottom_left,
                             kpng_8x4_drawImage_bottom_left));

  // test draw image on left boundary.
  bitmaps.clear();
  EXPECT_TRUE(Bitmap::LoadFromFile(
      g_service_locator, UTF8ToFilePath(fname_dst),
      image::PNG, &bitmaps));
  ASSERT_EQ(1u, bitmaps.size());
  Bitmap::Ref bitmap_dest_left(bitmaps[0]);
  bitmap_dest_left->DrawImage(*bitmap_4x4_src, 0, 0, 0, 4, 4, 0, -1, 0, 4, 4);
  EXPECT_TRUE(TestBitmapData(*bitmap_dest_left, kpng_8x4_drawImage_left));

  // test scale up.
  bitmaps.clear();
  EXPECT_TRUE(Bitmap::LoadFromFile(
      g_service_locator, UTF8ToFilePath(fname_dst),
      image::PNG, &bitmaps));
  ASSERT_EQ(1u, bitmaps.size());
  Bitmap::Ref bitmap_dest_scale_up(bitmaps[0]);
  bitmap_dest_scale_up->DrawImage(
      *bitmap_2x2_src, 0, 0, 0, 2, 2, 0, 0, 0, 8, 4);
  EXPECT_TRUE(TestBitmapData(*bitmap_dest_scale_up,
                             kpng_8x4_drawImage_scale_up));

  // test scale down.
  bitmaps.clear();
  EXPECT_TRUE(Bitmap::LoadFromFile(
      g_service_locator, UTF8ToFilePath(fname_dst),
      image::PNG, &bitmaps));
  ASSERT_EQ(1u, bitmaps.size());
  Bitmap::Ref bitmap_dest_scale_down(bitmaps[0]);
  bitmap_dest_scale_down->DrawImage(
      *bitmap_8x8_src, 0, 0, 0, 8, 8, 0, 0, 0, 4, 4);
  EXPECT_TRUE(TestBitmapData(*bitmap_dest_scale_down,
                             kpng_8x4_drawImage_scale_down));

  // test scale up to a large size.
  bitmaps.clear();
  EXPECT_TRUE(Bitmap::LoadFromFile(
      g_service_locator, UTF8ToFilePath(fname_dst),
      image::PNG, &bitmaps));
  ASSERT_EQ(1u, bitmaps.size());
  Bitmap::Ref bitmap_dest_scale_out(bitmaps[0]);
  bitmap_dest_scale_out->DrawImage(
      *bitmap_8x8_src, 0, 0, 0, 8, 8, 0, -2, -4, 12, 12);
  EXPECT_TRUE(TestBitmapData(*bitmap_dest_scale_out,
                             kpng_8x4_drawImage_scale_out));

  // test flip an image on both x and y cooridnates.
  bitmaps.clear();
  EXPECT_TRUE(Bitmap::LoadFromFile(
      g_service_locator, UTF8ToFilePath(fname_dst),
      image::PNG, &bitmaps));
  ASSERT_EQ(1u, bitmaps.size());
  Bitmap::Ref bitmap_dest_flip(bitmaps[0]);
  bitmap_dest_flip->DrawImage(*bitmap_4x4_src, 0, 0, 0, 4, 4, 0, 5, 3, -4, -4);
  EXPECT_TRUE(TestBitmapData(*bitmap_dest_flip, kpng_8x4_drawImage_flip));

  // test draw image on argb8 format.
  String fname_dst_argb8 = *g_program_path +
                           "/bitmap_test/" +
                           "png-8x4-24bit-drawimage-argb8-dest.png";
  bitmaps.clear();
  EXPECT_TRUE(Bitmap::LoadFromFile(
      g_service_locator, UTF8ToFilePath(fname_dst_argb8),
      image::PNG, &bitmaps));
  ASSERT_EQ(1u, bitmaps.size());
  Bitmap::Ref bitmap_dest_argb8(bitmaps[0]);
  String fname_src_argb8 = *g_program_path +
                           "/bitmap_test/" +
                           "png-4x4-24bit-drawimage-argb8-src.png";
  bitmaps.clear();
  EXPECT_TRUE(Bitmap::LoadFromFile(
      g_service_locator, UTF8ToFilePath(fname_src_argb8),
      image::PNG, &bitmaps));
  ASSERT_EQ(1u, bitmaps.size());
  Bitmap::Ref bitmap_src_argb8(bitmaps[0]);
  bitmap_dest_argb8->DrawImage(*bitmap_src_argb8, 0, 0, 0, 4, 4, 0, 0, 0, 4, 4);
  EXPECT_TRUE(TestBitmapData(*bitmap_dest_argb8, kpng_8x4_drawImage_argb8));
}

TEST_F(BitmapTest, SetRect) {
  Bitmap::Ref bitmap(new Bitmap(g_service_locator));
  const int kWidth = 8;
  const int kHeight = 8;
  const int kLevels = 2;
  const int kDestMip = 1;
  const unsigned kDestX = 1u;
  const unsigned kDestY = 1u;
  bitmap->Allocate(o3d::Texture::R32F, kWidth, kHeight, kLevels, Bitmap::IMAGE);
  const float* pixels =
      reinterpret_cast<const float*>(bitmap->GetMipData(kDestMip));
  static const float kExpected1[] = {
    0.0f, 0.0f, 0.0f, 0.0f,
    0.0f, 0.0f, 0.0f, 0.0f,
    0.0f, 0.0f, 0.0f, 0.0f,
    0.0f, 0.0f, 0.0f, 0.0f,
  };
  EXPECT_EQ(0, memcmp(pixels, kExpected1, sizeof(kExpected1)));
  const int kSrcWidth = 2;
  const int kSrcHeight = 2;
  static const float kSourcePixels[] = {
    0.123f, 0.456f,
    0.789f, 123.0f,
  };
  const int kSourcePitch = sizeof(kSourcePixels[0]) * kSrcWidth;
  // normal copy
  bitmap->SetRect(kDestMip, kDestX, kDestY,
                  kSrcWidth, kSrcHeight, kSourcePixels, kSourcePitch);
  static const float kExpected2[] = {
    0.0f, 0.0f, 0.0f, 0.0f,
    0.0f, 0.123f, 0.456f, 0.0f,
    0.0f, 0.789f, 123.0f, 0.0f,
    0.0f, 0.0f, 0.0f, 0.0f,
  };
  EXPECT_EQ(0, memcmp(pixels, kExpected2, sizeof(kExpected2)));
  // flipped copy
  bitmap->SetRect(
      kDestMip, kDestX, kDestY,
      kSrcWidth, kSrcHeight,
      reinterpret_cast<const uint8*>(kSourcePixels) + kSourcePitch,
      -kSourcePitch);
  static const float kExpected3[] = {
    0.0f, 0.0f, 0.0f, 0.0f,
    0.0f, 0.789f, 123.0f, 0.0f,
    0.0f, 0.123f, 0.456f, 0.0f,
    0.0f, 0.0f, 0.0f, 0.0f,
  };
  EXPECT_EQ(0, memcmp(pixels, kExpected3, sizeof(kExpected3)));
}

}  // namespace
