// Copyright (c) 2009 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef GFX_GTK_UTIL_H_
#define GFX_GTK_UTIL_H_
#pragma once

#include <stdint.h>
#include <vector>

#include <glib-object.h>

#include "base/scoped_ptr.h"

typedef struct _GdkColor GdkColor;
typedef struct _GdkPixbuf GdkPixbuf;
typedef struct _GdkRegion GdkRegion;

class SkBitmap;

const int kSkiaToGDKMultiplier = 257;

// Define a macro for creating GdkColors from RGB values.  This is a macro to
// allow static construction of literals, etc.  Use this like:
//   GdkColor white = GDK_COLOR_RGB(0xff, 0xff, 0xff);
#define GDK_COLOR_RGB(r, g, b) {0, r * kSkiaToGDKMultiplier, \
        g * kSkiaToGDKMultiplier, b * kSkiaToGDKMultiplier}

namespace gfx {

class Rect;

extern const GdkColor kGdkWhite;
extern const GdkColor kGdkGray;
extern const GdkColor kGdkBlack;
extern const GdkColor kGdkGreen;

// Convert and copy a SkBitmap to a GdkPixbuf. NOTE: this uses BGRAToRGBA, so
// it is an expensive operation.  The returned GdkPixbuf will have a refcount of
// 1, and the caller is responsible for unrefing it when done.
GdkPixbuf* GdkPixbufFromSkBitmap(const SkBitmap* bitmap);

// Modify the given region by subtracting the given rectangles.
void SubtractRectanglesFromRegion(GdkRegion* region,
                                  const std::vector<Rect>& cutouts);

// Returns the resolution (DPI) used by pango. A negative values means the
// resolution hasn't been set.
double GetPangoResolution();

}  // namespace gfx

namespace {
// A helper class that will g_object_unref |p| when it goes out of scope.
// This never adds a ref, it only unrefs.
template <typename Type>
struct GObjectUnrefer {
  void operator()(Type* ptr) const {
    if (ptr)
      g_object_unref(ptr);
  }
};
}  // namespace

// It's not legal C++ to have a templatized typedefs, so we wrap it in a
// struct.  When using this, you need to include ::Type.  E.g.,
// ScopedGObject<GdkPixbufLoader>::Type loader(gdk_pixbuf_loader_new());
template<class T>
struct ScopedGObject {
  typedef scoped_ptr_malloc<T, GObjectUnrefer<T> > Type;
};

#endif  // GFX_GTK_UTIL_H_
