// Copyright (c) 2006-2008 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/tab_contents/infobar_delegate.h"

#include "app/l10n_util.h"
#include "base/logging.h"
#include "build/build_config.h"
#include "chrome/browser/tab_contents/navigation_entry.h"
#include "chrome/browser/tab_contents/navigation_controller.h"
#include "chrome/browser/tab_contents/tab_contents.h"
#include "grit/generated_resources.h"

// InfoBarDelegate: ------------------------------------------------------------

bool InfoBarDelegate::ShouldExpire(
    const NavigationController::LoadCommittedDetails& details) const {
  bool is_reload =
      PageTransition::StripQualifier(details.entry->transition_type()) ==
          PageTransition::RELOAD;
  return is_reload || (contents_unique_id_ != details.entry->unique_id());
}

InfoBarDelegate::InfoBarDelegate(TabContents* contents)
    : contents_unique_id_(0) {
  if (contents)
    StoreActiveEntryUniqueID(contents);
}

void InfoBarDelegate::StoreActiveEntryUniqueID(TabContents* contents) {
  NavigationEntry* active_entry = contents->controller().GetActiveEntry();
  contents_unique_id_ = active_entry ? active_entry->unique_id() : 0;
}

// AlertInfoBarDelegate: -------------------------------------------------------

bool AlertInfoBarDelegate::EqualsDelegate(InfoBarDelegate* delegate) const {
  AlertInfoBarDelegate* alert_delegate = delegate->AsAlertInfoBarDelegate();
  if (!alert_delegate)
    return false;

  return alert_delegate->GetMessageText() == GetMessageText();
}

AlertInfoBarDelegate::AlertInfoBarDelegate(TabContents* contents)
    : InfoBarDelegate(contents) {
}

// LinkInfoBarDelegate: --------------------------------------------------------

LinkInfoBarDelegate::LinkInfoBarDelegate(TabContents* contents)
    : InfoBarDelegate(contents) {
}

// ConfirmInfoBarDelegate: -----------------------------------------------------

string16 ConfirmInfoBarDelegate::GetButtonLabel(
    InfoBarButton button) const {
  if (button == BUTTON_OK)
    return l10n_util::GetStringUTF16(IDS_OK);
  if (button == BUTTON_CANCEL)
    return l10n_util::GetStringUTF16(IDS_CANCEL);
  NOTREACHED();
  return string16();
}

ConfirmInfoBarDelegate::ConfirmInfoBarDelegate(TabContents* contents)
    : AlertInfoBarDelegate(contents) {
}

// SimpleAlertInfoBarDelegate: -------------------------------------------------

SimpleAlertInfoBarDelegate::SimpleAlertInfoBarDelegate(
    TabContents* contents,
    const string16& message,
    SkBitmap* icon,
    bool auto_expire)
    : AlertInfoBarDelegate(contents),
      message_(message),
      icon_(icon),
      auto_expire_(auto_expire) {
}

bool SimpleAlertInfoBarDelegate::ShouldExpire(
      const NavigationController::LoadCommittedDetails& details) const {
  if (auto_expire_)
    return AlertInfoBarDelegate::ShouldExpire(details);

  return false;
}

string16 SimpleAlertInfoBarDelegate::GetMessageText() const {
  return message_;
}

SkBitmap* SimpleAlertInfoBarDelegate::GetIcon() const {
  return icon_;
}

void SimpleAlertInfoBarDelegate::InfoBarClosed() {
  delete this;
}
