#!/bin/sh
#
# tagcp - A checkpoint stamper
# Copyright (C) 2006 Nippon Telegraph and Telephone Corporation.
# This file is part of NILFS and is licensed under the GPL.
#
# tagcp.sh,v 1.1 2006/03/20 08:07:38 ryusuke Exp
#
# Written by Ryusuke Konishi <ryusuke@osrg.net>
#
cmd=`basename $0`
cdir=`dirname $0`

show_help() {
    echo "Usage: $cmd [options..] [<directory>]"
    echo "Options:"
    echo "  -h            show this help"
    echo "  -f <file>     add sketch data to the current NILFS snapshot"
    echo "  -t <string>   add a text-based tag to the current NILFS snapshot"
    echo "                This option can be used with -c option to specify text types"
    echo "  -c <type>     specify file type, otherwise assumed from the file"
    echo "                extension or the content. <type> is one of the followings:"
    echo "                  text, jpeg, gif, png, html, xml, wiki or other"
    echo "  -m            moderate mode; cancel tagging if the file cache of"
    echo "                target fs seems to be clean"
    echo "  -n            nosync mode; do not write the tag just yet"
    echo "  -a            append the tag instead of overwriting"
    echo "Note: "
    echo "  If <directory> is omitted, all NILFS instances are tagged."
}

get_file_type() {
    ext=`echo $1 | sed 'y/ABCDEFGHIJKLMNOPQRSTUVWXYZ/abcdefghijklmnopqrstuvwxyz/'`
    case "$ext" in
	text|txt) echo T;;
	jpeg|jpg) echo J;;
	gif) echo G;;
	png) echo P;;
	html|htm) echo H;;
	xml) echo X;;
	wiki) echo W;;
	uri|url) echo U;;
	elf|sh|pl) echo E;;
	other) echo O;;
    esac
}

while getopts ht:f:namc: o; do
    case $o in
	h)
	    show_help 1>&2
	    exit 0 ;;
	t)
	    tagname="$OPTARG"; : ${file_type:=T} ;;
	f)
	    file="$OPTARG" ;;
	n)
	    nosync=1 ;;
	a)
	    append=1 ;;
	m)
	    moderate=1 ;;
	c)
	    file_type=`get_file_type $OPTARG` ;;
	*)
	    show_help 1>&2
	    exit 1 ;;
    esac
done
shift `expr $OPTIND - 1`

if [ -z "$tagname" -a -z "$file" ]; then
    show_help 1>&2; exit 1
fi

function fs_dir2dev() {
    df -t $2 "$1" | sed -ne '2{s/^\(\S\+\)\s.*$/\1/;p;q}'
}

function is_autofs_dir() {
    sed -ne '\|^automount(\S\+)\s\+'"$1"'\s\+autofs\s|{x;s/^/y/;p;q}' /etc/mtab
}

if [ -z "$1" ]; then
    autofs_dirs=`awk '$3 == "autofs" {print $2}' /etc/mtab`
    for autofs_dir in $autofs_dirs; do [ -d $autofs_dir/cur ]; done
    devices=`awk '$3 == "nilfs" && $4 ~ /rw/ {print $1}' /etc/mtab | sort | uniq`
elif [ -d "$1" ]; then
    devices=`fs_dir2dev "$1" nilfs`
    if [ -z "$devices" ]; then
	autofs=`is_autofs_dir "$1"`
	if [ "$autofs" = y -a -d $1/cur ]; then
	    devices=`fs_dir2dev "$1/cur" nilfs`
	fi
    fi
    if [ -z "$devices" ]; then
	echo "Not NILFS directory" 1>&2; exit 1
    fi
else
    show_help 1>&2; exit 1
fi

put_text() {
    [ -n "$append" ] || echo -n > "$2"
    LANG=C; printf '!%c%0.4u%s' ${file_type:-T} ${#1} "$1" >> "$2"
}

put_file() {
    bytes=`stat -c "%s" "$1"`
    if [ $bytes -gt 9999 ]; then bytes=9999; fi
    [ -n "$append" ] || echo -n > "$2"
    printf '!%c%0.4u' ${file_type:-O} $bytes >> "$2"
    head -c $bytes "$1" >> "$2"
}

guess_file_type() {
    ext=`expr "$1" : ".*\.\([A-Za-z]\+\)$"`
    file_type=`get_file_type $ext`
    [ -z "$file_type" ] || return;
    [ -x "`which file`" ] || return;
    case "`file -biL $1`" in
	text/plain*|application/x-shellscript) file_type=T ;;
	text/html) file_type=H ;;
	image/jpeg) file_type=J ;;
	image/gif) file_type=G ;;
	image/png) file_type=P ;;
	application/*) file_type=E ;;
    esac
    [ -z "$file_type" ] || return;
    ext=`file -bL $1 | sed 's/^\(\S\+\)\s.*$/\1/'`
    file_type=`get_file_type $ext`
}

export -n file_type
if [ -n "$file" ]; then
    if [ -z "$file_type" ]; then
	guess_file_type $file
	if [ -z "$file_type" ]; then
	    echo "Cannot detect file type. Use -c option." 1>&2
	    exit 1
	fi
    fi
    if [ "$file_type" == E -o -x $file ]; then
	echo "Executable file is not permitted." 1>&2
	exit 1
    elif [ ! -e $file ]; then
	echo "File not found." 1>&2
	exit 1
    elif [ ! -r $file ]; then
	echo "File not readable." 1>&2
	exit 1
    fi
fi

for device in $devices; do
    mntdir=`awk '$1 == "'$device'" && $3 == "nilfs" && $4 ~ /rw/ {print $2; exit}' /etc/mtab`
    test -n "$mntdir" -a -d $mntdir || continue

    echo "Tagging $device" 1>&2
    sketch=`find $mntdir -maxdepth 1 -mount -inum 10 -print`
    if [ -z "$sketch" -o ! -f "$sketch" ]; then
	echo "Cannot find sketch file" 1>&2; continue
    elif [ ! -w $sketch ]; then
	echo "Cannot write to sketch file" 1>&2; continue
    fi

    if [ -n "$moderate" ]; then
	backup=/tmp/sketch.$PPID.$cmd
	cp $sketch $backup
    fi

    if [ -n "$tagname" ]; then
	put_text "$tagname" "$sketch"
    elif [ -n "$file" ]; then
	put_file "$file" "$sketch"
    fi

    if [ -n "$moderate" ]; then
	sync
	if [ -s $sketch ]; then 
	    cp $backup $sketch;
	    echo "Canceled tagging $device" 1>&2
	fi
	rm -f $backup
    elif [ -z "$nosync" ]; then
	sync
	if [ -s $sketch ]; then touch $mntdir; sync; fi
    fi
done
