/*
 * super.c - NILFS module and super block management.
 *
 * Copyright (C) 2005 Nippon Telegraph and Telephone Corporation.
 *
 * This file is part of NILFS.
 *
 * NILFS is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * NILFS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with NILFS; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * super.c,v 1.244 2006/07/13 02:16:18 ryusuke Exp
 *
 * Written by Ryusuke Konishi <ryusuke@osrg.net>
 */
/*
 *  linux/fs/ext2/super.c
 *
 * Copyright (C) 1992, 1993, 1994, 1995
 * Remy Card (card@masi.ibp.fr)
 * Laboratoire MASI - Institut Blaise Pascal
 * Universite Pierre et Marie Curie (Paris VI)
 *
 *  from
 *
 *  linux/fs/minix/inode.c
 *
 *  Copyright (C) 1991, 1992  Linus Torvalds
 *
 *  Big-endian to little-endian byte-swapping/bitmaps by
 *        David S. Miller (davem@caip.rutgers.edu), 1995
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/string.h>
#include <linux/slab.h>
#include <linux/init.h>
#include <linux/blkdev.h>
#include <linux/parser.h>
#include <linux/random.h>
#include <linux/smp_lock.h>
#include <linux/vfs.h>
#include <linux/writeback.h>
#include <linux/kobject.h>
#include "nilfs.h"

MODULE_AUTHOR("NTT Corp.");
MODULE_DESCRIPTION("A New Implementation of the Log-structured Filesystem (NILFS)");
MODULE_VERSION(NILFS_VERSION);
MODULE_LICENSE("GPL");

LIST_HEAD(nilfs_super_blocks);
spinlock_t nilfs_sb_lock = SPIN_LOCK_UNLOCKED;
struct shrinker *nilfs_shrinker;

static int nilfs_commit_super(struct super_block *sb,
			      struct nilfs_super_block *sbp, int sync);
static int nilfs_remount(struct super_block *sb, int *flags, char *data);
static int check_rw_mount(struct file_system_type *fs_type, struct block_device *bdev);

/**
 * nilfs_error() - report failure condition on a filesystem
 *
 * nilfs_error() sets an ERROR_FS flag on the superblock as well as 
 * reporting an error message.  It should be called when NILFS detects
 * incoherences or defects of meta data on disk.  As for sustainable
 * errors such as a single-shot I/O error, nilfs_warning() or the printk()
 * function should be used instead.
 *
 * The segment constructor must not call this function because it can
 * kill itself.
 */
void
nilfs_error(struct super_block *sb, const char *function,
	    const char *fmt, ...)
{
	struct nilfs_sb_info *sbi = NILFS_SB(sb);
	va_list args;

	va_start(args, fmt);
	printk(KERN_CRIT "NILFS error (device %s): %s: ", sb->s_id, function);
	vprintk(fmt, args);
	printk("\n");
	va_end(args);

	if (!(sb->s_flags & MS_RDONLY)) {
		struct the_nilfs *nilfs = sbi->s_nilfs;

		if (!nilfs_test_opt(sbi, ERRORS_CONT)) {
			nilfs_detach_writer(nilfs, sbi);
			nilfs_free_free_inode_list(sb);
			nilfs_clear_segment(sbi);
		} 
		down_write(&nilfs->ns_sem);
		if (!(nilfs->ns_mount_state & NILFS_ERROR_FS)) {
			nilfs->ns_mount_state |= NILFS_ERROR_FS;
			nilfs->ns_sbp->s_state |= cpu_to_le16(NILFS_ERROR_FS);
			nilfs_commit_super(sb, nilfs->ns_sbp, 1);
		}
		up_write(&nilfs->ns_sem);
		nilfs_dump_stack(NILFS_VERBOSE_FS, 2);

		if (nilfs_test_opt(sbi, ERRORS_RO)) {
			printk(KERN_CRIT "Remounting filesystem read-only\n");
			sb->s_flags |= MS_RDONLY;
		} 
	}

	if (nilfs_test_opt(sbi, ERRORS_PANIC))
		panic("NILFS (device %s): panic forced after error\n", sb->s_id);
}

void nilfs_warning(struct super_block *sb, const char *function,
		   const char *fmt, ...)
{
	va_list args;

	va_start(args, fmt);
	printk(KERN_WARNING "NILFS warning (device %s): %s: ",
	       sb->s_id, function);
	vprintk(fmt, args);
	printk("\n");
	va_end(args);
}

static kmem_cache_t *nilfs_inode_cachep;

static struct inode *nilfs_alloc_inode(struct super_block *sb)
{
	struct nilfs_inode_info *ip;

	ip = (struct nilfs_inode_info *)kmem_cache_alloc(nilfs_inode_cachep, SLAB_NOFS);
	if(!ip)
		return NULL;

	ip->vfs_inode.i_version = 1;
	return &ip->vfs_inode;
}

static void nilfs_destroy_inode(struct inode *inode)
{
	/* nilfs_clean_all_file_node_blk(inode); */
	radix_tree_64_force_empty(&NILFS_I(inode)->i_block_ntree);
	kmem_cache_free(nilfs_inode_cachep, NILFS_I(inode));
}

static void init_once(void *obj, kmem_cache_t *cachep, unsigned long flags)
{
	struct nilfs_inode_info *ip = (struct nilfs_inode_info *)obj;
	if((flags & (SLAB_CTOR_VERIFY | SLAB_CTOR_CONSTRUCTOR)) ==
	   SLAB_CTOR_CONSTRUCTOR) {
		rwlock_init(&ip->i_meta_lock);
		init_rwsem(&ip->i_block_sem);
		INIT_RADIX_TREE_64(&ip->i_block_ntree, GFP_ATOMIC);
		spin_lock_init(&ip->i_block_ntree_lock);
		INIT_LIST_HEAD(&ip->i_dirty);
		init_MUTEX(&ip->i_node_pages_sem);
		INIT_LIST_HEAD(&ip->i_partial_node_pages);
#ifdef CONFIG_NILFS_XATTR
		init_rwsem(&ip->xattr_sem);
#endif
		inode_init_once(&ip->vfs_inode);
	}
}

static int init_inodecache(void)
{
	nilfs_inode_cachep = kmem_cache_create("nilfs_inode_cache",
					       sizeof(struct nilfs_inode_info),
					       0, SLAB_RECLAIM_ACCOUNT,
					       init_once, NULL);
  
	return ((nilfs_inode_cachep == NULL) ? -ENOMEM : 0);
}

static void destroy_inodecache(void)
{
	if(kmem_cache_destroy(nilfs_inode_cachep)) {
		printk(KERN_INFO "NILFS: some inodes on memory were not freed\n");
	}
}

static void nilfs_clear_inode(struct inode *inode)
{
	struct nilfs_inode_info *ii = NILFS_I(inode);

#ifdef CONFIG_NILFS_POSIX_ACL
	if(ii->i_acl && ii->i_acl != NILFS_ACL_NOT_CACHED) {
		posix_acl_release(ii->i_acl);
		ii->i_acl = NILFS_ACL_NOT_CACHED;
	}
	if(ii->i_default_acl && ii->i_default_acl != NILFS_ACL_NOT_CACHED) {
		posix_acl_release(ii->i_default_acl);
		ii->i_default_acl = NILFS_ACL_NOT_CACHED;
	}
#endif
	if(likely(!is_bad_inode(inode))) {
		struct nilfs_transaction_info ti;
		struct nilfs_sb_info *sbi = NILFS_SB(inode->i_sb);

		/*
		 * Free resources allocated in nilfs_read_inode(), here.
		 */
		nilfs_transaction_begin(inode->i_sb, &ti);
		nilfs_btree_clear(&ii->i_block_root);
		nilfs_clean_all_file_node_blk(inode);

		spin_lock_irq(&sbi->s_segctor.dirty_files_lock);
		if(!list_empty(&ii->i_dirty)) {
			inode_debug(2, "canceling inode (ino=%lu) from dirty files list\n",
				    inode->i_ino);
			list_del_init(&ii->i_dirty);
		}
		brelse(ii->i_bh);
		ii->i_bh = NULL;
		spin_unlock_irq(&sbi->s_segctor.dirty_files_lock);

		if (unlikely(!list_empty(&ii->i_partial_node_pages)))
			inode_debug(0, "i_partial_node_pages is not empty (ino=%lu)\n",
				    ii->vfs_inode.i_ino);

		nilfs_check_radix_tree(__FUNCTION__, inode->i_mapping,
				       inode->i_blkbits);
		nilfs_check_radix_tree_64(__FUNCTION__, &ii->i_block_ntree,
					  &ii->i_block_ntree_lock,
					  inode->i_blkbits);

		nilfs_transaction_end(inode->i_sb);
#ifdef CONFIG_NILFS_DEBUG
		atomic_inc(&nilfs_released_inode_n);
#endif
	}
}

static int
nilfs_write_inode_page(struct page *page, struct writeback_control *wbc)
{
	struct nilfs_sb_info *sbi;
	int ret;

	page_debug(2, "called (page=%p, wbc nonblocking %d, wbc for_reclaim %d)\n",
		    page, wbc->nonblocking, wbc->for_reclaim);
	BUG_ON(!page->mapping);
	sbi = NILFS_AS_SB(page->mapping);
	unlock_page(page);
	if (wbc->sync_mode == WB_SYNC_ALL) {
		ret = nilfs_construct_segment(sbi->s_super);
		if (NILFS_SEG_ERR(ret))
			return ret;
	} else if (wbc->for_reclaim)
		nilfs_flush_segment(sbi, NILFS_SC_FLUSH_IBT);

	return 0;
}

static inline void
nilfs_register_super(struct nilfs_sb_info *sbi)
{
	spin_lock(&nilfs_sb_lock);
	list_add_tail(&sbi->s_list, &nilfs_super_blocks);
	spin_unlock(&nilfs_sb_lock);
}

static inline void
nilfs_unregister_super(struct nilfs_sb_info *sbi)
{
	spin_lock(&nilfs_sb_lock);
	list_del_init(&sbi->s_list);
	spin_unlock(&nilfs_sb_lock);
}

/**
 * nilfs_update_last_segment - change pointer to the latest segment
 * @sbi: nilfs_sb_info
 *
 * nilfs_update_last_segment() changes information in the super block
 * after a partial segment is written out successfully. The super
 * block is marked dirty. It will be written out at the next VFS sync
 * operations such as sync_supers() and generic_shutdown_super().
 */
void nilfs_update_last_segment(struct nilfs_sb_info *sbi)
{
	struct the_nilfs *nilfs = sbi->s_nilfs;

	nilfs_debug(2, "detected change of last checkpoint\n");
	down_write(&nilfs->ns_sem);
	nilfs->ns_sbp->s_last_segment = cpu_to_le64(nilfs->ns_last_pseg);
	nilfs->ns_sbp->s_last_seq = cpu_to_le64(nilfs->ns_last_seq);
	sbi->s_super->s_dirt = 1;
	nilfs_mark_buffer_dirty(nilfs->ns_sbh);
	up_write(&nilfs->ns_sem);
}

static int 
nilfs_commit_super(struct super_block *sb, struct nilfs_super_block *sbp, int sync)
{
	struct the_nilfs *nilfs = NILFS_SB(sb)->s_nilfs;
	int err = 0;

	/* nilfs->sem must be locked by the caller. */
	nilfs_debug(2, "called\n");
	sbp->s_free_blocks_count = cpu_to_le64(nilfs_count_free_blocks(NILFS_SB(sb)));
	sbp->s_wtime = cpu_to_le64(get_seconds());
	nilfs_mark_buffer_dirty(nilfs->ns_sbh);
	if (sync) {
#ifdef NILFS_SB_BARRIER
		struct nilfs_sb_info *sbi = NILFS_SB(sb);
		int barrier_done = 0;

		if (nilfs_test_opt(sbi, BARRIER)) {
			set_buffer_ordered(nilfs->ns_sbh);
			barrier_done = 1;
		}
	retry:
#endif
		err = nilfs_sync_dirty_buffer(nilfs->ns_sbh);
#ifdef NILFS_SB_BARRIER
		if (err == -EOPNOTSUPP && barrier_done) {
			nilfs_warning(sb, __FUNCTION__,
				      "barrier-based sync failed. "
				      "disabling barriers\n");
			nilfs_clear_opt(sbi, BARRIER);
			barrier_done = 0;
			clear_buffer_ordered(nilfs->ns_sbh);
			goto retry;
		}
#endif
		if (unlikely(err))
			printk("NILFS: unable to write superblock (err=%d)\n", err);
	}
	sb->s_dirt = 0;
	return err;
}

static void nilfs_put_super(struct super_block *sb)
{
	struct nilfs_sb_info *sbi = NILFS_SB(sb);
	struct the_nilfs *nilfs = sbi->s_nilfs;

	nilfs_detach_writer(nilfs, sbi);
	/*
	 * Nilfs_free_free_inode_list() may dirt the inode b-tree.
	 * So, we call it before the last segment construction.
	 * Although the inode_list is needed by nilfs_free_inode(),
	 * we can safely dispose it here because no files are dirty
	 * and no inodes will be freed in the last segment construction.
	 */
	nilfs_debug(2, "Calling nilfs_free_free_inode_list(sb)\n");
	nilfs_free_free_inode_list(sb);
	nilfs_debug(2, "Deactivating segment constructor\n");
	nilfs_clear_segment(sbi);

	if(!(sb->s_flags & MS_RDONLY)) {
		down_write(&nilfs->ns_sem);
		nilfs_debug(1, "Closing on-disk superblock\n");
		nilfs->ns_sbp->s_state = cpu_to_le16(nilfs->ns_mount_state);
		nilfs_commit_super(sb, nilfs->ns_sbp, 1);
		up_write(&nilfs->ns_sem);
	}
	list_del_init(&sbi->s_list);
	nilfs_debug(2, "Calling btree_clear(&sbi->s_inode_root)\n");
	nilfs_btree_clear(&sbi->s_inode_root);
	nilfs_debug(2, "Calling clean_inactive_node_pages\n");
	nilfs_clean_inactive_node_pages(sb);
	nilfs_debug(2, "Calling clean_all_inode_node_blk\n");
	nilfs_clean_all_inode_node_blk(sb);
	nilfs_debug(2, "Removing superblock from nilfs_super_blocks\n");
	nilfs_unregister_super(sbi);
	nilfs_debug(2, "Calling radix_tree_64_force_empty()\n");
	radix_tree_64_force_empty(&sbi->s_inode_ntree);
	nilfs_debug(2, "Calling free_inode_buffers()\n");
	nilfs_free_inode_buffers(sb);
	nilfs_check_radix_tree(__FUNCTION__, sbi->s_mapping,
			    sb->s_blocksize_bits);
	nilfs_check_radix_tree_64(__FUNCTION__, &sbi->s_inode_ntree,
			       &sbi->s_inode_ntree_lock,
			       sb->s_blocksize_bits);
	nilfs_debug(2, "Calling clean_free_inode_list()\n");
	nilfs_clean_free_inode_list(sb);

	sbi->s_super = NULL;
	sb->s_fs_info = NULL;

	put_nilfs(sbi->s_nilfs);
	kfree(sbi->s_mapping->backing_dev_info);
	kfree(sbi);
	nilfs_debug(1, "done\n");
}

static void __write_super(struct super_block *sb,
			  struct nilfs_super_block *sbp)
{
	nilfs_debug(1, "called\n");
	if(!(sb->s_flags & MS_RDONLY))
		nilfs_commit_super(sb, sbp, 1);
	sb->s_dirt = 0;
}

/**
 * nilfs_write_super - write super block(s) of NILFS
 * @sb: super_block
 *
 * nilfs_write_super() gets a fs-dependent lock, writes super block(s), and
 * clears s_dirt.  This function is called in the section protected by
 * lock_super().
 * 
 * The s_dirt flag is managed by each filesystem and we protect it by ns_sem
 * of the struct the_nilfs.  Lock order must be as follows:
 *
 *   1. lock_super()
 *   2.    down_write(&nilfs->ns_sem)
 * 
 * Inside NILFS, locking ns_sem is enough to protect s_dirt and the buffer
 * of the super block (nilfs->ns_sbp).
 *
 * In most cases, VFS functions call lock_super() before calling these 
 * methods.  So we must be careful not to bring on deadlocks when using
 * lock_super();  see generic_shutdown_super(), write_super(), and so on.
 *
 * Note that order of lock_kernel() and lock_super() depends on contexts
 * of VFS.  We should also note that lock_kernel() can be used in its
 * protective section and only the outermost one has an effect.
 */
static void nilfs_write_super(struct super_block *sb)
{
	struct the_nilfs *nilfs = NILFS_SB(sb)->s_nilfs;

	down_write(&nilfs->ns_sem);
	__write_super(sb, nilfs->ns_sbp);
	up_write(&nilfs->ns_sem);
}

static int nilfs_sync_fs(struct super_block *sb, int wait)
{
	int ret;
	/* This function is called when super block should be written back */
	nilfs_debug(2, "called(wait=%d)\n", wait);
	if (wait) {
		ret = nilfs_construct_segment(sb);
		if(NILFS_SEG_ERR(ret))
			return ret;
	}
	return 0;
}

static void
nilfs_complete_recovery(struct nilfs_sb_info *sbi, int seg_stat)
{
	struct the_nilfs *nilfs = sbi->s_nilfs;
	int err;
	int recovery_done = 1;

	BUG_ON(nilfs_test_opt(sbi, SNAPSHOT));

	if (sbi->s_super->s_flags & MS_RDONLY) {
		printk(KERN_INFO
		       "NILFS: read-only filesystem. recovery unfinished.\n");
		return;
	}

	if (seg_stat == RECOVERY_ROLLFORWARD_DONE) {
		err = nilfs_construct_segment(sbi->s_super);
		if (NILFS_SEG_ERR(err)) {
			printk(KERN_WARNING
			       "NILFS (device %s): Oops! recovery failed.\n",
			       sbi->s_super->s_id);
			return;
		}
	}
	down_write(&nilfs->ns_sem);
	if (seg_stat > 0) {
		nilfs->ns_sbp->s_last_segment = cpu_to_le64(nilfs->ns_last_pseg);
		nilfs->ns_sbp->s_last_seq = cpu_to_le64(nilfs->ns_last_seq);
		nilfs_commit_super(sbi->s_super, nilfs->ns_sbp, 1);
	} else if (nilfs->ns_mount_state & NILFS_VALID_FS) {
		recovery_done = 0;
	}
	nilfs->ns_mount_state |= NILFS_VALID_FS;
	up_write(&nilfs->ns_sem);

	if (recovery_done)
		printk(KERN_INFO "NILFS: recovery complete.\n");
}

static int nilfs_statfs(struct super_block *sb, struct kstatfs *buf)
{
	struct nilfs_sb_info *sbi = NILFS_SB(sb);
	unsigned long long blocks;
	unsigned long overhead;
	unsigned long reserved_blocks;
	struct the_nilfs *nilfs = sbi->s_nilfs;

	/*
	 * Compute all of the segment blocks
	 *
	 * The blocks before first segment and after last segment 
	 * are excluded.
	 */
	blocks = sbi->s_blocks_per_segment * sbi->s_nsegment
		- sbi->s_first_data_block;

	/*
	 * Compute the overhead
	 *
	 * When distributing meta data blocks outside semgent structure,
	 * We must count them as the overhead.
	 */
	overhead = 0;

	down_read(&nilfs->ns_sem);
	reserved_blocks = le32_to_cpu(nilfs->ns_sbp->s_r_segments_count)
		* sbi->s_blocks_per_segment;
	up_read(&nilfs->ns_sem);

	buf->f_type = NILFS_SUPER_MAGIC;
	buf->f_bsize = sb->s_blocksize;
	buf->f_blocks = blocks - overhead;
	buf->f_bfree = nilfs_count_free_blocks(sbi);
	buf->f_bavail = (buf->f_bfree >= reserved_blocks) ? 
		(buf->f_bfree - reserved_blocks) : 0;
	buf->f_files = atomic_read(&sbi->s_inodes_count);
	buf->f_ffree = 0; /* nilfs_count_free_inodes(sb); */
	buf->f_namelen = NILFS_NAME_LEN;
	return 0;
}

static struct super_operations nilfs_sops = {
	.alloc_inode    = nilfs_alloc_inode,
	.destroy_inode  = nilfs_destroy_inode,
	.read_inode     = nilfs_read_inode,
	.dirty_inode    = nilfs_dirty_inode,
	/* .write_inode    = nilfs_write_inode, */
	.put_inode      = nilfs_put_inode,
	/* .drop_inode	  = nilfs_drop_inode, */
	.delete_inode   = nilfs_delete_inode,
	.put_super      = nilfs_put_super,
	.write_super    = nilfs_write_super,
	.sync_fs        = nilfs_sync_fs,
	/* .write_super_lockfs */
	/* .unlockfs */
	.statfs         = nilfs_statfs,
	.remount_fs     = nilfs_remount,
	.clear_inode    = nilfs_clear_inode,
	/* .umount_begin */
	/* .show_options */
};

enum {
	Opt_err_cont, Opt_err_panic, Opt_err_ro,
	Opt_recovery, Opt_sync, Opt_barrier,
	Opt_snapshot, Opt_order,
	Opt_passive,
	Opt_err,
};

static match_table_t tokens = {
	{Opt_err_cont, "errors=continue"},
	{Opt_err_panic, "errors=panic"},
	{Opt_err_ro, "errors=remount-ro"},
	{Opt_recovery, "recovery=%s"},
	{Opt_sync, "sync=%s"},
	{Opt_barrier, "barrier=%s"},
	{Opt_snapshot, "cp=%u"},
	{Opt_order, "order=%s"},
	{Opt_passive, "passive=%s"},
	{Opt_err, NULL}
};

static int match_bool(substring_t *s, int *result)
{
	int len = s->to - s->from;

	if (strncmp(s->from, "on", len) == 0 ||
	    strncmp(s->from, "true", len) == 0)
		*result = 1;
	else if (strncmp(s->from, "off", len) == 0 ||
		 strncmp(s->from, "false", len) == 0 ||
		 strncmp(s->from, "none", len) == 0)
		*result = 0;
	else
		return 1;
	return 0;
}

static int parse_options(char *options, struct super_block *sb)
{
	struct nilfs_sb_info *sbi = NILFS_SB(sb);
	char *p;
	substring_t args[MAX_OPT_ARGS];
	int option;

	if(!options) return 1;

	while((p = strsep(&options, ",")) != NULL) {
		int token;
		if (!*p) continue;
    
		token = match_token(p, tokens, args);
		switch(token) {
		case Opt_recovery:
			if (match_bool(&args[0], &option))
				return 0;
			if (option)
				nilfs_write_opt(sbi, RECOVERY, RECOVERY_QUICK);
			else
				nilfs_clear_opt(sbi, RECOVERY);
			break;
		case Opt_sync:
			if (strcmp(args[0].from, "segment") == 0)
				nilfs_write_opt(sbi, SYNC_MODE, SYNC_SEGMENT);
			else if (strcmp(args[0].from, "bio") == 0)
				nilfs_write_opt(sbi, SYNC_MODE, SYNC_BIO);
			else if (match_bool(&args[0], &option) || option)
				return 0;
			else
				nilfs_clear_opt(sbi, SYNC_MODE);
			break;
		case Opt_barrier:
			if (match_bool(&args[0], &option))
				return 0;
			if (option)
				nilfs_set_opt(sbi, BARRIER);
			else
				nilfs_clear_opt(sbi, BARRIER);
			break;
		case Opt_order:
			if (strcmp(args[0].from, "relaxed") == 0)
				/* Ordered data semantics */
				nilfs_clear_opt(sbi, STRICT_ORDER);
			else if (strcmp(args[0].from, "strict") == 0)
				/* Strict in-order semantics */
				nilfs_set_opt(sbi, STRICT_ORDER);
			else
				return 0;
			break;
		case Opt_err_panic:
			nilfs_write_opt(sbi, ERROR_MODE, ERRORS_PANIC);
			break;
		case Opt_err_ro:
			nilfs_write_opt(sbi, ERROR_MODE, ERRORS_RO);
			break;
		case Opt_err_cont:
			nilfs_write_opt(sbi, ERROR_MODE, ERRORS_CONT);
			break;
		case Opt_snapshot:
			if (match_int(&args[0], &option))
				return 0;
			if (!(sb->s_flags & MS_RDONLY))
				return 0;
			sbi->s_snapshot_pseg = option;
			nilfs_set_opt(sbi, SNAPSHOT);
			break;
		case Opt_passive:
			if (match_bool(&args[0], &option))
				return 0;
			if (sb->s_flags & MS_RDONLY)
				return 0;
			if (option)
				nilfs_set_opt(sbi, PASSIVE);
			else
				nilfs_clear_opt(sbi, PASSIVE);
			break;
		default:
			printk(KERN_ERR "NILFS: Unrecognized mount option \"%s\"\n", p);
			return 0;
		}
	}
	return 1;
}

static inline void 
set_default_options(struct super_block *sb, struct nilfs_super_block *sbp)
{
	struct nilfs_sb_info *sbi = NILFS_SB(sb);

	sbi->s_mount_opt =
		NILFS_MOUNT_ERRORS_CONT |
		NILFS_MOUNT_RECOVERY_QUICK |
		NILFS_MOUNT_BARRIER |
		NILFS_MOUNT_SYNC_SEGMENT;
}

static loff_t nilfs_max_size(unsigned int bits)
{
	loff_t res;
	int max_bits = bits + min((loff_t)nilfs_btree_max_bits(),
				  nilfs_fbn_max_bits());
	if (max_bits > sizeof(loff_t) * 8 /* CHAR_BITS */ )
		res = ~((loff_t) 0);
	else
		res = ((loff_t)1 << max_bits);
	return res;
}

static int nilfs_setup_super(struct super_block *sb, struct nilfs_super_block *sbp)
{
	struct the_nilfs *nilfs = NILFS_SB(sb)->s_nilfs;
	int max_mnt_count = le16_to_cpu(sbp->s_max_mnt_count);
	int mnt_count = le16_to_cpu(sbp->s_mnt_count);

	/* nilfs->sem must be locked by the caller. */
	if (!(nilfs->ns_mount_state & NILFS_VALID_FS)) {
		printk(KERN_WARNING "NILFS warning: mounting unchecked fs, "
		       "running fsck is recommended\n");
	} else if(nilfs->ns_mount_state & NILFS_ERROR_FS) {
		printk(KERN_WARNING "NILFS warning: mounting fs with errors, "
		       "running fsck is recommended\n");
	} else if(max_mnt_count >= 0 && mnt_count >= max_mnt_count) {
		printk(KERN_WARNING "NILFS warning: maximal mount count reached, "
		       "running fsck is recommended\n");
	}
	if (!max_mnt_count) {
		sbp->s_max_mnt_count = cpu_to_le16(NILFS_DFL_MAX_MNT_COUNT);
	}
	sbp->s_mnt_count = cpu_to_le16(mnt_count + 1);
	sbp->s_state = cpu_to_le16(le16_to_cpu(sbp->s_state) & ~NILFS_VALID_FS);
	sbp->s_mtime = cpu_to_le64(get_seconds());
	nilfs_debug(2, "setting valid to 0\n");
	nilfs_commit_super(sb, sbp, 1);

	return 0;
}

struct address_space_operations def_s_aops = {
	.writepage		= nilfs_write_inode_page,
	.releasepage		= nilfs_release_inode_page,
};

struct nilfs_super_block *
nilfs_load_super_block(struct super_block *sb, struct buffer_head **pbh)
{
	int blocksize;
	unsigned long offset, sb_index;

	/*
	 * Adjusting block size
	 * Blocksize will be enlarged when it is smaller than hardware
	 * sector size.
	 * Disk format of superblock does not change. 
	 */
	blocksize = sb_min_blocksize(sb, BLOCK_SIZE);
	if (!blocksize) {
		printk("NILFS: unable to set blocksize of superblock\n");
		return NULL;
	}
	sb_index = NILFS_SB_OFFSET_BYTES / blocksize;
	offset = NILFS_SB_OFFSET_BYTES % blocksize;

	*pbh = nilfs_bread(sb, sb_index);
	if(!*pbh) {
		printk("NILFS: unable to read superblock\n");
		return NULL;
	}
	return (struct nilfs_super_block *)((char *)(*pbh)->b_data + offset);
}

struct nilfs_super_block *
nilfs_reload_super_block(struct super_block *sb, struct buffer_head **pbh,
			 int blocksize)
{
	struct nilfs_super_block *sbp;
	unsigned long offset, sb_index;
	int hw_blocksize = bdev_hardsect_size(sb->s_bdev);

	if (blocksize < hw_blocksize) {
		printk(KERN_ERR
		       "NILFS: blocksize %d too small for device "
		       "(sector-size = %d).\n",
		       blocksize, hw_blocksize);
		goto failed_sbh;
	}
	nilfs_brelse(*pbh);
	sb_set_blocksize(sb, blocksize);
	
	sb_index = NILFS_SB_OFFSET_BYTES / blocksize;
	offset = NILFS_SB_OFFSET_BYTES % blocksize;
	
	*pbh = nilfs_bread(sb, sb_index);
	if (!*pbh) {
		printk(KERN_ERR
		       "NILFS: cannot read superblock on 2nd try.\n");
		goto failed;
	}

	sbp = (struct nilfs_super_block *)((char *)(*pbh)->b_data + offset);
	if (sbp->s_magic != cpu_to_le16(NILFS_SUPER_MAGIC)) {
		printk(KERN_ERR
		       "NILFS: !? Magic mismatch on 2nd try.\n");
		goto failed_sbh;
	}
	return sbp;

 failed_sbh:
	nilfs_brelse(*pbh);

 failed:
	return NULL;
}

int nilfs_store_magic_and_option(struct super_block *sb,
				 struct nilfs_super_block *sbp,
				 char *data)
{
	struct nilfs_sb_info *sbi = NILFS_SB(sb);

	/* trying to fill super (1st stage) */
	sb->s_magic = le16_to_cpu(sbp->s_magic);

	/* FS independent flags */
#ifdef NILFS_ATIME_DISABLE
	sb->s_flags |= MS_NOATIME;
#endif
#if NEED_MS_ONE_SECOND
	sb->s_flags |= MS_ONE_SECOND;
#endif

	if (sb->s_magic != NILFS_SUPER_MAGIC) {
		printk("NILFS: Can't find nilfs on dev %s.\n", sb->s_id);
		return -EINVAL;
	}

	set_default_options(sb, sbp);

	sbi->s_resuid = le16_to_cpu(sbp->s_def_resuid);
	sbi->s_resgid = le16_to_cpu(sbp->s_def_resgid);
	sbi->s_segctor.interval = le32_to_cpu(sbp->s_c_interval);
	sbi->s_segctor.block_max = le32_to_cpu(sbp->s_c_block_max);

	if (!parse_options(data, sb))
		return -EINVAL;

	return 0;
}

int nilfs_store_disk_layout(struct super_block *sb,
			    struct nilfs_super_block *sbp)
{
	struct nilfs_sb_info *sbi = NILFS_SB(sb);

	if (le32_to_cpu(sbp->s_rev_level) != NILFS_CURRENT_REV ||
	    le16_to_cpu(sbp->s_minor_rev_level) != NILFS_MINOR_REV) {
		printk(KERN_ERR "NILFS: revision mismatch "
		       "(superblock rev.=%d.%d, current rev.=%d.%d). "
		       "Please check newfs version.\n",
		       le32_to_cpu(sbp->s_rev_level),
		       le16_to_cpu(sbp->s_minor_rev_level),
		       NILFS_CURRENT_REV, NILFS_MINOR_REV);
		return -EINVAL;
	}
	sb->s_maxbytes = nilfs_max_size(sb->s_blocksize_bits);

	sbi->s_inode_size = le16_to_cpu(sbp->s_inode_size);
	sbi->s_first_ino = le32_to_cpu(sbp->s_first_ino);
	if (sbi->s_inode_size != sizeof(struct nilfs_inode) ||
	    sbi->s_inode_size > sb->s_blocksize - sizeof(struct nilfs_inode_hdr)) {
		printk("NILFS: unsupported inode size: %d\n", sbi->s_inode_size);
		return -EINVAL;
	}

	sbi->s_inodes_per_block = (sb->s_blocksize - sizeof(struct nilfs_inode_hdr)) / sbi->s_inode_size;
	sbi->s_blocks_per_segment = le32_to_cpu(sbp->s_blocks_per_segment);
	if (sbi->s_blocks_per_segment < NILFS_SEG_MIN_BLOCKS) {
		printk(KERN_ERR "NILFS: too short segment. \n");
		return -EINVAL;
	}

	sbi->s_first_data_block = le64_to_cpu(sbp->s_first_data_block);
	sbi->s_nsegment = le32_to_cpu(sbp->s_nsegment);
	sbi->s_crc_seed = le32_to_cpu(sbp->s_crc_seed);
	return 0;
}

static int
nilfs_fill_super(struct super_block *sb, void *data, int silent,
		 struct the_nilfs *nilfs)
{
	struct nilfs_sb_info *sbi;
	struct inode *root;
	struct backing_dev_info *bdi;
	int err, seg_stat;

	nilfs_debug(1, "start(silent=%d)\n", silent);
	sbi = kmalloc(sizeof(*sbi), GFP_KERNEL);
	if (!sbi)
		return -ENOMEM;
	bdi = kmalloc(sizeof(*bdi), GFP_KERNEL);
	if (!bdi) {
		kfree(sbi);
		return -ENOMEM;
	}

	sb->s_fs_info = sbi;
	memset(sbi, 0, sizeof(*sbi));
	memset(bdi, 0, sizeof(*bdi));

	get_nilfs(nilfs);
	sbi->s_nilfs = nilfs;
	sbi->s_super = sb;

	err = init_nilfs(nilfs, sbi, sb, (char *)data);
	if (err)
		goto failed_sbi;

	err = nilfs_init_segment(sbi, sbi->s_blocks_per_segment);
	if (err)
		goto failed_sbi;

	INIT_RADIX_TREE(&sbi->s_data.page_tree, GFP_ATOMIC);
#if NEED_RWLOCK_FOR_PAGECACHE_LOCK
	rwlock_init(&sbi->s_data.tree_lock);
#else
	spin_lock_init(&sbi->s_data.tree_lock);
#endif
	spin_lock_init(&sbi->s_data.i_mmap_lock);
#if NEED_ATOMIC_COUNTER_FOR_TRUNCATE_COUNT
	atomic_set(&sbi->s_data.truncate_count, 0);
#else
	sbi->s_data.truncate_count = 0;
#endif
	INIT_LIST_HEAD(&sbi->s_data.private_list);
	spin_lock_init(&sbi->s_data.private_lock);
	INIT_PRIO_TREE_ROOT(&sbi->s_data.i_mmap);
	INIT_LIST_HEAD(&sbi->s_data.i_mmap_nonlinear);
	INIT_LIST_HEAD(&sbi->s_list);
	init_rwsem(&sbi->s_inode_root_sem);
	INIT_RADIX_TREE_64(&sbi->s_inode_ntree, GFP_ATOMIC);
	spin_lock_init(&sbi->s_inode_ntree_lock);
	INIT_LIST_HEAD(&sbi->s_free_inodes);
	init_MUTEX(&sbi->s_free_inodes_sem);
	init_MUTEX(&sbi->s_node_pages_sem);
	INIT_LIST_HEAD(&sbi->s_partial_node_pages);
	INIT_LIST_HEAD(&sbi->s_inactive_node_pages);
	spin_lock_init(&sbi->s_inactive_node_pages_lock);

	sbi->s_mapping = &sbi->s_data;
	sbi->s_mapping->backing_dev_info = bdi;
	sbi->s_mapping->a_ops = &def_s_aops;

	/*
	 * Following initialization is overlapped because
	 * nilfs_sb_info structure has been cleared at the beginning.
	 * But we reserve them to keep our interest and make ready
	 * for the future change.
	 */
	sbi->s_mapping->host = NULL;
	sbi->s_mapping->assoc_mapping = NULL;
	sbi->s_free_inodes_len = 0;
	sbi->s_free_inodes_count = 0;
	sbi->s_inactive_node_pages_cnt = 0;

	get_random_bytes(&sbi->s_next_generation, sizeof(u32));
	spin_lock_init(&sbi->s_next_gen_lock);

	sb->s_op = &nilfs_sops;
	sb->s_export_op = NULL; /* should be set to &nilfs_export_ops */

	/*
	 * Code checking last segment and finding inode root from the last
	 * valid checkpoint should be inserted here.
	 */
	if (nilfs_test_opt(sbi, SNAPSHOT)) {
		seg_stat = nilfs_load_snapshot(sbi);
		if (seg_stat) {
			err = nilfs_warn_segment_error(sbi, seg_stat);
			printk(KERN_ERR
			       "NILFS: error loading snapshot.\n");
			goto failed_segctor;
		}
	} else {
		seg_stat = nilfs_load_last_segment(sbi, nilfs);
		if (seg_stat < 0) {
			err = seg_stat;
			goto failed_segctor;
		}
		if (sb->s_flags & MS_RDONLY) {
			sbi->s_snapshot_pseg = nilfs->ns_last_pseg;
			sbi->s_snapshot_cp = nilfs->ns_last_cp;
			nilfs_set_opt(sbi, SNAPSHOT);
		}
	}
	root = iget(sb, NILFS_ROOT_INO);
	sb->s_root = d_alloc_root(root);
	if (!sb->s_root) {
		iput(root);
		printk(KERN_ERR "NILFS: get root inode failed\n");
		err = -EINVAL;
		goto failed_btree;
	}
	if (!S_ISDIR(root->i_mode) || !root->i_blocks || !root->i_size) {
		dput(sb->s_root);
		sb->s_root = NULL;
		printk(KERN_ERR "NILFS: corrupt root inode. run fsck\n");
		err = -EINVAL;
		goto failed_btree;
	}

	if (!(sb->s_flags & MS_RDONLY)) {
		down_write(&nilfs->ns_sem);
		nilfs_setup_super(sb, nilfs->ns_sbp);
		up_write(&nilfs->ns_sem);
		nilfs_attach_writer(nilfs, sbi);
	}

	nilfs_complete_recovery(sbi, seg_stat);
	nilfs_register_super(sbi);

	nilfs_debug(1, "mounted filesystem\n");
	return 0;

 failed_btree:
	nilfs_free_free_inode_list(sb);
	nilfs_btree_clear(&sbi->s_inode_root);

 failed_segctor:
	nilfs_clear_segment(sbi);

 failed_sbi:
	put_nilfs(nilfs);
	sb->s_fs_info = NULL;
	kfree(bdi);
	kfree(sbi);
	nilfs_debug(1, "aborted\n");
	return err;
}

static int nilfs_remount(struct super_block *sb, int *flags, char *data)
{
	struct nilfs_sb_info *sbi = NILFS_SB(sb);
	struct nilfs_super_block *sbp;
	struct the_nilfs *nilfs = sbi->s_nilfs;
	unsigned long old_sb_flags;
	struct nilfs_mount_options old_opts;
	int err, seg_stat;

	nilfs_debug(1, "start\n");
	old_sb_flags = sb->s_flags;
	old_opts.mount_opt = sbi->s_mount_opt;
	old_opts.snapshot_pseg = sbi->s_snapshot_pseg;

	if(!parse_options(data, sb)) {
		err = -EINVAL;
		goto restore_opts;
	}
	sb->s_flags = (sb->s_flags & ~MS_POSIXACL);

	if ((*flags & MS_RDONLY) &&
	    sbi->s_snapshot_pseg != old_opts.snapshot_pseg) {
		printk(KERN_WARNING "NILFS (device %s): couldn't "
		       "remount to a different snapshot. \n",
		       sb->s_id);
		err = -EINVAL;
		goto restore_opts;
	}

	if((*flags & MS_RDONLY) == (sb->s_flags & MS_RDONLY))
		goto out;
	if (*flags & MS_RDONLY) {
		/* Shutting down the segment constructor */
		nilfs_detach_writer(nilfs, sbi);
		nilfs_free_free_inode_list(sb);
		nilfs_clear_segment(sbi);
		sb->s_flags |= MS_RDONLY;

		sbi->s_snapshot_pseg = nilfs->ns_last_pseg;
		sbi->s_snapshot_cp = nilfs->ns_last_cp;
		nilfs_set_opt(sbi, SNAPSHOT);

		/*
		 * Remounting a valid RW partition RDONLY, so set
		 * the RDONLY flag and then mark the partition as valid again.
		 */
		down_write(&nilfs->ns_sem);
		sbp = nilfs->ns_sbp;
		if (!(sbp->s_state & le16_to_cpu(NILFS_VALID_FS)) &&
		    (nilfs->ns_mount_state & NILFS_VALID_FS))
			sbp->s_state = cpu_to_le16(nilfs->ns_mount_state);
		sbp->s_mtime = cpu_to_le64(get_seconds());
		nilfs_commit_super(sb, sbp, 1);
		up_write(&nilfs->ns_sem);
	} else {
		/*
		 * Mounting a RDONLY partition read-write, so reread and
		 * store the current valid flag.  (It may have been changed
		 * by fsck since we originally mounted the partition.)
		 */
#if NEED_MOUNT_SEMAPHORE
		down(&sb->s_bdev->bd_mount_sem);
#else
		mutex_lock(&sb->s_bdev->bd_mount_mutex);
#endif
		/* Check existing RW-mount */
		if (check_rw_mount(sb->s_type, sb->s_bdev)) {
			printk(KERN_WARNING "NILFS (device %s): couldn't "
			       "remount because a RW-mount exists.\n",
			       sb->s_id);
			err = -EBUSY;
			goto rw_remount_failed;
		}
		sb->s_flags &= ~MS_RDONLY;
		nilfs_clear_opt(sbi, SNAPSHOT);

		err = nilfs_init_segment(sbi, sbi->s_blocks_per_segment);
		if (err)
			goto rw_remount_failed;

		seg_stat = nilfs_load_last_segment_for_remount(sbi, nilfs); /* locks nilfs->ns_sem */
		if (seg_stat < 0) {
			err = seg_stat;
			nilfs_clear_segment(sbi);
			goto rw_remount_failed;
		}
		sbp = nilfs->ns_sbp;
		nilfs_setup_super(sb, sbp);
		up_write(&nilfs->ns_sem);

		nilfs_attach_writer(nilfs, sbi);
		nilfs_complete_recovery(sbi, seg_stat);

#if NEED_MOUNT_SEMAPHORE
		up(&sb->s_bdev->bd_mount_sem);
#else
		mutex_unlock(&sb->s_bdev->bd_mount_mutex);
#endif
	}
 out:
	nilfs_debug(1, "remounted filesystem\n");
	return 0;

 rw_remount_failed:
#if NEED_MOUNT_SEMAPHORE
	up(&sb->s_bdev->bd_mount_sem);
#else
	mutex_unlock(&sb->s_bdev->bd_mount_mutex);
#endif
 restore_opts:
	sb->s_flags = old_sb_flags;
	sbi->s_mount_opt = old_opts.mount_opt;
	sbi->s_snapshot_pseg = old_opts.snapshot_pseg;
	return err;
}

struct nilfs_super_data {
	struct block_device *bdev;
	dbn_t seg_block;
	int flags;
	int mode;
#define NILFS_MODE_PASSIVE      1
};

static int nilfs_identify(char *data, struct nilfs_super_data *sd)
{
	char *p, *options = data;
	substring_t args[MAX_OPT_ARGS];
	int option, token;
	int ret = 0;

	do {
		if ((p = strsep(&options, ",")) != NULL && *p) {
			token = match_token(p, tokens, args);
			if (token == Opt_snapshot) {
				if (!(sd->flags & MS_RDONLY))
					ret++;
				else if (!(ret = match_int(&args[0], &option)))
					sd->seg_block = option;
			} else if (token == Opt_passive) {
				if (match_bool(&args[0], &option) || (sd->flags & MS_RDONLY))
					ret++;
				else 
					sd->mode = (option ? NILFS_MODE_PASSIVE : 0);
			}
			if (ret)
				printk(KERN_ERR "NILFS: invalid mount option: %s\n", p);
		}
		if (!options)
			break;
		BUG_ON(options == data);
		*(options - 1) = ',';
	} while (!ret);
	return ret;
}

static int nilfs_set_bdev_super(struct super_block *s, void *data)
{
	struct nilfs_super_data *sd = data;

	s->s_bdev = sd->bdev;
	s->s_dev = s->s_bdev->bd_dev;
	return 0;
}

static int nilfs_test_bdev_super(struct super_block *s, void *data)
{
	struct nilfs_super_data *sd = data;

	return (s->s_bdev == sd->bdev);
}

static int nilfs_test_bdev_super2(struct super_block *s, void *data)
{
	struct nilfs_super_data *sd = data;
	int ret;

	if (s->s_bdev != sd->bdev)
		return 0;

	if (!((s->s_flags | sd->flags) & MS_RDONLY))
		return 1; /* Reuse the old R/W-mode super_block */

	if (s->s_flags & sd->flags & MS_RDONLY) {
		if (down_read_trylock(&s->s_umount)) {
			ret = sd->seg_block && s->s_root &&
				(sd->seg_block == NILFS_SB(s)->s_snapshot_pseg);
			up_read(&s->s_umount);
			/*
			 * This path is locked with sb_lock by sget().
			 * So, drop_super() causes deadlock.
			 */
			return ret;
		}
	}
	return 0;
}

static struct super_block *
nilfs_get_sb(struct file_system_type *fs_type, int flags,
	  const char *dev_name, void *data)
{
	struct nilfs_super_data sd;
	struct super_block *s, *s2;
	struct the_nilfs *nilfs = NULL;
	int err, need_to_close = 1;

	sd.bdev = open_bdev_excl(dev_name, flags, fs_type);
	if (IS_ERR(sd.bdev))
		return (struct super_block *)sd.bdev;

	sd.seg_block = 0;
	sd.mode = 0;
	sd.flags = flags;
	if (nilfs_identify((char *)data, &sd)) {
		close_bdev_excl(sd.bdev);
		return ERR_PTR(-EINVAL);
	}
	
	/*
	 * once the super is inserted into the list by sget, s_umount
	 * will protect the lockfs code from trying to start a snapshot
	 * while we are mounting
	 */
#if NEED_MOUNT_SEMAPHORE
	down(&sd.bdev->bd_mount_sem);
#else
	mutex_lock(&sd.bdev->bd_mount_mutex);
#endif

	/*
	 * Phase-1: search any existent instance and get the_nilfs
	 */
	s = sget(fs_type, nilfs_test_bdev_super, nilfs_set_bdev_super, &sd);
	if (IS_ERR(s))
		goto out;

	if (!s->s_root) {
		err = -ENOMEM;
		nilfs = alloc_nilfs(sd.bdev);
		if (!nilfs)
			goto cancel_new;
	} else {
		struct nilfs_sb_info *sbi = NILFS_SB(s);

		BUG_ON(!sbi || !sbi->s_nilfs);
                /*
		 * s_umount protects super_block from unmount process;
		 * It covers pointers of nilfs_sb_info and the_nilfs.
		 */
		nilfs = sbi->s_nilfs;
		get_nilfs(nilfs);
		up_write(&s->s_umount);

		/*
		 * Phase-2: search specified snapshot or R/W mode super_block
		 */
		if (!sd.seg_block) {
			/* trying to get the segment block which has
			   the latest checkpoint.  */
			spin_lock_irq(&nilfs->ns_lock);
			sd.seg_block = nilfs->ns_last_pseg;
			spin_unlock_irq(&nilfs->ns_lock);
		}
		s2 = sget(fs_type, nilfs_test_bdev_super2, nilfs_set_bdev_super, &sd);
		deactivate_super(s);
                /*
		 * Although deactivate_super() invokes close_bdev_excl() at
		 * kill_block_super().  Here, s is an existent mount; we need
		 * one more close_bdev_excl() call.
		 */
		s = s2;
		if (IS_ERR(s))
			goto out;
	}

	if (!s->s_root) {
		char b[BDEVNAME_SIZE];

		s->s_flags = flags;
		strlcpy(s->s_id, bdevname(sd.bdev, b), sizeof(s->s_id));
#if NEED_S_OLD_BLOCKSIZE
		s->s_old_blocksize = block_size(sd.bdev);
		sb_set_blocksize(s, s->s_old_blocksize);
#else
		sb_set_blocksize(s, block_size(sd.bdev));
#endif
		err = nilfs_fill_super(s, data, flags & MS_VERBOSE, nilfs);
		if (err)
			goto cancel_new;

		s->s_flags |= MS_ACTIVE;
		if (sd.bdev->bd_disk) {
			struct kobject *kobj = &sd.bdev->bd_disk->kobj;

			if (sd.bdev->bd_part)
				kobj = &sd.bdev->bd_part->kobj;

#if NEED_KOBJECT_UEVENT_ATTRIBUTE_ARG
			kobject_uevent(kobj, KOBJ_MOUNT, NULL);
#else
			kobject_uevent(kobj, KOBJ_MOUNT);
#endif
		}
		need_to_close = 0;
	} else if (!(s->s_flags & MS_RDONLY)) {
		err = -EBUSY;
		if((sd.mode != NILFS_MODE_PASSIVE) ^ !nilfs_test_opt(NILFS_SB(s), PASSIVE)) {
			printk(KERN_ERR "NILFS: A passive mode mount and a normal rw-mount "
			       "are done at at time; they are exclusive.\n");
			goto cancel_new;  /* Mode is different */
		}
	}
 out:
#if NEED_MOUNT_SEMAPHORE
	up(&sd.bdev->bd_mount_sem);
#else
	mutex_unlock(&sd.bdev->bd_mount_mutex);
#endif
        /*
	 * This unlocking is delayed until the_nilfs is attached to 
	 * nilfs_sb_info. This ensures that the_nilfs is identical for 
	 * the same device in the list of fs_supers.
	 */
	if (nilfs)
		put_nilfs(nilfs);
	if (need_to_close)
		close_bdev_excl(sd.bdev);
	return s;

 cancel_new:
	/* Abandoning the newly allocated superblock */
#if NEED_MOUNT_SEMAPHORE
	up(&sd.bdev->bd_mount_sem); 
#else
	mutex_unlock(&sd.bdev->bd_mount_mutex);
#endif
	if (nilfs)
		put_nilfs(nilfs);
	up_write(&s->s_umount);
	deactivate_super(s);
	/*
	 * deactivate_super() invokes close_bdev_excl().
	 * We must finish all clean-up processing before this call;
	 * put_nilfs() and unlocking of bd_mount_sem depend on the block device.
	 */
	return ERR_PTR(err);
}

static int __false_bdev_super(struct super_block *s, void *data)
{
	return -EFAULT;
}

/**
 * check_rw_mount - check whether a RW mount exists or not.
 * fs_type: filesystem type
 * bdev: block device
 *
 * This function must be called within a section protected by bd_mount_mutex.
 */
static int check_rw_mount(struct file_system_type *fs_type, struct block_device *bdev)
{
	struct super_block *s;
	struct nilfs_super_data sd = { .flags = 0, .bdev = bdev };

	s = sget(fs_type, nilfs_test_bdev_super2, __false_bdev_super, &sd);
	if (IS_ERR(s))
		return 0; /* Not found */
	deactivate_super(s);
	return 1;  /* Found */
}

struct file_system_type nilfs_fs_type = {
	.owner    = THIS_MODULE,
	.name     = "nilfs",
	.get_sb   = nilfs_get_sb,
	.kill_sb  = kill_block_super,
	.fs_flags = FS_REQUIRES_DEV,
};

/*
 * nilfs_shrink_func()
 *  shrinker function
 */

#ifndef NILFS_SHRINKER_DISABLE
static int nilfs_shrink_func(int nr, GFP_T gfp_mask)
{
	struct super_block *sb;
	struct list_head *p, *n;
	int num = 0, ret = 0, negative = 0;

	shrink_debug(2, "called %d %u\n", nr, gfp_mask);
#if 0 /* just a example */
	if (nr && !(gfp_mask & __GFP_FS)) {
		shrink_debug(2, "return -1 for mask 0x%x\n", gfp_mask);
		return -1; 
	}
#endif
	/* shrink inode pages and inactive node pages for each super block */
	spin_lock(&nilfs_sb_lock);
	list_for_each_safe(p, n, &nilfs_super_blocks) {
		sb = nilfs_sb_entry(p)->s_super;
		if (sb->s_type == &nilfs_fs_type) {
			if (down_read_trylock(&sb->s_umount)) {
				sb->s_count++;
				spin_unlock(&nilfs_sb_lock);
				shrink_debug(3, "found and get lock %p\n", sb);
				if (sb->s_root) {
					ret = nilfs_shrink_inactive_node_pages(sb, nr);
					if (ret == -1) {
						negative = 1;
					} else {
						num += ret;
					}
				}
				drop_super(sb);
				spin_lock(&nilfs_sb_lock);
			} else {
				shrink_debug(3, "trylock failed %p\n", sb);
				negative = 1;
			}
		}
	}
	spin_unlock(&nilfs_sb_lock);
	if (negative) {
		shrink_debug(1, "ret -1, num %d\n", num);
		return -1;
	} else {
		shrink_debug(1, "num %d\n", num);
		/* return (num / 100) * 100; */
		return num;
	}
}

static inline int nilfs_set_shrinker(void)
{
	nilfs_shrinker = set_shrinker(DEFAULT_SEEKS, nilfs_shrink_func);
	return !nilfs_shrinker;
}

static inline void nilfs_remove_shrinker(void)
{
	remove_shrinker(nilfs_shrinker);
}
#endif /* !NILFS_SHRINKER_DISABLE */


static int __init init_nilfs_fs(void)
{
	int err;

	nilfs_init_counters();
	nilfs_init_debug_info();
	radix_tree_64_init();

	err = init_inodecache();
	if (err)
		goto failed;

	err = nilfs_init_transaction_cache();
	if (err)
		goto failed_inode_cache;

#ifndef NILFS_SHRINKER_DISABLE
	err = nilfs_set_shrinker();
	if (err)
		goto failed_transaction_cache;
#endif
	err = nilfs_init_proc_entries();
	if (err)
		goto failed_set_shrink;

	err = nilfs_btree_path_cache_init();
	if (err) {
		err = nilfs_convert_btree_error(err);
		goto failed_proc_entries;
	}

	err = register_filesystem(&nilfs_fs_type);
	if (err)
		goto failed_btree_path_cache;

	return 0;

 failed_btree_path_cache:
	nilfs_btree_path_cache_destroy();

 failed_proc_entries:
	nilfs_remove_proc_entries();

 failed_set_shrink:
#ifndef NILFS_SHRINKER_DISABLE
	nilfs_remove_shrinker();
	
 failed_transaction_cache:
#endif
	nilfs_destroy_transaction_cache();
	
 failed_inode_cache:
	destroy_inodecache();

 failed:
	radix_tree_64_destroy();
	return err;
}

static void __exit exit_nilfs_fs(void)
{
	nilfs_remove_proc_entries();
#ifndef NILFS_SHRINKER_DISABLE
	nilfs_remove_shrinker();
#endif
	nilfs_destroy_transaction_cache();
	destroy_inodecache();
	radix_tree_64_destroy();
	nilfs_btree_path_cache_destroy();
	unregister_filesystem(&nilfs_fs_type);
}

module_init(init_nilfs_fs)
module_exit(exit_nilfs_fs)

/* Local Variables:	*/
/* eval: (c-set-style "linux")	*/
/* End:			*/
