/*
 * nilfs_btree.h - NILFS B-tree operations.
 *
 * Copyright (C) 2005 Nippon Telegraph and Telephone Corporation.
 *
 * This file is part of NILFS.
 *
 * NILFS is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * NILFS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with NILFS; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * nilfs_btree.h,v 1.1 2005/09/15 09:25:09 koji Exp
 *
 * Written by Koji Sato <koji@osrg.net>
 */

#ifndef _LINUX_NILFS_BTREE_H
#define _LINUX_NILFS_BTREE_H

/**
 * nilfs_btree_node - B-tree node
 * @bn_level: level of node
 * @bn_nchildren: number of children
 * @bn_pad: padding
 * @bn_next: pointer to next node on list (only used for on-memory node)
 *
 * The size of a B-tree node is the logical block size of the file
 * system.
 */
struct nilfs_btree_node {
	__le16 bn_level;
	__le16 bn_nchildren;
	__le32 bn_pad;
	__le64 bn_next;
};

/* B-tree key and pointer on disk */
typedef __le64 nilfs_btree_dkey_t;
typedef __le64 nilfs_btree_dptr_t;

#define NILFS_BTREE_NODE_MAX_NCHILDREN(blksize)			\
	(((blksize) - sizeof(struct nilfs_btree_node)) /	\
	 (sizeof(nilfs_btree_dkey_t) + sizeof(nilfs_btree_dptr_t)))
#define NILFS_BTREE_NODE_MIN_NCHILDREN(blksize)			\
	((NILFS_BTREE_NODE_MAX_NCHILDREN(blksize) - 1) / 2 + 1)
#define NILFS_BTREE_NODE_DKEYS(node, blksize)	\
	((nilfs_btree_dkey_t *)((char *)(node) + sizeof(struct nilfs_btree_node)))
#define NILFS_BTREE_NODE_DPTRS(node, blksize)				\
	((nilfs_btree_dptr_t *)((char *)(node) +			\
			  sizeof(struct nilfs_btree_node) +		\
			  NILFS_BTREE_NODE_MAX_NCHILDREN(blksize) *	\
				sizeof(nilfs_btree_dkey_t)))

#endif	/* _LINUX_NILFS_BTREE_H */

/* Local Variables:	*/
/* eval: (c-set-style "linux")	*/
/* End:			*/
