/*
 *  Copyright (C) 2005 Kouji TAKAO <kouji@netlab.jp>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <glib/gi18n.h>
#include <gtk/gtk.h>
#include <gnome.h>

#include "gpass/configuration.h"
#include "gpass/error.h"
#include "application.h"
#include "summary.h"

static void
set_gpass_description(GtkTextView *summary)
{
    GtkTextBuffer *buffer;
    GtkTextIter iter;

    buffer = gtk_text_view_get_buffer(summary);
    gtk_text_buffer_set_text(buffer, "", 0);
    gtk_text_buffer_get_start_iter(buffer, &iter);
    gtk_text_buffer_insert(buffer, &iter, "\n", -1);
    gtk_text_buffer_insert_with_tags_by_name
        (buffer, &iter, "GPass " VERSION "\n", -1, "name", NULL);
    gtk_text_buffer_insert(buffer, &iter, "\n", -1);
    gtk_text_buffer_insert_with_tags_by_name
        (buffer, &iter,
         _("A simple password manager for GNOME 2\n"),
         -1, "description", NULL);
    gtk_text_buffer_insert(buffer, &iter, "\n\n", -1);
    gtk_text_buffer_insert_with_tags_by_name
        (buffer, &iter,
         _("GPass is a simple GNOME application, "
           "released under the GNU GPL licence, "
           "that lets you manage a collection of passwords.\n"
           "\n"
           " The password collection is stored in an encrypted file, "
           "protected by a master-password.\n"),
         -1, "program description", NULL);
}

void
gpass_summary_initialize(GPassWindow *window)
{
    GtkTextView *summary;
    GtkTextBuffer *buffer;

    summary = GTK_TEXT_VIEW
        (glade_xml_get_widget(GPASS_VIEW(window)->xml, "summary"));
    buffer = gtk_text_view_get_buffer(summary);
    gtk_text_buffer_create_tag(buffer, "icon",
                               "justification", GTK_JUSTIFY_CENTER,
                               NULL);
    gtk_text_buffer_create_tag(buffer, "name",
                               "weight", 500, "scale", 1.5,
                               "justification", GTK_JUSTIFY_CENTER,
                               NULL);
    gtk_text_buffer_create_tag(buffer, "description",
                               "justification", GTK_JUSTIFY_CENTER,
                               NULL);
    gtk_text_buffer_create_tag(buffer, "program description",
                               "scale", 0.8, NULL);
    gtk_text_buffer_create_tag(buffer, "property name",
                               "weight", 500,
                               "justification", GTK_JUSTIFY_CENTER,
                               "pixels_below_lines", 2,
                               NULL);
    gtk_text_buffer_create_tag(buffer, "property value",
                               "justification", GTK_JUSTIFY_CENTER,
                               "pixels_below_lines", 2,
                               NULL);
    set_gpass_description(summary);
}

static const gchar *
get_string(GPassAttributeList *attributes, const gchar *name)
{
    GPassAttribute *attr = gpass_attribute_list_lookup(attributes, name);
    const gchar *str;

    if (attr == NULL) {
        return NULL;
    }
    gpass_attribute_get(attr, &str);
    return str;
}

static void
write_attribute(GtkTextBuffer *buffer, GtkTextIter *iter,
                GPassAttribute *attribute)
{
    gchar *str;
    gint i;
    gboolean b;
    time_t t;
    struct tm tm;
    gchar t_str[1024];
    size_t result;
    GPassConfiguration *config;
    gboolean visible_secrets;
    
    gtk_text_buffer_insert_with_tags_by_name
        (buffer, iter, attribute->nick, -1, "property name", NULL);
    gtk_text_buffer_insert_with_tags_by_name
        (buffer, iter, ": ", -1, "property name", NULL);
    switch (attribute->type) {
    case GPASS_ATTRIBUTE_TYPE_INTEGER:
        gpass_attribute_get(attribute, &i);
        str = g_strdup_printf("%d", i);
        gtk_text_buffer_insert_with_tags_by_name
            (buffer, iter, str, -1, "property value", NULL);
        g_free(str);
        break;
    case GPASS_ATTRIBUTE_TYPE_BOOLEAN:
        gpass_attribute_get(attribute, &b);
        str = b ? _("Yes") : _("No");
        gtk_text_buffer_insert_with_tags_by_name
            (buffer, iter, str, -1, "property value", NULL);
        break;
    case GPASS_ATTRIBUTE_TYPE_TIME:
        gpass_attribute_get(attribute, &t);
        if (t == 0) {
            gtk_text_buffer_insert_with_tags_by_name
                (buffer, iter, "-", -1, "property value", NULL);
            break;
        }
        localtime_r(&t, &tm);
        result = strftime(t_str, 1024, "%F %H:%M:%S", &tm);
        if (result == 0) {
            gtk_text_buffer_insert_with_tags_by_name
                (buffer, iter, t_str, -1, "property value", NULL);
            break;
        }
        t_str[result] = '\0';
        gtk_text_buffer_insert_with_tags_by_name
            (buffer, iter, t_str, -1, "property value", NULL);
        break;
    case GPASS_ATTRIBUTE_TYPE_STRING:
    case GPASS_ATTRIBUTE_TYPE_TEXT:
    case GPASS_ATTRIBUTE_TYPE_URL:
    case GPASS_ATTRIBUTE_TYPE_ENTRY_TYPE:
        gpass_attribute_get(attribute, &str);
        gtk_text_buffer_insert_with_tags_by_name
            (buffer, iter, str, -1, "property value", NULL);
        break;
    case GPASS_ATTRIBUTE_TYPE_PASSWORD:
        config = gpass_configuration_instance();
        g_object_get(config, "visible_secrets", &visible_secrets, NULL);
        gpass_attribute_get(attribute, &str);
        if (visible_secrets) {
            gtk_text_buffer_insert_with_tags_by_name
                (buffer, iter, str, -1, "property value", NULL);
        }
        else {
            gchar *buf = g_strnfill(strlen(str), '*');
            
            gtk_text_buffer_insert_with_tags_by_name
                (buffer, iter, buf, -1, "property value", NULL);
            g_free(buf);
        }
        break;
    default:
        gtk_text_buffer_insert_with_tags_by_name
            (buffer, iter, "-", -1, "property value", NULL);
        break;
    }
    gtk_text_buffer_insert(buffer, iter, "\n", -1);
}

static void
write_summary(GtkTextView *summary, GPassEntry *entry,
              GPassAttributeList *attributes)
{
    GtkTextBuffer *buffer;
    GtkTextIter iter, start_iter;
    GtkTextMark *mark;
    const gchar *icon_id;
    GdkPixbuf *pixbuf;
    const gchar *string;
    GPassAttributeListCursor *cursor;
    GPassAttribute *attr, *mtime_attr, *etime_attr;
    
    buffer = gtk_text_view_get_buffer(summary);
    gtk_text_buffer_set_text(buffer, "", 0);
    gtk_text_buffer_get_start_iter(buffer, &iter);

    mark = gtk_text_buffer_create_mark(buffer, "icon", &iter, TRUE);
    icon_id = gpass_entry_class_icon_id
        (GPASS_ENTRY_GET_CLASS(entry));
    pixbuf = gtk_widget_render_icon(GTK_WIDGET(summary), icon_id,
                                    GTK_ICON_SIZE_DIALOG, NULL);
    gtk_text_buffer_insert_pixbuf(buffer, &iter, pixbuf);
    gtk_text_buffer_get_iter_at_mark(buffer, &start_iter, mark);
    gtk_text_buffer_apply_tag_by_name(buffer, "icon", &start_iter, &iter);
    gtk_text_buffer_delete_mark(buffer, mark);
    gtk_text_buffer_insert(buffer, &iter, "\n", -1);

    string = get_string(attributes, "name");
    if (string == NULL || *string == '\0') {
        string = _("<no name>");
    }
    gtk_text_buffer_insert_with_tags_by_name
        (buffer, &iter, string, -1, "name", NULL);
    gtk_text_buffer_insert(buffer, &iter, "\n\n", -1);
    gpass_attribute_list_remove(attributes, "name");
    
    string = get_string(attributes, "description");
    if (string == NULL || *string == '\0') {
        string = _("<no description>");
    }
    gtk_text_buffer_insert_with_tags_by_name
        (buffer, &iter, string, -1, "description", NULL);
    gtk_text_buffer_insert(buffer, &iter, "\n\n", -1);
    gpass_attribute_list_remove(attributes, "description");

    mtime_attr = gpass_attribute_list_lookup(attributes, "modification-time");
    if (mtime_attr != NULL) {
        g_object_ref(mtime_attr);
    }
    etime_attr = NULL;
    attr = gpass_attribute_list_lookup(attributes, "expiration");
    if (attr != NULL) {
        gboolean expiration;
        
        gpass_attribute_get(attr, &expiration);
        if (expiration) {
            etime_attr = gpass_attribute_list_lookup(attributes,
                                                     "expiration-time");
            if (etime_attr != NULL) {
                g_object_ref(etime_attr);
            }
        }
    }
    gpass_attribute_list_remove(attributes, "creation-time");
    gpass_attribute_list_remove(attributes, "modification-time");
    gpass_attribute_list_remove(attributes, "expiration");
    gpass_attribute_list_remove(attributes, "expiration-time");

    cursor = gpass_attribute_list_create_cursor(attributes);
    while (!gpass_attribute_list_cursor_is_done(cursor)) {
        GPassAttribute *attr;
        GError *error;
        
        g_object_get(cursor, "attribute", &attr, NULL);
        write_attribute(buffer, &iter, attr);
        error = gpass_attribute_list_cursor_next(cursor);
        if (error != NULL) {
            gpass_error_show_and_exit(error);
        }
    }
    g_object_unref(cursor);

    if (mtime_attr != NULL) {
        write_attribute(buffer, &iter, mtime_attr);
        g_object_unref(mtime_attr);
    }
    if (etime_attr != NULL) {
        write_attribute(buffer, &iter, etime_attr);
        g_object_unref(etime_attr);
    }
}

void
gpass_summary_update(GPassWindow *window)
{
    GtkTextView *summary;

    summary = GTK_TEXT_VIEW
        (glade_xml_get_widget(GPASS_VIEW(window)->xml, "summary"));
    if (window->current == NULL) {
        set_gpass_description(summary);
    }
    else {
        GPassEntry *entry = window->current;
        GPassAttributeList *attributes =
            gpass_entry_class_attributes(GPASS_ENTRY_GET_CLASS(entry));
        
        gpass_entry_get_attributes(entry, attributes);
        write_summary(summary, entry, attributes);
        g_object_unref(attributes);
    }
}
