/*
 *  Notes - panel plugin for Xfce Desktop Environment
 *  Copyright (C) 2003  Jakob Henriksson <b0kaj+dev@lysator.liu.se>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "notes.h"

#define MARGIN 4

gboolean title_changing = FALSE;

/* definied in notes_applet.c */
extern _note_applet *notes_applet;

gboolean
cb_note_delete (void)
{
   /* prevent Alt-F4 from closing note */
   return TRUE;
}

gboolean
cb_title_change(GtkWidget *widget, GdkEventButton *event, gpointer data)
{
    _note *note = (_note *)data;
    gchar *title_text;

    if (event->type == GDK_2BUTTON_PRESS) {
	
	/* get text from label */
	title_text = 
	  (gchar *)gtk_label_get_text(GTK_LABEL(note->label_title));
	/* remove label from container */
	gtk_widget_hide(note->label_title);
	gtk_container_remove(GTK_CONTAINER(widget), note->label_title);
	/* update the entry with the text */
	gtk_entry_set_text(GTK_ENTRY(note->entry_title), title_text);
	/* add entry to container instead */
	gtk_container_add(GTK_CONTAINER(widget), note->entry_title);
	gtk_widget_show(note->entry_title);
	gtk_widget_grab_focus(note->entry_title);

	title_changing = TRUE;
    }
    
    return FALSE;
}

void
on_title_change_done(_note *note)
{
    /* reverse the effect of on_title_change */

    gtk_label_set_text(GTK_LABEL(note->label_title),
		       gtk_editable_get_chars(GTK_EDITABLE(note->entry_title),
					      0, -1));
    /* remove entry */
    gtk_widget_hide(note->entry_title);
    gtk_container_remove(GTK_CONTAINER(note->event_box_top_middle),
			 note->entry_title);
    /* add the label again */
    gtk_container_add(GTK_CONTAINER(note->event_box_top_middle), 
		      note->label_title);
    gtk_widget_show(note->label_title);
    
    title_changing = FALSE;

    return;
}

gboolean
cb_title_key_press(GtkWidget *widget, GdkEventKey *event, gpointer data)
{
    if (event->type == GDK_KEY_PRESS) {
	if ((event->keyval == GDK_Return) || (event->keyval == GDK_KP_Enter)) {
	    on_title_change_done((_note *)data);
	}
    }
    return FALSE;
}

gboolean
cb_text_view_focus_in(GtkWidget *widget, GdkEventFocus *ev, gpointer data)
{
    /* update title if we were editing it */
    if (title_changing == TRUE) {
	on_title_change_done((_note *)data);
    }

    return FALSE;
}

gboolean
cb_move_window(GtkWidget *widget, GdkEventButton *event, gpointer data)
{
    GtkWidget *window = (GtkWidget *)data;

    if (event->type == GDK_BUTTON_PRESS) {
	if (event->button == 1) {
	    gtk_window_begin_move_drag(GTK_WINDOW(window),
				       event->button, 
				       event->x_root, event->y_root, 
				       event->time);
	}
    } 

    return FALSE;
}


gboolean
cb_resize_window(GtkWidget *widget, GdkEventButton *event, gpointer data)
{
  GtkWidget *window = gtk_widget_get_toplevel(widget);
  if (GTK_WIDGET_TOPLEVEL(window)) {

    if (event->type == GDK_BUTTON_PRESS) {
      if (event->button == 1) {
	gtk_window_begin_resize_drag(GTK_WINDOW(window),
				     (GdkWindowEdge)data,
				     event->button, 
				     event->x_root, event->y_root, 
				     event->time);
      }
    }
  }
  
    return FALSE;
}

_note*
notes_create_note (void)
{

  _note *note;
  
  /* create or note structure */
  note = g_new(_note, 1);

  GtkWidget *main_w;
  GtkWidget *table;
  GtkWidget *text_view;
  GtkWidget *event_box_top_middle;
  GtkWidget *label_title;
  GtkWidget *entry_title;
  GtkWidget *event_box_middle_left;
  GtkWidget *event_box_bottom_left;
  GtkWidget *event_box_middle_right;
  GtkWidget *event_box_bottom_middle;
  GtkWidget *event_box_top_left;
  GtkWidget *event_box_top_right;
  GtkWidget *event_box_bottom_right;
  GtkWidget *button_close;

  GtkWidget *hbox;

  GtkWidget *button_close_image;

  main_w = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title (GTK_WINDOW (main_w), _("main_w"));
  gtk_window_set_default_size (GTK_WINDOW (main_w), 200, 200);

  /* hints */
  gtk_window_set_decorated(GTK_WINDOW(main_w), FALSE);
  gtk_window_set_skip_pager_hint(GTK_WINDOW(main_w), TRUE);
  gtk_window_set_skip_taskbar_hint(GTK_WINDOW(main_w), TRUE);

  /* keep notes on top */
  gtk_window_set_keep_above(GTK_WINDOW(main_w), TRUE);

  /* prevent Alt-F4 from closing note */
  g_signal_connect (main_w, "delete-event", G_CALLBACK (cb_note_delete), 
		    NULL);

  table = gtk_table_new (3, 3, FALSE);
  gtk_widget_show (table);
  gtk_container_add (GTK_CONTAINER (main_w), table);

  text_view = gtk_text_view_new ();
  gtk_widget_show (text_view);
  gtk_table_attach (GTK_TABLE (table), text_view, 1, 2, 1, 2,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), 0, 0);
  gtk_text_view_set_wrap_mode (GTK_TEXT_VIEW (text_view), GTK_WRAP_WORD);
  gtk_text_view_set_left_margin (GTK_TEXT_VIEW (text_view), 5);
  gtk_text_view_set_right_margin (GTK_TEXT_VIEW (text_view), 5);

  /* hbox to hold close button and title entry */
  hbox = gtk_hbox_new (FALSE, 0);
  gtk_widget_show (hbox);
  gtk_table_attach (GTK_TABLE (table), hbox, 1, 2, 0, 1,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_FILL), 0, 0);

  event_box_top_middle = gtk_event_box_new ();
  gtk_widget_show (event_box_top_middle);
  gtk_box_pack_start (GTK_BOX (hbox), event_box_top_middle, TRUE, TRUE, 0);
  /* set a tooltip */
  gtk_tooltips_set_tip(notes_applet->tooltips, event_box_top_middle,
		       "Double-click to edit title", NULL);
  gtk_widget_realize(event_box_top_middle);
  /* cursor - move */
  gdk_window_set_cursor(event_box_top_middle->window,
			gdk_cursor_new(GDK_FLEUR));
  

  /* label */
  label_title = gtk_label_new (_("untitled note"));
  g_object_ref(label_title);
  gtk_widget_show (label_title);
  gtk_container_add (GTK_CONTAINER (event_box_top_middle), label_title);

  button_close = gtk_button_new ();
  gtk_widget_show (button_close);
  gtk_box_pack_start (GTK_BOX (hbox), button_close, FALSE, FALSE, 0);
  gtk_widget_set_size_request (button_close, 24, 20);
  gtk_button_set_relief (GTK_BUTTON (button_close), GTK_RELIEF_NONE);

  button_close_image = gtk_image_new_from_stock ("gtk-close", 
						 GTK_ICON_SIZE_BUTTON);
  gtk_widget_show (button_close_image);
  gtk_container_add (GTK_CONTAINER (button_close), button_close_image);

  entry_title = gtk_entry_new ();
  /* need to save title entry even when removed from a container */
  g_object_ref(entry_title);

  gtk_entry_set_has_frame (GTK_ENTRY (entry_title), FALSE);

  event_box_middle_left = gtk_event_box_new ();
  gtk_widget_show (event_box_middle_left);
  gtk_table_attach (GTK_TABLE (table), event_box_middle_left, 0, 1, 1, 2,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), 0, 0);
  gtk_widget_set_size_request (event_box_middle_left, 8, -1);
  gtk_widget_realize(event_box_middle_left);
  /* cursor - move */
  gdk_window_set_cursor(event_box_middle_left->window,
			gdk_cursor_new(GDK_FLEUR));

  event_box_bottom_left = gtk_event_box_new ();
  gtk_widget_show (event_box_bottom_left);
  gtk_table_attach (GTK_TABLE (table), event_box_bottom_left, 0, 1, 2, 3,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_FILL), 0, 0);
  gtk_widget_set_size_request (event_box_bottom_left, 8, 8);
  gtk_widget_realize(event_box_bottom_left);
  /* cursor - resize */
  gdk_window_set_cursor(event_box_bottom_left->window, 
			gdk_cursor_new(GDK_BOTTOM_LEFT_CORNER));

  event_box_middle_right = gtk_event_box_new ();
  gtk_widget_show (event_box_middle_right);
  gtk_table_attach (GTK_TABLE (table), event_box_middle_right, 2, 3, 1, 2,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), 0, 0);
  gtk_widget_set_size_request (event_box_middle_right, 8, -1);
  gtk_widget_realize(event_box_middle_right);
  /* cursor - move */
  gdk_window_set_cursor(event_box_middle_right->window,
			gdk_cursor_new(GDK_FLEUR));

  event_box_bottom_middle = gtk_event_box_new ();
  gtk_widget_show (event_box_bottom_middle);
  gtk_table_attach (GTK_TABLE (table), event_box_bottom_middle, 1, 2, 2, 3,
                    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
                    (GtkAttachOptions) (GTK_FILL), 0, 0);
  gtk_widget_set_size_request (event_box_bottom_middle, -1, 8);
  gtk_widget_realize(event_box_bottom_middle);
  /* cursor - move */
  gdk_window_set_cursor(event_box_bottom_middle->window,
			gdk_cursor_new(GDK_FLEUR));

  event_box_top_left = gtk_event_box_new ();
  gtk_widget_show (event_box_top_left);
  gtk_table_attach (GTK_TABLE (table), event_box_top_left, 0, 1, 0, 1,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_FILL), 0, 0);
  gtk_widget_set_size_request (event_box_top_left, 8, 8);
  gtk_widget_realize(event_box_top_left);
  /* cursor - resize */
  gdk_window_set_cursor(event_box_top_left->window, 
			gdk_cursor_new(GDK_TOP_LEFT_CORNER));

  event_box_top_right = gtk_event_box_new ();
  gtk_widget_show (event_box_top_right);
  gtk_table_attach (GTK_TABLE (table), event_box_top_right, 2, 3, 0, 1,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_FILL), 0, 0);
  gtk_widget_set_size_request (event_box_top_right, 8, 8);
  gtk_widget_realize(event_box_top_right);
  /* cursor - resize */
  gdk_window_set_cursor(event_box_top_right->window, 
			gdk_cursor_new(GDK_TOP_RIGHT_CORNER));

  event_box_bottom_right = gtk_event_box_new ();
  gtk_widget_show (event_box_bottom_right);
  gtk_table_attach (GTK_TABLE (table), event_box_bottom_right, 2, 3, 2, 3,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_FILL), 0, 0);
  gtk_widget_set_size_request (event_box_bottom_right, 8, 8);
  gtk_widget_realize(event_box_bottom_right);
  /* cursor - resize */
  gdk_window_set_cursor(event_box_bottom_right->window, 
			gdk_cursor_new(GDK_BOTTOM_RIGHT_CORNER));

  /* focus on text view */
  gtk_widget_grab_focus(text_view);

  /* when double-clicked the title becomes visible, send Note as data */
  g_signal_connect((gpointer) event_box_top_middle, "button-press-event",
		     G_CALLBACK(cb_move_window), 
		   (gpointer)main_w);
  g_signal_connect((gpointer) event_box_top_middle, "button-press-event",
		   G_CALLBACK(cb_title_change), 
		   (gpointer)note);
  g_signal_connect((gpointer) entry_title, "key-press-event", 
 		   G_CALLBACK(cb_title_key_press),  
 		   (gpointer)note); 
  g_signal_connect((gpointer) text_view, "focus-in-event",
		   G_CALLBACK(cb_text_view_focus_in), 
		   (gpointer)note);
  

  g_signal_connect ((gpointer) label_title, "key-press-event",
                    G_CALLBACK (cb_title_key_press),
		    (gpointer)note);
  g_signal_connect ((gpointer) event_box_middle_left, "button-press-event",
                    G_CALLBACK (cb_move_window),
		    (gpointer)main_w);
  g_signal_connect ((gpointer) event_box_bottom_left, "button-press-event",
                    G_CALLBACK (cb_resize_window),
		    (gpointer)GDK_WINDOW_EDGE_SOUTH_WEST);
  g_signal_connect ((gpointer) event_box_middle_right, "button-press-event",
                    G_CALLBACK (cb_move_window),
		    (gpointer)main_w);
  g_signal_connect ((gpointer) event_box_bottom_middle, "button-press-event",
                    G_CALLBACK (cb_move_window),
		    (gpointer)main_w);
  g_signal_connect ((gpointer) event_box_top_left, "button-press-event",
                    G_CALLBACK (cb_resize_window), 
		    (gpointer)GDK_WINDOW_EDGE_NORTH_WEST);
  g_signal_connect ((gpointer) event_box_top_right, "button-press-event",
                    G_CALLBACK (cb_resize_window), 
		    (gpointer)GDK_WINDOW_EDGE_NORTH_EAST);
  g_signal_connect ((gpointer) event_box_bottom_right, "button-press-event",
                    G_CALLBACK (cb_resize_window),
		    (gpointer)GDK_WINDOW_EDGE_SOUTH_EAST);

  /* add widgets to note structure */

  note->note_w = main_w;

  note->text_view = text_view;
  note->button_close = button_close;

  note->entry_title = entry_title;
  note->label_title = label_title;
  note->event_box_top_left = event_box_top_left;
  note->event_box_top_middle = event_box_top_middle;
  note->event_box_top_right = event_box_top_right;
  note->event_box_middle_left = event_box_middle_left;
  note->event_box_middle_right = event_box_middle_right;
  note->event_box_bottom_left = event_box_bottom_left;  
  note->event_box_bottom_middle = event_box_bottom_middle;  
  note->event_box_bottom_right = event_box_bottom_right;
  
  return note;
}


    
