/* # skkinput (Simple Kana-Kanji Input)
 * FontMgr.c
 * This file is part of skkinput.
 * Copyright (C) 1997
 * Takashi SAKAMOTO (sakamoto@yajima.kuis.kyoto-u.ac.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with skkinput; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <stdio.h>
#include <stdlib.h>
#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <X11/Intrinsic.h>

#include "commondef.h"
#include "kanji.h"
#define fontMgr_Static
#include "FontMgr.h"
#include "config.h"

extern unsigned char *getOneFontsetFromFontsetList
( unsigned char **string ) ;

/*
 * skkinput ɬפȤƤեȤ̾ġ
 */
/* iso8859 κȾʬѤʤΤǡɤȤäƤסĤȻפ
 * jisx0201 ȤäƤޤ backslash ȤΤȤä
 * ġ*/
/*
 * եȴꥹȡ
 */
static struct skkinputManagedFont *skkinputManagedFontListTop = NULL ;
static struct skkinputManagedFont *skkinputFreeFontListTop = NULL ;
static int freeFontListLength = 0 ;

/*
 * ǥեȤΥեȡ
 */
static struct skkinputManagedFont *defaultFontSet[ NUMBER_OF_CHARSET ] ;
static struct skkinputManagedFont *defaultMinibufFontSet[ NUMBER_OF_CHARSET ] ;

#ifdef DEBUG
/*
 *
 */
static void dump_fontlist( void )
{
  struct skkinputManagedFont *node = skkinputManagedFontListTop ;
  int i ;

  printf( "----- Manaed Font List ------\n" ) ;
  for( i = 0 ; node != NULL ; node = node->next, i ++ ){
    printf( "(%2d) ... \"%s\"\n", i, node->name ) ;
  }
  fflush( stdout ) ;
  return ; 
}

static void dump_freefontlist( void )
{
  struct skkinputManagedFont *node = skkinputFreeFontListTop ;
  int i ;

  printf( "----- Free Font List ------\n" ) ;
  for( i = 0 ; node != NULL ; node = node->next, i ++ ){
    printf( "(%2d) ... \"%s\"\n", i, node->name ) ;
  }
  fflush( stdout ) ;
  return ; 
}
#endif

/*
 *
 */
static struct skkinputManagedFont *fontMgr_AllocManagedFontListNode
( void )
{
  struct skkinputManagedFont *node ;

  node = ( struct skkinputManagedFont *)malloc
    ( sizeof( struct skkinputManagedFont ) ) ;
  /* ա֤ꤵ󡢤ɤޤ礦꤬ʤʤäƤޤޤġ*/
  /* ֤ʤ衢աġäơϤɤ褦ʤ */
  if( node != NULL ){
    /* ȽƤ*/
    node->next = NULL ;
    node->name = NULL ;
    node->font = NULL ;
    node->refer_count = 0 ;
  }
  return node ;
}

/*
 * ˥եȤϿƤ뤫ɤåؿ
 */
static struct skkinputManagedFont *fontMgr_FindManagedFontByName
( struct skkinputManagedFont *top, unsigned char *fontName )
{
  struct skkinputManagedFont *node = top ;
  unsigned char *ptr1, *ptr2 ;
  int chara1, chara2 ;

  /* ꥹȤƬ NULL ä顢ꥹȼΤǤ*/
  if( node == NULL )
    return NULL ;
  /* 硹˥ꥹȤ򤿤ɤޤ礦͡ */
  while( node != NULL ){
    ptr1 = node->name ;
    ptr2 = fontName ;
    while( *ptr1 != '\0' ){
      chara1 = ( *ptr1 >= 'A' && *ptr1 <= 'Z' )?
	*ptr1 + ( 'a' - 'A' ) : *ptr1 ;
      chara2 = ( *ptr2 >= 'A' && *ptr2 <= 'Z' )?
	*ptr2 + ( 'a' - 'A' ) : *ptr2 ;
      if( chara1 != chara2 )
	break ;
      ptr1 ++ ;
      ptr2 ++ ;
    }
    if( *ptr2 == '\0' && *ptr1 == '\0' )
      return node ;
    node = node->next ;
  }
  /* դޤǤ͡*/
  return NULL ;
}

/*
 * եȤؿ
 */
static struct skkinputManagedFont *fontMgr_AllocNewFont
( Display *disp, unsigned char *name )
{
  struct skkinputManagedFont *node, *pNode ;

  /* ¸ߤƤ뤫ɤǧ롣*/
  node = fontMgr_FindManagedFontByName
    ( skkinputManagedFontListTop, name ) ;
  if( node == NULL ){
    node = fontMgr_FindManagedFontByName
      ( skkinputFreeFontListTop, name ) ;
    if( node == NULL ){
      /* ¸ߤƤʤä顢ݤ롣*/
      if( ( node = fontMgr_AllocManagedFontListNode() ) == NULL )
	return NULL ;
      /* եȤɤ߹ࡣ*/
      if( ( node->font = XLoadQueryFont( disp, name ) ) == NULL ){
#ifdef DEBUG
	printf( "Find No Font: \"%s\" --> Error.\n", name ) ;
#endif
	free( node ) ;
	return NULL ;
      }
#ifdef DEBUG
      printf( "Find No Font: \"%s\" --> load.\n", name ) ;
#endif
      /* եȤ̾ϿƤ*/
      if( ( node->name = malloc( strlen( name ) + 1 ) ) == NULL ){
	XFreeFont( disp, node->font ) ;
	free( node ) ;
	return NULL ;
      }
      strcpy( node->name, name ) ;
      /* ꥹȤˤĤʤ*/
      node->next   = skkinputManagedFontListTop ;
      skkinputManagedFontListTop = node ;
#ifdef DEBUG
      fprintf
	( stderr, "----- New Font(Name):%s\n", name ) ;
#endif
    } else {
      if( node == skkinputFreeFontListTop ){
	skkinputFreeFontListTop = node->next ;
      } else {
	for( pNode = skkinputFreeFontListTop ;
	     pNode->next != node ; pNode = pNode->next )
	  ;
	pNode->next = node->next ;
      }
#ifdef DEBUG
      printf( "Find Cached Font: \"%s\"\n", node->name ) ;
#endif
      freeFontListLength -- ;
      /* ꥹȤˤĤʤ*/
      node->next = skkinputManagedFontListTop ;
      skkinputManagedFontListTop = node ;
    }
    /* ʬȤƤʤ顢Ȥ 1 Ǥ롣*/
    node->refer_count = 1 ;
  } else {
#ifdef DEBUG
    printf( "Find Currently Used Font: \"%s\"\n", node->name ) ;
#endif
    /* ¸ߤƤΤʤ顢ȥȤ 1 䤷Ƥ*/
    node->refer_count ++ ;
  }
  return ( node ) ;
}

/*
 * եȤ̾ǧؿ
 *----
 * skkinput ɬפʥեȤɤåؿƽФ
 * Ȥˤʤ롣
 */
static int fontMgr_compareFontName
( unsigned char *fontName, unsigned char *identifier, int a_not_equal_A )
{
  unsigned char *fptr, *iptr ;

  fptr = fontName ;
  iptr = identifier ;

  while( *iptr != '\0' ){
    switch( *iptr ){
    case '*' :
      if( *fptr == '\0' ){
	/* ʸϺǸޤǹԤäƤޤäƤΤǡʬ "*" 
	 * ʸ "*"  '\0' ǤʤФʤʤ*/
	iptr ++ ;
      } else {
	/* iptr  fpt + 1 ӤԤ
	 * "*-hogehoge"  "abc-hogehoge" Ӥäˡ
	 * ޤ"*-hogehoge"  "bc-hogehoge" ӤԤȤƤ
	 * ΤǤ롣*/
	if( fontMgr_compareFontName( fptr + 1, iptr, a_not_equal_A ) ){
	  /* פνĤޤꡢפȤȤ
	   * ʤΤ顢return True */
	  return True ;
	} else {
	  /* פʤäνξˤϡ"*"  match 
	   * ͽۤĹְäƤȽǤ롣Ĥޤꡢȡ
	   * "-hogehoge"  "abc-hogehoge" ӤԤȤˤʤ롣*/
	  iptr ++ ;
	}
      }
      break ;
    case '?' :
      /* ξ硢Ǥդΰʸ match 뤳Ȥˤʤ롣*/
      if( *fptr == '\0' )
	return False ;
      iptr ++ ;
      fptr ++ ;
      break ;
    case '\\' :
      /* ξˤϡʸ̵뤷ƼʸӤ˰ܤ뤳Ȥˤ
       * 롣μʸ *, ? ǤäƤ⡢ʸȤư*/
      iptr ++ ;
      if( *iptr == '\0' )
	return False ;
    default :
      if( *iptr != *fptr ){
	int chara1, chara2 ;
	/* ⤷ʸʸζ̤򤷤ʤΤʤ顢Ф˰ۤʤʸ
	 * ȴ롣*/
	if( a_not_equal_A )
	  return False ;
	/* ʸʸζ̤򤷤ʤΤǡե٥åȤʤƾ
	 * ʸľƤӤľ*/
	chara1 = ( *iptr >= 'A' && *iptr <= 'Z' )?
	  *iptr + ( 'a' - 'A' ) : *iptr ;
	chara2 = ( *fptr >= 'A' && *fptr <= 'Z' )?
	  *fptr + ( 'a' - 'A' ) : *fptr ;
	/* ʸƱΤӤƱǤ ۤʤСФ˰㤦ʸ
	 * (㤦ʸƤ롢Ƚʡ )*/
	if( chara1 != chara2 )
	  return False ;
      }
      /* ĤʸϺΤȤפƤΤǡʸӤ
       * Ԥޤ礦*/
      iptr ++ ;
      fptr ++ ;
      break ;
    }
  }
  return True ;
}

/*
 * skkinput ѤեȤǤ뤫ɤȽ̤ؿ
 * ---
 * "jisx0208.1983-0", "iso8859-1", "jisx0201.1976-0" 
 */
static int fontMgr_GetCharsetOfFontName
( unsigned char *fontName, unsigned char *charsetlist )
{
  unsigned char ***setptr, **ptr ;
  int charset_num, ret ;

#ifdef MAJIME
  /* ä˿ܤӤС*/
  ret = False ;
  for( setptr = fontset_identifiers, charset_num = 0 ;
       *setptr != NULL && charset_num < NUMBER_OF_CHARSET ;
       setptr ++, charset_num ++ ){
    charsetlist[ charset_num ] = False ;
    for( ptr = *setptr ; *ptr != NULL ; ptr ++ ){
      if( fontMgr_compareFontName( fontName, *ptr, False ) ){
	charsetlist[ charset_num ] = True ;
	ret = True ;
	break ;
      }
    }
  }
#else
  int length = strlen( fontName ) ;
  int flength ;

#ifdef DEBUG
  printf( "(charset check) \"%s\"\n", fontName ) ;
#endif
  /* äϼȴޤΥСƬ * ʤ顢
   * פɤȤСʤ櫓Ǥ*/ 
  ret = False ;
  for( setptr = fontset_identifiers, charset_num = 0 ;
       *setptr != NULL && charset_num < NUMBER_OF_CHARSET ;
       setptr ++, charset_num ++ ){
    charsetlist[ charset_num ] = False ;
    for( ptr = *setptr ; *ptr != NULL ; ptr ++ ){
      flength = strlen( 1 + *ptr ) ;
      if( length < flength )
	continue ;
      if( !strcasecmp( fontName + length - flength, 1 + *ptr ) ){
#ifdef DEBUG
	printf( "(%d)[Font] %s\n", charset_num, fontName ) ;
#endif
	charsetlist[ charset_num ] = True ;
	ret = True ;
	break ;
      }
    }
  }
#endif
  return ret ;
}

/*
 * skkinput ѤեȤǤ뤫ɤȽ̤ؿ
 * ---
 * "jisx0208.1983-0", "iso8859-1", "jisx0201.1976-0" 
 */
static int fontMgr_GetFontName
( unsigned char *fontName, unsigned char **charsetlist )
{
  unsigned char ***setptr, **ptr, *string ;
  int charset_num, count ;
  int length, flength ;

  count       = 0 ;
  setptr      = fontset_identifiers ;
  charset_num = 0 ;
  length      = strlen( fontName ) ;

  /* charset ֤˸ƹԤ*/
  while( *setptr != NULL && charset_num < NUMBER_OF_CHARSET ){
    /* ˳ݤƤΤʤ̵*/
    if( charsetlist[ charset_num ] == NULL ){
      /* ݤƤʤΤʤ顢줬 charset ΥեȤʤΤ
       * 롣*/
      for( ptr = *setptr ; *ptr != NULL ; ptr ++ ){
	flength = strlen( 1 + *ptr ) ;
	if( length < flength )
	  continue ;
	/* ⤷եȤ¸ߤΤʤ顣*/
	if( !strcasecmp( fontName + length - flength, 1 + *ptr ) ){
	  if( ( string = malloc( length + 1 ) ) != NULL ){
	    strcpy( string, fontName ) ;
	    charsetlist[ charset_num ] = string ;
	  }
	  count ++ ;
	  break ;
	}
      }
    } else {
      count ++ ;
    }
    setptr      ++ ;
    charset_num ++ ;
  }
  return count ;
}


/*
 * եȤ롣
 */
void fontMgr_FreeFont
( Display *disp, struct skkinputManagedFont *font )
{
  struct skkinputManagedFont *node ;

  if( font == NULL )
    return ;
  
  if( font->refer_count > 1 ){
    /* ¾ˤ⻲ȤƤͤΤǡõǤʤ*/
    font->refer_count -- ;
    return ;
  }
  /* ⤦ȤƤͤϼʬʤΤǡõƤޤ*/
  if( font == skkinputManagedFontListTop ){
    skkinputManagedFontListTop = font->next ;
  } else {
    node = skkinputManagedFontListTop ;
    while( node != NULL ){
      if( node->next == font )
	break ;
      node  = node->next ;
    }
    node->next = font->next ;
  }
  if( freeFontListLength < FONT_CACHE_SIZE ){
    /* ե꡼ꥹȤˤĤʤ롣*/
    font->next              = skkinputFreeFontListTop ;
    skkinputFreeFontListTop = font ;
    freeFontListLength ++ ;
  } else {
    /* ޤեȤơ*/
    XFreeFont( disp, font->font ) ;
    /* եȤ̾롣*/
    free( font->name ) ;
    /* ΡɼΤ롣*/
    free( font ) ;
  }
  return ;
}

void fontMgr_PrepareFont
( Display *disp, struct skkinputManagedFont **fontset,
  int charset, unsigned char *exactname )
{
  /* եȤݤ롣*/
  struct skkinputManagedFont *font ;
  /* ˤˤϲեȤƤƤΤǤС롣*/
  if( ( font = fontMgr_AllocNewFont( disp, exactname ) ) != NULL ){
    if( fontset[ charset ] != NULL ){
      fontMgr_FreeFont( disp, fontset[ charset ] ) ;
    }
    fontset[ charset ] = font ;
  }
#ifdef DEBUG
  dump_fontlist() ;
  dump_freefontlist() ;
#endif
  return ;
}

/*
 * fontset ꤷʸ󤫤顢font ؿ
 */
void fontMgr_PrepareFontByName
( Display *disp, struct skkinputManagedFont **fontset,
  unsigned char *string )
{
  unsigned char *fontset_string ;
  unsigned char *charsetlist[ NUMBER_OF_CHARSET ] ;
  char **fontslist ;
  int actual_count, i, j ;

  /* ⤷եȤꤷȦʸ󤬶ʤ饨顼ˤʤ롣*/
  if( string == NULL )
    return ;

  /* charset νԤ*/
  for( i = 0 ; i < NUMBER_OF_CHARSET ; i ++ )
    charsetlist[ i ] = NULL ;

  j = 0 ;
  /* ʸåơեڤФƤߤ롣*/
  while( *string != '\0' && j < NUMBER_OF_CHARSET ){
    fontset_string = getOneFontsetFromFontsetList( &string ) ;
    if( fontset_string == NULL )
      continue ;
#ifdef DEBUG
    printf( "(%d) \"%s\"\n", strlen( fontset_string ), fontset_string ) ;
#endif
    /* եȤΤ̾롣Ŭ̾ǤϡȤƤ⤸ʤ
     * ȽǤδˤϤʤʤ顣*/
    fontslist = XListFonts
      ( disp, ( char * )fontset_string,
	MAX_AVAILABLE_FONTS, &actual_count ) ;
    /* Υե̾˰פեȤºߤΤ */
    if( fontslist != NULL ){
      /* ºߤΤʤ顢ΤϿ롣*/
      for( i = 0 ; i < actual_count ; i ++ ){
#ifdef DEBUG
	printf( "(No. %2d) \"%s\"\n", i, fontslist[ i ] ) ;
#endif
	j = fontMgr_GetFontName( fontslist[ i ], charsetlist ) ;
	if( j >= NUMBER_OF_CHARSET )
	  break ;
      }
      XFreeFontNames( fontslist ) ;
    }
    free( fontset_string ) ;
  }
  /* եȤݤ롣*/
  for( i = 0 ; i < NUMBER_OF_CHARSET ; i ++ ){
    if( charsetlist[ i ] != NULL ){
      fontMgr_PrepareFont( disp, fontset, i, charsetlist[ i ] ) ;
      free( charsetlist[ i ] ) ;
    }
  }
  return ;
}


/*
 *  2 äƤΤʿɡĤäѤ #define Ȥʡ
 */
void fontMgr_PrepareFontByAtom
( Display *disp, struct skkinputManagedFont **fontset,
  Atom *fontatoms, Cardinal num_of_fonts )
{
  Atom *aptr ;
  unsigned char *fontlistName ;
  unsigned char charsetlist[ NUMBER_OF_CHARSET ] ;
  int i, j ;

  /* 褿եȤ󤿤ͤĥå롣*/
  for( i = 0, aptr = fontatoms ; i < num_of_fonts ; i ++, aptr ++ ){
    /* ޤեȤ̾ȴФ*/
    if( ( fontlistName = XGetAtomName( disp, *aptr ) ) == NULL )
      continue ;
#ifdef DEBUG
    printf( "(ByAtom) \"%s\"\n", fontlistName ) ;
#endif
    if( fontMgr_GetCharsetOfFontName( fontlistName, charsetlist ) ){
      for( j = 0 ; j < NUMBER_OF_CHARSET ; j ++ ){
	if( !charsetlist[ j ] )
	  continue ;
	fontMgr_PrepareFont
	  ( disp, fontset, j, fontlistName ) ;
      }
    }
    /* XGetAtomName  malloc 㤦ͤʤΤǡXFree ʤܡ*/
    XFree( fontlistName ) ;
  }
  return ;
}

int fontMgr_GetFontSetInfo
( struct skkinputManagedFont **fontset, int *ret_height, int *ret_ascent )
{
  int i, max_descent, max_ascent, ret ;

  ret = False ;
  max_descent = max_ascent = 0 ;
  for( i = 0 ; i < NUMBER_OF_CHARSET ; i ++ ){
    if( fontset[ i ] == NULL )
      continue ;
    ret  = True ;
    if( max_descent < fontset[ i ]->font->descent )
      max_descent = fontset[ i ]->font->descent ;
    if( max_ascent < fontset[ i ]->font->ascent )
      max_ascent  = fontset[ i ]->font->ascent ;
  }
  if( ret ){
    *ret_height = max_ascent + max_descent ;
    *ret_ascent = max_ascent ;
  }
  return ret ;
}

static struct skkinputManagedFont *fontMgr_CopyFont
( struct skkinputManagedFont *node )
{
  node->refer_count ++ ;
  return node ;
}

void fontMgr_CopyFontSet
( Display *disp, struct skkinputManagedFont **dest, 
  struct skkinputManagedFont **source )
{
  int i ;

  for( i = 0 ; i < NUMBER_OF_CHARSET ; i ++, dest ++, source ++ ){
    if( *source == NULL )
      continue ;
    /* ˥եȤʤ ʤ̣*/
    if( *dest != NULL ){
      fontMgr_FreeFont( disp, *dest ) ;
    }
    /* ȥ󥿤䤹Ȥʤ˥ԡ free ˤϤޤ롣*/
    fontMgr_CopyFont( *source ) ;
    *dest = *source ;
  }
  return ;
}

/*
 * եȥåȤ϶̤ǤĤäߤʤȤȤ顢٤
 * ǳݤƤɤʤȤå
 */
void fontMgr_initDefaultFontSet
( Display *disp, 
  unsigned char *defaultFontSetName,
  unsigned char *defaultMinibufFontSetName )
{
  int i ;
  for( i = 0 ; i < NUMBER_OF_CHARSET ; i ++ ){
    defaultFontSet[ i ] = defaultMinibufFontSet[ i ] = NULL ;
  }
  /* ǥեȤΥեȤݤ롣*/
  fontMgr_PrepareFontByName
    ( disp, defaultFontSet, defaultFontSetName ) ;
  fontMgr_PrepareFontByName
    ( disp, defaultMinibufFontSet, defaultMinibufFontSetName ) ;
  return ;
}

/*
 * ǥեȤΥեȥåȤؿå
 */
void fontMgr_closeDefaultFontSet( Display *disp )
{
  int i ;
  for( i = 0 ; i < NUMBER_OF_CHARSET ; i ++ ){
    fontMgr_FreeFont( disp, defaultFontSet[ i ] ) ;
    fontMgr_FreeFont( disp, defaultMinibufFontSet[ i ] ) ;
  }
  return ;
}

void fontMgr_copyDefaultFontSet
( Display *disp, struct skkinputManagedFont **dest )
{
  fontMgr_CopyFontSet( disp, dest, defaultFontSet ) ;
  return ;
}

void fontMgr_copyDefaultMinibufFontSet
( Display *disp, struct skkinputManagedFont **dest )
{
  fontMgr_CopyFontSet( disp, dest, defaultMinibufFontSet ) ;
  return ;
}
