/*
 * SCIM Bridge
 *
 * Copyright (c) 2006 Ryo Dairiki <ryo-dairiki@users.sourceforge.net>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.*
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.See the
 * GNU Lesser General Public License for more details.*
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 */

#include <assert.h>
#include <malloc.h>
#include <pthread.h>
#include <stdarg.h>
#include <stdio.h>
#include <string.h>

#include "scim-bridge-exception.h"
#include "scim-bridge-output.h"

/* Data structs */
typedef struct _ScimBridgeException
{
    scim_bridge_exception_code_t code;

    char *message;
    size_t message_capacity;

    char **stacks;
    size_t *stack_line_capacities;
    size_t stack_capacity;
    size_t stack_begin;
    size_t stack_end;
} ScimBridgeException;

/* Private variables */
static pthread_key_t exception_key;

static pthread_once_t static_initializer = PTHREAD_ONCE_INIT;

/* Predefinition */
static void free_exception (void *buffer);
static void static_initialize ();

/* Private Functions*/
static ScimBridgeException *do_alloc_exception ()
{
    ScimBridgeException *exception = malloc (sizeof (ScimBridgeException));
    exception->message_capacity = 50;
    exception->message = malloc (sizeof (char) * (exception->message_capacity + 1));
    exception->message[0] = '\0';

    exception->stack_begin = 0;
    exception->stack_end = 0;
    exception->stack_capacity = 10;
    exception->stacks = malloc (sizeof (char*) * exception->stack_capacity);
    exception->stack_line_capacities = malloc (sizeof (size_t) * exception->stack_capacity);
    int i;
    for (i = 0; i < exception->stack_capacity; ++i) {
        exception->stack_line_capacities[i] = 25;
        exception->stacks[i] = malloc (sizeof (char) * (exception->stack_line_capacities[i] + 1));
        exception->stacks[i][0] = '\0';
    }

    return exception;
}


static void do_free_exception (ScimBridgeException *exception)
{
    int i;
    for (i = 0; i < exception->stack_capacity; ++i) {
        free (exception->stacks[i]);
    }
    free (exception->message);
    free (exception->stacks);
    free (exception->stack_line_capacities);
    free (exception);
}


static ScimBridgeException *do_get_exception ()
{
    pthread_once (&static_initializer, static_initialize);

    ScimBridgeException *exception = (ScimBridgeException*) pthread_getspecific (exception_key);
    if (exception == NULL) {
        exception = do_alloc_exception ();
        pthread_setspecific (exception_key, exception);
    }

    return exception;
}


/* Functions */
void scim_bridge_exception_clear ()
{
    ScimBridgeException *exception = do_get_exception ();
    exception->code = NO_EXCEPTION;
    exception->message[0] = '\0';
    exception->stack_begin = 0;
    exception->stack_end = 0;
}


void scim_bridge_exception_occured (scim_bridge_exception_code_t code, const char *message,...)
{
    assert (code != NO_EXCEPTION);

    ScimBridgeException *exception = do_get_exception ();

    exception->stack_begin = 0;
    exception->stack_end = 0;

    exception->code = code;

    va_list ap;
    va_start (ap, message);
    const size_t message_length = vsnprintf (NULL, 0, message, ap);
    va_end (ap);

    if (message_length > exception->message_capacity) {
        free (exception->message);
        exception->message_capacity = message_length;
        exception->message = malloc (sizeof (char) * (exception->message_capacity + 1));
    }
    va_start (ap, message);
    vsnprintf (exception->message, exception->message_capacity + 1, message, ap);
    va_end (ap);
}


void scim_bridge_exception_push_stack (const char *stack_line)
{
    ScimBridgeException *exception = do_get_exception ();

    if (exception->stack_end >= exception->stack_capacity) {
        ++exception->stack_capacity;
        exception->stacks = realloc (exception->stacks, sizeof (const char**) * exception->stack_capacity);
        exception->stack_line_capacities = realloc (exception->stack_line_capacities, sizeof (size_t) * exception->stack_capacity);

        exception->stack_line_capacities[exception->stack_end] = 25;
        exception->stacks[exception->stack_end] = malloc (sizeof (char*) * (exception->stack_line_capacities[exception->stack_end] + 1));
    }

    const size_t stack_line_length = strlen (stack_line);
    if (stack_line_length > exception->stack_line_capacities[exception->stack_end]) {
        exception->stack_line_capacities[exception->stack_end] = stack_line_length;
        free (exception->stacks[exception->stack_end]);
        exception->stacks[exception->stack_end] = malloc (sizeof (char*) * (exception->stack_line_capacities[exception->stack_end] + 1));
    }
    strcpy (exception->stacks[exception->stack_end], stack_line);
    ++exception->stack_end;
}


const char *scim_bridge_exception_poll_stack ()
{
    ScimBridgeException *exception = do_get_exception ();

    if (exception->stack_begin < exception->stack_end) {
        const char *stack = exception->stacks[exception->stack_begin];
        ++exception->stack_begin;
        return stack;
    } else {
        return NULL;
    }
}


const char *scim_bridge_exception_get_message ()
{
    ScimBridgeException *exception = do_get_exception ();
    return exception->message;
}


scim_bridge_exception_code_t scim_bridge_exception_get_code ()
{
    ScimBridgeException *exception = do_get_exception ();
    return exception->code;
}


void scim_bridge_exception_output ()
{
    ScimBridgeException *exception = do_get_exception ();

    switch (exception->code) {
        case OUT_OF_BOUNDS_EXCEPTION:
            scim_bridge_perror ("Out of bounds exception has been occured: ");
            break;
        case NULL_POINTER_EXCEPTION:
            scim_bridge_perror ("Null pointer exception has been occured: ");
            break;
        case ILLEGAL_STATE_EXCEPTION:
            scim_bridge_perror ("Illegal state exception has been occured: ");
            break;
        case INVALID_ARGUMENT_EXCEPTION:
            scim_bridge_perror ("Invalid argument exception has been occured: ");
            break;
        case IO_EXCEPTION:
            scim_bridge_perror ("IO exception has been occured: ");
            break;
        case NUMBER_FORMAT_EXCEPTION:
            scim_bridge_perror ("Number format exception has been occured: ");
            break;
        case NO_SUCH_ELEMENT_EXCEPTION:
            scim_bridge_perror ("No such element exception has been occured: ");
            break;
        case INVALID_ESCAPE_EXCEPTION:
            scim_bridge_perror ("Invalid escape element exception has been occured: ");
            break;
        case UNEXPECTED_VALUE_EXCEPTION:
            scim_bridge_perror ("Unexpected value element exception has been occured: ");
            break;
        case INVALID_UTF8_EXCEPTION:
            scim_bridge_perror ("Invalid utf8 exception has been occured: ");
            break;
        case INVALID_UCS4_EXCEPTION:
            scim_bridge_perror ("Invalid ucs4 exception has been occured: ");
            break;
        case NO_ENOUGH_BUFFER_EXCEPTION:
            scim_bridge_perror ("No enough buffer exception has been occured: ");
            break;
        case OVER_FLOW_EXCEPTION:
            scim_bridge_perror ("Overflow exception has been occured: ");
            break;
        case NO_EXCEPTION:
            break;
        case UNKNOWN_EXCEPTION:
        default:
            scim_bridge_perror ("Unknown exception has been occured: ");
            break;
    }
    scim_bridge_perrorln ("%s", exception->message);
    scim_bridge_perrorln ("stacktrace: ");
    int i;
    for (i = exception->stack_begin; i < exception->stack_end; ++i) {
        scim_bridge_perrorln (" at %s", exception->stacks[i]);
    }
}


/* Helper Functions */
static void free_exception (void *buffer)
{
    ScimBridgeException *exception = (ScimBridgeException*) buffer;
    do_free_exception (exception);
}


static void static_initialize ()
{
    pthread_key_create (&exception_key, free_exception);
}
