/*
 * SCIM Bridge
 *
 * Copyright (c) 2006 Ryo Dairiki <ryo-dairiki@users.sourceforge.net>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.*
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.See the
 * GNU Lesser General Public License for more details.*
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 */

#include <pthread.h>
#include <stdlib.h>
#include <string.h>

#include "scim-bridge-debug.h"
#include "scim-bridge-string.h"

/* Static variables */
static scim_bridge_debug_level_t debug_level;
static scim_bridge_debug_flag_t debug_flags;

static pthread_once_t static_initializer = PTHREAD_ONCE_INIT;

/* Private function */
static void static_initialize ()
{
    unsigned int debug_level_uint;
    char *debug_level_str = getenv ("SCIM_BRIDGE_DEBUG_LEVEL");
    if (debug_level_str == NULL || scim_bridge_string_to_uint (&debug_level_uint, debug_level_str)) {
        debug_level = 0;
    } else {
        debug_level = debug_level_uint;
        if (debug_level > 10) debug_level = 10;
    }

    debug_flags = SCIM_BRIDGE_DEBUG_ALL;
    char *debug_flags_str = getenv ("SCIM_BRIDGE_DEBUG_FLAGS");
    if (debug_flags_str == NULL) return;

    char *tmp_str = alloca (sizeof (char) * (strlen (debug_flags_str) + 1));
    strcpy (tmp_str, debug_flags_str);
    char *p;
    char *token = strtok_r (tmp_str, " ", &p);
    while (token != NULL) {
        if (strcmp (token, "messenger") == 0) {
            debug_flags |= SCIM_BRIDGE_DEBUG_MESSENGER;
        } else if (strcmp (token, "agent") == 0) {
            debug_flags |= SCIM_BRIDGE_DEBUG_AGENT;
        } else if (strcmp (token, "imcontext") == 0) {
            debug_flags |= SCIM_BRIDGE_DEBUG_IMCONTEXT;
        } else if (strcmp (token, "client") == 0) {
            debug_flags |= SCIM_BRIDGE_DEBUG_CLIENT;
        } else if (strcmp (token, "scim") == 0) {
            debug_flags |= SCIM_BRIDGE_DEBUG_SCIM;
        } else if (strcmp (token, "all") == 0) {
            debug_flags = SCIM_BRIDGE_DEBUG_ALL;
        } else if (strcmp (token, "none") == 0) {
            debug_flags = SCIM_BRIDGE_DEBUG_NONE;
        }
        token = strtok_r (NULL, " ", &p);
    }
}


/* Implementations */
scim_bridge_debug_level_t scim_bridge_debug_get_level ()
{
    pthread_once (&static_initializer, &static_initialize);
    return debug_level;
}


scim_bridge_debug_flag_t scim_bridge_debug_get_flags ()
{
    pthread_once (&static_initializer, &static_initialize);
    return debug_flags;
}
