/*
 * SCIM Bridge
 *
 * Copyright (c) 2006 Ryo Dairiki <ryo-dairiki@users.sourceforge.net>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.*
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.See the
 * GNU Lesser General Public License for more details.*
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 */

#include <errno.h>
#include <string.h>

#include <sys/types.h>
#include <sys/wait.h>

#define Uses_SCIM_CONFIG_PATH

#include <scim.h>

#ifndef SCIM_PANEL_PROGRAM
#define SCIM_PANEL_PROGRAM (SCIM_LIBEXECDIR "/scim-panel-gtk")
#endif

#include "scim-bridge-agent-panel-client.h"
#include "scim-bridge-output.h"

using namespace scim;

typedef Signal1<void, int> ScimBridgeAgentPanelClientSignalVoid;
typedef Signal2<void, int, int> ScimBridgeAgentPanelClientSignalInt;
typedef Signal2<void, int, const String&> ScimBridgeAgentPanelClientSignalString;
typedef Signal2<void, int, const WideString&> ScimBridgeAgentPanelClientSignalWideString;
typedef Signal4<void, int, const String&, const String&, const Transaction&> ScimBridgeAgentPanelClientSignalStringStringTransaction;
typedef Signal2<void, int, const KeyEvent&> ScimBridgeAgentPanelClientSignalKeyEvent;

class ScimBridgeAgentPanelClientImpl: public ScimBridgeAgentPanelClient
{

    public:

        ScimBridgeAgentPanelClientImpl (const String &config_name, const ScimBridgeDisplay *display);
        ~ScimBridgeAgentPanelClientImpl ();

        int open_connection ();
        void close_connection ();
        int get_connection_number () const;
        bool is_connected () const;

        bool has_pending_event () const;
        retval_t filter_event ();

        retval_t prepare (scim_bridge_imcontext_id_t imcontext_id);
        retval_t send ();

        void turn_on ();
        void turn_off ();
        void update_screen ();
        void show_help (const String &help);
        void show_factory_menu (const std::vector<PanelFactoryInfo> &menu);
        void focus_in (const scim::String &uuid);
        void focus_out ();
        void update_factory_info (const PanelFactoryInfo &info);
        void update_spot_location (int x, int y);
        void show_preedit_string ();
        void show_aux_string ();
        void show_lookup_table ();
        void hide_preedit_string ();
        void hide_aux_string ();
        void hide_lookup_table ();
        void update_preedit_string (const WideString &str, const AttributeList &attrs);
        void update_preedit_caret (int cursor_position);
        void update_aux_string (const WideString &str, const AttributeList &attrs);
        void update_lookup_table (const LookupTable &table);
        void register_properties (const PropertyList &properties);
        void update_property (const Property &property);
        void start_helper (const String &helper_uuid);
        void stop_helper (const String &helper_uuid);
        void send_helper_event (const String &helper_uuid, const Transaction &trans);
        void register_input_context (const String &uuid);
        void remove_input_context ();

        Connection signal_connect_reload_config (ScimBridgeAgentPanelClientSlotVoid *slot);
        Connection signal_connect_exit (ScimBridgeAgentPanelClientSlotVoid *slot);
        Connection signal_connect_update_lookup_table_page_size (ScimBridgeAgentPanelClientSlotInt *slot);
        Connection signal_connect_lookup_table_page_up (ScimBridgeAgentPanelClientSlotVoid *slot);
        Connection signal_connect_lookup_table_page_down (ScimBridgeAgentPanelClientSlotVoid *slot);
        Connection signal_connect_trigger_property (ScimBridgeAgentPanelClientSlotString *slot);
        Connection signal_connect_process_helper_event (ScimBridgeAgentPanelClientSlotStringStringTransaction *slot);
        Connection signal_connect_move_preedit_caret (ScimBridgeAgentPanelClientSlotInt *slot);
        Connection signal_connect_select_candidate (ScimBridgeAgentPanelClientSlotInt *slot);
        Connection signal_connect_process_key_event (ScimBridgeAgentPanelClientSlotKeyEvent *slot);
        Connection signal_connect_commit_string (ScimBridgeAgentPanelClientSlotWideString *slot);
        Connection signal_connect_forward_key_event (ScimBridgeAgentPanelClientSlotKeyEvent *slot);
        Connection signal_connect_request_help (ScimBridgeAgentPanelClientSlotVoid *slot);
        Connection signal_connect_request_factory_menu (ScimBridgeAgentPanelClientSlotVoid *slot);
        Connection signal_connect_change_factory (ScimBridgeAgentPanelClientSlotString *slot);

    private:

        String config_name;
        ScimBridgeDisplay *display;

        SocketClient socket_client;
        int socket_timeout;
        uint32 socket_magic_key;
        Transaction send_transaction;
        scim_bridge_imcontext_id_t current_imcontext_id;
        int send_refcount;

        ScimBridgeAgentPanelClientSignalVoid  signal_reload_config;
        ScimBridgeAgentPanelClientSignalVoid signal_exit;
        ScimBridgeAgentPanelClientSignalInt signal_update_lookup_table_page_size;
        ScimBridgeAgentPanelClientSignalVoid signal_lookup_table_page_up;
        ScimBridgeAgentPanelClientSignalVoid signal_lookup_table_page_down;
        ScimBridgeAgentPanelClientSignalString signal_trigger_property;
        ScimBridgeAgentPanelClientSignalStringStringTransaction signal_process_helper_event;
        ScimBridgeAgentPanelClientSignalInt signal_move_preedit_caret;
        ScimBridgeAgentPanelClientSignalInt signal_select_candidate;
        ScimBridgeAgentPanelClientSignalKeyEvent signal_process_key_event;
        ScimBridgeAgentPanelClientSignalWideString signal_commit_string;
        ScimBridgeAgentPanelClientSignalKeyEvent signal_forward_key_event;
        ScimBridgeAgentPanelClientSignalVoid signal_request_help;
        ScimBridgeAgentPanelClientSignalVoid signal_request_factory_menu;
        ScimBridgeAgentPanelClientSignalString signal_change_factory;

        void launch_panel () const;

};

/* Implementations */
ScimBridgeAgentPanelClient *ScimBridgeAgentPanelClient::alloc (const String &config_name, const ScimBridgeDisplay *display)
{
    return new ScimBridgeAgentPanelClientImpl (config_name, display);
}


ScimBridgeAgentPanelClient::ScimBridgeAgentPanelClient ()
{
}


ScimBridgeAgentPanelClient::~ScimBridgeAgentPanelClient ()
{
}


ScimBridgeAgentPanelClientImpl::ScimBridgeAgentPanelClientImpl (const String &new_config_name, const ScimBridgeDisplay *new_display):
config_name (new_config_name), display (NULL), socket_timeout (scim_get_default_socket_timeout ()), socket_magic_key (0), current_imcontext_id (-1), send_refcount (0)
{
    if (new_display != NULL) {
        display = scim_bridge_alloc_display ();
        scim_bridge_copy_display (display, new_display);
    }
}


ScimBridgeAgentPanelClientImpl::~ScimBridgeAgentPanelClientImpl ()
{
    scim_bridge_free_display (display);
}


void ScimBridgeAgentPanelClientImpl::launch_panel () const
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT, 7, "launch_panel ()");

    String panel_program = scim_global_config_read (SCIM_GLOBAL_CONFIG_DEFAULT_PANEL_PROGRAM, String (SCIM_PANEL_PROGRAM));
    if (access (panel_program.c_str (), X_OK) != 0) panel_program = String (SCIM_PANEL_PROGRAM);
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT, 5, "Launch \"%s\"", panel_program.c_str ());

    if (display != NULL) {
        const pid_t retval = fork ();
        if (retval == 0) {
            if (daemon (0, 0) == 0) {
                char *display_name_cstr = strdup (scim_bridge_display_get_name (display));
                char *conf_name_cstr = strdup (config_name.c_str ());
                char *panel_program_cstr = strdup (panel_program.c_str ());
                char *my_argv[] = {panel_program_cstr, "--no-stay", "--display", display_name_cstr, "-c", conf_name_cstr, "-d", 0};
                if (execv (panel_program_cstr, my_argv)) abort ();
            } else {
                scim_bridge_perrorln ("Failed to forking for a panel process: %s", errno == 0 ? "Unknown reason":strerror (errno));
                abort ();
            }
        } else if (retval > 0) {
            waitpid (retval, NULL, 0);
        } else {
            scim_bridge_perrorln ("Failed to launch a panel: %s", errno == 0 ? "Unknown reason":strerror (errno));
        }
    } else {
        scim_bridge_println ("Not yet supported!");
        abort ();
    }
}


int ScimBridgeAgentPanelClientImpl::open_connection ()
{
    SocketAddress address (scim_get_default_panel_socket_address (scim_bridge_display_get_name (display)));
    if (socket_client.is_connected ()) close_connection ();

    bool retval;
    int count = 0;

    // Try three times.
    while (true) {
        if ((retval = socket_client.connect (address)) == false) {
            usleep (100000);
            launch_panel ();
            for (int i = 0; i < 200; ++i) {
                if (socket_client.connect (address)) {
                    retval = true;
                    break;
                }
                usleep (100000);
            }
        }

        if (retval && scim_socket_open_connection (socket_magic_key, String ("FrontEnd"), String ("Panel"), socket_client, socket_timeout))
            break;

        socket_client.close ();

        if (count++ >= 3) break;

        usleep (100000);
    }

    return socket_client.get_id ();
}


void ScimBridgeAgentPanelClientImpl::close_connection ()
{
    socket_client.close ();
    socket_magic_key = 0;
}


int ScimBridgeAgentPanelClientImpl::get_connection_number () const
{
    return socket_client.get_id ();
}


bool ScimBridgeAgentPanelClientImpl::is_connected () const
{
    return socket_client.is_connected ();
}


bool ScimBridgeAgentPanelClientImpl::has_pending_event () const
{
    return socket_client.is_connected () && socket_client.wait_for_data (0) > 0;
}


retval_t ScimBridgeAgentPanelClientImpl::filter_event ()
{
    Transaction recv_transaction;

    if (!socket_client.is_connected () || !recv_transaction.read_from_socket (socket_client, socket_timeout)) return RETVAL_FAILED;

    int cmd;
    uint32 context = static_cast<uint32> (-1);

    if (!recv_transaction.get_command (cmd) || cmd != SCIM_TRANS_CMD_REPLY) return RETVAL_SUCCEEDED;

    // No context id available, so there will be some global command.
    if (recv_transaction.get_data_type () == SCIM_TRANS_DATA_COMMAND) {
        while (recv_transaction.get_command (cmd)) {
            switch (cmd) {
                case SCIM_TRANS_CMD_RELOAD_CONFIG:
                    signal_reload_config (context);
                    break;
                case SCIM_TRANS_CMD_EXIT:
                    signal_exit (context);
                    break;
                default:
                    break;
            }
        }
        return RETVAL_SUCCEEDED;
    }

    // Now for context related command.
    if (!recv_transaction.get_data (context)) return RETVAL_SUCCEEDED;

    while (recv_transaction.get_command (cmd)) {
        switch (cmd) {
            case SCIM_TRANS_CMD_UPDATE_LOOKUP_TABLE_PAGE_SIZE:
            {
                uint32 size;
                if (recv_transaction.get_data (size)) signal_update_lookup_table_page_size (context, (int) size);
            }
            break;
            case SCIM_TRANS_CMD_LOOKUP_TABLE_PAGE_UP:
            {
                signal_lookup_table_page_up (context);
            }
            break;
            case SCIM_TRANS_CMD_LOOKUP_TABLE_PAGE_DOWN:
            {
                signal_lookup_table_page_down (context);
            }
            break;
            case SCIM_TRANS_CMD_TRIGGER_PROPERTY:
            {
                String property;
                if (recv_transaction.get_data (property)) signal_trigger_property (context, property);
            }
            break;
            case SCIM_TRANS_CMD_PROCESS_HELPER_EVENT:
            {
                String target_uuid;
                String helper_uuid;
                Transaction trans;
                if (recv_transaction.get_data (target_uuid) && recv_transaction.get_data (helper_uuid) && recv_transaction.get_data (trans)) signal_process_helper_event (context, target_uuid, helper_uuid, trans);
            }
            break;
            case SCIM_TRANS_CMD_MOVE_PREEDIT_CARET:
            {
                uint32 caret;
                if (recv_transaction.get_data (caret)) signal_move_preedit_caret (context, caret);
            }
            break;
            case SCIM_TRANS_CMD_SELECT_CANDIDATE:
            {
                uint32 item;
                if (recv_transaction.get_data (item)) signal_select_candidate (context, item);
            }
            break;
            case SCIM_TRANS_CMD_PROCESS_KEY_EVENT:
            {
                KeyEvent key;
                if (recv_transaction.get_data (key)) signal_process_key_event (context, key);
            }
            break;
            case SCIM_TRANS_CMD_COMMIT_STRING:
            {
                WideString wstr;
                if (recv_transaction.get_data (wstr)) signal_commit_string (context, wstr);
            }
            break;
            case SCIM_TRANS_CMD_FORWARD_KEY_EVENT:
            {
                KeyEvent key;
                if (recv_transaction.get_data (key)) signal_forward_key_event (context, key);
            }
            break;
            case SCIM_TRANS_CMD_PANEL_REQUEST_HELP:
            {
                signal_request_help (context);
            }
            break;
            case SCIM_TRANS_CMD_PANEL_REQUEST_FACTORY_MENU:
            {
                signal_request_factory_menu (context);
            }
            break;
            case SCIM_TRANS_CMD_PANEL_CHANGE_FACTORY:
            {
                String factory_uuid;
                if (recv_transaction.get_data (factory_uuid)) signal_change_factory (context, factory_uuid);
            }
            break;
            default:
                break;
        }
    }

    return RETVAL_SUCCEEDED;
}


retval_t ScimBridgeAgentPanelClientImpl::prepare (scim_bridge_imcontext_id_t imcontext_id)
{
    if (!socket_client.is_connected ()) return RETVAL_FAILED;

    int cmd;
    uint32 data;
    int new_imcontext_id = imcontext_id;

    if (send_refcount <= 0) {
        current_imcontext_id = imcontext_id;
        send_transaction.clear ();
        send_transaction.put_command (SCIM_TRANS_CMD_REQUEST);
        send_transaction.put_data (socket_magic_key);
        send_transaction.put_data ((uint32) new_imcontext_id);

        send_transaction.get_command (cmd);
        send_transaction.get_data (data);
        send_transaction.get_data (data);
        send_refcount = 0;
    }

    if (new_imcontext_id == current_imcontext_id) {
        ++send_refcount;
        return RETVAL_SUCCEEDED;
    } else {
        return RETVAL_FAILED;
    }
}


retval_t ScimBridgeAgentPanelClientImpl::send ()
{
    if (!socket_client.is_connected ()) return RETVAL_SUCCEEDED;

    if (send_refcount <= 0) return RETVAL_SUCCEEDED;
    --send_refcount;

    if (send_refcount > 0) return RETVAL_SUCCEEDED;

    current_imcontext_id = -1;
    if (send_transaction.get_data_type () != SCIM_TRANS_DATA_UNKNOWN) {
        return send_transaction.write_to_socket (socket_client, 0x4d494353);
    } else {
        return RETVAL_SUCCEEDED;
    }
}


void ScimBridgeAgentPanelClientImpl::turn_on ()
{
    if (send_refcount > 0) send_transaction.put_command (SCIM_TRANS_CMD_PANEL_TURN_ON);
}


void ScimBridgeAgentPanelClientImpl::turn_off ()
{
    if (send_refcount > 0) send_transaction.put_command (SCIM_TRANS_CMD_PANEL_TURN_OFF);
}


void ScimBridgeAgentPanelClientImpl::update_screen ()
{
    if (send_refcount > 0 && display != NULL) {
        send_transaction.put_command (SCIM_TRANS_CMD_UPDATE_SCREEN);
        send_transaction.put_data ((uint32) scim_bridge_display_get_screen_number (display));
    }
}


void ScimBridgeAgentPanelClientImpl::show_help (const String &help)
{
    if (send_refcount > 0) {
        send_transaction.put_command (SCIM_TRANS_CMD_PANEL_SHOW_HELP);
        send_transaction.put_data (help);
    }
}


void ScimBridgeAgentPanelClientImpl::show_factory_menu (const std::vector<PanelFactoryInfo> &menu)
{
    if (send_refcount > 0) {
        send_transaction.put_command (SCIM_TRANS_CMD_PANEL_SHOW_FACTORY_MENU);
        for (size_t i = 0; i < menu.size (); ++i) {
            send_transaction.put_data (menu[i].uuid);
            send_transaction.put_data (menu[i].name);
            send_transaction.put_data (menu[i].lang);
            send_transaction.put_data (menu[i].icon);
        }
    }
}


void ScimBridgeAgentPanelClientImpl::focus_in (const String &uuid)
{
    if (send_refcount > 0) {
        send_transaction.put_command (SCIM_TRANS_CMD_FOCUS_IN);
        send_transaction.put_data (uuid);
    }
}


void ScimBridgeAgentPanelClientImpl::focus_out ()
{
    if (send_refcount > 0) send_transaction.put_command (SCIM_TRANS_CMD_FOCUS_OUT);
}


void ScimBridgeAgentPanelClientImpl::update_factory_info (const PanelFactoryInfo &info)
{
    if (send_refcount > 0) {
        send_transaction.put_command (SCIM_TRANS_CMD_PANEL_UPDATE_FACTORY_INFO);
        send_transaction.put_data (info.uuid);
        send_transaction.put_data (info.name);
        send_transaction.put_data (info.lang);
        send_transaction.put_data (info.icon);
    }
}


void ScimBridgeAgentPanelClientImpl::update_spot_location (int x, int y)
{
    if (send_refcount > 0) {
        send_transaction.put_command (SCIM_TRANS_CMD_UPDATE_SPOT_LOCATION);
        send_transaction.put_data ((uint32) x);
        send_transaction.put_data ((uint32) y);
    }
}


void ScimBridgeAgentPanelClientImpl::show_preedit_string ()
{
    if (send_refcount > 0) send_transaction.put_command (SCIM_TRANS_CMD_SHOW_PREEDIT_STRING);
}


void ScimBridgeAgentPanelClientImpl::show_aux_string ()
{
    if (send_refcount > 0) send_transaction.put_command (SCIM_TRANS_CMD_SHOW_AUX_STRING);
}


void ScimBridgeAgentPanelClientImpl::show_lookup_table ()
{
    if (send_refcount > 0) send_transaction.put_command (SCIM_TRANS_CMD_SHOW_LOOKUP_TABLE);
}


void ScimBridgeAgentPanelClientImpl::hide_preedit_string ()
{
    if (send_refcount > 0) send_transaction.put_command (SCIM_TRANS_CMD_HIDE_PREEDIT_STRING);
}


void ScimBridgeAgentPanelClientImpl::hide_aux_string ()
{
    if (send_refcount > 0) send_transaction.put_command (SCIM_TRANS_CMD_HIDE_AUX_STRING);
}


void ScimBridgeAgentPanelClientImpl::hide_lookup_table ()
{
    if (send_refcount > 0) send_transaction.put_command (SCIM_TRANS_CMD_HIDE_LOOKUP_TABLE);
}


void ScimBridgeAgentPanelClientImpl::update_preedit_string (const WideString &str, const AttributeList &attrs)
{
    if (send_refcount > 0) {
        send_transaction.put_command (SCIM_TRANS_CMD_UPDATE_PREEDIT_STRING);
        send_transaction.put_data (utf8_wcstombs (str));
        send_transaction.put_data (attrs);
    }
}


void ScimBridgeAgentPanelClientImpl::update_preedit_caret (int caret)
{
    if (send_refcount > 0) {
        send_transaction.put_command (SCIM_TRANS_CMD_UPDATE_PREEDIT_CARET);
        send_transaction.put_data ((uint32) caret);
    }
}


void ScimBridgeAgentPanelClientImpl::update_aux_string (const WideString &str, const AttributeList &attrs)
{
    if (send_refcount > 0) {
        send_transaction.put_command (SCIM_TRANS_CMD_UPDATE_AUX_STRING);
        send_transaction.put_data (utf8_wcstombs (str));
        send_transaction.put_data (attrs);
    }
}


void ScimBridgeAgentPanelClientImpl::update_lookup_table (const LookupTable &table)
{
    if (send_refcount > 0) {
        send_transaction.put_command (SCIM_TRANS_CMD_UPDATE_LOOKUP_TABLE);
        send_transaction.put_data (table);
    }
}


void ScimBridgeAgentPanelClientImpl::register_properties (const PropertyList &properties)
{
    if (send_refcount > 0) {
        send_transaction.put_command (SCIM_TRANS_CMD_REGISTER_PROPERTIES);
        send_transaction.put_data (properties);
    }
}


void ScimBridgeAgentPanelClientImpl::update_property (const Property &property)
{
    if (send_refcount > 0) {
        send_transaction.put_command (SCIM_TRANS_CMD_UPDATE_PROPERTY);
        send_transaction.put_data (property);
    }
}


void ScimBridgeAgentPanelClientImpl::start_helper (const String &helper_uuid)
{
    if (send_refcount > 0) {
        send_transaction.put_command (SCIM_TRANS_CMD_START_HELPER);
        send_transaction.put_data (helper_uuid);
    }
}


void ScimBridgeAgentPanelClientImpl::stop_helper (const String &helper_uuid)
{
    if (send_refcount > 0) {
        send_transaction.put_command (SCIM_TRANS_CMD_STOP_HELPER);
        send_transaction.put_data (helper_uuid);
    }
}


void ScimBridgeAgentPanelClientImpl::send_helper_event (const String &helper_uuid, const Transaction &trans)
{
    if (send_refcount > 0) {
        send_transaction.put_command (SCIM_TRANS_CMD_SEND_HELPER_EVENT);
        send_transaction.put_data (helper_uuid);
        send_transaction.put_data (trans);
    }
}


void ScimBridgeAgentPanelClientImpl::register_input_context (const String &uuid)
{
    if (send_refcount > 0) {
        send_transaction.put_command (SCIM_TRANS_CMD_PANEL_REGISTER_INPUT_CONTEXT);
        send_transaction.put_data (uuid);
    }
}


void ScimBridgeAgentPanelClientImpl::remove_input_context ()
{
    if (send_refcount > 0) send_transaction.put_command (SCIM_TRANS_CMD_PANEL_REMOVE_INPUT_CONTEXT);
}


Connection ScimBridgeAgentPanelClientImpl::signal_connect_reload_config (ScimBridgeAgentPanelClientSlotVoid *slot)
{
    return signal_reload_config.connect (slot);
}


Connection ScimBridgeAgentPanelClientImpl::signal_connect_exit (ScimBridgeAgentPanelClientSlotVoid *slot)
{
    return signal_exit.connect (slot);
}


Connection ScimBridgeAgentPanelClientImpl::signal_connect_update_lookup_table_page_size (ScimBridgeAgentPanelClientSlotInt *slot)
{
    return signal_update_lookup_table_page_size.connect (slot);
}


Connection ScimBridgeAgentPanelClientImpl::signal_connect_lookup_table_page_up (ScimBridgeAgentPanelClientSlotVoid *slot)
{
    return signal_lookup_table_page_up.connect (slot);
}


Connection ScimBridgeAgentPanelClientImpl::signal_connect_lookup_table_page_down (ScimBridgeAgentPanelClientSlotVoid *slot)
{
    return signal_lookup_table_page_down.connect (slot);
}


Connection ScimBridgeAgentPanelClientImpl::signal_connect_trigger_property (ScimBridgeAgentPanelClientSlotString *slot)
{
    return signal_trigger_property.connect (slot);
}


Connection ScimBridgeAgentPanelClientImpl::signal_connect_process_helper_event (ScimBridgeAgentPanelClientSlotStringStringTransaction *slot)
{
    return signal_process_helper_event.connect (slot);
}


Connection ScimBridgeAgentPanelClientImpl::signal_connect_move_preedit_caret (ScimBridgeAgentPanelClientSlotInt *slot)
{
    return signal_move_preedit_caret.connect (slot);
}


Connection ScimBridgeAgentPanelClientImpl::signal_connect_select_candidate (ScimBridgeAgentPanelClientSlotInt *slot)
{
    return signal_select_candidate.connect (slot);
}


Connection ScimBridgeAgentPanelClientImpl::signal_connect_process_key_event (ScimBridgeAgentPanelClientSlotKeyEvent *slot)
{
    return signal_process_key_event.connect (slot);
}


Connection ScimBridgeAgentPanelClientImpl::signal_connect_commit_string (ScimBridgeAgentPanelClientSlotWideString *slot)
{
    return signal_commit_string.connect (slot);
}


Connection ScimBridgeAgentPanelClientImpl::signal_connect_forward_key_event (ScimBridgeAgentPanelClientSlotKeyEvent *slot)
{
    return signal_forward_key_event.connect (slot);
}


Connection ScimBridgeAgentPanelClientImpl::signal_connect_request_help (ScimBridgeAgentPanelClientSlotVoid *slot)
{
    return signal_request_help.connect (slot);
}


Connection ScimBridgeAgentPanelClientImpl::signal_connect_request_factory_menu (ScimBridgeAgentPanelClientSlotVoid *slot)
{
    return signal_request_factory_menu.connect (slot);
}


Connection ScimBridgeAgentPanelClientImpl::signal_connect_change_factory (ScimBridgeAgentPanelClientSlotString *slot)
{
    return signal_change_factory.connect (slot);
}
