/* Copyright (C) 2001-2004 Peter Selinger.
   This file is part of potrace. It is free software and it is covered
   by the GNU general public license. See the file COPYING for details. */

/* $Id: ztops.c,v 1.6 2004/03/06 06:14:28 selinger Exp $ */

/* Note: the code in this file does *not* implement LZW compression.
   It merely converts the ".Z" file format, which is e.g. generated by
   the "compress" program, to a format used by PostScript Level 2. In
   particular, the code contained in this file does not infringe on
   Unisys's U.S. patent 4,558,302. Said patent covers only compression
   and decompression, but not a change of file format on compressed
   data. */

/* The .Z file format.

   The ".Z" file format consists of the three bytes 1f 9d 90, followed
   by a logical bitstream. The logical bitstream is encoded as a
   physical byte stream in the little endian sense, i.e. the first bit
   of the bit stream is encoded as the least significant bit of the
   first byte. At the end, the bitstream is padded with 0 bits to a
   byte boundary.

   The logical bit stream in turns represents a logical stream of
   integers of varying bit length. These integers are represented in
   the little-endian way, i.e., the first bit of the bitstream
   represents the least significant bit of the first integer, etc. In
   this way, the bit stream is interpreted as a sequence of: A1 9-bit
   integers, followed by A2 10-bit integers, followed by A3 11-bit
   integers, followed by A4 12-bit integers. It is possible for a
   stream to encode fewer integers than this, but the maximal number
   is A1+A2+A3+A4. The meaning of these integers is of no particular
   concern here.
   
   The Postscript level 2 compressed stream format

   The Postscript level 2 compressed stream format consists of a
   logical bitstream. This bitstream is encoded as a byte stream in
   the big endian sense, i.e., the first bit of the bit stream is
   encoded as the most significant bit of the first byte. 

   The Postscript level 2 bitstream represents an abstract stream of
   integers of varying number of bits. These integers are represented
   in the big-endian sense, i.e. the first bit of the bit stream is
   the most significant bit of the first integer. 

   The bit stream is interpreted in this way as the 9-bit integer 256,
   followed by one or more blocks of integers, each of which consist
   of at most B1 9-bit integers, followed by B2 10-bit integers,
   followed by B3 11-bit integers, followed by B4 12-bit integers.
   The last integer in each block is 256 or 257; it is 257 in the last
   block, and 256 in every other block. Each block may contain fewer
   than B1+B2+B3+B4 integers. The integers 256 and 257 do not occur
   elsewhere.
   
   Relationship between the two integer streams.

   Divide the Postscript abstract integer stream into blocks by
   removing all occurrences of 256 and 257 and treating them as
   separators. In each block, decrement all the integers that are
   greater than 256 by 1. Now each block represents an integer stream
   in the ".Z" sense. Thus, multiple consecutive ".Z" streams of at
   most min(A1+A2+A3+A4, B1+B2+B3+B4) integers can be represented as a
   single Postscript stream.

   The translation from a sequence of ".Z" streams to a Postscript
   stream thus proceeds in a pipeline, roughly:

   bytestream -> bitstream -> integer stream -> integer stream ->
   bitstream -> bytestream.
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include "ztops.h"

#define A1 0x100
#define A2 0x200
#define A3 0x400
#define A4 0x800

#define B1 0x0FE
#define B2 0x200
#define B3 0x400
#define B4 0x800

static int A[13] = {0,0,0,0,0,0,0,0,0, A1, A1+A2, A1+A2+A3, A1+A2+A3+A4};
static int B[13] = {0,0,0,0,0,0,0,0,0, B1, B1+B2, B1+B2+B3, B1+B2+B3+B4};

/* magic number for .Z files */
static int zheader[] = { 0x1f, 0x9d, 0x90 };

/* prototypes */
static int ztops_writeint(ztops_state *s, int n);

/* ---------------------------------------------------------------------- */
/* all the ztops_* functions return 0 on success, or 1 on
   error with s->error set to an error message */

/* initialize the state */
int ztops_init(ztops_state *s) {
  int r;

  s->inheader = 0;
  s->inbits = 0;
  s->inbuf = 0;
  s->incount = 0;
  s->inwidth = 9;
  s->outcount = 0;
  s->outwidth = 9;
  s->outbits = 0;
  s->outbuf = 0;
  s->eod = 0;
  s->error = NULL;
  r = ztops_writeint(s, 256);
  s->outcount = 0;  /* 256 is not part of first block */
  return r;
}

/* receive a byte from .Z stream */
int ztops_inbyte(ztops_state *s, int b) {
  unsigned int n;

  b &= 0xff;

  /* ignore? */
  if (s->inheader < 3) {
    if (b != zheader[s->inheader]) {
      s->error = "wrong magic for .Z file";
      return 1;
    }
    s->inheader++;
    return 0;
  }
  
  /* add the byte to the bit buffer */
  s->inbuf |= b << s->inbits;
  s->inbits += 8;

  /* can we read an integer? */
  if (s->inbits < s->inwidth) {
    return 0;
  }

  /* read an integer */
  n = s->inbuf & ((1<<s->inwidth)-1);
  s->inbuf >>= s->inwidth;
  s->inbits -= s->inwidth;
  s->incount++;
  if (s->incount >= A[s->inwidth]) {
    s->inwidth++;
    if (s->inwidth > 12) {
      s->error = "input block too long";
      return 1;
    }
  }

  /* transform the integer */
  if (n > 256) {
    n++;
  }

  /* write it to the output */
  return ztops_writeint(s, n);
}

/* start a new .Z stream */
int ztops_newblock(ztops_state *s) {
  int r;

  /* send the last block's terminating 256 */
  r = ztops_writeint(s, 256);
  if (r) {
    return 1;
  }

  /* reset input stream: throw away padding of old stream */
  s->inheader = 0;
  s->inbits = 0;
  s->inbuf = 0;
  s->incount = 0;
  s->inwidth = 9;

  /* reset output stream, but do not pad */
  s->outcount = 0;
  s->outwidth = 9;
  
  return 0;
}

/* finish. After this, the state is invalid, so there is no need to
   update it. */
int ztops_finish(ztops_state *s) {
  int r, b;

  /* send the last block's terminating 257 */
  r = ztops_writeint(s, 257);
  if (r) {
    return 1;
  }

  /* ship the remaining bits, 0-padded */
  while (s->outbits > 0) {
    /* take byte from bit buffer */
    b = (s->outbuf >> (8*sizeof(int)-8)) & 0xff;
    s->outbuf <<= 8;
    s->outbits -= 8;
    
    /* and add it to output buffer */
    if (s->avail_out <= 0) {
      s->error = "output buffer overflow";
      return 1;
    }
    *s->next_out = b;
    s->next_out++;
    s->avail_out--;
  }

  /* make sure nothing else is written */
  s->eod = 1;

  return 0;
}

int ztops_writeint(ztops_state *s, int n) {
  int b;

  /* do not write anything after eod */
  if (s->eod) {
    return 0;
  }

  /* add it to the outgoing bit buffer */
  s->outbuf |= n << (8*sizeof(int) - s->outwidth - s->outbits);
  s->outbits += s->outwidth;

  s->outcount++;
  if (s->outcount >= B[s->outwidth]) {
    s->outwidth++;
    if (s->outwidth > 12) {
      s->error = "output block too long";
      return 1;
    }
  }

  /* while there are enough bits in the buffer, ship them */
  while (s->outbits >= 8) {
    /* take byte from bit buffer */
    b = (s->outbuf >> (8*sizeof(int)-8)) & 0xff;
    s->outbuf <<= 8;
    s->outbits -= 8;

    /* and add it to output buffer */
    if (s->avail_out <= 0) {
      s->error = "output buffer overflow";
      return 1;
    }
    *s->next_out = b;
    s->next_out++;
    s->avail_out--;
  }

  return 0;
}

/* ---------------------------------------------------------------------- */

/* these functions call the "compress" program to compress a
   stream. Call psl2_init, then call psl2_compress with buffers set.
   psl2_compress must be called again iff g->avail_out == 0 after the
   call. Normally, call with mode = PSL2_NORMAL, and at the end,
   *after* all the input characters have been flushed, call with
   PSL2_FINISH, until all the output is flushed. */

int psl2_init(psl2_state *g) {
  int r;

  g->bufsize = 0;
  g->outbufsize = 0;
  g->combufsize = 0;
  g->next_com = g->combuf;
  g->s.next_out = g->combuf;
  g->s.avail_out = 8192;
  r = ztops_init(&g->s);
  if (r) {
    g->error = g->s.error;
    return 1;
  }
  g->combufsize = g->s.next_out - g->combuf;
  return 0;
}

int psl2_compress(psl2_state *g, int mode) {
  char tmpfile[100];
  char command[100];
  char *tmpdir;
  int fd;
  int i, r;
  FILE *f;
  int finish = 0;
  int isize;

  while (1) {

    /* copy input to buffer */
    while (g->avail_in && g->bufsize < 16384) {
      g->buf[g->bufsize] = *g->next_in;
      g->bufsize++;
      g->next_in++;
      g->avail_in--;
    }
    
    /* copy combuf to output */
    while (g->next_com < g->combuf + g->combufsize && g->avail_out) {
      *g->next_out = *g->next_com;
      g->next_com++;
      g->next_out++;
      g->avail_out--;
    }
    
    /* return if nothing to do */
    if (g->next_com < g->combuf + g->combufsize 
	|| (g->bufsize < 16384 && mode != PSL2_FINISH)) {
      return 0;
    }
    
    if (finish) {
      return 0;
    }

    if (g->bufsize < 16384) {
      finish = 1;
    }

    /* write buffer contents to a file */
    tmpdir = getenv("TEMPDIR");
    if (tmpdir == NULL) {
      tmpdir = "/tmp";
    }
    strcpy(tmpfile, tmpdir);
    strcat(tmpfile, "/ztops.XXXXXX");
    fd = mkstemp(tmpfile);
    if (fd < 0) {
      g->error = "error creating temporary file";
      return 1;
    }
    i = 0;
    while (i < g->bufsize) {
      r = write(fd, g->buf+i, g->bufsize-i);
      if (r < 0) {
	close(fd);
	unlink(tmpfile);
	g->error = "error writing to temporary file";
	return 1;
      }
      i += r;
    }
    close(fd);
    
    isize = g->bufsize;

    sprintf(command, "compress < %s", tmpfile);

    while (1) {
      /* call "compress" on the file */
      f = popen(command, "r");
      if (!f) {
	unlink(tmpfile);
	g->error = "error opening pipeline";
	return 1;
      }
      
      /* read the compressed output */
      g->outbufsize = fread(g->outbuf, 1, 16384, f);
      if (ferror(f)) {
	pclose(f);
	unlink(tmpfile);
	g->error = "error receiving data from \'compress\'";
	return 1;
      }
      pclose(f);
      
      /* 5406 is the size of the longest .Z file which can be encoded
	 in a single Postscript level 2 block */
      if (g->outbufsize <= 5406) {
	break;
      }

      /* compressed data is too long - try to compress a smaller chunk
	 of data. */
      isize = 5406 * isize / g->outbufsize - 50;

      r = truncate(tmpfile, isize);
      if (r < 0) {
	unlink(tmpfile);
	g->error = "error while truncating temporary file";
	return 1;
      }
    }

    unlink(tmpfile);

    /* adjust input buffer - move remaining data to the front */
    g->bufsize -= isize;
    memmove(g->buf, g->buf+isize, g->bufsize);

    /* we have successfully transfered some data from buf to outbuf */
    
    /* now translate data from outbuf to combuf */
    g->s.next_out = g->combuf;
    g->s.avail_out = 8192;
    
    for (i=0; i<g->outbufsize; i++) {
      r = ztops_inbyte(&g->s, g->outbuf[i]);
      if (r) {
	g->error = g->s.error;
	return 1;
      }
    }

    if (finish && g->bufsize == 0) {
      r = ztops_finish(&g->s);
    } else {
      r = ztops_newblock(&g->s);
      finish = 0;
    }
    if (r) {
      g->error = g->s.error;
      return 1;
    }
    /* the data is now in combuf */
    g->combufsize = g->s.next_out - g->combuf;
    g->next_com = g->combuf;
    
    /* reset outbuf */
    g->outbufsize = 0;

  } /* while: repeat the whole process */
}

/* ---------------------------------------------------------------------- */

#ifdef MAIN

#define TRY(x) if (x) goto try_error

int main1() {
  int c;
  ztops_state psl2;
  ztops_state *s = &psl2;
  int r;
  char buf[100];

  s->next_out = buf;
  s->avail_out = 100;

  TRY(ztops_init(s));
  if (s->avail_out < ZTOPS_MINOUT) {
    fwrite(buf, 1, s->next_out - buf, stdout);
    s->next_out = buf;
    s->avail_out = 100;
  }
  while (1) {
    c = fgetc(stdin);
    if (c==EOF) {
      break;
    }
    TRY(ztops_inbyte(s, c));
    if (s->avail_out < ZTOPS_MINOUT) {
      fwrite(buf, 1, s->next_out - buf, stdout);
      s->next_out = buf;
      s->avail_out = 100;
    }
  }
  TRY(ztops_finish(s));
  if (s->avail_out < ZTOPS_MINOUT) {
    fwrite(buf, 1, s->next_out - buf, stdout);
    s->next_out = buf;
    s->avail_out = 100;
  }
  return 0;

 try_error:
  fprintf(stderr, "Error: %s\n", s->error); 
  exit(1);
}

#define BUFSIZE 10000

int main() {
  psl2_state psl2;
  psl2_state *g = &psl2;
  
  int ch;
  char inbuf[BUFSIZE];
  char outbuf[BUFSIZE];
  int i, r;

  r = psl2_init(g);
  if (r) {
    goto error;
  }
  while (1) {
    /* fill inbuf */
    for (i=0; i<BUFSIZE; i++) {
      ch = fgetc(stdin);
      if (ch==EOF) {
	break;
      }
      inbuf[i] = ch;
    }

    /* compress */
    g->next_in = inbuf;
    g->avail_in = i;
    do {
      g->next_out = outbuf;
      g->avail_out = BUFSIZE;
      r = psl2_compress(g, PSL2_NORMAL);
      if (r) {
	goto error;
      }
      fwrite(outbuf, 1, BUFSIZE-g->avail_out, stdout);
    } while (g->avail_out == 0);
    if (i<BUFSIZE) {
      break;
    }
  }
  g->avail_in = 0;
  do {
    g->next_out = outbuf;
    g->avail_out = BUFSIZE;
    r = psl2_compress(g, PSL2_FINISH);
    if (r) {
      goto error;
    }
    fwrite(outbuf, 1, BUFSIZE-g->avail_out, stdout);
  } while (g->avail_out == 0);
  fflush(stdout);

  return 0;

 error:
  fprintf(stderr, "psl2: %s\n", g->error);
  return 1;

}
  
#endif
