/*
 * inspect.h - NILFS tool inspector, declares
 *
 * Copyright (C) 2005, 2006 Nippon Telegraph and Telephone Corporation.
 *
 * This file is part of NILFS.
 *
 * NILFS is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * NILFS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with NILFS; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * inspect.h,v 1.7 2006/03/23 06:04:05 ryusuke Exp
 *
 * Written by Amagai Yoshiji <amagai@osrg.net>
 */

#define _XOPEN_SOURCE 600
#define __need_timeval
#define _FILE_OFFSET_BITS 64
#define _LARGEFILE_SOURCE
#define __USE_FILE_OFFSET64

#include <stdio.h>
#include <unistd.h>
#include <linux/unistd.h>
#include <stdlib.h>
#include <stdint.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <ctype.h>
#include <string.h>
#include <strings.h>
#include <asm/types.h>
#include <endian.h>
#include <time.h>
#include <uuid/uuid.h>
#include <malloc.h>
#include <setjmp.h>
#include "../newfs/newfs.h"
#include "../fs/nilfs_types.h"

/* command perse */
#define NARG 16
#define LARG 128

struct ins {
	char *com;
	int   comn;
	int   argn;
};

extern char vcom[NARG][LARG];
extern struct ins *ccom;
extern int gcnarg;
extern struct ins inscom[];

int getcom(FILE *, char *);
extern int gnum(char *, int);

struct nilfs_sb_info;

struct super_block {
	unsigned long s_blocksize;
	struct nilfs_sb_info *sbi;
};

struct nilfs_sb_info {
	int s_rev_level;		/* Revision level */
	int s_minor_rev_level;		/* minor revision level */

	unsigned long s_first_data_block; /* 1st seg disk block number */
	unsigned long s_nsegment;	/* number of segment in filesystem */
        dbn_t s_last_pseg;		/* point to summary block of last
					   partial segment */
	dbn_t s_inode_root;		/* inode root dbn */
	__u32 s_crc_seed;               /* Seed value of CRC32 calculation */
	unsigned long s_blocks_per_segment;
	unsigned short s_mount_state;
	struct super_block *s_super;	/* ->sb */
	struct super_block sb;
};

extern int nilfs_fill_super();
extern dbn_t nilfs_sb_pcopy_blk(struct nilfs_sb_info *, int, int);

extern struct nilfs_super_block superblock;	/* nilfs super block in disk */
extern struct nilfs_sb_info sb_info;	/* mini sb info */
extern struct nilfs_seg_info seg_info;	/* segment info in memory */

extern int blksperseg;	/* blocks per segment */
extern off_t blocksize; /* block size in [byte] */
extern dbn_t c_blk_n;	/* current_block_number */

#define O_DIRECT 040000
extern int readstruct(void *, int, off_t);

extern __u32 crc32c_le(__u32, unsigned char const *, size_t);

void mygetopt(int, char *[]);
void usage();
/* command line option */
extern int sketch_blk;
extern char *sketch_out;

extern int date_in_epoch;

extern char *devname;
extern char *progname;

/**************** copy from nilfs.h ****************/
/*
 * On-memory checkpoint information (inspect version)
 */
struct nilfs_cp_info {
	unsigned int            flags;        /* checkpoint flags */
	unsigned long           nblk_lseg;    /* number of logical segment 
						 in block count */
	unsigned int		cp_sum;
	unsigned int		sketch_size;  /* sketch data size in byte */
	unsigned char		*sketch;      /* sketch data */
	dbn_t                   inode_root;
	unsigned long           inodes_count;
	unsigned long           blocks_count;
};
/*
 * On-memory segment summary
 */
struct nilfs_seg_info {
        unsigned int            nfinfo;       /* number of file information
					         structures */
	unsigned int            nfbinfo;      /* number of fbinfo structures */
        unsigned int            nblocks;      /* number of blocks in this
					         partial segment */
        unsigned int            nblk_sum;     /* number of summary blocks */
        unsigned int            nblk_file;    /* number of file blocks */
        unsigned int            nblk_fbt;     /* number of file B-tree blocks */
        unsigned int            nblk_inode;   /* number of inode blocks */
        unsigned int            nblk_ibt;     /* number of inode B-tree blocks */
	struct timeval		ctime;	      /* creation date */
	unsigned int            flags;
};

static inline unsigned long
nilfs_recalc_segsum_size(struct nilfs_seg_info *ssi, unsigned int blocksize)
{
	unsigned long sum_bytes;

	sum_bytes = sizeof(struct nilfs_seg_summary) +
		sizeof(struct nilfs_finfo) * ssi->nfinfo +
		sizeof(__le64) * ssi->nblk_file +
		sizeof(struct nilfs_fbinfo) * ssi->nfbinfo +
		sizeof(struct nilfs_iinfo) * ssi->nblk_inode;

	ssi->nblk_sum = (sum_bytes - 1) / blocksize + 1;
	return sum_bytes;
}

static inline unsigned long
nilfs_recalc_cp_size(struct nilfs_cp_info *cpi, unsigned int blocksize)
{
	return sizeof(struct nilfs_checkpoint);
}

static inline unsigned long
nilfs_seg_blocks(const struct nilfs_seg_info *ssi, unsigned int nblk_cp)
{
	return ssi->nblk_sum + ssi->nblk_file + ssi->nblk_fbt +
		ssi->nblk_inode + ssi->nblk_ibt + nblk_cp;
}

/**************** copy from nilfs_fs.h ****************/

#define NILFS_PSEG_MIN_BLOCKS  2           /* Minimum number of blocks in
					      a partial segment */

/**************** copy from sb.h ****************/
#define NILFS_SB(sb) (sb)->sbi
#define le64_to_cpu(x) __le64_to_cpu(x)


/**************** copy end ****************/

/* Local Variables:	*/
/* eval: (c-set-style "linux")	*/
/* End:			*/
