/*
 * ibt.c - NILFS B-tree operations.
 *
 * Copyright (C) 2005 Nippon Telegraph and Telephone Corporation.
 *
 * This file is part of NILFS.
 *
 * NILFS is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * NILFS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with NILFS; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * ibt.c,v 1.3 2006/02/27 05:32:27 amagai Exp
 *
 * Written by Koji Sato <koji@osrg.net>
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include "nilfs.h"
#include "ibt.h"

static struct buffer_head *nilfs_ibt_get_node(struct nilfs_btree *btree, dbn_t dbn)
{
	return nilfs_get_inode_node_blk(btree->bt_owner, dbn);
}

static struct buffer_head *nilfs_ibt_get_new_node(struct nilfs_btree *btree)
{
	return nilfs_get_new_inode_node_blk(btree->bt_owner);
}

static void nilfs_ibt_put_node(struct nilfs_btree *btree, struct buffer_head *bh)
{
	nilfs_put_inode_node_blk(bh);
}

static void nilfs_ibt_delete_node(struct nilfs_btree *btree, struct buffer_head *bh)
{
	nilfs_delete_inode_node_blk(bh);
}

static void nilfs_ibt_delete_all_node(struct nilfs_btree *btree)
{
	nilfs_delete_all_inode_node_blk(btree->bt_owner);
}

static void nilfs_ibt_mark_node_dirty(struct nilfs_btree *btree, struct buffer_head *bh)
{
	nilfs_mark_inode_node_blk_dirty(bh);
}

static void nilfs_ibt_mark_node_prepare_dirty(struct nilfs_btree *btree, struct buffer_head *bh)
{
	nilfs_mark_inode_node_blk_prepare_dirty(bh);
}

static int nilfs_ibt_find_unused_key(struct nilfs_btree *btree, nilfs_btree_key_t *rkey)
{
	struct nilfs_ibt_info *ibi;
	nilfs_btree_key_t key;
	int ret;

	ibi = (struct nilfs_ibt_info *)btree->bt_info;
	if ((ret = nilfs_btree_find_unused_key_in_range(btree,
				ibi->i_min_key, ULONG_MAX,
				ibi->i_last_key + 1, &key)) < 0)
		return ret;
	if (rkey != NULL)
		*rkey = key;
	ibi->i_last_key = key;
	return ret;
}

static struct nilfs_btree_operations nilfs_ibt_ops = {
	.bop_get_node			= nilfs_ibt_get_node,
	.bop_get_new_node		= nilfs_ibt_get_new_node,
	.bop_put_node			= nilfs_ibt_put_node,
	.bop_delete_node		= nilfs_ibt_delete_node,
	.bop_delete_all_node		= nilfs_ibt_delete_all_node,
	.bop_mark_node_dirty		= nilfs_ibt_mark_node_dirty,
	.bop_mark_node_prepare_dirty	= nilfs_ibt_mark_node_prepare_dirty,
	.bop_find_unused_key		= nilfs_ibt_find_unused_key,
};

/**
 * nilfs_ibt_init - initialize an inode B-tree
 * @btree: B-tree
 * @dbn: Disk block number of B-tree root
 * @data: B-tree-type-specific data
 *
 * Description: nilfs_ibt_init() initializes the inode B-tree data structure
 * specified by @btree.
 *
 * Return Value: On success, 0 is returned. On error, a following negative
 * error code is returned.
 *
 * %-NILFS_BTREE_ENOMEM - Insufficient amount of memory available.
 */
int nilfs_ibt_init(struct nilfs_btree *btree, dbn_t dbn, void *data)
{
	struct super_block *sb;
	struct nilfs_sb_info *sbi;
	struct nilfs_ibt_info *ibi;

	sb = (struct super_block *)data;
	sbi = NILFS_SB(sb);
	btree->bt_ops = &nilfs_ibt_ops;
	btree->bt_blksize = sb->s_blocksize;
	btree->bt_blkbits = sb->s_blocksize_bits;
	btree->bt_rtree = &sbi->s_inode_ntree;
	btree->bt_rtree_lock = &sbi->s_inode_ntree_lock;
	btree->bt_inactive_list_lock = &sbi->s_inactive_node_pages_lock;
	if ((ibi = kmalloc(sizeof(struct nilfs_ibt_info), GFP_KERNEL)) == NULL) {
		nilfs_btree_clear(btree);
		return -NILFS_BTREE_ENOMEM;
	}
	ibi->i_min_key =
		((sbi->s_first_ino / sbi->s_inodes_per_block) >>
		 (PAGE_CACHE_SHIFT - sb->s_blocksize_bits)) + 1;
	ibi->i_last_key = ibi->i_min_key - 1;
	btree->bt_info = ibi;

	return 0;
}

/* Local Variables:	*/
/* eval: (c-set-style "linux")	*/
/* End:			*/
