/*
 * gc.c - NILFS module and segment garbage collector, allocator
 *
 * Copyright (C) 2005 Nippon Telegraph and Telephone Corporation.
 *
 * This file is part of NILFS.
 *
 * NILFS is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * NILFS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with NILFS; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * gc.c,v 1.26 2006/05/30 16:46:49 ryusuke Exp
 *
 * Written by Amagai Yoshiji <amagai@osrg.net>
 */

#include "nilfs.h"

#define usage_size	sizeof(struct segusage)
#define usageh_size	sizeof(struct segusage_hdr)

/*
 * NILFS disk layout
 */

/*
In following paragraphs,  use these parameters for example.
disk block size: 4096 byte
full segment size: 1024 block == 4 mega byte
disk partition size: 20480 segment == 80 giga byte

[full segment]

Disk partition is divided into fixed sizes.  This management unit is
called a full segment.  Writing out is done sequentially in full
segments.  Size of usable data area of full segments differs depending
upon the segment.  For example, first (zero-th) segment includes a
boot sector and super block, these blocks are reserved for file system
management.  Second (1st) segment includes a replica of the superblock.

|                          disk partition                           |
| 0 full segment | 1 full segment | 2 full segment | ....           |


[partial segment]

Write units.  Dirty buffers are written out as partial segments.  The
partial segment does not exceed the full segment boundaries.  A size of
partial segment is not fixed.


|                    full segment                                       |
| partial segment | partial segment | partial segment | partial segment |


[logical segment]

The partial segment sequence includes inseparable directory
operations.  In the recovery operations, these partial segments are
treated as one inseparable segment.

|        full segment              |        full segment              |
| pseg | pseg | pseg | pseg | pseg | pseg | pseg | pseg | pseg | pseg |
              |         logical segment          |


[segment management block]  struct segusage_hdr {}; struct segusage {};

The block is comprised of maximum 50 chunks for storing segment usage
information, which is used by the garbage collector and segment
allocator.  The position of the block is fixed, so there is one
replica per chunk.  

|                          disk partition                    |
| 0 segment chunk | 1 segment chunk | ... | 49 segment chunk |

The management block consists of one header and segusages.

|                       block                          |
| segusage_hdr | segusage | segusage | segusage | ...  |

A struct segusage is allocated for each full segment.  
Current implementation of NILFS, header is 40 byte and usage is 20 byte.
So, 1 block (4096 byte) can hold one header and 202 usages.

Size of segment chunk is not fixed.  For 20480 full segment, 
chunk 0 consists of 606 full segment  (3 blocks for management block),
chunk 1 consists of 482 full segment  (3 blocks),
chunk 2 - 49 consists of 404 full segment (2 blocks).

So, usable data area of 605th full segment is shorter by 3 blocks.
A replica of the management blocks are placed on half position of the
next segment chunk.  For example, the replica of the first segment chunk
placed on 846th full segment (846 <- 605 + 482/2).

There are maximum one hundred group of the usage tables, it takes
about one second to read all tables, distributed over the whole disk
(10-msec for one head seek operation).

*/

/*
 * NILFS partial segment layout
 */

/*

The partial segment consists of three parts.

|           partial segment                 |
| segment summary | data area | check point |

[segment summary]  struct nilfs_seg_summary {};  (nilfs_fs.h)

The segment summary keeps the block usage information of the partial
segment.  The main contents are checksums of the data area, the
segment summary, the length of the partial segment, and partial
segment creation time.  On the usual data access, the segment summary
is not referred.  It is needed by the garbage collector and file
system recovery process.


[data area]

There are file data blocks, file data B-tree node blocks, inode
blocks, and inode block B-tree node blocks in order.


[check point]  struct nilfs_checkpoint {};  (nilfs_fs.h)

A checkpoint is placed on the last tail of the partial segment.  The
checkpoint includes a checksum of the checkpoint itself. The
checkpoint accuracy means successfully writing the partial segment to
the disk.  The most important information in the checkpoint is the
root block number of the inode block B-tree.  The block number is
written out last, and the whole file system state is updated.

*/

/*
 *  Segment chunk and segment usage table
 *
 *  unsigned long n_usage_b = (blocksize - usageh_size) / usage_size;
 *  long onchunk = (s_nsegment + n_usage_b - 1) / n_usage_b;
 *
 ** if nchunk <= NILFS_MAX_SU_NCHUNK,
 *  let L = onchunk - 1;
 *
 *     0   1   2      L-2 L-1  L
 *   |---|---|---|...|---|---|---|
 *   |<- type F            ->|t G|
 *
 *   type F (chunk 0..L-2): chunk has n_usage_b usage.
 *   type G (chunk L): 2 <= N <= n_usage_b where N: number of usage.
 *
 ** if nchunk <= NILFS_MAX_SU_NCHUNK and s_nsegment % n_usage_b == 1,
 *
 *     0   1   2      L-2 L-1  L
 *   |---|---|---|...|---|---|---|
 *   |<- type F        ->|t H|t I|
 *
 *   type H (chunk L-1): chunk has n_usage_b - 1  usage.
 *   type I (chunk L): chunk has just two usage.
 *
 ** if NILFS_MAX_SU_NCHUNK < onchunk
 *
 * let nb = onchunk / NILFS_MAX_SU_NCHUNK;
 * let nbm = onchunk % NILFS_MAX_SU_NCHUNK;
 * let c as chunk number,
 *
 *    0   1   2          nbm          L
 *  |---|---|---|...|---|---|---|...|---|
 *  |<- type A    ->|t B|<-  Type C   ->|
 *
 *  type A:  0 <= c < nbm - 1
 *  type B:  c == nbm - 1
 *  type C:  nbm <= c <= L
 *    where L = onchunk - 1;
 *
 *  type A: chunk has (nb + 1) * n_usage_b  usage,
 *          usages consists of nb+1 blocks.
 *  type B: chunk has nb * n_usage_b + s  usage,
 *             where s = s_nsegment % n_usage_b;
 *          usages consists of nb+1 blocks.
 *  type C: chunk has nb * n_usage_b  usage,
 *          usages consists of nb blocks.
 *
 ** if NILFS_MAX_SU_NCHUNK < onchunk and nbm == 0
 *
 *   nbm  1   2           L
 *  |---|---|---|...|---|---|
 *  |<- type D        ->|t E|
 *
 *  type D: chunk has nb * n_usage_b  usage,
 *          usages consists of nb blocks.
 *  type E: chunk has (nb - 1) * n_usage_b + s usage,
 *             where s = s_nsegment % n_usage_b;
 *          usages consists of nb blocks.
 */

/**
 * nilfs_segchunk_seg_len - get number of segments of the segment chunk
 * @sbi: on memory super block 
 * @segchunknum: segment chunk number
 *
 * Description: nilfs_segchunk_seg_len returns a number of segments of the segment chunk.
 *
 * Return Value: number of segments of the segment chunk.  On parameter range error,
 * call BUG_ON().
 *
 */

unsigned long
nilfs_segchunk_seg_len(struct nilfs_sb_info *sbi, long segchunknum)
{
	unsigned long n_usage_b = (sbi->s_super->s_blocksize - usageh_size) / usage_size;
	long onchunk = (sbi->s_nsegment + n_usage_b - 1) / n_usage_b;
	long nb, nbm, s;

	BUG_ON(segchunknum < 0 || NILFS_MAX_SU_NCHUNK <= segchunknum);

	if (onchunk <= NILFS_MAX_SU_NCHUNK) {
		BUG_ON(onchunk <= segchunknum);
		if (segchunknum == onchunk - 1) { /* last segment chunk */
			if (sbi->s_nsegment % n_usage_b == 1) /* at least 2 segments */ /* type I */
				return 2;
			else
				return sbi->s_nsegment % n_usage_b;	/* type G */
		} else if (segchunknum == onchunk - 2 && sbi->s_nsegment % n_usage_b == 1)
			return n_usage_b - 1;	/* type H */
		else	/* type F */
			return n_usage_b;
	}

	nb = onchunk / NILFS_MAX_SU_NCHUNK;
	nbm = onchunk % NILFS_MAX_SU_NCHUNK;
	if (segchunknum < nbm - 1) { 	/* type A */
		return (nb + 1) * n_usage_b;
	} else if (segchunknum == nbm - 1) {	/* type B */
		if ((s = sbi->s_nsegment % n_usage_b))
			return nb * n_usage_b + s;
		else
			return (nb + 1) * n_usage_b;
	} else if (segchunknum == NILFS_MAX_SU_NCHUNK - 1 && nbm == 0 && (s = sbi->s_nsegment % n_usage_b)) { /* type E */
		return (nb - 1) * n_usage_b + s;
	} else {	/* type C, D */
		return nb * n_usage_b;
	}
}

/**
 * nilfs_segchunk_seg_num - get segment number of the segment chunk
 * @sbi: on memory super block 
 * @segchunknum: segment chunk number
 *
 * Description: nilfs_segchunk_seg_num returns a first segment number of the segment chunk
 *
 * Return Value: segment number. On parameter range error, call BUG_ON().
 *
 */

unsigned long
nilfs_segchunk_seg_num(struct nilfs_sb_info *sbi, long segchunknum)
{
	unsigned long n_usage_b = (sbi->s_super->s_blocksize - usageh_size) / usage_size;
	long onchunk = (sbi->s_nsegment + n_usage_b - 1) / n_usage_b;
	long nb, nbm;

	BUG_ON(segchunknum < 0 || NILFS_MAX_SU_NCHUNK <= segchunknum);
	if (onchunk <= NILFS_MAX_SU_NCHUNK) {
		BUG_ON(onchunk <= segchunknum);
		if (segchunknum == onchunk - 1 && sbi->s_nsegment % n_usage_b == 1)
			return segchunknum * n_usage_b - 1;	/* type I */
		else	/* type F, G, H */
			return segchunknum * n_usage_b;
	}

	nb = onchunk / NILFS_MAX_SU_NCHUNK;
	nbm = onchunk % NILFS_MAX_SU_NCHUNK;
	if (segchunknum < nbm) {	/* type A, B */
		return segchunknum * (nb + 1) * n_usage_b;
	} else if (nbm == 0) {	/* type D, E */
		return segchunknum * nb * n_usage_b;
	} else {		/* type C */
		return sbi->s_nsegment - nb * n_usage_b * (NILFS_MAX_SU_NCHUNK - segchunknum);
	}
}

/**
 * nilfs_susage_blk_len - get number of blocks for segment usage table of segment chunk
 * @sbi: on memory super block 
 * @segchunknum: segment chunk number
 *
 * Description: nilfs_susage_blk_len returns number of blocks for segment usage table of segment chunk.
 *
 * Return Value: number of blocks.  On parameter range error, call BUG_ON().
 *
 */

unsigned long
nilfs_susage_blk_len(struct nilfs_sb_info *sbi, long segchunknum)
{
	unsigned long n_usage_b = (sbi->s_super->s_blocksize - usageh_size) / usage_size;
	long onchunk = (sbi->s_nsegment + n_usage_b - 1) / n_usage_b;

	BUG_ON(segchunknum < 0 || NILFS_MAX_SU_NCHUNK <= segchunknum);
	if (onchunk <= NILFS_MAX_SU_NCHUNK) /* type F, G, H, I */
		return 1;
	else if (segchunknum < onchunk % NILFS_MAX_SU_NCHUNK) /* type A, B */
		return onchunk / NILFS_MAX_SU_NCHUNK + 1;
	else			/* type C, D, E */
		return onchunk / NILFS_MAX_SU_NCHUNK;
}

/**
 * nilfs_susage_blk_num - get block number of segment usage table header
 * @sbi: on memory super block
 * @secondary: 0 for primary table, 1 for secondary table
 * @segchunknum: chunk number
 *
 * Description: nilfs_susage_blk_num returns a block number of segment usage table header.
 *
 * Return Value: block number of the usage talbe header.
 * On parameter range error, call BUG_ON().
 */

dbn_t
nilfs_susage_blk_num(struct nilfs_sb_info *sbi, int secondary, long segchunknum)
{
	unsigned long n_usage_b = (sbi->s_super->s_blocksize - usageh_size) / usage_size;
	long onchunk = (sbi->s_nsegment + n_usage_b - 1) / n_usage_b;
	long nb, nbm;

	BUG_ON(segchunknum < 0 || NILFS_MAX_SU_NCHUNK <= segchunknum);
	if (secondary) {
		long sseg = segchunknum + 1;
		long o = onchunk < NILFS_MAX_SU_NCHUNK ? onchunk : NILFS_MAX_SU_NCHUNK;

		if (sseg == o)
			sseg = 0;
		return (nilfs_segchunk_seg_num(sbi, sseg) + nilfs_segchunk_seg_len(sbi, sseg) / 2)
			* sbi->s_blocks_per_segment - nilfs_susage_blk_len(sbi, segchunknum);
	}
	if (onchunk <= NILFS_MAX_SU_NCHUNK) {
		BUG_ON(onchunk <= segchunknum);
		if (segchunknum == onchunk - 1)	/* type G, I */
			return sbi->s_blocks_per_segment * sbi->s_nsegment - 1;
		else if (segchunknum == onchunk - 2 && sbi->s_nsegment % n_usage_b == 1) /* last segment chunk has 2 segments */ /* type H */
			return (segchunknum + 1 ) * n_usage_b * sbi->s_blocks_per_segment - sbi->s_blocks_per_segment - 1;
		else	/* type F */
			return (segchunknum + 1) * n_usage_b * sbi->s_blocks_per_segment - 1;
	}
	nb = onchunk / NILFS_MAX_SU_NCHUNK;
	nbm = onchunk % NILFS_MAX_SU_NCHUNK;
	if (segchunknum < nbm - 1) {	/* type A */
		return n_usage_b * (nb + 1) * sbi->s_blocks_per_segment * (segchunknum + 1) - (nb + 1);
	} else if (segchunknum == nbm - 1) {	/* type B */
		return sbi->s_blocks_per_segment * sbi->s_nsegment
			- (n_usage_b * nb * sbi->s_blocks_per_segment * (NILFS_MAX_SU_NCHUNK - segchunknum - 1))
			- (nb + 1);
	} else if (nbm == 0) {
		if (segchunknum == NILFS_MAX_SU_NCHUNK - 1)	/* type E */
			return sbi->s_blocks_per_segment * sbi->s_nsegment - nb;
		else	/* type D */
			return (segchunknum + 1) * n_usage_b * nb * sbi->s_blocks_per_segment - nb;
	} else {	/* type C */
		return sbi->s_blocks_per_segment * sbi->s_nsegment
			- (n_usage_b * nb * sbi->s_blocks_per_segment * (NILFS_MAX_SU_NCHUNK - segchunknum - 1))
			- nb;
	}
}

/**
 * nilfs_seg_segchunk_num - get segment chunk number that includes the segment
 * @sbi: on memory super block 
 * @segnum: segment number
 *
 * Description:  nilfs_seg_segchunk_num returns a segment chunk number that includes the segment
 *
 * Return Value: segment number.  On parameter range error, call BUG_ON().
 *
 */

unsigned long
nilfs_seg_segchunk_num(struct nilfs_sb_info *sbi, segnum_t segnum)
{
	unsigned long n_usage_b = (sbi->s_super->s_blocksize - usageh_size) / usage_size;
	long onchunk = (sbi->s_nsegment + n_usage_b - 1) / n_usage_b;
	long nb, nbm, segchunknum;

	BUG_ON(sbi->s_nsegment <= segnum);
	if (onchunk <= NILFS_MAX_SU_NCHUNK) {
		if (sbi->s_nsegment % n_usage_b == 1 && segnum == sbi->s_nsegment - 2)
			segchunknum = segnum / n_usage_b + 1;	/* type I */
		else
			segchunknum = segnum / n_usage_b; 	/* type F, G, H */
	} else {
		nb = onchunk / NILFS_MAX_SU_NCHUNK;
		nbm = onchunk % NILFS_MAX_SU_NCHUNK;

		if (0 < nbm) {	/* type A, B, C */
			if (sbi->s_nsegment - (NILFS_MAX_SU_NCHUNK - nbm) * nb * n_usage_b <= segnum) {/* type C */
				segchunknum = NILFS_MAX_SU_NCHUNK - (sbi->s_nsegment - segnum - 1) / (nb * n_usage_b) - 1;
			} else if (segnum < (nb + 1) * n_usage_b * (nbm - 1)) {/*  type A */
				segchunknum = segnum / ((nb + 1) * n_usage_b);
			} else	{/* type B */
				segchunknum = nbm - 1;
			}
		} else {	/* type D, E */
			segchunknum = segnum / (nb * n_usage_b);
		}
	}
	return segchunknum;
}

/**
 * nilfs_seg_start_blk - get first block number of the usable data block in the segment
 * @sbi: on memory super block 
 * @segnum: segment number
 *
 * Description: nilfs_seg_start_blk returns a first block number of the usable data block in the segment
 *
 * Return Value: block number.  On parameter range error, call BUG_ON().
 *
 */

dbn_t
nilfs_seg_start_blk(struct nilfs_sb_info *sbi, segnum_t segnum)
{
        if (segnum == 0)
                return sbi->s_first_data_block;
        else if (segnum == 1)	/* super block replica */
                return sbi->s_blocks_per_segment + sbi->s_first_data_block;
        else
                return (dbn_t)segnum * sbi->s_blocks_per_segment;
}

/**
 * nilfs_seg_blklen - get number of blocks of the usable data block in the segment
 * @sbi: on memory super block 
 * @segnum: segment number
 *
 * Description: nilfs_seg_blklen returns a number of blocks of the usable data block in the segment
 *
 * Return Value: number of blocks.  On parameter range error, call BUG_ON().
 *
 */

unsigned int
nilfs_seg_blklen(struct nilfs_sb_info *sbi, segnum_t segnum)
{
	unsigned long segchunknum = nilfs_seg_segchunk_num(sbi, segnum);
	unsigned long chunk_start = nilfs_segchunk_seg_num(sbi, segchunknum);
	unsigned long chunk_len = nilfs_segchunk_seg_len(sbi, segchunknum);
	unsigned long blklen;

	if (segnum == chunk_start + chunk_len - 1) {	/* last segment in the chunk */
		blklen = sbi->s_blocks_per_segment - nilfs_susage_blk_len(sbi, segchunknum);
	} else {
		unsigned long n_usage_b = (sbi->s_super->s_blocksize - usageh_size) / usage_size;
		long onchunk = (sbi->s_nsegment + n_usage_b - 1) / n_usage_b;
		long o = onchunk < NILFS_MAX_SU_NCHUNK ? onchunk : NILFS_MAX_SU_NCHUNK;
		long sseg = segchunknum - 1;

		if (sseg < 0)
			sseg = o - 1;
		if (segnum == chunk_start + chunk_len / 2 - 1) {
			/* secondary usage table */
			blklen = sbi->s_blocks_per_segment - nilfs_susage_blk_len(sbi, sseg);
		} else {
			blklen = sbi->s_blocks_per_segment;
		}
	}
	if (segnum == 0)	/* super block */
		blklen--;
	else if (segnum == 1)
		blklen--;	/* super block replica */
	return blklen;
}

/**
 * nilfs_alloc_segment - allocate an empty full segment.
 * @sbi: on memory super block 
 * @segnum: integer pointer for storing the newly allocated segment.
 * @wait: flag whether permitting blocking or not.
 *
 * Inherent return codes should be defined.
 */
int
nilfs_alloc_segment(struct nilfs_sb_info *sbi, segnum_t *segnum, int wait)
{
	struct the_nilfs *nilfs = sbi->s_nilfs;
	int err = -ENOSPC;

	spin_lock_irq(&nilfs->ns_lock);
	if (nilfs->ns_free_segments_count) {
		*segnum = nilfs->ns_segnum + 1;
		nilfs->ns_free_segments_count--;
		err = 0;
	}
	spin_unlock_irq(&nilfs->ns_lock);
	return err;
}

/**
 * nilfs_find_next_segment - return the number of successive full segment.
 * @sbi: on memory super block 
 * @segnum: number of reference (full) segment
 * @next: integer pointer for storing successor
 */
int
nilfs_find_next_segment(struct nilfs_sb_info *sbi, segnum_t segnum, segnum_t *next)
{
	BUG_ON(segnum >= sbi->s_nsegment);

	if (unlikely(segnum == sbi->s_nsegment - 1))
		return -ESRCH;
	*next = segnum + 1;
	return 0;
}

/**
 * nilfs_find_prev_segment - return the number of previous full segment.
 * @sbi: on memory super block 
 * @segnum: number of reference (full) segment
 * @prev: integer pointer for storing previous one
 */
int
nilfs_find_prev_segment(struct nilfs_sb_info *sbi, segnum_t segnum, segnum_t *prev)
{
	BUG_ON(segnum >= sbi->s_nsegment);

	if (unlikely(segnum == 0))
		return -ESRCH;
	*prev = segnum - 1;
	return 0;
}

/**
 * nilfs_count_free_blocks - return the number of unused full segments.
 * @sbi: on memory super block 
 */
unsigned long
nilfs_count_free_blocks(struct nilfs_sb_info *sbi)
{
	struct the_nilfs *nilfs = sbi->s_nilfs;
	unsigned long segments;

	spin_lock_irq(&nilfs->ns_lock);
	segments = nilfs->ns_free_segments_count;
	spin_unlock_irq(&nilfs->ns_lock);
	return segments * sbi->s_blocks_per_segment;
}

/**
 * nilfs_get_segment_range - get region of a full segment
 * @sbi: on memory super block 
 * @segnum: full segment number
 * @seg_start: pointer to a dbn_t type variable to store start DBN
 * @seg_end: pointer to a dbn_t type variable to store end DBN
 *
 * nilfs_get_segment_range() returns start DBN and end DBN of 
 * the segment specified by @segnum. The start DBN is stored in
 * @seg_start and the end DBN is stored in @seg_end.
 */
void
nilfs_get_segment_range(struct nilfs_sb_info *sbi, segnum_t segnum,
			dbn_t *seg_start, dbn_t *seg_end)
{
        *seg_start = (dbn_t)nilfs_seg_start_blk(sbi, segnum);
	*seg_end = *seg_start + nilfs_seg_blklen(sbi, segnum) - 1;
}

/* Local Variables:	*/
/* eval: (c-set-style "linux")	*/
/* End:			*/
