/*
 * bbt.c - NILFS B-tree operations.
 *
 * Copyright (C) 2005 Nippon Telegraph and Telephone Corporation.
 *
 * This file is part of NILFS.
 *
 * NILFS is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * NILFS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with NILFS; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * bbt.c,v 1.1 2005/09/15 09:47:19 koji Exp
 *
 * Written by Koji Sato <koji@osrg.net>
 */

#include <linux/module.h>
#include "nilfs.h"
#include "bbt.h"

static struct buffer_head *nilfs_bbt_get_node(struct nilfs_btree *btree, dbn_t dbn)
{
	return nilfs_get_file_node_blk(btree->bt_owner, dbn);
}

static struct buffer_head *nilfs_bbt_get_new_node(struct nilfs_btree *btree)
{
	return nilfs_get_new_file_node_blk(btree->bt_owner);
}

static void nilfs_bbt_put_node(struct nilfs_btree *btree, struct buffer_head *bh)
{
	nilfs_put_file_node_blk(bh);
}

static void nilfs_bbt_delete_node(struct nilfs_btree *btree, struct buffer_head *bh)
{
	nilfs_delete_file_node_blk(bh);
}

static void nilfs_bbt_delete_all_node(struct nilfs_btree *btree)
{
	nilfs_delete_all_file_node_blk(btree->bt_owner);
}

static void nilfs_bbt_mark_node_dirty(struct nilfs_btree *btree, struct buffer_head *bh)
{
	nilfs_mark_file_node_blk_dirty(bh);
}

static void nilfs_bbt_mark_node_prepare_dirty(struct nilfs_btree *btree, struct buffer_head *bh)
{
	nilfs_mark_file_node_blk_prepare_dirty(bh);
}

static int nilfs_bbt_find_unused_key(struct nilfs_btree *btree, nilfs_btree_key_t *rkey)
{
	return -NILFS_BTREE_ENOOP;
}

static struct nilfs_btree_operations nilfs_bbt_ops = {
	.bop_get_node			= nilfs_bbt_get_node,
	.bop_get_new_node		= nilfs_bbt_get_new_node,
	.bop_put_node			= nilfs_bbt_put_node,
	.bop_delete_node		= nilfs_bbt_delete_node,
	.bop_delete_all_node		= nilfs_bbt_delete_all_node,
	.bop_mark_node_dirty		= nilfs_bbt_mark_node_dirty,
	.bop_mark_node_prepare_dirty	= nilfs_bbt_mark_node_prepare_dirty,
	.bop_find_unused_key		= nilfs_bbt_find_unused_key,
};

/**
 * nilfs_bbt_init - initialize a block mapping B-tree
 * @btree: B-tree
 * @dbn: Disk block number of B-tree root
 * @data: B-tree-type-specific data
 *
 * Description: nilfs_bbt_init() initializes the block mapping B-tree data
 * structure specified by @btree.
 *
 * Return Value: 0 is always returned.
 */
int nilfs_bbt_init(struct nilfs_btree *btree, dbn_t dbn, void *data)
{
	struct inode *inode;
	struct nilfs_inode_info *ii;

	inode = (struct inode *)data;
	ii = NILFS_I(inode);
	btree->bt_ops = &nilfs_bbt_ops;
	btree->bt_blksize = 1UL << inode->i_blkbits;
	btree->bt_blkbits = inode->i_blkbits;
	btree->bt_rtree = &ii->i_block_ntree;
	btree->bt_rtree_lock = &ii->i_block_ntree_lock;
	btree->bt_inactive_list_lock =
		&NILFS_SB(inode->i_sb)->s_inactive_node_pages_lock;
	btree->bt_info = NULL;

	return 0;
}

/* Local Variables:	*/
/* eval: (c-set-style "linux")	*/
/* End:			*/
