#!/bin/sh
#
# copynilfs - backup tool of NILFS partitions
# Copyright (C) 2006 Nippon Telegraph and Telephone Corporation.
# This file is part of NILFS and is licensed under the GPL.
#
# copynilfs.sh,v 1.9 2006/04/06 02:51:21 ryusuke Exp
#
# Written by Ryusuke Konishi <ryusuke@osrg.net>
#
cmd=`basename $0`

inspect=/sbin/inspect
sharedir=/usr/share/nilfsprogs
filterdir=$sharedir/copynilfs/filter.d
tmpdir=/tmp

show_help() {
    echo "Usage: $cmd [options..] <source device> <destination device>"
    echo "Options:"
#    echo "  -a                append mode"
    echo "  -h                show this help"
    echo "  -f <cp-list>      read first columns of the <cp-list> file as"
    echo "                    selected checkpoints (list of block # of segment)"
    echo "  -F <filter>       specify a filter command that selects checkpoints"
    echo "  -t <nr-blocks>    set threshold value of dirty data blocks"
    echo "                    This value affects granularity of the segments"
    echo "                    to be written on the new device"
    echo "  -X                Do not preserve hard links"
    echo "  -q                quiet mode"
    echo "Newfs options:"
    echo "  -b <block-size>   specify blocksize of the destination device"
    echo "  -B <integer>      specify number of blocks per segment"
    echo "  -L <label>        specify label of new device"
}

function parse_filter() {
    local fcmd=`expr "$1" : "\(\S\+\)"`
    local fargs=`expr "$1" : "\S\+\s\+\(.*\)$"`

    if [ -x "$fcmd" ]; then :
	test `expr substr "$fcmd" 1 1` = '/' || fcmd=`pwd`/"$fcmd"
    elif [ -x "$filterdir/$fcmd" ]; then
	fcmd="$filterdir/$fcmd"
    else
	local fpath=`which $fcmd`
	if [ -z "$fpath" -o ! -x "$fpath" ]; then
	    echo "Not executable filter: $fcmd" 1>&2; exit 1
	fi
    fi
    echo "$fcmd $fargs"
}

quiet=0
copy_devices=1
copy_hardlinks=1
cpfilter=""
unset block_per_seg block_size vlabel

while getopts ahf:F:qb:B:L:t:X o; do
    case $o in
	h)
	    show_help 1>&2
	    exit 0 ;;
	a)
	    echo "Not supported yet"
	    exit 1 ;;
	f)
	    cplist="$OPTARG" ;;
	F)
	    fline=`parse_filter "$OPTARG"` || exit $?
	    cpfilter=${cpfilter:+"$cpfilter | "}"$fline"' $1' ;;
	q)
	    quiet=1 ;;
	b)
	    block_size="$OPTARG" ;;
	B)
	    block_per_seg="$OPTARG" ;;
	L)
	    vlabel="$OPTARG" ;;
	t)
	    threshold="$OPTARG" ;;
	X)
	    copy_hardlinks=0 ;;
	*)
	    show_help 1>&2
	    exit 1 ;;
    esac
done
shift `expr $OPTIND - 1`

# Sub-routines
function sysfs_nilfs_dir() {
    local bdev=`basename $1`
    local hdd=`expr $bdev : "\([[:alnum:]]*[[:alpha:]]\)[0-9]\+$"`

    if [ -z "$hdd" -o ! -d "/sys/block/$hdd/$bdev/nilfs" ]; then
	echo "/sys/block/$bdev/nilfs"
    else
	echo "/sys/block/$hdd/$bdev/nilfs"
    fi
}

# Checks
if [ -z "$1" -o -z "$2" ]; then
    show_help 1>&2; exit 1
elif [ ! -b $1 ]; then
    echo "Not device: $1" 1>&2; exit 1
elif [ ! -b $2 ]; then
    echo "Not device: $2" 1>&2; exit 1
elif [ "$1" == "$2" ]; then
    echo "Identical devices" 1>&2; exit 1
elif [ ! -r $1 ]; then
    echo "Device not readable: $1" 1>&2; exit 1
elif [ ! -r $2 ]; then
    echo "Device not readable: $2" 1>&2; exit 1
elif [ ! -w $2 ]; then
    echo "Device not writable: $2" 1>&2; exit 1
fi

mntdir=`awk '$1 == "'$2'"{print $2}' /etc/mtab`
if [ -n "$mntdir" ]; then
    echo "Device $2 is mounted on $mntdir" 1>&2; exit 1
fi

if [ ! -x $inspect ]; then
    echo "Please install inspect" 1>&2; exit 1;
elif [ ! -x `which rsync` ]; then
    echo "Cannot find rsync command" 1>&2; exit 1;
elif [ -n "$cplist" -a ! -f "$cplist" ]; then
    echo "Cannot find checkpoint list file: $cplist"; exit 1
fi

function getcp() {
    (echo "listcp"; echo "quit") | $inspect -e $1 | sed -ne '/MajorCP/{s/^nilfs> //;p}'
}

echo "$cmd will overwrite all existing data on $2." 1>&2
if ! read -p "Do you proceed [y/N]? " yn || [ "$yn" != "y" -a "$yn" != "Y" ]; then
    exit 0;
fi

# Get superblock information
function getsb() {
    echo "quit" | $inspect -e $1 |
    ( while read h v o; do
	case "$h" in
	    s_ctime) 
		ctime="$v" ;;
	    block_size)
		test "$v" -eq 4096 || : ${block_size:="$v"} ;;
	    s_blocks_per_segment)
		test "$v" -eq 1024 || : ${block_per_seg:="$v"} ;;
	    s_volume_name)
		test -z "$v" || : ${vlabel:="$v"} ;;
#	s_uuid)
#	    : ${uuid:="$v"} ;;
	esac
    done ;
    echo "${ctime:+-P $ctime}${block_size:+ -b $block_size}${block_per_seg:+ -B $block_per_seg}${vlabel:+ -L $vlabel}" )
}

# Main part
function copy_loop() {
    local sk=.sketch a=""

    while read b c t o; do
	test $b -ne 1 || continue
	if [ -n "$cplist" ]; then
	    test -n "$a" || read a o <&3 || break
	    until [ $a -eq $b ]; do
		test $a -gt $b || read a o <&3 || break 2
		continue 2
	    done
	fi
	test $quiet -ne 0 || echo "segblk=$b"
	if mount -t nilfs -r -o cp=$b $1 $sdir; then
	    if [ -n "$sk" -o ! -f "$sdir/$sk" ]; then
		sk=`cd $sdir; find -maxdepth 1 -mount -inum 10 -print | sed -e 's|^\./||'`
	    fi
	    echo "$t" > $sysfs_dir/sc_ctime
	    err=1
	    if rsync -ax $3 --inplace ${sk:+--exclude="$sk"} --del $sdir/ $ddir/; then
		if [ -f "$ddir/$sk" ]; then
		    if [ -f "$tdir/sketch.$b" ]; then
			cp -a "$tdir/sketch.$b" "$ddir/$sk"
		    elif [ -f "$sdir/$sk" ]; then
			cp -a "$sdir/$sk" "$ddir/$sk"
		    fi
		fi
		echo 1 > $sysfs_dir/sync
		err=$?
	    fi
	    umount $sdir
	    if [ $err -ne 0 ]; then
		echo "Failed to write snapshot(cp=$b)" 1>&2; return $err
	    fi
	else
	    echo "Failed to mount $1(cp=$b) on $sdir. skipped" 1>&2
	fi
	a=""
    done
}

function copy_nilfs() {
    local opt=""

    test $copy_hardlinks -eq 0 || opt="-H"
    test $copy_devices -eq 0 -o $UID -ne 0 || opt="$opt -D"

    test $quiet -ne 0 || echo "Copying checkpoints:" 1>&2
    if [ -n "$cplist" ]; then
	copy_loop $1 $2 "$opt" 3< "$cplist"
    else
	copy_loop $1 $2 "$opt"
    fi
    err=$?
    test $err -eq 0 || return $err
    test $quiet -ne 0 || echo "Done." 1>&2
}

function mounted() { test `df -t nilfs $1 | wc -l` -ne 1; }

function abort() { 
    if [ -n "$1" ]; then echo "$1" 1>&2; else echo "Aborting.." 1>&2; fi
    if mounted $sdir; then umount $sdir; fi
    if mounted $ddir; then umount $ddir; fi
    rm -rf $tdir; exit 1
}

newfs_opt=`getsb $1`

tdir=$tmpdir/$cmd.$PPID
sdir=$tdir/src.d
ddir=$tdir/dst.d
mkdir -p $tdir; chmod go-rxw $tdir
mkdir -p $sdir $ddir || abort "Cannot make $srcdir or $dstdir"

trap abort SIGINT SIGTERM

mkfs -t nilfs $newfs_opt $2 || abort "mkfs failed"
mount -t nilfs -o passive=on $2 $ddir || abort "mount failed"
sysfs_dir=`sysfs_nilfs_dir $2`

test -w $sysfs_dir/sc_ctime || abort "Cannot find sc_ctime."
test -w $sysfs_dir/sync || abort "Cannot find sync file."
if [ -n "$threshold" -a -w $sysfs_dir/sc_threshold ]; then
    echo "$threshold" > $sysfs_dir/sc_threshold
fi

if [ -n "$cpfilter" ]; then
    getcp $1 | eval '(cd $tdir; '"$cpfilter"')' | copy_nilfs $1 $2
else
    getcp $1 | copy_nilfs $1 $2
fi
err=$?

umount $ddir
rm -rf $tdir
trap "" SIGINT SIGTERM

exit $err
