/*  Window monitor / application launcher
 *  Copyright (C) 2004 UCHINO Satoshi.  All Rights Reserved.
 *
 *  This is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This software is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 *  USA.
 */

#include "windowmonitor.h"
#include <libxml/parser.h>
#include <libxml/tree.h>

typedef struct launcherItem {
  unsigned long id;   /* id of this item */
  char *commandpath;  /* command path */
  char *menupath;     /* menu path, e.g. "/Internet/Web Browser" */
  struct launcherItem *next;
} LauncherItem;

static LauncherItem *launcherList = NULL;

static int readConfigFile(void);

/* public functions */

int initLauncher(void) {
  return readConfigFile();
}

int launch(int id) {
  LauncherItem *launcher;
  for (launcher = launcherList; launcher; launcher = launcher->next) {
    if (launcher->id == id) {
      DEBUG_PRINTF(("launching id=0x%X, command path='%s', menu path='%s'\n",
                    launcher->id, 
                    launcher->commandpath, launcher->menupath));
      g_spawn_command_line_async(launcher->commandpath, NULL);
    }
  }
  return -1;  /* error */
}

int sendAllLaunchers(void) {
  LauncherItem *launcher;
  for (launcher = launcherList; launcher; launcher = launcher->next)
    SendWindowState(launcher->id, rfbWindowStateLauncher, launcher->menupath);
}

/* private functions */

static void addLauncherList(xmlNode *a_node, char *menuPathBuf);

/* read config file
 * uses some portion from http://xmlsoft.org/examples/tree1.c
 */
static int readConfigFile(void) {
  xmlDoc *doc = NULL;
  xmlNode *root_element = NULL;
  xmlNode *cur_node;
  char menuPathBuf[1024]; /* XXX assume menu path is not so long */

  if (appData.launcherFile == NULL)
    appData.launcherFile = DEFAULT_LAUNCHER_CONFIG_FILE;

  /*
   * this initialize the library and check potential ABI mismatches
   * between the version it was compiled for and the actual shared
   * library used.
   */
  LIBXML_TEST_VERSION

  DEBUG_PRINTF(("parsing config file %s...\n", appData.launcherFile));
  /*parse the file and get the DOM */
  doc = xmlParseFile(appData.launcherFile);
    
  if (doc == NULL) {
    fprintf(stderr, "ERROR: could not parse file %s\n", appData.launcherFile);
    return -1;
  }

  /*Get the root element node */
  root_element = xmlDocGetRootElement(doc);

  /* initialize menu path */
  strcpy(menuPathBuf, MENU_PATH_SEPARATOR);       /* root */

  /* find menu tag */
  for (cur_node = root_element; cur_node; cur_node = cur_node->next) {
    if (cur_node->type == XML_ELEMENT_NODE &&
        strcmp(cur_node->name, MENU_TAG) == 0) {  /* menu tag */
      addLauncherList(cur_node->children, menuPathBuf);
      break;
    }
  }
  if (!cur_node) {
    fprintf(stderr, "ERROR: could not find menu tag in file %s\n", appData.launcherFile);
    return -1;
  }

  /* for debugging */
  if (appData.debug) {
    LauncherItem *launcher;
    for (launcher = launcherList; launcher; launcher = launcher->next) {
      printf("launcher id=0x%X, command path='%s', menu path='%s'\n",
             launcher->id, 
             launcher->commandpath, launcher->menupath);
    }
  }
  return 0;
}

static void getFolderAttr(xmlNode *a_node, char **foldername);
static void getLauncherAttr(xmlNode *a_node, char **name, char **path);
static void assignId(LauncherItem *launcher);

/* walk tree and add items to launcherList */
static void addLauncherList(xmlNode *a_node, char *menuPathBuf) {
  xmlNode *cur_node = NULL;

  for (cur_node = a_node; cur_node; cur_node = cur_node->next) {
    if (cur_node->type != XML_ELEMENT_NODE)
      continue;
    if (strcmp(cur_node->name, FOLDER_TAG) == 0) {  /* folder tag */
      char *foldername, *prevPathEnd;
      getFolderAttr(cur_node, &foldername);
      if (!foldername)
        continue;

      prevPathEnd = menuPathBuf + strlen(menuPathBuf);
      /* append foldername to menu path */
      strcat(menuPathBuf, foldername);
      strcat(menuPathBuf, MENU_PATH_SEPARATOR);
      /* walk tree */
      addLauncherList(cur_node->children, menuPathBuf);
      /* remove foldername from menu path */
      *prevPathEnd = '\0';
    } else if (strcmp(cur_node->name, LAUNCHER_TAG) == 0) {
      char *name, *path;
      LauncherItem *newLauncher;
      getLauncherAttr(cur_node, &name, &path);
      if (!(name && path))
        continue;

      /* create new Launcher item structure */
      newLauncher = (LauncherItem *)malloc(sizeof(LauncherItem));
      if (!newLauncher) {
        fprintf(stderr, "ERROR: memory allocation error\n");
        return;
      }

      newLauncher->commandpath = (char *)malloc(strlen(path) + 1);
      newLauncher->menupath = (char *)malloc(strlen(name) + strlen(menuPathBuf) + 1);
      if (!newLauncher->commandpath || !newLauncher->menupath) {
        fprintf(stderr, "ERROR: memory allocation error\n");
        return;
      }
      strcpy(newLauncher->commandpath, path);
      strcpy(newLauncher->menupath, menuPathBuf);
      strcat(newLauncher->menupath, name);

      assignId(newLauncher);
      /* add newLauncher to launcherList */
      newLauncher->next = launcherList;
      launcherList = newLauncher;
    }
  }
}

static void getFolderAttr(xmlNode *a_node, char **foldername) {
  struct _xmlAttr *attr = a_node->properties;
  *foldername = NULL;
  for (; attr; attr = attr->next) {
    if (!attr->type == XML_ATTRIBUTE_NODE)
      continue;
    if (strcmp(attr->name, FOLDER_NAME_ATTR) == 0) {
      xmlNode *value = attr->children;
      for (; value; value = value->next) {
        if (value->type = XML_TEXT_NODE) {
          *foldername = value->content;
          break;      /* use first one */
        }
      }
    }
  }
  if (!*foldername) {
    fprintf(stderr, "ERROR: could not find folder name\n");
    return;
  }
}

static void getLauncherAttr(xmlNode *a_node, char **name, char **path) {
  struct _xmlAttr *attr = a_node->properties;
  *name = NULL;
  *path = NULL;
  for (; attr; attr = attr->next) {
    if (!attr->type == XML_ATTRIBUTE_NODE)
      continue;
    if (strcmp(attr->name, LAUNCHER_NAME_ATTR) == 0) {
      xmlNode *value = attr->children;
      for (; value; value = value->next) {
        if (value->type = XML_TEXT_NODE) {
          *name = value->content;
          break;      /* use first one */
        }
      }
    }
    if (strcmp(attr->name, LAUNCHER_PATH_ATTR) == 0) {
      xmlNode *value = attr->children;
      for (; value; value = value->next) {
        if (value->type = XML_TEXT_NODE) {
          *path = value->content;
          break;      /* use first one */
        }
      }
    }
  }
  if (!*name) {
    fprintf(stderr, "ERROR: could not find launcher name\n");
    return;
  }
  if (!*path) {
    fprintf(stderr, "ERROR: could not find command path\n");
    return;
  }
}

/* assign id */
static void assignId(LauncherItem *launcher) {
  static unsigned long id = 1;
  launcher->id = id++;
}
