/* ***** BEGIN LICENSE BLOCK *****
 * Source last modified: $Id: minifmt.cpp,v 1.8.4.2 2004/07/09 01:59:28 hubbe Exp $
 * 
 * Portions Copyright (c) 1995-2004 RealNetworks, Inc. All Rights Reserved.
 * 
 * The contents of this file, and the files included with this file,
 * are subject to the current version of the RealNetworks Public
 * Source License (the "RPSL") available at
 * http://www.helixcommunity.org/content/rpsl unless you have licensed
 * the file under the current version of the RealNetworks Community
 * Source License (the "RCSL") available at
 * http://www.helixcommunity.org/content/rcsl, in which case the RCSL
 * will apply. You may also obtain the license terms directly from
 * RealNetworks.  You may not use this file except in compliance with
 * the RPSL or, if you have a valid RCSL with RealNetworks applicable
 * to this file, the RCSL.  Please see the applicable RPSL or RCSL for
 * the rights, obligations and limitations governing use of the
 * contents of the file.
 * 
 * Alternatively, the contents of this file may be used under the
 * terms of the GNU General Public License Version 2 or later (the
 * "GPL") in which case the provisions of the GPL are applicable
 * instead of those above. If you wish to allow use of your version of
 * this file only under the terms of the GPL, and not to allow others
 * to use your version of this file under the terms of either the RPSL
 * or RCSL, indicate your decision by deleting the provisions above
 * and replace them with the notice and other provisions required by
 * the GPL. If you do not delete the provisions above, a recipient may
 * use your version of this file under the terms of any one of the
 * RPSL, the RCSL or the GPL.
 * 
 * This file is part of the Helix DNA Technology. RealNetworks is the
 * developer of the Original Code and owns the copyrights in the
 * portions it created.
 * 
 * This file, and the files included with this file, is distributed
 * and made available on an 'AS IS' basis, WITHOUT WARRANTY OF ANY
 * KIND, EITHER EXPRESS OR IMPLIED, AND REALNETWORKS HEREBY DISCLAIMS
 * ALL SUCH WARRANTIES, INCLUDING WITHOUT LIMITATION, ANY WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, QUIET
 * ENJOYMENT OR NON-INFRINGEMENT.
 * 
 * Technology Compatibility Kit Test Suite(s) Location:
 *    http://www.helixcommunity.org/content/tck
 * 
 * Contributor(s):
 * 
 * ***** END LICENSE BLOCK ***** */

#include "hxcom.h"
#include "hlxclib/string.h"
#include "hxtypes.h"
#include "hxwintyp.h"
#include "hxvsurf.h"
#include "ciddefs.h"


#include "minifmt.h"

#define _FOURCC         1
#define _BITCOUNT       2
#define _BITMASK        4


CFmtObj::CFmtObj()
 :  m_pFmtList(NULL)
{
    CreateFmtList();
}

CFmtObj::~CFmtObj()
{
    DestroyFmtList();
}

int CFmtObj::MakeBitmap(HXBitmapInfo* lpbi,
                        int nBISize, int cid,
                        ULONG32 dwWidth, ULONG32 dwHeight,
                        void* lppe, int nColors)
{
    int setPalette, bitmapinfoSize;

    // check input parameters
    if (!lpbi ||
       (int)dwWidth <= 0 ||  (int)dwHeight <= 0 ||
        !m_pFmtList)
        return 0;

    memset( lpbi, 0, sizeof(*lpbi) );
    
    // find entry
    FmtEntry* p = FindEntry(cid);
    if (!p)
        return 0;

    // calculate bitmapinfo size
    setPalette = 0;
    bitmapinfoSize = sizeof(HXBitmapInfoHeader);
    if (p->dwFlags & _BITMASK)
    {
        bitmapinfoSize += 3 * sizeof(ULONG32);
    }
    else if ((p->dwFlags & (_FOURCC|_BITCOUNT)) == (_FOURCC|_BITCOUNT) &&
              p->dwFourCC == BI_RGB && p->dwBitCount <= 8 &&
              nColors)
    {
        // check palette parameters
        if (lppe == NULL || nColors < 0 || nColors > 256)
            return 0;
#ifdef _WIN32        
        bitmapinfoSize += nColors * sizeof(PALETTEENTRY);
        setPalette = 1;
#endif        
    }

    // check if we have sufficient amount of memory
    if (nBISize < bitmapinfoSize)
        return 0;

    // initialize bitmapinfo structure
    memset(lpbi, 0, bitmapinfoSize);
    lpbi->bmiHeader.biSize = sizeof(HXBitmapInfoHeader);

    // set image parameters
    lpbi->bmiHeader.biWidth = dwWidth;
    lpbi->bmiHeader.biHeight = dwHeight;
    lpbi->bmiHeader.biPlanes = 1;
    lpbi->bmiHeader.biSizeImage = ImageSize (cid, dwWidth, dwHeight);

    // set color format
    SetBitmapColor (lpbi, cid);

#ifdef _WIN32
    /* set palette: */
//    if (setPalette)
//        SetBitmapPalette (lpbi, lppe, nColors);
#endif
    
    return bitmapinfoSize;  /* the number of bytes written */
}

int CFmtObj::SetBitmapColor(HXBitmapInfo* lpHXbi, int cid)
{
    // check input parameters
    if (!lpHXbi || !m_pFmtList)
        return -1;

    // find entry
    FmtEntry* p = FindEntry(cid);
    if (!p)
        return -1;

    // set color-format releated fields in BITMAPINFO
    lpHXbi->bmiHeader.biCompression = p->dwFourCC;
    lpHXbi->bmiHeader.biBitCount    = (UINT16)p->dwBitCount;

    // set color masks
    if (p->dwFlags & _BITMASK)
    {
        // get offset of color masks in the bitmap
        ULONG32 *lpColors = (ULONG32*)&lpHXbi->un.dwBitMask[0];
        lpColors[0] = p->dwBitMask[0];
        lpColors[1] = p->dwBitMask[1];
        lpColors[2] = p->dwBitMask[2];
    }

    return 0;
}

int CFmtObj::ImageSize (int cid, ULONG32 dwWidth, ULONG32 dwHeight)
{
    FmtEntry* p = FindEntry(cid);
    if (!p)
        return 0;
    
    int pitch = dwWidth * p->dwBitCount / 8;

    if (p->dwFlags & _BITCOUNT)
        pitch = (pitch + 3) & ~3;

    return dwHeight * pitch;
}


int CFmtObj::GetBitmapPitch (HXBitmapInfo* lpbi)
{
    register int cid, pitch;

    // check bitmap pointer & format
    cid = GetBitmapColor (lpbi);
    
    if (cid == -1)
        return 0;

    if (cid == CID_XING)
        return 768;
    
    // find entry
    FmtEntry* p = FindEntry(cid);
    if (!p)
        return 0;
    
    // calculate image pitch
    pitch = lpbi->bmiHeader.biWidth * p->nBPP;
    
    if (cid == BI_RGB || cid == BI_BITFIELDS)
#if defined(_MACINTOSH) || defined(_UNIX)
        pitch = ((pitch + 3) & ~3);
#else
        pitch = -((pitch + 3) & ~3);
#endif

    // return pitch
    return pitch;
}

int CFmtObj::GetBitmapColor(HXBitmapInfo* lpHXbi)
{
    int cid = -1;

    // check bitmap pointer/size
    if (lpHXbi == NULL)
        return CID_UNKNOWN;

    // get bitmap parameters to compare
    for (FmtEntry* p=m_pFmtList; p; p=p->pNext)
    {
        if (ChkColor (p, 
                      lpHXbi->bmiHeader.biCompression,
                      lpHXbi->bmiHeader.biBitCount, 
                      (ULONG32*)&lpHXbi->un.dwBitMask[0]))
        {
            cid = p->cid;
            break;
        }
    }
    
    return cid;
}

int CFmtObj::ChkColor (FmtEntry* p,
                       ULONG32 dwFourCC,
                       ULONG32 dwBitCount,
                       ULONG32 *lpBitMask)
{
    // clear match flags
    ULONG32 dwMatch = 0;

    // check FourCC
    if ((p->dwFlags & _FOURCC) && p->dwFourCC == dwFourCC)
        dwMatch |= _FOURCC;

    // check BitCount
    if ((p->dwFlags & _BITCOUNT) && p->dwBitCount == dwBitCount)
        dwMatch |= _BITCOUNT;

    // check BitMasks
    if ((p->dwFlags & _BITMASK) && lpBitMask &&
         p->dwBitMask[0] == lpBitMask[0] &&
         p->dwBitMask[1] == lpBitMask[1] &&
         p->dwBitMask[2] == lpBitMask[2])
         dwMatch |= _BITMASK;

    // combine results
    return dwMatch == p->dwFlags;
}

void CFmtObj::CreateFmtList()
{
    CreateFmtEntry( _FOURCC, MAKEFOURCC('I','4','2','0'),    CID_I420, 12, 1);
    CreateFmtEntry( _FOURCC, MAKEFOURCC('Y', 'V', '1', '2'), CID_YV12, 12, 1);
    CreateFmtEntry( _FOURCC, MAKEFOURCC('Y','U','Y','2'),    CID_YUY2, 16, 2);
    CreateFmtEntry( _FOURCC, MAKEFOURCC('U', 'Y', 'V', 'Y'), CID_UYVY, 16, 2);
    
    CreateFmtEntry( _FOURCC+_BITCOUNT, BI_RGB,  CID_RGB32,   32, 4);
    CreateFmtEntry( _FOURCC+_BITCOUNT, HX_ARGB, CID_ARGB32,  32, 4); 
    CreateFmtEntry( _FOURCC+_BITCOUNT, BI_RGB,  CID_RGB24,   24, 3);
    CreateFmtEntry( _FOURCC+_BITCOUNT, BI_RGB,  CID_RGB444,  12, 2);
    
    CreateFmtEntry( _FOURCC+_BITCOUNT+_BITMASK,BI_BITFIELDS, CID_RGB565, 16, 2, 0xF800,0x07E0,0x001F);
    CreateFmtEntry( _FOURCC+_BITCOUNT+_BITMASK,BI_BITFIELDS, CID_RGB555, 16, 2, 0x7C00,0x03E0,0x001F);
    CreateFmtEntry( _FOURCC+_BITCOUNT+_BITMASK,BI_BITFIELDS, CID_RGB444, 16, 2, 0x0F00,0x00F0,0x000F);
        
}

void CFmtObj::DestroyFmtList()
{
    if (!m_pFmtList)
        return;

    for (FmtEntry* p=m_pFmtList; p; m_pFmtList=p)
    {
        p = p->pNext;
        delete m_pFmtList;
    }
}


FmtEntry* CFmtObj::CreateFmtEntry( ULONG32 dwFlags,
                                   ULONG32 dwFourCC,
                                   int     cid,
                                   ULONG32 dwBitCount,
                                   ULONG32 nBPP,
                                   ULONG32 dwBitMask1,
                                   ULONG32 dwBitMask2,
                                   ULONG32 dwBitMask3
                                   )
 {
    FmtEntry* pEntry = new FmtEntry;
    
    if (pEntry)
    {
        memset(pEntry, 0, sizeof(*pEntry));

        if (!m_pFmtList)
            m_pFmtList = pEntry;
        else
        {
            FmtEntry* p;
            for (p=m_pFmtList; p->pNext; p=p->pNext)
                ;
            p->pNext = pEntry;
        }
    }

    pEntry->dwFlags      = dwFlags;
    pEntry->dwFourCC     = dwFourCC;
    pEntry->cid          = cid;
    pEntry->dwBitCount   = dwBitCount;
    pEntry->nBPP         = nBPP;
    pEntry->dwBitMask[0] = dwBitMask1;
    pEntry->dwBitMask[1] = dwBitMask2;
    pEntry->dwBitMask[2] = dwBitMask3;
        
    return pEntry;
}

FmtEntry* CFmtObj::FindEntry(int cid)
{
    FmtEntry* p;

    for (p=m_pFmtList; p && p->cid != cid; p=p->pNext)
        ;
    
    return p;
}
