/*
 * Diff text map header
 *
 * This module provides a mapping between internal buffer(mmap'd buffer)
 * and text widget(GtkText).
 * In gtkdiff, the operations are almost driven by the internal buffer.
 * On the other hand, UI level has text widget, and its content is not same
 * as the internal buffer.
 * Thus, I have to map a line number between (mmap'd)buffer and text(widget).
 * Through this module, I'm doing the mapping.
 *
 * Naming convention:
 * buf: internal buffer(mmap'd buffer)
 * text: text widget(GtkText's buffer)
 *
 * Copyright INOUE Seiichiro <inoue@ainet.or.jp>, licensed under the GPL.
 */
#ifndef __GDIFF_DTEXTMAP_H__
#define __GDIFF_DTEXTMAP_H__

#include "mbuffer.h"

#ifdef __cplusplus 
extern "C" {
#endif /* __cplusplus */

/* Constant numbers */
/* Forward declarations of types. */
typedef struct _DispLines DispLines;

/* DispLines instance has the following bit-wised attributes. */
typedef enum {
	DA_COMMON =	(1<<0),		/* Common parts between two files */
	DA_CHANGE =	(1<<1),		/* Changed parts */
	DA_CHANGE_O =	(1<<2),	/* Changed parts of the other file */
	DA_ONLY =	(1<<3),		/* Only parts */
	DA_O_ONLY =	(1<<4),		/* If the other file has only parts related to this part of base file.
							   This is normally displayed. */
	DA_ONLY_O =	(1<<5),		/* Only parts of the other file */
	DA_FILL =	(1<<6),		/* Fill parts. used in multipane-widget */
	DA_MARK = 	(1<<7),		/* Mark. used in merge-widget */

	DA_CHANGE_O2 =	(1<<8),	/* Changed parts of the other2 file */
	DA_O2_ONLY =	(1<<9),	/* If the other2 file has only parts related to this part of base file.
							   This is normally displayed. */
	DA_ONLY_O2 =	(1<<10),/* Only parts of the other2 file */

	/* the following two attributes are kludge */
	DA_O_ONLY_TOP =	(1<<16),/* gtkdiff suffers from virtual zero'th line.
							   This is the virtual zero'th line. */
#if 0	
	DA_O2_ONLY_TOP = (1<<17),/* gtkdiff suffers from virtual zero'th line.
								This is the virtual zero'th line. */
#endif	
	DA_HIDE =	(1<<18),	/* Hidden parts. used in merge-widget */
	
	DA_DIFF =	(DA_CHANGE | DA_ONLY),
	DA_BASE =	(DA_COMMON | DA_DIFF | DA_O_ONLY | DA_O2_ONLY),
	DA_OTHER =	(DA_CHANGE_O | DA_ONLY_O),
	DA_OTHER2 =	(DA_CHANGE_O2 | DA_ONLY_O2),

	DA_ANY = 	~0
} DispAttr;

/* 
 * Data structure definitions
 */
/* Buffer-Text mapper data structure.
 * It manages a linked list of DispLines.
 * And for performance, it keeps the current node and last node.
 */
struct _DTextMap {
	/* private */
	int total_nl;		/* total number of lines in text */
	GList *displn_list;	/* Doubly linked list of DispLines */
	GList *cur_node;	/* for performance */	
	GList *last_node;	/* for performance */

	gboolean b_has_o_only_top;	/* Ugly hack. TRUE if there is DA_O_ONLY_TOP node */
};

/* Displayed lines data structure.
 * If displayed lines has a related buffer, it keeps the buffer by mbuf.
 * mbuf doesn't allocate a memory by itself, so it is lighter than it looks. 
 */
struct _DispLines {
	MBuffer mbuf; /* DispLines is a derived type from MBuffer. */

	DispAttr attr;
	int pos;	/* position in text */
	int len;	/* length in text (character-length<GtkText spec>) */
};	


/* Global function declarations */
extern DTextMap* dtmap_new(int buf_nl);
extern void dtmap_delete(DTextMap *dtmap);

extern int dtmap_map_b2t(const DTextMap *dtmap, int bufln);
extern int dtmap_map_t2b(const DTextMap *dtmap, int text_ln);
extern int dtmap_bufln_by_displn(const DTextMap *dtmap, const DispLines *displn);

extern DispLines* dtmap_append_displn(DTextMap *dtmap, DispAttr attr, int pos, int len, const char *buf, int buf_lenb, int buf_nl);
extern DispLines* dtmap_insert_displn(DTextMap *dtmap, DispAttr attr, int pos, int len, const char *buf, int buf_lenb, int buf_nl);
extern gboolean dtmap_remove_displn(DTextMap *dtmap, DispLines *displn);

extern DispLines* dtmap_first_displn(DTextMap *dtmap, DispAttr attr);
extern DispLines* dtmap_next_displn(DTextMap *dtmap, DispAttr attr);
extern DispLines* dtmap_lookup_by_bufln(DTextMap *dtmap, int buf_ln);
extern DispLines* dtmap_lookup_next(DTextMap *dtmap, DispLines *displn);
extern DispLines* dtmap_lookup_prev(DTextMap *dtmap, DispLines *displn);

extern void dtmap_inc_displn(DTextMap *dtmap, DispLines *displn, int i_len);
extern void dtmap_dec_displn(DTextMap *dtmap, DispLines *displn, int i_len);

extern void dtmap_show_displn(DTextMap *dtmap, DispLines *displn);
extern void dtmap_hide_displn(DTextMap *dtmap, DispLines *displn);

#ifdef __cplusplus
} 
#endif /* __cplusplus */

#endif /* __GDIFF_DTEXTMAP_H__ */
