/*
 * Copyright (C)2003 USAGI/WIDE Project
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
/*
 * Authors:
 *      Noriaki TAKAMIYA @USAGI
 *	Masahide NAKAMURA @USAGI
 */

#include <assert.h>
#include <errno.h>
#include <signal.h>
#include <stdio.h>
/*#define _GNU_SOURCE*/ /* strsignal */
#define __USE_GNU /* strsignal */
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/select.h>
#include <asm/types.h>
#include <netinet/ip6.h>
#include <netinet/icmp6.h>
#include <linux/capability.h>
#include <linux/netlink.h>
#include <linux/xfrm.h>


#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <netinet/icmp6.h>
#include <net/if.h>

#include <mip6.h>
#include <mip6log.h>

#include "util.h"
#include "mip6d.h"

#define BUFLEN (8192)


static inline int nlroute_input(struct mip6d_handle *handle,
				void *buf, ssize_t len)
{
	__dprintf("called\n");
	return 0;
}

static inline int nlxfrm_input(struct mip6d_handle *handle,
			       void *buf, ssize_t len)
{
	__dprintf("called\n");
	return 0;
}

#include <netinet/icmp6.h>
static inline int icmp_input(struct mip6d_handle *handle,
			     struct mip6_msg_info *msg_info,
			     struct icmp6_hdr *icmp, ssize_t len)
{
	__dprintf("called\n");
	return handle->icmp_input(handle, msg_info, icmp, len);
}

extern int mh_input(struct mip6d_handle *handle,
		    struct mip6_msg_info *msg_info,
		    struct mip6_mh_hdr *mh, int len);

/*
 * returns:
 *   <0 error occured
 *    0 hao not found
 *   >0 hao found
 */
static int dstopt_input(void *msg, struct in6_addr *hoa)
{
	struct ip6_dest *hdr = (struct ip6_dest *)msg;
	int offset = 0;
	int len = (hdr->ip6d_len+1) << 3;
	int ret = 0;

	__dprintf("TLV parsing\n");

	while (len > 0) {
		void *tlv = (hdr + 1 + offset);
		unsigned char opt_type = *(unsigned char *)tlv;
		unsigned char opt_len;
		switch (opt_type) {
		case IPV6_TLV_PAD0:
			opt_len = 1;
			__dprintf("PAD0 received\n");
			break;
		case IPV6_TLV_PADN:
			opt_len = *(unsigned char *)(tlv + 1);
			__dprintf("PADN received: len=%d\n", opt_len);
			break;
		case IPV6_TLV_HAO:
		{
			struct destopt_hao *hao = (struct destopt_hao *)tlv;
			opt_len = hao->length;

			/* HAO's address is CoA which has been already replaced
			 * when receiving in kernel.
			 */
			__dprintf("HAO received: addr(must be CoA)=%s\n", in6_straddr(&hao->addr));

			if (ret > 0) {
				/* XXX: hao found more than one time */
			}

			ipv6_addr_copy(hoa, &hao->addr);

			ret = 1;
			break;
		}
		default:
			/* Currently unknown type is treated as error here.
			 * Strictly, it is required to check type as
			 * RFC2460(Section 4.2) says.
			 * (Of course, kernel can be implemented to do it.
			 * If kernel does so, it is error here.)
			 */

			__eprintf("unknown TLV type:%d (just treat error)\n", opt_type);

			ret = -1;
			break;
		}

		if (ret < 0)
			break;

		offset += opt_len;
		len -= opt_len;
	}

	__dprintf("TLV parsed\n");
	return ret;
}

static int pkt_recv(int sock,
		    struct mip6_msg_info *msg_info,
		    void *buf, unsigned int bufsize, int *len)
{
	struct msghdr	msg;
	struct cmsghdr	*cmsg;
	struct iovec	iov;
	unsigned char	cbuf[BUFLEN];

	memset(&msg, 0, sizeof(msg));
	memset(cbuf, 0, sizeof(cbuf));

	msg.msg_name = &msg_info->from_addr;
	msg.msg_namelen = sizeof(msg_info->from_addr);
	msg.msg_iov = &iov;
	msg.msg_iovlen = 1;
	msg.msg_control = cbuf;
	msg.msg_controllen = sizeof(cbuf);

	iov.iov_base = buf;
	iov.iov_len = bufsize;

	*len = recvmsg(sock, &msg, 0);
	if (*len < 0) {
		__perror("recvmsg");
		return -1;
	}

	for (cmsg = CMSG_FIRSTHDR(&msg);
	     cmsg != NULL;
	     cmsg = CMSG_NXTHDR(&msg, cmsg)) {

		if (cmsg->cmsg_level != IPPROTO_IPV6) {
			__eprintf("not IPPROTO_IPV6, discarded.\n");
			continue;
		}

		switch(cmsg->cmsg_type) {
		case IPV6_RECVPKTINFO:
			msg_info->pktinfo = (struct in6_pktinfo *)CMSG_DATA(cmsg);
			msg_info->msg_type |= cmsg->cmsg_type;
			{
				char buf[100];
				printf("%s:daddr = %s\n", __FUNCTION__, inet_ntop(PF_INET6, &msg_info->pktinfo->ipi6_addr, buf, 100));
			}
			break;
		case IPV6_RECVDSTOPTS:
		{
			int ret;
			struct in6_addr hoa;

			__dprintf("DSTOPT received\n");

			memset(&hoa, 0, sizeof(hoa));

			ret = dstopt_input(CMSG_DATA(cmsg), &hoa);
			if (ret < 0) {
				__eprintf("DSTOPT error\n");
				break;
			} else if(ret == 0) {
				__eprintf("DSTOPT error: no HAO\n");
				break;
			}

			/* kernel replaced address, so re-replaced here */
			ipv6_addr_copy(&msg_info->dstopt_hoa, &msg_info->from_addr.sin6_addr);
			ipv6_addr_copy(&msg_info->from_addr.sin6_addr, &hoa);

			msg_info->msg_type |= cmsg->cmsg_type;

			__dprintf("DSTOPT ok\n");
			break;
		}
		case IPV6_RECVRTHDR:
		{
			struct rt2_hdr {
				struct ip6_rthdr	rthdr;
				__u32			reserved;
				struct in6_addr		addr;
			} *rthdr = (struct rt2_hdr *)CMSG_DATA(cmsg);

			__dprintf("RT received\n");

			/* ignore other than Type 2 */
			if (rthdr->rthdr.ip6r_type != 2) {
				__dprintf("not type 2, so ignored)\n");
				break;
			}

			__dprintf("RT2 received: addr=%s\n", in6_straddr(&rthdr->addr));
			/* XXX: */
			ipv6_addr_copy(&msg_info->rthdr_hoa, &rthdr->addr);
			msg_info->msg_type |= cmsg->cmsg_type;

			__dprintf("RT2 ok\n");
			break;
		}
		default:
			__eprintf("unexpected cmsg type = %d\n", cmsg->cmsg_type);
			msg_info->msg_type |= cmsg->cmsg_type;
			break;
		}
	}

	return 1;
}

static int observe(struct mip6d_handle *handle)
{
	int ret;

	ret = handle->observe(handle);
	if (ret)
		return ret;

#if 0
	/* update Binding Cache (checking lifetime) */
	mip6_bc_update();
#endif
	return 0;
}

static int loop(struct mip6d_handle *handle)
{
	fd_set fds_def;
	/*
	 * handle timeval to know real-passed-time.
	 * (only for Linux. see select(2))
	 */
	struct timeval tv_def;
	struct timeval tv;
	int maxsock = 0;
	int ret = 0;

	FD_ZERO(&fds_def);
	FD_SET(handle->nlroute_sock, &fds_def);
	FD_SET(handle->nlxfrm_sock, &fds_def);
	FD_SET(handle->mh_sock, &fds_def);
	FD_SET(handle->icmp_sock, &fds_def);

	tv_def.tv_sec = 1;
	tv_def.tv_usec = 0;
	tv.tv_sec = 0;
	tv.tv_usec = 0;

	maxsock = handle->nlroute_sock;
	if (handle->nlxfrm_sock > maxsock)
		maxsock = handle->nlxfrm_sock;
	if (handle->mh_sock > maxsock)
		maxsock = handle->mh_sock;
	if (handle->icmp_sock > maxsock)
		maxsock = handle->icmp_sock;

	if (handle->conf->node_type == NODE_HA)
		set_anycast(handle, handle->conf->home_link_dev);

	while(1) {
		fd_set fds = fds_def; /* reinitialize */

		if (tv.tv_sec <= 0 && tv.tv_usec <= 0) {
			/*
			 * listen to the movement detection
			 * or timer handling...
			 */
			ret = observe(handle);
			if (ret < 0)
				break;

			tv = tv_def; /* reinitialize */
		}

		ret = select(maxsock+1, &fds, NULL, NULL, &tv); 
		if (ret < 0) {
			if (errno == EINTR)
				continue;
			__perror("select");
			break;
		}

		if (FD_ISSET(handle->nlroute_sock, &fds)) {
			int len;
			char buf[BUFLEN];
			memset(buf, 0, sizeof(buf));

			__dprintf("processing NETLINK route packets\n");

			len = recv(handle->nlroute_sock, buf, sizeof(buf), 0);
			if (len < 0) {
				__perror("recv");
				break;
			}

			ret = nlroute_input(handle, buf, len);
			if (ret < 0)
				break;
		}
		if (FD_ISSET(handle->nlxfrm_sock, &fds)) {
			int len;
			char buf[BUFLEN];
			memset(buf, 0, sizeof(buf));

			__dprintf("processing NETLINK xfrm packets\n");

			len = recv(handle->nlxfrm_sock, buf, sizeof(buf), 0);
			if (len < 0) {
				__perror("recv");
				break;
			}

			ret = nlxfrm_input(handle, buf, len);
			if (ret < 0)
				break;
		}
		if (FD_ISSET(handle->mh_sock, &fds)) {
			struct mip6_msg_info msg_info;
			int len = 0;
			char buf[BUFLEN];
			memset(&msg_info, 0, sizeof(msg_info));
			memset(buf, 0, sizeof(buf));

			__dprintf("processing Mobility packets\n");

			ret = pkt_recv(handle->mh_sock, &msg_info, buf, sizeof(buf), &len);
{
	char buf[100];
	printf("%s: daddr = %s\n", __FUNCTION__, inet_ntop(PF_INET6, &msg_info.pktinfo->ipi6_addr, buf, 100));
}
			if (ret < 0)
				break;

			ret = mh_input(handle, &msg_info, (struct mip6_mh_hdr *)buf, len);
			if (ret < 0)
				break;
		}
		if (FD_ISSET(handle->icmp_sock, &fds)) {
			struct mip6_msg_info msg_info;
			int len = 0;
			char buf[BUFLEN];
			memset(&msg_info, 0, sizeof(msg_info));
			memset(buf, 0, sizeof(buf));

			__dprintf("processing ICMPv6 packets\n");

			ret = pkt_recv(handle->icmp_sock, &msg_info, buf, sizeof(buf), &len);
			if (ret < 0)
				break;

			if (len < sizeof(struct icmp6_hdr)) {
				__eprintf("too short message:%d\n", len);
				break;
			}

			ret = icmp_input(handle, &msg_info, (struct icmp6_hdr *)buf, len);
			if (ret < 0)
				break;
		}
	}
	return ret;
}

static int sock_bind(struct mip6d_handle *handle)
{
	struct sockaddr_nl	nladdr;
	struct sockaddr_in6	myaddr;
	struct icmp6_filter	filter;
	const int		on = 1;
	int			ret;

	memset(&nladdr, 0, sizeof(nladdr));
	nladdr.nl_family = AF_NETLINK;
#if 0
	nladdr.nl_groups = CAP_NET_ADMIN;
#else
	nladdr.nl_groups = XFRMGRP_NOTIFY;
#endif
	memset(&myaddr, 0, sizeof(myaddr));
	myaddr.sin6_family = PF_INET6;

	/* XXX: */
	switch (handle->conf->node_type) {
	case NODE_CN:
		break;
	case NODE_HA:
		memcpy(&myaddr.sin6_addr, &handle->conf->ha_addr,
		       sizeof(struct in6_addr));
		break;
	case NODE_MN:
		break;
	}

	/*
	 * NETLINK xfrm bind
	 */
	ret = setsockopt(handle->nlxfrm_sock, SOL_SOCKET, SO_REUSEADDR,
			 &on, sizeof(&on));
	if (ret != 0) {
		__perror("setsockopt");
		goto error;
	}
	ret = bind(handle->nlxfrm_sock, (struct sockaddr *)&nladdr, sizeof(nladdr));
	if (ret != 0) {
		__perror("bind");
		goto error;
	}

	/*
	 * Mobility Header bind
	 */
	/* set socket option to be able to get interface index */
	ret = setsockopt(handle->mh_sock, IPPROTO_IPV6, IPV6_RECVPKTINFO,
			 &on, sizeof(on));
	if (ret < 0) {
		__perror("setsockopt");
		goto error;
	}
	/* Destination Options Header */
	ret = setsockopt(handle->mh_sock, IPPROTO_IPV6, IPV6_RECVDSTOPTS,
			 &on, sizeof(on));
	if (ret < 0) {
		__perror("setsockopt");
		goto error;
	}
	/* Routing Header */
	ret = setsockopt(handle->mh_sock, IPPROTO_IPV6, IPV6_RECVRTHDR,
			 &on, sizeof(on));
	if (ret < 0) {
		__perror("setsockopt");
		goto error;
	}

	ret = bind(handle->mh_sock, (struct sockaddr *)&myaddr, sizeof(myaddr));
	if (ret != 0) {
		__perror("bind");
		goto error;
	}

	/*
	 * ICMPv6 bind
	 */
	/* set socket option to be able to get interface index */
	ret = setsockopt(handle->icmp_sock, IPPROTO_IPV6, IPV6_RECVPKTINFO,
			 &on, sizeof(on));
	if (ret < 0) {
		__perror("setsockopt");
		goto error;
	}
	/* set icmp filter to listen */
	ICMP6_FILTER_SETBLOCKALL(&filter);
	ICMP6_FILTER_SETPASS(ICMPV6_DHAAD_REQUEST, &filter);
	ICMP6_FILTER_SETPASS(ICMPV6_DHAAD_REPLY, &filter);
	ICMP6_FILTER_SETPASS(ND_ROUTER_ADVERT, &filter);
	ICMP6_FILTER_SETPASS(ND_ROUTER_SOLICIT, &filter);
	ICMP6_FILTER_SETPASS(ICMPV6_MOBILE_PREFIX_SOLICIT, &filter);
	ICMP6_FILTER_SETPASS(ICMPV6_MOBILE_PREFIX_ADVERT, &filter);
	ret = setsockopt(handle->icmp_sock, IPPROTO_ICMPV6, ICMP6_FILTER, &filter, sizeof(filter));
	if (ret < 0) {
		__perror("setsockopt");
		goto error;
	}

	/* ICMPv6 socket must be bound to multiple IPv6 addresses */
	ret = setsockopt(handle->icmp_sock, SOL_SOCKET, SO_REUSEADDR,
			 &on, sizeof(&on));
	if (ret != 0) {
		__perror("setsockopt");
		goto error;
	}

	ret = bind(handle->icmp_sock, (struct sockaddr *)&myaddr, sizeof(myaddr));
	if (ret != 0) {
		__perror("bind");
		goto error;
	}

 error:
	return ret;
}

static int sock_init(struct mip6d_handle *handle)
{
	int		sock = 0;

	/*
	 * NETLINK route socket
	 */
	sock = socket(PF_NETLINK, SOCK_RAW, NETLINK_ROUTE);
	if (sock <= 0) {
		__perror("socket");
		goto error;
	}
	handle->nlroute_sock = sock;

	/*
	 * NETLINK xfrm socket
	 */
	sock = socket(PF_NETLINK, SOCK_RAW, NETLINK_XFRM);
	if (sock <= 0) {
		__perror("socket");
		goto error;
	}
	handle->nlxfrm_sock = sock;

	/*
	 * Mobility Header socket
	 */
	sock = socket(AF_INET6, SOCK_RAW, IPPROTO_MOBILITY);
	if (sock <= 0) {
		__perror("socket");
		goto error;
	}
	handle->mh_sock = sock;

	/*
	 * ICMPv6 socket
	 */
	sock = socket(AF_INET6, SOCK_RAW, IPPROTO_ICMPV6);
	if (sock <= 0) {
		__perror("socket");
		goto error;
	}
	handle->icmp_sock = sock;

	return 1;

 error:
	return -errno;
}

static void sock_fini(struct mip6d_handle *handle)
{
	if (handle->nlroute_sock > 0) {
		close(handle->nlroute_sock);
		handle->nlroute_sock = 0;
	}
	if (handle->nlxfrm_sock > 0) {
		close(handle->nlxfrm_sock);
		handle->nlxfrm_sock = 0;
	}
	if (handle->mh_sock > 0) {
		close(handle->mh_sock);
		handle->mh_sock = 0;
	}
	if (handle->icmp_sock > 0) {
		close(handle->icmp_sock);
		handle->icmp_sock = 0;
	}
}

static void signal_caught(int signal)
{
	__dprintf("signal caught: %s.\n", strsignal(signal));

	switch (signal) {
	case SIGHUP:
		__dprintf("currently do nothing.\n");
		break;
	case SIGUSR1:
		__dprintf("currently do nothing.\n");
		break;
	default:
		__dprintf("unknown signal!\n");
		break;
	}
}

static int init(struct mip6d_handle *handle)
{
	int ret;

	if (signal(SIGHUP, signal_caught) == SIG_ERR) {
		__eprintf("signal(%s) failed.\n", strsignal(SIGHUP));
		return -1;
	}
	if (signal(SIGUSR1, signal_caught) == SIG_ERR) {
		__eprintf("signal(%s) failed.\n", strsignal(SIGUSR1));
		return -1;
	}

	ret = handle->init(handle);
	if (ret < 0)
		return ret;

	ret = sock_init(handle);
	if (ret < 0)
		return ret;

	ret = sock_bind(handle);
	if (ret < 0)
		return ret;

	return ret;
}

static void fini(struct mip6d_handle *handle)
{
	sock_fini(handle);

	handle->fini(handle);
}

int net_main(struct mip6d_handle *handle)
{
	int ret;

	ret = init(handle);
	if (ret)
		goto fin;

	ret = loop(handle);
	if (ret)
		goto fin;

 fin:
	fini(handle);
	return ret;
}
