/*
 * Copyright (C)2003 USAGI/WIDE Project
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
/*
 * Authors:
 *      Noriaki TAKAMIYA @USAGI
 *	Masahide NAKAMURA @USAGI
 */

#include <assert.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <linux/xfrm.h>

#include <mip6.h>
#include <mip6mh.h>

#include "util.h"
#include "mip6d.h"

extern struct mip6d_handle *handle;


int ba_output(struct mip6d_handle *handle,
	      struct in6_addr *daddr, struct in6_addr *saddr,
	      int ifindex, unsigned char status, unsigned char flags,
	      unsigned short seq, unsigned short lifetime,
	      void *baopt, int baopt_len)
{
	return mip6mh_ba_output(handle->mh_sock,
				daddr, saddr, ifindex, status, flags, seq, lifetime,
				baopt, baopt_len);
}

int be_output(struct mip6d_handle *handle, struct in6_addr *daddr,
	      unsigned char status, struct in6_addr *be_hoa)
{
	return mip6mh_be_output(handle->mh_sock,
				daddr, status, be_hoa);
}

/* input functions */

static inline int brr_input(struct mip6d_handle *handle,
			    struct mip6_msg_info *msg_info,
			    struct mip6_mh_hdr *mh, ssize_t len)
{
	__dprintf("called\n");
	return 0;
}
static inline int hoti_input(struct mip6d_handle *handle,
			     struct mip6_msg_info *msg_info,
			     struct mip6_mh_hdr *mh, ssize_t len)
{
	__dprintf("called\n");
	return 0;
}
static inline int coti_input(struct mip6d_handle *handle,
			     struct mip6_msg_info *msg_info,
			     struct mip6_mh_hdr *mh, ssize_t len)
{
	__dprintf("called\n");
	return 0;
}
static inline int hot_input(struct mip6d_handle *handle,
			    struct mip6_msg_info *msg_info,
			    struct mip6_mh_hdr *mh, ssize_t len)
{
	__dprintf("called\n");
	return 0;
}
static inline int cot_input(struct mip6d_handle *handle,
			    struct mip6_msg_info *msg_info,
			    struct mip6_mh_hdr *mh, ssize_t len)
{
	__dprintf("called\n");
	return 0;
}

/* XXX: input bu common...
 */
static int bu_input(struct mip6d_handle *handle,
		    struct mip6_msg_info *msg_info,
		    struct mip6_mh_hdr *mh, ssize_t len)
{
	struct mip6_mh_bu *bu = (struct mip6_mh_bu *)mh->data;
	int a_bit = !!(bu->flags & MIP6_BU_F_ACK);
	int h_bit = !!(bu->flags & MIP6_BU_F_HR);
	int l_bit = !!(bu->flags & MIP6_BU_F_LL);
	int k_bit = !!(bu->flags & MIP6_BU_F_KM);
	unsigned char lifetime;
	struct mip6_mh_opt_alt_coa *alt_coa;
	struct mip6_mh_opt_nonce_index *ni;
	struct mip6_mh_opt_binding_auth *auth;
	unsigned char auth_len = 0;
	struct mip6_bc_entry *bce;
	struct in6_addr *hoa, *coa, *saddr, daddr;
	int req_cache;
	int err_code = 0;
	int first_regist = 0;
	int ifindex = 0;
	int ret;

	__dprintf("msg_info->from_addr.sin6_addr = %s\n",
		  in6_straddr(&msg_info->from_addr.sin6_addr));
	__dprintf("msg_info->pktinfo->ipi6_addr = %s\n",
		  in6_straddr(&msg_info->pktinfo->ipi6_addr));
	__dprintf("msg_info->dstopt_hoa = %s\n",
		  in6_straddr(&msg_info->dstopt_hoa));

	memcpy(&daddr, &msg_info->pktinfo->ipi6_addr, sizeof(struct in6_addr));
	ifindex = msg_info->pktinfo->ipi6_ifindex;

	hoa = (msg_info->msg_type & IPV6_RECVDSTOPTS) ?
		&msg_info->dstopt_hoa : &msg_info->from_addr.sin6_addr;

	if (!mip6_addr_unicast_routable(hoa)) {
		__dprintf("not a unicast routable, discarded.\n");
		goto discard;
	}

	/* get mobility options from packet */
	mip6mh_opt_get(mh, len, MIP6_OPT_ALT_COA,
		   (struct mip6_mh_opt **)&alt_coa);
	mip6mh_opt_get(mh, len, MIP6_OPT_NONCE_INDEX,
		   (struct mip6_mh_opt **)&ni);
	mip6mh_opt_get(mh, len, MIP6_OPT_BINDING_AUTH,
		   (struct mip6_mh_opt **)&auth);

	coa = (alt_coa) ? &alt_coa->alt_coa : &msg_info->from_addr.sin6_addr;
	auth_len = (auth) ? (sizeof(*auth) + auth->len) : 0;

	saddr = hoa;

	__dprintf("hoa=%s\n", in6_straddr(hoa));
	__dprintf("coa=%s\n", in6_straddr(coa));


	/* get binding cache entry */
	ret = mip6_bc_lookup(hoa, &bce);
	if (ret < 0) { /* lookup error */
		err_code = ret;
		goto error;
	} else if (ret == 0) { /* not found */

		/*
		 * If HA, create new entry for MN
		 * XXX: The safety is depend on IPsec
		 */
		if (handle->conf->node_type == NODE_HA) {
			__dprintf("hoa = %s\n", in6_straddr(hoa));
			__dprintf("coa = %s\n", in6_straddr(coa));

			memcpy(&handle->conf->hoa, hoa, sizeof(struct in6_addr));
			memcpy(&handle->conf->coa, coa, sizeof(struct in6_addr));
			mip6_ha_set_policy(XFRM_POLICY_OUT, IPPROTO_MOBILITY, IPPROTO_ROUTING, handle);
			ret = mip6_bc_add(hoa, coa, ntohs(bu->lifetime) * 4,
				bu->flags, ntohs(bu->seq), &bce);

			if (ret < 0) {
				bce = NULL;
				err_code = ret;
				goto error;
			}
			first_regist = 1;
		} else {
			/*
			 * CN or MN
			 * CN or MN MUST validate the noce index and binding
			 * authorization data.
			 */
		}
	} else {
		/* found */
		__dprintf("bc entry is found.\n");
	}

	/* XXX: check seq*/
	if (bce && !first_regist) {
		if (!mip6_seq_chk(bu->seq, bce->seq)) {
			__dprintf("sequence check failed bu=%d, bce=%d.\n",
				  bu->seq, bce->seq);

			ba_output(handle, saddr, &daddr, ifindex,
				  MIP6_BA_SEQ_OUT_OF_WINDOW,
				  0, bce->seq, 0,
				  auth, auth_len);
			goto fin;
		}
	}

	if (h_bit) {
		/* Nonce Indices mobility option MUST NOT be present */
		if (ni)
			goto discard;
	} else {
		/* XXX: Nonce */
		if (!ni)
			goto discard;
		;
		/* XXX: regenerating home/care-of keygen token */
		;
		/* XXX: binding auth data */
		if (!auth)
			goto discard;
		/* XXX: binding auth data MUST be the last option and
		 * MUST NOT have trailing padding
		 */
		;
	}

	if (bce) {
		if (h_bit != !!(bce->flags & MIP6_BU_F_HR)) {
			__dprintf("returning ba:139 Registration type change disallowed.\n");
			ba_output(handle, saddr, &daddr, ifindex,
				  MIP6_BA_REG_TYPE_CHANGE_DISALLOWED,
				  0, bu->seq, 0, auth, auth_len);
			goto fin;
		}
	}

	if (!h_bit) {
		/* XXX: required to check and reply a ba
		 * 136 Expired home nonce index
		 * 137 Expired care-of nonce index
		 * 138 Expired nonces
		 * */
		;
	}

	/* XXX: fix me!
	 * XXX: below code is very quick hacking...
	 * XXX: using 'goto' here because of avoiding increasing indent only...
	 */
	if (h_bit)
		goto home_registration;
	else
		goto node_registration;


	/* Requests to Cache/Delete a Binding (maybe CN operation) */
 node_registration:
	req_cache = (bu->lifetime > 0) && (ipv6_addr_cmp(coa, hoa) != 0);
	if (req_cache) {
		if (mip6_bc_may_overflow()) {
			ba_output(handle, saddr, &daddr, ifindex,
				  MIP6_BA_INSUFFICIENT_RESOURCE,
				  0, bu->seq, 0, auth, auth_len);
			goto fin;
		}
	}

	lifetime = bu->lifetime;

	if (!bce) { /* binding cache is not found */
		if (!req_cache) {
			__dprintf("the first BU requests to delete, ignored.\n");
			goto fin;
		}

		ret = mip6_bc_add(hoa, coa,
				  lifetime, bu->flags, bu->seq, &bce);
	} else { /* binding cache is found */
		if (req_cache) {
			ret = mip6_bc_set(bce, coa,
					  lifetime, bu->flags, bu->seq);
		} else {
			/* XXX: CN must remember last nonces... */
			;

			ret = mip6_bc_del(bce);
		}
	}
	if (ret < 0) {
		err_code = ret;
		goto error;
	}

	if (a_bit) {
		/* XXX: bra needed */
		ba_output(handle, saddr, &daddr, ifindex, MIP6_BA_ACCEPTED,
			  0, bu->seq, lifetime, auth, auth_len);
	}

	goto fin;


	/* Primary Care-of Address Registration/De-Registration */
 home_registration:
	if (h_bit) {
		if (handle->conf->node_type != NODE_HA) {
			ba_output(handle, saddr, &daddr, ifindex,
				  MIP6_BA_HOME_REG_NOT_SUPPORTED,
				  0, bu->seq, 0, auth, auth_len);
			goto fin;
		}
	}

	if (ha_prefix_cmp(hoa) != 0) {
		ba_output(handle, saddr, &daddr, ifindex,
			  MIP6_BA_NOT_HOME_SUBNET,
			  0, bu->seq, 0, NULL, 0);
		goto fin;
	}

	req_cache = (bu->lifetime > 0) && (ipv6_addr_cmp(coa, hoa) != 0);
	if (req_cache) {
		if (mip6_bc_may_overflow()) {
			ba_output(handle, saddr, &daddr, ifindex,
				  MIP6_BA_INSUFFICIENT_RESOURCE,
				  0, bu->seq, 0, auth, auth_len);
			goto fin;
		}
	}

	if (!(msg_info->msg_type & IPV6_RECVDSTOPTS)) {
		__dprintf("DSTOPT-HAO is not found.\n");
		/* XXX: */
		goto discard;
	}


	/* XXX: check lifetime */
	/* MUST NOT be greater than the remaining valid lifetime for the
	 * subnet prefix in the mobile node's home address
	 */
	/*ha_prefix_lifetime_get(hoa);*/
	;

	/* XXX: check whether ba's status is 1 or 0
	 * (bu's lifetime is/will_be deprecated or not)
	 */
	;
	lifetime = bu->lifetime;

	if (!bce) { /* binding cache is not found */
		struct mip6_bc_entry *bce_new;

		if (!req_cache) {
			ba_output(handle, saddr, &daddr, ifindex,
				  MIP6_BA_NOT_HOME_AGENT,
				  0, bu->seq, 0, auth, auth_len);
			goto fin;
		}

		ret = mip6_bc_add(hoa, coa,
				  lifetime, bu->flags, bu->seq, &bce_new);
		if (ret < 0) {
			err_code = ret;
			goto error;
		}

		/* DAD */
		ret = ha_dad(hoa, l_bit);
		if (ret < 0) {
			__eprintf("failed dad\n");

			ba_output(handle, saddr, &daddr, ifindex,
				  MIP6_BA_DAD_FAILED,
				  0, bu->seq, 0, auth, auth_len);
			goto fin;
		}

		/* HA needs configure for intercepting packets sent to MN's hoa. */
		ret = ha_tunnel_add(handle, bce_new);
		if (ret < 0) {
			__eprintf("failed to add tunnel\n");

			mip6_bc_del(bce_new);
			err_code = ret;
			goto error;
		}

		/* XXX: enable RT. Just testing. I'm not sure here is correct.  */
		ret = ha_rthdr_add(handle, bce_new);
		if (ret < 0) {
			__eprintf("failed to add rthdr\n");

			ha_tunnel_del(handle, bce_new);
			mip6_bc_del(bce_new);
			err_code = ret;
			goto error;
		}

	} else { /* binding cache is found */
		if (req_cache) {
			ret = mip6_bc_set(bce, coa,
					  lifetime, bu->flags, bu->seq);
			if (ret < 0) {
				err_code = ret;
				goto error;
			}
		} else {
			/* HA needs configure not to intercept packets sent to MN's hoa. */
			ret = ha_tunnel_del(handle, bce);
			if (ret < 0) {
				__eprintf("failed to remove tunnel\n");

				err_code = ret;
				goto error;
			}

			ret = mip6_bc_del(bce);
			if (ret < 0) {
				ha_tunnel_add(handle, bce); /* need? */
				err_code = ret;
				goto error;
			}
		}
	}

	if (a_bit) {
		/* XXX: bra needed */
		ba_output(handle, saddr, &daddr, ifindex,
			  MIP6_BA_ACCEPTED,
			  0, bu->seq, lifetime, auth, auth_len);
	}

	goto fin;
 error:
	__dprintf("error occured in handling bu\n");
	return err_code;
 discard:
	__dprintf("discarded bu\n");
	return 0;
 fin:
	__dprintf("result of handling bu: %d\n", err_code);
	return err_code;
}

static int ba_input(struct mip6d_handle *handle,
		    struct mip6_msg_info *msg_info,
		    struct mip6_mh_hdr *mh, ssize_t len)
{
	__dprintf("called\n");

	/* XXX: currently mn only */
	if (handle->conf->node_type = NODE_MN)
		return mn_ba_input(handle, msg_info, mh, len);

	return 0;
}
static inline int be_input(struct mip6d_handle *handle,
			   struct mip6_msg_info *msg_info,
			   struct mip6_mh_hdr *mh, ssize_t len)
{
	__dprintf("called\n");
	return 0;
}


int mh_input(struct mip6d_handle *handle,
	     struct mip6_msg_info *msg_info,
	     struct mip6_mh_hdr *mh, int len)
{
	struct mip6_mh_opt *opt;
	int err_code = 0;

	__dprintf("MH received\n");

	if (len < sizeof(struct mip6_mh_hdr) ||
	    len < sizeof(struct mip6_mh_hdr) + mh->hdrlen) {
		__eprintf("too short message:%d\n", len);
		err_code = -EINVAL;
		goto fin;
	}

/* XXX: There's no need to verify here because kernel might already verify. */
#if 1
	/* payload proto */
	if (mh->nexthdr != IPPROTO_NONE) {
		__eprintf("%s: invalid payload proto:%d\n", __FUNCTION__,
			mh->nexthdr);
		err_code = -EINVAL;
		goto fin;
	}
	/* header length */
	if (mh->hdrlen <= 0) {
		__eprintf("%s: invalid header length:%d\n", __FUNCTION__,
			  mh->hdrlen);
		err_code = -EINVAL;
		goto fin;
	}

	/* reserved */
	__dprintf("reserved area is: %d(0x%x)\n", mh->reserved, mh->reserved);
#endif

	__dprintf("mh type=%s\n", strmh(mh->type));

	/* process according to the mobility header type */
	switch (mh->type) {
	case MIP6_MH_BRR:	/* Binding Refresh Request */
	case MIP6_MH_HOTI:	/* Home Test Init */
	case MIP6_MH_COTI:	/* Care-of Test Init */
		break;
	case MIP6_MH_HOT:	/* Home Test */
	case MIP6_MH_COT:	/* Care-of Test */
	{
		struct mip6_mh_test *t = (struct mip6_mh_test *)mh->data;
		t->nonce_index = ntohs(t->nonce_index);
		break;
	}
	case MIP6_MH_BU:	/* Binding Update */
	{
		struct mip6_mh_bu *bu = (struct mip6_mh_bu *)mh->data;
		bu->lifetime = ntohs(bu->lifetime);
		bu->seq = ntohs(bu->seq);
		break;
	}
	case MIP6_MH_BA:	/* Binding Acknowledgement */
	{
		struct mip6_mh_ba *ba = (struct mip6_mh_ba *)mh->data;
		ba->lifetime = ntohs(ba->lifetime);
		ba->seq = ntohs(ba->seq);
		break;
	}
	case MIP6_MH_BE:	/* Binding Error */
		break;
	default:		/* Unknown MH type */
	{
		struct in6_addr *hoa = NULL;

		__eprintf("%s: unrecognized Mobility Header type received:%d\n", __FUNCTION__, mh->type);
		/* required to be sent Binding Error */

		if ((msg_info->msg_type & IPV6_RECVDSTOPTS) != 0)
			hoa = &msg_info->dstopt_hoa;

		be_output(handle, &msg_info->from_addr.sin6_addr, MIP6_BE_UNRECOGNIZED_MH, hoa);
		goto fin;
	}}

	/* mobility options: only nonce index has members who need ntohs */
	if (mip6mh_opt_get(mh, len, MIP6_OPT_NONCE_INDEX, &opt) >= 0) {
		struct mip6_mh_opt_nonce_index *ni = (struct mip6_mh_opt_nonce_index *)opt;
		ni->home_nonce_index = ntohs(ni->home_nonce_index);
		ni->careof_nonce_index = ntohs(ni->careof_nonce_index);
	}

	/* XXX: call functions */
	switch (mh->type) {
	case MIP6_MH_BRR:
		err_code = brr_input(handle, msg_info, mh, len);
		break;
	case MIP6_MH_HOTI:
		err_code = hoti_input(handle, msg_info, mh, len);
		break;
	case MIP6_MH_COTI:
		err_code = coti_input(handle, msg_info, mh, len);
		break;
	case MIP6_MH_HOT:
		err_code = hot_input(handle, msg_info, mh, len);
		break;
	case MIP6_MH_COT:
		err_code = cot_input(handle, msg_info, mh, len);
		break;
	case MIP6_MH_BU:
		err_code = bu_input(handle, msg_info, mh, len);
		break;
	case MIP6_MH_BA:
		err_code = ba_input(handle, msg_info, mh, len);
		break;
	case MIP6_MH_BE:
		err_code = be_input(handle, msg_info, mh, len);
		break;
	default:
		break;
	}

 fin:
	return err_code;
}
