/* $USAGI: strutil.c,v 1.2 2004/04/24 07:13:17 nakam Exp $ */

/*
 * Copyright (C)2004 USAGI/WIDE Project
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
/*
 * Authors:
 *	Masahide NAKAMURA @USAGI
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#include <time.h>
#include <linux/xfrm.h>

#include "mip6.h"

const struct sockaddr *getaddr(const char *str, int hints_family,
			       size_t *addrlen, int *family)
{
	static struct sockaddr_storage ss;
	struct addrinfo hints;
	struct addrinfo *res = NULL;
	int ret;

	memset(&hints, 0, sizeof(hints));
	hints.ai_socktype = SOCK_RAW;
	hints.ai_family = hints_family;
	hints.ai_flags = AI_PASSIVE;

	ret = getaddrinfo(str, NULL, &hints, &res);
	if (ret != 0) {
		printf("getaddrinfo: \"%s\" :%s\n", str, gai_strerror(ret));
		goto fin;
	}
	if (res->ai_addrlen > sizeof(ss)) {
		printf("getaddrinfo: \"%s\" : sockaddr too large:%d\n",
		       str, res->ai_addrlen);
		ret = -1;
		goto fin;
	}
	memcpy(&ss, res->ai_addr, res->ai_addrlen);
	if (addrlen)
		*addrlen = res->ai_addrlen;
	if (family)
		*family = res->ai_family;

 fin:
	if (res)
		freeaddrinfo(res);

	if (ret == 0)
		return (const struct sockaddr *)&ss;
	else
		return NULL;
}

const xfrm_address_t *getxfaddr(const char *str, int hints_family)
{
	const struct sockaddr *sa;
	size_t salen = 0;
	int family = PF_UNSPEC;
	xfrm_address_t *xa = NULL;

	sa = getaddr(str, hints_family, &salen, &family);
	if (!sa)
		return NULL;

	switch (family) {
	case PF_INET:
		if (salen > sizeof(struct sockaddr_in)) {
			printf("sockaddr too large: %d\n", salen);
			return NULL;
		}
		xa = (xfrm_address_t *)&((struct sockaddr_in *)sa)->sin_addr;
		break;
	case PF_INET6:
		if (salen > sizeof(struct sockaddr_in6)) {
			printf("sockaddr too large: %d\n", salen);
			return NULL;
		}
		xa = (xfrm_address_t *)&((struct sockaddr_in6 *)sa)->sin6_addr;
		break;
	default:
		printf("unsupported family: %d\n", family);
		break;
	}

	return xa;
}

const char *straddr(const struct sockaddr *sa, int salen)
{
	static char str[NI_MAXHOST];
	int ret;

	ret = getnameinfo(sa, salen, str, sizeof(str),
			  NULL, 0, NI_NUMERICHOST);
	if (ret != 0) {
		printf("getnameinfo:%s\n", gai_strerror(ret));
		return NULL;
	}

	return str;
}

const char *strxfaddr(const xfrm_address_t *xa, int family)
{
	struct sockaddr_storage ss;
	struct sockaddr *sa;
	int salen = 0;

	memset(&ss, 0, sizeof(ss));

	sa = (struct sockaddr *)&ss;
	sa->sa_family = family;

	switch (family) {
	case PF_INET:
	{
		struct sockaddr_in *sin = (struct sockaddr_in *)sa;
		memcpy(&sin->sin_addr, xa, sizeof(sin->sin_addr));
		salen = sizeof(*sin);
		break;
	}
	case PF_INET6:
	{
		struct sockaddr_in6 *sin6 = (struct sockaddr_in6 *)sa;
		memcpy(&sin6->sin6_addr, xa, sizeof(sin6->sin6_addr));
		salen = sizeof(*sin6);
		break;
	}
	default:
		printf("unsupported family: %d\n", family);
		return NULL;
	}

	return straddr(sa, salen);
}

const char *strproto(__u8 proto)
{
	static char str[32];
	struct protoent *pp;

	if (proto == 0) {
		strcpy(str, "any");
		goto fin;
	}

	pp = getprotobynumber(proto);
	if (!pp)
		sprintf(str, "%d", proto);
	else
		strcpy(str, pp->p_name);
 fin:
	return str;
}

const char *mip6_strproto(__u8 proto)
{
	static char str[64];
	const char *str0;

	str0 = strproto(proto);

	switch (proto) {
	case IPPROTO_ROUTING:
		sprintf(str, "route2(%s)", str0);
		break;
	case IPPROTO_DSTOPTS:
		sprintf(str, "hao(%s)", str0);
		break;
#if 0
	case IPPROTO_MH:
		strcpy(str, "ipv6-mh");
		break;
#endif
	default:
		strcpy(str, str0);
		break;
	}
	return str;
}

int getproto(const char *str)
{
	struct protoent *pp;
	__u8 proto;

	if (strcmp(str, "any") == 0) {
		proto = 0;
		goto fin;
	}
	pp = getprotobyname(str);
	if (!pp)
		proto = strtol(str, NULL, 0);
	else
		proto = pp->p_proto;

 fin:
	return proto;
}


int mip6_getproto(const char *str)
{
	__u8 proto;

	if (strcmp(str, "route2") == 0)
		proto = IPPROTO_ROUTING;
	else if (strcmp(str, "hao") == 0)
		proto = IPPROTO_DSTOPTS;
#if 0
	else if (strcmp(str, "ipv6-mh") == 0)
		proto = IPPROTO_MH;
#endif
	else
		proto = getproto(str);

	return proto;
}

const char *strmh(__u8 type)
{
	static char str[32];

	switch (type) {
	case MIP6_MH_BRR:
		/* Binding Refresh Request */
		strcpy(str, "brr");
		break;
	case MIP6_MH_HOTI:
		/* Home Test Init */
		strcpy(str, "hoti");
		break;
	case MIP6_MH_COTI:
		/* Care-of Test Init */
		strcpy(str, "coti");
		break;
	case MIP6_MH_HOT:
		/* Home Test */
		strcpy(str, "hot");
		break;
	case MIP6_MH_COT:
		/* Care-of Test */
		strcpy(str, "cot");
		break;
	case MIP6_MH_BU:
		/* Binding Update */
		strcpy(str, "bu");
		break;
	case MIP6_MH_BA:
		/* Binding Acknowledgement */
		strcpy(str, "ba");
		break;
	case MIP6_MH_BE:
		/* Binding Error */
		strcpy(str, "be");
		break;
	default:
		sprintf(str, "unknown-mh(%d)", type);
		break;
	}

	return str;
}

/* XXX: currently mip6 related types only. */
const char *stricmp6(__u8 type)
{
	static char str[32];

	switch(type) {
	case ICMPV6_DHAAD_REQUEST:
		strcpy(str, "dhaad request");
		break;
	case ICMPV6_DHAAD_REPLY:
		strcpy(str, "dhaad reply");
		break;
	case ICMPV6_MOBILE_PREFIX_SOLICIT:
		strcpy(str, "mps");
		break;
	case ICMPV6_MOBILE_PREFIX_ADVERT:
		strcpy(str, "mpa");
		break;
	default:
		sprintf(str, "%d", type);
		break;
	}

	return str;
}

const char *strxfpolicy_dir(__u8 dir)
{
	static char str[32];

	switch (dir) {
	case XFRM_POLICY_IN:
		strcpy(str, "in");
		break;
	case XFRM_POLICY_OUT:
		strcpy(str, "out");
		break;
	case XFRM_POLICY_FWD:
		strcpy(str, "fwd");
		break;
	default:
		sprintf(str, "unknown-dir(%d)", dir);
		break;
	}

	return str;
}

const char *strxfpolicy_action(__u8 action)
{
	static char str[32];

	switch (action) {
	case XFRM_POLICY_ALLOW:
		strcpy(str, "allow");
		break;
	case XFRM_POLICY_BLOCK:
		strcpy(str, "block");
		break;
	default:
		sprintf(str, "uknown-action(%d)", action);
		break;
	}

	return str;
}

const char *strxfpolicy_flags(__u8 flags)
{
	static char str[256];


	memset(str, '\0', sizeof(str));

	if (flags == 0)
		strcpy(str, "none");

	else if (flags & XFRM_POLICY_LOCALOK) {
		if (strlen(str) > 0)
			strcat(str, ",");
		strcat(str, "localok");
	} else
		sprintf(str, "unknown-flags(%d)", flags);

	return str;
}

const char *strxfshare(__u8 share)
{
	static char str[32];
	switch (share) {
	case XFRM_SHARE_ANY:		/* No limitations */
		strcpy(str, "any");
		break;
	case XFRM_SHARE_SESSION:	/* For this session only */
		strcpy(str, "session");
		break;
	case XFRM_SHARE_USER:	/* For this user only */
		strcpy(str, "user");
		break;
	case XFRM_SHARE_UNIQUE:	/* Use once */
		strcpy(str, "unique");
		break;
	default:
		sprintf(str, "unknown-share(%d)", share);
		break;
	}
	return str;
}

const char *strxflimit(__u64 limit)
{
	static char str[32];
	if (limit == XFRM_INF)
		strcpy(str, "(INF)");
	else
		sprintf(str, "%llu", limit);

	return str;
}

const char *strxftime(__u64 time)
{
	static char str[32];
	struct tm *tp;
	time_t t;

	if (time == 0) {
		strcpy(str, "(undefined)");
	} else {
		/* XXX: treat time in the same manner of xfrm_user.c, xfrm_state.c */
		t = (long)time;
		tp = localtime(&t);

		sprintf(str, "%04d/%02d/%02d %02d:%02d:%02d",
			tp->tm_year + 1900, tp->tm_mon + 1, tp->tm_mday,
			tp->tm_hour, tp->tm_min, tp->tm_sec);
	}

	return str;
}
