/* Copyright (c) 1996, 1997, 1998, 1999 Thorsten Kukuk
   Author: Thorsten Kukuk <kukuk@suse.de>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#define _GNU_SOURCE

#ifdef HAVE_GETOPT_H
#include <getopt.h>
#else
#include "lib/compat/getopt.h"
#endif
#include <stdio.h>
#include <locale.h>
#include <libintl.h>
#include <rpcsvc/nis.h>

#ifndef _
#define _(String) gettext (String)
#endif

/* Print the version information.  */
static inline void
print_version (void)
{
  fprintf (stdout, "nisdefaults (%s) %s\n", PACKAGE, VERSION);
  fprintf (stdout, gettext ("\
Copyright (C) %s Thorsten Kukuk.\n\
This is free software; see the source for copying conditions.  There is NO\n\
warranty; not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.\n\
"), "1998, 1999");
  /* fprintf (stdout, _("Written by %s.\n"), "Thorsten Kukuk"); */
}

static inline void
print_usage (void)
{
  fputs (_("Usage: nisdefaults [-adghprstv]\n"), stdout);
}

static void
print_help (void)
{
  print_usage ();
  fputs (_("nisdefaults - display NIS+ default values\n\n"),  stdout);

  fputs (_("  -a             Print defaults in a terse format\n"), stdout);
  fputs (_("  -d             Print default domain name\n"), stdout);
  fputs (_("  -g             Print default group name\n"), stdout);
  fputs (_("  -h             Print default hostname\n"), stdout);
  fputs (_("  -p             Print default principal name\n"), stdout);
  fputs (_("  -r             Print default access rights for new objects\n"),
	 stdout);
  fputs (_("  -s             Print default directory search path\n"), stdout);
  fputs (_("  -t             Print default time to live value\n"), stdout);
  fputs (_("  -v, --verbose  Print defaults in a verbose format\n"), stdout);
  fputs (_("  --help         Give this help list\n"), stdout);
  fputs (_("  --usage        Give a short usage message\n"), stdout);
  fputs (_("  --version      Print program version\n"), stdout);
}

static void
print_error (void)
{
  const char *program = "nisdefaults";

  fprintf (stderr,
           _("Try `%s --help' or `%s --usage' for more information.\n"),
           program, program);
}

int
main (int argc, char *argv[])
{
  int p_domain = 0;
  int p_group = 0;
  int p_host = 0;
  int p_principal = 0;
  int p_access = 0;
  int p_search = 0;
  int p_time = 0;
  int p_verbose = 0;

  setlocale (LC_MESSAGES, "");
  bindtextdomain (PACKAGE, LOCALEDIR);
  textdomain (PACKAGE);

  while (1)
    {
      int c;
      int option_index = 0;
      static struct option long_options[] =
      {
        {"version", no_argument, NULL, '\255'},
        {"usage", no_argument, NULL, '\254'},
        {"help", no_argument, NULL, '\253'},
	{"verbose", no_argument, NULL, 'v'},
	{NULL, 0, NULL, '\0'}
      };

      c = getopt_long (argc, argv, "adghprstv", long_options, &option_index);
      if (c == (-1))
	break;
      switch (c)
	{
	case 'a':
	  p_domain = p_group = p_host = p_principal =
	    p_access = p_search = p_time = 1;
	  break;
	case 'd':
	  p_domain = 1;
	  break;
	case 'g':
	  p_group = 1;
	  break;
	case 'h':
	  p_host = 1;
	  break;
	case 'p':
	  p_principal = 1;
	  break;
	case 'r':
	  p_access = 1;
	  break;
	case 's':
	  p_search = 1;
	  break;
	case 't':
	  p_time = 1;
	  break;
	case 'v':
	  p_verbose = 1;
	  break;
        case '\253':
          print_help ();
          return 0;
        case '\255':
          print_version ();
          return 0;
        case '\254':
          print_usage ();
          return 0;
        default:
          print_error ();
          return 1;
        }
    }

  argc -= optind;
  argv += optind;

  if (argc != 0)
    {
      fputs (_("nisdefaults: To many arguments\n"), stderr);
      print_error ();
      return 1;
    }

  if (p_domain == 0 && p_group == 0 && p_host == 0 && p_principal == 0 &&
      p_access == 0 && p_search == 0 && p_time == 0)
    {
      p_domain = p_group = p_host = p_principal =
	p_access = p_search = p_time = 1;
      p_verbose = 1;
    }

  if (p_principal)
    {
      char local_principal[NIS_MAXNAMELEN];

      strcpy (local_principal, __nis_default_owner (NULL));
      if (!p_verbose)
	fputs (local_principal, stdout);
      else
	{
	  fprintf (stdout, "Principal Name : %s", local_principal);
	  if (!key_secretkey_is_set())
	    fputs (_(" (not authenticated)"), stdout);
	}
      fputs ("\n", stdout);
    }

  if (p_domain)
    {
      char local_domain[NIS_MAXNAMELEN];

      if (nis_local_directory () != NULL)
	strcpy (local_domain, nis_local_directory ());
      else
	strcpy (local_domain,
		_("Broken nis_local_directory() function in libc!"));
      if (!p_verbose)
	fprintf (stdout, "%s\n", local_domain);
      else
	fprintf (stdout, "Domain Name    : %s\n", local_domain);
    }

  if (p_host)
    {
      char local_host[NIS_MAXNAMELEN];

      strcpy (local_host, nis_local_host ());
      if (!p_verbose)
	fprintf (stdout, "%s\n", local_host);
      else
	fprintf (stdout, "Host Name      : %s\n", local_host);
    }

  if (p_group)
    {
      char local_group[NIS_MAXNAMELEN];

      strcpy (local_group, __nis_default_group (NULL));
      if (!p_verbose)
	fprintf (stdout, "%s\n", local_group);
      else
	fprintf (stdout, "Group Name     : %s\n", local_group);
    }

  if (p_access)
    {
      if (!(!p_verbose))
	fprintf (stdout, "Access Rights  : ");
      nis_print_rights (__nis_default_access (NULL, 0));
      fprintf (stdout, "\n");
    }

  if (p_time)
    {
      unsigned long int ttltime;

      ttltime = __nis_default_ttl (NULL);
      if (!p_verbose)
	fprintf (stdout, "%lu\n", ttltime);
      else
	{
	  unsigned long int s, m, h;
	  h = ttltime / (60 * 60);
	  ttltime %= (60 * 60);
	  m = ttltime / 60;
	  ttltime %= 60;
	  s = ttltime;
	  fprintf (stdout, "Time to live   : ");
	  if (h != 0)
	    fprintf (stdout, "%lu:", h);
	  if (h != 0)
	    fprintf (stdout, "%02lu:", m);
	  else
	    fprintf (stdout, "%lu:", m);
	  if (h != 0 || m != 0)
	    fprintf (stdout, "%02lu\n", s);
	  else
	    fprintf (stdout, "%lu\n", s);
	}
    }

  if (p_search)
    {
      nis_name *path;

      path = nis_getnames ("");
      if (path == NULL || *path == NULL)
	return 1;
      else
	{
	  int i = 1;

	  if (!p_verbose)
	    fprintf (stdout, "%s\n", path[0] + 1);
	  else
	    fprintf (stdout, "Search Path    : %s\n", path[0] + 1);

	  while (path[i] != NULL)
	    {
	      if (!p_verbose)
		fprintf (stdout, "%s\n", path[i] + 1);
	      else
		fprintf (stdout, "               : %s\n", path[i] + 1);
	      ++i;
	    }
	}
    }

  return 0;
}
