/***************************************************************************
*   Copyright (C) 2003 by Mark Kretschmann                                *
*   markey@web.de                                                         *
*   based on work by Charles Samuels <charles@kde.org>                    *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

#include <sys/types.h>
#include <sys/sem.h>
#include <sys/stat.h>
#include <sys/shm.h>
#include <sys/wait.h>
#include <sys/mman.h>

#include <assert.h>
#include <errno.h>
#include <fcntl.h>
#include <math.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string>
#include <map>

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include <audiosubsys.h>
#include <stdsynthmodule.h>
#include <convert.h>
#include <debug.h>

#include "libmodplug/stdafx.h"
#include "libmodplug/sndfile.h"

#include "configparser/configparser.h"
#include "modplug.h"
#include "ModplugPlayObject_impl.h"


using namespace Arts;


ModplugPlayObject_impl::ModplugPlayObject_impl()
        : m_map( NULL )
        , m_bufLen( 1000000 )
        , m_pBuffer( new signed short[ m_bufLen ] )
        , m_state( posIdle )
{
    loadConfig();
}


ModplugPlayObject_impl::~ModplugPlayObject_impl()
{
    halt();
    saveConfig();

    ::munmap( m_map, m_size );
    ::close( m_fd );
    m_map = NULL;

    delete[] m_pBuffer;
}


void ModplugPlayObject_impl::loadConfig()
{
    std::string path = getenv( "HOME" );
    path += "/.modplugartsrc";
    m_configParser.readConfig( path );

    xbass_depth( m_configParser.readUintEntry( "xbass_depth", 0 ) );
    xbass_range( m_configParser.readUintEntry( "xbass_range", 0 ) );

    reverb_depth( m_configParser.readUintEntry( "reverb_depth", 0 ) );
    reverb_delay( m_configParser.readUintEntry( "reverb_delay", 0 ) );

    surround_depth( m_configParser.readUintEntry( "surround_depth", 0 ) );
    surround_delay( m_configParser.readUintEntry( "surround_delay", 0 ) );

    m_resamplingMap[ "nearest" ] = SRCMODE_NEAREST;
    m_resamplingMap[ "linear" ] = SRCMODE_LINEAR;
    m_resamplingMap[ "spline" ] = SRCMODE_SPLINE;
    m_resamplingMap[ "polyphase" ] = SRCMODE_POLYPHASE;

    resampling_mode( m_configParser.readStringEntry( "resampling_mode", "polyphase" ) );

    m_mod.SetRepeatCount( 0 );
}


void ModplugPlayObject_impl::saveConfig()
{
    m_configParser.writeEntry( "version", APP_VERSION );

    m_configParser.writeEntry( "xbass_depth", m_xbassDepth );
    m_configParser.writeEntry( "xbass_range", m_xbassRange );

    m_configParser.writeEntry( "reverb_depth", m_reverbDepth );
    m_configParser.writeEntry( "reverb_delay", m_reverbDelay );

    m_configParser.writeEntry( "surround_depth", m_surroundDepth );
    m_configParser.writeEntry( "surround_delay", m_surroundDelay ) ;

    m_configParser.writeEntry( "resampling_mode", m_resamplingMode );

    m_configParser.writeConfig();
}


void ModplugPlayObject_impl::updateConfig()
{
    CSoundFile::SetXBassParameters( m_xbassDepth, m_xbassRange );
    CSoundFile::SetReverbParameters( m_reverbDepth, m_reverbDelay );
    CSoundFile::SetSurroundParameters( m_surroundDepth, m_surroundDelay );

    CSoundFile::SetWaveConfig( samplingRate, 16, 2 );
    CSoundFile::SetWaveConfigEx( true, false, true, true, true, true, false );

    CSoundFile::SetResamplingMode( m_resamplingMap[ m_resamplingMode ] );
}



bool ModplugPlayObject_impl::loadMedia( const std::string &filename )
{
    arts_debug( "Modplug: loadMedia %s", filename.c_str() );
    m_fd = ::open( filename.c_str(), O_RDONLY );

    struct stat lStat;

    // open and mmap the file
    if ( m_fd == -1 )
        return false;

    ::fstat( m_fd, &lStat );
    m_size = lStat.st_size;

    m_map = ( unsigned char* ) ::mmap( 0, m_size, PROT_READ, MAP_PRIVATE, m_fd, 0 );
    
    if ( !m_map )
    {
        ::close( m_fd );
        arts_debug( "Modplug: mmap error!" );
        return false;
    }
    else
    {
        m_mod.Create( m_map, m_size );
        m_len = m_mod.GetSongTime();
        updateConfig();
        return true;
    }
}


std::string ModplugPlayObject_impl::description()
{
    return "modplug artsplugin";
}


poTime ModplugPlayObject_impl::currentTime()
{
    poTime time;

    float pos = ( ( float ) m_mod.GetCurrentPos() / ( float ) m_mod.GetMaxPosition() ) * ( float ) m_mod.GetSongTime();

    time.ms = 0;
    time.seconds = ( int ) pos;

    return time;
}


poTime ModplugPlayObject_impl::overallTime()
{
    poTime time;
    time.ms = 0;
    time.seconds = m_mod.GetSongTime();

    return time;
}


poCapabilities ModplugPlayObject_impl::capabilities()
{
    return static_cast<poCapabilities>( capPause | capSeek );
}


std::string ModplugPlayObject_impl::mediaName()
{
    return std::string( m_mod.GetTitle() );
}


poState ModplugPlayObject_impl::state()
{
    return m_state;
}


void ModplugPlayObject_impl::play()
{
    m_state = posPlaying;
}


void ModplugPlayObject_impl::halt()
{
    m_state = posIdle;
}


void ModplugPlayObject_impl::seek( const class poTime &t )
{
    float pos = ( ( float ) t.seconds / ( float ) m_mod.GetSongTime() ) * ( float ) m_mod.GetMaxPosition();

    m_mod.SetCurrentPos( ( unsigned int ) pos );
}


void ModplugPlayObject_impl::pause()
{
    m_state = posPaused;
}


void ModplugPlayObject_impl::streamInit()
{
    arts_debug( "modplug: streamInit" );
}


void ModplugPlayObject_impl::streamStart()
{
    arts_debug( "modplug: streamStart" );
}


void ModplugPlayObject_impl::calculateBlock( unsigned long samples )
{
    if ( m_mod.GetCurrentPos() >= m_mod.GetMaxPosition() )
    {
        m_state = posPaused ;
        halt();
    }

    if ( m_state == posPlaying )
    {
        if ( m_mod.Read( ( LPVOID ) m_pBuffer, samples * 2 * 2 ) )

            for ( unsigned int i = 0; i < samples; i++ )
            {
                left[ i ] = ( float ) ( m_pBuffer[ i * 2 + 0 ] ) / 32768.0;
                right[ i ] = ( float ) ( m_pBuffer[ i * 2 + 1 ] ) / 32768.0;
            }
    }

    else
    {
        for ( unsigned int i = 0; i < samples; i++ )
        {
            memset( left, 0, samples * sizeof( float ) );
            memset( right, 0, samples * sizeof( float ) );
        }
    }
}


void ModplugPlayObject_impl::streamEnd()
{
    arts_debug( "modplug: streamEnd" );
}


// XBASS ---------------

float ModplugPlayObject_impl::xbass_depth()
{
    return static_cast<float>( m_xbassDepth );
}


void ModplugPlayObject_impl::xbass_depth( float newval )
{
    m_xbassDepth = static_cast<unsigned int>( newval );
    updateConfig();
}


float ModplugPlayObject_impl::xbass_range()
{
    return static_cast<float>( m_xbassRange );
}


void ModplugPlayObject_impl::xbass_range( float newval )
{
    m_xbassRange = static_cast<unsigned int>( newval );
    updateConfig();
}


// REVERB ---------------

float ModplugPlayObject_impl::reverb_depth()
{
    return static_cast<float>( m_reverbDepth );
}


void ModplugPlayObject_impl::reverb_depth( float newval )
{
    m_reverbDepth = static_cast<unsigned int>( newval );
    updateConfig();
}


float ModplugPlayObject_impl::reverb_delay()
{
    return static_cast<float>( m_reverbDelay );
}


void ModplugPlayObject_impl::reverb_delay( float newval )
{
    m_reverbDelay = static_cast<unsigned int>( newval );
    updateConfig();
}


// SURROUND ----------------

float ModplugPlayObject_impl::surround_depth()
{
    return static_cast<float>( m_surroundDepth );
}


void ModplugPlayObject_impl::surround_depth( float newval )
{
    m_surroundDepth = static_cast<unsigned int>( newval );
    updateConfig();
}


float ModplugPlayObject_impl::surround_delay()
{
    return static_cast<float>( m_surroundDelay );
}


void ModplugPlayObject_impl::surround_delay( float newval )
{
    m_surroundDelay = static_cast<unsigned int>( newval );
    updateConfig();
}


// RESAMPLING ----------------

string ModplugPlayObject_impl::resampling_mode()
{
    return m_resamplingMode;
}


void ModplugPlayObject_impl::resampling_mode( const string &newval )
{
    m_resamplingMode = newval;
    updateConfig();
}



REGISTER_IMPLEMENTATION( ModplugPlayObject_impl );
